;+
; Project     : STEREO - PLASTIC
;
; Name        : plastic_process_sphk
;
; Purpose     : Parse Level 0 conversted spacecraft hk (csh) file, and put in daily CDF file.
;
; Category    : Telemetry, Data-extraction
;
;
; Syntax      : IDL> plastic_process_sphk
;
; Inputs      : infilename
;
; Opt. Inputs : infilepath, outfilepath, errorfilepath
;
; Outputs     : none
;
; Opt. Outputs: none
;
; Common      : none
;
; Restrictions: requires environmental variable SKELETONCDF be set
;               requires skeleton file PLA_L1_SP_00000000_V01.skt in data/skeleton/
;               directory
;
; Side effects: creates cdf file in filepath directory
;
; History     : version 1, 10/20/2006, LB Ellis, written
;               version 3, 12/20/2006 LBEllis changed skt files
;               version 4, 12/21/2006 LBE added pwr_vt and surv_pwr_vt
;               to skt file
;               version 5, 01/24/2007 LBE changed HK skt
;               version 6, 01/31/2007 LBE no change from v. 5
;               version 7, 02/07/2007 LBE no change from v. 6
;                          02/13/2007 LBE Added want_to_read
;               version 8, 02/15/2007 LBE no change from v. 7
;               version 9, 03/20/2007 LBE no change from v. 8
;                          08/27/08 Removed Katherine Singer from email
;               version 10, 06/02/2011 LBE no change from V09
;                           02/05/2013 LBE Add skip_email keyword.
;                           02/18/2014 LBE Change so Toni only gets email for first hk error log.
;                                          Add John Gaidos to email for first hk error log.
;                           03/24/15 Only send email to me when behind sun
;                           05/11/200  LBE Add Nick Lubinsky to email
;
; Contact     : Lorna Ellis (Lorna.Ellis@unh.edu)
;-


PRO plastic_process_sphk, infilename, infilepath=infilepath, outfilepath = outfilepath, errorfilepath = errorfilepath, cdfreadfilepath = cdfreadfilepath, version = version, skip_email = skip_email
compile_opt strictarrsubs
COMMON share1
hkskeleton_name = '/data1/SKT/PLA_L1_SC_00000000_V'+string(version, format='(I2.2)')
;hkskeleton_name = '/Users/lorna/Stereo/Level1/SKT/PLA_L1_SC_00000000_V'+string(version, format='(I2.2)')

pl_ebox_tmp_limit     = [-25.00, -20.00,  30.00,  40.00] ;   // email from Mark Nov. 17, 2006
pl_ebox_tmp_out = 0 ; set to one if goes out of limits
; pl_inst_msg_pwr_dwn_req = delta
pl_pwr_cur_limit      = [  0.10,   0.20,   0.46,   0.52]
pl_pwr_cur_out = 0
; pl_pwr_tt = delta
pl_surv_pwr_cur_limit = [ -1.00,  -1.00,   0.50,   0.60] ; if -1 in limit, check code
pl_surv_pwr_cur_out = 0
; pl_surv_pwr_tt = delta
pl_tof_tmp_limit      = [-25.00, -20.00,  30.00,  40.00]
pl_tof_tmp_out = 0

pl_pwr_vt_limit       = [  0.10,  23.00,  34.00,  36.00]
pl_pwr_vt_out = 0
pl_surv_pwr_vt_limit  = [  0.10,  23.00,  34.00,  36.00]
pl_surv_pwr_vt_out = 0

old_pl_ebox_tmp = !values.f_nan
old_pl_inst_msg_pwr_dwn_req = ''
old_pl_pwr_cur = !values.f_nan
old_pl_pwr_tt = ''
old_pl_surv_pwr_cur = !values.f_nan
old_pl_surv_pwr_tt = ''
old_pl_tof_tmp = !values.f_nan
old_pl_pwr_vt = !values.f_nan
old_pl_surv_pwr_vt = !values.f_nan

temp = strmid(infilename, 0, 1)
IF temp[0] EQ 'a' THEN BEGIN    ; 'ahead'
    instrument_id = 'FM1'
    spacecraft = 'ahead'
    year_st = strmid(infilename, 6, 4)
    doy_st = strmid(infilename, 11, 3)
ENDIF ELSE IF temp[0] EQ 'b' THEN BEGIN 
    instrument_id = 'FM2'
    spacecraft = 'behind'
    year_st = strmid(infilename, 7, 4)
    doy_st = strmid(infilename, 12, 3)
ENDIF ELSE BEGIN
    print, 'invalid infile name: ', infilename
    stop
ENDELSE 
year = fix(year_st)
doy = fix(doy_st)
if n_elements(infilepath) eq 1 then infile = concat_dir(infilepath, infilename) ELSE infile = infilename
openr, in_lun, infile, /get_lun
hkerrorfilename = infilename+'.sphkerrors'
if n_elements(errorfilepath) eq 1 then hkerrorfile = concat_dir(errorfilepath, hkerrorfilename) ELSE hkerrorfile = hkerrorfilename
openw, hkerror_lun, hkerrorfile, /get_lun

IF spacecraft EQ 'ahead' THEN temp_spacecraft = 'A' ELSE temp_spacecraft = 'B'
IF year LT 2001 OR year GT 2999 THEN print, "Error: invalid year", year
time_utc = doy2utc(doy, year)
time_st = utc2str(time_utc)
month_st = strmid(time_st, 5, 2)
day_st = strmid(time_st, 8, 2)

; sphk cdf
hkfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_SC_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then hkfileprefix = concat_dir(outfilepath, hkfileprefix)
hkfilename = hkfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+hkfileprefix+' -delete -fillval '+hkskeleton_name
spawn, command
hkcdf_id = cdf_open(hkfilename)
temp_infilename = infilename
WHILE strlen(temp_infilename) LT 32 DO BEGIN
    temp_infilename = temp_infilename+' '
ENDWHILE 
cdf_varput, hkcdf_id, 'level0_file', temp_infilename
temp_systime = systime(0, /utc)
WHILE strlen(temp_systime) LT 32 DO BEGIN
    temp_systime = temp_systime + ' '
ENDWHILE 
cdf_varput, hkcdf_id, 'processing_date', temp_systime
temp_string = 'Housekeeping errors for file ' + temp_infilename
printf, hkerror_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, hkerror_lun, temp_string

; loop through data
line = ''
tab = STRING(9B)
FOR i = 0, 4 DO BEGIN ; read through header
    readf, in_lun, line
ENDFOR 
num_records = 0L
WHILE eof(in_lun) EQ 0 DO BEGIN
    readf, in_lun, line
    parts = strsplit(line, tab, count = num_parts, /extract, /preserve_null)
    want_to_read = 0
    IF n_elements(parts) GT 169 THEN BEGIN
        IF parts[169] NE '' OR parts[170] NE '' THEN want_to_read = 1
    ENDIF 
    IF parts[120] NE '' OR parts[121] NE '' OR parts[122] NE '' OR parts[123] NE '' OR parts[124] NE '' OR $
      parts[125] NE '' OR parts[126] NE '' OR want_to_read EQ 1 THEN BEGIN 
        datetime = parts[0]
        year_st = strmid(datetime, 0, 4)
        doy = fix(strmid(datetime, 5, 3))
        temp = doy2ex(doy, fix(year_st))
        month = temp[5]
        day = temp[4]
        time_st = strmid(datetime, 9, 12)
        timestamp = year_st+'-'+string(month, format = '(I2.2)')+'-'+string(day, format = '(I2.2)')+'T'+time_st+'Z'
        epoch = anytim2cdf(timestamp)
        cdf_varput, hkcdf_id, 'epoch', epoch, rec_start = num_records
        IF parts[120] NE '' THEN BEGIN 
            ; pl_ebox_tmp
            pl_ebox_tmp = double(parts[120])
            cdf_varput, hkcdf_id, 'pl_ebox_tmp', pl_ebox_tmp, rec_start = num_records
            IF finite(old_pl_ebox_tmp) EQ 0 OR old_pl_ebox_tmp NE pl_ebox_tmp THEN BEGIN 
                CASE 1 OF 
                    pl_ebox_tmp LT pl_ebox_tmp_limit[0]: BEGIN
                        IF pl_ebox_tmp_out EQ 0 THEN BEGIN
                            printf, hkerror_lun, timestamp, " Red limit: pl_ebox_tmp too low ", pl_ebox_tmp
                            pl_ebox_tmp_out = 1
                        ENDIF 
                    END 
                   pl_ebox_tmp LT pl_ebox_tmp_limit[1]: BEGIN 
                        IF pl_ebox_tmp_out EQ 0 THEN BEGIN
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_ebox_tmp too low ", pl_ebox_tmp
                            pl_ebox_tmp_out = 1
                        ENDIF 
                    END 
                    pl_ebox_tmp GT pl_ebox_tmp_limit[2]: BEGIN 
                        IF pl_ebox_tmp_out EQ 0 THEN BEGIN
                            printf, hkerror_lun, timestamp, " Red limit: pl_ebox_tmp too high ", pl_ebox_tmp
                            pl_ebox_tmp_out = 1
                        ENDIF 
                    END 
                    pl_ebox_tmp GT pl_ebox_tmp_limit[3]: BEGIN 
                        IF pl_ebox_tmp_out EQ 0 THEN BEGIN
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_ebox_tmp too high ", pl_ebox_tmp
                            pl_ebox_tmp_out = 1
                        ENDIF 
                    END 
                    pl_ebox_tmp_out EQ 1: BEGIN 
                        printf, hkerror_lun, timestamp, " pl_ebox_tmp back in limits ", pl_ebox_tmp
                        pl_ebox_tmp_out = 0
                    END 
                    ELSE: BREAK
                ENDCASE
            ENDIF
            old_pl_ebox_tmp = pl_ebox_tmp
        ENDIF 
        IF parts[121] NE '' THEN BEGIN 
            ; pl_inst_msg_pwr_dwn_req
            pl_inst_msg_pwr_dwn_req = parts[121]
            IF old_pl_inst_msg_pwr_dwn_req NE '' AND pl_inst_msg_pwr_dwn_req NE old_pl_inst_msg_pwr_dwn_req THEN $
              printf, hkerror_lun, timestamp, " Change in value for pl_inst_msg_pwr_dwn_req, ", old_pl_inst_msg_pwr_dwn_req, ' ', pl_inst_msg_pwr_dwn_req
            old_pl_inst_msg_pwr_dwn_req = pl_inst_msg_pwr_dwn_req
            WHILE strlen(pl_inst_msg_pwr_dwn_req) LT 32 DO BEGIN 
                pl_inst_msg_pwr_dwn_req = pl_inst_msg_pwr_dwn_req+' '
            ENDWHILE 
            cdf_varput, hkcdf_id, 'pl_inst_msg_pwr_dwn_req', pl_inst_msg_pwr_dwn_req, rec_start = num_records
        ENDIF 
        IF parts[122] NE '' THEN BEGIN 
            ; pl_pwr_cur
            pl_pwr_cur = double(parts[122])
            cdf_varput, hkcdf_id, 'pl_pwr_cur', pl_pwr_cur, rec_start = num_records
            IF finite(old_pl_pwr_cur) EQ 0 OR old_pl_pwr_cur NE pl_pwr_cur THEN BEGIN 
                CASE 1 OF 
                    pl_pwr_cur LT pl_pwr_cur_limit[0]: BEGIN
                        IF pl_pwr_cur_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Red limit: pl_pwr_cur too low ", pl_pwr_cur
                            pl_pwr_cur_out = 1
                        ENDIF 
                    END 
                    pl_pwr_cur LT pl_pwr_cur_limit[1]: BEGIN 
                        IF pl_pwr_cur_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_pwr_cur too low ", pl_pwr_cur
                            pl_pwr_cur_out = 1
                        ENDIF 
                    END 
                    pl_pwr_cur GT pl_pwr_cur_limit[2]: BEGIN 
                        IF pl_pwr_cur_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Red limit: pl_pwr_cur too high ", pl_pwr_cur
                            pl_pwr_cur_out = 1
                        ENDIF 
                    END 
                    pl_pwr_cur GT pl_pwr_cur_limit[3]: BEGIN 
                        IF pl_pwr_cur_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_pwr_cur too high ", pl_pwr_cur
                            pl_pwr_cur_out = 1
                        ENDIF 
                    END 
                    pl_pwr_cur_out EQ 1: BEGIN 
                        printf, hkerror_lun, timestamp, " pl_pwr_cur back in limits ", pl_pwr_cur
                        pl_pwr_cur_out = 0
                    END 
                    ELSE: BREAK
                ENDCASE
            ENDIF
            old_pl_pwr_cur = pl_pwr_cur
        ENDIF 
        IF parts[123] NE '' THEN BEGIN 
            ; pl_pwr_tt
            pl_pwr_tt = parts[123]
            IF old_pl_pwr_tt NE '' AND pl_pwr_tt NE old_pl_pwr_tt THEN $
              printf, hkerror_lun, timestamp, " Change in value for pl_pwr_tt, ", old_pl_pwr_tt, ' ', pl_pwr_tt
            old_pl_pwr_tt = pl_pwr_tt
            WHILE strlen(pl_pwr_tt) LT 32 DO BEGIN 
                pl_pwr_tt = pl_pwr_tt+' '
            ENDWHILE 
            cdf_varput, hkcdf_id, 'pl_pwr_tt', pl_pwr_tt, rec_start = num_records
        ENDIF 
        IF parts[124] NE '' THEN BEGIN 
            ; pl_surv_pwr_cur
            pl_surv_pwr_cur = double(parts[124])
            cdf_varput, hkcdf_id, 'pl_surv_pwr_cur', pl_surv_pwr_cur, rec_start = num_records
             IF finite(old_pl_surv_pwr_cur) EQ 0 OR old_pl_surv_pwr_cur NE pl_surv_pwr_cur THEN BEGIN 
                CASE 1 OF 
                    pl_surv_pwr_cur GT pl_surv_pwr_cur_limit[2]: BEGIN 
                        IF pl_surv_pwr_cur_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Red limit: pl_surv_pwr_cur too high ", pl_surv_pwr_cur
                            pl_surv_pwr_cur_out = 1
                        ENDIF 
                    END 
                    pl_surv_pwr_cur GT pl_surv_pwr_cur_limit[3]: BEGIN 
                        IF pl_surv_pwr_cur_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_surv_pwr_cur too high ", pl_surv_pwr_cur
                            pl_surv_pwr_cur_out = 1
                        ENDIF 
                    END 
                    pl_surv_pwr_cur_out EQ 1: BEGIN 
                        printf, hkerror_lun, timestamp, " pl_surv_pwr_cur back in limits ", pl_surv_pwr_cur
                        pl_surv_pwr_cur_out = 0
                    END 
                    ELSE: BREAK
                ENDCASE
            ENDIF
            old_pl_surv_pwr_cur = pl_surv_pwr_cur
        ENDIF 
        IF parts[125] NE '' THEN BEGIN 
            ; pl_surv_pwr_tt
            pl_surv_pwr_tt = parts[125]
            IF old_pl_surv_pwr_tt NE '' AND pl_surv_pwr_tt NE old_pl_surv_pwr_tt THEN $
              printf, hkerror_lun, timestamp, " Change in value for pl_surv_pwr_tt, ", old_pl_surv_pwr_tt, ' ', pl_surv_pwr_tt
            old_pl_surv_pwr_tt = pl_surv_pwr_tt
            WHILE strlen(pl_surv_pwr_tt) LT 32 DO BEGIN 
                pl_surv_pwr_tt = pl_surv_pwr_tt+' '
            ENDWHILE 
            cdf_varput, hkcdf_id, 'pl_surv_pwr_tt', pl_surv_pwr_tt, rec_start = num_records
        ENDIF 
        IF parts[126] NE '' THEN BEGIN 
            ; pl_tof_tmp
            pl_tof_tmp = double(parts[126])
            cdf_varput, hkcdf_id, 'pl_tof_tmp', pl_tof_tmp, rec_start = num_records
             IF finite(old_pl_tof_tmp) EQ 0 OR old_pl_tof_tmp NE pl_tof_tmp THEN BEGIN 
                CASE 1 OF 
                    pl_tof_tmp LT pl_tof_tmp_limit[0]: BEGIN
                        IF pl_tof_tmp_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Red limit: pl_tof_tmp too low ", pl_tof_tmp
                            pl_tof_tmp_out = 1
                        ENDIF 
                    END 
                    pl_tof_tmp LT pl_tof_tmp_limit[1]: BEGIN 
                        IF pl_tof_tmp_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_tof_tmp too low ", pl_tof_tmp
                            pl_tof_tmp_out = 1
                        ENDIF 
                    END 
                    pl_tof_tmp GT pl_tof_tmp_limit[2]: BEGIN 
                        IF pl_tof_tmp_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Red limit: pl_tof_tmp too high ", pl_tof_tmp
                            pl_tof_tmp_out = 1
                        ENDIF 
                    END 
                    pl_tof_tmp GT pl_tof_tmp_limit[3]: BEGIN 
                        IF pl_tof_tmp_out EQ 0 THEN BEGIN 
                            printf, hkerror_lun, timestamp, " Yellow limit: pl_tof_tmp too high ", pl_tof_tmp
                            pl_tof_tmp_out = 1
                        ENDIF 
                    END 
                    pl_tof_tmp_out EQ 1: BEGIN 
                        printf, hkerror_lun, timestamp, " pl_tof_tmp back in limits ", pl_tof_tmp
                        pl_tof_tmp_out = 0
                    END 
                    ELSE: BREAK
                ENDCASE
            ENDIF
            old_pl_tof_tmp = pl_tof_tmp
        ENDIF 
        IF want_to_read EQ 1 THEN BEGIN 
            IF parts[169] NE '' THEN BEGIN 
                                ; pl_pwr_vt
                pl_pwr_vt = double(parts[169])
                cdf_varput, hkcdf_id, 'pl_pwr_vt', pl_pwr_vt, rec_start = num_records
                IF finite(old_pl_pwr_vt) EQ 0 OR old_pl_pwr_vt NE pl_pwr_vt THEN BEGIN 
                    CASE 1 OF 
                        pl_pwr_vt LT pl_pwr_vt_limit[0]: BEGIN
                            IF pl_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Red limit: pl_pwr_vt too low ", pl_pwr_vt
                                pl_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_pwr_vt LT pl_pwr_vt_limit[1]: BEGIN 
                            IF pl_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Yellow limit: pl_pwr_vt too low ", pl_pwr_vt
                                pl_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_pwr_vt GT pl_pwr_vt_limit[2]: BEGIN 
                            IF pl_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Red limit: pl_pwr_vt too high ", pl_pwr_vt
                                pl_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_pwr_vt GT pl_pwr_vt_limit[3]: BEGIN 
                            IF pl_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Yellow limit: pl_pwr_vt too high ", pl_pwr_vt
                                pl_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_pwr_vt_out EQ 1: BEGIN 
                            printf, hkerror_lun, timestamp, " pl_pwr_vt back in limits ", pl_pwr_vt
                            pl_pwr_vt_out = 0
                        END 
                        ELSE: BREAK
                    ENDCASE
                ENDIF
                old_pl_pwr_vt = pl_pwr_vt
            ENDIF 
            IF parts[170] NE '' THEN BEGIN 
                                ; pl_surv_pwr_vt
                pl_surv_pwr_vt = double(parts[170])
                cdf_varput, hkcdf_id, 'pl_surv_pwr_vt', pl_surv_pwr_vt, rec_start = num_records
                IF finite(old_pl_surv_pwr_vt) EQ 0 OR old_pl_surv_pwr_vt NE pl_surv_pwr_vt THEN BEGIN 
                    CASE 1 OF 
                        pl_surv_pwr_vt LT pl_surv_pwr_vt_limit[0]: BEGIN
                            IF pl_surv_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Red limit: pl_surv_pwr_vt too low ", pl_surv_pwr_vt
                                pl_surv_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_surv_pwr_vt LT pl_surv_pwr_vt_limit[1]: BEGIN 
                            IF pl_surv_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Yellow limit: pl_surv_pwr_vt too low ", pl_surv_pwr_vt
                                pl_surv_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_surv_pwr_vt GT pl_surv_pwr_vt_limit[2]: BEGIN 
                            IF pl_surv_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Red limit: pl_surv_pwr_vt too high ", pl_surv_pwr_vt
                                pl_surv_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_surv_pwr_vt GT pl_surv_pwr_vt_limit[3]: BEGIN 
                            IF pl_surv_pwr_vt_out EQ 0 THEN BEGIN 
                                printf, hkerror_lun, timestamp, " Yellow limit: pl_surv_pwr_vt too high ", pl_surv_pwr_vt
                                pl_surv_pwr_vt_out = 1
                            ENDIF 
                        END 
                        pl_surv_pwr_vt_out EQ 1: BEGIN 
                            printf, hkerror_lun, timestamp, " pl_surv_pwr_vt back in limits ", pl_surv_pwr_vt
                            pl_surv_pwr_vt_out = 0
                        END 
                        ELSE: BREAK
                    ENDCASE
                ENDIF
                old_pl_surv_pwr_vt = pl_surv_pwr_vt
            ENDIF 
        ENDIF 
        IF num_records MOD 10000 EQ 0 THEN print, num_records, ' ', timestamp
        num_records = num_records + 1
    ENDIF 
ENDWHILE

close, in_lun
free_lun, in_lun
close, hkerror_lun
free_lun, hkerror_lun
cdf_close, hkcdf_id

; send error logs
IF keyword_set(skip_email) OR $
  ((temp_spacecraft EQ 'B') AND ((year EQ 2014) AND (doy GT 269) OR (year EQ 2015))) $
  THEN addresses = ['lellis@spam.sr.unh.edu'] ELSE BEGIN 
    length = strlen(hkerrorfilename)
    IF strmid(hkerrorfilename, length-17, 6) EQ '00.csh' THEN $
      addresses = ['lellis@spam.sr.unh.edu', 'popecki@atlas.sr.unh.edu', 'tgalvin@atlas.sr.unh.edu', $
                   'john.gaidos@unh.edu', 'dfhplastic@gmail.com', 'nkx22@wildcats.unh.edu'] ELSE $
      addresses = ['lellis@spam.sr.unh.edu', 'popecki@atlas.sr.unh.edu']
ENDELSE 
FOR i = 0, n_elements(addresses)-1 DO BEGIN 
    command = 'mail -s '+hkerrorfilename+' < '+hkerrorfile+' '+addresses[i]
    spawn, command
ENDFOR 

; copy cdfs to read_cdf dir
command = 'cp '+hkfilename+' '+cdfreadfilepath+'/.'
spawn, command

END 
