;+
; Project     : STEREO - PLASTIC
;
; Name        : plastic_process_l0_stand_along
;
; Purpose     : Parse Level 0 PTP file, and put in daily CDF file.
;
; Category    : Telemetry, Data-extraction
;
; Explanation : Takes a PLASTIC Level 0 data PTP file structure
;               as created by READ_STEREO_PKT, parese it, and
;               puts the raw data into cdf files. There is one cdf
;               file created for each 24 hour day. Data is
;               uncompressed. There are variables for raw data
;               as well as converted data.
;
; Syntax      : IDL> plastic_process_l0
;
; Inputs      : infilename
;
; Opt. Inputs : infilepath, outfilepath, errorfilepath
;
; Outputs     : outfilename
;
; Opt. Outputs: none
;
; Common      : none
;
; Restrictions: requires environmental variable SKELETONCDF be set
;               requires skeleton file PLA_L1_00000000_V01.skt in data/skeleton/
;               directory
;
; Side effects: creates cdf file in filepath directory
;
; History     : version 1, 10/01/10, LB Ellis, created stand-alone version
;               01/31/2011 Update for post_327 DPU.
;               05/20/2011 Add dpu_post_327 to common block.

; Contact     : Lorna Ellis (Lorna.Ellis@unh.edu)
;-


; For the data array, byte 0 is correct, odd bytes +1, even bytes -1

PRO uncompress8, input, output
compile_opt strictarrsubs

input = long(input)
output = input

msb = ishft(input, -4) AND '0F'x
lsb = input AND '0F'x

IF input GE '20'x THEN output = (lsb OR '10'x) * (2 ^ (msb-1))

END 

PRO uncompress15, input, output
compile_opt strictarrsubs

output = double(input)
s = ishft(input, -15) AND '1'x  ; sign bit
e = ishft(input,  -9) AND '3F'x ; exponent
m = input AND '1FF'x            ; mantissa

output = ((-1) ^ s) * (512 + m) * (double(2) ^ (e-10))

END 

PRO uncompress_moment, comp, uncomp
compile_opt strictarrsubs
uncomp = 0D
IF comp NE 0 THEN BEGIN 
    temp = comp AND '8000'x
    s = ishft(temp, -15)
    temp = comp AND '7E00'x
    e = ishft(temp, -9)
    m = comp AND '01FF'x
    temp1 = (-1) ^ s
    temp2 = 512+m
    temp3 = 2 ^ (double(e-10))
    uncomp = temp1 * temp2 * temp3
ENDIF 
END 


PRO plastic_process_l0_stand_alone, infilename, infilepath=infilepath, outfilepath = outfilepath, errorfilepath = errorfilepath, outfilename
compile_opt strictarrsubs
COMMON share1, instrument_id, conv_file, epochs1, epochs5_mon, epochs5_heavy, cycles1, cycles5_mon, cycles5_heavy, num_cycles1, num_cycles5_mon, num_cycles5_heavy, num_analog_cycles, num_digital_cycles, num_classifier_cycles, num_memory_cycles, schan1, schan5_mon, schan5_heavy, apid, timestamp, error_lun, hkerror_lun, status_lun, error_cycles, cdf_id, suspect1, suspect5_heavy, suspect5_mon, pkt_epoch, version, dpu_post_327
version = 10
infilepath      = "~/Stereo/Level1/IDL/Input/"
outfilepath     = "~/Stereo/Level1/IDL/Output/"
errorfilepath   = "~/Stereo/Level1/IDL/Output/"
;skeleton_name   = '/guamdata1/SKT/PLA_L1_00000000_V'+string(version, format='(I2.2)')
skeleton_name   = '~/Stereo/Level1/SKT/PLA_L1_00000000_V'+string(version, format='(I2.2)')
;hkskeleton_name = '/guamdata1/SKT/PLA_L1_HK_00000000_V'+string(version, format='(I2.2)')
hkskeleton_name = '~/Stereo/Level1/SKT/PLA_L1_HK_00000000_V'+string(version, format='(I2.2)')
;clskeleton_name = '/guamdata1/SKT/PLA_L1_CL_00000000_V'+string(version, format='(I2.2)') ; has raw classifier data (apids 328 & 329)
clskeleton_name = '~/Stereo/Level1/SKT/PLA_L1_CL_00000000_V'+string(version, format='(I2.2)') ; has raw classifier data (apids 328 & 329)

temp = strmid(infilename, 0, 1)
IF temp[0] EQ 'A' THEN BEGIN
    instrument_id = 'FM1'
    spacecraft = 'ahead'
    year_st = '2001'
    doy_st = '000'
ENDIF ELSE IF temp[0] EQ 'B' THEN BEGIN 
    instrument_id = 'FM2'
    spacecraft = 'behind'
    year_st = '2001'
    doy_st = '000'
ENDIF ELSE BEGIN 
    temp = strmid(infilename, 7, 1)
    IF temp[0] EQ 'a' THEN BEGIN ; 'ahead'
        instrument_id = 'FM1'
        spacecraft = 'ahead'
        year_st = strmid(infilename, 13, 4)
        doy_st = strmid(infilename, 18, 3)
    ENDIF ELSE IF temp[0] EQ 'b' THEN BEGIN 
        instrument_id = 'FM2'
        spacecraft = 'behind'
        year_st = strmid(infilename, 14, 4)
        doy_st = strmid(infilename, 19, 3)
    ENDIF ELSE BEGIN
        print, 'invalid infile name: ', infilename
        stop
    ENDELSE 
ENDELSE 
set_com_stand_alone
year = fix(year_st)
doy = fix(doy_st)
if n_elements(infilepath) eq 1 then infile = concat_dir(infilepath, infilename) ELSE infile = infilename
openr, in_lun, infile, /get_lun
errorfilename = infilename+'.errors'
if n_elements(errorfilepath) eq 1 then errorfile = concat_dir(errorfilepath, errorfilename) ELSE errorfile = errorfilename
openw, error_lun, errorfile, /get_lun
hkerrorfilename = infilename+'.hkerrors'
if n_elements(errorfilepath) eq 1 then hkerrorfile = concat_dir(errorfilepath, hkerrorfilename) ELSE hkerrorfile = hkerrorfilename
openw, hkerror_lun, hkerrorfile, /get_lun
statusfilename = infilename+'.status'
if n_elements(errorfilepath) eq 1 then statusfile = concat_dir(errorfilepath, statusfilename) ELSE statusfile = statusfilename
openw, status_lun, statusfile, /get_lun

IF spacecraft EQ 'ahead' THEN temp_spacecraft = 'A' ELSE temp_spacecraft = 'B'
IF year LT 2001 OR year GT 2999 THEN print, "Error: invalid year", year
time_utc = doy2utc(doy, year)
time_st = utc2str(time_utc)
month_st = strmid(time_st, 5, 2)
day_st = strmid(time_st, 8, 2)

; data cdf
outfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then outfileprefix = concat_dir(outfilepath, outfileprefix) 
outfilename = outfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+outfileprefix+' -delete -fillval '+skeleton_name
spawn, command
cdf_id = cdf_open(outfilename)
temp_infilename = infilename
WHILE strlen(temp_infilename) LT 32 DO BEGIN
    temp_infilename = temp_infilename+' '
ENDWHILE 
cdf_varput, cdf_id, 'level0_file', temp_infilename
temp_systime = systime(0, /utc)
WHILE strlen(temp_systime) LT 32 DO BEGIN
    temp_systime = temp_systime + ' '
ENDWHILE 
cdf_varput, cdf_id, 'processing_date', temp_systime
temp_string = 'Errors for file ' + temp_infilename
printf, error_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, error_lun, temp_string

; hk cdf
hkfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_HK_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then hkfileprefix = concat_dir(outfilepath, hkfileprefix)
hkfilename = hkfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+hkfileprefix+' -delete -fillval '+hkskeleton_name
spawn, command
hkcdf_id = cdf_open(hkfilename)
cdf_varput, hkcdf_id, 'level0_file', temp_infilename
cdf_varput, hkcdf_id, 'processing_date', temp_systime
temp_string = 'Housekeeping errors for file ' + temp_infilename
printf, hkerror_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, hkerror_lun, temp_string
temp_string = 'Instrument status changes for file ' + temp_infilename
printf, status_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, status_lun, temp_string

; cl cdf
clfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_CL_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then clfileprefix = concat_dir(outfilepath, clfileprefix)
clfilename = clfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+clfileprefix+' -delete -fillval '+clskeleton_name
spawn, command
clcdf_id = cdf_open(clfilename)
cdf_varput, clcdf_id, 'level0_file', temp_infilename
cdf_varput, clcdf_id, 'processing_date', temp_systime


; loop through data
packet = 0
num_packet = 0L
dummy = '1958-01-01T00:00:00.000'
WHILE n_elements(packet) NE 0 DO BEGIN
    num_packet = num_packet + 1
    read_stereo_pkt, in_lun, packet
    IF n_elements(packet) NE 0 THEN BEGIN 
        IF tag_exist(packet, 'pkt') THEN BEGIN 
            apid = parse_stereo_pkt(packet, /APID)
            timestamp = parse_stereo_pkt(packet, /PKT_DATE, /ccsds) ; /ccsds is for anytim2utc
            pkt_epoch = anytim2cdf(timestamp)
            IF num_packet MOD 10000 EQ 0 THEN print, num_packet, ' ', timestamp
            ;IF strcmp(timestamp, dummy) NE 1 THEN BEGIN ; ignore packets with timestamp bytes of 0
            IF strcmp(timestamp, dummy) NE 1 AND timestamp GE '2011-05-11T00:00:00.000' THEN BEGIN ; ignore packets with timestamp bytes of 0
                CASE apid OF
                    '200'x: read_200, packet, hkcdf_id
                    '313'x: read_313, packet, hkcdf_id
                    '315'x: read_315, packet, temp_spacecraft
                    '316'x: read_316, packet
                    '317'x: read_317, packet, temp_spacecraft
                    '318'x: read_318, packet, clcdf_id
                    '319'x: IF version GE 10 THEN read_319_v10, packet, temp_spacecraft $
                      ELSE read_319, packet, temp_spacecraft
                    '31A'x: read_31a, packet, temp_spacecraft
                    '31B'x: read_31b, packet, temp_spacecraft
                    '31C'x: read_31c, packet, temp_spacecraft
                    '31D'x: read_31d, packet, temp_spacecraft
                    '31E'x: read_31e, packet, temp_spacecraft
                    '31F'x: read_31f, packet, temp_spacecraft
                    '320'x: read_320, packet, temp_spacecraft
                    '321'x: read_321, packet, temp_spacecraft
                    '322'x: read_322, packet, temp_spacecraft
                    '323'x: read_323, packet, temp_spacecraft
                    '324'x: read_324, packet, temp_spacecraft
                    '325'x: read_325, packet
                    '326'x: read_326, packet, temp_spacecraft
                    '327'x: read_327, packet
                    '328'x: read_328, packet, clcdf_id
                    '329'x: read_329, packet, clcdf_id
                    ELSE: BREAK 
                ENDCASE 
            ENDIF 
        ENDIF 
    ENDIF 
ENDWHILE
write_315, 1
write_316, 1
write_317, 1, 0
write_324, 1, 0
write_325, 1, 0
write_326, 1, 0
write_327, 1, 0
IF version LT 10 THEN final_write_heavies
write_hkerror, hkcdf_id
check_data

change_epoch, data_id = cdf_id, hk_id = hkcdf_id, version

; Check for hand-written error times
IF temp_spacecraft EQ 'B' AND year EQ 2008 AND doy EQ 304 THEN BEGIN 
    cdf_control, cdf_id, get_var_info = info, var = 'epoch1', /zvariable
    IF info.maxrec NE 1444 THEN printf, error_lun, 'plastic_process_l0: Error with handwritten error' $
      ELSE cdf_varput, cdf_id, 'error1', 1, rec_start = 910, /zvariable
ENDIF
IF temp_spacecraft EQ 'B' AND year EQ 2008 AND doy EQ 352 THEN BEGIN 
    cdf_control, cdf_id, get_var_info = info, var = 'epoch5_heavy', /zvariable
    cdf_varput, cdf_id, 'error5_heavy', [1, 1, 1], rec_start = 248, /zvariable
ENDIF

close, in_lun
free_lun, in_lun
close, error_lun
free_lun, error_lun
close, hkerror_lun
free_lun, hkerror_lun
close, status_lun
free_lun, status_lun
cdf_close, cdf_id
cdf_close, hkcdf_id
cdf_close, clcdf_id

; check for empty files
temp_i = where(epochs1       NE -1, count_1)
temp_i = where(epochs5_mon   NE -1, count_5_mon)
temp_i = where(epochs5_heavy NE -1, count_5_heavy)

; send error logs
addresses = ['lellis@spam.sr.unh.edu']
FOR i = 0, n_elements(addresses)-1 DO BEGIN 
    command = 'mail -s '+errorfilename+' < '+errorfile+' '+addresses[i]
    spawn, command
    command = 'mail -s '+hkerrorfilename+' < '+hkerrorfile+' '+addresses[i]
    spawn, command
ENDFOR 


END 
