;+
; Project     : STEREO - PLASTIC
;
; Name        : plastic_process_l0
;
; Purpose     : Parse Level 0 PTP file, and put in daily CDF file.
;
; Category    : Telemetry, Data-extraction
;
; Explanation : Takes a PLASTIC Level 0 data PTP file structure
;               as created by READ_STEREO_PKT, parese it, and
;               puts the raw data into cdf files. There is one cdf
;               file created for each 24 hour day. Data is
;               uncompressed. There are variables for raw data
;               as well as converted data.
;
; Syntax      : IDL> plastic_process_l0
;
; Inputs      : infilename
;
; Opt. Inputs : infilepath, outfilepath, errorfilepath
;
; Outputs     : outfilename
;
; Opt. Outputs: none
;
; Common      : none
;
; Restrictions: requires environmental variable SKELETONCDF be set
;               requires skeleton file PLA_L1_00000000_V01.skt in data/skeleton/
;               directory
;
; Side effects: creates cdf file in filepath directory
;
; History     : version 1, 3/3/06, LB Ellis, written
;             : version 3, 12/19/06 LB ELLIS, changed version number
;             to match new skeleton files
;             : version 4, 12/21/06 LBE changed skt
;             : version 5, 01/24/07 LBE added power_warning,
;             thruster_warning, coarse_pointing to 200, and added
;             instrument status log
;             : version 6, 01/31/07 LBE added unscale factor to moments
;             : version 7, 02/07/07 LBE changed fillval for moments
;             : version 8, 02/15/07 LBE fixed write_moments
;                          03/13/07 Added part to run ReadCdf
;             : version 9, 03/23/07 Changed for new DPU
;             :            06/25/07 Added outfilename as an output variable
;                          11/06/07 Added call to change_epoch
;                          08/27/08 Removed Katherine Singer from email
;                          01/29/09 Added check on B doy 352, 2008
;                          03/19/09 Add spacecraft argument to read_317.
;                          05/20/09 Add spacecraft argument to read_319.
;                          07/18/09 Add spacecraft argument to read_31c.
;                          01/27/10 Add spacecraft argument to read_324, read_326.
;                          02/09/10 Add check for tag_exist(packet, 'pkt').
;             : version 10, 05/31/11 Add infor for dpu post 327.
;                           06/28/11 Changed call to set_com_l0.
;                           07/05/11 Check for bit-flip problem (321 on B 7/2-3/2011)
;                           07/18/11 Check for dpu change on 6/17 for A.
;                           08/09/11 Check for A 6/27/2011 (bad bit on board)
;                           09/06/11 Check for B 7/30/2011
;                           09/23/11 Check for A 8/23/2011
;                           09/26/11 Check for B 8/23/2011
;                           09/27/11 Check for B 8/26/2011
;                           10/04/11 Check for B 8/28/2011
;                           02/05/13 Add skip_email keyword.
;                           12/17/13 Put checks in separate procedures.
;                           02/18/14 Change so Toni only gets email for first hk error log.
;                                    Add John Gaidos to email for first hk error log.
;                           08/27/14 Check for missing apids. Add to error log.
;                           09/10/14 Add have_apids and have_telem_vers
;                           09/17/14 Add Apid370
;                           11/24/14 Add setpterrors
;                           02/17/15 Move check_error for 1min from check_data because
;                                    has to be before check_mom.
;                           03/24/15 Only send email to me when behind sun
;                           03/05/19 Sent setpt errors to error
;                           directory
;                           03/10/20 Check that 370 in right day!
;                           05/11/200  LBE Add Nick Lubinsky to email
;
; Contact     : Lorna Ellis (Lorna.Ellis@unh.edu)
;-


; For the data array, byte 0 is correct, odd bytes +1, even bytes -1

PRO uncompress8, input, output
compile_opt strictarrsubs
input = long(input)
output = input
msb = ishft(input, -4) AND '0F'x
lsb = input AND '0F'x
IF input GE '20'x THEN output = (lsb OR '10'x) * (2 ^ (msb-1))
END 

PRO uncompress15, input, output
compile_opt strictarrsubs
output = double(input)
s = ishft(input, -15) AND '1'x  ; sign bit
e = ishft(input,  -9) AND '3F'x ; exponent
m = input AND '1FF'x            ; mantissa
output = ((-1) ^ s) * (512 + m) * (double(2) ^ (e-10))
END 

PRO uncompress_moment, comp, uncomp
compile_opt strictarrsubs
uncomp = 0D
IF comp NE 0 THEN BEGIN 
    temp = comp AND '8000'x
    s = ishft(temp, -15)
    temp = comp AND '7E00'x
    e = ishft(temp, -9)
    m = comp AND '01FF'x
    temp1 = (-1) ^ s
    temp2 = 512+m
    temp3 = 2 ^ (double(e-10))
    uncomp = temp1 * temp2 * temp3
ENDIF 
END 


PRO plastic_process_l0, infilename, infilepath=infilepath, outfilepath = outfilepath, errorfilepath = errorfilepath, cdfreadfilepath = cdfreadfilepath, outfilename, version = version, skip_email = skip_email
compile_opt strictarrsubs
COMMON share1

skeleton_name = '/data1/SKT/PLA_L1_00000000_V'+string(version, format='(I2.2)')
;skeleton_name = '/Users/lornaellis/Lorna/Level1/SKT/PLA_L1_00000000_'+version
hkskeleton_name = '/data1/SKT/PLA_L1_HK_00000000_V'+string(version, format='(I2.2)')
;hkskeleton_name = '/Users/lornaellis/Lorna/Level1/SKT/PLA_L1_HK_00000000_'+version
clskeleton_name = '/data1/SKT/PLA_L1_CL_00000000_V'+string(version, format='(I2.2)') ; has raw classifier data (apids 328 & 329)
;clskeleton_name = '/Users/lornaellis/Lorna/Level1/SKT/PLA_L1_CL_00000000_'+version ; has raw classifier data (apids 328 & 329)

temp = strmid(infilename, 0, 1)
IF temp[0] EQ 'A' THEN BEGIN
    instrument_id = 'FM1'
    spacecraft = 'ahead'
    year_st = '2001'
    doy_st = '000'
ENDIF ELSE IF temp[0] EQ 'B' THEN BEGIN 
    instrument_id = 'FM2'
    spacecraft = 'behind'
    year_st = '2001'
    doy_st = '000'
ENDIF ELSE BEGIN 
    temp = strmid(infilename, 7, 1)
    IF temp[0] EQ 'a' THEN BEGIN ; 'ahead'
        instrument_id = 'FM1'
        spacecraft = 'ahead'
        year_st = strmid(infilename, 13, 4)
        doy_st = strmid(infilename, 18, 3)
    ENDIF ELSE IF temp[0] EQ 'b' THEN BEGIN 
        instrument_id = 'FM2'
        spacecraft = 'behind'
        year_st = strmid(infilename, 14, 4)
        doy_st = strmid(infilename, 19, 3)
    ENDIF ELSE BEGIN
        print, 'invalid infile name: ', infilename
        stop
    ENDELSE 
ENDELSE 
set_com_l0
year = fix(year_st)
doy = fix(doy_st)
if n_elements(infilepath) eq 1 then infile = concat_dir(infilepath, infilename) ELSE infile = infilename
openr, in_lun, infile, /get_lun
errorfilename = infilename+'.errors'
if n_elements(errorfilepath) eq 1 then errorfile = concat_dir(errorfilepath, errorfilename) ELSE errorfile = errorfilename
openw, error_lun, errorfile, /get_lun
hkerrorfilename = infilename+'.hkerrors'
if n_elements(errorfilepath) eq 1 then hkerrorfile = concat_dir(errorfilepath, hkerrorfilename) ELSE hkerrorfile = hkerrorfilename
openw, hkerror_lun, hkerrorfile, /get_lun
statusfilename = infilename+'.status'
if n_elements(errorfilepath) eq 1 then statusfile = concat_dir(errorfilepath, statusfilename) ELSE statusfile = statusfilename
openw, status_lun, statusfile, /get_lun
setptfilename = infilename+'.setpterrors'
if n_elements(errorfilepath) eq 1 then setptfile = concat_dir(errorfilepath, setptfilename) ELSE setptfile = setptfilename
openw, setpt_lun, setptfile, /get_lun

IF spacecraft EQ 'ahead' THEN temp_spacecraft = 'A' ELSE temp_spacecraft = 'B'
IF year LT 2001 OR year GT 2999 THEN print, "Error: invalid year", year
time_utc = doy2utc(doy, year)
time_st = utc2str(time_utc)
month_st = strmid(time_st, 5, 2)
day_st = strmid(time_st, 8, 2)

; data cdf
outfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then outfileprefix = concat_dir(outfilepath, outfileprefix) 
outfilename = outfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+outfileprefix+' -delete -fillval '+skeleton_name
spawn, command
cdf_id = cdf_open(outfilename)
temp_infilename = infilename
WHILE strlen(temp_infilename) LT 32 DO BEGIN
    temp_infilename = temp_infilename+' '
ENDWHILE 
cdf_varput, cdf_id, 'level0_file', temp_infilename
temp_systime = systime(0, /utc)
WHILE strlen(temp_systime) LT 32 DO BEGIN
    temp_systime = temp_systime + ' '
ENDWHILE 
cdf_varput, cdf_id, 'processing_date', temp_systime
temp_string = 'Errors for file ' + temp_infilename
printf, error_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, error_lun, temp_string

; hk cdf
hkfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_HK_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then hkfileprefix = concat_dir(outfilepath, hkfileprefix)
hkfilename = hkfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+hkfileprefix+' -delete -fillval '+hkskeleton_name
spawn, command
hkcdf_id = cdf_open(hkfilename)
cdf_varput, hkcdf_id, 'level0_file', temp_infilename
cdf_varput, hkcdf_id, 'processing_date', temp_systime
temp_string = 'Housekeeping errors for file ' + temp_infilename
printf, hkerror_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, hkerror_lun, temp_string
temp_string = 'Instrument status changes for file ' + temp_infilename
printf, status_lun, temp_string
temp_string = 'Processed on ' + temp_systime
printf, status_lun, temp_string

; cl cdf
clfileprefix = 'ST'+temp_spacecraft+'_L1_PLA_CL_'+year_st+month_st+day_st+'_'+doy_st+'_V'+string(version, format='(I2.2)')
if n_elements(outfilepath) eq 1 then clfileprefix = concat_dir(outfilepath, clfileprefix)
clfilename = clfileprefix+".cdf"
command = '$SKELETONCDF -cdf '+clfileprefix+' -delete -fillval '+clskeleton_name
spawn, command
clcdf_id = cdf_open(clfilename)
cdf_varput, clcdf_id, 'level0_file', temp_infilename
cdf_varput, clcdf_id, 'processing_date', temp_systime


; loop through data
packet           = 0
num_packet       = 0L
have_setpt_error = 0
have_apids       = intarr(22) ;200, 313, 315-327, 370
have_telem_vers  = intarr(4)
dummy            = '1958-01-01T00:00:00.000'
WHILE n_elements(packet) NE 0 DO BEGIN
    num_packet = num_packet + 1
    read_stereo_pkt, in_lun, packet
    IF n_elements(packet) NE 0 THEN BEGIN 
        IF tag_exist(packet, 'pkt') THEN BEGIN 
            apid = parse_stereo_pkt(packet, /APID)
            timestamp = parse_stereo_pkt(packet, /PKT_DATE, /ccsds) ; /ccsds is for anytim2utc
            pkt_epoch = anytim2cdf(timestamp)
            CASE temp_spacecraft OF ; 0=orig, 1=red.tel, 2=side1, 3=side2
                'A': BEGIN
                    IF timestamp LT '2014-01-01T12:30:00.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2014-07-11T14:15:00.000' THEN telem_vers = 1 ELSE $
                      IF timestamp LT '2014-07-12T15:00:00.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2014-08-05T02:30:00.000' THEN telem_vers = 1 ELSE $
                      IF timestamp LT '2014-08-05T07:55:00.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2014-08-05T16:00:00.000' THEN telem_vers = 2 ELSE $
                      IF timestamp LT '2014-08-05T23:55:00.000' THEN telem_vers = 3 ELSE $
                      IF timestamp LT '2014-08-06T14:55:00.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2014-08-06T19:11:00.000' THEN telem_vers = 3 ELSE $
                      IF timestamp LT '2014-08-06T22:30:00.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2014-08-18T22:30:00.000' THEN telem_vers = 1 ELSE $
                      IF timestamp LT '2014-08-19T00:00:00.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2015-01-05T19:45:00.000' THEN telem_vers = 2 ELSE $
                      IF timestamp LT '2015-08-17T17:00:00.000' THEN telem_vers = 3 ELSE $
                      IF timestamp LT '2015-11-17T19:35:00.000' THEN telem_vers = 2 ELSE $
                      IF timestamp LT '2016-05-15T00:00:00.000' THEN telem_vers = 1 ELSE telem_vers = 0
                END 
                'B': BEGIN
                    IF timestamp LT '2013-12-30T23:02:01.000' THEN telem_vers = 0 ELSE $
                      IF timestamp LT '2014-02-26T23:30:58.270' THEN telem_vers = 1 ELSE $
                      IF timestamp LT '2014-02-27T23:00:00.000' THEN telem_vers = 0 ELSE telem_vers = 1
                END 
            ENDCASE 
            have_telem_vers[telem_vers] = 1
            IF num_packet MOD 10000 EQ 0 THEN print, num_packet, ' ', timestamp
            IF strcmp(timestamp, dummy) NE 1 THEN BEGIN ; ignore packets with timestamp bytes of 0
               ; 2018 time we were wrongly in mode 3 after a momentum dump
                IF ((temp_spacecraft EQ 'A') AND (timestamp GE '2011-06-27T13:47:00.000') AND $
                    (timestamp LE '2011-06-27T20:42:00.000')) OR $
                   ((temp_spacecraft EQ 'A') AND (timestamp GE '2018-10-02T11:37:00.000') AND $
                    (timestamp LE '2018-10-08T00:00:00.000') AND $
                    (apid NE '200'x) AND (apid NE '313'x) AND (apid NE '318'x)) THEN dummy = dummy ELSE BEGIN
                   ;IF timestamp GE '2017-01-04T06:05:55.000' and apid eq '325'x THEN stop
                    CASE apid OF
                        '200'x: BEGIN
                            have_apids[0] = 1
                            read_200, packet, hkcdf_id
                        END 
                        '313'x: BEGIN 
                            read_313, packet, hkcdf_id, temp_spacecraft, have_apids[1]
                            have_apids[1] = 1
                        END 
                        '315'x: BEGIN 
                            have_apids[2] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN 
                                check_bad_315_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_315, packet, temp_spacecraft
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_315_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_315, packet, temp_spacecraft
                            ENDIF ELSE read_315, packet, temp_spacecraft
                        END 
                        '316'x: BEGIN 
                            have_apids[3] = 1
                            read_316, packet
                        END 
                        '317'x: BEGIN
                            have_apids[4] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN
                                check_bad_317_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_317, packet, temp_spacecraft
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_317_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_317, packet, temp_spacecraft
                            ENDIF ELSE read_317, packet, temp_spacecraft
                        END 
                        '318'x: BEGIN 
                            read_318, packet, clcdf_id
                            have_apids[5] = 1
                        END 
                        '319'x: BEGIN
                            have_apids[6] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2013-01-14/00:00:00.000') THEN BEGIN
                                bad_times = [time_epoch('2013-01-13/18:25:40.586')]
                                found_i = where(pkt_epoch EQ bad_times, count)
                                IF count GT 0 THEN dummy = dummy ELSE BEGIN 
                                    IF version GE 10 THEN read_319_v10, packet, temp_spacecraft, version $
                                    ELSE read_319, packet, temp_spacecraft
                                ENDELSE 
                            ENDIF ELSE BEGIN 
                                IF version GE 10 THEN read_319_v10, packet, temp_spacecraft, version $
                                ELSE read_319, packet, temp_spacecraft
                            ENDELSE 
                        END 
                        '31A'x: BEGIN 
                            have_apids[7] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-28/00:00:00.000') THEN BEGIN
                                check_bad_31a_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31a, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2013-12-17/00:00:00.000') THEN BEGIN
                                check_bad_31a_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31a, packet, temp_spacecraft, version
                            ENDIF ELSE read_31a, packet, temp_spacecraft, version
                        END 
                        '31B'x: BEGIN 
                            have_apids[8] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-28/00:00:00.000') THEN BEGIN
                                check_bad_31b_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31b, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2013-12-17/00:00:00.000') THEN BEGIN
                                check_bad_31b_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31b, packet, temp_spacecraft, version
                            ENDIF ELSE read_31b, packet, temp_spacecraft, version
                        END 
                        '31C'x: BEGIN 
                            have_apids[9] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-28/00:00:00.000') THEN BEGIN
                                check_bad_31c_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31c, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-08-02/00:00:00.000') THEN BEGIN
                                check_bad_31c_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31c, packet, temp_spacecraft, version
                            ENDIF ELSE read_31c, packet, temp_spacecraft, version
                        END 
                        '31D'x: BEGIN 
                            have_apids[10] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-28/00:00:00.000') THEN BEGIN
                                check_bad_31d_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31d, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-08-02/00:00:00.000') THEN BEGIN
                                check_bad_31d_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31d, packet, temp_spacecraft, version
                            ENDIF ELSE read_31d, packet, temp_spacecraft, version
                        END  
                        '31E'x: BEGIN 
                            have_apids[11] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-28/00:00:00.000') THEN BEGIN
                                check_bad_31e_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31e, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2013-12-17/00:00:00.000') THEN BEGIN
                                check_bad_31e_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31e, packet, temp_spacecraft, version
                            ENDIF ELSE read_31e, packet, temp_spacecraft, version
                        END 
                        '31F'x: BEGIN 
                            have_apids[12] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-28/00:00:00.000') THEN BEGIN
                                check_bad_31f_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31f, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2013-12-17/00:00:00.000') THEN BEGIN
                                check_bad_31f_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_31f, packet, temp_spacecraft, version
                            ENDIF ELSE read_31f, packet, temp_spacecraft, version
                        END 
                        '320'x: BEGIN 
                            have_apids[13] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN
                                check_bad_320_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_320, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2013-12-14/00:00:00.000') THEN BEGIN
                                check_bad_320_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_320, packet, temp_spacecraft, version
                            ENDIF ELSE read_320, packet, temp_spacecraft, version
                        END 
                        '321'x: BEGIN
                            have_apids[14] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN
                                check_bad_321_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_321, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2011-07-04/00:00:00.000') THEN BEGIN
                                bad_start = time_epoch('2011-07-02/02:11:38.563')
                                bad_stop  = time_epoch('2011-07-03/16:46:38.566')
                                IF (pkt_epoch GE bad_start) AND (pkt_epoch LE bad_stop) THEN dummy = dummy ELSE $
                                  read_321, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_321_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_321, packet, temp_spacecraft, version
                            ENDIF ELSE read_321, packet, temp_spacecraft, version
                        END 
                        '322'x: BEGIN 
                            have_apids[15] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN
                                check_bad_322_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_322, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-07-09/00:00:00.000') THEN BEGIN
                                check_bad_322_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_322, packet, temp_spacecraft, version
                            ENDIF ELSE read_322, packet, temp_spacecraft, version
                        END 
                        '323'x: BEGIN 
                            have_apids[16] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN
                                check_bad_323_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_323, packet, temp_spacecraft, version
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-07-09/00:00:00.000') THEN BEGIN
                                check_bad_323_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_323, packet, temp_spacecraft, version
                            ENDIF ELSE read_323, packet, temp_spacecraft, version
                        END 
                        '324'x: BEGIN
                            have_apids[17] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-11-06/00:00:00.000') THEN BEGIN
                               check_bad_324_a, pkt_epoch, count
                               IF count GT 0 THEN dummy = dummy ELSE read_324, packet, temp_spacecraft
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_324_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_324, packet, temp_spacecraft
                            ENDIF ELSE read_324, packet, temp_spacecraft
                        END 
                        '325'x: BEGIN 
                           have_apids[18] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-03-11/00:00:00.000') THEN BEGIN
                                check_bad_325_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_325, packet, temp_spacecraft
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_325_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_325, packet, temp_spacecraft
                            ENDIF ELSE read_325, packet, temp_spacecraft
                        END 
                        '326'x: BEGIN 
                            have_apids[19] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2021-01-11/00:00:00.000') THEN BEGIN
                                check_bad_326_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_326, packet, temp_spacecraft
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_326_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_326, packet, temp_spacecraft
                            ENDIF ELSE read_326, packet, temp_spacecraft
                        END 
                        '327'x: BEGIN 
                            have_apids[20] = 1
                            IF temp_spacecraft EQ 'A' AND pkt_epoch LT time_epoch('2020-10-27/00:00:00.000') THEN BEGIN
                                check_bad_327_a, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_327, packet, temp_spacecraft
                            ENDIF ELSE IF temp_spacecraft EQ 'B' AND pkt_epoch LT time_epoch('2014-09-21/00:00:00.000') THEN BEGIN
                                check_bad_327_b, pkt_epoch, count
                                IF count GT 0 THEN dummy = dummy ELSE read_327, packet, temp_spacecraft
                            ENDIF ELSE read_327, packet, temp_spacecraft
                        END 
                        '328'x: read_328, packet, clcdf_id
                        '329'x: read_329, packet, clcdf_id
                        '370'x: BEGIN 
                            have_apids[21] = 1
                            read_370, packet, temp_spacecraft, time_st
                        END 
                        ELSE: BREAK 
                    ENDCASE 
                ENDELSE 
            ENDIF 
        ENDIF 
    ENDIF 
ENDWHILE
write_315, 1
write_316, 1
write_317, 1, 0
write_324, 1, 0
write_325, 1, 0
write_326, 1, 0
write_327, 1, 0

IF version LT 10 THEN final_write_heavies
write_hkerror, hkcdf_id
IF temp_spacecraft EQ 'A' AND year EQ 2011 AND doy EQ 168 THEN dpu_post_327 = 0 ; changes at hour 20, no data after that

; NB: After check_mom, epoch1 doesn't match all epoch1 variables!!!!
FOR i1 = 0, num_cycles1-1 DO BEGIN 
    check_error, i1, -1, -1
ENDFOR 
IF num_cycles1 GT 0 THEN BEGIN
    temp_data = suspect1[0:num_cycles1-1]
    cdf_varput, cdf_id, 'error1', temp_data, /zvariable
ENDIF 
IF timestamp NE ' ' THEN check_mom, temp_spacecraft
check_data, version, temp_spacecraft ; OK after check_mom because read variables fresh
change_epoch, data_id = cdf_id, hk_id = hkcdf_id, version, temp_spacecraft

; Check for hand-written error times
IF temp_spacecraft EQ 'B' AND year EQ 2008 AND doy EQ 304 THEN BEGIN 
    cdf_control, cdf_id, get_var_info = info, var = 'epoch1', /zvariable
    IF info.maxrec NE 1444 THEN printf, error_lun, 'plastic_process_l0: Error with handwritten error' $
      ELSE cdf_varput, cdf_id, 'error1', 1, rec_start = 910, /zvariable
ENDIF
IF temp_spacecraft EQ 'B' AND year EQ 2008 AND doy EQ 352 THEN BEGIN 
    cdf_control, cdf_id, get_var_info = info, var = 'epoch5_heavy', /zvariable
    cdf_varput, cdf_id, 'error5_heavy', [1, 1, 1], rec_start = 248, /zvariable
ENDIF

; Check for missing apids. Add to error log.
check_missing_apids, temp_spacecraft, year, doy, error_lun, have_apids

check_set_points, temp_spacecraft

close,    in_lun
free_lun, in_lun
close,    error_lun
free_lun, error_lun
close,    hkerror_lun
free_lun, hkerror_lun
close,    status_lun
free_lun, status_lun
close,    setpt_lun
free_lun, setpt_lun
cdf_close, cdf_id
cdf_close, hkcdf_id
cdf_close, clcdf_id

; check for empty files
temp_i = where(epochs1       NE -1, count_1)
temp_i = where(epochs5_mon   NE -1, count_5_mon)
temp_i = where(epochs5_heavy NE -1, count_5_heavy)


; send error logs
; data error
IF keyword_set(skip_email) OR $
  ((temp_spacecraft EQ 'B') AND ((year EQ 2014) AND (doy GT 269) OR (year EQ 2015))) $
  THEN addresses = ['lellis@spam.sr.unh.edu'] ELSE $
  addresses = ['lellis@spam.sr.unh.edu', 'popecki@atlas.sr.unh.edu']
FOR i = 0, n_elements(addresses)-1 DO BEGIN 
    command = 'mail -s '+errorfilename+' < '+errorfile+' '+addresses[i]
    spawn, command
ENDFOR 
; hk error
IF keyword_set(skip_email) OR $
  ((temp_spacecraft EQ 'B') AND ((year EQ 2014) AND (doy GT 269) OR (year EQ 2015))) $
  THEN addresses = ['lellis@spam.sr.unh.edu'] ELSE BEGIN
    length = strlen(hkerrorfilename)
    IF strmid(hkerrorfilename, length-15, 6) EQ '00.ptp' THEN $
      addresses = ['lellis@spam.sr.unh.edu', 'popecki@atlas.sr.unh.edu', 'tgalvin@atlas.sr.unh.edu', $
                   'john.gaidos@unh.edu', 'dfhplastic@gmail.com', 'nkx22@wildcats.unh.edu'] ELSE $
      addresses = ['lellis@spam.sr.unh.edu', 'popecki@atlas.sr.unh.edu']
ENDELSE 
FOR i = 0, n_elements(addresses)-1 DO BEGIN 
    command = 'mail -s '+hkerrorfilename+' < '+hkerrorfile+' '+addresses[i]
    spawn, command
ENDFOR 
; set pt error
IF have_setpt_error EQ 1 THEN BEGIN 
    IF keyword_set(skip_email) OR $
      ((temp_spacecraft EQ 'B') AND ((year EQ 2014) AND (doy GT 269) OR (year EQ 2015))) $
      THEN addresses = ['lellis@spam.sr.unh.edu'] ELSE $
      addresses = ['lellis@spam.sr.unh.edu', 'popecki@atlas.sr.unh.edu', 'tgalvin@atlas.sr.unh.edu', $
                   'john.gaidos@unh.edu', 'dfhplastic@gmail.com', 'nkx22@wildcats.unh.edu'] 
    FOR i = 0, n_elements(addresses)-1 DO BEGIN 
        command = 'mail -s '+setptfilename+' < '+setptfile+' '+addresses[i]
        spawn, command
    ENDFOR 
ENDIF 

; copy cdfs to read_cdf dir
command = 'cp '+outfilename+' '+cdfreadfilepath+'/.'
spawn, command
command = 'cp '+hkfilename+' '+cdfreadfilepath+'/.'
spawn, command
command = 'cp '+clfilename+' '+cdfreadfilepath+'/.'
spawn, command

END 
