;+
;PROCEDURE: plastic_find_l0.pro
;PURPOSE:
;  Find level 0 files that have not been processed, and process them.
;  Searches for files in the last 31 days.
;
;OPTIONAL INPUTS:
; start_doy: if want to run specified day(s), use this with year 
; stop_doy: if want to run more than on specified day, use this with start_doy and year 
; year: for running specific doy or doys
; sat: if only want to run files for one satellite (string 'A' or 'B')
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 10/17/2006
;
;MODIFICATION HISTORY:
;    10/17/2006 Created
;    03/13/2007 Added part to run ReadCdf. Always run sphk if ran
;    others.
;    05/24/2007 Changed ganymede to fiji.
;    06/25/2007 Added call to calculate_svalid_ratrig_ratio
;    07/24/2007 Added call to calculate_pri_ratio
;    09/24/2007 Update ratio file by year.
;    10/23/2007 No longer update ratio file -- now done in Level 2.
;    05/20/2011 Add dpu_post_327 to common block.
;    02/05/2013 Add skip_email keyword.
;    09/17/2013 Change path for SPHK to include year.
;    09/09/2014 Added telem_vers: 0=orig, 1=reduced, 2=sidelobe1, 3=sidelobe2
;    11/24/2014 Added setpt_lun and have_setpt_error to share1.
;    09/12/2018 Only do STA.
;-
PRO plastic_find_l0, start_doy = start_doy, stop_doy = stop_doy, year = year, sat = sat, skip_email = skip_email
COMMON share1, instrument_id, conv_file, epochs1, epochs5_mon, epochs5_heavy, cycles1, cycles5_mon, cycles5_heavy, num_cycles1, num_cycles5_mon, num_cycles5_heavy, num_analog_cycles, num_digital_cycles, num_classifier_cycles, num_memory_cycles, schan1, schan5_mon, schan5_heavy, apid, timestamp, error_lun, hkerror_lun, setpt_lun, status_lun, error_cycles, cdf_id, suspect1, suspect5_heavy, suspect5_mon, pkt_epoch, dpu_post_327, telem_vers, have_telem_vers, have_setpt_error
compile_opt strictarrsubs
datadir  = '/data1/Spacecraft_' ; this needs to have 'A' or 'B' added in program
;datadir = '/Users/lornaellis/Lorna/Level1/Spacecraft_'
indir    = 'Level0'        ; directory to search for data (subdir of datadir)
sphkdir  = 'SPHK'          ; directory to search for spacecraft housekeeping (subdir of datadir)
cdfdir   = 'Level1/CDFs'   ; directory in which to put cdfs -- should put in year subdirectory
errordir = 'Level1/Errors' ; directory for error logs -- should put in year subdirectory
logdir   = 'Level1/Logs'   ; directory in which to put write log (one file per month) & to check file_list
                                ; file_list is a list for the year of
; the files that have been processed -- i.e. the doy, version of ptp
;                                       file and version of fin file
cdfreadpath          = '~/Files/'
cdfoutpath           = '~/Files/Output/'
zippath_a            = 'oahu:"/data2/Ascii/Level1/A/'
zippath_b            = 'oahu:"/data2/Ascii/Level1/B/'
tempplotpath         = '~/Plots/'
finalplotpath        = 'fiji:/data1/Plots/SciencePlots/'
calibrationpath      = '/usr/local/bin/ssw/stereo/plastic/data/calibrations/' ; for maui
;calibrationpath     = '/usr/local/ssw/stereo/plastic/data/calibrations/'  ; for daisy
num_versions_allowed = 25 ; set this to increase the number of possible versions in the list

; set up array of days to check for
doy_specified = 0
IF n_elements(start_doy) GT 0 AND n_elements(year) GT 0 THEN BEGIN  ; specified doy and year
    doy_specified = 1 ; so process, even if processed before
    IF n_elements(stop_doy) GT 0 THEN BEGIN 
        doy_array = indgen(stop_doy-start_doy+1)
        doy_array = doy_array+start_doy ; set up array of day-of-years
    ENDIF ELSE doy_array = [start_doy] ; if only one day
    year_array = intarr(n_elements(doy_array))
    year_array[*] = year  ; only do this one year
    year_st = string(year, format = '(I4.4)')
    year_list = [year]
    month_array = intarr(n_elements(doy_array))
    FOR ii = 0, n_elements(month_array)-1 DO BEGIN 
        temp_doy = doy_array[ii]
        temp = doy2ex(temp_doy, year)
        month_array[ii] = temp[5]
    ENDFOR 
ENDIF ELSE IF n_elements(start_doy) GT 0 OR n_elements(year) GT 0 THEN BEGIN 
    print, "Must set start_doy and year together"
    stop
ENDIF ELSE BEGIN                     ; start from today and do previous 30 days
; find current year and doy
    time_struct = bin_date(systime(0, /utc))
    IF time_struct[1] GE 10 THEN month_st = string(time_struct[1], format = '(I2.2)') ELSE $
      month_st = '0'+string(time_struct[1], format = '(I1.1)')
    IF time_struct[2] GE 10 THEN day_st = string(time_struct[2], format = '(I2.2)') ELSE $
      day_st = '0'+string(time_struct[2], format = '(I1.1)')
    year = time_struct[0]
    year_st = string(year, format = '(I4.4)')
    year_list = [year]
    time_string = year_st+'-'+month_st+'-'+day_st
    utc = str2utc(time_string)
    doy = utc2doy(utc)
    doy_array = intarr(40)
    year_array = intarr(40)
    month_array = intarr(40)
    FOR ii = 0, 39 DO BEGIN 
        temp_doy = doy - ii
        IF temp_doy LT 1 THEN BEGIN 
            temp_year = year-1
            year_list = [year_list, temp_year]
            IF leap_year(temp_year) EQ 1 THEN temp_doy = 366+temp_doy ELSE temp_doy = 365+temp_doy ; temp_doy was negative
        ENDIF ELSE temp_year = year
        doy_array[39-ii] = temp_doy
        year_array[39-ii] = temp_year
        temp = doy2ex(temp_doy, temp_year)
        month_array[39-ii] = temp[5]
    ENDFOR 
ENDELSE 

FOR ii = 0, 0 DO BEGIN ; one for each satellite
;FOR ii = 0, 1 DO BEGIN ; one for each satellite
    IF n_elements(sat) GT 0 THEN IF sat EQ 'B' THEN ii = 1 ; ignore sat A
    IF ii EQ 0 THEN temp_datadir = datadir+'A/' ELSE temp_datadir = datadir+'B/'
; read in the list of files already processed
    line = ''
    logpath = temp_datadir+logdir
    version_list_now = strarr(367, num_versions_allowed) ; need 366, and start at 1
    version_list_now[*, *] = ''
    version_sphk_now = strarr(367, num_versions_allowed) ; need 366, and start at 1
    version_sphk_now[*, *] = ''
    date_version_list_now = strarr(367) ; need 366, and start at 1
    date_version_list_now[*, *] = ''
    date_version_sphk_now = strarr(367) ; need 366, and start at 1
    date_version_sphk_now[*, *] = ''
    IF ii EQ 0 THEN BEGIN 
        list_filename_now = 'files_processed_'+string(year, format = '(I4.4)')+'_A' 
        list_sphk_now = 'sphk_processed_'+string(year, format = '(I4.4)')+'_A' 
        date_list_filename_now = 'processing_dates_'+string(year, format = '(I4.4)')+'_A' 
        date_list_sphk_now = 'sphk_processing_dates_'+string(year, format = '(I4.4)')+'_A' 
    ENDIF ELSE BEGIN 
        list_filename_now = 'files_processed_'+string(year, format = '(I4.4)')+'_B'
        list_sphk_now = 'sphk_processed_'+string(year, format = '(I4.4)')+'_B' 
        date_list_filename_now = 'processing_dates_'+string(year, format = '(I4.4)')+'_B' 
        date_list_sphk_now = 'sphk_processing_dates_'+string(year, format = '(I4.4)')+'_B' 
    ENDELSE 
    ; data files processed this year
    list_filepath_now = concat_dir(logpath, list_filename_now)
    IF file_test(list_filepath_now) EQ 1 THEN BEGIN 
        openr, list_now_lun, list_filepath_now, /get_lun
        readf, list_now_lun, line ; skip header line
        WHILE eof(list_now_lun) EQ 0 DO BEGIN 
            readf, list_now_lun, line
            parts = strsplit(line, /extract)
            temp_doy = fix(parts[0])
            FOR jj = 1, n_elements(parts)-1 DO BEGIN 
                version_list_now[temp_doy, jj-1] = parts[jj]
            ENDFOR 
        ENDWHILE 
        close, list_now_lun
        free_lun, list_now_lun
    ENDIF 
    ; processing dates for data files
    date_list_filepath_now = concat_dir(logpath, date_list_filename_now)
    IF file_test(date_list_filepath_now) EQ 1 THEN BEGIN 
        openr, date_list_now_lun, date_list_filepath_now, /get_lun
        readf, date_list_now_lun, line ; skip header line
        WHILE eof(date_list_now_lun) EQ 0 DO BEGIN 
            readf, date_list_now_lun, line
            parts = strsplit(line, /extract)
            temp_doy = fix(parts[0])
            IF n_elements(parts) GT 1 THEN date_version_list_now[temp_doy] = parts[1]
        ENDWHILE 
        close, date_list_now_lun
        free_lun, date_list_now_lun
    ENDIF 
    ; spacecraft housekeeping files processed this year
    list_sphk_now = concat_dir(logpath, list_sphk_now)
    IF file_test(list_sphk_now) EQ 1 THEN BEGIN 
        openr, list_now_lun, list_sphk_now, /get_lun
        readf, list_now_lun, line ; skip header line
        WHILE eof(list_now_lun) EQ 0 DO BEGIN 
            readf, list_now_lun, line
            parts = strsplit(line, /extract)
            temp_doy = fix(parts[0])
            FOR jj = 1, n_elements(parts)-1 DO BEGIN 
                version_sphk_now[temp_doy, jj-1] = parts[jj]
            ENDFOR 
        ENDWHILE 
        close, list_now_lun
        free_lun, list_now_lun
    ENDIF 
    ; processing dates for sphk files
    date_list_sphk_now = concat_dir(logpath, date_list_sphk_now)
    IF file_test(date_list_sphk_now) EQ 1 THEN BEGIN 
        openr, date_list_now_lun, date_list_sphk_now, /get_lun
        readf, date_list_now_lun, line ; skip header line
        WHILE eof(date_list_now_lun) EQ 0 DO BEGIN 
            readf, date_list_now_lun, line
            parts = strsplit(line, /extract)
            temp_doy = fix(parts[0])
            IF n_elements(parts) GT 1 THEN date_version_sphk_now[temp_doy] = parts[1]
        ENDWHILE 
        close, date_list_now_lun
        free_lun, date_list_now_lun
    ENDIF 
    IF ii EQ 0 THEN BEGIN
        version_list_now_a = version_list_now 
        version_sphk_now_a = version_sphk_now 
        date_version_list_now_a = date_version_list_now 
        date_version_sphk_now_a = date_version_sphk_now 
    ENDIF ELSE BEGIN 
        version_list_now_b = version_list_now
        version_sphk_now_b = version_sphk_now
        date_version_list_now_b = date_version_list_now
        date_version_sphk_now_b = date_version_sphk_now
    ENDELSE 
    version_list_prev = strarr(367, num_versions_allowed) ; need 366, and start at 1
    version_list_prev[*, *] = ''
    version_sphk_prev = strarr(367, num_versions_allowed) ; need 366, and start at 1
    version_sphk_prev[*, *] = ''
    date_version_list_prev = strarr(367) ; need 366, and start at 1
    date_version_list_prev[*, *] = ''
    date_version_sphk_prev = strarr(367) ; need 366, and start at 1
    date_version_sphk_prev[*, *] = ''
    ; last year
    IF n_elements(year_list) GT 1 THEN BEGIN 
        IF ii EQ 0 THEN BEGIN
            list_filename_prev = 'files_processed_'+string(year-1, format = '(I4.4)')+'_A' 
            list_sphk_prev = 'sphk_processed_'+string(year-1, format = '(I4.4)')+'_A' 
            date_list_filename_prev = 'processing_dates_'+string(year-1, format = '(I4.4)')+'_A' 
            date_list_sphk_prev = 'sphk_processing_dates_'+string(year-1, format = '(I4.4)')+'_A' 
        ENDIF ELSE BEGIN 
            list_filename_prev = 'files_processed_'+string(year-1, format = '(I4.4)')+'_B'
            list_sphk_prev = 'sphk_processed_'+string(year-1, format = '(I4.4)')+'_B'
            date_list_filename_prev = 'processing_dates_'+string(year-1, format = '(I4.4)')+'_B'
            date_list_sphk_prev = 'sphk_processing_dates_'+string(year-1, format = '(I4.4)')+'_B'
        ENDELSE 
        ; data files 
        list_filepath_prev = concat_dir(logpath, list_filename_prev)
        IF file_test(list_filepath_prev) EQ 1 THEN BEGIN 
            openr, list_prev_lun, list_filepath_prev, /get_lun
            readf, list_prev_lun, line ; skip header line
            WHILE eof(list_prev_lun) EQ 0 DO BEGIN 
                readf, list_prev_lun, line
                parts = strsplit(line, /extract)
                temp_doy = fix(parts[0])
                FOR jj = 1, n_elements(parts)-1 DO BEGIN 
                    version_list_prev[temp_doy, jj-1] = parts[jj]
                ENDFOR 
            ENDWHILE 
            close, list_prev_lun
            free_lun, list_prev_lun
        ENDIF 
        ; processing dates for data files 
        date_list_filepath_prev = concat_dir(logpath, date_list_filename_prev)
        IF file_test(date_list_filepath_prev) EQ 1 THEN BEGIN 
            openr, date_list_prev_lun, date_list_filepath_prev, /get_lun
            readf, date_list_prev_lun, line ; skip header line
            WHILE eof(date_list_prev_lun) EQ 0 DO BEGIN 
                readf, date_list_prev_lun, line
                parts = strsplit(line, /extract)
                temp_doy = fix(parts[0])
                IF n_elements(parts) GT 1 THEN date_version_list_prev[temp_doy] = parts[1]
            ENDWHILE 
            close, date_list_prev_lun
            free_lun, date_list_prev_lun
        ENDIF 
        ; spacecraft housekeeping files 
        list_filepath_prev = concat_dir(logpath, list_sphk_prev)
        IF file_test(list_filepath_prev) EQ 1 THEN BEGIN 
            openr, list_prev_lun, list_filepath_prev, /get_lun
            readf, list_prev_lun, line ; skip header line
            WHILE eof(list_prev_lun) EQ 0 DO BEGIN 
                readf, list_prev_lun, line
                parts = strsplit(line, /extract)
                temp_doy = fix(parts[0])
                FOR jj = 1, n_elements(parts)-1 DO BEGIN 
                    version_sphk_prev[temp_doy, jj-1] = parts[jj]
                ENDFOR 
            ENDWHILE 
            close, list_prev_lun
            free_lun, list_prev_lun
        ENDIF 
        ; processing dates for spacecraft housekeeping files 
        date_list_filepath_prev = concat_dir(logpath, date_list_sphk_prev)
        IF file_test(date_list_filepath_prev) EQ 1 THEN BEGIN 
            openr, date_list_prev_lun, date_list_filepath_prev, /get_lun
            readf, date_list_prev_lun, line ; skip header line
            WHILE eof(date_list_prev_lun) EQ 0 DO BEGIN 
                readf, date_list_prev_lun, line
                parts = strsplit(line, /extract)
                temp_doy = fix(parts[0])
                IF n_elements(parts) GT 1 THEN date_version_sphk_prev[temp_doy] = parts[1]
            ENDWHILE 
            close, date_list_prev_lun
            free_lun, date_list_prev_lun
        ENDIF 
    ENDIF 
    IF ii EQ 0 THEN BEGIN 
        version_list_prev_a = version_list_prev 
        version_sphk_prev_a = version_sphk_prev 
        date_version_list_prev_a = date_version_list_prev 
        date_version_sphk_prev_a = date_version_sphk_prev 
    ENDIF ELSE BEGIN
        version_list_prev_b = version_list_prev
        version_sphk_prev_b = version_sphk_prev
        date_version_list_prev_b = date_version_list_prev
        date_version_sphk_prev_b = date_version_sphk_prev
    ENDELSE 

; check and process files
    tab = STRING(9B)
    IF ii EQ 0 THEN logname = 'processing_log_'+string(year, format = '(I4.4)')+'_A' $
    ELSE logname = 'processing_log_'+string(year, format = '(I4.4)')+'_B'
    full_logname = concat_dir(logpath, logname)
    openw, log_now_lun, full_logname, /get_lun, /append
    IF n_elements(year_list) GT 1 THEN BEGIN 
        logname = 'processing_log_'+string(year-1, format = '(I4.4)')
        full_logname = concat_dir(logpath, logname)
        openw, log_prev_lun, full_logname, /get_lun, /append
    ENDIF 
    wrote_to_now_log = 0
    wrote_to_prev_log = 0
    IF ii EQ 0 THEN BEGIN
        version_list_now       = version_list_now_a
        version_list_prev      = version_list_prev_a
        version_sphk_now       = version_sphk_now_a
        version_sphk_prev      = version_sphk_prev_a
        date_version_list_now  = date_version_list_now_a
        date_version_list_prev = date_version_list_prev_a
        date_version_sphk_now  = date_version_sphk_now_a
        date_version_sphk_prev = date_version_sphk_prev_a
        sat_st = 'ahead'
    ENDIF ELSE BEGIN 
        version_list_now       = version_list_now_b
        version_list_prev      = version_list_prev_b
        version_sphk_now       = version_sphk_now_b
        version_sphk_prev      = version_sphk_prev_b
        date_version_list_now  = date_version_list_now_b
        date_version_list_prev = date_version_list_prev_b
        date_version_sphk_now  = date_version_sphk_now_b
        date_version_sphk_prev = date_version_sphk_prev_b
        sat_st = 'behind'
    ENDELSE 
    svalid_ratrig_ratio = fltarr(32, n_elements(doy_array))
    ratio_valid = intarr(n_elements(doy_array))
    pri_ratio = fltarr(128, n_elements(doy_array))
    pri_ratio_valid = intarr(n_elements(doy_array))
    FOR jj = 0, n_elements(doy_array)-1 DO BEGIN ; do all specified days
        print, "starting ", string(ii), string(doy_array[jj])
        doy_st = string(doy_array[jj], format = '(I3.3)')
        year_st = string(year_array[jj], format = '(I4.4)')
        month_st = string(month_array[jj], format = '(I2.2)')
        inpath = temp_datadir+indir+'/'+year_st+'/'+month_st
        cdfpath = temp_datadir+cdfdir+'/'+year_st
        errorpath = temp_datadir+errordir+'/'+year_st
        IF year GT 2011 OR (year EQ 2011 AND doy_array[jj] GE 152) THEN skt_version = 10 ELSE skt_version = 9
        ; search for data files
        temp_str = 'plastc_'+sat_st+'_'+year_st+'_'+doy_st+'*'
        print, 'Starting file: ', temp_str
        temp_str = concat_dir(inpath, temp_str)
        files_found = file_search(temp_str, count = num_files)
        IF num_files GT 0 THEN BEGIN  ; find most recent version and .fin
            files_found = file_basename(files_found)
            ptp_version = -1
            fin_version = -1
            FOR kk = 0, num_files-1 DO BEGIN 
                temp = strmid(files_found[kk], strlen(files_found[kk])-6, 2)
                temp_version = fix(temp)
                suffix = strmid(files_found[kk], strlen(files_found[kk])-3, 3)
                IF suffix EQ 'fin' THEN BEGIN 
                    IF temp_version GT fin_version THEN BEGIN 
                        fin_version = temp_version
                        best_file = files_found[kk]
                    ENDIF
                ENDIF ELSE BEGIN 
                    IF (temp_version GT ptp_version) AND (fin_version EQ -1) THEN BEGIN 
                        ptp_version = temp_version
                        best_file = files_found[kk]
                    ENDIF
                ENDELSE 
            ENDFOR 
            name_end = strmid(best_file, strlen(best_file)-6, 6)
            IF year_array[jj] EQ year THEN BEGIN
                temp_list = version_list_now[doy_array[jj], *] 
            ENDIF ELSE BEGIN 
                temp_list = version_list_prev[doy_array[jj], *] 
            ENDELSE 
            found = 0
            next_index = -1
            FOR ll = 0, num_versions_allowed-1 DO BEGIN 
                IF temp_list[ll] EQ name_end THEN found = 1 $ ; file already processed, ignore
                ELSE IF (temp_list[ll] EQ '') AND (next_index EQ -1) THEN next_index = ll
            ENDFOR 
            IF doy_specified EQ 1 THEN found = 0
            IF found EQ 0 THEN BEGIN 
                IF ll EQ num_versions_allowed AND next_index EQ -1 THEN BEGIN 
                    print, 'more versions than allowed in variable, change num_versions_allowed: ', num_versions_allowed
                    stop
                ENDIF ELSE BEGIN 
                    plastic_process_l0, best_file, infilepath = inpath, outfilepath = cdfpath, errorfilepath = errorpath, $
                                        cdfreadfilepath = cdfreadpath, outfilename, version = skt_version, skip_email = skip_email
                    ;calculate_svalid_ratrig_ratio, outfilename, year_array[jj], doy_array[jj], temp_ratio
                    ;svalid_ratrig_ratio[*, jj] = temp_ratio
                    ;ratio_valid[jj] = 1
                    ;calculate_pri0_pri0_3_ratio, outfilename, year_array[jj], doy_array[jj], temp_ratio
                    ;pri_ratio[*, jj] = temp_ratio
                    ;pri_ratio_valid[jj] = 1
                    IF year_array[jj] EQ year THEN BEGIN
                        IF wrote_to_now_log EQ 0 THEN BEGIN
                            printf, log_now_lun, systime(0, /utc)
                            wrote_to_now_log = 1
                        ENDIF 
                        temp_st = tab+best_file
                        printf, log_now_lun, temp_st
                        version_list_now[doy_array[jj], next_index] = name_end 
                        date_version_list_now[doy_array[jj]] = systime(0, /utc)
                    ENDIF ELSE BEGIN 
                        IF wrote_to_prev_log EQ 0 THEN BEGIN
                            printf, log_prev_lun, systime(0, /utc)
                            wrote_to_prev_log = 1
                        ENDIF 
                        temp_st = tab+best_file
                        printf, log_prev_lun, temp_st
                        version_list_prev[doy_array[jj], next_index] = name_end 
                        date_version_list_prev[doy_array[jj]] = systime(0, /utc)
                    ENDELSE 
                ENDELSE 
            ENDIF 
        ENDIF 
        ; search for converted spacecraft housekeeping files
        inpath = temp_datadir+sphkdir+'/'+year_st
        temp_str = sat_st+'_'+year_st+'_'+doy_st+'_*'
        temp_str = concat_dir(inpath, temp_str)
        files_found = file_search(temp_str, count = num_files)
        IF num_files GT 0 THEN BEGIN  ; find most recent version
            files_found = file_basename(files_found)
            version = -1
            FOR kk = 0, num_files-1 DO BEGIN 
                temp = strmid(files_found[kk], strlen(files_found[kk])-6, 2)
                temp_version = fix(temp)
                suffix = strmid(files_found[kk], strlen(files_found[kk])-3, 3)
                IF suffix EQ 'csh' THEN BEGIN 
                    IF temp_version GT version THEN BEGIN 
                        version = temp_version
                        best_file = files_found[kk]
                    ENDIF
                ENDIF 
            ENDFOR 
            name_end = strmid(best_file, strlen(best_file)-6, 6)
            IF year_array[jj] EQ year THEN BEGIN
                temp_list = version_sphk_now[doy_array[jj], *] 
            ENDIF ELSE BEGIN 
                temp_list = version_sphk_prev[doy_array[jj], *] 
            ENDELSE 
            found = 0
            next_index = -1
            FOR ll = 0, num_versions_allowed-1 DO BEGIN 
                IF temp_list[ll] EQ name_end THEN found = 1 $ ; file already processed, ignore
                ELSE IF (temp_list[ll] EQ '') AND (next_index EQ -1) THEN next_index = ll
            ENDFOR 
            IF doy_specified EQ 1 THEN found = 0
            IF found EQ 0 THEN BEGIN 
                IF ll EQ num_versions_allowed AND next_index EQ -1 THEN BEGIN 
                    print, 'more versions than allowed in variable, change num_versions_allowed: ', num_versions_allowed
                    stop
                ENDIF ELSE BEGIN 
                    plastic_process_sphk, best_file, infilepath = inpath, outfilepath = cdfpath, errorfilepath = errorpath, $
                      cdfreadfilepath = cdfreadpath, version = skt_version
                    IF year_array[jj] EQ year THEN BEGIN
                        IF wrote_to_now_log EQ 0 THEN BEGIN
                            printf, log_now_lun, systime(0, /utc)
                            wrote_to_now_log = 1
                        ENDIF 
                        temp_st = tab+best_file
                        printf, log_now_lun, temp_st
                        version_sphk_now[doy_array[jj], next_index] = name_end 
                    ENDIF ELSE BEGIN 
                        IF wrote_to_prev_log EQ 0 THEN BEGIN
                            printf, log_prev_lun, systime(0, /utc)
                            wrote_to_prev_log = 1
                        ENDIF 
                        temp_st = tab+best_file
                        printf, log_prev_lun, temp_st
                        version_sphk_prev[doy_array[jj], next_index] = name_end 
                    ENDELSE 
                ENDELSE 
            ENDIF 
        ENDIF 
    ENDFOR 
    ; write out ratio -- assume no more than 2 years in year_array
    ;temp_year = year_array[0]
    ;temp_i = where(year_array EQ temp_year, count)
    ;IF count EQ n_elements(year_array) THEN BEGIN ; all in one year
    ;    update_ratio_file, ii, calibrationpath, year_array, doy_array, svalid_ratrig_ratio, ratio_valid
    ;    update_pri_ratio_file, ii, calibrationpath, year_array, doy_array, pri_ratio, pri_ratio_valid
    ;ENDIF ELSE BEGIN ; two years in year array
    ;    update_ratio_file, ii, calibrationpath, year_array[temp_i], doy_array[temp_i], svalid_ratrig_ratio[temp_i], ratio_valid[temp_i]
    ;    update_pri_ratio_file, ii, calibrationpath, year_array[temp_i], doy_array[temp_i], pri_ratio[temp_i], pri_ratio_valid[temp_i]
    ;    temp_i = where(year_array NE temp_year, count)
    ;    update_ratio_file, ii, calibrationpath, year_array[temp_i], doy_array[temp_i], svalid_ratrig_ratio[temp_i], ratio_valid[temp_i]
    ;    update_pri_ratio_file, ii, calibrationpath, year_array[temp_i], doy_array[temp_i], pri_ratio[temp_i], pri_ratio_valid[temp_i]
    ;ENDELSE 
    IF ii EQ 0 THEN BEGIN
        version_list_now_a = version_list_now
        version_list_prev_a = version_list_prev
        version_sphk_now_a = version_sphk_now
        version_sphk_prev_a = version_sphk_prev
        date_version_list_now_a = date_version_list_now
        date_version_list_prev_a = date_version_list_prev
        date_version_sphk_now_a = date_version_sphk_now
        date_version_sphk_prev_a = date_version_sphk_prev
    ENDIF ELSE BEGIN 
        version_list_now_b = version_list_now
        version_list_prev_b = version_list_prev
        version_sphk_now_b = version_sphk_now
        version_sphk_prev_b = version_sphk_prev
        date_version_list_now_b = date_version_list_now
        date_version_list_prev_b = date_version_list_prev
        date_version_sphk_now_b = date_version_sphk_now
        date_version_sphk_prev_b = date_version_sphk_prev
    ENDELSE 
    IF wrote_to_now_log EQ 0 THEN BEGIN 
        temp_st = systime(0, /utc)
        temp_st = temp_st+tab+'No Files Processed'
        printf, log_now_lun, temp_st
    ENDIF 
    close, log_now_lun
    free_lun, log_now_lun
    IF n_elements(year_list) GT 1 THEN BEGIN 
        close, log_prev_lun
        free_lun, log_prev_lun
    ENDIF     

; write out updated list files
    IF ii EQ 0 THEN BEGIN 
        sat_st = 'A'
        version_list_now = version_list_now_a
        version_list_prev = version_list_prev_a
        version_sphk_now = version_sphk_now_a
        version_sphk_prev = version_sphk_prev_a
        date_version_list_now = date_version_list_now_a
        date_version_list_prev = date_version_list_prev_a
        date_version_sphk_now = date_version_sphk_now_a
        date_version_sphk_prev = date_version_sphk_prev_a
    ENDIF ELSE BEGIN 
        sat_st = 'B'
        version_list_now = version_list_now_b
        version_list_prev = version_list_prev_b
        version_sphk_now = version_sphk_now_b
        version_sphk_prev = version_sphk_prev_b
        date_version_list_now = date_version_list_now_b
        date_version_list_prev = date_version_list_prev_b
        date_version_sphk_now = date_version_sphk_now_b
        date_version_sphk_prev = date_version_sphk_prev_b
    ENDELSE 
    ; data file
    list_filename_now = 'files_processed_'+string(year, format = '(I4.4)')+'_'+sat_st
    list_filepath_now = concat_dir(logpath, list_filename_now)
    openw, list_now_lun, list_filepath_now, /get_lun
    temp_string = 'Ptp and Fin files processed for spacecraft '+sat_st+' for '+string(year, format = '(I4.4)')
    printf, list_now_lun, temp_string
    FOR jj = 1, 366 DO BEGIN 
        temp_string = string(jj, format = '(I3.3)')
        FOR kk = 0, num_versions_allowed-1 DO BEGIN 
            temp_string = temp_string+tab+version_list_now[jj, kk]
        ENDFOR
        printf, list_now_lun, temp_string
    ENDFOR
    close, list_now_lun
    free_lun, list_now_lun
    ; date file for data files
    date_list_filename_now = 'processing_dates_'+string(year, format = '(I4.4)')+'_'+sat_st
    date_list_filepath_now = concat_dir(logpath, date_list_filename_now)
    openw, date_list_now_lun, date_list_filepath_now, /get_lun
    temp_string = 'Most recent processing date for Ptp and Fin files for spacecraft '+sat_st+' for '+string(year, format = '(I4.4)')
    printf, date_list_now_lun, temp_string
    FOR jj = 1, 366 DO BEGIN 
        temp_string = string(jj, format = '(I3.3)')
        temp_string = temp_string+tab+date_version_list_now[jj]
        printf, date_list_now_lun, temp_string
    ENDFOR
    close, date_list_now_lun
    free_lun, date_list_now_lun
    ; spacecraft housekeeping file
    list_filename_now = 'sphk_processed_'+string(year, format = '(I4.4)')+'_'+sat_st
    list_filepath_now = concat_dir(logpath, list_filename_now)
    openw, list_now_lun, list_filepath_now, /get_lun
    temp_string = 'Spacecraft housekeeping files processed for spacecraft '+sat_st+' for '+string(year, format = '(I4.4)')
    printf, list_now_lun, temp_string
    FOR jj = 1, 366 DO BEGIN 
        temp_string = string(jj, format = '(I3.3)')
        FOR kk = 0, num_versions_allowed-1 DO BEGIN 
            temp_string = temp_string+tab+version_sphk_now[jj, kk]
        ENDFOR
        printf, list_now_lun, temp_string
    ENDFOR
    close, list_now_lun
    free_lun, list_now_lun
    ; date file for spacecraft housekeeping files
    date_list_filename_now = 'sphk_processing_dates_'+string(year, format = '(I4.4)')+'_'+sat_st
    date_list_filepath_now = concat_dir(logpath, date_list_filename_now)
    openw, date_list_now_lun, date_list_filepath_now, /get_lun
    temp_string = 'Most recent processing date for spacecraft housekeeping files for spacecraft '+sat_st+' for '+string(year, format = '(I4.4)')
    printf, date_list_now_lun, temp_string
    FOR jj = 1, 366 DO BEGIN 
        temp_string = string(jj, format = '(I3.3)')
        temp_string = temp_string+tab+date_version_sphk_now[jj]
        printf, date_list_now_lun, temp_string
    ENDFOR
    close, date_list_now_lun
    free_lun, date_list_now_lun
    IF n_elements(year_list) GT 1 THEN BEGIN 
        ; data file
        list_filename_prev = 'files_processed_'+string(year-1, format = '(I4.4)')+'_'+sat_st
        list_filepath_prev = concat_dir(logpath, list_filename_prev)
        openw, list_prev_lun, list_filepath_prev, /get_lun
        temp_string = 'Ptp and Fin files processed for spacecraft '+sat_st+' for '+string(year-1, format = '(I4.4)')
        printf, list_prev_lun, temp_string
        FOR jj = 1, 366 DO BEGIN 
            temp_string = string(jj, format = '(I3.3)')
            FOR kk = 0, num_versions_allowed-1 DO BEGIN 
                temp_string = temp_string+tab+version_list_prev[jj, kk]
            ENDFOR
            printf, list_prev_lun, temp_string
        ENDFOR
        close, list_prev_lun
        free_lun, list_prev_lun
                                ; date file for data files
        date_list_filename_prev = 'processing_dates_'+string(year, format = '(I4.4)')+'_'+sat_st
        date_list_filepath_prev = concat_dir(logpath, date_list_filename_prev)
        openw, date_list_prev_lun, date_list_filepath_prev, /get_lun
        temp_string = 'Most recent processing date for Ptp and Fin files for spacecraft '+sat_st+' for '+string(year, format = '(I4.4)')
        printf, date_list_prev_lun, temp_string
        FOR jj = 1, 366 DO BEGIN 
            temp_string = string(jj, format = '(I3.3)')
            temp_string = temp_string+tab+date_version_list_prev[jj]
            printf, date_list_prev_lun, temp_string
        ENDFOR
        close, date_list_prev_lun
        free_lun, date_list_prev_lun
        ; spacecraft housekeeping file
        list_filename_prev = 'sphk_processed_'+string(year-1, format = '(I4.4)')+'_'+sat_st
        list_filepath_prev = concat_dir(logpath, list_filename_prev)
        openw, list_prev_lun, list_filepath_prev, /get_lun
        temp_string = 'Spacecraft housekeeping files processed for spacecraft '+sat_st+' for '+string(year-1, format = '(I4.4)')
        printf, list_prev_lun, temp_string
        FOR jj = 1, 366 DO BEGIN 
            temp_string = string(jj, format = '(I3.3)')
            FOR kk = 0, num_versions_allowed-1 DO BEGIN 
                temp_string = temp_string+tab+version_sphk_prev[jj, kk]
            ENDFOR
            printf, list_prev_lun, temp_string
        ENDFOR
        close, list_prev_lun
        free_lun, list_prev_lun
                                ; date file for spacecraft housekeeping files
        date_list_filename_prev = 'sphk_processing_dates_'+string(year, format = '(I4.4)')+'_'+sat_st
        date_list_filepath_prev = concat_dir(logpath, date_list_filename_prev)
        openw, date_list_prev_lun, date_list_filepath_prev, /get_lun
        temp_string = 'Most recent processing date for spacecraft housekeeping files for spacecraft '+$
                      sat_st+' for '+string(year, format = '(I4.4)')
        printf, date_list_prev_lun, temp_string
        FOR jj = 1, 366 DO BEGIN 
            temp_string = string(jj, format = '(I3.3)')
            temp_string = temp_string+tab+date_version_sphk_prev[jj]
            printf, date_list_prev_lun, temp_string
        ENDFOR
        close, date_list_prev_lun
        free_lun, date_list_prev_lun
    ENDIF
    IF n_elements(sat) GT 0 THEN IF sat EQ 'A' THEN ii = 2
ENDFOR 
print, "Finished CDFs"

; run read_cdf
check_cdfs, datadir, cdfdir, cdfreadpath
;!PATH = '/Users/lornaellis/Lorna/InstrTest/ReadCdf:' + !PATH
!PATH = '~/ReadCdf:' + !PATH
;command = 'cp /Users/lornaellis/Lorna/InstrTest/ReadCdf/read_cdf.settings .'
command = 'cp ~/ReadCdf/read_cdf.settings .'
spawn, command
read_cdf
command = 'rm read_cdf.settings'
spawn, command
zip_copy, cdfoutpath, zippath_a, zippath_b
print, "Finished read_cdf"

; run science_overview
;!PATH = '/Users/lornaellis/Lorna/InstrTest/Classifier/Plot/CDF/:' + !PATH
!PATH = '~/ScienceOverview/:' + !PATH
;command = 'cp /Users/lornaellis/Lorna/InstrTest/Classifier/Plot/CDF/science_overview.settings .'
command = 'cp ~/ScienceOverview/science_overview.settings .'
spawn, command
;command = 'cp /Users/lornaellis/Lorna/InstrTest/Classifier/Plot/CDF/*.txt .'
command = 'cp ~/ScienceOverview/*.txt .'
spawn, command
science_overview
command = 'rm science_overview.settings'
spawn, command
plot_copy, tempplotpath, finalplotpath 
command = 'rm '+cdfreadpath+'/*'
spawn, command

command = 'echo "finished_level1  '+systime()+'" | mail lellis@spam.sr.unh.edu'
spawn, command
print, "Done"
END 
