; Lorna Ellis
; 05/23/2007
; find_norm_data.pro

; This program takes the raw pha counts, and normalizes them.

; Input: counts -- [num_esa, 16, 4, 16, num_epoch]  
;                  esa, defl (summed across 2), priority, counts per bin (not all used), valid epochs
;        pri_counts -- [num_esa, 16, 4 or 2, num_epoch] same as above, but priority taken from priority bit
;        when_sum_defl    : 0=sum before normalization, 1=sum after normalization
;        temp_file_start  : string with beginning of file name for error files
;        temp_file_stop   : string with beginning of file name for error files
;
; Output: _norm -- [esa, defl (for sw side), num_bins, num_epoch]
;         summed_..._counts_time
;         num_intervals : the number of time intervals

; 10/15/2007 Output no longer summed over time.
; 12/04/2007 Added summed_..._counts_time to output
;            Added norm_cycles
; 01/24/2008 Changed format of start_time and stop_time
; 01/25/2008 Add option for summing defl before or after normalization
; 01/29/2008 Added num_intervals
; 02/21/2008 Only print error if in range of wanted ESA
; 04/18/2008 Added check on plots_wanted when denom is 0
; 05/27/2008 Added check for data gap when checking cycles
; 07/10/2008 Fixed bug when normalizing over whole file.
; 07/16/2008 Added converted_valid_epoch5.
; 07/17/2008 Fixed bug with lining up epochs.
; 07/28/2008 Remove 5 minute data where schan = -1
;            Changed so timing set on epoch rather than cycle
; 10/07/2008 Changed so buffer is in pri_counts arrays.
; 07/13/2010 Added finished_5 and finished_1 checks (for 12/20/2009 on A).
; 07/20/2010 Fixed bug where norm_cycles_time missing if normalized over whole day.
; 04/04/2011 Add check for when cycle = -1.
; 04/06/2011 Fixed bug when small data gap.
; 04/28/2011 Fixed bug with fill values in pri data, and with data gaps.
; 05/03/2011 Check for ii-kk<0 when checking for data gap.
; 09/29/2011 Fixed bug when no 1 min data in 5 min.
; 10/05/2011 Fixed bug when finished 5 min.
; 10/12/2011 Significant rewrite for efficiency.
; 10/26/2011 _counts now only have needed ESAs.
; 02/16/2012 Fixed bugs relating to num cycles counted.
; 04/02/2012 Remove num cycles when only want section 0 or 1.
; 01/23/2014 Change to allow base rate greater than 5.
; 03/26/2014 Change doubles to floats

PRO normalize, nom, denom, rate, have_defl, num_intervals, temp_values, num_esa
compile_opt strictarrsubs
; in: nom          : numerator of normalization (from summed_..._counts_time)
;     denom        : denominator of normalization (from summed_..._pri_counts_time)
;     rate         : from ..._pri_data_time
;     have_defl    : 0=don't have defl (already summed), arrays are smaller; 1=still have defl in arrays
;     num_intervals: number of time intervals (for size of arrays)
; out: temp_values: normalized data for this bin and priority

IF have_defl EQ 0 THEN temp_values = fltarr(num_esa, num_intervals) ELSE temp_values = fltarr(num_esa, 16, num_intervals)
invalid_divisor_index = where(denom EQ 0, invalid_divisor_count)
valid_divisor_index   = where(denom NE 0, valid_divisor_count)
IF invalid_divisor_count GT 0 THEN BEGIN 
    IF valid_divisor_count GT 0 THEN BEGIN 
        temp_values[valid_divisor_index] = (float(rate[valid_divisor_index])*$
                                            float(nom [valid_divisor_index])/float(denom[valid_divisor_index])) 
    ENDIF ELSE BEGIN
        IF have_defl EQ 0 THEN temp_values[*, *] = 0 ELSE temp_values[*, *, *] = 0
    ENDELSE 
ENDIF ELSE BEGIN 
    IF have_defl EQ 0 THEN temp_values[*, *] = (float(rate)*float(nom)/float(denom)) ELSE $
      temp_values[*, *, *] = (float(rate)*float(nom)/float(denom))
ENDELSE 
END 

PRO write_error_log, filename, pri, have_defl, denom, rate, num_esa
; in: filename     : string
;     pri          : priority (0-3)
;     have_defl    : 0=don't have defl (already summed), arrays are smaller; 1=still have defl in arrays
;     denom        : denominator of normalization (from summed_..._pri_counts_time)
;     rate         : from ..._pri_data_time
compile_opt strictarrsubs
COMMON com_pha_play
tab = STRING(9B)

temp_filename = filename+string(pri, format = '(I1.1)')+'.txt'
openw, error_lun, temp_filename, /get_lun
temp_string = 'Statistical data for '+filename+' for priority '+string(pri, format = '(I1.1)')
printf, error_lun, temp_string
temp_string = 'Normalization interval: '+string(norm_interval, format = '(I4)')+' Results summed over time.'
printf, error_lun, temp_string
temp_string = 'ESA step'+tab+'num (pha=0 rate>0)'+tab+'num (pha>0 rate>0)'+tab+'ratio'$
              +tab+'counts in primary buffer'+tab+'counts in secondary buffer'+tab+'total counts in pha'+tab+$
              'counts in rate'+tab+'ratio'
printf, error_lun, temp_string
FOR ll = 0, num_esa-1 DO BEGIN 
    index = ll+esa_start
    IF ll MOD norm_esa_sum EQ 0 THEN BEGIN 
        IF have_defl EQ 0 THEN BEGIN ; no defl
            temp_primary         = reform(denom[ll, 0, *])
            temp_secondary       = reform(denom[ll, 1, *])
            temp_denom           = temp_primary+temp_secondary
            temp_rate            = reform(rate [ll, *])
            temp_primary_count   = total(temp_primary)
            temp_secondary_count = total(temp_secondary)
            temp_denom_count     = total(temp_denom)
            temp_rate_count      = total(rate[ll, *])
        ENDIF ELSE BEGIN        ; still has defl
            temp_primary         = reform(denom[ll, *, 0, *])
            temp_secondary       = reform(denom[ll, *, 1, *])
            temp_denom           = temp_primary+temp_secondary
            temp_rate            = reform(rate[ll, *, *])
            temp_primary_count   = total(temp_primary)
            temp_secondary_count = total(temp_secondary)
            temp_denom_count     = total(temp_denom)
            temp_rate_count      = total(rate[ll, *, *])
        ENDELSE 
        bad_stats_index  = where(temp_denom EQ 0 AND temp_rate NE 0, bad_stats_count)
        good_stats_index = where(temp_denom GT 0 AND temp_rate GT 0, good_stats_count)
        IF norm_esa_sum EQ 2 THEN temp_esa_string = string(index, format = '(I3.3)')+'/'+$
          string(index+1, format = '(I3.3)') ELSE temp_esa_string = string(index, format = '(I3.3)')
        temp_string = temp_esa_string+tab+string(bad_stats_count)+tab+string(good_stats_count)+tab+$
                      string(float(bad_stats_count)/good_stats_count)
        IF want_buffer_info EQ 1 THEN temp_string = temp_string+tab+string(temp_primary_count)+tab+$
                      string(temp_secondary_count) ELSE temp_string = temp_string+tab+tab
        temp_string = temp_string+tab+string(temp_denom_count)+$
                      tab+string(temp_rate_count)+tab+string(float(temp_denom_count)/temp_rate_count)
        printf, error_lun, temp_string
    ENDIF 
ENDFOR 
close, error_lun
free_lun, error_lun
END 

PRO find_norm_data, num_bins, swpha_counts, wappha_ssd_counts, wappha_nossd_counts, supranoe_ssd_counts, supranoe_nossd_counts, $
                    suprawide_counts, swz_counts, swhalpha_counts, $
                    swpha_pri_counts, wappha_ssd_pri_counts, wappha_nossd_pri_counts, $
                    when_sum_defl, temp_file_start, temp_file_end, $
                    swpha_norm, wappha_ssd_norm, wappha_nossd_norm, supranoe_ssd_norm, supranoe_nossd_norm, $
                    suprawide_norm, swz_norm, swhalpha_norm, $
                    summed_swpha_counts_time, summed_wappha_ssd_counts_time, summed_wappha_nossd_counts_time, $
                    summed_supranoe_ssd_counts_time, $
                    summed_supranoe_nossd_counts_time, summed_suprawide_counts_time, summed_swz_counts_time, $
                    summed_swhalpha_counts_time, $
                    num_intervals, sw_pri, wap_pri_ssd, wap_pri_nossd, cdf_base_brief
COMMON com_pha_play
compile_opt strictarrsubs
tab = STRING(9B)

num_esa = esa_stop-esa_start+1
swpha_norm          = lonarr(num_esa, 16, num_bins[0]) 
wappha_ssd_norm     = lonarr(num_esa,     num_bins[1]) 
wappha_nossd_norm   = lonarr(num_esa, 16, num_bins[2]) 
supranoe_ssd_norm   = lonarr(num_esa, 16, num_bins[3]) 
supranoe_nossd_norm = lonarr(num_esa, 16, num_bins[4]) 
suprawide_norm      = lonarr(num_esa, 16, num_bins[5]) 
swz_norm            = lonarr(num_esa, 16, num_bins[6]) 
swhalpha_norm       = lonarr(num_esa, 16, num_bins[7])

IF n_elements(converted_valid_epoch1) GT 0 THEN have_data1 = 1

IF have_data1 EQ 1 THEN BEGIN     
    ; check arrays
    size_sw_pri        = size(sw_pri)        ; num_esax16x4 [esa,defl,class,time]
    size_wap_pri_ssd   = size(wap_pri_ssd)   ; num_esax2    [esa, class,time]
    size_wap_pri_nossd = size(wap_pri_nossd) ; num_esax2    [esa,class,time]
    IF (size_sw_pri[4] NE size_wap_pri_ssd[3]) OR (size_sw_pri[4] NE size_wap_pri_nossd[3]) OR $
      (size_sw_pri[4] NE n_elements(converted_valid_epoch5)) OR $
      (size_sw_pri[4] NE n_elements(s_chan5))      $
      THEN BEGIN
        print, 'Error in find_norm_data: False assumption about array sizes.'
        stop
    ENDIF 

    ; combine every 2 esas if requested
    IF norm_esa_sum EQ 2 THEN BEGIN 
        FOR ii = 0, (num_esa/2)-1 DO BEGIN 
            IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN 
                swpha_counts           [ (ii*2),    *, *, *, *] = swpha_counts         [(ii*2), *, *, *, *] + $
                  swpha_counts         [((ii*2)+1), *, *, *, *]
                swpha_counts           [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN 
                wappha_ssd_counts      [ (ii*2),    *, *, *, *] = wappha_ssd_counts    [(ii*2), *, *, *, *] + $
                  wappha_ssd_counts    [((ii*2)+1), *, *, *, *]
                wappha_ssd_counts      [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN 
                wappha_nossd_counts    [ (ii*2),    *, *, *, *] = wappha_nossd_counts  [(ii*2), *, *, *, *] + $
                  wappha_nossd_counts  [((ii*2)+1), *, *, *, *]
                wappha_nossd_counts    [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[3] EQ 1 THEN BEGIN 
                supranoe_ssd_counts    [ (ii*2),    *, *, *, *] = supranoe_ssd_counts  [(ii*2), *, *, *, *] + $
                  supranoe_ssd_counts  [((ii*2)+1), *, *, *, *]
                supranoe_ssd_counts    [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[4] EQ 1 THEN BEGIN 
                supranoe_nossd_counts  [ (ii*2),    *, *, *, *] = supranoe_nossd_counts[(ii*2), *, *, *, *] + $
                  supranoe_nossd_counts[((ii*2)+1), *, *, *, *]
                supranoe_nossd_counts  [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[5] EQ 1 THEN BEGIN 
                suprawide_counts       [ (ii*2),    *, *, *, *] = suprawide_counts     [(ii*2), *, *, *, *] + $
                  suprawide_counts     [((ii*2)+1), *, *, *, *]
                suprawide_counts       [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[6] EQ 1 THEN BEGIN 
                swz_counts             [ (ii*2),    *, *, *, *] = swz_counts           [(ii*2), *, *, *, *] + $
                  swz_counts           [((ii*2)+1), *, *, *, *]
                swz_counts             [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            IF plots_wanted[7] EQ 1 THEN BEGIN 
                swhalpha_counts        [ (ii*2),    *, *, *, *] = swhalpha_counts      [(ii*2), *, *, *, *] + $
                  swhalpha_counts      [((ii*2)+1), *, *, *, *]
                swhalpha_counts        [((ii*2)+1), *, *, *, *] = 0
            ENDIF 
            sw_pri                   [(ii*2),     *, *, *   ] = sw_pri                 [(ii*2), *, *, *   ] + $
              sw_pri                 [((ii*2)+1), *, *, *   ]
            wap_pri_ssd              [(ii*2),     *, *      ] = wap_pri_ssd            [(ii*2), *, *      ] + $
              wap_pri_ssd            [((ii*2)+1), *, *      ]
            wap_pri_nossd            [(ii*2),     *, *      ] = wap_pri_nossd          [(ii*2), *, *      ] + $
              wap_pri_nossd          [((ii*2)+1), *, *      ]
            swpha_pri_counts         [(ii*2),     *, *, *, *] = swpha_pri_counts       [(ii*2), *, *, *, *] + $
              swpha_pri_counts       [((ii*2)+1), *, *, *, *]
            wappha_ssd_pri_counts    [(ii*2),     *, *, *, *] = wappha_ssd_pri_counts  [(ii*2), *, *, *, *] + $
              wappha_ssd_pri_counts  [((ii*2)+1), *, *, *, *]
            wappha_nossd_pri_counts  [(ii*2),     *, *, *, *] = wappha_nossd_pri_counts[(ii*2), *, *, *, *] + $
              wappha_nossd_pri_counts[((ii*2)+1), *, *, *, *]
            sw_pri                   [((ii*2)+1), *, *, *   ] = 0
            wap_pri_ssd              [((ii*2)+1), *, *      ] = 0
            wap_pri_nossd            [((ii*2)+1), *, *      ] = 0
            swpha_pri_counts         [((ii*2)+1), *, *, *, *] = 0
            wappha_ssd_pri_counts    [((ii*2)+1), *, *, *, *] = 0
            wappha_nossd_pri_counts  [((ii*2)+1), *, *, *, *] = 0
        ENDFOR 
    ENDIF 

    ; line up epoch1 with epoch5 
    num_records5                   = n_elements( converted_valid_epoch5)
    norm_cycles                    = intarr(num_esa,           num_records5)
    summed_swpha_pri_counts        = fltarr(num_esa, 16, 4, 2, num_records5)
    summed_wappha_ssd_pri_counts   = fltarr(num_esa, 16, 2, 2, num_records5)
    summed_wappha_nossd_pri_counts = fltarr(num_esa, 16, 2, 2, num_records5)
    summed_swpha_pri_counts       [*, *, *, *, *] = !values.f_nan
    summed_wappha_nossd_pri_counts[*, *, *, *, *] = !values.f_nan
    summed_wappha_ssd_pri_counts  [*, *, *, *, *] = !values.f_nan
    IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN 
        summed_swpha_counts                         = fltarr(num_esa, 16, 4, num_bins[0], num_records5) 
        summed_swpha_counts         [*, *, *, *, *] = !values.f_nan
    ENDIF 
    IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN
        summed_wappha_ssd_counts                    = fltarr(num_esa, 16, 4, num_bins[1], num_records5) 
        summed_wappha_ssd_counts    [*, *, *, *, *] = !values.f_nan
    ENDIF 
    IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN
        summed_wappha_nossd_counts                  = fltarr(num_esa, 16, 4, num_bins[2], num_records5) 
        summed_wappha_nossd_counts  [*, *, *, *, *] = !values.f_nan
    ENDIF 
    IF plots_wanted[3] EQ 1                         THEN BEGIN
        summed_supranoe_ssd_counts                  = fltarr(num_esa, 16, 4, num_bins[3], num_records5) 
        summed_supranoe_ssd_counts  [*, *, *, *, *] = !values.f_nan
    ENDIF 
    IF plots_wanted[4] EQ 1                         THEN BEGIN
        summed_supranoe_nossd_counts                = fltarr(num_esa, 16, 4, num_bins[4], num_records5) 
        summed_supranoe_nossd_counts[*, *, *, *, *] = !values.f_nan
    ENDIF
    IF plots_wanted[5] EQ 1                         THEN BEGIN
        summed_suprawide_counts                     = fltarr(num_esa, 16, 4, num_bins[5], num_records5) 
        summed_suprawide_counts     [*, *, *, *, *] = !values.f_nan
    ENDIF 
    IF plots_wanted[6] EQ 1                         THEN BEGIN
        summed_swz_counts                           = fltarr(num_esa, 16, 4, num_bins[6], num_records5) 
        summed_swz_counts           [*, *, *, *, *] = !values.f_nan
    ENDIF 
    IF plots_wanted[7] EQ 1                         THEN BEGIN
        summed_swhalpha_counts                      = fltarr(num_esa, 16, 4, num_bins[7], num_records5)
        summed_swhalpha_counts      [*, *, *, *, *] = !values.f_nan
    ENDIF 

    num_valid_epoch1 = n_elements(converted_valid_epoch1)
    FOR jj = 0, n_elements(converted_valid_epoch5)-1 DO BEGIN 
        wanted_i = where(converted_valid_epoch1 GE (converted_valid_epoch5[jj]-10) AND $
                         converted_valid_epoch1 LT (converted_valid_epoch5[jj]+(base_intervals[jj]*60)-10), count)
        IF jj EQ n_elements(converted_valid_epoch5)-1 AND count LT 5 THEN count = 0 ; drop end interval
        IF count EQ 1 THEN BEGIN 
            IF s_chan1[wanted_i] EQ s_chan5[jj] THEN norm_cycles[*, jj] = norm_cycles[*, jj] + 1 ELSE $
              IF s_chan1[wanted_i] LT s_chan5[jj] THEN BEGIN 
                IF esa_start GE s_chan5[jj] OR esa_stop LT s_chan1[wanted_i] THEN $
                  norm_cycles[*, jj] = norm_cycles[*, jj] + 1 ELSE BEGIN 
                    IF s_chan1[wanted_i]-1-esa_start GE 0 THEN $
                      norm_cycles [0:s_chan1[wanted_i]-esa_start-1,      jj] = norm_cycles[0:s_chan1[wanted_i]-esa_start-1,       jj] + 1
                    IF s_chan5[jj]-esa_start LE num_esa-1 THEN $
                      norm_cycles [s_chan5[jj]-esa_start:num_esa-1,      jj] = norm_cycles[s_chan5[jj]-esa_start:num_esa-1,       jj] + 1
                ENDELSE 
            ENDIF ELSE IF s_chan1[wanted_i] GT s_chan5[jj] THEN BEGIN 
                IF esa_start GE s_chan1[wanted_i] OR esa_stop LT s_chan5[jj] THEN $
                  norm_cycles[*, jj] = norm_cycles[*, jj] + 1 ELSE BEGIN 
                    IF s_chan5[jj]-1-esa_start GE 0 THEN $
                      norm_cycles[0:s_chan5[jj]-esa_start-1,             jj] = norm_cycles[0:s_chan5[jj]-esa_start-1,             jj] + 1
                    IF s_chan1[wanted_i]-esa_start LE num_esa-1 THEN $
                      norm_cycles[s_chan1[wanted_i]-esa_start:num_esa-1, jj] = norm_cycles[s_chan1[wanted_i]-esa_start:num_esa-1, jj] + 1
                ENDELSE 
            ENDIF 
            IF section_wanted EQ 0 THEN BEGIN ; only want main channel
                IF s_chan5[jj] GE esa_start THEN BEGIN 
                    IF s_chan5[jj] LE esa_stop THEN norm_cycles[s_chan5[jj]-esa_start:num_esa-1, jj] = 0
                ENDIF ELSE BEGIN 
                    norm_cycles[*, jj] = 0
                ENDELSE 
            ENDIF 
            IF section_wanted EQ 1 THEN BEGIN ; only want S channel
                IF s_chan5[jj] LE esa_stop THEN BEGIN 
                    IF s_chan5[jj] GE esa_start THEN norm_cycles[0:s_chan5[jj]-esa_start, jj] = 0
                ENDIF ELSE BEGIN 
                    norm_cycles[*, jj] = 0
                ENDELSE 
            ENDIF 
            summed_swpha_pri_counts         [*, *, *, *, jj] = swpha_pri_counts       [*, *, *, *, wanted_i]
            summed_wappha_ssd_pri_counts    [*, *, *, *, jj] = wappha_ssd_pri_counts  [*, *, *, *, wanted_i]
            summed_wappha_nossd_pri_counts  [*, *, *, *, jj] = wappha_nossd_pri_counts[*, *, *, *, wanted_i]
            IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN $
              summed_swpha_counts           [*, *, *, *, jj] = swpha_counts           [*, *, *, *, wanted_i]
            IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN $
              summed_wappha_ssd_counts      [*, *, *, *, jj] = wappha_ssd_counts      [*, *, *, *, wanted_i]
            IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN $
              summed_wappha_nossd_counts    [*, *, *, *, jj] = wappha_nossd_counts    [*, *, *, *, wanted_i]
            IF plots_wanted[3] EQ 1 THEN $
              summed_supranoe_ssd_counts    [*, *, *, *, jj] = supranoe_ssd_counts    [*, *, *, *, wanted_i]
            IF plots_wanted[4] EQ 1 THEN $
              summed_supranoe_nossd_counts  [*, *, *, *, jj] = supranoe_nossd_counts  [*, *, *, *, wanted_i]
            IF plots_wanted[5] EQ 1 THEN $
              summed_suprawide_counts       [*, *, *, *, jj] = suprawide_counts       [*, *, *, *, wanted_i]
            IF plots_wanted[6] EQ 1 THEN $
              summed_swz_counts             [*, *, *, *, jj] = swz_counts             [*, *, *, *, wanted_i]
            IF plots_wanted[7] EQ 1 THEN $
              summed_swhalpha_counts        [*, *, *, *, jj] = swhalpha_counts        [*, *, *, *, wanted_i]
        ENDIF ELSE IF count GT 0 THEN BEGIN 
            equal_i = where(s_chan1[wanted_i] EQ s_chan5[jj], equal_count)
            lt_i    = where(s_chan1[wanted_i] LT s_chan5[jj], lt_count)
            gt_i    = where(s_chan1[wanted_i] GT s_chan5[jj], gt_count) 
            IF equal_count GT 0 THEN norm_cycles[*, jj] = norm_cycles[*, jj] + equal_count
            IF lt_count GT 0 THEN BEGIN 
                FOR kk = 0, lt_count-1 DO BEGIN 
                    IF esa_start GE s_chan5[jj] OR esa_stop LT s_chan1[wanted_i[lt_i[kk]]] THEN $
                      norm_cycles  [*,        jj] = $
                        norm_cycles[*,        jj] + 1 $
                    ELSE BEGIN 
                        IF s_chan1[wanted_i[lt_i[kk]]]-1-esa_start GE 0 THEN $
                          norm_cycles  [0:s_chan1[wanted_i[lt_i[kk]]]-1-esa_start,        jj] = $
                            norm_cycles[0:s_chan1[wanted_i[lt_i[kk]]]-1-esa_start,        jj] + 1 
                        IF s_chan5[jj]-esa_start LE num_esa-1 THEN $
                          norm_cycles  [s_chan5[jj]-esa_start:num_esa-1,        jj] = $
                            norm_cycles[s_chan5[jj]-esa_start:num_esa-1,        jj] + 1 
                    ENDELSE 
                ENDFOR 
            ENDIF 
            IF gt_count GT 0 THEN BEGIN 
                FOR kk = 0, gt_count-1 DO BEGIN 
                    IF esa_start GE s_chan1[wanted_i[gt_i[kk]]] OR esa_stop LT s_chan5[jj] THEN $
                      norm_cycles  [*,  jj] = $
                        norm_cycles[*,  jj] + 1 $
                    ELSE BEGIN 
                        IF s_chan5[jj]-1-esa_start GE 0 THEN $
                          norm_cycles  [0:s_chan5[jj]-esa_start-1,  jj] = $
                            norm_cycles[0:s_chan5[jj]-esa_start-1,  jj] + 1
                        IF s_chan1[wanted_i[gt_i[kk]]]-esa_start LE num_esa-1 THEN $
                          norm_cycles  [s_chan1[wanted_i[gt_i[kk]]]-esa_start:num_esa-1,  jj] = $
                            norm_cycles[s_chan1[wanted_i[gt_i[kk]]]-esa_start:num_esa-1,  jj] + 1
                    ENDELSE 
                ENDFOR 
            ENDIF 
            IF section_wanted EQ 0 THEN BEGIN ; only want main channel
                IF s_chan5[jj] GE esa_start THEN BEGIN 
                    IF s_chan5[jj] LE esa_stop THEN norm_cycles[s_chan5[jj]-esa_start:num_esa-1, jj] = 0
                ENDIF ELSE BEGIN 
                    norm_cycles[*, jj] = 0
                ENDELSE 
            ENDIF 
            IF section_wanted EQ 1 THEN BEGIN ; only want S channel
                IF s_chan5[jj] LE esa_stop THEN BEGIN 
                    IF s_chan5[jj] GE esa_start THEN norm_cycles[0:s_chan5[jj]-esa_start, jj] = 0
                ENDIF ELSE BEGIN 
                    norm_cycles[*, jj] = 0
                ENDELSE 
            ENDIF 
            summed_swpha_pri_counts         [*, *, *, *, jj] = total(swpha_pri_counts       [*, *, *, *, wanted_i], 5, /nan)
            summed_wappha_ssd_pri_counts    [*, *, *, *, jj] = total(wappha_ssd_pri_counts  [*, *, *, *, wanted_i], 5, /nan)
            summed_wappha_nossd_pri_counts  [*, *, *, *, jj] = total(wappha_nossd_pri_counts[*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN $
              summed_swpha_counts           [*, *, *, *, jj] = total(swpha_counts           [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN $
              summed_wappha_ssd_counts      [*, *, *, *, jj] = total(wappha_ssd_counts      [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN $
              summed_wappha_nossd_counts    [*, *, *, *, jj] = total(wappha_nossd_counts    [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[3] EQ 1 THEN $
              summed_supranoe_ssd_counts    [*, *, *, *, jj] = total(supranoe_ssd_counts    [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[4] EQ 1 THEN $
              summed_supranoe_nossd_counts  [*, *, *, *, jj] = total(supranoe_nossd_counts  [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[5] EQ 1 THEN $
              summed_suprawide_counts       [*, *, *, *, jj] = total(suprawide_counts       [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[6] EQ 1 THEN $
              summed_swz_counts             [*, *, *, *, jj] = total(swz_counts             [*, *, *, *, wanted_i], 5, /nan)
            IF plots_wanted[7] EQ 1 THEN $
              summed_swhalpha_counts        [*, *, *, *, jj] = total(swhalpha_counts        [*, *, *, *, wanted_i], 5, /nan)
        ENDIF 
    ENDFOR 

    ; normalize
    have_data = 1
    IF norm_interval EQ -1 THEN BEGIN ; normalize over whole file
        num_intervals = 1
        norm_times = [converted_valid_epoch5[0]]
        norm_cycles_time = norm_cycles
        IF size(sw_pri,                              /n_dimensions) EQ 4 THEN $
          sw_pri_data_time                         = total(sw_pri,                         4, /nan) $ ; sum over time
          ELSE sw_pri_data_time                    =       sw_pri
        IF size(wap_pri_ssd,                         /n_dimensions) EQ 3 THEN $
          wap_pri_ssd_data_time                    = total(wap_pri_ssd,                    3, /nan) $ ; sum over time
          ELSE wap_pri_ssd_data_time               =       wap_pri_ssd
        IF size(wap_pri_nossd,                       /n_dimensions) EQ 3 THEN $
          wap_pri_nossd_data_time                  = total(wap_pri_nossd,                  3, /nan) $ ; sum over time
          ELSE wap_pri_nossd_data_time             =       wap_pri_nossd
        IF size(summed_swpha_pri_counts_time,        /n_dimensions) EQ 5 THEN $
          summed_swpha_pri_counts_time             = total(summed_swpha_pri_counts,        5, /nan) $
          ELSE summed_swpha_pri_counts_time        =       summed_swpha_pri_counts
        IF size(summed_wappha_ssd_pri_counts_time,   /n_dimensions) EQ 5 THEN $
          summed_wappha_ssd_pri_counts_time        = total(summed_wappha_ssd_pri_counts,   5, /nan) $
          ELSE summed_wappha_ssd_pri_counts_time   =       summed_wappha_ssd_pri_counts
        IF size(summed_wappha_nossd_pri_counts_time, /n_dimensions) EQ 5 THEN $
          summed_wappha_nossd_pri_counts_time      = total(summed_wappha_nossd_pri_counts, 5, /nan) $
          ELSE summed_wappha_nossd_pri_counts_time =       summed_wappha_nossd_pri_counts
        FOR ii = 0, 7 DO BEGIN
            IF plots_wanted[ii] EQ 1 OR (ii LE 2 AND files_wanted[2] EQ 1) THEN BEGIN
                CASE ii OF
                    0: BEGIN
                        IF size(summed_swpha_counts,          /n_dimensions) EQ 5 THEN $
                          summed_swpha_counts_time               = total(summed_swpha_counts,          5, /nan) $ ; sum over time
                          ELSE summed_swpha_counts_time          =       summed_swpha_counts
                    END
                    1: BEGIN
                        IF size(summed_wappha_ssd_counts,     /n_dimensions) EQ 5 THEN $
                          summed_wappha_ssd_counts_time          = total(summed_wappha_ssd_counts,     5, /nan) $ 
                          ELSE summed_wappha_ssd_counts_time     =       summed_wappha_ssd_counts
                    END
                    2: BEGIN
                        IF size(summed_wappha_nossd_counts,   /n_dimensions) EQ 5 THEN $
                          summed_wappha_nossd_counts_time        = total(summed_wappha_nossd_counts,   5, /nan) $ 
                          ELSE summed_wappha_nossd_counts_time   =       summed_wappha_nossd_counts
                    END
                    3: BEGIN
                        IF size(summed_supranoe_ssd_counts,   /n_dimensions) EQ 5 THEN $
                          summed_supranoe_ssd_counts_time        = total(summed_supranoe_ssd_counts,   5, /nan) $ 
                          ELSE summed_supranoe_ssd_counts_time   =       summed_supranoe_ssd_counts
                    END
                    4: BEGIN
                        IF size(summed_supranoe_nossd_counts, /n_dimensions) EQ 5 THEN $
                          summed_supranoe_nossd_counts_time      = total(summed_supranoe_nossd_counts, 5, /nan) $ 
                          ELSE summed_supranoe_nossd_counts_time =       summed_supranoe_nossd_counts
                    END
                    5: BEGIN
                        IF size(summed_suprawide_counts,      /n_dimensions) EQ 5 THEN $
                          summed_suprawide_counts_time           = total(summed_suprawide_counts,      5, /nan) $ 
                          ELSE summed_suprawide_counts_time      =        summed_suprawide_counts
                    END
                    6: BEGIN
                        IF size(summed_swz_counts,            /n_dimensions) EQ 5 THEN $
                          summed_swz_counts_time                 = total(summed_swz_counts,            5, /nan) $ ; sum over time
                          ELSE summed_swz_counts_time            =       summed_swz_counts
                    END
                    7: BEGIN
                        IF size(summed_swhalpha_counts,       /n_dimensions) EQ 5 THEN $
                          summed_swhalpha_counts_time            = total(summed_swhalpha_counts,       5, /nan) $ 
                          ELSE summed_swhalpha_counts_time       =       summed_swhalpha_counts
                    END
                ENDCASE
            ENDIF
        ENDFOR
    ENDIF ELSE BEGIN ; not going over whole day
        norm_times = indgen(fix((stop_time-start_time)/(norm_interval*60)))*(norm_interval*60.0)+start_time
        num_intervals = n_elements(norm_times)
        ; sum over time interval
        IF num_intervals GT 1 THEN BEGIN 
            norm_cycles_time                                               = fltarr(num_esa,                     num_intervals)
            sw_pri_data_time                                               = fltarr(num_esa, 16, 4,              num_intervals)
            wap_pri_ssd_data_time                                          = fltarr(num_esa,                  2, num_intervals)
            wap_pri_nossd_data_time                                        = fltarr(num_esa,                  2, num_intervals)
            summed_swpha_pri_counts_time                                   = fltarr(num_esa, 16, 4,           2, num_intervals)
            summed_wappha_ssd_pri_counts_time                              = fltarr(num_esa, 16, 2,           2, num_intervals)
            summed_wappha_nossd_pri_counts_time                            = fltarr(num_esa, 16, 2,           2, num_intervals)
            norm_cycles_time                               [*,          *] = !values.f_nan 
            sw_pri_data_time                               [*, *, *,    *] = !values.f_nan 
            wap_pri_ssd_data_time                          [*,       *, *] = !values.f_nan 
            wap_pri_nossd_data_time                        [*,       *, *] = !values.f_nan 
            summed_swpha_pri_counts_time                   [*, *, *, *, *] = !values.f_nan 
            summed_wappha_ssd_pri_counts_time              [*, *, *, *, *] = !values.f_nan 
            summed_wappha_nossd_pri_counts_time            [*, *, *, *, *] = !values.f_nan 
            IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN 
                summed_swpha_counts_time                         = fltarr(num_esa, 16, 4, num_bins[0], num_intervals)
                summed_swpha_counts_time         [*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN
                summed_wappha_ssd_counts_time                    = fltarr(num_esa, 16, 4, num_bins[1], num_intervals)
                summed_wappha_ssd_counts_time    [*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN BEGIN 
                summed_wappha_nossd_counts_time                  = fltarr(num_esa, 16, 4, num_bins[2], num_intervals)
                summed_wappha_nossd_counts_time  [*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[3] EQ 1                         THEN BEGIN 
                summed_supranoe_ssd_counts_time                  = fltarr(num_esa, 16, 4, num_bins[3], num_intervals)
                summed_supranoe_ssd_counts_time  [*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[4] EQ 1                         THEN BEGIN
                summed_supranoe_nossd_counts_time                = fltarr(num_esa, 16, 4, num_bins[4], num_intervals)
                summed_supranoe_nossd_counts_time[*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[5] EQ 1                         THEN BEGIN 
                summed_suprawide_counts_time                     = fltarr(num_esa, 16, 4, num_bins[5], num_intervals)
                summed_suprawide_counts_time     [*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[6] EQ 1                         THEN BEGIN
                summed_swz_counts_time                           = fltarr(num_esa, 16, 4, num_bins[6], num_intervals)
                summed_swz_counts_time           [*, *, *, *, *] = !values.f_nan
            ENDIF 
            IF plots_wanted[7] EQ 1                         THEN BEGIN
                summed_swhalpha_counts_time                      = fltarr(num_esa, 16, 4, num_bins[7], num_intervals)
                summed_swhalpha_counts_time      [*, *, *, *, *] = !values.f_nan
            ENDIF 
            
            FOR jj = 0, num_intervals-1 DO BEGIN ; combine over time
                IF jj EQ num_intervals-1 THEN wanted_i = where(converted_valid_epoch5 GE norm_times[jj], count) ELSE $
                  wanted_i = where(converted_valid_epoch5 GE norm_times[jj] AND converted_valid_epoch5 LT norm_times[jj+1], count)
                IF count EQ 1 THEN BEGIN 
                    norm_cycles_time                      [*,           jj] = $
                            norm_cycles                   [*,          wanted_i]
                    sw_pri_data_time                      [*, *, *,     jj] = $
                            sw_pri                        [*, *, *,    wanted_i]
                    wap_pri_ssd_data_time                 [*, *,       jj] = $
                            wap_pri_ssd                   [*, *,       wanted_i]
                    wap_pri_nossd_data_time               [*, *,       jj] = $
                            wap_pri_nossd                 [*, *,       wanted_i]
                    summed_swpha_pri_counts_time          [*, *, *, *, jj] = $
                            summed_swpha_pri_counts       [*, *, *, *, wanted_i]
                    summed_wappha_ssd_pri_counts_time     [*, *, *, *, jj] = $
                            summed_wappha_ssd_pri_counts  [*, *, *, *, wanted_i]
                    summed_wappha_nossd_pri_counts_time   [*, *, *, *, jj] = $
                            summed_wappha_nossd_pri_counts[*, *, *, *, wanted_i]
                    IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN $
                      summed_swpha_counts_time            [*, *, *, *, jj] = $
                              summed_swpha_counts         [*, *, *, *, wanted_i]
                    IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN $
                      summed_wappha_ssd_counts_time       [*, *, *, *, jj] = $
                              summed_wappha_ssd_counts    [*, *, *, *, wanted_i]
                    IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN $
                      summed_wappha_nossd_counts_time     [*, *, *, *, jj] = $
                              summed_wappha_nossd_counts  [*, *, *, *, wanted_i]
                    IF plots_wanted[3] EQ 1 THEN $
                      summed_supranoe_ssd_counts_time     [*, *, *, *, jj] = $
                              summed_supranoe_ssd_counts  [*, *, *, *, wanted_i]
                    IF plots_wanted[4] EQ 1 THEN $
                      summed_supranoe_nossd_counts_time   [*, *, *, *, jj] = $
                              summed_supranoe_nossd_counts[*, *, *, *, wanted_i]
                    IF plots_wanted[5] EQ 1 THEN $
                      summed_suprawide_counts_time        [*, *, *, *, jj] = $
                              summed_suprawide_counts     [*, *, *, *, wanted_i]
                    IF plots_wanted[6] EQ 1 THEN $
                      summed_swz_counts_time              [*, *, *, *, jj] = $
                              summed_swz_counts           [*, *, *, *, wanted_i]
                    IF plots_wanted[7] EQ 1 THEN $
                      summed_swhalpha_counts_time         [*, *, *, *, jj] = $
                              summed_swhalpha_counts      [*, *, *, *, wanted_i]
                ENDIF ELSE IF count GT 1 THEN BEGIN 
                    norm_cycles_time                      [*,           jj] = $
                      total(norm_cycles                   [*,          wanted_i], 2, /nan)
                    sw_pri_data_time                      [*, *, *,     jj] = $
                      total(sw_pri                        [*, *, *,    wanted_i], 4, /nan)
                    wap_pri_ssd_data_time                 [*, *,       jj] = $
                      total(wap_pri_ssd                   [*, *,       wanted_i], 3, /nan)
                    wap_pri_nossd_data_time               [*, *,       jj] = $
                      total(wap_pri_nossd                 [*, *,       wanted_i], 3, /nan)
                    summed_swpha_pri_counts_time          [*, *, *, *, jj] = $
                      total(summed_swpha_pri_counts       [*, *, *, *, wanted_i], 5, /nan)
                    summed_wappha_ssd_pri_counts_time     [*, *, *, *, jj] = $
                      total(summed_wappha_ssd_pri_counts  [*, *, *, *, wanted_i], 5, /nan)
                    summed_wappha_nossd_pri_counts_time   [*, *, *, *, jj] = $
                      total(summed_wappha_nossd_pri_counts[*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN $
                      summed_swpha_counts_time            [*, *, *, *, jj] = $
                        total(summed_swpha_counts         [*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN $
                      summed_wappha_ssd_counts_time       [*, *, *, *, jj] = $
                        total(summed_wappha_ssd_counts    [*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN $
                      summed_wappha_nossd_counts_time     [*, *, *, *, jj] = $
                        total(summed_wappha_nossd_counts  [*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[3] EQ 1 THEN $
                      summed_supranoe_ssd_counts_time     [*, *, *, *, jj] = $
                        total(summed_supranoe_ssd_counts  [*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[4] EQ 1 THEN $
                      summed_supranoe_nossd_counts_time   [*, *, *, *, jj] = $
                        total(summed_supranoe_nossd_counts[*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[5] EQ 1 THEN $
                      summed_suprawide_counts_time        [*, *, *, *, jj] = $
                        total(summed_suprawide_counts     [*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[6] EQ 1 THEN $
                      summed_swz_counts_time              [*, *, *, *, jj] = $
                        total(summed_swz_counts           [*, *, *, *, wanted_i], 5, /nan)
                    IF plots_wanted[7] EQ 1 THEN $
                      summed_swhalpha_counts_time         [*, *, *, *, jj] = $
                        total(summed_swhalpha_counts      [*, *, *, *, wanted_i], 5, /nan)
                ENDIF 
            ENDFOR 
        ENDIF  ELSE BEGIN
            num_intervals = n_elements(converted_valid_epoch5)
            norm_times = converted_valid_epoch5
            IF plots_wanted[0] EQ 1 OR files_wanted[2] EQ 1 THEN summed_swpha_counts_time        = summed_swpha_counts
            IF plots_wanted[1] EQ 1 OR files_wanted[2] EQ 1 THEN summed_wappha_ssd_counts_time   = summed_wappha_ssd_counts
            IF plots_wanted[2] EQ 1 OR files_wanted[2] EQ 1 THEN summed_wappha_nossd_counts_time = summed_wappha_nossd_counts
            IF plots_wanted[3] EQ 1 THEN summed_supranoe_ssd_counts_time   = summed_supranoe_ssd_counts
            IF plots_wanted[4] EQ 1 THEN summed_supranoe_nossd_counts_time = summed_supranoe_nossd_counts
            IF plots_wanted[5] EQ 1 THEN summed_suprawide_counts_time      = summed_suprawide_counts
            IF plots_wanted[6] EQ 1 THEN summed_swz_counts_time            = summed_swz_counts
            IF plots_wanted[7] EQ 1 THEN summed_swhalpha_counts_time       = summed_swhalpha_counts
            norm_cycles_time                    = norm_cycles
            sw_pri_data_time                    = sw_pri
            wap_pri_ssd_data_time               = wap_pri_ssd
            wap_pri_nossd_data_time             = wap_pri_nossd
            summed_swpha_pri_counts_time        = summed_swpha_pri_counts 
            summed_wappha_ssd_pri_counts_time   = summed_wappha_ssd_pri_counts
            summed_wappha_nossd_pri_counts_time = summed_wappha_nossd_pri_counts
       ENDELSE 
    ENDELSE 

        ; divide
    IF have_data EQ 1 THEN BEGIN 
        temp_file_start = plot_dir+'/norm_error_'
        temp_file_end = cdf_base_brief
        IF esa_range_format EQ 1 THEN temp_file_end = temp_file_end+esa_st
        temp_file_end = temp_file_end+'_'+suffix+'_'

        FOR ii = 0, 7 DO BEGIN
            IF plots_wanted[ii] EQ 1 OR (ii LE 2 AND files_wanted[2] EQ 1) THEN BEGIN
                CASE ii OF
                    0: BEGIN
                        IF when_sum_defl EQ 0 THEN data = fltarr(num_esa, num_bins[ii], num_intervals) ELSE $
                          data = fltarr(num_esa, 16, num_bins[ii], num_intervals)
                        filename = temp_file_start+'swpha_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom        = reform(summed_swpha_counts_time    [*, *, jj, kk, *]) ; (esa, defl, pri,    bin, time)
                                temp_denom = reform(summed_swpha_pri_counts_time[*, *, jj,  *, *]) ; (esa, defl, pri, buffer, time)
                                denom      = total (temp_denom, 3) ; sum over buffer
                                rate       = reform(sw_pri_data_time            [*, *, jj,     *]) ; (esa, defl, pri,         time)
                                IF when_sum_defl EQ 0 THEN BEGIN ; sum defl before normalization
                                    nom        = total(nom,        2, /nan)
                                    denom      = total(denom,      2, /nan)
                                    temp_denom = total(temp_denom, 2, /nan)
                                    rate       = total(rate,       2, /nan)
                                ENDIF 
                                normalize, nom, denom, rate, when_sum_defl, num_intervals, temp_values, num_esa
                                IF when_sum_defl EQ 0 THEN data[*, kk, *] = data[*, kk, *] + temp_values ELSE $
                                  data[*, *, kk, *] = data[*, *, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, when_sum_defl, temp_denom, rate, num_esa
                                ENDIF
                            ENDFOR 
                        ENDFOR 
                        swpha_norm = data
                    END
                    1: BEGIN
                        data     = fltarr(num_esa, num_bins[ii], num_intervals)
                        filename = temp_file_start+'wappha_ssd_'+temp_file_end 
                        FOR kk = 0, (num_bins[ii]/2)-1 DO BEGIN ; for each bin
                            bad_stats  = 0L
                            good_stats = 0L
                            FOR jj = 0, 3 DO BEGIN 
                                nom        = reform(summed_wappha_ssd_counts_time    [*, *,    jj, (kk*2), *]) 
                                nom        = total (nom,        2) ; sum over deflection
                                temp_denom = reform(summed_wappha_ssd_pri_counts_time[*, *, (jj/2),     *, *])
                                denom      = total (temp_denom, 3) ; sum over buffer
                                denom      = total (denom,      2) ; sum over deflection
                                temp_denom = total (temp_denom, 2)
                                rate       = reform(wap_pri_ssd_data_time[*, (jj/2), *])
                                normalize, nom, denom, rate, 0, num_intervals, temp_values, num_esa
                                data[*, (kk*2),   *] = data[*, (kk*2),   *] + temp_values
                                data[*, (kk*2)+1, *] = data[*, (kk*2)+1, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, 0, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        wappha_ssd_norm = data
                    END
                    2: BEGIN
                        data     = fltarr(num_esa, num_bins[ii], num_intervals)
                        filename = temp_file_start+'wappha_nossd_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom        = reform(summed_wappha_nossd_counts_time    [*, *,     jj, kk, *]) 
                                nom        = total (nom,        2) ; sum over deflection
                                temp_denom = reform(summed_wappha_nossd_pri_counts_time[*, *, (jj/2),  *, *])
                                denom      = total (temp_denom, 3) ; sum over buffer
                                denom      = total (denom,      2) ; sum over deflection
                                temp_denom = total (temp_denom, 2)
                                rate       = reform(wap_pri_nossd_data_time[*, (jj/2), *])
                                normalize, nom, denom, rate, 0, num_intervals, temp_values, num_esa
                                data[*, kk, *] = data[*, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, 0, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        wappha_nossd_norm = data
                    END
                    3: BEGIN
                        data     = fltarr(num_esa, num_bins[ii], num_intervals)
                        filename = temp_file_start+'supranoe_ssd_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom        = reform(summed_supranoe_ssd_counts_time  [*, *,     jj, kk, *]) 
                                nom        = total (nom,        2) ; sum over deflection
                                temp_denom = reform(summed_wappha_ssd_pri_counts_time[*, *, (jj/2),  *, *])
                                denom      = total (temp_denom, 3) ; sum over buffer
                                denom      = total (denom,      2) ; sum over deflection
                                temp_denom = total (temp_denom, 2)
                                rate       = reform(wap_pri_ssd_data_time[*, (jj/2), *])
                                normalize, nom, denom, rate, 0, num_intervals, temp_values, num_esa
                                data[*, kk, *] = data[*, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, 0, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        supranoe_ssd_norm = data
                    END
                    4: BEGIN
                        data     = fltarr(num_esa, num_bins[ii], num_intervals)
                        filename = temp_file_start+'supranoe_nossd_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom        = reform(summed_supranoe_nossd_counts_time  [*, *,     jj, kk, *]) 
                                nom        = total (nom,        2) ; sum over deflection
                                temp_denom = reform(summed_wappha_nossd_pri_counts_time[*, *, (jj/2),  *, *])
                                denom      = total (temp_denom, 3) ; sum over buffer
                                denom      = total (denom,      2) ; sum over deflection
                                temp_denom = total (temp_denom, 2)
                                rate       = reform(wap_pri_nossd_data_time[*, (jj/2), *])
                                normalize, nom, denom, rate, 0, num_intervals, temp_values, num_esa
                                data[*, kk, *] = data[*, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, 0, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        supranoe_nossd_norm = data
                    END
                    5: BEGIN
                        data     = fltarr(num_esa, num_bins[ii], num_intervals)
                        filename = temp_file_start+'suprawide_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom        = reform(summed_suprawide_counts_time     [*, *,     jj, kk, *]) 
                                nom        = total (nom,        2) ; sum over deflection
                                temp_denom = reform(summed_wappha_ssd_pri_counts_time[*, *, (jj/2),  *, *])
                                denom      = total (temp_denom, 3) ; sum over buffer
                                denom      = total (denom,      2) ; sum over deflection
                                temp_denom = total (temp_denom, 2)
                                rate       = reform(wap_pri_ssd_data_time[*, (jj/2), *])
                                normalize, nom, denom, rate, 0, num_intervals, temp_values, num_esa
                                data[*, kk, *] = data[*, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, 0, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        suprawide_norm = data
                    END
                    6: BEGIN
                        IF when_sum_defl EQ 0 THEN data = fltarr(num_esa, num_bins[ii], num_intervals) ELSE $
                          data = fltarr(num_esa, 16, num_bins[ii], num_intervals)
                        filename = temp_file_start+'swz_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom            = reform(summed_swz_counts_time      [*, *, jj, kk, *])
                                temp_denom     = reform(summed_swpha_pri_counts_time[*, *, jj,  *, *])
                                denom          = total (temp_denom, 3) ; sum over buffer
                                rate           = reform(sw_pri_data_time            [*, *, jj,     *])
                                IF when_sum_defl EQ 0 THEN BEGIN ; sum over defl before normalization
                                    nom        = total (nom,        2)
                                    denom      = total (denom,      2)
                                    temp_denom = total (temp_denom, 2)
                                    rate       = total (rate,       2)
                                ENDIF 
                                normalize, nom, denom, rate, when_sum_defl, num_intervals, temp_values, num_esa
                                IF when_sum_defl EQ 0 THEN data[*, kk, *] = data[*, kk, *] + temp_values ELSE $
                                  data[*, *, kk, *] = data[*, *, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, when_sum_defl, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        swz_norm = data
                    END
                    7: BEGIN
                        IF when_sum_defl EQ 0 THEN data = fltarr(num_esa, num_bins[ii], num_intervals) ELSE $
                          data = fltarr(num_esa, 16, num_bins[ii], num_intervals)
                        filename = temp_file_start+'swhalpha_'+temp_file_end 
                        FOR kk = 0, num_bins[ii]-1 DO BEGIN ; for each bin
                            FOR jj = 0, 3 DO BEGIN ; for each priority
                                nom            = reform(summed_swhalpha_counts_time [*, *, jj, kk, *])
                                temp_denom     = reform(summed_swpha_pri_counts_time[*, *, jj,  *, *])
                                denom          = total (temp_denom, 3) ; sum over buffer
                                rate           = reform(sw_pri_data_time            [*, *, jj,     *])
                                IF when_sum_defl EQ 0 THEN BEGIN ; sum deflection before normalization
                                    nom        = total (nom,        2)
                                    denom      = total (denom,      2)
                                    temp_denom = total (temp_denom, 2)
                                    rate       = total (rate,       2)
                                ENDIF 
                                normalize, nom, denom, rate, when_sum_defl, num_intervals, temp_values, num_esa
                                IF when_sum_defl EQ 0 THEN data[*, kk, *] = data[*, kk, *] + temp_values ELSE $
                                  data[*, *, kk, *] = data[*, *, kk, *] + temp_values
                                ; write error logs
                                IF kk EQ 0 THEN BEGIN 
                                    write_error_log, filename, jj, when_sum_defl, temp_denom, rate, num_esa
                                ENDIF 
                            ENDFOR 
                        ENDFOR 
                        swhalpha_norm = data
                    END
                ENDCASE
            ENDIF
        ENDFOR
    ENDIF 
ENDIF  
END 
