; Lorna Ellis
; 05/21/2007
; find_bins_pp.pro

; This is used by pha_play to find the bins for each category, given
; its limits

; in : limits        : array of limits -- 12 categories, max_num_bins (not all
;                      used), 5 (index, m/q min, m/q max, mass min, mass max)
;      num_bins      : array (size 12) with the number of bins for each
;                      category
;      mass_max      : constant for Mmax
;      mass_min      : constant for Mmin
;      mq_max        : constant for MQmax
;      mq_min        : constant for MQmin
;      n_max         : constant for NMAX
;      nq_max        : constant for NQMAX
;      d_nossd       : constant for distance between plates non ssd
;      d_ssd         : constant for distance between plates ssd
;      quad_style_boxes = 0 if old limits files (min and max mass and m/q), 1 if define 4 points for each box
; out: bins          : array of bins -- also 7 x max_num_bins x 4
;      mass_low_bound: array with low bounds for mass
;      mq_low_bound  : array with low bounds for m/q  (nonSSD and SSD)

; 08/04/2007 added -2 option for m/q
; 10/16/2007 Changed 5.8 to d_nossd and 64 to d_ssd^2
; 09/08/2008 Added quad_style_boxes.
; 01/18/2013 Fix quad_style_boxes.
; 05/21/2013 Add error messages for quad_style boxes.

PRO find_bins_pp, limits, num_bins, mass_max, mass_min, mq_max, mq_min, n_max, nq_max, d_nossd, d_ssd, quad_style_boxes, $
               bins, mass_low_bound, mq_low_bound
compile_opt strictarrsubs

max_num_bins = max(num_bins)
IF quad_style_boxes EQ 0 THEN BEGIN
    num_columns = 5
    start_mq    = 1
    stop_mq     = 2
    start_mass  = 3
    stop_mass   = 4
ENDIF ELSE BEGIN 
    num_columns = 9
    start_mq    = 1
    stop_mq     = 7
    start_mass  = 2
    stop_mass   = 8
ENDELSE 
bins = INTARR(12, max_num_bins, num_columns)

; Convert the mass-per-charge to a bin number
mq_low_bound = fltarr(2, 256)   ; nonSSD, SSD -- 256 bins
mq_low_bound[0, 0] = 0.0
mq_low_bound[1, 0] = 0.0
k_q = (mq_max/mq_min)^(1/nq_max)
FOR i = 1, nq_max DO BEGIN       ; set min bounds for each bin
    mq_low_bound[0, i] = mq_min*(k_q^(i-1))
    mq_low_bound[1, i] = mq_low_bound[0, i] * d_nossd * d_nossd / (d_ssd * d_ssd) ; 5.8 * 5.8 / 64 
ENDFOR 
mq_low_bound[0, 255] = 60
mq_low_bound[1, 255] = mq_low_bound[0, 255] * d_nossd * d_nossd / (d_ssd * d_ssd) ; 5.8 * 5.8 / 64 
; Convert mass to a bin number
mass_low_bound = fltarr(92)
mass_low_bound[0] = 0.0
k_m = (mass_max/mass_min)^(1/n_max)
for i = 1, n_max do begin
    mass_low_bound[i] = mass_min*(k_m^(i-1))
endfor
mass_low_bound[91] = 95.0

; Convert the bounds to bins for bin_dat_word
FOR i = 0, 11 DO BEGIN   ; 12 categories
    IF i MOD 2 EQ 0 THEN mq_index = 1 ELSE mq_index = 0 ; 1:SSD, 0:noSSD
    FOR j = 0, num_bins[i]-1 DO BEGIN 
        bins[i, j, 0] = limits[i, j, 0]    ; copy bin number
        FOR k = start_mq, stop_mq DO BEGIN ; m/q
            IF limits[i, j, k] EQ -1 THEN bins[i, j, k] = -1  $ ; -1 mean bin all
            ELSE IF limits[i, j, k] EQ -2 THEN bin =  0   $ ; -2 mean bin 0
                 ELSE BEGIN
                bin = 0
                WHILE (bin GT -1) AND (bin LT 255) DO BEGIN
                    IF limits[i, j, k] GE mq_low_bound[mq_index, bin] AND limits[i, j, k] LT mq_low_bound[mq_index, bin+1] $
                      THEN BEGIN
                        bins[i, j, k] = bin
                        bin = -1
                    ENDIF ELSE bin = bin + 1
                ENDWHILE
                IF bin EQ 255 THEN BEGIN
                    bins[i, j, k] = bin
                ENDIF
            ENDELSE 
            IF quad_style_boxes EQ 1 THEN k = k+1 ; want every other bin
        ENDFOR 

        FOR k = start_mass, stop_mass DO BEGIN 
            IF limits[i, j, k] EQ -1 THEN bin = -1  $ ; -1 means bin all
            ELSE IF limits[i, j, k] EQ -2 THEN bin =  0   $ ; -2 mean bin 0
                 ELSE  BEGIN 
                found = 0
                bin = 0
                WHILE found EQ 0 AND bin LT 91 DO BEGIN 
                    IF limits[i, j, k] GE mass_low_bound[bin] AND limits[i, j, k] LT mass_low_bound[bin+1] THEN BEGIN 
                        found = 1
                    ENDIF ELSE bin = bin + 1
                ENDWHILE 
                IF bin EQ 91 AND limits[i, j, k] GT mass_low_bound[91] $
                  then print, 'Error in find_bins: Mass out of bounds', limits[i, j, k], mass_low_bound[90]
            ENDELSE 
            bins[i, j, k] = bin
            IF quad_style_boxes EQ 1 THEN k = k+1 ; want every other bin
        ENDFOR
        
        IF quad_style_boxes EQ 0 THEN BEGIN 
            IF bins[i, j, 1] GT bins[i, j, 2] THEN BEGIN 
                print, "ERROR in find_bins: invalid M/Q bounds (bin numbers):", bins[i, j, 1], bins[i, j, 2]
                stop
            ENDIF
            IF bins[i, j, 3] GT bins[i, j, 4] THEN BEGIN
                print, "ERROR in find_bins: invalid mass bounds (bin numbers):", bins[i, j, 3], bins[i, j, 4]
                stop
            ENDIF 
        ENDIF ELSE BEGIN 
            IF bins[i, j, 1] GT bins[i, j, 3] THEN BEGIN 
                print, "ERROR in find_bins: invalid M/Q bounds (bin numbers):", bins[i, j, 1], bins[i, j, 3], bins[i, j, 5], bins[i, j, 7]
                stop
            ENDIF
            IF (bins[i, j, 4] GT bins[i, j, 2]) THEN BEGIN 
                print, "ERROR in find_bins: invalid mass bounds (bin numbers):", bins[i, j, 2], bins[i, j, 4], bins[i, j, 6], bins[i, j, 8]
                stop
            ENDIF
            IF ((bins[i, j, 6] GE bins[i, j, 2]) OR (bins[i, j, 8] GE bins[i, j, 2])) AND (bins[i, j, 2] GT 0) THEN BEGIN 
                print, "ERROR in find_bins: invalid mass bounds (bin numbers):", bins[i, j, 2], bins[i, j, 4], bins[i, j, 6], bins[i, j, 8]
                stop
            ENDIF
            IF ((bins[i, j, 6] GT bins[i, j, 2]) OR (bins[i, j, 8] GT bins[i, j, 2])) THEN BEGIN 
                print, "ERROR in find_bins: invalid mass bounds (bin numbers):", bins[i, j, 2], bins[i, j, 4], bins[i, j, 6], bins[i, j, 8]
                stop
            ENDIF
            IF (bins[i, j, 4] EQ bins[i, j, 2]) AND (bins[i, j, 1] GT bins[i, j, 3]) AND (bins[i, j, 1] NE -1) THEN BEGIN 
                print, "ERROR in find_bins: invalid mass bounds (bin numbers):", bins[i, j, 2], bins[i, j, 4], bins[i, j, 6], bins[i, j, 8]
                stop
            ENDIF
            IF bins[i, j, 1] EQ bins[i, j, 7] AND bins[i, j, 2] LT bins[i, j, 8] THEN BEGIN ; if square, check first is upper
                print, "ERROR in find_bins: invalid bounds (bin numbers):", bins[i, j, 1], bins[i, j, 7], bins[i, j, 2], bins[i, j, 8]
                stop
            ENDIF
        ENDELSE 
    ENDFOR
ENDFOR 

END 
