; Lorna Ellis
; 03/25/2010
; calculate_mass_mq_kiel.pro

; This program takes an array of PHA words and calculates mass, mass per
; charge and e/q. It is intended as part of the pha_play program. 

; This version uses the "Kiel" algorithm from Lars Berger and Christian Drews.

; Input: pha           : 16 element array [swpe, swpd, quad, ssd_id, ssde, tof, position, section, priority, 
;                                          new bin, nm_orig, nq_orig, nm_new, nq_new, which_products, schan]
;        want_bins     : flag for if want bin number returned
;        c2            : constant from file
;        d_nossd       : constant from file
;        d_ssd         : constant from file
;        d1            : constant from file
;        mass_max      : constant from file
;        mass_min      : constant from file
;        t_max         : constant from file
;        mass_low_bound: from find_bins [92] -- only needed if want bins
;        mq_low_bound  : from find_bins [2,256] -- only needed if want bins
;
; Output: swpe    : from pha
;         swpd    : from pha
;         quadrant: from pha
;         position: from pha
;         section : from pha
;         priority: from pha
;         half    : based on section
;         mass    : mass in keV -- returns NaN when ssde_un eq 0
;         nm      : mass bin
;         m_per_q : mass per charge
;         nq      : mass per charge bin
;         e_per_q : energy per charge in keV

;  03/12/2014 Removed arguments that aren't used.


PRO calculate_mass_mq_kiel, pha, want_bins, c2, d_nossd, d_ssd, d1, mass_max, mass_min, t_max, mass_low_bound, mq_low_bound, swpe, swpd, quadrant, position, section, priority, half, mass, nm, m_per_q, nq, e_per_q
compile_opt strictarrsubs

d2  = 1.048954246
pac = 19.7

line = ''
infile = 'MtableSTA.dat'
openr, in_lun, infile, /get_lun
kiel_mass_table = fltarr(1024, 1024) ; ssde_un, tof
FOR ii = 0, 5 DO BEGIN          ; read header
    readf, in_lun, line
ENDFOR
FOR ssde_i = 0, 1023 DO BEGIN 
    readf, in_lun, line
    parts = float(strsplit(line, /extract))
    kiel_mass_table[*, ssde_i] = parts
ENDFOR

temp = size(pha, /dimensions)
IF n_elements(pha) GT 16 THEN num_pha = temp[1] ELSE num_pha = 1
swpe     = reform(long(pha[0, *]))         ; convert to a long
swpd     = reform(long(pha[1, *]))

temp_i = where(swpe MOD 2 EQ 1, count)
IF count GT 0 THEN swpd[temp_i] = 31-swpd[temp_i] ; reverse deflection on odd esas
quadrant = reform(long(pha[2, *]))         ; no-SSD half has PHA quadrants 3 & 4
ssd_id   = reform(long(pha[3, *]))
ssde_un  = reform(long(pha[4, *]))
tof      = reform(long(pha[5, *]))
position = reform(long(pha[6, *]))
section  = reform(long(pha[7, *]))
priority = reform(long(pha[8, *]))
yes_i = where(section EQ 3, yes_count, complement = no_i, ncomplement = no_count)
half = lonarr(num_pha)
IF yes_count GT 0 THEN half[yes_i] = 1L
IF no_count GT 0 THEN half[no_i] = 0L

; Calculate NQ table
; Convert time channels (Tch) to nanoseconds (Tns)
tns = dblarr(num_pha)
tns = tof * 0.2372 + (0.00574698 * (swpe^1.47412)) + 5.5
; Convert E/Q steps to keV
s = 128 - swpe  ; default used 127
e_per_q =  d1 * d2^s
; Calculate Mass per charge for each T -- E/Q pair
c1 = (4.38E7)^2/(1E9)^2/(d_nossd)^2
m_per_q = c1 * (pac + e_per_q - c2) * (tns)^2
half0_i = where(half EQ 0, half0_count, complement = half1_i, ncomplement = half1_count)
IF half0_count GT 0 THEN m_per_q[half0_i] = m_per_q[half0_i] * (d_nossd)^2 / (d_ssd)^2
; get bin 
IF want_bins EQ 1 THEN BEGIN 
    nq = intarr(num_pha)
    nq[*] = 255 ; if not found elsewhere, it will be here
    IF half0_count GT 0 THEN BEGIN
        ;mq_index[half0_i] = 1
        temp_m_q = m_per_q[half0_i]
        temp_nq = nq[half0_i]
        FOR ii = 0, 254 DO BEGIN ; 256 limit bins -- this loop is small compared with num_pha
            temp_i = where(mq_low_bound[1, ii] LE temp_m_q AND mq_low_bound[1, ii+1] GT temp_m_q, temp_count)
            IF temp_count GT 0 THEN temp_nq[temp_i] = ii
        ENDFOR 
        nq[half0_i] = temp_nq
    ENDIF 
    IF half1_count GT 0 THEN BEGIN
        ;mq_index[half1_i] = 0
        temp_m_q = m_per_q[half1_i]
        temp_nq = nq[half1_i]
        FOR ii = 0, 254 DO BEGIN ; 256 limit bins -- this loop is small compared with num_pha
            temp_i = where(mq_low_bound[0, ii] LE temp_m_q AND mq_low_bound[0, ii+1] GT temp_m_q, temp_count)
            IF temp_count GT 0 THEN temp_nq[temp_i] = ii
        ENDFOR 
        nq[half1_i] = temp_nq
    ENDIF 
ENDIF 
; Calculate NM 
; use table
mass = kiel_mass_table[ssde_un, tof]
; Convert to a bin number
IF want_bins EQ 1 THEN BEGIN 
    nm = intarr(num_pha)
    nm[*] = 91
    FOR ii = 0, 90 DO BEGIN    ; 92 limit bins -- this loop is small compared with num_pha
        temp_i = where(mass_low_bound[ii] LE mass AND mass_low_bound[ii+1] GT mass, temp_count)
        IF temp_count GT 0 THEN nm[temp_i] = ii
    ENDFOR 
    ; now overwrite with error cases
    temp_i = where(mass LE mass_min, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 92
    temp_i = where(mass GE mass_max, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 91
    temp_i = where(tns GE t_max, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 95
    temp_i = where(finite(tns) NE 1, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 94
    temp_i = where(ssde_un EQ 0, temp_count)
    IF temp_count GT 0 THEN BEGIN
        nm  [temp_i] = 0
        mass[temp_i] = 0
    ENDIF 
ENDIF 

END 
