; calculate_mass_mq_abg_new.pro

; This program takes a PLASTIC PHA word and calculates mass, mass per
; charge and e/q. It is intended as part of the pha_play program.


;  Modification History
;  Authors Lorna Ellis - modified by ABGalvin to use specialized algorithms
;  03/25/2007: LBE: Original version
;  Spring 2008:  ABG version
;  10/21/2008: LBE: Changed to "new" to deal with whole array of values instead of one at a time
;  11/07/2008: LBE: Changed c2 to c2_abg because changed in this procedure. 
;  01/23/2012: LBE: Fix where c2_abg should be c2_abg[temp_i]
;  05/21/2012: ABG: Included e/q tables used in FeQ routine processing and checks on energy etc
;  05/22/2012: ABG: Now have input files for STA E/Q instead of array lists
;  05/27/2012: ABG: added which algorithm to the routine call, used for setting restrictions
;  11/01/2012: LBE: Get proton bulk speed and thermal speed.
;  05/16/2013: LBE: Changed checkon size to 16 from 15 (because added schan to valid_pha)

; Input: pha           : 9 element array 
;                        [swpe, swpd, quad, ssd_id, ssde, tof, position, section, priority]
;        want_bins     : flag for if want NM, NQ bin numbers returned
;        a1            : constant from classconstants file 
;        a2            : constant from classconstants file 
;        a3            : constant from classconstants file 
;        a4            : constant from classconstants file 
;        a5            : constant from classconstants file 
;        a6            : constant from classconstants file 
;        a_e           : constant from classconstants file 
;        a_t           : constant from classconstants file           
;        b_e           : constant from classconstants file 
;        b_t           : constant from classconstants file 
;        c2            : constant from classconstants file 
;        d_nossd       : constant from classconstants file 
;        d_ssd         : constant from classconstants file 
;        d1            : constant from classconstants file 
;        d2            : constant from classconstants file 
;        e_max         : constant from classconstants file 
;        e_min         : constant from classconstants file 
;        mass_max      : constant from classconstants file 
;        mass_min      : constant from classconstants file 
;        t_max         : constant from classconstants file 
;        t_min         : constant from classconstants file 
;        pac           : constant from settings file
;        mass_low_bound: from find_bins [92] -- only needed if want bins
;        mq_low_bound  : from find_bins [2,256] -- only needed if want bins
;        sat           : 'A' or 'B'
;        want_bins     : set to 1 in call from fill_bins_new
;
; Internally defined:  alternative time and energy conversions from those used onboard
;         tslope        
;         tintercept 
;         eslope
;         eintercept
;
; Output: swpe    : from pha
;         swpd    : from pha
;         quadrant: from pha
;         position: from pha
;         section : from pha
;         priority: from pha
;         half    : based on section (defines TOF distance)
;         mass    : mass in amu -- returns NaN or zero when ssde_un eq 0
;         nm      : logarithmic mass bin
;         m_per_q : mass per charge
;         nq      : logarithmic mass per charge bin
;         e_per_q : energy per charge in keV
;         e_meas  : energy [keV]
;         tns     : time [ns]

PRO calculate_mass_mq_abg_new, which_algorithm, which_section, pha, want_bins, $
                 a1, a2, a3, a4, a5, a6, a_e, a_t, b_e, b_t, c2, $
                 d_nossd, d_ssd, d1, d2, e_max, e_min, mass_max, mass_min, t_max, t_min, pac, $
                 mass_low_bound, mq_low_bound, swpe, swpd, quadrant, position, $
                 section, priority, half, mass, nm, m_per_q, nq, e_per_q, e_meas, tns
compile_opt strictarrsubs   ;lbe
COMMON com_pha_play

want_proton_speed = 1 

IF want_proton_speed EQ 1 THEN BEGIN
    pha_play_read_proton_data, proton_from_valid, pr_error, pr_caution, pr_speed, pr_therm
ENDIF 

cdf_base_brief = strmid(cdf_base, 0, 4)+strmid(cdf_base, 11, 8)

; INPUTS BY HAND
routine_check = 1           ; if wanting e vs t plots, set to 0
speed_min     = 220.        ; min speed in routine check
speed_max     = 900.        ; max speed in routine check
print, ' Have entered abg_new, routine processing check = ', $
         routine_check, 'speed range ', speed_min, speed_max

check = 1
print, ' Have entered abg_new, consistency checks = ', check
; check = 0 means do not clean up pos0 for STA/B nor make consistency check on SSD id and RAs
; check = 1 means make pos0 clean on STA/B
; check = 2 means make pos0 clean up if in STA/B, and make SSD id vs RA checks for either s/c


speed_known = 0
speed_solar_wind = 650
print, ' Have entered abg_new, solar wind speed known = ', speed_known
IF speed_known GT 0 THEN print, ' speed = ', speed_solar_wind
; if sw speed is known, then more stringent checks are possible for Alg 4

temp = size(pha, /dimensions)
IF n_elements(pha) GT 16 THEN num_pha = temp[1] ELSE num_pha = 1

;;;;;;;;;;;;;;
; PHA WORD
;;;;;;;;;;;;;;
swpe     = reform(long(pha[0, *]))               ; ESA Step, convert to a long
swpd     = reform(long(pha[1, *]))               ; DEFL Step, convert to a long
temp_i = where(swpe MOD 2 EQ 1, count)           ; reverse deflection on odd esas
IF count GT 0 THEN swpd[temp_i] = 31-swpd[temp_i]
quadrant = reform(long(pha[2, *]))               ; no-SSD half has PHA quadrants 2 & 3
ssd_id   = reform(long(pha[3, *]))               ; ssd pixels 0-15
ssde_un  = reform(long(pha[4, *]))               ; ssd measured energy in channel number (10bit)
tof      = reform(long(pha[5, *]))               ; time of flight in channel number
position = reform(long(pha[6, *]))               ; RA or POS bin
section  = reform(long(pha[7, *]))               ; 0=main, 1=schn, 2=wapssd, 3=wapnossd
priority = reform(long(pha[8, *]))               ; pha range or priority

; if section 3, different TOF-distance and different TAC Board
yes_i = where(section EQ 3, yes_count, complement = no_i, ncomplement = no_count)
half = lonarr(num_pha)
IF yes_count GT 0 THEN half[yes_i] = 1L    ; TAC2
IF no_count GT 0 THEN half[no_i] = 0L      ; TAC0

; set values for time_of_flight distance
tof_dist = fltarr(num_pha)
tof_dist[*] = 8.0
temp_i = where(half EQ 1, temp_count) ; check for non-ssd side
IF temp_count GT 0 THEN tof_dist[temp_i] = 5.80

pacHK = 20.0  ; STB
IF sat EQ 'A' THEN pacHK = 19.95

;;;;;;;;;;;;;;;;;;;
; TIME CONVERSION and TOF Chamber speed
;;;;;;;;;;;;;;;;;
; Convert time channels (Tch) to nanoseconds (Tns) 

; initialize arrays
tslope     = fltarr(num_pha)
tintercept = fltarr(num_pha)
tns        = fltarr(num_pha)
speed_tof  = fltarr(num_pha)  			; speed of ion in chamber after post acceleration, derived
										; from tof
;default values  
tslope(*) = 0.25
tintercept(*) = 0.
tns(*) = 0.
speed_tof (*) = -1.

IF sat EQ 'A' THEN BEGIN               ; STA
    IF yes_count GT 0 THEN BEGIN       ; TAC2
        tslope[yes_i] = 0.250          ; from galvin/gustafson for TAC2 (placeholder)
        tintercept[yes_i] = 4.7
    ENDIF 
    IF no_count GT 0 THEN BEGIN        ; TAC0 
        tslope[no_i] = 0.250                
;       tintercept[no_i] = 3.5         ; from MAP for TAC0
;       tintercept[no_i] = 4.8		   ; from Check for O+6 ;abg
;       tintercept[no_i] = 5.283       ; from check for O+6 ssd 5&6 - abg
;       tintercept[no_i] = 6.0         ; average from scattered down protons
        tintercept[no_i] = 5.6         ; proton, helium mq
    ENDIF 

ENDIF ELSE BEGIN                      ; STB
    IF yes_count GT 0 THEN BEGIN      ; TAC2
        tslope[yes_i] = 0.240         ; from galvin/gustafson for TAC2 (placeholder)
        tintercept[yes_i] = 3.5
    ENDIF
    IF no_count GT 0 THEN BEGIN       ; TAC0
        tslope[no_i] = 0.250          ; from MAP for TAC0
;       tintercept[no_i] = 2.34       ; from MAP
        tintercept[no_i] = 4.7        ; from scattered down protons in Schn
    ENDIF 
ENDELSE 

tns  =  tof * tslope + tintercept        ; convert TOF channel to time (ns)
 
 ;clean up
temp_i = where(tof LT 0, temp_count)
IF temp_count GT 0 THEN print, ' negative tof ', tof(temp_i) ; should not happen
temp_i = where(tof LE 0, temp_count)
IF temp_count GT 0 THEN tns[temp_i] = 0  ; zero TOF means no time measured
temp_i = where(tns LT 0, temp_count)     ; conversion should not go negative
IF temp_count GT 0 then tns[temp_i] = 0
IF temp_count GT 0 then print, ' negative time calculated'; neg tof means no time measured

; calculate fastest possible chamber speed based on tn and straight flight tof chamber distance
   temp_i = where(tns GT 10, temp_count)                           ; tof_dist depends on ssd/nossd
   IF temp_count GT 0 then speed_tof = (tof_dist*1E-5)/(tns*1E-9) ; speed [kps] in TOF chamber


;;;;;;;;;;;;;;;;;;;;
;  E/Q CONVERSION
;;;;;;;;;;;;;;;;;;;;
; Convert ESA step number to keV/e

ESA_AS_BenchModTK = fltarr(128)            ; PLASTIC A Schannel E/Q array
ESA_AM_BenchModTK = fltarr(128)            ; PLASTIC A Main Channel E/Q array
ESA_BM_ioncal0    = fltarr(128)            ; PLASTIC B first attempt array
ESA_BS_kds        = fltarr(128)
e_per_q           = fltarr(num_pha)        ; energy per charge for each pha event (added 10.9.12)

; read the e/q values for each esa step
line = " "
tab = string(9B)
esa_file_M = 'STA_MEQ_Tables_43_8240.txt'
esa_file_S = 'STA_SEQ_Tables_431_8683.txt'     ; changed to revised table in August 2012
esa_file_B = 'STB_esa_steps_ioncal0.txt'

openr, esa_M_lun, esa_file_M, /GET_LUN
readf, esa_M_lun, line ; read 1 header lines
readf, esa_M_lun, line ; read 1 header lines
readf, esa_M_lun, line ; read 1 header lines
FOR ii = 0, 127 DO BEGIN ; 128 ESA steps
    readf, esa_M_lun, line
    parts = strsplit(line, /extract, count = count)
    IF count EQ 5 THEN BEGIN 								; expect five columns
        IF parts[0] NE ii THEN BEGIN
            print, 'Problem in reading ESA_Main file'
            stop
        ENDIF ELSE ESA_AM_BenchModTK[ii] = float(parts[3]) 	; choose bench_mod
    ENDIF ELSE IF count NE 0 THEN BEGIN
        print, 'Invalid line in file: '+esa_file_M
        stop
    ENDIF
ENDFOR 
close, esa_M_lun
free_lun, esa_M_lun

line = " "
openr, esa_S_lun, esa_file_S, /GET_LUN
readf, esa_S_lun, line ; read 1 header lines
readf, esa_S_lun, line ; read 1 header lines
readf, esa_S_lun, line ; read 1 header lines
FOR ii = 0, 127 DO BEGIN ; 128 ESA steps
    readf, esa_S_lun, line
    parts = strsplit(line, /extract, count = count)
    IF count EQ 3 THEN BEGIN 								; expect three columns
        IF parts[0] NE ii THEN BEGIN
            print, 'Problem in reading ESA_Schannel file'
            stop
        ENDIF ELSE ESA_AS_BenchModTK[ii] = float(parts[2]) 	; choose bench_mod
    ENDIF ELSE IF count NE 0 THEN BEGIN
        print, 'Invalid line in file: '+esa_file_S
        stop
    ENDIF
ENDFOR 
close, esa_S_lun
free_lun, esa_S_lun

line = " "
openr, esa_B_lun, esa_file_B, /GET_LUN
readf, esa_B_lun, line ; read 1 header lines
FOR ii = 0, 127 DO BEGIN ; 128 ESA steps
    readf, esa_B_lun, line
    parts = strsplit(line, /extract, count = count)
    IF count EQ 3 THEN BEGIN 								; expect three columns
        IF parts[0] NE ii THEN BEGIN
            print, 'Problem in reading ESA_B file'
            stop
        ENDIF ELSE BEGIN
            ESA_BM_ioncal0[ii] = float(parts[1]) 	; choose ion cal
            ESA_BS_kds[ii] = float(parts[2])        ; choose kds for 63-127
        ENDELSE
    ENDIF ELSE IF count NE 0 THEN BEGIN
        print, 'Invalid line in file: '+esa_file_B
        stop
    ENDIF
ENDFOR 
close, esa_B_lun
free_lun, esa_B_lun


; NOTE:  we need to know schannel switch to find what ESA to use for sections 2 and 3
;        ...future task

IF sat EQ 'A' THEN BEGIN                 
   e_per_q = ESA_AM_BenchModTK(swpe)		; Main Channel E/Qs
   temp_i = where(section EQ 1, temp_count) ; Small Channel
   IF temp_count GT 0 THEN e_per_q[temp_i] = ESA_AS_BenchModTK(swpe[temp_i])  ; Small Chn E/Qs

ENDIF ELSE IF sat EQ 'B' THEN BEGIN
   e_per_q = ESA_BM_ioncal0(swpe)		; Main Channel E/Qs
   temp_i = where(section EQ 1, temp_count) ; Small Channel
   IF temp_count GT 0 THEN e_per_q[temp_i] = ESA_BS_kds(swpe[temp_i])  ; Small Chn E/Qs
   
 ENDIF
 ;clean up
temp_i = where(swpe LT 0 OR swpe GT 127, temp_count)
IF temp_count GT 0 THEN print, ' swpe out of range ', swpe(temp_i) ; should not happen


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; MEASURED ENERGY CONVERSION
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Convert uncompressed (10bit) energy channel to keV
; ssd_offset are commanded
; slope and intercepts are from source calibrations

; initialize arrays
ssd_offset = fltarr(num_pha)
eslope = fltarr(num_pha)
eintercept = fltarr(num_pha)

IF sat EQ 'A' THEN BEGIN          ; STA
    temp_i = where(ssd_id EQ 0, temp_count) ; ssd pixel 0
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 1
        eslope[temp_i] = 2.276
        eintercept[temp_i] = -0.838
    ENDIF  
    temp_i = where(ssd_id EQ 1, temp_count) ; ssd pixel 1
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 5
        eslope[temp_i] = 2.302
        eintercept[temp_i] = 8.09
    ENDIF 
    temp_i = where(ssd_id EQ 2, temp_count) ; ssd pixel 2
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 2
        eslope[temp_i] = 2.291
        eintercept[temp_i] = 1.429
    ENDIF 
    temp_i = where(ssd_id EQ 3, temp_count) ; ssd pixel 3
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 3
        eslope[temp_i] = 2.263
        eintercept[temp_i] = 3.738
    ENDIF 
    temp_i = where(ssd_id EQ 4, temp_count) ; ssd pixel 4
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 5
        eslope[temp_i] = 2.235
        eintercept[temp_i] = 7.495
    ENDIF 
    temp_i = where(ssd_id EQ 5, temp_count) ; ssd pixel 5
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 7
        eslope[temp_i] = 2.227
        eintercept[temp_i] = 12.909
    ENDIF 
    temp_i = where(ssd_id EQ 6, temp_count) ; ssd pixel 6
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.225
        eintercept[temp_i] = 5.283
    ENDIF 
    temp_i = where(ssd_id EQ 7, temp_count) ; ssd pixel 7
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 3
        eslope[temp_i] = 2.214
        eintercept[temp_i] = 4.03
    ENDIF 
    temp_i = where(ssd_id EQ 8, temp_count) ; ssd pixel 8
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 8
        eslope[temp_i] = 2.211
        eintercept[temp_i] = 14.252
    ENDIF 
    temp_i = where(ssd_id EQ 9, temp_count) ; ssd pixel 9
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.16
        eintercept[temp_i] = 6.74
    ENDIF 
    temp_i = where(ssd_id EQ 10, temp_count) ; ssd pixel 10
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 2
        eslope[temp_i] = 2.219
        eintercept[temp_i] = 0.977
    ENDIF 
    temp_i = where(ssd_id EQ 11, temp_count) ; ssd pixel 11
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 6
        eslope[temp_i] = 2.22
        eintercept[temp_i] = 10.795
    ENDIF 
    temp_i = where(ssd_id EQ 12, temp_count) ; ssd pixel 12
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.223
        eintercept[temp_i] = 5.155
    ENDIF 
    temp_i = where(ssd_id EQ 13, temp_count) ; ssd pixel 13
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.223
        eintercept[temp_i] = 5.014
    ENDIF 
    temp_i = where(ssd_id EQ 14, temp_count) ; ssd pixel 14
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 0
        eslope[temp_i] = 2.269
        eintercept[temp_i] = -3.128
    ENDIF 
    temp_i = where(ssd_id EQ 15, temp_count) ; ssd pixel 15
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 0
        eslope[temp_i] = 2.357
        eintercept[temp_i] = -2.912
    ENDIF
    
ENDIF ELSE BEGIN                  ; STB
    temp_i = where(ssd_id EQ 0, temp_count) ; ssd pixel 0
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 2
        eslope[temp_i] = 2.16
        eintercept[temp_i] = -21.06
    ENDIF 
    temp_i = where(ssd_id EQ 1, temp_count) ; ssd pixel 1
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 1
        eslope[temp_i] = 2.13
        eintercept[temp_i] = -23.2
    ENDIF 
    temp_i = where(ssd_id EQ 2, temp_count) ; ssd pixel 2
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 0
        eslope[temp_i] = 2.24
        eintercept[temp_i] = -26.29
    ENDIF 
    temp_i = where(ssd_id EQ 3, temp_count) ; ssd pixel 3
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 0
        eslope[temp_i] = 2.09
        eintercept[temp_i] = -24.
    ENDIF 
    temp_i = where(ssd_id EQ 4, temp_count) ; ssd pixel 4
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 2
        eslope[temp_i] = 2.19
        eintercept[temp_i] = -20.27
    ENDIF 
    temp_i = where(ssd_id EQ 5, temp_count) ; ssd pixel 5
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.16
        eintercept[temp_i] = -17.49
    ENDIF 
    temp_i = where(ssd_id EQ 6, temp_count) ; ssd pixel 6
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 2
        eslope[temp_i] = 2.13
        eintercept[temp_i] = -20.46
    ENDIF 
    temp_i = where(ssd_id EQ 7, temp_count) ; ssd pixel 7
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 2
        eslope[temp_i] = 2.14
        eintercept[temp_i] = -19.95
    ENDIF 
    temp_i = where(ssd_id EQ 8, temp_count) ; ssd pixel 8
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 7
        eslope[temp_i] = 2.09
        eintercept[temp_i] = -10.66
    ENDIF 
    temp_i = where(ssd_id EQ 9, temp_count) ; ssd pixel 9
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 6
        eslope[temp_i] = 2.1
        eintercept[temp_i] = -12.87
    ENDIF 
    temp_i = where(ssd_id EQ 10, temp_count) ; ssd pixel 10
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.01
        eintercept[temp_i] = -14.73
    ENDIF 
    temp_i = where(ssd_id EQ 11, temp_count) ; ssd pixel 11
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 3
        eslope[temp_i] = 2.11
        eintercept[temp_i] = -19.02
    ENDIF 
    temp_i = where(ssd_id EQ 12, temp_count) ; ssd pixel 12
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 1
        eslope[temp_i] = 2.18
        eintercept[temp_i] = -23.17
    ENDIF 
    temp_i = where(ssd_id EQ 13, temp_count) ; ssd pixel 13
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 5
        eslope[temp_i] = 2.09
        eintercept[temp_i] = -15.13
    ENDIF 
    temp_i = where(ssd_id EQ 14, temp_count) ; ssd pixel 14
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 4
        eslope[temp_i] = 2.24
        eintercept[temp_i] = -18.14
    ENDIF 
    temp_i = where(ssd_id EQ 15, temp_count) ; ssd pixel 15
    IF temp_count GT 0 THEN BEGIN 
        ssd_offset[temp_i] = 0
        eslope[temp_i] = 2.24
        eintercept[temp_i] = -25.56
    ENDIF 
ENDELSE 

 e_meas = fltarr(num_pha)                      ; initialize array
 temp_i = where(ssde_un GT 0, temp_count)
 IF temp_count GT 0 THEN $
    e_meas[temp_i] = (ssde_un[temp_i] - ssd_offset[temp_i])*eslope[temp_i] + eintercept[temp_i]

;clean up
 temp_i = where(ssde_un EQ 0, temp_count)
 IF temp_count GT 0 THEN e_meas[temp_i] = 0  ; zero ssde_un means no energy measured
; check for negative energy
 temp_i = where(e_meas LT 0, temp_count)
 IF temp_count GT 0 THEN e_meas[temp_i] = 0  
 IF temp_count GT 0 THEN print, 'negative energy calculated'



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; FIRST ITERATION IN DETERMING MASS-PER-CHARGE
; USES NOMINAL CARBON FOIL CORRECTION OF 1.5
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Calculate Mass per charge default value

; initialize arrays
m_per_q_init1 			= fltarr(num_pha)
m_per_q_init2 			= fltarr(num_pha)
m_per_q       			= fltarr(num_pha)
c2_abg        			= fltarr(num_pha)

m_per_q_init1[*] = 0.
m_per_q_init2[*] = 0.
m_per_q[*] = 0.

c1 = 2.99936E-05*(8.0/tof_dist)^2
c2_abg[*] = 1.5         ; first guess for carbon foil correction

;calculate first estimate for m/q
m_per_q_init1 = c1 * (pacHK + e_per_q - c2_abg) * (tns)^2



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; DETERMING MASS - initial estimate
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; initialize arrays
  mass_init1 = fltarr(num_pha)
  mass_init2 = fltarr(num_pha)
  mass = fltarr(num_pha)
  x = fltarr(num_pha)
  y = fltarr(num_pha)
 
 ;default values
  mass_init1[*] = 0.
  mass_init2[*] = -1
  mass[*]       = 0.


; Arrays for second mass iteration
  Rnuc = fltarr(15)
  lnE = fltarr(15, num_pha)
  Rnuc=[0.7, 1.00 , 2.0, 2.5, 3.5, 5.5, 10., 12., 16.0 , 20.02 , 28 , 40, 56, 80, 95]
  jflagmax = 15


;valid range
  temp_i = where((e_meas LE 2000) AND (e_meas GT 15) AND (tns GT 10), temp_count)
  IF temp_count GT 0 THEN BEGIN  ; mass calculation loop1
    x[temp_i] = alog(e_meas[temp_i]) 
    y[temp_i] = alog(tns[temp_i])

     
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; FIRST MASS ITERATION - Default coefficients
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
    mass_init1[temp_i] = EXP(a1 + a2*x[temp_i] + a3*y[temp_i] + a4*x[temp_i]*y[temp_i] + $
                         a5*x[temp_i]^2 + a6*y[temp_i]^3)
    mass[temp_i]       = mass_init1[temp_i]
                                                           
ENDIF                           ; big mass calculation loop1

 
;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
;  Creating some species tracks in E vs ESA space
;                           and in T vs ESA space
;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
 
   mqHe_lowbound     = fltarr(num_pha)
   mqHe_lowbound (*) = 100.
   mqHe_lowbound     = mass * 0.41961 + 0.91302
   
   
; the following are a function of esa step, not associated with a particular pha event   
   emeas_Proton_expect          = fltarr(num_pha)    
   emeas_Proton_expect_lowEQ    = fltarr(num_pha)    ;account for 7% bandwidth in esa
   emeas_Proton_expect_highEQ   = fltarr(num_pha)    ;account for 7% bandwidth in esa
   emeas_Proton_upper           = fltarr(num_pha)
   emeas_Proton_lower           = fltarr(num_pha)
   emeas_Proton_pileup2         = fltarr(num_pha)

   emeas_HePlus_expect          = fltarr(num_pha)
   emeas_HePlus_expect_lowEQ    = fltarr(num_pha)
   emeas_HePlus_expect_highEQ   = fltarr(num_pha)
   emeas_HePlus_upper           = fltarr(num_pha)
   emeas_HePlus_lower           = fltarr(num_pha)

   emeas_HePlus2_expect         = fltarr(num_pha)
   emeas_HePlus2_expect_lowEQ   = fltarr(num_pha)
   emeas_HePlus2_expect_highEQ  = fltarr(num_pha)
   emeas_HePlus2_upper          = fltarr(num_pha)
   emeas_HePlus2_lower          = fltarr(num_pha)
   emeas_HePlus2_pileup2        = fltarr(num_pha) ; enery pileup by double impacts

   emeas_CPlus1_expect          = fltarr(num_pha)
   emeas_CPlus1_expect_lowEQ    = fltarr(num_pha)
   emeas_CPlus1_expect_highEQ   = fltarr(num_pha)
   emeas_CPlus1_upper           = fltarr(num_pha)
   emeas_CPlus1_lower           = fltarr(num_pha)
 
   emeas_OPlus6_expect          = fltarr(num_pha)
   emeas_OPlus6_expect_lowEQ    = fltarr(num_pha)
   emeas_OPlus6_expect_highEQ   = fltarr(num_pha)
   emeas_OPlus6_upper           = fltarr(num_pha)
   emeas_OPlus6_lower           = fltarr(num_pha)
  
   emeas_OPlus1_expect          = fltarr(num_pha)
   emeas_OPlus1_expect_lowEQ    = fltarr(num_pha)
   emeas_OPlus1_expect_highEQ   = fltarr(num_pha)
   emeas_OPlus1_upper           = fltarr(num_pha)
   emeas_OPlus1_lower           = fltarr(num_pha)

   emeas_OPlus2_expect          = fltarr(num_pha)
   emeas_OPlus2_expect_lowEQ    = fltarr(num_pha)
   emeas_OPlus2_expect_highEQ   = fltarr(num_pha)
   emeas_OPlus2_upper           = fltarr(num_pha)
   emeas_OPlus2_lower           = fltarr(num_pha)

   emeas_NePlus1_expect         = fltarr(num_pha)
   emeas_NePlus1_expect_lowEQ   = fltarr(num_pha)
   emeas_NePlus1_expect_highEQ  = fltarr(num_pha)
   emeas_NePlus1_upper          = fltarr(num_pha)
   emeas_NePlus1_lower          = fltarr(num_pha)
 
   emeas_FePlus7_expect         = fltarr(num_pha)
   emeas_FePlus7_expect_lowEQ   = fltarr(num_pha)
   emeas_FePlus7_expect_highEQ  = fltarr(num_pha)
   emeas_FePlus7_upper          = fltarr(num_pha)
   emeas_FePlus7_lower          = fltarr(num_pha)
 
   emeas_FePlus6_expect         = fltarr(num_pha)
   emeas_FePlus6_expect_lowEQ   = fltarr(num_pha)
   emeas_FePlus6_expect_highEQ  = fltarr(num_pha)
   emeas_FePlus6_upper          = fltarr(num_pha)
   emeas_FePlus6_lower          = fltarr(num_pha)
                            
   nFWHM_Proton          		= fltarr(num_pha)
   HWHM_Proton       			= fltarr(num_pha)

   nFWHM_HeliumPlus  			= fltarr(num_pha)
   HWHM_HeliumPlus   			= fltarr(num_pha)
   nFWHM_HeliumPlus2 			= fltarr(num_pha)
   HWHM_HeliumPlus2  			= fltarr(num_pha)

   nFWHM_CPlus1     			= fltarr(num_pha)
   HWHM_CPlus1      			= fltarr(num_pha)
  
   nFWHM_OPlus1     			= fltarr(num_pha)
   HWHM_OPlus1      			= fltarr(num_pha)
   nFWHM_OPlus2     			= fltarr(num_pha)
   HWHM_OPlus2      			= fltarr(num_pha)
   nFWHM_OPlus6     			= fltarr(num_pha)
   HWHM_OPlus6      			= fltarr(num_pha)

   nFWHM_NePlus1     			= fltarr(num_pha)
   HWHM_NePlus1      			= fltarr(num_pha)

   nFWHM_FePlus7     			= fltarr(num_pha)
   HWHM_FePlus7      			= fltarr(num_pha)
   nFWHM_FePlus6     			= fltarr(num_pha)
   HWHM_FePlus6      			= fltarr(num_pha)

   tns_Proton_expect            = fltarr(num_pha)
   tns_Proton_upper             = fltarr(num_pha)
   tns_Proton_lower             = fltarr(num_pha)
   cf_protons                   = fltarr(num_pha)
   
   tns_HePlus_expect            = fltarr(num_pha)
   tns_HePlus_upper             = fltarr(num_pha)
   tns_HePlus_lower             = fltarr(num_pha)
   cf_HePlus                    = fltarr(num_pha)

   tns_HePlus2_expect           = fltarr(num_pha)
   tns_HePlus2_upper            = fltarr(num_pha)
   tns_HePlus2_lower            = fltarr(num_pha)
   cf_HePlus2                   = fltarr(num_pha)
   cf_HePlus2_hiEQ              = fltarr(num_pha)
   cf_HePlus2_loEQ              = fltarr(num_pha)

   tns_CPlus1_expect            = fltarr(num_pha)
   tns_CPlus1_upper             = fltarr(num_pha)
   tns_CPlus1_lower             = fltarr(num_pha)
   cf_CPlus1                    = fltarr(num_pha)
   cf_CPlus1_hiEQ               = fltarr(num_pha)
   cf_CPlus1_loEQ               = fltarr(num_pha)

   tns_OPlus1_expect            = fltarr(num_pha)
   tns_OPlus1_upper             = fltarr(num_pha)
   tns_OPlus1_lower             = fltarr(num_pha)
   cf_OPlus1                    = fltarr(num_pha)
   cf_OPlus1_hiEQ               = fltarr(num_pha)
   cf_OPlus1_loEQ               = fltarr(num_pha)

   tns_OPlus2_expect            = fltarr(num_pha)
   tns_OPlus2_upper             = fltarr(num_pha)
   tns_OPlus2_lower             = fltarr(num_pha)
   cf_OPlus2                    = fltarr(num_pha)
   cf_OPlus2_hiEQ               = fltarr(num_pha)
   cf_OPlus2_loEQ               = fltarr(num_pha)

   tns_OPlus6_expect            = fltarr(num_pha)
   tns_OPlus6_upper             = fltarr(num_pha)
   tns_OPlus6_lower             = fltarr(num_pha)
   cf_OPlus6                    = fltarr(num_pha)
   cf_OPlus6_hiEQ               = fltarr(num_pha)
   cf_OPlus6_loEQ               = fltarr(num_pha)

   tns_H3OPlus1_expect          = fltarr(num_pha)
   tns_H3OPlus1_upper           = fltarr(num_pha)
   tns_H3OPlus1_lower           = fltarr(num_pha)
   cf_H3OPlus1                  = fltarr(num_pha)
   cf_H3OPlus1_hiEQ             = fltarr(num_pha)
   cf_H3OPlus1_loEQ             = fltarr(num_pha)


   tns_NePlus1_expect           = fltarr(num_pha)
   tns_NePlus1_upper            = fltarr(num_pha)
   tns_NePlus1_lower            = fltarr(num_pha)
   cf_NePlus1                   = fltarr(num_pha)
   cf_NePlus1_hiEQ              = fltarr(num_pha)
   cf_NePlus1_loEQ              = fltarr(num_pha)

   tns_SPlus1_expect            = fltarr(num_pha)
   tns_SPlus1_upper             = fltarr(num_pha)
   tns_SPlus1_lower             = fltarr(num_pha)
   cf_SPlus1                    = fltarr(num_pha)
   cf_SPlus1_hiEQ               = fltarr(num_pha)
   cf_SPlus1_loEQ               = fltarr(num_pha)

   tns_FePlus7_expect           = fltarr(num_pha)
   cf_FePlus7                   = fltarr(num_pha)
   cf_FePlus7_hiEQ              = fltarr(num_pha)
   cf_FePlus7_loEQ              = fltarr(num_pha)
 
   tns_FePlus6_expect           = fltarr(num_pha)
   cf_FePlus6                   = fltarr(num_pha)
   cf_FePlus6_hiEQ              = fltarr(num_pha)
   cf_FePlus6_loEQ              = fltarr(num_pha)

; Sat A values: 
; Energy detection threshold   
   threshold  = 29    ; approximate threshold in keV for STA based on a visual cutoff near chn 14 (28.5 keV)
   thres_bit  = 4     ;give a bit of leaway for check criteria, same for both s/c

;  Energy Noise add pulse height defect to upper and lower boundaries
   eNoise = 15
   width_factor   = 1.15        ; add a bit since half max only covers part of distribution
   width_factorFe = 0.55        ; lower fwhm seems to work for multiply charged ions
                       
; TOF width is empirically broader than nominal due to noise on power supplies 
; - add another 25% to A, 10% to B (may depend on MCP bias or gain)
   tof_broaden = 0.25

   IF sat EQ 'B' or which_section EQ 3 THEN BEGIN
          threshold = 26    ; cutoff for STB near chn 23 (25-26 keV)
          eNoise = 15
          width_factor   = 1.0         ; add a bit since half max only covers part of distribution
          width_factorFe = 0.5         ; factor = 0.5 is hwhm, factor = 1 is fwhm,
          tof_broaden = 0.10
   ENDIF                                    
         

; Protons Energy 
   emeas_Proton_expect =(0.0518*ALOG(e_per_q+pacHK) + 0.7307)* $
                         ((e_per_q+pacHK)-(-0.0003*(e_per_q+pacHK)^2+0.0386*(e_per_q+pacHK)+1.2715))
   emeas_Proton_expect_lowEQ =(0.0518*ALOG(e_per_q*0.95+pacHK) + 0.7307)* $
                         ((e_per_q*0.95+pacHK)-(-0.0003*(e_per_q*0.95+pacHK)^2+ $
                          0.0386*(e_per_q*0.95+pacHK)+1.2715))
   emeas_Proton_expect_highEQ =(0.0518*ALOG(e_per_q*1.05+pacHK) + 0.7307)* $
                         ((e_per_q*1.05+pacHK)-(-0.0003*(e_per_q*1.05+pacHK)^2+ $
                          0.0386*(e_per_q*1.05+pacHK)+1.2715))

   nFWHM_Proton = 1.0445*ALOG(((e_per_q+pacHK)-(-0.0003*(e_per_q+pacHK)^2+$
                  0.0386*(e_per_q+pacHK)+1.2715))*1/1) -1.2626
   HWHM_Proton  = 0.55 * sqrt(eNoise^2 + nFWHM_Proton^2)                ; normal hwhm+smidgeon


   emeas_Proton_upper  = emeas_Proton_expect_highEQ + HWHM_Proton      ; allow for noise in measured E
   emeas_Proton_lower  = emeas_Proton_expect_lowEQ  - HWHM_Proton
   emeas_Proton_pileup2 = 1.8 * emeas_Proton_expect
   temp_i = where (emeas_Proton_pileup2(*) LE 0, temp_count)
   IF temp_count GT 0 THEN emeas_Proton_pileup2(temp_i) = threshold
   
; negative energy corrections   - this will not allow threshold counts
   temp_i = where ( emeas_Proton_lower(*) LT 0, temp_count)
   IF temp_count GT 0 THEN emeas_Proton_lower(temp_i) = 0.
   temp_i = where ( emeas_Proton_upper(*) LT 0, temp_count)
   IF temp_count GT 0 THEN emeas_Proton_upper(temp_i) = 0.
   temp_i = where ( emeas_Proton_expect(*) LT 0, temp_count)
   IF temp_count GT 0 THEN emeas_Proton_expect(temp_i) = 0.

; Protons time of flight
   mq_proton  = 1.0
   cf_protons = -0.0003*(pacHK+e_per_q)^2+0.0386*(pacHK+e_per_q)+1.2715
   tns_Proton_expect = sqrt(mq_proton/(c1*(pacHK + e_per_q - cf_protons)))
   tns_Proton_lower  = (1-tof_broaden)*sqrt(mq_proton/(c1*(pacHK + 1.05*e_per_q - cf_protons)))
   tns_Proton_upper  = (1+tof_broaden)*sqrt(mq_proton/(c1*(pacHK + 0.95*e_per_q - cf_protons)))  


;;;;;;;;;;;;;;;;;;;;;
; HELIUM PLUS 1
   
; Helium Plus Energy
   emeas_HePlus_expect=(0.1132*ALOG((e_per_q+pacHK)*1/3.97)+0.519)*                  $
                       ((e_per_q+pacHK)-(-0.0002*(e_per_q+pacHK)^2+0.0451*(e_per_q+pacHK)+1.657))*1
   emeas_HePlus_expect_lowEQ=(0.1132*ALOG((e_per_q*0.95+pacHK)*1/3.97)+0.519)*       $
                       ((e_per_q*0.95+pacHK)-(-0.0002*(e_per_q*0.95+pacHK)^2+        $
                       0.0451*(e_per_q*0.95+pacHK)+1.657))*1
   emeas_HePlus_expect_highEQ=(0.1132*ALOG((e_per_q*1.05+pacHK)*1/3.97)+0.519)*      $
                       ((e_per_q*1.05+pacHK)-(-0.0002*(e_per_q*1.05+pacHK)^2+        $
                       0.0451*(e_per_q*1.05+pacHK)+1.657))*1

   nFWHM_HeliumPlus = 1.0822*ALOG(((e_per_q+pacHK)-(-0.0002*(e_per_q+pacHK)^2+       $
                      0.0451*(e_per_q+pacHK)+1.657))*1/3.97) + 2.3728
   HWHM_HeliumPlus  = 0.55 * sqrt(eNoise^2 + nFWHM_HeliumPlus^2)               ; normal hwhm+smidgeon

   emeas_HePlus_upper  = emeas_HePlus_expect_highEQ + HWHM_HeliumPlus    ; allow for fwhm in measured E
   emeas_HePlus_lower  = emeas_HePlus_expect_lowEQ - HWHM_HeliumPlus 

;  negative energy corrections
   temp_i = where ( emeas_HePlus_lower(*) LT 0, temp_count)
   IF temp_count GT 0 THEN emeas_HePlus_lower(temp_i) = 0.
   temp_i = where ( emeas_HePlus_upper(*) LT 0, temp_count)
   IF temp_count GT 0 THEN emeas_HePlus_upper(temp_i) = 0.
   temp_i = where ( emeas_HePlus_expect(*) LT 0, temp_count)
   IF temp_count GT 0 THEN emeas_HePlus_expect(temp_i) = 0.

; He+ time of flight
   mq_HePlus  = 4.0
   cf_HePlus = -0.0002*(pacHK+e_per_q)^2+0.0451*(pacHK+e_per_q)+1.657
   tns_HePlus_expect = sqrt(mq_HePlus/(c1*(pacHK + e_per_q - cf_HePlus)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B   
   tns_HePlus_upper  = (1+tof_broaden)*sqrt(mq_HePlus/(c1*(pacHK + 0.95*e_per_q - cf_HePlus)))
   tns_HePlus_lower  = (1-tof_broaden)*sqrt(mq_HePlus/(c1*(pacHK + 1.05*e_per_q - cf_HePlus)))

;;;;;;;;;;;;;;;;;;;;;
; HELIUM PLUS 2

; He+2 time of flight
   mq_HePlus2  = 2.0
   cf_HePlus2 = -0.0001*(pacHK+e_per_q)^2+0.0267*(pacHK+e_per_q)+1.0986
   cf_HePlus2_hiEQ = -0.0001*(pacHK+e_per_q*1.05)^2+0.0267*(pacHK+e_per_q*1.05)+1.0986
   cf_HePlus2_loEQ = -0.0001*(pacHK+e_per_q*0.95)^2+0.0267*(pacHK+e_per_q*0.95)+1.0986

   tns_HePlus2_expect = sqrt(mq_HePlus2/(c1*(pacHK + e_per_q - cf_HePlus2)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_HePlus2_upper  = (1+tof_broaden)*sqrt(mq_HePlus2/(c1*(pacHK + 0.95*e_per_q - cf_HePlus2_loEQ)))
   tns_HePlus2_lower  = (1-tof_broaden)*sqrt(mq_HePlus2/(c1*(pacHK + 1.05*e_per_q - cf_HePlus2_hiEQ)))

; Helium +2 Energy                     
   emeas_HePlus2_expect=(0.1132*ALOG((e_per_q+pacHK)*2/3.97)+0.519)*                 $
                       (e_per_q+pacHK-cf_HePlus2)*2
   emeas_HePlus2_expect_lowEQ=(0.1132*ALOG((e_per_q*0.95+pacHK)*2/3.97)+0.519)*      $
                       (e_per_q*0.95+pacHK-cf_HePlus2_loEQ)*2
   emeas_HePlus2_expect_highEQ=(0.1132*ALOG((e_per_q*1.05+pacHK)*2/3.97)+0.519)*     $
                       (e_per_q*1.05+pacHK-cf_HePlus2_hiEQ)*2

   nFWHM_HeliumPlus2 = 1.0822*ALOG(((e_per_q+pacHK)-(-0.0001*(e_per_q+pacHK)^2+      $
                       0.0267*(e_per_q+pacHK)+1.0986))*2/3.97) + 2.3728
   ; to fully cover, make hwhm more than "half", use 0.6 instead of 0.5
   HWHM_HeliumPlus2  = 0.55 * sqrt(eNoise^2 + nFWHM_HeliumPlus2^2)         ; normal hwhm+smidgeon

   emeas_HePlus2_upper   = emeas_HePlus2_expect_highEQ + HWHM_HeliumPlus2
   emeas_HePlus2_lower   = emeas_HePlus2_expect_lowEQ  - HWHM_HeliumPlus2
   emeas_HePlus2_pileup2 = 1.8* emeas_HePlus2_expect

   temp_i = where ( emeas_HePlus2_lower(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_HePlus2_lower(temp_i) = 0.
   temp_i = where ( emeas_HePlus2_upper(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_HePlus2_upper(temp_i) = 0.
   temp_i = where ( emeas_HePlus2_expect(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_HePlus2_expect(temp_i) = 0.

;;;;;;;;;;;;;;;;;;;;;
; Carbon Plus 1  - right now has oxygen stuff is in here

; C+1 time of flight  y = -9E-05x2 + 0.0423x + 4.1292 (C+)
   mq_CPlus1  = 12/1.
   cf_CPlus1 = -9E-05*(pacHK+e_per_q)^2+0.0423*(pacHK+e_per_q)+4.1292
   cf_CPlus1_hiEQ = -9E-05*(pacHK+e_per_q*1.05)^2+0.0423*(pacHK+e_per_q*1.05)+4.1292
   cf_CPlus1_loEQ = -9E-05*(pacHK+e_per_q*0.95)^2+0.0423*(pacHK+e_per_q*0.95)+4.1292

   tns_CPlus1_expect = sqrt(mq_CPlus1/(c1*(pacHK + e_per_q - cf_CPlus1)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_CPlus1_upper  = (1+tof_broaden)*sqrt(mq_CPlus1/(c1*(pacHK + 0.95*e_per_q - cf_CPlus1_loEQ)))
   tns_CPlus1_lower  = (1-tof_broaden)*sqrt(mq_CPlus1/(c1*(pacHK + 1.05*e_per_q - cf_CPlus1_hiEQ)))

; Carbon Plus Energy   = y = 0.1717ln(x) + 0.3246              
   emeas_CPlus1_expect=(0.1717*ALOG((e_per_q+pacHK)*1/12)+0.3246)*                 $
                       (e_per_q+pacHK-cf_CPlus1)*1
   emeas_CPlus1_expect_lowEQ=(0.1717*ALOG((e_per_q*0.95+pacHK)*1/12)+0.3246)*      $
                       (e_per_q*0.95+pacHK-cf_CPlus1_loEQ)*1
   emeas_CPlus1_expect_highEQ=(0.1717*ALOG((e_per_q*1.05+pacHK)*1/12)+0.3246)*     $
                       (e_per_q*1.05+pacHK-cf_CPlus1_hiEQ)*1

; Carbon y = 5.1545ln(x) + 4.9174 
   nFWHM_CPlus1 = 5.1545*ALOG((e_per_q+pacHK-cf_CPlus1)*1/12) + 4.9174 
   HWHM_CPlus1  = width_factor * sqrt(eNoise^2 + nFWHM_CPlus1^2)

   emeas_CPlus1_upper   = emeas_CPlus1_expect_highEQ + HWHM_CPlus1
   emeas_CPlus1_lower   = emeas_CPlus1_expect_lowEQ  - HWHM_CPlus1

   temp_i = where ( emeas_CPlus1_lower(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_CPlus1_lower(temp_i) = 0.
   temp_i = where ( emeas_CPlus1_upper(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_CPlus1_upper(temp_i) = 0.
   temp_i = where ( emeas_CPlus1_expect(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_CPlus1_expect(temp_i) = 0.

;;;;;;;;;;;;;;;;;;;;;
; Oxygen Plus 1

; O+1 time of flight  -0.0001x2 + 0.0506x + 4.1868
   mq_OPlus1  = 15.87/1.
   cf_OPlus1 = -0.0001*(pacHK+e_per_q)^2+0.0506*(pacHK+e_per_q)+4.1868
   cf_OPlus1_hiEQ = -0.0001*(pacHK+e_per_q*1.05)^2+0.0506*(pacHK+e_per_q*1.05)+4.1868
   cf_OPlus1_loEQ = -0.0001*(pacHK+e_per_q*0.95)^2+0.0506*(pacHK+e_per_q*0.95)+4.1868

   tns_OPlus1_expect = sqrt(mq_OPlus1/(c1*(pacHK + e_per_q - cf_OPlus1)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_OPlus1_upper  = (1+tof_broaden)*sqrt(mq_OPlus1/(c1*(pacHK + 0.95*e_per_q - cf_OPlus1_loEQ)))
   tns_OPlus1_lower  = (1-tof_broaden)*sqrt(mq_OPlus1/(c1*(pacHK + 1.05*e_per_q - cf_OPlus1_hiEQ)))

; Oxygen Plus Energy   = 0.164706ln(x) + 0.287961               
   emeas_OPlus1_expect=(0.164706*ALOG((e_per_q+pacHK)*1/15.87)+0.287961)*                 $
                       (e_per_q+pacHK-cf_OPlus1)*1
   emeas_OPlus1_expect_lowEQ=(0.164706*ALOG((e_per_q*0.95+pacHK)*1/15.87)+0.287961)*      $
                       (e_per_q*0.95+pacHK-cf_OPlus1_loEQ)*1
   emeas_OPlus1_expect_highEQ=(0.164706*ALOG((e_per_q*1.05+pacHK)*1/15.87)+0.287961)*     $
                       (e_per_q*1.05+pacHK-cf_OPlus1_hiEQ)*1

; Oxygen noise 9.434*LN(enuc)+4.0139
   nFWHM_OPlus1 = 9.434*ALOG((e_per_q+pacHK-cf_OPlus1)*1/15.87) + 4.0139
   HWHM_OPlus1  = width_factor * sqrt(eNoise^2 + nFWHM_OPlus1^2)

   emeas_OPlus1_upper   = emeas_OPlus1_expect_highEQ + HWHM_OPlus1
   emeas_OPlus1_lower   = emeas_OPlus1_expect_lowEQ  - HWHM_OPlus1

   temp_i = where ( emeas_OPlus1_lower(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_OPlus1_lower(temp_i) = 0.
   temp_i = where ( emeas_OPlus1_upper(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_OPlus1_upper(temp_i) = 0.
   temp_i = where ( emeas_OPlus1_expect(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_OPlus1_expect(temp_i) = 0.

;;;;;;;;;;;;;;;;;;;;;
; Oxygen Plus 2

; O+2 time of flight  y = -0.0002x2 + 0.0506x + 2.0934
   mq_OPlus2  = 15.87/2.
   cf_OPlus2 = -0.0002*(pacHK+e_per_q)^2+0.0506*(pacHK+e_per_q)+2.0934
   cf_OPlus2_hiEQ = -0.0002*(pacHK+e_per_q*1.05)^2+0.0506*(pacHK+e_per_q*1.05)+2.0934
   cf_OPlus2_loEQ = -0.0002*(pacHK+e_per_q*0.95)^2+0.0506*(pacHK+e_per_q*0.95)+2.0934

   tns_OPlus2_expect = sqrt(mq_OPlus2/(c1*(pacHK + e_per_q - cf_OPlus2)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_OPlus2_upper  = (1+tof_broaden)*sqrt(mq_OPlus2/(c1*(pacHK + 0.95*e_per_q - cf_OPlus2_loEQ)))
   tns_OPlus2_lower  = (1-tof_broaden)*sqrt(mq_OPlus2/(c1*(pacHK + 1.05*e_per_q - cf_OPlus2_hiEQ)))

; Oxygen Plus 2 Energy   = 0.164706ln(x) + 0.287961               
   emeas_OPlus2_expect=(0.164706*ALOG((e_per_q+pacHK)*2/15.87)+0.287961)*                 $
                       (e_per_q+pacHK-cf_OPlus2)*2
   emeas_OPlus2_expect_lowEQ=(0.164706*ALOG((e_per_q*0.95+pacHK)*2/15.87)+0.287961)*      $
                       (e_per_q*0.95+pacHK-cf_OPlus2_loEQ)*2
   emeas_OPlus2_expect_highEQ=(0.164706*ALOG((e_per_q*1.05+pacHK)*2/15.87)+0.287961)*     $
                       (e_per_q*1.05+pacHK-cf_OPlus2_hiEQ)*2

; Oxygen noise 9.434*LN(enuc)+4.0139
   nFWHM_OPlus2 = 9.434*ALOG((e_per_q+pacHK-cf_OPlus2)*2/15.87) + 4.0139
   HWHM_OPlus2  = width_factor * sqrt(eNoise^2 + nFWHM_OPlus2^2)

   emeas_OPlus2_upper   = emeas_OPlus2_expect_highEQ + HWHM_OPlus2
   emeas_OPlus2_lower   = emeas_OPlus2_expect_lowEQ  - HWHM_OPlus2

   temp_i = where ( emeas_OPlus2_lower(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_OPlus2_lower(temp_i) = 0.
   temp_i = where ( emeas_OPlus2_upper(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_OPlus2_upper(temp_i) = 0.
   temp_i = where ( emeas_OPlus2_expect(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_OPlus2_expect(temp_i) = 0.

;;;;;;;;;;;;;;;;;;;;;
; Oxygen Plus 6

; O+6 time of flight  y = -7E-05x2 + 0.0265x + 0.9374
   mq_OPlus6  = 15.87/6.
   cf_OPlus6 = -0.00007*(pacHK+e_per_q)^2+0.0265*(pacHK+e_per_q)+0.9374
   cf_OPlus6_hiEQ = -0.00007*(pacHK+e_per_q*1.05)^2+0.0265*(pacHK+e_per_q*1.05)+0.9374
   cf_OPlus6_loEQ = -0.00007*(pacHK+e_per_q*0.95)^2+0.0265*(pacHK+e_per_q*0.95)+0.9374

   tns_OPlus6_expect = sqrt(mq_OPlus6/(c1*(pacHK + e_per_q - cf_OPlus6)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_OPlus6_upper  = (1+tof_broaden)*sqrt(mq_OPlus6/(c1*(pacHK + 0.95*e_per_q - cf_OPlus6_loEQ)))
   tns_OPlus6_lower  = (1-tof_broaden)*sqrt(mq_OPlus6/(c1*(pacHK + 1.05*e_per_q - cf_OPlus6_hiEQ)))

; Oxygen Plus 6 Energy   = 0.164706ln(x) + 0.287961               
   emeas_OPlus6_expect=(0.164706*ALOG((e_per_q+pacHK)*6/15.87)+0.287961)*                 $
                       (e_per_q+pacHK-cf_OPlus6)*6
   emeas_OPlus6_expect_lowEQ=(0.164706*ALOG((e_per_q*0.95+pacHK)*6/15.87)+0.287961)*      $
                       (e_per_q*0.95+pacHK-cf_OPlus6_loEQ)*6
   emeas_OPlus6_expect_highEQ=(0.164706*ALOG((e_per_q*1.05+pacHK)*6/15.87)+0.287961)*     $
                       (e_per_q*1.05+pacHK-cf_OPlus6_hiEQ)*6

; Oxygen noise 9.434*LN(enuc)+4.0139
   nFWHM_OPlus6 = 9.434*ALOG((e_per_q+pacHK-cf_OPlus6)*6/15.87) + 4.0139
   HWHM_OPlus6  = width_factor * sqrt(eNoise^2 + nFWHM_OPlus6^2)

   emeas_OPlus6_upper   = emeas_OPlus6_expect_highEQ + HWHM_OPlus6
   emeas_OPlus6_lower   = emeas_OPlus6_expect_lowEQ  - HWHM_OPlus6

;;;;;;;;;;;;;;;;;;;;;
; H3O Plus 1

; O+ used for CF   -0.0001x2 + 0.0506x + 4.1868
   mq_H3OPlus1  = 19./1.
   cf_H3OPlus1 = -0.0001*(pacHK+e_per_q)^2+0.0506*(pacHK+e_per_q)+4.1868
   cf_H3OPlus1_hiEQ = -0.0001*(pacHK+e_per_q*1.05)^2+0.0506*(pacHK+e_per_q*1.05)+4.1868
   cf_H3OPlus1_loEQ = -0.0001*(pacHK+e_per_q*0.95)^2+0.0506*(pacHK+e_per_q*0.95)+4.1868

   tns_H3OPlus1_expect = sqrt(mq_H3OPlus1/(c1*(pacHK + e_per_q - cf_H3OPlus1)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_H3OPlus1_upper  = (1+tof_broaden)*sqrt(mq_H3OPlus1/(c1*(pacHK + 0.95*e_per_q - cf_H3OPlus1_loEQ)))
   tns_H3OPlus1_lower  = (1-tof_broaden)*sqrt(mq_H3OPlus1/(c1*(pacHK + 1.05*e_per_q - cf_H3OPlus1_hiEQ)))


;;;;;;;;;;;;;;;;;;;;;
; Neon Plus 1

; Ne+ time of flight  1.1292E-05x2 + 2.5980E-02x + 1.0271E+01
   mq_NePlus1  = 20.02/1.
   cf_NePlus1 = -0.000011292*(pacHK+e_per_q)^2+0.02598*(pacHK+e_per_q)+10.0271
   cf_NePlus1_hiEQ = -0.000011292*(pacHK+e_per_q*1.05)^2+0.02598*(pacHK+e_per_q*1.05)+10.0271
   cf_NePlus1_loEQ = -0.000011292*(pacHK+e_per_q*0.95)^2+0.02598*(pacHK+e_per_q*0.95)+10.0271

   tns_NePlus1_expect = sqrt(mq_NePlus1/(c1*(pacHK + e_per_q - cf_NePlus1)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_NePlus1_upper  = (1+tof_broaden)*sqrt(mq_NePlus1/(c1*(pacHK + 0.95*e_per_q - cf_NePlus1_loEQ)))
   tns_NePlus1_lower  = (1-tof_broaden)*sqrt(mq_NePlus1/(c1*(pacHK + 1.05*e_per_q - cf_NePlus1_hiEQ)))

; Neon Plus Energy   y = 0.1443ln(x) + 0.2722             
   emeas_NePlus1_expect=(0.1443*ALOG((e_per_q+pacHK)*1/20.02)+0.2722)*              $
                       (e_per_q+pacHK-cf_NePlus1)*1
   emeas_NePlus1_expect_lowEQ=(0.1443*ALOG((e_per_q*0.95+pacHK)*1/20.02)+0.2722)*    $
                       (e_per_q*0.95+pacHK-cf_NePlus1_loEQ)*1
   emeas_NePlus1_expect_highEQ=(0.1443*ALOG((e_per_q*1.05+pacHK)*1/20.02)+0.2722)*   $
                       (e_per_q*1.05+pacHK-cf_NePlus1_hiEQ)*1

; Neon Noise  y = 13.721ln(x) + 3.1264
   nFWHM_NePlus1 = 13.721*ALOG((e_per_q+pacHK-cf_NePlus1)*1/20.02) + 3.1264
   HWHM_NePlus1  = width_factor * sqrt(eNoise^2 + nFWHM_NePlus1^2)

   emeas_NePlus1_upper   = emeas_NePlus1_expect_highEQ + HWHM_NePlus1
   emeas_NePlus1_lower   = emeas_NePlus1_expect_lowEQ  - HWHM_NePlus1

   temp_i = where ( emeas_NePlus1_lower(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_NePlus1_lower(temp_i) = 0.
   temp_i = where ( emeas_NePlus1_upper(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_NePlus1_upper(temp_i) = 0.
   temp_i = where ( emeas_NePlus1_expect(*) LT 0., temp_count)
   IF temp_count GT 0 THEN emeas_NePlus1_expect(temp_i) = 0.

;;;;;;;;;;;;;;;;;;;;;
; Sulfur Plus 1

; S+ time of flight  y = -1E-05x2 + 0.0277x + 5.7431
   mq_SPlus1  = 32/1.
   cf_SPlus1 = -1E-05*(pacHK+e_per_q)^2+0.0277*(pacHK+e_per_q)+5.7431
   cf_SPlus1_hiEQ = -1E-05*(pacHK+e_per_q*1.05)^2+0.0277*(pacHK+e_per_q*1.05)+5.7431
   cf_SPlus1_loEQ = -1E-05*(pacHK+e_per_q*0.95)^2+0.0277*(pacHK+e_per_q*0.95)+5.7431

   tns_SPlus1_expect = sqrt(mq_SPlus1/(c1*(pacHK + e_per_q - cf_SPlus1)))
;  width is empirically broader than nominal - add another 25% to A, 11% to B 
   tns_SPlus1_upper  = (1+tof_broaden)*sqrt(mq_CPlus1/(c1*(pacHK + 0.95*e_per_q - cf_SPlus1_loEQ)))
   tns_SPlus1_lower  = (1-tof_broaden)*sqrt(mq_CPlus1/(c1*(pacHK + 1.05*e_per_q - cf_SPlus1_hiEQ)))


;;;;;;;;;;;;;;;;;;;;;
; IRON PLUS 7

; Fe+7 time of flight
   mq_FePlus7   = 55.41/7
   cf_FePlus7 = -0.00005*(pacHK+e_per_q)^2+0.0046*(pacHK+e_per_q)+1.5463
   cf_FePlus7_hiEQ = -0.00005*(pacHK+e_per_q*1.05)^2+0.0046*(pacHK+e_per_q*1.05)+1.5463
   cf_FePlus7_loEQ = -0.00005*(pacHK+e_per_q*0.95)^2+0.0046*(pacHK+e_per_q*0.95)+1.5463
   
   tns_FePlus7_expect = sqrt(mq_FePlus7/(c1*(pacHK + e_per_q - cf_FePlus7)))
   tns_FePlus7_upper  = (1+tof_broaden)*sqrt(mq_FePlus7/(c1*(pacHK + 0.95*e_per_q - cf_FePlus7_loEQ)))
   tns_FePlus7_lower  = (1-tof_broaden)*sqrt(mq_FePlus7/(c1*(pacHK + 1.05*e_per_q - cf_FePlus7_hiEQ)))

; Fe +7 Energy   0.1442ln(x) + 0.1525                  
   emeas_FePlus7_expect=(0.1142*ALOG((e_per_q+pacHK)*7/55.41)+0.1525)*                 $
                       (e_per_q+pacHK-cf_FePlus7)*7
   emeas_FePlus7_expect_lowEQ=(0.1142*ALOG((e_per_q*0.95+pacHK)*7/55.41)+0.1525)*      $
                       (e_per_q*0.95+pacHK-cf_FePlus7_loEQ)*7
   emeas_FePlus7_expect_highEQ=(0.1142*ALOG((e_per_q*1.05+pacHK)*7/55.41)+0.1525)*     $
                       (e_per_q*1.05+pacHK-cf_FePlus7_hiEQ)*7

   nFWHM_FePlus7 = 27.416*ALOG((e_per_q+pacHK-cf_FePlus7)*7/55.41) + 8.2339
   HWHM_FePlus7  = width_factorFe * sqrt(eNoise^2 + nFWHM_FePlus7^2)

   emeas_FePlus7_upper   = emeas_FePlus7_expect_highEQ + HWHM_FePlus7
   emeas_FePlus7_lower   = emeas_FePlus7_expect_lowEQ  - HWHM_FePlus7

;;;;;;;;;;;;;;;;;;;;;
; IRON PLUS 6

; Fe+6 time of flight = 9E-06x2 + 0.0094x + 1.7118
   mq_FePlus6   = 55.41/6
   cf_FePlus6 = -0.000009*(pacHK+e_per_q)^2+0.0094*(pacHK+e_per_q)+1.7118
   cf_FePlus6_hiEQ = -0.000009*(pacHK+e_per_q*1.05)^2+0.0094*(pacHK+e_per_q*1.05)+1.7118
   cf_FePlus6_loEQ = -0.000009*(pacHK+e_per_q*0.95)^2+0.0094*(pacHK+e_per_q*0.95)+1.7118
   
   tns_FePlus6_expect = sqrt(mq_FePlus6/(c1*(pacHK + e_per_q - cf_FePlus6)))
   tns_FePlus6_upper  = (1+tof_broaden)*sqrt(mq_FePlus7/(c1*(pacHK + 0.95*e_per_q - cf_FePlus6_loEQ)))
   tns_FePlus6_lower  = (1-tof_broaden)*sqrt(mq_FePlus7/(c1*(pacHK + 1.05*e_per_q - cf_FePlus6_hiEQ)))

; Fe +6 Energy   0.1442ln(x) + 0.1525                  
   emeas_FePlus6_expect=(0.1142*ALOG((e_per_q+pacHK)*6/55.41)+0.1525)*                 $
                       (e_per_q+pacHK-cf_FePlus6)*6
   emeas_FePlus6_expect_lowEQ=(0.1142*ALOG((e_per_q*0.95+pacHK)*6/55.41)+0.1525)*      $
                       (e_per_q*0.95+pacHK-cf_FePlus6_loEQ)*6
   emeas_FePlus6_expect_highEQ=(0.1142*ALOG((e_per_q*1.05+pacHK)*6/55.41)+0.1525)*     $
                       (e_per_q*1.05+pacHK-cf_FePlus6_hiEQ)*6

   nFWHM_FePlus6 = 27.416*ALOG((e_per_q+pacHK-cf_FePlus6)*6/55.41) + 8.2339
   HWHM_FePlus6  = width_factorFe * sqrt(eNoise^2 + nFWHM_FePlus6^2)

   emeas_FePlus6_upper   = emeas_FePlus6_expect_highEQ + HWHM_FePlus6
   emeas_FePlus6_lower   = emeas_FePlus6_expect_lowEQ  - HWHM_FePlus6


;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
; SECOND TERATION IN DETERMING MASS-PER-CHARGE 
; USES SPECIES CARBON FOIL CORRECTION FROM CFOIL
; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   m_per_q_init2 = m_per_q_init1

;
; Case for most solar wind species
;
temp_i = where((m_per_q_init1 GT .1) AND (m_per_q_init1 LE 40.), temp_count)
IF temp_count GT 0 THEN $   ;   using derived carbon foil loss at total E/Q = 22 as a mark
   c2_abg[temp_i] = 0.0057*(m_per_q_init1[temp_i])^2 - 0.0093*m_per_q_init1[temp_i] + 1.5059

temp_i = where((m_per_q_init1 GT 40), temp_count)   ; the Wap section 3 extends to m/q 60
IF temp_count GT 0 THEN $   ;   using derived carbon foil loss at total E/Q = 22 as a mark
   c2_abg[temp_i] = 0.0057*(40)^2 - 0.0093*40 + 1.5059

; artificial high m/q ions in s channel due to pileup, do not want to over correct for carbon foil
temp_i = where ( ((section EQ 1) OR (section EQ 0 AND position EQ 0)) AND $
         (m_per_q_init1 GT 3), temp_count)
IF temp_count GT 0 THEN c2_abg[temp_i] = 1.5   ; use default

; O+ mass non-zero- verify E chn measured falls below proton Echn measured curve
temp_i = where((m_per_q_init1 GE 10.) AND (mass GT 5 AND mass LE 40) AND (section LT 3) $
                AND (e_meas LE emeas_Proton_lower), temp_count)
IF temp_count GT 0 THEN BEGIN   ;         assume pickup oxygen
     c2_abg[temp_i] = 0.0385*(pacHK + e_per_q[temp_i]) +  4.4253
ENDIF


; Case for He+, mass non zero - verify E chn measured falls below proton expect measured curve
; before applying. Unfortunately, proton and helium measured energies are very close.
temp_i = where ( (mass GT 2.7 AND mass LT 12) AND (m_per_q_init1 GT 2.4 AND m_per_q_init1 LT 6.4 $
         AND m_per_q_init1 GE mqHe_lowbound) AND (e_meas LT emeas_HePlus_upper) AND $
         (section LT 3), temp_count)
IF temp_count GT 0 THEN $ ;   assume pickup helium, but some misplaced protons can still get here
       c2_abg[temp_i] = -0.0002*(pacHK + e_per_q[temp_i])^2 + 0.0423*(pacHK + e_per_q[temp_i]) + 1.6977

temp_i = where(c2_abg GE 10, temp_count)
IF temp_count GT 0 THEN c2_abg[temp_i] = 10.
   

;  Now apply   
temp_i = where((tns EQ 0 OR tns GT 10), temp_count)
IF temp_count GT 0 THEN $
m_per_q_init2[temp_i] = c1[temp_i] * (pacHK + e_per_q[temp_i] - c2_abg[temp_i]) * (tns[temp_i])^2
   
 m_per_q =  m_per_q_init2
 
; print,  'e/q =', e_per_q, '    mass = ', mass, '    m/q_1,final = ', m_per_q_init1, m_per_q



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;CALCULATION OF Q and v
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

q_init = fltarr(num_pha)
v_init = fltarr(num_pha)
q_init[*] = 0
v_init[*] = 0.

temp_i = where(m_per_q GT 1 AND mass GT 0, temp_count) 
IF temp_count GT 0 THEN q_init[temp_i] = mass[temp_i] / m_per_q[temp_i]

temp_i = where(m_per_q GT .5, temp_count) 
IF temp_count GT 0 THEN v_init[temp_i] = 438*sqrt(e_per_q[temp_i]/m_per_q[temp_i])


; print, mass, m_per_q , q_init, v_init  ; check on screen


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Clean up and Consistency checks
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; NOTE: Algorithm 2 applies consistency checks that should always be true
;                 4 applies these, but places restrictions for solar wind ions
;                   which can result in the loss of pickup or magnetospheric ions.
;                   If "check" is set, strict restrictions on the SSD vs RA position.
;                   These SSD-RA restrictions are to tighten up the TOF resolution
;                   due to scattering in the carbon foil.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  INSTRUMENTAL ELECTRONICS AND MECHANICAL (CF SCATTER) CHECKS

  bad_tof = 45
  IF sat EQ 'B' THEN bad_tof = 60
; Check I-1: BAD TOF
; instrumental effect - real particle but bad time of flight; messes up mass and m/q
  temp_i = where((tof GT 0) AND (tof LE bad_tof) AND (e_meas LE e_min) $
                  AND (section LT 3), temp_count)
  IF temp_count GT 0 THEN m_per_q[temp_i] = 1000  ; instrumental effect - use 1000 as flag Nq=255
  IF temp_count GT 0 THEN mass[temp_i] = 1000	  ; this will be assigned Nm = 96  
                                                  ; note - will cut off high energy protons
                                                  ; only, repeat only, if they don't have a measured energy

; Check I-2:  BAD TOF, frequency depends on flux
;             these are not always from protons - can be any high flux
;             but no species properly measured can have a speed (tof) faster than protons
   temp_i = where (tns LT  tns_Proton_lower, temp_count); lower limit tns from protons 
   IF temp_count GT 0 THEN mass[temp_i]    = 1000    ; this will be assigned Nm = 96
   IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255


; Check I-3:  Energy Value out of Range
  temp_i = where((tns GT 10) AND (e_meas GT 2000) AND (section LT 3), temp_count)
  IF temp_count GT 0 THEN mass[temp_i]    = 1000	; this will be assigned Nm = 96
  IF temp_count GT 0 THEN m_per_q[temp_i] = 1000	; this will be assigned Nq = 255


; Check I-4:  Multiple or Unknown positions in section 0 
; position zero means either zero or unknown so ends up in Main Section 0. 
; NOTE in STA this zero position seems to occur a lot when the electronics is 
; overwhelmed by lots of accidentals.  Accidentals ends up in wrong mass 
; and/or mass per charge.
;

; always clean up for IRON
  temp_i = where ( (section EQ 0 AND position EQ 0) AND $
                   (mass GT 40), temp_count)
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255                                          
    IF temp_count GT 0 THEN mass[temp_i]    = 1000	  ; this will be assigned Nm = 96  
                  
; otherwise clean up if "check" is set
  IF sat EQ 'A' AND (check EQ 1 OR check EQ 2) THEN BEGIN
    temp_i = where( (section EQ 0 AND position EQ 0), temp_count); unknown position, usually pileup
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255                                          
    IF temp_count GT 0 THEN mass[temp_i]    = 1000	  ; this will be assigned Nm = 96  
  ENDIF


;   For STB, position zero mostly are ions from the WAP side.  
;   Eliminating these events affects a lot of the proton
;   suprathermal tails (real events, but not really in the SWS section).  Probably see because
;   the MCP bias on STB is so high, that the non-scrubbed WAP is extremely sensitive, so can
;   see the trails even though the SWS does not, but the "loud" MCP pulses cause a position=0
;   Use the "check" condition at start of module to utilize.

    IF sat EQ 'B' AND (check EQ 1 OR check EQ 2) THEN BEGIN
      temp_i = where( (section EQ 0 AND position EQ 0), temp_count)
      IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; unknown position, may be wap
                                                        ; this will be assigned Nq = 255                                             
      IF temp_count GT 0 THEN mass[temp_i]    = 1000	; this will be assigned Nm = 96  
    ENDIF

 ;  Check I-5 : Modest SSD pixel scatter restrictions (scatter due to carbon foil)
 ;  SSD id not consistent with section, allows some scatter beyond section, specifically
 ;  pixels 0, 9, and 10 (i.e., the most adjacent pixels)
    temp_i = where( ((section EQ 0 OR section EQ 1) AND ssd_id GT 11), temp_count)
    IF temp_count GT 0 THEN mass[temp_i]    = 1000    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  PROTON CHECK AREA


;  Check P-1:  Proton Energy Pileup
;  Pileup energies for protons fall onto the He+2 track; need to rely on m/q and m
;  If not Protons (H+), then m/q must be greater than one. 3He+2 would be m/q=1.5
;                  H2+ will be mass 1 and m/q 2
;  Give a little slack for broad time/energy
;  NOTE:  a number of threshold events are eliminated as pileup
   temp_i = where ( (section LT 3) AND              $        ; section with energy
        (mass GT 0) AND                             $        ; non zero mass events
        (m_per_q LE 0.75 )  AND                     $        ; leave margin for 3He+2 (0.5*mq=1.5)
        (e_meas GE 1.5*emeas_Proton_pileup2),       $        ; limit for triple hit pileup
        temp_count)                                          ; does not allow threshold counts
    IF (temp_count GT 0) THEN BEGIN
                     mass [temp_i]   = 1000           ; this will be assigned Nm = 96
                     m_per_q[temp_i] = 1000           ; this will be assigned Nq = 255
    ENDIF

;  Check P-2a:  Proton and He+ higher TOF Pileup. Check tof above He+ (m/q 4).
;  Identify as H or He+ from the emeas vs ESA track.  
;  NOTE that emeasured is almost identical for He+ as for Protons, O+ would be lower 
;  than protons or He+, O+ clearly separated except at higher energies (> 35 keV/e)
   temp_i = where ( (section LT 3)          AND $     ; ssd sections
        (mass GT 0 AND mass LT 40)          AND $     ; non zero mass events, protect Fe
        (tns GT tns_HePlus_upper)           AND $     ; mq > 4, Fe = or < 14 is in this range
        (e_meas GE emeas_HePlus_lower)      AND $     ; lower limit energy from lower Helium +1
        (e_meas LE emeas_Proton_upper),         $     ; upper limit energy from protons
        temp_count)                                   ; does not allow threshold counts
   temp_Oplus = where ( (mass GE 10)        AND $     ; in the C, O mass range
        (e_meas GE 0.95*emeas_OPlus1_lower) AND $     ; check O+ emeas area
        (e_meas LE 1.00*emeas_OPlus1_upper) AND $
        (tns GE tns_OPlus1_lower)           AND $     ; check O+ tmeas area  
        (tns LE tns_OPlus1_upper)           AND $
        (position NE 0),                        $     ; except if position = 0, don't trust  
        temp_Opluscount)
   IF (temp_count GT 0 AND temp_Opluscount LE 0) THEN BEGIN
        mass[temp_i]    = 1000    ; this will be assigned Nm = 96
        m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255
   ENDIF


;  Check P-2b:  Proton higher TOF pileup below He+
;  Protect 3He+, which will have mass of 3 but m/q of 1.5, keep strict emeas range
;  Protect H2+, which will have mass of H, but m/q of 2
   temp_i = where ( (section EQ 1) AND          $     ; solar wind protons in schn
        (mass GT 1.7 AND mass LT 40) AND        $     ; Mass>Proton, and protect Fe, H2+
        (tns LT  tns_HePlus_lower)  AND         $     ; target range below He+, mq<4
        (tns GT  tns_Proton_upper) AND          $     ; and above Protons, mq>1
        (e_meas GE emeas_Proton_lower) AND      $     ; lower limit energy from protons
        (e_meas LE emeas_Proton_upper),         $     ; upper limit energy from protons
        temp_count)                                   
    IF temp_count GT 0 THEN BEGIN                     ;  these are protons, redefine   
         mass[temp_i]    = 0.2*(mass[temp_i]-1.0)/1.0 + 1.0
         m_per_q[temp_i] = 0.2*(m_per_q[temp_i]-1.0)/1.0 + 1.0
    ENDIF
   ;IF temp_count GT 0 THEN mass[temp_i]    = 1000    ; this will be assigned Nm = 96
   ;IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255


;  Check P-2c:  Proton TOF Pileup through m/q 4
;  Identify as H or He+ from the emeas vs ESA track. Take values gt He upper  
;  NOTE that emeasured is almost identical for He+ as for Protons
;  Looking to help clean up protons in the He+
   temp_i = where ( (section EQ 1) AND          $     ; proton bulk solar wind pileup is in sch
        (mass GT 0 AND mass LT 40) AND          $     ; non zero mass events
        (tns GT  tns_HePlus2_upper) AND         $     ; mq > 2, 
        (e_meas GE emeas_HePlus_upper) AND      $     ; lower limit energy from upper Helium +1
        (e_meas LE emeas_Proton_upper) AND      $     ; upper limit energy from proton
        (e_meas LT emeas_HePlus2_lower),        $     ; but keep He+2 safe
        temp_count)                                   ; 
   IF temp_count GT 0 THEN mass[temp_i]    = 1000    ; this will be assigned Nm = 96
   IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Helium check area
;;

temp_helium = where (    (section LT 3) AND     $       ; section with ENERGY
                         (e_meas GT 0 ) AND     $       ; have energy, therefore a mass
                         (e_meas GT 0.9*emeas_HePlus_lower)    AND  $ ;lower energy range helium plus
                         (e_meas LT 1.5*emeas_HePlus2_pileup2) AND  $ ;upper energy range triple pileup
                         (tns GT tns_HePlus2_lower),                $ ;lower tof range He+2, He+ will be higher tof
                          count_helium_area        )
                          
IF count_helium_area GT 0 THEN BEGIN
;  Check He2-1a:  Helium+2 energy pileup
;  NOTE - checking cases where pileup is 2*measured energy 
;        for triple pile up - have a problem - C6 and O7 will have the same emeas as
;        triple 4He2
;  Check I5a:  safe for C6, N7, O8 etc through mq restriction lt 1.6
   temp_i = where ( (section LT 3) AND $                      ; section with energy
        (m_per_q LT 1.6 )  AND $                              ; m/q range he+2,leave margin for C6
        (e_meas GE emeas_HePlus2_pileup2), temp_count)        ; lower limit double hits  
    IF temp_count GT 0 THEN mass[temp_i]    = 1000    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255


;  Check He2-2a:  Helium2 higher TOF Pileup check, if have energy to cross check
;  Identify as He+2 from emeas vs ESA.  Check tof.
;  Safe for 3He+2 (tns, or mq, criteria), safe for He+ and O+ (emeas criteria)
;  To protect Fe+3 to Fe+9, 2keV/e checks sw He+2 through 438 km/s and is ok for
;  these Fe's down to 250 km/s.  Higher Fe charge states are not close to
;  He+2 emeas track (although will depend on chosen width_factor!!
   temp_i = where ( (section LT 3)           AND $     ; ssd sections
        (mass   GT 0)                        AND $     ; non zero mass events
        (tns    GT tns_HePlus2_upper)        AND $     ; mq > 2, or high tof tail for He+2
        (e_meas GT threshold)                AND $     ; want to be above threshold to verify He+2
        (e_meas GE 1.05*emeas_HePlus2_lower) AND $     ; lower limit energy from He+2, smidge buffer
        (e_meas LE 1.05*emeas_HePlus2_upper),    $     ; upper limit energy from He+2         
        temp_count)                                   ; does not allow threshold counts
   temp_Fe = where ( (mass GE 40)            AND $     ; in the Fe mass range
        (e_meas GE 0.95*emeas_FePlus6_lower) AND $     ; check Fe6-7 area
        (e_meas LE 1.05*emeas_FePlus7_upper) AND $
        (tns GE 0.95*tns_FePlus7_lower)      AND $     ; check Fe6-7 area
        (tns LE 1.05*tns_FePlus6_upper),         $
        temp_Fecount)
    IF (temp_count GT 0 AND temp_Fecount LE 0) THEN BEGIN
         mass[temp_i]    = 1000    ; this will be assigned Nm = 96
         m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255
    ENDIF

;  Check He2-2b:  Helium2 higher TOF Pileup check, if have energy to cross check
;  Identify as He+2 from emeas vs ESA.  Check tof. Make sure don't mess with Fe or O+
;  This check does not limit mass or eq area: Allows check through Fe 7 and O+, but
;  is not as thorough as Check He2-2a.  Currently superceded by 2a.
   temp_i = where ( (section LT 3) AND          $     ; ssd sections
        (mass GT 0) AND                         $     ; non zero mass events
        (tns  GT tns_HePlus2_upper) AND         $     ; mq > 2, or high tof tail for He+2
        (e_meas GT threshold) AND               $     ; want to be above threshold to verify He+2
        (e_meas GE .95*emeas_HePlus2_lower) AND $     ; lower limit energy from He+2
        (e_meas LE 1.05*emeas_HePlus2_upper),   $     ; upper limit energy from He+2         
        temp_count)                                   ; does not allow threshold counts
    temp_Oplus = where ( (mass GE 10)       AND $     ; in the O mass range
        (e_meas GE 0.95*emeas_OPlus1_lower) AND $     ; check O+ area
        (e_meas LE 1.05*emeas_OPlus1_upper) AND $
        (tns GE 0.95*tns_OPlus1_lower)      AND $     ; check O+ area
        (tns LE 1.05*tns_OPlus1_upper)      AND $
        (position NE 0),                        $     ; except if position = 0, don't trust
        temp_Opluscount)
   temp_Oplus2 = where ( (mass GE 10)       AND $     ; in the O mass range
        (e_meas GE 0.95*emeas_OPlus2_lower) AND $     ; check O+ area
        (e_meas LE 1.05*emeas_OPlus2_upper) AND $
        (tns GE 0.95*tns_OPlus2_lower)      AND $     ; check O+2 area
        (tns LE 1.05*tns_OPlus2_upper)      AND $
        (position NE 0),                        $     ; except if position = 0, don't trust, 
        temp_Oplus2count)
    temp_Neplus = where ( (mass GE 14)       AND $     ; in the Ne mass range
        (e_meas GE 0.95*emeas_NePlus1_lower) AND $     ; check Ne+ area
        (e_meas LE 1.05*emeas_NePlus1_upper) AND $
        (tns GE 0.95*tns_NePlus1_lower)      AND $     ; check Ne+ area
        (tns LE 1.05*tns_NePlus1_upper)      AND $
        (position NE 0),                         $     ; except if position = 0, don't trust,
        temp_Nepluscount)        
    temp_Fe = where ( (mass GE 40)           AND $     ; in the Fe mass range
        (e_meas GE 0.95*emeas_FePlus7_lower) AND $     ; check Fe7 area
        (e_meas LE 1.05*emeas_FePlus7_upper) AND $
        (tns GE 0.95*tns_FePlus7_lower)      AND $     ; check Fe7 area
        (tns LE 1.05*tns_FePlus7_upper)      AND $
        (position NE 0),                         $     ; except if position = 0, don't trust
        temp_Fecount)
    IF (temp_count GT 0                     AND $   ; in He+2 energy range, but 
        temp_Fecount LE 0                   AND $   ; allow low energy Fe, if pos ne 0 
        temp_Opluscount LE 0                AND $   ; allow O+, if pos ne 0
        temp_Oplus2count LE 0               AND $   ; allow O+2, if pos ne 0
        temp_Nepluscount LE 0) THEN BEGIN       $   ; allow Ne+, if pos ne 0  
        mass[temp_i]    = 1000    ; this will be assigned Nm = 96
        m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255
    ENDIF

;  Check He1-3:  Helium1 higher TOF Pileup check, if have energy to cross check
;  Identify as He+1 from emeas vs ESA.  Check tof.
   temp_i = where ( (section LT 3)           AND  $     ; ssd sections
        (mass   GT 0)                        AND  $     ; non zero mass events
        (m_per_q GT 6)                       AND  $     ; past the He+ range
        (tns    GT tns_HePlus_upper)         AND  $     ; high tof tail for He+1
        (e_meas GT threshold)                AND  $     ; want to be above threshold to verify He+1
        (e_meas GE 1.05*emeas_HePlus_lower)  AND  $     ; lower limit energy from He+1, smidge O+ buffer
        (e_meas LE 1.05*emeas_HePlus_upper),      $     ; upper limit energy from He+1         
        temp_count)                                   ; does not allow threshold counts
    IF temp_count GT 0 THEN mass[temp_i]    = 1000    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1000    ; this will be assigned Nq = 255

 ENDIF      ; overall helium area



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; PHYSICS CONSISTENCY CHECKS
;  Note: H2+ will show as m/q =2, but mass = 1. Deuterium would show as m/q=2, but
;  mass equal 2.


 E_EQ_Fe           = fltarr(num_pha)
 E_EQ_CNO          = fltarr(num_pha)
 E_EQ_PUI          = fltarr(num_pha)
 V_EQ_PUI          = fltarr(num_pha)
 time_EQ_PUI       = fltarr(num_pha)
 E_EQ_H            = fltarr(num_pha)
 V_EQ_H            = fltarr(num_pha)
 time_EQ_H         = fltarr(num_pha)
 E_speed_Fe        = fltarr(num_pha)
 E_speed_H         = fltarr(num_pha)
 time_EQ_Emeas     = fltarr(num_pha)
 tns_mq_O6_pha     = fltarr(num_pha)
 mq_speed_low      = fltarr(num_pha)
 mq_speed_high     = fltarr(num_pha)
 
 E_EQ_Fe [*]       = 2000.  ; maximum energy for adc
 E_EQ_CNO [*]      = 2000.
 E_speed_Fe [*]    = 2000.
 E_speed_H [*]     = 0.
 E_EQ_PUI [*]      = 0.
 V_EQ_PUI [*]      = 0.
 time_EQ_PUI [*]   = 250.			; max tn allowed by TAC board
 E_EQ_H [*]        = 0.
 V_EQ_H [*]        = 0.
 time_EQ_H [*]     = 10.
 time_EQ_Emeas [*] = 250.
 tns_mq_O6_pha [*] = 0.
 mq_speed_low [*]  = 100.          ; maximum mq possible in instrument
 mq_speed_high [*] = 0.3           ; minimum mq possible in instrument
  
 E_EQ_PUI = (pacHK + e_per_q*0.85) ; assume 15% hwhm, subtract to make higher tof

; Check Phys - 1a:  speed-mq limit on mq values
; nominal m/q based on nominal instrument range OR based on nominal speed range and e/q
; if mq>1000, already flagged elsewhere
  IF routine_check EQ 1 THEN BEGIN  ; use nominal speed range
   temp_i = where (m_per_q GT 0 AND m_per_q LT 1000 AND section LT 3, temp_count)
   IF temp_count GT 0 THEN $
      mq_speed_low[temp_i]  = 1.05*1.035*e_per_q[temp_i]/(speed_min/437.8)^2        ;eq & mq bandwidths
      mq_speed_high[temp_i] = 0.95*0.965*e_per_q[temp_i]/(speed_max/437.8)^2        ;eq & mq bandwidths 
  ENDIF
  temp_mq = where (m_per_q GT mq_speed_low OR m_per_q LT mq_speed_high, count_mq)
  temp_m = where (mass GT 0, count_mass)
  ;IF (count_mq GT 0 AND count_mass GT 0) THEN mass[temp_m]     = 1100	; this will be assigned Nm = 96
  IF (count_mq GT 0)                     THEN m_per_q[temp_mq] = 1100   ; this will be assigned Nq = 255 

; Check Phys-2a : upper limit tof
; check for consistent tof from energy, 
; based on E= 1/2 mV^2, and using e_meas as lower limit for Etotal (e_meas will have defect)
; but don't want to lose pui's coming in just at threshold
; checks upper limit on tof
    upper_limit_mass = 56
    temp_i = where ( (section LT 3) AND (e_meas GT 0.), temp_count)
    If temp_count GT 0 THEN $
       time_EQ_Emeas[temp_i] = $
            1E9*tof_dist[temp_i]*1E-5/(437.8*sqrt(e_meas[temp_i]/upper_limit_mass))
    temp_i = where (e_meas GT 0. AND e_meas LE 2000 AND tns GT time_EQ_Emeas, temp_count)
    IF temp_count GT 0 THEN BEGIN
            mass[temp_i]    = 1100	; this will be assigned Nm = 96
            m_per_q[temp_i] = 1100  ; this will be assigned Nq = 255 
    ENDIF 


;  Check Phys-2b : upper limit tof; all sections
;  Check on highest tof consistency: incident energy/charge = (pacHK+e/q+e/q_bandwidth)
;  calibration bandwidth Fwhm for ESA is 6-7%; assume half width of 15%
;  cross check using maximum m/q for pickup ions
;  TOF should be less than tof + tof_bandwidth (assume 15%)
    mq_pui_all80 = 40                 ; pui through full m/q range ssd side 80mm
    mq_pui_all58 = 100                ; pui through full m/q range section 3 58mm
    V_EQ_PUI     = 437.8*sqrt(E_EQ_PUI/mq_pui_all80)
    temp_section3 = where(section EQ 3, count_section3)
    IF count_section3 GT 0 THEN  V_EQ_PUI = 437.8*sqrt(E_EQ_PUI/mq_pui_all58)
    time_EQ_PUI = (1E-5*tof_dist/V_EQ_PUI)*1E9 * 1.50   
                  ; conv km, ns and use 50% tof half bandwidth
    temp_i = where( tns GT time_EQ_PUI, temp_count )
    IF temp_count GT 0 AND count_section3 LT 0 THEN BEGIN
   	        mass[temp_i] = 1100  	    ; this will be assigned Nm = 96 
    ENDIF
    IF temp_count GT 0 THEN BEGIN
   	        m_per_q[temp_i] = 1100  	; this will be assigned Nq = 255  
    ENDIF


;  Check Phys-3 : Upper limit for energy
;  This criteria has little or no discernable effect
;  Check on total energy consistency: incident energy = charge*(pacHK+e/q+e/q_bandwidth)
;  calibration bandwidth Fwhm for ESA is 6-7%; assume half width of 15%
;  cross check using maximum charge Fe+26 
;  Emeas should be less than Emax
    E_EQ_Fe26 = (pacHK + e_per_q*1.15)*26
    temp_i = where( (mass GT 0) AND (e_meas GT E_EQ_Fe26) AND (section LT 3), temp_count)
    IF temp_count GT 0 THEN mass[temp_i]    = 1100	; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1100  ; this will be assigned Nq = 255  
                  ; energy is higher than possible for any species at this ESA step


;  Check Phys-4 : Upper limit for energy
;  This criteria has little or no discernable effect
;  check for consistent tof and energy, based on E= 1/2 mV^2, assumes no pulse height defect
;  uses measured tof to determine speed in tof chamber
;  checks upper limit for emeas
    E_speed_Fe = (1.05*speed_tof/438)^2 * 56        ; max species iron, phd assumed 0
                                                    ; allow speed_tof to be 5% higher
                                                    ; note if speed_tof is not measured it = -1
    temp_i = where( (e_meas GT 0.) AND (speed_tof GT 0.)  AND (section LT 3) $
                AND (e_meas GT E_speed_Fe) AND (e_meas LE 2000), temp_count)
    IF temp_count GT 0 THEN BEGIN
            mass[temp_i]    = 1100	; this will be assigned Nm = 96
            m_per_q[temp_i] = 1100  ; this will be assigned Nq = 255 
    ENDIF                           ; don't expect this criteria to be triggered often

; Check Phys-5 : lower limit for energy 
; check for consistent tof and energy, based on E= 1/2 mV^2, assumes 50% pulse height defect
; checks lower limit for emeas (for emeas ne 0)
    E_speed_H = (0.95*speed_tof/438)^2 * 1 * 0.5  ; min species hydrogen, include 50% defect
                                                  ; allow 5% lower speed
                                                  ; check speed_tof (-1 means not determined)
    temp_i = where( (e_meas GT 0. AND e_meas LE 2000) AND (speed_tof GT 0.)  AND $
                    (e_meas LT E_speed_H) AND (section LT 3), temp_count)               
    IF temp_count GT 0 THEN BEGIN
            mass[temp_i]    = 1100	; this will be assigned Nm = 96
            m_per_q[temp_i] = 1100  ; this will be assigned Nq = 255 
    ENDIF 


;  Check Phys-6 : lower limit mq; all sections
    temp_i = where( m_per_q LE 0.3 AND section LT 3, temp_count)
    IF temp_count GT 0 THEN BEGIN
            mass[temp_i]    = 1100	; this will be assigned Nm = 96
            m_per_q[temp_i] = 1100  ; this will be assigned Nq = 255 
    ENDIF                           
    temp_i = where( m_per_q LE 0.3 AND section EQ 3, temp_count)
    IF temp_count GT 0 THEN BEGIN
            m_per_q[temp_i] = 1100  ; this will be assigned Nq = 255 
    ENDIF                           


 IF check EQ 2 THEN BEGIN 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; STRICT SSD ID vs RA SCATTER CLEANUP
; the following imposes strict ssd to RA alignment. Is implemented only if "check" is set to 2

;  Check pos vs ssd id to minimize scatter and/or wrong ssd triggered, use adjusted RA
;         and allow +/- two detector widths - giving 3.3% tof resolution
;         although always true, using just with solar wind algorithm 4
 
    temp_i = where( (section EQ 0 OR section EQ 1) AND $
            (ssd_id EQ 1 AND e_meas GT 0 AND position LT 16 OR position GT 30), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1) AND $
            (ssd_id EQ 2 AND e_meas GT 0 AND position LT 16 OR position GT 33), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1)  AND $
            (ssd_id EQ 3 AND e_meas GT 0 AND position LT 16 OR position GT 37), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1)  AND $
            (ssd_id EQ 4 AND e_meas GT 0 AND position LT 21 OR position GT 41), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1)  AND $
            (ssd_id EQ 5 AND e_meas GT 0 AND position LT 25 OR position GT 45), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1) AND $
            (ssd_id EQ 6 AND e_meas GT 0 AND position LT 29 OR position GT 47), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1) AND $
            (ssd_id EQ 7 AND e_meas GT 0 AND position LT 33 OR position GT 47), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255

    temp_i = where( (section EQ 0 OR section EQ 1) AND  $
            (ssd_id EQ 8 AND e_meas GT 0 AND position LT 37 OR position GT 47), temp_count)
    IF temp_count GT 0 THEN mass[temp_i] = 1500    ; this will be assigned Nm = 96
    IF temp_count GT 0 THEN m_per_q[temp_i] = 1500 ; this will be assigned Nq = 255
  
 ENDIF  ; ssd vs RA checks concluded
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; STRICT SOLAR WIND CHECK
; the following imposes strict mq limits based on speed. Is implemented only if alg=4
; and speed is known

  IF which_algorithm EQ 4 THEN BEGIN 
    print, ' For solar wind-only constraints are being checked'

;   If solar wind speed is known, then can do a tighter fit.
;   Create a m/q spectrum from the E/Q table plus solar wind speed

    ESA_MQmax_Table    = fltarr(128)
    ESA_MQmax_Table[*] = 200.
    
    IF speed_known EQ 1 THEN BEGIN
       print, ' started check against known speed, speed = ',speed_solar_wind
       proton_EQ_solar_wind = (speed_solar_wind/437.8)^2 * 1./1
    ; high speed minor ions - could be higher speed by alfven.  Ok.
    ; low speed minor ions - could have speed less than protons, adjust.
       IF proton_EQ_solar_wind LT 350 THEN proton_EQ_solar_wind = proton_EQ_solar_wind - 30.
       ESA_MQmax_Table = 1.10*e_per_q / proton_EQ_solar_wind    ; factor for upper eq bandwidth

    ;  check if already flagged mq>1000, and solar wind imposed mq upper limit              
       temp_i = where ( m_per_q LT 1000 AND m_per_q GT ESA_MQmax_Table, temp_count)
       temp_section = where (section LT 3 AND mass GT 0, count_ssd)
       IF (temp_count GT 0 AND cound_ssd GT 0 ) THEN mass[temp_i] = 1800 ; this will be assigned Nm = 96
       IF temp_count GT 0 THEN m_per_q[temp_i] = 1800 ; this will be assigned Nq = 255
       IF temp_count GT 0 THEN print, ' speed known is being utilized'       
    ENDIF

 ENDIF  ; consistencies for solar wind (Algorithm 4) do not use for pickup ions

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;CALCULATION OF NM
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Convert to a bin number
IF want_bins EQ 1 THEN BEGIN 
    nm = intarr(num_pha)
    nm[*] = 95                 ; set masses that are above highest nm bin
    FOR ii = 0, 90 DO BEGIN    ; 91 limit bins -- this loop is small compared with num_pha
        temp_i = where(mass_low_bound[ii] LE mass AND mass_low_bound[ii+1] GT mass, temp_count)
        IF temp_count GT 0 THEN nm[temp_i] = ii
    ENDFOR 
    
    
    temp_i = where(mass GE 95.0 AND mass LT 100.7, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 91  ; other routines may treat 91 like mass max 

    temp_i = where(mass GE 100.7 AND mass LT 106.75, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 92  ; other routines may treat 91 like mass max 

    temp_i = where(mass GE 106.75 AND mass LT 113.16, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 93  ; other routines may treat 91 like mass max 

    temp_i = where(mass GE 113.16 AND mass LT 119.95, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 94  ; other routines may treat 91 like mass max 

;   this leaves nm = 95 for mass gt 119.95, until next step checks above 900

;   contraint checks given large masses end up as 96
    temp_i = where(mass GE 900, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 96     
        
    temp_i = where(mass LE mass_min, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 97

    temp_i = where(tns GE t_max, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 98

    temp_i = where(tns LE t_min, temp_count) 
    IF temp_count GT 0 THEN nm[temp_i] = 98

    temp_i = where(finite(tns) NE 1, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 98

    temp_i = where(e_meas GE e_max, temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 99

    temp_i = where((e_meas EQ 0) OR (e_meas LE e_min), temp_count)
    IF temp_count GT 0 THEN nm[temp_i] = 0  ; not an error - mass zero
    
ENDIF 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;CALCULATION OF NQ
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; get bin 
half0_i = where(half EQ 0, half0_count, complement = half1_i, ncomplement = half1_count)
IF half0_count GT 0 THEN m_per_q[half0_i] = m_per_q[half0_i] 
IF want_bins EQ 1 THEN BEGIN 
    nq = intarr(num_pha)
    nq[*] = 255 ; if not found elsewhere, it will be here
    IF half0_count GT 0 THEN BEGIN
        ;mq_index[half0_i] = 1
        temp_m_q = m_per_q[half0_i]
        temp_nq = nq[half0_i]
    FOR ii = 0, 254 DO BEGIN ; 256 limit bins -- this loop is small compared with num_pha
        temp_i = $
           where(mq_low_bound[1, ii] LE temp_m_q AND mq_low_bound[1, ii+1] GT temp_m_q, temp_count) 
        IF temp_count GT 0 THEN temp_nq[temp_i] = ii
    ENDFOR 
        nq[half0_i] = temp_nq
    ENDIF 
    IF half1_count GT 0 THEN BEGIN
        ;mq_index[half1_i] = 0
        temp_m_q = m_per_q[half1_i]
        temp_nq = nq[half1_i]
    FOR ii = 0, 254 DO BEGIN ; 256 limit bins -- this loop is small compared with num_pha
        temp_i = $
           where(mq_low_bound[0, ii] LE temp_m_q AND mq_low_bound[0, ii+1] GT temp_m_q, temp_count)
        IF temp_count GT 0 THEN temp_nq[temp_i] = ii
   ENDFOR 
        nq[half1_i] = temp_nq
    ENDIF
    
; error case:
    temp_i = where(m_per_q LE 0.1, temp_count)
    IF temp_count GT 0 THEN nq[temp_i] = 255
    
ENDIF 

; Some plots for checking data, but don't use if into routine processing = 1 (takes forever)

 IF routine_check EQ 0 THEN BEGIN

 IF which_section EQ 3 THEN BEGIN  
     title = 'IonTracks_MQ-1-2-4-8-12-16-19-32_'+ cdf_base_brief
     tmax_scale = 260.
     plot_tns_vs_esastep_pha_all, nq, nm, tns, swpe,  tmax_scale,              $
                  tns_Proton_expect ,  $
                  tns_HePlus2_expect,  $
                  tns_HePlus_expect ,  $
                  tns_OPlus2_expect,   $ 
                  tns_CPlus1_expect,   $
                  tns_OPlus1_expect ,  $
                  tns_H3OPlus1_expect, $
                  tns_SPlus1_expect,   $ 
                  title

     title = 'IonTracks_MQ-1-2-4-8-16_'+ cdf_base_brief
     tmax_scale = 150.
     plot_tns_vs_esastep_pha, nq, nm, tns, swpe,  tmax_scale,                  $
                  tns_Proton_expect , tns_Proton_lower , tns_Proton_upper,     $
                  tns_HePlus_expect , tns_HePlus_lower , tns_HePlus_upper,     $
                  tns_HePlus2_expect, tns_HePlus2_lower, tns_HePlus2_upper,    $
                  tns_OPlus1_expect , tns_OPlus1_lower , tns_OPlus1_upper,     $
                  tns_OPlus2_expect, tns_OPlus2_lower  , tns_OPlus2_upper,     $ 
                  title
 ENDIF

 IF which_section LT 3 THEN BEGIN                ; plots that include energy
;   title = 'Fe_expect_'+ cdf_base_brief
;   plot_ekev_vs_tns_pha, nq, nm, e_meas, tns,                                $
;                                 tns_FePlus7_expect,  emeas_FePlus7_expect,  $
;                                 emeas_FePlus7_lower, emeas_FePlus7_upper,   $
;                                 tns_FePlus6_expect,  emeas_FePlus6_expect,  $
;                                 emeas_FePlus6_lower, emeas_FePlus6_upper, title
 
;   title = 'He_expect_'+ cdf_base_brief
;   plot_ekev_vs_tns_pha, nq, nm, e_meas, tns,                                $
;                                 tns_HePlus2_expect,  emeas_HePlus2_expect,  $
;                                 emeas_HePlus2_lower, emeas_HePlus2_upper,   $
;                                 tns_HePlus_expect,   emeas_HePlus_expect,   $
;                                 emeas_HePlus_lower,  emeas_HePlus_upper, title
;
     title = 'IonTracks_MQ-1-2-4-8-12-16-19-32_'+ cdf_base_brief
     tmax_scale = 260.
     plot_tns_vs_esastep_pha_all, nq, nm, tns, swpe,  tmax_scale,              $
                  tns_Proton_expect ,  $
                  tns_HePlus2_expect,  $
                  tns_HePlus_expect ,  $
                  tns_OPlus2_expect,   $ 
                  tns_CPlus1_expect,   $
                  tns_OPlus1_expect ,  $
                  tns_H3OPlus1_expect, $
                  tns_SPlus1_expect,   $ 
                  title

     title = 'IonTracks_H1He2O6He1Fe6_'+ cdf_base_brief
     tmax_scale = 150.
     plot_tns_vs_esastep_pha, nq, nm, tns, swpe,  tmax_scale,                  $
                  tns_Proton_expect , tns_Proton_lower , tns_Proton_upper,     $
                  tns_HePlus_expect , tns_HePlus_lower , tns_HePlus_upper,     $
                  tns_HePlus2_expect, tns_HePlus2_lower, tns_HePlus2_upper,    $
                  tns_OPlus6_expect , tns_OPlus6_lower , tns_OPlus6_upper,     $
                  tns_FePlus6_expect, tns_FePlus6_lower, tns_FePlus6_upper,    $ 
                  title


     title = 'IonTracks_He1H1He2O6Fe6_'+ cdf_base_brief
     emax_scale = 600.
     plot_ekev_vs_esastep_pha, nq, nm, e_meas, swpe,  emax_scale,                   $
                  emeas_Proton_expect , emeas_Proton_lower , emeas_Proton_upper,    $
                  emeas_HePlus_expect , emeas_HePlus_lower , emeas_HePlus_upper,    $
                  emeas_HePlus2_expect, emeas_HePlus2_lower, emeas_HePlus2_upper,   $
                  emeas_OPlus6_expect , emeas_OPlus6_lower , emeas_OPlus6_upper,    $ 
                  emeas_FePlus6_expect , emeas_FePlus6_lower , emeas_FePlus6_upper, $ 
                  title
                               
   title = 'H_expect_'+ cdf_base_brief
   plot_ekev_vs_tns_pha, nq, nm, e_meas, tns,                                $
                                 tns_Proton_expect,  emeas_Proton_expect,    $
                                 emeas_Proton_lower, emeas_Proton_upper,     $
                                 tns_Proton_expect,   emeas_Proton_expect,   $
                                 emeas_Proton_lower,  emeas_Proton_upper, title

                                
;    title = 'Oxygen1-2_'+ cdf_base_brief'
;   plot_ekev_vs_tns_pha, nq, nm, e_meas, tns,                                $
;                                 tns_OPlus1_expect,  emeas_OPlus1_expect,    $
;                                 emeas_OPlus1_lower, emeas_OPlus1_upper,     $
;                                 tns_OPlus2_expect,  emeas_OPlus2_expect,    $
;                                 emeas_OPlus2_lower, emeas_OPlus2_upper, title
 ENDIF ; section with ssd's                                
 ENDIF ; routine processing
 
END 
