; Lorna Ellis
; 08/21/2009
; circleplot.pro

; 09/28/2009 Added lorad as argument.
;            Added ytitle and yplace as keywords.

; This is a revision of Mark Conde's spiroplt.pro program which he
; used to create his Condegrams (see
; http://www.gedds.alaska.edu/spiral/).

; Notes from Mark Conde:
;The routine takes 6 positional parameters as input, plus a number of optional keyword parameters. I is called as:

;   pro spiroplt, x, y, period, pitch, yscale, phaseref, $
;       title = ti, $
;       tithick = ttk, $
;       ticolor = ticul, $
;       tifont = tifon, $
;       baseline = bl, $
;       major_ticks = majtix, $
;       minor_ticks = mintix, $
;       ticklab = tlab, $
;       data_color = datcul, $
;       axis_color = axecul, $
;       ylab = ylb, $
;       ybar = yb, $
;       ygrid = yg, $
;       spirogrid = sg, $
;       gridcolor = gc, $
;       yzero = yz, $
;       units = unts, $
;       gap_angle = gapang, $
;       label_invert = labinv, $
;       timestamp = tst, $
;       ythick = ytk, $
;       period=perlab, $
;       jd0=jd0

;The parameters are:

;x: input array of times, one time for each data point. Times are in minutes (yuck!) and I think the zero time should correspond to the oldest data point in your data.

;y: This is an input array containing the actual data values to be plotted.

;period: This specifies the number of minutes that correspond to one rotation of the spiral. I normally set it to 27 days * 1440 minutes per day.

;pitch: This specifies how "tightly wound" the spiral is. From memory, a larger pitch will mean the spiral starts closer to the middle of the page. Pitch is specified as a fraction of the radius of the outer turn of the spiral.

;yscale: This specifies the scale of the "y" deflection; it corresponds to the y-value increment that would shift the data line radially by an amount equal to the radial separation between successive "laps" of the spiral.

;phaseref: This allows the latest data in the spiral to occur at some clock angle other than the default "12-o'clock" position. I always specify a value of -1e-8 (radians) for this parameter.

;Most keywords should be obvious I hope. Note that you need to supply arrays specifying the times at which the major and minor time ticks will be drawn, in the same units as the "x" input parameter. You also need to supply an array of strings which are used as time labels for the major time ticks.

;The yzero parameter allows you to specify that the y-value that falls on the "y-axis" is something other than zero. (So for example to plot solar wind speed in the range 300 to 700 km/sec you specify yzero=300 and yrange=400. The routine will happily plot values outside this range, but they'll "spill over into the next or previous lap of the spiral.)

;If the jd0 keyword parameter is supplied, it is interpreted as the Julian day (including fractional part) of the first data point. It is then used to determine the absolute Julian day of each data point. This information is ultimately used to plot icons for the new moon and full moon times. If jd0 is not specified, the program does not know the absolute date/times of the data, and the moon icons are not drawn.

;The gap_angle parameter is used to specify the largest time interval between successive data points that will will be drawn with a connected line. Data gaps longer than this will not be connected by a line.

;The routine calls the IDL ASTRO "mphase" routine, and possibly some other library routines (I can't recall right now.)

; range (LBE added), string array to print of time range, with years.

function getspot, rad, phase, yval, cen
compile_opt strictarrsubs
xrd = rad + yval
return, cen + xrd*[sin(phase), cos(phase)]
end

pro circleplot, x, y, period, pitch, yscale, phaseref, lorad, $
                title = ti, $
                tithick = ttk, $
                ticolor = ticul, $
                tifont = tifon, $
                baseline = bl, $
                major_ticks = majtix, $
                minor_ticks = mintix, $
                ticklab = tlab, $
                data_color = datcul, $
                axis_color = axecul, $
                ylab = ylb, $
                ybar = yb, $
                ygrid = yg, $
                spirogrid = sg, $
                gridcolor = gc, $
                yzero = yz, $
                units = unts, $
                gap_angle = gapang, $
                label_invert = labinv, $
                timestamp = tst, $
                ythick = ytk, $
                period = perlab, $
                jd0 = jd0, $
                range = range, $
                ytitle = ytitle, $
                yplace = yplace
compile_opt strictarrsubs

   if n_elements(datcul) eq 0       then datcul = 0.25*!d.n_colors
   if n_elements(datcul) lt 2       then datcul = [datcul,datcul]
   if not(keyword_set(gc))          then gc     = 0.3*!d.n_colors
;  if not(keyword_set(axecul))      then axecul = 0.25*!d.n_colors
   if not(keyword_set(yz))          then yz = 0
   if not(keyword_set(gapang))      then gapang = 5.*!pi/180.
   if not(keyword_set(tst))         then    tst = 1 else tst  = 0
   if not(keyword_set(perlab))      then   perl = 0 else perl = 1
   if not(keyword_set(ytk))         then ytk    = 2
   if not(keyword_set(labinv))      then labinv = 0
   if not(keyword_set(ttk))         then ttk = 1
   if not(keyword_set(ticul))       then ticul = 0
   if not(keyword_set(tifon))       then tifon = 3

;  This routine plots time-series data in a polar spiral.

   box    = convert_coord(.999, .98, /normal, /to_device)
   pix    = min(box(0:1))
   cen    = [pix/1.98, pix/2.05]

   period = float(period)
   tz = min(x)
   if keyword_set(mintix) then tz = min([tz, mintix])
   if keyword_set(majtix) then tz = min([tz, majtix])

   npts  = n_elements(y)
   laps  = 1 ; lbe
   y2pix = pix/2*pitch/yscale

   phaseref = phaseref*!pi/180.
   if phaseref ge 0 then lophase = phaseref else lophase = abs(phaseref) - 2*!pi*laps
   ;lorad  = (pix/2)*(1 - pitch*(laps + 1))
;   lorad = 200 ; lbe

   tz = min(x)
   if keyword_set(mintix) then tz = min([tz, mintix])
   if keyword_set(majtix) then tz = min([tz, majtix])

   if keyword_set(sg) then begin
      for k=1,sg do begin
          nbl  = 180.*laps
          rad = lorad + (float(k)/sg)*yscale*y2pix
          spot_1 = getspot(rad, lophase, 0., cen)
          plots, spot_1, /device
          for j=1,nbl-1 do begin
              dx    = (x(npts-1) - tz)*float(j)/float(nbl-1) 
              phase = lophase + 2*!pi*dx/period
              ;rad   = lorad   +  pix/2*pitch*dx/period + (float(k)/sg)*yscale*y2pix
              spot  = getspot(rad, phase, 0., cen)
              plots, spot, /continue, /device, color=gc ; this plots the spiral grid (xlines)
          ENDFOR
          plots, spot_1, /continue, /device, color = gc ; complete circle
      endfor
   endif

;  This loop draws full and new moon icons on the "y=0 axis":
   if keyword_set(jd0) then begin
;      jd = findgen(900)*(max(x) - min(x))/(900.*1440d) + double(jd0)
      jd = x/1440d + double(jd0)
      xx =  x(0) + dindgen(900)*(max(x)  - min(x))/900d
      jd = jd(0) + dindgen(900)*(max(jd) - min(jd))/900d
      lopack = min(jd) - (max(jd) - (jd))
      hipack = jd + max(jd) - min(jd)
      jd = [lopack(0:898) , jd, hipack(1:*)]

      if n_elements(jd) gt 30000 then begin
         mphase, jd(0:29999), phlun
         mphase, jd(30000:*), qhlun
         phlun = [phlun, qhlun]
      endif else mphase, jd, phlun
      phlun  = smooth(phlun, 0.005*n_elements(phlun))
      ;phlun  = phlun(899:1798)
      fullmoon = where(phlun gt 0.98 and phlun - shift(phlun, 1) gt 0 and phlun - shift(phlun, -1) gt 0, nfull)
      fullmoon = fullmoon(where(fullmoon ge 899 and fullmoon le 1798)) - 899
      newmoon  = where(phlun lt 0.02 and phlun - shift(phlun, 1) lt 0 and phlun - shift(phlun, -1) lt 0, nnew)
      newmoon  = newmoon(where(newmoon ge 899 and newmoon le 1798)) - 899
      xyouts, 35, 975, 'New Moon', align=0, color=axecul, /device, charsize=0.8
      xyouts, 35, 945, 'Full Moon', align=0, color=axecul, /device, charsize=0.8

      for k=1,10 do tvcircle, (k/10.)*yscale*y2pix/5., 20, 980, color=gc, thick=2
      tvcircle, yscale*y2pix/5., 20, 950, color=gc, thick=2

      for j=0,n_elements(newmoon) - 1 do begin
          dx = (xx(newmoon(j)) - tz)
          phase = lophase + 2*!pi*dx/period
          rad   = lorad   +  pix/2*pitch*dx/period
          spot  = getspot(rad, phase, 0., cen)
          for k=1,10 do tvcircle, (k/10.)*yscale*y2pix/5., spot(0), spot(1), color=gc, thick=2
      endfor
      for j=0,n_elements(fullmoon)-1 do begin
          dx = (xx(fullmoon(j)) - tz)
          phase = lophase + 2*!pi*dx/period
          rad   = lorad   +  pix/2*pitch*dx/period
          spot  = getspot(rad, phase, 0., cen)
          tvcircle, yscale*y2pix/5., spot(0), spot(1), color=gc, thick=2
      endfor
   endif

;  This loop plots the data:
   phase = lophase + 2*!pi*(x(0) - tz)/period
   rad   = lorad   +  pix/2*pitch*(x(0) - tz)/period
   spot  = getspot(rad, phase, (y(0)-yz)*y2pix, cen)
   plots, spot, /device
   oldphase = lophase
   for j=1l,npts-1 do begin
       phase = lophase + 2*!pi*(x(j) - tz)/period
       pointcul = datcul(0) + (datcul(1) - datcul(0))*(x(j) - tz)/(period*laps)
       ctx   = 1
       if phase - oldphase gt gapang then ctx = 0
       spot  = getspot(rad, phase, (y(j)-yz)*y2pix, cen)
       plots, spot, continue = ctx, /device, color = pointcul, thick = ytk ; this plots the data
       oldphase = phase
   endfor

   if keyword_set(mintix) then BEGIN ; these plot tick marks
       rad   = lorad   + pix/2*pitch*(mintix(0) - tz)/period
       for j = 0, n_elements(mintix)-1 do begin
           phase = lophase + 2*!pi*(mintix(j) - tz)/period
           spot  = getspot(rad, phase, -pitch*pix/25., cen)
           plots, spot, /device, color = axecul 
           spot  = getspot(rad, phase, pitch*pix/25., cen)
           plots, spot, /device, color = axecul, /continue
       endfor
   endif

   if keyword_set(majtix) then begin
       rad   = lorad   + pix/2*pitch*(majtix(0) - tz)/period
      for j=0, n_elements(majtix)-1 do begin
          phase = lophase + 2*!pi*(majtix(j) - tz)/period
          spot  = getspot(rad, phase, 0., cen)
          if  keyword_set(yg) and (yg gt 0 or j eq n_elements(majtix)-1) then begin
              ygg  = abs(yg)
              spt2 = getspot(rad + yscale*y2pix, phase, 0., cen)
              plots, [spot(0), spot(1)], /device, color = gc
              plots, [spt2(0), spt2(1)], /device, color=gc, /continue, thick=1
              for k=1,ygg do begin
                  spt3 = getspot(rad + (float(k)/ygg)*yscale*y2pix, phase - .5*!pi/180, 0., cen)
                  spt4 = getspot(rad + (float(k)/ygg)*yscale*y2pix, phase + .5*!pi/180, 0., cen)
                  plots, [spt3(0), spt3(1)], /device, color=gc
                  plots, [spt4(0), spt4(1)], /device, color=gc, /continue, thick=1
              endfor
              spt3 = getspot(rad + yscale*y2pix, phase - 1.*!pi/180, 0., cen)
              spt4 = getspot(rad + yscale*y2pix, phase + 1.*!pi/180, 0., cen)
              plots, [spt3(0), spt3(1)], /device, color=gc
              plots, [spt4(0), spt4(1)], /device, color=gc, /continue, thick=1
          endif
          spot  = getspot(rad, phase, -pitch*pix/15., cen)
          plots, spot, /device, color=axecul
          spot  = getspot(rad, phase, pitch*pix/15., cen)
          plots, spot, /device, color=axecul, /continue
      endfor
   endif
   if keyword_set(tlab) then begin
       rad   = lorad   + pix/2*pitch*(majtix(0) - tz)/period
       for j = 0, n_elements(majtix)-2 do BEGIN
           yoff  = -0.8*pitch*pix/4
;---------Advance tick labels by 0.3 days so they fall entirely within their "own" day:
           phase = lophase + 2*!pi*(majtix(j) - tz + (.5*1440))/period
          ;phase = lophase + 2*!pi*(majtix(j) - tz)/period
          ;phase = lophase + 2*!pi*(0.6*1440 + 0.9978*majtix(j) - tz)/period
           pang  = atan(sin(phase), cos(phase))
           pang  = -180.*pang/!pi
           if (pang lt -90 or pang gt 90) and labinv then begin
               pang = pang + 180
               yoff = yoff/2
           endif
           spot  = getspot(rad, phase, yoff, cen)
           xyouts, spot(0), spot(1), tlab(j), align = 0.5, color = axecul, orientation = pang, /device
       endfor
   endif

   if keyword_set(bl) then begin
      nbl  = 180.*laps
      spot_1 = getspot(lorad, lophase, 0., cen)
      plots, spot_1, /device
      for j=1,nbl-1 do begin
          dx    = (x(npts-1) - tz)*float(j)/float(nbl-1)
          phase = lophase + 2*!pi*dx/period
          spot  = getspot(lorad, phase, 0., cen)
          plots, spot, /continue, /device, color = axecul
      endfor
      plots, spot_1, /continue, /device, color = axecul
   endif

   if keyword_set(ti) then begin
          fnt = '!' + strcompress(string(tifon), /remove_all)
          xyouts, cen(0), cen(1) + (pix/2)*1.5*pitch, fnt + ti + '!3', $
                  align=0.5, color=ticul, /device, charsize=2.0, charthick=ttk
   endif

      IF keyword_set(range) THEN BEGIN
          range_string = range[0]+' - '+range[1]
          xyouts, cen[0], cen[1]+25, range_string, align = 0.5, color = ticul, /device, charsize = 1.5
      ENDIF 

   if keyword_set(yb) then begin
          plots, [cen(0)-pix/30, cen(1) - pix/2*1.5*pitch], /device, color=axecul
          plots, [cen(0)-pix/30, cen(1) - pix/2*1.5*pitch - yscale*y2pix], /device, color=axecul, /continue, thick=2
          plots, [cen(0)-pix/30 - pix/80, cen(1) - pix/2*1.5*pitch], /device, color=axecul
          plots, [cen(0)-pix/30 + pix/80, cen(1) - pix/2*1.5*pitch], /device, color=axecul, /continue, thick=2
          plots, [cen(0)-pix/30 - pix/80, cen(1) - pix/2*1.5*pitch - yscale*y2pix], /device, color=axecul
          plots, [cen(0)-pix/30 + pix/80, cen(1) - pix/2*1.5*pitch - yscale*y2pix], /device, color=axecul, /continue, thick=2
          legend = string(fix(yz)) + " "
          if keyword_set(unts) then legend = legend + unts
          legend = strtrim(legend, 2)
          xyouts, cen(0), cen(1) - pix/2*1.5*pitch - yscale*y2pix, legend, align=0, color=axecul, /device, charsize=1
          legend = string(fix(yz+yscale)) + " "
          if keyword_set(unts) then legend = legend + unts
          legend = strtrim(legend, 2)
          xyouts, cen(0), cen(1) - pix/2*1.5*pitch, legend, align=0, color=axecul, /device, charsize=1
      ENDIF

      IF keyword_set(yplace) THEN BEGIN
          CASE yplace OF
              1: xyouts, cen[0], cen[1]- 50,   ytitle, align = 0.5, color = datcul, /device, charsize = 1
              2: xyouts, cen[0], cen[1]- 75, ytitle, align = 0.5, color = datcul, /device, charsize = 1
              3: xyouts, cen[0], cen[1]-100, ytitle, align = 0.5, color = datcul, /device, charsize = 1
          ENDCASE
      ENDIF 

   if tst gt 0 then begin
      stm = systime(/ut)
      cln = strpos(stm, ':')
      lbl = 'Condegram plotted ' + strmid(stm, 4, 6) + ', ' + strmid(stm, strlen(stm)-4, 4) + ' at ' + $
            strmid(stm, cln-2, 5) + ' UT'
      xyouts, box(0)-5, 5, lbl, align=1, color=axecul, /device, charsize=0.8
      ;lbl = 'GI-UAF ' + strmid(stm, cln+7, 4)
      ;xyouts, 5, 5, lbl, align=0, color=axecul, /device, charsize=0.8
   endif

   if perl gt 0 then begin
      xyouts, box(0)-5, box(1) - 10, perlab, align=1, color=axecul, /device, charsize=0.8
   endif

  end
