;+
; PROCEDURE: read_schan_conv
;
; PURPOSE: to get the schan conversion factor from calibration files
;
; INPUT: sat   - S/C id - 'A' or 'B'
;        dat   - data structure
;
; OUTPUT: schan_conv: the schan conversion factor for each calibration file
;         cal_start : start time for each calibration file (as double)
;
; CREATED BY: L. Ellis
;
; LAST MODIFICATION:
;   01/11/2008 Created
;
; MODIFICATION LIST:
;
;-

PRO read_schan_conv, sat, prod, dat, units
compile_opt strictarrsubs

get_timespan, ts
ts_str = time_string(ts)
start_time = ts_str[0] ; need to change format to SolarSoft time format
strput, start_time, 'T', 10
start_time = start_time+'.000'
stop_time = ts_str[0] ; need to change format to SolarSoft time format
strput, stop_time, 'T', 10
stop_time = stop_time+'.000'

; find most recent catalog file
temp_files = file_search('$PLACAL/*.cal', count = num_files)
version = 1
catalog_file = ''
FOR ii = 0, num_files-1 DO BEGIN 
    temp_file = file_basename(temp_files[ii])
    temp_version = fix(strmid(temp_file, 21, 2))
    temp_sat = strmid(temp_file, 2, 1)
    IF temp_version GE version AND temp_sat EQ sat THEN BEGIN
        version = temp_version
        catalog_file = temp_files[ii]
    ENDIF 
ENDFOR 

; find calibration file(s)
openr, cat_lun, catalog_file, /get_lun
cat_files_found = 0
cat_files = ['']
finished = 0
line = ''
mom_cal_start = [-2D]           ; this first element will be removed
mom_cal_stop = [-2D]            ; this first element will be removed
WHILE eof(cat_lun) EQ 0 AND finished EQ 0 DO BEGIN ; need this loop to get to MOM
    readf, cat_lun, line
    parts_current = strsplit(line, /extract)
    IF parts_current[0] EQ 'MOM' THEN BEGIN 
        
        WHILE eof(cat_lun) EQ 0 AND finished EQ 0 DO BEGIN 
            readf, cat_lun, line
            parts_next = strsplit(line, /extract)
            IF parts_next[0] NE 'MOM' THEN BEGIN ; finished MOM section
                finished = 1
                cat_files_found = cat_files_found + 1
                cat_files = [cat_files, parts_current[2]]
                mom_cal_start = [mom_cal_start, time_double(parts_current[1])]
                mom_cal_stop = [mom_cal_stop, -1D]
            ENDIF ELSE BEGIN    ; current and next are both MOM
                IF start_time GE time_double(parts_next[1]) THEN parts_current = parts_next $ ; skip current file
                ELSE BEGIN 
                    cat_files_found = cat_files_found + 1
                    cat_files = [cat_files, parts_current[2]]
                    mom_cal_start = [mom_cal_start, time_double(parts_current[1])]
                    IF stop_time LT time_double(parts_next[1]) THEN BEGIN ; skip rest of files
                        finished = 1 
                        mom_cal_stop = [mom_cal_stop, -1D]
                    ENDIF ELSE BEGIN
                        mom_cal_stop = [mom_cal_stop, time_double(parts_next[1])]
                        parts_current = parts_next
                    ENDELSE 
                ENDELSE 
            ENDELSE 
        ENDWHILE 
    ENDIF 
ENDWHILE 

IF cat_files_found GT 0 THEN cat_files = cat_files[1:cat_files_found]
close, cat_lun
free_lun, cat_lun

; read calibration files
IF cat_files_found GT 0 THEN BEGIN 
    geom = dblarr(cat_files_found, 2) ; M,S
    FOR ii = 0, cat_files_found-1 DO BEGIN 
        filename = '$PLACAL/'+cat_files[ii]
        openr, cal_lun, filename, /get_lun
        found_geom_factor = 0
        WHILE eof(cal_lun) EQ 0 AND found_geom_factor EQ 0 DO BEGIN 
            readf, cal_lun, line
            parts = strsplit(line, /extract)
            IF parts[0] EQ 'geom_factor' THEN BEGIN 
                IF parts[1] NE 'S' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE geom[ii, 1] = double(parts[2])
                readf, cal_lun, line
                parts = strsplit(line, /extract)
                IF parts[0] NE 'geom_factor' OR parts[1] NE 'M' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE geom[ii, 0] = double(parts[2])
                found_geom_factor = 1
            ENDIF
        ENDWHILE 
        close, cal_lun
        free_lun, cal_lun
    ENDFOR
ENDIF 

schan_conv = geom[*, 0]/geom[*, 1]
cal_start = mom_cal_start[1:n_elements(mom_cal_start)-1]
num_cal = n_elements(cal_start)
SWITCH prod OF 
    'h_alpha': BEGIN
        ep = dat.energy_peak_i
        dat.units_name = units
        temp_i = where(ep LT 4, count)  ; deal with end cases
        IF count GT 0 THEN ep[temp_i] = 4
        temp_i = where(ep GT 127-15, count) 
        IF count GT 0 THEN ep[temp_i] = 112
        FOR ii = 0, num_cal-1 DO BEGIN ; for if more than one calibration file
            IF ii EQ num_cal-1 THEN $
              temp_i = where(dat.time GE cal_start[ii] AND ep GE dat.s_chan_i-4, count) $
            ELSE $
              temp_i = where(dat.time GE cal_start[ii] AND dat.time LT cal_start[ii+1] AND ep GE dat.s_chan_i-15, count)
            IF count GT 0 THEN BEGIN
                FOR jj = 0, count-1 DO BEGIN ; go through times
                    energy_indices = indgen(20)+ep[temp_i[jj]]-15 ; create array of esa steps used at this time
                    temp_j = where(energy_indices GE dat.s_chan_i[temp_i[jj]], count_j)
                    IF count_j GT 0 THEN dat.data[temp_i[jj], *, *, temp_j] = dat.data[temp_i[jj], *, *, temp_j] * schan_conv[ii]
                ENDFOR
            ENDIF 
        ENDFOR
        BREAK 
    END 
    'h+peak':
    'he++peak':
    'he++tcr': BEGIN
        ep = dat.energy_peak_i
        dat.units_name = units
        temp_i = where(ep LT 4, count)  ; deal with end cases
        IF count GT 0 THEN ep[temp_i] = 4
        temp_i = where(ep GT 127-5, count) 
        IF count GT 0 THEN ep[temp_i] = 122
        FOR ii = 0, num_cal-1 DO BEGIN ; for if more than one calibration file
            IF ii EQ num_cal-1 THEN $
              temp_i = where(dat.time GE cal_start[ii] AND ep GE dat.s_chan_i-4, count) $
            ELSE $
              temp_i = where(dat.time GE cal_start[ii] AND dat.time LT cal_start[ii+1] AND ep GE dat.s_chan_i-5, count)
            IF count GT 0 THEN BEGIN
                FOR jj = 0, count-1 DO BEGIN ; go through times
                    energy_indices = indgen(10)+ep[temp_i[jj]]-5 ; create array of esa steps used at this time
                    temp_j = where(energy_indices GE dat.s_chan_i[temp_i[jj]], count_j)
                    IF count_j GT 0 THEN dat.data[temp_i[jj], *, *, temp_j] = dat.data[temp_i[jj], *, *, temp_j] * schan_conv[ii]
                ENDFOR
            ENDIF 
        ENDFOR
        BREAK 
    END 
    'sw_z>2_h': BEGIN 
        dat.units_name = units
        FOR ii = 0, num_cal-1 DO BEGIN ; for if more than one calibration file
            IF ii EQ num_cal-1 THEN $
              temp_i = where(dat.time GE cal_start[ii], count) $
            ELSE $
              temp_i = where(dat.time GE cal_start[ii] AND dat.time LT cal_start[ii+1], count)
            IF count GT 0 THEN BEGIN
                FOR jj = 0, count-1 DO BEGIN ; go through times
                    temp_schan = dat.s_chan_i[temp_i[jj]]
                    IF temp_schan LE 127 THEN dat.data[temp_i[jj], *, *, temp_schan:127] = $
                      dat.data[temp_i[jj], *, *, temp_schan:127] * schan_conv[ii]
                ENDFOR
            ENDIF 
        ENDFOR
        BREAK 
    END 
    'sw_z>2_l':
    'sw_priority': 
    'wap_ssd_tcr':
    'wap_ssd_dcr':
    'wap_no_ssd_dcr': BEGIN
        dat.units_name = units
        FOR ii = 0, num_cal-1 DO BEGIN ; for if more than one calibration file
            IF ii EQ num_cal-1 THEN $
              temp_i = where(dat.time GE cal_start[ii], count) $
            ELSE $
              temp_i = where(dat.time GE cal_start[ii] AND dat.time LT cal_start[ii+1], count)
            IF count GT 0 THEN BEGIN
                FOR jj = 0, count-1 DO BEGIN ; go through times
                    temp_schan = dat.s_chan_i[temp_i[jj]]
                    IF temp_schan LE 127 THEN dat.data[temp_i[jj], *, temp_schan:127] = dat.data[temp_i[jj], *, temp_schan:127] * schan_conv[ii]
                ENDFOR
            ENDIF 
        ENDFOR
        BREAK 
    END 
    'wap_priority_ssd':
    'wap_priority_no_ssd': BEGIN 
        dat.units_name = units
        FOR ii = 0, num_cal-1 DO BEGIN ; for if more than one calibration file
            IF ii EQ num_cal-1 THEN $
              temp_i = where(dat.time GE cal_start[ii], count) $
            ELSE $
              temp_i = where(dat.time GE cal_start[ii] AND dat.time LT cal_start[ii+1], count)
            IF count GT 0 THEN BEGIN
                FOR jj = 0, count-1 DO BEGIN ; go through times
                    temp_schan = dat.s_chan_i[temp_i[jj]]
                    IF temp_schan LE 127 THEN dat.data[temp_i[jj], temp_schan:127] = dat.data[temp_i[jj], temp_schan:127] * schan_conv[ii]
                ENDFOR
            ENDIF 
        ENDFOR
        BREAK 
    END 
    ELSE: stop
ENDSWITCH  
END 
