;+
;PROCEDURE: pl_read_cdf_mom
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch: 0(for variable), 1, or 5 minute resolution
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;    02/07/2007: LBE changed fillval for moments
;    02/16/2007: LBE added _mom to name
;    03/28/2007: LBE added check to make sure have any data (check
;                maxrec in read_product_mom), and set file_name to ''
;                if no records
;    04/10/2007: LBE changed name of check_indices, and fixed it,
;                changed index variable to triplets from pairs
;    07/10/2007: LBE fixed remove_fill, and made data_index into longs
;    09/19/2007: LBE Added cdf_close
;    02/11/2008: LBE Get last minute. Account for 2 timestamps in one minute.
;-
PRO read_product_mom, cdf_id, prod_name, data_var, data_index, epoch_maxrec
compile_opt strictarrsubs
; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 
cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
IF varinfo.maxrec GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec
IF maxrec GT 0 THEN BEGIN 
    cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
    num_dim = size(data_var, /n_dimensions)
    CASE num_dim OF 
        1: BEGIN
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[data_index[kk]:(data_index[kk]+data_index[kk+2]-1)] = $
                  temp_data[data_index[kk+1]:(data_index[kk+1]+data_index[kk+2]-1)]
                kk = kk+2
            ENDFOR 
        END 
        2: BEGIN
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+2]-1)] = $
                  temp_data[*, data_index[kk+1]:(data_index[kk+1]+data_index[kk+2]-1)]
                kk = kk+2
            ENDFOR 
        END 
        3: BEGIN
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+2]-1)] = $
                  temp_data[*, *, data_index[kk+1]:(data_index[kk+1]+data_index[kk+2]-1)]
                kk = kk+2
            ENDFOR 
        END 
        ELSE: BEGIN 
            print, "Lorna: Add more dimensions"
            stop
        END
    ENDCASE 
ENDIF ELSE BEGIN
    data_index = -1
ENDELSE
END 

PRO remove_fill_mom, fill_val, data, error
compile_opt strictarrsubs
; fill_val: fill_value to be replaced by NaN
; data: array of data values
; error: array that matches data array, 1 means error
indice = where(float(data) EQ float(fill_val), count) ; I don't understand why the float is necessary
FOR ii = 0, count-1 DO BEGIN
    data[indice[ii]] = !values.f_nan
ENDFOR 
indice = where(error EQ 1, count)
FOR ii = 0, count-1 DO BEGIN
    IF size(data, /n_dimensions) GT 1 THEN $
      data[*, indice[ii]] = !values.f_nan $
    ELSE data[indice[ii]] = !values.f_nan 
ENDFOR 
END 


PRO init_mom, sat, num_records1, mom_initialized
compile_opt strictarrsubs
; initialize all the moments variabales (in com_mom)
COMMON com_moment
COMMON share1
CASE sat OF
    'A': BEGIN 
        mom_meta_a             = intarr(4, num_records1)
        mom_density_main_a     = dblarr(num_records1)
        mom_density_s_a        = dblarr(num_records1)
        mom_velocity_main_a    = dblarr(3, num_records1)
        mom_velocity_s_a       = dblarr(3, num_records1)
        mom_heat_flux_main_a   = dblarr(3, num_records1)
        mom_heat_flux_s_a      = dblarr(3, num_records1)
        mom_temperature_main_a = dblarr(6, num_records1)
        mom_temperature_s_a    = dblarr(6, num_records1)
        error1_a               = intarr(num_records1)
        mom_meta_a[*, *]             = !values.f_nan
        mom_density_main_a[*]        = !values.f_nan
        mom_density_s_a[*]           = !values.f_nan
        mom_velocity_main_a[*, *]    = !values.f_nan
        mom_velocity_s_a[*, *]       = !values.f_nan
        mom_heat_flux_main_a[*, *]   = !values.f_nan
        mom_heat_flux_s_a[*, *]      = !values.f_nan
        mom_temperature_main_a[*, *] = !values.f_nan
        mom_temperature_s_a[*, *]    = !values.f_nan
    END 
    'B': BEGIN 
        mom_meta_b             = intarr(4, num_records1)
        mom_density_main_b     = dblarr(num_records1)
        mom_density_s_b        = dblarr(num_records1)
        mom_velocity_main_b    = dblarr(3, num_records1)
        mom_velocity_s_b       = dblarr(3, num_records1)
        mom_heat_flux_main_b   = dblarr(3, num_records1)
        mom_heat_flux_s_b      = dblarr(3, num_records1)
        mom_temperature_main_b = dblarr(6, num_records1)
        mom_temperature_s_b    = dblarr(6, num_records1)
        error1_b               = intarr(num_records1)
        mom_meta_b[*, *]             = !values.f_nan
        mom_density_main_b[*]        = !values.f_nan
        mom_density_s_b[*]           = !values.f_nan
        mom_velocity_main_b[*, *]    = !values.f_nan
        mom_velocity_s_b[*, *]       = !values.f_nan
        mom_heat_flux_main_b[*, *]   = !values.f_nan
        mom_heat_flux_s_b[*, *]      = !values.f_nan
        mom_temperature_main_b[*, *] = !values.f_nan
        mom_temperature_s_b[*, *]    = !values.f_nan
    END 
ENDCASE 
mom_initialized = 1
END 

PRO read_moments, cdf_id, sat, mom_read, data1_index, epoch1maxrec
compile_opt strictarrsubs
COMMON com_moment
COMMON share1
CASE sat OF 
    'A': BEGIN
        read_product_mom, cdf_id, 'moment_meta',      mom_meta_a,             data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'density_main',     mom_density_main_a,     data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'density_s',        mom_density_s_a,        data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'velocity_main',    mom_velocity_main_a,    data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'velocity_s',       mom_velocity_s_a,       data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'heat_flux_main',   mom_heat_flux_main_a,   data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'heat_flux_s',      mom_heat_flux_s_a,      data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'temperature_main', mom_temperature_main_a, data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'temperature_s',    mom_temperature_s_a,    data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'error1',           error1_a,               data1_index, epoch1maxrec+1
        remove_fill_mom, double(-1e31), mom_meta_a, error1_a
        remove_fill_mom, double(-1e31), mom_density_main_a, error1_a
        remove_fill_mom, double(-1e31), mom_density_s_a, error1_a
        remove_fill_mom, double(-1e31), mom_velocity_main_a, error1_a
        remove_fill_mom, double(-1e31), mom_velocity_s_a, error1_a
        remove_fill_mom, double(-1e31), mom_heat_flux_main_a, error1_a
        remove_fill_mom, double(-1e31), mom_heat_flux_s_a, error1_a
        remove_fill_mom, double(-1e31), mom_temperature_main_a, error1_a
        remove_fill_mom, double(-1e31), mom_temperature_s_a, error1_a
    END
    'B': BEGIN
        read_product_mom, cdf_id, 'moment_meta',      mom_meta_b,             data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'density_main',     mom_density_main_b,     data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'density_s',        mom_density_s_b,        data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'velocity_main',    mom_velocity_main_b,    data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'velocity_s',       mom_velocity_s_b,       data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'heat_flux_main',   mom_heat_flux_main_b,   data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'heat_flux_s',      mom_heat_flux_s_b,      data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'temperature_main', mom_temperature_main_b, data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'temperature_s',    mom_temperature_s_b,    data1_index, epoch1maxrec+1
        read_product_mom, cdf_id, 'error1',           error1_b,               data1_index, epoch1maxrec+1
        remove_fill_mom, double(-1e31), mom_meta_b, error1_b
        remove_fill_mom, double(-1e31), mom_density_main_b, error1_b
        remove_fill_mom, double(-1e31), mom_density_s_b, error1_b
        remove_fill_mom, double(-1e31), mom_velocity_main_b, error1_b
        remove_fill_mom, double(-1e31), mom_velocity_s_b, error1_b
        remove_fill_mom, double(-1e31), mom_heat_flux_main_b, error1_b
        remove_fill_mom, double(-1e31), mom_heat_flux_s_b, error1_b
        remove_fill_mom, double(-1e31), mom_temperature_main_b, error1_b
        remove_fill_mom, double(-1e31), mom_temperature_s_b, error1_b
    END
ENDCASE
mom_read = 1
END 

PRO check_indices_mom, index, epoch, resolution, file_num
compile_opt strictarrsubs
; checks for missing records from the cdf epoch variable
; index: array of triplets [start_index_time,start_index_cdf,num_records] (could be [0,0,10,15,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1 or 5 depending on epoch (may need to add variable)
; file_num: index of this file in the file array
first_index = long(file_num)*(1440/resolution)
start_index_time = 0
num_records = 0
in_valid_area = 0
ii = 0L
index = [-1L]
in_synch = 1 ; needed for when duplicate time-stamps -- only takes first
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN  
            FOR min = 0, 59 DO BEGIN 
                WHILE in_synch EQ 1 DO BEGIN 
                    IF ii LT n_elements(epoch) THEN BEGIN 
                        time = time_struct(epoch[ii])
                        IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                            IF in_valid_area EQ 0 THEN BEGIN
                                start_index_time = first_index+(hour*60)+min
                                start_index_cdf = ii
                                num_records = 1
                                in_valid_area = 1
                            ENDIF ELSE BEGIN ; already in valid area
                                num_records = num_records+1
                            ENDELSE 
                            ii = ii+1
                            in_synch = 0
                        ENDIF ELSE BEGIN
                            IF ii GT 0 THEN BEGIN ; check for two valid timestamps in the minute
                                last_time = time_struct(epoch[ii-1])
                                IF last_time.hour EQ time.hour AND last_time.min EQ time.min AND last_time.sec NE time.sec THEN BEGIN 
                                    num_records = num_records+1
                                    ii = ii+1
                                    in_synch = 0
                                    min = min-1 ; do this minute again
                                ENDIF ELSE BEGIN 
                                    IF in_valid_area EQ 1 THEN BEGIN
                                        in_valid_area = 0
                                        IF index[0] EQ -1 THEN index = $
                                          [long(start_index_time), long(start_index_cdf), long(num_records)] $
                                        ELSE index = [index, long(start_index_time), long(start_index_cdf), long(num_records)]
                                    ENDIF 
                                    IF time.hour LT hour OR (hour EQ time.hour AND time.min LT min) THEN ii = ii+1 ELSE in_synch = 0
                                ENDELSE 
                            ENDIF ELSE IF time.hour LT hour OR (hour EQ time.hour AND time.min LT min) THEN ii = ii+1 ELSE in_synch = 0
                        ENDELSE 
                    ENDIF ELSE BEGIN 
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = $
                              [long(start_index_time), long(start_index_cdf), long(num_records)] $
                            ELSE index = [index, long(start_index_time), long(start_index_cdf), long(num_records)]                
                        ENDIF 
                        in_synch = 0
                    ENDELSE 
                ENDWHILE 
                in_synch = 1
            ENDFOR 
        ENDFOR 
        IF in_valid_area EQ 1 THEN BEGIN
            IF index[0] EQ -1 THEN index = $
              [long(start_index_time), long(start_index_cdf), long(num_records)] $
            ELSE index = [index, long(start_index_time), long(start_index_cdf), long(num_records)]                
        ENDIF 
    END
    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf_mom, sat, files, products, which_epoch
compile_opt strictarrsubs
COMMON share1
COMMON com_moment
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat

; initialize output variables (from common block)
need_epoch1 = 0
need_epoch5 = 0
FOR ii = 0, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
        1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
        END 
        5: BEGIN 
            num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 1
        END 
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 
mom_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
        'D'      : 
        'VX'     : 
        'VY'     : 
        'VZ'     : 
        'VXYZ'   : 
        'VT'     : 
        'TXX'    : 
        'TYY'    : 
        'TZZ'    : 
        'TXXYYZZ':
        'PXX'    : 
        'PYY'    : 
        'PZZ'    : 
        'PXXYYZZ': 
        'EX'     :  
        'EY'     : 
        'EZ'     : 
        'HX'     : 
        'HY'     :
        'HZ'     : IF mom_initialized EQ 0 THEN init_mom, sat, num_records1, mom_initialized
    ENDSWITCH 
ENDFOR 

; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch1', temp_epoch, rec_count = epoch1info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_mom, data1_index, converted_epoch, which_epoch, ii
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+2]-1)] = $
                              converted_epoch[data1_index[kk+1]:(data1_index[kk+1]+data1_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR 
                    ENDIF ELSE files[ii] = ''
                END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_a = epoch5 
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
        ENDELSE 

                                ; get other data
        mom_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
                'D'      : 
                'VX'     : 
                'VY'     : 
                'VZ'     : 
                'VXYZ'   : 
                'VT'     : 
                'TXX'    : 
                'TYY'    : 
                'TZZ'    : 
                'TXXYYZZ': 
                'PXX'    : 
                'PYY'    : 
                'PZZ'    : 
                'PXXYYZZ': 
                'EX'     : 
                'EY'     : 
                'EZ'     : 
                'HX'     : 
                'HY'     : 
                'HZ'     : BEGIN 
                    IF mom_read EQ 0 THEN read_moments, cdf_id, sat, mom_read, data1_index, epoch1info.maxrec
                    BREAK
                END 
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
        cdf_close, cdf_id
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
    IF need_epoch5 EQ 1 THEN data5_index_a = data5_index 
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
    IF need_epoch5 EQ 1 THEN data5_index_b = data5_index
ENDELSE 
END 
