;+
; PROCEDURE: pl_calc_alpha
; PURPOSE:
;  This creates tplot variables for alpha data.
;
; INPUT
; PARAMETERS: sats     : Array of sats wanted ('A' or 'B' or both)
;             sources  : 00: SW-all                 : h_alpha
;                        01: SW-H(D)                : h+peak
;                        02: SW-alpha(D)            : he++peak
;                        03: SW-alpha(T)            : he++trc
;                        04: SW PHA PRIORITY RATES  : sw_priority_00
;                        05: SW PHA PRIORITY RATES  : sw_priority_01
;         
;             functions: 00: f(v) = 70*EXP(.000003*v/SQRT(2))
;                        01: Kristin's Efficiency table
;                        02: f(v) = 120/((-6.30369334716773E-19*(v^3))-(8.63692363656727E-13*(v^2))+(5.9564934852831E-7*v)+(6.59619620444739E-1))
;             ave_time : time (in minutes) over which to create alpha products
;
; CREATED BY: L. Ellis
;
; MODIFICATION HISTORY:
;    11/07/2008: LBE: Created
;    11/17/2008: LBE: When read vel, divide by sqrt(2)
;    01/15/2009: LBE: Added which_channel
;    02/04/2009: LBE: Changed to _peak_esa is after alpha fit.
;                     Changed fit to +/- 4 (from -2, +4)
;    02/09/2009: LBE: Changed fit to -12, +4
;    04/03/2009: LBE: Changed velocity_bulk to bulk_speed
;    05/14/2009: LBE: Changed so which_channel comes from alpha_peak_esa rather than expected peak
;    05/19/2009: LBE: Adjusted so flux_t is beginning of time interval
;    05/27/2009: LBE: Add Toni_files keyword. Changes so separate esa files for s and main channels.
;    06/09/2009: LBE: Added check for non-finite schan.
;    06/10/2009: LBE: Added check for -1 schan in ew.
;    06/15/2009: LBE: Multiply density by 32 for unknown reason.
;    06/19/2009: LBE: Check for when schan didn't switch.
;    06/30/2009: LBE: Changed from Counts to c/s.
;    07/08/2009: LBE: Changed fit to -12, +6
;    07/09/2009: LBE: Added Kristin's ra efficiency tables.
;    07/21/2009: LBE: Added background subtraction.
;    07/22/2009: LBE: Made background subtraction an option.
;    07/27/2009: LBE: Added check for if no sw_pri data.
;    10/09/2009: LBE: Changed from leakage_energy_per_charge.txt to 8/17-21/09 ESA tables from Kristin.
;    11/06/2009: LBE: Drop EW data if necessary to line up with flux_t
;                     NB: EW time may not match flux_t
;    11/13/2009: LBE: Remove data from ESA step of schan switch.
;    11/16/2009: LBE: Remove data from next ESA step as well.
;    11/19/2009: LBE: Added background subtraction to schan-only and main_only.
;    11/20/2009: LBE: Make each esa step "pure" schan or main chan.
;    11/25/2009: LBE: Changed from proton to alpha mass in temperature calculation.
;    11/30/2009: LBE: Fixed bug so set num_summed to 0 when remove counts in schan.
;                     Now store alpha_efficiency.
;    12/01/2009: LBE: Do proton subtraction to schan_only and main_only.
;    12/11/2009: LBE: Add lots of code to have better measure_errors.
;    12/21/2009: LBE: No longer use average_tplot_variable; so average within each hour.
;    12/22/2009: LBE: Multiply background error by 16.
;    12/23/2009: LBE: Fix bug where multiply rel error by dist twice.
;                     Fix measure_errors for when not just background subtraction.
;    01/11/2010: LBE: Add WAP to Pri.
;    01/12/2010: LBE: If fit fails, redo with shifted dist.
;                     Fix bug so which_channel is int.
;    01/13/2010: LBE: Change meaning of alpha_flag. 0=first fit,
;                     1=second fit, 2= both fits failed
;    01/15/2010: LBE: Fixed bug where schan_esa_hour_d was actually ave schan for whole day, not one hour
;                     Added which_channel2 
;                        0 = Main  ( 0 -  3 steps of dist in schan)
;                        1 = Schan (10 - 13 steps of dist in schan)
;                        2 = Mixed ( 4 -  9 steps of dist in schan)
;    02/01/2010: LBE: Trying new table "Revised_alpha_schan_Table_012910.txt
;    02/03/2010: LBE: Streamline A. Use Pri1 instead of svalid for B efficiency.
;    02/10/2010: LBE: When labeled Schan, only  use steps in schan (for A).
;    02/12/2010: LBE: Subtract aberration.
;    03/05/2010: LBE: Added which_channel2 = 3 (outside expected range).
;                     Removed old parts of code that are no longer used.
;                     When labeled Main, only use steps in main.
;                     When labeled schan, only use steps in schan (for both, not just A).
;    03/08/2010: LBE: Changed vel to vel_inst and vel_inert.
;    05/07/2010: LBE: For A: Add vel dependent ra eff. Change schan adj.
;    05/10/2010: LBE: Added outlier checks.
;    05/18/2010: LBE: For B: Add vel dependent ra eff. Change schan adj.
;    06/16/2010: LBE: For B: Changed ra eff and schan adj per Lynn's 6/8 email. 
;-

PRO read_ra_eff, file, ra_eff
compile_opt strictarrsubs
table_location = '~/splat/Tables/' ; maui
path = table_location+file
ra_eff = dblarr(128)            ; Efficiency tables have one value for each ESA step
openr, in_lun, path, /GET_LUN
line = ","
readf, in_lun, line           ; read past first line of header
FOR i = 0, 127 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    ra_eff[i] = parts[3]			
ENDFOR
close, in_lun
free_lun, in_lun
END 



PRO pl_calc_alpha, sats, sources, functions, ave_time, bkgd_sub = bkgd_sub, proton_sub = proton_sub
compile_opt strictarrsubs

FOR ii = 0, n_elements(sats)-1 DO BEGIN
    sat = sats[ii]

    ; schan energy table
    IF sat EQ 'A' THEN ESA_file_s = '~/splat/Tables/Revised_alpha_schan_Table_012910.txt' $ ;ESA_table_A_2009_08_17.txt' 
    ELSE ESA_file_s = '~/splat/Tables/ESA_Schan_Alpha_Table_B_20100615.txt'
                                ;ESA_Schan_Table_B_20100203.txt' ;ESA_table_B_2009_08_21.txt' 
    openr, in_lun, ESA_file_s, /GET_LUN
    line = " "
    alpha_vel_table_s = fltarr(128)
    readf, in_lun, line         ; read past first line
    IF sat EQ 'A' THEN BEGIN 
        readf, in_lun, line     ; read past first line
        readf, in_lun, line     ; Revised has 3 lines of header
    ENDIF 
    FOR i = 0, 127 DO BEGIN
        readf, in_lun, line
        parts = STRSPLIT(line, /EXTRACT) 
        IF sat EQ 'A' THEN BEGIN
            IF keyword_set(toni_files) THEN alpha_vel_table_s[i] = 439.28*1000*sqrt(float(parts[1])*2/4.0026) $ ; convert E/Q to V (m/s)
            ELSE alpha_vel_table_s[i] = float(parts[4]) ; alpha V (m/s)
        ENDIF ELSE alpha_vel_table_s[i] = float(parts[1]) ; alpha V (m/s) (was parts[3])
    ENDFOR
    close, in_lun
    free_lun, in_lun
    ; main energy table
    ESA_file_m = '~/splat/Tables/main_esa_table_20090527.txt' 
    openr, in_lun, ESA_file_m, /GET_LUN
    line = " "
    alpha_vel_table_m = fltarr(128)
    readf, in_lun, line         ; read past first line
    FOR i = 0, 127 DO BEGIN
        readf, in_lun, line
        parts = STRSPLIT(line, /EXTRACT)
        alpha_vel_table_m[i] = 439.28*1000*sqrt(float(parts[1])*2/4.0026) ; convert E/Q to V (m/s)
    ENDFOR
    close, in_lun
    free_lun, in_lun

    FOR jj = 0, n_elements(sources)-1 DO BEGIN 
        ; monitor rates
        mon_spec_prod = ['s_valid', 'ra_trig']
        pla_plot_mon_spec, sat, mon_spec_prod, 'counts'
        ; distribution
        source = sources[jj]
        ; Load Level 1 data
        IF source GE 0 AND source LE 3 THEN en_spec_prod = [sat+string(source+1, format = '(I2.2)')] ELSE $
          IF source EQ 4 THEN en_spec_prod = [sat+'10/0'] ELSE $
          IF source EQ 5 THEN en_spec_prod = [sat+'10/1'] ELSE BEGIN 
            print, 'invalid source'
            stop
        ENDELSE 

        ;IF keyword_set(bkgd_sub) THEN BEGIN 
        IF source EQ 5 THEN BEGIN ; assuming sw pri 1 at the moment
            dat     = get_pla_sw_priority     (sat, 1)
            wap_dat = get_pla_wap_priority_ssd(sat, 0)
            IF size(dat, /type) EQ 8 THEN BEGIN ; check if have data
                ;error_num_counts = dat.data
                pri1_orig        = dat.data
                pri1_t           = dat.time
                dat     = convert_plastic_units(dat,     'c/s')
                wap_dat = convert_plastic_units(wap_dat, 'c/s')
                var1 = 'ST'+sat+'_PLA_C/S_SW_PRIORITY_01'
                var2 = 'ST'+sat+'_PLA_C/S_WAP_PRIORITY_00'
                store_data, var1, data = {x: dat.time, y: dat.data}
                store_data, var2, data = {x: dat.time, y: wap_dat.data}
                ; subtract background
                get_data, var1, pri_t, pri_d
                temp_pri_d = pri_d
                norm  = dat.nposition * dat.ndeflection
                ; main only & schan only
                pri_main_d             = temp_pri_d
                pri_schan_d            = temp_pri_d
                wap_main_d             = wap_dat.data
                wap_schan_d            = wap_dat.data
                FOR kk = 0, n_elements(pri_t)-1 DO BEGIN 
                    IF dat.s_chan_i[kk] EQ -1 THEN BEGIN 
                        pri_main_d            [kk, *, *] = !values.f_nan
                        pri_schan_d           [kk, *, *] = !values.f_nan
                        wap_main_d            [kk, *   ] = !values.f_nan
                        wap_schan_d           [kk, *   ] = !values.f_nan
                    ENDIF ELSE BEGIN 
                        IF dat.s_chan_i[kk] LT 128 THEN BEGIN 
                            pri_main_d           [kk, *, dat.s_chan_i[kk]:127] = !values.f_nan
                            wap_main_d           [kk,    dat.s_chan_i[kk]:127] = !values.f_nan
                        ENDIF 
                        pri_schan_d           [kk, *, 0:dat.s_chan_i[kk]-1] = !values.f_nan
                        wap_schan_d           [kk,    0:dat.s_chan_i[kk]-1] = !values.f_nan
                    ENDELSE 
                ENDFOR 
                background_main        = (pri_main_d            [*, 0, *]+pri_main_d            [*, 15, *]) / 2
                background_schan       = (pri_schan_d           [*, 0, *]+pri_schan_d           [*, 15, *]) / 2
                IF keyword_set(bkgd_sub) EQ 1 THEN BEGIN 
                    FOR kk = 0, 15 DO BEGIN 
                        pri_main_d [*, kk, *] = pri_main_d [*, kk, *] - background_main
                        pri_schan_d[*, kk, *] = pri_schan_d[*, kk, *] - background_schan
                    ENDFOR 
                ENDIF 
                bad_main_i  = where(pri_main_d  LT 0, bad_main_count)
                bad_schan_i = where(pri_schan_d LT 0, bad_schan_count)
                IF bad_main_count  GT 0 THEN BEGIN
                    pri_main_d           [bad_main_i] = 0
                ENDIF 
                IF bad_schan_count GT 0 THEN BEGIN 
                    pri_schan_d           [bad_schan_i] = 0
                ENDIF 
                zdata_main                    = TOTAL(pri_main_d (*, *, *),   2, /NaN) / norm
                zdata_schan                   = TOTAL(pri_schan_d(*, *, *),   2, /NaN) / norm
                ; have to reset NaN values (lost in sum)
                bad_main_i = where(finite(pri_main_d[*, 0, *]) EQ 0, bad_main_count) ; assume all defl finite or not
                IF bad_main_count GT 0 THEN BEGIN
                    new_bad_main_i = array_indices(pri_main_d[*, 0, *], bad_main_i)
                    zdata_main                  [new_bad_main_i[0, *, *], new_bad_main_i[2, *, *]] = !values.f_nan
                ENDIF 
                bad_schan_i = where(finite(pri_schan_d[*, 0, *]) EQ 0, bad_schan_count) ; assume all defl finite or not
                IF bad_schan_count GT 0 THEN BEGIN
                    new_bad_schan_i = array_indices(pri_schan_d[*, 0, *], bad_schan_i)
                    zdata_schan                  [new_bad_schan_i[0, *, *], new_bad_schan_i[2, *, *]] = !values.f_nan
                ENDIF 
                ; add wap to pri
                IF sat EQ 'A' THEN zdata_main  = zdata_main  + wap_main_d
                IF sat EQ 'A' THEN zdata_schan = zdata_schan + wap_schan_d
                var_main  = 'ST'+sat+'_PLA_ENSPEC_C/S_SW_PRIORITY_01_main_bkgd'
                var_schan = 'ST'+sat+'_PLA_ENSPEC_C/S_SW_PRIORITY_01_schan_bkgd'
                store_data, var_main,  data = {x: pri_t, y: zdata_main}
                store_data, var_schan, data = {x: pri_t, y: zdata_schan}
            ENDIF 
        ENDIF ELSE stop         ; need to add code here for other arrays
        ;ENDIF 
        pla_plot_en_spec, en_spec_prod, 'c/s', diagn = 0, nodisplay = 1

        ; Adjust reduced dist to 128 arrays (and multiply by schan factor?)
        IF source LE 3 THEN adjust_reduced_dist, 'STA_PLA_ENSPEC_C/S_H_ALPHA', 1

        ; Adjust schan based on function
        IF source EQ 5 THEN schan_var = 'ST'+sat+'_PLA_ENSPEC_C/S_SW_PRIORITY_01_s_chan_esa' ELSE stop ; need to fill in

        FOR kk = 0, n_elements(functions)-1 DO BEGIN
            func = functions[kk]

            CASE source OF
                0: var = 'ST'+sat+'_PLA_ENSPEC_C/S_H_ALPHA'
                1: var = 'ST'+sat+'_PLA_ENSPEC_C/S_H+PEAK'
                2: var = 'ST'+sat+'_PLA_ENSPEC_C/S_HE++PEAK'
                3: var = 'ST'+sat+'_PLA_ENSPEC_C/S_H'
                4: var = 'ST'+sat+'_PLA_ENSPEC_C/S_SW_PRIORITY_00'
                5: var = 'ST'+sat+'_PLA_ENSPEC_C/S_SW_PRIORITY_01'
            ENDCASE 
            var_schan_only = var+'_schan_only'
            var_main_only  = var+'_main_only'
            var_schan_only = var+'_schan_bkgd'
            var_main_only  = var+'_main_bkgd'
            get_data, var_main_only,  var_main_t,  var_main_d
            get_data, var_schan_only, var_schan_t, var_schan_d

            IF n_elements(var_main_t) GT 1 THEN BEGIN ; check that we have data
                num_summed = dat.summed
                var_schan_esa = var+'_s_chan_esa'
                get_data, var_schan_esa, schan_esa_t, schan_esa_d
                IF n_elements(var_main_t) NE n_elements(schan_esa_t) THEN stop
                ; ra eff
                temp_vel = alpha_vel_table_s / 1000 ; change from m/s to km/s
                FOR esa_i = 0, 127 DO BEGIN 
                    IF sat EQ 'A' THEN BEGIN 
                        IF temp_vel[esa_i] LT 275 THEN ra_eff = 0.10 $
                        ELSE IF temp_vel[esa_i] GE 275 AND temp_vel[esa_i] LT 570 THEN $
                          ra_eff = (8E-7*(temp_vel[esa_i]^2)) $
                                   + (0.0009 * temp_vel[esa_i]) - 0.2015 $
                             ELSE ra_eff = 0.57
                    ENDIF ELSE BEGIN ; B
                        IF temp_vel[esa_i] LT 275 THEN ra_eff = 0.143 $
                        ELSE IF temp_vel[esa_i] GE 275 AND temp_vel[esa_i] LT 525 THEN $
                          ra_eff = (-1.208E-5 *(temp_vel[esa_i]^2)) $
                                   + (0.01208 * temp_vel[esa_i]) - 2.265 $
                             ELSE ra_eff = 0.75
                    ENDELSE 
                    var_main_d [*, esa_i] = var_main_d [*, esa_i] / ra_eff
                    var_schan_d[*, esa_i] = var_schan_d[*, esa_i] / ra_eff
                ENDFOR 
                store_data, var_main,    data = {x:var_main_t,  y:var_main_d}
                store_data, var_schan,   data = {x:var_schan_t, y:var_schan_d}
                
                adjust_alpha_schan, var_main_only,  schan_var, func, alpha_vel_table_s, sat
                adjust_alpha_schan, var_schan_only, schan_var, func, alpha_vel_table_s, sat

            ; adjust alpha dist
                var_schan_only_flux = var_schan_only+'_alpha_flux'
                var_main_only_flux  = var_main_only +'_alpha_flux'
                schan_esa = var+'_s_chan_esa'
                get_data, schan_esa, schan_t, schan_esa_d
                schan_esa_d = float(schan_esa_d)
                temp_i = where(schan_esa_d EQ -1, count)
                IF count GT 0 THEN BEGIN
                    schan_esa_d[temp_i] = !values.f_nan
                    store_data, schan_esa, data = {x:schan_t, y:schan_esa_d}
                ENDIF 
                schan_esa_5min = schan_esa_d
                schan_t_5min   = schan_t

                ; Average data
                IF ave_time MOD 60 EQ 0 THEN BEGIN ; assuming mulitple of hour
                    get_data, var_main_only_flux,  flux_t,  flux_main_only_d
                    get_data, var_schan_only_flux, flux_t,  flux_schan_only_d
                    get_data, schan_esa,           schan_t, schan_esa_d
                    get_timespan, a
                    start_time_st      = time_string(a[0])
                    temp_time_st       = strmid(start_time_st, 0, 11)+'00:00:00' ; start at 0 o'clock
                    temp_time          = time_double(temp_time_st)
                    num_times          = 24/(ave_time / 60)
                    flux_hour_t        = dblarr(num_times)
                    flux_main_hour_d   = fltarr(num_times, 128)
                    flux_schan_hour_d  = fltarr(num_times, 128)
                    schan_esa_hour_d   = fltarr(num_times)
                    FOR hour_i = 0, (24/(ave_time / 60))-1 DO BEGIN 
                        ; time
                        wanted_i = where(flux_t GE temp_time AND flux_t LT (temp_time+(ave_time*60)), wanted_count)
                        IF wanted_count GE 1 THEN BEGIN 
                            flux_hour_t[hour_i]        = flux_t[wanted_i[0]]
                        ENDIF ELSE BEGIN 
                            flux_hour_t[hour_i]        = temp_time
                        ENDELSE 
                        FOR esa_i = 0, 127 DO BEGIN 
                            ; flux_main_only
                            wanted_i = where(flux_t GE temp_time AND flux_t LT (temp_time+(ave_time*60)) AND $
                                             finite(flux_main_only_d[*, esa_i]) EQ 1, wanted_count)
                            IF wanted_count GT 1 THEN BEGIN
                                mom_flux_main      = moment(flux_main_only_d[wanted_i, esa_i], /nan)
                                flux_main_hour_d[hour_i, esa_i]   = mom_flux_main  [0]
                            ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
                                flux_main_hour_d[hour_i, esa_i]   = flux_main_only_d  [wanted_i, esa_i]
                            ENDIF ELSE BEGIN 
                                flux_main_hour_d[hour_i, esa_i]   = !values.f_nan
                            ENDELSE 
                            ; flux_schan_only
                            wanted_i = where(flux_t GE temp_time AND flux_t LT (temp_time+(ave_time*60)) AND $
                                             finite(flux_schan_only_d[*, esa_i]) EQ 1, wanted_count)
                            IF wanted_count GT 1 THEN BEGIN
                                mom_flux_schan     = moment(flux_schan_only_d[wanted_i, esa_i], /nan)
                                flux_schan_hour_d[hour_i, esa_i]  = mom_flux_schan  [0]
                            ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
                                flux_schan_hour_d[hour_i, esa_i]  = flux_schan_only_d  [wanted_i, esa_i]
                            ENDIF ELSE BEGIN 
                                flux_schan_hour_d[hour_i, esa_i]  = !values.f_nan
                            ENDELSE 
                        ENDFOR 
                        ; schan_esa
                        wanted_i = where(schan_t GE temp_time AND schan_t LT (temp_time+(ave_time*60)) AND $
                                         finite(schan_esa_d) EQ 1, wanted_count)
                        IF wanted_count GT 1 THEN BEGIN
                            mom_schan_esa      = moment(schan_esa_d[wanted_i], /nan)
                            schan_esa_hour_d[hour_i]   = mom_schan_esa  [0]
                        ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
                            schan_esa_hour_d[hour_i]   = schan_esa_d  [wanted_i]
                        ENDIF ELSE BEGIN 
                            schan_esa_hour_d[hour_i]   = !values.f_nan
                        ENDELSE 
                        ; increment time
                        temp_time = temp_time + (ave_time * 60) ; increment in seconds
                    ENDFOR 
                ENDIF ELSE stop ; need to add code here

                store_data, var_main_only_flux,  data = {x:flux_hour_t, y:flux_main_hour_d}
                store_data, var_schan_only_flux, data = {x:flux_hour_t, y:flux_schan_hour_d}
                store_data, schan_esa,           data = {x:flux_hour_t, y:schan_esa_hour_d}

                ; make dist "pure" by esa
                flux_hour_d     = flux_main_hour_d ; for size
                chan_by_esa     = intarr(num_times, 128)
                flux_hour_d [*] = !values.f_nan
                chan_by_esa [*] = -1  ; 0=main, 1=schan
                FOR time_i = 0, n_elements(flux_hour_t)-1 DO BEGIN
                    FOR esa_i = 0, 127 DO BEGIN 
                        main_i  = where(schan_esa_5min GT esa_i AND $
                                        schan_t_5min GE flux_hour_t[time_i] AND $
                                        schan_t_5min LT (flux_hour_t[time_i]+(ave_time * 60)), main_count)
                        IF main_count GT 0 THEN num_main = total(num_summed[main_i, esa_i]) ELSE num_main = 0
                        schan_i = where(schan_esa_5min LE esa_i AND schan_esa_5min GT 0 AND $
                                        schan_t_5min GE flux_hour_t[time_i] AND $
                                        schan_t_5min LT (flux_hour_t[time_i]+(ave_time * 60)), schan_count)
                        IF schan_count GT 0 THEN num_schan = total(num_summed[schan_i, esa_i]) ELSE num_schan = 0
                        total_num = num_main + num_schan
                        IF float(num_main)/float(total_num) GT .25 THEN BEGIN
                            flux_hour_d [time_i, esa_i] = flux_main_hour_d [time_i, esa_i] 
                            chan_by_esa [time_i, esa_i] = 0
                        ENDIF ELSE BEGIN 
                            flux_hour_d [time_i, esa_i] = flux_schan_hour_d [time_i, esa_i] 
                            chan_by_esa [time_i, esa_i] = 1
                        ENDELSE 
                    ENDFOR
                ENDFOR
                new_flux_hour_d = flux_hour_d/7.1E-8 ; m^2 sr eV/eV (divide by main channel geometric factor) -- need to change for B!
                new_flux_schan_hour_d = flux_schan_hour_d/7.1E-8 
                new_flux_main_hour_d  = flux_main_hour_d /7.1E-8 
                size = size(new_flux_hour_d)
                ra_eff = .8     ; temporary value
                FOR mm = 0, size[1]-1 DO BEGIN 
                    new_flux_hour_d      [mm, *] = new_flux_hour_d      [mm, *] / ra_eff ; RA_TRIG efficiency
                    new_flux_schan_hour_d[mm, *] = new_flux_schan_hour_d[mm, *] / ra_eff
                    new_flux_main_hour_d [mm, *] = new_flux_main_hour_d [mm, *] / ra_eff
                ENDFOR 
                dist_constant = 0.0218*0.7854*32 ; .0218*.7854=angular bin size in radians (theta, phi); 32 converts to counts/sec
                FOR mm = 0L, n_elements(flux_hour_t)-1 DO BEGIN
                    IF finite(schan_esa_hour_d[mm]) EQ 1 THEN BEGIN 
                        temp_schan_esa = round(schan_esa_hour_d[mm])
                        new_flux_hour_d[mm, 0:temp_schan_esa-1] = $
                          dist_constant*new_flux_hour_d[mm, 0:temp_schan_esa-1]/(alpha_vel_table_m[0:temp_schan_esa-1]^2)
                        IF temp_schan_esa LT 128 THEN new_flux_hour_d[mm, temp_schan_esa:127] = $
                          dist_constant*new_flux_hour_d[mm, temp_schan_esa:127]/(alpha_vel_table_s[temp_schan_esa:127]^2)
                        new_flux_schan_hour_d[mm, *] = dist_constant*new_flux_schan_hour_d[mm, *]/(alpha_vel_table_s^2)
                        new_flux_main_hour_d [mm, *] = dist_constant*new_flux_main_hour_d [mm, *]/(alpha_vel_table_m^2)
                    ENDIF ELSE BEGIN
                        new_flux_hour_d      [mm, *] = !values.f_nan
                        new_flux_schan_hour_d[mm, *] = !values.f_nan
                        new_flux_main_hour_d [mm, *] = !values.f_nan
                    ENDELSE 
                ENDFOR 

            ; find alpha peak and proton spillover peak via proton velocity
                pl_plot_l2_1dmax, sat, order = 3
                proton_vel = sat+'_l2_1dmax_proton_bulk_speed'
                get_data, proton_vel, proton_t, proton_vel_d
                proton_spill_vel_d = proton_vel_d / (sqrt(2)) ; put in alpha space
                alpha_peak_i  = lonarr(n_elements(flux_hour_t))
                proton_peak_i = lonarr(n_elements(flux_hour_t))
                FOR mm = 0L, n_elements(flux_hour_t)-1 DO BEGIN
                    wanted_i = where(proton_t GE flux_hour_t[mm] AND proton_t LT flux_hour_t[mm]+(ave_time*60) $
                                     AND finite(proton_vel_d) EQ 1, count)
                    IF count GT 0 THEN BEGIN
                        IF count GT 1 THEN BEGIN 
                            mom = moment(proton_vel_d[wanted_i], /nan)
                            proton_vel = mom[0] * 1000 ; change from km/s to m/s
                            mom = moment(proton_spill_vel_d[wanted_i], /nan)
                            proton_spill_vel = mom[0] * 1000 ; change from km/s to m/s
                        ENDIF ELSE proton_vel = proton_vel_d[wanted_i] * 1000
                    ; need to check both tables (Main and s) and where each peak falls
                    ; Alphas
                        diff_m = abs(alpha_vel_table_m-proton_vel) 
                        diff_s = abs(alpha_vel_table_s-proton_vel) 
                        peak_value_m = min(diff_m, temp_alpha_peak_i_m)
                        peak_value_s = min(diff_s, temp_alpha_peak_i_s)
                        temp_schan_esa = round(schan_esa_hour_d[mm])
                        IF temp_alpha_peak_i_m LT temp_schan_esa AND temp_alpha_peak_i_s LT temp_schan_esa THEN $
                          alpha_peak_i[mm] = temp_alpha_peak_i_m $
                        ELSE IF temp_alpha_peak_i_m GE temp_schan_esa AND temp_alpha_peak_i_s GE temp_schan_esa THEN $
                          alpha_peak_i[mm] = temp_alpha_peak_i_s $
                             ELSE alpha_peak_i[mm] = temp_alpha_peak_i_m ; random choice between main and s
                    ; Proton spillover
                        diff_m = abs(alpha_vel_table_m-proton_spill_vel) 
                        diff_s = abs(alpha_vel_table_s-proton_spill_vel) 
                        peak_value_m = min(diff_m, temp_proton_peak_i_m)
                        peak_value_s = min(diff_s, temp_proton_peak_i_s)
                        temp_schan_esa = round(schan_esa_hour_d[mm])
                        IF temp_proton_peak_i_m LT temp_schan_esa AND temp_proton_peak_i_s LT temp_schan_esa THEN $
                          proton_peak_i[mm] = temp_proton_peak_i_m $
                        ELSE IF temp_proton_peak_i_m GE temp_schan_esa AND temp_proton_peak_i_s GE temp_schan_esa THEN $
                          proton_peak_i[mm] = temp_proton_peak_i_s $
                             ELSE proton_peak_i[mm] = temp_proton_peak_i_m ; random choice between main and s
                    ENDIF ELSE  new_flux_hour_d[mm, *] = !values.f_nan
                ENDFOR 

            ; adjust by ratio -- divide all esa of counts by ratio at peak
            ; Create svalid/ratrig ratio
                get_data, 's_valid_counts_norm_science', svalid_t, svalid_d ;, svalid_v
                get_data, 'ra_trig_counts_norm_science', ratrig_t, ratrig_d ;, ratrig_v
                IF sat EQ 'B' THEN BEGIN 
                    temp_pri1_d = total(pri1_orig, 2, /nan) ; sum over deflection
                    pri1_d      = fltarr(n_elements(pri1_t), 32)
                    FOR esa_i = 0, 31 DO BEGIN
                        pri1_d[*, esa_i] = temp_pri1_d[*, (esa_i*4+0)] + temp_pri1_d[*, (esa_i*4+1)] + $
                                           temp_pri1_d[*, (esa_i*4+2)] + temp_pri1_d[*, (esa_i*4+3)]
                    ENDFOR 
                ENDIF 
                IF array_equal(svalid_t, ratrig_t) NE 1 THEN stop ; We don't expect this to happen!!!!
                replace_mon_norm_overflow, svalid_d
                replace_mon_norm_overflow, ratrig_d
                alpha_efficiency_d = fltarr(n_elements(flux_hour_t))
                FOR mm = 0L, n_elements(flux_hour_t)-1 DO BEGIN 
                      wanted_i   = where(ratrig_t GE (flux_hour_t[mm]) AND ratrig_t LT (flux_hour_t[mm]+(ave_time*60)), wanted_count) 
                    IF sat EQ 'B' THEN $
                      wanted_b_i = where(pri1_t   GE (flux_hour_t[mm]) AND pri1_t   LT (flux_hour_t[mm]+(ave_time*60)), wanted_count) 
                    IF wanted_count GT 1 THEN BEGIN
                        IF alpha_peak_i[mm] LT 0 OR alpha_peak_i[mm] GT 127 THEN BEGIN
                            print, "invalid entry for esa"
                            stop
                        ENDIF
                        IF sat EQ 'A' THEN svalid_m = moment(svalid_d[wanted_i,   alpha_peak_i[mm]/4]) $
                          ELSE             pri1_m   = moment(pri1_d  [wanted_b_i, alpha_peak_i[mm]/4])
                        ratrig_m = moment(ratrig_d[wanted_i,   alpha_peak_i[mm]/4])
                        IF sat EQ 'A' THEN ratio = svalid_m[0]/ratrig_m[0] ELSE ratio = pri1_m[0]/ratrig_m[0]
                        new_flux_hour_d      [mm, *] = new_flux_hour_d      [mm, *] / ratio
                        new_flux_schan_hour_d[mm, *] = new_flux_schan_hour_d[mm, *] / ratio
                        new_flux_main_hour_d [mm, *] = new_flux_main_hour_d [mm, *] / ratio
                    ENDIF ELSE IF count EQ 1 THEN BEGIN 
                        IF alpha_peak_i[mm] LT 0 OR alpha_peak_i[mm] GT 127 THEN BEGIN
                            print, "invalid entry for esa"
                            stop
                        ENDIF
                        ratio = svalid_m[wanted_i]/ratrig_m[wanted_i]
                        new_flux_hour_d      [mm, *] = new_flux_hour_d      [mm, *] / ratio
                        new_flux_schan_hour_d[mm, *] = new_flux_schan_hour_d[mm, *] / ratio
                        new_flux_main_hour_d [mm, *] = new_flux_main_hour_d [mm, *] / ratio
                    ENDIF ELSE ratio = !values.f_nan
                    alpha_efficiency_d[mm] = ratio
                ENDFOR 
                alpha_dist            = sat+'_alpha_dist'
                alpha_dist_schan_only = sat+'_alpha_dist_schan_only'
                alpha_dist_main_only  = sat+'_alpha_dist_main_only'
                alpha_peak_esa        = sat+'_alpha_expected_peak_esa'
                alpha_eff             = sat+'_alpha_efficiency'
                store_data, alpha_dist,            data = {x:flux_hour_t, y:new_flux_hour_d}
                store_data, alpha_dist_schan_only, data = {x:flux_hour_t, y:new_flux_schan_hour_d}
                store_data, alpha_dist_main_only,  data = {x:flux_hour_t, y:new_flux_main_hour_d}
                store_data, alpha_peak_esa,        data = {x:flux_hour_t, y:alpha_peak_i}
                store_data, alpha_eff,             data = {x:flux_hour_t, y:alpha_efficiency_d}

            ; To make it easier for IDL to get a good fit to the curve, multiply the distribution by
            ; 1e5.  Remember to divide the density by this factor later on!
                dist                = new_flux_hour_d
                dist                = dist           *(1.0E5)
                coeff               = dblarr(3) ; A matrix in which to store the fit coefficients 
                density             = dblarr(n_elements(flux_hour_t))
                avg_v               = dblarr(n_elements(flux_hour_t))
                temp_k              = dblarr(n_elements(flux_hour_t))
                v_th                = dblarr(n_elements(flux_hour_t))
                katelyn_dist        = dblarr(n_elements(flux_hour_t), 128) ; 17 elements in fit at each time
                katelyn_fit         = dblarr(n_elements(flux_hour_t), 128) ; 17 elements in fit at each time
                proton_dist         = dblarr(n_elements(flux_hour_t), 128)
                proton_fit          = dblarr(n_elements(flux_hour_t), 128)
                proton_fit_full     = dblarr(n_elements(flux_hour_t), 128)
                proton_chisq        = dblarr(n_elements(flux_hour_t))
                alpha_chisq         = dblarr(n_elements(flux_hour_t))
                alpha_flag          = dblarr(n_elements(flux_hour_t))
                proton_flag         = dblarr(n_elements(flux_hour_t))
                which_channel2_d    = intarr(n_elements(flux_hour_t))
                katelyn_dist    [*] = !values.f_nan
                katelyn_fit     [*] = !values.f_nan
                proton_dist     [*] = !values.f_nan
                proton_fit      [*] = !values.f_nan
                proton_fit_full [*] = !values.f_nan
                which_channel2_d[*] = -1
                FOR mm = 0L, n_elements(flux_hour_t)-1 DO BEGIN 
                    ; Proton Spillover
                    valid = 1
                    IF (proton_peak_i[mm] GE 4) AND (proton_peak_i[mm] LE 121) THEN BEGIN
                        k = proton_peak_i[mm]-4
                        m = proton_peak_i[mm]+6  
                    ENDIF ELSE IF proton_peak_i[mm] GE 4 THEN BEGIN
                        k = proton_peak_i[mm]-4	
                        m = 127
                    ENDIF ELSE BEGIN ; peak lt 4
                        k = 0	
                        m = proton_peak_i[mm]+6
                    ENDELSE 
                    proton_dist[mm, k:m] = dist[mm, k:m]
                    temp_schan_esa = round(schan_esa_hour_d[mm])
                    bad_i = where(dist eq 0, count)
                    if count gt 0 then dist[bad_i] = !values.f_nan
                    temp_dist = reform(dist[mm, k:m])
                    temp_alpha_vel_table = alpha_vel_table_s[k:m]
                    temp_proton_fit = dblarr(m-k+1)
                    temp_proton_fit[*] = !values.f_nan
                    bad_i = where(finite(temp_dist) eq 0 OR temp_dist LT 0, bad_count, $
                                  complement = good_i, ncomplement = good_count)
                    if bad_count gt 0 and good_count gt 3 then begin
                        temp_dist_a = temp_dist[good_i]
                        temp_alpha_vel_table_a = temp_alpha_vel_table[good_i]
                        FIT_MATRIX = GAUSSFIT(double(temp_alpha_vel_table_a), temp_dist_a, coeff, $
                                              nterms = 3, measure_errors = SQRT(temp_dist_a), chisq = temp_chisq) 
                        temp_proton_fit[good_i] = fit_matrix
                        proton_fit[mm, k:m] = temp_proton_fit
                    endif else if good_count gt 3 then begin
                        FIT_MATRIX = GAUSSFIT(double(alpha_vel_table_s[k:m]), reform(dist[mm, k:m]), coeff, $
                                              nterms = 3, measure_errors = SQRT(reform(dist[mm, k:m])), chisq = temp_chisq)
                        proton_fit[mm, k:m] = fit_matrix
                    endif else BEGIN
                        print, "not enough valid data at time index", mm
                        coeff[*] = !values.f_nan
                        temp_chisq = !values.f_nan
                    ENDELSE 
                    x = alpha_vel_table_s
                    z = (x - coeff[1]) / abs(coeff[2])
                    proton_fit_full[mm, *] = coeff[0] * (2.71827^((-1.0*(z^2))/2))
                    proton_chisq[mm] = temp_chisq 
                    IF coeff[1] GT alpha_vel_table_s[k] THEN BEGIN
                        proton_flag[mm] = 1
                    ENDIF ELSE IF coeff[1] LT alpha_vel_table_s[m] THEN BEGIN
                        proton_flag[mm] = 1
                    ENDIF ELSE BEGIN
                        proton_flag[mm] = 0     
                    ENDELSE          

                    ; Alphas
                    IF keyword_set(proton_sub) AND proton_flag[mm] EQ 0 THEN BEGIN
                        dist           [mm, *] = dist           [mm, *] - proton_fit_full[mm, *]
                    ENDIF 
                    valid = 1
                    IF (alpha_peak_i[mm] GE 6) AND (alpha_peak_i[mm] LE 121) THEN BEGIN
                        k = alpha_peak_i[mm]-6
                        m = alpha_peak_i[mm]+6  
                        schan_i = where(chan_by_esa[mm, k:m] EQ 1, num_schan, complement = main_i, ncomplement = num_main)
                        IF num_schan LE 3 THEN which_channel2_d[mm] = 0 ELSE $ ; main channel
                          IF num_schan GE 10 THEN which_channel2_d[mm] = 1 ELSE $ ; s_channel
                          which_channel2_d[mm] = 2 ; mixed
                        IF which_channel2_d[mm] EQ 1 AND num_main GT 0 THEN BEGIN ; if labeled schan, don't use main
                            temp_dist = reform(dist[mm, k:m])
                            temp_dist[main_i] = !values.f_nan
                            dist[mm, k:m] = temp_dist
                        ENDIF 
                        IF which_channel2_d[mm] EQ 0 AND num_schan GT 0 THEN BEGIN ; if labeled main, don't use schan
                            temp_dist = reform(dist[mm, k:m])
                            temp_dist[schan_i] = !values.f_nan
                            dist[mm, k:m] = temp_dist
                        ENDIF 
                    ENDIF ELSE IF alpha_peak_i[mm] GE 6 THEN BEGIN
                        k = alpha_peak_i[mm]-6
                        m = 127
                        which_channel2_d[mm] = 3 ; peak too low
                    ENDIF ELSE BEGIN ; peak lt 6
                        k = 0	
                        m = alpha_peak_i[mm]+6
                        which_channel2_d[mm] = 3 ; peak too high
                    ENDELSE 
                    katelyn_dist[mm, k:m] = dist[mm, k:m]
                    temp_schan_esa = round(schan_esa_hour_d[mm])
                    bad_i = where(dist eq 0, count)
                    if count gt 0 then dist[bad_i] = !values.f_nan
                    temp_dist            = reform(dist[mm, k:m])
                    temp_alpha_vel_table = alpha_vel_table_s[k:m]
                    temp_katelyn_fit = dblarr(m-k+1)
                    temp_katelyn_fit[*] = !values.f_nan
                    bad_i = where(finite(temp_dist) eq 0 OR temp_dist LT 0, bad_count, $
                                  complement = good_i, ncomplement = good_count)
                    if bad_count gt 0 and good_count gt 3 then begin
                        temp_dist_a            = temp_dist           [good_i]
                        delta_dist_funct       = fltarr(good_count)
                        delta_dist_funct[*]    = 0.05 * max(temp_dist_a)
                        temp_alpha_vel_table_a = temp_alpha_vel_table[good_i]
                        FIT_MATRIX = GAUSSFIT(double(temp_alpha_vel_table_a), temp_dist_a, coeff, $
                                              nterms = 3, measure_errors = delta_dist_funct, chisq = temp_chisq) 
                        temp_katelyn_fit[good_i] = fit_matrix
                        katelyn_fit[mm, k:m] = temp_katelyn_fit
                    endif else if good_count gt 3 then BEGIN
                        delta_dist_funct       = fltarr(m-k+1)
                        delta_dist_funct[*]    = 0.05 * max(temp_dist)
                        FIT_MATRIX = GAUSSFIT(double(alpha_vel_table_s[k:m]), temp_dist, coeff, $
                                              nterms = 3, measure_errors = delta_dist_funct, chisq = temp_chisq)
                        katelyn_fit[mm, k:m] = fit_matrix
                    endif else BEGIN
                        print, "not enough valid data at time index", mm
                        coeff[*] = !values.f_nan
                        temp_chisq = !values.f_nan
                    ENDELSE 

                    alpha_chisq[mm] = temp_chisq                  
                    density[mm] = coeff[0]*((2*!PI)^(0.5))*abs(coeff[2])*1.0E-11 ; 1/cc
                    IF density[mm] LT 0 THEN BEGIN
                        print, mm, ' ', time_string(flux_hour_t[mm]), density[mm]
                        stop
                    ENDIF 
                    IF coeff[1] GT alpha_vel_table_s[k] THEN BEGIN
                        alpha_flag[mm] = 2
                    ENDIF ELSE IF coeff[1] LT alpha_vel_table_s[m] THEN BEGIN
                        alpha_flag[mm] = 2
                    ENDIF ELSE BEGIN
                        alpha_flag[mm] = 0
                    ENDELSE

                    ; redo fit with shifted dist (to 2 steps higher energy) if first try failed
                    IF alpha_flag[mm] EQ 2 THEN BEGIN 
                        katelyn_fit[mm, *] = !values.f_nan
                        IF (alpha_peak_i[mm]-2 GE 6) AND (alpha_peak_i[mm]-2 LE 121) THEN BEGIN
                            k = alpha_peak_i[mm]-2-6
                            m = alpha_peak_i[mm]-2+6  
                            IF num_schan LE 3 THEN which_channel2_d[mm] = 0 ELSE $ ; main channel
                              IF num_schan GE 10 THEN which_channel2_d[mm] = 1 ELSE $ ; s_channel
                              which_channel2_d[mm] = 2 ; mixed
                            IF which_channel2_d[mm] EQ 1 AND num_main GT 0 AND sat EQ 'A' THEN BEGIN ; if labeled schan, don't use main
                                temp_dist = reform(dist[mm, k:m])
                                temp_dist[main_i] = !values.f_nan
                                dist[mm, k:m] = temp_dist
                            ENDIF 
                            IF which_channel2_d[mm] EQ 0 AND num_schan GT 0 AND sat EQ 'A' THEN BEGIN ; if labeled main, don't use schan
                                temp_dist = reform(dist[mm, k:m])
                                temp_dist[schan_i] = !values.f_nan
                                dist[mm, k:m] = temp_dist
                            ENDIF 
                        ENDIF ELSE IF alpha_peak_i[mm]-2 GE 6 THEN BEGIN
                            k = alpha_peak_i[mm]-2-6
                            m = 127
                            which_channel2_d[mm] = 3 ; peak too low
                        ENDIF ELSE BEGIN ; peak lt 6
                            k = 0	
                            m = alpha_peak_i[mm]-2+6
                            which_channel2_d[mm] = 3 ; peak too high
                        ENDELSE 
                        katelyn_dist[mm, k:m] = dist[mm, k:m]
                        temp_schan_esa = round(schan_esa_hour_d[mm])
                        bad_i = where(dist eq 0, count)
                        if count gt 0 then dist[bad_i] = !values.f_nan
                        temp_dist            = reform(dist[mm, k:m])
                        temp_alpha_vel_table = alpha_vel_table_s[k:m]
                        temp_katelyn_fit = dblarr(m-k+1)
                        temp_katelyn_fit[*] = !values.f_nan
                        bad_i = where(finite(temp_dist) eq 0 OR temp_dist LT 0, bad_count, $
                                      complement = good_i, ncomplement = good_count)
                        if bad_count gt 0 and good_count gt 3 then begin
                            temp_dist_a            = temp_dist           [good_i]
                            delta_dist_funct       = fltarr(good_count)
                            delta_dist_funct[*]    = 0.05 * max(temp_dist_a)
                            temp_alpha_vel_table_a = temp_alpha_vel_table[good_i]
                            FIT_MATRIX = GAUSSFIT(double(temp_alpha_vel_table_a), temp_dist_a, coeff, $
                                                  nterms = 3, measure_errors = delta_dist_funct, chisq = temp_chisq) 
                            temp_katelyn_fit[good_i] = fit_matrix
                            katelyn_fit[mm, k:m] = temp_katelyn_fit
                        endif else if good_count gt 3 then BEGIN
                            delta_dist_funct       = fltarr(m-k+1)
                            delta_dist_funct[*]    = 0.05 * max(temp_dist)
                            FIT_MATRIX = GAUSSFIT(double(alpha_vel_table_s[k:m]), temp_dist, coeff, $
                                                  nterms = 3, measure_errors = delta_dist_funct, chisq = temp_chisq)
                            katelyn_fit[mm, k:m] = fit_matrix
                        endif else BEGIN
                            print, "not enough valid data at time index", mm
                            coeff[*] = !values.f_nan
                            temp_chisq = !values.f_nan
                        ENDELSE 
                        alpha_chisq[mm] = temp_chisq                  
                                ; katelyn_fit[mm, k:m] =fit_matrix
                        density[mm] = coeff[0]*((2*!PI)^(0.5))*abs(coeff[2])*1.0E-11 ; 1/cc
                        IF density[mm] LT 0 THEN BEGIN
                            print, mm, ' ', time_string(flux_hour_t[mm]), density[mm]
                            stop
                        ENDIF 
                        IF coeff[1] GT alpha_vel_table_s[k] THEN BEGIN
                            alpha_flag[mm] = 2
                        ENDIF ELSE IF coeff[1] LT alpha_vel_table_s[m] THEN BEGIN
                            alpha_flag[mm] = 2 ; failed
                        ENDIF ELSE BEGIN
                            alpha_flag[mm] = 1 ; worked on second try
                        ENDELSE
                    ENDIF 
                    avg_v[mm] = coeff[1]*1.0E-3 ; km/s
                    temp_k[mm] = (abs(coeff[2])^2)*4*(1.67E-27)/(1.38E-23) ;deg K
                    v_th[mm] = SQRT(2)*(abs(coeff[2]))*1.0E-3 ; km/s
                    bad_i = where(finite(coeff) EQ 0, bad_count)
                    IF bad_count GT 0 THEN BEGIN
                        density[mm] = !values.f_nan
                        avg_v[mm]   = !values.f_nan
                        temp_k[mm]  = !values.f_nan
                        v_th[mm]    = !values.f_nan
                    ENDIF 

                ENDFOR 

                IF keyword_set(proton_sub) THEN BEGIN
                    temp_dist            = dist            / 1.0E5
                    store_data, alpha_dist,            data = {x:flux_hour_t, y:temp_dist}
                ENDIF 

            ; find _peak_esa
                alpha_peak_esa_d = intarr(n_elements(avg_v))
                FOR mm = 0L, n_elements(avg_v)-1 DO BEGIN
                    IF finite(avg_v[mm]) EQ 1 THEN BEGIN 
                                ; need to check both tables (Main and s) and where each peak falls
                        diff_m = abs(alpha_vel_table_m-(avg_v[mm]*1000))
                        diff_s = abs(alpha_vel_table_s-(avg_v[mm]*1000))
                        peak_value_m = min(diff_m, temp_peak_i_m)
                        peak_value_s = min(diff_s, temp_peak_i_s)
                        temp_schan_esa = round(schan_esa_hour_d[mm])
                        IF temp_peak_i_m LT temp_schan_esa AND temp_peak_i_s LT temp_schan_esa THEN BEGIN ; main channel
                            alpha_peak_esa_d[mm] = temp_peak_i_m 
                        ENDIF ELSE IF temp_peak_i_m GE temp_schan_esa AND temp_peak_i_s GE temp_schan_esa THEN BEGIN ; s channel
                            alpha_peak_esa_d[mm] = temp_peak_i_s 
                        ENDIF ELSE BEGIN ; could be either, choose main (randomly)
                            alpha_peak_esa_d[mm] = temp_peak_i_m 
                        ENDELSE 
                    ENDIF ELSE alpha_peak_esa_d[mm, *] = -1
                ENDFOR 
                alpha_peak_esa = sat+'_alpha_peak_esa'
                store_data, alpha_peak_esa, data = {x:flux_hour_t, y:alpha_peak_esa_d}

                ; aberration angle
                temp_time = time_string(flux_hour_t)
                strput, temp_time, 'T', 10 ; change time string format for SolarSoft
                state = get_stereo_coord(temp_time, sat, system = 'hci')
                spcrft_vel = sqrt((state[3, *]^2)+(state[4, *]^2)+(state[5, *]^2))
                alpha_vel_inst  = avg_v
                alpha_vel_inert = sqrt((avg_v^2)-(spcrft_vel^2))

                ; remove outliers 
                times_st = time_string(flux_hour_t)
                bad_i = where(times_st GE '2007-02-00/00:00:00' AND times_st LT '2007-03-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 02
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 02
                ENDIF 
                bad_i = where(times_st GE '2007-03-00/00:00:00' AND times_st LT '2007-04-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 03
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 03
                ENDIF 
                bad_i = where(times_st GE '2007-04-00/00:00:00' AND times_st LT '2007-05-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 04
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 04
                ENDIF 
                bad_i = where(times_st GE '2007-05-00/00:00:00' AND times_st LT '2007-06-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 05
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 05
                ENDIF 
                bad_i = where(times_st GE '2007-06-00/00:00:00' AND times_st LT '2007-07-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 06
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 06
                ENDIF 
                bad_i = where(times_st GE '2007-07-00/00:00:00' AND times_st LT '2007-08-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 07
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 07
                ENDIF 
                bad_i = where(times_st GE '2007-08-00/00:00:00' AND times_st LT '2007-09-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 08
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 08
                ENDIF 
                bad_i = where(times_st GE '2007-09-00/00:00:00' AND times_st LT '2007-10-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 09
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 09
                ENDIF 
                bad_i = where(times_st GE '2007-10-00/00:00:00' AND times_st LT '2007-11-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 10
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 10
                ENDIF 
                bad_i = where(times_st GE '2007-11-00/00:00:00' AND times_st LT '2007-12-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 11
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 11
                ENDIF 
                bad_i = where(times_st GE '2007-12-00/00:00:00' AND times_st LT '2008-01-00/00:00:00', bad_count)
                IF bad_count GT 0 THEN BEGIN 
                    set_bad_alpha_tem,     times_st,          temp_k, v_th, sat, 2007, 12
                    set_bad_alpha_den_tem, times_st, density, temp_k, v_th, sat, 2007, 12
                ENDIF 

            ; create variables
                var_den               = sat+'_alpha_den'
                var_vel_inst          = sat+'_alpha_vel_inst'
                var_vel_inert         = sat+'_alpha_vel_inertial'
                var_temp              = sat+'_alpha_temp'
                var_therm             = sat+'_alpha_therm'
                var_which_channel     = sat+'_which_channel'
                store_data, 'proton_dist',     data = {x:flux_hour_t, y:proton_dist}
                store_data, 'proton_fit',      data = {x:flux_hour_t, y:proton_fit}
                store_data, 'proton_fit_full', data = {x:flux_hour_t, y:proton_fit_full}
                store_data, 'katelyn_dist',    data = {x:flux_hour_t, y:katelyn_dist}
                store_data, 'katelyn_fit',     data = {x:flux_hour_t, y:katelyn_fit}
                temp_time = flux_hour_t[0]
                store_data, 'vel_table_s',     data = {x:temp_time,   y:alpha_vel_table_s}
                store_data, 'vel_table_m',     data = {x:temp_time,   y:alpha_vel_table_m}
                store_data, 'alpha_chisq',     data = {x:flux_hour_t, y:alpha_chisq}
                store_data, 'proton_chisq',    data = {x:flux_hour_t, y:proton_chisq}
                store_data, 'alpha_flag',      data = {x:flux_hour_t, y:alpha_flag}
                store_data, 'proton_flag',     data = {x:flux_hour_t, y:proton_flag}
                store_data, var_den,           data = {x:flux_hour_t, y:density} 
                store_data, var_vel_inst,      data = {x:flux_hour_t, y:alpha_vel_inst} 
                store_data, var_vel_inert,     data = {x:flux_hour_t, y:alpha_vel_inert} 
                store_data, var_temp,          data = {x:flux_hour_t, y:temp_k} 
                store_data, var_therm,         data = {x:flux_hour_t, y:v_th} 
                store_data, var_which_channel, data = {x:flux_hour_t, y:which_channel2_d} 

            ; empty arrays to create memory space
                density               = 0
                temp_k                = 0 
                v_th                  = 0 
                which_channel         = 0 
                schan_t               = 0
                schan_esa_hour_d      = 0
                flux_hour_d           = 0
                flux_schan_hour_d     = 0
                flux_main_hour_d      = 0
                new_flux_hour_d       = 0
                new_flux_schan_hour_d = 0
                new_flux_main_hour_d  = 0
                proton_t              = 0
                proton_vel_d          = 0
                svalid_t              = 0
                svalid_d              = 0
                ratrig_t              = 0
                ratrig_d              = 0
                dist                  = 0
                katelyn_dist          = 0
                katelyn_fit           = 0
                proton_dist           = 0
                proton_fit            = 0
                proton_fit_full       = 0
                proton_chisq          = 0
                chisq                 = 0
                alpha_flag            = 0
                proton_flag           = 0
                
            ; N/S
;                sw_pri1_struct = get_pla_data(sat, 'sw_priority', 1) ; for N/S angle
;                ns_schan_only = sat+'_alpha_ns_schan_only'
;                ns_main_only  = sat+'_alpha_ns_main_only'
;                IF size(sw_pri1_struct, /type) EQ 8 THEN BEGIN 
;                    ns_t     = sw_pri1_struct.time
;                    ns_d     = sw_pri1_struct.data
;                    ns_schan = sw_pri1_struct.s_chan_i
;                    ns_schan_only_d = ns_d
;                    ns_main_only_d  = ns_d
;                    temp_i = where(finite(ns_schan) EQ 0 OR ns_schan EQ 0, temp_count)
;                    IF temp_count GT 0 THEN stop ; don't expect this
;                    FOR mm = 0L, n_elements(ns_t)-1 DO BEGIN 
;                        IF ns_schan[mm] EQ -1 THEN BEGIN 
;                            ns_schan_only_d[mm, *] = !values.f_nan
;                            ns_main_only_d[mm, *] = !values.f_nan
;                        ENDIF ELSE BEGIN 
;                            ns_schan_only_d[mm, *, 0:ns_schan[mm]-1] = !values.f_nan
;                            IF ns_schan[mm] LT 128 THEN ns_main_only_d [mm, *, ns_schan[mm]:127] = !values.f_nan
;                        ENDELSE 
;                    ENDFOR 
;                    store_data, ns_schan_only, data = {x:ns_t, y:ns_schan_only_d}
;                    store_data, ns_main_only,  data = {x:ns_t, y:ns_main_only_d}
;
;
;                    ; Average data
;                    IF ave_time MOD 60 EQ 0 THEN BEGIN ; assuming mulitple of hour
;                        get_timespan, a
;                        start_time_st      = time_string(a[0])
;                        temp_time_st       = strmid(start_time_st, 0, 11)+'00:00:00' ; start at 0 o'clock
;                        temp_time          = time_double(temp_time_st)
;                        num_times          = 24/(ave_time / 60)
;                        ns_hour_t        = dblarr(num_times)
;                        ns_main_hour_d   = fltarr(num_times, 16, 128)
;                        ns_schan_hour_d  = fltarr(num_times, 16, 128)
;                        FOR hour_i = 0, (24/(ave_time / 60))-1 DO BEGIN 
;                        ; time
;                            wanted_i = where(ns_t GE temp_time AND ns_t LT (temp_time+(ave_time*60)), wanted_count)
;                            IF wanted_count GE 1 THEN BEGIN 
;                                ns_hour_t[hour_i]        = ns_t[wanted_i[0]]
;                            ENDIF ELSE BEGIN 
;                                ns_hour_t[hour_i]        = temp_time
;                            ENDELSE 
;                            FOR mid_i = 0, 15 DO BEGIN 
;                                FOR esa_i = 0, 127 DO BEGIN 
;                                    ; ns_main_only
;                                    wanted_i = where(ns_t GE temp_time AND ns_t LT (temp_time+(ave_time*60)) AND $
;                                                     finite(ns_main_only_d[*, mid_i, esa_i]) EQ 1, wanted_count)
;                                    IF wanted_count GT 1 THEN BEGIN
;                                        mom_ns_main      = moment(ns_main_only_d[wanted_i, mid_i, esa_i], /nan)
;                                        ns_main_hour_d[hour_i, mid_i, esa_i]   = mom_ns_main  [0]
;                                    ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
;                                        ns_main_hour_d[hour_i, mid_i, esa_i]   = ns_main_only_d  [wanted_i, mid_i, esa_i]
;                                    ENDIF ELSE BEGIN 
;                                        ns_main_hour_d[hour_i, mid_i, esa_i]   = !values.f_nan
;                                    ENDELSE 
;                                    ; ns_schan_only
;                                    wanted_i = where(ns_t GE temp_time AND ns_t LT (temp_time+(ave_time*60)) AND $
;                                                     finite(ns_schan_only_d[*, mid_i, esa_i]) EQ 1, wanted_count)
;                                    IF wanted_count GT 1 THEN BEGIN
;                                        mom_ns_schan      = moment(ns_schan_only_d[wanted_i, mid_i, esa_i], /nan)
;                                        ns_schan_hour_d[hour_i, mid_i, esa_i]   = mom_ns_schan  [0]
;                                    ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
;                                        ns_schan_hour_d[hour_i, mid_i, esa_i]   = ns_schan_only_d  [wanted_i, mid_i, esa_i]
;                                    ENDIF ELSE BEGIN 
;                                        ns_schan_hour_d[hour_i, mid_i, esa_i]   = !values.f_nan
;                                    ENDELSE 
;                                ENDFOR 
;                            ENDFOR 
;                            ; increment time
;                            temp_time = temp_time + (ave_time * 60) ; increment in seconds
;                        ENDFOR 
;                    ENDIF ELSE stop ; need to add code here
;
;                    store_data, ns_schan_only, data = {x:ns_hour_t, y:ns_schan_hour_d}
;                    store_data, ns_main_only,  data = {x:ns_hour_t, y:ns_main_hour_d}
;                    IF array_equal(ns_hour_t, flux_hour_t) NE 1 THEN stop
;                    temp_ns_schan_hour_d = fltarr(n_elements(flux_hour_t), 16) ; 16 deflection bins
;                    temp_ns_main_hour_d  = fltarr(n_elements(flux_hour_t), 16) ; 16 deflection bins
;                    FOR mm = 0L, n_elements(flux_hour_t)-1 DO BEGIN 
;                        temp_peak_i = alpha_peak_esa_d[mm]
;                        IF temp_peak_i LT 2   THEN temp_peak_i = 2
;                        IF temp_peak_i GT 125 THEN temp_peak_i = 125
;                        temp_schan_hour_data = reform(ns_schan_hour_d[mm, *, temp_peak_i-2:temp_peak_i+2])
;                        temp_main_hour_data = reform(ns_main_hour_d[mm, *, temp_peak_i-2:temp_peak_i+2])
;                        temp_ns_schan_hour_d[mm, *] = total(temp_schan_hour_data, 2, /nan) ; sum over esa
;                        temp_ns_main_hour_d [mm, *] = total(temp_main_hour_data,  2, /nan) ; sum over esa
;                    ENDFOR 
;                    avg_defl_schan_hour_d = $
;                      total(temp_ns_schan_hour_d * sw_pri1_struct.deflection[0:n_elements(flux_hour_t)-1, *], 2, /nan)/$
;                      total(temp_ns_schan_hour_d, 2, /nan)
;                    avg_defl_main_hour_d = $
;                      total(temp_ns_main_hour_d * sw_pri1_struct.deflection[0:n_elements(flux_hour_t)-1, *], 2, /nan)/$
;                      total(temp_ns_main_hour_d, 2, /nan)
;                    IF sat EQ 'A' THEN BEGIN
;                ;avg_defl_main_hour_d = -23.25+1.5*avg_defl_main_hour_d ; convert to angle (degrees) from bin number
;                ;avg_defl_schan_hour_d = -19.375 + 1.25*avg_defl_d ; convert to angle (degrees) from bin number (done above)
;                        avg_defl_schan_hour_d_scaled = 3.0*(avg_defl_schan_hour_d + 3.5) ; modified 27 August, 2008  
;                        avg_defl_main_hour_d_scaled = 3.0*(avg_defl_main_hour_d + 3.5) ; modified 27 August, 2008  
;                        ; 3.5 if shift center of distribution, 3.0 is to spread out distribution      
;                        FOR k = 0L, n_elements(flux_hour_t)-1 DO BEGIN
;                            avg_defl_schan_hour_d(k) = avg_defl_schan_hour_d_scaled(k) + $
;                              (-2.0854E-7)*(avg_v(k)^3)+(3.0371E-4)*(avg_v(k)^2) + $
;                              (-1.2454E-1)*avg_v(k)+1.0983E1 ; modified 27 August, 2008
;                            avg_defl_main_hour_d(k) = avg_defl_main_hour_d_scaled(k) + $
;                              (-2.0854E-7)*(avg_v(k)^3)+(3.0371E-4)*(avg_v(k)^2) + $
;                              (-1.2454E-1)*avg_v(k)+1.0983E1 ; modified 27 August, 2008
;                   ; this comes from velocity dependence of leakage
;                            IF ABS(avg_defl_schan_hour_d(k)) GT 20.0 THEN avg_defl_schan_hour_d(k) = !values.f_nan ; added 02 September, 2008
;                            IF ABS(avg_defl_main_hour_d (k)) GT 20.0 THEN avg_defl_main_hour_d(k) = !values.f_nan ; added 02 September, 2008
;                        ENDFOR
;                    ENDIF ELSE BEGIN
;                        avg_defl_main_hour_d  = 31.0 - avg_defl_main_hour_d
;                        avg_defl_schan_hour_d = 31.0 - avg_defl_schan_hour_d ; flip distribution  (LBE: Do I need this?)
;                ;avg_defl_main_hour_d = -23.25+1.5*avg_defl_main_hour_d
;                ;avg_defl_schan_hour_d = -19.375 + 1.25*avg_defl_schan_hour_d -- don't need, done above
;                        avg_defl_schan_hour_d_scaled = 1.75*(avg_defl_schan_hour_d - 4.62) ; modified 14 August, 2008
;                        avg_defl_main_hour_d_scaled  = 1.75*(avg_defl_main_hour_d  - 4.62) ; modified 14 August, 2008
;                        FOR k = 0L, n_elements(flux_hour_t)-1 DO BEGIN
;                            avg_defl_schan_hour_d(k) = avg_defl_schan_hour_d_scaled(k) + $
;                              (-2.6404E-7)*(avg_v(k)^3) + (4.3726E-4)*(avg_v(k)^2) + $
;                              (-2.3922E-1)*avg_v(k)+4.2639E1
;                            avg_defl_main_hour_d(k) = avg_defl_main_hour_d_scaled(k) + $
;                              (-2.6404E-7)*(avg_v(k)^3) + (4.3726E-4)*(avg_v(k)^2) + $
;                              (-2.3922E-1)*avg_v(k)+4.2639E1
;                            IF ABS(avg_defl_schan_hour_d(k)) GT 20.0 THEN avg_defl_schan_hour_d(k) = !values.f_nan ; added 02 September, 2008
;                            IF ABS(avg_defl_main_hour_d (k)) GT 20.0 THEN avg_defl_main_hour_d(k) = !values.f_nan ; added 02 September, 2008
;                        ENDFOR 
;                    ENDELSE
;                    ns_schan_hour_d = avg_defl_schan_hour_d
;                    ns_main_hour_d  = avg_defl_main_hour_d
;
;            ; create variables
;                    var_ns_schan_only     = sat+'_alpha_ns_schan_only'
;                    var_ns_main_only      = sat+'_alpha_ns_main_only'
;                    store_data, var_ns_schan_only,     data = {x:flux_hour_t,   y:ns_schan_hour_d} 
;                    store_data, var_ns_main_only,      data = {x:flux_hour_t,   y:ns_main_hour_d} 
;                ENDIF 
;                                ; empty arrays to create memory space
;                sw_pri1_struct               = 0
;                ns_hour_t                    = 0
;                ns_d                         = 0
;                ns_schan                     = 0
;                ns_schan_hour_d              = 0
;                ns_main_hour_d               = 0
;                temp_ns_schan_hour_d         = 0
;                temp_ns_main_hour_d          = 0
;                avg_defl_schan_hour_d        = 0
;                avg_defl_main_hour_d         = 0
;                avg_defl_schan_hour_d_scaled = 0
;                avg_defl_main_hour_d_scaled  = 0

            ; E/W
;                sw_all_struct = get_pla_data(sat, 'h_alpha', 0) ; for E/W angle
;                IF size(sw_all_struct, /type) EQ 8 THEN BEGIN 
;                    ew_schan_only = sat+'_alpha_ew_schan_only'
;                    ew_main_only  = sat+'_alpha_ew_main_only'
;                    ew_t     = sw_all_struct.time
;                    ew_d     = sw_all_struct.full_array
;                    ew_schan = sw_all_struct.s_chan_i
;                ; line up with beginning of flux_hour_t
;                    early_i = where((ew_t LT flux_hour_t[0]) AND (abs(ew_t - flux_hour_t[0]) GT 1.0), early_count, complement = good_i)
;                    IF early_count GT 0 THEN BEGIN
;                        ew_t = ew_t[good_i]
;                        ew_d = ew_d[good_i, *, *, *]
;                    ENDIF 
;                    flux_i = 1
;                    WHILE abs(ew_t[0] - flux_hour_t[flux_i-1]) GT 1 DO BEGIN ; drop data to line up with a flux_hour_t
;                        early_i = where((ew_t LT flux_hour_t[flux_i]) AND (abs(ew_t - flux_hour_t[flux_i]) GT 1.0), $
;                                        early_count, complement = good_i)
;                        IF early_count GT 0 THEN BEGIN
;                            ew_t = ew_t[good_i]
;                            ew_d = ew_d[good_i, *, *, *]
;                        ENDIF 
;                        flux_i = flux_i + 1
;                    ENDWHILE 
;                    ew_schan_only_d = ew_d
;                    ew_main_only_d  = ew_d
;                    FOR mm = 0L, n_elements(ew_t)-1 DO BEGIN 
;                        IF mm MOD 10000 EQ 0 THEN print, 'in ew ', mm, ' ', n_elements(ew_t)
;                        IF ew_schan[mm] NE -1 THEN BEGIN ; valid schan
;                            IF ew_schan[mm] GT   0 THEN ew_schan_only_d[mm, *, *, 0:ew_schan[mm]-1] = !values.f_nan
;                            IF ew_schan[mm] LT 128 THEN ew_main_only_d [mm, *, *, ew_schan[mm]:127] = !values.f_nan 
;                        ENDIF ELSE BEGIN
;                            ew_schan_only_d[mm, *, *, *] = !values.f_nan
;                            ew_main_only_d [mm, *, *, *] = !values.f_nan
;                        ENDELSE 
;                    ENDFOR 
;
;                    ; Average data
;                    IF ave_time MOD 60 EQ 0 THEN BEGIN ; assuming mulitple of hour
;                        get_timespan, a
;                        start_time_st      = time_string(a[0])
;                        temp_time_st       = strmid(start_time_st, 0, 11)+'00:00:00' ; start at 0 o'clock
;                        temp_time          = time_double(temp_time_st)
;                        num_times          = 24/(ave_time / 60)
;                        ew_hour_t        = dblarr(num_times)
;                        ew_main_hour_d   = fltarr(num_times, 32, 8, 128)
;                        ew_schan_hour_d  = fltarr(num_times, 32, 8, 128)
;                        FOR hour_i = 0, (24/(ave_time / 60))-1 DO BEGIN 
;                        ; time
;                            wanted_i = where(ew_t GE temp_time AND ew_t LT (temp_time+(ave_time*60)), wanted_count)
;                            IF wanted_count GE 1 THEN BEGIN 
;                                ew_hour_t[hour_i]        = ew_t[wanted_i[0]]
;                            ENDIF ELSE BEGIN 
;                                ew_hour_t[hour_i]        = temp_time
;                            ENDELSE 
;                            FOR defl_i = 0, 31 DO BEGIN 
;                                FOR pos_i = 0, 7 DO BEGIN 
;                                    FOR esa_i = 0, 127 DO BEGIN 
;                                        ; ew_main_only
;                                        wanted_i = where(ew_t GE temp_time AND ew_t LT (temp_time+(ave_time*60)) AND $
;                                                         finite(ew_main_only_d[*, defl_i, pos_i, esa_i]) EQ 1, wanted_count)
;                                        IF wanted_count GT 1 THEN BEGIN
;                                            mom_ew_main      = moment(ew_main_only_d[wanted_i, defl_i, pos_i, esa_i], /nan)
;                                            ew_main_hour_d[hour_i, defl_i, pos_i, esa_i] = mom_ew_main  [0]
;                                        ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
;                                            ew_main_hour_d[hour_i, defl_i, pos_i, esa_i] = ew_main_only_d[wanted_i, defl_i, pos_i, esa_i]
;                                        ENDIF ELSE BEGIN 
;                                            ew_main_hour_d[hour_i, defl_i, pos_i, esa_i] = !values.f_nan
;                                        ENDELSE 
;                                        ; ew_schan_only
;                                        wanted_i = where(ew_t GE temp_time AND ew_t LT (temp_time+(ave_time*60)) AND $
;                                                         finite(ew_schan_only_d[*, defl_i, pos_i, esa_i]) EQ 1, wanted_count)
;                                        IF wanted_count GT 1 THEN BEGIN
;                                            mom_ew_schan      = moment(ew_schan_only_d[wanted_i, defl_i, pos_i, esa_i], /nan)
;                                            ew_schan_hour_d[hour_i, defl_i, pos_i, esa_i] = mom_ew_schan  [0]
;                                        ENDIF ELSE IF wanted_count EQ 1 THEN BEGIN 
;                                            ew_schan_hour_d[hour_i, defl_i, pos_i, esa_i] = ew_schan_only_d[wanted_i, defl_i, pos_i, esa_i]
;                                        ENDIF ELSE BEGIN 
;                                            ew_schan_hour_d[hour_i, defl_i, pos_i, esa_i] = !values.f_nan
;                                        ENDELSE 
;                                    ENDFOR 
;                                ENDFOR 
;                            ENDFOR 
;                            ; increment time
;                            temp_time = temp_time + (ave_time * 60) ; increment in seconds
;                        ENDFOR 
;                    ENDIF ELSE stop ; need to add code here
;
;                    ew_schan_hour_d = total(ew_schan_hour_d, 2, /nan) ; sum over defl
;                    ew_main_hour_d  = total(ew_main_hour_d,  2, /nan) ; sum over defl
;                    temp_ew_schan_hour_d = fltarr(n_elements(ew_hour_t), 8) ; 8 position (summed over 4, 0 excluded)
;                    temp_ew_main_hour_d  = fltarr(n_elements(ew_hour_t), 8) ; 8 position (summed over 4, 0 excluded)
;                    FOR mm = 0L, n_elements(ew_hour_t)-1 DO BEGIN 
;                        IF mm LT n_elements(flux_hour_t) THEN BEGIN 
;                            temp_peak_i = alpha_peak_esa_d[mm]
;                            IF temp_peak_i EQ 127 THEN temp_peak_i = 126
;                            IF temp_peak_i EQ 0 THEN temp_peak_i = 1
;                            IF temp_peak_i EQ -1 THEN BEGIN
;                                temp_ew_schan_hour_d[mm, *] = !values.f_nan
;                                temp_ew_main_hour_d [mm, *] = !values.f_nan
;                            ENDIF ELSE BEGIN 
;                                temp_schan_hour_data = reform(ew_schan_hour_d[mm, *, temp_peak_i-1:temp_peak_i+1])
;                                temp_main_hour_data  = reform(ew_main_hour_d [mm, *, temp_peak_i-1:temp_peak_i+1])
;                                temp_ew_schan_hour_d[mm, *] = total(temp_schan_hour_data, 2, /nan) ; sum over esa
;                                temp_ew_main_hour_d [mm, *] = total(temp_main_hour_data,  2, /nan) ; sum over esa
;                            ENDELSE 
;                        ENDIF ELSE BEGIN ; don't know peak
;                            temp_ew_schan_hour_d[mm, *] = !values.f_nan
;                            temp_ew_main_hour_d [mm, *] = !values.f_nan
;                        ENDELSE 
;                    ENDFOR 
;                    avg_pos_schan_hour_d = $
;                      total(temp_ew_schan_hour_d  * sw_all_struct.position[0:n_elements(ew_hour_t)-1, *], 2, /nan)/$
;                      total(temp_ew_schan_hour_d, 2, /nan)
;                    avg_pos_main_hour_d = $
;                      total(temp_ew_main_hour_d  * sw_all_struct.position[0:n_elements(ew_hour_t)-1, *], 2, /nan)/$
;                      total(temp_ew_main_hour_d, 2, /nan)
;                    amp_toni = 11.25
;                    fullscale_toni = 19.6875
;                    alpha_ew_schan_hour_d = (amp_toni*(sinh((avg_pos_schan_hour_d/fullscale_toni))))
;                    alpha_ew_main_hour_d  = (amp_toni*(sinh((avg_pos_main_hour_d /fullscale_toni))))
;                    ew_schan_hour_d = alpha_ew_schan_hour_d
;                    ew_main_hour_d  = alpha_ew_main_hour_d
;
;            ; create variables
;                    var_ew_schan_only     = sat+'_alpha_ew_schan_only'
;                    var_ew_main_only      = sat+'_alpha_ew_main_only'
;                    store_data, var_ew_schan_only,     data = {x:ew_hour_t,     y:ew_schan_hour_d} 
;                    store_data, var_ew_main_only,      data = {x:ew_hour_t,     y:ew_main_hour_d} 
;            ; empty arrays to create memory space
;                    sw_all_struct   = 0
;                    ew_hour_t       = 0
;                    ew_d            = 0
;                    ew_schan        = 0
;                    ew_schan_hour_d = 0
;                    ew_main_hour_d  = 0
;                ENDIF 
;            ENDIF 
            ENDIF 
        ENDFOR 
    ENDFOR 

    store_data, ['s_valid_counts_norm_science', 'ra_trig_counts_norm_science'], /DEL
ENDFOR 

END 
