;+
; PROCEDURE: get_pla_mon_def_spec
;
; PURPOSE: to get the product related data and create the
;          corresponding  tplot variable
;
; INPUT: sat   - S/C id - 'A' or 'B'
;        prod  - product
;        class - classification
;        units - units
;
; KEYWORDS: diagn - if set extra diagnostic tplot variables are created
;
; OUTPUT: the output is provided via the tplot variables.
;
; CREATED BY: L. Ellis
;
; LAST MODIFICATION:
;
; MODIFICATION LIST:
;    12/18/2014 LBE Created from get_pla_mon_spec.pro
;    06/02/2015 LBE Change z range
;    06/04/2015 LBE Added main and schan versions
;-
PRO get_pla_mon_def_spec, sat, prod, units, diagn = diagn
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

seek_files, sat, 'science', filesfound
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN
ENDIF

n_prod = prod+'_norm_science'
;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))

        prod_name = 'epoch5_mon'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 1 THEN BEGIN
            
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            prod_name = n_prod
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable

            cdf_data = FLOAT(cdf_data)
            ovfl = WHERE(cdf_data EQ 507904,  covfl)
            IF covfl GT 0 THEN BEGIN
                cdf_data(ovfl) = !VALUES.F_NaN
            ENDIF

            fill = WHERE(cdf_data EQ -1,  cfill)
            IF cfill GT 0 THEN BEGIN
                cdf_data(fill) = !VALUES.F_NaN
            ENDIF

            ; make versions for main and schan
            cdf_data_main  = cdf_data
            cdf_data_schan = cdf_data
            num_esa_cdf_main   = intarr(n_elements(epoch)) ; for number of esa steps we have
            num_esa_cdf_schan  = intarr(n_elements(epoch))
            cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
            IF epoch1info.maxrec GE 1 THEN BEGIN
                cdf_varget, cdf_id, 'epoch1',  epoch1, rec_count = epoch1info.maxrec+1, /zvariable
                cdf_varget, cdf_id, 's_chan1', schan1, rec_count = epoch1info.maxrec+1, /zvariable
                no_switch = where(schan1 EQ 0, count)
                IF count GT 0 THEN schan1[no_switch] = 128
                FOR epoch_i = 0L, n_elements(epoch)-1 DO BEGIN 
                    mins = where(epoch1 GE epoch[epoch_i]-30000 AND epoch1 LE epoch[epoch_i]+270000 AND $
                                 finite(schan1) EQ 1 AND schan1 GT 0, num_min)
                    IF num_min GT 5 THEN BEGIN 
                        FOR min_i = 0, num_min-2 DO BEGIN ; look for duplicates
                            IF abs(epoch1[mins[min_i]] -  epoch1[mins[min_i+1]]) LT 30 THEN BEGIN 
                                IF schan1[mins[min_i]] EQ schan1[mins[min_i+1]] THEN BEGIN 
                                    num_min = num_min-1 ; duplicate found
                                ENDIF 
                            ENDIF 
                        ENDFOR 
                        IF num_min GT 5 THEN stop ; don't expect this
                    ENDIF 
                    IF num_min NE 5 THEN BEGIN 
                        cdf_data_main [*, *, epoch_i] = !values.f_nan
                        cdf_data_schan[*, *, epoch_i] = !values.f_nan
                    ENDIF ELSE BEGIN 
                        bad_schan = where(finite(schan1[mins]) EQ 0 OR schan1[mins] LT 1, count)
                        IF count GT 0 THEN BEGIN 
                            cdf_data_main [*, *, epoch_i] = !values.f_nan
                            cdf_data_schan[*, *, epoch_i] = !values.f_nan
                        ENDIF 
                        min_esa = min(schan1[mins], /nan)
                        max_esa = max(schan1[mins], /nan) 
                        min_esa_i = min_esa/4 ; convert from 128 steps to 32 bins
                        max_esa_i = max_esa/4
                        IF max_esa_i EQ 32 THEN max_esa_i = 31
                        IF min_esa_i LT 32 THEN cdf_data_main [min_esa_i:31, *, epoch_i] = !values.f_nan
                        cdf_data_schan[ 0:max_esa_i, *, epoch_i] = !values.f_nan
                        num_esa_cdf_main [epoch_i] = min_esa_i * 4
                        num_esa_cdf_schan[epoch_i] = (31-max_esa_i) * 4
                    ENDELSE 
                ENDFOR 
            ENDIF ELSE BEGIN 
                cdf_data_main [*, *, *] = !values.f_nan
                cdf_data_schan[*, *, *] = !values.f_nan
            ENDELSE 

            prod_sum_en = TOTAL(cdf_data,       1, /NaN)
            main_sum    = TOTAL(cdf_data_main,  1, /NaN)
            schan_sum   = TOTAL(cdf_data_schan, 1, /NaN)

            prod_name = 's_chan5_mon'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan5_mon, rec_count = epochinfo.maxrec+1, /zvariable

            prod_name = 'error5_mon'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error5_mon, rec_count = epochinfo.maxrec+1, /zvariable

            IF ifwd EQ 0 THEN BEGIN
                converted_epoch = REFORM   (time_double(epoch, /epoch))
                data            = TRANSPOSE(prod_sum_en)
                full_data       = TRANSPOSE(cdf_data)
                data_main       = TRANSPOSE(main_sum)
                data_schan      = TRANSPOSE(schan_sum)
                num_esa_main    = REFORM   (num_esa_cdf_main)
                num_esa_schan   = REFORM   (num_esa_cdf_schan)
                s_chan5_mon     = REFORM   (cdf_s_chan5_mon)
                error5_mon      = REFORM   (cdf_error5_mon)
            ENDIF ELSE BEGIN
                converted_epoch = [converted_epoch, REFORM   (time_double(epoch, /epoch))]
                data            = [data,            TRANSPOSE(prod_sum_en)]
                full_data       = [full_data,       TRANSPOSE(cdf_data)]
                data_main       = [data_main,       TRANSPOSE(main_sum)]
                data_schan      = [data_schan,      TRANSPOSE(schan_sum)]
                num_esa_main    = [num_esa_main,    REFORM   (num_esa_cdf_main)]
                num_esa_schan   = [num_esa_schan,   REFORM   (num_esa_cdf_schan)]
                s_chan5_mon     = [s_chan5_mon,     REFORM   (cdf_s_chan5_mon)]
                error5_mon      = [error5_mon,      REFORM   (cdf_error5_mon)]
            ENDELSE
            ifwd =  ifwd + 1
        ENDIF
        cdf_close, cdf_id
    ENDIF
ENDFOR

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
IF n_elements(converted_epoch) GT 0 THEN BEGIN 
    findfill = WHERE(error5_mon EQ 0, cff)
    IF cff GT 0 THEN BEGIN
        converted_epoch = converted_epoch(findfill)
        s_chan5_mon     = s_chan5_mon    (findfill)
        error5_mon      = error5_mon     (findfill)
        data            = data           (findfill, *)
        full_data       = full_data      (findfill, *, *)
        data_main       = data_main      (findfill, *)
        data_schan      = data_schan     (findfill, *)
    ENDIF
    get_pla_energies, sat, esa_step

    ntime     = N_ELEMENTS(converted_epoch)
    varr      = FLTARR(ntime, 8) ; 8 defl bins
    defl_step = float(indgen(8))
    FOR iff = 0L, ntime-1 DO varr(iff, *) = defl_step

;------------------------------------------------------------------
; Convert Units
; 0.0128 seconds per defl step * 32 steps = 0.4096 seconds at each ESA step
; 0.16384 only 1 ESA step (instead of 4) at full resolution 
;------------------------------------------------------------------
    IF STRUPCASE(units) EQ 'C/S' THEN BEGIN
        ;data = data / 8.192
        stop ; need to figure this out for defl step
    ENDIF
    IF STRUPCASE(units) EQ 'C/MIN' THEN BEGIN
        data       = data       / 5 ; because 5-min data
        data_main  = data_main  / 5 ; because 5-min data
        data_schan = data_schan / 5 ; because 5-min data
    ENDIF

    ; divide by 4 because 4 defl steps in each group
    data       = data       / 4
    data_main  = data_main  / 4
    data_schan = data_schan / 4

    ; divide by 128 because 128 esa steps
    data       = data       / 128
    FOR epoch_i = 0L, n_elements(converted_epoch)-1 DO BEGIN 
        data_main [epoch_i, *] = data_main [epoch_i, *] / num_esa_main [epoch_i]
        data_schan[epoch_i, *] = data_schan[epoch_i, *] / num_esa_schan[epoch_i]
    ENDFOR 

;------------------------------------------------------------------
    vname = prod+'_'+units+'_norm_science'
    main_name  = vname+'_main'
    schan_name = vname+'_schan'

    IF STRUPCASE(units) EQ 'C/S' THEN BEGIN
        ztitle = 'C/S!C!C/Def/ESA'
    ENDIF ELSE IF STRUPCASE(units) EQ 'C/MIN' THEN BEGIN
        ztitle = 'C/Min/!C!CDefl/ESA'
    ENDIF ELSE BEGIN
        ztitle = 'COUNTS/!C!CDefl/ESA'
    ENDELSE
    ytitle = prod+'!C!CDefl. Step'

    store_data, vname, data = {x:converted_epoch, y:data, v:varr}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ylog:0, $
                        zlog:1, $
                        ystyle:1, $
                        yrange:[0, 7], $
                        zrange:[1, 1e2], $
                        ytitle:ytitle, $
                        ztitle:ztitle $
                       }

    store_data, main_name, data = {x:converted_epoch, y:data_main, v:varr}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ylog:0, $
                        zlog:1, $
                        ystyle:1, $
                        yrange:[0, 7], $
                        zrange:[1, 1e2], $
                        ytitle:ytitle, $
                        ztitle:ztitle $
                       }

    store_data, schan_name, data = {x:converted_epoch, y:data_schan, v:varr}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ylog:0, $
                        zlog:1, $
                        ystyle:1, $
                        yrange:[0, 7], $
                        zrange:[1, 1e2], $
                        ytitle:ytitle, $
                        ztitle:ztitle $
                       }
ENDIF 

END
