;+
; FUNCTION: get_pla_h1_alpha
;
; PURPOSE: To create the data structure of the 'h_alpha' product
;
; INPUT: sat: S/C id - 'A' or 'B'
;
; OUTPUT: the data structure is returned
;       : data.data is [time, 8, 8, 20] for [time, DP-3:DP+4, summed pos, EP+4:EP-15]
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;   01-15-2007: LBE
;               Added indices for energy peak and schan
;   04-28-2009: LBE: Added full_counts (puts into full array)
;   06-02-2009: LBE: Check for if have enough memory for full_counts
;   09-02-2011: LBE: Add index for deflection peak.
;   03-15-2012: LBE: Check for schan = -1.
;   09-27-2012: LBE: Added set_to_beg keyword.
;   11-28-2016: LBE: Added from_full keyword.
;   07-23-2020: LBE: Increase number of records we can handle.
;-

FUNCTION get_pla_h1_alpha, sat, set_to_beg = set_to_beg, from_full = from_full
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose
get_err_no = 0 & get_err_msg = ''

prod        = 'h_alpha'
IF keyword_set(from_full) THEN nenergy     = 128 ELSE nenergy     =  20
IF keyword_set(from_full) THEN ndeflection =  32 ELSE ndeflection =  8
nposition   =  8

seek_files, sat, 'science', filesfound ; find files that correspond to time interval
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN, 0
ENDIF

;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))

        prod_name = 'epoch1'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'h_alpha'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'proton_peak'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_proton_peak, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 's_chan1'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan1, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'error1'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error1, rec_count = epochinfo.maxrec+1, /zvariable

            IF ifwd EQ 0 THEN BEGIN
                converted_epoch = REFORM(time_double(epoch, /epoch))
                data            = TRANSPOSE(cdf_data)
                energy_peak     = REFORM(cdf_proton_peak(0, *))
                position_peak   = REFORM(cdf_proton_peak(1, *))
                deflection_peak = REFORM(cdf_proton_peak(2, *))
                source_array    = REFORM(cdf_proton_peak(3, *))
                s_chan1         = REFORM(cdf_s_chan1)
                error1          = REFORM(cdf_error1)
            ENDIF ELSE BEGIN
                converted_epoch = [converted_epoch, REFORM(time_double(epoch, /epoch))]
                data            = [data,            TRANSPOSE(cdf_data)]
                energy_peak     = [energy_peak,     REFORM(cdf_proton_peak(0, *))]
                position_peak   = [deflection_peak, REFORM(cdf_proton_peak(1, *))]
                deflection_peak = [deflection_peak, REFORM(cdf_proton_peak(2, *))]
                source_array    = [source_array,    REFORM(cdf_proton_peak(3, *))]
                s_chan1         = [s_chan1,         REFORM(cdf_s_chan1)]
                error1          = [error1,          REFORM(cdf_error1)]
                
            ENDELSE
            
            ifwd =  ifwd + 1
            
        ENDIF
        
        cdf_close, cdf_id

    ENDIF
    
ENDFOR

IF ifwd EQ 0 THEN BEGIN

    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0

ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error1 EQ 0, cff)
IF cff GT 0 THEN BEGIN
    
    converted_epoch = converted_epoch(findfill)
    energy_peak = energy_peak(findfill)
    position_peak = position_peak(findfill)
    deflection_peak = deflection_peak(findfill)
    source_array = source_array(findfill)
    s_chan1 = s_chan1(findfill)
    error1 = error1(findfill)
    data = data(findfill, *, *, *)
    
ENDIF

;------------------------------------------------------------------
; Check individual arrays for value range validity
;------------------------------------------------------------------

icepoch =  WHERE(converted_epoch LT (time_double('1958-01-01')) OR $
                 converted_epoch GT (time_double('2050-01-01')), cicepoch)
IF cicepoch GT 0 THEN STOP

data =  FLOAT(data)
icdata = WHERE(data EQ -1, cicdata)
IF cicdata GT 0 THEN BEGIN
    data(icdata) =  !VALUES.F_NAN
ENDIF

icenp = WHERE(energy_peak LT 0 OR energy_peak GT 127, cicenp)
;IF cicenp GT 0 THEN STOP

icpop = WHERE(position_peak LT 0 OR position_peak GT 31, cicpop)
;IF cicpop GT 0 THEN STOP

icdep = WHERE(deflection_peak LT 0 OR deflection_peak GT 31, cicdep)
;IF cicdep GT 0 THEN STOP

isar =  WHERE(source_array NE 0 AND source_array NE 1, cisar)
;IF cisar GT 0 THEN STOP

isch =  WHERE(s_chan1 LT 0 OR s_chan1 GT 127,  cisch)
;IF cisch GT 0 THEN STOP

ierr =  WHERE(error1 NE 0 AND error1 NE 1,  cierr)
;IF cierr GT 0 THEN STOP

;------------------------------------------------------------------
;convert energy/deflection/position/s_channel steps to actual values
;------------------------------------------------------------------
get_pla_energies, sat, esa_step             ; get energy bin values range
get_pla_theta_angles, sat, prod, theta_step ; get polar (deflection) angles range
get_pla_phi_angles, sat, prod, phi_step     ; get azimuthal (position) angles range
theta_bins = indgen(32)
phi_bins   = indgen(8)

position     = FLTARR(N_ELEMENTS(converted_epoch), nposition)
position_i   = FLTARR(N_ELEMENTS(converted_epoch), nposition)
deflection   = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
deflection_i = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
energy       = FLTARR(N_ELEMENTS(converted_epoch), nenergy)
IF n_elements(converted_epoch) LE 50000 THEN BEGIN  ; limit of memory -- check when have new computer
    full_counts = fltarr(N_elements(converted_epoch), 32, 8, 128) ; [time, defl, summed pos, esa]
    full_counts[*] = !values.f_nan
    have_full_counts = 1
ENDIF ELSE have_full_counts = 0
FOR ii = 0l, N_ELEMENTS(converted_epoch)-1 DO BEGIN

    ;-----------------------------------------------------------------
    ; If the energy peak is close to the boundaries the 20 energy
    ; steps start from 0 or 127. This results in the peak not being
    ; in the middle.
    ;-----------------------------------------------------------------
    IF keyword_set(from_full) THEN energy[ii, *] = esa_step ELSE BEGIN 
       IF energy_peak(ii) LT 15 THEN BEGIN
          energy(ii, *) = esa_step(0:19)
       ENDIF ELSE BEGIN
          IF energy_peak(ii) GT 123 THEN BEGIN
             energy(ii, *) = esa_step(108:127)
          ENDIF ELSE BEGIN
             energy(ii, *) = esa_step((energy_peak(ii)-15):(energy_peak(ii)+4))
          ENDELSE
       ENDELSE
    endelse 

    ;-----------------------------------------------------------------
    ; If the deflection peak is close to the boundaries the 
    ; 8 deflection steps start from 0 or 31. This results in the 
    ; peak not being in the middle.
    ;-----------------------------------------------------------------
    IF keyword_set(from_full) THEN deflection[ii, *] = theta_step ELSE BEGIN 
       IF deflection_peak(ii) GT 27 THEN BEGIN
          deflection  (ii, *) = theta_step(24:31)
          deflection_i(ii, *) = theta_bins(24:31)
       ENDIF ELSE BEGIN
          IF deflection_peak(ii) LT 3 THEN BEGIN
             deflection  (ii, *) = theta_step(0:7)
             deflection_i(ii, *) = theta_bins(0:7)
          ENDIF ELSE BEGIN
             deflection  (ii, *) = theta_step((deflection_peak(ii)-3):(deflection_peak(ii)+4))
             deflection_i(ii, *) = theta_bins((deflection_peak(ii)-3):(deflection_peak(ii)+4))
          ENDELSE
       ENDELSE
    endelse 

    ;-----------------------------------------------------------------
    ; The position peak is not used. Instead all positions are bined
    ; in 8 bins.
    ;-----------------------------------------------------------------
    position  (ii, *) = phi_step
    position_i(ii, *) = phi_bins

    ; fill full_counts
    IF have_full_counts EQ 1 THEN BEGIN 
        IF energy_peak    [ii] LT 15 THEN ep = 15 ELSE IF energy_peak    [ii] GT 123 THEN ep = 123 ELSE ep = energy_peak    [ii]
        IF deflection_peak[ii] LT  3 THEN dp =  3 ELSE IF deflection_peak[ii] GT  27 THEN dp =  27 ELSE dp = deflection_peak[ii]
        full_counts[ii, dp-3:dp+4, *, ep-15:ep+4] = data[ii, *, *, *]
    ENDIF 
ENDFOR

;convert s_chan1 to energy
s_chan_i = s_chan1
s_chan1  = float(s_chan1)
bad_i = where(s_chan1 LT 0, bad_count, complement = good_i, ncomplement = good_count)
IF  bad_count GT 0 THEN s_chan1[ bad_i] = !values.f_nan
IF good_count GT 0 THEN s_chan1[good_i] = esa_step(s_chan1[good_i])
no_schan_i = where(s_chan_i EQ 0, count)
IF count GT 0 THEN BEGIN ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan1 [no_schan_i] = 0.1
ENDIF 

;------------------------------------------------------------------
; Time tag of the middle of the bin
;------------------------------------------------------------------
IF keyword_set(set_to_beg) EQ 0 THEN $
  converted_epoch = converted_epoch + 30. ; time shift by half minute

;------------------------------------------------------------------
; Integration time
;------------------------------------------------------------------
packets = N_ELEMENTS(converted_epoch)
integ_t = FLTARR(packets, ndeflection, nposition, nenergy) + 12.8e-3 ; ms

;------------------------------------------------------------------
; Construct the data structure
;------------------------------------------------------------------
IF have_full_counts EQ 1 THEN $
  retdata = {$
          project:           'STEREO/PLASTIC', $
          prod_name:         prod, $
          units_name:        'Counts', $
          ntime:             N_ELEMENTS(converted_epoch), $
          time:              converted_epoch, $
          integ_t:           integ_t, $
          data:              data, $
          valid:             1, $
          nenergy:           nenergy, $
          energy:            energy, $
          energy_peak_i:     energy_peak, $
          nposition:         nposition, $
          position:          position, $
          position_i:        position_i, $
          ndeflection:       ndeflection, $
          deflection:        deflection, $
          deflection_i:      deflection_i, $
          deflection_peak_i: deflection_peak, $     
          source:            source_array, $
          s_chan:            s_chan1, $
          s_chan_i:          s_chan_i, $
          error:             error1, $
          full_array:        full_counts $
          } $
ELSE $
  retdata = {$
          project:           'STEREO/PLASTIC', $
          prod_name:         prod, $
          units_name:        'Counts', $
          ntime:             N_ELEMENTS(converted_epoch), $
          time:              converted_epoch, $
          integ_t:           integ_t, $
          data:              data, $
          valid:             1, $
          nenergy:           nenergy, $
          energy:            energy, $
          energy_peak_i:     energy_peak, $
          nposition:         nposition, $
          position:          position, $
          position_i:        position_i, $
          ndeflection:       ndeflection, $
          deflection:        deflection, $
          deflection_i:      deflection_i, $
          deflection_peak_i: deflection_peak, $     
          source:            source_array, $
          s_chan:            s_chan1, $
          s_chan_i:          s_chan_i, $
          error:             error1 $
          }

RETURN, retdata

END
