;+
; PROCEDURE: get_pla_en_spec
;
; PURPOSE: to get the product related data and create the
;          corresponding  tplot variable
;
; INPUT: sat   - S/C id - 'A' or 'B'
;        prod  - product
;        class - classification
;        units - units
;
; KEYWORDS: diagn - if set extra diagnostic tplot variables are created
;
; OUTPUT: the output is provided via the tplot variables.
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION LIST:
;   01/11/2008 LBE: Added ajustment for schan switch.
;   03/26/2008 LBE: Changed ylim for s_chan
;   06/10/2008 LBE: Added _s_chan_esa
;   08/12/2008 LBE: Added _peak [ESA, Pos, Defl, Array]
;   08/13/2008 LBE: Added nodisplay keyword
;   11/07/2008 LBE: Fixed bug in when to plot _s_chan and _peak
;   12/10/2008 LBE: Added variables for schan only and main only
;   01/13/2009 LBE: Changed loop variable to long.
;   04/15/2010 LBE: Added num_summed tplot variable.
;   05/05/2010 LBE: Only use num_summed for 5-min products.
;   05/17/2010 LBE: Added positions keyword.  
;   07/12/2010 LBE: Fix so catches pos 0 by itself.
;   07/13/2010 LBE: Fixed bug and change name when lim pos.
;   07/28/2011 LBE: Change for s_chan_i NaN.
;   12/22/2011 LBE: Fix sw_pri for when only one record.
;   02/02/2012 LBE: Fix bug in wap_ssd_tcr when only one record.
;   09/27/2012 LBE: Added set_to_beg keyword.
;   01/23/2013 LBE: Add erase gaps keywords (to make spears in plots disappear).
;   07/24/2013 LBE: Fix bug in sw_priority when only one record.
;   07/14/2014 LBE: Fix bug in wap_no_ssd_dcr when only one record.
;-
PRO get_pla_en_spec, sat, prod, class, units, diagn=diagn, nodisplay = nodisplay, positions = positions, $
                     set_to_beg = set_to_beg, erase_gaps = erase_gaps
compile_opt strictarrsubs  
  
COMMON get_error, get_err_no, get_err_msg, default_verbose
          
@tplot_com

dat = get_pla_data(sat, prod, class, set_to_beg = set_to_beg, erase_gaps = erase_gaps)

IF strlowcase(units) EQ 'adj_counts' OR strlowcase(units) EQ 'adj_c/s' THEN BEGIN
    read_schan_conv, sat, prod, dat, units
ENDIF 
IF get_err_no NE 0 THEN RETURN

;------------------------------------------------------------------
; Convert Units
;------------------------------------------------------------------
IF STRUPCASE(units) NE 'COUNTS' AND strupcase(units) NE 'ADJ_COUNTS' THEN BEGIN
    dat = convert_plastic_units(dat, units)
ENDIF
;------------------------------------------------------------------
  
ytitle = 'ST' + STRUPCASE(sat) +'/PLA' + '!C!C'

; !!!! Be carefull: NaN data when totaled give zero.
    
IF strlowcase(dat.units_name) EQ 'counts' OR strlowcase(dat.units_name) EQ 'adj_counts' THEN norm = 1. ELSE norm =  dat.nposition * dat.ndeflection

prodname = 'ST' + sat + '_PLA_ENSPEC_' + STRUPCASE(units) + '_' + STRUPCASE(prod)
IF prod EQ 'h_alpha' OR $
  prod EQ 'h+peak' OR $
  prod EQ 'he++peak' OR $
  prod EQ 'he++tcr' THEN BEGIN
    ytitle = ytitle + prod
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        ytitle = ytitle + ' lim_pos'
        bad_i = where(positions LT 0 OR positions GT 7, bad_count)
        IF bad_count GT 0 THEN BEGIN
            print, 'ERROR: bad position'
            stop
        ENDIF 
        IF norm NE 1 THEN norm = (n_elements(positions)) * dat.ndeflection
        temp_data = total(dat.data, 2, /nan) ; sum over defl
        zdata = total(temp_data[*, positions, *], 2, /nan) / norm
        ; don't need to replace /Nan because not using full distribution
    ENDIF ELSE zdata = TOTAL(TOTAL(dat.data, 2, /NaN), 2, /NaN) / norm
ENDIF

IF prod EQ 'sw_z>2_h' THEN BEGIN
    ytitle = ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        ytitle = ytitle + ' lim_pos'
        bad_i = where(positions LT 0 OR positions GT 7, bad_count)
        IF bad_count GT 0 THEN BEGIN
            print, 'ERROR: bad position'
            stop
        ENDIF 
        IF norm NE 1 THEN norm = (n_elements(positions)) * dat.ndeflection
        temp_data = total(dat.data, 2, /nan) ; sum over defl
        zdata = total(temp_data[*, positions, *], 2, /nan) / norm
        ; don't need to replace /Nan because not using full distribution
    ENDIF ELSE zdata = TOTAL(TOTAL(dat.data(*, *, *, *), 2, /NaN), 2, /NaN) / norm
    prodname = prodname + '_' + STRING(class, FORMAT = '(i2.2)')
ENDIF

IF prod EQ 'sw_z>2_l' THEN BEGIN
    ytitle = ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        ytitle = ytitle + ' lim_pos'
        bad_i = where(positions LT 0 OR positions GT 7, bad_count)
        IF bad_count GT 0 THEN BEGIN
            print, 'ERROR: bad position'
            stop
        ENDIF 
        IF norm NE 1 THEN norm = (n_elements(positions)) * dat.ndeflection
        zdata = total(dat.data[*, positions, *], 2, /nan) / norm
    ENDIF ELSE zdata = TOTAL(REFORM(dat.data(*, *, *)), 2, /NaN) / norm
    prodname = prodname + '_' + STRING(class, FORMAT = '(i2.2)')
ENDIF
  
IF prod EQ 'sw_priority' THEN BEGIN
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        print, 'ERROR: sw_priority does not have position information'
        stop
    ENDIF ELSE BEGIN
        size = size(dat.data)
        IF size[0] LT 3 THEN zdata = TOTAL(REFORM(dat.data(*, *)), 1, /NaN) / norm $ ; only one record
          ELSE zdata = TOTAL(REFORM(dat.data(*, *, *)), 2, /NaN) / norm
    ENDELSE 
    prodname = prodname + '_' + STRING(class, FORMAT = '(i2.2)')
    ytitle = ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
ENDIF

IF prod EQ 'wap_ssd_tcr' OR prod EQ 'wap_ssd_dcr' OR prod EQ 'wap_no_ssd_dcr' THEN BEGIN
    ytitle = ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        ytitle = ytitle + ' lim_pos'
        IF prod EQ 'wap_no_ssd_dcr' THEN $
          bad_i = where(positions LT 0 OR positions GT 7, bad_count) ELSE $
          bad_i = where(positions LT 0 OR positions GT 3, bad_count) 
        IF bad_count GT 0 THEN BEGIN
            print, 'ERROR: bad position'
            stop
        ENDIF 
        IF norm NE 1 THEN norm = (n_elements(positions)) * dat.ndeflection
        zdata = total(dat.data[*, positions, *], 2, /nan) / norm
    ENDIF ELSE BEGIN
        temp_size = size(dat.data)
        IF temp_size[0] EQ 2 THEN zdata = TOTAL(dat.data(*, *), 1, /NaN) / norm $ ; only one time
          ELSE zdata = TOTAL(dat.data(*, *, *), 2, /NaN) / norm
    ENDELSE 
    prodname = prodname + '_' + STRING(class, FORMAT = '(i2.2)')
ENDIF

IF prod EQ 'wap_priority_ssd' OR prod EQ 'wap_priority_no_ssd' THEN BEGIN
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        print, 'ERROR: sw_priority does not have position information'
        stop
    ENDIF ELSE zdata = dat.data / norm
    prodname = prodname + '_' + STRING(class, FORMAT = '(i2.2)')
    ytitle = ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
ENDIF

IF prod EQ 'class' THEN BEGIN
    ytitle = ytitle + prod + '!C!CEnergy (keV)'
    IF size(positions, /dimensions) GT 0 THEN BEGIN 
        ytitle = ytitle + ' lim_pos'
        bad_i = where((positions LT 0) OR (positions GT 7), bad_count)
        IF bad_count GT 0 THEN BEGIN
            print, 'ERROR: bad position'
            stop
        ENDIF 
        IF norm NE 1 THEN norm = (n_elements(positions)) * dat.ndeflection
        temp_data = total(dat.data, 2, /nan) ; sum over defl
        zdata = total(temp_data[*, positions, *], 2, /nan) / norm
    ENDIF ELSE zdata =   TOTAL(TOTAL(dat.data, 2, /NaN), 2, /NaN) / norm
ENDIF

;------------------------------------------------------------------
; Save spectra in tplot variables
;------------------------------------------------------------------
temp_ytitle = ytitle + '!C!CE/q (keV/q)'
ztitle = STRUPCASE(dat.units_name)

IF size(positions, /dimensions) GT 0 THEN BEGIN
    prodname = prodname+'_pos'
    FOR ii = 0, n_elements(positions)-1 DO BEGIN 
        prodname = prodname+'_'+strcompress(string(positions[ii]), /remove_all)
    ENDFOR 
ENDIF 
IF size(positions, /dimensions) EQ 0 THEN temp_positions = indgen(32) ELSE $
  temp_positions = positions
IF ztitle EQ 'COUNTS' OR ztitle EQ 'C/S' THEN zmax = 1000 ELSE zmax = 100000 ; higher range for with schan conv
store_data, prodname, data = {x:REFORM(dat.time), y:zdata, v:dat.energy}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:1, $
                    zlog:1, $
                    yrange:[0.2, 100], $
                    zrange:[0.1, zmax], $
                    ytitle:temp_ytitle, $
                    positions: temp_positions, $
                    ztitle:ztitle $
                   }
tplot_var = prodname
IF keyword_set(nodisplay) EQ 0 THEN tplot, tplot_var,  add_var = 999 ; add panel to the bottom of the plot

store_data, prodname+'_s_chan', data = {x:REFORM(dat.time), y:dat.s_chan}, $
            dlim = {panel_size:1, $
                    ylog:1, $
                    yrange:[0.2, 100], $
                    ytitle:'S-Chan (keV)'}

store_data, prodname+'_s_chan_esa', data = {x:REFORM(dat.time), y:dat.s_chan_i}, $
            dlim = {panel_size:1, $
                    ylog:0, $
                    yrange:[0, 127], $
                    ytitle:'S-Chan ESA Step'}

IF prod EQ 'h_alpha' OR $
  prod EQ 'h+peak' OR $
  prod EQ 'he++peak' OR $
  prod EQ 'he++tcr' THEN BEGIN
    store_data, prodname+'_peak_esa', data = {x:reform(dat.time), y:dat.energy_peak_i}, $
                dlim = {panel_size:1, $
                        ylog:0, $
                        yrange:[0, 127], $
                        ytitle:'Energy Peak ESA Step'}
    
ENDIF

; Set up variables for schan only and main channel only
schan_data = zdata
main_data  = zdata
; check for invalid schan
bad_i = where(dat.s_chan_i EQ -1, count)
IF count GT 0 THEN BEGIN
    schan_data[bad_i, *] = !values.f_nan
    main_data [bad_i, *] = !values.f_nan
ENDIF 
IF prod EQ 'h_alpha' THEN BEGIN 
    all_main_i = where(dat.s_chan_i GT dat.energy_peak_i+4 , count) ; all main channel
    IF count GT 0 THEN schan_data[all_main_i, *] = !values.f_nan
    mixed_i = where(dat.s_chan_i GE dat.energy_peak_i-15 AND dat.s_chan_i LE dat.energy_peak_i+4, count) ; has some of each channel
    IF count GT 0 THEN BEGIN
        FOR ii = 0L, count-1 DO BEGIN
            relative_i = dat.s_chan_i[mixed_i[ii]] - dat.energy_peak_i[mixed_i[ii]] + 15
            IF relative_i GT 0 THEN schan_data[mixed_i[ii], 0:relative_i-1] = !values.f_nan 
            main_data[mixed_i[ii], relative_i:19] = !values.f_nan
        ENDFOR
    ENDIF 
    all_schan_i = where(dat.s_chan_i LT dat.energy_peak_i-15, count) ; all s-chan
    IF count GT 0 THEN main_data[all_schan_i, *] = !values.f_nan
ENDIF ELSE IF prod EQ 'h+peak' OR $
  prod EQ 'he++peak' OR $
  prod EQ 'he++tcr' THEN BEGIN
    all_main_i = where(dat.s_chan_i GT dat.energy_peak_i+4 , count) ; all main channel
    IF count GT 0 THEN schan_data[all_main_i, *] = !values.f_nan
    mixed_i = where(dat.s_chan_i GE dat.energy_peak_i-5 AND dat.s_chan_i LE dat.energy_peak_i+4, count) ; has some of each channel
    IF count GT 0 THEN BEGIN
        FOR ii = 0L, count-1 DO BEGIN
            relative_i = dat.s_chan_i[mixed_i[ii]] - dat.energy_peak_i[mixed_i[ii]] + 5
            IF relative_i GT 0 THEN schan_data[mixed_i[ii], 0:relative_i-1] = !values.f_nan
            main_data[mixed_i[ii], relative_i:9] = !values.f_nan
        ENDFOR
    ENDIF 
    all_schan_i = where(dat.s_chan_i LT dat.energy_peak_i-5, count) ; all s-chan
    IF count GT 0 THEN main_data[all_schan_i, *] = !values.f_nan
ENDIF ELSE BEGIN
    size = size(zdata)
    IF size[0] EQ 0 THEN BEGIN 
        IF finite(dat.s_chan_i) EQ 1 THEN schan_data = !values.f_nan
        IF finite(dat.s_chan_i) EQ 1 AND dat.s_chan_i LT 128 THEN main_data = !values.f_nan
    ENDIF ELSE IF size[0] EQ 1 THEN BEGIN 
        IF finite(dat.s_chan_i[0]) EQ 1 THEN schan_data[0:dat.s_chan_i[0]-1] = !values.f_nan
        IF finite(dat.s_chan_i[0]) EQ 1 AND dat.s_chan_i[0] LT 128 THEN main_data[dat.s_chan_i[0]:127] = !values.f_nan
    ENDIF ELSE BEGIN 
        FOR ii = 0L, size[1]-1 DO BEGIN
            IF finite(dat.s_chan_i[ii]) EQ 1 THEN schan_data[ii, 0:dat.s_chan_i[ii]-1] = !values.f_nan
            IF finite(dat.s_chan_i[ii]) EQ 1 AND dat.s_chan_i[ii] LT 128 THEN main_data[ii, dat.s_chan_i[ii]:127] = !values.f_nan
        ENDFOR
    ENDELSE 
ENDELSE 

IF prod NE 'h_alpha' AND prod NE 'h+peak' AND prod NE 'he++peak' AND prod NE 'he++tcr' THEN BEGIN 
    temp_ytitle = ytitle + ' NumSummed'
    store_data, prodname+'_num_summed', data = {x:REFORM(dat.time), y:dat.summed, v:dat.energy}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ytitle:temp_ytitle $
                       }
ENDIF 

temp_ytitle = ytitle + ' S-Chan!C!CE/q (keV/q)'
store_data, prodname+'_schan_only', data = {x:REFORM(dat.time), y:schan_data, v:dat.energy}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:1, $
                    zlog:1, $
                    positions: temp_positions, $
                    yrange:[0.2, 100], $
                    zrange:[0.1, zmax], $
                    ytitle:temp_ytitle, $
                    ztitle:ztitle $
                   }

temp_ytitle = ytitle + ' Main Ch.!C!CE/q (keV/q)'
store_data, prodname+'_main_only', data = {x:REFORM(dat.time), y:main_data, v:dat.energy}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:1, $
                    zlog:1, $
                    positions: temp_positions, $
                    yrange:[0.2, 100], $
                    zrange:[0.1, zmax], $
                    ytitle:temp_ytitle, $
                    ztitle:ztitle $
                   }


END
