;+
;PROCEDURE: calc_alpha_bulk
;PURPOSE:
;  Calculate proton bulk parameters from 1 minute resolution RA_Trigger rate.
;
;  This is the second version of calc_proton_bulk.  The original version is
;  at best slow and memory intensive.  This version is intended to run more
;  efficiently.
;
;CREATED BY: K. Simunac
;
;LAST MODIFICATION: 12 September 2011
;
;MODIFICATION HISTORY:
;	01 November 2010:  KDCS created
;	12 September 2011: added routines to create 1 hour and 10 min averages.
;       09/24/2020: LBE: Add table_location to call to calc_alpha_data
;-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;							  ;
; Below is a routine to calculate density, bulk speed,	  ;
; temperature and thermal speed using full	 	  ;
; resolution rate data from RA_Trigger or SF0.		  ;
;	 						  ;					 	  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO calc_alpha_bulk, sat, files, which_epoch, time, num_days, table_location

COMMON share1_high_res
COMMON com_high_res
tab = string(9B)
tplot_options, 'color', 0

num_records1 = final_count ; final count of number of records from pl_read_cdf_high_res
have_data = 1

; Define some variables for storing error flags and s-channel switch information
    s_chan = intarr(num_records1)
    s_chan(*) = 0 
    error1 = intarr(num_records1)
    rate_name = ' '
    std_error = intarr(num_records1) ; A matrix in which to store possible errors
    std_error(*) = 0
    caution_flag = intarr(num_records1)
    caution_flag(*) = 0


;; As of November 2010 ESA range is still set to steps 64 to 127
;; Highest proton speed that can be measured is about 900 km/second

;; Geometric Factors for Main Channel and S Channel (full SWS)
;; G = Active Area*Angular Extent of Slit*Bandwidth/2
;; For example, for the main channel of FM1:
;; 	Active Area = 8.9E-5 m^2
;;	Deflection FWHM = 1.9 degrees = 0.033 radians
;;	Azimuth Acceptance = 45 degrees = 0.785 radians
;;	Energy Bandwidth = 6.12%
;;	G = (8.9E-5)*(0.033)*(0.785)*0.0612/2 = 7.1E-8 m^2 sr eV/eV

A_G_Main = 7.1E-8  ; geometric factor for PLASTIC/A main channel
A_G_Sch = 2.3E-11  ; geometric factor for PLASTIC/A small channel
B_G_Main = 6.0E-8  ; geometric factor for PLASTIC/B main channel
B_G_Sch = 1.7E-11  ; geometric factor for PLASTIC/B small channel

;; Dead times for SF0 and RA_TRIG

A_SF0_dead_time = 5.0E-6 ;sec
A_RA_TRIG_dead_time = 2.0E-6 ;sec
B_SF0_dead_time = 5.0E-6	;sec
B_RA_TRIG_dead_time = 2.0E-6	;sec ; CHANGED ON 02 APRIL, 2007 FROM 2.5E-6


;; Read in tables of estimated efficiencies
;; The location of the efficiency tables must be specified in the crib file.

IF sat EQ 'A' THEN eff_file = table_location+'leakage_A_2010_05_07.txt'
;IF sat EQ 'B' THEN eff_file = table_location+'leakage_B_2011_05_17.txt' 
IF sat EQ 'B' THEN eff_file = table_location+'leakage_B_2013_06_10.txt' 
eff = fltarr(128)
openr, in_lun_1, eff_file, /GET_LUN
line = ","
readf, in_lun_1, line ; read the header line
FOR i = 64, 127 DO BEGIN
	readf, in_lun_1, line
	parts = STRSPLIT(line, ",", /EXTRACT)
	eff[i]=parts[1]
ENDFOR
close, in_lun_1
free_lun, in_lun_1

print, 'Efficiency tables have been read.'

;---------------------------------------------------------
IF sat EQ 'A' THEN BEGIN
   ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_a_full, /nan) LE 0 AND TOTAL(ra_trig_a_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        STOP
    ENDIF
   

   ;; Check to see if there are RA_Trig counts.  If so, continue.  
   ;; If not, assume the full resolution rate is SF0.
   ;;
   ;; RA_Trig was set as the full resolution rate on 14 February, 2007 for A
   ;; and on 1 March, 2007 for B.

    IF TOTAL(ra_trig_a_full, /nan) GT 0 THEN BEGIN
        full_data = ra_trig_a_full
        ra_trig_a_full = 0
        rate_name = '_RA_Trig'
        dead_time = A_RA_trig_dead_time
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_a
        error1 = error1_a
    ENDIF ELSE BEGIN
        full_data = sf0_a_full
        sf0_a_full = 0
        rate_name = '_SF0'
        dead_time = A_SF0_dead_time
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_a
        error1 = error1_a
    ENDELSE



ENDIF ELSE BEGIN ; data is from spacecraft B
    ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_b_full, /nan) LE 0 AND TOTAL(ra_trig_b_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        have_data = 0
    ENDIF
    

    ;; Check to see if there are RA_Trig counts.  If so, continue.  
    ;; If not, assum SF0 is the full resolution rate.
    ;;
    ;; RA_Trig was set as the full resolution rate on 14 February, 2007 on A
    ;; and on 1 March, 2007 on B.

    IF TOTAL(ra_trig_b_full, /nan) GT 0 THEN BEGIN
        full_data = ra_trig_b_full
        ra_trig_b_full = 0	
        rate_name = '_RA_Trig'
        dead_time = B_RA_trig_dead_time
        G_Main = B_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_b
        error1 = error1_b
    ENDIF ELSE BEGIN
        full_data = sf0_b_full
        sf0_b_full = 0
        rate_name = '_SF0'
        dead_time = B_SF0_dead_time
	G_Main = B_G_Main
        G_Sch = B_G_Sch
        s_chan = s_chan1_b
        error1 = error1_b
   ENDELSE
ENDELSE

;; Get the time associated with the first file to label output files
start_time=' '
start_time=strmid(time,0,10)


print, 'Starting calculations.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove the fill value of -1 from the files.
remove_fill_high_res, -1, full_data
print, 'Removed fill values.'

;; Check for value indicating overflow.  If present, mark record with error 3.
FOR m=0L, num_records1 - 1 DO BEGIN
    FOR k = 0, 127 DO BEGIN
        FOR l = 0, 31 DO BEGIN
            IF full_data(k, l, m) GE 507904 THEN BEGIN
                full_data(k, l, m) = !values.f_nan
                std_error(m) = 3
                PRINT, 'Overflow value detected.  Time, ESA, Defl:', m, k, l
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
;print,'Maximum Counts:',max(full_data,/NAN)
;IF max(full_data) GT 32000 THEN STOP


;; Subtract background.

; Create arrays to store data after background subtraction.
full_data_no_bkgd2 = fltarr(128,32,num_records1)
full_data_no_bkgd2(*,*,*) = !values.f_nan
delta_counts_less_bkgd = fltarr(128,32,num_records1)
delta_counts_less_bkgd(*,*,*) = !values.f_nan
calc_alpha_background_subtract, full_data, caution_flag, num_records1, full_data_no_bkgd2, delta_counts_less_bkgd, sat


;; Correct for deflection wobble
;; -- an effect possibly due to an offset in deflector power supplies

FOR j = 1, 31 DO BEGIN          ; DEFL Steps
    FOR i = 64, 126, 2 DO BEGIN ; Even ESA Steps only
        FOR k = 0L, num_records1-1 DO BEGIN ; Time Steps
            full_data_no_bkgd2(i, j-1, k) = full_data_no_bkgd2(i, j, k)
	    delta_counts_less_bkgd(i,j-1,k) = delta_counts_less_bkgd(i,j,k)
            IF j EQ 31 THEN BEGIN
		full_data_no_bkgd2(i, j, k) = !values.f_nan
		delta_counts_less_bkgd(i,j,k) = !values.f_nan
	    ENDIF
        ENDFOR	
    ENDFOR
ENDFOR

;; Convert to Hz -- divide through by accumulation time of 12.8 milliseconds
full_data_hz = fltarr(128,32,num_records1) ; changed from double to float to save memory 28 Sept, 2007
full_data_hz[*,*,*] = !values.f_nan
full_data_hz = FLOAT(full_data_no_bkgd2)/0.0128 ; changed from double to float to save memory 28 Sept, 2007
delta_counts_hz = fltarr(128,32,num_records1)
delta_counts_hz = !values.f_nan
delta_counts_hz = delta_counts_less_bkgd/0.0128
print, 'Converted to Hz.'

full_data_no_bkgd2 = 0 ; set to 0 to free up memory 
delta_counts_less_bkgd = 0 ; free up memory

;; Correct for dead time -- real rate = recorded rate/(1-recorded rate*dead time)
delta_counts_hz = delta_counts_hz/((1.0-(full_data_hz*dead_time))^2.0)
full_data_hz = full_data_hz/(1.0-(full_data_hz*dead_time)) ; changed 1 to 1.0 13 Jan, 2009

;; See if dead time correction results in negative rate.  If so, stop.
FOR k = 0L, num_records1-1 DO BEGIN
    FOR i = 0, 127 DO BEGIN
        FOR j = 0, 31 DO BEGIN
            IF full_data_hz(i, j, k) LT 0 THEN BEGIN
                PRINT, '(', i, j, k, ') Invalid countrate -- deadtime is too big.'
                STOP
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
print, 'Dead time correction completed.'

; Divide by geometric factor to account for different apertures
full_data_hz = full_data_hz/G_Main ; Divide by G to get into to the right units
delta_counts_hz = delta_counts_hz/G_Main
FOR hh=0L, num_records1-1 DO BEGIN
    IF sat EQ 'A' THEN F_Switch = 120.0 ; approximate ratio between main and s channel based on in-flight data
    IF sat EQ 'B' THEN F_Switch = 625.0 ; 
    FOR ii = 0, 127 DO BEGIN			
        IF (ii GE s_chan(hh)) THEN BEGIN ; if data taken with S channel then multiply to normalize with main channel
            full_data_hz(ii, *, hh) = full_data_hz(ii, *, hh)*F_Switch
	    delta_counts_hz(ii,*,hh) = delta_counts_hz(ii,*,hh)*F_Switch
        ENDIF
    ENDFOR
ENDFOR
print, 'Division by geometric factor completed.'

; Sum over deflection (out-of-ecliptic plane) steps to get a 1-D count distribution
sum_full_data = FLTARR(num_records1,128) 
sum_full_data(*,*) = 0.0
sum_delta_counts = FLTARR(num_records1,128)
sum_delta_counts(*,*) = 0.0
peak_ESA = INTARR(num_records1)
peak_ESA(*) = -1
calc_proton_sum_defl, full_data_hz, delta_counts_hz, sum_full_data, sum_delta_counts, peak_ESA, num_records1


; Divide through by a flat efficiency for both main and s-channel based on pre-launch calibration.
IF sat EQ 'A' THEN BEGIN
	sum_full_data = sum_full_data/0.80
	sum_delta_counts = sum_delta_counts/0.80
ENDIF
IF sat EQ 'B' THEN BEGIN
	sum_full_data = sum_full_data/4.0
	sum_delta_counts = sum_delta_counts/4.0
ENDIF
 
print, 'Divison by efficiency completed.'

; Get E/Q and proton speed associated with each ESA step so that distribution functions can be found
; The location of the ESA table must be specified in the crib file.

IF sat EQ 'B' THEN ESA_file = table_location+'ESA_table_B_Toni_benchmod3.txt' 
;IF sat EQ 'B' THEN ESA_file = table_location+'ESA_table_B_2010_06_25b.txt' 
IF sat EQ 'A' THEN ESA_file = table_location+'ESA_table_A_Toni_fall_2012_v3.txt' ; Toni's revised table
;IF sat EQ 'A' THEN ESA_file = table_location+'ESA_table_A_2009_08_17.txt' ; KDCS E/Q
openr, in_lun, ESA_file, /GET_LUN
line = " "
energy_S = fltarr(128) ; changed from double to float (August 2009)
energy_S(*) = !values.f_nan
v_S = fltarr(128) ; changed from double to float (August 2009)
v_S(*) = !values.f_nan

readf, in_lun, line   ; read past first lines

FOR i = 64, 127 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    energy_S[i] = parts[1]			;keV/q
    v_S[i] = parts[2]
ENDFOR

	
close, in_lun
free_lun, in_lun
print, 'Finished reading S channel ESA table.'

IF sat EQ 'B' THEN ESA_file_main = table_location+'Main_channel_B_empirical_v2.txt' 
;IF sat EQ 'A' THEN ESA_file_main = table_location+'Main_channel_A_empirical_v3.txt'
;IF sat EQ 'A' THEN ESA_file_main = table_location+'ESA_table_A_Toni_fall_2012_v3.txt'
IF sat EQ 'A' THEN ESA_file_main = table_location+'ESA_table_A_main_all_steps_Toni_v3.txt'
openr, in_lun, ESA_file_main, /GET_LUN
line = " "
energy_main = fltarr(128) ; changed from double to float (August 2009)
energy_main(*) = !values.f_nan
v_main = fltarr(128) ; changed from double to float (August 2009)
v_main(*) = !values.f_nan

readf, in_lun, line   ; read past first lines

;IF sat EQ 'B' THEN BEGIN
;FOR i = 64, 127 DO BEGIN
;    readf, in_lun, line
;    parts = STRSPLIT(line, " ,", /EXTRACT)
;    energy_main[i] = parts[1]			;keV/q
;    v_main[i] = parts[2]
;ENDFOR
;ENDIF ELSE BEGIN
FOR i = 0, 127 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    energy_main[i] = parts[1]			;keV/q
    v_main[i] = parts[2]
ENDFOR
;ENDELSE

	
close, in_lun
free_lun, in_lun
print, 'Finished reading main channel ESA table.'

; Multiply by another energy dependent "leakage efficiency"
FOR ii = 0L, num_records1 - 1 DO BEGIN
	FOR jj = 64, 127 DO BEGIN
		IF jj GE s_chan(ii) THEN BEGIN 
		sum_full_data(ii,jj) = sum_full_data(ii,jj)*eff(jj)
		sum_delta_counts(ii,jj) = sum_delta_counts(ii,jj)*eff(jj)
		ENDIF
	ENDFOR
ENDFOR

; Now multiply each ESA step by (delta theta*delta phi)/(v^2) to get distribution function
dist_funct = dblarr(128,num_records1)
dist_funct(*,*) = !values.d_nan
peak_dist = intarr(num_records1)
peak_dist(*) = -1
delta_dist_funct = dblarr(128,num_records1)
delta_dist_funct(*,*) = !values.d_nan

FOR i = 0L, num_records1-1 DO BEGIN
    peak_count = 0.0
    FOR j = 0, 127 DO BEGIN
        IF sum_full_data(i, j) GT 0 THEN BEGIN
		IF j GE s_chan(i) THEN BEGIN
            	dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
	    	delta_dist_funct(j,i) = (0.0218*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		ENDIF ELSE BEGIN
            	dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v_main(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
	    	delta_dist_funct(j,i) = (0.0218*0.7854/((v_main(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_main(j))^2.0)*(0.05*v_main(j))^2.0)

		ENDELSE
        ENDIF ELSE BEGIN
            dist_funct(j, i) = 0.0
	    delta_dist_funct(j,i) = 0.0
        ENDELSE

; Average repeated ESA steps
		;IF sat EQ 'B' THEN BEGIN
		;	dist_funct(122,i) = DOUBLE(0.5*(dist_funct(121,i)+dist_funct(123,i)))
		;	dist_funct(125,i) = DOUBLE(0.5*(dist_funct(124,i)+dist_funct(126,i)))
		;	delta_dist_funct(122,i) = DOUBLE(0.5*SQRT((delta_dist_funct(121,i))^2.0+(delta_dist_funct(123,i))^2.0))
		;	delta_dist_funct(125,i) = DOUBLE(0.5*SQRT((delta_dist_funct(124,i))^2.0+(delta_dist_funct(126,i))^2.0))
		;ENDIF ELSE BEGIN
		;	dist_funct(116,i) = DOUBLE(0.5*(dist_funct(115,i)+dist_funct(117,i)))
		;	dist_funct(118,i) = DOUBLE(0.5*(dist_funct(117,i)+dist_funct(119,i)))
		;	dist_funct(123,i) = DOUBLE(0.5*(dist_funct(122,i)+dist_funct(124,i)))
		;	dist_funct(126,i) = DOUBLE(0.5*(dist_funct(125,i)+dist_funct(127,i)))
		;	delta_dist_funct(116,i) = DOUBLE(0.5*SQRT((delta_dist_funct(115,i))^2.0+(delta_dist_funct(117,i))^2.0))
		;	delta_dist_funct(118,i) = DOUBLE(0.5*SQRT((delta_dist_funct(117,i))^2.0+(delta_dist_funct(119,i))^2.0))
		;	delta_dist_funct(123,i) = DOUBLE(0.5*SQRT((delta_dist_funct(122,i))^2.0+(delta_dist_funct(124,i))^2.0))
		;	delta_dist_funct(126,i) = DOUBLE(0.5*SQRT((delta_dist_funct(125,i))^2.0+(delta_dist_funct(127,i))^2.0))
		;ENDELSE

        
        IF (dist_funct(j, i) GT peak_count) THEN peak_count = dist_funct(j, i)
        IF (dist_funct(j, i) GE peak_count) THEN peak_dist(i) = j
    ENDFOR
ENDFOR

sum_full_data = 0 ; added 28 Sept, 2007 to use save memory
sum_delta_counts = 0


; To make it easier for IDL to get a good fit to the curve, multiply the distribution by
; 1e5.  Remember to divide the density by this factor later on!
dist_funct = dist_funct*(1.0E5)
delta_dist_funct = delta_dist_funct*(1.0E5)
print, 'Distribution functions complete.'

; Fit the distribution function versus velocity data to a 1-D Gaussian.
coeff = dblarr(num_records1,3) ; A matrix in which to store the proton fit coefficients
coeff(*,*,*) = !values.d_nan
density = dblarr(num_records1)
density(*) = !values.f_nan
avg_v= dblarr(num_records1)
avg_v(*) = !values.f_nan
temp_K = dblarr(num_records1)
temp_K(*) = !values.f_nan
v_th = dblarr(num_records1)
v_th(*) = !values.f_nan
good_fit = fltarr(num_records1)
good_fit(*) = !values.f_nan


calc_proton_fit_maxwellian, sat, num_records1, coeff, density, avg_v, temp_K, v_th, good_fit, $
	peak_dist, dist_funct, delta_dist_funct, v_S, v_main, std_error, error1, s_chan, epoch1

;Uncomment below to define variables for alpha data and then fit the alpha distribution
coeff_alpha = dblarr(num_records1,3) ; A matrix in which to store the proton fit coefficients
coeff_alpha(*,*,*) = !values.d_nan
density_alpha = dblarr(num_records1)
density_alpha(*) = !values.f_nan
avg_v_alpha= dblarr(num_records1)
avg_v_alpha(*) = !values.f_nan
temp_K_alpha = dblarr(num_records1)
temp_K_alpha(*) = !values.f_nan
v_th_alpha = dblarr(num_records1)
v_th_alpha(*) = !values.f_nan
good_fit_alpha = fltarr(num_records1)
good_fit_alpha(*) = !values.f_nan
expected_peak_channel = INTARR(num_records1)
expected_peak_channel(*) = -1
Na_to_Np = fltarr(num_records1)
Na_to_Np(*) = !values.f_nan
Va_Vp = fltarr(num_records1)
Va_Vp(*) = !values.f_nan
alpha_caution = INTARR(num_records1)
alpha_caution(*) = 0


;calc_alpha_data_edit, v_S, v_main, avg_v, density, v_th, dist_funct, delta_dist_funct, coeff_alpha, $
calc_alpha_data, v_S, v_main, avg_v, density, v_th, dist_funct, delta_dist_funct, coeff_alpha, $
	density_alpha, avg_v_alpha, temp_K_alpha, v_th_alpha, good_fit_alpha, $
	coeff, num_records1, expected_peak_channel, s_chan, sat, epoch1, std_error, caution_flag, $
	Na_to_Np, Va_Vp, alpha_caution, table_location

;; Un-comment below to print raw output file
calc_alpha_raw_ascii, sat, epoch1, num_records1, density, avg_v, temp_K, v_th, good_fit, avg_v_alpha, $
	density_alpha, v_th_alpha, good_fit_alpha, expected_peak_channel, std_error, Na_to_Np, Va_Vp, alpha_caution
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Calculate hourly averages;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRINT, 'Calculating hourly averages.'

k = FIX(num_records1/60)+1

    hour_start_time = dblarr(k)
    hour_start_time(*) = !values.d_nan
    hour_v_alpha = fltarr(k)
    hour_v_alpha(*) = !values.f_nan
    hour_n_alpha = fltarr(k)
    hour_n_alpha(*) = !values.f_nan
    hour_vth_alpha = fltarr(k)
    hour_vth_alpha(*) = !values.f_nan
    hour_NatoNp = fltarr(k)
    hour_NatoNp(*) = !values.f_nan
    hour_Va_Vp = fltarr(k)
    hour_Va_vp(*) = !values.f_nan
    hour_count = fltarr(k)
    hour_count(*) = !values.f_nan
    hour_count_v = fltarr(k)
    hour_count_v(*) = !values.f_nan
    hour_count_NatoNp = fltarr(k)
    hour_count_NatoNp(*) = !values.f_nan

    ih = 0 ; index into 1 hour data
    found_first = 0
    on_first = 1
    FOR im = 0L, n_elements(epoch1)-1 DO BEGIN ; index into 1 minute data
        now_time = time_struct(epoch1(im))
        IF now_time.year NE 1970 THEN BEGIN ; skip invalid times
            IF found_first EQ 0 THEN BEGIN 
                base_1hr = now_time.hour
            ENDIF 
            now_1hr = now_time.hour
            IF now_time.hour NE base_1hr OR found_first EQ 0 THEN BEGIN ; new base
                found_first = 1
                IF on_first EQ 0 THEN BEGIN ; write old variables
                    IF n_elements(temp_n_alpha) GT 0 AND FINITE(temp_n_alpha) EQ 1 THEN  hour_n_alpha[ih] = temp_n_alpha/sixty_count
                    IF n_elements(temp_v_alpha) GT 0 AND FINITE(temp_v_alpha) EQ 1 THEN hour_v_alpha[ih] = temp_v_alpha/sixty_count_v
                    IF n_elements(temp_vth_alpha) GT 0 AND FINITE(temp_vth_alpha) EQ 1 THEN hour_vth_alpha[ih] = temp_vth_alpha/sixty_count
                    IF n_elements(temp_NatoNp) GT 0 AND FINITE(temp_NatoNp) EQ 1 THEN hour_NatoNp[ih] = temp_NatoNp/sixty_count_NatoNp
		    IF n_elements(temp_Va_Vp) GT 0 AND FINITE(temp_Va_Vp) EQ 1 THEN hour_Va_Vp[ih] = temp_Va_Vp/sixty_count_v
                    hour_count[ih] = sixty_count
		    hour_count_v[ih] = sixty_count_v
		    hour_count_NatoNp[ih] = sixty_count_NatoNp
		ENDIF 
                base_1hr = now_1hr
                IF on_first EQ 1 THEN on_first = 0 ELSE ih = ih + 1 ; so don't get empty first record
                IF ih EQ n_elements(hour_start_time) THEN BEGIN  ; need to resize arrays
		    tmp_arr  = intarr(10)
		    tmp_arr[*] = !values.f_nan
                    hour_count = [hour_count, tmp_arr]
		    hour_count_v = [hour_count_v, tmp_arr]
		    hour_count_NatoNp = [hour_count_NatoNp, tmp_arr]
                    hour_start_time = [hour_start_time, tmp_arr]
                    hour_n_alpha = [hour_n_alpha, tmp_arr]
                    hour_v_alpha = [hour_v_alpha, tmp_arr]
                    hour_vth_alpha = [hour_vth_alpha, tmp_arr]
                    hour_NatoNp = [hour_NatoNp, tmp_arr]
		    hour_Va_Vp = [hour_Va_Vp, tmp_arr]
                ENDIF 
                hour_start_time[ih] = epoch1[im] ; LBE changed 9/11/09 to make actual start time
                IF finite(density_alpha[im]) EQ 1 THEN temp_n_alpha = density_alpha[im] ELSE temp_n_alpha = !values.f_nan 
                IF finite(avg_v_alpha[im]) EQ 1 THEN temp_v_alpha = avg_v_alpha[im] ELSE temp_v_alpha = !values.f_nan
                IF finite(v_th_alpha[im]) EQ 1 THEN temp_vth_alpha = v_th_alpha[im] ELSE temp_vth_alpha = !values.f_nan
                IF finite(Na_to_Np[im]) EQ 1 THEN temp_NatoNp = Na_to_Np[im] ELSE temp_NatoNp = !values.f_nan
		IF finite(Va_Vp[im]) EQ 1 THEN temp_Va_Vp = Va_Vp[im] ELSE temp_Va_Vp = !values.f_nan
               
                IF FINITE(temp_n_alpha) EQ 1 THEN sixty_count = 1.0 ELSE sixty_count = 0.0  
		IF FINITE(temp_v_alpha) EQ 1 THEN sixty_count_v = 1.0 ELSE sixty_count_v = 0.0 
		IF FINITE(temp_NatoNp) EQ 1 THEN sixty_count_NatoNp = 1.0 ELSE sixty_count_NatoNp = 0.0
		
            ENDIF ELSE BEGIN   ; further in same 1 hour space
                IF finite(density_alpha[im]) EQ 1 THEN $
                  IF FINITE(temp_n_alpha) EQ 0 THEN temp_n_alpha = density_alpha[im] $
                  ELSE temp_n_alpha = temp_n_alpha + density_alpha[im]
                IF finite(avg_v_alpha[im]) EQ 1 THEN $
                  IF FINITE(temp_v_alpha) EQ 0 THEN temp_v_alpha = avg_v_alpha[im] $
                  ELSE temp_v_alpha = temp_v_alpha + avg_v_alpha[im]
                IF finite(v_th_alpha[im]) EQ 1 THEN $
                  IF FINITE(temp_vth_alpha) EQ 0 THEN temp_vth_alpha = v_th_alpha[im] $
                  ELSE temp_vth_alpha = temp_vth_alpha + v_th_alpha[im]
                IF finite(Na_to_Np[im]) EQ 1 THEN $
                  IF FINITE(temp_NatoNp) EQ 0 THEN temp_NatoNp = Na_to_Np[im] $
		  ELSE temp_NatoNp = temp_NatoNp + Na_to_Np[im]
                IF finite(Va_Vp[im]) EQ 1 THEN $
		  IF FINITE(temp_Va_Vp) EQ 0 THEN temp_Va_Vp = Va_Vp[im] $
		  ELSE temp_Va_Vp = temp_Va_Vp + Va_Vp[im]
          
               
                IF (FINITE(density_alpha[im]) EQ 1) THEN sixty_count = sixty_count+1.0 ; Make sure time step has a valid result
		IF (FINITE(avg_v_alpha[im]) EQ 1) THEN sixty_count_v = sixty_count_v+1.0
		IF (FINITE(Na_to_Np[im]) EQ 1) then sixty_count_NatoNp = sixty_count_NatoNp + 1.0

            ENDELSE 
        ENDIF 
    ENDFOR 
;; add final value
   
    IF n_elements(temp_n_alpha) GT 0 THEN $
      IF ih EQ n_elements(hour_n_alpha) THEN hour_n_alpha = [hour_n_alpha, temp_n_alpha/sixty_count] ELSE $
      hour_n_alpha[ih] = temp_n_alpha/sixty_count
    IF n_elements(temp_v_alpha) GT 0 THEN $
      IF ih EQ n_elements(hour_v_alpha) THEN hour_v_alpha = [hour_v_alpha, temp_v_alpha/sixty_count_v] ELSE $
      hour_v_alpha[ih] = temp_v_alpha/sixty_count_v
    IF n_elements(temp_vth_alpha) GT 0 THEN $
      IF ih EQ n_elements(hour_vth_alpha) THEN hour_vth_alpha = [hour_vth_alpha, temp_vth_alpha/sixty_count] ELSE $
      hour_vth_alpha[ih] = temp_vth_alpha/sixty_count
    IF n_elements(temp_Na_to_Np) GT 0 THEN $
      IF ih EQ n_elements(hour_NatoNp) THEN hour_NatoNp = [hour_NatoNp, temp_NatoNp/sixty_count_NatoNp] ELSE $
      hour_NatoNp[ih] = temp_NatoNp/sixty_count_NatoNp
    IF n_elements(temp_Va_Vp) GT 0 THEN $
      IF ih EQ n_elements(hour_Va_Vp) THEN hour_Va_Vp = [hour_Va_Vp, temp_Va_Vp/sixty_count_v] ELSE $
      hour_Va_Vp[ih] = temp_Va_Vp/sixty_count_v

    IF ih EQ n_elements(hour_count)    THEN hour_count = [hour_count, sixty_count]          ELSE hour_count[ih] = sixty_count
    IF ih EQ n_elements(hour_count_v)  THEN hour_count_v = [hour_count_v, sixty_count_v]    ELSE hour_count_v[ih] = sixty_count_v
    IF ih EQ n_elements(hour_count_NatoNp) THEN hour_count_NatoNp = [hour_count_NatoNp, sixty_count_NatoNp] ELSE $
	hour_count_NatoNp[ih] = sixty_count_NatoNp 

;; Un-comment below to print raw 1 hour output file
calc_alpha_raw_ascii_1hour, sat, hour_start_time, k, hour_n_alpha, hour_v_alpha, hour_vth_alpha, hour_NatoNp, hour_Va_Vp, hour_count, hour_count_v, hour_count_NatoNp

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;Calculate 10 minute averages.;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRINT, 'Calculating 10 minute averages.'

IF num_records1 MOD 10 EQ 0 THEN k = (num_records1/10) ELSE k = FIX(num_records1/10) + 1

    ten_min_start_time = dblarr(k)
    ten_min_start_time(*) = !values.d_nan
    ten_min_v_alpha = fltarr(k)
    ten_min_v_alpha(*) = !values.f_nan
    ten_min_n_alpha = fltarr(k)
    ten_min_n_alpha(*) = !values.f_nan
    ten_min_vth_alpha = fltarr(k)
    ten_min_vth_alpha(*) = !values.f_nan
    ten_min_NatoNp = fltarr(k)
    ten_min_NatoNp(*) = !values.f_nan
    ten_min_Va_Vp = fltarr(k)
    ten_min_Va_vp(*) = !values.f_nan
    ten_min_count = fltarr(k)
    ten_min_count(*) = !values.f_nan
    ten_min_count_v = fltarr(k)
    ten_min_count_v(*) = !values.f_nan
    ten_min_count_NatoNp = fltarr(k)
    ten_min_count_NatoNp(*) = !values.f_nan

   i10 = 0 ; index into 1 hour data
    found_first = 0
    on_first = 1
    FOR i1 = 0L, n_elements(epoch1)-1 DO BEGIN ; index into 1 minute data
        now_time = time_struct(epoch1(i1))
        IF now_time.year NE 1970 THEN BEGIN ; skip invalid times
	    IF i10 EQ 0 THEN base_hour = now_time.hour
            IF found_first EQ 0 THEN BEGIN 
                base_10min = now_time.min/10
            ENDIF 
	    now_10min = now_time.min/10
           IF now_time.hour NE base_hour OR found_first EQ 0 OR $
              ((now_time.hour EQ base_hour) AND (now_10min NE base_10min)) THEN BEGIN ; new base
                base_hour = now_time.hour
		found_first = 1
                IF on_first EQ 0 THEN BEGIN ; write old variables
 		    IF n_elements(temp_n_alpha) GT 0 AND FINITE(temp_n_alpha) EQ 1 THEN  ten_min_n_alpha[i10] = temp_n_alpha/ten_count
                    IF n_elements(temp_v_alpha) GT 0 AND FINITE(temp_v_alpha) EQ 1 THEN ten_min_v_alpha[i10] = temp_v_alpha/ten_count_v
                    IF n_elements(temp_vth_alpha) GT 0 AND FINITE(temp_vth_alpha) EQ 1 THEN ten_min_vth_alpha[i10] = temp_vth_alpha/ten_count
                    IF n_elements(temp_NatoNp) GT 0 AND FINITE(temp_NatoNp) EQ 1 THEN ten_min_NatoNp[i10] = temp_NatoNp/ten_count_NatoNp
		    IF n_elements(temp_Va_Vp) GT 0 AND FINITE(temp_Va_Vp) EQ 1 THEN ten_min_Va_Vp[i10] = temp_Va_Vp/ten_count_v
                    ten_min_count[i10] = ten_count
		    ten_min_count_v[i10] = ten_count_v
		    ten_min_count_NatoNp[i10] = ten_count_NatoNp
		ENDIF 
		base_10min = now_10min
                IF on_first EQ 1 THEN on_first = 0 ELSE i10 = i10 + 1
                IF i10 EQ n_elements(ten_min_start_time) THEN BEGIN  ; need to resize arrays
    		    tmp_arr  = intarr(10)
		    tmp_arr[*] = !values.f_nan
                    ten_min_count = [ten_min_count, tmp_arr]
		    ten_min_count_v = [ten_min_count_v, tmp_arr]
		    ten_min_count_NatoNp = [ten_min_count_NatoNp, tmp_arr]
                    ten_min_start_time = [ten_min_start_time, tmp_arr]
                    ten_min_n_alpha = [ten_min_n_alpha, tmp_arr]
                    ten_min_v_alpha = [ten_min_v_alpha, tmp_arr]
                    ten_min_vth_alpha = [ten_min_vth_alpha, tmp_arr]
                    ten_min_NatoNp = [ten_min_NatoNp, tmp_arr]
		    ten_min_Va_Vp = [ten_min_Va_Vp, tmp_arr]
                ENDIF 

		ten_min_start_time[i10] = epoch1[i1] ; LBE changed 9/11/09 to make actual start time
                IF finite(density_alpha[i1]) EQ 1 THEN temp_n_alpha = density_alpha[i1] ELSE temp_n_alpha = !values.f_nan 
                IF finite(avg_v_alpha[i1]) EQ 1 THEN temp_v_alpha = avg_v_alpha[i1] ELSE temp_v_alpha = !values.f_nan
                IF finite(v_th_alpha[i1]) EQ 1 THEN temp_vth_alpha = v_th_alpha[i1] ELSE temp_vth_alpha = !values.f_nan
                IF finite(Na_to_Np[i1]) EQ 1 THEN temp_NatoNp = Na_to_Np[i1] ELSE temp_NatoNp = !values.f_nan
		IF finite(Va_Vp[i1]) EQ 1 THEN temp_Va_Vp = Va_Vp[i1] ELSE temp_Va_Vp = !values.f_nan
               
                IF FINITE(temp_n_alpha) EQ 1 THEN ten_count = 1.0 ELSE ten_count = 0.0  
		IF FINITE(temp_v_alpha) EQ 1 THEN ten_count_v = 1.0 ELSE ten_count_v = 0.0 
		IF FINITE(temp_NatoNp) EQ 1 THEN ten_count_NatoNp = 1.0 ELSE ten_count_NatoNp = 0.0

 ENDIF ELSE BEGIN   ; further in same 10 min space
                IF finite(density_alpha[i1]) EQ 1 THEN $
                  IF FINITE(temp_n_alpha) EQ 0 THEN temp_n_alpha = density_alpha[i1] $
                  ELSE temp_n_alpha = temp_n_alpha + density_alpha[i1]
                IF finite(avg_v_alpha[i1]) EQ 1 THEN $
                  IF FINITE(temp_v_alpha) EQ 0 THEN temp_v_alpha = avg_v_alpha[i1] $
                  ELSE temp_v_alpha = temp_v_alpha + avg_v_alpha[i1]
                IF finite(v_th_alpha[i1]) EQ 1 THEN $
                  IF FINITE(temp_vth_alpha) EQ 0 THEN temp_vth_alpha = v_th_alpha[i1] $
                  ELSE temp_vth_alpha = temp_vth_alpha + v_th_alpha[i1]
                IF finite(Na_to_Np[i1]) EQ 1 THEN $
                  IF FINITE(temp_NatoNp) EQ 0 THEN temp_NatoNp = Na_to_Np[i1] $
		  ELSE temp_NatoNp = temp_NatoNp + Na_to_Np[i1]
                IF finite(Va_Vp[i1]) EQ 1 THEN $
		  IF FINITE(temp_Va_Vp) EQ 0 THEN temp_Va_Vp = Va_Vp[i1] $
		  ELSE temp_Va_Vp = temp_Va_Vp + Va_Vp[i1]
          
               
                IF (FINITE(density_alpha[i1]) EQ 1) THEN ten_count = ten_count+1.0 ; Make sure time step has a valid result
		IF (FINITE(avg_v_alpha[i1]) EQ 1) THEN ten_count_v = ten_count_v+1.0
		IF (FINITE(Na_to_Np[i1]) EQ 1) then ten_count_NatoNp = ten_count_NatoNp + 1.0

            ENDELSE 
        ENDIF 
    ENDFOR 

;; add final value
   
    IF n_elements(temp_n_alpha) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_n_alpha) THEN ten_min_n_alpha = [ten_min_n_alpha, temp_n_alpha/ten_count] ELSE $
      ten_min_n_alpha[i10] = temp_n_alpha/ten_count
    IF n_elements(temp_v_alpha) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_v_alpha) THEN ten_min_v_alpha = [ten_min_v_alpha, temp_v_alpha/ten_count_v] ELSE $
      ten_min_v_alpha[i10] = temp_v_alpha/ten_count_v
    IF n_elements(temp_vth_alpha) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_vth_alpha) THEN ten_min_vth_alpha = [ten_min_vth_alpha, temp_vth_alpha/ten_count] ELSE $
      ten_min_vth_alpha[i10] = temp_vth_alpha/ten_count
    IF n_elements(temp_Na_to_Np) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_NatoNp) THEN ten_min_NatoNp = [ten_min_NatoNp, temp_NatoNp/ten_count_NatoNp] ELSE $
      ten_min_NatoNp[ih] = temp_NatoNp/ten_count_NatoNp
    IF n_elements(temp_Va_Vp) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_Va_Vp) THEN ten_min_Va_Vp = [ten_min_Va_Vp, temp_Va_Vp/ten_count_v] ELSE $
      ten_min_Va_Vp[i10] = temp_Va_Vp/ten_count_v

    IF i10 EQ n_elements(ten_min_count)    THEN ten_min_count = [ten_min_count, ten_count]          ELSE ten_min_count[i10] = ten_count
    IF i10 EQ n_elements(ten_min_count_v)  THEN ten_min_count_v = [ten_min_count_v, ten_count_v]    ELSE ten_min_count_v[i10] = ten_count_v
    IF i10 EQ n_elements(ten_min_count_NatoNp) THEN ten_min_count_NatoNp = [ten_min_count_NatoNp, ten_count_NatoNp] ELSE $
	ten_min_count_NatoNp[i10] = ten_count_NatoNp 

;; Un-comment below to print raw 10 min output file
calc_alpha_raw_ascii_10min, sat, ten_min_start_time, k, ten_min_n_alpha, ten_min_v_alpha, ten_min_vth_alpha, ten_min_NatoNp, ten_min_Va_Vp, ten_min_count, ten_min_count_v, ten_min_count_NatoNp



END 
