;+
; PROCEDURE: adjust_alpha_schan
;
; PURPOSE: To adjust alpha data based on a function.
;
; INPUT: var -> tplot variable name (a string)
;        which_func -> an index into which function to use:
;               0: f(v) = 70*EXP(.000003*v/SQRT(2))
;               1: Kristin's Efficiency table
;               2: f(v) = 120/((-6.30369334716773E-19*(v^3))-(8.63692363656727E-13*(v^2))+(5.9564934852831E-7*v)+(6.59619620444739E-1))
;        alpha_vel -> alpha velocity from energy table (this used to be proton vel 10/12/2009)
;
; MODIFICATION HISTORY:
;           11/11/2008: LBE: Created. Based on plot_spec_line_equation.
;           12/10/2008: LBE: Adjusted for if schan only or main only
;           10/12/2009: LBE: Removed sqrt(2) from func 0 per Lynn's doc (10/6/2009)
;           11/19/2009: LBE: Added schan_var
;           12/14/2009: LBE: Added function 6 (from Kristin's 'leakage_A_2009_08_17.xls'
;           01/06/2010: LBE: Multiply main channel by 4 for A
;           05/07/2010: LBE: Remove main channel factor of 4. 
;                            Add func 7 (for A)
;           05/18/2010: LBE: Add func 8 (for B)
;           05/24/2010: LBE: Fixed bug in 8.
;           06/16/2010: LBE: Changed func 8 per Lynn's 6/8 email.
;-
PRO adjust_alpha_schan, var, schan_var, which_func, alpha_vel, sat
compile_opt strictarrsubs

;---------------------------------------------------------------------
; INPUT parameters
;---------------------------------------------------------------------

fact_file = '~/splat/splat_user/Cribs/Alphas/RA_Efficiency_with_SChan.txt'

;---------------------------------------------------------------------

  ; check function
IF which_func LT 0 OR which_func GT 8 THEN BEGIN
    print, "invalid entry for which_func"
    stop
ENDIF 

  ; If Kristin's is a table...
IF which_func EQ 1 THEN BEGIN
    line = ''
    fact_table = dblarr(128)
    openr, fact_lun, fact_file, /get_lun
    readf, fact_lun, line
    FOR kk = 0, 127 DO BEGIN
        readf, fact_lun, line
        parts = strsplit(line, /extract)
        fact_table[kk] = double(parts[7])
    ENDFOR
    close, fact_lun
    free_lun, fact_lun
ENDIF 

get_timespan, time_interval
t_s = gettime(time_interval(0)) ; start time in tplot-time
t_e = gettime(time_interval(1)) ; end time in tplot-time 
  
var_len = strlen(var)
IF var_len GT 10 THEN BEGIN 
    IF strcmp(strmid(var, var_len-11, 11), '_schan_only') EQ 1 THEN temp_var = strmid(var, 0, var_len-11) $
      ELSE IF strcmp(strmid(var, var_len-10, 10), '_main_only') EQ 1 THEN temp_var = strmid(var, 0, var_len-10) $
	ELSE IF strcmp(strmid(var, var_len-5), '_bkgd') EQ 1 THEN temp_var = strmid(var, 0, var_len-5)$ 
        ELSE temp_var = var
ENDIF ELSE temp_var = var

get_data, var, data = data, dlimit = dlimit, limit = limit
get_data, schan_var, data = schan_data, dlimit = schan_dlimit, limit = schan_limit

; check for missing schan data
temp_i = where(schan_data.y EQ -1, temp_count)
IF temp_count GT 0 THEN data.y[temp_i, *] = !values.f_nan

; adjust schan
FOR ii = 0, 127 DO BEGIN ; this is smaller loop than going by num_records
    temp_i = where(schan_data.y LE ii, temp_count)
    IF temp_count GT 0 THEN BEGIN 
        IF which_func EQ 2 AND ii LT 64 THEN BEGIN
            schan_factor = 4000
        ENDIF ELSE BEGIN 
            CASE which_func OF 
                0: schan_factor = 70*EXP(.000003*alpha_vel[ii]) ;alpha
                1: schan_factor = fact_table[ii] ;proton
                2: schan_factor = 120/((-6.30369334716773E-19*(alpha_vel[ii]^3))-(8.63692363656727E-13*(alpha_vel[ii]^2))+$
                                       (5.95649348532831E-7*(alpha_vel[ii]))+(6.59619620444739E-1)) ;proton
		3: schan_factor = 181.5926 ; find P1 average in excel
                4: schan_factor = 65/(3.11E6*(alpha_vel[ii]*sqrt(2)/1000.)^(-2.47)) ; Kristin's function
                5: IF alpha_vel[ii] GT 420000 THEN BEGIN schan_factor = 185
                ENDIF ELSE BEGIN
                    schan_factor = (65/(3.11E6*(alpha_vel[ii]*sqrt(2)/1000.)^(-2.47)))*(0.812)
                ENDELSE
                6: schan_factor = (((7.799889476347E-12 )*((sqrt(2)*alpha_vel[ii]/1000)^4)) - $ ; Kristin 12/14/2009
                                  ((1.103592099922E-08 )*((sqrt(2)*alpha_vel[ii]/1000)^3)) + $ ; convert Alpha vel (m/s) to
                                  ((0.00001225352247071)*((sqrt(2)*alpha_vel[ii]/1000)^2)) - $ ; proton vel (km/s)
                                  ((0.007106614285786  )*((sqrt(2)*alpha_vel[ii]/1000)  )) + 2.646607699963) * 120
                7: IF alpha_vel[ii] LT 275000 THEN schan_factor = 28 $
                   ELSE IF alpha_vel[ii] GE 275000 AND alpha_vel[ii] LT 626000 THEN schan_factor =  $
                     (6.5444E-7*((alpha_vel[ii]/1000)^3)) $
                     - (3.37E-4 *((alpha_vel[ii]/1000)^2)) $
                     + (0.21097 * (alpha_vel[ii]/1000)) - 18.103 $
                     ELSE schan_factor = 140
                8: IF alpha_vel[ii] LT 275000 THEN schan_factor = 6.4 $
                   ELSE IF alpha_vel[ii] GE 275000 AND alpha_vel[ii] LT 375000 THEN schan_factor = $
                     (0.00365 *((alpha_vel[ii]/1000)^2)) $
                     - (1.8895 * (alpha_vel[ii]/1000)) + 250.39 $
                     ELSE IF alpha_vel[ii] GE 375000 AND alpha_vel[ii] LT 475000 THEN schan_factor = $
                       (-0.00688 *((alpha_vel[ii]/1000)^2)) $
                       + (6.5574 * (alpha_vel[ii]/1000)) - 1437. $
                     ELSE IF alpha_vel[ii] GE 475000 AND alpha_vel[ii] LT 675000 THEN schan_factor = $
                       (0.00285 *((alpha_vel[ii]/1000)^2)) $
                       -  (3.223 * (alpha_vel[ii]/1000)) + 1013.5 $
                       ELSE schan_factor = 140
                ELSE: stop
            ENDCASE 
        ENDELSE 
        data.y[temp_i, ii] = data.y[temp_i, ii] * schan_factor
    ENDIF 
ENDFOR 

temp_var = var + '_alpha_flux'
store_data, temp_var, data = {x:data.x, y:data.y} 
END


