;+
;  PROCEDURE: plot_wind_swe
;
;  PURPOSE: to read and plot WIND/SWE cdf data
;
;  CREATED BY: C. Mouikis
;
;  MODIFICATION HISTORY:
;        2008-01-01: (CGM) Time tags are checked for "bad" values
;        before the tplot variables are created.
;        2008-11-04: LBE: Look for value within 1 min if only want one
;        time.
;        2010-12-13: LBE: Add Vth, and check quality flag.
;        2011-01-03: LBE: Have possibility of no times.
;        2014-02-17: LBE: Add checkes for all bad data.
;-

PRO r_wind_swe_cdf, path, fln
compile_opt strictarrsubs
  
  COMMON wind_swe_data, wind_time, wind_swe
  
  ;--------------------------------------------------------------------
  ; Find original cdf file
  ;--------------------------------------------------------------------

  find = FINDFILE(path+fln, count=ff)
  IF ff EQ 0 THEN BEGIN
    print, 'FILE: '+fln+' was not found'
    get_err_no = 1
    get_err_msg= 'Data NOT found in time interval'
    RETURN
  ENDIF
  
  fln_r = find(ff-1)
  
  read_cdf, IN_FILE = fln_r, $
    VAR = 'Np', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data=data
  Np_xarr = REFORM(data.x(0,*))
  Np_yarr = REFORM(data.y(*,0))
  store_data, mout_data, /del

  read_cdf, IN_FILE = fln_r, $
    VAR = 'QF_Np', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data = data
  QF_Np_xarr = REFORM(data.x(0,*))
  QF_Np_yarr = REFORM(data.y(*,0))
  store_data, mout_data, /del

  read_cdf, IN_FILE = fln_r, $
    VAR = 'QF_Vth', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data = data
  QF_Vth_xarr = REFORM(data.x(0,*))
  QF_Vth_yarr = REFORM(data.y(*,0))
  store_data, mout_data, /del

  read_cdf, IN_FILE = fln_r, $
    VAR = 'QF_V', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data = data
  QF_V_xarr = REFORM(data.x(0,*))
  QF_V_yarr = data.y
  store_data, mout_data, /del

  read_cdf, IN_FILE = fln_r, $
    VAR = 'THERMAL_SPD', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data = data
  Vth_xarr = REFORM(data.x(0,*))
  Vth_yarr = REFORM(data.y(*,0))
  store_data, mout_data, /del

  read_cdf, IN_FILE = fln_r, $
    VAR = 'V_GSE', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data=data
  V_gse_xarr = REFORM(data.x(0,*))
  V_gse_yarr = data.y
  store_data, mout_data, /del

  read_cdf, IN_FILE = fln_r, $
    VAR = 'V_GSM', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data=data
  V_gsm_xarr = REFORM(data.x(0,*))
  V_gsm_yarr = data.y
  store_data, mout_data, /del
  
  read_cdf, IN_FILE = fln_r, $
    VAR = 'SC_pos_gse', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data=data
  X_gse_xarr = REFORM(data.x(0,*))
  X_gse_yarr = data.y / 6372.0 ; Re
  store_data, mout_data, /del
  
  read_cdf, IN_FILE = fln_r, $
    VAR = 'SC_pos_GSM', $
    EPOCH = 'Epoch', OUT_DATA = mout_data
  
  get_data, mout_data, data=data
  X_gsm_xarr = REFORM(data.x(0,*))
  X_gsm_yarr = data.y / 6372.0 ; Re
  store_data, mout_data, /del
  
  ; check quality flags
  bad_den_i = where(QF_Np_yarr  NE 0, bad_den_count)
  bad_vth_i = where(QF_Vth_yarr NE 0, bad_vth_count)
  bad_vel_i = where(QF_V_yarr   NE 0, bad_vel_count)
  ;IF bad_den_count GT 0 THEN Np_yarr   [bad_den_i] = !values.f_nan
  ;IF bad_vth_count GT 0 THEN Vth_yarr  [bad_vth_i] = !values.f_nan
  ;IF bad_vel_count GT 0 THEN V_gse_yarr[bad_vel_i] = !values.f_nan
  ;IF bad_vel_count GT 0 THEN V_gsm_yarr[bad_vel_i] = !values.f_nan

  ;--------------------------------------------------------------------
  ; Re-open the file and read data. If file is zipped and the
  ; idl version is not 5.4 then read the unziped file from temp directory
  ;--------------------------------------------------------------------
  lc = N_ELEMENTS(Np_xarr)
  dummy_str = {$,
               Np:0.0, $
               Vth:0.0, $
               V_GSE_X:0.0, $
               V_GSE_Y:0.0, $
               V_GSE_Z:0.0, $
               V_GSM_X:0.0, $
               V_GSM_Y:0.0, $
               V_GSM_Z:0.0, $
               X_GSE_X:0.0, $
               X_GSE_Y:0.0, $
               X_GSE_Z:0.0, $
               X_GSM_X:0.0, $
               X_GSM_Y:0.0, $
               X_GSM_Z:0.0 $
              }
  
  wind_swe = REPLICATE(dummy_str, lc)
  wind_time = Np_xarr
  wind_swe.Np = Np_yarr
  wind_swe.Vth = Vth_yarr
  wind_swe.V_GSE_X = REFORM(V_gse_yarr(*,0))
  wind_swe.V_GSE_Y = REFORM(V_gse_yarr(*,1))
  wind_swe.V_GSE_Z = REFORM(V_gse_yarr(*,2))
  wind_swe.V_GSM_X = REFORM(V_gsm_yarr(*,0))
  wind_swe.V_GSM_Y = REFORM(V_gsm_yarr(*,1))
  wind_swe.V_GSM_Z = REFORM(V_gsm_yarr(*,2))
  wind_swe.X_GSE_X = REFORM(X_gse_yarr(*,0))
  wind_swe.X_GSE_Y = REFORM(X_gse_yarr(*,1))
  wind_swe.X_GSE_Z = REFORM(X_gse_yarr(*,2))
  wind_swe.X_GSM_X = REFORM(X_gsm_yarr(*,0))
  wind_swe.X_GSM_Y = REFORM(X_gsm_yarr(*,1))
  wind_swe.X_GSM_Z = REFORM(X_gsm_yarr(*,2))

END


PRO plot_wind_swe, $
                   ALL=ALL, $
                   DENS=DENS, $
                   V_GSE=V_GSE, $
                   V_GSM=V_GSM, $
                   X_GSE=X_GSE, $
                   X_GSM=X_GSM, $
                   PRESS=PRESS
compile_opt strictarrsubs
  
COMMON get_error, get_err_no, get_err_msg, default_verbose
COMMON wind_swe_data, wind_time, wind_swe
  
get_err_no = 0
get_err_msg = 'Data found in time interval'
  
WIND_SWE_PATH = GETENV('WINDDATA')+'/swe/'

file_list = FINDFILE(WIND_SWE_PATH + '/*swe*', COUNT = fcount)
  
get_timespan, time_interval
t_s = gettime(time_interval(0)) ; start time in tplot-time
t_e = gettime(time_interval(1)) ; end time in tplot-time  
  
t_s_str = time_struct(t_s)      ; start_time tplot time structure
t_e_str = time_struct(t_e)      ; end_time tplot time structure
  
mjd_s = julday(t_s_str.month, t_s_str.date, t_s_str.year) ;start julian day
mjd_e = julday(t_e_str.month, t_e_str.date, t_e_str.year) ; end julian day

no_of_files = (mjd_e - mjd_s) + 1 ; number of days to be loaded
  
  ;Last day is not included if hour=min=sec=0
IF t_e_str.hour EQ 0 AND t_e_str.min EQ 0 AND t_e_str.sec EQ 0 THEN $
  no_of_files = no_of_files - 1
  
  ;--------------------------------------------------------------------
  ; Read all 1 day files that correspond to requested time interval
  ;--------------------------------------------------------------------
fc = 0                          ; Files-found counter
FOR ff = 0, no_of_files-1 DO BEGIN
    caldat, mjd_s + ff, month, day, year ; find caledar date

    filename = 'wi_k0_swe_' + $
      STRING(year,month,day,format='(i4.4,i2.2,i2.2)') + '_v*.cdf'
    
    r_wind_swe_cdf, wind_swe_path, filename
    
    IF n_elements(wind_swe) GT 0 THEN BEGIN 
        IF ff EQ 0 THEN BEGIN
            wind_swe_all = wind_swe
            wind_time_all = wind_time
        ENDIF ELSE BEGIN
            wind_swe_all = [wind_swe_all, wind_swe]
            wind_time_all = [wind_time_all, wind_time]
        ENDELSE
    ENDIF 
    
ENDFOR

  ;-------------------------------------------------------------------
  ; Find bad time tags
  ;-------------------------------------------------------------------
IF n_elements(wind_time_all) GT 0 THEN BEGIN 
  time_sort = SORT(wind_time_all)
  wind_time_all = wind_time_all(time_sort)
  wind_swe_all = wind_swe_all(time_sort)

  ; Clean bad time tags
  good_time = WHERE(wind_time_all GE t_s AND wind_time_all LE t_e, igood_time)
  IF igood_time GT 0 THEN BEGIN
      wind_time_all = wind_time_all(good_time)
      wind_swe_all = wind_swe_all(good_time)

      tvstr = 'WIND_SWE_'
  
      IF KEYWORD_SET(DENS) OR KEYWORD_SET(ALL) THEN BEGIN
          fillind = WHERE(wind_swe_all.Np NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              density = wind_swe_all(fillind).Np
              str = {x:time, y:density}
              store_data, tvstr + 'DENSITY', data = str
              options, tvstr + 'DENSITY', 'ytitle', 'n (cm!U-3!N)'
          ENDIF 
      ENDIF

      IF KEYWORD_SET(ALL) THEN BEGIN
          fillind = WHERE(wind_swe_all.Vth NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              thermal_vel = wind_swe_all(fillind).Vth
              str = {x:time, y:thermal_vel}
              store_data, tvstr + 'THERMAL_VEL', data = str
              options, tvstr + 'THERMAL_VEL', 'ytitle', 'Vth (km/s)'
          ENDIF 
      ENDIF

      IF KEYWORD_SET(X_GSE) OR KEYWORD_SET(ALL) THEN BEGIN
          
          fillind = WHERE(wind_swe_all.X_GSE_X NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              X_gse_x = wind_swe_all(fillind).X_GSE_X
              str = {x:time, y:X_gse_x}
              store_data, tvstr + 'X_GSE_X', data = str
              options, tvstr + 'X_GSE_X', 'ytitle', 'X gse'
          ENDIF 

          fillind = WHERE(wind_swe_all.X_GSE_Y NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              X_gse_y = wind_swe_all(fillind).X_GSE_Y
              str = {x:time, y:X_gse_y}
              store_data, tvstr + 'X_GSE_Y', data = str
              options, tvstr + 'X_GSE_Y', 'ytitle', 'Y gse'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.X_GSE_Z NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              X_gse_z = wind_swe_all(fillind).X_GSE_Z
              str = {x:time, y:X_gse_z}
              store_data, tvstr + 'X_GSE_Z', data = str
              options, tvstr + 'X_GSE_Z', 'ytitle', 'Z gse'
          ENDIF 
  
      ENDIF
 
      IF KEYWORD_SET(X_GSM) OR KEYWORD_SET(ALL) THEN BEGIN
          
          fillind = WHERE(wind_swe_all.X_GSM_X NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              X_gsm_x = wind_swe_all(fillind).X_GSM_X
              str = {x:time, y:X_gsm_x}
              store_data, tvstr + 'X_GSM_X', data = str
              options, tvstr + 'X_GSM_X', 'ytitle', 'X gsm'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.X_GSM_Y NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              X_gsm_y = wind_swe_all(fillind).X_GSM_Y
              str = {x:time, y:X_gsm_y}
              store_data, tvstr + 'X_GSM_Y', data = str
              options, tvstr + 'X_GSM_Y', 'ytitle', 'Y gsm'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.X_GSM_Z NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              X_gsm_z = wind_swe_all(fillind).X_GSM_Z
              str = {x:time, y:X_gsm_z}
              store_data, tvstr + 'X_GSM_Z', data = str
              options, tvstr + 'X_GSM_Z', 'ytitle', 'Z gsm'
          ENDIF 
  
      ENDIF
  
      IF KEYWORD_SET(V_GSE) OR KEYWORD_SET(ALL) THEN BEGIN
    
          fillind = WHERE(wind_swe_all.V_GSE_X NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              velocity = wind_swe_all(fillind).V_GSE_X
              str = {x:time, y:velocity}
              store_data, tvstr + 'V_GSE_X', data = str
              options, tvstr + 'V_GSE_X', 'ytitle', 'V!LX!N (km s!U-1!N)'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.V_GSE_Y NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              velocity = wind_swe_all(fillind).V_GSE_Y
              str = {x:time, y:velocity}
              store_data, tvstr + 'V_GSE_Y', data = str
              options, tvstr + 'V_GSE_Y', 'ytitle', 'V!LY!N (km s!U-1!N)'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.V_GSE_Z NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              velocity = wind_swe_all(fillind).V_GSE_Z
              str = {x:time, y:velocity}
              store_data, tvstr + 'V_GSE_Z', data = str
              options, tvstr + 'V_GSE_Z', 'ytitle', 'V!LZ!N (km s!U-1!N)'
          ENDIF 
  
      ENDIF

      IF KEYWORD_SET(V_GSM) OR KEYWORD_SET(ALL) THEN BEGIN
    
          fillind = WHERE(wind_swe_all.V_GSM_X NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              velocity = wind_swe_all(fillind).V_GSM_X
              str = {x:time, y:velocity}
              store_data, tvstr + 'V_GSM_X', data = str
              options, tvstr + 'V_GSM_X', 'ytitle', 'V!LX!N (km s!U-1!N)'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.V_GSM_Y NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              velocity = wind_swe_all(fillind).V_GSM_Y
              str = {x:time, y:velocity}
              store_data, tvstr + 'V_GSM_Y', data = str
              options, tvstr + 'V_GSM_Y', 'ytitle', 'V!LY!N (km s!U-1!N)'
          ENDIF 
  
          fillind = WHERE(wind_swe_all.V_GSM_Z NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              velocity = wind_swe_all(fillind).V_GSM_Z
              str = {x:time, y:velocity}
              store_data, tvstr + 'V_GSM_Z', data = str
              options, tvstr + 'V_GSM_Z', 'ytitle', 'V!LZ!N (km s!U-1!N)'
          ENDIF 
      ENDIF
  
      IF KEYWORD_SET(PRESS) OR KEYWORD_SET(ALL) THEN BEGIN
    
          velocity = SQRT(wind_swe_all.V_GSE_X^2 + $
                          wind_swe_all.V_GSE_Y^2 + $
                          wind_swe_all.V_GSE_Z^2)
    
          fillind = WHERE(wind_swe_all.Np NE -1.0e+31 AND $
                          velocity NE -1.0e+31, good_count)
          IF good_count GT 0 THEN BEGIN 
              time = wind_time_all(fillind)
              density = wind_swe_all(fillind).Np
              velocity = velocity
    
              mp = 1.67e-27     ; (kg)
              press = $
                density * 1.e6 * mp *$
                (velocity*1000.)^2*1.e9
              str = {x:time, y:press}
              store_data, tvstr + 'PRESSURE', data = str
              options, tvstr + 'PRESSURE', 'ytitle', 'P (nPa)'
          ENDIF 
      ENDIF
  ENDIF 
ENDIF 
  
END
