;+
;PROCEDURE: create_ra_mom_cdfs
;PURPOSE:
;   Take tplot variables with ra moments in them. Check for when
;   entrance system is off (and exclude that data). Put data in cdfs.
;
;   NOTE: Must be run in SPLAT.
;
;CREATED BY: L. Ellis
;
;MODIFICATION HISTORY:
;    02/08/2008 Created.
;    04/01/2008 Changed to incorporate Kristin's program.
;    04/28/2008 Added different directory for tplot variables.
;    05/20/2008 Added 10 min and 1 hr data.
;    05/27/2008 Added attitude_flag.
;    06/02/2008 Added spacecraft location and Carrington info.
;    06/18/2008 Change print format if NaN for velocity.
;    09/04/2008 Added check for if no data.
;    09/11/2008 Changed to V02.
;    09/22/2008 Fixed bug where set ts for B
;    10/08/2008 Added bad_times_start and _stop
;    11/26/2008 Changed to V03.
;    12/10/2008 Removed part that saved tplot variables.
;    03/03/2009 Added variables from pl_calc_ew_proton.
;    03/11/2009 Added check for when no valid 10min and 1hr data.
;               Fixed bug with speed.
;    03/16/2009 Added _1min to filenames. Changed ascii headers.
;    03/18/2009 Removed _hertn and _rtn from bulk speed. 
;               Added num cycles for n/s.
;               Added ew_inst and ns_inst.
;    04/13/2009 Fixed bug with spacecraft latitude.
;    04/15/2009 Added want_gse option to create gse ascii.
;    04/29/2009 Update yearly and monthly ascii files.
;    05/06/2009 Fixed uncertainty strings.
;    05/07/2009 Changed heliospheric to heliocentric, ascii file names.
;    05/15/2009 Fixed EW/NS bug. Put Non_valid in directory.
;    07/20/2009 Added empty_1min flag.
;    09/11/2009 Added error=8 for bad times. 
;               Changed 10min and 1hr ascii so print all times.
;    09/29/2009 Changed round( to round(float because rounding a double doesn't work with .5
;               Fixed bug in bad_times code (exclude partial 10min 1hr times)
;    10/01/2009 Changed ra_t_struct to ts_struct in 10min & 1hr update.
;    10/08/2009 Changed bad_times based on Kristin's bad_times_KDCS.xls
;    12/07/2009 Check for temperatures below 1000.
;    01/19/2010 Add monthly 1min files.
;    02/10/2010 Fix bug in bad_stop_time_1hr.
;    03/15/2010 Fix bug so that monthly 1min files update correctly.
;    04/20/2010 Put in Bad Time for B. (4/5-6/2007)
;    07/15/2010 Changed 10min and 1hr ascii so print fake times.
;    09/13/2010 Changed error code from -1 to 5 for missing data.
;    09/23/2010 Update monthly 1min file even if have no data.
;    10/08/2010 Set red_chi2 to NaN when remove data.
;    06/30/2011 Change for V05 skt files (ISTP compliant -- ReducedChi changed)
;    07/12/2011 Change for new SKT filenames (STA/STB)
;    08/16/2011 Removed bad time for B (4/5-6/2007).
;    12/20/2011 Added key_test keyword for when testing new code.
;    03/16/2012 Added roll_test keyword.
;    04/04/2012 Added swz_test keyword.
;    04/06/2012 Added checks for if ew data missing.
;    05/23/2013 Update to add Spacecraft info when no data.
;    04/14/2014 Change name of skt file (should have happened in 2011!)
;    11/20/2014 Fix ew_source_flag_st.
;    11/25/2014 Change so ascii for 10min and 1hr has error 0 when no data.
;    12/19/2014 Add bad density time for STB 9/1/2014
;    03/05/2020 Changes in calc_proton_bulk_v1.pro
;               Check for sweeping with dacm_esa_ctl. Lower esa_cm_dc limit.
;    05/07/2020 Change den decimal places to 3 for small
;    06/15/2020 Add PAC efficiency.
;-

PRO create_1min_header, temp_string
compile_opt strictarrsubs
tab = string(9B)
temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+'sec'+tab+'millisec'+tab+$
              'date and time'+tab+'1 keV/q time'+tab+$
              'Np [1/cc]'+tab+$
              'Bulk Speed [km/s]'+tab+$
              'Tkin [deg K]'+tab+$
              'v_th [km/s]'+tab+$
              'N/S Inst. [deg]'+tab+$
              'E/W Inst. [deg]'+tab+$
              'Vr HERTN [km/s]'+tab+$
              'Vt HERTN [km/s]'+tab+$
              'Vn HERTN [km/s]'+tab+$
              'N/S HERTN [deg]'+tab+$
              'E/W HERTN [deg]'+tab+$
              'Vr RTN [km/s]'+tab+$
              'Vt RTN [km/s]'+tab+$
              'Vn RTN [km/s]'+tab+$
              'N/S RTN [deg]'+tab+$
              'E/W RTN [deg]'+tab+$
              'Error Code'+tab+'Caution'+tab+$
              'Attitude Flag (roll)'+tab+'Attitude Flag (yaw)'+tab+'Attitude Flag (pitch)'+tab+$
              'E/W Source Flag'+tab+'E/W Missed Peak Flag'+tab+$
              'Reduced Chi^2'+tab+'E/W Rel Stat Uncertainty (%)'+tab+$
              'Carr.Rot'+tab+'Spcrft. Long. [Carr, degrees]'+tab+'Heliocentric Distance [km]'+tab+$
              'Spcrft. Long. [HEE, degrees]'+tab+'Spcrft. Lat. [HEE, degrees]'+tab+$
              'Spcrft. Long. [HEEQ, degrees]'+tab+'Spcrft. Lat. [HEEQ, degrees]'+tab+$
              'Spcrft. Long. [HCI, degrees]'+tab+'Spcrft. Lat. [HCI, degrees]'
END 

PRO update_1min, have_data_1min, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, ns_inst_st, ew_inst_st, $
                 vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, vr_rtn_st, vt_rtn_st, vn_rtn_st, $
                 ns_rtn_st, ew_rtn_st, error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                 carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                 out_lun, temp_doy, temp_year, sat
compile_opt strictarrsubs
doy_to_month_date, temp_year, temp_doy, temp_month, temp_date
tab = string(9B)
time_i = 0
IF have_data_1min EQ 1 THEN BEGIN 
    ra_t_struct = time_struct(ra_t)
    ra_t_string = time_string(ra_t)
    kev_string = time_string(epoch_1kev)
ENDIF
FOR kk = 0, 23 DO BEGIN        ; hours
    FOR mm = 0, 59 DO BEGIN    ; minutes
        IF have_data_1min EQ 0 OR time_i EQ n_elements(ra_t) THEN BEGIN 
            temp_string = string(temp_year, format = '(I4.4)')+tab+string(temp_doy, format = '(I3.3)')+tab+$
                          string(kk,        format = '(I2.2)')+tab+string(mm,       format = '(I2.2)')+tab+$
                          '00'+tab+'00' ; seconds and milli
            time_string = string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; time
                          string(temp_date, format = '(I2.2)')+'/'+$
                          string(kk,        format = '(I2.2)')+':'+string(mm,         format = '(I2.2)')+':00'
            temp_string = temp_string+tab+time_string+tab+$
                          string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; kev time
                          string(temp_date, format = '(I2.2)')+'/'+$
                          string(kk,        format = '(I2.2)')+':'+string(mm,         format = '(I2.2)')+':00'
            FOR nn = 0, 15 DO BEGIN 
                temp_string = temp_string + tab+'-1.0E+31'
            ENDFOR 
            temp_string = temp_string + tab+'5' ; error code
            FOR nn = 0, 05 DO BEGIN 
                temp_string = temp_string + tab+'-1'
            ENDFOR 
            FOR nn = 0, 1 DO BEGIN 
                temp_string = temp_string + tab+'-1.0E+31'
            ENDFOR 
            ; add spacecraft data
            strput, time_string, 'T', 10 ; change time string format for SolarSoft
            temp_carr_num = get_stereo_carr_rot(time_string, sat)
            temp_value = get_stereo_lonlat(time_string, sat, system = 'carrington', /degrees)
            temp_carr_lon = reform(temp_value[1, *])
            IF temp_carr_lon LT 0 THEN temp_carr_lon = temp_carr_long + 360 ; range should be 0-360
            temp_helio_dist = reform(temp_value[0, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HEE', /degrees)
            temp_hee_lon = reform(temp_value[1, *])
            temp_hee_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HEEQ', /degrees)
            temp_heeq_lon = reform(temp_value[1, *])
            temp_heeq_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HCI', /degrees)
            temp_hci_lon = reform(temp_value[1, *])
            IF temp_hci_lon LT 0 THEN temp_hci_lon = temp_hci_lon + 360 ; range should be 0-360
            temp_hci_lat = reform(temp_value[2, *])
            temp_string = temp_string + tab +$
                          string(temp_carr_num,   format = '(F10.4)')+tab+$
                          string(temp_carr_lon,   format = '(F10.4)')+tab+$
                          string(temp_helio_dist, format = '(E12.5)')+tab+$
                          string(temp_hee_lon,    format = '(F10.4)')+tab+$
                          string(temp_hee_lat,    format = '(F10.4)')+tab+$
                          string(temp_heeq_lon,   format = '(F10.4)')+tab+$
                          string(temp_heeq_lat,   format = '(F10.4)')+tab+$
                          string(temp_hci_lon,    format = '(F10.4)')+tab+$
                          string(temp_hci_lat,    format = '(F10.4)')
        ENDIF ELSE BEGIN 
            found = 0
            WHILE finite(ra_t[time_i]) EQ 0 DO BEGIN 
                time_i = time_i + 1
            ENDWHILE 
            IF ra_t_struct[time_i].hour LT kk OR $
              (ra_t_struct[time_i].hour EQ kk AND ra_t_struct[time_i].min LT mm) THEN BEGIN ; check for 2 in a minute
                IF ra_t[time_i]-ra_t[time_i-1] LT 60 THEN mm = mm - 1 ELSE stop ; shouldn't get here
            ENDIF 
            IF ra_t_struct[time_i].hour GT kk OR $
              (ra_t_struct[time_i].hour EQ kk AND ra_t_struct[time_i].min GE (mm+1)) THEN BEGIN ; no valid time
                temp_string = string(temp_year, format = '(I4.4)')+tab+string(temp_doy, format = '(I3.3)')+tab+$
                              string(kk,        format = '(I2.2)')+tab+string(mm,       format = '(I2.2)')+tab+$
                              '00'+tab+'00' ; seconds and milli
                time_string = string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; time
                              string(temp_date, format = '(I2.2)')+'/'+$
                              string(kk,        format = '(I2.2)')+':'+string(mm,         format = '(I2.2)')+':00'
                temp_string = temp_string+tab+time_string+tab+$
                              string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; kev time
                              string(temp_date, format = '(I2.2)')+'/'+$
                              string(kk,        format = '(I2.2)')+':'+string(mm,         format = '(I2.2)')+':00'
                FOR nn = 0, 15 DO BEGIN 
                    temp_string = temp_string + tab+'-1.0E+31'
                ENDFOR 
                temp_string = temp_string + tab+'5' ; error code
                FOR nn = 0, 05 DO BEGIN 
                    temp_string = temp_string + tab+'-1'
                ENDFOR 
                FOR nn = 0, 1 DO BEGIN 
                    temp_string = temp_string + tab+'-1.0E+31'
                ENDFOR 
            ; add spacecraft data
                strput, time_string, 'T', 10 ; change time string format for SolarSoft
                temp_carr_num = get_stereo_carr_rot(time_string, sat)
                temp_value = get_stereo_lonlat(time_string, sat, system = 'carrington', /degrees)
                temp_carr_lon = reform(temp_value[1, *])
                IF temp_carr_lon LT 0 THEN temp_carr_lon = temp_carr_lon + 360 ; range should be 0-360
                temp_helio_dist = reform(temp_value[0, *])
                temp_value = get_stereo_lonlat(time_string, sat, system = 'HEE', /degrees)
                temp_hee_lon = reform(temp_value[1, *])
                temp_hee_lat = reform(temp_value[2, *])
                temp_value = get_stereo_lonlat(time_string, sat, system = 'HEEQ', /degrees)
                temp_heeq_lon = reform(temp_value[1, *])
                temp_heeq_lat = reform(temp_value[2, *])
                temp_value = get_stereo_lonlat(time_string, sat, system = 'HCI', /degrees)
                temp_hci_lon = reform(temp_value[1, *])
                IF temp_hci_lon LT 0 THEN temp_hci_lon = temp_hci_lon + 360 ; range should be 0-360
                temp_hci_lat = reform(temp_value[2, *])
                temp_string = temp_string + tab +$
                              string(temp_carr_num,   format = '(F10.4)')+tab+$
                              string(temp_carr_lon,   format = '(F10.4)')+tab+$
                              string(temp_helio_dist, format = '(E12.5)')+tab+$
                              string(temp_hee_lon,    format = '(F10.4)')+tab+$
                              string(temp_hee_lat,    format = '(F10.4)')+tab+$
                              string(temp_heeq_lon,   format = '(F10.4)')+tab+$
                              string(temp_heeq_lat,   format = '(F10.4)')+tab+$
                              string(temp_hci_lon,    format = '(F10.4)')+tab+$
                              string(temp_hci_lat,    format = '(F10.4)')
            ENDIF ELSE BEGIN 
                temp_string = string(ra_t_struct[time_i].year, format = '(I4.4)')+tab+$
                              string(ra_t_struct[time_i].doy,  format = '(I3.3)')+tab+$
                              string(ra_t_struct[time_i].hour, format = '(I2.2)')+tab+$
                              string(ra_t_struct[time_i].min,  format = '(I2.2)')+tab+$
                              string(ra_t_struct[time_i].sec,  format = '(I2.2)')+tab+$
                              string(ra_t_struct[time_i].fsec, format = '(F0.3)')+tab+$
                              ra_t_string[time_i]+tab+$
                              kev_string [time_i]+tab+$
                              den_st     [time_i]+tab+$
                              speed_st   [time_i]+tab+$ 
                              temp_st    [time_i]+tab+$
                              v_therm_st [time_i]+tab+$
                              ns_inst_st [time_i]+tab+$
                              ew_inst_st [time_i]+tab+$
                              vr_hertn_st[time_i]+tab+$
                              vt_hertn_st[time_i]+tab+$
                              vn_hertn_st[time_i]+tab+$
                              ns_hertn_st[time_i]+tab+$
                              ew_hertn_st[time_i]+tab+$
                              vr_rtn_st  [time_i]+tab+$
                              vt_rtn_st  [time_i]+tab+$
                              vn_rtn_st  [time_i]+tab+$
                              ns_rtn_st  [time_i]+tab+$
                              ew_rtn_st  [time_i]+tab+$
                              string(error     [time_i   ])+tab+string(caution   [time_i   ])+tab+$
                              string(attitude  [0, time_i])+tab+string(attitude  [1, time_i])+tab+$
                              string(attitude  [2, time_i])+tab+$
                              ew_source_flag_st[time_i, 0] +tab+ew_source_flag_st[time_i, 1] +tab+$ 
                              red_chi_sq_st[time_i]+tab+uncert_st[time_i]+tab+$
                              string(carr_rot  [time_i], format = '(F10.4)')+tab+string(lon_carr[time_i],   format = '(F10.4)')+tab+$
                              string(helio_dist[time_i], format = '(E12.5)')+tab+$
                              string(lon_hee   [time_i], format = '(F10.4)')+tab+string(lat_hee [time_i],   format = '(F10.4)')+tab+$
                              string(lon_heeq  [time_i], format = '(F10.4)')+tab+string(lat_heeq[time_i],   format = '(F10.4)')+tab+$
                              string(lon_hci   [time_i], format = '(F10.4)')+tab+string(lat_hci [time_i],   format = '(F10.4)') 
                time_i = time_i + 1
            ENDELSE 
        ENDELSE 
        printf, out_lun, temp_string
    ENDFOR 
ENDFOR 
END 

PRO create_10min_header, temp_string
compile_opt strictarrsubs
tab = string(9B)
temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+$
              'date and time'+tab+$
              'Np [1/cc]'+tab+$
              'Bulk Speed [km/s]'+tab+$
              'Tkin [deg K]'+tab+$
              'v_th [km/s]'+tab+$
              'N/S Inst. [deg]'+tab+$
              'E/W Inst. [deg]'+tab+$
              'Vr HERTN [km/s]'+tab+$
              'Vt HERTN [km/s]'+tab+$
              'Vn HERTN [km/s]'+tab+$
              'N/S HERTN [deg]'+tab+$
              'E/W HERTN [deg]'+tab+$
              'Vr RTN [km/s]'+tab+$
              'Vt RTN [km/s]'+tab+$
              'Vn RTN [km/s]'+tab+$
              'N/S RTN [deg]'+tab+$
              'E/W RTN [deg]'+tab+$
              'Num Cycles (Np, v_therm, temp)'+tab+'Num Cycles (Speed)'+tab+$
              'Num Cycles (NS_inst)'+tab+'Num Cycles (Vel. Com., Other Angles)'+tab+$
              'Error Code'+tab+'Caution'+tab+$
              'Attitude Flag (roll)'+tab+'Attitude Flag (yaw)'+tab+'Attitude Flag (pitch)'+tab+$
              'E/W Source Flag'+tab+'E/W Missed Peak Flag'+tab+$
              'Reduced Chi^2'+tab+'E/W Rel Stat Uncertainty (%)'+tab+$
              'Carr.Rot'+tab+'Spcrft. Long. [Carr, degrees]'+tab+'Heliospheric Distance [km]'+tab+$
              'Spcrft. Long. [HEE, degrees]'+tab+'Spcrft. Lat. [HEE, degrees]'+tab+$
              'Spcrft. Long. [HEEQ, degrees]'+tab+'Spcrft. Lat. [HEEQ, degrees]'+tab+$
              'Spcrft. Long. [HCI, degrees]'+tab+'Spcrft. Lat. [HCI, degrees]'
END 

PRO update_10min, have_data_10min,     ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                  v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                  vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                  vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                  cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                  error_10min,         caution_10min,      attitude_10min,       ew_source_flag_st_10min, $
                  red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                  lon_carr_10min,      helio_dist_10min, $
                  lon_hee_10min,       lat_hee_10min,    $
                  lon_heeq_10min,      lat_heeq_10min,   $
                  lon_hci_10min,       lat_hci_10min,    out_lun, temp_doy, temp_year, sat
compile_opt strictarrsubs
doy_to_month_date, temp_year, temp_doy, temp_month, temp_date
tab = string(9B)
time_i = 0
IF have_data_10min EQ 1 THEN BEGIN 
    ra_t_struct = time_struct(ra_t_10min)
    ra_t_string = time_string(ra_t_10min) 
ENDIF 
FOR kk = 0, 23 DO BEGIN         ; hours
    FOR mm = 0, 5 DO BEGIN      ; 10 minutes
        IF have_data_10min EQ 0 OR time_i EQ n_elements(ra_t_10min) THEN BEGIN 
            temp_string = string(temp_year, format = '(I4.4)')+tab+string(temp_doy,   format = '(I3.3)')+tab+$
                          string(kk,        format = '(I2.2)')+tab+string((mm*10),    format = '(I2.2)')+tab
            time_string = string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; time
                          string(temp_date, format = '(I2.2)')+'/'+$
                          string(kk,        format = '(I2.2)')+':'+string((mm*10),    format = '(I2.2)')+':00'
            temp_string = temp_string+time_string
            FOR nn = 0, 15 DO BEGIN 
                temp_string = temp_string + tab+'-1.0E+31'
            ENDFOR 
            FOR nn = 0, 3 DO BEGIN 
                temp_string = temp_string + tab+'0' ; (change from -1 starting data 8/2014)
            ENDFOR 
            temp_string = temp_string + tab+'0' ; error code (change from 5 starting data 8/2014)
            FOR nn = 0, 5 DO BEGIN 
                temp_string = temp_string + tab+'-1'
            ENDFOR 
            FOR nn = 0, 1 DO BEGIN 
                temp_string = temp_string + tab+'-1.0E+31'
            ENDFOR 
            ; add spacecraft data
            strput, time_string, 'T', 10 ; change time string format for SolarSoft
            temp_carr_num = get_stereo_carr_rot(time_string, sat)
            temp_value = get_stereo_lonlat(time_string, sat, system = 'carrington', /degrees)
            temp_carr_lon = reform(temp_value[1, *])
            IF temp_carr_lon LT 0 THEN temp_carr_lon = temp_carr_lon + 360 ; range should be 0-360
            temp_helio_dist = reform(temp_value[0, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HEE', /degrees)
            temp_hee_lon = reform(temp_value[1, *])
            temp_hee_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HEEQ', /degrees)
            temp_heeq_lon = reform(temp_value[1, *])
            temp_heeq_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HCI', /degrees)
            temp_hci_lon = reform(temp_value[1, *])
            IF temp_hci_lon LT 0 THEN temp_hci_lon = temp_hci_lon + 360 ; range should be 0-360
            temp_hci_lat = reform(temp_value[2, *])
            temp_string = temp_string + tab +$
                          string(temp_carr_num,   format = '(F10.4)')+tab+$
                          string(temp_carr_lon,   format = '(F10.4)')+tab+$
                          string(temp_helio_dist, format = '(E12.5)')+tab+$
                          string(temp_hee_lon,    format = '(F10.4)')+tab+$
                          string(temp_hee_lat,    format = '(F10.4)')+tab+$
                          string(temp_heeq_lon,   format = '(F10.4)')+tab+$
                          string(temp_heeq_lat,   format = '(F10.4)')+tab+$
                          string(temp_hci_lon,    format = '(F10.4)')+tab+$
                          string(temp_hci_lat,    format = '(F10.4)')
        ENDIF ELSE BEGIN 
            found = 0
            WHILE finite(ra_t_10min[time_i]) EQ 0 DO BEGIN 
                time_i = time_i + 1
            ENDWHILE 
            IF ra_t_struct[time_i].hour LT kk OR $
              (ra_t_struct[time_i].hour EQ kk AND ra_t_struct[time_i].min LT (mm*10)) THEN stop $ ; shouldn't get here
            ELSE IF ra_t_struct[time_i].hour GT kk OR $
              (ra_t_struct[time_i].hour EQ kk AND ra_t_struct[time_i].min GE ((mm+1)*10)) THEN BEGIN ; no valid time
                temp_string = string(temp_year, format = '(I4.4)')+tab+string(temp_doy, format = '(I3.3)')+tab+$
                              string(kk,        format = '(I2.2)')+tab+string((mm*10),  format = '(I2.2)')+tab
                time_string = string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; time
                              string(temp_date, format = '(I2.2)')+'/'+$
                              string(kk,        format = '(I2.2)')+':'+string((mm*10),    format = '(I2.2)')+':00'
                temp_string = temp_string+time_string
                FOR nn = 0, 15 DO BEGIN 
                    temp_string = temp_string + tab+'-1.0E+31'
                ENDFOR 
                FOR nn = 0, 3 DO BEGIN 
                    temp_string = temp_string + tab+'0' ; (change from -1 starting data 8/2014)
                ENDFOR 
                temp_string = temp_string + tab+'0' ; error code (change from 5 starting data 8/2014)
                FOR nn = 0, 5 DO BEGIN 
                    temp_string = temp_string + tab+'-1'
                ENDFOR 
                FOR nn = 0, 1 DO BEGIN 
                    temp_string = temp_string + tab+'-1.0E+31'
                ENDFOR 
            ; add spacecraft data
                strput, time_string, 'T', 10 ; change time string format for SolarSoft
                temp_carr_num = get_stereo_carr_rot(time_string, sat)
                temp_value = get_stereo_lonlat(time_string, sat, system = 'carrington', /degrees)
                temp_carr_lon = reform(temp_value[1, *])
                IF temp_carr_lon LT 0 THEN temp_carr_lon = temp_carr_lon + 360 ; range should be 0-360
                temp_helio_dist = reform(temp_value[0, *])
                temp_value = get_stereo_lonlat(time_string, sat, system = 'HEE', /degrees)
                temp_hee_lon = reform(temp_value[1, *])
                temp_hee_lat = reform(temp_value[2, *])
                temp_value = get_stereo_lonlat(time_string, sat, system = 'HEEQ', /degrees)
                temp_heeq_lon = reform(temp_value[1, *])
                temp_heeq_lat = reform(temp_value[2, *])
                temp_value = get_stereo_lonlat(time_string, sat, system = 'HCI', /degrees)
                temp_hci_lon = reform(temp_value[1, *])
                IF temp_hci_lon LT 0 THEN temp_hci_lon = temp_hci_lon + 360 ; range should be 0-360
                temp_hci_lat = reform(temp_value[2, *])
                temp_string = temp_string + tab +$
                              string(temp_carr_num,   format = '(F10.4)')+tab+$
                              string(temp_carr_lon,   format = '(F10.4)')+tab+$
                              string(temp_helio_dist, format = '(E12.5)')+tab+$
                              string(temp_hee_lon,    format = '(F10.4)')+tab+$
                              string(temp_hee_lat,    format = '(F10.4)')+tab+$
                              string(temp_heeq_lon,   format = '(F10.4)')+tab+$
                              string(temp_heeq_lat,   format = '(F10.4)')+tab+$
                              string(temp_hci_lon,    format = '(F10.4)')+tab+$
                              string(temp_hci_lat,    format = '(F10.4)')
            ENDIF ELSE BEGIN 
                temp_string = string(ra_t_struct[time_i].year, format = '(I4.4)')+tab+$
                              string(ra_t_struct[time_i].doy,  format = '(I3.3)')+tab+$
                              string(ra_t_struct[time_i].hour, format = '(I2.2)')+tab+$
                              string((mm*10), format = '(I2.2)')+tab+$
                              ra_t_string      [time_i]+tab+$
                              den_10min_st     [time_i]+tab+$
                              speed_10min_st   [time_i]+tab+$
                              temp_10min_st    [time_i]+tab+$
                              v_therm_10min_st [time_i]+tab+$
                              ns_inst_10min_st [time_i]+tab+$
                              ew_inst_10min_st [time_i]+tab+$
                              vr_hertn_10min_st[time_i]+tab+$
                              vt_hertn_10min_st[time_i]+tab+$
                              vn_hertn_10min_st[time_i]+tab+$ 
                              ns_hertn_10min_st[time_i]+tab+$ 
                              ew_hertn_10min_st[time_i]+tab+$ 
                              vr_rtn_10min_st  [time_i]+tab+$ 
                              vt_rtn_10min_st  [time_i]+tab+$ 
                              vn_rtn_10min_st  [time_i]+tab+$ 
                              ns_rtn_10min_st  [time_i]+tab+$ 
                              ew_rtn_10min_st  [time_i]+tab+$ 
                              string(cycles_10min        [time_i   ])+tab+string(cycles_speed_10min  [time_i   ])+tab+$
                              string(cycles_ns_inst_10min[time_i   ])+tab+       cycles_ew_10min_st  [time_i   ] +tab+$
                              string(error_10min         [time_i   ])+tab+string(caution_10min       [time_i   ])+tab+$
                              string(attitude_10min      [0, time_i])+tab+string(attitude_10min      [1, time_i])+tab+$
                              string(attitude_10min      [2, time_i])+tab+$
                              ew_source_flag_st_10min    [time_i, 0] +tab+ew_source_flag_st_10min    [time_i, 1] +tab+$ 
                              red_chi_sq_10min_st[time_i]+tab+$
                              uncert_10min_st    [time_i]+tab+$
                              string(carr_rot_10min  [time_i], format = '(F10.4)')+tab+$
                              string(lon_carr_10min  [time_i], format = '(F10.4)')+tab+$
                              string(helio_dist_10min[time_i], format = '(E12.5)')+tab+$
                              string(lon_hee_10min   [time_i], format = '(F10.4)')+tab+$
                              string(lat_hee_10min   [time_i], format = '(F10.4)')+tab+$
                              string(lon_heeq_10min  [time_i], format = '(F10.4)')+tab+$
                              string(lat_heeq_10min  [time_i], format = '(F10.4)')+tab+$
                              string(lon_hci_10min   [time_i], format = '(F10.4)')+tab+$
                              string(lat_hci_10min   [time_i], format = '(F10.4)') 
                time_i = time_i + 1
            ENDELSE 
        ENDELSE 
        printf, out_lun, temp_string
    ENDFOR
ENDFOR 
END  

PRO create_1hr_header, temp_string
compile_opt strictarrsubs
tab = string(9B)
temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+$
              'date and time'+tab+$
              'Np [1/cc]'+tab+$
              'Bulk Speed [km/s]'+tab+$
              'Tkin [deg K]'+tab+$
              'v_th [km/s]'+tab+$
              'N/S Inst. [deg]'+tab+$
              'E/W Inst. [deg]'+tab+$
              'Vr HERTN [km/s]'+tab+$
              'Vt HERTN [km/s]'+tab+$
              'Vn HERTN [km/s]'+tab+$
              'N/S HERTN [deg]'+tab+$
              'E/W HERTN [deg]'+tab+$
              'Vr RTN [km/s]'+tab+$
              'Vt RTN [km/s]'+tab+$
              'Vn RTN [km/s]'+tab+$
              'N/S RTN [deg]'+tab+$
              'E/W RTN [deg]'+tab+$
              'Num Cycles (Np, v_therm, temp)'+tab+'Num Cycles (Speed)'+tab+$
              'Num Cycles (NS_inst)'+tab+'Num Cycles (Vel. Com., Other Angles)'+tab+$
              'Error Code'+tab+'Caution'+tab+$
              'Attitude Flag (roll)'+tab+'Attitude Flag (yaw)'+tab+'Attitude Flag (pitch)'+tab+$
              'E/W Source Flag'+tab+'E/W Missed Peak Flag'+tab+$
              'Reduced Chi^2'+tab+'E/W Rel Stat Uncertainty (%)'+tab+$
              'Carr.Rot'+tab+'Spcrft. Long. [Carr, degrees]'+tab+'Heliospheric Distance [km]'+tab+$
              'Spcrft. Long. [HEE, degrees]'+tab+'Spcrft. Lat. [HEE, degrees]'+tab+$
              'Spcrft. Long. [HEEQ, degrees]'+tab+'Spcrft. Lat. [HEEQ, degrees]'+tab+$
              'Spcrft. Long. [HCI, degrees]'+tab+'Spcrft. Lat. [HCI, degrees]'
END 

PRO update_1hr, have_data_1hr,     ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                v_therm_1hr_st,                                          ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                lon_carr_1hr,      helio_dist_1hr, $
                lon_hee_1hr,       lat_hee_1hr,    $
                lon_heeq_1hr,      lat_heeq_1hr,   $
                lon_hci_1hr,       lat_hci_1hr,    out_lun, temp_doy, temp_year, sat
compile_opt strictarrsubs
doy_to_month_date, temp_year, temp_doy, temp_month, temp_date
tab = string(9B)
time_i = 0
IF have_data_1hr EQ 1 THEN BEGIN 
    ra_t_struct = time_struct(ra_t_1hr)
    ra_t_string = time_string(ra_t_1hr)
ENDIF 
FOR kk = 0, 23 DO BEGIN         ; hours
    IF have_data_1hr EQ 0 OR time_i EQ n_elements(ra_t_1hr) THEN BEGIN 
        temp_string = string(temp_year, format = '(I4.4)')+tab+string(temp_doy, format = '(I3.3)')+tab+$
                      string(kk,        format = '(I2.2)')+tab
        time_string = string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; time
                      string(temp_date, format = '(I2.2)')+'/'+$
                      string(kk,        format = '(I2.2)')+':00:00'
        temp_string = temp_string+time_string
        FOR nn = 0, 15 DO BEGIN 
            temp_string = temp_string + tab+'-1.0E+31'
        ENDFOR 
        FOR nn = 0, 3 DO BEGIN 
            temp_string = temp_string + tab+'0' ; (change from -1 starting data 8/2014)
        ENDFOR 
        temp_string = temp_string + tab+'0' ; error code (change from 5 starting data 8/2014)
        FOR nn = 0, 5 DO BEGIN 
            temp_string = temp_string + tab+'-1'
        ENDFOR 
        FOR nn = 0, 1 DO BEGIN 
            temp_string = temp_string + tab+'-1.0E+31'
        ENDFOR 
            ; add spacecraft data
        strput, time_string, 'T', 10 ; change time string format for SolarSoft
        temp_carr_num = get_stereo_carr_rot(time_string, sat)
        temp_value = get_stereo_lonlat(time_string, sat, system = 'carrington', /degrees)
        temp_carr_lon = reform(temp_value[1, *])
        IF temp_carr_lon LT 0 THEN temp_carr_lon = temp_carr_lon + 360 ; range should be 0-360
        temp_helio_dist = reform(temp_value[0, *])
        temp_value = get_stereo_lonlat(time_string, sat, system = 'HEE', /degrees)
        temp_hee_lon = reform(temp_value[1, *])
        temp_hee_lat = reform(temp_value[2, *])
        temp_value = get_stereo_lonlat(time_string, sat, system = 'HEEQ', /degrees)
        temp_heeq_lon = reform(temp_value[1, *])
        temp_heeq_lat = reform(temp_value[2, *])
        temp_value = get_stereo_lonlat(time_string, sat, system = 'HCI', /degrees)
        temp_hci_lon = reform(temp_value[1, *])
        IF temp_hci_lon LT 0 THEN temp_hci_lon = temp_hci_lon + 360 ; range should be 0-360
        temp_hci_lat = reform(temp_value[2, *])
        temp_string = temp_string + tab +$
                      string(temp_carr_num,   format = '(F10.4)')+tab+$
                      string(temp_carr_lon,   format = '(F10.4)')+tab+$
                      string(temp_helio_dist, format = '(E12.5)')+tab+$
                      string(temp_hee_lon,    format = '(F10.4)')+tab+$
                      string(temp_hee_lat,    format = '(F10.4)')+tab+$
                      string(temp_heeq_lon,   format = '(F10.4)')+tab+$
                      string(temp_heeq_lat,   format = '(F10.4)')+tab+$
                      string(temp_hci_lon,    format = '(F10.4)')+tab+$
                      string(temp_hci_lat,    format = '(F10.4)')
    ENDIF ELSE BEGIN 
        WHILE finite(ra_t_1hr[time_i]) EQ 0 DO BEGIN 
            time_i = time_i + 1
        ENDWHILE 
        IF ra_t_struct[time_i].hour LT kk THEN stop $ ; shouldn't get here
        ELSE IF ra_t_struct[time_i].hour GT kk THEN BEGIN ; no valid time
            temp_string = string(temp_year, format = '(I4.4)')+tab+string(temp_doy, format = '(I3.3)')+tab+$
                          string(kk,        format = '(I2.2)')+tab
            time_string = string(temp_year, format = '(I4.4)')+'-'+string(temp_month, format = '(I2.2)')+'-'+$ ; time
                          string(temp_date, format = '(I2.2)')+'/'+$
                          string(kk,        format = '(I2.2)')+':00:00'
            temp_string = temp_string+time_string
            FOR nn = 0, 15 DO BEGIN 
                temp_string = temp_string + tab+'-1.0E+31'
            ENDFOR 
            FOR nn = 0, 3 DO BEGIN 
                temp_string = temp_string + tab+'0' ; (change from -1 starting data 8/2014)
            ENDFOR 
            temp_string = temp_string + tab+'0' ; error code (change from 5 starting data 8/2014)
            FOR nn = 0, 5 DO BEGIN 
                temp_string = temp_string + tab+'-1'
            ENDFOR 
            FOR nn = 0, 1 DO BEGIN 
                temp_string = temp_string + tab+'-1.0E+31'
            ENDFOR 
            ; add spacecraft data
            strput, time_string, 'T', 10 ; change time string format for SolarSoft
            temp_carr_num = get_stereo_carr_rot(time_string, sat)
            temp_value = get_stereo_lonlat(time_string, sat, system = 'carrington', /degrees)
            temp_carr_lon = reform(temp_value[1, *])
            IF temp_carr_lon LT 0 THEN temp_carr_lon = temp_carr_lon + 360 ; range should be 0-360
            temp_helio_dist = reform(temp_value[0, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HEE', /degrees)
            temp_hee_lon = reform(temp_value[1, *])
            temp_hee_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HEEQ', /degrees)
            temp_heeq_lon = reform(temp_value[1, *])
            temp_heeq_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(time_string, sat, system = 'HCI', /degrees)
            temp_hci_lon = reform(temp_value[1, *])
            IF temp_hci_lon LT 0 THEN temp_hci_lon = temp_hci_lon + 360 ; range should be 0-360
            temp_hci_lat = reform(temp_value[2, *])
            temp_string = temp_string + tab +$
                          string(temp_carr_num,   format = '(F10.4)')+tab+$
                          string(temp_carr_lon,   format = '(F10.4)')+tab+$
                          string(temp_helio_dist, format = '(E12.5)')+tab+$
                          string(temp_hee_lon,    format = '(F10.4)')+tab+$
                          string(temp_hee_lat,    format = '(F10.4)')+tab+$
                          string(temp_heeq_lon,   format = '(F10.4)')+tab+$
                          string(temp_heeq_lat,   format = '(F10.4)')+tab+$
                          string(temp_hci_lon,    format = '(F10.4)')+tab+$
                          string(temp_hci_lat,    format = '(F10.4)')
        ENDIF ELSE BEGIN 
            temp_string = string(ra_t_struct[time_i].year, format = '(I4.4)')+tab+$
                          string(ra_t_struct[time_i].doy, format = '(I3.3)')+tab+$
                          string(ra_t_struct[time_i].hour, format = '(I2.2)')+tab+$
                          ra_t_string    [time_i]+tab+$
                          den_1hr_st     [time_i]+tab+$
                          speed_1hr_st   [time_i]+tab+$ 
                          temp_1hr_st    [time_i]+tab+$
                          v_therm_1hr_st [time_i]+tab+$
                          ns_inst_1hr_st [time_i]+tab+$ 
                          ew_inst_1hr_st [time_i]+tab+$ 
                          vr_hertn_1hr_st[time_i]+tab+$ 
                          vt_hertn_1hr_st[time_i]+tab+$ 
                          vn_hertn_1hr_st[time_i]+tab+$ 
                          ns_hertn_1hr_st[time_i]+tab+$ 
                          ew_hertn_1hr_st[time_i]+tab+$ 
                          vr_rtn_1hr_st  [time_i]+tab+$ 
                          vt_rtn_1hr_st  [time_i]+tab+$ 
                          vn_rtn_1hr_st  [time_i]+tab+$ 
                          ns_rtn_1hr_st  [time_i]+tab+$ 
                          ew_rtn_1hr_st  [time_i]+tab+$ 
                          string(cycles_1hr        [time_i   ])+tab+string(cycles_speed_1hr  [time_i   ])+tab+$
                          string(cycles_ns_inst_1hr[time_i   ])+tab+       cycles_ew_1hr_st  [time_i   ] +tab+$
                          string(error_1hr         [time_i   ])+tab+string(caution_1hr       [time_i   ])+tab+$
                          string(attitude_1hr      [0, time_i])+tab+string(attitude_1hr      [1, time_i])+tab+$
                          string(attitude_1hr      [2, time_i])+tab+$
                          ew_source_flag_1hr_st    [time_i, 0] +tab+ew_source_flag_1hr_st    [time_i, 1] +tab+$ 
                          red_chi_sq_1hr_st[time_i]+tab+$
                          uncert_1hr_st    [time_i]+tab+$
                          string(carr_rot_1hr  [time_i], format = '(F10.4)')+tab+$
                          string(lon_carr_1hr  [time_i], format = '(F10.4)')+tab+$
                          string(helio_dist_1hr[time_i], format = '(E12.5)')+tab+$
                          string(lon_hee_1hr   [time_i], format = '(F10.4)')+tab+$
                          string(lat_hee_1hr   [time_i], format = '(F10.4)')+tab+$
                          string(lon_heeq_1hr  [time_i], format = '(F10.4)')+tab+$
                          string(lat_heeq_1hr  [time_i], format = '(F10.4)')+tab+$
                          string(lon_hci_1hr   [time_i], format = '(F10.4)')+tab+$
                          string(lat_hci_1hr   [time_i], format = '(F10.4)') 
            time_i = time_i + 1
        ENDELSE
    ENDELSE 
    printf, out_lun, temp_string
ENDFOR
END


PRO create_ra_mom_cdfs, start_time, stop_time, gse = gse, sat = sat, key_test = key_test, roll_test = roll_test, swz_test = swz_test, temp_test = temp_test
compile_opt strictarrsubs
tab = string(9B)
tic
IF keyword_set(gse) EQ 1 THEN want_gse = 1 ELSE want_gse = 0
out_a_dir_base = '/data1/' ; maui
out_b_dir_base = '/data2/' ; maui
ascii_out_dir_base='/nfs/oahudata2/Ascii/Level2/1DMax/Non_valid/' ; maui		
table_location = '~/splat/Tables/' ; maui
;out_a_dir_base = '/home/lorna/splat/splat_user/Cribs/Temp/' ; fiji
;out_b_dir_base = '/home/lorna/splat/splat_user/Cribs/Temp/' ; fiji
;ascii_out_dir_base='/data1/Ascii/Level2/1DMax/Non_valid/' ; fiji		
;table_location = '/nfs/mauidata1/Software/SPLAT_latest/Tables/' ; fiji
;out_a_dir_base = '/mauidata1/' ; daisy
;out_a_dir_base = '~/Temp/Kristin/' ; daisy
;out_b_dir_base = '/mauidata2/' ; daisy
;ascii_out_dir_base='~/Temp/Kristin/' ; daisy		
;table_location = '~/splat/Tables/' ; daisy
;out_a_dir_base = '~/Temp/'

IF keyword_set(key_test) OR keyword_set(roll_test) OR keyword_set(swz_test) OR keyword_set(temp_test) THEN BEGIN 
    out_a_dir_base = '/data2/TestDir/'  ; maui
    out_b_dir_base = '/data2/TestDir/'  ; maui
    ascii_out_dir_base = '/data2/TestDir/'
ENDIF 
version = 'V05'

;start_time = '2008-01-01/00:00:00'
;stop_time  = '2008-01-02/00:00:00'

IF keyword_set(sat) THEN sats = [sat] ELSE sats = ['A', 'B']
;sats = ['B']
;sats = ['A']

;bad_sat = ['A', 'B']
;bad_start_time = ['2008-05-10/14:56:09', '2007-04-05/00:00:00']
;bad_stop_time = ['2008-05-13/19:14:09', '2007-04-07/00:00:00']
bad_sat        = ['A', 'B']
bad_start_time = ['2008-05-10/14:56:09', $
                  '2014-09-01/14:30:00']
bad_stop_time  = ['2008-05-13/19:14:09', $
                  '2014-09-08/16:00:00']
bad_prods      = [-1, 0] ; -1:all, 0=den

ts = time_struct(start_time)
te = time_struct(stop_time)

nofdays = te.daynum - ts.daynum

set_com
set_com_mon
set_com_high_res
FOR ii = 0, n_elements(sats)-1 DO BEGIN 
    ts = time_double(start_time)
    sat = sats[ii]
    ;skeleton_name = '~/Lorna/Level2/SKT/PLA_L2_1DMAX_00000000_'+version ; daisy
    ;skeleton_name       = '~/Level2/SKT/ST'+sat+'_PLA_L2_Protons_1DMax_1min_00000000_' +version ; maui
    skeleton_name       = '~/Level2/SKT/ST'+sat+'_L2_PLA_1DMax_1min_00000000_' +version ; maui
    skeleton_name_10min = '~/Level2/SKT/ST'+sat+'_L2_PLA_1DMax_10min_00000000_'+version ; maui
    skeleton_name_1hr   = '~/Level2/SKT/ST'+sat+'_L2_PLA_1DMax_1hr_00000000_'  +version ; maui
    FOR jj = 0, nofdays-1 DO BEGIN

        timespan, ts, 1, /DAYS  ; SECONDS, MINUTES, HOURS, DAYS (DEFAULT)
        ts_struct = time_struct(ts)
        time_string = time_string(ts)
        get_timespan, tt
        IF sat EQ 'A' THEN out_dir_base = out_a_dir_base ELSE out_dir_base = out_b_dir_base

        ;; get ra data
        if (ts_struct.year gt 2020) or ((ts_struct.year eq 2020) and (ts_struct.month ge 4)) then begin
           ;; 500 came from what Toni added for high counts
           ;; 2.73 came from comparing OBMoments to Toni's 6/29/2020
           pl_calc_proton_bulk_from_crib, sat, bkgd, time_string, 1, table_location, /INST_COORD, pac_eff = (500 * 2.73)
        endif else begin
           pl_calc_proton_bulk_from_crib, sat, bkgd, time_string, 1, table_location, /INST_COORD
        endelse 
        tplot_names, names = names

        ; calculate E/W angles
        IF want_gse EQ 1 THEN BEGIN 
           pl_calc_ew_proton, sat, /gse 
        ENDIF ELSE IF keyword_set(key_test)  THEN BEGIN 
           pl_calc_ew_proton, sat, /key_eff 
        ENDIF ELSE IF keyword_set(roll_test) THEN BEGIN 
           pl_calc_ew_proton_roll, sat 
        ENDIF ELSE IF keyword_set(swz_test)  THEN BEGIN 
           pl_calc_ew_proton_swz, sat 
        ENDIF ELSE pl_calc_ew_proton, sat

        have_data       = 0
        have_data_10min = 0
        have_data_1hr   = 0

        IF n_elements(names) GT 0 THEN BEGIN ; have tplot variables
            IF size(names, /type) EQ 7 AND names[0] NE '' THEN BEGIN ; have tplot variables
                get_data, '1_keV_time',                      ra_t_1kev,                   epoch_1kev
                get_data, 'density',                         ra_t_den,                    den
                get_data, 'density_10min',                   ra_t_den_10min,              den_10min
                get_data, 'density_1hr',                     ra_t_den_1hr,                den_1hr
                get_data, 'speed',                           ra_t_speed,                  speed
                get_data, 'speed_10min',                     ra_t_speed_10min,            speed_10min
                get_data, 'speed_1hr',                       ra_t_speed_1hr,              speed_1hr
                get_data, 'temperature',                     ra_t_temp,                   temp
                get_data, 'temperature_10min',               ra_t_temp_10min,             temp_10min
                get_data, 'temperature_1hr',                 ra_t_temp_1hr,               temp_1hr
                get_data, 'v_thermal',                       ra_t_v_therm,                v_therm
                get_data, 'v_thermal_10min',                 ra_t_v_therm_10min,          v_therm_10min
                get_data, 'v_thermal_1hr',                   ra_t_v_therm_1hr,            v_therm_1hr
                get_data, 'north_south',                     ra_t_ns_inst,                ns_inst 
                get_data, 'north_south_10min',               ra_t_ns_inst_10min,          ns_inst_10min 
                get_data, 'north_south_1hr',                 ra_t_ns_inst_1hr,            ns_inst_1hr 
                get_data, 'error_code',                      ra_t_error,                  error
                get_data, 'error_code_10min',                ra_t_error_10min,            error_10min
                get_data, 'error_code_1hr',                  ra_t_error_1hr,              error_1hr
                get_data, 'caution_code',                    ra_t_caution,                caution
                get_data, 'caution_code_10min',              ra_t_caution_10min,          caution_10min
                get_data, 'caution_code_1hr',                ra_t_caution_1hr,            caution_1hr
                get_data, 'attitude_flag',                   ra_t_attitude,               attitude
                get_data, 'attitude_flag_10min',             ra_t_attitude_10min,         attitude_10min
                get_data, 'attitude_flag_1hr',               ra_t_attitude_1hr,           attitude_1hr
                get_data, 'reduced_chi_squared',             ra_t_red_chi_sq,             red_chi_sq
                get_data, 'reduced_chi_squared_10min',       ra_t_red_chi_sq_10min,       red_chi_sq_10min
                get_data, 'reduced_chi_squared_1hr',         ra_t_red_chi_sq_1hr,         red_chi_sq_1hr
                get_data, 'cycles_10min',                    ra_t_cycles_10min,           cycles_10min
                get_data, 'cycles_1hr',                      ra_t_cycles_1hr,             cycles_1hr
                get_data, 'cycles_10min_v',                  ra_t_cycles_speed_10min,     cycles_speed_10min
                get_data, 'cycles_1hr_v',                    ra_t_cycles_speed_1hr,       cycles_speed_1hr
                get_data, 'cycles_10min_ns',                 ra_t_cycles_ns_inst_10min,   cycles_ns_inst_10min
                get_data, 'cycles_1hr_ns',                   ra_t_cycles_ns_inst_1hr,     cycles_ns_inst_1hr
                get_data, 'carrington_rotation',             ra_t_carr_rot,               carr_rot
                get_data, 'carrington_rotation_10min',       ra_t_carr_rot_10min,         carr_rot_10min
                get_data, 'carrington_rotation_1hr',         ra_t_carr_rot_1hr,           carr_rot_1hr
                get_data, 'carrington_longitude',            ra_t_lon_carr,               lon_carr
                get_data, 'carrington_longitude_10min',      ra_t_lon_carr_10min,         lon_carr_10min
                get_data, 'carrington_longitude_1hr',        ra_t_lon_carr_1hr,           lon_carr_1hr
                get_data, 'heliocentric_distance',           ra_t_helio_dist,             helio_dist
                get_data, 'heliocentric_distance_10min',     ra_t_helio_dist_10min,       helio_dist_10min
                get_data, 'heliocentric_distance_1hr',       ra_t_helio_dist_1hr,         helio_dist_1hr
                get_data, 'spacecraft_longitude_HEE',        ra_t_lon_hee,                lon_hee
                get_data, 'spacecraft_longitude_HEE_10min',  ra_t_lon_hee_10min,          lon_hee_10min
                get_data, 'spacecraft_longitude_HEE_1hr',    ra_t_lon_hee_1hr,            lon_hee_1hr
                get_data, 'spacecraft_latitude_HEE',         ra_t_lat_hee,                lat_hee
                get_data, 'spacecraft_latitude_HEE_10min',   ra_t_lat_hee_10min,          lat_hee_10min
                get_data, 'spacecraft_latitude_HEE_1hr',     ra_t_lat_hee_1hr,            lat_hee_1hr
                get_data, 'spacecraft_longitude_HEEQ',       ra_t_lon_heeq,               lon_heeq
                get_data, 'spacecraft_longitude_HEEQ_10min', ra_t_lon_heeq_10min,         lon_heeq_10min
                get_data, 'spacecraft_longitude_HEEQ_1hr',   ra_t_lon_heeq_1hr,           lon_heeq_1hr
                get_data, 'spacecraft_latitude_HEEQ',        ra_t_lat_heeq,               lat_heeq
                get_data, 'spacecraft_latitude_HEEQ_10min',  ra_t_lat_heeq_10min,         lat_heeq_10min
                get_data, 'spacecraft_latitude_HEEQ_1hr',    ra_t_lat_heeq_1hr,           lat_heeq_1hr
                get_data, 'spacecraft_longitude_HCI',        ra_t_lon_hci,                lon_hci
                get_data, 'spacecraft_longitude_HCI_10min',  ra_t_lon_hci_10min,          lon_hci_10min
                get_data, 'spacecraft_longitude_HCI_1hr',    ra_t_lon_hci_1hr,            lon_hci_1hr
                get_data, 'spacecraft_latitude_HCI',         ra_t_lat_hci,                lat_hci
                get_data, 'spacecraft_latitude_HCI_10min',   ra_t_lat_hci_10min,          lat_hci_10min
                get_data, 'spacecraft_latitude_HCI_1hr',     ra_t_lat_hci_1hr,            lat_hci_1hr
                get_data, 'stereo_ew_inst',                  ew_t_ew_inst,                ew_inst
                get_data, 'stereo_ew_inst_10min',            ew_t_ew_inst_10min,          ew_inst_10min
                get_data, 'stereo_ew_inst_1hr',              ew_t_ew_inst_1hr,            ew_inst_1hr
                get_data, 'stereo_vr_hertn',                 ew_t_vr_hertn,               vr_hertn
                get_data, 'stereo_vr_hertn_10min',           ew_t_vr_hertn_10min,         vr_hertn_10min
                get_data, 'stereo_vr_hertn_1hr',             ew_t_vr_hertn_1hr,           vr_hertn_1hr
                get_data, 'stereo_vt_hertn',                 ew_t_vt_hertn,               vt_hertn
                get_data, 'stereo_vt_hertn_10min',           ew_t_vt_hertn_10min,         vt_hertn_10min
                get_data, 'stereo_vt_hertn_1hr',             ew_t_vt_hertn_1hr,           vt_hertn_1hr
                get_data, 'stereo_vn_hertn',                 ew_t_vn_hertn,               vn_hertn
                get_data, 'stereo_vn_hertn_10min',           ew_t_vn_hertn_10min,         vn_hertn_10min
                get_data, 'stereo_vn_hertn_1hr',             ew_t_vn_hertn_1hr,           vn_hertn_1hr
                get_data, 'stereo_ew_hertn',                 ew_t_ew_hertn,               ew_hertn
                get_data, 'stereo_ew_hertn_10min',           ew_t_ew_hertn_10min,         ew_hertn_10min
                get_data, 'stereo_ew_hertn_1hr',             ew_t_ew_hertn_1hr,           ew_hertn_1hr
                get_data, 'stereo_ns_hertn',                 ew_t_ns_hertn,               ns_hertn
                get_data, 'stereo_ns_hertn_10min',           ew_t_ns_hertn_10min,         ns_hertn_10min
                get_data, 'stereo_ns_hertn_1hr',             ew_t_ns_hertn_1hr,           ns_hertn_1hr
                get_data, 'stereo_vr_rtn',                   ew_t_vr_rtn,                 vr_rtn
                get_data, 'stereo_vr_rtn_10min',             ew_t_vr_rtn_10min,           vr_rtn_10min
                get_data, 'stereo_vr_rtn_1hr',               ew_t_vr_rtn_1hr,             vr_rtn_1hr
                get_data, 'stereo_vt_rtn',                   ew_t_vt_rtn,                 vt_rtn
                get_data, 'stereo_vt_rtn_10min',             ew_t_vt_rtn_10min,           vt_rtn_10min
                get_data, 'stereo_vt_rtn_1hr',               ew_t_vt_rtn_1hr,             vt_rtn_1hr
                get_data, 'stereo_vn_rtn',                   ew_t_vn_rtn,                 vn_rtn
                get_data, 'stereo_vn_rtn_10min',             ew_t_vn_rtn_10min,           vn_rtn_10min
                get_data, 'stereo_vn_rtn_1hr',               ew_t_vn_rtn_1hr,             vn_rtn_1hr
                get_data, 'stereo_ew_rtn',                   ew_t_ew_rtn,                 ew_rtn
                get_data, 'stereo_ew_rtn_10min',             ew_t_ew_rtn_10min,           ew_rtn_10min
                get_data, 'stereo_ew_rtn_1hr',               ew_t_ew_rtn_1hr,             ew_rtn_1hr
                get_data, 'stereo_ns_rtn',                   ew_t_ns_rtn,                 ns_rtn
                get_data, 'stereo_ns_rtn_10min',             ew_t_ns_rtn_10min,           ns_rtn_10min
                get_data, 'stereo_ns_rtn_1hr',               ew_t_ns_rtn_1hr,             ns_rtn_1hr
                get_data, 'ew_source_flag',                  ew_t_ew_source_flag,         ew_source_flag
                get_data, 'ew_source_flag_10min',            ew_t_ew_source_flag_10min,   ew_source_flag_10min
                get_data, 'ew_source_flag_1hr',              ew_t_ew_source_flag_1hr,     ew_source_flag_1hr
                get_data, 'num_in_ave_10min',                ew_t_cycles_ew_10min,        cycles_ew_10min
                get_data, 'num_in_ave_1hr',                  ew_t_cycles_ew_1hr,          cycles_ew_1hr
                get_data, 'uncertainty',                     ew_t_uncert,                 uncert
                get_data, 'uncertainty_10min',               ew_t_uncert_10min,           uncert_10min
                get_data, 'uncertainty_1hr',                 ew_t_uncert_1hr,             uncert_1hr
                IF want_gse EQ 1 THEN BEGIN
                    get_data, 'stereo_vr_gse',               ew_t_vr_gse,                 vr_gse
                    get_data, 'stereo_vt_gse',               ew_t_vt_gse,                 vt_gse
                    get_data, 'stereo_vn_gse',               ew_t_vn_gse,                 vn_gse
                    get_data, 'stereo_ew_gse',               ew_t_ew_gse,                 ew_gse
                    get_data, 'stereo_ns_gse',               ew_t_ns_gse,                 ns_gse
                ENDIF 
                finite_t_1kev                 = where(finite(ra_t_1kev)                 EQ 1)
                finite_t_den                  = where(finite(ra_t_den)                  EQ 1)
                finite_t_den_10min            = where(finite(ra_t_den_10min)            EQ 1)
                finite_t_den_1hr              = where(finite(ra_t_den_1hr)              EQ 1)
                finite_t_speed                = where(finite(ra_t_speed)                EQ 1)
                finite_t_speed_10min          = where(finite(ra_t_speed_10min)          EQ 1)
                finite_t_speed_1hr            = where(finite(ra_t_speed_1hr)            EQ 1)
                finite_t_temp                 = where(finite(ra_t_temp)                 EQ 1)
                finite_t_temp_10min           = where(finite(ra_t_temp_10min)           EQ 1)
                finite_t_temp_1hr             = where(finite(ra_t_temp_1hr)             EQ 1)
                finite_t_ns_inst              = where(finite(ra_t_ns_inst)              EQ 1)
                finite_t_ns_inst_10min        = where(finite(ra_t_ns_inst_10min)        EQ 1)
                finite_t_ns_inst_1hr          = where(finite(ra_t_ns_inst_1hr)          EQ 1)
                finite_t_v_therm              = where(finite(ra_t_v_therm)              EQ 1)
                finite_t_v_therm_10min        = where(finite(ra_t_v_therm_10min)        EQ 1)
                finite_t_v_therm_1hr          = where(finite(ra_t_v_therm_1hr)          EQ 1)
                finite_t_error                = where(finite(ra_t_error)                EQ 1)
                finite_t_error_10min          = where(finite(ra_t_error_10min)          EQ 1)
                finite_t_error_1hr            = where(finite(ra_t_error_1hr)            EQ 1)
                finite_t_caution              = where(finite(ra_t_caution)              EQ 1)
                finite_t_caution_10min        = where(finite(ra_t_caution_10min)        EQ 1)
                finite_t_caution_1hr          = where(finite(ra_t_caution_1hr)          EQ 1)
                finite_t_attitude             = where(finite(ra_t_attitude)             EQ 1)
                finite_t_attitude_10min       = where(finite(ra_t_attitude_10min)       EQ 1)
                finite_t_attitude_1hr         = where(finite(ra_t_attitude_1hr)         EQ 1)
                finite_t_red_chi_sq           = where(finite(ra_t_red_chi_sq)           EQ 1)
                finite_t_red_chi_sq_10min     = where(finite(ra_t_red_chi_sq_10min)     EQ 1)
                finite_t_red_chi_sq_1hr       = where(finite(ra_t_red_chi_sq_1hr)       EQ 1)
                finite_t_cycles_10min         = where(finite(ra_t_cycles_10min)         EQ 1)
                finite_t_cycles_1hr           = where(finite(ra_t_cycles_1hr)           EQ 1)
                finite_t_cycles_speed_10min   = where(finite(ra_t_cycles_speed_10min)   EQ 1)
                finite_t_cycles_speed_1hr     = where(finite(ra_t_cycles_speed_1hr)     EQ 1)
                finite_t_cycles_ns_inst_10min = where(finite(ra_t_cycles_ns_inst_10min) EQ 1)
                finite_t_cycles_ns_inst_1hr   = where(finite(ra_t_cycles_ns_inst_1hr)   EQ 1)
                finite_t_carr_rot             = where(finite(ra_t_carr_rot)             EQ 1)
                finite_t_carr_rot_10min       = where(finite(ra_t_carr_rot_10min)       EQ 1)
                finite_t_carr_rot_1hr         = where(finite(ra_t_carr_rot_1hr)         EQ 1)
                finite_t_lon_carr             = where(finite(ra_t_lon_carr)             EQ 1)
                finite_t_lon_carr_10min       = where(finite(ra_t_lon_carr_10min)       EQ 1)
                finite_t_lon_carr_1hr         = where(finite(ra_t_lon_carr_1hr)         EQ 1)
                finite_t_helio_dist           = where(finite(ra_t_helio_dist)           EQ 1)
                finite_t_helio_dist_10min     = where(finite(ra_t_helio_dist_10min)     EQ 1)
                finite_t_helio_dist_1hr       = where(finite(ra_t_helio_dist_1hr)       EQ 1)
                finite_t_lon_hee              = where(finite(ra_t_lon_hee)              EQ 1)
                finite_t_lon_hee_10min        = where(finite(ra_t_lon_hee_10min)        EQ 1)
                finite_t_lon_hee_1hr          = where(finite(ra_t_lon_hee_1hr)          EQ 1)
                finite_t_lat_hee              = where(finite(ra_t_lat_hee)              EQ 1)
                finite_t_lat_hee_10min        = where(finite(ra_t_lat_hee_10min)        EQ 1)
                finite_t_lat_hee_1hr          = where(finite(ra_t_lat_hee_1hr)          EQ 1)
                finite_t_lon_heeq             = where(finite(ra_t_lon_heeq)             EQ 1)
                finite_t_lon_heeq_10min       = where(finite(ra_t_lon_heeq_10min)       EQ 1)
                finite_t_lon_heeq_1hr         = where(finite(ra_t_lon_heeq_1hr)         EQ 1)
                finite_t_lat_heeq             = where(finite(ra_t_lat_heeq)             EQ 1)
                finite_t_lat_heeq_10min       = where(finite(ra_t_lat_heeq_10min)       EQ 1)
                finite_t_lat_heeq_1hr         = where(finite(ra_t_lat_heeq_1hr)         EQ 1)
                finite_t_lon_hci              = where(finite(ra_t_lon_hci)              EQ 1)
                finite_t_lon_hci_10min        = where(finite(ra_t_lon_hci_10min)        EQ 1)
                finite_t_lon_hci_1hr          = where(finite(ra_t_lon_hci_1hr)          EQ 1)
                finite_t_lat_hci              = where(finite(ra_t_lat_hci)              EQ 1)
                finite_t_lat_hci_10min        = where(finite(ra_t_lat_hci_10min)        EQ 1)
                finite_t_lat_hci_1hr          = where(finite(ra_t_lat_hci_1hr)          EQ 1)
                finite_t_ew_inst              = where(finite(ew_t_ew_inst)              EQ 1)
                finite_t_ew_inst_10min        = where(finite(ew_t_ew_inst_10min)        EQ 1)
                finite_t_ew_inst_1hr          = where(finite(ew_t_ew_inst_1hr)          EQ 1)
                finite_t_vr_hertn             = where(finite(ew_t_vr_hertn)             EQ 1)
                finite_t_vr_hertn_10min       = where(finite(ew_t_vr_hertn_10min)       EQ 1)
                finite_t_vr_hertn_1hr         = where(finite(ew_t_vr_hertn_1hr)         EQ 1)
                finite_t_vt_hertn             = where(finite(ew_t_vt_hertn)             EQ 1)
                finite_t_vt_hertn_10min       = where(finite(ew_t_vt_hertn_10min)       EQ 1)
                finite_t_vt_hertn_1hr         = where(finite(ew_t_vt_hertn_1hr)         EQ 1)
                finite_t_vn_hertn             = where(finite(ew_t_vn_hertn)             EQ 1)
                finite_t_vn_hertn_10min       = where(finite(ew_t_vn_hertn_10min)       EQ 1)
                finite_t_vn_hertn_1hr         = where(finite(ew_t_vn_hertn_1hr)         EQ 1)
                finite_t_ew_hertn             = where(finite(ew_t_ew_hertn)             EQ 1)
                finite_t_ew_hertn_10min       = where(finite(ew_t_ew_hertn_10min)       EQ 1)
                finite_t_ew_hertn_1hr         = where(finite(ew_t_ew_hertn_1hr)         EQ 1)
                finite_t_ns_hertn             = where(finite(ew_t_ns_hertn)             EQ 1)
                finite_t_ns_hertn_10min       = where(finite(ew_t_ns_hertn_10min)       EQ 1)
                finite_t_ns_hertn_1hr         = where(finite(ew_t_ns_hertn_1hr)         EQ 1)
                finite_t_vr_rtn               = where(finite(ew_t_vr_rtn)               EQ 1)
                finite_t_vr_rtn_10min         = where(finite(ew_t_vr_rtn_10min)         EQ 1)
                finite_t_vr_rtn_1hr           = where(finite(ew_t_vr_rtn_1hr)           EQ 1)
                finite_t_vt_rtn               = where(finite(ew_t_vt_rtn)               EQ 1)
                finite_t_vt_rtn_10min         = where(finite(ew_t_vt_rtn_10min)         EQ 1)
                finite_t_vt_rtn_1hr           = where(finite(ew_t_vt_rtn_1hr)           EQ 1)
                finite_t_vn_rtn               = where(finite(ew_t_vn_rtn)               EQ 1)
                finite_t_vn_rtn_10min         = where(finite(ew_t_vn_rtn_10min)         EQ 1)
                finite_t_vn_rtn_1hr           = where(finite(ew_t_vn_rtn_1hr)           EQ 1)
                finite_t_ew_rtn               = where(finite(ew_t_ew_rtn)               EQ 1)
                finite_t_ew_rtn_10min         = where(finite(ew_t_ew_rtn_10min)         EQ 1)
                finite_t_ew_rtn_1hr           = where(finite(ew_t_ew_rtn_1hr)           EQ 1)
                finite_t_ns_rtn               = where(finite(ew_t_ns_rtn)               EQ 1)
                finite_t_ns_rtn_10min         = where(finite(ew_t_ns_rtn_10min)         EQ 1)
                finite_t_ns_rtn_1hr           = where(finite(ew_t_ns_rtn_1hr)           EQ 1)
                finite_t_ew_source_flag       = where(finite(ew_t_ew_source_flag)       EQ 1)
                finite_t_ew_source_flag_10min = where(finite(ew_t_ew_source_flag_10min) EQ 1)
                finite_t_ew_source_flag_1hr   = where(finite(ew_t_ew_source_flag_1hr)   EQ 1)
                finite_t_uncert               = where(finite(ew_t_uncert)               EQ 1)
                finite_t_uncert_10min         = where(finite(ew_t_uncert_10min)         EQ 1)
                finite_t_uncert_1hr           = where(finite(ew_t_uncert_1hr)           EQ 1)
                finite_t_cycles_ew_10min      = where(finite(ew_t_cycles_ew_10min)      EQ 1)
                finite_t_cycles_ew_1hr        = where(finite(ew_t_cycles_ew_1hr)        EQ 1)
                IF want_gse EQ 1 THEN BEGIN
                    finite_t_vr_gse           = where(finite(ew_t_vr_gse)               EQ 1)
                    finite_t_vt_gse           = where(finite(ew_t_vt_gse)               EQ 1)
                    finite_t_vn_gse           = where(finite(ew_t_vn_gse)               EQ 1)
                    finite_t_ew_gse           = where(finite(ew_t_ew_gse)               EQ 1)
                    finite_t_ns_gse           = where(finite(ew_t_ns_gse)               EQ 1)
                ENDIF 
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_1kev                [finite_t_1kev])                 NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_speed               [finite_t_speed])                NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_speed_10min         [finite_t_speed_10min])          NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_speed_1hr           [finite_t_speed_1hr])            NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_temp                [finite_t_temp])                 NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_temp_10min          [finite_t_temp_10min])           NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_temp_1hr            [finite_t_temp_1hr])             NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_ns_inst             [finite_t_ns_inst])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_ns_inst_10min       [finite_t_ns_inst_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_ns_inst_1hr         [finite_t_ns_inst_1hr])          NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_v_therm             [finite_t_v_therm])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_v_therm_10min       [finite_t_v_therm_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_v_therm_1hr         [finite_t_v_therm_1hr])          NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_error               [finite_t_error])                NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_error_10min         [finite_t_error_10min])          NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_error_1hr           [finite_t_error_1hr])            NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_caution             [finite_t_caution])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_caution_10min       [finite_t_caution_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_caution_1hr         [finite_t_caution_1hr])          NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_attitude            [finite_t_attitude])             NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_attitude_10min      [finite_t_attitude_10min])       NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_attitude_1hr        [finite_t_attitude_1hr])         NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_red_chi_sq          [finite_t_red_chi_sq])           NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_red_chi_sq_10min    [finite_t_red_chi_sq_10min])     NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_red_chi_sq_1hr      [finite_t_red_chi_sq_1hr])       NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_cycles_10min        [finite_t_cycles_10min])         NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_cycles_1hr          [finite_t_cycles_1hr])           NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_cycles_speed_10min  [finite_t_cycles_speed_10min])   NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_cycles_speed_1hr    [finite_t_cycles_speed_1hr])     NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_cycles_ns_inst_10min[finite_t_cycles_ns_inst_10min]) NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_cycles_ns_inst_1hr  [finite_t_cycles_ns_inst_1hr])   NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_carr_rot            [finite_t_carr_rot])             NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_carr_rot_10min      [finite_t_carr_rot_10min])       NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_carr_rot_1hr        [finite_t_carr_rot_1hr])         NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lon_carr            [finite_t_lon_carr])             NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lon_carr_10min      [finite_t_lon_carr_10min])       NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lon_carr_1hr        [finite_t_lon_carr_1hr])         NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_helio_dist          [finite_t_helio_dist])           NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_helio_dist_10min    [finite_t_helio_dist_10min])     NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_helio_dist_1hr      [finite_t_helio_dist_1hr])       NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lon_hee             [finite_t_lon_hee])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lon_hee_10min       [finite_t_lon_hee_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lon_hee_1hr         [finite_t_lon_hee_1hr])          NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lat_hee             [finite_t_lat_hee])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lat_hee_10min       [finite_t_lat_hee_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lat_hee_1hr         [finite_t_lat_hee_1hr])          NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lon_heeq            [finite_t_lon_heeq])             NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lon_heeq_10min      [finite_t_lon_heeq_10min])       NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lon_heeq_1hr        [finite_t_lon_heeq_1hr])         NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lat_heeq            [finite_t_lat_heeq])             NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lat_heeq_10min      [finite_t_lat_heeq_10min])       NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lat_heeq_1hr        [finite_t_lat_heeq_1hr])         NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lon_hci             [finite_t_lon_hci])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lon_hci_10min       [finite_t_lon_hci_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lon_hci_1hr         [finite_t_lon_hci_1hr])          NE 1 THEN stop
                IF array_equal(ra_t_den      [finite_t_den],       ra_t_lat_hci             [finite_t_lat_hci])              NE 1 THEN stop
                IF array_equal(ra_t_den_10min[finite_t_den_10min], ra_t_lat_hci_10min       [finite_t_lat_hci_10min])        NE 1 THEN stop
                IF array_equal(ra_t_den_1hr  [finite_t_den_1hr],   ra_t_lat_hci_1hr         [finite_t_lat_hci_1hr])          NE 1 THEN stop
                IF want_gse EQ 1 THEN BEGIN
                    IF array_equal(ra_t_den  [finite_t_den],       ew_t_vr_gse              [finite_t_vr_gse])               NE 1 THEN stop
                    IF array_equal(ra_t_den  [finite_t_den],       ew_t_vt_gse              [finite_t_vt_gse])               NE 1 THEN stop
                    IF array_equal(ra_t_den  [finite_t_den],       ew_t_vn_gse              [finite_t_vn_gse])               NE 1 THEN stop
                    IF array_equal(ra_t_den  [finite_t_den],       ew_t_ew_gse              [finite_t_ew_gse])               NE 1 THEN stop
                    IF array_equal(ra_t_den  [finite_t_den],       ew_t_ns_gse              [finite_t_ns_gse])               NE 1 THEN stop
                ENDIF 
                empty_1min  = 0
                empty_10min = 0
                empty_1hr   = 0
                IF n_elements(ra_t_den) EQ 1 THEN BEGIN  ; there isn't any valid data
                    IF ra_t_den[0] EQ 0 OR $
                      (finite(den    ) EQ 0 AND $
                       finite(speed  ) EQ 0 AND $
                       finite(temp   ) EQ 0 AND $
                       finite(v_therm) EQ 0 AND $
                       finite(ns_inst) EQ 0 ) THEN BEGIN 
                        empty_1min = 1
                        finite_ew_inst        = where(finite(ew_inst)        EQ 1, good_ew_inst)
                        finite_vr_hertn       = where(finite(vr_hertn)       EQ 1, good_vr_hertn)
                        finite_vt_hertn       = where(finite(vt_hertn)       EQ 1, good_vt_hertn)
                        finite_vn_hertn       = where(finite(vn_hertn)       EQ 1, good_vn_hertn)
                        finite_ew_hertn       = where(finite(ew_hertn)       EQ 1, good_ew_hertn)
                        finite_ns_hertn       = where(finite(ns_hertn)       EQ 1, good_ns_hertn)
                        finite_vr_rtn         = where(finite(vr_rtn)         EQ 1, good_vr_rtn)
                        finite_vt_rtn         = where(finite(vt_rtn)         EQ 1, good_vt_rtn)
                        finite_vn_rtn         = where(finite(vn_rtn)         EQ 1, good_vn_rtn)
                        finite_ew_rtn         = where(finite(ew_rtn)         EQ 1, good_ew_rtn)
                        finite_ns_rtn         = where(finite(ns_rtn)         EQ 1, good_ns_rtn)
                        finite_ew_source_flag = where(ew_source_flag         GE 0, good_ew_source_flag)
                        finite_uncert         = where(finite(uncert)         EQ 1, good_uncert)
                        IF good_ew_inst        NE 0 AND ew_t_ew_inst        NE 0 THEN stop
                        IF good_vr_hertn       NE 0 AND ew_t_vr_hertn       NE 0 THEN stop
                        IF good_vt_hertn       NE 0 AND ew_t_vt_hertn       NE 0 THEN stop
                        IF good_vn_hertn       NE 0 AND ew_t_vn_hertn       NE 0 THEN stop
                        IF good_ew_hertn       NE 0 AND ew_t_ew_hertn       NE 0 THEN stop
                        IF good_ns_hertn       NE 0 AND ew_t_ns_hertn       NE 0 THEN stop
                        IF good_vr_rtn         NE 0 AND ew_t_vr_rtn         NE 0 THEN stop
                        IF good_vt_rtn         NE 0 AND ew_t_vt_rtn         NE 0 THEN stop
                        IF good_vn_rtn         NE 0 AND ew_t_vn_rtn         NE 0 THEN stop
                        IF good_ew_rtn         NE 0 AND ew_t_ew_rtn         NE 0 THEN stop
                        IF good_ns_rtn         NE 0 AND ew_t_ns_rtn         NE 0 THEN stop
                        IF good_ew_source_flag NE 0 AND ew_t_ew_source_flag NE 0 THEN stop
                        IF good_uncert         NE 0 AND ew_t_uncert         NE 0 THEN stop
                    ENDIF 
                ENDIF 
                IF empty_1min EQ 0 THEN BEGIN 
                    IF n_elements(ew_inst) GT 1 THEN BEGIN ; have ew data
                        IF array_equal(ra_t_den[finite_t_den], ew_t_ew_inst       [finite_t_ew_inst ])       NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_vr_hertn      [finite_t_vr_hertn])       NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_vt_hertn      [finite_t_vt_hertn])       NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_vn_hertn      [finite_t_vn_hertn])       NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_ew_hertn      [finite_t_ew_hertn])       NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_ns_hertn      [finite_t_ns_hertn])       NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_vr_rtn        [finite_t_vr_rtn])         NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_vt_rtn        [finite_t_vt_rtn])         NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_vn_rtn        [finite_t_vn_rtn])         NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_ew_rtn        [finite_t_ew_rtn])         NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_ns_rtn        [finite_t_ns_rtn])         NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_ew_source_flag[finite_t_ew_source_flag]) NE 1 THEN stop
                        IF array_equal(ra_t_den[finite_t_den], ew_t_uncert        [finite_t_uncert])         NE 1 THEN stop
                    ENDIF ELSE BEGIN ; no ew data
                        IF ew_t_ew_inst        NE 0 THEN stop
                        IF ew_t_vr_hertn       NE 0 THEN stop
                        IF ew_t_vt_hertn       NE 0 THEN stop
                        IF ew_t_vn_hertn       NE 0 THEN stop
                        IF ew_t_ew_hertn       NE 0 THEN stop
                        IF ew_t_ns_hertn       NE 0 THEN stop
                        IF ew_t_vr_rtn         NE 0 THEN stop
                        IF ew_t_vt_rtn         NE 0 THEN stop
                        IF ew_t_vn_rtn         NE 0 THEN stop
                        IF ew_t_ew_rtn         NE 0 THEN stop
                        IF ew_t_ns_rtn         NE 0 THEN stop
                        IF ew_t_ew_source_flag NE 0 THEN stop
                        IF ew_t_uncert         NE 0 THEN stop
                        ew_inst        = fltarr(n_elements(ra_t_den))
                        vr_hertn       = fltarr(n_elements(ra_t_den))
                        vt_hertn       = fltarr(n_elements(ra_t_den))
                        vn_hertn       = fltarr(n_elements(ra_t_den))
                        ns_hertn       = fltarr(n_elements(ra_t_den))
                        ew_hertn       = fltarr(n_elements(ra_t_den))
                        vr_rtn         = fltarr(n_elements(ra_t_den))
                        vt_rtn         = fltarr(n_elements(ra_t_den))
                        vn_rtn         = fltarr(n_elements(ra_t_den))
                        ns_rtn         = fltarr(n_elements(ra_t_den))
                        ew_rtn         = fltarr(n_elements(ra_t_den))
                        uncert         = fltarr(n_elements(ra_t_den))
                        ew_source_flag = fltarr(n_elements(ra_t_den), 2)
                        ew_inst       [*] = !values.f_nan
                        vr_hertn      [*] = !values.f_nan
                        vt_hertn      [*] = !values.f_nan
                        vn_hertn      [*] = !values.f_nan
                        ns_hertn      [*] = !values.f_nan
                        ew_hertn      [*] = !values.f_nan
                        vr_rtn        [*] = !values.f_nan
                        vt_rtn        [*] = !values.f_nan
                        vn_rtn        [*] = !values.f_nan
                        ns_rtn        [*] = !values.f_nan
                        ew_rtn        [*] = !values.f_nan
                        uncert        [*] = !values.f_nan
                        ew_source_flag[*, *] = !values.f_nan
                    ENDELSE 
                ENDIF 
                IF n_elements(ra_t_den_10min) EQ 1 AND ra_t_den_10min[0] EQ 0 THEN BEGIN  ; there isn't any valid data
                    empty_10min = 1
                    finite_ew_inst_10min        = where(finite(ew_inst_10min)        EQ 1, good_ew_inst_10min)
                    finite_vr_hertn_10min       = where(finite(vr_hertn_10min)       EQ 1, good_vr_hertn_10min)
                    finite_vt_hertn_10min       = where(finite(vt_hertn_10min)       EQ 1, good_vt_hertn_10min)
                    finite_vn_hertn_10min       = where(finite(vn_hertn_10min)       EQ 1, good_vn_hertn_10min)
                    finite_ew_hertn_10min       = where(finite(ew_hertn_10min)       EQ 1, good_ew_hertn_10min)
                    finite_ns_hertn_10min       = where(finite(ns_hertn_10min)       EQ 1, good_ns_hertn_10min)
                    finite_vr_rtn_10min         = where(finite(vr_rtn_10min)         EQ 1, good_vr_rtn_10min)
                    finite_vt_rtn_10min         = where(finite(vt_rtn_10min)         EQ 1, good_vt_rtn_10min)
                    finite_vn_rtn_10min         = where(finite(vn_rtn_10min)         EQ 1, good_vn_rtn_10min)
                    finite_ew_rtn_10min         = where(finite(ew_rtn_10min)         EQ 1, good_ew_rtn_10min)
                    finite_ns_rtn_10min         = where(finite(ns_rtn_10min)         EQ 1, good_ns_rtn_10min)
                    finite_ew_source_flag_10min = where(ew_source_flag_10min         GE 0, good_ew_source_flag_10min)
                    finite_uncert_10min         = where(finite(uncert_10min)         EQ 1, good_uncert_10min)
                    IF good_ew_inst_10min        NE 0 AND ew_t_ew_inst_10min        NE 0 THEN stop
                    IF good_vr_hertn_10min       NE 0 AND ew_t_vr_hertn_10min       NE 0 THEN stop
                    IF good_vt_hertn_10min       NE 0 AND ew_t_vt_hertn_10min       NE 0 THEN stop
                    IF good_vn_hertn_10min       NE 0 AND ew_t_vn_hertn_10min       NE 0 THEN stop
                    IF good_ew_hertn_10min       NE 0 AND ew_t_ew_hertn_10min       NE 0 THEN stop
                    IF good_ns_hertn_10min       NE 0 AND ew_t_ns_hertn_10min       NE 0 THEN stop
                    IF good_vr_rtn_10min         NE 0 AND ew_t_vr_rtn_10min         NE 0 THEN stop
                    IF good_vt_rtn_10min         NE 0 AND ew_t_vt_rtn_10min         NE 0 THEN stop
                    IF good_vn_rtn_10min         NE 0 AND ew_t_vn_rtn_10min         NE 0 THEN stop
                    IF good_ew_rtn_10min         NE 0 AND ew_t_ew_rtn_10min         NE 0 THEN stop
                    IF good_ns_rtn_10min         NE 0 AND ew_t_ns_rtn_10min         NE 0 THEN stop
                    IF good_ew_source_flag_10min NE 0 AND ew_t_ew_source_flag_10min NE 0 THEN stop
                    IF good_uncert_10min         NE 0 AND ew_t_uncert_10min         NE 0 THEN stop
                ENDIF ELSE BEGIN  
                    ew_size = size(ew_inst_10min)
                    IF ew_size[0] GT 0 THEN BEGIN 
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_ew_inst_10min       [finite_t_ew_inst_10min])        NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_vr_hertn_10min      [finite_t_vr_hertn_10min])       NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_vt_hertn_10min      [finite_t_vt_hertn_10min])       NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_vn_hertn_10min      [finite_t_vn_hertn_10min])       NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_ew_hertn_10min      [finite_t_ew_hertn_10min])       NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_ns_hertn_10min      [finite_t_ns_hertn_10min])       NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_vr_rtn_10min        [finite_t_vr_rtn_10min])         NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_vt_rtn_10min        [finite_t_vt_rtn_10min])         NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_vn_rtn_10min        [finite_t_vn_rtn_10min])         NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_ew_rtn_10min        [finite_t_ew_rtn_10min])         NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_ns_rtn_10min        [finite_t_ns_rtn_10min])         NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_ew_source_flag_10min[finite_t_ew_source_flag_10min]) NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_cycles_ew_10min     [finite_t_cycles_ew_10min])      NE 1 THEN stop
                        IF array_equal(ra_t_den_10min[finite_t_den_10min], ew_t_uncert_10min        [finite_t_uncert_10min])         NE 1 THEN stop
                    ENDIF ELSE BEGIN 
                        IF ew_t_ew_inst_10min        NE 0 THEN stop
                        IF ew_t_vr_hertn_10min       NE 0 THEN stop
                        IF ew_t_vt_hertn_10min       NE 0 THEN stop
                        IF ew_t_vn_hertn_10min       NE 0 THEN stop
                        IF ew_t_ew_hertn_10min       NE 0 THEN stop
                        IF ew_t_ns_hertn_10min       NE 0 THEN stop
                        IF ew_t_vr_rtn_10min         NE 0 THEN stop
                        IF ew_t_vt_rtn_10min         NE 0 THEN stop
                        IF ew_t_vn_rtn_10min         NE 0 THEN stop
                        IF ew_t_ew_rtn_10min         NE 0 THEN stop
                        IF ew_t_ns_rtn_10min         NE 0 THEN stop
                        IF ew_t_ew_source_flag_10min NE 0 THEN stop
                        IF ew_t_uncert_10min         NE 0 THEN stop
                        ew_inst_10min        = fltarr(n_elements(ra_t_den_10min))
                        vr_hertn_10min       = fltarr(n_elements(ra_t_den_10min))
                        vt_hertn_10min       = fltarr(n_elements(ra_t_den_10min))
                        vn_hertn_10min       = fltarr(n_elements(ra_t_den_10min))
                        ns_hertn_10min       = fltarr(n_elements(ra_t_den_10min))
                        ew_hertn_10min       = fltarr(n_elements(ra_t_den_10min))
                        vr_rtn_10min         = fltarr(n_elements(ra_t_den_10min))
                        vt_rtn_10min         = fltarr(n_elements(ra_t_den_10min))
                        vn_rtn_10min         = fltarr(n_elements(ra_t_den_10min))
                        ns_rtn_10min         = fltarr(n_elements(ra_t_den_10min))
                        ew_rtn_10min         = fltarr(n_elements(ra_t_den_10min))
                        uncert_10min         = fltarr(n_elements(ra_t_den_10min))
                        cycles_ew_10min      = fltarr(n_elements(ra_t_den_10min))
                        ew_source_flag_10min = intarr(n_elements(ra_t_den_10min), 2)
                        ew_inst_10min       [*] = !values.f_nan
                        vr_hertn_10min      [*] = !values.f_nan
                        vt_hertn_10min      [*] = !values.f_nan
                        vn_hertn_10min      [*] = !values.f_nan
                        ns_hertn_10min      [*] = !values.f_nan
                        ew_hertn_10min      [*] = !values.f_nan
                        vr_rtn_10min        [*] = !values.f_nan
                        vt_rtn_10min        [*] = !values.f_nan
                        vn_rtn_10min        [*] = !values.f_nan
                        ns_rtn_10min        [*] = !values.f_nan
                        ew_rtn_10min        [*] = !values.f_nan
                        uncert_10min        [*] = !values.f_nan
                        cycles_ew_10min     [*] = 0
                        ew_source_flag_10min[*, *] = -1
                    ENDELSE 
                ENDELSE 
                IF n_elements(ra_t_den_1hr) EQ 1 AND ra_t_den_1hr[0] EQ 0 THEN BEGIN  ; there isn't any valid data
                    empty_1hr = 1
                    finite_ew_inst_1hr        = where(finite(ew_inst_1hr)        EQ 1, good_ew_inst_1hr)
                    finite_vr_hertn_1hr       = where(finite(vr_hertn_1hr)       EQ 1, good_vr_hertn_1hr)
                    finite_vt_hertn_1hr       = where(finite(vt_hertn_1hr)       EQ 1, good_vt_hertn_1hr)
                    finite_vn_hertn_1hr       = where(finite(vn_hertn_1hr)       EQ 1, good_vn_hertn_1hr)
                    finite_ew_hertn_1hr       = where(finite(ew_hertn_1hr)       EQ 1, good_ew_hertn_1hr)
                    finite_ns_hertn_1hr       = where(finite(ns_hertn_1hr)       EQ 1, good_ns_hertn_1hr)
                    finite_vr_rtn_1hr         = where(finite(vr_rtn_1hr)         EQ 1, good_vr_rtn_1hr)
                    finite_vt_rtn_1hr         = where(finite(vt_rtn_1hr)         EQ 1, good_vt_rtn_1hr)
                    finite_vn_rtn_1hr         = where(finite(vn_rtn_1hr)         EQ 1, good_vn_rtn_1hr)
                    finite_ew_rtn_1hr         = where(finite(ew_rtn_1hr)         EQ 1, good_ew_rtn_1hr)
                    finite_ns_rtn_1hr         = where(finite(ns_rtn_1hr)         EQ 1, good_ns_rtn_1hr)
                    finite_ew_source_flag_1hr = where(ew_source_flag_1hr         GE 0, good_ew_source_flag_1hr)
                    finite_uncert_1hr         = where(finite(uncert_1hr)         EQ 1, good_uncert_1hr)
                    IF good_ew_inst_1hr        NE 0 AND ew_t_ew_inst_1hr        NE 0 THEN stop
                    IF good_vr_hertn_1hr       NE 0 AND ew_t_vr_hertn_1hr       NE 0 THEN stop
                    IF good_vt_hertn_1hr       NE 0 AND ew_t_vt_hertn_1hr       NE 0 THEN stop
                    IF good_vn_hertn_1hr       NE 0 AND ew_t_vn_hertn_1hr       NE 0 THEN stop
                    IF good_ew_hertn_1hr       NE 0 AND ew_t_ew_hertn_1hr       NE 0 THEN stop
                    IF good_ns_hertn_1hr       NE 0 AND ew_t_ns_hertn_1hr       NE 0 THEN stop
                    IF good_vr_rtn_1hr         NE 0 AND ew_t_vr_rtn_1hr         NE 0 THEN stop
                    IF good_vt_rtn_1hr         NE 0 AND ew_t_vt_rtn_1hr         NE 0 THEN stop
                    IF good_vn_rtn_1hr         NE 0 AND ew_t_vn_rtn_1hr         NE 0 THEN stop
                    IF good_ew_rtn_1hr         NE 0 AND ew_t_ew_rtn_1hr         NE 0 THEN stop
                    IF good_ns_rtn_1hr         NE 0 AND ew_t_ns_rtn_1hr         NE 0 THEN stop
                    IF good_ew_source_flag_1hr NE 0 AND ew_t_ew_source_flag_1hr NE 0 THEN stop
                    IF good_uncert_1hr         NE 0 AND ew_t_uncert_1hr         NE 0 THEN stop
                ENDIF ELSE BEGIN 
                    ew_size = size(ew_inst_1hr)
                    IF ew_size[0] GT 0 THEN BEGIN 
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_ew_inst_1hr       [finite_t_ew_inst_1hr])        NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_vr_hertn_1hr      [finite_t_vr_hertn_1hr])       NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_vt_hertn_1hr      [finite_t_vt_hertn_1hr])       NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_vn_hertn_1hr      [finite_t_vn_hertn_1hr])       NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_ew_hertn_1hr      [finite_t_ew_hertn_1hr])       NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_ns_hertn_1hr      [finite_t_ns_hertn_1hr])       NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_vr_rtn_1hr        [finite_t_vr_rtn_1hr])         NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_vt_rtn_1hr        [finite_t_vt_rtn_1hr])         NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_vn_rtn_1hr        [finite_t_vn_rtn_1hr])         NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_ew_rtn_1hr        [finite_t_ew_rtn_1hr])         NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_ns_rtn_1hr        [finite_t_ns_rtn_1hr])         NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_ew_source_flag_1hr[finite_t_ew_source_flag_1hr]) NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_cycles_ew_1hr     [finite_t_cycles_ew_1hr])      NE 1 THEN stop
                        IF array_equal(ra_t_den_1hr[finite_t_den_1hr], ew_t_uncert_1hr        [finite_t_uncert_1hr])         NE 1 THEN stop
                    ENDIF ELSE BEGIN 
                        IF ew_t_ew_inst_1hr        NE 0 THEN stop
                        IF ew_t_vr_hertn_1hr       NE 0 THEN stop
                        IF ew_t_vt_hertn_1hr       NE 0 THEN stop
                        IF ew_t_vn_hertn_1hr       NE 0 THEN stop
                        IF ew_t_ew_hertn_1hr       NE 0 THEN stop
                        IF ew_t_ns_hertn_1hr       NE 0 THEN stop
                        IF ew_t_vr_rtn_1hr         NE 0 THEN stop
                        IF ew_t_vt_rtn_1hr         NE 0 THEN stop
                        IF ew_t_vn_rtn_1hr         NE 0 THEN stop
                        IF ew_t_ew_rtn_1hr         NE 0 THEN stop
                        IF ew_t_ns_rtn_1hr         NE 0 THEN stop
                        IF ew_t_ew_source_flag_1hr NE 0 THEN stop
                        IF ew_t_uncert_1hr         NE 0 THEN stop
                        ew_inst_1hr        = fltarr(n_elements(ra_t_den_1hr))
                        vr_hertn_1hr       = fltarr(n_elements(ra_t_den_1hr))
                        vt_hertn_1hr       = fltarr(n_elements(ra_t_den_1hr))
                        vn_hertn_1hr       = fltarr(n_elements(ra_t_den_1hr))
                        ns_hertn_1hr       = fltarr(n_elements(ra_t_den_1hr))
                        ew_hertn_1hr       = fltarr(n_elements(ra_t_den_1hr))
                        vr_rtn_1hr         = fltarr(n_elements(ra_t_den_1hr))
                        vt_rtn_1hr         = fltarr(n_elements(ra_t_den_1hr))
                        vn_rtn_1hr         = fltarr(n_elements(ra_t_den_1hr))
                        ns_rtn_1hr         = fltarr(n_elements(ra_t_den_1hr))
                        ew_rtn_1hr         = fltarr(n_elements(ra_t_den_1hr))
                        uncert_1hr         = fltarr(n_elements(ra_t_den_1hr))
                        cycles_ew_1hr      = fltarr(n_elements(ra_t_den_1hr))
                        ew_source_flag_1hr = intarr(n_elements(ra_t_den_1hr), 2)
                        ew_inst_1hr       [*] = !values.f_nan
                        vr_hertn_1hr      [*] = !values.f_nan
                        vt_hertn_1hr      [*] = !values.f_nan
                        vn_hertn_1hr      [*] = !values.f_nan
                        ns_hertn_1hr      [*] = !values.f_nan
                        ew_hertn_1hr      [*] = !values.f_nan
                        vr_rtn_1hr        [*] = !values.f_nan
                        vt_rtn_1hr        [*] = !values.f_nan
                        vn_rtn_1hr        [*] = !values.f_nan
                        ns_rtn_1hr        [*] = !values.f_nan
                        ew_rtn_1hr        [*] = !values.f_nan
                        uncert_1hr        [*] = !values.f_nan
                        cycles_ew_1hr     [*] = 0
                        ew_source_flag_1hr[*, *] = -1
                    ENDELSE 
                ENDELSE 
                ra_t = ra_t_den
                ra_t_10min = ra_t_den_10min
                ra_t_1hr = ra_t_den_1hr

                ; Check for hand-written bad times
                have_data = 1
                have_data_10min = 1
                have_data_1hr = 1
                FOR kk = 0, n_elements(bad_start_time)-1 DO BEGIN 
                    IF bad_sat[kk] EQ sat THEN BEGIN 
                        temp_bad_start       = time_double(       bad_start_time[kk])
                        temp_bad_stop        = time_double(       bad_stop_time [kk])
                        temp_bad_start_10min = time_double(strmid(bad_start_time[kk], 0, 15)+ '0:00')
                        temp_bad_stop_10min  = time_double(strmid(bad_stop_time [kk], 0, 15)+ '9:59')
                        temp_bad_start_1hr   = time_double(strmid(bad_start_time[kk], 0, 14)+'00:00')
                        temp_bad_stop_1hr    = time_double(strmid(bad_stop_time [kk], 0, 14)+'59:59')
                        bad_time = where(ra_t GE temp_bad_start AND ra_t LE temp_bad_stop, bad_count, $
                                         complement = good_time, ncomplement = good_count)
                        bad_time_10min = where(ra_t_10min GE temp_bad_start_10min AND ra_t_10min LE temp_bad_stop_10min, bad_count_10min, $
                                               complement = good_time_10min, ncomplement = good_count_10min)
                        bad_time_1hr = where(ra_t_1hr GE temp_bad_start_1hr AND ra_t_1hr LE temp_bad_stop_1hr, bad_count_1hr, $
                                             complement = good_time_1hr, ncomplement = good_count_1hr)
                        IF bad_count GT 0 THEN BEGIN 
                            error     [bad_time] = 8 ; removed by hand
                            CASE bad_prods[kk] OF
                                -1: BEGIN 
                                    den       [bad_time] = !values.f_nan
                                    speed     [bad_time] = !values.f_nan
                                    temp      [bad_time] = !values.f_nan
                                    v_therm   [bad_time] = !values.f_nan
                                    ns_inst   [bad_time] = !values.f_nan
                                    ew_inst   [bad_time] = !values.f_nan
                                    vr_hertn  [bad_time] = !values.f_nan
                                    vt_hertn  [bad_time] = !values.f_nan
                                    vn_hertn  [bad_time] = !values.f_nan
                                    ew_hertn  [bad_time] = !values.f_nan
                                    ns_hertn  [bad_time] = !values.f_nan
                                    vr_rtn    [bad_time] = !values.f_nan
                                    vt_rtn    [bad_time] = !values.f_nan
                                    vn_rtn    [bad_time] = !values.f_nan
                                    ew_rtn    [bad_time] = !values.f_nan
                                    ns_rtn    [bad_time] = !values.f_nan
                                    red_chi_sq[bad_time] = !values.f_nan
                                    IF want_gse EQ 1 THEN BEGIN
                                        vr_gse[bad_time] = !values.f_nan
                                        vt_gse[bad_time] = !values.f_nan
                                        vn_gse[bad_time] = !values.f_nan
                                        ew_gse[bad_time] = !values.f_nan
                                        ns_gse[bad_time] = !values.f_nan
                                    ENDIF
                                END 
                                0: den        [bad_time] = !values.f_nan                                   
                            ENDCASE 
                        ENDIF 
                        IF bad_count_10min GT 0 THEN BEGIN
                            error_10min         [bad_time_10min] = 8 ; removed by hand
                            CASE bad_prods[kk] OF
                                -1: BEGIN 
                                    cycles_10min        [bad_time_10min] = 0
                                    cycles_speed_10min  [bad_time_10min] = 0
                                    cycles_ns_inst_10min[bad_time_10min] = 0
                                    cycles_ew_10min     [bad_time_10min] = 0
                                    den_10min           [bad_time_10min] = !values.f_nan
                                    speed_10min         [bad_time_10min] = !values.f_nan
                                    temp_10min          [bad_time_10min] = !values.f_nan
                                    v_therm_10min       [bad_time_10min] = !values.f_nan
                                    ns_inst_10min       [bad_time_10min] = !values.f_nan
                                    ew_inst_10min       [bad_time_10min] = !values.f_nan
                                    vr_hertn_10min      [bad_time_10min] = !values.f_nan
                                    vt_hertn_10min      [bad_time_10min] = !values.f_nan
                                    vn_hertn_10min      [bad_time_10min] = !values.f_nan
                                    ew_hertn_10min      [bad_time_10min] = !values.f_nan
                                    ns_hertn_10min      [bad_time_10min] = !values.f_nan
                                    vr_rtn_10min        [bad_time_10min] = !values.f_nan
                                    vt_rtn_10min        [bad_time_10min] = !values.f_nan
                                    vn_rtn_10min        [bad_time_10min] = !values.f_nan
                                    ew_rtn_10min        [bad_time_10min] = !values.f_nan
                                    ns_rtn_10min        [bad_time_10min] = !values.f_nan
                                    red_chi_sq_10min    [bad_time_10min] = !values.f_nan
                                END 
                                0: den_10min            [bad_time_10min] = !values.f_nan
                            ENDCASE 
                        ENDIF 
                        IF bad_count_1hr GT 0 THEN BEGIN
                            error_1hr         [bad_time_1hr] = 8 ; removed by hand
                            CASE bad_prods[kk] OF
                                -1: BEGIN 
                                    cycles_1hr        [bad_time_1hr] = 0
                                    cycles_speed_1hr  [bad_time_1hr] = 0
                                    cycles_ns_inst_1hr[bad_time_1hr] = 0
                                    cycles_ew_1hr     [bad_time_1hr] = 0
                                    den_1hr           [bad_time_1hr] = !values.f_nan
                                    speed_1hr         [bad_time_1hr] = !values.f_nan
                                    temp_1hr          [bad_time_1hr] = !values.f_nan
                                    v_therm_1hr       [bad_time_1hr] = !values.f_nan
                                    ns_inst_1hr       [bad_time_1hr] = !values.f_nan
                                    ew_inst_1hr       [bad_time_1hr] = !values.f_nan
                                    vr_hertn_1hr      [bad_time_1hr] = !values.f_nan
                                    vt_hertn_1hr      [bad_time_1hr] = !values.f_nan
                                    vn_hertn_1hr      [bad_time_1hr] = !values.f_nan
                                    ew_hertn_1hr      [bad_time_1hr] = !values.f_nan
                                    ns_hertn_1hr      [bad_time_1hr] = !values.f_nan
                                    vr_rtn_1hr        [bad_time_1hr] = !values.f_nan
                                    vt_rtn_1hr        [bad_time_1hr] = !values.f_nan
                                    vn_rtn_1hr        [bad_time_1hr] = !values.f_nan
                                    ew_rtn_1hr        [bad_time_1hr] = !values.f_nan
                                    ns_rtn_1hr        [bad_time_1hr] = !values.f_nan
                                    red_chi_sq_1hr    [bad_time_1hr] = !values.f_nan
                                END
                                0: den_1hr            [bad_time_1hr] = !values.f_nan
                            ENDCASE 
                        ENDIF 
                    ENDIF 
                ENDFOR 

                ;; adjust decimal places
                ;; density
                den_big_i         = where(finite(den)       EQ 1 AND den       GE 100,                     den_big_c)
                den_mid_i         = where(finite(den)       EQ 1 AND den       LT 100 AND den GE 10,       den_mid_c)
                den_small_i       = where(finite(den)       EQ 1 AND den       LT 10,                      den_small_c)
                den_10min_big_i   = where(finite(den_10min) EQ 1 AND den_10min GE 100,                     den_10min_big_c)
                den_10min_mid_i   = where(finite(den_10min) EQ 1 AND den_10min LT 100 AND den_10min GE 10, den_10min_mid_c)
                den_10min_small_i = where(finite(den_10min) EQ 1 AND den_10min LT 10,                      den_10min_small_c)
                den_1hr_big_i     = where(finite(den_1hr)   EQ 1 AND den_1hr   GE 100,                     den_1hr_big_c)
                den_1hr_mid_i     = where(finite(den_1hr)   EQ 1 AND den_1hr   LT 100 AND den_1hr GE 10,   den_1hr_mid_c)
                den_1hr_small_i   = where(finite(den_1hr)   EQ 1 AND den_1hr   LT 10,                      den_1hr_small_c)
                ; no decimal places
                IF den_big_c       GT 0 THEN den      [den_big_i]       = round(float(den      [den_big_i]))
                IF den_10min_big_c GT 0 THEN den_10min[den_10min_big_i] = round(float(den_10min[den_10min_big_i]))
                IF den_1hr_big_c   GT 0 THEN den_1hr  [den_1hr_big_i]   = round(float(den_1hr  [den_1hr_big_i]))
                ; one decimal place
                IF den_mid_c       GT 0 THEN den      [den_mid_i]       = float(round(float(den      [den_mid_i]       * 10))) / 10
                IF den_10min_mid_c GT 0 THEN den_10min[den_10min_mid_i] = float(round(float(den_10min[den_10min_mid_i] * 10))) / 10
                IF den_1hr_mid_c   GT 0 THEN den_1hr  [den_1hr_mid_i]   = float(round(float(den_1hr  [den_1hr_mid_i]   * 10))) / 10
                ;; three decimal places
                ;; Changed this 5/7/2020 
                IF den_small_c       GT 0 THEN den      [den_small_i]       = float(round(float(den      [den_small_i]       * 1000))) / 1000
                IF den_10min_small_c GT 0 THEN den_10min[den_10min_small_i] = float(round(float(den_10min[den_10min_small_i] * 1000))) / 1000
                IF den_1hr_small_c   GT 0 THEN den_1hr  [den_1hr_small_i]   = float(round(float(den_1hr  [den_1hr_small_i]   * 1000))) / 1000
                ; temperature
                temp_2_i       = where(finite(temp)       EQ 1 AND temp       GE 1E2 AND temp       LT 1E3, temp_2_c)
                temp_2_10min_i = where(finite(temp_10min) EQ 1 AND temp_10min GE 1E2 AND temp_10min LT 1E3, temp_2_10min_c)
                temp_2_1hr_i   = where(finite(temp_1hr)   EQ 1 AND temp_1hr   GE 1E2 AND temp_1hr   LT 1E3, temp_2_1hr_c)
                temp_3_i       = where(finite(temp)       EQ 1 AND temp       GE 1E3 AND temp       LT 1E4, temp_3_c)
                temp_3_10min_i = where(finite(temp_10min) EQ 1 AND temp_10min GE 1E3 AND temp_10min LT 1E4, temp_3_10min_c)
                temp_3_1hr_i   = where(finite(temp_1hr)   EQ 1 AND temp_1hr   GE 1E3 AND temp_1hr   LT 1E4, temp_3_1hr_c)
                temp_4_i       = where(finite(temp)       EQ 1 AND temp       GE 1E4 AND temp       LT 1E5, temp_4_c)
                temp_4_10min_i = where(finite(temp_10min) EQ 1 AND temp_10min GE 1E4 AND temp_10min LT 1E5, temp_4_10min_c)
                temp_4_1hr_i   = where(finite(temp_1hr)   EQ 1 AND temp_1hr   GE 1E4 AND temp_1hr   LT 1E5, temp_4_1hr_c)
                temp_5_i       = where(finite(temp)       EQ 1 AND temp       GE 1E5 AND temp       LT 1E6, temp_5_c)
                temp_5_10min_i = where(finite(temp_10min) EQ 1 AND temp_10min GE 1E5 AND temp_10min LT 1E6, temp_5_10min_c)
                temp_5_1hr_i   = where(finite(temp_1hr)   EQ 1 AND temp_1hr   GE 1E5 AND temp_1hr   LT 1E6, temp_5_1hr_c)
                temp_6_i       = where(finite(temp)       EQ 1 AND temp       GE 1E6,                       temp_6_c)
                temp_6_10min_i = where(finite(temp_10min) EQ 1 AND temp_10min GE 1E6,                       temp_6_10min_c)
                temp_6_1hr_i   = where(finite(temp_1hr)   EQ 1 AND temp_1hr   GE 1E6,                       temp_6_1hr_c)
                IF temp_2_c       GT 0 THEN temp      [temp_2_i]       = float(round(float(temp      [temp_2_i]       / 1   ))) * 1    ; 1s place
                IF temp_2_10min_c GT 0 THEN temp_10min[temp_2_10min_i] = float(round(float(temp_10min[temp_2_10min_i] / 1   ))) * 1 
                IF temp_2_1hr_c   GT 0 THEN temp_1hr  [temp_2_1hr_i]   = float(round(float(temp_1hr  [temp_2_1hr_i]   / 1   ))) * 1 
                IF temp_3_c       GT 0 THEN temp      [temp_3_i]       = float(round(float(temp      [temp_3_i]       / 10  ))) * 10   ; 10s place
                IF temp_3_10min_c GT 0 THEN temp_10min[temp_3_10min_i] = float(round(float(temp_10min[temp_3_10min_i] / 10  ))) * 10
                IF temp_3_1hr_c   GT 0 THEN temp_1hr  [temp_3_1hr_i]   = float(round(float(temp_1hr  [temp_3_1hr_i]   / 10  ))) * 10
                IF temp_4_c       GT 0 THEN temp      [temp_4_i]       = float(round(float(temp      [temp_4_i]       / 10E1))) * 10E1 ; 100s place
                IF temp_4_10min_c GT 0 THEN temp_10min[temp_4_10min_i] = float(round(float(temp_10min[temp_4_10min_i] / 10E1))) * 10E1
                IF temp_4_1hr_c   GT 0 THEN temp_1hr  [temp_4_1hr_i]   = float(round(float(temp_1hr  [temp_4_1hr_i]   / 10E1))) * 10E1
                IF temp_5_c       GT 0 THEN temp      [temp_5_i]       = float(round(float(temp      [temp_5_i]       / 10E2))) * 10E2 ; 1,000s place
                IF temp_5_10min_c GT 0 THEN temp_10min[temp_5_10min_i] = float(round(float(temp_10min[temp_5_10min_i] / 10E2))) * 10E2
                IF temp_5_1hr_c   GT 0 THEN temp_1hr  [temp_5_1hr_i]   = float(round(float(temp_1hr  [temp_5_1hr_i]   / 10E2))) * 10E2
                IF temp_6_c       GT 0 THEN temp      [temp_6_i]       = float(round(float(temp      [temp_6_i]       / 10E3))) * 10E3 ; 10,000s place
                IF temp_6_10min_c GT 0 THEN temp_10min[temp_6_10min_i] = float(round(float(temp_10min[temp_6_10min_i] / 10E3))) * 10E3
                IF temp_6_1hr_c   GT 0 THEN temp_1hr  [temp_6_1hr_i]   = float(round(float(temp_1hr  [temp_6_1hr_i]   / 10E3))) * 10E3
                ; the rest
                speed_i             = where(finite(speed)                EQ 1, speed_c)
                speed_10min_i       = where(finite(speed_10min)          EQ 1, speed_10min_c)
                speed_1hr_i         = where(finite(speed_1hr)            EQ 1, speed_1hr_c)
                v_therm_i           = where(finite(v_therm)              EQ 1, v_therm_c)
                v_therm_10min_i     = where(finite(v_therm_10min)        EQ 1, v_therm_10min_c)
                v_therm_1hr_i       = where(finite(v_therm_1hr)          EQ 1, v_therm_1hr_c)
                ns_inst_i           = where(finite(ns_inst)              EQ 1, ns_inst_c)
                ns_inst_10min_i     = where(finite(ns_inst_10min)        EQ 1, ns_inst_10min_c)
                ns_inst_1hr_i       = where(finite(ns_inst_1hr)          EQ 1, ns_inst_1hr_c)
                ew_inst_i           = where(finite(ew_inst)              EQ 1, ew_inst_c)
                ew_inst_10min_i     = where(finite(ew_inst_10min)        EQ 1, ew_inst_10min_c)
                ew_inst_1hr_i       = where(finite(ew_inst_1hr)          EQ 1, ew_inst_1hr_c)
                vr_hertn_i          = where(finite(vr_hertn)             EQ 1, vr_hertn_c)
                vr_hertn_10min_i    = where(finite(vr_hertn_10min)       EQ 1, vr_hertn_10min_c)
                vr_hertn_1hr_i      = where(finite(vr_hertn_1hr)         EQ 1, vr_hertn_1hr_c)
                vt_hertn_i          = where(finite(vt_hertn)             EQ 1, vt_hertn_c)
                vt_hertn_10min_i    = where(finite(vt_hertn_10min)       EQ 1, vt_hertn_10min_c)
                vt_hertn_1hr_i      = where(finite(vt_hertn_1hr)         EQ 1, vt_hertn_1hr_c)
                vn_hertn_i          = where(finite(vn_hertn)             EQ 1, vn_hertn_c)
                vn_hertn_10min_i    = where(finite(vn_hertn_10min)       EQ 1, vn_hertn_10min_c)
                vn_hertn_1hr_i      = where(finite(vn_hertn_1hr)         EQ 1, vn_hertn_1hr_c)
                ns_hertn_i          = where(finite(ns_hertn)             EQ 1, ns_hertn_c)
                ns_hertn_10min_i    = where(finite(ns_hertn_10min)       EQ 1, ns_hertn_10min_c)
                ns_hertn_1hr_i      = where(finite(ns_hertn_1hr)         EQ 1, ns_hertn_1hr_c)
                ew_hertn_i          = where(finite(ew_hertn)             EQ 1, ew_hertn_c)
                ew_hertn_10min_i    = where(finite(ew_hertn_10min)       EQ 1, ew_hertn_10min_c)
                ew_hertn_1hr_i      = where(finite(ew_hertn_1hr)         EQ 1, ew_hertn_1hr_c)
                vr_rtn_i            = where(finite(vr_rtn)               EQ 1, vr_rtn_c)
                vr_rtn_10min_i      = where(finite(vr_rtn_10min)         EQ 1, vr_rtn_10min_c)
                vr_rtn_1hr_i        = where(finite(vr_rtn_1hr)           EQ 1, vr_rtn_1hr_c)
                vt_rtn_i            = where(finite(vt_rtn)               EQ 1, vt_rtn_c)
                vt_rtn_10min_i      = where(finite(vt_rtn_10min)         EQ 1, vt_rtn_10min_c)
                vt_rtn_1hr_i        = where(finite(vt_rtn_1hr)           EQ 1, vt_rtn_1hr_c)
                vn_rtn_i            = where(finite(vn_rtn)               EQ 1, vn_rtn_c)
                vn_rtn_10min_i      = where(finite(vn_rtn_10min)         EQ 1, vn_rtn_10min_c)
                vn_rtn_1hr_i        = where(finite(vn_rtn_1hr)           EQ 1, vn_rtn_1hr_c)
                ns_rtn_i            = where(finite(ns_rtn)               EQ 1, ns_rtn_c)
                ns_rtn_10min_i      = where(finite(ns_rtn_10min)         EQ 1, ns_rtn_10min_c)
                ns_rtn_1hr_i        = where(finite(ns_rtn_1hr)           EQ 1, ns_rtn_1hr_c)
                ew_rtn_i            = where(finite(ew_rtn)               EQ 1, ew_rtn_c)
                ew_rtn_10min_i      = where(finite(ew_rtn_10min)         EQ 1, ew_rtn_10min_c)
                ew_rtn_1hr_i        = where(finite(ew_rtn_1hr)           EQ 1, ew_rtn_1hr_c)
                red_chi_sq_i        = where(finite(red_chi_sq)           EQ 1, red_chi_sq_c) 
                red_chi_sq_10min_i  = where(finite(red_chi_sq_10min)     EQ 1, red_chi_sq_10min_c) 
                red_chi_sq_1hr_i    = where(finite(red_chi_sq_1hr)       EQ 1, red_chi_sq_1hr_c) 
                uncert_i            = where(finite(uncert)               EQ 1, uncert_c)
                uncert_10min_i      = where(finite(uncert_10min)         EQ 1, uncert_10min_c)
                uncert_1hr_i        = where(finite(uncert_1hr)           EQ 1, uncert_1hr_c)
                ew_source_i         = where(finite(ew_source_flag)       EQ 1, ew_source_c)
                ew_source_10min_i   = where(finite(ew_source_flag_10min) EQ 1, ew_source_10min_c)
                ew_source_1hr_i     = where(finite(ew_source_flag_1hr)   EQ 1, ew_source_1hr_c)
                IF want_gse THEN BEGIN
                    vr_gse_i        = where(finite(vr_gse)               EQ 1, vr_gse_c)
                    vt_gse_i        = where(finite(vt_gse)               EQ 1, vt_gse_c)
                    vn_gse_i        = where(finite(vn_gse)               EQ 1, vn_gse_c)
                    ns_gse_i        = where(finite(ns_gse)               EQ 1, ns_gse_c)
                    ew_gse_i        = where(finite(ew_gse)               EQ 1, ew_gse_c)
            ENDIF 
                ; no decimal places
            IF speed_c             GT 0 THEN speed            [speed_i]             = float(round(float(speed            [speed_i])))
            IF speed_10min_c       GT 0 THEN speed_10min      [speed_10min_i]       = float(round(float(speed_10min      [speed_10min_i]))) 
            IF speed_1hr_c         GT 0 THEN speed_1hr        [speed_1hr_i]         = float(round(float(speed_1hr        [speed_1hr_i])))
            IF uncert_c            GT 0 THEN uncert           [uncert_i]            = float(round(float(uncert           [uncert_i])))
            IF uncert_10min_c      GT 0 THEN uncert_10min     [uncert_10min_i]      = float(round(float(uncert_10min     [uncert_10min_i]))) 
            IF uncert_1hr_c        GT 0 THEN uncert_1hr       [uncert_1hr_i]        = float(round(float(uncert_1hr       [uncert_1hr_i])))
            IF ew_source_c         GT 0 THEN ew_source_flag      [ew_source_i]       = float(round(float(ew_source_flag      [ew_source_i])))
            IF ew_source_10min_c   GT 0 THEN ew_source_flag_10min[ew_source_10min_i] = float(round(float(ew_source_flag_10min[ew_source_10min_i]))) 
            IF ew_source_1hr_c     GT 0 THEN ew_source_flag_1hr  [ew_source_1hr_i]   = float(round(float(ew_source_flag_1hr  [ew_source_1hr_i])))
                ; one decimal place
            IF v_therm_c           GT 0 THEN v_therm          [v_therm_i]           = float(round(float(v_therm       [v_therm_i]        * 10))) / 10
            IF v_therm_10min_c     GT 0 THEN v_therm_10min    [v_therm_10min_i]     = float(round(float(v_therm_10min [v_therm_10min_i]  * 10))) / 10
            IF v_therm_1hr_c       GT 0 THEN v_therm_1hr      [v_therm_1hr_i]       = float(round(float(v_therm_1hr   [v_therm_1hr_i]    * 10))) / 10
            IF ns_inst_c           GT 0 THEN ns_inst          [ns_inst_i]           = float(round(float(ns_inst       [ns_inst_i]        * 10))) / 10
            IF ns_inst_10min_c     GT 0 THEN ns_inst_10min    [ns_inst_10min_i]     = float(round(float(ns_inst_10min [ns_inst_10min_i]  * 10))) / 10
            IF ns_inst_1hr_c       GT 0 THEN ns_inst_1hr      [ns_inst_1hr_i]       = float(round(float(ns_inst_1hr   [ns_inst_1hr_i]    * 10))) / 10
            IF ew_inst_c           GT 0 THEN ew_inst          [ew_inst_i]           = float(round(float(ew_inst       [ew_inst_i]        * 10))) / 10
            IF ew_inst_10min_c     GT 0 THEN ew_inst_10min    [ew_inst_10min_i]     = float(round(float(ew_inst_10min [ew_inst_10min_i]  * 10))) / 10
            IF ew_inst_1hr_c       GT 0 THEN ew_inst_1hr      [ew_inst_1hr_i]       = float(round(float(ew_inst_1hr   [ew_inst_1hr_i]    * 10))) / 10
            IF vr_hertn_c          GT 0 THEN vr_hertn         [vr_hertn_i]          = float(round(float(vr_hertn      [vr_hertn_i]       * 10))) / 10
            IF vr_hertn_10min_c    GT 0 THEN vr_hertn_10min   [vr_hertn_10min_i]    = float(round(float(vr_hertn_10min[vr_hertn_10min_i] * 10))) / 10
            IF vr_hertn_1hr_c      GT 0 THEN vr_hertn_1hr     [vr_hertn_1hr_i]      = float(round(float(vr_hertn_1hr  [vr_hertn_1hr_i]   * 10))) / 10
            IF vt_hertn_c          GT 0 THEN vt_hertn         [vt_hertn_i]          = float(round(float(vt_hertn      [vt_hertn_i]       * 10))) / 10
            IF vt_hertn_10min_c    GT 0 THEN vt_hertn_10min   [vt_hertn_10min_i]    = float(round(float(vt_hertn_10min[vt_hertn_10min_i] * 10))) / 10
            IF vt_hertn_1hr_c      GT 0 THEN vt_hertn_1hr     [vt_hertn_1hr_i]      = float(round(float(vt_hertn_1hr  [vt_hertn_1hr_i]   * 10))) / 10
            IF vn_hertn_c          GT 0 THEN vn_hertn         [vn_hertn_i]          = float(round(float(vn_hertn      [vn_hertn_i]       * 10))) / 10
            IF vn_hertn_10min_c    GT 0 THEN vn_hertn_10min   [vn_hertn_10min_i]    = float(round(float(vn_hertn_10min[vn_hertn_10min_i] * 10))) / 10
            IF vn_hertn_1hr_c      GT 0 THEN vn_hertn_1hr     [vn_hertn_1hr_i]      = float(round(float(vn_hertn_1hr  [vn_hertn_1hr_i]   * 10))) / 10
            IF ns_hertn_c          GT 0 THEN ns_hertn         [ns_hertn_i]          = float(round(float(ns_hertn      [ns_hertn_i]       * 10))) / 10
            IF ns_hertn_10min_c    GT 0 THEN ns_hertn_10min   [ns_hertn_10min_i]    = float(round(float(ns_hertn_10min[ns_hertn_10min_i] * 10))) / 10
            IF ns_hertn_1hr_c      GT 0 THEN ns_hertn_1hr     [ns_hertn_1hr_i]      = float(round(float(ns_hertn_1hr  [ns_hertn_1hr_i]   * 10))) / 10
            IF ew_hertn_c          GT 0 THEN ew_hertn         [ew_hertn_i]          = float(round(float(ew_hertn      [ew_hertn_i]       * 10))) / 10
            IF ew_hertn_10min_c    GT 0 THEN ew_hertn_10min   [ew_hertn_10min_i]    = float(round(float(ew_hertn_10min[ew_hertn_10min_i] * 10))) / 10
            IF ew_hertn_1hr_c      GT 0 THEN ew_hertn_1hr     [ew_hertn_1hr_i]      = float(round(float(ew_hertn_1hr  [ew_hertn_1hr_i]   * 10))) / 10
            IF vr_rtn_c            GT 0 THEN vr_rtn           [vr_rtn_i]            = float(round(float(vr_rtn        [vr_rtn_i]         * 10))) / 10
            IF vr_rtn_10min_c      GT 0 THEN vr_rtn_10min     [vr_rtn_10min_i]      = float(round(float(vr_rtn_10min  [vr_rtn_10min_i]   * 10))) / 10
            IF vr_rtn_1hr_c        GT 0 THEN vr_rtn_1hr       [vr_rtn_1hr_i]        = float(round(float(vr_rtn_1hr    [vr_rtn_1hr_i]     * 10))) / 10
            IF vt_rtn_c            GT 0 THEN vt_rtn           [vt_rtn_i]            = float(round(float(vt_rtn        [vt_rtn_i]         * 10))) / 10
            IF vt_rtn_10min_c      GT 0 THEN vt_rtn_10min     [vt_rtn_10min_i]      = float(round(float(vt_rtn_10min  [vt_rtn_10min_i]   * 10))) / 10
            IF vt_rtn_1hr_c        GT 0 THEN vt_rtn_1hr       [vt_rtn_1hr_i]        = float(round(float(vt_rtn_1hr    [vt_rtn_1hr_i]     * 10))) / 10
            IF vn_rtn_c            GT 0 THEN vn_rtn           [vn_rtn_i]            = float(round(float(vn_rtn        [vn_rtn_i]         * 10))) / 10
            IF vn_rtn_10min_c      GT 0 THEN vn_rtn_10min     [vn_rtn_10min_i]      = float(round(float(vn_rtn_10min  [vn_rtn_10min_i]   * 10))) / 10
            IF vn_rtn_1hr_c        GT 0 THEN vn_rtn_1hr       [vn_rtn_1hr_i]        = float(round(float(vn_rtn_1hr    [vn_rtn_1hr_i]     * 10))) / 10
            IF ns_rtn_c            GT 0 THEN ns_rtn           [ns_rtn_i]            = float(round(float(ns_rtn        [ns_rtn_i]         * 10))) / 10
            IF ns_rtn_10min_c      GT 0 THEN ns_rtn_10min     [ns_rtn_10min_i]      = float(round(float(ns_rtn_10min  [ns_rtn_10min_i]   * 10))) / 10
            IF ns_rtn_1hr_c        GT 0 THEN ns_rtn_1hr       [ns_rtn_1hr_i]        = float(round(float(ns_rtn_1hr    [ns_rtn_1hr_i]     * 10))) / 10
            IF ew_rtn_c            GT 0 THEN ew_rtn           [ew_rtn_i]            = float(round(float(ew_rtn        [ew_rtn_i]         * 10))) / 10
            IF ew_rtn_10min_c      GT 0 THEN ew_rtn_10min     [ew_rtn_10min_i]      = float(round(float(ew_rtn_10min  [ew_rtn_10min_i]   * 10))) / 10
            IF ew_rtn_1hr_c        GT 0 THEN ew_rtn_1hr       [ew_rtn_1hr_i]        = float(round(float(ew_rtn_1hr    [ew_rtn_1hr_i]     * 10))) / 10
            IF want_gse EQ 1 THEN BEGIN
                IF vr_gse_c        GT 0 THEN vr_gse           [vr_gse_i]            = float(round(float(vr_gse        [vr_gse_i]         * 10))) / 10
                IF vt_gse_c        GT 0 THEN vt_gse           [vt_gse_i]            = float(round(float(vt_gse        [vt_gse_i]         * 10))) / 10
                IF vn_gse_c        GT 0 THEN vn_gse           [vn_gse_i]            = float(round(float(vn_gse        [vn_gse_i]         * 10))) / 10
                IF ns_gse_c        GT 0 THEN ns_gse           [ns_gse_i]            = float(round(float(ns_gse        [ns_gse_i]         * 10))) / 10
                IF ew_gse_c        GT 0 THEN ew_gse           [ew_gse_i]            = float(round(float(ew_gse        [ew_gse_i]         * 10))) / 10
            ENDIF 
                ; two decimal places
            IF red_chi_sq_c       GT 0 THEN red_chi_sq      [red_chi_sq_i]       = float(round(float(red_chi_sq      [red_chi_sq_i]       * 100))) / 100
            IF red_chi_sq_10min_c GT 0 THEN red_chi_sq_10min[red_chi_sq_10min_i] = float(round(float(red_chi_sq_10min[red_chi_sq_10min_i] * 100))) / 100
            IF red_chi_sq_1hr_c   GT 0 THEN red_chi_sq_1hr  [red_chi_sq_1hr_i]   = float(round(float(red_chi_sq_1hr  [red_chi_sq_1hr_i]   * 100))) / 100

                ; create formatted strings for printing 
            den_st       = string(den)
            den_10min_st = string(den_10min)
            den_1hr_st   = string(den_1hr)
            IF den_big_c         GT 0 THEN den_st      [den_big_i]         = string(den      [den_big_i],         format = '(F0.0)')
            IF den_10min_big_c   GT 0 THEN den_10min_st[den_10min_big_i]   = string(den_10min[den_10min_big_i],   format = '(F0.0)')
            IF den_1hr_big_c     GT 0 THEN den_1hr_st  [den_1hr_big_i]     = string(den_1hr  [den_1hr_big_i],     format = '(F0.0)')
                ; one decimal place
            IF den_mid_c         GT 0 THEN den_st      [den_mid_i]         = string(den      [den_mid_i],         format = '(F0.1)')
            IF den_10min_mid_c   GT 0 THEN den_10min_st[den_10min_mid_i]   = string(den_10min[den_10min_mid_i],   format = '(F0.1)')
            IF den_1hr_mid_c     GT 0 THEN den_1hr_st  [den_1hr_mid_i]     = string(den_1hr  [den_1hr_mid_i],     format = '(F0.1)')
                ; two decimal places
            IF den_small_c       GT 0 THEN den_st      [den_small_i]       = string(den      [den_small_i],       format = '(F0.2)')
            IF den_10min_small_c GT 0 THEN den_10min_st[den_10min_small_i] = string(den_10min[den_10min_small_i], format = '(F0.2)')
            IF den_1hr_small_c   GT 0 THEN den_1hr_st  [den_1hr_small_i]   = string(den_1hr  [den_1hr_small_i],   format = '(F0.2)')
            speed_st             = string(speed,            format = '(F0.1)')
            speed_10min_st       = string(speed_10min,      format = '(F0.1)')
            speed_1hr_st         = string(speed_1hr,        format = '(F0.1)')
            temp_st              = string(temp,             format = '(E10.2)')
            temp_10min_st        = string(temp_10min,       format = '(E10.2)')
            temp_1hr_st          = string(temp_1hr,         format = '(E10.2)')
            v_therm_st           = string(v_therm,          format = '(F0.1)')
            v_therm_10min_st     = string(v_therm_10min,    format = '(F0.1)')
            v_therm_1hr_st       = string(v_therm_1hr,      format = '(F0.1)')
            ns_inst_st           = string(ns_inst,          format = '(F0.1)')
            ns_inst_10min_st     = string(ns_inst_10min,    format = '(F0.1)')
            ns_inst_1hr_st       = string(ns_inst_1hr,      format = '(F0.1)')
            ew_inst_st           = string(ew_inst,          format = '(F0.1)')
            ew_inst_10min_st     = string(ew_inst_10min,    format = '(F0.1)')
            ew_inst_1hr_st       = string(ew_inst_1hr,      format = '(F0.1)')
            vr_hertn_st          = string(vr_hertn,         format = '(F0.1)')
            vr_hertn_10min_st    = string(vr_hertn_10min,   format = '(F0.1)')
            vr_hertn_1hr_st      = string(vr_hertn_1hr,     format = '(F0.1)')
            vt_hertn_st          = string(vt_hertn,         format = '(F0.1)')
            vt_hertn_10min_st    = string(vt_hertn_10min,   format = '(F0.1)')
            vt_hertn_1hr_st      = string(vt_hertn_1hr,     format = '(F0.1)')
            vn_hertn_st          = string(vn_hertn,         format = '(F0.1)')
            vn_hertn_10min_st    = string(vn_hertn_10min,   format = '(F0.1)')
            vn_hertn_1hr_st      = string(vn_hertn_1hr,     format = '(F0.1)')
            ew_hertn_st          = string(ew_hertn,         format = '(F0.1)')
            ew_hertn_10min_st    = string(ew_hertn_10min,   format = '(F0.1)')
            ew_hertn_1hr_st      = string(ew_hertn_1hr,     format = '(F0.1)')
            ns_hertn_st          = string(ns_hertn,         format = '(F0.1)')
            ns_hertn_10min_st    = string(ns_hertn_10min,   format = '(F0.1)')
            ns_hertn_1hr_st      = string(ns_hertn_1hr,     format = '(F0.1)')
            vr_rtn_st            = string(vr_rtn,           format = '(F0.1)')
            vr_rtn_10min_st      = string(vr_rtn_10min,     format = '(F0.1)')
            vr_rtn_1hr_st        = string(vr_rtn_1hr,       format = '(F0.1)')
            vt_rtn_st            = string(vt_rtn,           format = '(F0.1)')
            vt_rtn_10min_st      = string(vt_rtn_10min,     format = '(F0.1)')
            vt_rtn_1hr_st        = string(vt_rtn_1hr,       format = '(F0.1)')
            vn_rtn_st            = string(vn_rtn,           format = '(F0.1)')
            vn_rtn_10min_st      = string(vn_rtn_10min,     format = '(F0.1)')
            vn_rtn_1hr_st        = string(vn_rtn_1hr,       format = '(F0.1)')
            ew_rtn_st            = string(ew_rtn,           format = '(F0.1)')
            ew_rtn_10min_st      = string(ew_rtn_10min,     format = '(F0.1)')
            ew_rtn_1hr_st        = string(ew_rtn_1hr,       format = '(F0.1)')
            ns_rtn_st            = string(ns_rtn,           format = '(F0.1)')
            ns_rtn_10min_st      = string(ns_rtn_10min,     format = '(F0.1)')
            ns_rtn_1hr_st        = string(ns_rtn_1hr,       format = '(F0.1)')
            red_chi_sq_st        = string(red_chi_sq,       format = '(F0.2)')
            red_chi_sq_10min_st  = string(red_chi_sq_10min, format = '(F0.2)')
            red_chi_sq_1hr_st    = string(red_chi_sq_1hr,   format = '(F0.2)')
            uncert_st            = string(uncert,           format = '(F0.2)')
            uncert_10min_st      = string(uncert_10min,     format = '(F0.2)')
            uncert_1hr_st        = string(uncert_1hr,       format = '(F0.2)')
            ew_source_flag_st       = string(fix(ew_source_flag))       
            ew_source_flag_10min_st = string(fix(ew_source_flag_10min)) 
            ew_source_flag_1hr_st   = string(fix(ew_source_flag_1hr))   
            cycles_ew_10min_st      = string(fix(cycles_ew_10min)) 
            cycles_ew_1hr_st        = string(fix(cycles_ew_1hr))   
            IF want_gse EQ 1 THEN BEGIN
                vr_gse_st        = string(vr_gse,           format = '(F0.1)')
                vt_gse_st        = string(vt_gse,           format = '(F0.1)')
                vn_gse_st        = string(vn_gse,           format = '(F0.1)')
                ew_gse_st        = string(ew_gse,           format = '(F0.1)')
                ns_gse_st        = string(ns_gse,           format = '(F0.1)')
            ENDIF 

                ; put in strings for NaN values
            bad_epoch_1kev_i       = where(finite(epoch_1kev)       NE 1, bad_epoch_1kev_c)
            bad_red_chi_sq_i       = where(finite(red_chi_sq)       NE 1, bad_red_chi_sq_c)
            bad_red_chi_sq_10min_i = where(finite(red_chi_sq_10min) NE 1, bad_red_chi_sq_10min_c)
            bad_red_chi_sq_1hr_i   = where(finite(red_chi_sq_1hr)   NE 1, bad_red_chi_sq_1hr_c)
            bad_den_i              = where(finite(den)              NE 1, bad_den_c)
            bad_den_10min_i        = where(finite(den_10min)        NE 1, bad_den_10min_c)
            bad_den_1hr_i          = where(finite(den_1hr)          NE 1, bad_den_1hr_c)
            bad_uncert_i           = where(finite(uncert)           NE 1, bad_uncert_c)
            bad_uncert_10min_i     = where(finite(uncert_10min)     NE 1, bad_uncert_10min_c)
            bad_uncert_1hr_i       = where(finite(uncert_1hr)       NE 1, bad_uncert_1hr_c)
            bad_speed_i            = where(finite(speed)            NE 1, bad_speed_c)
            bad_speed_10min_i      = where(finite(speed_10min)      NE 1, bad_speed_10min_c)
            bad_speed_1hr_i        = where(finite(speed_1hr)        NE 1, bad_speed_1hr_c)
            bad_temp_i             = where(finite(temp)             NE 1, bad_temp_c)
            bad_temp_10min_i       = where(finite(temp_10min)       NE 1, bad_temp_10min_c)
            bad_temp_1hr_i         = where(finite(temp_1hr)         NE 1, bad_temp_1hr_c)
            bad_v_therm_i          = where(finite(v_therm)          NE 1, bad_v_therm_c)
            bad_v_therm_10min_i    = where(finite(v_therm_10min)    NE 1, bad_v_therm_10min_c)
            bad_v_therm_1hr_i      = where(finite(v_therm_1hr)      NE 1, bad_v_therm_1hr_c)
            bad_ns_inst_i          = where(finite(ns_inst)          NE 1, bad_ns_inst_c)
            bad_ns_inst_10min_i    = where(finite(ns_inst_10min)    NE 1, bad_ns_inst_10min_c)
            bad_ns_inst_1hr_i      = where(finite(ns_inst_1hr)      NE 1, bad_ns_inst_1hr_c)
            bad_ew_inst_i          = where(finite(ew_inst)          NE 1, bad_ew_inst_c)
            bad_ew_inst_10min_i    = where(finite(ew_inst_10min)    NE 1, bad_ew_inst_10min_c)
            bad_ew_inst_1hr_i      = where(finite(ew_inst_1hr)      NE 1, bad_ew_inst_1hr_c)
            bad_vr_hertn_i         = where(finite(vr_hertn)         NE 1, bad_vr_hertn_c)
            bad_vr_hertn_10min_i   = where(finite(vr_hertn_10min)   NE 1, bad_vr_hertn_10min_c)
            bad_vr_hertn_1hr_i     = where(finite(vr_hertn_1hr)     NE 1, bad_vr_hertn_1hr_c)
            bad_vt_hertn_i         = where(finite(vt_hertn)         NE 1, bad_vt_hertn_c)
            bad_vt_hertn_10min_i   = where(finite(vt_hertn_10min)   NE 1, bad_vt_hertn_10min_c)
            bad_vt_hertn_1hr_i     = where(finite(vt_hertn_1hr)     NE 1, bad_vt_hertn_1hr_c)
            bad_vn_hertn_i         = where(finite(vn_hertn)         NE 1, bad_vn_hertn_c)
            bad_vn_hertn_10min_i   = where(finite(vn_hertn_10min)   NE 1, bad_vn_hertn_10min_c)
            bad_vn_hertn_1hr_i     = where(finite(vn_hertn_1hr)     NE 1, bad_vn_hertn_1hr_c)
            bad_ew_hertn_i         = where(finite(ew_hertn)         NE 1, bad_ew_hertn_c)
            bad_ew_hertn_10min_i   = where(finite(ew_hertn_10min)   NE 1, bad_ew_hertn_10min_c)
            bad_ew_hertn_1hr_i     = where(finite(ew_hertn_1hr)     NE 1, bad_ew_hertn_1hr_c)
            bad_ns_hertn_i         = where(finite(ns_hertn)         NE 1, bad_ns_hertn_c)
            bad_ns_hertn_10min_i   = where(finite(ns_hertn_10min)   NE 1, bad_ns_hertn_10min_c)
            bad_ns_hertn_1hr_i     = where(finite(ns_hertn_1hr)     NE 1, bad_ns_hertn_1hr_c)
            bad_vr_rtn_i           = where(finite(vr_rtn)           NE 1, bad_vr_rtn_c)
            bad_vr_rtn_10min_i     = where(finite(vr_rtn_10min)     NE 1, bad_vr_rtn_10min_c)
            bad_vr_rtn_1hr_i       = where(finite(vr_rtn_1hr)       NE 1, bad_vr_rtn_1hr_c)
            bad_vt_rtn_i           = where(finite(vt_rtn)           NE 1, bad_vt_rtn_c)
            bad_vt_rtn_10min_i     = where(finite(vt_rtn_10min)     NE 1, bad_vt_rtn_10min_c)
            bad_vt_rtn_1hr_i       = where(finite(vt_rtn_1hr)       NE 1, bad_vt_rtn_1hr_c)
            bad_vn_rtn_i           = where(finite(vn_rtn)           NE 1, bad_vn_rtn_c)
            bad_vn_rtn_10min_i     = where(finite(vn_rtn_10min)     NE 1, bad_vn_rtn_10min_c)
            bad_vn_rtn_1hr_i       = where(finite(vn_rtn_1hr)       NE 1, bad_vn_rtn_1hr_c)
            bad_ew_rtn_i           = where(finite(ew_rtn)           NE 1, bad_ew_rtn_c)
            bad_ew_rtn_10min_i     = where(finite(ew_rtn_10min)     NE 1, bad_ew_rtn_10min_c)
            bad_ew_rtn_1hr_i       = where(finite(ew_rtn_1hr)       NE 1, bad_ew_rtn_1hr_c)
            bad_ns_rtn_i           = where(finite(ns_rtn)           NE 1, bad_ns_rtn_c)
            bad_ns_rtn_10min_i     = where(finite(ns_rtn_10min)     NE 1, bad_ns_rtn_10min_c)
            bad_ns_rtn_1hr_i       = where(finite(ns_rtn_1hr)       NE 1, bad_ns_rtn_1hr_c)
            bad_ew_source_i        = where(finite(ew_source_flag)       NE 1, bad_ew_source_c)
            bad_ew_source_10min_i  = where(finite(ew_source_flag_10min) NE 1, bad_ew_source_10min_c)
            bad_ew_source_1hr_i    = where(finite(ew_source_flag_1hr)   NE 1, bad_ew_source_1hr_c)
            bad_cycles_ew_10min_i  = where(finite(cycles_ew_10min)      NE 1, bad_cycles_ew_10min_c)
            bad_cycles_ew_1hr_i    = where(finite(cycles_ew_1hr)        NE 1, bad_cycles_ew_1hr_c)
            IF want_gse EQ 1 THEN BEGIN
                bad_vr_gse_i       = where(finite(vr_gse)           NE 1, bad_vr_gse_c)
                bad_vt_gse_i       = where(finite(vt_gse)           NE 1, bad_vt_gse_c)
                bad_vn_gse_i       = where(finite(vn_gse)           NE 1, bad_vn_gse_c)
                bad_ew_gse_i       = where(finite(ew_gse)           NE 1, bad_ew_gse_c)
                bad_ns_gse_i       = where(finite(ns_gse)           NE 1, bad_ns_gse_c)
            ENDIF

            IF bad_epoch_1kev_c       GT 0 THEN BEGIN
                epoch_1kev         [bad_epoch_1kev_i]       = -1e31
            END 
            IF bad_red_chi_sq_c       GT 0 THEN BEGIN
                red_chi_sq         [bad_red_chi_sq_i]       = -1e31
                red_chi_sq_st      [bad_red_chi_sq_i]       = '-1.0E+31'
            END 
            IF bad_red_chi_sq_10min_c GT 0 THEN BEGIN
                red_chi_sq_10min   [bad_red_chi_sq_10min_i] = -1e31
                red_chi_sq_10min_st[bad_red_chi_sq_10min_i] = '-1.0E+31'
            END 
            IF bad_red_chi_sq_1hr_c   GT 0 THEN BEGIN
                red_chi_sq_1hr     [bad_red_chi_sq_1hr_i]   = -1e31
                red_chi_sq_1hr_st  [bad_red_chi_sq_1hr_i]   = '-1.0E+31'
            END 
            IF bad_uncert_c           GT 0 THEN BEGIN
                uncert             [bad_uncert_i]           = -1e31
                uncert_st          [bad_uncert_i]           = '-1.0E+31'
            END 
            IF bad_uncert_10min_c     GT 0 THEN BEGIN
                uncert_10min       [bad_uncert_10min_i]     = -1e31
                uncert_10min_st    [bad_uncert_10min_i]     = '-1.0E+31'
            END 
            IF bad_uncert_1hr_c       GT 0 THEN BEGIN
                uncert_1hr         [bad_uncert_1hr_i]       = -1e31
                uncert_1hr_st      [bad_uncert_1hr_i]       = '-1.0E+31'
            END 
            IF bad_den_c              GT 0 THEN BEGIN
                den                [bad_den_i]              = -1e31
                den_st             [bad_den_i]              = '-1.0E+31'
            END 
            IF bad_den_10min_c        GT 0 THEN BEGIN
                den_10min          [bad_den_10min_i]        = -1e31
                den_10min_st       [bad_den_10min_i]        = '-1.0E+31'
            END 
            IF bad_den_1hr_c          GT 0 THEN BEGIN
                den_1hr            [bad_den_1hr_i]          = -1e31
                den_1hr_st         [bad_den_1hr_i]          = '-1.0E+31'
            END 
            IF bad_speed_c            GT 0 THEN BEGIN
                speed              [bad_speed_i]            = -1e31
                speed_st           [bad_speed_i]            = '-1.0E+31'
            END 
            IF bad_speed_10min_c      GT 0 THEN BEGIN
                speed_10min        [bad_speed_10min_i]      = -1e31
                speed_10min_st     [bad_speed_10min_i]      = '-1.0E+31'
            END 
            IF bad_speed_1hr_c        GT 0 THEN BEGIN
                speed_1hr          [bad_speed_1hr_i]        = -1e31
                speed_1hr_st       [bad_speed_1hr_i]        = '-1.0E+31'
            END 
            IF bad_temp_c             GT 0 THEN BEGIN
                temp               [bad_temp_i]             = -1e31
                temp_st            [bad_temp_i]             = '-1.0E+31'
            END 
            IF bad_temp_10min_c       GT 0 THEN BEGIN
                temp_10min         [bad_temp_10min_i]       = -1e31
                temp_10min_st      [bad_temp_10min_i]       = '-1.0E+31'
            END 
            IF bad_temp_1hr_c         GT 0 THEN BEGIN
                temp_1hr           [bad_temp_1hr_i]         = -1e31
                temp_1hr_st        [bad_temp_1hr_i]         = '-1.0E+31'
            END 
            IF bad_v_therm_c          GT 0 THEN BEGIN
                v_therm            [bad_v_therm_i]          = -1e31
                v_therm_st         [bad_v_therm_i]          = '-1.0E+31'
            END 
            IF bad_v_therm_10min_c    GT 0 THEN BEGIN
                v_therm_10min      [bad_v_therm_10min_i]    = -1e31
                v_therm_10min_st   [bad_v_therm_10min_i]    = '-1.0E+31'
            END 
            IF bad_v_therm_1hr_c      GT 0 THEN BEGIN
                v_therm_1hr        [bad_v_therm_1hr_i]      = -1e31
                v_therm_1hr_st     [bad_v_therm_1hr_i]      = '-1.0E+31'
            END 
            IF bad_ns_inst_c          GT 0 THEN BEGIN
                ns_inst            [bad_ns_inst_i]          = -1e31
                ns_inst_st         [bad_ns_inst_i]          = '-1.0E+31'
            END 
            IF bad_ns_inst_10min_c    GT 0 THEN BEGIN
                ns_inst_10min      [bad_ns_inst_10min_i]    = -1e31
                ns_inst_10min_st   [bad_ns_inst_10min_i]    = '-1.0E+31'
            END 
            IF bad_ns_inst_1hr_c      GT 0 THEN BEGIN
                ns_inst_1hr        [bad_ns_inst_1hr_i]      = -1e31
                ns_inst_1hr_st     [bad_ns_inst_1hr_i]      = '-1.0E+31'
            END 
            IF bad_ew_inst_c          GT 0 THEN BEGIN
                ew_inst            [bad_ew_inst_i]          = -1e31
                ew_inst_st         [bad_ew_inst_i]          = '-1.0E+31'
            END 
            IF bad_ew_inst_10min_c    GT 0 THEN BEGIN
                ew_inst_10min      [bad_ew_inst_10min_i]    = -1e31
                ew_inst_10min_st   [bad_ew_inst_10min_i]    = '-1.0E+31'
            END 
            IF bad_ew_inst_1hr_c      GT 0 THEN BEGIN
                ew_inst_1hr        [bad_ew_inst_1hr_i]      = -1e31
                ew_inst_1hr_st     [bad_ew_inst_1hr_i]      = '-1.0E+31'
            END 
            IF bad_vr_hertn_c         GT 0 THEN BEGIN
                vr_hertn           [bad_vr_hertn_i]         = -1e31
                vr_hertn_st        [bad_vr_hertn_i]         = '-1.0E+31'
            END 
            IF bad_vr_hertn_10min_c   GT 0 THEN BEGIN
                vr_hertn_10min     [bad_vr_hertn_10min_i]   = -1e31
                vr_hertn_10min_st  [bad_vr_hertn_10min_i]   = '-1.0E+31'
            END 
            IF bad_vr_hertn_1hr_c     GT 0 THEN BEGIN
                vr_hertn_1hr       [bad_vr_hertn_1hr_i]     = -1e31
                vr_hertn_1hr_st    [bad_vr_hertn_1hr_i]     = '-1.0E+31'
            END 
            IF bad_vt_hertn_c         GT 0 THEN BEGIN
                vt_hertn           [bad_vt_hertn_i]         = -1e31
                vt_hertn_st        [bad_vt_hertn_i]         = '-1.0E+31'
            END 
            IF bad_vt_hertn_10min_c   GT 0 THEN BEGIN
                vt_hertn_10min     [bad_vt_hertn_10min_i]   = -1e31
                vt_hertn_10min_st  [bad_vt_hertn_10min_i]   = '-1.0E+31'
            END 
            IF bad_vt_hertn_1hr_c     GT 0 THEN BEGIN
                vt_hertn_1hr       [bad_vt_hertn_1hr_i]     = -1e31
                vt_hertn_1hr_st    [bad_vt_hertn_1hr_i]     = '-1.0E+31'
            END 
            IF bad_vn_hertn_c         GT 0 THEN BEGIN
                vn_hertn           [bad_vn_hertn_i]         = -1e31
                vn_hertn_st        [bad_vn_hertn_i]         = '-1.0E+31'
            END 
            IF bad_vn_hertn_10min_c   GT 0 THEN BEGIN
                vn_hertn_10min     [bad_vn_hertn_10min_i]   = -1e31
                vn_hertn_10min_st  [bad_vn_hertn_10min_i]   = '-1.0E+31'
            END 
            IF bad_vn_hertn_1hr_c     GT 0 THEN BEGIN
                vn_hertn_1hr       [bad_vn_hertn_1hr_i]     = -1e31
                vn_hertn_1hr_st    [bad_vn_hertn_1hr_i]     = '-1.0E+31'
            END 
            IF bad_ew_hertn_c         GT 0 THEN BEGIN
                ew_hertn           [bad_ew_hertn_i]         = -1e31
                ew_hertn_st        [bad_ew_hertn_i]         = '-1.0E+31'
            END 
            IF bad_ew_hertn_10min_c   GT 0 THEN BEGIN 
                ew_hertn_10min     [bad_ew_hertn_10min_i]   = -1e31
                ew_hertn_10min_st  [bad_ew_hertn_10min_i]   = '-1.0E+31'
            END 
            IF bad_ew_hertn_1hr_c     GT 0 THEN BEGIN 
                ew_hertn_1hr       [bad_ew_hertn_1hr_i]     = -1e31
                ew_hertn_1hr_st    [bad_ew_hertn_1hr_i]     = '-1.0E+31'
            END 
            IF bad_ns_hertn_c         GT 0 THEN BEGIN 
                ns_hertn           [bad_ns_hertn_i]         = -1e31
                ns_hertn_st        [bad_ns_hertn_i]         = '-1.0E+31'
            END 
            IF bad_ns_hertn_10min_c   GT 0 THEN BEGIN 
                ns_hertn_10min     [bad_ns_hertn_10min_i]   = -1e31
                ns_hertn_10min_st  [bad_ns_hertn_10min_i]   = '-1.0E+31'
            END 
            IF bad_ns_hertn_1hr_c     GT 0 THEN BEGIN 
                ns_hertn_1hr       [bad_ns_hertn_1hr_i]     = -1e31
                ns_hertn_1hr_st    [bad_ns_hertn_1hr_i]     = '-1.0E+31'
            END 
            IF bad_vr_rtn_c           GT 0 THEN BEGIN 
                vr_rtn             [bad_vr_rtn_i]           = -1e31
                vr_rtn_st          [bad_vr_rtn_i]           = '-1.0E+31'
            END 
            IF bad_vr_rtn_10min_c     GT 0 THEN BEGIN 
                vr_rtn_10min       [bad_vr_rtn_10min_i]     = -1e31
                vr_rtn_10min_st    [bad_vr_rtn_10min_i]     = '-1.0E+31'
            END 
            IF bad_vr_rtn_1hr_c       GT 0 THEN BEGIN 
                vr_rtn_1hr         [bad_vr_rtn_1hr_i]       = -1e31
                vr_rtn_1hr_st      [bad_vr_rtn_1hr_i]       = '-1.0E+31'
            END 
            IF bad_vt_rtn_c           GT 0 THEN BEGIN 
                vt_rtn             [bad_vt_rtn_i]           = -1e31
                vt_rtn_st          [bad_vt_rtn_i]           = '-1.0E+31'
            END 
            IF bad_vt_rtn_10min_c     GT 0 THEN BEGIN 
                vt_rtn_10min       [bad_vt_rtn_10min_i]     = -1e31
                vt_rtn_10min_st    [bad_vt_rtn_10min_i]     = '-1.0E+31'
            END 
            IF bad_vt_rtn_1hr_c       GT 0 THEN BEGIN 
                vt_rtn_1hr         [bad_vt_rtn_1hr_i]       = -1e31
                vt_rtn_1hr_st      [bad_vt_rtn_1hr_i]       = '-1.0E+31'
            END 
            IF bad_vn_rtn_c           GT 0 THEN BEGIN 
                vn_rtn             [bad_vn_rtn_i]           = -1e31
                vn_rtn_st          [bad_vn_rtn_i]           = '-1.0E+31'
            END 
            IF bad_vn_rtn_10min_c     GT 0 THEN BEGIN 
                vn_rtn_10min       [bad_vn_rtn_10min_i]     = -1e31
                vn_rtn_10min_st    [bad_vn_rtn_10min_i]     = '-1.0E+31'
            END 
            IF bad_vn_rtn_1hr_c       GT 0 THEN BEGIN
                vn_rtn_1hr         [bad_vn_rtn_1hr_i]       = -1e31
                vn_rtn_1hr_st      [bad_vn_rtn_1hr_i]       = '-1.0E+31'
            END 
            IF bad_ew_rtn_c           GT 0 THEN BEGIN 
                ew_rtn             [bad_ew_rtn_i]           = -1e31
                ew_rtn_st          [bad_ew_rtn_i]           = '-1.0E+31'
            END 
            IF bad_ew_rtn_10min_c     GT 0 THEN BEGIN
                ew_rtn_10min       [bad_ew_rtn_10min_i]     = -1e31
                ew_rtn_10min_st    [bad_ew_rtn_10min_i]     = '-1.0E+31'
            END 
            IF bad_ew_rtn_1hr_c       GT 0 THEN BEGIN 
                ew_rtn_1hr         [bad_ew_rtn_1hr_i]       = -1e31
                ew_rtn_1hr_st      [bad_ew_rtn_1hr_i]       = '-1.0E+31'
            END 
            IF bad_ns_rtn_c           GT 0 THEN BEGIN 
                ns_rtn             [bad_ns_rtn_i]           = -1e31
                ns_rtn_st          [bad_ns_rtn_i]           = '-1.0E+31'
            END 
            IF bad_ns_rtn_10min_c     GT 0 THEN BEGIN 
                ns_rtn_10min       [bad_ns_rtn_10min_i]     = -1e31
                ns_rtn_10min_st    [bad_ns_rtn_10min_i]     = '-1.0E+31'
            END 
            IF bad_ns_rtn_1hr_c       GT 0 THEN BEGIN 
                ns_rtn_1hr         [bad_ns_rtn_1hr_i]       = -1e31
                ns_rtn_1hr_st      [bad_ns_rtn_1hr_i]       = '-1.0E+31'
            END 
            IF bad_ew_source_c        GT 0 THEN BEGIN
                ew_source_flag         [bad_ew_source_i]       = -1
                ew_source_flag_st      [bad_ew_source_i]       = '-1'
            END 
            IF bad_ew_source_10min_c  GT 0 THEN BEGIN
                ew_source_flag_10min   [bad_ew_source_10min_i] = -1
                ew_source_flag_10min_st[bad_ew_source_10min_i] = '-1'
            END 
            IF bad_ew_source_1hr_c    GT 0 THEN BEGIN
                ew_source_flag_1hr     [bad_ew_source_1hr_i]   = -1
                ew_source_flag_1hr_st  [bad_ew_source_1hr_i]   = '-1'
            END 
            IF bad_cycles_ew_10min_c  GT 0 THEN BEGIN
                cycles_ew_10min    [bad_cycles_ew_10min_i]  = -1e31
                cycles_ew_10min_st [bad_cycles_ew_10min_i]  = '-1'
            END 
            IF bad_cycles_ew_1hr_c    GT 0 THEN BEGIN
                cycles_ew_1hr      [bad_cycles_ew_1hr_i]    = -1e31
                cycles_ew_1hr_st   [bad_cycles_ew_1hr_i]    = '-1'
            END 
            IF want_gse EQ 1 THEN BEGIN
                IF bad_vr_gse_c       GT 0 THEN BEGIN 
                    vr_gse         [bad_vr_gse_i]           = -1e31
                    vr_gse_st      [bad_vr_gse_i]           = '-1.0E+31'
                END 
                IF bad_vt_gse_c       GT 0 THEN BEGIN 
                    vt_gse         [bad_vt_gse_i]           = -1e31
                    vt_gse_st      [bad_vt_gse_i]           = '-1.0E+31'
                END 
                IF bad_vn_gse_c       GT 0 THEN BEGIN 
                    vn_gse         [bad_vn_gse_i]           = -1e31
                    vn_gse_st      [bad_vn_gse_i]           = '-1.0E+31'
                END 
                IF bad_ew_gse_c       GT 0 THEN BEGIN 
                    ew_gse         [bad_ew_gse_i]           = -1e31
                    ew_gse_st      [bad_ew_gse_i]           = '-1.0E+31'
                END 
                IF bad_ns_gse_c       GT 0 THEN BEGIN 
                    ns_gse         [bad_ns_gse_i]           = -1e31
                    ns_gse_st      [bad_ns_gse_i]           = '-1.0E+31'
                END 
            ENDIF 

                ; convert epoch
            epoch_cdf            = time_epoch(ra_t)
            epoch_cdf_10min      = time_epoch(ra_t_10min)
            epoch_cdf_1hr        = time_epoch(ra_t_1hr)
            epoch_cdf_1kev       = time_epoch(epoch_1kev)
            
                                ; put in CDFs
                                ; 1 min.
            out_dir = out_dir_base+'Spacecraft_'+sat+'/Level2/CDFs/1DMax/Non_valid/1min/'+$
                      string(ts_struct.year, format = '(I4.4)')+'/' 
            out_file = 'ST'+sat+'_L2_PLA_1DMax_1min_'+string(ts_struct.year, format = '(I4.4)')+$
                       string(ts_struct.month, format = '(I2.2)')+$
                       string(ts_struct.date, format = '(I2.2)')+'_'+version
            out_path = out_dir+out_file
            command = '$SKELETONCDF -cdf '+out_path+' -delete -fillval '+skeleton_name
            spawn, command
            temp_out_name = out_path+'.cdf'
            cdf_id = cdf_open(temp_out_name)
            date = systime()
            WHILE strlen(date) LT 32 DO BEGIN
                date = date+' '
            ENDWHILE 

                ; load into cdf
            cdf_varput, cdf_id, 'processing_date',     date,       /zvariable
            IF have_data EQ 1 AND empty_1min EQ 0 THEN BEGIN 
                attitude       = transpose(attitude)
                ew_source_flag = transpose(ew_source_flag)
                cdf_varput, cdf_id, 'epoch',                       epoch_cdf,      /zvariable
                cdf_varput, cdf_id, 'epoch_1kev',                  epoch_cdf_1kev, /zvariable
                cdf_varput, cdf_id, 'error',                       error,          /zvariable
                cdf_varput, cdf_id, 'caution',                     caution,        /zvariable
                cdf_varput, cdf_id, 'attitude_flag',               attitude,       /zvariable
                cdf_varput, cdf_id, 'ew_source_flag',              ew_source_flag, /zvariable
                cdf_varput, cdf_id, 'Reduced_Chi2',                red_chi_sq,     /zvariable
                cdf_varput, cdf_id, 'stat_uncertainty_angle',      uncert,         /zvariable
                cdf_varput, cdf_id, 'proton_number_density',       den,            /zvariable
                cdf_varput, cdf_id, 'proton_bulk_speed',           speed,          /zvariable
                cdf_varput, cdf_id, 'proton_temperature',          temp,           /zvariable
                cdf_varput, cdf_id, 'proton_thermal_speed',        v_therm,        /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_inst',  ns_inst,        /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_inst',  ew_inst,        /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_HERTN', ns_hertn,       /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_HERTN', ew_hertn,       /zvariable
                cdf_varput, cdf_id, 'proton_Vr_HERTN',             vr_hertn,       /zvariable
                cdf_varput, cdf_id, 'proton_Vt_HERTN',             vt_hertn,       /zvariable
                cdf_varput, cdf_id, 'proton_Vn_HERTN',             vn_hertn,       /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_RTN',   ns_rtn,         /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_RTN',   ew_rtn,         /zvariable
                cdf_varput, cdf_id, 'proton_Vr_RTN',               vr_rtn,         /zvariable
                cdf_varput, cdf_id, 'proton_Vt_RTN',               vt_rtn,         /zvariable
                cdf_varput, cdf_id, 'proton_Vn_RTN',               vn_rtn,         /zvariable
                cdf_varput, cdf_id, 'carrington_rotation',         carr_rot,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_carr',             lon_carr,       /zvariable
                cdf_varput, cdf_id, 'heliocentric_dist',           helio_dist,     /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_hee',              lon_hee,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_hee',              lat_hee,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_heeq',             lon_heeq,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_heeq',             lat_heeq,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_hci',              lon_hci,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_hci',              lat_hci,        /zvariable
            ENDIF 
            cdf_close, cdf_id
            print, "Finished ", out_file
                ; 10 min.
            out_dir = out_dir_base+'Spacecraft_'+sat+'/Level2/CDFs/1DMax/Non_valid/10min/'+$
                      string(ts_struct.year, format = '(I4.4)')+'/' 
            out_file = 'ST'+sat+'_L2_PLA_1DMax_10min_'+string(ts_struct.year, format = '(I4.4)')+$
                       string(ts_struct.month, format = '(I2.2)')+$
                       string(ts_struct.date, format = '(I2.2)')+'_'+version
            out_path = out_dir+out_file
            command = '$SKELETONCDF -cdf '+out_path+' -delete -fillval '+skeleton_name_10min
            spawn, command
            temp_out_name = out_path+'.cdf'
            cdf_id = cdf_open(temp_out_name)
            date = systime()
            WHILE strlen(date) LT 32 DO BEGIN
                date = date+' '
            ENDWHILE 
                ; load into cdf
            cdf_varput, cdf_id, 'processing_date_10min',     date,             /zvariable
            IF have_data_10min EQ 1 AND empty_10min EQ 0 THEN BEGIN 
                attitude_10min       = transpose(attitude_10min)
                ew_source_flag_10min = transpose(ew_source_flag_10min)
                cdf_varput, cdf_id, 'epoch_10min',                       epoch_cdf_10min,      /zvariable
                cdf_varput, cdf_id, 'error_10min',                       error_10min,          /zvariable
                cdf_varput, cdf_id, 'caution_10min',                     caution_10min,        /zvariable
                cdf_varput, cdf_id, 'attitude_flag_10min',               attitude_10min,       /zvariable
                cdf_varput, cdf_id, 'ew_source_flag_10min',              ew_source_flag_10min, /zvariable
                cdf_varput, cdf_id, 'Reduced_Chi2_10min',                red_chi_sq_10min,     /zvariable
                cdf_varput, cdf_id, 'stat_uncertainty_angle_10min',      uncert_10min,         /zvariable
                cdf_varput, cdf_id, 'proton_number_density_10min',       den_10min,            /zvariable
                cdf_varput, cdf_id, 'proton_bulk_speed_10min',           speed_10min,          /zvariable
                cdf_varput, cdf_id, 'proton_temperature_10min',          temp_10min,           /zvariable
                cdf_varput, cdf_id, 'proton_thermal_speed_10min',        v_therm_10min,        /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_inst_10min',  ns_inst_10min,        /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_inst_10min',  ew_inst_10min,        /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_HERTN_10min', ns_hertn_10min,       /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_HERTN_10min', ew_hertn_10min,       /zvariable
                cdf_varput, cdf_id, 'proton_Vr_HERTN_10min',             vr_hertn_10min,       /zvariable
                cdf_varput, cdf_id, 'proton_Vt_HERTN_10min',             vt_hertn_10min,       /zvariable
                cdf_varput, cdf_id, 'proton_Vn_HERTN_10min',             vn_hertn_10min,       /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_RTN_10min',   ns_rtn_10min,         /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_RTN_10min',   ew_rtn_10min,         /zvariable
                cdf_varput, cdf_id, 'proton_Vr_RTN_10min',               vr_rtn_10min,         /zvariable
                cdf_varput, cdf_id, 'proton_Vt_RTN_10min',               vt_rtn_10min,         /zvariable
                cdf_varput, cdf_id, 'proton_Vn_RTN_10min',               vn_rtn_10min,         /zvariable
                cdf_varput, cdf_id, 'cycles_den_10min',                  cycles_10min,         /zvariable
                cdf_varput, cdf_id, 'cycles_speed_10min',                cycles_speed_10min,   /zvariable
                cdf_varput, cdf_id, 'cycles_ns_inst_10min',              cycles_ns_inst_10min, /zvariable
                cdf_varput, cdf_id, 'cycles_vel_10min',                  cycles_ew_10min,      /zvariable
                cdf_varput, cdf_id, 'carrington_rotation_10min',         carr_rot_10min,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_carr_10min',             lon_carr_10min,       /zvariable
                cdf_varput, cdf_id, 'heliocentric_dist_10min',           helio_dist_10min,     /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_hee_10min',              lon_hee_10min,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_hee_10min',              lat_hee_10min,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_heeq_10min',             lon_heeq_10min,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_heeq_10min',             lat_heeq_10min,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_hci_10min',              lon_hci_10min,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_hci_10min',              lat_hci_10min,        /zvariable
            ENDIF 
            cdf_close, cdf_id
            print, "Finished ", out_file
                ; 1 hr.
            out_dir = out_dir_base+'Spacecraft_'+sat+'/Level2/CDFs/1DMax/Non_valid/1hr/'+$
                      string(ts_struct.year, format = '(I4.4)')+'/' 
            out_file = 'ST'+sat+'_L2_PLA_1DMax_1hr_'+string(ts_struct.year, format = '(I4.4)')+$
                       string(ts_struct.month, format = '(I2.2)')+$
                       string(ts_struct.date, format = '(I2.2)')+'_'+version
            out_path = out_dir+out_file
            command = '$SKELETONCDF -cdf '+out_path+' -delete -fillval '+skeleton_name_1hr
            spawn, command
            temp_out_name = out_path+'.cdf'
            cdf_id = cdf_open(temp_out_name)
            date = systime()
            WHILE strlen(date) LT 32 DO BEGIN
                date = date+' '
            ENDWHILE 
                ; load into cdf
            cdf_varput, cdf_id, 'processing_date_1hr',     date,           /zvariable
            IF have_data_1hr EQ 1 AND empty_1hr EQ 0 THEN BEGIN 
                attitude_1hr       = transpose(attitude_1hr)
                ew_source_flag_1hr = transpose(ew_source_flag_1hr)
                cdf_varput, cdf_id, 'epoch_1hr',                       epoch_cdf_1hr,      /zvariable
                cdf_varput, cdf_id, 'error_1hr',                       error_1hr,          /zvariable
                cdf_varput, cdf_id, 'caution_1hr',                     caution_1hr,        /zvariable
                cdf_varput, cdf_id, 'attitude_flag_1hr',               attitude_1hr,       /zvariable
                cdf_varput, cdf_id, 'ew_source_flag_1hr',              ew_source_flag_1hr, /zvariable
                cdf_varput, cdf_id, 'Reduced_Chi2_1hr',                red_chi_sq_1hr,     /zvariable
                cdf_varput, cdf_id, 'stat_uncertainty_angle_1hr',      uncert_1hr,         /zvariable
                cdf_varput, cdf_id, 'proton_number_density_1hr',       den_1hr,            /zvariable
                cdf_varput, cdf_id, 'proton_bulk_speed_1hr',           speed_1hr,          /zvariable
                cdf_varput, cdf_id, 'proton_temperature_1hr',          temp_1hr,           /zvariable
                cdf_varput, cdf_id, 'proton_thermal_speed_1hr',        v_therm_1hr,        /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_inst_1hr',  ns_inst_1hr,        /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_inst_1hr',  ew_inst_1hr,        /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_HERTN_1hr', ns_hertn_1hr,       /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_HERTN_1hr', ew_hertn_1hr,       /zvariable
                cdf_varput, cdf_id, 'proton_Vr_HERTN_1hr',             vr_hertn_1hr,       /zvariable
                cdf_varput, cdf_id, 'proton_Vt_HERTN_1hr',             vt_hertn_1hr,       /zvariable
                cdf_varput, cdf_id, 'proton_Vn_HERTN_1hr',             vn_hertn_1hr,       /zvariable
                cdf_varput, cdf_id, 'proton_n_s_flow_angle_RTN_1hr',   ns_rtn_1hr,         /zvariable
                cdf_varput, cdf_id, 'proton_e_w_flow_angle_RTN_1hr',   ew_rtn_1hr,         /zvariable
                cdf_varput, cdf_id, 'proton_Vr_RTN_1hr',               vr_rtn_1hr,         /zvariable
                cdf_varput, cdf_id, 'proton_Vt_RTN_1hr',               vt_rtn_1hr,         /zvariable
                cdf_varput, cdf_id, 'proton_Vn_RTN_1hr',               vn_rtn_1hr,         /zvariable
                cdf_varput, cdf_id, 'cycles_den_1hr',                  cycles_1hr,         /zvariable
                cdf_varput, cdf_id, 'cycles_speed_1hr',                cycles_speed_1hr,   /zvariable
                cdf_varput, cdf_id, 'cycles_ns_inst_1hr',              cycles_ns_inst_1hr, /zvariable
                cdf_varput, cdf_id, 'cycles_vel_1hr',                  cycles_ew_1hr,      /zvariable
                cdf_varput, cdf_id, 'carrington_rotation_1hr',         carr_rot_1hr,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_carr_1hr',             lon_carr_1hr,       /zvariable
                cdf_varput, cdf_id, 'heliocentric_dist_1hr',           helio_dist_1hr,     /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_hee_1hr',              lon_hee_1hr,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_hee_1hr',              lat_hee_1hr,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_heeq_1hr',             lon_heeq_1hr,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_heeq_1hr',             lat_heeq_1hr,       /zvariable
                cdf_varput, cdf_id, 'spcrft_lon_hci_1hr',              lon_hci_1hr,        /zvariable
                cdf_varput, cdf_id, 'spcrft_lat_hci_1hr',              lat_hci_1hr,        /zvariable
            ENDIF 
            cdf_close, cdf_id
            print, "Finished ", out_file

                ; write ascii files
                ; 1 min.
            IF empty_1min EQ 0 THEN BEGIN 
                time_i = 0
                IF have_data EQ 1 THEN BEGIN 
                    WHILE finite(ra_t[time_i]) EQ 0 DO BEGIN 
                        time_i = time_i + 1
                    ENDWHILE 
                ENDIF 
                label_time = time_struct(ra_t(time_i))
                label_year = STRING(label_time.year, FORMAT = '(I04)')
                label_month = STRING(label_time.month, FORMAT = '(I02)')
                label_day = STRING(label_time.date, FORMAT = '(I02)')
                    ; write daily file
                outfile = ascii_out_dir_base+'/1min/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+$
                          'ST'+sat+'_L2_PLA_1DMax_1min_'+$
                          label_year+label_month+label_day+'_'+string(label_time.doy, format = '(I3.3)')+'_'+version+'.txt'
                openw, out_lun, outfile, /GET_LUN 
                temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat
                printf, out_lun, temp_string
                create_1min_header, temp_string
                printf, out_lun, temp_string
                IF have_data EQ 1 THEN BEGIN 
                    update_1min, 1, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, ns_inst_st, ew_inst_st, $
                                 vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, vr_rtn_st, vt_rtn_st, vn_rtn_st, $
                                 ns_rtn_st, ew_rtn_st, error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                                 carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                                 out_lun, ts_struct.doy, ts_struct.year, sat
                ENDIF 
                close, out_lun
                free_lun, out_lun
            ENDIF 

                ; GSE
            IF want_gse EQ 1 THEN BEGIN
                outfile = ascii_out_dir_base+'/1min/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+'ST'+sat+$
                          '_L2_PLA_1DMax_1min_GSE_'+$
                          label_year+label_month+label_day+'_'+string(label_time.doy, format = '(I3.3)')+'_'+version+'.txt'
                openw, out_lun, outfile, /GET_LUN 
                temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat
                printf, out_lun, temp_string
                temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+'sec'+tab+'millisec'+tab+$
                              'date and time'+tab+'1 keV/q time'+tab+$
                              'Np [1/cc]'+tab+$
                              'Bulk Speed [km/s]'+tab+$
                              'Tkin [deg K]'+tab+$
                              'v_th [km/s]'+tab+$
                              'N/S Inst. [deg]'+tab+$
                              'E/W Inst. [deg]'+tab+$
                              'Vr GSE [km/s]'+tab+$
                              'Vt GSE [km/s]'+tab+$
                              'Vn GSE [km/s]'+tab+$
                              'N/S GSE [deg]'+tab+$
                              'E/W GSE [deg]'+tab+$
                              'Error Code'+tab+'Caution'+tab+$
                              'Attitude Flag (roll)'+tab+'Attitude Flag (yaw)'+tab+'Attitude Flag (pitch)'+tab+$
                              'E/W Source Flag'+tab+'E/W Missed Peak Flag'+tab+$
                              'Reduced Chi^2'+tab+'E/W Rel Stat Uncertainty (%)'+tab+$
                              'Carr.Rot'+tab+'Spcrft. Long. [Carr, degrees]'+tab+'Heliocentric Distance [km]'+tab+$
                              'Spcrft. Long. [HEE, degrees]'+tab+'Spcrft. Lat. [HEE, degrees]'+tab+$
                              'Spcrft. Long. [HEEQ, degrees]'+tab+'Spcrft. Lat. [HEEQ, degrees]'+tab+$
                              'Spcrft. Long. [HCI, degrees]'+tab+'Spcrft. Lat. [HCI, degrees]'
                printf, out_lun, temp_string
                IF have_data EQ 1 THEN BEGIN 
                    ra_t_struct = time_struct(ra_t)
                    ra_t_string = time_string(ra_t)
                    kev_string = time_string(epoch_1kev)
                    FOR kk = 0L, N_ELEMENTS(ra_t) - 1 DO BEGIN
                        temp_string = string(ra_t_struct[kk].year, format = '(I4.4)')+tab+$
                                      string(ra_t_struct[kk].doy,  format = '(I3.3)')+tab+$
                                      string(ra_t_struct[kk].hour, format = '(I2.2)')+tab+$
                                      string(ra_t_struct[kk].min,  format = '(I2.2)')+tab+$
                                      string(ra_t_struct[kk].sec,  format = '(I2.2)')+tab+$
                                      string(ra_t_struct[kk].fsec, format = '(F0.3)')+tab+$
                                      ra_t_string[kk]+tab+$
                                      kev_string [kk]+tab+$
                                      den_st     [kk]+tab+$
                                      speed_st   [kk]+tab+$ 
                                      temp_st    [kk]+tab+$
                                      v_therm_st [kk]+tab+$
                                      ns_inst_st [kk]+tab+$
                                      ew_inst_st [kk]+tab+$
                                      vr_gse_st  [kk]+tab+$
                                      vt_gse_st  [kk]+tab+$
                                      vn_gse_st  [kk]+tab+$
                                      ns_gse_st  [kk]+tab+$
                                      ew_gse_st  [kk]+tab+$
                                      string(error         [kk   ])+tab+string(caution       [kk   ])+tab+$
                                      string(attitude      [0, kk])+tab+string(attitude      [1, kk])+tab+string(attitude[2, kk])+tab+$
                                      string(ew_source_flag[0, kk])+tab+string(ew_source_flag[1, kk])+tab+$ 
                                      red_chi_sq_st[kk]+tab+uncert_st[kk]+tab+$
                                      string(carr_rot  [kk], format = '(F10.4)')+tab+string(lon_carr[kk],   format = '(F10.4)')+tab+$
                                      string(helio_dist[kk], format = '(E12.5)')+tab+$
                                      string(lon_hee   [kk], format = '(F10.4)')+tab+string(lat_hee [kk],   format = '(F10.4)')+tab+$
                                      string(lon_heeq  [kk], format = '(F10.4)')+tab+string(lat_heeq[kk],   format = '(F10.4)')+tab+$
                                      string(lon_hci   [kk], format = '(F10.4)')+tab+string(lat_hci [kk],   format = '(F10.4)') 
                        printf, out_lun, temp_string
                    ENDFOR
                ENDIF 
                close, out_lun
                free_lun, out_lun
            ENDIF 
        ENDIF 
    ENDIF                   

        ; update 10 min and 1 hr (and 1min monthly) even if have no data
        label_year = STRING(ts_struct.year, FORMAT = '(I04)')
        label_month = STRING(ts_struct.month, FORMAT = '(I02)')
        label_day = STRING(ts_struct.date, FORMAT = '(I02)')
        ; 1 min update monthly files
        outfile = ascii_out_dir_base+'/1min/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+$
                  'ST'+sat+'_L2_PLA_1DMax_1min_'+$
                  label_year+label_month+'_'+version+'.txt'
        CASE ts_struct.month OF 
            1: temp_doy = 1
            2: temp_doy = 32
            3: temp_doy = 60
            4: temp_doy = 91
            5: temp_doy = 121
            6: temp_doy = 152
            7: temp_doy = 182
            8: temp_doy = 213
            9: temp_doy = 244
            10: temp_doy = 274
            11: temp_doy = 305
            12: temp_doy = 335
        ENDCASE 
        IF (ts_struct.year EQ 2008 OR ts_struct.year EQ 2012 OR ts_struct.year EQ 2016 OR ts_struct.year EQ 2020) $
          AND ts_struct.month GT 2 THEN temp_doy = temp_doy + 1
        found_file = file_test(outfile)
        IF found_file EQ 1 THEN BEGIN ; need to update
            temp_outfile = ascii_out_dir_base+'/1min/temp_'+sat+'_'+label_year+label_month+'.txt'
            IF file_test(temp_outfile) EQ 1 THEN stop ; could have file conflict
            openw, out_lun, temp_outfile, /GET_LUN
            openr, in_lun, outfile, /get_lun
            line = ''
            readf,  in_lun,  line ; two lines of header
            printf, out_lun, line
            readf,  in_lun,  line
            printf, out_lun, line
            found_place = 0
            WHILE found_place EQ 0 AND eof(in_lun) EQ 0 DO BEGIN
                readf, in_lun, line
                parts = strsplit(line, /extract)
                WHILE temp_doy LT parts[1] DO BEGIN 
                    update_1min, 0, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, $
                                 ns_inst_st, ew_inst_st, $
                                 vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, $
                                 vr_rtn_st, vt_rtn_st, vn_rtn_st, ns_rtn_st, ew_rtn_st, $
                                 error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                                 carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                                 out_lun, temp_doy, ts_struct.year, sat
                    temp_doy = temp_doy + 1
                ENDWHILE 
                IF parts[1] LT ts_struct.doy THEN BEGIN
                    printf, out_lun, line 
                    IF parts[2] EQ 0 AND parts[3] EQ 0 THEN temp_doy = temp_doy + 1
                ENDIF ELSE found_place = 1
            ENDWHILE 
            IF found_place EQ 0 THEN BEGIN 
                WHILE temp_doy LT ts_struct.doy DO BEGIN 
                    update_1min, 0, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, $
                                 ns_inst_st, ew_inst_st, $
                                 vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, $
                                 vr_rtn_st, vt_rtn_st, vn_rtn_st, ns_rtn_st, ew_rtn_st, $
                                 error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                                 carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                                 out_lun, temp_doy, ts_struct.year, sat
                    temp_doy = temp_doy + 1
                ENDWHILE 
            ENDIF ELSE temp_doy = ts_struct.doy
            update_1min, have_data, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, $
                         ns_inst_st, ew_inst_st, $
                         vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, $
                         vr_rtn_st, vt_rtn_st, vn_rtn_st, ns_rtn_st, ew_rtn_st, $
                         error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                         carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                         out_lun, ts_struct.doy, ts_struct.year, sat
            WHILE eof(in_lun) EQ 0 DO BEGIN 
                readf, in_lun, line
                parts = strsplit(line, /extract)
                IF parts[1] NE ts_struct[0].doy THEN printf, out_lun, line ; write later days
            ENDWHILE 
            close, in_lun
            close, out_lun
            free_lun, in_lun
            free_lun, out_lun
            command = 'mv '+temp_outfile+' '+outfile
            spawn, command
        ENDIF ELSE BEGIN        ; need to start new file
            openw, out_lun, outfile, /GET_LUN
            temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat
            printf, out_lun, temp_string
            create_1min_header, temp_string
            printf, out_lun, temp_string
            WHILE temp_doy LT ts_struct.doy DO BEGIN 
                update_1min, 0, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, ns_inst_st, ew_inst_st, $
                             vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, vr_rtn_st, vt_rtn_st, vn_rtn_st, $
                             ns_rtn_st, ew_rtn_st, error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                             carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                             out_lun, temp_doy, ts_struct.year, sat
                temp_doy = temp_doy + 1
            ENDWHILE 
            update_1min, have_data, ra_t, epoch_1kev, den_st, speed_st, temp_st, v_therm_st, ns_inst_st, ew_inst_st, $
                         vr_hertn_st, vt_hertn_st, vn_hertn_st, ns_hertn_st, ew_hertn_st, vr_rtn_st, vt_rtn_st, vn_rtn_st, $
                         ns_rtn_st, ew_rtn_st, error, caution, attitude, ew_source_flag_st, red_chi_sq_st, uncert_st, $
                         carr_rot, lon_carr, helio_dist, lon_hee, lat_hee, lon_heeq, lat_heeq, lon_hci, lat_hci, $
                         out_lun, ts_struct.doy, ts_struct.year, sat
            close, out_lun
            free_lun, out_lun
        ENDELSE 
        ; 10 min.
        ; write daily file
        outfile = ascii_out_dir_base+'/10min/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+$
                  'ST'+sat+'_L2_PLA_1DMax_10min_'+$
                  label_year+label_month+label_day+'_'+string(ts_struct.doy, format = '(I3.3)')+'_'+version+'.txt'
        openw, out_lun, outfile, /GET_LUN
        temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat+' 10 minute averages'
        printf, out_lun, temp_string
        create_10min_header, temp_string
        printf, out_lun, temp_string
        update_10min, have_data_10min,     ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                      v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                      vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                      vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                      cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                      error_10min,         caution_10min,      attitude_10min,       ew_source_flag_10min_st, $
                      red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                      lon_carr_10min,      helio_dist_10min, $
                      lon_hee_10min,       lat_hee_10min,    $
                      lon_heeq_10min,      lat_heeq_10min,   $
                      lon_hci_10min,       lat_hci_10min,    out_lun, ts_struct.doy, ts_struct.year, sat
        close, out_lun
        free_lun, out_lun
        ; update monthly files
        outfile = ascii_out_dir_base+'/10min/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+$
                  'ST'+sat+'_L2_PLA_1DMax_10min_'+$
                  label_year+label_month+'_'+version+'.txt'
        CASE ts_struct.month OF 
            1: temp_doy = 1
            2: temp_doy = 32
            3: temp_doy = 60
            4: temp_doy = 91
            5: temp_doy = 121
            6: temp_doy = 152
            7: temp_doy = 182
            8: temp_doy = 213
            9: temp_doy = 244
            10: temp_doy = 274
            11: temp_doy = 305
            12: temp_doy = 335
        ENDCASE 
        IF (ts_struct.year EQ 2008 OR ts_struct.year EQ 2012 OR ts_struct.year EQ 2016 OR ts_struct.year EQ 2020) $
          AND ts_struct.month GT 2 THEN temp_doy = temp_doy + 1
        found_file = file_test(outfile)
        IF found_file EQ 1 THEN BEGIN ; need to update
            temp_outfile = ascii_out_dir_base+'/10min/temp_'+sat+'_'+label_year+label_month+'.txt'
            IF file_test(temp_outfile) EQ 1 THEN stop ; could have file conflict
            openw, out_lun, temp_outfile, /GET_LUN
            openr, in_lun, outfile, /get_lun
            line = ''
            readf,  in_lun,  line ; two lines of header
            printf, out_lun, line
            readf,  in_lun,  line
            printf, out_lun, line
            found_place = 0
            WHILE found_place EQ 0 AND eof(in_lun) EQ 0 DO BEGIN
                readf, in_lun, line
                parts = strsplit(line, /extract)
                WHILE temp_doy LT parts[1] DO BEGIN 
                    update_10min, 0,                   ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                                  v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                                  vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                                  vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                                  cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                                  error_10min,         caution_10min,      attitude_10min,       ew_source_flag_10min_st, $
                                  red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                                  lon_carr_10min,      helio_dist_10min, $
                                  lon_hee_10min,       lat_hee_10min,    $
                                  lon_heeq_10min,      lat_heeq_10min,   $
                                  lon_hci_10min,       lat_hci_10min,    out_lun, temp_doy, ts_struct.year, sat
                    temp_doy = temp_doy + 1
                ENDWHILE 
                IF parts[1] LT ts_struct.doy THEN BEGIN
                    printf, out_lun, line 
                    IF parts[2] EQ 0 AND parts[3] EQ 0 THEN temp_doy = temp_doy + 1
                ENDIF ELSE found_place = 1
            ENDWHILE
            IF found_place EQ 0 THEN BEGIN 
                WHILE temp_doy LT ts_struct.doy DO BEGIN 
                    update_10min, 0,                   ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                                  v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                                  vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                                  vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                                  cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                                  error_10min,         caution_10min,      attitude_10min,       ew_source_flag_10min_st, $
                                  red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                                  lon_carr_10min,      helio_dist_10min, $
                                  lon_hee_10min,       lat_hee_10min,    $
                                  lon_heeq_10min,      lat_heeq_10min,   $
                                  lon_hci_10min,       lat_hci_10min,    out_lun, temp_doy, ts_struct.year, sat
                    temp_doy = temp_doy + 1
                ENDWHILE 
            ENDIF ELSE temp_doy = ts_struct.doy
            update_10min, have_data_10min,     ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                          v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                          vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                          vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                          cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                          error_10min,         caution_10min,      attitude_10min,       ew_source_flag_10min_st, $
                          red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                          lon_carr_10min,      helio_dist_10min, $
                          lon_hee_10min,       lat_hee_10min,    $
                          lon_heeq_10min,      lat_heeq_10min,   $
                          lon_hci_10min,       lat_hci_10min,    out_lun, ts_struct.doy, ts_struct.year, sat
            WHILE eof(in_lun) EQ 0 DO BEGIN 
                readf, in_lun, line
                parts = strsplit(line, /extract)
                IF parts[1] NE ts_struct[0].doy THEN printf, out_lun, line ; write later days
            ENDWHILE 
            close, in_lun
            close, out_lun
            free_lun, in_lun
            free_lun, out_lun
            command = 'mv '+temp_outfile+' '+outfile
            spawn, command
        ENDIF ELSE BEGIN        ; need to start new file
            openw, out_lun, outfile, /GET_LUN
            temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat+' 10 minute averages'
            printf, out_lun, temp_string
            create_10min_header, temp_string
            printf, out_lun, temp_string
            WHILE temp_doy LT ts_struct.doy DO BEGIN 
                update_10min, 0,                   ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                              v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                              vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                              vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                              cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                              error_10min,         caution_10min,      attitude_10min,       ew_source_flag_10min_st, $
                              red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                              lon_carr_10min,      helio_dist_10min, $
                              lon_hee_10min,       lat_hee_10min,    $
                              lon_heeq_10min,      lat_heeq_10min,   $
                              lon_hci_10min,       lat_hci_10min,    out_lun, temp_doy, ts_struct.year, sat
                temp_doy = temp_doy + 1
            ENDWHILE 
            update_10min, have_data_10min,     ra_t_10min,         den_10min_st,         speed_10min_st,    temp_10min_st,     $
                          v_therm_10min_st,                                              ns_inst_10min_st,  ew_inst_10min_st,  $ 
                          vr_hertn_10min_st,   vt_hertn_10min_st,  vn_hertn_10min_st,    ns_hertn_10min_st, ew_hertn_10min_st, $
                          vr_rtn_10min_st,     vt_rtn_10min_st,    vn_rtn_10min_st,      ns_rtn_10min_st,   ew_rtn_10min_st,   $
                          cycles_10min,        cycles_speed_10min, cycles_ns_inst_10min, cycles_ew_10min_st,      $
                          error_10min,         caution_10min,      attitude_10min,       ew_source_flag_10min_st, $
                          red_chi_sq_10min_st, uncert_10min_st,    carr_rot_10min,                             $
                          lon_carr_10min,      helio_dist_10min, $
                          lon_hee_10min,       lat_hee_10min,    $
                          lon_heeq_10min,      lat_heeq_10min,   $
                          lon_hci_10min,       lat_hci_10min,    out_lun, ts_struct.doy, ts_struct.year, sat
            close, out_lun
            free_lun, out_lun
        ENDELSE 
        ; 1 hr.
        label_year = STRING(ts_struct.year, FORMAT = '(I04)')
        label_month = STRING(ts_struct.month, FORMAT = '(I02)')
        label_day = STRING(ts_struct.date, FORMAT = '(I02)')
        ; write daily file
        outfile = ascii_out_dir_base+'/1hr/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+$
                  'ST'+sat+'_L2_PLA_1DMax_1hr_'+$
                  label_year+label_month+label_day+'_'+string(ts_struct.doy, format = '(I3.3)')+'_'+version+'.txt'
        openw, out_lun, outfile, /GET_LUN
        temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat+' 1 hour averages'
        printf, out_lun, temp_string
        create_1hr_header, temp_string
        printf, out_lun, temp_string
        update_1hr, have_data_1hr,     ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                    v_therm_1hr_st,                                              ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                    vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                    vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                    cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                    error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                    red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                    lon_carr_1hr,      helio_dist_1hr, $
                    lon_hee_1hr,       lat_hee_1hr,    $
                    lon_heeq_1hr,      lat_heeq_1hr,   $
                    lon_hci_1hr,       lat_hci_1hr,    out_lun, ts_struct.doy, ts_struct.year, sat
        close, out_lun
        free_lun, out_lun
        ; update yearly files
        outfile = ascii_out_dir_base+'/1hr/'+sat+'/'+string(ts_struct.year, format = '(I4.4)')+'/'+$
                  'ST'+sat+'_L2_PLA_1DMax_1hr_'+$
                  label_year+'_'+version+'.txt'
        temp_doy = 1 ; start at beginning of year
        found_file = file_test(outfile)
        IF found_file EQ 1 THEN BEGIN ; need up update
            temp_outfile = ascii_out_dir_base+'/1hr/temp_'+sat+'_'+label_year+'.txt'
            print, temp_outfile
            IF file_test(temp_outfile) EQ 1 THEN stop ; could have file conflict
            openw, out_lun, temp_outfile, /GET_LUN
            openr, in_lun, outfile, /get_lun
            line = ''
            readf,  in_lun,  line ; two lines of header
            printf, out_lun, line
            readf,  in_lun,  line
            printf, out_lun, line
            found_place = 0
            WHILE found_place EQ 0 AND eof(in_lun) EQ 0 DO BEGIN
                readf, in_lun, line
                parts = strsplit(line, /extract)
                WHILE temp_doy LT parts[1] DO BEGIN 
                    update_1hr, 0,                 ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                                v_therm_1hr_st,                                              ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                                vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                                vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                                cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                                error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                                red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                                lon_carr_1hr,      helio_dist_1hr, $
                                lon_hee_1hr,       lat_hee_1hr,    $
                                lon_heeq_1hr,      lat_heeq_1hr,   $
                                lon_hci_1hr,       lat_hci_1hr,    out_lun, temp_doy, ts_struct.year, sat
                    temp_doy = temp_doy + 1
                ENDWHILE 
                IF parts[1] LT ts_struct[0].doy THEN BEGIN
                    printf, out_lun, line 
                    IF parts[2] EQ 0 THEN temp_doy = temp_doy + 1
                ENDIF ELSE found_place = 1
            ENDWHILE
            IF found_place EQ 0 THEN BEGIN 
                WHILE temp_doy LT ts_struct.doy DO BEGIN 
                    update_1hr, 0,                 ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                                v_therm_1hr_st,                                              ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                                vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                                vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                                cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                                error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                                red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                                lon_carr_1hr,      helio_dist_1hr, $
                                lon_hee_1hr,       lat_hee_1hr,    $
                                lon_heeq_1hr,      lat_heeq_1hr,   $
                                lon_hci_1hr,       lat_hci_1hr,    out_lun, temp_doy, ts_struct.year, sat
                    temp_doy = temp_doy + 1
                ENDWHILE 
            ENDIF ELSE temp_doy = ts_struct.doy
            update_1hr, have_data_1hr,     ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                        v_therm_1hr_st,                                              ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                        vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                        vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                        cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                        error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                        red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                        lon_carr_1hr,      helio_dist_1hr, $
                        lon_hee_1hr,       lat_hee_1hr,    $
                        lon_heeq_1hr,      lat_heeq_1hr,   $
                        lon_hci_1hr,       lat_hci_1hr,    out_lun, ts_struct.doy, ts_struct.year, sat
            WHILE eof(in_lun) EQ 0 DO BEGIN 
                readf, in_lun, line
                parts = strsplit(line, /extract)
                IF parts[1] NE ts_struct[0].doy THEN printf, out_lun, line ; write later days
            ENDWHILE 
            close, in_lun
            close, out_lun
            free_lun, in_lun
            free_lun, out_lun
            command = 'mv '+temp_outfile+' '+outfile
            spawn, command
        ENDIF ELSE BEGIN        ; need to start new file
            openw, out_lun, outfile, /GET_LUN
            temp_string = '1D Maxwellian Proton Moments for Spacecraft '+sat+' 1 hour averages'
            printf, out_lun, temp_string
            create_1hr_header, temp_string
            printf, out_lun, temp_string
            WHILE temp_doy LT ts_struct.doy DO BEGIN 
                update_1hr, 0,                 ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                            v_therm_1hr_st,                                              ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                            vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                            vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                            cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                            error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                            red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                            lon_carr_1hr,      helio_dist_1hr, $
                            lon_hee_1hr,       lat_hee_1hr,    $
                            lon_heeq_1hr,      lat_heeq_1hr,   $
                            lon_hci_1hr,       lat_hci_1hr,    out_lun, temp_doy, ts_struct.year, sat
                temp_doy = temp_doy + 1
            ENDWHILE 
            update_1hr, have_data_1hr,     ra_t_1hr,         den_1hr_st,         speed_1hr_st,    temp_1hr_st,     $
                        v_therm_1hr_st,                                          ns_inst_1hr_st,  ew_inst_1hr_st,  $ 
                        vr_hertn_1hr_st,   vt_hertn_1hr_st,  vn_hertn_1hr_st,    ns_hertn_1hr_st, ew_hertn_1hr_st, $
                        vr_rtn_1hr_st,     vt_rtn_1hr_st,    vn_rtn_1hr_st,      ns_rtn_1hr_st,   ew_rtn_1hr_st,   $
                        cycles_1hr,        cycles_speed_1hr, cycles_ns_inst_1hr, cycles_ew_1hr_st,      $
                        error_1hr,         caution_1hr,      attitude_1hr,       ew_source_flag_1hr_st, $
                        red_chi_sq_1hr_st, uncert_1hr_st,    carr_rot_1hr,                             $
                        lon_carr_1hr,      helio_dist_1hr, $
                        lon_hee_1hr,       lat_hee_1hr,    $
                        lon_heeq_1hr,      lat_heeq_1hr,   $
                        lon_hci_1hr,       lat_hci_1hr,    out_lun, ts_struct.doy, ts_struct.year, sat
            close, out_lun
            free_lun, out_lun
        ENDELSE 

        tplot_names, '*', names = names
        IF n_elements(names) GT 0 THEN store_data, names, /DEL
        ts = ts + 86400.
    ENDFOR
ENDFOR 
print, "Done"
toc
END 
