; Lorna Ellis
; 05/13/2010
; run_ave.pro

; This procedure takes a running average of the input data.

; This assumes that the time dimension is first.

; inpu   t:  data    : the data array, or a string to the tplot name
;            num_ave : the number of points to average (must be odd)
;            weights : array of weights (must have same # of elements as num_ave)
;                    : ex: [.25, .5, .25] -- should add to 1
;
; output  : data     : the modified data array
;
; keywords: accum_min    : if set, then run accumulation instead of average. 
;                      In this case, weights are meaningless

; 06/08/2010 LBE Added accum_min keyword.
; 02/04/2010 LBE Check if array too small in run_ave_time.
; 01/06/2011 LBE Change LT to LE in first line of run_ave_time.
; 02/09/2012 LBE Fix run_ave_time.
; 06/17/2014 LBE Fix bug at ends of arrays (when NaN)
; 02/05/2018 LBE Fix bug in run_ave_time when some bad points in very small array.

; sub-routine for the internal average
PRO run_ave_time, data, new_data, half_num, num_ave, weights, num_records
compile_opt strictarrsubs
IF num_records LT num_ave THEN BEGIN ; array very small
    FOR ii = 0, num_records-1 DO BEGIN ; have missing data
        bad_i = where(finite(data) EQ 0, bad_count, $
                      complement = good_i, ncomplement = good_count)
        IF good_count EQ 0 THEN new_data[ii] = !values.f_nan ELSE BEGIN 
            temp_data = 0
            good_i = good_i + (num_ave/2) - ii ; set to middle
            IF ii             GT num_ave/2   THEN start_point = ii - (num_ave/2) ELSE start_point = 0
            IF (num_ave/2)+ii LT good_count THEN stop_point  = (num_ave/2) + ii ELSE stop_point  = good_count-1
            good_i = good_i[start_point:stop_point] ; truncate
            good_weight  = total(weights[good_i])
            temp_weights = weights[good_i]
            FOR weight_i = 0, stop_point-start_point DO BEGIN
                IF finite(data[weight_i]) EQ 1 THEN BEGIN
                    my_weight = temp_weights[weight_i] / good_weight
                    temp_data = temp_data + (data[weight_i] * my_weight)
                ENDIF 
            ENDFOR 
            new_data[ii] = temp_data
        ENDELSE 
    ENDFOR 
ENDIF ELSE BEGIN 
; beginning of array
    FOR ii = 0, half_num-1 DO BEGIN ; have missing data
        bad_i = where(finite(data[0:ii+half_num]) EQ 0, bad_count, $
                      complement = good_i, ncomplement = good_count)
        IF good_count EQ 0 THEN new_data[ii] = !values.f_nan ELSE BEGIN 
            temp_data = 0
            good_weight = total(weights[good_i+half_num-ii]) 
            FOR weight_i = 0, num_ave-1 DO BEGIN
                IF ii-half_num+weight_i GE 0 THEN BEGIN 
                    IF finite(data[ii-half_num+weight_i]) EQ 1 THEN BEGIN
                        my_weight = weights[weight_i] / good_weight
                        temp_data = temp_data + (data[ii-half_num+weight_i] * my_weight)
                        ;print, weight_i, weights[weight_i], good_weight, my_weight, temp_data, data[ii-half_num+weight_i]
                    ENDIF
                ENDIF 
            ENDFOR 
            ;stop
            new_data[ii] = temp_data
        ENDELSE 
    ENDFOR 
; middle of array
    FOR ii = half_num, num_records-half_num-1 DO BEGIN 
        IF array_equal(finite(data[ii-half_num:ii+half_num]), 1) EQ 1 THEN BEGIN 
            temp_data = 0
            FOR weight_i = 0, num_ave-1 DO BEGIN
                temp_data = temp_data + (data[ii-half_num+weight_i] * weights[weight_i])
            ENDFOR
            new_data[ii] = temp_data
        ENDIF ELSE BEGIN        ; missing data
            bad_i = where(finite(data[ii-half_num:ii+half_num]) EQ 0, bad_count, $
                          complement = good_i, ncomplement = good_count)
            IF good_count EQ 0 THEN new_data[ii] = !values.f_nan ELSE BEGIN 
                temp_data = 0
                good_weight = total(weights[good_i])
                FOR weight_i = 0, num_ave-1 DO BEGIN
                    IF finite(data[ii-half_num+weight_i]) EQ 1 THEN BEGIN
                        my_weight = weights[weight_i] / good_weight
                        temp_data = temp_data + (data[ii-half_num+weight_i] * my_weight)
                    ENDIF
                ENDFOR 
                new_data[ii] = temp_data
            ENDELSE 
        ENDELSE 
    ENDFOR 
; end of array
    FOR ii = num_records-half_num, num_records-1 DO BEGIN ; have missing data
        bad_i = where(finite(data[ii-half_num:num_records-1]) EQ 0, bad_count, $
                      complement = good_i, ncomplement = good_count)
        IF good_count EQ 0 THEN new_data[ii] = !values.f_nan ELSE BEGIN 
            temp_data = 0
            good_weight = total(weights[good_i])
            FOR weight_i = 0, num_ave-1 DO BEGIN
                IF ii-half_num+weight_i LT num_records THEN BEGIN 
                    IF finite(data[ii-half_num+weight_i]) EQ 1 THEN BEGIN
                        my_weight = weights[weight_i] / good_weight
                        temp_data = temp_data + (data[ii-half_num+weight_i] * my_weight)
                    ENDIF
                ENDIF 
            ENDFOR 
            new_data[ii] = temp_data
        ENDELSE 
    ENDFOR 
ENDELSE 
END 

; sub-routine for the internal accumulation
PRO run_accum_time, data, new_data, half_num, num_ave, weights, num_records
FOR ii = 0L, num_records-1 DO BEGIN 
    IF finite(data[ii]) EQ 0 OR data[ii] EQ 0 THEN new_data[ii] = !values.f_nan ELSE BEGIN 
        IF ii-half_num LT 0           THEN start_ii = 0             ELSE start_ii = ii-half_num
        IF ii+half_num GE num_records THEN stop_ii  = num_records-1 ELSE stop_ii  = ii+half_num
        new_data[ii] = total(data[start_ii:stop_ii], /nan)
    ENDELSE 
ENDFOR 
END 


; main program
PRO run_ave, data, num_ave, weights, accum_min = accum_min
compile_opt strictarrsubs

type = size(data, /type)
IF type EQ 7 THEN BEGIN ; given tplot_name
    data_name = data
    get_data, data_name, time, data
ENDIF 

size = size(data)

IF keyword_set(accum_min) EQ 0 THEN BEGIN 
; check weight array
    IF n_elements(weights) NE num_ave THEN BEGIN 
        print, 'invalid weights array in run_ave'
        stop
    ENDIF 
    IF num_ave MOD 2 NE 1 THEN stop ; should be odd
    IF abs(total(weights)-1.0) GT .1 THEN stop ; should add to 1
ENDIF 

num_records = size[1] ; # of elements in time dimension
new_data    = data ; for size
new_data[*] = !values.f_nan
half_num    = num_ave / 2      ; num elements in each direction
; one-dim array
IF size[0] EQ 1 THEN BEGIN 
    run_ave_time, data, new_data, half_num, num_ave, weights, num_records
ENDIF ELSE IF size[0] EQ 2 THEN BEGIN ; 2-dim array
    FOR jj = 0, size[2]-1 DO BEGIN 
        partial_data        = reform(data[*, jj])
        new_partial_data    = partial_data ; for size
        new_partial_data[*] = !values.f_nan
        IF keyword_set(accum_min) EQ 1 THEN run_accum_time, partial_data, new_partial_data, half_num, num_ave, weights, num_records $
          ELSE run_ave_time, partial_data, new_partial_data, half_num, num_ave, weights, num_records
        new_data[*, jj]     = new_partial_data
    ENDFOR 
ENDIF ELSE IF size[0] EQ 3 THEN BEGIN ; 3-dim array
    FOR jj = 0, size[2]-1 DO BEGIN 
        print, 'run_ave ', jj, size[2]
        FOR kk = 0, size[3]-1 DO BEGIN 
            partial_data        = reform(data[*, jj, kk])
            new_partial_data    = partial_data ; for size
            new_partial_data[*] = !values.f_nan
            IF keyword_set(accum_min) EQ 1 THEN run_accum_time, partial_data, new_partial_data, half_num, num_ave, weights, num_records $
              ELSE run_ave_time, partial_data, new_partial_data, half_num, num_ave, weights, num_records
            new_data[*, jj, kk] = new_partial_data
        ENDFOR 
    ENDFOR 
ENDIF ELSE IF size[0] EQ 4 THEN BEGIN ; 4-dim array
    FOR jj = 0, size[2]-1 DO BEGIN 
        print, 'run_ave ', jj, size[2]
        FOR kk = 0, size[3]-1 DO BEGIN 
            FOR ll = 0, size[4]-1 DO BEGIN 
                partial_data            = reform(data[*, jj, kk, ll])
                new_partial_data        = partial_data ; for size
                new_partial_data[*]     = !values.f_nan
                IF keyword_set(accum_min) THEN run_accum_time, partial_data, new_partial_data, half_num, num_ave, weights, num_records $
                  ELSE run_ave_time, partial_data, new_partial_data, half_num, num_ave, weights, num_records
                new_data[*, jj, kk, ll] = new_partial_data
            ENDFOR 
        ENDFOR 
    ENDFOR 
ENDIF ELSE BEGIN
    print, 'data array has too many dimensions. Ask Lorna to add some to program.'
    stop
ENDELSE 
data = new_data

IF type EQ 7 THEN BEGIN ; given tplot_name
    store_data, data_name, data = {x:time, y:data}
    data = data_name
ENDIF 

accum_min = 0

END 
