;+
; PROCEDURE: average_tplot_variable
;
; PURPOSE: to time average any tplot variable
;
; INPUT:
;        var -> variable name - Array of strings or numbers. The '*'
;                               is also accepted.
;        avg -> averaging time in seconds
;
; KEYWORD:
;         NEW_NAME   -> the averaged variable is stored with a new name that
;                       includes the avereging time
;         START_TIME -> 'yyyy-mm-dd/hh:mm:ss', if set, use as start time
;
; CREATED by: C. Mouikis
;
; LAST MODIFIED: 09/25/01
;
; MODIFICATION HISTORY:
;   09/25/01 - The time step loop index is declared as a long.
;   10/17/07 - The averaged time grid is fixed and the empty bins take
;              the NaN value
;   03/27/08 - Bug fixed. The averaged arrays were two elements short.
;   09/10/08 - Changed loop variable to long. (LBE)
;   10/21/08 - Bug fixed. The V array in the 2D case was not properly
;              defined (CGM)
;   04/30/09 - The routine can handle up to four dimensions (of course
;              the averaging is done in the first (time) dimension
;              only) (CGM)
;   06/03/10 - Added start_time keyword (LBE)
;   06/04/10 - Added num_avg (LBE)
;   12/01/10 - Add check for time=0 in 1 dim array.
;   10/10/11 - Fix bug when start_time if after first time (LBE)
;   09/27/18 - Change format for new_name from 5 to 8 (LBE)
;-
PRO average_tplot_variable, var, avg, NEW_NAME=NEW_NAME, CLUSTER = CLUSTER, START_TIME = start_time
compile_opt strictarrsubs
  
IF avg LE 0 THEN BEGIN
    PRINT, 'the avg time must be > 0'
    RETURN
ENDIF

;--------------------------------------------------------------------
; Find the names of the variables to be averaged
;--------------------------------------------------------------------
tplot_names, var, NAMES = var_names
var_id = WHERE(var_names NE '', c_var_id)
IF c_var_id GT 0 THEN BEGIN
    var_names = var_names(var_id)
ENDIF ELSE BEGIN
    PRINT, 'No tplot variable with such name/number exists'
    RETURN
ENDELSE

;--------------------------------------------------------------------
; Loop through all variables
;--------------------------------------------------------------------
FOR iv = 0, N_ELEMENTS(var_names)-1 DO BEGIN
    
    ;------------------------------------------------------------------
    ; Already averaged variables are excluded
    ;------------------------------------------------------------------
    IF STRPOS(var_names(iv), '_AVG') NE -1 THEN BEGIN
        PRINT, 'The variable: ' + var_names(iv) + $
               ' is already averaged. Time average the original variable' 
        RETURN
    ENDIF
    
    ;------------------------------------------------------------------
    ; Extract specie and s/c information from the variable name
    ;------------------------------------------------------------------
    IF STRPOS(var_names(iv), '_SP') NE -1 THEN BEGIN
        specie = STRMID(var_names(iv), STRPOS(var_names(iv), '_SP') + 3, 1)
    ENDIF
    sat    = STRMID(var_names(iv), STRPOS(var_names(iv), '_SC') + 3, 1)

    ;------------------------------------------------------------------
    ; Extract data information from tplot variable
    ;------------------------------------------------------------------
    get_data, var_names(iv), data = data, dlim = dlim, lim = lim
    
    time = data.x
    yray = data.y
    str_element, data, 'V', value, SUCCESS = vfound
    IF vfound EQ 1 THEN vray = data.v

    IF keyword_set(start_time) EQ 1 THEN BEGIN
        wanted_i = where(time GE time_double(start_time), wanted_count)
        IF wanted_count GT 0 THEN BEGIN 
            time = time[wanted_i]
            size = size(yray)
            CASE size[0] OF 
                1: BEGIN 
                    yray = yray[wanted_i]
                    temp_yray = fltarr(1)
                END
                2: BEGIN 
                    yray = yray[wanted_i, *]
                    temp_yray = fltarr(1, size[2]) 
                END 
                3: BEGIN 
                    yray = yray[wanted_i, *, *]
                    temp_yray = fltarr(1, size[2], size[3]) 
                END
                4: BEGIN 
                    yray = yray[wanted_i, *, *, *]
                    temp_yray = fltarr(1, size[2], size[3], size[4]) 
                END 
                ELSE: BEGIN 
                    print, 'ERROR: Too many dimensions'
                    stop
                END 
            ENDCASE 
            temp_yray[*] = !values.f_nan
            yray = [temp_yray, yray]
        ENDIF 
        time = [time_double(start_time), time]
    ENDIF 
    yray_size = size(yray)
    dim_numb = yray_size(0)    

    ;------------------------------------------------------------------
    ; Filter out time tags with infinite values
    ; !!! This only works for 1-D data arrays !!!
    ;------------------------------------------------------------------
    IF dim_numb EQ 1 THEN BEGIN
        finite_time = FINITE(time)
        fnan = WHERE((finite_time EQ 1) AND (time GT 0), ifnan)
        IF ifnan GT 0 THEN BEGIN
            time = time(fnan)
            yray = yray(fnan)
        ENDIF
    ENDIF

    ;------------------------------------------------------------------
    ; Set variables/arrays
    ;------------------------------------------------------------------
    specie_str = ['H!U+!N', 'He!U++!N', 'He!U+!N', 'O!U+!N']
    
    dt = time(N_ELEMENTS(time)-1) - time(0) ; time interval in seconds
    n_avg = CEIL(dt/avg) 
    no_data = 0
    IF n_avg GT 0 THEN BEGIN 
        time_avg = DBLARR(n_avg)
        IF dim_numb EQ 1 THEN BEGIN
            yray_avg = DBLARR(n_avg)
        ENDIF ELSE BEGIN
            IF dim_numb EQ 2 THEN BEGIN
                yray_avg = DBLARR(n_avg, yray_size(2))
                IF vfound EQ 1 THEN vray_avg = FLTARR(n_avg, yray_size(2))
            ENDIF ELSE BEGIN
                IF dim_numb EQ 3 THEN BEGIN
                    yray_avg = DBLARR(n_avg, yray_size(2), yray_size(3))
                    IF vfound EQ 1 THEN vray_avg = FLTARR(n_avg, yray_size(2), yray_size(3))
                ENDIF ELSE BEGIN
                    IF dim_numb EQ 4 THEN BEGIN
                        yray_avg = DBLARR(n_avg, yray_size(2), yray_size(3), yray_size(4))
                        IF vfound EQ 1 THEN vray_avg = FLTARR(n_avg, yray_size(2), yray_size(3), yray_size(4))
                    ENDIF ELSE BEGIN
                        PRINT, 'Data array has more than four dimensions'
                        RETURN
                    ENDELSE
                ENDELSE
            ENDELSE
        ENDELSE
        num_avg = yray_avg      ; for size
    ENDIF ELSE no_data = 1

    ;------------------------------------------------------------------
    ; Time average
    ;------------------------------------------------------------------
    jj = 0l
    FOR ii = 0l, n_avg - 1 DO BEGIN
        IF KEYWORD_SET(CLUSTER) THEN BEGIN
            ; This is how I did the time averaging
            ; in the CCAT software (CM)
            av_ind = WHERE((time - time(0)) GE (ii*avg) AND $
                           (time - time(0)) LT ((ii+1)*avg), av_cnt)
            IF av_cnt GT 0 THEN BEGIN ; if there are measurements within 
                                ; an AVG sec interval
                time_avg(jj)   = TOTAL(time(av_ind))   / av_cnt
                IF dim_numb EQ 1 THEN BEGIN
                    yray_avg(jj) = TOTAL(yray(av_ind)) / av_cnt
                ENDIF ELSE BEGIN
                    yray_avg(jj, *) = TOTAL(yray(av_ind, *), 1) / av_cnt
                ENDELSE
                jj = jj + 1
            ENDIF
        ENDIF ELSE BEGIN
            ; Here new time tags are centered and
            ; empty bins are retained (CM)
            ; Empty bins are filled with NaN value
            av_ind = WHERE((time - time(0)) GE (ii*avg) AND $
                           (time - time(0)) LT ((ii+1)*avg), av_cnt)
            time_avg(jj) = time(0) + (ii+0.5)*avg
            IF av_cnt GT 0 THEN BEGIN ; if there are measurements within 
                                ; an AVG sec interval
                IF dim_numb EQ 1 THEN BEGIN
                    yray_avg(jj) = MEAN(yray(av_ind), /NaN)
                    good_i = where(finite(yray[av_ind]) EQ 1, good_count)
                    num_avg[jj] = good_count
                ENDIF ELSE BEGIN
                    IF dim_numb EQ 2 THEN BEGIN
                        FOR ij = 0L, N_ELEMENTS(yray(0, *))-1 DO BEGIN
                            yray_avg(jj, ij) = MEAN(yray(av_ind, ij), /NaN)
                            good_i = where(finite(yray[av_ind, ij]) EQ 1, good_count)
                            num_avg[jj, ij] = good_count
                            IF vfound EQ 1 THEN vray_avg(jj, ij) = vray(0, ij)
                        ENDFOR
                    ENDIF ELSE BEGIN
                        IF dim_numb EQ 3 THEN BEGIN
                            FOR ij = 0L, N_ELEMENTS(yray(0, *))-1 DO BEGIN
                                FOR ik = 0L, N_ELEMENTS(yray(0, 0, *))-1 DO BEGIN
                                    yray_avg(jj, ij, ik) = MEAN(yray(av_ind, ij, ik), /NaN)
                                    good_i = where(finite(yray[av_ind, ij, ik]) EQ 1, good_count)
                                    num_avg[jj, ij, ik] = good_count
                                    IF vfound EQ 1 THEN vray_avg(jj, ij, ik) = vray(0, ij, ik)
                                ENDFOR
                            ENDFOR
                        ENDIF ELSE BEGIN
                            IF dim_numb EQ 4 THEN BEGIN
                                FOR ij = 0L, N_ELEMENTS(yray(0, *))-1 DO BEGIN
                                    FOR ik = 0L, N_ELEMENTS(yray(0, 0, *))-1 DO BEGIN
                                        FOR il = 0L, N_ELEMENTS(yray(0, 0, 0, *))-1 DO BEGIN
                                            yray_avg(jj, ij, ik, il) = MEAN(yray(av_ind, ij, ik, il), /NaN)
                                            good_i = where(finite(yray[av_ind, ij, ik, il]) EQ 1, good_count)
                                            num_avg[jj, ij, ik, il] = good_count
                                            IF vfound EQ 1 THEN vray_avg(jj, ij, ik, il) = vray(0, ij, ik, il)
                                        ENDFOR
                                    ENDFOR
                                ENDFOR
                            ENDIF
                        ENDELSE
                    ENDELSE
                ENDELSE
            ENDIF ELSE BEGIN
                IF dim_numb EQ 1 THEN BEGIN
                    yray_avg(jj) = !VALUES.F_NAN
                ENDIF ELSE BEGIN
                    yray_avg(jj, *) = !VALUES.F_NAN
                ENDELSE
            ENDELSE
            jj = jj + 1
        ENDELSE
    ENDFOR
    
    IF no_data EQ 0 THEN BEGIN 
        time = time_avg(0:jj-1)
        IF dim_numb EQ 1 THEN BEGIN
            yray = yray_avg(0:jj-1)
        ENDIF ELSE BEGIN
            IF dim_numb EQ 2 THEN BEGIN
                yray = yray_avg(0:jj-1, *)
                IF vfound EQ 1 THEN vray = vray_avg(0:jj-1, *)
            ENDIF ELSE BEGIN
                IF dim_numb EQ 3 THEN BEGIN
                    yray = yray_avg(0:jj-1, *, *)
                    IF vfound EQ 1 THEN vray = vray_avg(0:jj-1, *, *)
                ENDIF ELSE BEGIN
                    IF dim_numb EQ 4 THEN BEGIN
                        yray = yray_avg(0:jj-1, *, *, *)
                        IF vfound EQ 1 THEN vray = vray_avg(0:jj-1, *, *, *)
                    ENDIF
                ENDELSE
            ENDELSE
        ENDELSE
    ENDIF 

    ;------------------------------------------------------------------
    ; If keyword NEW_NAME is set the time averaged variable is stored
    ; with a new name that includes the averaging time
    ;------------------------------------------------------------------
    IF KEYWORD_SET(NEW_NAME) THEN BEGIN
      new_var_name = var_names(iv) + '_AVG' + $
        STRCOMPRESS(STRING(avg, format='(i8)'), /REMOVE_ALL)
    ENDIF ELSE BEGIN
      new_var_name = var_names(iv)
    ENDELSE
    
    IF dim_numb EQ 1 THEN BEGIN
      datastr = {x:time, y:yray}
    ENDIF ELSE BEGIN
      IF vfound EQ 1 THEN BEGIN
        datastr = {x:time, y:yray, v:vray}
      ENDIF ELSE BEGIN
        datastr = {x:time, y:yray}
      ENDELSE
    ENDELSE

    IF no_data EQ 0 THEN BEGIN 
        store_data, new_var_name, data = datastr, dlim = dlim, lim = lim

    ; Add tplot_variable for num_avg
        num_avg_name = new_var_name+'_num_avg'
        store_data, num_avg_name, data = {x:time, y:num_avg}
    ENDIF ELSE store_data, new_var_name, /delete

    
    ;------------------------------------------------------------------
    ; Set plot attributes for new variable names
    ;------------------------------------------------------------------
    IF KEYWORD_SET(NEW_NAME) THEN BEGIN
      
      IF STRPOS(var_names(iv), '_UN') NE -1 THEN BEGIN
        dumstr = STRMID(var_names(iv), $
                        strpos(var_names(iv), '_UN')+3,$
                        strlen(var_names(iv))-1)
        units_name = STRMID(dumstr, 0, strpos(dumstr, '_'))
        CASE STRUPCASE(units_name) OF
          'COUNTS': uname = 'COUNTS'
          'BCOUNTS': uname = '1/tof_bin'
          'NCOUNTS': uname = '1/bin'
          'RATE': uname = '1/s'
          'NRATE': uname = '1/s-bin'
          'EFLUX': uname = 'eV/cm!E2!N-s-sr-eV'
          'DIFFFLUX': uname = '1/cm!E2!N-s-sr-(eV/e)'
          'DISTFUNC': uname = 's!E3!N/cm!E3!N-km!E3!N'
        ENDCASE
        
      ENDIF
      
      IF STRMID(var_names(iv),0,6) EQ 'ENSPEC' THEN BEGIN
        options, new_var_name, 'spec',1
        options, new_var_name, 'x_no_interp',1
        options, new_var_name, 'y_no_interp',1
        options, new_var_name, 'ytitle', $
          'SC' + string(sat, format='(i1.1)') + '!C!C' + $
          specie_str(specie)+' (eV)'
        options, new_var_name, 'ztitle', uname
        ylim,    new_var_name,  20., 4.5e4, 1
        
;        zlim, '*SP0*', 1.e0,  1.e3, 1
;        zlim, '*SP1*', 1.e-1, 1.e2, 1
;        zlim, '*SP2*', 1.e-2, 1.e1, 1
;        zlim, '*SP3*', 1.e-1, 1.e1, 1
      ENDIF
      
      IF STRMID(var_names(iv),0,6) EQ 'PASPEC' THEN BEGIN
        options, new_var_name, 'spec',1
        options, new_var_name, 'x_no_interp',1
        options, new_var_name, 'y_no_interp',1
        options, new_var_name, 'ytitle', $
          'SC' + string(sat, format='(i1.1)') + '!C!C' + $
          specie_str(specie)+' (deg)'
        options, new_var_name, 'ztitle', uname
        ylim,    new_var_name,  0., 360., 0
        
;        zlim, '*SP0*', 1.e0,  1.e3, 1
;        zlim, '*SP1*', 1.e-1, 1.e2, 1
;        zlim, '*SP2*', 1.e-2, 1.e1, 1
;        zlim, '*SP3*', 1.e-1, 1.e1, 1
      ENDIF
      
      IF STRMID(var_names(iv),0,3) EQ 'MAG' THEN BEGIN
        ylim, new_var_name,  0., 0., 0
      ENDIF
      
    ENDIF
    
    next:
  ENDFOR
END
  
