;+
;PROCEDURE: stack_plot
;PURPOSE:
;  Calculate proton distribution functions and make stack plots.
;
;PARAMETERS:   in:  sat: 'A' or 'B'
;		    bkgd: 1 or 2 
;                   files : Array of file paths to read from (strings)
;                   which_epoch: 1 (for 1 minute resolution)
;		    time:  starting date and time
;		    table_location:  where efficiency and ESA tables are found
; 		    output_dir:  where to write the output files
;				
;
;CREATED BY: K. Simunac
;
;LAST MODIFICATION: 06/10/2011
;
;MODIFICATION HISTORY:
;    May 2009:  created
;	October 2010: updated to agree with revised version of calc_proton_bulk_v2
;	June 2011: again updated to match calc_proton_bulk_v2
;       10/27/2011 LBE: no longer halt when no data
;       08/29/2014 LBE: create plot when no data
;       08/29/2014 LBE: change doubles to floats
;-


PRO stack_plot, sat, files, which_epoch, time, num_days, table_location, output_dir
compile_opt strictarrsubs

COMMON share1_high_res
COMMON com_high_res
tab = string(9B)
tplot_options, 'color', 0

num_records1 = final_count ; final count of number of records from pl_read_cdf_high_res
have_data = 1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;							  ;
; Below is a routine to calculate density, bulk speed,	  ;
; temperature and thermal speed using full	 	  ;
; resolution rate data from RA_Trigger.		  	  ;
;	 						  ;					 	  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; As of 31 March, 2008 ESA range is set to steps 64 to 127
; Highest proton speed is about 900 km/second (with some thermal spread)
; For table of step, E/Q, and proton velocity see ESA_table.txt


; Geometric Factors for Main Channel and S Channel (full SWS)

; Reto's Values
;---------------------------------
;A_G_Main = 3.15E-7 ;m^2 sr eV/eV 
;A_G_Sch = 6.0E-11 
;B_G_Main = 2.52E-7 ;m^2 sr eV/eV
;B_G_Sch = 6.3E-11

;Kristin's Values using Lynn's method
;-------------------------------------
; G = Active Area*Angular Extent of Slit*Bandwidth/2
; For example, for the main channel of FM1:
; 	Active Area = 8.9E-5 m^2
;	Deflection FWHM = 1.9 degrees = 0.033 radians
;	Azimuth Acceptance = 45 degrees = 0.785 radians
;	Energy Bandwidth = 6.12%
;	G = (8.9E-5)*(0.033)*(0.785)*0.0612/2 = 7.1E-8 m^2 sr eV/eV

A_G_Main = 7.1E-8 
A_G_Sch = 2.3E-11 
B_G_Main = 6.0E-8
B_G_Sch = 1.7E-11 


; Active areas -- not for calculations, but here for reference
;----------------------------------------------------------------
; S Channel value is divided by ratio of deflection bin size to
; instantaneous field of view (though not for geometric factor
; calculation above):  
;
; 1.33 degrees / 0.37 degrees = 3.6 for A
; 1.33 degrees / 0.27 degrees = 4.9 for B

 A_Area_Main = 8.9E-1 ; cm^2
 A_Area_Sch = 1.41E-3/3.6 ;cm^2
 B_Area_Main = 7.5E-1 ; cm^2
 B_Area_Sch = 1.50E-3/4.9 ;cm^2

; Dead times for SF0 and RA_TRIG
;--------------------------------
A_SF0_dead_time = 5.0E-6 ;sec
A_RA_TRIG_dead_time = 2.0E-6 ;sec
B_SF0_dead_time = 5.0E-6	;sec
B_RA_TRIG_dead_time = 2.0E-6	;sec ; CHANGED ON 02 APRIL, 2007 FROM 2.5E-6


;; Read in tables of estimated efficiencies
;; The location of the efficiency tables must be specified in the crib file.

IF sat EQ 'A' THEN eff_file = table_location+'leakage_A_2010_05_07.txt'
IF sat EQ 'B' THEN eff_file = table_location+'leakage_B_2011_05_17.txt' 
eff = fltarr(128)
openr, in_lun_1, eff_file, /GET_LUN
line = ","
readf, in_lun_1, line ; read the header line
FOR i = 64, 127 DO BEGIN
	readf, in_lun_1, line
	parts = STRSPLIT(line, ",", /EXTRACT)
	eff[i]=parts[1]
ENDFOR
close, in_lun_1
free_lun, in_lun_1

print, 'Efficiency tables have been read.'

;---------------------------------------------------------
IF sat EQ 'A' THEN BEGIN
   ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_a_full, /nan) LE 0 AND TOTAL(ra_trig_a_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        have_data = 0
    ENDIF ELSE BEGIN 
        s_chan = intarr(num_records1)
        error1 = intarr(num_records1)
        rate_name = ' '
        std_error = intarr(num_records1) ; A matrix in which to store possible errors
        std_error(*) = 0

   ;; Check to see if there are RA_Trig counts.  If so, continue.  
   ;; If not, assume the full resolution rate is SF0.
   ;;
   ;; RA_Trig was set as the full resolution rate on 14 February, 2007 for A
   ;; and on 1 March, 2007 for B.

        IF TOTAL(ra_trig_a_full, /nan) GT 0 THEN BEGIN
            full_data = ra_trig_a_full
            ra_trig_a_full = 0
            rate_name = '_RA_Trig'
            dead_time = A_RA_trig_dead_time
            G_Main = A_G_Main
            G_Sch = A_G_Sch
            s_chan = s_chan1_a
            error1 = error1_a
            error1_a = 0
        ENDIF ELSE BEGIN
            full_data = sf0_a_full
            sf0_a_full = 0
            rate_name = '_SF0'
            dead_time = A_SF0_dead_time
            G_Main = A_G_Main
            G_Sch = A_G_Sch
            s_chan = s_chan1_a
            error1 = error1_a
            error1_a = 0
        ENDELSE
    ENDELSE 
ENDIF ELSE BEGIN ; data is from spacecraft B
    ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_b_full, /nan) LE 0 AND TOTAL(ra_trig_b_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        have_data = 0
    ENDIF ELSE BEGIN 
        full_data = intarr(128, 32, num_records1) ; changed from float to integer to save memory 28 Sept, 2007
        rate_name = ' '
        std_error = intarr(num_records1) ; A matrix in which to store possible errors
        std_error(*) = 0

    ;; Check to see if there are RA_Trig counts.  If so, continue.  
    ;; If not, assum SF0 is the full resolution rate.
    ;;
    ;; RA_Trig was set as the full resolution rate on 14 February, 2007 on A
    ;; and on 1 March, 2007 on B.

        IF TOTAL(ra_trig_b_full, /nan) GT 0 THEN BEGIN
            full_data = ra_trig_b_full
            ra_trig_b_full = 0	
            rate_name = '_RA_Trig'
            dead_time = B_RA_trig_dead_time
            G_Main = B_G_Main
            G_Sch = A_G_Sch
            s_chan = s_chan1_b
            error1 = error1_b
            error1_b = 0
        ENDIF ELSE BEGIN
            full_data = sf0_b_full
            sf0_b_full = 0
            rate_name = '_SF0'
            dead_time = B_SF0_dead_time
            G_Main = B_G_Main
            G_Sch = B_G_Sch
            s_chan = s_chan1_b
            error1 = error1_b
            error1_b = 0
        ENDELSE
    ENDELSE 
ENDELSE

;; Get the time associated with the first file to label output files
start_year = ' '
start_year = strmid(time, 0, 4)
start_month = ' '
start_month = strmid(time, 5, 2)
start_day = ' '
start_day = strmid(time, 8, 2)

IF have_data EQ 1 THEN BEGIN 
    print, 'Starting calculations.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove the fill value of -1 from the files.
    remove_fill_high_res, -1, full_data
    print, 'Removed fill values.'

;; Check for value indicating overflow.  If present, mark record with error 3.
    FOR m = 0L, num_records1 - 1 DO BEGIN
        FOR k = 0, 127 DO BEGIN
            FOR l = 0, 31 DO BEGIN
                IF full_data(k, l, m) GE 507904 THEN BEGIN
                    full_data(k, l, m) = !values.f_nan
                    error1(m) = 3
                    PRINT, 'Overflow value detected.  Time, ESA, Defl:', m, k, l
                ENDIF
            ENDFOR
        ENDFOR
    ENDFOR
    print, 'Maximum Counts:', max(full_data, /NAN)
    IF max(full_data) GT 32000 THEN STOP

;;---------------------------------------------------------------------------------
;; Subtract out different background level for each ESA step -- added 31 August, 2007
;; to see if this removes contamination from WAP
    back_ground = fltarr(128, num_records1) ; changed from integer to float (8 May, 2008)
    back_ground(*, *) = 0.0
    delta_bkgd = fltarr(128,num_records1)
    delta_bkgd[*,*] = !values.f_nan
    full_data_no_bkgd = fltarr(128, 32, num_records1) ; changed from integer to float (8 May, 2008)
    full_data_no_bkgd[*, *, *] = 0
    delta_counts = fltarr(128,32,num_records1)
    delta_counts(*,*,*) = !values.f_nan
    delta_counts = SQRT(full_data)
    delta_counts_less_bkgd = fltarr(128,32,num_records1)
    delta_counts_less_bkgd(*,*,*) = !values.f_nan
    FOR mm = 0L, num_records1-1 DO BEGIN
        FOR nn = 0, 127 DO BEGIN
            back_ground[nn, mm] = ((full_data(nn, 31, mm)+full_data(nn, 0, mm))/2.0)
	    delta_bkgd[nn,mm] = 0.5*SQRT(full_data(nn, 31, mm)+full_data(nn, 0, mm))
            IF (back_ground(nn, mm)) LT 1 THEN back_ground(nn, mm) = 1.0
	; Subtract out background
            full_data_no_bkgd[nn, *, mm] = full_data(nn, *, mm)- back_ground(nn, mm)
	    delta_counts_less_bkgd[nn,*,mm]=SQRT(delta_counts(nn,*,mm)^2.0+delta_bkgd[nn,mm]^2.0)
        ENDFOR
    ENDFOR
    full_data = 0               ; reduce memory usage
    print, 'Background has been subtracted out.'


; Check to see if negative counts have resulted from background subtraction.
; If yes, set those values to zero.
    FOR yy = 0L, n_elements(full_data_no_bkgd)-1 DO BEGIN
        IF (full_data_no_bkgd(yy) LT 0) THEN full_data_no_bkgd(yy) = 0.0
    ENDFOR 

;; Correct for deflection wobble
;; -- an effect possibly due to an offset in deflector power supplies
    FOR j = 1, 31 DO BEGIN      ; DEFL Steps
        FOR i = 64, 126, 2 DO BEGIN ; Even ESA Steps only
            FOR k = 0L, num_records1-1 DO BEGIN ; Time Steps
                full_data_no_bkgd(i, j-1, k) = full_data_no_bkgd(i, j, k)
                delta_counts_less_bkgd(i, j-1, k) = delta_counts_less_bkgd(i, j, k)
                IF j EQ 31 THEN BEGIN
                    full_data_no_bkgd(i, j, k) = !values.f_nan
                    delta_counts_less_bkgd(i, j, k) = !values.f_nan
                ENDIF
            ENDFOR	
        ENDFOR
    ENDFOR



; Convert to Hz -- divide through by accumulation time of 12.8 milliseconds
;------------------------------------------------------------------------------
    full_data_hz = fltarr(128, 32, num_records1) ; changed from double to float to save memory 28 Sept, 2007
    full_data_hz[*, *, *] = !values.f_nan
    full_data_hz = FLOAT(full_data_no_bkgd)/0.0128 ; changed from double to float to save memory 28 Sept, 2007
    delta_counts_hz = fltarr(128, 32, num_records1)
    delta_counts_hz = !values.f_nan
    delta_counts_hz = delta_counts_less_bkgd/0.0128
    print, 'Converted to Hz.'
    
    full_data_no_bkgd = 0
    delta_counts_less_bkgd = 0 

; Correct for dead time -- real rate = recorded rate/(1-recorded rate*dead time)
    delta_counts_hz = delta_counts_hz/((1.0-(full_data_hz*dead_time))^2.0)
    full_data_hz = full_data_hz/(1.0-(full_data_hz*dead_time)) ; changed 1 to 1.0 13 Jan, 2009


;See if dead time correction results in negative rate.  If so, stop.
    FOR k = 0L, num_records1-1 DO BEGIN
        FOR i = 0, 127 DO BEGIN
            FOR j = 0, 31 DO BEGIN
                IF full_data_hz(i, j, k) LT 0 THEN BEGIN
                    PRINT, '(', i, j, k, ') Invalid countrate -- deadtime is too big.'
                    STOP
                ENDIF
            ENDFOR
        ENDFOR
    ENDFOR
    print, 'Dead time correction completed.'

; Divide by geometric factor to account for different apertures
;----------------------------------------------------------------
    full_data_hz = full_data_hz/G_Main ; Divide by G to get into to the right units
    delta_counts_hz = delta_counts_hz/G_Main
    FOR hh = 0L, num_records1-1 DO BEGIN
        IF sat EQ 'A' THEN F_Switch = 120.0 ; approximate ratio between main and s channel based on in-flight data
        IF sat EQ 'B' THEN F_Switch = 625.0
        FOR ii = 0, 127 DO BEGIN			
            IF (ii GE s_chan(hh)) THEN BEGIN ; if data taken with S channel then multiply to normalize with main channel
                full_data_hz(ii, *, hh) = full_data_hz(ii, *, hh)*F_Switch
                delta_counts_hz(ii, *, hh) = delta_counts_hz(ii, *, hh)*F_Switch
            ENDIF
        ENDFOR
    ENDFOR
    print, 'Division by geometric factor completed.'

; Sum over deflections steps for each ESA step.
    Defl_count = INTARR(num_records1)
    Defl_count(*) = 0

    peak_ESA = intarr(num_records1)
    peak_ESA(*) = -1
    sum_full_data = fltarr(num_records1, 128) ; changed from double to float to save memory
    sum_full_data(*, *) = 0.0
    sum_delta_counts = fltarr(num_records1, 128)
    sum_delta_counts(*, *) = 0.0

    FOR j = 0L, num_records1-1 DO BEGIN
        FOR k = 0, 127 DO BEGIN	
            temp_count = 0		
            FOR ll = 0, 31 DO BEGIN
                IF full_data_hz(k, ll, j) GT 0 THEN BEGIN		
                    sum_full_data[j, k] = sum_full_data(j, k)+(full_data_hz(k, ll, j))*(COS(-0.33816+0.02182*ll))
				; multiply by cosine of deflection angle to account for projection effect
                    sum_delta_counts[j, k] = SQRT(sum_delta_counts(j, k)^2.0+(delta_counts_hz(k, ll, j)*COS(-0.33816+0.02182*ll))^2.0)
                    Defl_count[j] = Defl_count(j)+1
                    temp_count = temp_count + 1
                ENDIF
            ENDFOR
            IF temp_count LE 0 THEN BEGIN
                sum_full_data[j, k] = 0.0
                sum_delta_counts[j, k] = 0.0
            ENDIF 
        ENDFOR		
        peak_count = 0
        FOR g = 0, 127 DO BEGIN
            IF (sum_full_data(j, g) GT peak_count) THEN peak_count = sum_full_data(j, g)
            IF (sum_full_data(j, g) GE peak_count) THEN peak_ESA(j) = g
        ENDFOR
    ENDFOR
    print, 'Finished summing over deflection steps.'

; Divide through by a flat efficiency
    IF sat EQ 'A' THEN BEGIN
        sum_full_data = sum_full_data/0.80
        sum_delta_counts = sum_delta_counts/0.80
    ENDIF
    IF sat EQ 'B' THEN BEGIN
        sum_full_data = sum_full_data/4.0
        sum_delta_counts = sum_delta_counts/4.0
    ENDIF

    print, 'Divison by efficiency completed.'

; Get E/Q and V associated with each ESA step so that distribution functions can be found
; The location of the ESA table must be specified in the crib file.
    IF sat EQ 'B' THEN ESA_file = table_location+'ESA_table_B_2010_06_25b.txt' 
    IF sat EQ 'A' THEN ESA_file = table_location+'ESA_table_A_2009_08_17.txt'

    openr, in_lun, ESA_file, /GET_LUN
    line = " "
    energy_S = fltarr(128)
    energy_S(*) = !values.f_nan
    v_S = fltarr(128)
    v_S(*) = !values.f_nan
    
    readf, in_lun, line         ; read past first lines
 
    FOR i = 64, 127 DO BEGIN
        readf, in_lun, line
        parts = STRSPLIT(line, " ,", /EXTRACT)
        energy_S[i] = parts[1]  ;keV/q
        v_S[i] = parts[2]	
    ENDFOR
	
    close, in_lun
    free_lun, in_lun
    print, 'Finished reading S channel ESA table.'


    IF sat EQ 'B' THEN ESA_file_main = table_location+'Main_channel_B_empirical_v2.txt' 
    IF sat EQ 'A' THEN ESA_file_main = table_location+'ESA_table_A_2009_08_17.txt'
    openr, in_lun, ESA_file_main, /GET_LUN
    line = " "
    energy_main = fltarr(128)
    energy_main(*) = !values.f_nan
    v_main = fltarr(128)
    v_main(*) = !values.f_nan

    readf, in_lun, line         ; read past first lines
    
    IF SAT EQ 'B' THEN BEGIN
        FOR i = 0, 127 DO BEGIN
            readf, in_lun, line
            parts = STRSPLIT(line, ",", /EXTRACT)
            energy_main[i] = parts[1]
            v_main[i] = parts[2]
        ENDFOR
    ENDIF ELSE BEGIN            ; sat is A
        FOR i = 64, 127 DO BEGIN
            readf, in_lun, line
            parts = STRSPLIT(line, " ,", /EXTRACT)
            energy_main[i] = parts[1] ;keV/q
            v_main[i] = parts[2]	
        ENDFOR
    ENDELSE
	
    close, in_lun
    free_lun, in_lun
    print, 'Finished reading main channel ESA table.'

; Multiply by another energy dependent "leakage efficiency"

    FOR ii = 0L, num_records1 - 1 DO BEGIN
        FOR jj = 64, 127 DO BEGIN
            IF jj GE s_chan(ii) THEN BEGIN 
                sum_full_data(ii, jj) = sum_full_data(ii, jj)*eff(jj)
                sum_delta_counts(ii, jj) = sum_delta_counts(ii, jj)*eff(jj)
            ENDIF
        ENDFOR
    ENDFOR


; Now multiply each ESA step by (delta theta*delta phi)/(v^2) to get distribution function
    dist_funct = fltarr(128, num_records1)
    dist_funct(*, *) = !values.d_nan
    peak_dist = intarr(num_records1)
    peak_dist(*) = -1
    delta_dist_funct = fltarr(128, num_records1)
    delta_dist_funct(*, *) = !values.d_nan
    
    FOR i = 0L, num_records1-1 DO BEGIN
        peak_count = 0.0
        FOR j = 0, 127 DO BEGIN
            IF sum_full_data(i, j) GT 0 THEN BEGIN
                IF j GE s_chan(i) THEN BEGIN
                    dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
                    delta_dist_funct(j, i) = (0.0218*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                                 ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
                ENDIF ELSE BEGIN
                    dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v_main(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
                    delta_dist_funct(j, i) = $
                      (0.0218*0.7854/((v_main(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                             ((2.0*sum_full_data(i, j)/v_main(j))^2.0)*(0.05*v_main(j))^2.0)
                ENDELSE
            ENDIF ELSE BEGIN
                dist_funct(j, i) = 0.0
                delta_dist_funct(j, i) = 0.0
            ENDELSE

; Average repeated ESA steps
            IF sat EQ 'B' THEN BEGIN
                dist_funct(122, i) = (0.5*(dist_funct(121, i)+dist_funct(123, i)))
                dist_funct(125, i) = (0.5*(dist_funct(124, i)+dist_funct(126, i)))
                delta_dist_funct(122, i) = (0.5*SQRT((delta_dist_funct(121, i))^2.0+(delta_dist_funct(123, i))^2.0))
                delta_dist_funct(125, i) = (0.5*SQRT((delta_dist_funct(124, i))^2.0+(delta_dist_funct(126, i))^2.0))
            ENDIF ELSE BEGIN
                dist_funct(116, i) = (0.5*(dist_funct(115, i)+dist_funct(117, i)))
                dist_funct(118, i) = (0.5*(dist_funct(117, i)+dist_funct(119, i)))
                dist_funct(123, i) = (0.5*(dist_funct(122, i)+dist_funct(124, i)))
                dist_funct(126, i) = (0.5*(dist_funct(125, i)+dist_funct(127, i)))
                delta_dist_funct(116, i) = (0.5*SQRT((delta_dist_funct(115, i))^2.0+(delta_dist_funct(117, i))^2.0))
                delta_dist_funct(118, i) = (0.5*SQRT((delta_dist_funct(117, i))^2.0+(delta_dist_funct(119, i))^2.0))
                delta_dist_funct(123, i) = (0.5*SQRT((delta_dist_funct(122, i))^2.0+(delta_dist_funct(124, i))^2.0))
                delta_dist_funct(126, i) = (0.5*SQRT((delta_dist_funct(125, i))^2.0+(delta_dist_funct(127, i))^2.0))
            ENDELSE

        
            IF (dist_funct(j, i) GT peak_count) THEN peak_count = dist_funct(j, i)
            IF (dist_funct(j, i) GE peak_count) THEN peak_dist(i) = j
        ENDFOR
    ENDFOR

    sum_full_data = 0           ; added 28 Sept, 2007 to use save memory
    sum_delta_counts = 0
ENDIF ELSE BEGIN ; dummy values for when no data
    num_records1     = 10087
    dist_funct       = fltarr(128, num_records1)
    s_chan           = intarr(     num_records1)
    dist_funct[*, *] = !values.f_nan
    rate_name        = '_RA_Trig' 
ENDELSE 

print, 'Making stack plot.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;Make stack plots.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
data_matrix = dist_funct
dummy_matrix = fltarr(128, num_records1)
dummy_matrix(*, *) = !values.f_nan

ThisDevice = !D.Name
SET_PLOT, 'PS'

; Here is the filename for the stack plot output..
plot_output_1 = output_dir+sat+'_'+start_year+'_'+start_month+'_'+start_day+'_v11_lin.ps'
plot_output_2 = output_dir+sat+'_'+start_year+'_'+start_month+'_'+start_day+'_v11_log.ps'

DEVICE, FILENAME = plot_output_1, XSIZE = 8.0, YSIZE = 10.5, /INCHES, $
        /COLOR, Bits_Per_Pixel = 8, PORTRAIT = 1, XOFFSET = 0.25, YOFFSET = 0.25

LOADCT, 39, NCOLORS = 256
;!P.BACKGROUND = 255
!P.MULTI = [1, 1, 1]

y_space = -1
y_label = ' '


IF num_records1 GT 1441 THEN BEGIN
    !Y.TICKNAME = ['0', '1', '2', '3', '4', '5', '6', '7'] ; this label for days
    y_space = 1440
    y_label = 'DAYS beginning on '+start_year+'-'+start_month+'-'+start_day
ENDIF ELSE BEGIN
    !Y.TICKNAME = ['0', '3', '6', '9', '12', '15', '18', '21', '24'] ; use this label for hours
    y_space = 180
    y_label = 'HOURS beginning on '+start_year+'-'+start_month+'-'+start_day
ENDELSE

sub_label = ' '
;IF sat EQ 'A' THEN BEGIN
;	IF rate_name EQ '_SF0' THEN BEGIN
;	sub_label = 'Eff_Table_1, Dead_Time = 5e-6 sec'
;	ENDIF ELSE BEGIN
;	sub_label = 'Eff_Table_4, Dead_Time = 2e-6 sec' ; table 4 is identical to 3, but multiplied by root 2 pi
;	ENDELSE
;ENDIF ELSE BEGIN
;	IF rate_name EQ '_SF0' THEN BEGIN
;	sub_label = 'Eff_Table_1, Dead_Time = 5e-6 sec'
;	ENDIF ELSE BEGIN
;	sub_label = 'Eff_Table_4, Dead_Time = 2e-6 sec' ; table 4 is really multiple tables varying with time
;	ENDELSE
;ENDELSE

switch_data = fltarr(128, num_records1)
switch_data(*, *) = !values.d_nan


FOR i = 0L, num_records1 - 1 DO BEGIN
    FOR j = 0, 127 DO BEGIN
        IF j LT 64 THEN data_matrix(j, i) = !values.d_nan
        IF (j EQ s_chan(i)) AND (s_chan(i) NE 0) THEN BEGIN
            switch_data(j, i) =  data_matrix(j, i) ; 0.01 
            IF switch_data(j, i) EQ 0 THEN switch_data(j, i) = !values.d_nan ;0.0001 ; to keep finite boundary in log plot
            IF j LT 127 THEN switch_data(j+1, i) = data_matrix(j+1, i) ; 0.01
            IF j LT 127 THEN BEGIN
                IF switch_data(j+1, i) EQ 0 THEN switch_data(j+1, i) = !values.d_nan ;0.0001
            ENDIF
			;IF j GT 64 THEN switch_data(j-1,i) = data_matrix(j-1,i)
			;IF j GT 64 AND switch_data(j-1,i) EQ 0 THEN switch_data(j-1,i) = 0.001
        ENDIF
    ENDFOR
    x = time_struct(epoch1(i))
    IF x.year EQ 1970 THEN BEGIN
        data_matrix(*, i) = !values.d_nan
        switch_data(*, i) = !values.d_nan
    ENDIF
ENDFOR

surface, data_matrix, xrange = [130, 60], yrange = [0, num_records1-1], zrange = [0, 1000], ytickinterval = y_space, $
         yminor = 3, xstyle = 1, ystyle = 1, zstyle = 1, zaxis = -1, ax = 75, az = 0, charsize = 2, $
         /horizontal, xtitle = 'ESA Step', ytitle = y_label, /save,  min_value = 0.0, xthick = 2.0, ythick = 2.0, charthick = 2.0

surface, switch_data, xrange = [130, 60], yrange = [0, num_records1-1], zrange = [0, 1000], ytickinterval = y_space, $
         yminor = 3, xstyle = 1, ystyle = 1, zstyle = 1, zaxis = -1, ax = 75, az = 0, charsize = 2, $
         /horizontal, min_value = 0.0, color = 250, /noerase, /T3D, xthick = 2.0, ythick = 2.0, charthick = 2.0 

CONTOUR, dummy_matrix, $        ;TITLE = sat+rate_name+' Full Resolution', subtitle = sub_label, $
         xrange = [130, 60], yrange = [0, num_records1-1], zrange = [0, 1000], ytickinterval = y_space, $
         yminor = 3, xstyle = 1, ystyle = 1, zstyle = 1, zaxis = -1, charsize = 2, xtitle = 'ESA Step', ytitle = y_label, $
         min_value = 0.0, color = 0, /noerase, /T3D, xthick = 2.0, ythick = 2.0, charthick = 2.0

contour, data_matrix, /nodata, title = sat+rate_name+' Full Resolution', subtitle = sub_label, $
         charsize = 2.0, xstyle = 4, ystyle = 4, zstyle = 4, charthick = 2.0, /noerase

DEVICE, /close_file
;SET_PLOT, ThisDevice

DEVICE, FILENAME = plot_output_2, XSIZE = 8.0, YSIZE = 10.5, /INCHES, $
        /COLOR, Bits_Per_Pixel = 8, PORTRAIT = 1, XOFFSET = 0.25, YOFFSET = 0.25

LOADCT, 39, NCOLORS = 256
;!P.BACKGROUND = 255
!P.MULTI = [1, 1, 1]

surface, data_matrix, xrange = [130, 60], yrange = [0, num_records1-1], zrange = [0.01, 1000], $
         ytickinterval = y_space, yminor = 3, xstyle = 1, ystyle = 1, zstyle = 1, zaxis = -1, ax = 75, az = 0, charsize = 2, /zlog, $
         /horizontal, /save, xtitle = 'ESA Step', ytitle = y_label, min_value = 0.01, xthick = 2.0, ythick = 2.0, charthick = 2.0

surface, switch_data, xrange = [130, 60], yrange = [0, num_records1-1], zrange = [0.01, 1000], $
         ytickinterval = y_space, yminor = 3, xstyle = 1, ystyle = 1, zstyle = 1, zaxis = -1, ax = 75, az = 0, charsize = 2, /zlog, $
         /horizontal, /noerase, /T3D, min_value = 0.01, COLOR = 250, xthick = 2.0, ythick = 2.0, charthick = 2.0

contour, dummy_matrix, /zlog, $ ;title=sat+rate_name+' Full Resolution', subtitle = sub_label, $
         xrange = [130, 60], yrange = [0, num_records1-1], zrange = [0.01, 1000], ytickinterval = y_space, $
         yminor = 3, xstyle = 1, ystyle = 1, zstyle = 1, zaxis = -1, charsize = 2, xtitle = 'ESA Step', ytitle = y_label, $
         color = 0, /noerase, /T3D, xthick = 2.0, ythick = 2.0, charthick = 2.0, min_value = 0.01
    
contour, data_matrix, /nodata, title = sat+rate_name+' Full Resolution', /zlog, charsize = 2.0, xstyle = 4, ystyle = 4, zstyle = 4, $
         subtitle = sub_label, charthick = 2.0, /noerase

DEVICE, /close_file
SET_PLOT, ThisDevice

outname = plot_output_1
temp_name = output_dir+sat+'_'+start_year+'_'+start_month+'_'+start_day+'_v11_lin'
command_string = 'convert -flatten -density 288 -geometry 25% -sharpen 3 ' + outname + ' ' +temp_name + '.png'
spawn, command_string
    
outname = plot_output_2
temp_name = output_dir+sat+'_'+start_year+'_'+start_month+'_'+start_day+'_v11_log'
command_string = 'convert -flatten -density 288 -geometry 25% -sharpen 3 ' + outname + ' '+temp_name + '.png'
spawn, command_string
    
print, 'Stack plot finished.'

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


END
