;+
;PROCEDURE: seek_files
;PURPOSE:
;  Find the files needed for the timespan and satellites given.
;
;PARAMETERS:   in:  sat:        'A' or 'B'
;                   file_type:  'science', 'hk', 'cl' (classifier), or 'sc' (spacecraft housekeeping)
;              out: files_a:    Array of filenames (and paths) needed for
;                               given timespan 
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;
;	November 2006:  K. Simunac modified version number and prefix for housekeeping files
;	3 Jan, 2007:  K. Simunac modifies to look for latest version
;	of file
;       9 Jan, 2007: L. Ellis fixed bug with looping
;	9 Jan, 2007: K. Simunac adds type for spacecraft housekeeping
;       1 Feb, 2007: C. Mouikis - An extra day was read when time
;                                interval ended in 00:00:00 time
;       26 Feb, 2007: C. Mouikis - the level1 data environment
;                                  variable points to the cdf directory
;       26 Feb, 2007: C. Mouikis - two level1 data environment variables
;                                  are intriduced; one for each S/C
;       06 Feb, 2008: L. Ellis - removed browsemom, changed l2mom to
;                                l2obmom
;       26 Feb, 2008: L. Ellis -- added l2mom
;       30 Jul, 2008: L. Ellis -- changed type_string for l2mom
;       21 Oct, 2008: L. Ellis -- removed doy from l2obmom filename
;       31 Oct, 2008: L. Ellis -- changed l2mom to l2_1dmax
;       02 Feb, 2009: L. Ellis -- added ace_swics
;       09 Mar, 2009: L. Ellis -- Changed 1dmax to valid and prelim
;       19 Mar, 2009: L. Ellis -- Added _1min to 1DMax filenames
;       25 Mar, 2009: L. Ellis -- updated l2_obmom
;       14 May, 2009: L. Ellis -- changed wind_pm to wind_3dp
;       30 Oct, 2009: L. Ellis -- added wind_orbit
;       17 Nov, 2009: L. Ellis -- added ace_swics_2hr
;       18 Nov, 2009: L. Ellis -- added ace_swics_q
;       19 Feb, 2010: L. Ellis -- added l2_alphas_prelim
;       14 Jul, 2010: L. Ellis -- added stereo_mag
;       23 Sep, 2010: L. Ellis -- added beacon
;       22 Oct, 2010: L. Ellis -- put Non_valid in OBMom path.
;       26 Oct, 2010: L. Ellis -- added l2_alpha_ra
;       04 Nov, 2010: L. Ellis -- Made obmom valid and prelim.
;       29 Nov, 2010: L. Ellis -- added l2_alpha_ra_10min
;       03 Aug, 2011: L. Ellis -- added swea
;       12 Oct, 2011: L. Ellis -- fix typo in swea A
;       21 Dec, 2011: L. Ellis -- add l2_1dmax_test
;       14 Apr, 2012: L. Ellis -- add l2_1dmax_test2
;       22 Aug, 2012: L. Ellis -- put browsemom back in
;       23 Aug, 2012: L. Ellis -- add beacon_mag
;       15 Apr, 2014: L. Ellis -- add l2_1dmax_5min
;       17 Mar, 2015: L. Ellis -- add ace_swics_v2_2hr
;       24 Mar, 2016: L. Ellis -- add l3_heplus_psd
;-
PRO seek_files, sat, file_type, files, browse = browse
compile_opt strictarrsubs

CASE file_type OF
    'l2_alpha_ra'      : type_string = 'Alpha_RA_1DMax'
    'l2_alpha_ra_10min': type_string = 'Alpha_RA_1DMax_10min'
    'l2_alphas_prelim' : type_string = 'Alphas_prelim'
    'l2_obmom_valid'   : type_string = 'OBMom_valid'
    'l2_obmom_prelim'  : type_string = 'OBMom_prelim'
    'l2_1dmax_valid'   : type_string = '1DMax_valid'
    'l2_1dmax_prelim'  : type_string = '1DMax_prelim'
    'l2_1dmax_test'    : type_string = '1DMax_test'
    'l2_1dmax_test2'   : type_string = '1DMax_test2'
    'l2_1dmax_5min'    : type_string = '1DMax_5min'
    'l3_heplus_psd'    : type_string = ''
    'science'          : type_string = ''
    'hk'               : type_string = 'HK_' ; K. Simunac removed underscore before HK
    'cl'               : type_string = 'CL_' ; K. Simunac removed underscore before CL
    'sc'               : type_string = 'SC_'
    'beacon'           : type_string = 'LB_'
    'browsemom'        : type_string = 'LB_'
    'beacon_mag'       : type_string = 'LB_'
    'beacon_raw'       : type_string = 'LB_'
    'ace_swics'        : type_string = ''
    'ace_swics_2hr'    : type_string = ''
    'ace_swics_v2_2hr' : type_string = '' ; for 2012 - 
    'ace_swics_q'      : type_string = ''
    'stereo_mag'       : type_string = ''
    'soho'             : type_string = ''
    'wind_3dp'         : type_string = ''
    'wind_orbit'       : type_string = ''
    'swea_mom'         : type_string = ''
    'swea_pad'         : type_string = ''
    ELSE : BEGIN 
        print, "invalid file type: ", file_type
        stop
    ENDELSE 
ENDCASE 

IF strcmp(sat, 'A') NE 1 AND strcmp(sat, 'B') NE 1 AND strcmp(sat, 'W') NE 1 AND strcmp(sat, 'A_S') NE 1 THEN BEGIN
    print, "sat must be 'A' or 'B'"
    stop
ENDIF 

get_timespan, times
times_struct = time_struct(times)

; Subtract a day if last day time is 00:00:00 - C. Mouikis
IF (times_struct[1].hour EQ 0) AND (times_struct[1].min) EQ 0 AND (times_struct[1].sec EQ 0) THEN BEGIN
    IF times_struct[1].doy GT 1 THEN BEGIN
        times_struct[1].doy = times_struct[1].doy - 1
    ENDIF ELSE BEGIN
        times_struct[1].year = times_struct[1].year - 1
        times_struct[1].doy = 365
        IF FIX(times_struct[1].year/4.) EQ (times_struct[1].year/4.) THEN times_struct[1].doy = 366
    ENDELSE
    
ENDIF

year = times_struct[0].year
doy = times_struct[0].doy
first = 1
WHILE (year LT times_struct[1].year) OR ((doy LE times_struct[1].doy) AND (year EQ times_struct[1].year)) DO BEGIN 
    doy_to_month_date, year, doy, month, date

    IF file_type EQ 'l2_alpha_ra' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_ALPHA_RA')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_ALPHA_RA')+'/'
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_'+type_string+'_1hr_'
        file_prefix = file_prefix+string(year, format = '(I4.4)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_alpha_ra_10min' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_ALPHA_RA_10MIN')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_ALPHA_RA_10MIN')+'/'
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_'+type_string+'_'
        file_prefix = file_prefix+string(year, format = '(I4.4)')+string(month, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_alphas_prelim' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_ALPHAS_PRELIM')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_ALPHAS_PRELIM')+'/'
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_Alphas_Pri_'
        IF sat EQ 'A' THEN file_prefix = file_prefix+'1hr_' ELSE file_prefix = file_prefix+'6hr_'
        file_prefix = file_prefix+string(year, format = '(I4.4)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_obmom_valid' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_OBMOM_VALID')+'/1min/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_OBMOM_VALID')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_OBMom_1min_valid_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_obmom_prelim' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_OBMOM_PRELIM')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_OBMOM_PRELIM')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_OBMom_1min_nonvalid_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_1dmax_valid' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_1DMAX_VALID')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_1DMAX_VALID')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_1DMax_1min_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_1dmax_prelim' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_1DMAX_PRELIM')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_1DMAX_PRELIM')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_1DMax_1min_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_1dmax_test' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_1DMAX_TEST')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_1DMAX_TEST')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_1DMax_1min_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_1dmax_test2' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_1DMAX_TEST2')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_1DMAX_TEST2')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_1DMax_1min_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l2_1dmax_5min' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L2_1DMAX_5MIN')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L2_1DMAX_5MIN')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L2_PLA_1DMax_5min_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'l3_heplus_psd' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L3_HePlus_PSD')+'/' $
        ELSE print, 'ERROR: No Data for STB'
        file_prefix = temp_dir+'ST'+sat+'_L3_PLA_HePlus_F_Vsw_24hr_'+string(year, format = '(I4.4)')+'0101_V'

    ENDIF ELSE IF file_type EQ 'beacon' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_LB_DATA') $
        ELSE temp_dir = getenv('STB_PLA_LB_DATA')
        temp_dir = temp_dir+'/'+string(year, format = '(I4.4)')+'/'+string(month, format = '(I2.2)')+'/'
        file_prefix = temp_dir+'ST'+sat+'_LB_PLASTIC_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'browsemom' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_LB_DATA') $
        ELSE temp_dir = getenv('STB_PLA_LB_DATA')
        temp_dir = temp_dir+'/'+string(year, format = '(I4.4)')+'/'+string(month, format = '(I2.2)')+'/'
        file_prefix = temp_dir+'ST'+sat+'_LB_PLA_BROWSE_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'beacon_raw' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_LB_RAW') $
        ELSE temp_dir = getenv('STB_PLA_LB_RAW')
        file_prefix = temp_dir+'/'+string(year, format = '(I4.4)')+'/ST'+sat+'_LB_PLA_Raw_MOM_1min_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'beacon_mag' THEN BEGIN

        IF sat EQ 'A' THEN temp_dir = getenv('STA_LB_IMPACT') $
        ELSE temp_dir = getenv('STB_LB_IMPACT')
        file_prefix = temp_dir+'/ST'+sat+'_LB_IMPACT_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V'

    ENDIF ELSE IF file_type EQ 'wind_3dp' THEN BEGIN 

        temp_dir = getenv('WINDDATA')+'/3dp/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'wi_pm_3dp_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_v'

    ENDIF ELSE IF file_type EQ 'wind_orbit' THEN BEGIN 
         
        temp_dir = getenv('WINDDATA')+'/orbit/'+'/' 
        file_prefix = temp_dir+'wi_or_pre_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_v'

    ENDIF ELSE IF file_type EQ 'ace_swics' THEN BEGIN 

        temp_dir = getenv('ACEDATA')+'/swics/'
        file_prefix = temp_dir+'ACE_SWICS_Data_'+string(year, format = '(I4.4)')+'.txt'

    ENDIF ELSE IF file_type EQ 'ace_swics_2hr' THEN BEGIN 

        temp_dir = getenv('ACEDATA')+'/swics/'
        file_prefix = temp_dir+'ACE_SWICS_Data_'+string(year, format = '(I4.4)')+'_2hr.txt'

    ENDIF ELSE IF file_type EQ 'ace_swics_v2_2hr' THEN BEGIN 

        temp_dir = getenv('ACEDATA')+'/swics/'
        file_prefix = temp_dir+'ACE_SWICS2_Data_2hr_'+string(year, format = '(I4.4)')+'.txt'

    ENDIF ELSE IF file_type EQ 'ace_swics_q' THEN BEGIN 

        temp_dir = getenv('ACEDATA')+'/swics/'
        file_prefix = temp_dir+'ACE_SWICS_QdistData_'+string(year, format = '(I4.4)')+'.txt'

    ENDIF ELSE IF file_type EQ 'stereo_mag' THEN BEGIN 

        IF sat EQ 'A' THEN temp_dir = getenv('STA_MAG_DATA')+'/' ELSE $
          temp_dir = getenv('STB_MAG_DATA')+'/'
        file_prefix = temp_dir+'ST'+sat+'_L1_MAG_Transformed_'+string(year, format = '(I4.4)')+'_V01.txt'

    ENDIF ELSE IF file_type EQ 'soho' THEN BEGIN 

        temp_dir = getenv('SOHODATA')+'/'
        file_prefix = temp_dir+'so_mtof_ele_'+type_string+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_v'

    ENDIF ELSE IF file_type EQ 'swea_mom' THEN BEGIN 

        IF sat EQ 'A' THEN temp_dir = getenv('STA_SWEA_MOM_DATA')+'/' ELSE $
          temp_dir = getenv('STB_SWEA_MOM_DATA')+'/'
        file_prefix = temp_dir+'ST'+sat+'_L2_SWEA_MOM_'+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V04.cef'

    ENDIF ELSE IF file_type EQ 'swea_pad' THEN BEGIN 

        IF sat EQ 'A' THEN temp_dir = getenv('STA_SWEA_PAD_DATA')+'/' ELSE $
          temp_dir = getenv('STB_SWEA_PAD_DATA')+'/'
        IF year EQ 2007 AND month EQ 6 AND date EQ 20 THEN $
          file_prefix = temp_dir+'ST'+sat+'_L2_SWEA_PAD_'+string(year, format = '(I4.4)')$
                        +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V03.cef' $
        ELSE $
          file_prefix = temp_dir+'ST'+sat+'_L2_SWEA_PAD_'+string(year, format = '(I4.4)')$
          +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')+'_V04.cef'

    ENDIF ELSE BEGIN
        
        IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L1DATA')+'/'+string(year, format = '(I4.4)')+'/' $
        ELSE temp_dir = getenv('STB_PLA_L1DATA')+'/'+string(year, format = '(I4.4)')+'/' 
        file_prefix = temp_dir+'ST'+sat+'_L1_PLA_'+type_string+string(year, format = '(I4.4)')$
                      +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')$
                      +'_'+string(doy, format = '(I3.3)')+'_V'
    ENDELSE

    IF file_type EQ 'ace_swics' OR file_type EQ 'ace_swics_2hr' OR file_type EQ 'ace_swics_v2_2hr' OR $
      file_type EQ 'ace_swics_q' OR file_type EQ 'stereo_mag' OR $
      file_type EQ 'swea_mom' OR file_type EQ 'swea_pad' THEN BEGIN
        file_string = file_prefix
        file_found = file_test(file_string, /read)
    ENDIF ELSE BEGIN 
        more = 1 
        version = 99            ; largest expected version number
        file_found = 0
        WHILE (more EQ 1) AND (version GT 0) DO BEGIN ; stop looking if there isn't a version 1 file
            file_string = file_prefix+string(version, format = '(I2.2)')+'.cdf'
            file_found = file_test(file_string, /read) ; returns 1 if file exists
            IF file_found EQ 1 THEN BEGIN
                more = 0        ; the most recent version has been found, stop looking for lower version numbers
            ENDIF ELSE BEGIN
                version = version - 1 ; try the next lower version number
            ENDELSE 
        ENDWHILE
    ENDELSE 

    IF file_found EQ 0 THEN file_string = ''
    IF first EQ 1 THEN BEGIN 
        files = [file_string]
        first = 0
    ENDIF ELSE files = [files, file_string]
    IF file_type EQ 'ace_swics' OR file_type EQ 'ace_swics_2hr' OR file_type EQ 'ace_swics_v2_2hr' OR file_type EQ 'ace_swics_q' OR $ ; yearly
      file_type EQ 'l2_alphas_prelim' OR file_type EQ 'stereo_mag' OR file_type EQ 'l2_alpha_ra' OR file_type EQ 'l3_heplus_psd' THEN BEGIN 
        year = year+1          ; swics files are yearly
        doy = 1
    ENDIF ELSE IF file_type EQ 'l2_alpha_ra_10min' THEN BEGIN ; monthly
        month = month+1
        IF month EQ 13 THEN BEGIN
            month = 1
            year  = year + 1
        ENDIF 
        temp_st = time_struct(string(year, format = '(I4.4)')+'-'+string(month, format = '(I2.2)')+'-01/00:00:00')
        doy = temp_st.doy
    ENDIF ELSE BEGIN 
        IF month EQ 12 AND date EQ 31 THEN BEGIN 
            doy = 1 
            year = year + 1
        ENDIF ELSE doy = doy + 1
    ENDELSE 
ENDWHILE


END 
