;+
; PROCEDURE: pla_plot_en_spec
;
; PURPOSE:
;  To plot plastic energy spectra. This routine can be called from a
;  crib or from the command line.
;
; INPUT:
;       sprod - Spacecraft ('A' or 'B') & product ID.
;
; POSSIBLE PRODUCTS:
; 
;          SOURCE ARRAY             REDUCED ARRAY
;
;      01: SW-all                 : h_alpha
;      02: SW-H(D)                : h+peak
;      03: SW-alpha(D)            : he++peak
;      04: SW-alpha(T)            : he++trc
;      05: SW Z>2                 : sw_z>2_h              class:0-1
;      06: SW Z>2                 : sw_z>2_l              class:0-12
;      07: WIDE ANGLE             : wap_ssd_tcr           class:0-14
;      08: WIDE ANGLE(D)          : wap_ssd_dcr           class:0-6
;      09: WIDE ANGLE(D)          : wap_no_ssd_dcr        class:0-6
;      10: SW PHA PRIORITY RATES  : sw_priority           class:0-3
;      11: WAP PHA PRIORITY RATES : wap_priority_ssd      class:0-1
;      12: wap_priority_no_ssd   class:0-1
;      13: class
;
; KEYWORDS: positions - an array of ints for which pos to include (in
;                       pos bin for product)
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   08/13/2008 LBE: Added nodisplay keyword.  
;   05/17/2010 LBE: Added positions keyword.
;   09/27/2012 LBE: Added set_to_beg keyword.
;   01/23/2013 LBE: Add erase gaps keywords (to make spears in plots disappear).
;
;-
PRO pla_plot_en_spec, sprod, units, diagn = diagn,  help = help, nodisplay = nodisplay, positions = positions, $
  set_to_beg = set_to_beg, erase_gaps = erase_gaps
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

get_err_no = 0 & get_err_msg = ''

IF KEYWORD_SET(HELP) THEN BEGIN
    PRINT, '- sprod: prod identifier string'
    PRINT, '- units: units string'
    PRINT, '- diagn = diagn: create diagnostic plots'
    PRINT, '- help = help: show procedure inputs'
    
    RETURN
ENDIF

; Check for gaps in the stings of the input variables
sprod = STRCOMPRESS(sprod, /REMOVE_ALL)
units = STRCOMPRESS(units, /REMOVE_ALL)

prod_str = ['h_alpha', 'h+peak', 'he++peak', 'he++tcr', $
            'sw_z>2_h', 'sw_z>2_l', 'wap_ssd_tcr', 'wap_ssd_dcr', $
            'wap_no_ssd_dcr', 'sw_priority', 'wap_priority_ssd', 'wap_priority_no_ssd', $
            'class' $
           ]

; Transform the input string variable SPROD into the variables SAT/PROD/CLASS
get_input_param, sprod, sat, prod, class
IF get_err_no NE 0 THEN RETURN

prodtmp = prod_str(prod-1)
prod = prodtmp

IF keyword_set(erase_gaps) THEN temp_erase = 1 ELSE temp_erase = 0
; Loop over all sat/product combinations
FOR ii = 0, N_ELEMENTS(prod)-1 DO BEGIN

    get_err_no = 0 & get_err_msg = '' ; reset error indicator
    IF temp_erase EQ 1 THEN erase_gaps = 1
    get_pla_en_spec, sat(ii), prod(ii), class(ii), units, $
                    diagn = diagn, nodisplay = nodisplay, positions = positions, set_to_beg = set_to_beg, $
                     erase_gaps = erase_gaps

    IF get_err_no NE 0 THEN BEGIN
        err_msg, 'No data for product: ' + prod(ii)
    ENDIF

ENDFOR

END
