;+
;PROCEDURE: pl_read_mom_conv
;PURPOSE: Find appropriate calibration files and read in calibration values.
;
;INPUT
;PARAMETERS:  sat: 'A' or 'B'
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 10/12/2006
;
;MODIFICATION HISTORY:
;    10/12/2006 Created
;    03/28/2007 Added check for if no valid epoch.
;    06/27/2007 Added read of svalid/ratrig ratios
;    07/24/2007 Added read of pri 0 ratios
;    09/10/2007 Added vel_groups, removed mom_eff, sw_h, sw_all
;    09/12/2007 Fixed bug with unclosed files
;    09/17/2007 Changed end of mom_cal_stop to string
;    09/24/2007 Changed so ratio files are by year.
;    09/24/2007 Changed end of mom_cal_stop back to -1
;    01/08/2008 Added check for leapyear.
;-
PRO pl_read_mom_conv, sat
compile_opt strictarrsubs
COMMON share1
COMMON com_moment

; find start and end times
IF sat EQ 'A' THEN epoch1 = epoch1_a ELSE epoch1 = epoch1_b
valid_epochs = where(finite(epoch1), count)
IF count GT 0 THEN BEGIN 
    start_time = epoch1[valid_epochs[0]]
    stop_time = epoch1[valid_epochs[count-1]]
; find most recent catalog file
    temp_files = file_search('$PLACAL/*.cal', count = num_files)
    version = 1
    catalog_file = ''
    FOR ii = 0, num_files-1 DO BEGIN 
        temp_file = file_basename(temp_files[ii])
        temp_version = fix(strmid(temp_file, 21, 2))
        temp_sat = strmid(temp_file, 2, 1)
        IF temp_version GE version AND temp_sat EQ sat THEN BEGIN
            version = temp_version
            catalog_file = temp_files[ii]
        ENDIF 
    ENDFOR 

; find calibration file(s)
    openr, cat_lun, catalog_file, /get_lun
    cat_files_found = 0
    cat_files = ['']
    finished = 0
    line = ''
    mom_cal_start = [-2D]       ; this first element will be removed
    mom_cal_stop = [-2D]        ; this first element will be removed
    WHILE eof(cat_lun) EQ 0 AND finished EQ 0 DO BEGIN ; need this loop to get to MOM
        readf, cat_lun, line
        parts_current = strsplit(line, /extract)
        IF parts_current[0] EQ 'MOM' THEN BEGIN 
            
            WHILE eof(cat_lun) EQ 0 AND finished EQ 0 DO BEGIN 
                readf, cat_lun, line
                parts_next = strsplit(line, /extract)
                IF parts_next[0] NE 'MOM' THEN BEGIN ; finished MOM section
                    finished = 1
                    cat_files_found = cat_files_found + 1
                    cat_files = [cat_files, parts_current[2]]
                    mom_cal_start = [mom_cal_start, time_double(parts_current[1])]
                    mom_cal_stop = [mom_cal_stop, -1D]
                ENDIF ELSE BEGIN ; current and next are both MOM
                    IF start_time GE time_double(parts_next[1]) THEN parts_current = parts_next $ ; skip current file
                    ELSE BEGIN 
                        cat_files_found = cat_files_found + 1
                        cat_files = [cat_files, parts_current[2]]
                        mom_cal_start = [mom_cal_start, time_double(parts_current[1])]
                        IF stop_time LT time_double(parts_next[1]) THEN BEGIN ; skip rest of files
                            finished = 1 
                            mom_cal_stop = [mom_cal_stop, -1D]
                        ENDIF ELSE BEGIN
                            mom_cal_stop = [mom_cal_stop, time_double(parts_next[1])]
                            parts_current = parts_next
                        ENDELSE 
                    ENDELSE 
                ENDELSE 
            ENDWHILE 
        ENDIF 
    ENDWHILE 

    IF cat_files_found GT 0 THEN cat_files = cat_files[1:cat_files_found]
    close, cat_lun
    free_lun, cat_lun
ENDIF ELSE cat_files_found = 0

; read calibration files
IF cat_files_found GT 0 THEN BEGIN 
    step_var = dblarr(cat_files_found)
    table_norm = dblarr(cat_files_found, 4) ; D,V,P,H
    geom = dblarr(cat_files_found, 2) ; M,S
    ra_trig_eff = dblarr(cat_files_found, 128, 4) ; 128 lines, esa step, Kristin vel, table vel, efficiency
    vel_groups = dblarr(cat_files_found, 32) ; average of our velocities in ra_trig_eff
    FOR ii = 0, cat_files_found-1 DO BEGIN 
        filename = '$PLACAL/'+cat_files[ii]
        openr, cal_lun, filename, /get_lun
        found_step = 0
        WHILE eof(cal_lun) EQ 0 AND found_step EQ 0 DO BEGIN 
            readf, cal_lun, line
            parts = strsplit(line, /extract)
            IF parts[0] EQ 'step_variable' THEN BEGIN 
                step_var[ii] = double(parts[1])
                found_step = 1
            ENDIF
        ENDWHILE 
        found_table_norm = 0
        WHILE eof(cal_lun) EQ 0 AND found_table_norm EQ 0 DO BEGIN 
            readf, cal_lun, line
            parts = strsplit(line, /extract)
            IF parts[0] EQ 'table_norm_values' THEN BEGIN 
                IF parts[1] NE 'D' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE table_norm[ii, 0] = double(parts[2])
                readf, cal_lun, line
                parts = strsplit(line, /extract)
                IF parts[0] NE 'table_norm_values' OR parts[1] NE 'V' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE table_norm[ii, 1] = double(parts[2])
                readf, cal_lun, line
                parts = strsplit(line, /extract)
                IF parts[0] NE 'table_norm_values' OR parts[1] NE 'P' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE table_norm[ii, 2] = double(parts[2])
                readf, cal_lun, line
                parts = strsplit(line, /extract)
                IF parts[0] NE 'table_norm_values' OR parts[1] NE 'H' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE table_norm[ii, 3] = double(parts[2])
                found_table_norm = 1
            ENDIF
        ENDWHILE 
        found_geom_factor = 0
        WHILE eof(cal_lun) EQ 0 AND found_geom_factor EQ 0 DO BEGIN 
            readf, cal_lun, line
            parts = strsplit(line, /extract)
            IF parts[0] EQ 'geom_factor' THEN BEGIN 
                IF parts[1] NE 'S' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE geom[ii, 1] = double(parts[2])
                readf, cal_lun, line
                parts = strsplit(line, /extract)
                IF parts[0] NE 'geom_factor' OR parts[1] NE 'M' THEN BEGIN
                    print, 'Error in reading moments calibration'
                    stop
                ENDIF ELSE geom[ii, 0] = double(parts[2])
                found_geom_factor = 1
            ENDIF
        ENDWHILE 
        found_ra_trig_eff = 0
        WHILE eof(cal_lun) EQ 0 AND found_ra_trig_eff EQ 0 DO BEGIN 
            readf, cal_lun, line
            parts = strsplit(line, /extract)
            IF parts[0] EQ 'ra_trig' THEN BEGIN 
                ra_trig_eff[ii, 0, *] = double(parts[1:4])
                FOR jj = 1, 127 DO BEGIN ; 128 lines total
                    readf, cal_lun, line
                    parts = strsplit(line, /extract)
                    IF parts[0] NE  'ra_trig' THEN BEGIN 
                        print, "unexpected number of lines"
                        stop
                    ENDIF ELSE ra_trig_eff[ii, jj, *] = double(parts[1:4])
                ENDFOR
                found_ra_trig_eff = 1
            ENDIF
        ENDWHILE
        FOR jj = 0, 31 DO BEGIN 
            vel_groups[ii, jj] = total(ra_trig_eff[ii, (jj*4):(jj*4+3), 2])/4
        ENDFOR 
        close, cal_lun
        free_lun, cal_lun
    ENDFOR
ENDIF 

temp_mom_cal_start = mom_cal_start[1:n_elements(mom_cal_start)-1]
temp_mom_cal_stop = mom_cal_stop[1:n_elements(mom_cal_stop)-1]
IF sat EQ 'A' THEN BEGIN
    IF cat_files_found GT 0 THEN BEGIN 
        mom_cal_start_a = temp_mom_cal_start
        mom_cal_stop_a = temp_mom_cal_stop
        step_var_a = step_var
        table_norm_a = table_norm
        geom_a = geom
    ENDIF 
ENDIF ELSE BEGIN
    IF cat_files_found GT 0 THEN BEGIN 
        mom_cal_start_b = mom_cal_start[1:n_elements(mom_cal_start)-1]
        mom_cal_stop_b = mom_cal_stop[1:n_elements(mom_cal_stop)-1]
        step_var_b = step_var
        table_norm_b = table_norm
        geom_b = geom
    ENDIF 
ENDELSE 

; get ready for moments efficiencies
get_timespan, times
times_struct = time_struct(times)
start_year = times_struct[0].year
IF start_year EQ 2008 OR start_year EQ 2012 OR start_year EQ 2016 OR start_year EQ 2020 THEN leapyear = 1 ELSE $
  leapyear = 0
stop_year = times_struct[n_elements(times_struct)-1].year
start_doy = times_struct[0].doy
stop_doy = times_struct[n_elements(times_struct)-1].doy
IF start_year EQ stop_year THEN num_days = stop_doy-start_doy+1 ELSE IF leapyear EQ 0 THEN $
  num_days = stop_doy-start_doy+1+365 ELSE num_days = stop_doy-start_doy+1+366
doys = intarr(num_days)
years = intarr(num_days)
ratio = fltarr(32, num_days)
IF start_year EQ stop_year THEN BEGIN ; all in same year
    years[*] = start_year
    FOR ii = 0, num_days-1 DO BEGIN
        doys[ii] = start_doy+ii
    ENDFOR 
ENDIF ELSE BEGIN 
    ii = start_year
    jj = start_doy
    FOR kk = 0, num_days-1 DO BEGIN
        years[kk] = ii
        doys[kk] = jj
        jj = jj+1
        IF jj EQ 366 AND leapyear EQ 0 THEN BEGIN
            jj = 1
            ii = ii+1
        ENDIF
        IF jj EQ 367 AND leapyear EQ 1 THEN BEGIN
            jj = 1
            ii = ii+1
        ENDIF
    ENDFOR
ENDELSE 
;doys = doys - 2               ; for checking beacon

; read s_valid/ra_trig ratio (32 for each day -- based on energy)
ratio = dblarr(32, num_days)
ii = 0
year_index = uniq(years)
FOR kk = 0, n_elements(year_index)-1 DO BEGIN  ; start here need to make subarrays
    temp = where(years EQ years[year_index[kk]], count)
    cal_file = 'ST'+sat+'_svalid_ratrig_ratio_'+string(years[year_index[kk]], format = '(I4.4)')+'.tab'
    cal_path = '$PLACAL/'+cal_file
    found = file_test(cal_path)
    openr, cal_lun, cal_path, /get_lun
    readf, cal_lun, line
    readf, cal_lun, line
    readf, cal_lun, line        ; 3 header lines
    WHILE eof(cal_lun) EQ 0 AND ii LT count DO BEGIN 
        readf, cal_lun, line
        parts = strsplit(line, /extract)
        temp_year = fix(parts[0])
        temp_doy = fix(parts[1])
        IF temp_year NE years[ii] THEN print, 'Error in pl_read_mom_conv: wrong year for svalid/ratrig' $
        ELSE IF temp_year EQ years[ii] AND temp_doy LT doys[ii] THEN dummy = 0 $ ; not yet at right day
             ELSE IF temp_year EQ years[ii] AND temp_doy EQ doys[ii] THEN BEGIN ; found it!
            FOR jj = 0, 31 DO BEGIN
                ratio[jj, ii] = float(parts[2+jj])
            ENDFOR 
            ii = ii+1
        ENDIF ELSE print, 'Error in pl_read_mom_conv1: unknown svalid/ratrig ratio for this day'
    ENDWHILE 
close, cal_lun
free_lun, cal_lun
ENDFOR 
IF ii NE num_days THEN print, 'Error in pl_read_mom_conv2: unknown svalid/ratrig ratio for this day'

; read sw pri 0/sw pri 0:3 ratio (128 for each day -- based on energy)
pri_ratio = dblarr(128, num_days)
ii = 0
year_index = uniq(years)
FOR kk = 0, n_elements(year_index)-1 DO BEGIN  ; start here need to make subarrays
    temp = where(years EQ years[year_index[kk]], count)
    cal_file = 'ST'+sat+'_pri_ratio_'+string(years[year_index[kk]], format = '(I4.4)')+'.tab'
    cal_path = '$PLACAL/'+cal_file
    openr, cal_lun, cal_path, /get_lun
    readf, cal_lun, line
    readf, cal_lun, line        ; 2 header lines
    WHILE eof(cal_lun) EQ 0 AND ii LT count DO BEGIN 
        readf, cal_lun, line
        parts = strsplit(line, /extract)
        temp_year = fix(parts[0])
        temp_doy = fix(parts[1])
        IF temp_year LT years[ii] THEN print, 'Error in pl_read_mom_conv: wrong year for pri 0 ratio' $
        ELSE IF temp_year EQ years[ii] AND temp_doy LT doys[ii] THEN dummy = 0 $ ; not yet at right day
             ELSE IF temp_year EQ years[ii] AND temp_doy EQ doys[ii] THEN BEGIN ; found it!
            FOR jj = 0, 127 DO BEGIN
                pri_ratio[jj, ii] = float(parts[2+jj])
            ENDFOR 
            ii = ii+1
        ENDIF ELSE print, 'Error in pl_read_mom_conv1: unknown pri 0 ratio for this day'
    ENDWHILE 
ENDFOR 
close, cal_lun
free_lun, cal_lun
IF ii NE num_days THEN print, 'Error in pl_read_mom_conv2: unknown pri 0 ratio for this day'

; THIS IS TEMPORARY CODE FOR COMPARING WITH KRISTIN
; read sw_pri/(sw_pri+wap_ssd_pri) ratio 
; (128 for each day -- based on energy)
;swpri_ratio = dblarr(128, num_days)
;ii = 0
;cal_file = 'ST'+sat+'_swpri_ratio.tab'
;cal_path = '$PLACAL/'+cal_file
;openr, cal_lun, cal_path, /get_lun
;readf, cal_lun, line
;readf, cal_lun, line
;readf, cal_lun, line ; 3 header lines
;WHILE eof(cal_lun) EQ 0 AND ii LT num_days DO BEGIN 
;    readf, cal_lun, line
;    parts = strsplit(line, /extract)
;    temp_year = fix(parts[0])
;    temp_doy = fix(parts[1])
;    IF temp_year LT years[ii] THEN dummy = 0 $ ; not yet at right year
;    ELSE IF temp_year EQ years[ii] AND temp_doy LT doys[ii] THEN dummy = 0 $ ; not yet at right day
;         ELSE IF temp_year EQ years[ii] AND temp_doy EQ doys[ii] THEN BEGIN ; found it!
;        FOR jj = 0, 127 DO BEGIN
;            swpri_ratio[jj, ii] = float(parts[2+jj])
;        ENDFOR 
;        ii = ii+1
;    ENDIF ELSE print, 'Error in pl_read_mom_conv1: unknown swpri ratio for this day'
;ENDWHILE 
;IF ii NE num_days THEN print, 'Error in pl_read_mom_conv2: unknown swpri ratio for this day'
;close, cal_lun
;free_lun, cal_lun

;doys = doys + 2               ; for checking beacon

IF sat EQ 'A' THEN BEGIN
    pri_ratio_a = pri_ratio
    ;swpri_ratio_a = swpri_ratio
    svalid_ratrig_a = ratio
    ra_trig_eff_a = ra_trig_eff
    vel_groups_a = vel_groups
    doys_a = doys
    years_a = years
ENDIF ELSE BEGIN
    pri_ratio_b = pri_ratio
    ;swpri_ratio_b = swpri_ratio
    svalid_ratrig_b = ratio
    ra_trig_eff_b = ra_trig_eff
    vel_groups_b = vel_groups
    doys_b = doys
    years_b = years
ENDELSE 

END 
