;+
;PROCEDURE: pl_read_cdf_mon_full
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch: 0(for variable), 1, or 5 minute resolution
;				
;
;CREATED BY: L. Ellis (pl_read_cdf)
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;
;	December 2006 K.Simunac modifies for monitor rates
;	4 Jan, 2007:  modified to check for error flags
;	7 Jan, 2007:  modified to output counts per second (if desired)
;	8 Jan, 2007:  modified to calculate select ratios
;	12 Jan, 2007:  modified variable and program names to avoid confusion
;		read_product -> read_product_mon
;		remove_fill -> remove_fill_mon
;		init_mom -> init_mon
;		mom_initialized -> mon_initialized
;		mom_read -> mon_read
;		check_indices -> check_indices_mon
;	24 Jan, 2007:  IMPORTANT FIX to sum_defl... index system
;    05-15-2007: CM
;                Actual maxrec is epochinfo.maxrec+1
;    09-19-2007: LBE
;                Added cdf_close
;-

PRO read_product_mon_full, cdf_id, prod_name, data_var, data_index, epoch_maxrec
compile_opt strictarrsubs

; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 

cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable

IF varinfo.maxrec+1 GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec

cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
num_dim = size(data_var, /n_dimensions)

CASE num_dim OF 
    1: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    2: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    3: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, *, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    ELSE: BEGIN 
        print, "Lorna: Add more dimensions"
        stop
    END
ENDCASE 
END 

PRO remove_fill_mon_full, fill_val, data
compile_opt strictarrsubs

; fill_val: fill_value to be replaced by NaN
; data: array of data values
indice = where(data EQ fill_val, count)
FOR ii = 0L, count-1 DO BEGIN
    data[indice[ii]] = !values.d_nan
ENDFOR 

END 


PRO init_mon_full, sat, num_records1, mon_full_initialized, E_step, products
compile_opt strictarrsubs

; initialize all the monitor rate variables (in com_mon_full)

num_E = n_elements(E_step)

COMMON com_mon_full
num_rows1 = num_records1

CASE sat OF
    'A': BEGIN 
	error1_a	= fltarr(num_records1) ; error flag
	error1_a[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_a_full	= dblarr(128,32,num_records1) ;0
		s_valid_a_full[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_a_full = dblarr(num_records1,128) ;0
		sum_defl_s_valid_a_full[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_a_full	= dblarr(128,32,num_records1) ; 1
		s_e_not_req_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_a_full = dblarr(num_records1,128) ;1
		sum_defl_s_e_not_req_a_full[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_a_full	= dblarr(128,32,num_records1) ; 2
		s_e_req_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_a_full = dblarr(num_records1,128) ;2
		sum_defl_s_e_req_a_full[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_a_full	= dblarr(128,32,num_records1) ; 3
		s_no_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_a_full = dblarr(num_records1,128) ;3
		sum_defl_s_no_pos_a_full[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_a_full	= dblarr(128,32,num_records1) ; 4
		s_mult_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_a_full = dblarr(num_records1,128) ;4
		sum_defl_s_mult_pos_a_full[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_a_full	= dblarr(128,32,num_records1) ; 5
		s_no_e_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_a_full = dblarr(num_records1,128) ;5
		sum_defl_s_no_e_a_full[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_a_full	= dblarr(128,32,num_records1) ; 6
		s_mult_e_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_a_full = dblarr(num_records1,128) ;6
		sum_defl_s_mult_e_a_full[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_a_full	= dblarr(128,32,num_records1) ; 7
		ra_sat_a_a_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_a_full = dblarr(num_records1,128) ;7
		sum_defl_ra_sat_a_a_full[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_a_full	= dblarr(128,32,num_records1) ; 8
		ra_sat_b_a_full[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_a_full = dblarr(num_records1,128) ;8
		sum_defl_ra_sat_b_a_full[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_a_full	= dblarr(128,32,num_records1) ; 9
		ra_sat_both_a_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_a_full = dblarr(num_records1,128) ;9
		sum_defl_ra_sat_both_a_full[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_a_full	= dblarr(128,32,num_records1) ; 10
		ssd_sw_a_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_a_full = dblarr(num_records1,128) ;10
		sum_defl_ssd_sw_a_full[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_a_full	= dblarr(128,32,num_records1) ; 11
		ssd_st_a_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_a_full = dblarr(num_records1,128) ;11
		sum_defl_ssd_st_a_full[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_a_full	= dblarr(128,32,num_records1) ; 12
		sf0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_a_full[*,*] = !values.d_nan ;12		
		END
	'sfr0':  BEGIN
		sfr0_a_full	= dblarr(128,32,num_records1) ; 13
		sfr0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_a_full[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_a_full	= dblarr(128,32,num_records1) ; 14
		stp0_a_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_a_full[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_a_full	= dblarr(128,32,num_records1) ; 15
		ra_trig_a_full[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_full = dblarr(num_records1,128) ;15
		sum_defl_ra_trig_a_full[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_a_full	= dblarr(128,32,num_records1) ; 16
		pos1_0_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_a_full = dblarr(num_records1,128) ;16
		sum_defl_pos1_0_a_full[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_a_full	= dblarr(128,32,num_records1) ; 17
		pos1_1_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_a_full = dblarr(num_records1,128) ;17
		sum_defl_pos1_1_a_full[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_a_full	= dblarr(128,32,num_records1) ; 18
		w_no_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_a_full = dblarr(num_records1,128) ;18
		sum_defl_w_no_pos_a_full[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_a_full	= dblarr(128,32,num_records1) ; 19
		w_mult_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_a_full = dblarr(num_records1,128) ;19
		sum_defl_w_mult_pos_a_full[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_a_full	= dblarr(128,32,num_records1) ; 20
		w_valid_a_full[*,*,*]	= !values.d_nan
		sum_defl_w_valid_a_full = dblarr(num_records1,128) ;20
		sum_defl_w_valid_a_full[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_a_full	= dblarr(128,32,num_records1) ; 21
		sf2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_a_full[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_a_full	= dblarr(128,32,num_records1) ; 22
		sfr2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_a_full[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_a_full	= dblarr(128,32,num_records1) ; 23
		stp2_a_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_a_full[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_a_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_a_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_a_full[*,*] = !values.d_nan ;24	
		END
	'pos2_1':  BEGIN
		pos2_1_a_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_a_full[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_a_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_a_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_a_full[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_a_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_a_full[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_a_full	= dblarr(128,32,num_records1) ; 28
		pos3_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_a_full = dblarr(num_records1,128) ;28
		sum_defl_pos3_0_a_full[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_a_full	= dblarr(128,32,num_records1) ; 29
		pos3_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_full = dblarr(num_records1,128) ;29
		sum_defl_pos3_1_a_full[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_a_full	= dblarr(128,32,num_records1) ; 30
		pos3_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_full = dblarr(num_records1,128) ;30
		sum_defl_pos3_2_a_full[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_a_full	= dblarr(128,32,num_records1) ; 31
		pos3_3_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_a_full = dblarr(num_records1,128) ;31
		sum_defl_pos3_3_a_full[*,*] = !values.d_nan ;31
		END
	'pos2':  BEGIN 
		pos2_a_full	= dblarr(128,32,num_records1) 
		pos2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_a_full = dblarr(num_records1,128) 
		sum_defl_pos2_a_full[*,*] = !values.d_nan  

		pos2_0_a_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_a_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_a_full[*,*] = !values.d_nan ;24	

		pos2_1_a_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_a_full[*,*] = !values.d_nan ;25

		pos2_2_a_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_a_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_a_full[*,*] = !values.d_nan ;26

		pos2_3_a_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_a_full[*,*] = !values.d_nan ;27
		
		END
	'pos3':  BEGIN 
		pos3_a_full	= dblarr(128,32,num_records1) 
		pos3_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_a_full = dblarr(num_records1,128) 
		sum_defl_pos3_a_full[*,*] = !values.d_nan 

		pos3_0_a_full	= dblarr(128,32,num_records1) ; 24
		pos3_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_a_full = dblarr(num_records1,128) ;24
		sum_defl_pos3_0_a_full[*,*] = !values.d_nan ;24	

		pos3_1_a_full	= dblarr(128,32,num_records1) ; 25
		pos3_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_full = dblarr(num_records1,128) ;25
		sum_defl_pos3_1_a_full[*,*] = !values.d_nan ;25

		pos3_2_a_full	= dblarr(128,32,num_records1) ; 26
		pos3_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_full = dblarr(num_records1,128) ;26
		sum_defl_pos3_2_a_full[*,*] = !values.d_nan ;26

		pos3_3_a_full	= dblarr(128,32,num_records1) ; 27
		pos3_3_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_a_full = dblarr(num_records1,128) ;27
		sum_defl_pos3_3_a_full[*,*] = !values.d_nan ;27
		
		END


	'stop0':  BEGIN 
		stop0_a_full	= dblarr(128,32,num_records1) 
		stop0_a_full[*,*,*]	= !values.d_nan
		sum_defl_stop0_a_full = dblarr(num_records1,128) 
		sum_defl_stop0_a_full[*,*] = !values.d_nan  

		sf0_a_full	= dblarr(128,32,num_records1) ; 12
		sf0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_a_full[*,*] = !values.d_nan ;12	

		sfr0_a_full	= dblarr(128,32,num_records1) ; 13
		sfr0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_a_full[*,*] = !values.d_nan ;13
	
		END

	'start0':  BEGIN
		start0_a_full	= dblarr(128,32,num_records1) 
		start0_a_full[*,*,*]	= !values.d_nan
		sum_defl_start0_a_full = dblarr(num_records1,128)
		sum_defl_start0_a_full[*,*] = !values.d_nan

		sfr0_a_full	= dblarr(128,32,num_records1) ; 13
		sfr0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_a_full[*,*] = !values.d_nan ;13

		stp0_a_full	= dblarr(128,32,num_records1) ; 14
		stp0_a_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_a_full[*,*] = !values.d_nan ;14

		END
	'stop2':  BEGIN
		stop2_a_full	= dblarr(128,32,num_records1)
		stop2_a_full[*,*,*]	= !values.d_nan
		sum_defl_stop2_a_full = dblarr(num_records1,128)
		sum_defl_stop2_a_full[*,*] = !values.d_nan

		sf2_a_full	= dblarr(128,32,num_records1) ; 21
		sf2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_a_full[*,*] = !values.d_nan ;21
		
		sfr2_a_full	= dblarr(128,32,num_records1) ; 22
		sfr2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_a_full[*,*] = !values.d_nan ;22
		END

	'start2':  BEGIN
		start2_a_full	= dblarr(128,32,num_records1) 
		start2_a_full[*,*,*]	= !values.d_nan
		sum_defl_start2_a_full = dblarr(num_records1,128) 
		sum_defl_start2_a_full[*,*] = !values.d_nan 

		sfr2_a_full	= dblarr(128,32,num_records1) ; 22
		sfr2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_a_full[*,*] = !values.d_nan ;22

		stp2_a_full	= dblarr(128,32,num_records1) ; 23
		stp2_a_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_a_full[*,*] = !values.d_nan ;23
		END

	ENDCASE
	ENDFOR			
    END 
    'B': BEGIN 
	error1_b	= fltarr(num_records1) ; error flag
	error1_b[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_b_full	= dblarr(128,32,num_records1) ;0
		s_valid_b_full[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_b_full = dblarr(num_records1,128) ;0
		sum_defl_s_valid_b_full[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_b_full	= dblarr(128,32,num_records1) ; 1
		s_e_not_req_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_b_full = dblarr(num_records1,128) ;1
		sum_defl_s_e_not_req_b_full[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_b_full	= dblarr(128,32,num_records1) ; 2
		s_e_req_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_b_full = dblarr(num_records1,128) ;2
		sum_defl_s_e_req_b_full[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_b_full	= dblarr(128,32,num_records1) ; 3
		s_no_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_b_full = dblarr(num_records1,128) ;3
		sum_defl_s_no_pos_b_full[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_b_full	= dblarr(128,32,num_records1) ; 4
		s_mult_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_b_full = dblarr(num_records1,128) ;4
		sum_defl_s_mult_pos_b_full[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_b_full	= dblarr(128,32,num_records1) ; 5
		s_no_e_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_b_full = dblarr(num_records1,128) ;5
		sum_defl_s_no_e_b_full[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_b_full	= dblarr(128,32,num_records1) ; 6
		s_mult_e_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_b_full = dblarr(num_records1,128) ;6
		sum_defl_s_mult_e_b_full[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_b_full	= dblarr(128,32,num_records1) ; 7
		ra_sat_a_b_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_b_full = dblarr(num_records1,128) ;7
		sum_defl_ra_sat_a_b_full[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_b_full	= dblarr(128,32,num_records1) ; 8
		ra_sat_b_b_full[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_b_full = dblarr(num_records1,128) ;8
		sum_defl_ra_sat_b_b_full[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_b_full	= dblarr(128,32,num_records1) ; 9
		ra_sat_both_b_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_b_full = dblarr(num_records1,128) ;9
		sum_defl_ra_sat_both_b_full[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_b_full	= dblarr(128,32,num_records1) ; 10
		ssd_sw_b_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_b_full = dblarr(num_records1,128) ;10
		sum_defl_ssd_sw_b_full[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_b_full	= dblarr(128,32,num_records1) ; 11
		ssd_st_b_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_b_full = dblarr(num_records1,128) ;11
		sum_defl_ssd_st_b_full[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_b_full	= dblarr(128,32,num_records1) ; 12
		sf0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_b_full[*,*] = !values.d_nan ;12		
		END
	'sfr0':  BEGIN
		sfr0_b_full	= dblarr(128,32,num_records1) ; 13
		sfr0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_b_full[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_b_full	= dblarr(128,32,num_records1) ; 14
		stp0_b_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_b_full[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_b_full	= dblarr(128,32,num_records1) ; 15
		ra_trig_b_full[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_full = dblarr(num_records1,128) ;15
		sum_defl_ra_trig_b_full[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_b_full	= dblarr(128,32,num_records1) ; 16
		pos1_0_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_b_full = dblarr(num_records1,128) ;16
		sum_defl_pos1_0_b_full[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_b_full	= dblarr(128,32,num_records1) ; 17
		pos1_1_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_b_full = dblarr(num_records1,128) ;17
		sum_defl_pos1_1_b_full[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_b_full	= dblarr(128,32,num_records1) ; 18
		w_no_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_b_full = dblarr(num_records1,128) ;18
		sum_defl_w_no_pos_b_full[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_b_full	= dblarr(128,32,num_records1) ; 19
		w_mult_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_b_full = dblarr(num_records1,128) ;19
		sum_defl_w_mult_pos_b_full[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_b_full	= dblarr(128,32,num_records1) ; 20
		w_valid_b_full[*,*,*]	= !values.d_nan
		sum_defl_w_valid_b_full = dblarr(num_records1,128) ;20
		sum_defl_w_valid_b_full[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_b_full	= dblarr(128,32,num_records1) ; 21
		sf2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_b_full[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_b_full	= dblarr(128,32,num_records1) ; 22
		sfr2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_b_full[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_b_full	= dblarr(128,32,num_records1) ; 23
		stp2_b_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_b_full[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_b_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_b_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_b_full[*,*] = !values.d_nan ;24	
		END
	'pos2_1':  BEGIN
		pos2_1_b_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_b_full[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_b_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_b_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_b_full[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_b_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_b_full[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_b_full	= dblarr(128,32,num_records1) ; 28
		pos3_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_b_full = dblarr(num_records1,128) ;28
		sum_defl_pos3_0_b_full[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_b_full	= dblarr(128,32,num_records1) ; 29
		pos3_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_full = dblarr(num_records1,128) ;29
		sum_defl_pos3_1_b_full[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_b_full	= dblarr(128,32,num_records1) ; 30
		pos3_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_full = dblarr(num_records1,128) ;30
		sum_defl_pos3_2_b_full[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_b_full	= dblarr(128,32,num_records1) ; 31
		pos3_3_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_b_full = dblarr(num_records1,128) ;31
		sum_defl_pos3_3_b_full[*,*] = !values.d_nan ;31
		END
	'pos2':  BEGIN 
		pos2_b_full	= dblarr(128,32,num_records1) 
		pos2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_b_full = dblarr(num_records1,128) 
		sum_defl_pos2_b_full[*,*] = !values.d_nan  

		pos2_0_b_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_b_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_b_full[*,*] = !values.d_nan ;24	

		pos2_1_b_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_b_full[*,*] = !values.d_nan ;25

		pos2_2_b_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_b_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_b_full[*,*] = !values.d_nan ;26

		pos2_3_b_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_b_full[*,*] = !values.d_nan ;27
		
		END
	'pos3':  BEGIN 
		pos3_b_full	= dblarr(128,32,num_records1) 
		pos3_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_b_full = dblarr(num_records1,128) 
		sum_defl_pos3_b_full[*,*] = !values.d_nan 

		pos3_0_b_full	= dblarr(128,32,num_records1) ; 24
		pos3_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_b_full = dblarr(num_records1,128) ;24
		sum_defl_pos3_0_b_full[*,*] = !values.d_nan ;24	

		pos3_1_b_full	= dblarr(128,32,num_records1) ; 25
		pos3_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_full = dblarr(num_records1,128) ;25
		sum_defl_pos3_1_b_full[*,*] = !values.d_nan ;25

		pos3_2_b_full	= dblarr(128,32,num_records1) ; 26
		pos3_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_full = dblarr(num_records1,128) ;26
		sum_defl_pos3_2_b_full[*,*] = !values.d_nan ;26

		pos3_3_b_full	= dblarr(128,32,num_records1) ; 27
		pos3_3_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_b_full = dblarr(num_records1,128) ;27
		sum_defl_pos3_3_b_full[*,*] = !values.d_nan ;27
		
		END

	'stop0':  BEGIN 
		stop0_b_full	= dblarr(128,32,num_records1) 
		stop0_b_full[*,*,*]	= !values.d_nan
		sum_defl_stop0_b_full = dblarr(num_records1,128) 
		sum_defl_stop0_b_full[*,*] = !values.d_nan  

		sf0_b_full	= dblarr(128,32,num_records1) ; 12
		sf0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_b_full[*,*] = !values.d_nan ;12	

		sfr0_b_full	= dblarr(128,32,num_records1) ; 13
		sfr0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_b_full[*,*] = !values.d_nan ;13
	
		END

	'start0':  BEGIN
		start0_b_full	= dblarr(128,32,num_records1) 
		start0_b_full[*,*,*]	= !values.d_nan
		sum_defl_start0_b_full = dblarr(num_records1,128)
		sum_defl_start0_b_full[*,*] = !values.d_nan

		sfr0_b_full	= dblarr(128,32,num_records1) ; 13
		sfr0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_b_full[*,*] = !values.d_nan ;13

		stp0_b_full	= dblarr(128,32,num_records1) ; 14
		stp0_b_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_b_full[*,*] = !values.d_nan ;14

		END
	'stop2':  BEGIN
		stop2_b_full	= dblarr(128,32,num_records1)
		stop2_b_full[*,*,*]	= !values.d_nan
		sum_defl_stop2_b_full = dblarr(num_records1,128)
		sum_defl_stop2_b_full[*,*] = !values.d_nan

		sf2_b_full	= dblarr(128,32,num_records1) ; 21
		sf2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_b_full[*,*] = !values.d_nan ;21
		
		sfr2_b_full	= dblarr(128,32,num_records1) ; 22
		sfr2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_b_full[*,*] = !values.d_nan ;22
		END

	'start2':  BEGIN
		start2_b_full	= dblarr(128,32,num_records1) 
		start2_b_full[*,*,*]	= !values.d_nan
		sum_defl_start2_b_full = dblarr(num_records1,128) 
		sum_defl_start2_b_full[*,*] = !values.d_nan 

		sfr2_b_full	= dblarr(128,32,num_records1) ; 22
		sfr2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_b_full[*,*] = !values.d_nan ;22

		stp2_b_full	= dblarr(128,32,num_records1) ; 23
		stp2_b_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_b_full[*,*] = !values.d_nan ;23
		END

	ENDCASE
	ENDFOR
	END
ENDCASE 
mon_full_initialized = 1

END 

PRO read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, prod_name, full_values, sum_full
compile_opt strictarrsubs
COMMON com_mon_full

temp_name = prod_name+'_full'

read_product_mon_full, cdf_id, temp_name, full_values, data1_index, epoch1maxrec

sum_full=DBLARR(num_records1,128)
sum_full[*, *] = !values.d_nan

remove_fill_mon_full, -1, full_values
temp = where(finite(full_values) EQ 1,  count)

IF count GT 0 THEN BEGIN 
    FOR j = 0, num_records1-1 DO BEGIN
        FOR k = 0, 127 DO BEGIN	
            sum_full[j, k] = TOTAL(full_values(k, *, j), /NAN)
        ENDFOR
    ENDFOR
ENDIF

END 

PRO read_mon_full, cdf_id, sat, mon_full_read, data1_index, epoch1maxrec, products, num_records1, E_step
compile_opt strictarrsubs

COMMON com_mon_full
CASE sat OF 
    'A': BEGIN
	
	read_product_mon_full, cdf_id, 'error1', error1_a, data1_index, epoch1maxrec
	remove_fill_mon_full, -1, error1_a
	IF (TOTAL(error1_a, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'


        FOR ii = 0, n_elements(products)-1 DO BEGIN

            CASE products[ii] OF
                's_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_valid', s_valid_a_full, sum_defl_s_valid_a_full
                's_e_not_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_not_req', s_e_not_req_a_full, sum_defl_s_e_not_req_a_full
		's_e_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_req', s_e_req_a_full, sum_defl_s_e_req_a_full
		's_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_pos', s_no_pos_a_full, sum_defl_s_no_pos_a_full
		's_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_pos', s_mult_pos_a_full, sum_defl_s_mult_pos_a_full
		's_no_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_e', s_no_e_a_full, sum_defl_s_no_e_a_full
		's_mult_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_e', s_mult_e_a_full, sum_defl_s_mult_e_a_full
		'ra_sat_a': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_a', ra_sat_a_a_full, sum_defl_ra_sat_a_a_full
		'ra_sat_b': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_b', ra_sat_b_a_full, sum_defl_ra_sat_b_a_full
		'ra_sat_both': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_both', ra_sat_both_a_full, sum_defl_ra_sat_both_a_full
		'ssd_sw': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_sw', ssd_sw_a_full, sum_defl_ssd_sw_a_full
		'ssd_st': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_st', ssd_st_a_full, sum_defl_ssd_st_a_full
 		'sf0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf0', sf0_a_full, sum_defl_sf0_a_full
		'sfr0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr0', sfr0_a_full, sum_defl_sfr0_a_full
		'stp0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp0', stp0_a_full, sum_defl_stp0_a_full
		'ra_trig': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_trig', ra_trig_a_full, sum_defl_ra_trig_a_full
		'pos1_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_0', pos1_0_a_full, sum_defl_pos1_0_a_full
		'pos1_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_1', pos1_1_a_full, sum_defl_pos1_1_a_full
		'w_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_no_pos', w_no_pos_a_full, sum_defl_w_no_pos_a_full
		'w_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_mult_pos', w_mult_pos_a_full, sum_defl_w_mult_pos_a_full
		'w_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_valid', w_valid_a_full, sum_defl_w_valid_a_full
		'sf2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf2', sf2_a_full, sum_defl_sf2_a_full
		'sfr2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr2', sfr2_a_full, sum_defl_sfr2_a_full
		'stp2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp2', stp2_a_full, sum_defl_stp2_a_full
		'pos2_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_0', pos2_0_a_full, sum_defl_pos2_0_a_full
		'pos2_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_1', pos2_1_a_full, sum_defl_pos2_1_a_full
		'pos2_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_2', pos2_2_a_full, sum_defl_pos2_2_a_full
		'pos2_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_3', pos2_3_a_full, sum_defl_pos2_3_a_full
		'pos3_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_0', pos3_0_a_full, sum_defl_pos3_0_a_full
		'pos3_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_1', pos3_1_a_full, sum_defl_pos3_1_a_full
		'pos3_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_2', pos3_2_a_full, sum_defl_pos3_2_a_full
		'pos3_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_3', pos3_3_a_full, sum_defl_pos3_3_a_full
		'pos2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_0', pos2_0_a_full, sum_defl_pos2_0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_1', pos2_1_a_full,  sum_defl_pos2_1_a_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_2', pos2_2_a_full,  sum_defl_pos2_2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_3', pos2_3_a_full, sum_defl_pos2_3_a_full
			
			FOR kk = 0D, n_elements(pos2_0_a_full) - 1 DO BEGIN
				pos2_a_full[kk] = pos2_0_a_full[kk]+pos2_1_a_full[kk]+pos2_2_a_full[kk]+pos2_3_a_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_a_full) - 1 DO BEGIN
				sum_defl_pos2_a_full[mm] = sum_defl_pos2_0_a_full[mm]+sum_defl_pos2_1_a_full[mm]+sum_defl_pos2_2_a_full[mm]+sum_defl_pos2_3_a_full[mm]
			ENDFOR
			
			END
		'pos3':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_0', pos3_0_a_full,  sum_defl_pos3_0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_1', pos3_1_a_full,  sum_defl_pos3_1_a_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_2', pos3_2_a_full,  sum_defl_pos3_2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_3', pos3_3_a_full,  sum_defl_pos3_3_a_full
			
			FOR kk = 0D, n_elements(pos3_0_a_full) - 1 DO BEGIN
				pos3_a_full[kk] = pos3_0_a_full[kk]+pos3_1_a_full[kk]+pos3_2_a_full[kk]+pos3_3_a_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_a_full) - 1 DO BEGIN
				sum_defl_pos3_a_full[mm] = sum_defl_pos3_0_a_full[mm]+sum_defl_pos3_1_a_full[mm]+sum_defl_pos3_2_a_full[mm]+sum_defl_pos3_3_a_full[mm]
			ENDFOR
			
			END

		'stop0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf0', sf0_a_full, sum_defl_sf0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_a_full, sum_defl_sfr0_a_full
			
			FOR kk = 0D, n_elements(sf0_a_full) - 1 DO BEGIN
			IF (sf0_a_full[kk] NE 0) AND (FINITE(sf0_a_full[kk]) EQ 1) AND (FINITE(sfr0_a_full[kk]) EQ 1) THEN BEGIN
				stop0_a_full[kk] = sfr0_a_full[kk]/sf0_a_full[kk]
			ENDIF ELSE BEGIN
				stop0_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_a_full) - 1 DO BEGIN
			IF (sum_defl_sf0_a_full[mm] NE 0) AND (FINITE(sum_defl_sf0_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_a_full[mm] = sum_defl_sfr0_a_full[mm]/sum_defl_sf0_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'start0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp0', stp0_a_full, sum_defl_stp0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_a_full, sum_defl_sfr0_a_full
			
			FOR kk = 0D, n_elements(stp0_a_full) - 1 DO BEGIN
			IF (stp0_a_full[kk] NE 0) AND (FINITE(stp0_a_full[kk]) EQ 1) AND (FINITE(sfr0_a_full[kk]) EQ 1) THEN BEGIN
				start0_a_full[kk] = sfr0_a_full[kk]/stp0_a_full[kk]
			ENDIF ELSE BEGIN
				start0_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_a_full) - 1 DO BEGIN
			IF (sum_defl_stp0_a_full[mm] NE 0) AND (FINITE(sum_defl_stp0_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_a_full[mm] = sum_defl_sfr0_a_full[mm]/sum_defl_stp0_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'stop2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf2', sf2_a_full, sum_defl_sf2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_a_full, sum_defl_sfr2_a_full
			
			FOR kk = 0D, n_elements(sf2_a_full) - 1 DO BEGIN
			IF (sf2_a_full[kk] NE 0) AND (FINITE(sf2_a_full[kk]) EQ 1) AND (FINITE(sfr2_a_full[kk]) EQ 1) THEN BEGIN
				stop2_a_full[kk] = sfr2_a_full[kk]/sf2_a_full[kk]
			ENDIF ELSE BEGIN
				stop2_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_a_full) - 1 DO BEGIN
			IF (sum_defl_sf2_a_full[mm] NE 0) AND (FINITE(sum_defl_sf2_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_a_full[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_sf2_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

		'start2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp2', stp2_a_full, sum_defl_stp2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_a_full, sum_defl_sfr2_a_full
			
			FOR kk = 0D, n_elements(stp2_a_full) - 1 DO BEGIN
			IF (stp2_a_full[kk] NE 0) AND (FINITE(stp2_a_full[kk]) EQ 1) AND (FINITE(sfr2_a_full[kk]) EQ 1) THEN BEGIN
				start2_a_full[kk] = sfr2_a_full[kk]/stp2_a_full[kk]
			ENDIF ELSE BEGIN
				start2_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_a_full) - 1 DO BEGIN
			IF (sum_defl_stp2_a_full[mm] NE 0) AND (FINITE(sum_defl_stp2_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_a_full[mm] = sum_defl_sfr2_a_full[mm]/sum_defl_stp2_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

            ENDCASE
        ENDFOR
    END
    'B': BEGIN

	read_product_mon_full, cdf_id, 'error1', error1_b, data1_index, epoch1maxrec
	remove_fill_mon_full, -1, error1_b
	IF (TOTAL(error1_b, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'


        FOR ii = 0, n_elements(products)-1 DO BEGIN

            CASE products[ii] OF
                's_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_valid', s_valid_b_full, sum_defl_s_valid_b_full
                's_e_not_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_not_req', s_e_not_req_b_full, sum_defl_s_e_not_req_b_full
		's_e_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_req', s_e_req_b_full, sum_defl_s_e_req_b_full
		's_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_pos', s_no_pos_b_full, sum_defl_s_no_pos_b_full
		's_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_pos', s_mult_pos_b_full, sum_defl_s_mult_pos_b_full
		's_no_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_e', s_no_e_b_full, sum_defl_s_no_e_b_full
		's_mult_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_e', s_mult_e_b_full, sum_defl_s_mult_e_b_full
		'ra_sat_a': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_a', ra_sat_a_b_full, sum_defl_ra_sat_a_b_full
		'ra_sat_b': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_b', ra_sat_b_b_full, sum_defl_ra_sat_b_b_full
		'ra_sat_both': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_both', ra_sat_both_b_full, sum_defl_ra_sat_both_b_full
		'ssd_sw': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_sw', ssd_sw_b_full, sum_defl_ssd_sw_b_full
		'ssd_st': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_st', ssd_st_b_full, sum_defl_ssd_st_b_full
 		'sf0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf0', sf0_b_full, sum_defl_sf0_b_full
		'sfr0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr0', sfr0_b_full, sum_defl_sfr0_b_full
		'stp0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp0', stp0_b_full, sum_defl_stp0_b_full
		'ra_trig': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_trig', ra_trig_b_full, sum_defl_ra_trig_b_full
		'pos1_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_0', pos1_0_b_full, sum_defl_pos1_0_b_full
		'pos1_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_1', pos1_1_b_full, sum_defl_pos1_1_b_full
		'w_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_no_pos', w_no_pos_b_full, sum_defl_w_no_pos_b_full
		'w_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_mult_pos', w_mult_pos_b_full, sum_defl_w_mult_pos_b_full
		'w_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_valid', w_valid_b_full, sum_defl_w_valid_b_full
		'sf2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf2', sf2_b_full, sum_defl_sf2_b_full
		'sfr2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr2', sfr2_b_full, sum_defl_sfr2_b_full
		'stp2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp2', stp2_b_full, sum_defl_stp2_b_full
		'pos2_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_0', pos2_0_b_full, sum_defl_pos2_0_b_full
		'pos2_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_1', pos2_1_b_full, sum_defl_pos2_1_b_full
		'pos2_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_2', pos2_2_b_full, sum_defl_pos2_2_b_full
		'pos2_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_3', pos2_3_b_full, sum_defl_pos2_3_b_full
		'pos3_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_0', pos3_0_b_full, sum_defl_pos3_0_b_full
		'pos3_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_1', pos3_1_b_full, sum_defl_pos3_1_b_full
		'pos3_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_2', pos3_2_b_full, sum_defl_pos3_2_b_full
		'pos3_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_3', pos3_3_b_full, sum_defl_pos3_3_b_full
		'pos2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_0', pos2_0_b_full, sum_defl_pos2_0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_1', pos2_1_b_full,  sum_defl_pos2_1_b_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_2', pos2_2_b_full,  sum_defl_pos2_2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_3', pos2_3_b_full, sum_defl_pos2_3_b_full
			
			FOR kk = 0D, n_elements(pos2_0_b_full) - 1 DO BEGIN
				pos2_b_full[kk] = pos2_0_b_full[kk]+pos2_1_b_full[kk]+pos2_2_b_full[kk]+pos2_3_b_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_b_full) - 1 DO BEGIN
				sum_defl_pos2_b_full[mm] = sum_defl_pos2_0_b_full[mm]+sum_defl_pos2_1_b_full[mm]+sum_defl_pos2_2_b_full[mm]+sum_defl_pos2_3_b_full[mm]
			ENDFOR
			
			END
		'pos3':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_0', pos3_0_b_full,  sum_defl_pos3_0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_1', pos3_1_b_full,  sum_defl_pos3_1_b_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_2', pos3_2_b_full,  sum_defl_pos3_2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_3', pos3_3_b_full,  sum_defl_pos3_3_b_full
			
			FOR kk = 0D, n_elements(pos3_0_b_full) - 1 DO BEGIN
				pos3_b_full[kk] = pos3_0_b_full[kk]+pos3_1_b_full[kk]+pos3_2_b_full[kk]+pos3_3_b_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_b_full) - 1 DO BEGIN
				sum_defl_pos3_b_full[mm] = sum_defl_pos3_0_b_full[mm]+sum_defl_pos3_1_b_full[mm]+sum_defl_pos3_2_b_full[mm]+sum_defl_pos3_3_b_full[mm]
			ENDFOR
			
			END

		'stop0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf0', sf0_b_full, sum_defl_sf0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_b_full, sum_defl_sfr0_b_full
			
			FOR kk = 0D, n_elements(sf0_b_full) - 1 DO BEGIN
			IF (sf0_b_full[kk] NE 0) AND (FINITE(sf0_b_full[kk]) EQ 1) AND (FINITE(sfr0_b_full[kk]) EQ 1) THEN BEGIN
				stop0_b_full[kk] = sfr0_b_full[kk]/sf0_b_full[kk]
			ENDIF ELSE BEGIN
				stop0_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_b_full) - 1 DO BEGIN
			IF (sum_defl_sf0_b_full[mm] NE 0) AND (FINITE(sum_defl_sf0_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_b_full[mm] = sum_defl_sfr0_b_full[mm]/sum_defl_sf0_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'start0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp0', stp0_b_full, sum_defl_stp0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_b_full, sum_defl_sfr0_b_full
			
			FOR kk = 0D, n_elements(stp0_b_full) - 1 DO BEGIN
			IF (stp0_b_full[kk] NE 0) AND (FINITE(stp0_b_full[kk]) EQ 1) AND (FINITE(sfr0_b_full[kk]) EQ 1) THEN BEGIN
				start0_b_full[kk] = sfr0_b_full[kk]/stp0_b_full[kk]
			ENDIF ELSE BEGIN
				start0_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_b_full) - 1 DO BEGIN
			IF (sum_defl_stp0_b_full[mm] NE 0) AND (FINITE(sum_defl_stp0_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_b_full[mm] = sum_defl_sfr0_b_full[mm]/sum_defl_stp0_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'stop2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf2', sf2_b_full, sum_defl_sf2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_b_full, sum_defl_sfr2_b_full
			
			FOR kk = 0D, n_elements(sf2_b_full) - 1 DO BEGIN
			IF (sf2_b_full[kk] NE 0) AND (FINITE(sf2_b_full[kk]) EQ 1) AND (FINITE(sfr2_b_full[kk]) EQ 1) THEN BEGIN
				stop2_b_full[kk] = sfr2_b_full[kk]/sf2_b_full[kk]
			ENDIF ELSE BEGIN
				stop2_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_b_full) - 1 DO BEGIN
			IF (sum_defl_sf2_b_full[mm] NE 0) AND (FINITE(sum_defl_sf2_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_b_full[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_sf2_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

		'start2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp2', stp2_b_full, sum_defl_stp2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_b_full, sum_defl_sfr2_b_full
			
			FOR kk = 0D, n_elements(stp2_b_full) - 1 DO BEGIN
			IF (stp2_b_full[kk] NE 0) AND (FINITE(stp2_b_full[kk]) EQ 1) AND (FINITE(sfr2_b_full[kk]) EQ 1) THEN BEGIN
				start2_b_full[kk] = sfr2_b_full[kk]/stp2_b_full[kk]
			ENDIF ELSE BEGIN
				start2_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_b_full) - 1 DO BEGIN
			IF (sum_defl_stp2_b_full[mm] NE 0) AND (FINITE(sum_defl_stp2_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_b_full[mm] = sum_defl_sfr2_b_full[mm]/sum_defl_stp2_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

            ENDCASE
        ENDFOR
	END
    ENDCASE
    mon_full_read = 1
END 

PRO check_indices_mon_full, index, epoch, resolution, file_num
compile_opt strictarrsubs

; checks for missing records from the cdf epoch variable
; index: array of pairs [start_index,num_records] (could be [0,10,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1,5, or 3 depending on epoch (may need to add variable)
; file_num: index of this file in the file array

first_index=0L
first_index = file_num*(1440/resolution)

num_records = 0L
in_valid_area = 0
ii = 0L
index = [-1L]
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 59 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*60)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]                
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
;    5: BEGIN 
;        FOR hour = 0, 23 DO BEGIN 
;            FOR min = 0, 11 DO BEGIN 
;                IF ii LT n_elements(epoch) THEN BEGIN 
;                    time = time_struct(epoch[ii])
;                    IF (hour EQ time.hour AND time.min GE (min*5) AND time.min LT (min*5+5)) OR $
;                      (time.hour EQ hour+1 AND time.min LE 4 AND min EQ 11) THEN BEGIN 
;                        IF in_valid_area EQ 0 THEN BEGIN
;                            start_index = first_index+(hour*12)+min
;                            num_records = 1
;                            in_valid_area = 1
;                        ENDIF ELSE BEGIN ; already in valid area
;                            num_records = num_records+1
;                        ENDELSE 
;                        ii = ii+1
;                    ENDIF ELSE BEGIN
;                        IF in_valid_area EQ 1 THEN BEGIN
;                            in_valid_area = 0
;                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
;                            ELSE index = [index, start_index, num_records]
;                        ENDIF 
;                    ENDELSE 
;                ENDIF ELSE BEGIN 
;                    IF in_valid_area EQ 1 THEN BEGIN
;                        in_valid_area = 0
;                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
;                        ELSE index = [index, start_index, num_records]
;                    ENDIF 
;                ENDELSE 
;            ENDFOR 
;        ENDFOR 
;    END
    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf_mon_full, sat, files, products, which_epoch, E_step
compile_opt strictarrsubs

COMMON share1_mon_full
COMMON com_mon_full
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat

; initialize output variables (from common block)
need_epoch1 = 0
;need_epoch5 = 0

FOR ii = 0L, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
        1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
;	    num_records5 = n_elements(files)*289
;            epoch5 = dblarr(num_records5)
;            epoch5(*) = !values.f_nan
;            need_epoch5 = 5

        END 
;        5: BEGIN 
;            num_records5 = n_elements(files)*289
;            epoch5 = dblarr(num_records5)
;            epoch5(*) = !values.f_nan
;            need_epoch5 = 5
;        END 
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 
mon_full_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	's_valid' : ;0
	's_e_not_req': ;1
	's_e_req': ;2
	's_no_pos': ;3
	's_mult_pos': ;4
	's_no_e': ;5
	's_mult_e': ;6
	'ra_sat_a': ;7
	'ra_sat_b': ;8
	'ra_sat_both': ;9
	'ssd_sw': ;10
	'ssd_st': ;11
	'sf0': ;12
	'sfr0': ;13
	'stp0': ;14
	'ra_trig': ;15
	'pos1_0': ;16
	'pos1_1': ;17
	'w_no_pos': ;18
	'w_mult_pos': ;19
	'w_valid': ;20
	'sf2': ;21
	'sfr2': ;22
	'stp2': ;23
	'pos2_0': ;24
	'pos2_1': ;25
	'pos2_2': ;26
	'pos2_3': ;27
	'pos3_0': ;28
	'pos3_1': ;29
	'pos3_2': ;30
	'pos3_3': ;31
	'stop0':
	'start0':
	'stop2':
	'start2':
	'pos2':
	'pos3': IF mon_full_initialized EQ 0 THEN init_mon_full, sat, num_records1, mon_full_initialized, E_step, products
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch1', temp_epoch, rec_count = epoch1info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_mon_full, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data1_index[kk+1]-1)]
                            converted_index = converted_index+data1_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF
		remove_fill_mon_full, -1E31, epoch1
;		cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5_mon', /zvariable
;                    IF epoch5info.maxrec GE 0 THEN BEGIN 
;                        cdf_varget, cdf_id, 'epoch5_mon', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
;			converted_epoch5 = time_double(temp_epoch, /epoch)
;                        check_indices_mon_full, data5_index, converted_epoch5, 5, ii
;                        converted_index5 = 0
;                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
;                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
;                              converted_epoch5[converted_index5:(converted_index5+data5_index[kk+1]-1)]
;                            converted_index5 = converted_index5+data5_index[kk+1]
;                            kk = kk+1
;                        ENDFOR 
;                    ENDIF 
               END 
;                5: BEGIN 
;                    cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5_mon', /zvariable
;                    IF epoch5info.maxrec GE 0 THEN BEGIN 
;                        cdf_varget, cdf_id, 'epoch5_mon', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
;                        converted_epoch = time_double(temp_epoch, /epoch)
;                        check_indices_mon_full, data5_index, converted_epoch, which_epoch, ii
;                        converted_index = 0
;                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
;                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
;                              converted_epoch[converted_index:(converted_index+data5_index[kk+1]-1)]
;                            converted_index = converted_index+data5_index[kk+1]
;                            kk = kk+1
;                        ENDFOR 
;                    ENDIF 
;                END 
                0: BEGIN 
                    print, "Lorna: add variable epoch"
                    stop
                END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            ;IF need_epoch5 EQ 5 THEN epoch5_a = epoch5
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            ;IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
        ENDELSE 

        ; get other data
        mon_full_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		's_valid' : ;0
		's_e_not_req': ;1
		's_e_req': ;2
		's_no_pos': ;3
		's_mult_pos': ;4
		's_no_e': ;5
		's_mult_e': ;6
		'ra_sat_a': ;7
		'ra_sat_b': ;8
		'ra_sat_both': ;9
		'ssd_sw': ;10
		'ssd_st': ;11
		'sf0': ;12
		'sfr0': ;13
		'stp0': ;14
		'ra_trig': ;15
		'pos1_0': ;16
		'pos1_1': ;17
		'w_no_pos': ;18
		'w_mult_pos': ;19
		'w_valid': ;20
		'sf2': ;21
		'sfr2': ;22
		'stp2': ;23
		'pos2_0': ;24
		'pos2_1': ;25
		'pos2_2': ;26
		'pos2_3': ;27
		'pos3_0': ;28
		'pos3_1': ;29
		'pos3_2': ;30
		'pos3_3': ;31
		'stop0':
		'start0':
		'stop2':
		'start2':
		'pos2':
		'pos3': $
	IF mon_full_read EQ 0 THEN read_mon_full, cdf_id, sat, mon_full_read, data1_index, epoch1info.maxrec+1, products, num_records1, E_step
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
        cdf_close, cdf_id
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
   ; IF need_epoch5 EQ 5 THEN data5_index_a = data5_index
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
   ; IF need_epoch5 EQ 5 THEN data5_index_b = data5_index
ENDELSE 

END 
