;+
;PROCEDURE: pl_plot_obmom_from_crib
;PURPOSE:
;  It is a crib sheet for plotting stereo-plastic OB moments using
;  the TPLOT package
;
;INPUT
;PARAMETERS:   sat_a_mom: Array of moments wanted for sat A
;              sat_b_mom: Array of moments wanted for sat B
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 09/11/2006
;
;MODIFICATION HISTORY:
;    09/11/2006 Adjusted from C. Mouikis's plot_obmom_from_crib
;    02/09/2007 Changed set_plot_attributes to set_plot_attributes_mom
;
;    02/28/2007: The PS output option is taken out (C. Mouikis)	
;    03/28/2007: Added check for empty files (LBE)
;    04/02/2009: Removed set_plot_attributes. (LBE)
;    10/19/2009: Added update ratios. Now assumes only one day.
;    05/27/2015: Add closed_entrance keyword.
;    10/12/2015: Added svalid_ratio keyword.
;    11/02/2015: Changed make_moment_struct to _a and _b
;    12/18/2015: Added read_only keyword so can run without updating ratios
;    12/19/2015: Add no_display keyword.
;-
PRO check_prod, prod
compile_opt strictarrsubs
; in: string with product abbreviation
SWITCH prod OF 
    'D':
    'VX':
    'VY':
    'VZ':
    'VXYZ':
    'VT': 
    'TXX':
    'TYY':
    'TZZ':
    'TXXYYZZ':
    'PXX':
    'PYY':
    'PZZ':
    'PXXYYZZ': BREAK 
    ELSE: BEGIN 
        print, 'Invalid element in moment array: ', prod
        stop
    ENDELSE
ENDSWITCH
END 


PRO check_moment_array, moment_array, a_array, b_array
compile_opt strictarrsubs
; in: moment_array -- array from crib sheet
; out: a_array,b_array -- same as moment array, but divided by satellite
FOR ii = 0, N_ELEMENTS(moment_array)-1 DO BEGIN
    temp_st = moment_array[ii]
    IF strmid(temp_st, 0, 1) EQ 'A' THEN BEGIN 
        prod = strmid(temp_st, 2, strlen(temp_st)-2)
        IF n_elements(a_array) EQ 0 THEN a_array = [prod] ELSE a_array = [a_array, prod]
        check_prod, prod
    ENDIF ELSE BEGIN 
        IF strmid(temp_st, 0, 1) EQ 'B' THEN BEGIN 
            prod = strmid(temp_st, 2, strlen(temp_st)-2)
            IF n_elements(b_array) EQ 0 THEN b_array = [prod] ELSE b_array = [b_array, prod]
            check_prod, prod
        ENDIF ELSE BEGIN 
            print, 'Invalid element in moment array: ', moment_array[ii]
            stop
        ENDELSE   
    ENDELSE 
ENDFOR  
END


PRO pl_plot_obmom_from_crib, sat_moment, $
  INST_COORD = INST_COORD, closed_entrance = closed_entrance, $
  read_only = read_only, no_display = no_display
compile_opt strictarrsubs

  COMMON get_error, get_err_no, get_err_msg, default_verbose
  COMMON com_moment
  specie_str = 'H!U+!N'

  check_moment_array, sat_moment, sat_a_mom, sat_b_mom

  ; First update ratios, just copying from yesterday
  get_timespan, tt
  time_st = time_struct(tt)
  IF time_st[0].doy EQ 1 THEN BEGIN
      read_ratios, 'A', time_st[0].year-1, old_svalid_ratio_a, old_pri0_ratio_a
      read_ratios, 'B', time_st[0].year-1, old_svalid_ratio_b, old_pri0_ratio_b
  ENDIF 
  read_ratios, 'A', time_st[0].year,      svalid_ratio_a, pri0_ratio_a
  read_ratios, 'B', time_st[0].year,      svalid_ratio_b, pri0_ratio_b
  filled_a = where(abs(svalid_ratio_a[time_st[0].doy-1, *]) GT .0001, filled_a_count)
  filled_b = where(abs(svalid_ratio_b[time_st[0].doy-1, *]) GT .0001, filled_b_count)
  IF filled_a_count EQ 0 THEN BEGIN 
      IF time_st[0].doy EQ 1 THEN BEGIN
          IF time_st[0].year-1 EQ 2008 OR time_st[0].year-1 EQ 2012 OR time_st[0].year-1 EQ 2016 OR time_st[0].year-1 EQ 2020 THEN BEGIN
              svalid_ratio_a[0, *] = old_svalid_ratio_a[365, *]
          ENDIF ELSE BEGIN
              svalid_ratio_a[0, *] = old_svalid_ratio_a[364, *]
          ENDELSE 
      ENDIF ELSE BEGIN 
          svalid_ratio_a[time_st[0].doy-1, *] = svalid_ratio_a[time_st[0].doy-2, *]
      ENDELSE 
  ENDIF 
  IF filled_b_count EQ 0 THEN BEGIN 
      IF time_st[0].doy EQ 1 THEN BEGIN
          IF time_st[0].year-1 EQ 2008 OR time_st[0].year-1 EQ 2012 OR time_st[0].year-1 EQ 2016 OR time_st[0].year-1 EQ 2020 THEN BEGIN
              svalid_ratio_b[0, *] = old_svalid_ratio_b[365, *]
          ENDIF ELSE BEGIN
              svalid_ratio_b[0, *] = old_svalid_ratio_b[364, *]
          ENDELSE 
      ENDIF ELSE BEGIN 
          svalid_ratio_b[time_st[0].doy-1, *] = svalid_ratio_b[time_st[0].doy-2, *]
      ENDELSE 
  ENDIF 
  filled_a = where(abs(pri0_ratio_a[time_st[0].doy-1, *]) GT .0001, filled_a_count)
  filled_b = where(abs(pri0_ratio_b[time_st[0].doy-1, *]) GT .0001, filled_b_count)
  IF filled_a_count EQ 0 THEN BEGIN 
      IF time_st[0].doy EQ 1 THEN BEGIN
          IF time_st[0].year-1 EQ 2008 OR time_st[0].year-1 EQ 2012 OR time_st[0].year-1 EQ 2016 OR time_st[0].year-1 EQ 2020 THEN BEGIN
              pri0_ratio_a[0, *] = old_pri0_ratio_a[365, *]
          ENDIF ELSE BEGIN
              pri0_ratio_a[0, *] = old_pri0_ratio_a[364, *]
          ENDELSE 
      ENDIF ELSE BEGIN 
          pri0_ratio_a[time_st[0].doy-1, *] = pri0_ratio_a[time_st[0].doy-2, *]
      ENDELSE 
    ENDIF 
  IF filled_b_count EQ 0 THEN BEGIN 
      IF time_st[0].doy EQ 1 THEN BEGIN
          IF time_st[0].year-1 EQ 2008 OR time_st[0].year-1 EQ 2012 OR time_st[0].year-1 EQ 2016 OR time_st[0].year-1 EQ 2020 THEN BEGIN
              pri0_ratio_b[0, *] = old_pri0_ratio_b[365, *]
          ENDIF ELSE BEGIN
              pri0_ratio_b[0, *] = old_pri0_ratio_b[364, *]
          ENDELSE 
      ENDIF ELSE BEGIN 
          pri0_ratio_b[time_st[0].doy-1, *] = pri0_ratio_b[time_st[0].doy-2, *]
      ENDELSE 
  ENDIF 

  IF keyword_set(read_only) EQ 0 THEN BEGIN 
      update_ratios,    time_st[0].year, 'A', svalid_ratio_a, pri0_ratio_a
      update_ratios,    time_st[0].year, 'B', svalid_ratio_b, pri0_ratio_b
  ENDIF 

  ; Figure out which files we need to read from
  IF n_elements(sat_a_mom) GT 0 THEN seek_files, 'A', 'science', files_a ELSE files_a = ['']
  IF n_elements(sat_b_mom) GT 0 THEN seek_files, 'B', 'science', files_b ELSE files_b = ['']
  dummy = where(strcmp(files_a, '') NE 1, num_a_valid_files)
  dummy = where(strcmp(files_b, '') NE 1, num_b_valid_files)
  IF num_a_valid_files EQ 0 AND n_elements(sat_a_mom) GT 0 THEN print, "No files found for spacecraft A"
  IF num_b_valid_files EQ 0 AND n_elements(sat_b_mom) GT 0 THEN print, "No files found for spacecraft B"

  ; Read data
  IF num_a_valid_files GT 0 AND n_elements(sat_a_mom) GT 0 THEN pl_read_cdf_mom, 'A', files_a, sat_a_mom, 1
  IF num_b_valid_files GT 0 AND n_elements(sat_b_mom) GT 0 THEN pl_read_cdf_mom, 'B', files_b, sat_b_mom, 1

  ; Check for empty files
  IF num_a_valid_files GT 0 THEN BEGIN 
      dummy = where(strcmp(files_a, '') NE 1, num_a_valid_files)
      IF num_a_valid_files EQ 0 AND n_elements(sat_a_mom) GT 0 THEN print, "Only empty files found for spacecraft A"
  ENDIF 
  IF num_b_valid_files GT 0 THEN BEGIN 
      dummy = where(strcmp(files_b, '') NE 1, num_b_valid_files)
      IF num_b_valid_files EQ 0 AND n_elements(sat_b_mom) GT 0 THEN print, "Only empty found for spacecraft B"
  ENDIF 

  ; Read conversion factor
  IF num_a_valid_files GT 0 AND n_elements(sat_a_mom) GT 0 THEN pl_read_mom_conv, 'A'
  IF num_b_valid_files GT 0 AND n_elements(sat_b_mom) GT 0 THEN pl_read_mom_conv, 'B'

  ; Put in tplot structures
  IF num_a_valid_files GT 0 AND n_elements(sat_a_mom) GT 0 THEN make_moment_struct_a, 'A', closed_entrance = closed_entrance, $
    read_only = read_only
  IF num_b_valid_files GT 0 AND n_elements(sat_b_mom) GT 0 THEN make_moment_struct_b, 'B', closed_entrance = closed_entrance, $
    svalid_ratio = svalid_ratio_b, read_only = read_only

  ; Make the plots
  IF keyword_set(no_display) EQ 0 THEN $
    IF (num_a_valid_files GT 0 AND n_elements(sat_a_mom) GT 0) $
      OR (num_b_valid_files GT 0 AND n_elements(sat_b_mom) GT 0) THEN $
    plot_moments, sat_moment, num_a_valid_files, num_b_valid_files

END
