;+
;PROCEDURE: make_moment_struct_b
;PURPOSE: Create tplot structures from data
;
; NB: Must be run in SolarSoft!!!
;
;PARAMETERS:   sat : 'A' or 'B'
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 10/14/2006
;
;MODIFICATION HISTORY:
;    10/14/2006 Created
;    01/23/2007 LBE added diagonalization
;    02/01/2007 LBE fixed bug in density
;    07/04/2007 LBE started using mom_eff
;    08/01/2007 LBE major rewrite of efficiencies
;    09/10/2007 LBE added vel_groups
;    09/21/2007 LBE removed low velocities from vel_box
;    10/03/2007 LBE look for zeros in ratios
;    12/06/2007 LBE added velocity theta and phi
;    12/18/2007 LBE added corr_vel (Lisa's corrections for components)
;    02/26/2008 LBE added corr_t
;    04/18/2008 LBE added check for valid data
;    04/30/2008 LBE added abberation angle (instead of 31) 
;    05/13/2008 LBE multiply angles by -1 for B
;    09/17/2008 LBE changed typo of valid to valid_i
;    12/19/2008 LBE Changed E/W to Toni algorithm.
;    03/12/2009 LBE Updated temporal offset. Rewrote for clarity.
;    03/31/2009 LBE Added interpolations after correction to Vx.
;    05/12/2009 LBE Fixed bug in when to use data_indices.
;    05/22/2009 LBE Change Vx->Vr, etc for HERTN, RTN.
;                   Fixed bug so epoch1 has right number of elements.
;    05/26/2009 LBE Put update ratios in here. Now assumes only running one day.
;    06/09/2009 LBE Added temporal offset for N/S and Temperature_xx.
;    06/15/2009 LBE Change how I flip the N/S
;    06/18/2009 LBE Added temporal offset starting at 9/1/2008.
;    06/19/2009 LBE Aberration now taken out in HERTN and RTN (to match RA method)
;    07/13/2009 LBE Added offset for temperature and NS.
;    07/20/2009 LBE Added offset for Density
;    07/21/2009 LBE Added saved_density for temperature processing
;    10/06/2009 LBE Artificially flip E/W for B to match RA. DON'T KNOW IF THIS IS CORRRECT!!!
;    10/07/2009 LBE Added E/W temporal offset through 9/1/09.
;    10/16/2009 LBE Update ratios when go off end of energies (low energy).
;    11/24/2009 LBE Update offsets through 9/30/2009.
;    12/23/2009 LBE Update offsets through 10/31/2009.
;    03/11/2010 LBE Update offsets through 01/31/2010.
;    04/13/2010 LBE Update offsets through 02/28/2010.
;    07/01/2010 LBE Update offsets.
;    07/27/2010 LBE Add run_ave.
;    10/20/2010 LBE Update offsets.
;    11/01/2010 LBE Check for all zeros and set to NaN.
;                   Don't remove ns_inst when > 20.
;    11/09/2010 LBE Add E/W offset for B. Remove N/S offset.
;    11/23/2010 LBE Put back in N/S offset for A.
;    12/21/2010 LBE Remove E/W offset for B after 2007.
;    02/04/2011 LBE Check for invalid pri ratio (missing data).
;    06/14/2011 LBE Update offsets through 3/01/2011.
;    11/29/2011 LBE Update offsets through 4/11/2011.
;    07/17/2012 LBE Update E/W offsets.
;    07/24/2013 LBE Fix bug when one sw_priority value.
;    09/29/2014 LBE Check for finite temp_ratio
;    05/27/2015 LBE Add closed_entrance keyword.
;    05/28/2015 LBE Don't copy zero values in svalid_ratrig eff.
;    06/18/2015 LBE New approach for efficiencies. Take 1-hr boxcar average of density and velocity. If den < .5, don't use that vel
;    for finding efficiencies.
;    07/03/2015 LBE Fix but that had < rather than LT.
;    08/03/2015 LBE Take into account if temp_ratio = NaN
;    10/13/2015 LBE Take running median of svalid_ratrig efficiency for STB.
;    10/14/2015 LBE Large rewrite to simplify density calculation.
;    10/22/2015 LBE Fix 90 day running median (had is as 80 days.)
;    11/02/2015 LBE Changed to make_moment_struct_b
;    11/03/2015 LBE Allow 90 day running median to cross years.
;    12/18/2015 LBE Added read_only keyword
;    12/22/2015 LBE Put in values for eff [16:22]
;    11/02/2016 LBE Fixed bug so poor_stats flag set when too few counts to get average
;    09/13/2017 LBE Fix problem when no valid ratio data (from _a 8/5/2015)
;-
PRO make_moment_struct_b, sat, closed_entrance = closed_entrance, svalid_ratio = svalid_ratio, read_only = read_only
compile_opt strictarrsubs
COMMON share1
COMMON com_moment

amp_toni = 11.25
fullscale_toni = 19.6875

epoch1            = epoch1_b
error1            = error1_b
mom_meta          = mom_meta_b
velocity_main     = mom_velocity_main_b 
velocity_s        = mom_velocity_s_b
density_main      = mom_density_main_b 
density_s         = mom_density_s_b
heat_flux_main    = mom_heat_flux_main_b
heat_flux_s       = mom_heat_flux_s_b 
temperature_main  = mom_temperature_main_b
temperature_s     = mom_temperature_s_b 
step_var          = step_var_b
table_norm        = table_norm_b
geom              = geom_b
doys              = doys_b
years             = years_b
svalid_ratrig     = svalid_ratrig_b
ra_trig_eff       = ra_trig_eff_b
vel_groups        = vel_groups_b
pri_ratio         = pri_ratio_b
mom_cal_start     = mom_cal_start_b
mom_cal_stop      = mom_cal_stop_b

bad_i = where(finite(epoch1) NE 1, bad_count, complement = good_i, ncomplement = good_count)
IF bad_count GT 0 THEN BEGIN
    IF good_count GT 0 THEN BEGIN 
        epoch1            = epoch1          [   good_i]
        error1            = error1          [   good_i]
        mom_meta          = mom_meta        [*, good_i]
        velocity_main     = velocity_main   [*, good_i]
        velocity_s        = velocity_s      [*, good_i]
        density_main      = density_main    [   good_i]
        density_s         = density_s       [   good_i]
        heat_flux_main    = heat_flux_main  [*, good_i]
        heat_flux_s       = heat_flux_s     [*, good_i]
        temperature_main  = temperature_main[*, good_i]
        temperature_s     = temperature_s   [*, good_i]
    ENDIF ELSE stop ; no valid data -- have to deal with this
ENDIF 
bad_i = where(velocity_main   [0, *] EQ 0 AND velocity_main   [1, *] EQ 0 AND velocity_main   [2, *] EQ 0 AND $
              velocity_s      [0, *] EQ 0 AND velocity_s      [1, *] EQ 0 AND velocity_s      [2, *] EQ 0 AND $
              density_main    [   *] EQ 0 AND $
              density_s       [   *] EQ 0 AND $
              heat_flux_main  [0, *] EQ 0 AND heat_flux_main  [1, *] EQ 0 AND heat_flux_main  [2, *] EQ 0 AND $
              heat_flux_s     [0, *] EQ 0 AND heat_flux_s     [1, *] EQ 0 AND heat_flux_s     [2, *] EQ 0 AND $
              temperature_main[0, *] EQ 0 AND temperature_main[1, *] EQ 0 AND temperature_main[2, *] EQ 0 AND $
              temperature_main[3, *] EQ 0 AND temperature_main[4, *] EQ 0 AND temperature_main[5, *] EQ 0 AND $
              temperature_s   [0, *] EQ 0 AND temperature_s   [1, *] EQ 0 AND temperature_s   [2, *] EQ 0 AND $
              temperature_s   [3, *] EQ 0 AND temperature_s   [4, *] EQ 0 AND temperature_s   [5, *] EQ 0, $
              bad_count, complement = good_i, ncomplement = good_count)
IF bad_count GT 0 THEN BEGIN
    IF good_count GT 0 THEN BEGIN 
        epoch1            = epoch1          [   good_i]
        error1            = error1          [   good_i]
        mom_meta          = mom_meta        [*, good_i]
        velocity_main     = velocity_main   [*, good_i]
        velocity_s        = velocity_s      [*, good_i]
        density_main      = density_main    [   good_i]
        density_s         = density_s       [   good_i]
        heat_flux_main    = heat_flux_main  [*, good_i]
        heat_flux_s       = heat_flux_s     [*, good_i]
        temperature_main  = temperature_main[*, good_i]
        temperature_s     = temperature_s   [*, good_i]
    ENDIF ELSE BEGIN ; take down to one record so will be removed
        epoch1            = epoch1          [   0]
        error1            = error1          [   0]
        mom_meta          = mom_meta        [*, 0]
        velocity_main     = velocity_main   [*, 0]
        velocity_s        = velocity_s      [*, 0]
        density_main      = density_main    [   0]
        density_s         = density_s       [   0]
        heat_flux_main    = heat_flux_main  [*, 0]
        heat_flux_s       = heat_flux_s     [*, 0]
        temperature_main  = temperature_main[*, 0]
        temperature_s     = temperature_s   [*, 0]
    ENDELSE 
ENDIF 

find_cal_indices, epoch1, density_s, mom_cal_start, mom_cal_stop, data_indices, cal_file_index
num_records = n_elements(data_indices)
velocity    = dblarr(3, num_records)
density     = dblarr(num_records)
heat_flux   = dblarr(3, num_records)
pressure_t  = dblarr(6, num_records)

IF num_records LT 2 THEN BEGIN
    temp_i = n_elements(doys)
    temp_string = 'No valid data for '+sat+' from '+string(doys[0])+'/'+string(years[0])+' to '+$
                  string(doys[temp_i-1])+'/'+string(years[temp_i-1])
    print, temp_string
ENDIF ELSE BEGIN 
    IF num_records NE n_elements(epoch1) THEN BEGIN 
        epoch1           = epoch1          [   data_indices]
        error1           = error1          [   data_indices]
        mom_meta         = mom_meta        [*, data_indices]
        velocity_main    = velocity_main   [*, data_indices]
        velocity_s       = velocity_s      [*, data_indices]
        density_main     = density_main    [   data_indices]
        density_s        = density_s       [   data_indices]
        heat_flux_main   = heat_flux_main  [*, data_indices]
        heat_flux_s      = heat_flux_s     [*, data_indices]
        temperature_main = temperature_main[*, data_indices]
        temperature_s    = temperature_s   [*, data_indices]
    ENDIF 
; errors
    labels = ''
    name = sat+'_obmom_error'
    datastr = {x:epoch1, y:error1}
    store_data, name, data = datastr, dlim = {ylog:0, labels:labels, panel_size:2.} 

; take running average of all data
;weights = [.25, .5, .25] 
;weights = [.1111, .2222, .3334, .2222, .1111] 
;weights = [.0625, .125, .1875, .25, .1875, .125, .0625]
    weights = [.04, .08, .12, .16, .20, .16, .12, .08, .04]
    velocity_main    = transpose(velocity_main)
    velocity_s       = transpose(velocity_s)
    heat_flux_main   = transpose(heat_flux_main)
    heat_flux_s      = transpose(heat_flux_s)
    temperature_main = transpose(temperature_main)
    temperature_s    = transpose(temperature_s)
    run_ave, velocity_main,    n_elements(weights), weights, accum = 0
    run_ave, velocity_s,       n_elements(weights), weights, accum = 0
    run_ave, density_main,     n_elements(weights), weights, accum = 0
    run_ave, density_s,        n_elements(weights), weights, accum = 0
    run_ave, heat_flux_main,   n_elements(weights), weights, accum = 0
    run_ave, heat_flux_s,      n_elements(weights), weights, accum = 0
    run_ave, temperature_main, n_elements(weights), weights, accum = 0
    run_ave, temperature_s,    n_elements(weights), weights, accum = 0
    velocity_main    = transpose(velocity_main)
    velocity_s       = transpose(velocity_s)
    heat_flux_main   = transpose(heat_flux_main)
    heat_flux_s      = transpose(heat_flux_s)
    temperature_main = transpose(temperature_main)
    temperature_s    = transpose(temperature_s)

; do actual calibration
;velocity
    FOR ii = 0L, num_records-1 DO BEGIN 
        i_cal = cal_file_index[ii]
        prefix = table_norm[i_cal, 1]/table_norm[i_cal, 0] ; Bv/Bd in Lynn's document
        ;denom = (density_s[ii]/geom[i_cal, 1])
        denom = (density_s[ii]/geom[i_cal, 1]) + $ ; Ds/Gs + Dm/Gm
                (density_main[ii]/geom[i_cal, 0]) 
        FOR jj = 0, 2 DO BEGIN  ; x,y,z
            ;numer = (velocity_s[jj, ii]/geom[i_cal, 1])
            numer = (velocity_s[jj, ii]/geom[i_cal, 1]) + $ ; Vx,y,zs/Gs + Vx,y,zm/Gm
                    (velocity_main[jj, ii]/geom[i_cal, 0]) 
            velocity[jj, ii] = prefix * numer / denom
        ENDFOR
    ENDFOR

; instrument coordinates
    vx_inst_d              = reform(velocity[0, *])
    vy_inst_d              = reform(velocity[1, *])
    vz_inst_d              = reform(velocity[2, *])
    temp_bulk_speed_inst_d = sqrt((vx_inst_d^2)+(vy_inst_d^2)+(vz_inst_d^2))

; LBE temp code 6/23/2016
; first store old data
    ;bulk_speed_inst_d    = dblarr(num_records)
    ;bulk_speed_inst_d[*] = !values.f_nan
    ;FOR ii = 0L, num_records-1 DO BEGIN 
    ;   i_cal = cal_file_index[ii]
    ;   bulk_speed_inst_d[ii] = interpol(reform(ra_trig_eff[i_cal, *, 1]), reform(ra_trig_eff[i_cal, *, 2]), $
    ;                                    temp_bulk_speed_inst_d[ii])
    ;ENDFOR
    ;store_data,'vel_raw',data={x:epoch1,y:temp_bulk_speed_inst_d}
    ;store_data,'vel_orig_table',data={x:epoch1,y:bulk_speed_inst_d}
; trying a different Moments's table
    ;temp_file = '~/Temp/Plastic_Moments_table_070413.txt'
    ;openr, temp_lun, temp_file, /get_lun
    ;line = ''
    ;FOR ii = 0, 11 DO BEGIN
    ;   readf, temp_lun, line
    ;ENDFOR
    ;FOR ii = 0, 127 DO BEGIN
    ;   readf, temp_lun, line
    ;   parts = strsplit(line, /extract)
    ;   ra_trig_eff[i_cal, ii, 2] = float(parts[6])*440
    ;ENDFOR 
    ;close, temp_lun
    ;free_lun, temp_lun

; LBE temp code 4/26/2016
;temp_peak = intarr(n_elements(epoch1))
;FOR ii = 0, n_elements(epoch1)-1 DO BEGIN 
;    temp_peak[ii] = min(where(temp_bulk_speed_inst_d[ii] GE ra_trig_eff[i_cal, *, 2]))
;ENDFOR
;store_data, 'ob_peak_backwards', data = {x:epoch1, y:temp_peak}
;stop

; reform velocity using Kristin's velocities
    bulk_speed_inst_d    = dblarr(num_records)
    bulk_speed_inst_d[*] = !values.f_nan
    FOR ii = 0L, num_records-1 DO BEGIN 
        i_cal = cal_file_index[ii]
        bulk_speed_inst_d[ii] = interpol(reform(ra_trig_eff[i_cal, *, 1]), reform(ra_trig_eff[i_cal, *, 2]), $
                                         temp_bulk_speed_inst_d[ii])
    ENDFOR

; LBE temp code 6/23/2016
    ;store_data,'vel_070413_table',data={x:epoch1,y:bulk_speed_inst_d}
    ;stop
    
    ; angles
    ; ns
    ns_inst_d = 180 / !PI * ATAN(vz_inst_d / SQRT((vx_inst_d ^ 2)+(vy_inst_d ^ 2)))
    ;ns_inst_d = (1.5D * ns_inst_d) + 4.75  ; from old version (pre 3/17/2009)
    ; Kristin's algorithm for N/S
    IF sat EQ 'A' THEN BEGIN  ; NOTE: OB needs an offset because of deflection wobble (3/17/2009)
        scaled_ns = 3.0*(ns_inst_d + 3.5) ; 3.5 shifts center of dist., 3.0 spreads out dist.
        ns_inst_d = scaled_ns + (-2.0854E-7)*(bulk_speed_inst_d^3)+(3.0371E-4)*(bulk_speed_inst_d^2) + $
                  (-1.2454E-1)*bulk_speed_inst_d+1.0983E1 ; modified 27 August, 2008
                                ; above line comes from velocity dependence of leakage (compared to other spacecraft)
    ENDIF ELSE BEGIN 
        ns_inst_d = -1.0 * ns_inst_d ; flip distribution 
        scaled_ns = 1.75*(ns_inst_d - 4.62) ; modified 14 August, 2008
        ns_inst_d = scaled_ns + (-2.6404E-7)*(bulk_speed_inst_d^3) + (4.3726E-4)*(bulk_speed_inst_d^2) + $
                     (-2.3922E-1)*bulk_speed_inst_d+4.2639E1
    ENDELSE 

    ; ew
    ew_inst_d = 180 / !PI * ATAN(vy_inst_d / vx_inst_d)
    ;IF sat EQ 'B' THEN ew_inst_d = -1.0 * ew_inst_d
    ew_inst_d = (amp_toni*(sinh((ew_inst_d/fullscale_toni))))

    ; add temporal offset for E/W and N/S
    time_struct = time_struct(epoch1)
    temp_year = time_struct.year
    temp_doy = time_struct.doy
    indices = where(temp_year EQ 2008, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365
    indices = where(temp_year EQ 2009, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366
    indices = where(temp_year EQ 2010, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365
    indices = where(temp_year EQ 2011, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365
    indices = where(temp_year EQ 2012, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365
    indices = where(temp_year EQ 2013, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366
    indices = where(temp_year EQ 2014, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365
    indices = where(temp_year EQ 2015, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365+365
    indices = where(temp_year EQ 2016, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365+365+365
    indices = where(temp_year EQ 2017, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365+365+365+366
    indices = where(temp_year EQ 2018, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365+365+365+366+365
    indices = where(temp_year EQ 2019, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365+365+365+366+365+365
    indices = where(temp_year EQ 2020, count) 
    IF count GT 0 THEN temp_doy[indices] = temp_doy[indices]+365+366+365+365+365+366+365+365+365+366+365+365+365
    indices = where(temp_year GT 2020, count) ; NOTE: Only good through 2020
    IF count GT 0 THEN stop
    temp_doy = double(temp_doy)
    IF sat EQ 'A' THEN BEGIN 
        stop ; should only be STB!
    ENDIF ELSE BEGIN ; B
        indices = where (temp_doy LT 97, count) ; 01/01/2007 - 04/06/2007
        IF count GT 0 THEN BEGIN 
            ew_vel_offset      = (-1.04160E-04*(bulk_speed_inst_d[indices]^2)) $
                                 + (1.12090E-01*(bulk_speed_inst_d[indices])) - 1.63447E01
            ew_temporal_offset = 0
            ew_inst_d[indices] = ew_vel_offset + ew_temporal_offset - ew_inst_d[indices] ; this takes care of flip
        ENDIF 
        indices = where (temp_doy GE 97 AND temp_doy LT 121, count) ; 04/07/2007 - 4/30/2007
        IF count GT 0 THEN BEGIN 
            ew_vel_offset      = (0.02 * bulk_speed_inst_d[indices])
            ew_temporal_offset = (-0.00074927*(temp_doy[indices]^2)) $
                                 - (0.025417*(temp_doy[indices])) + 8.0658
            ew_inst_d[indices] = ew_vel_offset + ew_temporal_offset - ew_inst_d[indices] ; thiss takes care of flip
        ENDIF 
        indices = where (temp_doy GE 121 AND temp_doy LT 365, count) ; 05/01/2007 - 12/31/2007 (I made up end date)
        IF count GT 0 THEN BEGIN 
            ew_vel_offset      = (0.02 * bulk_speed_inst_d[indices])
            ew_temporal_offset = (-2.0613E-08 *(temp_doy[indices]^4)) $
                                 + (1.8326E-05*(temp_doy[indices]^3)) $
                                 - (5.8564E-03*(temp_doy[indices]^2)) $
                                 + (8.2148E-01*(temp_doy[indices])) - 4.7375E01
            ew_inst_d[indices] = ew_vel_offset + ew_temporal_offset - ew_inst_d[indices] ; this takes care of flip
        ENDIF 
    ENDELSE 

    ; Adjust OB Vx and find Vt
    vx_inst_d = vx_inst_d * cos(ew_inst_d * !pi / 180) / cos(20 * !pi / 180)
    bulk_speed_inst_sav_d = sqrt(vx_inst_d^2 + vy_inst_d^2 + vz_inst_d^2) ; used for finding density and updating ratios
    FOR ii = 0L, num_records-1 DO BEGIN 
        bulk_speed_inst_d[ii] = interpol(reform(ra_trig_eff[i_cal, *, 1]), reform(ra_trig_eff[i_cal, *, 2]), $
                                         bulk_speed_inst_sav_d[ii])
    ENDFOR 
    bulk_speed_inst  = sat+'_obmom_bulk_speed_inst'
    vx_inst          = sat+'_obmom_vx_inst'
    vy_inst          = sat+'_obmom_vy_inst'
    vz_inst          = sat+'_obmom_vz_inst'
    ns_inst          = sat+'_obmom_ns_inst'
    ew_inst          = sat+'_obmom_ew_inst'
    store_data, ns_inst,         data = {x: epoch1, y: ns_inst_d} ; degrees
    store_data, ew_inst,         data = {x: epoch1, y: ew_inst_d} ; degrees

    ; Unflip so coordinate transformation will work
    IF sat EQ 'B' THEN BEGIN
        ns_inst_d = -1.0 * ns_inst_d
        ew_inst_d = -1.0 * ew_inst_d
    ENDIF 

    ; Turn angles into coordinates
    vx_inst_d = bulk_speed_inst_d * cos(ns_inst_d * !pi / 180) * cos(ew_inst_d * !pi / 180)
    vy_inst_d = bulk_speed_inst_d * cos(ns_inst_d * !pi / 180) * sin(ew_inst_d * !pi / 180)
    vz_inst_d = bulk_speed_inst_d * sin(ns_inst_d * !pi / 180)

    store_data, bulk_speed_inst, data = {x: epoch1, y: bulk_speed_inst_d}
    store_data, vx_inst,         data = {x: epoch1, y: vx_inst_d}
    store_data, vy_inst,         data = {x: epoch1, y: vy_inst_d}
    store_data, vz_inst,         data = {x: epoch1, y: vz_inst_d}
    ;stop
    
    ; spacecraft coordinates  
    vx_spcrft_d         = (-1.0) * vx_inst_d ; multiplying by Lynn's matrix
    vy_spcrft_d         = vz_inst_d
    vz_spcrft_d         = (-1.0) * vy_inst_d
    ns_spcrft_d         = atan(vz_spcrft_d / sqrt((vx_spcrft_d^2)+(vy_spcrft_d^2)))
    ew_spcrft_d         = atan(vy_spcrft_d / vx_spcrft_d)
    bulk_speed_spcrft_d = sqrt((vx_spcrft_d^2)+(vy_spcrft_d^2)+(vz_spcrft_d^2))
    bulk_speed_spcrft = sat+'_obmom_bulk_speed_spcrft'
    vx_spcrft         = sat+'_obmom_vx_spcrft'
    vy_spcrft         = sat+'_obmom_vy_spcrft'
    vz_spcrft         = sat+'_obmom_vz_spcrft'
    ns_spcrft         = sat+'_obmom_ns_spcrft'
    ew_spcrft         = sat+'_obmom_ew_spcrft'
    store_data, bulk_speed_spcrft, data = {x: epoch1, y: bulk_speed_spcrft_d}
    store_data, vx_spcrft,         data = {x: epoch1, y: vx_spcrft_d}
    store_data, vy_spcrft,         data = {x: epoch1, y: vy_spcrft_d}
    store_data, vz_spcrft,         data = {x: epoch1, y: vz_spcrft_d}
    store_data, ns_spcrft,         data = {x: epoch1, y: ns_spcrft_d} ; degrees
    store_data, ew_spcrft,         data = {x: epoch1, y: ew_spcrft_d} ; degrees

    vel_spcrft_d = dblarr(3, num_records)
    vel_spcrft_d[0, *] = vx_spcrft_d
    vel_spcrft_d[1, *] = vy_spcrft_d
    vel_spcrft_d[2, *] = vz_spcrft_d

; Put in HERTN
    vel_hertn_d = dblarr(3, num_records)
    temp_time = time_string(epoch1)
    strput, temp_time, 'T', 10  ; change time string format for SolarSoft
    cmat_hertn  = get_stereo_cmat(temp_time, sat, system = 'hertn')
    FOR ii = 0L, num_records-1 DO BEGIN 
        temp_vel = reform(vel_spcrft_d[*, ii])
        temp_cmat = reform(cmat_hertn[*, *, ii])
        vel_hertn_d[*, ii] = temp_cmat # temp_vel
    ENDFOR 
    ;vel_hertn_d = cmat_hertn # vel_spcrft_d

    ; Add Aberration
    state = get_stereo_coord(temp_time, sat, system = 'hci')
    spcrft_vel = state[3:5, *]
    convert_stereo_coord, temp_time, spcrft_vel, 'hci', 'hertn', spacecraft = sat
    vel_hertn_d = vel_hertn_d + spcrft_vel

    ; recompute angles and bulk speed
    vr_hertn_d         = reform(vel_hertn_d[0, *])
    vt_hertn_d         = reform(vel_hertn_d[1, *])
    vn_hertn_d         = reform(vel_hertn_d[2, *])
    ns_hertn_d         = atan(vn_hertn_d / sqrt((vr_hertn_d^2)+(vt_hertn_d^2))) * 180 / !pi
    ew_hertn_d         = atan(vt_hertn_d / vr_hertn_d) * 180 / !pi
    bulk_speed_hertn_d = sqrt((vr_hertn_d^2)+(vt_hertn_d^2)+(vn_hertn_d^2))

    bulk_speed_hertn = sat+'_obmom_bulk_speed_hertn'
    vr_hertn         = sat+'_obmom_vr_hertn'
    vt_hertn         = sat+'_obmom_vt_hertn'
    vn_hertn         = sat+'_obmom_vn_hertn'
    ns_hertn         = sat+'_obmom_ns_hertn'
    ew_hertn         = sat+'_obmom_ew_hertn'
    store_data, bulk_speed_hertn, data = {x: epoch1, y: bulk_speed_hertn_d}
    store_data, vr_hertn,         data = {x: epoch1, y: vr_hertn_d}
    store_data, vt_hertn,         data = {x: epoch1, y: vt_hertn_d}
    store_data, vn_hertn,         data = {x: epoch1, y: vn_hertn_d}
    store_data, ns_hertn,         data = {x: epoch1, y: ns_hertn_d} ; degrees
    store_data, ew_hertn,         data = {x: epoch1, y: ew_hertn_d} ; degrees

; Put in RTN
    vel_rtn_d = dblarr(3, num_records)
    cmat_rtn  = get_stereo_cmat(temp_time, sat, system = 'rtn')
    ;vel_rtn_d = cmat_rtn # vel_spcrft_d
    FOR ii = 0L, num_records-1 DO BEGIN 
        temp_vel = reform(vel_spcrft_d[*, ii])
        temp_cmat = reform(cmat_rtn[*, *, ii])
        vel_rtn_d[*, ii] = temp_cmat # temp_vel
    ENDFOR 

    ; Add Aberration
    state = get_stereo_coord(temp_time, sat, system = 'hci')
    spcrft_vel = state[3:5, *]
    convert_stereo_coord, temp_time, spcrft_vel, 'hci', 'rtn', spacecraft = sat, /ignore_origin
    vel_rtn_d = vel_rtn_d + spcrft_vel

    ; recompute angles and bulk speed
    vr_rtn_d         = vel_rtn_d[0, *]
    vt_rtn_d         = vel_rtn_d[1, *]
    vn_rtn_d         = vel_rtn_d[2, *]
    ns_rtn_d         = atan(vn_rtn_d / sqrt((vr_rtn_d^2)+(vt_rtn_d^2))) * 180 / !pi
    ew_rtn_d         = atan(vt_rtn_d / vr_rtn_d) * 180 / !pi
    bulk_speed_rtn_d = sqrt((vr_rtn_d^2)+(vt_rtn_d^2)+(vn_rtn_d^2))

    bulk_speed_rtn = sat+'_obmom_bulk_speed_rtn'
    vr_rtn         = sat+'_obmom_vr_rtn'
    vt_rtn         = sat+'_obmom_vt_rtn'
    vn_rtn         = sat+'_obmom_vn_rtn'
    ns_rtn         = sat+'_obmom_ns_rtn'
    ew_rtn         = sat+'_obmom_ew_rtn'
    store_data, bulk_speed_rtn, data = {x: epoch1, y: bulk_speed_rtn_d}
    store_data, vr_rtn,         data = {x: epoch1, y: vr_rtn_d}
    store_data, vt_rtn,         data = {x: epoch1, y: vt_rtn_d}
    store_data, vn_rtn,         data = {x: epoch1, y: vn_rtn_d}
    store_data, ns_rtn,         data = {x: epoch1, y: ns_rtn_d} ; degrees
    store_data, ew_rtn,         data = {x: epoch1, y: ew_rtn_d} ; degrees

; fill in efficiency
    poor_stats_d       = intarr(num_records)
    efficiencies_d     = dblarr(num_records)
    pri_efficiencies_d = dblarr(num_records)
    ratios_d           = dblarr(num_records)
    vel_tmp_d          = dblarr(num_records)
    vel_sdev_d         = dblarr(num_records)
    eff_tmp_d          = dblarr(num_records)
    efficiencies_d    [*] = !values.f_nan
    pri_efficiencies_d[*] = !values.f_nan
    ratios_d          [*] = !values.f_nan 
    vel_tmp_d         [*] = !values.f_nan 
    vel_sdev_d        [*] = !values.f_nan
    eff_tmp_d         [*] = !values.f_nan 
    IF num_records GE 19 THEN     box_size = 19 ELSE     box_size = num_records 
    ;IF num_records GE 10 THEN box_size = 10 ELSE box_size = num_records ; temporary for checking beacon
    vel_box     = [bulk_speed_inst_sav_d[0]]
    FOR ii = 1, box_size-1 DO BEGIN
        vel_box = [vel_box, bulk_speed_inst_sav_d[ii]]
    ENDFOR 
    ; figure out eff ratios. 
    ; STA: remove zeros
    ; STB: use 90 day running median, and remove zeros.
    temp_time = time_struct(epoch1)
    IF array_equal(temp_time.doy, temp_time[0].doy) NE 1 THEN stop ; not all one day!
    doy = temp_time[0].doy
    IF sat EQ 'A' THEN BEGIN 
        i_doy = where(doys_a EQ doy)
        temp_svalid_ratrig = reform(svalid_ratrig[*, i_doy]) ; this is out of 2 days?
    ENDIF ELSE BEGIN 
        i_doy = where(doys_b EQ doy)
        IF doy LT 90 THEN BEGIN 
            temp_stb_ratio = fltarr(90, 32)
            read_ratios, 'B', temp_year[0]-1, old_svalid_ratio_b, old_pri0_ratio_b
            IF (temp_year[0]-1) MOD 4 EQ 0 THEN last_doy = 366 ELSE last_doy = 365
            temp_stb_ratio[ 0:(90-doy-1), *] = old_svalid_ratio_b[(last_doy-90+doy):(last_doy-1), *]
            temp_stb_ratio[(90-doy):89, *] =  svalid_ratio[0:(doy-1), *]
        ENDIF ELSE temp_stb_ratio = reform(svalid_ratio[(doy-90):doy-1, *]) ; this is out of full days
        temp_svalid_ratrig = fltarr(32)
        FOR esa_i = 24, 25 DO BEGIN
            FOR doy_i = 1, 89 DO BEGIN 
                IF abs(temp_stb_ratio[doy_i-1, esa_i]-temp_stb_ratio[doy_i, esa_i]) LT .0000001 THEN $
                  temp_stb_ratio[doy_i-1, esa_i] = !values.f_nan
            ENDFOR
            temp_svalid_ratrig[esa_i] = median(reform(temp_stb_ratio[*, esa_i]))
        ENDFOR 
    ENDELSE 
    IF temp_svalid_ratrig[24] EQ 0 OR temp_svalid_ratrig[25] EQ 0 THEN stop ; Not sure what to do here!
    mom = moment(temp_svalid_ratrig[24:25])
    temp_svalid_ratrig[*] = mom[0]
    ; pri ratio (128 ESA bins)
    temp_pri_ratio = pri_ratio[*, i_doy]
    FOR ii = 0, 126 DO BEGIN ; first fill in zeros from right
        jj = 126-ii
        IF temp_pri_ratio    [jj] EQ 0 AND temp_pri_ratio    [jj+1] NE 0 THEN temp_pri_ratio    [jj] = temp_pri_ratio    [jj+1]
    ENDFOR 
    FOR ii = 1, 127 DO BEGIN ; then fill from left
        IF temp_pri_ratio    [ii] EQ 0 AND temp_pri_ratio    [ii-1] NE 0 THEN temp_pri_ratio    [ii] = temp_pri_ratio    [ii-1]
    ENDFOR 
    bad_i = where(temp_svalid_ratrig EQ 0 OR finite(temp_svalid_ratrig) EQ 0, bad_svalid)
    bad_i = where(temp_pri_ratio     EQ 0 OR finite(temp_pri_ratio    ) EQ 0, bad_pri)
    IF bad_svalid GT 0 OR bad_pri GT 0 THEN stop ; shouldn't get here
        
                                ; starting in 2009, adjust low energy
; (vel_bins[27:31]) by the line created from ob_ra_den_vs_vel_monthly
; -- running Jan 2009 and doing an exponential fit of bins 26-30.
    ;a0_low = 132233
    ;a1_low = -.0356207
    ;vel_bins_low = [327., 309.725, 297., 281., 260.5] ; from ESA_table_B_2010_06_25b.txt
    ;y_fit_low = a0_low*exp(vel_bins_low*a1_low)

                                ; starting in 2009, adjust high energy
; (vel_bins[0:21]) by the line created from ob_ra_den_vs_vel_monthly
; -- running May 2012 and doing an exponential fit of bins 16:22.
    ;a0_high = .127571           ; from exp fit
    ;a1_high = 0.00580439        ; from exp fit
    ;a0_high = 1.93742e-15 ; from power law fit
    ;a1_high = 5.45204     ; from power law fit

; bins 0-15 from plastic_sweep_table_info.txt
; bins 16-21 from ESA_table_B_2010_06_25b.txt
    ;vel_bins_high = [100., 82.223, 67.606, 55.588, 45.706, 37.581, 30.9, $
    ;                 25.407, 20.89, 17.177, 14.123, 11.612, 9.548, 7.851, 6.455, 5.308, $
    ;                 896.361, 823.301, 754.636, 690.366, 630.492, 575.012]
    ;y_fit_high = a0_high*exp(vel_bins_high*a1_high) ; from exp fit
    ;y_fit_high = a0_high*(vel_bins_high^a1_high) ; from power law fit

    FOR ii = 0L, num_records-1 DO BEGIN 
        i_cal = cal_file_index[ii]
                                ; set up velocity box
        IF ii GT 8 AND ii LT num_records-9 THEN BEGIN
            vel_box = vel_box[1:18]
            vel_box = [vel_box, bulk_speed_inst_sav_d[ii+9]]
        ENDIF 
        ;IF ii GT 9 AND ii LT n_elements(data_indices) THEN BEGIN ; temporary for checking beacon algorithm
        ;    vel_box = vel_box[1:9]
        ;    vel_box = [vel_box, bulk_speed_inst_sav_d[data_indices[ii]]]
        ;ENDIF 
        temp_i = where(finite(vel_box) EQ 1, count)
        IF count GT 1 THEN BEGIN 
           temp = moment(vel_box, /nan, sdev = temp_sdev)
           vel_sdev_d[ii] = temp_sdev
           IF temp_sdev / bulk_speed_inst_sav_d[ii] GT 0.05 THEN poor_stats_d[ii] = 1
        ENDIF else poor_stats_d[ii] = 1

        temp_i = where((vel_box LT ra_trig_eff[i_cal, 127, 2]) OR (finite(vel_box) EQ 0), count)
        IF count GT 0 AND count LT 19 THEN vel_box[temp_i] = !values.f_nan
        IF count EQ 19 THEN count = 0 ; work around for if all values bad, just take bad average
        temp_velocity = total(vel_box, /nan)/(19-count)
        ;temp_velocity = total(vel_box, /nan)/(10-count) ; temporary code for checking beacon algorithm

    ; check for 0 values in ratio arrays (9/13/2017)
        valid_i = where(svalid_ratrig[*, i_doy] NE 0, count)
        IF count GT 0 THEN BEGIN
            IF valid_i[0] GT 0 THEN svalid_ratrig[0:valid_i[0]-1, i_doy] = svalid_ratrig[valid_i[0], i_doy]
            FOR jj = 0, count-2 DO BEGIN ; fill in between valid indices
                IF valid_i[jj] LT valid_i[jj+1]-1 THEN BEGIN ; have gap
                    svalid_ratrig[valid_i[jj]+1:valid_i[jj+1]-1, i_doy] = svalid_ratrig[valid_i[jj+1], i_doy] ; take one to the right
                ENDIF
            ENDFOR 
            IF valid_i[count-1] LT 31 THEN svalid_ratrig[valid_i[count-1]+1:31, i_doy] = svalid_ratrig[valid_i[count-1], i_doy]
        ENDIF 
        valid_i = where(pri_ratio[*, i_doy] NE 0, count)
        IF count GT 0 THEN BEGIN
            IF valid_i[0] GT 0 THEN pri_ratio[0:valid_i[0]-1, i_doy] = pri_ratio[valid_i[0], i_doy]
            FOR jj = 0, count-2 DO BEGIN ; fill in between valid indices
                IF valid_i[jj] LT valid_i[jj+1]-1 THEN BEGIN ; have gap
                    pri_ratio[valid_i[jj]+1:valid_i[jj+1]-1, i_doy] = pri_ratio[valid_i[jj+1], i_doy] ; take one to the right
                ENDIF
            ENDFOR 
            IF valid_i[count-1] LT 127 THEN pri_ratio[valid_i[count-1]+1:127, i_doy] = pri_ratio[valid_i[count-1], i_doy]
        ENDIF 

        ; search for closest efficiency
        ;IF i_doy GT 2 THEN i_doy = i_doy-2 ;remove -- for checking beacon algorithm
        found = 0
        jj = 0
        WHILE jj LT 127 AND found EQ 0 DO BEGIN ; this takes the lower value
            IF temp_velocity GT ra_trig_eff[i_cal, jj, 2] THEN BEGIN
                temp_bin               = jj / 4
                ratio_interp           = interpol(reform(temp_svalid_ratrig), reform(vel_groups[i_cal, *]), temp_velocity)
                efficiencies_d    [ii] = ra_trig_eff[i_cal, jj, 3] * ratio_interp
                                ;IF (temp_year[ii] GE 2009) AND (temp_bin GE 27) THEN $
                ;  efficiencies_d  [ii] = efficiencies_d[ii] * y_fit_low[temp_bin-27]
                ;IF (temp_year[ii] GE 2009) AND (temp_bin LE 21) THEN $
                ;  efficiencies_d  [ii] = efficiencies_d[ii] * y_fit_high[temp_bin]
                pri_efficiencies_d[ii] = double(temp_pri_ratio[jj])
                vel_tmp_d         [ii] = ra_trig_eff[i_cal, jj, 2] 
                ratios_d          [ii] = ratio_interp 
                eff_tmp_d         [ii] = ra_trig_eff[i_cal, jj, 3] 
                found = 1
            ENDIF ELSE jj = jj + 1
        ENDWHILE 
        IF found EQ 0 THEN BEGIN
            efficiencies_d    [ii] = double(ra_trig_eff[i_cal, 127, 3])
            ;IF temp_year[ii] GE 2009 THEN $
            ;  efficiencies_d  [ii] = efficiencies_d[ii] * y_fit_low[31-27]
            pri_efficiencies_d[ii] = double(temp_pri_ratio[127])
            vel_tmp_d         [ii] = ra_trig_eff[i_cal, 127, 2] 
            ratios_d          [ii] = temp_svalid_ratrig[127/4 ]
            eff_tmp_d         [ii] = ra_trig_eff[i_cal, 127, 3]
        ENDIF 
        IF mom_meta[3, ii] NE 1 THEN pri_efficiencies_d[ii] = 1 ; don't use for SW-all
    ENDFOR 

    ; Added 03/11/2016
    ; Artificial correction based on comparison to RA.
    efficiencies_d = efficiencies_d / 10 

    vel_tmp          = sat+'_obmom_look_up_vel'
    ratios           = sat+'_obmom_svalid_ratrig'
    eff_tmp          = sat+'_obmom_ra_eff'   
    pri_efficiencies = sat+'_obmom_pri0_ratio'
    vel_sdev         = sat+'_obmom_vel_sdev' 
    poor_stats       = sat+'_obmom_poor_stats'
    store_data, vel_tmp,          data = {x:epoch1, y:vel_tmp_d         }, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, ratios,           data = {x:epoch1, y:ratios_d          }, dlim = {ylog:1, labels:labels, panel_size:2.}
    ;store_data, ratios,           data = {x:epoch1, y:efficiencies_d    }, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, eff_tmp,          data = {x:epoch1, y:eff_tmp_d         }, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, pri_efficiencies, data = {x:epoch1, y:pri_efficiencies_d}, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, vel_sdev,         data = {x:epoch1, y:vel_sdev_d        }, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, poor_stats,       data = {x:epoch1, y:poor_stats_d      }, dlim = {ylog:1, labels:labels, panel_size:2.}

; the rest
    pi          = dblarr(6, num_records)
    pressure_6  = dblarr(6, num_records)
    energy_flux = dblarr(3, num_records)

    FOR ii = 0L, num_records-1 DO BEGIN 
        i_cal  = cal_file_index[ii]
        ; density
        density_main[ii]   =  density_main[ii]   * step_var    [i_cal] * $
          table_norm[i_cal, 0] / (geom        [i_cal, 0] * efficiencies_d[ii])
        density_s   [ii]   =  density_s   [ii]   * step_var    [i_cal] * $
          table_norm[i_cal, 0] / (geom        [i_cal, 1] * efficiencies_d[ii])
        ;density     [ii]   =  density_s   [ii]
        density     [ii]   =  density_main[ii]   + density_s   [ii]
        IF mom_meta[3, ii] EQ 1 THEN BEGIN ; if moments taken from SW-H
            density[ii] = density[ii]/pri_efficiencies_d[ii]
        ENDIF 
        ; pressure & temperature
        k = 1.38E-23            ; Boltzmann constant J/deg
        a_bp_e = step_var[i_cal] * table_norm[i_cal, 2] / efficiencies_d[ii]
        IF mom_meta[3, ii] EQ 1 THEN BEGIN ; if moments taken from SW-H
            a_bp_e = a_bp_e/pri_efficiencies_d[ii]
        ENDIF 
        mn = 1.67E-27 * density[ii] * (10 ^ 6D)
        FOR jj = 0, 5 DO BEGIN  ; xx,xy,xz,yy,yz,zz
            IF jj LT 3 THEN vel1 = velocity[0, ii] ELSE IF jj LT 5 THEN $
              vel1 = velocity[1, ii] ELSE vel1 = velocity[2, ii] 
            IF jj EQ 0 THEN vel2 = velocity[0, ii] ELSE IF (jj EQ 1 OR jj EQ 3) THEN $
              vel2 = velocity[1, ii] ELSE vel2 = velocity[2, ii]
            vel1 = vel1 * 1000
            vel2 = vel2 * 1000
            ;sum  = (temperature_s   [jj, ii] / geom[i_cal, 1]) 
            sum  = (temperature_s   [jj, ii] / geom[i_cal, 1]) + $
                   (temperature_main[jj, ii] / geom[i_cal, 0]) 
            pi        [jj, ii] = a_bp_e * sum
            pressure_6[jj, ii] = pi[jj, ii] - (mn * vel1 * vel2)
        ENDFOR
        ; energy flux density and heat flux
        a_bh_e = step_var[i_cal] * table_norm[i_cal, 3] / efficiencies_d[ii]
        IF mom_meta[3, ii] EQ 1 THEN BEGIN ; if moments taken from SW-H
            a_bh_e = a_bh_e/pri_efficiencies_d[ii]
        ENDIF 
        ;sum = (heat_flux_s[ii]/geom[i_cal, 1])
        sum = (heat_flux_main[ii]/geom[i_cal, 0]) + (heat_flux_s[ii]/geom[i_cal, 1])
        energy_flux[*, ii] = a_bh_e * sum
    ENDFOR

    ; add temporal offset ratio to density
    saved_density = density ; used for calculating temperature
    IF sat EQ 'A' THEN BEGIN 
        den_temporal_offset_ratio = 1.5
        density = density * den_temporal_offset_ratio
    ENDIF ELSE BEGIN ; B
        indices = where (temp_doy LT 288, count) ; until 10/15/2007
        IF count GT 0 THEN BEGIN
            den_temporal_offset_ratio = 1.5
            density[indices] = density[indices] * den_temporal_offset_ratio
        ENDIF 
        indices = where (temp_doy GE 288 AND temp_doy LT 411, count) ; 10/15/2007 - 2/14/2008
        IF count GT 0 THEN BEGIN
            den_temporal_offset_ratio = (0.0054*temp_doy[indices]) -  0.1663
            density[indices] = density[indices] * den_temporal_offset_ratio
        ENDIF 
        indices = where (temp_doy GE 411, count) ; 2/14/2008 - 
        IF count GT 0 THEN BEGIN
            den_temporal_offset_ratio = 2.0
            density[indices] = density[indices] * den_temporal_offset_ratio
        ENDIF 
    ENDELSE         

    ; density
    name = sat+'_obmom_density'  
    store_data, name, data = {x:epoch1, y:density}, dlim = {ylog:1, labels:labels, panel_size:2.}

    ; Inst coordinates
    ; pressure
    pressure_xx_inst_d = reform(pressure_6[0, *])
    pressure_xy_inst_d = reform(pressure_6[1, *])
    pressure_xz_inst_d = reform(pressure_6[2, *])
    pressure_yy_inst_d = reform(pressure_6[3, *])
    pressure_yz_inst_d = reform(pressure_6[4, *])
    pressure_zz_inst_d = reform(pressure_6[5, *])
    pressure_xx_inst = sat+'_obmom_pressure_xx_inst'
    pressure_xy_inst = sat+'_obmom_pressure_xy_inst'
    pressure_xz_inst = sat+'_obmom_pressure_xz_inst'
    pressure_yy_inst = sat+'_obmom_pressure_yy_inst'
    pressure_yz_inst = sat+'_obmom_pressure_yz_inst'
    pressure_zz_inst = sat+'_obmom_pressure_zz_inst'
    store_data, pressure_xx_inst, data = {x:epoch1, y:pressure_xx_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, pressure_xy_inst, data = {x:epoch1, y:pressure_xy_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, pressure_xz_inst, data = {x:epoch1, y:pressure_xz_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, pressure_yy_inst, data = {x:epoch1, y:pressure_yy_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, pressure_yz_inst, data = {x:epoch1, y:pressure_yz_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, pressure_zz_inst, data = {x:epoch1, y:pressure_zz_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 

    ; temperature
    temperature_6 = dblarr(6, num_records)
    FOR ii = 0, 5 DO BEGIN 
        temperature_6[ii, *] = pressure_6[ii, *]/(saved_density * (10 ^ 6D) * k) ; 10^6 to convert to 1/m3 for density
    ENDFOR 
    temperature_xx_inst_d = reform(temperature_6[0, *])
    temperature_xy_inst_d = reform(temperature_6[1, *])
    temperature_xz_inst_d = reform(temperature_6[2, *])
    temperature_yy_inst_d = reform(temperature_6[3, *])
    temperature_yz_inst_d = reform(temperature_6[4, *])
    temperature_zz_inst_d = reform(temperature_6[5, *])

    ; add temporal offset ratio to temperature_xx
    IF sat EQ 'A' THEN BEGIN 
        temp_xx_temporal_offset_ratio = 0.0001 * temp_doy + 0.2867
        temperature_xx_inst_d = temperature_xx_inst_d * temp_xx_temporal_offset_ratio
    ENDIF ELSE BEGIN ; B
        temp_xx_temporal_offset_ratio = 0.25
        temperature_xx_inst_d = temperature_xx_inst_d * temp_xx_temporal_offset_ratio
    ENDELSE         

    temperature_xx_inst = sat+'_obmom_temperature_xx_inst'
    temperature_xy_inst = sat+'_obmom_temperature_xy_inst'
    temperature_xz_inst = sat+'_obmom_temperature_xz_inst'
    temperature_yy_inst = sat+'_obmom_temperature_yy_inst'
    temperature_yz_inst = sat+'_obmom_temperature_yz_inst'
    temperature_zz_inst = sat+'_obmom_temperature_zz_inst'
    store_data, temperature_xx_inst, data = {x:epoch1, y:temperature_xx_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, temperature_xy_inst, data = {x:epoch1, y:temperature_xy_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, temperature_xz_inst, data = {x:epoch1, y:temperature_xz_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, temperature_yy_inst, data = {x:epoch1, y:temperature_yy_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, temperature_yz_inst, data = {x:epoch1, y:temperature_yz_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 
    store_data, temperature_zz_inst, data = {x:epoch1, y:temperature_zz_inst_d}, dlim = {ylog:0, labels:labels, panel_size:2.} 

    ; energy flux
    energy_flux_x_inst_d = reform(energy_flux[0, *])
    energy_flux_y_inst_d = reform(energy_flux[1, *])
    energy_flux_z_inst_d = reform(energy_flux[2, *])
    energy_flux_x_inst = sat+'_obmom_energy_flux_x_inst'
    energy_flux_y_inst = sat+'_obmom_energy_flux_y_inst'
    energy_flux_z_inst = sat+'_obmom_energy_flux_z_inst'
    store_data, energy_flux_x_inst, data = {x:epoch1, y:energy_flux_x_inst_d}, dlim = {ylog:1, labels:labels, panel_size:2.} 
    store_data, energy_flux_y_inst, data = {x:epoch1, y:energy_flux_y_inst_d}, dlim = {ylog:1, labels:labels, panel_size:2.} 
    store_data, energy_flux_z_inst, data = {x:epoch1, y:energy_flux_z_inst_d}, dlim = {ylog:1, labels:labels, panel_size:2.} 

    ; update ratios for valid velocities
    ; remove low density times
    speed_check_den_d = bulk_speed_inst_sav_d
    FOR ii = 0L, num_records-1 DO BEGIN 
        IF density[ii] LT .5 THEN speed_check_den_d[ii] = !values.f_nan
    ENDFOR 
    max_vel = max(speed_check_den_d, /nan) 
    min_vel = min(speed_check_den_d, /nan) 

    IF finite(min_vel) EQ 0 OR finite(max_vel) EQ 0 THEN invalid_min_max = 1 ELSE invalid_min_max = 0 
    IF array_equal(cal_file_index[0], cal_file_index) NE 1 THEN stop $ ; need to make more sophisticated. Cal file switches mid-day
      ELSE temp_ra_trig_eff = reform(ra_trig_eff[0, *, *])
    ; svalid/ratrig ratio
    ii = 0
    found = 0
    ;invalid_min_max = 0
    WHILE ii LT 127 AND found EQ 0 DO BEGIN  ; only take velocity indices inside min and max velocities
        IF max_vel LT temp_ra_trig_eff[ii, 2, 0] THEN ii = ii + 1 ELSE BEGIN
            max_vel_i = ii
            found = 1
        ENDELSE 
    ENDWHILE
    IF found NE 1 THEN invalid_min_max = 1
    ii = 0
    found = 0
    WHILE ii LT 31 AND found EQ 0 DO BEGIN  ; only take velocity indices inside min and max velocities
        IF max_vel LT vel_groups[0, ii] THEN ii = ii + 1 ELSE BEGIN
            max_vel_i = ii
            found = 1
        ENDELSE 
    ENDWHILE
    IF found NE 1 THEN invalid_min_max = 1
    ii = 0
    found = 0
    min_vel_i = -2
    WHILE ii LT 32 AND found EQ 0 DO BEGIN  ; only take velocity indices inside min and max velocities
        IF min_vel LT vel_groups[0, ii] THEN ii = ii + 1 ELSE BEGIN
            min_vel_i = ii-1
            found = 1
        ENDELSE 
    ENDWHILE
    IF found NE 1 OR min_vel_i LT 0 THEN min_vel_i = 31
    ; check if both vel between one box 06/18/2015
    IF (min_vel_i LT max_vel_i) AND (invalid_min_max EQ 0) THEN BEGIN
        IF min_vel_i EQ max_vel_i-1 THEN BEGIN ; reverse them
            temp_vel_i = max_vel_i
            max_vel_i  = min_vel_i
            min_vel_i  = temp_vel_i
        ENDIF ELSE stop ; don't expect this
    END 

    get_timespan, timespan
    time_st = time_struct(timespan[0])
    year = time_st.year
    doy = time_st.doy
    read_ratios, sat, year, svalid_ratio, pri0_ratio
    calculate_svalid_ratrig_ratio_b, sat, temp_ratio, closed_entrance = closed_entrance

    IF invalid_min_max EQ 0 AND n_elements(temp_ratio) GT 1 THEN BEGIN 
        IF max_vel_i GT min_vel_i THEN min_vel_i = max_vel_i
        good_i = where((finite(reform(temp_ratio[max_vel_i:min_vel_i])) EQ 1) AND $
                       (      (reform(temp_ratio[max_vel_i:min_vel_i])) GT 0), good_count)
        IF good_count GT 0 THEN BEGIN 
            indices = good_i+max_vel_i
            svalid_ratio[doy-1, indices] = temp_ratio[indices]
        ENDIF 
        IF doy GT 1 THEN BEGIN
            IF max_vel_i GT  0 THEN svalid_ratio[doy-1, 0:max_vel_i-1 ] = svalid_ratio[doy-2, 0:max_vel_i-1 ]
            IF min_vel_i LT 31 THEN svalid_ratio[doy-1, min_vel_i+1:31] = svalid_ratio[doy-2, min_vel_i+1:31]
        ENDIF ELSE BEGIN ; need to go back a year if possible
            IF year GT 2006 THEN BEGIN ; have previous year
                read_ratios, sat, year-1, prev_svalid_ratio, prev_pri0_ratio
                IF year-1 MOD 4 EQ 0 THEN old_doy_i = 365 ELSE old_doy_i = 364 ; check for leap year
                IF max_vel_i GT  0 THEN svalid_ratio[doy-1, 0:max_vel_i-1 ] = prev_svalid_ratio[old_doy_i, 0:max_vel_i-1 ]
                IF min_vel_i LT 31 THEN svalid_ratio[doy-1, min_vel_i+1:31] = prev_svalid_ratio[old_doy_i, min_vel_i+1:31]
            ENDIF ; otherwise leave it as is
        ENDELSE
    ENDIF ELSE BEGIN ; copy previous day 
        IF doy GT 1 THEN BEGIN
            svalid_ratio[doy-1, *] = svalid_ratio[doy-2, *]
        ENDIF ELSE IF year GT 2006 THEN BEGIN ; have previous year
            read_ratios, sat, year-1, prev_svalid_ratio, prev_pri0_ratio
            IF year-1 MOD 4 EQ 0 THEN old_doy_i = 365 ELSE old_doy_i = 364 ; check for leap year
            svalid_ratio[doy-1, *] = prev_svalid_ratio[old_doy_i, *]
        ENDIF                   ; otherwise leave it as is
    ENDELSE 

    ; pri0 ratio
    ii = 0
    found = 0
    invalid_min_max = 0
    WHILE ii LT 127 AND found EQ 0 DO BEGIN  ; only take velocity indices inside min and max velocities
        IF max_vel LT temp_ra_trig_eff[ii, 2] THEN ii = ii + 1 ELSE BEGIN
            max_vel_i = ii
            found = 1
        ENDELSE 
    ENDWHILE
    IF found NE 1 THEN invalid_min_max = 1
    ii = 0
    found = 0
    min_vel_i = -2
    WHILE ii LT 128 AND found EQ 0 DO BEGIN  ; only take velocity indices inside min and max velocities
        IF min_vel LT temp_ra_trig_eff[ii, 2] THEN ii = ii + 1 ELSE BEGIN
            min_vel_i = ii-1
            found = 1
        ENDELSE 
    ENDWHILE
    IF found NE 1 OR min_vel_i LT 0 THEN min_vel_i = 127 
    calculate_pri0_pri0_3_ratio, sat, temp_ratio, closed_entrance = closed_entrance
    IF n_elements(temp_ratio) EQ 1 AND temp_ratio[0] EQ 0 THEN invalid_min_max = 1
    IF invalid_min_max EQ 0 THEN BEGIN 
        IF max_vel_i GT min_vel_i THEN min_vel_i = max_vel_i
        IF n_elements(temp_ratio) EQ 1 THEN BEGIN 
            IF max_vel_i NE min_vel_i THEN stop ; problem -- should be one value, I think
            IF finite(temp_ratio) EQ 1 AND temp_ratio GT 0 THEN pri0_ratio[doy-1, min_vel_i] = temp_ratio
        ENDIF ELSE BEGIN 
            good_i = where((finite(reform(temp_ratio[max_vel_i:min_vel_i])) EQ 1) AND $
                           (      (reform(temp_ratio[max_vel_i:min_vel_i])) GT 0), good_count)
            IF good_count GT 0 THEN BEGIN 
                indices = good_i+max_vel_i
                pri0_ratio[doy-1, indices] = temp_ratio[indices]
            ENDIF 
            IF doy GT 1 THEN BEGIN
                IF max_vel_i GT   0 THEN pri0_ratio[doy-1, 0:max_vel_i-1  ] = pri0_ratio[doy-2, 0:max_vel_i-1  ]
                IF min_vel_i LT 127 THEN pri0_ratio[doy-1, min_vel_i+1:127] = pri0_ratio[doy-2, min_vel_i+1:127]
            ENDIF ELSE BEGIN    ; need to go back a year if possible
                IF year GT 2006 THEN BEGIN ; have previous year
                    IF year-1 MOD 4 EQ 0 THEN old_doy_i = 365 ELSE old_doy_i = 364 ; check for leap year
                    IF max_vel_i GT   0 THEN pri0_ratio[doy-1, 0:max_vel_i-1  ] = prev_pri0_ratio[old_doy_i, 0:max_vel_i-1  ]
                    IF min_vel_i LT 127 THEN pri0_ratio[doy-1, min_vel_i+1:127] = prev_pri0_ratio[old_doy_i, min_vel_i+1:127]
                ENDIF           ; otherwise leave it as is
            ENDELSE
        ENDELSE 
    ENDIF ELSE BEGIN ; copy previous day
        IF doy GT 1 THEN BEGIN
            pri0_ratio[doy-1, *] = pri0_ratio[doy-2, *]
        ENDIF ELSE IF year GT 2006 THEN BEGIN ; have previous year
            IF year-1 MOD 4 EQ 0 THEN old_doy_i = 365 ELSE old_doy_i = 364 ; check for leap year
            pri0_ratio[doy-1, *] = prev_pri0_ratio[old_doy_i, *]
        ENDIF                   ; otherwise leave it as is
    ENDELSE 
    IF keyword_set(read_only) EQ 0 THEN update_ratios, year, sat, svalid_ratio, pri0_ratio
ENDELSE 
END 
