;+
;PROCEDURE: JBField
;
;PURPOSE: Convert SPLAT Mag files into GSE coordinates. Then produce ascii files of the following:
;time, Bphi in gse, Btheta in gse, magnitude of B in gse, the magnitude of B in RTN (as a check),
;Spacecraft (s/c) phi in gse, S/Ctheta gse, Linear Miss of the B field and Earth 
;
;INPUT: sc ->Spacecraft('A' or 'B'), Time String, Bx, By, Bz 
;
;NOTE: Assumes files read in have Bfield data in RTN
;
; 09/30/2008 LBE: Added want_ascii
; 10/10/2008 LBE: Added rtn variables
; 12/05/2008 LBE: Fixed bug with _rtn name
;-
PRO jbfield, sc, t_st, x, y, z, want_ascii
compile_opt strictarrsubs

tab = string(9B)

LN = N_ELEMENTS(x)		;Reads the number of elements Bx 

strput, t_st,'T',10		;Formats the time so that it can be read into solarsoft

a_rtn = [x,y,z]		;Forms an array composing of the RTN Bfield components

a_rtn=dblarr(3,LN)
a_rtn[0,*]=x 
a_rtn[1,*]=y
a_rtn[2,*]=z

magbrtn=SQRT(a_rtn[0,*]^2+a_rtn[1,*]^2+a_rtn[2,*]^2)		;Forms an array of the magnitude of a in RTN

a_gse = a_rtn
convert_stereo_coord,t_st,a_gse,'RTN', 'GSE', spacecraft=sc		;Converts the tip of the assumed RTN bfield into GSE coordinates

s = get_stereo_coord(t_st, sc, system = 'GSE', novelocity=1)		;s becomes a variable containing the s/c position during the time interval

b = a_gse-s		; forms the b field in GSE centered at the Earth

magbgse=SQRT(b[0,*]^2+b[1,*]^2+b[2,*]^2)		;Forms an array of the magnitude of b in GSE
mag_sc_gse=SQRT(s[0,*]^2+s[1,*]^2+s[2,*]^2)		;Forms an array of the magnitude of s in GSE

PhiB=dblarr(1,LN)		;Creates the Bfield phi angle 
FOR i=0, LN-1 DO BEGIN
PhiB[0,i]=180/!PI*ATAN(b[1,i], b[0,i])
ENDFOR

PhiB_rtn=dblarr(1,LN)		;Creates the Bfield phi angle 
FOR i=0, LN-1 DO BEGIN
PhiB_rtn[0,i]=180/!PI*ATAN(a_rtn[1,i], a_rtn[0,i])
ENDFOR

Phisc=dblarr(1,LN)		;Creates the s/c phi angle
FOR j=0, LN-1 DO BEGIN
Phisc[0,j]=180/!PI*ATAN(s[1,j], s[0,j])
ENDFOR

R=SQRT(b[0,*]^2+b[1,*]^2)
R_rtn=SQRT(a_rtn[0,*]^2+a_rtn[1,*]^2)

ThetaB=dblarr(1,LN)		;Creates the Bfield theta angle
FOR k=0, LN-1 DO BEGIN
ThetaB[0,k]=180/!PI*ATAN(R[0,k], b[2,k])
ENDFOR

ThetaB_rtn=dblarr(1,LN)		;Creates the Bfield theta angle
FOR k=0, LN-1 DO BEGIN
ThetaB_rtn[0,k]=180/!PI*ATAN(R_rtn[0,k], a_rtn[2,k])
ENDFOR

T=SQRT(s[0,*]^2+s[1,*]^2)

Thetasc=dblarr(1,LN)		;Creates the s/c theta angle
FOR m=0, LN-1 DO BEGIN
Thetasc[0,m]=180/!PI*ATAN(T[0,m], s[2,m])
ENDFOR

U=SQRT(s[0,*]^2+s[1,*]^2+s[2,*]^2)
LinearMiss = DBLARR(1,LN)

FOR n=0, LN-1 DO BEGIN		;Creates an array of the linear miss of the Bfield and Earth in Re
LinearMiss[0,n] = SIN(ACOS((-s[0,n]*b[0,n]-s[1,n]*b[1,n]-s[2,n]*b[2,n])/(magbgse[0,n]*U[0,n])))*U[0,n]/6378.1
ENDFOR

vecangdif=DBLARR(1,LN)		;Defines an array which will contain the information on the angular difference in the s/c position and Bfield vectors

FOR p=0, LN-1 DO BEGIN

	vecangdif[0,p]=ACOS((s[0,p]*b[0,p]+s[1,p]*b[1,p]+s[2,p]*b[2,p])/(magbgse[0,p]*U[0,p]))*180/!PI
	IF (vecangdif[0,p] GT 90) THEN BEGIN		;Ensures that the angular miss is less than 90 since the B field can be parallel or anti parallel and S/C is connected
	vecangdif[0,p] = 180 - vecangdif[0,p]
	ENDIF
ENDFOR

header1 = 'Joshs Bfield data for suprathermal analysis. The coordinate system used is GSE'
header2 = 'time(yyyy-mm-dd/hh:mimi:ss)'+tab+'Bphi(deg)'+tab+'Btheta(deg)'+tab+'|B|gse(nT)'+tab+'|B|rtn(nT)'+tab+'S/Cphi(deg)'+tab+'S/Ctheta(deg)'+tab+'Linear_Miss(Re)'+tab+'Angular_Miss(deg)'

strput, t_st, '/', 10		;Fixes changes made to time format

time = STRARR(1,LN)		;Creates a string array for the read in time

time[0,*]= t_st

t_dbl = time_double(t_st)
phi_bfield = sc+'_phi_bfield_gse'
store_data, phi_bfield, data = {x: t_dbl, y: reform(PhiB)}
phi_bfield_rtn = sc+'_phi_bfield_rtn'
store_data, phi_bfield_rtn, data = {x: t_dbl, y: reform(PhiB_rtn)}
theta_bfield = sc+'_theta_bfield_gse'
store_data, theta_bfield, data = {x: t_dbl, y: reform(ThetaB)}
theta_bfield_rtn = sc+'_theta_bfield_rtn'
store_data, theta_bfield_rtn, data = {x: t_dbl, y: reform(ThetaB_rtn)}
mag_bfield = sc+'_mag_bfield'
store_data, mag_bfield, data = {x: t_dbl, y: reform(magbgse)}
phi_spcrft = sc+'_phi_spcrft'
store_data, phi_spcrft, data = {x: t_dbl, y: reform(Phisc)}
theta_spcrft = sc+'_theta_spcrft'
store_data, theta_spcrft, data = {x: t_dbl, y: reform(Thetasc)}
mag_spcrft = sc+'_mag_spcrft'
store_data, mag_spcrft, data = {x: t_dbl, y: reform(mag_sc_gse)}
angular_miss = sc+'_angular_miss'
store_data, angular_miss, data = {x: t_dbl, y: reform(vecangdif)}
linear_miss = sc+'_linear_miss'
store_data, linear_miss, data = {x: t_dbl, y: reform(LinearMiss)}

IF want_ascii EQ 1 THEN BEGIN 
    OpenW, lun, 'data4josh.asc', /Get_LUN
    PrintF, lun, header1
    printF, lun, header2
    FOR ii = 0, LN-1 DO BEGIN   ; write data
        temp_string = string(time[0, ii])
        temp_string = temp_string+tab+string(PhiB[0, ii])
        temp_string = temp_string+tab+string(ThetaB[0, ii])
        temp_string = temp_string+tab+string(magbgse[0, ii])
        temp_string = temp_string+tab+string(magbrtn[0, ii])
        temp_string = temp_string+tab+string(Phisc[0, ii])
        temp_string = temp_string+tab+string(Thetasc[0, ii])
        temp_string = temp_string+tab+string(LinearMiss[0, ii])
        temp_string = temp_string+tab+string(vecangdif[0, ii])
        printf, lun, temp_string
    ENDFOR 
    close, lun
    free_lun, lun
ENDIF 

print, 'end of program'
end
