;+
; PROCEDURE: get_transformed_stereo_mag
;
; PURPOSE: get the stereo magnetic field data
;
; INPUT: sc -> Spacecraft number
;
; CREATED BY: C. Mouikis
;
; MODIFICATION HISTORY:
;     06/16/2011 LBE: Created from old get_stereo_mag.pro
;     07/27/2011 LBE: Fixed bug when reading more than one month.
;-

PRO read_stereo_mag_onefile, sc, year, path, filename, times1, bx_rtn1, by_rtn1, bz_rtn1, bt_rtn1, $
bx_gse1, by_gse1, bz_gse1, bt_gse1, bx_gsm1, by_gsm1, bz_gsm1, bt_gsm1
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

get_err_no = 0 & get_err_msg = ''


ext = '.txt'
find = FINDFILE(path+'/'+filename+ext, count = ff)
IF ff EQ 0 THEN BEGIN
    PRINT, 'FILE: ' + path+filename+ext + ' was not found'
    get_err_no = 1
    get_err_msg = 'Data not found for time interval'

ENDIF ELSE BEGIN 

    fln_r = find(ff-1)          ; If different versions found get the latest

    line = ''
    OPENR, in_lun, fln_r, /GET_LUN

    READF, in_lun, line ; 2 lines header
    READF, in_lun, line

    index  = 0L
    times1  = dblarr(44640) ; 31 days x 1440 min
    bx_rtn1 = fltarr(44640)
    by_rtn1 = fltarr(44640)
    bz_rtn1 = fltarr(44640)
    bt_rtn1 = fltarr(44640)
    bx_gse1 = fltarr(44640)
    by_gse1 = fltarr(44640)
    bz_gse1 = fltarr(44640)
    bt_gse1 = fltarr(44640)
    bx_gsm1 = fltarr(44640)
    by_gsm1 = fltarr(44640)
    bz_gsm1 = fltarr(44640)
    bt_gsm1 = fltarr(44640)

    WHILE NOT EOF(in_lun) DO BEGIN
        READF, in_lun, line
        parts = strsplit(line, /extract)
        times1 [index] = time_double(parts[0])
        bx_rtn1[index] = float(parts[ 1])
        by_rtn1[index] = float(parts[ 2])
        bz_rtn1[index] = float(parts[ 3])
        bt_rtn1[index] = float(parts[ 4])
        bx_gse1[index] = float(parts[ 5])
        by_gse1[index] = float(parts[ 6])
        bz_gse1[index] = float(parts[ 7])
        bt_gse1[index] = float(parts[ 8])
        bx_gsm1[index] = float(parts[ 9])
        by_gsm1[index] = float(parts[10])
        bz_gsm1[index] = float(parts[11])
        bt_gsm1[index] = float(parts[12])
        index = index + 1
    ENDWHILE
    times1  = times1 [0:index-1]
    bx_rtn1 = bx_rtn1[0:index-1]
    by_rtn1 = by_rtn1[0:index-1]
    bz_rtn1 = bz_rtn1[0:index-1]
    bt_rtn1 = bt_rtn1[0:index-1]
    bx_gse1 = bx_gse1[0:index-1]
    by_gse1 = by_gse1[0:index-1]
    bz_gse1 = bz_gse1[0:index-1]
    bt_gse1 = bt_gse1[0:index-1]
    bx_gsm1 = bx_gsm1[0:index-1]
    by_gsm1 = by_gsm1[0:index-1]
    bz_gsm1 = bz_gsm1[0:index-1]
    bt_gsm1 = bt_gsm1[0:index-1]
        
    CLOSE, in_lun
    FREE_LUN, in_lun,  /FORCE
ENDELSE 
 
END


PRO get_transformed_stereo_mag, sc
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

get_err_no = 0
get_err_msg = 'Data found in time interval'

IF sc EQ 'A' THEN path = GETENV('STA_MAG_DATA') ELSE $
  path = GETENV('STB_MAG_DATA')

;path = './.' ; temporary

get_timespan, time_interval
t_s = gettime(time_interval(0)) ; start time in tplot-time
t_e = gettime(time_interval(1)) ; end time in tplot-time

t_s_str = time_struct(t_s)      ; start_time tplot time structure
t_e_str = time_struct(t_e)      ; end_time tplot time structure

mjd_s = julday(t_s_str.month, t_s_str.date, t_s_str.year) ;start julian day
mjd_e = julday(t_e_str.month, t_e_str.date, t_e_str.year) ; end julian day

no_of_files = ((t_e_str.year-1) * 12 + t_e_str.month) - $
              ((t_s_str.year-1) * 12 + t_s_str.month) + 1

;Last month is not included if hour=min=sec=0 and date=1
IF t_e_str.date EQ 1 AND t_e_str.hour EQ 0 AND t_e_str.min EQ 0 AND t_e_str.sec EQ 0 THEN $
  no_of_files = no_of_files - 1

;--------------------------------------------------------------------
; Read all 1 day files that correspond to requested time interval
;--------------------------------------------------------------------
fc = 0                          ; Files-found counter
FOR nd = 0, no_of_files-1 DO BEGIN ; Loop trough all days
    
    caldat, mjd_s, month, day, year ; find caledar date
    month = month+nd
    IF month EQ 13 THEN BEGIN
        year = year+1
        month = 1
    ENDIF 
    satstr = sc
    
    temp_path = path + '/' + STRING(year, FORMAT = '(i4.4)')+'/'

    filename = 'ST' + sc + '_L1_MAG_Transformed_' + $
               STRING(year, FORMAT = '(i4.4)') + $
               STRING(month, FORMAT = '(i2.2)') + '_V02'

    read_stereo_mag_onefile, sc, year, temp_path, filename, times1, bx_rtn1, by_rtn1, bz_rtn1, bt_rtn1, $
      bx_gse1, by_gse1, bz_gse1, bt_gse1, bx_gsm1, by_gsm1, bz_gsm1, bt_gsm1

    IF get_err_no EQ 0 THEN BEGIN ; if file is found
        PRINT, 'Magnetic field file loaded: ', path+filename
        fc = fc + 1
      
        IF fc EQ 1 THEN BEGIN
            times  = times1
            bx_rtn = bx_rtn1
            by_rtn = by_rtn1
            bz_rtn = bz_rtn1
            bt_rtn = bt_rtn1
            bx_gse = bx_gse1
            by_gse = by_gse1
            bz_gse = bz_gse1
            bt_gse = bt_gse1
            bx_gsm = bx_gsm1
            by_gsm = by_gsm1
            bz_gsm = bz_gsm1
            bt_gsm = bt_gsm1
        ENDIF ELSE BEGIN        ; if more than one day needs to be loaded
            times  = [ times,  times1] ; concatenate time arrays
            bx_rtn = [bx_rtn, bx_rtn1]
            by_rtn = [by_rtn, by_rtn1]
            bz_rtn = [bz_rtn, bz_rtn1]
            bt_rtn = [bt_rtn, bt_rtn1]
            bx_gse = [bx_gse, bx_gse1]
            by_gse = [by_gse, by_gse1]
            bz_gse = [bz_gse, bz_gse1]
            bt_gse = [bt_gse, bt_gse1]
            bx_gsm = [bx_gsm, bx_gsm1]
            by_gsm = [by_gsm, by_gsm1]
            bz_gsm = [bz_gsm, bz_gsm1]
            bt_gsm = [bt_gsm, bt_gsm1]
        ENDELSE
    ENDIF
ENDFOR

; check for invalid data flag (1E32)
bad_bx_rtn = where(bx_rtn LT -1E30, bad_bx_rtn_count)
bad_by_rtn = where(by_rtn LT -1E30, bad_by_rtn_count)
bad_bz_rtn = where(bz_rtn LT -1E30, bad_bz_rtn_count)
bad_bt_rtn = where(bt_rtn LT -1E30, bad_bt_rtn_count)
bad_bx_gse = where(bx_gse LT -1E30, bad_bx_gse_count)
bad_by_gse = where(by_gse LT -1E30, bad_by_gse_count)
bad_bz_gse = where(bz_gse LT -1E30, bad_bz_gse_count)
bad_bt_gse = where(bt_gse LT -1E30, bad_bt_gse_count)
bad_bx_gsm = where(bx_gsm LT -1E30, bad_bx_gsm_count)
bad_by_gsm = where(by_gsm LT -1E30, bad_by_gsm_count)
bad_bz_gsm = where(bz_gsm LT -1E30, bad_bz_gsm_count)
bad_bt_gsm = where(bt_gsm LT -1E30, bad_bt_gsm_count)

IF bad_bx_rtn_count GT 0 THEN  bx_rtn[bad_bx_rtn] = !values.f_nan
IF bad_by_rtn_count GT 0 THEN  by_rtn[bad_by_rtn] = !values.f_nan
IF bad_bz_rtn_count GT 0 THEN  bz_rtn[bad_bz_rtn] = !values.f_nan
IF bad_bt_rtn_count GT 0 THEN  bt_rtn[bad_bt_rtn] = !values.f_nan
IF bad_bx_gse_count GT 0 THEN  bx_gse[bad_bx_gse] = !values.f_nan
IF bad_by_gse_count GT 0 THEN  by_gse[bad_by_gse] = !values.f_nan
IF bad_bz_gse_count GT 0 THEN  bz_gse[bad_bz_gse] = !values.f_nan
IF bad_bt_gse_count GT 0 THEN  bt_gse[bad_bt_gse] = !values.f_nan
IF bad_bx_gsm_count GT 0 THEN  bx_gsm[bad_bx_gsm] = !values.f_nan
IF bad_by_gsm_count GT 0 THEN  by_gsm[bad_by_gsm] = !values.f_nan
IF bad_bz_gsm_count GT 0 THEN  bz_gsm[bad_bz_gsm] = !values.f_nan
IF bad_bt_gsm_count GT 0 THEN  bt_gsm[bad_bt_gsm] = !values.f_nan

limind = WHERE(times GE t_s AND times LE t_e, climind)

IF climind GT 0 THEN BEGIN
    store_data, 'ST' + sc + '_MAG_RTN_X', data = {x:times(limind), y:bx_rtn[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LX!N RTN (nT)'}
    store_data, 'ST' + sc + '_MAG_RTN_Y', data = {x:times(limind), y:by_rtn[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LY!N RTN (nT)'}
    store_data, 'ST' + sc + '_MAG_RTN_Z', data = {x:times(limind), y:bz_rtn[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LZ!N RTN (nT)'}
    store_data, 'ST' + sc + '_MAG_RTN_T', data = {x:times(limind), y:bt_rtn[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!Ltot!N RTN (nT)'}
    store_data, 'ST' + sc + '_MAG_GSE_X', data = {x:times(limind), y:bx_gse[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LX!N GSE (nT)'}
    store_data, 'ST' + sc + '_MAG_GSE_Y', data = {x:times(limind), y:by_gse[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LY!N GSE (nT)'}
    store_data, 'ST' + sc + '_MAG_GSE_Z', data = {x:times(limind), y:bz_gse[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LZ!N GSE (nT)'}
    store_data, 'ST' + sc + '_MAG_GSE_T', data = {x:times(limind), y:bt_gse[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!Ltot!N GSE (nT)'}
    store_data, 'ST' + sc + '_MAG_GSM_X', data = {x:times(limind), y:bx_gsm[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LX!N GSM (nT)'}
    store_data, 'ST' + sc + '_MAG_GSM_Y', data = {x:times(limind), y:by_gsm[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LY!N GSM (nT)'}
    store_data, 'ST' + sc + '_MAG_GSM_Z', data = {x:times(limind), y:bz_gsm[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LZ!N GSM (nT)'}
    store_data, 'ST' + sc + '_MAG_GSM_T', data = {x:times(limind), y:bt_gsm[limind]}, $
                dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!Ltot!N GSM (nT)'}

ENDIF

END
