;+
; PROCEDURE: get_stereo_mag
;
; PURPOSE: get the stereo magnetic field data
;
; INPUT: sc -> Spacecraft number
;
; CREATED BY: C. Mouikis
;
; MODIFICATION HISTORY:
;     10/02/2008 LBE: Added check for invalid flag.
;     10/07/2008 LBE: Added check for month 13.
;     10/24/2008 LBE: Added check for last day of month.
;     07/15/2009 LBE: Changed ytitle to RTN.
;     12/01/2009 LBE: Updated for version 2.
;     07/13/2010 LBE: Added gse and gsm keywords.
;     07/15/2010 LBE: Fix bug with coordinate transformations.
;     06/03/2011 LBE: Change for CDFs instead of ascii.
;-

PRO read_stereo_mag_onefile, sc, year, path, file_base, B_time, B_data
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

get_err_no = 0 & get_err_msg = ''

find = FINDFILE(path+'/'+file_base+'*', count = ff)
IF ff EQ 0 THEN BEGIN
    PRINT, 'FILE: ' + path+file_base + ' was not found'
    get_err_no = 1
    get_err_msg = 'Data not found for time interval'
ENDIF ELSE IF ff GT 1 THEN BEGIN 
    print, 'More than one file with base '+path+file_base+' found'
    get_err_no = 1
    get_err_msg = 'Duplicate data found for time interval'
ENDIF ELSE BEGIN 
    fln_r = find(ff-1)          ; If different versions found get the latest
    cdf_id = cdf_open(fln_r)
    cdf_control, cdf_id, get_var_info = info, var = 'Epoch', /zvariable
    IF info.maxrec GE 0 THEN BEGIN 
        cdf_varget, cdf_id, 'Epoch', epoch, rec_count = info.maxrec+1, /zvariable
        B_time = reform(time_double(epoch, /epoch))
        cdf_varget, cdf_id, 'BFIELD', B_data, rec_count = info.maxrec+1, /zvariable
        B_data = transpose(B_data)
    ENDIF 
    cdf_close, cdf_id
ENDELSE 
 
END


PRO get_stereo_mag, sc, GSE = GSE, GSM = GSM

COMMON get_error, get_err_no, get_err_msg, default_verbose

get_err_no = 0
get_err_msg = 'Data found in time interval'

IF sc EQ 'A' THEN path = GETENV('STA_MAG_DATA') ELSE $
  path = GETENV('STB_MAG_DATA')
pat = path+sc+'/'

;path = './.' ; temporary

get_timespan, time_interval
t_s = gettime(time_interval(0)) ; start time in tplot-time
t_e = gettime(time_interval(1)) ; end time in tplot-time

t_s_str = time_struct(t_s)      ; start_time tplot time structure
t_e_str = time_struct(t_e)      ; end_time tplot time structure

mjd_s = julday(t_s_str.month, t_s_str.date, t_s_str.year) ;start julian day
mjd_e = julday(t_e_str.month, t_e_str.date, t_e_str.year) ; end julian day

no_of_files = (mjd_e - mjd_s) + 1 ; number of days to be loaded

;Last day is not included if hour=min=sec=0
IF t_e_str.hour EQ 0 AND t_e_str.min EQ 0 AND t_e_str.sec EQ 0 THEN $
  no_of_files = no_of_files - 1

;--------------------------------------------------------------------
; Read all 1 day files that correspond to requested time interval
;--------------------------------------------------------------------
fc = 0                          ; Files-found counter
temp_time = t_s
FOR nd = 0, no_of_files-1 DO BEGIN ; Loop trough all days
    
    temp_struct = time_struct(temp_time)
    year        = temp_struct.year
    month       = temp_struct.month
    day         = temp_struct.date
    satstr      = sc
    
    temp_path = path + '/' + STRING(year, FORMAT = '(i4.4)') + '/'

    file_base = 'ST' + sc + '_L1_MAG_RTN_' + $
               STRING(year,  FORMAT = '(i4.4)') + $
               STRING(month, FORMAT = '(i2.2)') + $
               STRING(day,   FORMAT = '(I2.2)') + '_V'

    read_stereo_mag_onefile, sc, year, temp_path, file_base, B_time, B_data

    IF get_err_no EQ 0 THEN BEGIN ; if file is found
        PRINT, 'Magnetic field file loaded: ', path+file_base
        fc = fc + 1
      
        IF fc EQ 1 THEN BEGIN
            Btime = B_time
            Bdata = B_data
        ENDIF ELSE BEGIN        ; if more than one day needs to be loaded
            Btime = [Btime, B_time] ; concatenate time arrays
            Bdata = [Bdata, B_data] ; concatenate data arrays
        ENDELSE
    ENDIF

    temp_time = temp_time+86400 ; advance one day
ENDFOR

; check for invalid data flag (1E32)
IF n_elements(Bdata) GT 0 THEN BEGIN 
    invalid_i = where(Bdata GT 1E31, invalid_c)
    IF invalid_c GT 0 THEN Bdata[invalid_i] = !values.f_nan

    limind = WHERE(Btime GE t_s AND Btime LE t_E, climind)

    IF climind GT 0 THEN BEGIN
        store_data, 'ST' + sc + '_MAG_RTN_X', data = {x:Btime(limind), y:REFORM(Bdata(limind, 0))}, $
                    dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LX!N RTN (nT)'}
        store_data, 'ST' + sc + '_MAG_RTN_Y', data = {x:Btime(limind), y:REFORM(Bdata(limind, 1))}, $
                    dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LY!N RTN (nT)'}
        store_data, 'ST' + sc + '_MAG_RTN_Z', data = {x:Btime(limind), y:REFORM(Bdata(limind, 2))}, $
                    dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LZ!N RTN (nT)'}
        store_data, 'ST' + sc + '_MAG_RTN_T', data = {x:Btime(limind), y:REFORM(Bdata(limind, 3))}, $
                    dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!Ltot!N RTN (nT)'}

        IF keyword_set(GSM) OR keyword_set(GSE) THEN BEGIN 
            temp_time = time_string(Btime[limind])
            strput, temp_time, 'T', 10 ; change time string format for SolarSoft
            new_data = transpose(Bdata[limind, 0:2])
        
            IF keyword_set(GSE) THEN BEGIN 
                convert_stereo_coord, temp_time, new_data, 'RTN', 'GSE', spacecraft = sc, /ignore_origin
                prefix = 'GSE' 
            ENDIF ELSE BEGIN 
                convert_stereo_coord, temp_time, new_data, 'RTN', 'GSM', spacecraft = sc, /ignore_origin
                prefix = 'GSM'
            ENDELSE 

            new_total = sqrt((reform(new_data[0, *]) * reform(new_data[0, *])) + $
                             (reform(new_data[1, *]) * reform(new_data[1, *])) + $
                             (reform(new_data[2, *]) * reform(new_data[2, *])))

            store_data, 'ST' + sc + '_MAG_'+prefix+'_X', data = {x:Btime(limind), y:REFORM(new_data[0, *])}, $
                        dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LX!N RTN (nT)'}
            store_data, 'ST' + sc + '_MAG_'+prefix+'_Y', data = {x:Btime(limind), y:REFORM(new_data[1, *])}, $
                        dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LY!N RTN (nT)'}
            store_data, 'ST' + sc + '_MAG_'+prefix+'_Z', data = {x:Btime(limind), y:REFORM(new_data[2, *])}, $
                        dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!LZ!N RTN (nT)'}
            store_data, 'ST' + sc + '_MAG_'+prefix+'_T', data = {x:Btime(limind), y:new_total}, $
                        dlim = {panel_size:1, ytitle:'ST'+sc+'!C!C'+'B!Ltot!N RTN (nT)'}
        ENDIF 
    ENDIF
ENDIF 

END
