;+
; FUNCTION: get_pla_wap_priority_ssd
;
; PURPOSE: To create the data structure of the 'wap_priority_ssd' product
;
; INPUT: sat: S/C id - 'A' or 'B'
;
; OUTPUT: the data structure is returned
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   05-15-2007: CM:  Actual maxrec is epochinfo.maxrec+1
;   03-25-2007: LBE: Added indices for schan
;   05-05-2010: LBE: Added summed to structure.
;   06-03-2011: LBE Added check for CDF V10.
;   07-27-2011: LBE Make schan NaN when fill.
;   09-27-2012: LBE: Added set_to_beg keyword.
;   12-12-2012: LBE: Remove data where no num_summed info.
;   01-25-2013: LBE: Add erase gaps keywords (to make spears in plots disappear).
;   10-03-2014: LBE: Fix bug in check for new_inter.
;-

FUNCTION get_pla_wap_priority_ssd, sat, class, set_to_beg = set_to_beg, erase_gaps = erase_gaps
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose
get_err_no = 0 & get_err_msg = ''

prod        = 'wap_priority_ssd'
nenergy     =  128
nposition   =  1
ndeflection =  1

seek_files, sat, 'science', filesfound ; find files that correspond to time interval
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN, 0
ENDIF

;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))
        version = fix(strmid(filesfound[jj], strlen(filesfound[jj])-6, 2))
        IF version LT 10 THEN prod_name = 'epoch5_heavy' ELSE prod_name = 'epoch_heavy'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'wap_priority_ssd'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF version LT 10 THEN prod_name = 's_chan5_heavy' ELSE prod_name = 's_chan_wap_ssd_pri'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF version LT 10 THEN prod_name = 'heavy_ion_num_summed5' ELSE prod_name = 'num_summed_wap_ssd_pri'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_heavy_ion_num_summed5, rec_count = epochinfo.maxrec+1, /zvariable

            IF version LT 10 THEN prod_name = 'error5_heavy' ELSE prod_name = 'error_wap_ssd_pri'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF ifwd EQ 0 THEN BEGIN
                converted_epoch       =    REFORM(time_double(epoch, /epoch))
                data                  = TRANSPOSE(cdf_data)
                s_chan5_heavy         =    REFORM(cdf_s_chan5_heavy)
                heavy_ion_num_summed5 = TRANSPOSE(cdf_heavy_ion_num_summed5)
                error5_heavy          =    REFORM(cdf_error5_heavy)
            ENDIF ELSE BEGIN
                converted_epoch       = [converted_epoch,          REFORM(time_double(epoch, /epoch))]
                s_chan5_heavy         = [s_chan5_heavy,            REFORM(cdf_s_chan5_heavy)]
                heavy_ion_num_summed5 = [heavy_ion_num_summed5, TRANSPOSE(cdf_heavy_ion_num_summed5)]
                error5_heavy          = [error5_heavy,             REFORM(cdf_error5_heavy)]
                IF n_elements(epoch) EQ 1 THEN BEGIN ; only one record
                    temp_data = lonarr(1, 2, 128)
                    temp_data[0, *, *] = transpose(cdf_data)
                    data               = [data,             temp_data]
                ENDIF ELSE BEGIN 
                    data               = [data,   transpose(cdf_data)]
                ENDELSE                 
            ENDELSE
            ifwd =  ifwd + 1
        ENDIF
        cdf_close, cdf_id
    ENDIF
ENDFOR

IF ifwd EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0
ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error5_heavy EQ 0, cff)
IF cff GT 0 THEN BEGIN    
    converted_epoch = converted_epoch(findfill)
    s_chan5_heavy = s_chan5_heavy(findfill)
    error5_heavy = error5_heavy(findfill)
    heavy_ion_num_summed5 = heavy_ion_num_summed5(findfill, *)
    data = data(findfill, *, *)
ENDIF

good_i = where(s_chan5_heavy GE 0, good_count)
IF good_count GT 0 THEN BEGIN
    converted_epoch       = converted_epoch      [good_i]
    s_chan5_heavy         = s_chan5_heavy        [good_i]
    heavy_ion_num_summed5 = heavy_ion_num_summed5[good_i, *]
    error5_heavy          = error5_heavy         [good_i]
    data                  = data                 [good_i, *, *]
ENDIF

good_i = where(heavy_ion_num_summed5[*, 0] GE 0, good_count)
IF good_count GT 0 THEN BEGIN 
    converted_epoch       = converted_epoch      [good_i]
    s_chan5_heavy         = s_chan5_heavy        [good_i]
    heavy_ion_num_summed5 = heavy_ion_num_summed5[good_i, *]
    error5_heavy          = error5_heavy         [good_i]
    data                  = data                 [good_i, *, *]
ENDIF
;------------------------------------------------------------------
; Check individual arrays for value range validity
;------------------------------------------------------------------

icepoch =  WHERE(converted_epoch LT (time_double('1958-01-01')) OR $
                 converted_epoch GT (time_double('2050-01-01')), cicepoch)
IF cicepoch GT 0 THEN STOP

data =  FLOAT(data)
icdata = WHERE(data EQ -1, cicdata)
IF cicdata GT 0 THEN BEGIN
    data(icdata) =  !VALUES.F_NAN
ENDIF

s_chan5_heavy = float(s_chan5_heavy)
isch =  WHERE(s_chan5_heavy LT 0 ,  cisch)
IF cisch GT 0 THEN BEGIN
    s_chan5_heavy[isch] = !values.f_nan
ENDIF 

ierr =  WHERE(error5_heavy NE 0 AND error5_heavy NE 1,  cierr)
;IF cierr GT 0 THEN STOP

IF keyword_set(erase_gaps) THEN BEGIN 
; Fill data gaps with NaN
    get_timespan, temp_time
    counts     = n_elements(converted_epoch)
    interval   = 5 * 60 ; 5 min cadence
    num_recs   = round(((converted_epoch[counts-1] - converted_epoch[0]) / interval)) + 2
    new_times  = double(indgen(num_recs)) * interval + converted_epoch[0]
    new_schan  = fltarr(num_recs)
    new_summd  = intarr(num_recs, 128)
    new_error  = intarr(num_recs)
    new_data   = fltarr(num_recs, 2, 128)
    new_inter  = intarr(num_recs)
    new_schan[*      ] = !values.f_nan
    new_summd[*, *   ] = -1
    new_error[*      ] = -1
    new_data [*, *, *] = !values.f_nan
    new_inter[*      ] = interval
    
    FOR ii = 0L, n_elements(converted_epoch)-1 DO BEGIN 
        min = min(abs(converted_epoch[ii]-new_times), min_i, /nan)
        IF min GT new_inter[min_i]/2.0 THEN stop ; problem
        new_times[min_i      ] = converted_epoch      [ii      ]
        new_schan[min_i      ] = s_chan5_heavy        [ii      ]
        new_summd[min_i, *   ] = heavy_ion_num_summed5[ii, *   ]
        new_error[min_i      ] = error5_heavy         [ii      ]
        new_data [min_i, *, *] = data                 [ii, *, *]
    ENDFOR 
    converted_epoch       = new_times
    s_chan5_heavy         = new_schan
    heavy_ion_num_summed5 = new_summd
    error5_heavy          = new_error
    data                  = new_data
ENDIF 

;------------------------------------------------------------------
; Restrict data array to classification required
;------------------------------------------------------------------
data =  REFORM(data(*, class, *))

;------------------------------------------------------------------
;convert energy/deflection/position/s_channel steps to actual values
;------------------------------------------------------------------
get_pla_energies, sat, esa_step             ; get energy bin values range
get_pla_theta_angles, sat, prod, theta_step ; get polar (deflection) angles range
get_pla_phi_angles, sat, prod, phi_step     ; get azimuthal (position) angles range

;position   = FLTARR(N_ELEMENTS(converted_epoch), nposition)
deflection = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
energy = FLTARR(N_ELEMENTS(converted_epoch), 128)

FOR ii = 0, N_ELEMENTS(converted_epoch)-1 DO BEGIN
    energy(ii, *) =  esa_step
    deflection(ii, *) =  theta_step
;    position(ii, *) =  phi_step
ENDFOR

;convert s_chan5_heavy to energy
s_chan_i = s_chan5_heavy
good_i = where(finite(s_chan5_heavy) EQ 1, count)
IF count GT 0 THEN s_chan5_heavy[good_i] = esa_step(s_chan5_heavy[good_i])
no_schan_i = where(s_chan_i EQ 0, count)
IF count GT 0 THEN BEGIN        ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan5_heavy[no_schan_i] = 0.1
ENDIF 

;------------------------------------------------------------------
; Time tag of the middle of the bin
;------------------------------------------------------------------
IF keyword_set(set_to_beg) EQ 0 THEN $
  converted_epoch = converted_epoch + 150. ; time shift by 2.5 minutes

;------------------------------------------------------------------
; Integration time
;------------------------------------------------------------------
packets = N_ELEMENTS(converted_epoch)
dt =  12.8e-3 ; ms - Integration time for one deflection
integ_t = FLTARR(packets, nenergy)
FOR ip = 0, packets-1 DO BEGIN
    FOR ien = 0, nenergy-1 DO BEGIN
        integ_t(ip, ien) = dt * heavy_ion_num_summed5(ip, ien) * 32.
    ENDFOR
ENDFOR

;------------------------------------------------------------------
; Construct the dara structure
;------------------------------------------------------------------
retdata = {$
          project:     'STEREO/PLASTIC', $
          prod_name:   prod, $
          units_name:  'Counts', $
          ntime:       N_ELEMENTS(converted_epoch), $
          time:        converted_epoch, $
          integ_t:     integ_t, $
          data:        data, $
          nenergy:     nenergy, $
          energy:      energy, $
          nposition:   nposition, $
;          position:    position, $
          ndeflection: ndeflection, $
          deflection:  deflection, $
          s_chan:      s_chan5_heavy, $
          s_chan_i:    s_chan_i, $
          summed:      heavy_ion_num_summed5, $
          error:       error5_heavy $
          }

RETURN, retdata

END
