;+
; FUNCTION: get_pla_wap_no_ssd_dcr
;
; PURPOSE: To create the data structure of the 'wap_no_ssd_dcr' product
;
; INPUT: sat: S/C id - 'A' or 'B'
;
; OUTPUT: the data structure is returned
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   05-15-2007: CM  Actual maxrec is epochinfo.maxrec+1
;   02-19-2008: LBE Changed loop variable to long to deal with longer time
;               period.
;   03-25-2007: LBE Added indices for schan
;   12-08-2008: LBE Added check for no valid data.
;   04-16-2010: LBE Added summed to data structure.
;   06-03-2011: LBE Added check for CDF V10.
;   07-27-2011: LBE Make schan NaN when fill.
;   12-22-2011: LBE: Fix when only one record.
;   09-27-2012: LBE: Added set_to_beg keyword.
;   12-05-2012: LBE: Remove data where no num_summed info.
;   01-04-2013: LBE: Special case for STB for 6/28/2011.
;   01-28-2013: LBE: Add erase gaps keywords (to make smears in plots disappear).
;   03-11-2014: LBE: Changes in B accum in 2/2014.
;   04-08-2014: LBE: For memory problems, check before reallocated arrays if have all data
;   07-14-2014: LBE: Changes in A accum in 7/2014.
;   10-03-2014: LBE: Fix bug in check for new_inter.
;   01-07-2014: LBE: Check for data before erase_gaps.
;   06-20-2016: LBE: Changes in A accum.
;-

FUNCTION get_pla_wap_no_ssd_dcr, sat, class, set_to_beg = set_to_beg, erase_gaps = erase_gaps
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose
get_err_no = 0 & get_err_msg = ''

prod        = 'wap_no_ssd_dcr'
nenergy     =  128
nposition   =  8
ndeflection =  1

seek_files, sat, 'science', filesfound ; find files that correspond to time interval
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN, 0
ENDIF

;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))
        version = fix(strmid(filesfound[jj], strlen(filesfound[jj])-6, 2))
        IF version LT 10 THEN prod_name = 'epoch5_heavy' ELSE prod_name = 'epoch_heavy'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable

            prod_name = 'wap_no_ssd_dcr'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable

            IF strmid(filesfound[jj], strlen(filesfound[jj])-31, 31) EQ 'STB_L1_PLA_20110628_179_V10.cdf' THEN $
              prod_name = 's_chan_apid319' $ ; change to new DPU software
            ELSE IF version LT 10 THEN prod_name = 's_chan5_heavy' ELSE prod_name = 's_chan_wap_nossd_dcr'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF version LT 10 THEN prod_name = 'heavy_ion_num_summed10' ELSE prod_name = 'num_summed_wap_nossd_dcr'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_heavy_ion_num_summed10, rec_count = epochinfo.maxrec+1, /zvariable

            IF version LT 10 THEN prod_name = 'error5_heavy' ELSE prod_name = 'error_wap_nossd_dcr'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error5_heavy, rec_count = epochinfo.maxrec+1, /zvariable

            IF ifwd EQ 0 THEN BEGIN
                converted_epoch           = REFORM   (time_double(epoch, /epoch))
                data                      = TRANSPOSE(cdf_data)
                s_chan5_heavy             = REFORM   (cdf_s_chan5_heavy)
                heavy_ion_num_summed10    = TRANSPOSE(cdf_heavy_ion_num_summed10)
                error5_heavy              = REFORM   (cdf_error5_heavy)
            ENDIF ELSE BEGIN
                converted_epoch           = [converted_epoch,        REFORM   (time_double(epoch, /epoch))]
                s_chan5_heavy             = [s_chan5_heavy,          REFORM   (cdf_s_chan5_heavy)]
                heavy_ion_num_summed10    = [heavy_ion_num_summed10, TRANSPOSE(cdf_heavy_ion_num_summed10)]
                error5_heavy              = [error5_heavy,           REFORM   (cdf_error5_heavy)]
                IF n_elements(epoch) EQ 1 THEN BEGIN ; only one record
                    temp_data = lonarr(1, 7, 8, 128)
                    temp_data[0, *, *, *] = transpose(cdf_data)
                    data                  = [data,             temp_data]
                ENDIF ELSE BEGIN 
                    data                  = [data,   transpose(cdf_data)]
                ENDELSE 
            ENDELSE
            
            ifwd =  ifwd + 1
            
        ENDIF

        cdf_close, cdf_id

    ENDIF
    
ENDFOR

IF ifwd EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0
ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error5_heavy EQ 0, cff)
IF (cff GT 0) AND (cff NE n_elements(converted_epoch)) THEN BEGIN    
    converted_epoch        = converted_epoch       (findfill)
    s_chan5_heavy          = s_chan5_heavy         (findfill)
    error5_heavy           = error5_heavy          (findfill)
    heavy_ion_num_summed10 = heavy_ion_num_summed10(findfill, *)
    data                   = data                  (findfill, *, *, *)
ENDIF

good_i = where(s_chan5_heavy GE 0, good_count)
IF (good_count GT 0) AND (good_count NE n_elements(converted_epoch)) THEN BEGIN
    converted_epoch        = converted_epoch       [good_i         ]
    s_chan5_heavy          = s_chan5_heavy         [good_i         ]
    heavy_ion_num_summed10 = heavy_ion_num_summed10[good_i, *      ]
    error5_heavy           = error5_heavy          [good_i         ]
    data                   = data                  [good_i, *, *, *]
ENDIF

good_i = where(heavy_ion_num_summed10[*, 0] GE 0, good_count)
IF (good_count GT 0) AND (good_count NE n_elements(converted_epoch)) THEN BEGIN 
    converted_epoch        = converted_epoch       [good_i]
    s_chan5_heavy          = s_chan5_heavy         [good_i]
    heavy_ion_num_summed10 = heavy_ion_num_summed10[good_i, *]
    error5_heavy           = error5_heavy          [good_i]
    data                   = data                  [good_i, *, *, *]
ENDIF

;------------------------------------------------------------------
; Check individual arrays for value range validity
;------------------------------------------------------------------

icepoch =  WHERE(converted_epoch LT (time_double('1958-01-01')) OR $
                 converted_epoch GT (time_double('2050-01-01')), cicepoch)
IF cicepoch GT 0 THEN STOP

data =  FLOAT(data)
icdata = WHERE(data EQ -1, cicdata)
IF cicdata GT 0 THEN BEGIN
    data(icdata) =  !VALUES.F_NAN
ENDIF

s_chan5_heavy = float(s_chan5_heavy)
isch =  WHERE(s_chan5_heavy LT 0 ,  cisch)
IF cisch GT 0 THEN BEGIN
    s_chan5_heavy[isch] = !values.f_nan
ENDIF 

ierr =  WHERE(error5_heavy NE 0 AND error5_heavy NE 1,  cierr)
;IF cierr GT 0 THEN STOP

IF keyword_set(erase_gaps) THEN BEGIN 
    good_i = where(finite(data) EQ 1, good_c)
    IF good_c GT 0 THEN BEGIN 
; Fill data gaps with NaN
        get_timespan, temp_time
        IF sat EQ 'A' THEN BEGIN
            a1_time  = time_double('2011-06-24/21:22:16') ; 10 - 60
            a2_time  = time_double('2011-09-19/15:00:00') ; 60 - 10
            a3_time  = time_double('2011-09-28/15:00:16') ; 10 - 60
            a4_time  = time_double('2014-07-11/14:15:00') ; 60 - 10
            a5_time  = time_double('2016-05-15/00:00:00') ; 10 - 05
            a6_time  = time_double('2016-06-16/17:18:00') ; 05 - 10
            wanted0  = where(converted_epoch LT a1_time, count0)
            wanted1  = where(converted_epoch GE a1_time AND converted_epoch LT a2_time, count1)
            wanted2  = where(converted_epoch GE a2_time AND converted_epoch LT a3_time, count2)
            wanted3  = where(converted_epoch GE a3_time AND converted_epoch LT a4_time, count3)
            wanted4  = where(converted_epoch GE a4_time AND converted_epoch LT a5_time, count4)
            wanted5  = where(converted_epoch GE a5_time AND converted_epoch LT a6_time, count5)
            wanted6  = where(converted_epoch GE a6_time, count6)
            cadences = [10, 60, 10, 60, 10, 5, 10]
            counts   = [count0, count1, count2, count3, count4, count5, count6]
        ENDIF ELSE BEGIN 
            b1_time  = time_double('2011-07-13/01:20:38') ; 10 - 60
            b2_time  = time_double('2014-02-26/23:30:58') ; 30 -  5
            b3_time  = time_double('2014-02-27/23:00:00') ;  5 - 30
            wanted0  = where(converted_epoch LT b1_time, count0)
            wanted1  = where(converted_epoch GE b1_time AND converted_epoch LT b2_time, count1)
            wanted2  = where(converted_epoch GE b2_time AND converted_epoch LT b3_time, count2)
            wanted3  = where(converted_epoch GE b3_time, count3)
            cadences = [10, 60, 10, 60]
            counts   = [count0, count1, count2, count3]
        ENDELSE 
        
        FOR ii = 0, n_elements(cadences)-1 DO BEGIN 
            IF counts[ii] GT 0 THEN BEGIN 
                interval = cadences[ii] * 60
                CASE ii OF
                    0: wanted_i = wanted0
                    1: wanted_i = wanted1
                    2: wanted_i = wanted2
                    3: wanted_i = wanted3
                    4: wanted_i = wanted4
                    5: wanted_i = wanted5
                    6: wanted_i = wanted6
                ENDCASE 
                num_recs   = round(((converted_epoch[wanted_i[counts[ii]-1]] - converted_epoch[wanted_i[0]]) / interval)) + 2
                temp_times = double(indgen(num_recs)) * interval + converted_epoch[wanted_i[0]]
                temp_schan = fltarr(num_recs)
                temp_summd = intarr(num_recs, 128)
                temp_error = intarr(num_recs)
                temp_data  = fltarr(num_recs, 7, 8, 128)
                temp_inter = intarr(num_recs)
                temp_schan[*         ] = !values.f_nan
                temp_summd[*, *      ] = -1
                temp_error[*         ] = -1
                temp_data [*, *, *, *] = !values.f_nan
                temp_inter[*         ] = interval
                IF n_elements(new_times) EQ 0 THEN BEGIN 
                    new_times = temp_times
                    new_schan = temp_schan
                    new_summd = temp_summd
                    new_error = temp_error
                    new_data  = temp_data
                    new_inter = temp_inter
                ENDIF ELSE BEGIN 
                    good_i = where(new_times LT temp_times[0], count, ncomplement = bad_count) ; check for overlapping times
                    IF bad_count GT 0 THEN BEGIN
                        new_times = new_times[good_i         ]
                        new_schan = new_schan[good_i         ]
                        new_summd = new_summd[good_i, *      ]
                        new_error = new_error[good_i         ]
                        new_data  = new_data [good_i, *, *, *]
                        new_inter = new_inter[good_i         ]
                    ENDIF 
                    IF temp_times[0]-new_times[count-1] GT interval THEN BEGIN ; check for data gap
                        temp2_summd = intarr(1, 128)
                        temp2_data  = fltarr(1, 7, 8, 128)
                        temp2_summd[*, *      ]  = -1
                        temp2_data [*, *, *, *] = !values.f_nan
                        temp_times = [(temp_times[0]-interval), temp_times]
                        temp_schan = [!values.f_nan,            temp_schan]
                        temp_summd = [temp2_summd,              temp_summd]
                        temp_error = [-1,                       temp_error]
                        temp_data  = [temp2_data,               temp_data ]
                        temp_inter = [new_inter[count-1],       temp_inter]
                    ENDIF 
                    new_times = [new_times, temp_times]
                    new_schan = [new_schan, temp_schan]
                    new_summd = [new_summd, temp_summd]
                    new_error = [new_error, temp_error]
                    new_data  = [new_data,  temp_data ]
                    new_inter = [new_inter, temp_inter]
                ENDELSE 
            ENDIF 
        ENDFOR 
    
        FOR ii = 0L, n_elements(converted_epoch)-1 DO BEGIN 
            min = min(abs(converted_epoch[ii]-new_times), min_i, /nan)
            IF min GT new_inter[min_i]/2.0 THEN stop ; problem
            new_times[min_i         ] = converted_epoch       [ii         ]
            new_schan[min_i         ] = s_chan5_heavy         [ii         ]
            new_summd[min_i, *      ] = heavy_ion_num_summed10[ii, *      ]
            new_error[min_i         ] = error5_heavy          [ii         ]
            new_data [min_i, *, *, *] = data                  [ii, *, *, *]
        ENDFOR 
        converted_epoch        = new_times
        s_chan5_heavy          = new_schan
        heavy_ion_num_summed10 = new_summd
        error5_heavy           = new_error
        data                   = new_data
    ENDIF 
ENDIF 

;------------------------------------------------------------------
; Restrict data array to classification required
;------------------------------------------------------------------
data =  REFORM(data(*, class, *, *))

;------------------------------------------------------------------
;convert energy/deflection/position/s_channel steps to actual values
;------------------------------------------------------------------
get_pla_energies, sat, esa_step             ; get energy bin values range
get_pla_theta_angles, sat, prod, theta_step ; get polar (deflection) angles range
get_pla_phi_angles, sat, prod, phi_step     ; get azimuthal (position) angles range

position   = FLTARR(N_ELEMENTS(converted_epoch), nposition)
deflection = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
energy     = FLTARR(N_ELEMENTS(converted_epoch), nenergy)

FOR ii = 0L, N_ELEMENTS(converted_epoch)-1 DO BEGIN
    energy    (ii, *) =  esa_step
    deflection(ii, *) =  theta_step
    position  (ii, *) =  phi_step
ENDFOR

;convert s_chan5_heavy to energy
s_chan_i = s_chan5_heavy
good_i = where(finite(s_chan5_heavy) EQ 1, good_count)
IF good_count GT 0 THEN s_chan5_heavy[good_i] = esa_step(s_chan5_heavy[good_i])
no_schan_i = where(s_chan_i EQ 0, count)
IF count GT 0 THEN BEGIN ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan5_heavy[no_schan_i] = 0.1
ENDIF 

;------------------------------------------------------------------
; Time tag of the middle of the bin
;------------------------------------------------------------------
IF keyword_set(set_to_beg) EQ 0 THEN $
  converted_epoch = converted_epoch + 300. ; time shift by 5.0 minutes

;------------------------------------------------------------------
; Integration time
;------------------------------------------------------------------
packets = N_ELEMENTS(converted_epoch)

dt =  12.8e-3 ; ms

integ_t = FLTARR(packets, nposition, nenergy)
FOR ip = 0L, packets-1 DO BEGIN
    FOR ien = 0, nenergy-1 DO BEGIN
        
        integ_t(ip, *, ien) = dt * heavy_ion_num_summed10(ip, ien) * 32.

    ENDFOR
ENDFOR

;------------------------------------------------------------------
; Construct the dara structure
;------------------------------------------------------------------
retdata = {$
          project:     'STEREO/PLASTIC', $
          prod_name:   prod, $
          units_name:  'Counts', $
          ntime:       N_ELEMENTS(converted_epoch), $
          time:        converted_epoch, $
          integ_t:     integ_t, $
          data:        data, $
          nenergy:     nenergy, $
          energy:      energy, $
          nposition:   nposition, $
          position:    position, $
          ndeflection: ndeflection, $
          deflection:  deflection, $
          s_chan:      s_chan5_heavy, $
          s_chan_i:    s_chan_i, $
          summed:      heavy_ion_num_summed10, $
          error:       error5_heavy $
          }

RETURN, retdata

END
