;+
; FUNCTION: get_pla_sw_z2_l
;
; PURPOSE: To create the data structure of the 'sw z>2 l' product
;
; INPUT: sat: S/C id - 'A' or 'B'
;
; OUTPUT: the data structure is returned
;
; KEYWORDS: full_chan: 2=main,1=schan, default=both - use to specify which channel to include in the full_array
;           roll_eff : Use roll efficiencies for each position bin.
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   05-15-2007: CM  Actual maxrec is epochinfo.maxrec+1
;   02-19-2008: LBE Changed loop variable to long to deal with longer time period.
;   03-25-2007: LBE Added indices for schan
;   06-03-2011: LBE Added check for CDF V10.
;   07-27-2011: LBE Make schan NaN when fill.
;   12-22-2011: LBE Fix when only one record.
;   04-04-2012: LBE Added full_chan and roll_eff keywords.
;   05-02-2012: LBE Added position_i.
;   09-27-2012: LBE: Added set_to_beg keyword.
;   12-12-2012: LBE: Remove data where no num_summed info.
;   01-25-2013: LBE: Add erase gaps keywords (to make spears in plots disappear).
;   03-10-2014: LBE: Take into account change in accumulation time.
;   07-14-2014: LBE: Changes in A accum in 7/2014.
;   08-11-2014: LBE: Changes in A accum in 8/2014.
;   10-03-2014: LBE: Fix bug in check for new_inter.
;   06-20-2016: LBE: Changes in A accum.
;-

FUNCTION get_pla_sw_z2_l, sat,  class, full_chan = full_chan, roll_eff = roll_eff, set_to_beg = set_to_beg, $
  erase_gaps = erase_gaps
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose
get_err_no = 0 & get_err_msg = ''

prod        = 'sw_z>2_l'
nenergy     =  128
nposition   =  8
ndeflection =  1

seek_files, sat, 'science', filesfound ; find files that correspond to time interval
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN, 0
ENDIF

;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN
    IF filesfound(jj) NE '' THEN BEGIN
        cdf_id = cdf_open(filesfound(jj))
        version = fix(strmid(filesfound[jj], strlen(filesfound[jj])-6, 2))
        IF version LT 10 THEN prod_name = 'epoch5_heavy' ELSE prod_name = 'epoch_heavy'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'sw_z>2_l'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF version LT 10 THEN prod_name = 's_chan5_heavy' ELSE prod_name = 's_chan_sw_z>2_l'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
           
            IF version LT 10 THEN prod_name = 'heavy_ion_num_summed5' ELSE prod_name = 'num_summed_sw_z>2_l'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_heavy_ion_num_summed5, rec_count = epochinfo.maxrec+1, /zvariable

            IF version LT 10 THEN prod_name = 'error5_heavy' ELSE prod_name = 'error_sw_z>2_l'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF ifwd EQ 0 THEN BEGIN
                converted_epoch       = REFORM(time_double(epoch, /epoch))
                data                  = TRANSPOSE(cdf_data)
                s_chan5_heavy         = REFORM(cdf_s_chan5_heavy)
                heavy_ion_num_summed5 = TRANSPOSE(cdf_heavy_ion_num_summed5)
                error5_heavy          = REFORM(cdf_error5_heavy)
            ENDIF ELSE BEGIN
                converted_epoch       = [converted_epoch, REFORM(time_double(epoch, /epoch))]
                s_chan5_heavy         = [s_chan5_heavy,   REFORM(cdf_s_chan5_heavy)]
                heavy_ion_num_summed5 = [heavy_ion_num_summed5, TRANSPOSE(cdf_heavy_ion_num_summed5)]
                error5_heavy          = [error5_heavy,    REFORM(cdf_error5_heavy)]
                IF n_elements(epoch) EQ 1 THEN BEGIN ; only one record
                    temp_data = lonarr(1, 13, 8, 128)
                    temp_data[0, *, *, *] = transpose(cdf_data)
                    data                  = [data,             temp_data]
                ENDIF ELSE BEGIN 
                    data                  = [data,   transpose(cdf_data)]
                ENDELSE                 
            ENDELSE
            ifwd =  ifwd + 1
        ENDIF
        cdf_close, cdf_id
    ENDIF
ENDFOR

IF ifwd EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0
ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error5_heavy EQ 0, cff)
IF cff GT 0 THEN BEGIN    
    converted_epoch = converted_epoch(findfill)
    s_chan5_heavy = s_chan5_heavy(findfill)
    error5_heavy = error5_heavy(findfill)
    heavy_ion_num_summed5 = heavy_ion_num_summed5(findfill, *)
    data = data(findfill, *, *, *)
ENDIF

good_i = where(s_chan5_heavy GE 0, good_count)
IF good_count GT 0 THEN BEGIN
    converted_epoch       = converted_epoch      [good_i         ]
    s_chan5_heavy         = s_chan5_heavy        [good_i         ]
    heavy_ion_num_summed5 = heavy_ion_num_summed5[good_i, *      ]
    error5_heavy          = error5_heavy         [good_i         ]
    data                  = data                 [good_i, *, *, *]
ENDIF

good_i = where(heavy_ion_num_summed5[*, 0] GE 0, good_count)
IF good_count GT 0 THEN BEGIN 
    converted_epoch       = converted_epoch      [good_i         ]
    s_chan5_heavy         = s_chan5_heavy        [good_i         ]
    heavy_ion_num_summed5 = heavy_ion_num_summed5[good_i, *      ]
    error5_heavy          = error5_heavy         [good_i         ]
    data                  = data                 [good_i, *, *, *]
ENDIF
;------------------------------------------------------------------
; Check individual arrays for value range validity
;------------------------------------------------------------------

icepoch =  WHERE(converted_epoch LT (time_double('1958-01-01')) OR $
                 converted_epoch GT (time_double('2050-01-01')), cicepoch)
IF cicepoch GT 0 THEN STOP

data =  FLOAT(data)
icdata = WHERE(data EQ -1, cicdata)
IF cicdata GT 0 THEN BEGIN
    data(icdata) =  !VALUES.F_NAN
ENDIF

s_chan5_heavy = float(s_chan5_heavy)
isch =  WHERE(s_chan5_heavy LT 0 ,  cisch)
IF cisch GT 0 THEN BEGIN
    s_chan5_heavy[isch] = !values.f_nan
ENDIF 

ierr =  WHERE(error5_heavy NE 0 AND error5_heavy NE 1,  cierr)
;IF cierr GT 0 THEN STOP

IF keyword_set(erase_gaps) THEN BEGIN 
; Fill data gaps with NaN
    get_timespan, temp_time
    IF sat EQ 'A' THEN BEGIN
        a1_time  = time_double('2014-01-01/12:30:00') ;   5 - 120
        a2_time  = time_double('2014-07-11/14:15:00') ; 120 -   5
        a3_time  = time_double('2014-07-12/15:00:00') ;   5 - 120
        a4_time  = time_double('2014-08-05/02:30:00') ; 120 -   5
        a5_time  = time_double('2015-11-17/19:45:00') ;   5 - 120
        a6_time  = time_double('2016-06-16/17:18:00') ; 120 -   5
        wanted0  = where(converted_epoch LT a1_time, count0)
        wanted1  = where(converted_epoch GE a1_time AND converted_epoch LT a2_time, count1)
        wanted2  = where(converted_epoch GE a2_time AND converted_epoch LT a3_time, count2)
        wanted3  = where(converted_epoch GE a3_time AND converted_epoch LT a4_time, count3)
        wanted4  = where(converted_epoch GE a4_time AND converted_epoch LT a5_time, count4)
        wanted5  = where(converted_epoch GE a5_time AND converted_epoch LT a6_time, count5)
        wanted6  = where(converted_epoch GE a6_time, count6)
        cadences = [5, 120, 5, 120, 5, 120, 5]
        counts   = [count0, count1, count2, count3, count4, count5, count6]
    ENDIF ELSE BEGIN 
        b1_time  = time_double('2013-12-30/23:02:01') ;   5 - 120
        b2_time  = time_double('2014-02-26/23:30:58') ; 120 -   5
        b3_time  = time_double('2014-02-27/23:00:00') ;   5 - 120
        wanted0  = where(converted_epoch LT b1_time, count0)
        wanted1  = where(converted_epoch GE b1_time AND converted_epoch LT b2_time, count1)
        wanted2  = where(converted_epoch GE b2_time AND converted_epoch LT b3_time, count2)
        wanted3  = where(converted_epoch GE b3_time, count3)
        cadences = [5, 120, 5, 120]
        counts   = [count0, count1, count2, count3]
    ENDELSE 
    
    FOR ii = 0, n_elements(cadences)-1 DO BEGIN 
        IF counts[ii] GT 0 THEN BEGIN 
            interval = cadences[ii] * 60
            CASE ii OF
                0: wanted_i = wanted0
                1: wanted_i = wanted1
                2: wanted_i = wanted2
                3: wanted_i = wanted3
                4: wanted_i = wanted4
                5: wanted_i = wanted5
                6: wanted_i = wanted6
            ENDCASE 
            num_recs   = round(((converted_epoch[wanted_i[counts[ii]-1]] - converted_epoch[wanted_i[0]]) / interval)) + 2
            temp_times = double(indgen(num_recs)) * interval + converted_epoch[wanted_i[0]]
            temp_schan = fltarr(num_recs)
            temp_summd = intarr(num_recs, 128)
            temp_error = intarr(num_recs)
            temp_data  = fltarr(num_recs, 13, 8, 128)
            temp_inter = intarr(num_recs)
            temp_schan[*         ] = !values.f_nan
            temp_summd[*, *      ] = -1
            temp_error[*         ] = -1
            temp_data [*, *, *, *] = !values.f_nan
            temp_inter[*         ] = interval
            IF n_elements(new_times) EQ 0 THEN BEGIN 
                new_times = temp_times
                new_schan = temp_schan
                new_summd = temp_summd
                new_error = temp_error
                new_data  = temp_data
                new_inter = temp_inter
            ENDIF ELSE BEGIN 
                good_i = where(new_times LT temp_times[0], count, ncomplement = bad_count) ; check for overlapping times
                IF bad_count GT 0 THEN BEGIN
                    new_times = new_times[good_i         ]
                    new_schan = new_schan[good_i         ]
                    new_summd = new_summd[good_i, *      ]
                    new_error = new_error[good_i         ]
                    new_data  = new_data [good_i, *, *, *]
                    new_inter = new_inter[good_i         ]
                ENDIF 
                IF temp_times[0]-new_times[count-1] GT interval THEN BEGIN ; check for data gap
                    temp2_summd = intarr(1, 128)
                    temp2_data  = fltarr(1, 13, 8, 128)
                    temp2_summd[*, *      ] = -1
                    temp2_data [*, *, *, *] = !values.f_nan
                    temp_times = [(temp_times[0]-interval), temp_times]
                    temp_schan = [!values.f_nan,            temp_schan]
                    temp_summd = [temp2_summd,              temp_summd]
                    temp_error = [-1,                       temp_error]
                    temp_data  = [temp2_data,               temp_data ]
                    temp_inter = [new_inter[count-1],       temp_inter]
                ENDIF 
                new_times = [new_times, temp_times]
                new_schan = [new_schan, temp_schan]
                new_summd = [new_summd, temp_summd]
                new_error = [new_error, temp_error]
                new_data  = [new_data,  temp_data ]
                new_inter = [new_inter, temp_inter]
            ENDELSE 
        ENDIF
    ENDFOR 

    FOR ii = 0L, n_elements(converted_epoch)-1 DO BEGIN 
        min = min(abs(converted_epoch[ii]-new_times), min_i, /nan)
        IF min GT new_inter[min_i]/2.0 THEN stop ; problem
        new_times[min_i         ] = converted_epoch      [ii         ]
        new_schan[min_i         ] = s_chan5_heavy        [ii         ]
        new_summd[min_i, *      ] = heavy_ion_num_summed5[ii, *      ]
        new_error[min_i         ] = error5_heavy         [ii         ]
        new_data [min_i, *, *, *] = data                 [ii, *, *, *]
    ENDFOR 
    converted_epoch       = new_times
    s_chan5_heavy         = new_schan
    heavy_ion_num_summed5 = new_summd
    error5_heavy          = new_error
    data                  = new_data
ENDIF 

;------------------------------------------------------------------
; Restrict data array to classification required
;------------------------------------------------------------------
data =  REFORM(data(*, class, *, *))

;------------------------------------------------------------------
;convert energy/deflection/position/s_channel steps to actual values
;------------------------------------------------------------------
get_pla_energies, sat, esa_step             ; get energy bin values range
get_pla_theta_angles, sat, prod, theta_step ; get polar (deflection) angles range
get_pla_phi_angles, sat, prod, phi_step     ; get azimuthal (position) angles range
phi_bins   = indgen(8)

position   = FLTARR(N_ELEMENTS(converted_epoch), nposition)
position_i = FLTARR(N_ELEMENTS(converted_epoch), nposition)
deflection = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
energy     = FLTARR(N_ELEMENTS(converted_epoch), nenergy)

FOR ii = 0L, N_ELEMENTS(converted_epoch)-1 DO BEGIN
    energy    (ii, *) =  esa_step
    deflection(ii, *) =  theta_step
    position  (ii, *) =  phi_step
    position_i(ii, *) =  phi_bins
ENDFOR

;convert s_chan5_heavy to energy
s_chan_i = s_chan5_heavy
good_i = where(finite(s_chan5_heavy) EQ 1, good_count)
IF good_count GT 0 THEN s_chan5_heavy[good_i] = esa_step(s_chan5_heavy[good_i])
no_schan_i = where(s_chan_i EQ 0, count)
IF count GT 0 THEN BEGIN ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan5_heavy[no_schan_i] = 0.1
ENDIF 

; create one_channel data if requested
IF keyword_set(full_chan) EQ 1 THEN BEGIN 
    FOR ii = 0, n_elements(converted_epoch)-1 DO BEGIN 
        IF s_chan_i[ii] EQ -1 THEN BEGIN ; don't know chan
            data[ii, *, *] = !values.f_nan
        ENDIF ELSE IF full_chan EQ 2 THEN BEGIN ; want main only for full counts
            IF s_chan_i[ii] LT 128 THEN data[ii, *, s_chan_i[ii]:127] = !values.f_nan
        ENDIF ELSE IF full_chan EQ 1 THEN BEGIN ; want schan only for full counts
            IF s_chan_i[ii] LT 128 THEN data[ii, *, 0:s_chan_i[ii]] = !values.f_nan
        ENDIF ELSE BEGIN
            print, 'invalid entry for full_chan'
            stop
        ENDELSE 
    ENDFOR 
ENDIF 

; Roll efficiencies -- uses position efficiencies based on rolls 
; This assumes using SChan data
IF keyword_set(roll_eff) THEN BEGIN
    IF keyword_set(full_chan) EQ 0 THEN BEGIN
        print, 'expect full_chan = 2 for roll eff'
        stop
    ENDIF ELSE IF full_chan NE 2 THEN BEGIN 
        print, 'expect full_chan = 2 for roll eff'
        stop
    ENDIF 
    roll_eff_2007_051 = [0.041305245, 0.014150797, 0.013675338, 0.052118908, 0.110454493, 0.011372107, 0.001445319, 0.001927095]
    roll_eff_2007_295 = [0.039497735, 0.037631302, 0.041360508, 0.048342350, 0.046380930, 0.017264118, 0.008923757, 0.007625305]
    roll_eff_2008_003 = [0.049474358, 0.016177377, 0.019476233, 0.049837544, 0.089146412, 0.013431646, 0.005855153, 0.008647809]
    roll_eff_2008_092 = [0.023738214, 0.016222829, 0.015370522, 0.053786955, 0.084361338, 0.039551916, 0.010796689, 0.005162685]
    roll_eff_2008_178 = [0.018323150, 0.010136897, 0.036635246, 0.082810222, 0.053650751, 0.020104806, 0.016697538, 0.012815262]
    roll_eff_2008_274 = [0.019017400, 0.013717398, 0.013629288, 0.040619772, 0.122869685, 0.024224396, 0.004351472, 0.006975906]
    roll_eff_2008_337 = [0.018495200, 0.012116242, 0.010526281, 0.063376218, 0.097341849, 0.036733133, 0.007440306, 0.002890292]
    roll_eff_2009_069 = [0.021992713, 0.016771442, 0.017665345, 0.056578670, 0.086844337, 0.030037803, 0.013647249, 0.006158129]
    roll_eff_2010_138 = [0.008624228, 0.005508914, 0.007462410, 0.045879486, 0.111955018, 0.055944517, 0.012106563, 0.002241955]
    roll_eff_2011_207 = [0.015120461, 0.009307107, 0.033254825, 0.087209386, 0.042909622, 0.018412094, 0.019601436, 0.020036188]
    epoch_str = time_struct(converted_epoch)
    FOR year = (epoch_str.year)[0], (epoch_str.year)[n_elements(converted_epoch)-1] DO BEGIN 
        year_i = where(epoch_str.year EQ year, count)
        IF count EQ n_elements(converted_epoch) THEN BEGIN ; all in one year
            start_doy = (epoch_str.doy)[0]
            stop_doy  = (epoch_str.doy)[n_elements(converted_epoch)-1]
        ENDIF ELSE IF year EQ (epoch_str.year)[0] THEN BEGIN ; have end of year
            start_doy = (epoch_str.doy)[0]
            IF year MOD 4 EQ 0 THEN stop_doy = 366 ELSE stop_doy = 365
        ENDIF ELSE IF year EQ (epoch_str.year)[n_elements(converted_epoch)-1] THEN BEGIN ; have beginning of year
            start_doy = 1
            stop_doy = (epoch_str.doy)[n_elements(converted_epoch)-1]
        ENDIF ELSE BEGIN ; have whole year
            start_doy = 1
            IF year MOD 4 EQ 0 THEN stop_doy = 366 ELSE stop_doy = 365
        ENDELSE 
        FOR doy = start_doy, stop_doy DO BEGIN 
            wanted_i = where(epoch_str.doy EQ doy, count)
            eff = dblarr(8)
            CASE (epoch_str.year)[0] OF 
                2007: BEGIN
                    IF doy LT 51 THEN eff = roll_eff_2007_051 $
                    ELSE IF doy LT 295 THEN BEGIN ; doy 51 - 294 
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2007_051[eff_i], roll_eff_2007_295[eff_i]], $
                                                     (float((epoch_str.doy)[0])-51)/244)
                        ENDFOR 
                    ENDIF ELSE BEGIN ; doy 295 - 
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2007_295[eff_i], roll_eff_2008_092[eff_i]], $
                                                     (float((epoch_str.doy)[0])-295)/73)
                        ENDFOR 
                    ENDELSE 
                END 
                2008: BEGIN
                    IF doy LT 003 THEN BEGIN ; doy < 003
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2007_295[eff_i], roll_eff_2008_092[eff_i]], $
                                                     (float((epoch_str.doy)[0])+70)/73)
                        ENDFOR 
                    ENDIF ELSE IF doy LT 92 THEN BEGIN ; doy 003 - 091
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2007_295[eff_i], roll_eff_2008_092[eff_i]], $
                                                     (float((epoch_str.doy)[0])-003)/89)
                        ENDFOR 
                    ENDIF ELSE IF doy LT 178 THEN BEGIN ; doy 092 - 177
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2008_092[eff_i], roll_eff_2008_178[eff_i]], $
                                                     (float((epoch_str.doy)[0])-092)/86)
                        ENDFOR 
                    ENDIF ELSE IF doy LT 274 THEN BEGIN ; doy 178 - 273
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2008_178[eff_i], roll_eff_2008_274[eff_i]], $
                                                     (float((epoch_str.doy)[0])-178)/96)
                        ENDFOR 
                    ENDIF ELSE IF doy LT 337 THEN BEGIN ; doy 274 - 336
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2008_274[eff_i], roll_eff_2008_337[eff_i]], $
                                                     (float((epoch_str.doy)[0])-274)/63)
                        ENDFOR 
                    ENDIF ELSE BEGIN ; doy 337 -
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2008_337[eff_i], roll_eff_2009_069[eff_i]], $
                                                     (float((epoch_str.doy)[0])-337)/98)
                        ENDFOR 
                    ENDELSE 
                END 
                2009: BEGIN 
                    IF doy LT 69 THEN BEGIN ; doy < 069
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2008_337[eff_i], roll_eff_2009_069[eff_i]], $
                                                     (float((epoch_str.doy)[0])+29)/98)
                        ENDFOR 
                    ENDIF ELSE BEGIN ; doy 069 - 
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2009_069[eff_i], roll_eff_2010_138[eff_i]], $
                                                     (float((epoch_str.doy)[0])-69)/434)
                        ENDFOR 
                    ENDELSE 
                END 
                2010: BEGIN 
                    IF doy LT 138 THEN BEGIN ; doy < 138
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2009_069[eff_i], roll_eff_2010_138[eff_i]], $
                                                     (float((epoch_str.doy)[0])+296)/434)
                        ENDFOR 
                    ENDIF ELSE BEGIN ; doy 138 - 
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2010_138[eff_i], roll_eff_2011_207[eff_i]], $
                                                     (float((epoch_str.doy)[0])-138)/434)
                        ENDFOR 
                    ENDELSE 
                END 
                2011: BEGIN 
                    IF doy LT 207 THEN BEGIN ; doy < 207
                        FOR eff_i = 0, 7 DO BEGIN
                            eff[eff_i] = interpolate([roll_eff_2010_138[eff_i], roll_eff_2011_207[eff_i]], $
                                                     (float((epoch_str.doy)[0])+227)/434)
                        ENDFOR 
                    ENDIF ELSE BEGIN ; doy 207 - 
                        eff = roll_eff_2011_207
                    ENDELSE 
                END 
            ENDCASE 
            data[wanted_i, 00, *] = data[wanted_i, 00, *] * eff[00] 
            data[wanted_i, 01, *] = data[wanted_i, 01, *] * eff[01] 
            data[wanted_i, 02, *] = data[wanted_i, 02, *] * eff[02] 
            data[wanted_i, 03, *] = data[wanted_i, 03, *] * eff[03] 
            data[wanted_i, 04, *] = data[wanted_i, 04, *] * eff[04] 
            data[wanted_i, 05, *] = data[wanted_i, 05, *] * eff[05] 
            data[wanted_i, 06, *] = data[wanted_i, 06, *] * eff[06] 
            data[wanted_i, 07, *] = data[wanted_i, 07, *] * eff[07] 
        ENDFOR 
    ENDFOR 
ENDIF 

;------------------------------------------------------------------
; Time tag of the middle of the bin
;------------------------------------------------------------------
IF keyword_set(set_to_beg) EQ 0 THEN $
  converted_epoch = converted_epoch + 150. ; time shift by 2.5 minutes

;------------------------------------------------------------------
; Integration time
;------------------------------------------------------------------
packets = N_ELEMENTS(converted_epoch)

dt =  12.8e-3 ; ms

integ_t = FLTARR(packets, nposition, nenergy)
FOR ip = 0L, packets-1 DO BEGIN
    FOR ien = 0, nenergy-1 DO BEGIN
        integ_t(ip, *, ien) = dt * heavy_ion_num_summed5(ip, ien) * 32.
    ENDFOR
ENDFOR

;------------------------------------------------------------------
; Construct the data structure
;------------------------------------------------------------------
retdata = {$
          project:     'STEREO/PLASTIC', $
          prod_name:   prod, $
          units_name:  'Counts', $
          ntime:       N_ELEMENTS(converted_epoch), $
          time:        converted_epoch, $
          integ_t:     integ_t, $
          data:        data, $
          nenergy:     nenergy, $
          energy:      energy, $
          nposition:   nposition, $
          position:    position, $
          position_i:  position_i, $
          ndeflection: ndeflection, $
          deflection:  deflection, $
          s_chan:      s_chan5_heavy, $
          s_chan_i:    s_chan_i, $
          summed:      heavy_ion_num_summed5, $
          error:       error5_heavy $
          }

RETURN, retdata

END

