;+
; PROCEDURE: get_pla_pos_spec
;
; PURPOSE:
;
; INPUT:
;
; KEYWORDS: full_chan : 2=main,1=schan, default=both - use to specify which channel to include 
;           from_full : set if want spectrogram created from full distribution --
;                       necessary if have included running average
;           weights   : Array of floats for running average
;                       Should add to 1. Should have odd # of elements. (ex: [.25, .5, .25])
;           no_defl   : Sum over defl before running ave., then put in "data"
;           accum_min     : Run accumulation rather than average -- running accum,  remove zeros
;           key_eff   : Use position bin efficiencies before finding average.
;           accum_time: [num_min_accum, start_time]
; 
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION LIST:
;   05-14-2007: CM
;               Even if the energy range for a particular time step does not
;               have data, all time steps are checked instead of
;               exiting. The time steps with no data in the energy
;               range are filled with NaN
;   12-06-2007: LBE: Changed loop variables to longs.
;   12-10-2008: LBE: Added variables for schan only and main only.
;   01-07-2009: LBE: Changed more loop variables to longs.
;   01-15-2009: LBE: Added shan_esa and peak_esa.
;   05-20-2010: LBE: Added full_chan keyword (for specifying which channel to include)
;                    Added from_full keyword (for when using running average)
;                    Added weights keyword (for when using running average)
;   05-24-2010: LBE: Added no_defl keyword -- so use data (not full_counts) even when from_full set
;   05-28-2010: LBE: Put NaN back in after sum.
;   06-08-2010: LBE: Added accum_min keyword.
;   07-16-2010: LBE: Fixed bug when schan doesn't switch.
;   07-22-2010: LBE: Made another loop variable a long.
;   09-02-2011: LBE: Added bin keyword (to plot by bins rather than degrees)
;   09-08-2011: LBE: Check for if only one energy bin found in reduced dist.
;   12-16-2011: LBE: Added key_eff keyword.
;   03-16-2012: LBE: Added roll_eff.
;   05-03-2012: LBE: Added calib.
;   05-29-2012: LBE: Added accum_time.
;   01-30-2013: LBE: Added erase_gaps keyword.
;   11-28-2016: LBE: Adjust for full_array.
;   06-13-2017: LBE: Add plots to bottom of page.
;-

PRO get_pla_pos_spec, sat, prod, class, units, energy, phi, diagn = diagn, full_chan = full_chan, weights = weights, from_full = from_full, no_defl = no_defl, accum_min = accum_min, bin = bin, key_eff = key_eff, roll_eff = roll_eff, calib = calib, accum_time = accum_time, erase_gaps = erase_gaps
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

IF keyword_set(key_eff) EQ 1 AND prod NE 'h+peak' THEN BEGIN
    print, 'Error: Position efficiencies not determined for ', prod
    stop
ENDIF 

dat = get_pla_data(sat, prod, class, full_chan = full_chan, weights = weights, from_full = from_full, no_defl = no_defl, $
                   accum_min = accum_min, key_eff = key_eff, roll_eff = roll_eff, accum_time = accum_time, erase_gaps = erase_gaps) 

IF get_err_no NE 0 THEN RETURN

;------------------------------------------------------------------
; Convert Units
;------------------------------------------------------------------
IF units NE 'Counts' THEN BEGIN
    dat = convert_plastic_units(dat, units)
ENDIF
;------------------------------------------------------------------

ytitle = 'ST' + STRUPCASE(sat) +'/PLA' + '!C!C'

IF dat.units_name EQ 'Counts' THEN norm = 1. ELSE BEGIN
    norm =  dat.nposition * dat.ndeflection
    IF keyword_set(from_full) THEN BEGIN
        print, 'integration interval not set up when running average'
        stop
    ENDIF
ENDELSE 

; The enegry range is the actual energy range
get_pla_energies, sat, esa_step
esa_step =  esa_step * 1e3
ien = WHERE(esa_step GE energy(0) AND esa_step LE energy(1), cien)
IF cien GT 0 THEN BEGIN
    elow  = esa_step(ien(cien-1)) * (1.0-0.0489)
    ehigh = esa_step(ien(0)) * (1.0+0.0489)
    erangestr = STRING(elow,  FORMAT = '(i5.5)') + '_' + STRING(ehigh,  FORMAT = '(i5.5)')
ENDIF ELSE BEGIN
    erangestr = '*****'
ENDELSE

prodname = 'ST' + sat + '_PLA_POSSPEC_' + erangestr + '_' + STRUPCASE(units) + '_' + STRUPCASE(prod)

IF prod EQ 'h_alpha' OR $
  prod EQ 'h+peak' OR $
  prod EQ 'he++peak' OR $
  prod EQ 'he++tcr' THEN BEGIN
    ; limit to when have peak
   if (keyword_set(from_full) and (keyword_set(no_defl) eq 0)) then data_with_peak = dat.full_array else data_with_peak = dat.data
   data_with_peak[0, *, *, *] = !values.f_nan
    FOR ii = 1L, dat.ntime-1 DO BEGIN
        IF (prod EQ 'h_alpha' AND dat.energy_peak_i[ii-1] LT (dat.energy_peak_i[ii]-15)) OR $
          ( prod NE 'h_alpha' AND dat.energy_peak_i[ii-1] LT (dat.energy_peak_i[ii]-05)) OR $
          (dat.energy_peak_i    [ii-1] GT (dat.energy_peak_i    [ii]+4)) OR $
          (dat.deflection_peak_i[ii-1] LT (dat.deflection_peak_i[ii]-3)) OR $
          (dat.deflection_peak_i[ii-1] GT (dat.deflection_peak_i[ii]+4)) OR $
          (dat.energy_peak_i    [ii-1] LT 0) OR $
          (dat.deflection_peak_i[ii-1] LT 0) OR $
          (dat.energy_peak_i    [  ii] LT 0) OR $
          (dat.deflection_peak_i[  ii] LT 0) THEN data_with_peak[ii, *, *, *] = !values.f_nan
    ENDFOR 
    ; limit to just schan and just main
    zdata           = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    schan_data      = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    main_data       = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    schan_with_peak = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    main_with_peak  = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    if (keyword_set(from_full)) and (keyword_set(no_defl) eq 0) then begin 
       temp_data       = dat.full_array
       temp_schan_data = dat.full_array
       temp_main_data  = dat.full_array
    endif else begin 
       temp_data       = dat.data
       temp_schan_data = dat.data
       temp_main_data  = dat.data
    endelse 
    temp_schan_with_peak = data_with_peak
    temp_main_with_peak  = data_with_peak
    IF prod EQ 'h_alpha' THEN BEGIN 
        all_main_i = where(dat.s_chan_i GT dat.energy_peak_i+4, count) ; all main channel
        IF count GT 0 THEN temp_schan_data     [all_main_i, *, *, *] = !values.f_nan
        IF count GT 0 THEN temp_schan_with_peak[all_main_i, *, *, *] = !values.f_nan
        mixed_i = where(dat.s_chan_i GE dat.energy_peak_i-15 AND $
                        dat.s_chan_i LE dat.energy_peak_i+4, count) ; has some of each channel
        IF count GT 0 THEN BEGIN
            FOR ii = 0L, count-1 DO BEGIN
                relative_i = dat.s_chan_i[mixed_i[ii]] - dat.energy_peak_i[mixed_i[ii]] + 15
                IF relative_i GT 0 THEN temp_schan_data     [mixed_i[ii], *, *, 0:relative_i-1] = !values.f_nan
                IF relative_i GT 0 THEN temp_schan_with_peak[mixed_i[ii], *, *, 0:relative_i-1] = !values.f_nan
                temp_main_data     [mixed_i[ii], *, *, relative_i:19] = !values.f_nan
                temp_main_with_peak[mixed_i[ii], *, *, relative_i:19] = !values.f_nan
            ENDFOR
        ENDIF 
        all_schan_i = where(dat.s_chan_i LT dat.energy_peak_i-15, count) ; all s-chan
        IF count GT 0 THEN temp_main_data     [all_schan_i, *, *, *] = !values.f_nan
        IF count GT 0 THEN temp_main_with_peak[all_schan_i, *, *, *] = !values.f_nan
    ENDIF ELSE BEGIN 
        all_main_i = where(dat.s_chan_i GT dat.energy_peak_i+4, count) ; all main channel
        IF count GT 0 THEN IF keyword_set(no_defl) THEN BEGIN
            temp_schan_data     [all_main_i, *, *] = !values.f_nan 
            temp_schan_with_peak[all_main_i, *, *] = !values.f_nan 
        ENDIF ELSE BEGIN 
            temp_schan_data     [all_main_i, *, *, *] = !values.f_nan
            temp_schan_with_peak[all_main_i, *, *, *] = !values.f_nan
        ENDELSE 
        mixed_i = where(dat.s_chan_i GE dat.energy_peak_i-5 AND $
                        dat.s_chan_i LE dat.energy_peak_i+4, count) ; has some of each channel
        IF count GT 0 THEN BEGIN
            FOR ii = 0L, count-1 DO BEGIN
                IF dat.s_chan_i[mixed_i[ii]] EQ -1 THEN BEGIN
                    IF keyword_set(no_defl) THEN BEGIN 
                        temp_schan_data     [mixed_i[ii], *, *] = !values.f_nan
                        temp_schan_with_peak[mixed_i[ii], *, *] = !values.f_nan
                        temp_main_data      [mixed_i[ii], *, *] = !values.f_nan
                        temp_main_with_peak [mixed_i[ii], *, *] = !values.f_nan
                    ENDIF ELSE BEGIN 
                        temp_schan_data     [mixed_i[ii], *, *, *] = !values.f_nan
                        temp_schan_with_peak[mixed_i[ii], *, *, *] = !values.f_nan
                        temp_main_data      [mixed_i[ii], *, *, *] = !values.f_nan
                        temp_main_with_peak [mixed_i[ii], *, *, *] = !values.f_nan
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF keyword_set(from_full) THEN relative_i = dat.s_chan_i[mixed_i[ii]] ELSE $
                      relative_i = dat.s_chan_i[mixed_i[ii]] - dat.energy_peak_i[mixed_i[ii]] + 5
                    IF relative_i GT 0 THEN BEGIN 
                        IF keyword_set(no_defl) THEN BEGIN 
                           temp_schan_data     [mixed_i[ii], *, 0:relative_i-1] = !values.f_nan 
                           temp_schan_with_peak[mixed_i[ii], *, 0:relative_i-1] = !values.f_nan 
                        ENDIF ELSE BEGIN 
                           temp_schan_data     [mixed_i[ii], *, *, 0:relative_i-1] = !values.f_nan
                           temp_schan_with_peak[mixed_i[ii], *, *, 0:relative_i-1] = !values.f_nan
                        ENDELSE 
                    ENDIF 
                    IF keyword_set(no_defl) THEN BEGIN
                        IF relative_i LE 127 THEN BEGIN
                            temp_main_data     [mixed_i[ii], *, relative_i:127] = !values.f_nan 
                            temp_main_with_peak[mixed_i[ii], *, relative_i:127] = !values.f_nan 
                        ENDIF 
                    ENDIF ELSE BEGIN 
                        IF relative_i LE 9 THEN BEGIN
                            temp_main_data     [mixed_i[ii], *, *, relative_i:9] = !values.f_nan
                            temp_main_with_peak[mixed_i[ii], *, *, relative_i:9] = !values.f_nan
                        ENDIF 
                    ENDELSE 
                ENDELSE 
            ENDFOR
        ENDIF 
        all_schan_i = where(dat.s_chan_i LT dat.energy_peak_i-5, count) ; all s-chan
        IF count GT 0 THEN IF keyword_set(no_defl) THEN BEGIN
            temp_main_data     [all_schan_i, *, *] = !values.f_nan 
            temp_main_with_peak[all_schan_i, *, *] = !values.f_nan 
        ENDIF ELSE BEGIN 
            temp_main_data     [all_schan_i, *, *, *] = !values.f_nan
            temp_main_with_peak[all_schan_i, *, *, *] = !values.f_nan
        ENDELSE 
    ENDELSE 
    ; limit by energy
    FOR ii = 0L, N_ELEMENTS(dat.time)-1 DO BEGIN
        en = REFORM(dat.energy(ii, *))*1.e3
        ien = WHERE(en GE energy(0) AND en LE energy(1), cien)
        IF cien GT 0 THEN BEGIN 
            IF keyword_set(no_defl) THEN BEGIN 
                zdata          (ii, *) = REFORM(TOTAL(temp_data(ii, *, ien), 3, /NaN)) ; sum over energy
                schan_data     (ii, *) = REFORM(TOTAL(temp_schan_data(ii, *, ien), 3, /NaN))
                main_data      (ii, *) = REFORM(TOTAL(temp_main_data(ii, *, ien), 3, /NaN))
                schan_with_peak(ii, *) = REFORM(TOTAL(temp_schan_with_peak(ii, *, ien), 3, /NaN))
                main_with_peak (ii, *) = REFORM(TOTAL(temp_main_with_peak(ii, *, ien), 3, /NaN))
                FOR jj = 0L,  7 DO BEGIN ; put back in NaN (0-7=pos)
                    IF array_equal(finite(temp_data           [ii, jj, ien]),  0) EQ 1 THEN zdata          [ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_schan_data     [ii, jj, ien]),  0) EQ 1 THEN schan_data     [ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_schan_with_peak[ii, jj, ien]),  0) EQ 1 THEN schan_with_peak[ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_main_data      [ii, jj, ien]),  0) EQ 1 THEN main_data      [ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_main_with_peak [ii, jj, ien]),  0) EQ 1 THEN main_with_peak [ii, jj] = !values.f_nan
                ENDFOR 
            ENDIF ELSE BEGIN 
                IF cien EQ 1 THEN BEGIN ; sum over defl
                    zdata          (ii, *) = REFORM(TOTAL(           temp_data(ii, *, *, ien), 2, /NaN))
                    schan_data     (ii, *) = REFORM(TOTAL(     temp_schan_data(ii, *, *, ien), 2, /NaN))
                    main_data      (ii, *) = REFORM(TOTAL(      temp_main_data(ii, *, *, ien), 2, /NaN))
                    schan_with_peak(ii, *) = REFORM(TOTAL(temp_schan_with_peak(ii, *, *, ien), 2, /NaN))
                    main_with_peak(ii, *)  = REFORM(TOTAL( temp_main_with_peak(ii, *, *, ien), 2, /NaN))
                ENDIF ELSE BEGIN ; sum over defl and energy
                    zdata          (ii, *) = REFORM(TOTAL(TOTAL(           temp_data(ii, *, *, ien), 2, /NaN), 3, /NaN)) 
                    schan_data     (ii, *) = REFORM(TOTAL(TOTAL(     temp_schan_data(ii, *, *, ien), 2, /NaN), 3, /NaN))
                    main_data      (ii, *) = REFORM(TOTAL(TOTAL(      temp_main_data(ii, *, *, ien), 2, /NaN), 3, /NaN))
                    schan_with_peak(ii, *) = REFORM(TOTAL(TOTAL(temp_schan_with_peak(ii, *, *, ien), 2, /NaN), 3, /NaN))
                    main_with_peak(ii, *)  = REFORM(TOTAL(TOTAL( temp_main_with_peak(ii, *, *, ien), 2, /NaN), 3, /NaN))
                ENDELSE 
                FOR jj = 0,  7 DO BEGIN ; put back in NaN
                    IF array_equal(finite(temp_data           [ii, *, jj, ien]),  0) EQ 1 THEN zdata          [ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_schan_data     [ii, *, jj, ien]),  0) EQ 1 THEN schan_data     [ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_main_data      [ii, *, jj, ien]),  0) EQ 1 THEN main_data      [ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_schan_with_peak[ii, *, jj, ien]),  0) EQ 1 THEN schan_with_peak[ii, jj] = !values.f_nan
                    IF array_equal(finite(temp_main_with_peak [ii, *, jj, ien]),  0) EQ 1 THEN main_with_peak [ii, jj] = !values.f_nan
                ENDFOR 
            ENDELSE
        ENDIF ELSE BEGIN
            err_msg,  'Energy range not available'
            zdata          (ii, *) = !values.F_NAN
            schan_data     (ii, *) = !values.F_NAN
            main_data      (ii, *) = !values.F_NAN
            schan_with_peak(ii, *) = !values.F_NAN
            main_with_peak (ii, *) = !values.F_NAN
        ENDELSE
    ENDFOR
    ytitle =  ytitle + prod

    IF keyword_set(accum_time) THEN BEGIN 
        IF keyword_set(weight)           THEN stop
        IF keyword_set(no_defl)     EQ 0 THEN stop
        get_timespan, tt
        temp_tt          = accum_time[1]
        temp_time        = dat.time-30 ; back to beginning of time
        num_records      = fix(((tt[1]-tt[0])/(60*accum_time[0]))+1)
        accum_t          = dblarr(num_records)
        accum_zdata      = fltarr(num_records, 8)
        accum_schan_data = fltarr(num_records, 8)
        accum_main_data  = fltarr(num_records, 8)
        accum_schan_peak = fltarr(num_records, 8)
        accum_main_peak  = fltarr(num_records, 8)
        num_accum        = intarr(num_records)
        temp_i           = 0L
        accum_i          = 0L
        WHILE temp_tt LT tt[1] DO BEGIN
            accum_t             [accum_i      ] = temp_tt
            wanted_i = where(temp_time GE temp_tt AND temp_time LT (accum_time[1]+(accum_time[0]*60*(temp_i+1))), count)
            IF count GT 0 THEN BEGIN
                accum_zdata     [accum_i, *, *] = total(zdata          [wanted_i, *], 1, /nan)
                accum_schan_data[accum_i, *, *] = total(schan_data     [wanted_i, *], 1, /nan)
                accum_main_data [accum_i, *, *] = total(main_data      [wanted_i, *], 1, /nan)
                accum_schan_peak[accum_i, *, *] = total(schan_with_peak[wanted_i, *], 1, /nan)
                accum_main_peak [accum_i, *, *] = total(main_with_peak [wanted_i, *], 1, /nan)
                num_accum       [accum_i      ] = count
            ENDIF ELSE BEGIN 
                accum_zdata     [accum_i, *, *] = !values.f_nan
                accum_schan_data[accum_i, *, *] = !values.f_nan
                accum_main_data [accum_i, *, *] = !values.f_nan
                accum_schan_peak[accum_i, *, *] = !values.f_nan
                accum_main_peak [accum_i, *, *] = !values.f_nan
                num_accum       [accum_i      ] = 0
            ENDELSE
            accum_i  = accum_i + 1
            temp_i   = temp_i  + 1
            temp_tt  = temp_tt + (accum_time[0]*60)
        ENDWHILE 
        IF count GT 0 THEN rec_end = accum_i-1 ELSE rec_end = accum_i
        accum_t          = accum_t         [0:rec_end      ]
        accum_zdata      = accum_zdata     [0:rec_end, *, *]
        accum_schan_data = accum_schan_data[0:rec_end, *, *]
        accum_main_data  = accum_main_data [0:rec_end, *, *]
        accum_schan_peak = accum_schan_peak[0:rec_end, *, *]
        accum_main_peak  = accum_main_peak [0:rec_end, *, *]
        num_accum        = num_accum       [0:rec_end      ]
        num_records      = rec_end+1
        pos_bins_accum   = fltarr(num_records, 8)
    ENDIF 
ENDIF 

IF prod EQ 'sw_z>2_h' THEN BEGIN
    ; limit to just schan and just main
    zdata      = FLTARR(N_ELEMENTS(dat.time), dat.nposition)
    schan_data = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    main_data  = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    temp_schan_data = dat.data
    temp_main_data  = dat.data
    size = size(dat.data)
    FOR ii = 0L, size[1]-1 DO BEGIN
        temp_schan_data[ii,*, *, 0:dat.s_chan_i[ii]-1] = !values.f_nan
        IF dat.s_chan_i[ii] LT 128 THEN temp_main_data[ii, *, *, dat.s_chan_i[ii]:127] = !values.f_nan
    ENDFOR
    ; limit by energy
    FOR ii = 0L, N_ELEMENTS(dat.time)-1 DO BEGIN
        en = REFORM(dat.energy(ii, *))*1.e3
        ien = WHERE(en GE energy(0) AND en LE energy(1), cien)
        IF cien GT 0 THEN BEGIN
            IF cien EQ 1 THEN BEGIN
                zdata     (ii, *) = REFORM(TOTAL(       dat.data(ii, *, *, ien), 2, /NaN))
                schan_data(ii, *) = REFORM(TOTAL(temp_schan_data(ii, *, *, ien), 2, /NaN))
                main_data (ii, *) = REFORM(TOTAL( temp_main_data(ii, *, *, ien), 2, /NaN))
            ENDIF ELSE BEGIN
                zdata     (ii, *) = REFORM(TOTAL(TOTAL(       dat.data(ii, *, *, ien), 2, /NaN), 3, /NaN))
                schan_data(ii, *) = REFORM(TOTAL(TOTAL(temp_schan_data(ii, *, *, ien), 2, /NaN), 3, /NaN))
                main_data (ii, *) = REFORM(TOTAL(TOTAL( temp_main_data(ii, *, *, ien), 2, /NaN), 3, /NaN))
            ENDELSE
        ENDIF ELSE BEGIN
            err_msg,  'Energy range not available'
            zdata     (ii, *) = !values.F_NAN
            schan_data(ii, *) = !values.F_NAN
            main_data (ii, *) = !values.F_NAN
        ENDELSE
    ENDFOR
    ytitle =  ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
    prodname = 'pla_' + sat + '_posspec_' + prod + '_cl' + STRING(class, FORMAT = '(i2.2)')
ENDIF

IF prod EQ 'sw_z>2_l' OR prod EQ 'wap_ssd_tcr' OR prod EQ 'wap_ssd_dcr' OR prod EQ 'wap_no_ssd_dcr' THEN BEGIN
    ; limit to just schan and just main
    schan_data = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    main_data  = FLTARR(N_ELEMENTS(dat.time), dat.nposition)    
    temp_schan_data = dat.data
    temp_main_data  = dat.data
    size = size(dat.data)
    FOR ii = 0L, size[1]-1 DO BEGIN
        IF dat.s_chan_i[ii] EQ -1 OR finite(dat.s_chan_i[ii]) EQ 0 THEN BEGIN 
            temp_schan_data[ii, *, *] = !values.f_nan
            temp_main_data [ii, *, *] = !values.f_nan
        ENDIF ELSE BEGIN 
            temp_schan_data[ii, *, 0:dat.s_chan_i[ii]-1] = !values.f_nan
            IF dat.s_chan_i[ii] LT 128 THEN temp_main_data[ii, *, dat.s_chan_i[ii]:127] = !values.f_nan
        ENDELSE 
    ENDFOR
    ; limit by energy
    zdata = FLTARR(N_ELEMENTS(dat.time), dat.nposition)
    FOR ii = 0L, N_ELEMENTS(dat.time)-1 DO BEGIN
        en = REFORM(dat.energy(ii, *))*1.e3
        ien = WHERE(en GE energy(0) AND en LE energy(1), cien)
        IF cien GT 0 THEN BEGIN
            IF cien EQ 1 THEN BEGIN
                zdata     (ii, *) = REFORM(       dat.data(ii, *, ien))
                schan_data(ii, *) = REFORM(temp_schan_data(ii, *, ien))
                main_data (ii, *) = REFORM( temp_main_data(ii, *, ien))
            ENDIF ELSE BEGIN
                zdata     (ii, *) = REFORM(TOTAL(       dat.data(ii, *, ien), 3, /NaN))
                schan_data(ii, *) = REFORM(TOTAL(temp_schan_data(ii, *, ien), 3, /NaN))
                main_data (ii, *) = REFORM(TOTAL( temp_main_data(ii, *, ien), 3, /NaN))
            ENDELSE
        ENDIF ELSE BEGIN
            err_msg,  'Energy range not available'
            zdata     (ii, *) = !values.F_NAN
            schan_data(ii, *) = !values.F_NAN
            main_data (ii, *) = !values.F_NAN
        ENDELSE
    ENDFOR
    ytitle =  ytitle + prod + '_' + STRING(class, FORMAT = '(i2.2)')
    prodname = 'pla_' + sat + '_posspec_' +  prod + '_cl' + STRING(class, FORMAT = '(i2.2)')
ENDIF

;------------------------------------------------------------------
; Save spectra in tplot variables
;------------------------------------------------------------------
erangestr = STRING(elow/1e3,  FORMAT = '(f5.2)') + '-' + STRING(ehigh/1e3,  FORMAT = '(f5.2)')
ytitle1 = ytitle + '!C!C' + erangestr + ' (keV)!C!CPos. Angle '
IF keyword_set(bin) THEN ytitle1 = ytitle1+'(bins)' ELSE ytitle1 = ytitle1+'(deg)'
ztitle = STRUPCASE(dat.units_name)
temp_name = prodname
IF keyword_set(from_full) EQ 1 THEN BEGIN 
    ytitle1 = ytitle1+' (from full)'
    temp_name = temp_name+'_from_full'
ENDIF 
IF keyword_set(bin) THEN yrange = [0, 8] ELSE yrange = [-25, 25]
IF keyword_set(calib) THEN BEGIN 
    pos_bins = fltarr(n_elements(dat.time), 8)
    pos_bins[*, 0] = -26.14
    pos_bins[*, 1] = -20.50
    pos_bins[*, 2] = -15.84
    pos_bins[*, 3] = -07.00
    pos_bins[*, 4] = -02.00
    pos_bins[*, 5] =  02.99
    pos_bins[*, 6] =  05.49
    pos_bins[*, 7] =  15.99
    temp_pos_bins = pos_bins
    pos_bins[*, 0:6] = (-0.0003 * (pos_bins[*, 0:6]^4)) $
           - (0.0049 * (pos_bins[*, 0:6]^3)) + (0.0488 * (pos_bins[*, 0:6]^2)) $
           + (1.1968 *  pos_bins[*, 0:6]   ) +  2.1126
    pos_bins[*, 7] =  (0.0425 * (pos_bins[*, 7]^2)) $
           + (0.9947 *  pos_bins[*, 7]   ) +  2.027
ENDIF ELSE if keyword_set(bin) THEN pos_bins = dat.position_i ELSE pos_bins = dat.position

store_data, temp_name, data = {x:REFORM(dat.time), y:zdata, v:pos_bins}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:0, $
                    zlog:1, $
                    zrange:[0.1, 1000], $
                    ytitle:ytitle1, $
                    ztitle:ztitle $
                   }
tplot_var = prodname
tplot, tplot_var,  add_var = 999 ; add panel to the bottom of the plot

; Accum
IF keyword_set(accum_time) THEN BEGIN 
    suffix = 'accum'+string(accum_time[0], format = '(I3.3)')
    FOR ii = 0, num_records-1 DO BEGIN 
        pos_bins_accum[ii, *] = pos_bins[0, *]
    ENDFOR 
    erangestr = STRING(elow/1e3,  FORMAT = '(f5.2)') + '-' + STRING(ehigh/1e3,  FORMAT = '(f5.2)')
    ytitle1b = ytitle + '!C!C' + erangestr + ' (keV)!C!CPos. Angle '
    IF keyword_set(bin) THEN ytitle1b = ytitle1b+'(bins)' ELSE ytitle1b = ytitle1b+'(deg)'
    ztitle = STRUPCASE(dat.units_name)
    temp_name = prodname
    IF keyword_set(from_full) EQ 1 THEN BEGIN 
        ytitle1b = ytitle1b+' (from full, '+suffix+')'
        temp_name = temp_name+'_from_full_'+suffix
    ENDIF 
    store_data, temp_name, data = {x:accum_t, y:accum_zdata, v:pos_bins_accum}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ylog:0, $
                        zlog:1, $
                        zrange:[0.1, 1000], $
                        ytitle:ytitle1b, $
                        ztitle:ztitle $
                       }
    
    ytitle1c = ytitle+'!C!CNum Cycles'
    temp_name = prodname+'_num_'+suffix
    store_data, temp_name, data = {x:accum_t, y:num_accum}
ENDIF 

ytitle2 =  ytitle + '!C!C S-Channel (keV)'
store_data, prodname+'_s_chan', data = {x:REFORM(dat.time), y:dat.s_chan}, $
            dlim = {panel_size:1, $
                    ylog:1, $
                    yrange:[0.1, 10], $
                    ytitle:ytitle2}

; av_phi
IF prod EQ 'sw_z>2_l' THEN BEGIN ; need here because might have /nan
    avg_phi =  TOTAL((zdata * pos_bins), 2, /nan) / TOTAL(zdata, 2, /nan)
    FOR ii = 0, n_elements(dat.time)-1 DO BEGIN 
        bad_i = where(finite(zdata[ii, *]) EQ 0, bad_count)
        IF bad_count EQ 8 THEN avg_phi[ii] = !values.f_nan
    ENDFOR 
ENDIF ELSE BEGIN 
    avg_phi =  TOTAL((zdata * pos_bins), 2) / TOTAL(zdata, 2) ; original code
ENDELSE 
ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
temp_name = prodname+'_av_phi'
IF keyword_set(from_full) EQ 1 THEN BEGIN 
    ytitle3 = ytitle3+' (from full)'
    temp_name = temp_name+'_from_full'
ENDIF 
store_data, temp_name, data = {x:REFORM(dat.time), y:avg_phi}, $
            dlim = {panel_size:1, $
                    ylog:0, $
                    yrange:[-25,  25], $
                    ytitle:ytitle3}

IF keyword_set(accum_time) THEN BEGIN 
    ytitle3b =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
    IF keyword_set(bin) THEN ytitle3b = ytitle3b+'(bins)' ELSE ytitle3b = ytitle3b+'(deg)'
    avg_phi_accum =  TOTAL((accum_zdata * pos_bins_accum), 2) / TOTAL(accum_zdata, 2) ; original code
    temp_name = prodname+'_av_phi'
    IF keyword_set(from_full) EQ 1 THEN BEGIN 
        ytitle3b = ytitle3b+' (from full, '+suffix+')'
        temp_name = temp_name+'_from_full_'+suffix
    ENDIF 
    store_data, temp_name, data = {x:accum_t, y:avg_phi_accum}, $
                dlim = {panel_size:1, $
                        ylog:0, $
                        yrange:[-25,  25], $
                        ytitle:ytitle3b}
ENDIF 

; av_phi schan only
avg_phi_schan_only =  TOTAL((schan_data * pos_bins), 2) / TOTAL(schan_data, 2)
ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
temp_name = prodname+'_av_phi_schan_only'
IF keyword_set(from_full) EQ 1 THEN BEGIN 
    ytitle3 = ytitle3+' (from full)'
    temp_name = temp_name+'_from_full'
ENDIF 
store_data, temp_name, data = {x:REFORM(dat.time), y:avg_phi_schan_only}, $
            dlim = {panel_size:1, $
                    ylog:0, $
                    yrange:[-25,  25], $
                    ytitle:ytitle3}
IF keyword_set(accum_time) THEN BEGIN 
    avg_phi_schan_only_accum =  TOTAL((accum_schan_data * pos_bins_accum), 2) / TOTAL(accum_schan_data, 2)
    ;IF   prod EQ 'he++tcr' AND accum_time[0] EQ 10 THEN stop
    ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
    IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
    temp_name = prodname+'_av_phi_schan_only'
    IF keyword_set(from_full) EQ 1 THEN BEGIN 
        ytitle3 = ytitle3+' (from full, '+suffix+')'
        temp_name = temp_name+'_from_full_'+suffix
    ENDIF 
    store_data, temp_name, data = {x:accum_t, y:avg_phi_schan_only_accum}, $
                dlim = {panel_size:1, $
                        ylog:0, $
                        yrange:[-25,  25], $
                        ytitle:ytitle3}
ENDIF

; limited to when have peak
IF prod EQ 'h_alpha' OR $
  prod EQ 'h+peak' OR $
  prod EQ 'he++peak' OR $
  prod EQ 'he++tcr' THEN BEGIN
    avg_phi_schan_only_with_peak =  TOTAL((schan_with_peak * pos_bins), 2) / TOTAL(schan_with_peak, 2)
    ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
    IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
    temp_name = prodname+'_av_phi_schan_only_with_peak'
    store_data, temp_name, data = {x:REFORM(dat.time), y:avg_phi_schan_only_with_peak}, $
                dlim = {panel_size:1, $
                        ylog:0, $
                        yrange:[-25,  25], $
                        ytitle:ytitle3}

    avg_phi_main_only_with_peak =  TOTAL((main_with_peak * pos_bins), 2) / TOTAL(main_with_peak, 2)
    ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
    IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
    temp_name = prodname+'_av_phi_main_only_with_peak'
    store_data, temp_name, data = {x:REFORM(dat.time), y:avg_phi_main_only_with_peak}, $
                dlim = {panel_size:1, $
                        ylog:0, $
                        yrange:[-25,  25], $
                        ytitle:ytitle3}

    IF keyword_set(accum_time) THEN BEGIN 
        avg_phi_schan_only_with_peak_accum =  TOTAL((accum_schan_peak * pos_bins_accum), 2) / TOTAL(accum_schan_peak, 2)
        ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
        IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
        temp_name = prodname+'_av_phi_schan_only_with_peak_'+suffix
        store_data, temp_name, data = {x:accum_t, y:avg_phi_schan_only_with_peak_accum}, $
                    dlim = {panel_size:1, $
                            ylog:0, $
                            yrange:[-25,  25], $
                            ytitle:ytitle3}

        avg_phi_main_only_with_peak_accum =  TOTAL((accum_main_peak * pos_bins_accum), 2) / TOTAL(accum_main_peak, 2)
        ytitle3 =  ytitle + '!C!C' + erangestr + ' (keV)!C!CAvg. Pos. Angle '
        IF keyword_set(bin) THEN ytitle3 = ytitle3+'(bins)' ELSE ytitle3 = ytitle3+'(deg)'
        temp_name = prodname+'_av_phi_main_only_with_peak_'+suffix
        store_data, temp_name, data = {x:accum_t, y:avg_phi_main_only_with_peak_accum}, $
                    dlim = {panel_size:1, $
                            ylog:0, $
                            yrange:[-25,  25], $
                            ytitle:ytitle3}
    ENDIF
ENDIF 

IF keyword_set(from_full) EQ 0 AND keyword_set(full_chan) EQ 0 THEN BEGIN ; these variables assume reduced dist
    ; Store variables for schan only and main chan only
    ytitle1 = ytitle + ' SChan Only!C!C' + erangestr + ' (keV)!C!CPos. Angle '
    IF keyword_set(bin) THEN ytitle1 = ytitle1+'(bins)' ELSE ytitle1 = ytitle1+'(deg)'
    store_data, prodname+'_schan_only', data = {x:REFORM(dat.time), y:schan_data, v:pos_bins}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ylog:0, $
                        zlog:1, $
                        zrange:[0.1, 1000], $
                        ytitle:ytitle1, $
                        ztitle:ztitle $
                       }

    ytitle1 = ytitle + ' Main Only!C!C' + erangestr + ' (keV)!C!CPos. Angle '
    IF keyword_set(bin) THEN ytitle1 = ytitle1+'(bins)' ELSE ytitle1 = ytitle1+'(deg)'
    store_data, prodname+'_main_only', data = {x:REFORM(dat.time), y:main_data, v:pos_bins}, $
                dlim = {panel_size:1, $
                        spec:1, $
                        x_no_interp:1, $
                        y_no_interp:1, $
                        ylog:0, $
                        zlog:1, $
                        zrange:[0.1, 1000], $
                        ytitle:ytitle1, $
                        ztitle:ztitle $
                       }

ENDIF 

store_data, prodname+'_s_chan_esa', data = {x:REFORM(dat.time), y:dat.s_chan_i}, $
            dlim = {panel_size:1, $
                    ylog:0, $
                    yrange:[0, 127], $
                    ytitle:'S-Chan ESA Step'}

IF prod EQ 'h_alpha' OR $
  prod EQ 'h+peak' OR $
  prod EQ 'he++peak' OR $
  prod EQ 'he++tcr' THEN BEGIN
    store_data, prodname+'_peak_esa', data = {x:reform(dat.time), y:dat.energy_peak_i}, $
                dlim = {panel_size:1, $
                        ylog:0, $
                        yrange:[0, 127], $
                        ytitle:'Energy Peak ESA Step'}
ENDIF 
END
