;+
; PROCEDURE: get_pla_mon_full_spec
;
; PURPOSE: to get the product related data and create the
;          corresponding  tplot variable
;
; INPUT: sat   - S/C id - 'A' or 'B'
;        prod  - product
;        class - classification
;        units - units
;
; KEYWORDS: diagn - if set extra diagnostic tplot variables are created
;
; OUTPUT: the output is provided via the tplot variables.
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION LIST:
;     06/26/2008: LBE: Added s_chan_i and check on error
;     07/23/2008: CGM: Fill value is substituted by the NaN value.
;     08/01/2008: CGM: 	modified C/S conversion factor
;		        - previously conversion to Hz gave average counts per second during accumulation cycle
;			- new conversion gives counts/sec as though detector had larger solid angle and could
;		              observe all out-of-ecliptic angles simultaneously
;     02/10/2009: LBE: Limit by timerange.
;     11/12/2009: LBE: Changed yrange. Added schan_esa (in keV/q).
;     01/29/2010: LBE: Added check for no data.
;     12/17/2014: LBE: Remove times with no schan data.
;     06/03/2015: LBE: Change z range.
;     06/04/2015: LBE: Added main and schan versions
;-
PRO get_pla_mon_full_spec, sat, prod, units, diagn = diagn
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

seek_files, sat, 'science', filesfound
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN
ENDIF

n_prod = prod+'_full'
;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))

        prod_name = 'epoch1'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            prod_name = n_prod
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 's_chan1'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan1, rec_count = epochinfo.maxrec+1, /zvariable

            prod_name = 'error1'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error1, rec_count = epochinfo.maxrec+1, /zvariable

            cdf_data = FLOAT(cdf_data)
            fillval = WHERE(cdf_data LT 0, cfillval)
            IF cfillval GT 0 THEN BEGIN
                cdf_data(fillval) = !VALUES.F_NAN
            ENDIF

            ; make versions for main and schan
            cdf_data_main     = cdf_data
            cdf_data_schan    = cdf_data
            num_esa_cdf_main  = intarr(n_elements(epoch)) ; for number of esa steps we have
            num_esa_cdf_schan = intarr(n_elements(epoch))
            bad_schan = where(finite(cdf_s_chan1) EQ 0 OR cdf_s_chan1 LT 1, count)
            IF count GT 0 THEN BEGIN 
                cdf_data_main [*, *, bad_schan] = !values.f_nan
                cdf_data_schan[*, *, bad_schan] = !values.f_nan
            ENDIF 
            noswitch = where(cdf_s_chan1 EQ 0, count)
            IF count GT 0 THEN cdf_s_chan1[noswitch] = 128 
            FOR epoch_i = 0L, n_elements(epoch)-1 DO BEGIN 
                schan = cdf_s_chan1[epoch_i]
                IF finite(schan) EQ 1 AND schan GT 0 THEN BEGIN 
                    IF schan LT 128 THEN cdf_data_main [schan:127, *, epoch_i] = !values.f_nan
                    IF schan EQ 128 THEN cdf_data_schan[        *, *, epoch_i] = !values.f_nan $
                      ELSE cdf_data_schan[  0:schan, *, epoch_i] = !values.f_nan
                    num_esa_cdf_main [epoch_i] = schan
                    num_esa_cdf_schan[epoch_i] = (128-schan)
                ENDIF 
            ENDFOR 

            ra_trig_sum_defl = TOTAL(cdf_data,       2, /NaN)
            main_sum         = TOTAL(cdf_data_main,  2, /NaN)
            schan_sum        = TOTAL(cdf_data_schan, 2, /NaN)

            IF ifwd EQ 0 THEN BEGIN
                converted_epoch = REFORM(time_double(epoch, /epoch))
                data            = TRANSPOSE(ra_trig_sum_defl)
                data_main       = TRANSPOSE(main_sum)
                data_schan      = TRANSPOSE(schan_sum)
                num_esa_main    = REFORM   (num_esa_cdf_main)
                num_esa_schan   = REFORM   (num_esa_cdf_schan)
                s_chan1         = REFORM(cdf_s_chan1)
                error1          = REFORM(cdf_error1)
            ENDIF ELSE BEGIN
                converted_epoch = [converted_epoch, REFORM(time_double(epoch, /epoch))]
                data            = [data, TRANSPOSE(ra_trig_sum_defl)]
                data_main       = [data_main,       TRANSPOSE(main_sum)]
                data_schan      = [data_schan,      TRANSPOSE(schan_sum)]
                num_esa_main    = [num_esa_main,    REFORM   (num_esa_cdf_main)]
                num_esa_schan   = [num_esa_schan,   REFORM   (num_esa_cdf_schan)]
                s_chan1         = [s_chan1,         REFORM(cdf_s_chan1)]
                error1          = [error1,          REFORM(cdf_error1)]
            ENDELSE
            
            ifwd =  ifwd + 1
            
        ENDIF
        cdf_close, cdf_id
    ENDIF
ENDFOR

; Check for no data
IF n_elements(converted_epoch) EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No data found'
    RETURN
ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error1 LT 1, cff)
IF cff GT 0 THEN BEGIN
    converted_epoch = converted_epoch(findfill)
    s_chan1         = s_chan1        (findfill)
    error1          = error1         (findfill)
    data            = data           (findfill, *)
    data_main       = data_main      (findfill, *)
    data_schan      = data_schan     (findfill, *)
ENDIF

; no schan data
findfill = WHERE(s_chan1 GE 0, cff)
IF cff GT 0 THEN BEGIN
    converted_epoch = converted_epoch(findfill)
    s_chan1         = s_chan1        (findfill)
    error1          = error1         (findfill)
    data            = data           (findfill, *)
    data_main       = data_main      (findfill, *)
    data_schan      = data_schan     (findfill, *)
ENDIF

; outside timerange
get_timespan, tt
bad_i = where(converted_epoch LT tt[0] OR converted_epoch GT tt[1], bad_count, complement = good_i, ncomplement = good_count)
IF good_count EQ 0 THEN BEGIN 
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN
ENDIF ELSE IF bad_count GT 0 THEN BEGIN
    converted_epoch = converted_epoch[good_i]
    data            = data           [good_i, *]
    data_main       = data_main      [good_i, *]
    data_schab      = data_schan     [good_i, *]
    num_esa_main    = num_esa_main   [good_i, *]
    num_esa_schan   = num_esa_schan  [good_i, *]
    s_chan1         = s_chan1        [good_i]
    error1          = error1         [good_i]
ENDIF 

;convert s_chan1 to energy
get_pla_energies, sat, esa_step
s_chan_i   = s_chan1
no_schan_i = where(s_chan_i EQ 128, count, ncomplement = yes_count, complement = yes_i)
IF yes_count GT 0 THEN s_chan1[yes_i] = esa_step(s_chan1[yes_i])
IF count GT 0 THEN BEGIN ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan1 [no_schan_i] = 0.1
ENDIF

ntime = N_ELEMENTS(converted_epoch)
varr = FLTARR(ntime, 128)

FOR iff = 0L, ntime-1 DO varr(iff, *) = esa_step
;------------------------------------------------------------------
; Convert Units
; 0.0128 seconds per defl step 
; previously conversion to Hz gave average counts per second during accumulation cycle
; new conversion gives counts/sec as though detector had larger solid angle and could
; observe all out-of-ecliptic angles simultaneously
;------------------------------------------------------------------
IF STRUPCASE(units) EQ 'C/S' THEN BEGIN
    data = data / 0.0128
ENDIF
;------------------------------------------------------------------
vname = prod+'_'+units+'_full'
main_name  = vname+'_main'
schan_name = vname+'_schan'

IF STRUPCASE(units) EQ 'C/S' THEN BEGIN
    ztitle = '!C!CC/S'
ENDIF ELSE BEGIN
    ztitle = '!C!CC/Min'
ENDELSE
ytitle = prod+'!C!CE/q (keV/q)'
store_data, vname, data = {x:converted_epoch, y:data, v:varr}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:1, $
                    zlog:1, $
                    yrange:[0.1, 10.0], $
                    zrange:[1, 1e5], $
                    ytitle:ytitle, $
                    ztitle:ztitle $
                   }

store_data, main_name, data = {x:converted_epoch, y:data_main, v:varr}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:1, $
                    zlog:1, $
                    yrange:[0.1, 10.0], $
                    zrange:[1, 1e5], $
                    ytitle:ytitle, $
                    ztitle:ztitle $
                   }

store_data, schan_name, data = {x:converted_epoch, y:data_schan, v:varr}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    ylog:1, $
                    zlog:1, $
                    yrange:[0.1, 10.0], $
                    zrange:[1, 1e5], $
                    ytitle:ytitle, $
                    ztitle:ztitle $
                   }

; This _temp one is unnecessary (but I needed one in between)
store_data, prod+'_s_chan_esa', data = {x:converted_epoch, y:s_chan1}, $
            dlim = {panel_size:1, $
                    ylog:1, $
                    yrange:[0.1, 100.0], $
                    ytitle:'S-Chan ESA (keV/q)'}

store_data, prod+'_s_chan_esa_i', data = {x:converted_epoch, y:s_chan_i}, $
            dlim = {panel_size:1, $
                    ylog:0, $
                    yrange:[0, 127], $
                    ytitle:'S-Chan ESA Step'}

END
