;+
; PROCEDURE: get_pla_mon_full
;
; PURPOSE: to get the product related data and create the
;          corresponding  tplot variable. This one keeps all three
;          dimensions [time, defl, esa].
;
; INPUT: sat   - S/C id - 'A' or 'B'
;        prod  - product
;        class - classification
;        units - units
;
; OUTPUT: the output is provided via the tplot variables.
;
; CREATED BY: L. Ellis
;
; LAST MODIFICATION:
;
; MODIFICATION LIST:
;     06/15/2015: LBE: Created from get_pla_mon_full.pro.
;-

PRO get_pla_mon_full_5min, sat, prod, units
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

seek_files, sat, 'science', filesfound
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN
ENDIF

n_prod = prod+'_full'
;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))
        prod_name = 'epoch5_mon'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            prod_name = prod+'_norm_science'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable

            cdf_data = FLOAT(cdf_data)
            fillval  = WHERE(cdf_data LT 0, cfillval)
            IF cfillval GT 0 THEN BEGIN
                cdf_data(fillval) = !VALUES.F_NAN
            ENDIF

            prod_name = 's_chan5_mon'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan5, rec_count = epochinfo.maxrec+1, /zvariable

            prod_name = 'error5_mon'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error5, rec_count = epochinfo.maxrec+1, /zvariable

            IF ifwd EQ 0 THEN BEGIN
                converted_epoch = REFORM(time_double(epoch, /epoch))
                data            = TRANSPOSE(cdf_data)
                s_chan5         = REFORM(cdf_s_chan5)
                error5          = REFORM(cdf_error5)
            ENDIF ELSE BEGIN
                converted_epoch = [converted_epoch, REFORM(time_double(epoch, /epoch))]
                data            = [data,            TRANSPOSE(cdf_data)]
                s_chan5         = [s_chan5,         REFORM(cdf_s_chan5)]
                error5          = [error5,          REFORM(cdf_error5)]
            ENDELSE
            ifwd =  ifwd + 1
        ENDIF
        cdf_close, cdf_id
    ENDIF
ENDFOR

; Check for no data
IF n_elements(converted_epoch) EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No data found'
    RETURN
ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error5 LT 1, cff)
IF cff GT 0 THEN BEGIN
    converted_epoch = converted_epoch(findfill)
    s_chan5 = s_chan5(findfill)
    error5 = error5(findfill)
    data = data(findfill, *, *)
ENDIF

; outside timerange
get_timespan, tt
bad_i = where(converted_epoch LT tt[0] OR converted_epoch GT tt[1], bad_count, complement = good_i, ncomplement = good_count)
IF good_count EQ 0 THEN BEGIN 
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN
ENDIF ELSE IF bad_count GT 0 THEN BEGIN
    converted_epoch = converted_epoch[good_i]
    data            = data           [good_i, *, *]
    s_chan5         = s_chan5        [good_i]
    error5          = error5         [good_i]
ENDIF 

;convert s_chan5 to energy
get_pla_energies, sat, esa_step
s_chan_i = s_chan5
bad_i = where(s_chan5 LT 0, bad_count, complement = good_i, ncomplement = good_count)
IF bad_count  GT 0 THEN s_chan5[bad_i] = !values.f_nan
IF good_count GT 0 THEN s_chan5[good_i] = esa_step(s_chan5[good_i])
no_schan_i = where(s_chan_i EQ 0, count)
IF count GT 0 THEN BEGIN ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan5 [no_schan_i] = 0.1
ENDIF

ntime = N_ELEMENTS(converted_epoch)
varr  = FLTARR(ntime, 128)

FOR iff = 0L, ntime-1 DO varr(iff, *) = esa_step
;------------------------------------------------------------------
; Convert Units
; 0.0128 seconds per defl step 
; previously conversion to Hz gave average counts per second during accumulation cycle
; new conversion gives counts/sec as though detector had larger solid angle and could
; observe all out-of-ecliptic angles simultaneously
;------------------------------------------------------------------
IF STRUPCASE(units) EQ 'C/S' THEN BEGIN
    data = data / (0.0128*5)
ENDIF
;------------------------------------------------------------------
vname = prod+'_'+units+'_full_5min'

IF STRUPCASE(units) EQ 'C/S' THEN BEGIN
    ztitle = '!C!CC/S'
ENDIF ELSE BEGIN
    ztitle = '!C!CCOUNTS'
ENDELSE
store_data, vname, data = {x:converted_epoch, y:data, v:varr}, $
            dlim = {panel_size:1, $
                    spec:1, $
                    x_no_interp:1, $
                    y_no_interp:1, $
                    zlog:1, $
                    zrange:[1, 1e4], $
                    ztitle:ztitle $
                   }

store_data, prod+'_s_chan_esa_5min', data = {x:converted_epoch, y:s_chan5}, $
            dlim = {panel_size:1, $
                    ylog:1, $
                    yrange:[0.1, 100.0], $
                    ytitle:'S-Chan ESA (keV/q)'}

store_data, prod+'_s_chan_esa_i_5min', data = {x:converted_epoch, y:s_chan_i}, $
            dlim = {panel_size:1, $
                    ylog:0, $
                    yrange:[0, 127], $
                    ytitle:'S-Chan ESA Step'}

END
