;+
; FUNCTION: get_pla_h1_peak
;
; PURPOSE: To create the data structure of the 'h1_peak' product
;
; INPUT: sat: S/C id - 'A' or 'B'
;
; OUTPUT: the data structure is returned [time, defl, pos, energy]
;
; KEYWORDS: full_chan: 2=main,1=schan, default=both - use to specify which channel to include in the full_array
;           from_full: set if want spectrogram created from full distribution --
;                      necessary if have included running average
;           weights  : Array of floats for running average
;                      Should add to 1. Should have odd # of elements. (ex: [.25, .5, .25])
;           no_defl  : Sum over defl before running ave., then put in "data"
;           accum_min    : Run accumulation rather than average.
;           key_eff  : Use position bin efficiencies before finding average.
;           roll_eff : Use roll efficiencies for each position bin.
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;   12-07-2007: LBE
;               Changed loop variable to long.
;   03-26-2008: LBE: Added indices for energy peak and schan
;   02-12-2008: LBE: Added indices for pos peak and defl peak.
;                    Fixed bug with position (was reading defl).
;   07-20-2009: LBE: Check for case of 1 record.
;   05-19-2010: LBE: Added full_counts (puts into full array)
;   05-20-2010: LBE: Added full_chan keyword.
;                    Added from_full keyword (for when using running average)
;                    Added weights keyword (for full_array)
;   05-24-2010: LBE: Added no_defl keyword.
;   05-28-2010: LBE: Put NaN back in after sum.
;   06-08-2010: LBE: Added accum_min keyword.
;   09-06-2011: LBE: Add index for deflection peak.
;   12-16-2011: LBE: Added key_eff keyword.
;   01-30-2011: LBE: Change algorithm for key_eff.
;   02-09-2012: LBE: Added more eff for Pos 5-7 per Toni email.
;   02-10-2012: LBE: Fix when schan = -1.
;   02-21-2012: LBE: Different eff. for pos and neg angles.
;   03-16-2012: LBE: Added roll_eff.
;   07-02-2012: LBE: Added read_h1_eff_from_pha.
;   08-30-2012: LBE: Added read_h1_eff_from_pha4 (less hand-programming).
;   09-08-2012: LBE: Changed read_h1_eff_from_pha4 so divide by efficiencies.
;   09-27-2012: LBE: Added set_to_beg keyword.
;   10-03-2012: LBE: Added read_h1_eff_from_pha5 to deal with MCP changes better.
;   10-10-2012: LBE: Fixed read_h1_eff_from_pha5 for during eff times.
;   05-08-2014: LBE: Skip MCP changes before first calibration time.
;   12-05-2017: LBE: Update STA PHA_breaks file.
;   12-04-2018: LBE: Change STA_PHA_breaks date.
;   12-12-2018: LBE: Change STA_PHA_breaks date.
;   12-19-2018: LBE: Change STA_PHA_breaks date.
;-

PRO read_h1_eff_from_pha5, converted_epoch, full_counts, sat
compile_opt strictarrsubs

; read tables
eff_line = ''
bre_line = ''
eff_done = 0
bre_done = 0
IF sat EQ 'A' THEN BEGIN 
    openr, eff_lun, '~/splat/Tables/STA_PHA_ra_eff_20141107.txt', /get_lun 
    openr, bre_lun, '~/splat/Tables/STA_PHA_breaks_20181219.txt', /get_lun 
ENDIF ELSE BEGIN 
    openr, eff_lun, '~/splat/Tables/STB_PHA_ra_eff_20141014.txt', /get_lun
    openr, bre_lun, '~/splat/Tables/STB_PHA_breaks_20140630.txt', /get_lun 
ENDELSE 
readf, eff_lun, eff_line ; header
FOR ii = 0, 5 DO BEGIN 
    readf, bre_lun, bre_line ; header, and before first calibration time
ENDFOR 
temp_num_times = 0L
bre_index      = 0L
eff_index      = 0L
;eff            = fltarr(200, 32) ; 100 is arbitrary big number
eff            = fltarr(200, 8) ; 100 is arbitrary big number
bre_times      = dblarr(200)
start_times    = dblarr(200)
stop_times     = dblarr(200)
WHILE eof(eff_lun) EQ 0 DO BEGIN 
    readf, eff_lun, eff_line
    eff_parts = strsplit(eff_line, /extract)
    start_times[eff_index] = time_double(eff_parts[0])
    stop_times [eff_index] = time_double(eff_parts[1])
    FOR ii = 0, 7 DO BEGIN 
        IF ii EQ 0 THEN eff[eff_index, ii] = mean(float(eff_parts[2+16+1:2+16+3]), /nan) ELSE $
          eff[eff_index, ii] = mean(float(eff_parts[2+(ii*4)+16:2+(ii*4)+16+3]), /nan)
    ENDFOR 
    eff_index = eff_index + 1
ENDWHILE 
WHILE eof(bre_lun) EQ 0 DO BEGIN 
    readf, bre_lun, bre_line
    bre_parts = strsplit(bre_line, /extract)
    bre_times[bre_index] = time_double(bre_parts[0])
    bre_index = bre_index + 1
ENDWHILE 
close, eff_lun
close, bre_lun
free_lun, eff_lun
free_lun, bre_lun
eff         = eff        [0:eff_index-1, *]
bre_times   = bre_times  [0:bre_index-1   ]
start_times = start_times[0:eff_index-1   ]
stop_times  = stop_times [0:eff_index-1   ]

; artificially bridge gap caused by MCP changes
; use 10/10/2007 values through gap
IF sat EQ 'A' THEN BEGIN fill_i = where(start_times[*] GE time_double('2007-09-24') AND start_times[*] LT time_double('2007-10-10'), count)
    good_i = where(start_times[*] EQ time_double('2007-10-10'))
    FOR ii = 0, count-1 DO BEGIN 
        eff[fill_i[ii], *] = eff[good_i, *]
    ENDFOR 
ENDIF 

; running average -- over 3 points, but break at MCP changes
temp_num_times = n_elements(start_times)
smooth_eff = fltarr(temp_num_times, 8)
FOR jj = 0, 7 DO BEGIN 
    smooth_eff[0, jj] = mean(eff[0:1, jj], /nan)
ENDFOR 
FOR ii = 0, temp_num_times-1 DO BEGIN 
    IF ii EQ 0 THEN start_i = 0 ELSE BEGIN  ; find start of ave
        prev_bre_times = where(bre_times LT start_times[ii], count)
        IF count EQ 0 THEN start_i = ii-1 ELSE $
          IF bre_times[prev_bre_times[count-1]] LT start_times[ii-1] THEN start_i = ii-1 ELSE $
          start_i = ii
    ENDELSE 
    IF ii EQ temp_num_times-1 THEN stop_i = ii ELSE BEGIN ; find end of ave
        post_bre_times = where(bre_times GT stop_times[ii], count)
        IF count EQ 0 THEN stop_i = ii+1 ELSE $
          IF bre_times[post_bre_times[0]] GT stop_times[ii+1] THEN stop_i = ii+1 ELSE $
          stop_i = ii
    ENDELSE 
    FOR jj = 0, 7 DO BEGIN 
        smooth_eff[ii, jj] = mean(eff[start_i:stop_i, jj], /nan)
    ENDFOR
ENDFOR 
;smooth_eff = eff ; temporary code 05/24/2013 (for test run of calibration times)

temp_eff = fltarr(n_elements(converted_epoch), 8)
; find efficiencies
; before first
wanted_i = where(converted_epoch LT start_times[0], count)
IF count GT 0 THEN BEGIN 
    FOR jj = 0, 7 DO BEGIN 
        temp_eff[wanted_i, 0] = smooth_eff[0, 0]
        temp_eff[wanted_i, 1] = smooth_eff[0, 1]
        temp_eff[wanted_i, 2] = smooth_eff[0, 2]
        temp_eff[wanted_i, 3] = smooth_eff[0, 3]
        temp_eff[wanted_i, 4] = smooth_eff[0, 4]
        temp_eff[wanted_i, 5] = smooth_eff[0, 5]
        temp_eff[wanted_i, 6] = smooth_eff[0, 6]
        temp_eff[wanted_i, 7] = smooth_eff[0, 7]
    ENDFOR 
ENDIF 
; during times
FOR ii = 0, temp_num_times-1 DO BEGIN 
    wanted_i = where((converted_epoch GE start_times[ii]) AND (converted_epoch LE stop_times[ii]), count)
    IF count GT 0 THEN BEGIN 
        FOR jj = 0, 7 DO BEGIN 
            temp_eff[wanted_i, 0] = smooth_eff[ii, 0]
            temp_eff[wanted_i, 1] = smooth_eff[ii, 1]
            temp_eff[wanted_i, 2] = smooth_eff[ii, 2]
            temp_eff[wanted_i, 3] = smooth_eff[ii, 3]
            temp_eff[wanted_i, 4] = smooth_eff[ii, 4]
            temp_eff[wanted_i, 5] = smooth_eff[ii, 5]
            temp_eff[wanted_i, 6] = smooth_eff[ii, 6]
            temp_eff[wanted_i, 7] = smooth_eff[ii, 7]
        ENDFOR 
    ENDIF 
ENDFOR 
; between times
FOR ii = 1, temp_num_times-1 DO BEGIN 
    num_sec = (start_times[ii] - stop_times[ii-1])
    wanted_i = where((converted_epoch GT stop_times[ii-1]) AND (converted_epoch LT start_times[ii]), count)
    IF count GT 0 THEN BEGIN 
        FOR jj = wanted_i[0], wanted_i[count-1] DO BEGIN ; assumes contiguous times!
            FOR eff_i = 0, 7 DO BEGIN 
                temp_eff2 = interpolate([smooth_eff[ii-1, eff_i], smooth_eff[ii, eff_i]], $
                                       (converted_epoch[jj]-stop_times[ii-1])/num_sec)
                temp_eff[jj, eff_i] = temp_eff2
            ENDFOR 
        ENDFOR 
    ENDIF 
ENDFOR 
; around breaks (MCP changes, etc) -- NB: might overwrite last bit
FOR ii = 0, n_elements(bre_times)-1 DO BEGIN 
    pre_bre_i = where(stop_times LT bre_times[ii], pre_count)
    IF pre_count EQ 0 THEN stop ELSE BEGIN 
        wanted_i = where((converted_epoch GE stop_times[pre_bre_i[pre_count-1]]) AND (converted_epoch LE bre_times[ii]), count)
        IF count GT 0 THEN BEGIN 
            FOR jj = 0, 7 DO BEGIN 
                temp_eff[wanted_i, 0] = smooth_eff[pre_bre_i[pre_count-1], 0]
                temp_eff[wanted_i, 1] = smooth_eff[pre_bre_i[pre_count-1], 1]
                temp_eff[wanted_i, 2] = smooth_eff[pre_bre_i[pre_count-1], 2]
                temp_eff[wanted_i, 3] = smooth_eff[pre_bre_i[pre_count-1], 3]
                temp_eff[wanted_i, 4] = smooth_eff[pre_bre_i[pre_count-1], 4]
                temp_eff[wanted_i, 5] = smooth_eff[pre_bre_i[pre_count-1], 5]
                temp_eff[wanted_i, 6] = smooth_eff[pre_bre_i[pre_count-1], 6]
                temp_eff[wanted_i, 7] = smooth_eff[pre_bre_i[pre_count-1], 7]
            ENDFOR 
        ENDIF 
    ENDELSE 
    post_bre_i = where(start_times GT bre_times[ii], post_count)
    IF post_count GT 0 THEN BEGIN 
        wanted_i = where((converted_epoch LE start_times[post_bre_i[0]]) AND (converted_epoch GE bre_times[ii]), count)
        IF count GT 0 THEN BEGIN 
            FOR jj = 0, 7 DO BEGIN 
                temp_eff[wanted_i, 0] = smooth_eff[post_bre_i[0], 0]
                temp_eff[wanted_i, 1] = smooth_eff[post_bre_i[0], 1]
                temp_eff[wanted_i, 2] = smooth_eff[post_bre_i[0], 2]
                temp_eff[wanted_i, 3] = smooth_eff[post_bre_i[0], 3]
                temp_eff[wanted_i, 4] = smooth_eff[post_bre_i[0], 4]
                temp_eff[wanted_i, 5] = smooth_eff[post_bre_i[0], 5]
                temp_eff[wanted_i, 6] = smooth_eff[post_bre_i[0], 6]
                temp_eff[wanted_i, 7] = smooth_eff[post_bre_i[0], 7]
            ENDFOR 
        ENDIF 
    ENDIF 
ENDFOR 
; after last
wanted_i = where(converted_epoch GT stop_times[temp_num_times-1], count)
IF count GT 0 THEN BEGIN 
    FOR jj = 0, 7 DO BEGIN 
        temp_eff[wanted_i, 0] = smooth_eff[temp_num_times-1, 0]
        temp_eff[wanted_i, 1] = smooth_eff[temp_num_times-1, 1]
        temp_eff[wanted_i, 2] = smooth_eff[temp_num_times-1, 2]
        temp_eff[wanted_i, 3] = smooth_eff[temp_num_times-1, 3]
        temp_eff[wanted_i, 4] = smooth_eff[temp_num_times-1, 4]
        temp_eff[wanted_i, 5] = smooth_eff[temp_num_times-1, 5]
        temp_eff[wanted_i, 6] = smooth_eff[temp_num_times-1, 6]
        temp_eff[wanted_i, 7] = smooth_eff[temp_num_times-1, 7]
    ENDFOR 
ENDIF

; now divide
FOR ii = 0, n_elements(converted_epoch)-1 DO BEGIN 
    FOR jj = 0, 7 DO BEGIN 
        full_counts[ii, *, jj, *] = full_counts[ii, *, jj, *] / temp_eff[ii, jj]
    ENDFOR     
ENDFOR 
END 

FUNCTION get_pla_h1_peak, sat, full_chan = full_chan, weights = weights, from_full = from_full, no_defl = no_defl, accum_min = accum_min, $
  key_eff = key_eff, roll_eff = roll_eff, set_to_beg = set_to_beg
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose
get_err_no = 0 & get_err_msg = ''

prod        = 'h+peak'
IF keyword_set(from_full) THEN nenergy     = 128 ELSE nenergy     =  10
IF keyword_set(from_full) THEN ndeflection =  32 ELSE ndeflection =  8
nposition   =  8

seek_files, sat, 'science', filesfound ; find files that correspond to time interval
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN, 0
ENDIF

;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0 
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))

        prod_name = 'epoch1'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'h+peak'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'proton_peak'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_proton_peak, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 's_chan1'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan1, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'error1'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error1, rec_count = epochinfo.maxrec+1, /zvariable

            IF ifwd EQ 0 THEN BEGIN
                converted_epoch = REFORM(time_double(epoch, /epoch))
                data            = TRANSPOSE(cdf_data)
                energy_peak     = REFORM(cdf_proton_peak(0, *))
                position_peak   = REFORM(cdf_proton_peak(1, *))
                deflection_peak = REFORM(cdf_proton_peak(2, *))
                source_array    = REFORM(cdf_proton_peak(3, *))
                s_chan1         = REFORM(cdf_s_chan1)
                error1          = REFORM(cdf_error1)
            ENDIF ELSE BEGIN
                converted_epoch = [converted_epoch, REFORM(time_double(epoch, /epoch))]
                cdf_data_tr = transpose(cdf_data)
                IF epochinfo.maxrec EQ 0 THEN BEGIN
                    temp_data = lonarr(1, 8, 8, 10)
                    temp_data[0, *, *, *] = cdf_data_tr
                    cdf_data_tr = temp_data
                ENDIF 
                data            = [data,            cdf_data_tr]
                energy_peak     = [energy_peak,     REFORM(cdf_proton_peak(0, *))]
                position_peak   = [position_peak,   REFORM(cdf_proton_peak(1, *))]
                deflection_peak = [deflection_peak, REFORM(cdf_proton_peak(2, *))]
                source_array    = [source_array,    REFORM(cdf_proton_peak(3, *))]
                s_chan1         = [s_chan1,         REFORM(cdf_s_chan1)]
                error1          = [error1,          REFORM(cdf_error1)]
            ENDELSE
            ifwd =  ifwd + 1
        ENDIF
        cdf_close, cdf_id
    ENDIF
ENDFOR

IF ifwd EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0
ENDIF

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error1 EQ 0, cff)
IF cff GT 0 THEN BEGIN
    converted_epoch = converted_epoch(findfill)
    energy_peak     = energy_peak    (findfill)
    position_peak   = position_peak  (findfill)
    deflection_peak = deflection_peak(findfill)
    source_array    = source_array   (findfill)
    s_chan1         = s_chan1        (findfill)
    error1          = error1         (findfill)
    data            = data           (findfill, *, *, *)
ENDIF ELSE BEGIN 
    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0
ENDELSE 

;------------------------------------------------------------------
; Check individual arrays for value range validity
;------------------------------------------------------------------

icepoch =  WHERE(converted_epoch LT (time_double('1958-01-01')) OR $
                 converted_epoch GT (time_double('2050-01-01')), cicepoch)
IF cicepoch GT 0 THEN STOP

data =  FLOAT(data)
icdata = WHERE(data EQ -1, cicdata)
IF cicdata GT 0 THEN BEGIN
    data(icdata) =  !VALUES.F_NAN
ENDIF

icenp = WHERE(energy_peak LT 0 OR energy_peak GT 127, cicenp)
;IF cicenp GT 0 THEN STOP

icpop = WHERE(position_peak LT 0 OR position_peak GT 31, cicpop)
;IF cicpop GT 0 THEN STOP

icdep = WHERE(deflection_peak LT 0 OR deflection_peak GT 31, cicdep)
;IF cicdep GT 0 THEN STOP

icdep = WHERE(deflection_peak LT 0 OR deflection_peak GT 31, cicdep)
;IF cicdep GT 0 THEN STOP

isar =  WHERE(source_array NE 0 AND source_array NE 1, cisar)
;IF cisar GT 0 THEN STOP

isch =  WHERE(s_chan1 LT 0 OR s_chan1 GT 127,  cisch)
;IF cisch GT 0 THEN STOP

ierr =  WHERE(error1 NE 0 AND error1 NE 1,  cierr)
;IF cierr GT 0 THEN STOP

;------------------------------------------------------------------
;convert energy/deflection/position/s_channel steps to actual values
;------------------------------------------------------------------
get_pla_energies, sat, esa_step             ; get energy bin values range
get_pla_theta_angles, sat, prod, theta_step ; get polar (deflection) angles range
get_pla_phi_angles, sat, prod, phi_step     ; get azimuthal (position) angles range
theta_bins = indgen(32)
phi_bins   = indgen(8)

;convert s_chan1 to energy
s_chan_i = s_chan1
bad_i = where(s_chan1 LT 0, bad_count, complement = good_i, ncomplement = good_count)
IF bad_count GT 0 THEN BEGIN ; take care of -1
    s_chan1[bad_i] = !values.f_nan
ENDIF 
IF good_count GT 0 THEN BEGIN 
    s_chan1[good_i] = esa_step(s_chan1[good_i])
ENDIF 
no_schan_i = where(s_chan_i EQ 0, count)
IF count GT 0 THEN BEGIN ; take care of when schan didn't switch
    s_chan_i[no_schan_i] = 128
    s_chan1 [no_schan_i] = 0.1
ENDIF 

position     = FLTARR(N_ELEMENTS(converted_epoch), nposition)
position_i   = FLTARR(N_ELEMENTS(converted_epoch), nposition)
deflection   = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
deflection_i = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
energy       = FLTARR(N_ELEMENTS(converted_epoch), nenergy)

IF n_elements(converted_epoch) LE 20000 THEN BEGIN  ; limit of memory -- check when have new computer
    full_counts = fltarr(N_elements(converted_epoch), 32, 8, 128) ; [time, defl, summed pos, esa]
    full_counts[*] = !values.f_nan
    have_full_counts = 1
ENDIF ELSE have_full_counts = 0

FOR ii = 0L, N_ELEMENTS(converted_epoch)-1 DO BEGIN

    ;-----------------------------------------------------------------
    ; If the energy peak is close to the boundaries the 10 energy
    ; steps start from 0 or 127. This results in the peak not being
    ; in the middle.
    ;-----------------------------------------------------------------
    IF keyword_set(from_full) THEN energy[ii, *] = esa_step ELSE BEGIN 
        IF energy_peak(ii) LT 5 THEN BEGIN
            energy(ii, *) = esa_step(0:9)
        ENDIF ELSE BEGIN
            IF energy_peak(ii) GT 123 THEN BEGIN
                energy(ii, *) = esa_step(118:127)
            ENDIF ELSE BEGIN
                energy(ii, *) = esa_step((energy_peak(ii)-5):(energy_peak(ii)+4))
            ENDELSE
        ENDELSE
    ENDELSE 

    ;-----------------------------------------------------------------
    ; If the deflection peak is close to the boundaries the 
    ; 8 deflection steps start from 0 or 31. This results in the 
    ; peak not being in the middle.
    ;-----------------------------------------------------------------
    IF keyword_set(from_full) THEN deflection[ii, *] = theta_step ELSE BEGIN 
        IF deflection_peak(ii) GT 27 THEN BEGIN
            deflection  (ii, *) = theta_step(24:31)
            deflection_i(ii, *) = theta_bins(24:31)
        ENDIF ELSE BEGIN
            IF deflection_peak(ii) LT 3 THEN BEGIN
                deflection  (ii, *) = theta_step(0:7)
                deflection_i(ii, *) = theta_bins(0:7)
            ENDIF ELSE BEGIN
                deflection  (ii, *) = theta_step((deflection_peak(ii)-3):(deflection_peak(ii)+4))
                deflection_i(ii, *) = theta_bins((deflection_peak(ii)-3):(deflection_peak(ii)+4))
            ENDELSE
        ENDELSE
    ENDELSE 

    ;-----------------------------------------------------------------
    ; The position peak is not used. Instead all positions are bined
    ; in 8 bins.
    ;-----------------------------------------------------------------
    position  (ii, *) = phi_step
    position_i(ii, *) = phi_bins

    ; fill full_counts
    IF have_full_counts EQ 1 THEN BEGIN 
        IF energy_peak    [ii] LT 5 THEN ep = 5 ELSE IF energy_peak    [ii] GT 123 THEN ep = 123 ELSE ep = energy_peak    [ii]
        IF deflection_peak[ii] LT 3 THEN dp = 3 ELSE IF deflection_peak[ii] GT  27 THEN dp =  27 ELSE dp = deflection_peak[ii]
        full_counts[ii, dp-3:dp+4, *, ep-5:ep+4] = data[ii, *, *, *]

        ; create one-channel data if requested
        IF keyword_set(full_chan) EQ 1 THEN BEGIN
            temp_data = reform(full_counts[ii, *, *, *])
            IF s_chan_i[ii] EQ -1 THEN BEGIN ; don't know chan
                temp_data[*, *, *] = !values.f_nan
            ENDIF ELSE IF full_chan EQ 2 THEN BEGIN ; want main only for full counts
                temp_data[*, *, s_chan_i[ii]:127] = !values.f_nan
            ENDIF ELSE IF full_chan EQ 1 THEN BEGIN ; want schan only for full counts
                IF s_chan_i[ii] LT 128 THEN temp_data[*, *, 0:s_chan_i[ii]-1] = !values.f_nan
            ENDIF ELSE BEGIN
                print, 'invalid entry for full_chan'
                stop
            ENDELSE 
            full_counts[ii, *, *, *] = temp_data
        ENDIF 
    ENDIF 
ENDFOR


; Position efficiencies 
IF keyword_set(key_eff) THEN BEGIN 
    read_h1_eff_from_pha5, converted_epoch, full_counts, sat
ENDIF 
IF keyword_set(weights) THEN BEGIN
    IF have_full_counts EQ 1 THEN BEGIN
        IF keyword_set(no_defl) THEN BEGIN
            data = total(full_counts, 2, /nan) ; sum over defl
            temp_data = data ; for debugging
            size = size(full_counts)
            FOR ii = 0L, size[1]-1 DO BEGIN ; need to remove /nan after sum
                FOR jj = 0, 7 DO BEGIN ; pos
                    FOR kk = 0, 127 DO BEGIN ; esa
                        IF array_equal(finite(full_counts[ii, *, jj, kk]), 0) EQ 1 THEN $
                          data[ii, jj, kk] = !values.f_nan
                    ENDFOR
                ENDFOR 
            ENDFOR 
            run_ave, data,        n_elements(weights), weights, accum_min = accum_min
        ENDIF ELSE BEGIN 
            run_ave, full_counts, n_elements(weights), weights, accum_min = accum_min
        ENDELSE 
    ENDIF ELSE BEGIN
        print, 'not enough memory for running average'
        stop
    ENDELSE 
ENDIF 

;------------------------------------------------------------------
; Time tag of the middle of the bin
;------------------------------------------------------------------
IF keyword_set(set_to_beg) EQ 0 THEN $
  converted_epoch = converted_epoch + 30. ; time shift by half minute

;------------------------------------------------------------------
; Integration time
;------------------------------------------------------------------
packets = N_ELEMENTS(converted_epoch)
integ_t = FLTARR(packets, ndeflection, nposition, nenergy) + 12.8e-3 ; ms

;------------------------------------------------------------------
; Construct the data structure
;------------------------------------------------------------------
IF have_full_counts EQ 1 THEN $
retdata = {$
          project:           'STEREO/PLASTIC', $
          prod_name:         prod, $
          units_name:        'Counts', $
          ntime:             N_ELEMENTS(converted_epoch), $
          time:              converted_epoch, $
          integ_t:           integ_t, $
          data:              data, $
          nenergy:           nenergy, $
          energy:            energy, $
          energy_peak_i:     energy_peak, $
          nposition:         nposition, $
          position:          position, $
          position_i:        position_i, $
          position_peak_i:   position_peak, $
          ndeflection:       ndeflection, $
          deflection:        deflection, $
          deflection_i:      deflection_i, $
          deflection_peak_i: deflection_peak, $
          source:            source_array, $
          s_chan:            s_chan1, $
          s_chan_i:          s_chan_i, $
          error:             error1, $
          full_array:        full_counts $
          } $
ELSE $
retdata = {$
          project:           'STEREO/PLASTIC', $
          prod_name:         prod, $
          units_name:        'Counts', $
          ntime:             N_ELEMENTS(converted_epoch), $
          time:              converted_epoch, $
          integ_t:           integ_t, $
          data:              data, $
          nenergy:           nenergy, $
          energy:            energy, $
          energy_peak_i:     energy_peak, $
          nposition:         nposition, $
          position:          position, $
          position_i:        position_i, $
          position_peak_i:   position_peak, $
          ndeflection:       ndeflection, $
          deflection:        deflection, $
          deflection_i:      deflection_i, $
          deflection_peak_i: deflection_peak, $
          source:            source_array, $
          s_chan:            s_chan1, $
          s_chan_i:          s_chan_i, $
          error:             error1 $
          }
;stop
RETURN, retdata

END
