;+
;PROCEDURE: calc_proton_norm_science
;PURPOSE:
;  Calculate proton bulk parameters from the 5 minute resolution 
;  SF0 or RA_Trigger normal science rate.
;
;
;CREATED BY: K. Simunac
;
;LAST MODIFICATION: March 2013
;
;MODIFICATION HISTORY:
;	January 2011:  created
;	11/30/2011: set chi^2 to NaN if error flag is
;			equal to 3 (data overflow)
;
;	2012: Update efficiency table for STA RA_TRIG and SFRO
;       04/11/2014: LBE : Add tplot variables.
;       04/14/2014: LBE : Change doubles to floats.
;-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;							  ;
; Below is a routine to calculate proton density, 	  ;
; bulk speed, temperature and thermal speed using normal  ;
; resolution rate data from RA_Trigger or SF0.		  ;
;	 						  ;					 	  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO calc_proton_norm_science, sat, files, which_epoch, time, num_days, table_location, which_rate

COMMON share1_norm
COMMON com_norm

tab = string(9B)
tplot_options, 'color', 0

num_records5 = final_count_5 ; final count of number of records from pl_read_cdf_mon
num_records1 = 5.0*num_records5+4.0;LONG(n_elements(schan))
have_data = 1

; Define some variables for storing error flags and s-channel switch information
    s_chan = intarr(num_records5)
    s_chan(*) = 0
    s_chan_full_res = intarr(num_records1)
    s_chan_full_res(*) = 0
    error5 = intarr(num_records5)
    rate_name = ' '
    std_error = intarr(num_records5) ; A matrix in which to store possible errors
    std_error(*) = 0
    caution_flag = intarr(num_records5) 
	; caution_flag is a matrix to store a warning if background subtraction results are substantially different
    caution_flag(*) = 0

;; Geometric Factors for Main Channel and S Channel (full SWS)
;; G = Active Area*Angular Extent of Slit*Bandwidth/2
;; For example, for the main channel of FM1:
;; 	Active Area = 8.9E-5 m^2
;;	Deflection FWHM = 1.9 degrees = 0.033 radians
;;	Azimuth Acceptance = 45 degrees = 0.785 radians
;;	Energy Bandwidth = 6.12%
;;	G = (8.9E-5)*(0.033)*(0.785)*0.0612/2 = 7.1E-8 m^2 sr eV/eV

A_G_Main = 7.1E-8  ; geometric factor for PLASTIC/A main channel
A_G_Sch = 2.3E-11  ; geometric factor for PLASTIC/A small channel
B_G_Main = 6.0E-8  ; geometric factor for PLASTIC/B main channel
B_G_Sch = 1.7E-11  ; geometric factor for PLASTIC/B small channel

;; Dead times for SF0 and RA_TRIG

A_SF0_dead_time = 5.0E-6 ;sec
A_RA_TRIG_dead_time = 2.0E-6 ;sec
B_SF0_dead_time = 5.0E-6	;sec
B_RA_TRIG_dead_time = 2.0E-6	;sec ; CHANGED ON 02 APRIL, 2007 FROM 2.5E-6


;; Read in tables of estimated efficiencies
;; The location of the efficiency tables must be specified in the crib file.

eff = fltarr(32)
eff(*) = !values.f_nan ; change to !values.f_nan once tables have been created

IF which_rate EQ 'SFR0' THEN BEGIN
	IF sat EQ 'A' THEN eff_file = table_location+'leakage_A_SFR0_2012_11_07_V2.txt'
ENDIF ELSE BEGIN ; Use default RA_Trig table
; For RA_Trig use this table
IF sat EQ 'A' THEN eff_file = table_location+'leakage_A_2012_07_25.txt'; 2011_03_04
IF sat EQ 'B' THEN eff_file = table_location+'leakage_B_2011_10_03.txt'
ENDELSE
 
openr, in_lun_1, eff_file, /GET_LUN
line = ","
readf, in_lun_1, line ; read the header line
FOR i = 0, 31 DO BEGIN
	readf, in_lun_1, line
	parts = STRSPLIT(line, ",", /EXTRACT)
	eff[i]=parts[1]
ENDFOR
close, in_lun_1
free_lun, in_lun_1

;print, 'Efficiency tables have been read.'
;---------------------------------------------------------

; Get E/Q and V associated with each ESA step so that distribution functions can be found
; The location of the ESA table must be specified in the crib file.

IF sat EQ 'A' THEN ESA_file_main = table_location+'ESA_table_main_A_2012_08_10.txt' 
IF sat EQ 'B' THEN ESA_file_main = table_location+'ESA_table_main_B_2011_10_03.txt'
;IF sat EQ 'A' THEN ESA_file_S = table_location+'ESA_table_A_S_2011_03_04.txt'  ; KDCS
IF sat EQ 'A' THEN ESA_file_S = table_location+'ESA_table_A_S_Toni_v3_modified.txt' ; Toni's Table modified for lack of supression
;IF sat EQ 'A' THEN ESA_file_S = table_location+'ESA_table_A_S_Toni_v3.txt' ; Toni's Table ignoring lack of supression
IF sat EQ 'B' THEN ESA_file_S = table_location+'ESA_table_B_S_2011_03_04.txt'

energy_main = fltarr(32) ; changed from double to float (August 2009)
energy_main(*) = !values.f_nan
v_main = fltarr(32) ; changed from double to float (August 2009)
v_main(*) = !values.f_nan
energy_S = fltarr(32) ; changed from double to float (August 2009)
energy_S(*) = !values.f_nan
v_S = fltarr(32) ; changed from double to float (August 2009)
v_S(*) = !values.f_nan

; Read in main channel ESA table
openr, in_lun, ESA_file_main, /GET_LUN
line = " "
readf, in_lun, line   ; read past first lines
FOR i = 0, 31 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    energy_main[i] = parts[1]			;keV/q
    v_main[i] = parts[2]
ENDFOR

close, in_lun
free_lun, in_lun

; Read in S channel ESA table
openr, in_lun, ESA_file_S, /GET_LUN
line = " "
readf, in_lun, line   ; read past first lines
FOR i = 0, 31 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    energy_S[i] = parts[1]			;keV/q
    v_S[i] = parts[2]
ENDFOR

close, in_lun
free_lun, in_lun

;stop

print, 'Finished reading ESA tables.'


IF sat EQ 'A' THEN BEGIN
	get_schan_esa1, 'A'
        get_data, 'A_schan1_esa', schan_t, schan
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = FIX(s_chan5_a/4)
    	s_chan_full_res = FIX(schan/4)
        error5 = error5_a
	
    IF which_rate EQ 'RA_TRIG' THEN BEGIN
      ;; Check that there is actually data in the files.  If not, then stop.
       IF TOTAL(ra_trig_a_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No RA_TRIG data for this time.'
          STOP
       ENDIF
        full_data = ra_trig_a_sci
        ra_trig_a_sci = 0
        rate_name = '_RA_Trig'
        dead_time = A_RA_trig_dead_time
    ENDIF


    IF which_rate EQ 'SF0' THEN BEGIN
       IF TOTAL(sf0_a_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No SF0 data for this time.'
          STOP
       ENDIF
        full_data = sf0_a_sci
        sf0_a_sci = 0
        rate_name = '_SF0'
        dead_time = A_SF0_dead_time
    ENDIF

    IF which_rate EQ 'SFR0' THEN BEGIN
       IF TOTAL(sfr0_a_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No SFR0 data for this time.'
          STOP
       ENDIF
 	full_data = sfr0_a_sci
        sfr0_a_sci = 0
        rate_name = '_SFR0'
        dead_time = 5.0E-5 ; Estimated value
    ENDIF

   IF which_rate EQ 'S_VALID' THEN BEGIN 
       IF TOTAL(s_valid_a_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No S_VALID data for this time.'
          STOP
       ENDIF
        full_data = s_valid_a_sci
        s_valid_a_sci = 0
        rate_name = '_S_VALID'
        dead_time = 5.0E-5 ; Placeholder value
   ENDIF

   IF which_rate EQ 'STP0' THEN BEGIN
       IF TOTAL(stp0_a_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No STP0 data for this time.'
          STOP
       ENDIF
      	full_data = stp0_a_sci
        stp0_a_sci = 0
        rate_name = '_STP0'
        dead_time = 5.0E-5 ; Placeholder value
   ENDIF


   IF which_rate EQ 'SSD_SW' THEN BEGIN
       IF TOTAL(ssd_sw_a_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No SSD_SW data for this time.'
       ENDIF
      	full_data = ssd_sw_a_sci
        ssd_sw_a_sci = 0
        rate_name = '_SSD_SW'
        dead_time = 5.0E-5 ; Placeholder value
   ENDIF

ENDIF ELSE BEGIN ; data is from spacecraft B
	get_schan_esa1, 'B'
        get_data, 'B_schan1_esa', schan_t, schan
        G_Main = B_G_Main
        G_Sch = A_G_Sch
        s_chan = FIX(s_chan5_b/4)
    	s_chan_full_res = FIX(schan/4)
        error5 = error5_b
	
    IF which_rate EQ 'RA_TRIG' THEN BEGIN
    ;; Check that there is actually data in the files.  If not, then stop.
       IF TOTAL(ra_trig_b_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No RA_TRIG data for this time.'
          have_data = 0
       ENDIF
        full_data = ra_trig_b_sci
        ra_trig_b_sci = 0	
        rate_name = '_RA_Trig'
        dead_time = B_RA_trig_dead_time  
    ENDIF

    IF which_rate EQ 'SF0' THEN BEGIN
       IF TOTAL(sf0_b_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No SF0 data for this time.'
          have_data = 0
       ENDIF
        full_data = sf0_b_sci
        sf0_b_sci = 0
        rate_name = '_SF0'
        dead_time = B_SF0_dead_time
    ENDIF

    IF which_rate EQ 'SFR0' THEN BEGIN
       IF TOTAL(sfr0_b_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No SFR0 data for this time.'
          STOP
       ENDIF
 	full_data = sfr0_b_sci
        sfr0_b_sci = 0
        rate_name = '_SFR0'
        dead_time = 5.0E-5 ; Placeholder value
    ENDIF

   IF which_rate EQ 'S_VALID' THEN BEGIN 
       IF TOTAL(s_valid_b_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No S_VALID data for this time.'
          STOP
       ENDIF
        full_data = s_valid_b_sci
        s_valid_b_sci = 0
        rate_name = '_S_VALID'
        dead_time = 5.0E-5 ; Placeholder value
   ENDIF

   IF which_rate EQ 'STP0' THEN BEGIN
       IF TOTAL(stp0_b_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No STP0 data for this time.'
          STOP
       ENDIF
      	full_data = stp0_b_sci
        stp0_b_sci = 0
        rate_name = '_STP0'
        dead_time = 5.0E-5 ; Placeholder value
   ENDIF

   IF which_rate EQ 'SSD_SW' THEN BEGIN
       IF TOTAL(ssd_sw_b_sci, /nan) LE 0 THEN BEGIN
          PRINT, 'No SSD_SW data for this time.'
          STOP
       ENDIF
      	full_data = ssd_sw_b_sci
        ssd_sw_b_sci = 0
        rate_name = '_SSD_SW'
        dead_time = 5.0E-5 ; Placeholder value
   ENDIF

ENDELSE

;; Get the time associated with the first file to label output files
start_time=' '
start_time=strmid(time,0,10)


print, 'Starting calculations.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove the fill value of -1 from the files.
remove_fill_norm, -1, full_data
print, 'Removed fill values.'

;; Check for value indicating overflow.  If present, mark record with error 3.

FOR m=0L, num_records5 - 1 DO BEGIN
    FOR k = 0, 31 DO BEGIN ; ESA steps
        FOR l = 0, 7 DO BEGIN; DEFL steps
            IF full_data(k, l, m) GE 507904 THEN BEGIN
                full_data(k, l, m) = !values.f_nan
                std_error(m) = 3
		overflow_time = time_struct(epoch5(m))
		overflow_year = STRING(overflow_time.year, FORMAT = '(I04)')
		overflow_month = STRING(overflow_time.month, FORMAT = '(I02)')
		overflow_day = STRING(overflow_time.date, FORMAT = '(I02)')
		overflow_hour = STRING(overflow_time.hour, FORMAT = '(I02)')
		overflow_min = STRING(overflow_time.min, FORMAT = '(I02)')
		overflow_ESA = STRING(k)
		overflow_Defl = STRING(l)
		temp_string = ' '
		temp_string = overflow_year+'-'+overflow_month+'-'+overflow_day+'/'+overflow_hour+':'+overflow_min+' '+$
			overflow_ESA+' '+overflow_Defl
                PRINT, 'Overflow value detected.  Time, ESA, Defl:'
		PRINT, temp_string
		;STOP
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
;print,'Maximum Counts:',max(full_data,/NAN)
;IF max(full_data) GT 32000 THEN STOP


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Un-comment this section to make a contour plot of counts versus ESA and Defl Steps
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;FOR k = 0, num_records5-1 DO BEGIN
;
;time = time_struct(epoch5(k))
;YEAR = STRING(time.year, FORMAT = '(I04)')
;DOY = STRING(time.doy, FORMAT = '(I03)')
;MONTH = STRING(time.month, FORMAT = '(I02)')
;DATE = STRING(time.date, FORMAT = '(I02)')
;HR = STRING(time.hour, FORMAT = '(I02)')
;MIN = STRING(time.min, FORMAT = '(I02)')
;
;IF time.year EQ 2013 AND time.month EQ 11 AND (time.date GE 8 AND time.date LE 9) AND ((k MOD 3) EQ 0) THEN BEGIN
;
;	ThisDevice = !D.Name
;	SET_PLOT,'PS'
;
;	;; Here is the filename for the output.  Rename as needed.
;	plot_output='/home/ksimunac/Output/'+sat+rate_name+'_'+YEAR+MONTH+DATE+'_'+HR+MIN+'_contour_plot_raw.ps'
;
;	DEVICE, FILENAME=plot_output,XSIZE=8.0,YSIZE=8.0,/INCHES,$;
;;	/COLOR, Bits_Per_Pixel=8, PORTRAIT=1, XOFFSET=0.25, YOFFSET=0.25
;
;	!P.FONT = 0
;	DEVICE, SET_FONT = 'HELVETICA', /BOLD, FONT_SIZE = 16.0
;	
;	LOADCT,39, NCOLORS=256
;	!P.MULTI=[1,1,1]
;
;	count_data = fltarr(32,8)
;	count_data(*,*) = !values.f_nan
;	FOR i = 0,31 DO BEGIN
;		FOR j = 0,7 DO BEGIN
;			count_data(i,j) = full_data(i,j,k)
;		ENDFOR
;	ENDFOR
;	!X.TICKS = 8
;	!X.TICKNAME = ['128','112','96','80','64','48','32','16','0']
;
;	!Y.TICKS = 8
;	!Y.TICKNAME = ['0','4','8','12','16','20','24','28','32']
;
;	x_data = fltarr(2)
;	x_data(*) = s_chan(k)
;	y_data = fltarr(2)
;	y_data(0) = 0
;	y_data(1) = 8
;
;	
;	x_center = indgen(32)
;	y_center = fltarr(32)
;	IF sat EQ 'A' THEN y_center = (13.2*energy_S^(-0.064))/4.0
;	IF sat EQ 'B' THEN y_center = (11.9*energy_S^(-0.087))/4.0
;
;	FOR ll = 0, 31 DO BEGIN
;		IF x_center(ll) LT x_data(0) THEN y_center(ll) = !values.f_nan
;	ENDFOR
;
;
;
;	contour, count_data, xrange=[32,0], yrange = [0, 8], xstyle=1, ystyle=1,/fill, $
;		xtitle = 'ESA Step', ytitle = 'Defl Bin', title = 'Contour Plot of Counts '+YEAR+'-'+MONTH+'-'+DATE+' '+HR+':'+MIN, $
;		ticklen=1.0, xthick=2.0, ythick=2.0, LEVELS = [1,5,25,125,625,3125,15625,78125,390625]
;		;LEVELS = [1,10,100,1000,10000,100000,1000000];, LEVELS = [1,5,25,125,625,3125,15625,78125,390625]
;	OPLOT, x_data, y_data, COLOR = 250, THICK = 10.0
;	OPLOT, x_center, y_center, COLOR = 75, THICK = 10.0
;
;	;legend = 'AUTO SCALE TO ENHANCE CONTRAST'
;	legend = 'LOG SCALE POWERS OF 5'
;	XYOUTS, 120, -3, legend, /DEVICE, CHARSIZE = 0.9
;
;	;X_VERTEX = [-2.0,2.0,0.0,-2.0]
;	;Y_VERTEX = [3.5,3.5,0.0,3.5]
;	;USERSYM,X_VERTEX,Y_VERTEX, /FILL
;	;oplot, x_data, y_data,psym=8, color=250
;
;	DEVICE, /close_file
;	SET_PLOT, ThisDevice
;ENDIF
;
;ENDFOR
;
;print, 'Contour plots finished.'
;stop
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Subtract background, and see if two techniques have very different results.
;; If they differ by less than 5% caution flag is set to 0.  If they differ by 5 to
;; 10% the caution flag is set to 1.  If they differ by more than 10% the caution
;; flag is set to 2.
full_data_no_bkgd2 = fltarr(32,8,num_records5)
full_data_no_bkgd2(*,*,*) = !values.f_nan
delta_counts_less_bkgd = fltarr(32,8,num_records5)
delta_counts_less_bkgd(*,*,*) = !values.f_nan
calc_proton_norm_background_subtract, full_data, num_records5, caution_flag, full_data_no_bkgd2, delta_counts_less_bkgd


;;; Correct for deflection wobble
;;; -- an effect possibly due to an offset in deflector power supplies
;;FOR j = 1, 31 DO BEGIN          ; DEFL Steps
;;    FOR i = 64, 126, 2 DO BEGIN ; Even ESA Steps only
;;        FOR k = 0L, num_records1-1 DO BEGIN ; Time Steps
;;            full_data_no_bkgd2(i, j-1, k) = full_data_no_bkgd2(i, j, k)
;;	    delta_counts_less_bkgd(i,j-1,k) = delta_counts_less_bkgd(i,j,k)
;;            IF j EQ 31 THEN BEGIN
;;		full_data_no_bkgd2(i, j, k) = !values.f_nan
;;		delta_counts_less_bkgd(i,j,k) = !values.f_nan
;;	    ENDIF
;;        ENDFOR	
;;    ENDFOR
;;ENDFOR

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Un-comment this section to make a contour plot of counts versus ESA and Defl Steps
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;FOR k = 0, num_records5-2 DO BEGIN
;
;time = time_struct(epoch5(k))
;YEAR = STRING(time.year, FORMAT = '(I04)')
;DOY = STRING(time.doy, FORMAT = '(I03)')
;MONTH = STRING(time.month, FORMAT = '(I02)')
;DATE = STRING(time.date, FORMAT = '(I02)')
;HR = STRING(time.hour, FORMAT = '(I02)')
;MIN = STRING(time.min, FORMAT = '(I02)')
;next_time = time_struct(epoch5(k+1)) ; This is the next 5-min time stamp
;
;IF time.year EQ 2014 AND time.month EQ 2 AND (time.date GE 21 OR time.date LT 26) AND ((k MOD 30) EQ 0) THEN BEGIN
;
;	ThisDevice = !D.Name
;	SET_PLOT,'PS'
;
;	;; Here is the filename for the output.  Rename as needed.
;	plot_output='/home/ksimunac/Output/'+sat+rate_name+'_'+YEAR+MONTH+DATE+'_'+HR+MIN+'_contour_plot.ps'
;
;	DEVICE, FILENAME=plot_output,XSIZE=8.0,YSIZE=8.0,/INCHES,$
;	/COLOR, Bits_Per_Pixel=8, PORTRAIT=1, XOFFSET=0.25, YOFFSET=0.25
;
;	!P.FONT = 0
;	DEVICE, SET_FONT = 'HELVETICA', /BOLD, FONT_SIZE = 16.0
;	
;	LOADCT,39, NCOLORS=256
;	!P.MULTI=[1,1,1]
;
;	count_data = fltarr(32,8)
;	count_data(*,*) = !values.f_nan
;	FOR i = 0,31 DO BEGIN
;		FOR j = 0,7 DO BEGIN
;			count_data(i,j) = full_data_no_bkgd2(i,j,k)
;		ENDFOR
;	ENDFOR
;
;	!X.TICKS = 8
;	!X.TICKNAME = ['128','112','96','80','64','48','32','16','0']
;
;	!Y.TICKS = 8
;	!Y.TICKNAME = ['0','4','8','12','16','20','24','28','32']
;
;	contour, count_data, xrange=[32,0], yrange = [0, 8], xstyle=1, ystyle=1,/fill, $
;		xtitle = 'ESA Step', ytitle = 'Defl Bin', title = 'Contour Plot of Counts '+YEAR+'-'+MONTH+'-'+DATE+' '+HR+':'+MIN, $
;		ticklen=1.0, xthick=2.0, ythick=2.0, LEVELS = [1,5,25,125,625,3125,15625,78125]
;		;LEVELS = [1,10,100,1000,10000,100000,1000000];, LEVELS = [1,5,25,125,625,3125,15625,78125]
;
;	x_data =intarr(2)
;	x_data(*) = s_chan(k)
;	; Get S-Channel switch groups from 1 minute data to overplot
;	FOR ii = 0, n_elements(epoch1) - 5,0 DO BEGIN
;	epoch1_time_0 = time_struct(epoch1(ii))
;	epoch1_time_1 = time_struct(epoch1(ii+1))
;	epoch1_time_2 = time_struct(epoch1(ii+2))
;	epoch1_time_3 = time_struct(epoch1(ii+3))
;	epoch1_time_4 = time_struct(epoch1(ii+4))
;	x_data_0 = intarr(2)
;	x_data_0(*) = -1
;	x_data_1 = intarr(2)
;	x_data_1(*) = -1
;	x_data_2 = intarr(2)
;	x_data_2(*) = -1
;	x_data_3 = intarr(2)
;	x_data_3(*) = -1
;	x_data_4 = intarr(2)
;	x_data_4(*) = -1
;	IF (epoch1_time_0.date EQ time.date) AND (epoch1_time_0.hour EQ time.hour) AND (epoch1_time_0.min EQ time.min) AND $
;		(epoch1_time_0.sec EQ time.sec) THEN BEGIN
;		x_data_0(*) = s_chan_full_res(ii)
;		ii = ii+1
;		IF (epoch1_time_1.date EQ time.date) AND (epoch1_time_1.hour EQ time.hour) AND (epoch1_time_1.min GE time.min) AND $
;			(epoch1_time_1.min LT next_time.min) THEN BEGIN
;				x_data_1(*) = s_chan_full_res(ii+1)
;				ii = ii+1
;		ENDIF
;		IF (epoch1_time_2.date EQ time.date) AND (epoch1_time_2.hour EQ time.hour) AND (epoch1_time_2.min GE time.min) AND $
;			(epoch1_time_2.min LT next_time.min) THEN BEGIN
;				x_data_2(*) = s_chan_full_res(ii+1)
;				ii = ii+1
;		ENDIF
;		IF (epoch1_time_3.date EQ time.date) AND (epoch1_time_3.hour EQ time.hour) AND (epoch1_time_3.min GE time.min) AND $
;			(epoch1_time_3.min LT next_time.min) THEN BEGIN
;				x_data_3(*) = s_chan_full_res(ii+1)
;				ii = ii+1
;		ENDIF
;		IF (epoch1_time_4.date EQ time.date) AND (epoch1_time_4.hour EQ time.hour) AND (epoch1_time_4.min GE time.min) AND $
;			(epoch1_time_4.min LT next_time.min) THEN BEGIN
;				x_data_4(*) = s_chan_full_res(ii+1)
;				ii = ii+1
;		ENDIF
;	ENDIF ELSE BEGIN
;		ii = ii+1
;	ENDELSE
;	ENDFOR
;
;	y_data = intarr(2)
;	y_data(0) = 0
;	y_data(1) = 8
;	
;	;STOP
;
;	x_center = indgen(32)
;	y_center = fltarr(32)
;	IF sat EQ 'A' THEN y_center = (13.2*energy_S^(-0.064))/4.0
;	IF sat EQ 'B' THEN y_center = (11.9*energy_S^(-0.087))/4.0
;	
;	FOR ll = 0, 31 DO BEGIN
;		IF ll LT x_data(0) THEN y_center(ll) = !values.f_nan
;	ENDFOR
;
;	oplot, x_data, y_data,COLOR = 250, THICK = 10.0
;	oplot, x_center, y_center, COLOR = 75, THICK = 10.0
;	oplot, x_data_0, y_data, COLOR = 250, THICK = 10.0
;	oplot, x_data_1, y_data, COLOR = 250, THICK = 10.0
;	oplot, x_data_2, y_data, COLOR = 250, THICK = 10.0
;	oplot, x_data_3, y_data, COLOR = 250, THICK = 10.0
;	oplot, x_data_4, y_data, COLOR = 75, THICK = 10.0
;
;	legend = 'LOG SCALE POWERS OF 5'
;	XYOUTS, 120, -3, legend, /DEVICE, CHARSIZE = 0.9
;
;
;	DEVICE, /close_file
;	SET_PLOT, ThisDevice
;ENDIF
;
;ENDFOR
;
;print, 'Contour plots finished.'
;stop
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Fit the counts versus deflection steps with a gaussian to find the peak for the N/S flow angle
S_MAX = FLTARR(num_records5)
S_MAX(*) = 0.0
MAIN_MAX = FLTARR(num_records5) 
MAIN_MAX(*) = 0.0
calc_proton_defl_peak_norm, num_records5, full_data_no_bkgd2, s_chan, S_MAX, MAIN_MAX, epoch5, sat


;; Convert to Hz -- divide through by accumulation time of 
;12.8 milliseconds*5 cycles*4 ESA steps*4 Defl steps = 1.024  seconds
full_data_hz = fltarr(32,8,num_records5) ; changed from double to float to save memory 28 Sept, 2007
full_data_hz[*,*,*] = !values.f_nan
full_data_hz = FLOAT(full_data_no_bkgd2)/1.024 ; changed from double to float to save memory 28 Sept, 2007
delta_counts_hz = fltarr(32,8,num_records5)
delta_counts_hz = !values.f_nan
delta_counts_hz = delta_counts_less_bkgd/1.024
print, 'Converted to Hz.'

full_data_no_bkgd2 = 0 ; set to 0 to free up memory 
delta_counts_less_bkgd = 0 ; free up memory

;; Correct for dead time -- real rate = recorded rate/(1-recorded rate*dead time)
delta_counts_hz = delta_counts_hz/((1.0-(full_data_hz*dead_time))^2.0)
full_data_hz = full_data_hz/(1.0-(full_data_hz*dead_time)) ; changed 1 to 1.0 13 Jan, 2009

;; See if dead time correction results in negative rate.  If so, stop.
FOR k = 0L, num_records5-1 DO BEGIN
    FOR i = 0, 31 DO BEGIN
        FOR j = 0, 7 DO BEGIN
            IF full_data_hz(i, j, k) LT 0 THEN BEGIN
                PRINT, '(', i, j, k, ') Invalid countrate -- deadtime is too big.'
		full_data_hz(i,j,k) = !values.f_nan
                STOP
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
print, 'Dead time correction completed.'

; Divide by geometric factor to account for different apertures
full_data_hz = full_data_hz/G_Main ; Divide by G to get into to the right units
delta_counts_hz = delta_counts_hz/G_Main
FOR hh=0L, num_records5-1 DO BEGIN
    IF sat EQ 'A' THEN F_Switch = 120.0 ; approximate ratio between main and s channel based on in-flight data
    IF sat EQ 'B' THEN F_Switch = 625.0 ; 
    FOR ii = 0, 31 DO BEGIN			
        IF (ii GE s_chan(hh)) THEN BEGIN ; if data taken with S channel then multiply to normalize with main channel
            full_data_hz(ii, *, hh) = full_data_hz(ii, *, hh)*F_Switch
	    delta_counts_hz(ii,*,hh) = delta_counts_hz(ii,*,hh)*F_Switch
        ENDIF
    ENDFOR
ENDFOR
print, 'Division by geometric factor completed.'


; Sum over deflection (out-of-ecliptic plane) steps to get a 1-D count distribution
sum_full_data = FLTARR(num_records5,32) 
sum_full_data(*,*) = 0.0
sum_delta_counts = FLTARR(num_records5,32)
sum_delta_counts(*,*) = 0.0
peak_ESA = INTARR(num_records5)
peak_ESA(*) = -1
calc_proton_norm_sum_defl, full_data_hz, delta_counts_hz, sum_full_data, sum_delta_counts, peak_ESA, num_records5


; Divide through by a flat efficiency for both main and s-channel based on pre-launch calibration.
IF sat EQ 'A' THEN BEGIN
	sum_full_data = sum_full_data/0.80
	sum_delta_counts = sum_delta_counts/0.80

;; If using STA SFR0 then multiply through by 110.0 for May and July 2012 events
;; IF using STA SF0 multiply by 4.0
;; If using STA STP0 multiply by 2750.0
;	sum_full_data = 110.0*sum_full_data
;	sum_delta_counts = 110.0*sum_delta_counts

ENDIF

IF sat EQ 'B' THEN BEGIN
	sum_full_data = sum_full_data/4.0
	sum_delta_counts = sum_delta_counts/4.0

;;IF USING STB SF0 multiply by 25.0
;	sum_full_data = 25.0*sum_full_data
;	sum_delta_counts = 25.0*sum_delta_counts

ENDIF
 
print, 'Divison by efficiency completed.'

; Multiply by another energy dependent "leakage efficiency"
FOR ii = 0L, num_records5 - 1 DO BEGIN
	FOR jj = 0, 31 DO BEGIN
		IF jj GE s_chan(ii) THEN BEGIN 
		sum_full_data(ii,jj) = sum_full_data(ii,jj)*eff(jj)
		sum_delta_counts(ii,jj) = sum_delta_counts(ii,jj)*eff(jj)
		ENDIF
	ENDFOR
ENDFOR

; Now multiply each ESA step by (delta theta*delta phi)/(v^2) to get distribution function
dist_funct = fltarr(32,num_records5)
dist_funct(*,*) = !values.d_nan
peak_dist = intarr(num_records5)
peak_dist(*) = -1
delta_dist_funct = fltarr(32,num_records5)
delta_dist_funct(*,*) = !values.d_nan

FOR i = 0L, num_records5-1 DO BEGIN
    peak_count = 0.0
    FOR j = 0, 31 DO BEGIN
        IF sum_full_data(i, j) GT 0 THEN BEGIN
		IF j GE s_chan(i) THEN BEGIN ; S Channel data
		   IF j GE 8 THEN BEGIN
            	   dist_funct(j, i) = 0.08727*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
	    	   delta_dist_funct(j,i) = (0.08727*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   ; Account for defls maxing out at ESA step 33 (group 8)
		   IF j EQ 7 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/1.11)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) 
	    	   delta_dist_funct(j,i) = ((0.08727/1.11)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 6 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/1.35)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0)
	    	   delta_dist_funct(j,i) = ((0.08727/1.35)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 5 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/1.63)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) 
	    	   delta_dist_funct(j,i) = ((0.08727/1.63)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 4 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/1.97)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0)
	    	   delta_dist_funct(j,i) = ((0.08727/1.97)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 3 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/2.37)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) 
	    	   delta_dist_funct(j,i) = ((0.08727/2.37)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 2 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/2.87)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) 
	    	   delta_dist_funct(j,i) = ((0.08727/2.87)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 1 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/3.46)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) 
	    	   delta_dist_funct(j,i) = ((0.08727/3.46)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		   IF j EQ 0 THEN BEGIN
            	   dist_funct(j, i) = (0.08727/4.0)*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) 
	    	   delta_dist_funct(j,i) = ((0.08727/4.0)*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		   ENDIF
		ENDIF ELSE BEGIN ; Main channel
		dist_funct(j, i) = 0.08727*0.7854*sum_full_data(i, j)/((v_main(j))^2.0) 
	    	delta_dist_funct(j,i) = (0.08727*0.7854/((v_main(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_main(j))^2.0)
		ENDELSE
        ENDIF ELSE BEGIN
            dist_funct(j, i) = 0.0
	    delta_dist_funct(j,i) = 0.0
        ENDELSE

; Omit the ESA group where the S-Channel switch occurred.  There is a chance it could
; change between groups within the 5-minutes, so we may want to add a step to check
; the 1-minute S-channel switch values.
	
	IF j EQ s_chan(i) THEN dist_funct(j,i) = !values.f_nan

; For special events exclude some ESA steps 
;IF j GE 15 THEN dist_funct(j,i) = !values.f_nan
;IF j LE 10 THEN dist_funct(j,i) = !values.f_nan

        IF (dist_funct(j, i) GT peak_count) THEN peak_count = dist_funct(j, i)
        IF (dist_funct(j, i) GE peak_count) THEN peak_dist(i) = j
    ENDFOR
ENDFOR

sum_full_data = 0 ; added 28 Sept, 2007 to use save memory
sum_delta_counts = 0


; To make it easier for IDL to get a good fit to the curve, multiply the distribution by
; 1e5.  Remember to divide the density by this factor later on!
dist_funct = dist_funct*(1.0E5)
delta_dist_funct = delta_dist_funct*(1.0E5)
print, 'Distribution functions complete.'

; Fit the distribution function versus velocity data to a 1-D Gaussian.
coeff = fltarr(num_records5,3) ; A matrix in which to store the proton fit coefficients
coeff(*,*,*) = !values.d_nan
density = fltarr(num_records5)
density(*) = !values.f_nan
avg_v= fltarr(num_records5)
avg_v(*) = !values.f_nan
temp_K = fltarr(num_records5)
temp_K(*) = !values.f_nan
v_th = fltarr(num_records5)
v_th(*) = !values.f_nan
NS = fltarr(num_records5)
NS(*) = !values.f_nan
good_fit = fltarr(num_records5)
good_fit(*) = !values.f_nan

calc_proton_fit_maxwellian_norm, sat, num_records5, coeff, density, avg_v, temp_K, v_th, good_fit, $
	peak_dist, dist_funct, delta_dist_funct, v_main, v_S, std_error, error5, s_chan, epoch5

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Find the N/S Flow Angle  ; This needs more work and is not ready for public distribution
;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;        SMALL_S_MAX_SCALED = dblarr(num_records5)
;        SMALL_S_MAX_SCALED(*) = !values.d_nan
;	SMALL_MAIN_MAX = dblarr(num_records5)
;	SMALL_MAIN_MAX(*) = !values.d_nan
;	SMALL_S_MAX = dblarr(num_records5)
;	SMALL_S_MAX(*) = !values.d_nan
;
;        FOR x = 0L, num_records5-1 DO BEGIN
;            IF MAIN_MAX(x) EQ 0 THEN SMALL_MAIN_MAX(x) = !values.f_nan ELSE SMALL_MAIN_MAX(x) = MAIN_MAX(x)
;            IF S_MAX(x) EQ 0 THEN SMALL_S_MAX(x) = !values.f_nan ELSE SMALL_S_MAX(x) = S_MAX(x)
;        ENDFOR
;        IF SAT EQ 'A' THEN BEGIN
;            SMALL_MAIN_MAX = -19.25+5.5*SMALL_MAIN_MAX ; convert to angle (degrees) from bin number
;            SMALL_S_MAX = -17.5 + 5.0*SMALL_S_MAX  ; convert to angle (degrees) from bin number
;            SMALL_S_MAX_SCALED = SMALL_S_MAX
;            ;SMALL_S_MAX_SCALED = 2.2*(SMALL_S_MAX + 2.55) ;  
;                        ; 2.55 if shift center of distribution, 2.3 is to spread out distribution     
;			; Note that this is different from 1 minute processing because there is
;			; no correction for deflection wobble (which makes the offset from 0 
;			; larger because of the choice of which ESA steps to shift).  In the
;			; 1 minute processing the distribution is shifted by 3.5 degrees and 
;			; stretched by a factor of 3.0.
;            FOR k = 0L, num_records5-1 DO BEGIN
;                NS(k) = 2.0*(SMALL_S_MAX_SCALED(k) + (0.005)*avg_v(k)-1.0137)
;                ;NS(k) = 2.0*(SMALL_S_MAX_SCALED(k) + (0.0062)*avg_v(k)-1.7297)
;                   ; this comes from velocity/energy dependence of leakage
;                IF ABS(NS(k)) GT 20.0 THEN NS(k) = !values.f_nan ; added 02 September, 2008
;            ENDFOR
;        ENDIF ELSE BEGIN ; spacecraft B
;            SMALL_MAIN_MAX = 31.0 - SMALL_MAIN_MAX
;            SMALL_S_MAX = 31.0 - SMALL_S_MAX ; flip distribution because PLASTIC-B is upside-down
;            
;            SMALL_MAIN_MAX = -19.25+5.5*SMALL_MAIN_MAX
;            SMALL_S_MAX = -17.5 + 5.0*SMALL_S_MAX
;            SMALL_S_MAX_SCALED = 1.75*(SMALL_S_MAX - 4.62) ; modified 14 August, 2008
;            FOR k = 0L, num_records5-1 DO BEGIN
;                NS(k) = SMALL_S_MAX_SCALED(k) + (-2.6404E-7)*(avg_v(k)^3) + (4.3726E-4)*(avg_v(k)^2) + $
;                                 (-2.3922E-1)*avg_v(k)+4.2639E1
;                IF ABS(NS(k)) GT 20.0 THEN NS(k) = !values.f_nan ; added 02 September, 2008
;            ENDFOR 
;        ENDELSE
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Now, need to remove times when entrance system is closed
;Uncomment below to define variables for alpha data and then fit the alpha distribution
coeff_alpha = fltarr(num_records5,3) ; A matrix in which to store the proton fit coefficients
coeff_alpha(*,*,*) = !values.d_nan
density_alpha = fltarr(num_records5)
density_alpha(*) = !values.f_nan
avg_v_alpha= fltarr(num_records5)
avg_v_alpha(*) = !values.f_nan
temp_K_alpha = fltarr(num_records5)
temp_K_alpha(*) = !values.f_nan
v_th_alpha = fltarr(num_records5)
v_th_alpha(*) = !values.f_nan
good_fit_alpha = fltarr(num_records5)
good_fit_alpha(*) = !values.f_nan
expected_peak_channel = INTARR(num_records5)
expected_peak_channel(*) = -1

;calc_norm_alpha_data, v_main, v_S, avg_v, density, v_th, dist_funct, delta_dist_funct, coeff_alpha, $
;	density_alpha, avg_v_alpha, temp_K_alpha, v_th_alpha, good_fit_alpha, $
;	coeff, num_records5, expected_peak_channel, s_chan, sat, epoch5 

; CHECK FOR PROBLEMS (non-physical results)
; hand-checked bad times -- Lorna keeps updated list

IF sat EQ 'A' THEN BEGIN  ;outliers
   bad_i = where(time_string(epoch5) EQ '2007-01-01/00:00:00' OR $ ; to be filled in
                 time_string(epoch5) EQ '2007-01-01/00:00:55', $ ; to be filled in
                 bad_count)
   IF bad_count GT 0 THEN BEGIN 
       avg_v    [bad_i] = !values.f_nan
       density  [bad_i] = !values.f_nan
       temp_K   [bad_i] = !values.f_nan
       v_th     [bad_i] = !values.f_nan
       good_fit [bad_i] = !values.f_nan
       std_error[bad_i] = 8
   ENDIF
ENDIF 

IF sat EQ 'B' THEN BEGIN
    bad_i = where(time_string(epoch5) EQ '2007-01-01/00:00:00' OR $ ; to be filled in
                  time_string(epoch5) EQ '2007-01-01/00:00:55', $ ; to be filled in
                  bad_count)
    IF bad_count GT 0 THEN BEGIN 
        avg_v    [bad_i] = !values.f_nan
        density  [bad_i] = !values.f_nan
        temp_K   [bad_i] = !values.f_nan
	v_th     [bad_i] = !values.f_nan
	good_fit [bad_i] = !values.f_nan
        std_error[bad_i] = 8
    ENDIF
ENDIF 

; Check for 3 standard deviations or non-physical results

FOR k = 0L, num_records5 - 1 DO BEGIN
IF std_error(k) NE 3 THEN BEGIN
    IF avg_v(k) GT 2500 OR avg_v(k) LE 230 THEN std_error(k) = 1 ; changed upper limit August 2012
    IF density(k) LT 0.001 OR density(k) GE 500 THEN std_error(k) = 1
    IF (v_th(k) GT 400) THEN std_error(k) = 1
    IF (good_fit(k) LT 0.25) OR (good_fit(k) GT 10) THEN std_error(k) = 1 

    IF sat EQ 'B' AND v_th(k) LE 10 THEN BEGIN ; changed to from 12 to 10, June 2010
	IF std_error(k) NE 1 THEN std_error(k) = 7
    ENDIF

    IF sat EQ 'A' AND v_th(k) LE 10 THEN BEGIN
	IF std_error(k) NE 1 THEN std_error(k) = 7
    ENDIF

    IF error5(k) GT 0 THEN std_error(k) = 2

    IF FINITE(avg_v(k)) EQ 0 AND std_error(k) NE 5 THEN std_error(k) = 1
    IF FINITE(good_fit(k)) EQ 0 THEN std_error(k) = 1 ; added Feb 2009

    IF std_error(k) EQ 1 THEN BEGIN
	avg_v[k] = !values.f_nan
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan
	NS(k) = !values.f_nan
    ENDIF

    IF std_error(k) EQ 7 THEN BEGIN ; removed chi^2 criteria, KDCS, 12 October 2009
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan
    ENDIF
    
    IF FINITE(density(k)) EQ 1 THEN BEGIN
        IF k GE 5 AND k LE num_records5-7 THEN BEGIN
	IF FINITE(density(k+1)) EQ 1 AND FINITE(density(k+2)) EQ 1 THEN BEGIN
            mean_density = MEAN(density(k-5:k+6), /NAN)
            mean_avg_v = MEAN(avg_v(k-5:k+6), /NAN)
            mean_v_th = MEAN(v_th(k-5:k+6), /NAN)
	  
            stdev_density = STDDEV(density(k-5:k+6), /NAN)
            stdev_avg_v = STDDEV(avg_v(k-5:k+6), /NAN)
            stdev_v_th = STDDEV(v_th(k-5:k+6), /NAN)

            IF ABS(density(k) - mean_density) GE 3*stdev_density AND std_error(k) NE 1 AND std_error(k) NE 3 THEN std_error(k) = 4
            IF ABS(avg_v(k) - mean_avg_v) GE 3*stdev_avg_v AND std_error(k) NE 1 AND std_error(k) NE 3 THEN std_error(k) = 4
            IF ABS(v_th(k) - mean_v_th) GE 3*stdev_v_th AND std_error(k) NE 1 AND std_error(k) NE 3 THEN std_error(k) = 4
	ENDIF
        ENDIF
			
    ENDIF

ENDIF ELSE BEGIN ; data overflow flag is set

	avg_v[k] = !values.f_nan
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan
	NS(k) = !values.f_nan
ENDELSE
ENDFOR

prefix = 'ST'+sat+'_ra_mom_5min'
den = prefix+'_density'
vel = prefix+'_bulk_speed'
tem = prefix+'_temperature'
vth = prefix+'_v_thermal'
err = prefix+'_error'
chi = prefix+'_reduced_chi_squared'
ns1 = prefix+'_north_south'

store_data, den, data = {x:epoch5, y:density  }
store_data, vel, data = {x:epoch5, y:avg_v    }
store_data, tem, data = {x:epoch5, y:temp_K   }
store_data, vth, data = {x:epoch5, y:v_th     }
store_data, err, data = {x:epoch5, y:std_error}
store_data, chi, data = {x:epoch5, y:good_fit }
store_data, ns1, data = {x:epoch5, y:NS       }


; Create ascii files and stop
;calc_proton_raw_ascii_norm, sat, epoch5, num_records5, density, avg_v, temp_K, v_th, good_fit, std_error, NS
;calc_norm_alpha_raw_ascii, sat, epoch5, num_records5, density, avg_v, temp_K, v_th, good_fit, $
;	avg_v_alpha, density_alpha, v_th_alpha, good_fit_alpha, expected_peak_channel, std_error
 
;stop

END 
