;+
;PROCEDURE: calc_proton_bulk_v2
;PURPOSE:
;  Calculate proton bulk parameters from 1 minute resolution RA_Trigger rate.
;
;  This is the second version of calc_proton_bulk.  The original version is
;  at best slow and memory intensive.  This version is intended to run more
;  efficiently.
;
;CREATED BY: K. Simunac
;
;LAST MODIFICATION: 28 May 2011
;
;MODIFICATION HISTORY:
;	19 November 2009:  created
;	25 November 2009:  
;		added variable to store information
;		regarding whether alpha peak is in main or S channel
;	June 2010: 
;		-Modified STB S channel ESA table
;		-Modified efficiency tables for both STA and STB
;		-Changed the S channel switch factor for B after 
;		 studying the distribution functions before and after
;		 background subtraction
;		-Changed minimum acceptable thermal speed on STB from 12
;		 to 10 km/s (this agrees with STA)
;		-Updated list of "outlier" points
;  07/16/2010 LBE Added Outliers.
;  08/04/2010 LBE Added Outliers.
;  08/10/2010 LBE Change order so don't overwrite error code 8.
;  08/26/2010 LBE Added Outliers.
;  09/13/2010 LBE Added Outliers.
;  09/29/2010 KDCS Corrected averaging of reduced chi^2 values.
;  10/06/2010 KDCS Added check for error codes in averaging of
;		   chi^2.  Added check for error code 8 in all
;		   averaging sections.
;  10/07/2010 KDCS Set the reduced chi^2 values to NaN if
;		   error code is set to 7. 
;  10/18/2010 LBE  Added Outliers. Check for finite(epoch1[0]).
;  11/15/2010 LBE  Added outliers.
;  01/07/2011 LBE  Added outliers.
;  01/14/2011 LBE  Added outliers.
;  02/18/2011 LBE  Added outliers.
;  03/09/2011 LBE  Added outliers.
;  03/15/2011 LBE  Added outliers.
;  04/11/2011 LBE  Added outliers.
;  05/09/2011 LBE  Added outliers.
;  05/27/2011 KDCS Updated STB efficiency tables and S channel swich ratio.
;			Added sections of code to correctly account for protons in
;			the main channel.
;  05/28/2011 KDCS Inserted a temporary main channel table (really the S table)
;			for STA so that routine processing can be run.
;  05/31/2011 LBE  Removed B outliers since this is a new version.
;  06/08/2011 LBE  Added A outliers.
;  07/19/2011 LBE  Added B outliers.
;  08/04/2011 LBE  Added B outliers.
;  08/11/2011 LBE  Added B outliers.
;  09/13/2011 LBE  Added outliers.
;  09/20/2011 LBE  Update to take in 5-min rate data.
;  11/10/2011 LBE  Added outliers.
;  11/28/2011 LBE  Set 10min and 1hr to caution=5 if any 1min has that.
;  11/29/2011 LBE  Change STA 5min file to 201111129 (sets chi2 to NaN when error=3)
;  12/07/2011 LBE  Added outliers.
;  01/09/2012 LBE  Added outliers.
;  06/13/2012 LBE  Added separate check for removing just angle.
;  07/10/2012 LBE  Update priorities of error codes in averages.
;  07/10/2012 LBE  Added angle_error for averages.
;  03/13/2013 LBE  Moved ouliers to seperate procedures.
;  04/10/2014 LBE  Allow 10-second match for hk times.
;  05/01/2014 LBE  Change to new get_pla_hk procedure.
;  05/12/2014 LBE  Change line 1295 to check on_first for base_hour.
;  11/25/2014 LBE  Changes to deal with data gaps (error=5)
;  11/25/2014 LBE  Change so error 5 and 8 don't go into averages.
;  12/16/2014 LBE  Set have_data to 0 if all have error 5.
;  12/23/2015 LBE  Added /POST_CONJUNCTION to call for roll info.
;-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;							  ;
; Below is a routine to calculate density, bulk speed,	  ;
; temperature and thermal speed using full	 	  ;
; resolution rate data from RA_Trigger or SF0.		  ;
;	 						  ;		
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO calc_proton_bulk_v2, sat, files, which_epoch, time, num_days, table_location
compile_opt strictarrsubs

COMMON share1_high_res
COMMON com_high_res
tab = string(9B)
tplot_options, 'color', 0

num_records1 = final_count ; final count of number of records from pl_read_cdf_high_res
have_data = 1

; Define some variables for storing error flags and s-channel switch information
    s_chan = intarr(num_records1)
    s_chan(*) = 0 
    error1 = intarr(num_records1)
    rate_name = ' '
    std_error = intarr(num_records1) ; A matrix in which to store possible errors
    std_error(*) = 0
    caution_flag = intarr(num_records1) 
	; caution_flag is a matrix to store a warning if background subtraction results are substantially different
    caution_flag(*) = 0


;; As of June 2010 ESA range is still set to steps 64 to 127
;; Highest proton speed that can be measured is about 900 km/second

;; Geometric Factors for Main Channel and S Channel (full SWS)
;; G = Active Area*Angular Extent of Slit*Bandwidth/2
;; For example, for the main channel of FM1:
;; 	Active Area = 8.9E-5 m^2
;;	Deflection FWHM = 1.9 degrees = 0.033 radians
;;	Azimuth Acceptance = 45 degrees = 0.785 radians
;;	Energy Bandwidth = 6.12%
;;	G = (8.9E-5)*(0.033)*(0.785)*0.0612/2 = 7.1E-8 m^2 sr eV/eV

A_G_Main = 7.1E-8  ; geometric factor for PLASTIC/A main channel
A_G_Sch = 2.3E-11  ; geometric factor for PLASTIC/A small channel
B_G_Main = 6.0E-8  ; geometric factor for PLASTIC/B main channel
B_G_Sch = 1.7E-11  ; geometric factor for PLASTIC/B small channel

;; Dead times for SF0 and RA_TRIG

A_SF0_dead_time = 5.0E-6 ;sec
A_RA_TRIG_dead_time = 2.0E-6 ;sec
B_SF0_dead_time = 5.0E-6	;sec
B_RA_TRIG_dead_time = 2.0E-6	;sec ; CHANGED ON 02 APRIL, 2007 FROM 2.5E-6


;; Read in tables of estimated efficiencies
;; The location of the efficiency tables must be specified in the crib file.

IF sat EQ 'A' THEN eff_file = table_location+'leakage_A_2010_05_07.txt'
IF sat EQ 'B' THEN eff_file = table_location+'leakage_B_2011_05_17.txt' 
eff = fltarr(128)
openr, in_lun_1, eff_file, /GET_LUN
line = ","
readf, in_lun_1, line ; read the header line
FOR i = 64, 127 DO BEGIN
	readf, in_lun_1, line
	parts = STRSPLIT(line, ",", /EXTRACT)
	eff[i]=parts[1]
ENDFOR
close, in_lun_1
free_lun, in_lun_1

print, 'Efficiency tables have been read.'

;---------------------------------------------------------
IF sat EQ 'A' THEN BEGIN
   ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_a_full, /nan) LE 0 AND TOTAL(ra_trig_a_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        STOP
    ENDIF
   

   ;; Check to see if there are RA_Trig counts.  If so, continue.  
   ;; If not, assume the full resolution rate is SF0.
   ;;
   ;; RA_Trig was set as the full resolution rate on 14 February, 2007 for A
   ;; and on 1 March, 2007 for B.

    IF TOTAL(ra_trig_a_full, /nan) GT 0 THEN BEGIN
        full_data = ra_trig_a_full
        ra_trig_a_full = 0
        rate_name = '_RA_Trig'
        dead_time = A_RA_trig_dead_time
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_a
        error1 = error1_a
    ENDIF ELSE BEGIN
        full_data = sf0_a_full
        sf0_a_full = 0
        rate_name = '_SF0'
        dead_time = A_SF0_dead_time
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_a
        error1 = error1_a
    ENDELSE



ENDIF ELSE BEGIN ; data is from spacecraft B
    ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_b_full, /nan) LE 0 AND TOTAL(ra_trig_b_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        have_data = 0
    ENDIF
    

    ;; Check to see if there are RA_Trig counts.  If so, continue.  
    ;; If not, assum SF0 is the full resolution rate.
    ;;
    ;; RA_Trig was set as the full resolution rate on 14 February, 2007 on A
    ;; and on 1 March, 2007 on B.

    IF TOTAL(ra_trig_b_full, /nan) GT 0 THEN BEGIN
        full_data = ra_trig_b_full
        ra_trig_b_full = 0	
        rate_name = '_RA_Trig'
        dead_time = B_RA_trig_dead_time
        G_Main = B_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_b
        error1 = error1_b
    ENDIF ELSE BEGIN
        full_data = sf0_b_full
        sf0_b_full = 0
        rate_name = '_SF0'
        dead_time = B_SF0_dead_time
	G_Main = B_G_Main
        G_Sch = B_G_Sch
        s_chan = s_chan1_b
        error1 = error1_b
    ENDELSE
ENDELSE

;; Get the time associated with the first file to label output files
start_time=' '
start_time=strmid(time,0,10)

print, 'Starting calculations.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove the fill value of -1 from the files.
remove_fill_high_res, -1, full_data
print, 'Removed fill values.'

; Check for missing data. Mark record with error 5.
FOR m = 0L, num_records1-1 DO BEGIN
    bad_i = where(finite(full_data[*, *, m]) EQ 0, bad_count)
    IF bad_count GT 2048 THEN std_error[m] = 5 ; expect half the ESA steps
ENDFOR 

;; Check for value indicating overflow.  If present, mark record with error 3.
FOR m=0L, num_records1 - 1 DO BEGIN
    FOR k = 0, 127 DO BEGIN
        FOR l = 0, 31 DO BEGIN
            IF full_data(k, l, m) GE 507904 THEN BEGIN
                full_data(k, l, m) = !values.f_nan
                std_error(m) = 3
                PRINT, 'Overflow value detected.  Time, ESA, Defl:', m, k, l
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
;print,'Maximum Counts:',max(full_data,/NAN)
;IF max(full_data) GT 32000 THEN STOP

;; Subtract background, and see if two techniques have very different results.
;; If they differ by less than 5% caution flag is set to 0.  If they differ by 5 to
;; 10% the caution flag is set to 1.  If they differ by more than 10% the caution
;; flag is set to 2.
full_data_no_bkgd2 = fltarr(128,32,num_records1)
full_data_no_bkgd2(*,*,*) = !values.f_nan
delta_counts_less_bkgd = fltarr(128,32,num_records1)
delta_counts_less_bkgd(*,*,*) = !values.f_nan
calc_proton_background_subtract, full_data, num_records1, caution_flag, full_data_no_bkgd2, delta_counts_less_bkgd

;; Correct for deflection wobble
;; -- an effect possibly due to an offset in deflector power supplies
FOR j = 1, 31 DO BEGIN          ; DEFL Steps
    FOR i = 64, 126, 2 DO BEGIN ; Even ESA Steps only
        FOR k = 0L, num_records1-1 DO BEGIN ; Time Steps
            full_data_no_bkgd2(i, j-1, k) = full_data_no_bkgd2(i, j, k)
	    delta_counts_less_bkgd(i,j-1,k) = delta_counts_less_bkgd(i,j,k)
            IF j EQ 31 THEN BEGIN
		full_data_no_bkgd2(i, j, k) = !values.f_nan
		delta_counts_less_bkgd(i,j,k) = !values.f_nan
	    ENDIF
        ENDFOR	
    ENDFOR
ENDFOR

;; Fit the counts versus deflection steps with a gaussian to find the peak for the N/S flow angle
S_MAX = FLTARR(num_records1)
S_MAX(*) = 0.0
MAIN_MAX = FLTARR(num_records1) 
MAIN_MAX(*) = 0.0
calc_proton_defl_peak, num_records1, full_data_no_bkgd2, s_chan, S_MAX, MAIN_MAX

;; Convert to Hz -- divide through by accumulation time of 12.8 milliseconds
full_data_hz = fltarr(128,32,num_records1) ; changed from double to float to save memory 28 Sept, 2007
full_data_hz[*,*,*] = !values.f_nan
full_data_hz = FLOAT(full_data_no_bkgd2)/0.0128 ; changed from double to float to save memory 28 Sept, 2007
delta_counts_hz = fltarr(128,32,num_records1)
delta_counts_hz = !values.f_nan
delta_counts_hz = delta_counts_less_bkgd/0.0128
print, 'Converted to Hz.'

full_data_no_bkgd2 = 0 ; set to 0 to free up memory 
delta_counts_less_bkgd = 0 ; free up memory

;; Correct for dead time -- real rate = recorded rate/(1-recorded rate*dead time)
delta_counts_hz = delta_counts_hz/((1.0-(full_data_hz*dead_time))^2.0)
full_data_hz = full_data_hz/(1.0-(full_data_hz*dead_time)) ; changed 1 to 1.0 13 Jan, 2009

;; See if dead time correction results in negative rate.  If so, stop.
FOR k = 0L, num_records1-1 DO BEGIN
    FOR i = 0, 127 DO BEGIN
        FOR j = 0, 31 DO BEGIN
            IF full_data_hz(i, j, k) LT 0 THEN BEGIN
                PRINT, '(', i, j, k, ') Invalid countrate -- deadtime is too big.'
                STOP
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
print, 'Dead time correction completed.'

; Divide by geometric factor to account for different apertures
full_data_hz = full_data_hz/G_Main ; Divide by G to get into to the right units
delta_counts_hz = delta_counts_hz/G_Main
FOR hh=0L, num_records1-1 DO BEGIN
    IF sat EQ 'A' THEN F_Switch = 120.0 ; approximate ratio between main and s channel based on in-flight data
    IF sat EQ 'B' THEN F_Switch = 625.0 ; changed from 125 to 625 27 May 2011
    FOR ii = 0, 127 DO BEGIN			
        IF (ii GE s_chan(hh)) THEN BEGIN ; if data taken with S channel then multiply to normalize with main channel
            full_data_hz(ii, *, hh) = full_data_hz(ii, *, hh)*F_Switch
	    delta_counts_hz(ii,*,hh) = delta_counts_hz(ii,*,hh)*F_Switch
        ENDIF
    ENDFOR
ENDFOR
print, 'Division by geometric factor completed.'

; Sum over deflection (out-of-ecliptic plane) steps to get a 1-D count distribution
sum_full_data = FLTARR(num_records1,128) 
sum_full_data(*,*) = 0.0
sum_delta_counts = FLTARR(num_records1,128)
sum_delta_counts(*,*) = 0.0
peak_ESA = INTARR(num_records1)
peak_ESA(*) = -1
calc_proton_sum_defl, full_data_hz, delta_counts_hz, sum_full_data, sum_delta_counts, peak_ESA, num_records1
; LBE
;lbe1 = intarr(num_records1)
;FOR ii = 0, num_records1-1 DO BEGIN 
;    a = max(sum_full_data[ii, *], a_i, /nan)
;    lbe1[ii] = a_i
;ENDFOR 
;store_data, 'peak_sum_full', data = {x:epoch1, y:lbe1}

; Divide through by a flat efficiency for both main and s-channel.
; The value for A is based on pre-launch calibration.  The value for B
; is empirical.
IF sat EQ 'A' THEN BEGIN
	sum_full_data = sum_full_data/0.80
	sum_delta_counts = sum_delta_counts/0.80
ENDIF
IF sat EQ 'B' THEN BEGIN
	sum_full_data = sum_full_data/4.0
	sum_delta_counts = sum_delta_counts/4.0
ENDIF
 
print, 'Divison by efficiency completed.'

; Get E/Q and V associated with each ESA step so that distribution functions can be found
; The location of the ESA table must be specified in the crib file.

IF sat EQ 'B' THEN ESA_file = table_location+'ESA_table_B_2010_06_25b.txt' 
IF sat EQ 'A' THEN ESA_file = table_location+'ESA_table_A_2009_08_17.txt' ; V09
;IF sat EQ 'A' THEN ESA_file = table_location+'ESA_table_A_Toni_fall_2012_v3.txt' ; 03/27/2013
openr, in_lun, ESA_file, /GET_LUN
line = " "
energy_S = fltarr(128) ; changed from double to float (August 2009)
energy_S(*) = !values.f_nan
v_S = fltarr(128) ; changed from double to float (August 2009)
v_S(*) = !values.f_nan

readf, in_lun, line   ; read past first lines

FOR i = 64, 127 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    energy_S[i] = parts[1]			;keV/q
    v_S[i] = parts[2]
ENDFOR

close, in_lun
free_lun, in_lun
print, 'Finished reading S channel ESA table.'

IF sat EQ 'B' THEN ESA_file_main = table_location+'Main_channel_B_empirical_v2.txt'  
IF sat EQ 'A' THEN ESA_file_main = table_location+'ESA_table_A_2009_08_17.txt' ; V09
;IF sat EQ 'A' THEN ESA_file_main = table_location+'ESA_table_A_Toni_fall_2012_v3.txt' ; 03/27/2013
openr, in_lun, ESA_file_main, /GET_LUN
line = " "
energy_main = fltarr(128) ; changed from double to float (August 2009)
energy_main(*) = !values.f_nan
v_main = fltarr(128) ; changed from double to float (August 2009)
v_main(*) = !values.f_nan

readf, in_lun, line   ; read past first lines

IF SAT EQ 'B' THEN BEGIN
	FOR i = 0, 127 DO BEGIN
    	readf, in_lun, line
    	parts = STRSPLIT(line, " ,", /EXTRACT)
    	energy_main[i] = parts[1]			;keV/q
    	v_main[i] = parts[2]
	ENDFOR
ENDIF ELSE BEGIN ; SAT IS A
	FOR i = 64, 127 DO BEGIN
    	readf, in_lun, line
    	parts = STRSPLIT(line, " ,", /EXTRACT)
    	energy_main[i] = parts[1]			;keV/q
    	v_main[i] = parts[2]
	ENDFOR
ENDELSE
	
close, in_lun
free_lun, in_lun
print, 'Finished reading main channel ESA table.'


; Multiply by another energy dependent "leakage efficiency"
FOR ii = 0L, num_records1 - 1 DO BEGIN
	FOR jj = 64, 127 DO BEGIN
		IF jj GE s_chan(ii) THEN BEGIN 
		sum_full_data(ii,jj) = sum_full_data(ii,jj)*eff(jj)
		sum_delta_counts(ii,jj) = sum_delta_counts(ii,jj)*eff(jj)
		ENDIF
	ENDFOR
ENDFOR
; LBE
;lbe2 = intarr(num_records1)
;FOR ii = 0, num_records1-1 DO BEGIN 
;    a = max(sum_full_data[ii, *], a_i, /nan)
;    lbe2[ii] = a_i
;ENDFOR 
;store_data, 'peak_sum_full_after_eff', data = {x:epoch1, y:lbe2}

; Now multiply each ESA step by (delta theta*delta phi)/(v^2) to get distribution function
dist_funct = dblarr(128,num_records1)
dist_funct(*,*) = !values.d_nan
peak_dist = intarr(num_records1)
peak_dist(*) = -1
delta_dist_funct = dblarr(128,num_records1)
delta_dist_funct(*,*) = !values.d_nan
FOR i = 0L, num_records1-1 DO BEGIN
    peak_count = 0.0
    FOR j = 0, 127 DO BEGIN
        IF sum_full_data(i, j) GT 0 THEN BEGIN
        	IF j GE s_chan(i) THEN BEGIN
            		dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v_S(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
	    		delta_dist_funct(j,i) = (0.0218*0.7854/((v_S(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_S(j))^2.0)*(0.05*v_S(j))^2.0)
		ENDIF ELSE BEGIN
            		dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v_main(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
	    		delta_dist_funct(j,i) = (0.0218*0.7854/((v_main(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v_main(j))^2.0)*(0.05*v_main(j))^2.0)
		ENDELSE

        ENDIF ELSE BEGIN
            dist_funct(j, i) = 0.0
	    delta_dist_funct(j,i) = 0.0
        ENDELSE

; Average repeated ESA steps
		IF sat EQ 'B' THEN BEGIN
			dist_funct(122,i) = DOUBLE(0.5*(dist_funct(121,i)+dist_funct(123,i)))
			dist_funct(125,i) = DOUBLE(0.5*(dist_funct(124,i)+dist_funct(126,i)))
			delta_dist_funct(122,i) = DOUBLE(0.5*SQRT((delta_dist_funct(121,i))^2.0+(delta_dist_funct(123,i))^2.0))
			delta_dist_funct(125,i) = DOUBLE(0.5*SQRT((delta_dist_funct(124,i))^2.0+(delta_dist_funct(126,i))^2.0))
		ENDIF ELSE BEGIN
			dist_funct(116,i) = DOUBLE(0.5*(dist_funct(115,i)+dist_funct(117,i)))
			dist_funct(118,i) = DOUBLE(0.5*(dist_funct(117,i)+dist_funct(119,i)))
			dist_funct(123,i) = DOUBLE(0.5*(dist_funct(122,i)+dist_funct(124,i)))
			dist_funct(126,i) = DOUBLE(0.5*(dist_funct(125,i)+dist_funct(127,i)))
			delta_dist_funct(116,i) = DOUBLE(0.5*SQRT((delta_dist_funct(115,i))^2.0+(delta_dist_funct(117,i))^2.0))
			delta_dist_funct(118,i) = DOUBLE(0.5*SQRT((delta_dist_funct(117,i))^2.0+(delta_dist_funct(119,i))^2.0))
			delta_dist_funct(123,i) = DOUBLE(0.5*SQRT((delta_dist_funct(122,i))^2.0+(delta_dist_funct(124,i))^2.0))
			delta_dist_funct(126,i) = DOUBLE(0.5*SQRT((delta_dist_funct(125,i))^2.0+(delta_dist_funct(127,i))^2.0))
		ENDELSE
        
        IF (dist_funct(j, i) GT peak_count) THEN peak_count = dist_funct(j, i)
        IF (dist_funct(j, i) GE peak_count) THEN peak_dist(i) = j
    ENDFOR
ENDFOR

sum_full_data = 0 ; added 28 Sept, 2007 to use save memory
sum_delta_counts = 0
;stop
; LBE
;lbe3 = intarr(num_records1)
;FOR ii = 0, num_records1-1 DO BEGIN 
;    a = max(dist_funct[*, ii], a_i, /nan)
;ENDFOR 
;store_data, 'dist_func', data = {x:epoch1, y:lbe3}
;store_data, 'peak_dist', data = {x:epoch1, y:peak_dist}
;stop

; To make it easier for IDL to get a good fit to the curve, multiply the distribution by
; 1e5.  Remember to divide the density by this factor later on!
dist_funct = dist_funct*(1.0E5)
delta_dist_funct = delta_dist_funct*(1.0E5)
print, 'Distribution functions complete.'

; Fit the distribution function versus velocity data to a 1-D Gaussian.
coeff = dblarr(num_records1,3) ; A matrix in which to store the proton fit coefficients
coeff(*,*,*) = !values.d_nan
density = dblarr(num_records1)
density(*) = !values.f_nan
avg_v= dblarr(num_records1)
avg_v(*) = !values.f_nan
temp_K = dblarr(num_records1)
temp_K(*) = !values.f_nan
v_th = dblarr(num_records1)
v_th(*) = !values.f_nan
good_fit = fltarr(num_records1)
good_fit(*) = !values.f_nan
calc_proton_fit_maxwellian, sat, num_records1, coeff, density, avg_v, temp_K, v_th, good_fit, $
	peak_dist, dist_funct, delta_dist_funct, v_S, v_main, std_error, error1, s_chan, epoch1

;Uncomment below to define variables for alpha data and then fit the alpha distribution
coeff_alpha = dblarr(num_records1,3) ; A matrix in which to store the proton fit coefficients
coeff_alpha(*,*,*) = !values.d_nan
density_alpha = dblarr(num_records1)
density_alpha(*) = !values.f_nan
avg_v_alpha= dblarr(num_records1)
avg_v_alpha(*) = !values.f_nan
temp_K_alpha = dblarr(num_records1)
temp_K_alpha(*) = !values.f_nan
v_th_alpha = dblarr(num_records1)
v_th_alpha(*) = !values.f_nan
good_fit_alpha = fltarr(num_records1)
good_fit_alpha(*) = !values.f_nan
expected_peak_channel = INTARR(num_records1)
expected_peak_channel(*) = -1

;calc_proton_alpha_data, v, avg_v, v_th, dist_funct, delta_dist_funct, coeff_alpha, $
;	density_alpha, avg_v_alpha, temp_K_alpha, v_th_alpha, good_fit_alpha, $
;	coeff, num_records1, expected_peak_channel, s_chan, sat, epoch1 

;; Un-comment below to print raw output file
;calc_proton_raw_ascii, sat, epoch1, num_records1, density, avg_v, temp_K, v_th, good_fit, avg_v_alpha, $
;	density_alpha, v_th_alpha, good_fit_alpha, expected_peak_channel

; Check for 3 standard deviations or non-physical results
FOR k = 0L, num_records1 - 1 DO BEGIN
    IF (avg_v(k) GT 2000 OR avg_v(k) LE 230) AND std_error[k] NE 5 THEN std_error(k) = 1 ; changed lower limit to 230 per Toni 3/23/2009 
    IF (density(k) LT 0.001 OR density(k) GE 500) AND std_error[k] NE 5 THEN std_error(k) = 1
    IF (v_th(k) GT 400) AND std_error[k] NE 5 THEN std_error(k) = 1

    IF sat EQ 'B' AND v_th(k) LE 10 THEN BEGIN ; changed to from 12 to 10, June 2010
	IF std_error(k) NE 1 AND std_error[k] NE 5 THEN std_error(k) = 7
    ENDIF

    IF sat EQ 'A' AND v_th(k) LE 10 THEN BEGIN
	IF std_error(k) NE 1 AND std_error[k] NE 5 THEN std_error(k) = 7
    ENDIF

    IF error1(k) GT 0 THEN std_error(k) = 2

    IF FINITE(avg_v(k)) EQ 0 AND std_error(k) NE 5 THEN std_error(k) = 1
    IF FINITE(good_fit(k)) EQ 0 AND std_error[k] NE 5 THEN std_error(k) = 1 ; added Feb 2009
    IF std_error(k) EQ 1 OR std_error[k] EQ 5 THEN BEGIN
	avg_v[k] = !values.f_nan
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan
    ENDIF

    IF std_error(k) EQ 7 THEN BEGIN ; removed chi^2 criteria, KDCS, 12 October 2009
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan ; set to NaN added 7 October 2010 (KDCS)
    ENDIF
    
    IF FINITE(density(k)) EQ 1 THEN BEGIN
        IF k GE 29 AND k LE num_records1-31 THEN BEGIN
	IF FINITE(density(k+1)) EQ 1 AND FINITE(density(k+2)) EQ 1 THEN BEGIN
            mean_density = MEAN(density(k-29:k+30), /NAN)
            mean_avg_v = MEAN(avg_v(k-29:k+30), /NAN)
            mean_v_th = MEAN(v_th(k-29:k+30), /NAN)
	  
            stdev_density = STDDEV(density(k-29:k+30), /NAN)
            stdev_avg_v = STDDEV(avg_v(k-29:k+30), /NAN)
            stdev_v_th = STDDEV(v_th(k-29:k+30), /NAN)

            IF ABS(density(k) - mean_density) GE 3*stdev_density AND std_error(k) NE 1 THEN std_error(k) = 4
            IF ABS(avg_v(k) - mean_avg_v) GE 3*stdev_avg_v AND std_error(k) NE 1 THEN std_error(k) = 4
            IF ABS(v_th(k) - mean_v_th) GE 3*stdev_v_th AND std_error(k) NE 1 THEN std_error(k) = 4
	ENDIF
        ENDIF
    ENDIF
ENDFOR

; take off extra time on either side
temp_start = time_double(time)
temp_stop = temp_start+(num_days*24L*60*60)
wanted_i = where(epoch1 GE temp_start AND epoch1 LT temp_stop, wanted_count)
IF wanted_count LT 1 THEN BEGIN
    print, 'Error: calc_proton_bulk: No valid data.'
    have_data = 0
ENDIF ELSE BEGIN 
    small_epoch1    = epoch1      [wanted_i]
    small_density   = density     [wanted_i]
    small_avg_v     = avg_v       [wanted_i]
    small_temp_K    = temp_K      [wanted_i]
    small_v_th      = v_th        [wanted_i]
    small_MAIN_MAX  = MAIN_MAX    [wanted_i]
    small_S_MAX     = S_MAX       [wanted_i]
    small_std_error = std_error   [wanted_i]
    small_caution   = caution_flag[wanted_i]
    small_good_fit  = good_fit    [wanted_i]
    new_num_records = wanted_count
    angle_error     = intarr(n_elements(small_std_error))
    err5 = where(small_std_error EQ 5, err5_c)
    IF err5_c EQ wanted_count THEN have_data = 0
ENDELSE 

; CHECK FOR PROBLEMS (non-physical results)
; hand-checked bad times
bad_count = 0
IF have_data EQ 1 THEN BEGIN 
    IF sat EQ 'A' AND (time_string(small_epoch1[0]) LT '2016-08-01/00:00:00' OR finite(small_epoch1[0]) EQ 0) THEN BEGIN ;outliers
        check_proton_outliers_a, small_epoch1, bad_i, bad_count
        IF bad_count GT 0 THEN BEGIN 
            small_avg_v    [bad_i] = !values.f_nan
            small_density  [bad_i] = !values.f_nan
            small_temp_K   [bad_i] = !values.f_nan
            small_v_th     [bad_i] = !values.f_nan
            small_good_fit [bad_i] = !values.f_nan
            small_std_error[bad_i] = 8
        ENDIF
    ENDIF 

    IF sat EQ 'B' AND (time_string(small_epoch1[0]) LT '2014-10-01/00:00:00' OR finite(small_epoch1[0]) EQ 0) THEN BEGIN
        check_proton_outliers_b, small_epoch1, bad_i, bad_count
        IF bad_count GT 0 THEN BEGIN 
            small_avg_v    [bad_i] = !values.f_nan
            small_density  [bad_i] = !values.f_nan
            small_temp_K   [bad_i] = !values.f_nan
            small_v_th     [bad_i] = !values.f_nan
            small_good_fit [bad_i] = !values.f_nan
            small_std_error[bad_i] = 8
        ENDIF
    ENDIF 

    SMALL_S_MAX_SCALED = dblarr(new_num_records)
    SMALL_S_MAX_SCALED(*) = !values.f_nan
    
    FOR x = 0L, new_num_records-1 DO BEGIN
        IF SMALL_MAIN_MAX(x) EQ 0 THEN SMALL_MAIN_MAX(x) = !values.f_nan
        IF SMALL_S_MAX(x) EQ 0 THEN SMALL_S_MAX(x) = !values.f_nan		
    ENDFOR
    IF SAT EQ 'A' THEN BEGIN
        SMALL_MAIN_MAX = -23.25+1.5*SMALL_MAIN_MAX ; convert to angle (degrees) from bin number
        SMALL_S_MAX = -19.375 + 1.25*SMALL_S_MAX ; convert to angle (degrees) from bin number
        SMALL_S_MAX_SCALED = 3.0*(SMALL_S_MAX + 3.5) ; modified 27 August, 2008  
                                ; 3.5 if shift center of distribution, 3.0 is to spread out distribution      
        
        FOR k = 0L, new_num_records-1 DO BEGIN
                                ;SMALL_S_MAX_SCALED(k) = 3.0*(SMALL_S_MAX(k) + 1.72) ; old conversion
            
                                ;SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) - (3.6613E-5)*(small_avg_v(k)^2) + $
                                ;  5.2914E-2*small_avg_v(k)-1.5385E1 + 2.0 ; 
            SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) + (-2.0854E-7)*(small_avg_v(k)^3)+(3.0371E-4)*(small_avg_v(k)^2) + $
                             (-1.2454E-1)*small_avg_v(k)+1.0983E1 ; modified 27 August, 2008
                   ; this comes from velocity/energy dependence of leakage
            IF ABS(SMALL_S_MAX(k)) GT 20.0 THEN SMALL_S_MAX(k) = !values.f_nan ; added 02 September, 2008
        ENDFOR
    ENDIF ELSE BEGIN
        SMALL_MAIN_MAX = 31.0 - SMALL_MAIN_MAX
        SMALL_S_MAX = 31.0 - SMALL_S_MAX ; flip distribution because PLASTIC-B is upside-down
        
        SMALL_MAIN_MAX = -23.25+1.5*SMALL_MAIN_MAX
        SMALL_S_MAX = -19.375 + 1.25*SMALL_S_MAX
        SMALL_S_MAX_SCALED = 1.75*(SMALL_S_MAX - 4.62) ; modified 14 August, 2008
        FOR k = 0L, new_num_records-1 DO BEGIN
            SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) + (-2.6404E-7)*(small_avg_v(k)^3) + (4.3726E-4)*(small_avg_v(k)^2) + $
                             (-2.3922E-1)*small_avg_v(k)+4.2639E1
            IF ABS(SMALL_S_MAX(k)) GT 20.0 THEN SMALL_S_MAX(k) = !values.f_nan ; added 02 September, 2008
        ENDFOR 
    ENDELSE

; Now, need to remove times when entrance system is closed
; --------------------------------------------------------
; get hk data
    ;sat_hk = [28]
    ;pl_plot_hk_from_crib, sat_hk, sat, /INST_COORD
    ;temp_string = 'ST'+sat+'_Analog_HK_esa_cm_dc'
    ;get_data, temp_string, hk_time, hk_data
    sat_hk = ['esa_cm_dc_eng']
    get_pla_hk, sat, sat_hk
    temp_string = 'ST'+sat+'_Analog_HK_esa_cm_dc_eng'
    get_data, temp_string, hk_time, hk_data
; line up data
    good_i = where(finite(hk_time) EQ 1, good_count)
    IF good_count GT 0 THEN BEGIN 
        hk_time = hk_time[good_i] 
        hk_data = hk_data[good_i] 
    ENDIF ELSE stop             ; no good data
    good_i = where(hk_time GT 0, good_count)
    IF good_count GT 0 THEN BEGIN 
        hk_time = hk_time[good_i] 
        hk_data = hk_data[good_i] 
    ENDIF ELSE stop
    IF array_equal(small_epoch1, hk_time) NE 1 THEN BEGIN ; make sure time arrays line up
        temp_hk_time = dblarr(n_elements(small_epoch1))
        temp_hk_data = dblarr(n_elements(small_epoch1))
        hh = 0L
        rr = 0L
        WHILE rr LT n_elements(small_epoch1) AND hh LT n_elements(hk_time) DO BEGIN 
           ;IF small_epoch1[rr] EQ hk_time[hh] THEN BEGIN ; changed this 04/10/2014
            IF abs(small_epoch1[rr] - hk_time[hh]) LT 10 THEN BEGIN ; OK to have more hk values than ra
                temp_hk_time[rr] = hk_time[hh]
                temp_hk_data[rr] = hk_data[hh]
                rr = rr + 1
                hh = hh + 1
            ENDIF ELSE IF small_epoch1[rr] LT hk_time[hh] THEN rr = rr + 1 ELSE hh = hh + 1
        ENDWHILE 
        hk_time = temp_hk_time
        hk_data = temp_hk_data
    ENDIF 
; remove any indices that were added (where there was ra data but no hk data
    good_i = where(hk_time NE 0, good_count)
    IF good_count NE n_elements(hk_time) THEN BEGIN
        hk_time = hk_time[good_i]
        hk_data = hk_data[good_i]
    ENDIF 
; check for entrance system closed
    num_records = n_elements(small_epoch1)
    check_indices = intarr(num_records) ; make sure we don't overwrite data
    good_i = where(finite(hk_time) EQ 1, good_count) ; good_i[0] is where we start valid data
    temp_start_i = good_i[0]
    hh = 0                      ; for hk_time
    FOR kk = 0, num_records-1 DO BEGIN 
        IF finite(hk_time[hh+temp_start_i]) EQ 1 THEN BEGIN 
            good_i = where(hk_time EQ hk_time[hh+temp_start_i], count)
            IF count NE 1 THEN BEGIN ; check error cases
                IF count EQ 0 AND small_std_error[kk] GT 0 THEN check_indices[kk] = check_indices[kk]+1 $ ; no hk at this time, but error already set
                ELSE IF count EQ 0 THEN BEGIN ; check if nearest value was ok (data gap)
                    check_indices[kk] = check_indices[kk]+1
                    offset = 1
                    IF hh-offset GT 0 THEN last_i = where(hk_time EQ hk_time[hh-offset], last_count) ELSE last_count = 0
                    IF hh+offset LT num_records-1 THEN next_i = where(hk_time EQ hk_time[hh+offset], next_count) ELSE $
                      next_count = 0
                    WHILE last_count EQ 0 AND next_count EQ 0 DO BEGIN 
                        offset = offset+1
                        IF hh-offset GT 0 THEN last_i = where(hk_time EQ hk_time[hh-offset], last_count) ELSE last_count = 0
                        IF hh+offset LT num_records-1 THEN next_i = where(hk_time EQ hk_time[hh+offset], next_count) ELSE $
                          next_count = 0
                    ENDWHILE 
                    IF last_count GT 0 THEN BEGIN
                        IF hk_data[last_i] LE 50 THEN BEGIN 
                            small_avg_v[kk] = !values.f_nan
                            small_density[kk] = !values.f_nan
                            small_temp_K[kk] = !values.f_nan
                            small_v_th[kk] = !values.f_nan
                            small_s_max[kk] = !values.f_nan
                            small_std_error[kk] = 5
                            small_good_fit[kk] = !values.f_nan
                        ENDIF 
                    ENDIF ELSE IF next_count GT 0 THEN BEGIN 
                        IF hk_data[next_i] LE 50 THEN BEGIN 
                            small_avg_v[kk] = !values.f_nan
                            small_density[kk] = !values.f_nan
                            small_temp_K[kk] = !values.f_nan
                            small_v_th[kk] = !values.f_nan
                            small_s_max[kk] = !values.f_nan
                            small_std_error[kk] = 5
                            small_good_fit[kk] = !values.f_nan
                        ENDIF 
                    ENDIF ELSE stop ; no valid values found in either direction, what should I do?
                ENDIF ELSE stop ; count gt 1?
            ENDIF ELSE BEGIN    ; count eq 1
                check_indices[kk] = check_indices[kk]+1
                IF hk_data[good_i] LE 50 THEN BEGIN
                    small_avg_v[kk] = !values.f_nan
                    small_density[kk] = !values.f_nan
                    small_temp_K[kk] = !values.f_nan
                    small_v_th[kk] = !values.f_nan
                    small_s_max[kk] = !values.f_nan
                    small_std_error[kk] = 5
                    small_good_fit[kk] = !values.f_nan
                ENDIF
            ENDELSE 
        ENDIF ELSE IF finite(small_epoch1[kk]) EQ 0 THEN BEGIN ; time is invalid in error and ra
            check_indices[kk] = check_indices[kk]+1
            small_avg_v[kk] = !values.f_nan
            small_density[kk] = !values.f_nan
            small_temp_K[kk] = !values.f_nan
            small_v_th[kk] = !values.f_nan
            small_s_max[kk] = !values.f_nan
            small_std_error[kk] = 5
            small_good_fit[kk] = !values.f_nan
        ENDIF 
        IF kk LT num_records-1 AND hh LT n_elements(hk_time)-1 THEN IF small_epoch1[kk+1] GT hk_time[hh] THEN hh = hh + 1 ; 8/3/09
    ENDFOR 
    IF array_equal(check_indices, 1) NE 1 THEN stop

;; Get roll information and spacecraft coordinates
    good_i = where(finite(small_epoch1) EQ 1, good_count)
    IF good_count GT 0 THEN BEGIN
        temp_epoch_string = time_string(small_epoch1[good_i])
        strput, temp_epoch_string, 'T', 10 ; change time string format for SolarSoft
        roll = get_stereo_roll(temp_epoch_string, sat, yaw, pitch, system = 'HERTN', /POST_CONJUNCTION)
        attitude_flag = intarr(num_records, 3) ; roll, yaw, pitch
        flag_i = where(roll  GT 1 OR roll  LT -1, count)
        IF count GT 0 THEN attitude_flag[flag_i, 0] = 1
        temp_i = where(yaw LT 0, temp_count)
        IF temp_count GT 0 THEN yaw[temp_i] = yaw[temp_i] + 360
        flag_i = where(yaw GT 181 OR yaw LT 179, count)
        IF count GT 0 THEN attitude_flag[flag_i, 1] = 1
        flag_i = where(pitch GT 1 OR pitch LT -1, count)
        IF count GT 0 THEN attitude_flag[flag_i, 2] = 1
        carr_num = get_stereo_carr_rot(temp_epoch_string, sat)
        temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'carrington', /degrees)
        carr_lon = reform(temp_value[1, *])
        temp_i = where(carr_lon LT 0, temp_c)
        IF temp_c GT 0 THEN carr_lon[temp_i] = carr_lon[temp_i] + 360 ; range should be 0-360
        helio_dist = reform(temp_value[0, *])
        temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'HEE', /degrees)
        hee_lon = reform(temp_value[1, *])
        hee_lat = reform(temp_value[2, *])
        temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'HEEQ', /degrees)
        heeq_lon = reform(temp_value[1, *])
        heeq_lat = reform(temp_value[2, *])
        temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'HCI', /degrees)
        hci_lon = reform(temp_value[1, *])
        temp_i = where(hci_lon LT 0, temp_c)
        IF temp_c GT 0 THEN hci_lon[temp_i] = hci_lon[temp_i] + 360 ; range should be 0-360
        hci_lat = reform(temp_value[2, *])

; check for N/S outliers
        bad_count = 0
        IF sat EQ 'A' THEN BEGIN 
            check_ns_outliers_a, small_epoch1, bad_i, bad_count
            ; V09
            ;bad_i = where(time_string(small_epoch1) EQ '2012-04-30/04:51:02' OR $ ; KDCS 06/08/2012
            ;              time_string(small_epoch1) EQ '2012-04-30/04:54:02' OR $ ; KDCS 06/08/2012
            ;              time_string(small_epoch1) EQ '2012-04-30/04:59:02', $ ; KDCS 06/08/2012
            ;              bad_count)
        ENDIF ELSE IF sat EQ 'B' THEN BEGIN ; V09
            check_ns_outliers_b, small_epoch1, bad_i, bad_count
        ENDIF 
        IF bad_count GT 0 THEN BEGIN 
            SMALL_S_MAX    [bad_i] = !values.f_nan
            small_std_error[bad_i] = 8
            angle_error    [bad_i] = 1
        ENDIF
    ENDIF 
ENDIF 

; save 1 minute data to file
; ----------------------------
IF have_data EQ 1 THEN BEGIN
    small_density_2 = small_density
    IF sat EQ 'A' THEN kev_time = small_epoch1 + 41.6
    IF sat EQ 'B' THEN kev_time = small_epoch1 + 42.0
    ;percent_diff = dblarr(N_ELEMENTS(small_density_2))
    ;percent_diff(*) = !values.f_nan
        time_i = 0
    WHILE finite(small_epoch1[time_i]) EQ 0 DO BEGIN 
        time_i = time_i + 1
    ENDWHILE 
    label_time = time_struct(small_epoch1(time_i))
    label_year = STRING(label_time.year, FORMAT = '(I04)')
    label_month = STRING(label_time.month, FORMAT = '(I02)')
    label_day = STRING(label_time.date, FORMAT = '(I02)')
 


    good_indices = intarr(n_elements(small_density_2)) ; set to 1 if data should be included
    FOR p = 0L, N_ELEMENTS(small_density_2) - 1 DO BEGIN
       	  ;  percent_diff(p) = (density_1(p)-small_density_2(p))/small_density_2(p)
          ; IF finite(density_1[p]) EQ 0 OR finite(small_density_2[p]) EQ 0 THEN caution_flag[p] = 2
        ;IF ABS(percent_diff(p)) GE 0.1 THEN caution_flag(p) = 2
        ;IF ABS(percent_diff(p)) GE 0.05 AND ABS(percent_diff(p)) LT 0.1 THEN caution_flag(p) = 1 
        
	IF small_std_error(p) EQ 1 OR small_std_error(p) EQ 5 THEN BEGIN
            small_density(p) = !values.f_nan
            small_avg_v(p) = !values.f_nan
            small_temp_K(p) = !values.f_nan
            small_v_th(p) = !values.f_nan
            SMALL_S_MAX(p) = !values.f_nan
	    small_good_fit(p) = !values.f_nan
        ENDIF	
    
	
        IF finite(small_avg_v(p)) EQ 1 THEN good_indices[p] = 1 ; CHANGED from small_density to small_avg_v FEB 2009
 	
    ENDFOR	

                                ; Check for times when need to fill in 5-min data
    small_caution_orig = small_caution
    IF sat EQ 'A' AND (time_string(epoch1[0]) LT '2012-07-26/00:00:00' OR finite(epoch1[0]) EQ 0) THEN BEGIN 
        openr, in_lun, table_location+'STA_proton_5min_20130520.txt', /get_lun
        readf, in_lun, line     ; header
        WHILE eof(in_lun) EQ 0 DO BEGIN
            readf, in_lun, line
            parts = strsplit(line, /extract)
            temp_time = time_double(parts[5])
            wanted_i = where(small_epoch1 GE temp_time AND (small_epoch1 LT (temp_time+300)), time_count)
            IF time_count GT 0 THEN BEGIN
                den_i = where(finite(small_density    [wanted_i]) EQ 0, den_count)
                vel_i = where(finite(small_avg_v      [wanted_i]) EQ 0, vel_count)
                tem_i = where(finite(small_temp_K     [wanted_i]) EQ 0, tem_count)
                vth_i = where(finite(small_v_th       [wanted_i]) EQ 0, vth_count)
                n_s_i = where(finite(SMALL_S_MAX      [wanted_i]) EQ 0, n_s_count)
                FOR temp_i = 0, den_count-1 DO BEGIN ; if any products are missing (other than N/S), replace all
                    small_density  [wanted_i[den_i[temp_i]]] = float(parts[ 6])
                    small_avg_v    [wanted_i[den_i[temp_i]]] = float(parts[ 7])
                    small_temp_K   [wanted_i[den_i[temp_i]]] = float(parts[ 8])
                    small_v_th     [wanted_i[den_i[temp_i]]] = float(parts[ 9])
                    SMALL_S_MAX    [wanted_i[den_i[temp_i]]] = float(parts[10])
                    small_good_fit [wanted_i[den_i[temp_i]]] = float(parts[11])
                    small_std_error[wanted_i[den_i[temp_i]]] = float(parts[12])
                    IF array_equal(finite(float(parts[6:9])), 0) NE 1 THEN small_caution[wanted_i[den_i[temp_i]]] = 5
                ENDFOR 
                FOR temp_i = 0, vel_count-1 DO BEGIN 
                    small_density  [wanted_i[vel_i[temp_i]]] = float(parts[ 6])
                    small_avg_v    [wanted_i[vel_i[temp_i]]] = float(parts[ 7])
                    small_temp_K   [wanted_i[vel_i[temp_i]]] = float(parts[ 8])
                    small_v_th     [wanted_i[vel_i[temp_i]]] = float(parts[ 9])
                    SMALL_S_MAX    [wanted_i[vel_i[temp_i]]] = float(parts[10])
                    small_good_fit [wanted_i[vel_i[temp_i]]] = float(parts[11])
                    small_std_error[wanted_i[vel_i[temp_i]]] = float(parts[12])
                    IF array_equal(finite(float(parts[6:9])), 0) NE 1 THEN small_caution[wanted_i[den_i[temp_i]]] = 5
                ENDFOR 
                FOR temp_i = 0, tem_count-1 DO BEGIN 
                    small_density  [wanted_i[tem_i[temp_i]]] = float(parts[ 6])
                    small_avg_v    [wanted_i[tem_i[temp_i]]] = float(parts[ 7])
                    small_temp_K   [wanted_i[tem_i[temp_i]]] = float(parts[ 8])
                    small_v_th     [wanted_i[tem_i[temp_i]]] = float(parts[ 9])
                    SMALL_S_MAX    [wanted_i[tem_i[temp_i]]] = float(parts[10])
                    small_good_fit [wanted_i[tem_i[temp_i]]] = float(parts[11])
                    small_std_error[wanted_i[tem_i[temp_i]]] = float(parts[12])
                    IF array_equal(finite(float(parts[6:9])), 0) NE 1 THEN small_caution[wanted_i[den_i[temp_i]]] = 5
                ENDFOR 
                FOR temp_i = 0, vth_count-1 DO BEGIN 
                    small_density  [wanted_i[vth_i[temp_i]]] = float(parts[ 6])
                    small_avg_v    [wanted_i[vth_i[temp_i]]] = float(parts[ 7])
                    small_temp_K   [wanted_i[vth_i[temp_i]]] = float(parts[ 8])
                    small_v_th     [wanted_i[vth_i[temp_i]]] = float(parts[ 9])
                    SMALL_S_MAX    [wanted_i[vth_i[temp_i]]] = float(parts[10])
                    small_good_fit [wanted_i[vth_i[temp_i]]] = float(parts[11])
                    small_std_error[wanted_i[vth_i[temp_i]]] = float(parts[12])
                    IF array_equal(finite(float(parts[6:9])), 0) NE 1 THEN small_caution[wanted_i[den_i[temp_i]]] = 5
                ENDFOR 
            ENDIF 
        ENDWHILE 
        close, in_lun
        free_lun, in_lun
    ENDIF 

    labels = ' '
    store_data, 'density',                   data = {x:small_epoch1, y:small_density  }, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, 'speed',                     data = {x:small_epoch1, y:small_avg_v    }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'temperature',               data = {x:small_epoch1, y:small_temp_K   }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'v_thermal',                 data = {x:small_epoch1, y:small_v_th     }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'north_south',               data = {x:small_epoch1, y:SMALL_S_MAX    }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, '1_keV_time',                data = {x:small_epoch1, y:kev_time       }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'error_code',                data = {x:small_epoch1, y:small_std_error}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'caution_code',              data = {x:small_epoch1, y:small_caution  }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'attitude_flag',             data = {x:small_epoch1, y:attitude_flag  }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'reduced_chi_squared',       data = {x:small_epoch1, y:small_good_fit }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_rotation',       data = {x:small_epoch1, y:carr_num       }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_longitude',      data = {x:small_epoch1, y:carr_lon       }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'heliocentric_distance',     data = {x:small_epoch1, y:helio_dist     }, dlim = {ylog:0, labels:labels, panel_size:2.}
    
    store_data, 'spacecraft_longitude_HEE',  data = {x:small_epoch1, y:hee_lon        }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEE',   data = {x:small_epoch1, y:hee_lat        }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEEQ', data = {x:small_epoch1, y:heeq_lon       }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEEQ',  data = {x:small_epoch1, y:heeq_lat       }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HCI',  data = {x:small_epoch1, y:hci_lon        }, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HCI',   data = {x:small_epoch1, y:hci_lat        }, dlim = {ylog:0, labels:labels, panel_size:2.}

    print, 'Calculating hourly averages.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;Calculate hourly averages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    k = FIX(new_num_records/60)+1
    small_density_copy1 = small_density
    small_avg_v_copy1 = small_avg_v
    small_temp_K_copy1 = small_temp_K
    small_v_th_copy1 = small_v_th
    SMALL_S_MAX_copy1 = SMALL_S_MAX

    hour_start_time = dblarr(k)
    hour_start_time(*) = !values.d_nan
    hour_small_density = dblarr(k)
    hour_small_density(*) = !values.f_nan
    hour_speed = dblarr(k)
    hour_speed(*) = !values.f_nan
    hour_temp = dblarr(k)
    hour_temp(*) = !values.f_nan
    hour_small_v_th = dblarr(k)
    hour_small_v_th(*) = !values.f_nan
    hour_count = dblarr(k)
    hour_count(*) = !values.f_nan
    hour_count_v = dblarr(k)
    hour_count_v(*) = !values.f_nan
    hour_count_ns = dblarr(k)
    hour_count_ns(*) = !values.f_nan
    hour_ns = dblarr(k)
    hour_ns(*) = !values.f_nan
    hour_error = intarr(k)
    hour_caution = intarr(k)
    hour_attitude = intarr(k, 3)
    hour_red_chi = dblarr(k)
    hour_red_chi(*) = !values.f_nan
    hour_carr_num = dblarr(k)
    hour_carr_num(*) = !values.f_nan
    hour_carr_lon = dblarr(k)
    hour_carr_lon(*) = !values.f_nan
    hour_helio_dist = dblarr(k)
    hour_helio_dist(*) = !values.f_nan
    hour_hee_lon = dblarr(k)
    hour_hee_lon(*) = !values.f_nan
    hour_hee_lat = dblarr(k)
    hour_hee_lat(*) = !values.f_nan
    hour_heeq_lon = dblarr(k)
    hour_heeq_lon(*) = !values.f_nan
    hour_heeq_lat = dblarr(k)
    hour_heeq_lat(*) = !values.f_nan
    hour_hci_lon = dblarr(k)
    hour_hci_lon(*) = !values.f_nan
    hour_hci_lat = dblarr(k)
    hour_hci_lat(*) = !values.f_nan
    ih = 0 ; index into 1 hour data
    found_first = 0
    on_first = 1
    FOR im = 0L, n_elements(small_epoch1)-1 DO BEGIN ; index into 1 minute data
        now_time = time_struct(small_epoch1(im))
        IF now_time.year NE 1970 THEN BEGIN ; skip invalid times
            IF found_first EQ 0 THEN BEGIN 
                base_1hr = now_time.hour
            ENDIF 
            now_1hr = now_time.hour
            IF now_time.hour NE base_1hr OR found_first EQ 0 THEN BEGIN ; new base
                found_first = 1
                                ;IF ih GT 0 THEN BEGIN ; write old variables
                IF on_first EQ 0 THEN BEGIN ; write old variables
                    hour_error[ih] = temp_error
                    hour_caution[ih] = temp_caution
                    hour_attitude[ih, *] = temp_attitude
                    IF n_elements(temp_red_chi) GT 0 AND finite(temp_red_chi) EQ 1 THEN hour_red_chi[ih] = temp_red_chi/sixty_count ; corrected 29 Sept 2010
                    IF n_elements(temp_density) GT 0 AND FINITE(temp_density) EQ 1 THEN  hour_small_density[ih] = temp_density/sixty_count
                    IF n_elements(temp_speed) GT 0 AND FINITE(temp_speed) EQ 1 THEN hour_speed[ih] = temp_speed/sixty_count_v
                    IF n_elements(temp_temp) GT 0 AND FINITE(temp_temp) EQ 1 THEN hour_temp[ih] = temp_temp/sixty_count
                    IF n_elements(temp_v_th) GT 0 AND FINITE(temp_v_th) EQ 1 THEN hour_small_v_th[ih] = temp_v_th/sixty_count
                    IF n_elements(temp_ns) GT 0 AND FINITE(temp_ns) EQ 1 THEN hour_ns[ih] = temp_ns/sixty_count_ns
                    hour_count[ih] = sixty_count
                    hour_count_v[ih] = sixty_count_v
                    hour_count_ns[ih] = sixty_count_ns
                ENDIF 
                base_1hr = now_1hr
                IF on_first EQ 1 THEN on_first = 0 ELSE ih = ih + 1 ; so don't get empty first record
                IF ih EQ n_elements(hour_start_time) THEN BEGIN ; need to resize arrays
                    tmp_arr  = intarr(10)
                    tmp2_arr = intarr(10, 3)
                    hour_error = [hour_error, tmp_arr]
                    hour_caution = [hour_caution, tmp_arr]
                    hour_attitude = [hour_attitude, tmp2_arr]
                    tmp_arr = dblarr(10)
                    tmp_arr[*] = !values.f_nan
                    hour_count = [hour_count, tmp_arr]
                    hour_count_v = [hour_count_v, tmp_arr]
                    hour_count_ns = [hour_count_ns, tmp_arr]
                    hour_start_time = [hour_start_time, tmp_arr]
                    hour_red_chi = [hour_red_chi, tmp_arr]
                    hour_carr_num = [hour_carr_num, tmp_arr]
                    hour_carr_lon = [hour_carr_lon, tmp_arr]
                    hour_helio_dist = [hour_helio_dist, tmp_arr]
                    hour_hee_lon = [hour_hee_lon, tmp_arr]
                    hour_hee_lat = [hour_hee_lat, tmp_arr]
                    hour_heeq_lon = [hour_heeq_lon, tmp_arr]
                    hour_heeq_lat = [hour_heeq_lat, tmp_arr]
                    hour_hci_lon = [hour_hci_lon, tmp_arr]
                    hour_hci_lat = [hour_hci_lat, tmp_arr]
                    hour_small_density = [hour_small_density, tmp_arr]
                    hour_speed = [hour_speed, tmp_arr]
                    hour_temp = [hour_temp, tmp_arr]
                    hour_small_v_th = [hour_small_v_th, tmp_arr]
                    hour_ns = [hour_ns, tmp_arr]
                ENDIF 
                ;hour_start_time[ih] = time_double(string(now_time.year, format = '(I4.4)')+'-'+string(now_time.month, format = '(I2.2)')+'-'+$
                ;                                  string(now_time.date, format = '(I2.2)')+'/'+string(now_time.hour, format = '(I2.2)')+$
                ;                                  ':00:00')
                hour_start_time[ih] = small_epoch1[im] ; LBE changed 9/11/09 to make actual start time
                hour_carr_num[ih] = carr_num[im]
                hour_carr_lon[ih] = carr_lon[im]
                hour_helio_dist[ih] = helio_dist[im]
                hour_hee_lon[ih] = hee_lon[im]
                hour_hee_lat[ih] = hee_lat[im]
                hour_heeq_lon[ih] = heeq_lon[im]
                hour_heeq_lat[ih] = heeq_lat[im]
                hour_hci_lon[ih] = hci_lon[im]
                hour_hci_lat[ih] = hci_lat[im]
                IF finite(small_density_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) $
                  THEN temp_density = small_density_copy1[im] ELSE temp_density = !values.f_nan 
                IF finite(small_avg_v_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) $
                  THEN temp_speed = small_avg_v_copy1[im] ELSE temp_speed = !values.f_nan
                IF finite(small_temp_K_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) $
                  THEN temp_temp = small_temp_K_copy1[im] ELSE temp_temp = !values.f_nan
                IF finite(small_v_th_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) $
                  THEN temp_v_th = small_v_th_copy1[im] ELSE temp_v_th = !values.f_nan
                IF finite(SMALL_S_MAX_copy1[im]) EQ 1 AND small_std_error[im] NE 5 AND small_std_error[im] NE 8 $
                  THEN temp_ns = SMALL_S_MAX_copy1[im] ELSE temp_ns = !values.f_nan
                IF finite(small_good_fit[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5  AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) $
                  THEN temp_red_chi = small_good_fit[im] ELSE temp_red_chi = !values.f_nan ; corrected 29 Sept. 2010
                temp_caution = small_caution_orig[im]
                temp_attitude = attitude_flag[im, *]
                IF small_std_error[im] EQ 5 OR small_std_error[im] EQ 8 THEN temp_error = 0 $
                  ELSE temp_error = small_std_error[im]
                IF FINITE(temp_density) EQ 1 THEN sixty_count = 1.0 ELSE sixty_count = 0.0  
                IF FINITE(temp_speed) EQ 1 THEN sixty_count_v = 1.0 ELSE sixty_count_v = 0.0 
                IF FINITE(temp_ns) EQ 1 THEN sixty_count_ns = 1.0 ELSE sixty_count_ns = 0.0
            ENDIF ELSE BEGIN    ; further in same 1 hour space
                IF finite(small_density_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) THEN $
                  IF FINITE(temp_density) EQ 0 THEN temp_density = small_density_copy1[im] $
                  ELSE temp_density = temp_density + small_density_copy1[im]
                IF finite(small_avg_v_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) THEN $
                  IF FINITE(temp_speed) EQ 0 THEN temp_speed = small_avg_v_copy1[im] $
                  ELSE temp_speed = temp_speed + small_avg_v_copy1[im]
                IF finite(small_temp_K_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) THEN $
                  IF FINITE(temp_temp) EQ 0 THEN temp_temp = small_temp_K_copy1[im] $
                  ELSE temp_temp = temp_temp + small_temp_K_copy1[im]
                IF finite(small_v_th_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) THEN $
                  IF FINITE(temp_v_th) EQ 0 THEN temp_v_th = small_v_th_copy1[im] $
                  ELSE temp_v_th = temp_v_th + small_v_th_copy1[im]
                IF finite(SMALL_S_MAX_copy1[im]) EQ 1 AND small_std_error[im] NE 5 AND small_std_error[im] NE 8 THEN IF FINITE(temp_ns) EQ 0 THEN $
                  temp_ns = SMALL_S_MAX_copy1[im] ELSE temp_ns = temp_ns + SMALL_S_MAX_copy1[im]
                IF finite(small_good_fit[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) THEN $
                  IF finite(temp_red_chi) EQ 0 THEN temp_red_chi = small_good_fit[im] $
                  ELSE temp_red_chi = temp_red_chi + small_good_fit[im] ; corrected 29 Sept. 2010
                IF small_caution_orig[im] EQ 2 THEN temp_caution = 2
                IF small_caution_orig[im] EQ 1 AND temp_caution NE 2 THEN temp_caution = 1
                IF attitude_flag[im, 0] EQ 1 THEN temp_attitude[0] = 1
                IF attitude_flag[im, 1] EQ 1 THEN temp_attitude[1] = 1
                IF attitude_flag[im, 2] EQ 1 THEN temp_attitude[2] = 1
;; check which errors to carry over (priority is 3,7,4,6,2,1,8,5)
;; change starting 8/2014 so 5 & 8 don't go in average
                CASE small_std_error[im] OF
                    3: temp_error = 3
                    7: IF (temp_error NE 3) THEN temp_error = 7
                    4: IF (temp_error NE 3) AND (temp_error NE 7) THEN temp_error = 4
                    6: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) THEN temp_error = 6
                    2: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) THEN temp_error = 2
                    1: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                      AND (temp_error NE 2) THEN temp_error = 1
                    ;8: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                    ;  AND (temp_error NE 2) AND (temp_error NE 1) THEN temp_error = 8
                    8: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                      AND (temp_error NE 2) AND (temp_error NE 1) THEN temp_error = 0
                    ;5: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                    ;  AND (temp_error NE 2) AND (temp_error NE 1) AND (temp_error NE 8) THEN temp_error = 5
                    5: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                      AND (temp_error NE 2) AND (temp_error NE 1) AND (temp_error NE 8) THEN temp_error = 0
                    0: temp_error = temp_error
                ENDCASE 
                IF (FINITE(small_density_copy1[im]) EQ 1) AND (small_density_copy1[im] NE -1) AND $
                  small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) THEN $
                  sixty_count = sixty_count+1.0 ; Make sure time step has a valid result
                IF (FINITE(small_avg_v_copy1[im]) EQ 1) AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 AND $
                  ((small_std_error[im] NE 8) OR (small_std_error[im] EQ 8 AND angle_error[im] EQ 1)) $
                  THEN sixty_count_v = sixty_count_v+1.0
                IF (FINITE(SMALL_S_MAX_copy1[im]) EQ 1) AND small_std_error[im] NE 5 AND small_std_error[im] NE 8 $
                  THEN sixty_count_ns = sixty_count_ns+1.0
            ENDELSE 
        ENDIF 
    ENDFOR 
;; add final value
    IF ih EQ n_elements(hour_error)         THEN hour_error = [hour_error, temp_error]          ELSE hour_error[ih] = temp_error
    IF ih EQ n_elements(hour_caution)       THEN hour_caution = [hour_caution, temp_caution]    ELSE hour_caution[ih] = temp_caution
    IF ih EQ n_elements(hour_attitude)      THEN hour_attitude = [hour_attitude, temp_attitude] ELSE hour_attitude[ih, *] = temp_attitude
    IF n_elements(temp_density) GT 0 THEN $
      IF ih EQ n_elements(hour_small_density) THEN hour_small_density = [hour_small_density, temp_density/sixty_count] ELSE $
      hour_small_density[ih] = temp_density/sixty_count
    IF n_elements(temp_speed) GT 0 THEN $
      IF ih EQ n_elements(hour_speed) THEN hour_speed = [hour_speed, temp_speed/sixty_count_v] ELSE hour_speed[ih] = temp_speed/sixty_count_v
    IF n_elements(temp_temp) GT 0 THEN $
      IF ih EQ n_elements(hour_temp)  THEN hour_temp = [hour_temp, temp_temp/sixty_count]      ELSE hour_temp[ih] = temp_temp/sixty_count
    IF n_elements(temp_v_th) GT 0 THEN $
      IF ih EQ n_elements(hour_small_v_th) THEN hour_small_v_th = [hour_small_v_th, temp_v_th/sixty_count] ELSE $
      hour_small_v_th[ih] = temp_v_th/sixty_count
    IF n_elements(temp_ns) GT 0 THEN $
      IF ih EQ n_elements(hour_ns) THEN hour_ns = [hour_ns, temp_ns/sixty_count_ns]         ELSE hour_ns[ih] = temp_ns/sixty_count_ns
    IF n_elements(temp_red_chi) GT 0 THEN $
      IF ih EQ n_elements(hour_red_chi) THEN hour_red_chi = [hour_red_chi, temp_red_chi/sixty_count] $
      ELSE hour_red_chi[ih] = temp_red_chi/sixty_count ; corrected 29 Sept. 2010
    IF ih EQ n_elements(hour_count)    THEN hour_count = [hour_count, sixty_count]          ELSE hour_count[ih] = sixty_count
    IF ih EQ n_elements(hour_count_v)  THEN hour_count_v = [hour_count_v, sixty_count_v]    ELSE hour_count_v[ih] = sixty_count_v
    IF ih EQ n_elements(hour_count_ns) THEN hour_count_ns = [hour_count_ns, sixty_count_ns] ELSE hour_count_ns[ih] = sixty_count_ns

    good_indices = intarr(n_elements(hour_start_time)) ; set to 1 if record should be included
    found_bad_time = 0
    FOR p = 0L, N_ELEMENTS(hour_start_time) - 1 DO BEGIN
        IF finite(hour_start_time(p)) EQ 1 THEN good_indices[p] = 1 ELSE found_bad_time = 1
    ENDFOR
    IF found_bad_time EQ 1 THEN BEGIN
        good_i = where(good_indices EQ 1, good_count)
        IF good_count GT 0 THEN BEGIN 
            hour_start_time    = hour_start_time   [good_i]
            hour_small_density = hour_small_density[good_i]
            hour_speed         = hour_speed        [good_i]
            hour_temp          = hour_temp         [good_i]
            hour_small_v_th    = hour_small_v_th   [good_i]
            hour_ns            = hour_ns           [good_i]
            hour_error         = hour_error        [good_i]
            hour_caution       = hour_caution      [good_i]
            hour_attitude      = hour_attitude     [good_i, *]
            hour_red_chi       = hour_red_chi      [good_i]
            hour_count         = hour_count        [good_i]
            hour_count_v       = hour_count_v      [good_i]
            hour_count_ns      = hour_count_ns     [good_i]
            hour_carr_num      = hour_carr_num     [good_i]
            hour_carr_lon      = hour_carr_lon     [good_i]
            hour_helio_dist    = hour_helio_dist   [good_i]
            hour_hee_lon       = hour_hee_lon      [good_i]
            hour_hee_lat       = hour_hee_lat      [good_i]
            hour_heeq_lon      = hour_heeq_lon     [good_i]
            hour_heeq_lat      = hour_heeq_lat     [good_i]
            hour_hci_lon       = hour_hci_lon      [good_i]
            hour_hci_lat       = hour_hci_lat      [good_i]
        ENDIF ELSE stop         ; no valid times!
    ENDIF 

    ; check for caution eq 5
    FOR temp_i = 0, n_elements(hour_start_time)-2 DO BEGIN
        wanted_i = where(small_epoch1 GE hour_start_time[temp_i] AND small_epoch1 LT hour_start_time[temp_i+1], count)
        five_i = where(small_caution[wanted_i] EQ 5, five_count)
        IF five_count GT 0 THEN hour_caution[temp_i] = 5
    ENDFOR 
    wanted_i = where(small_epoch1 GE hour_start_time[temp_i], count)
    five_i = where(small_caution[wanted_i] EQ 5, five_count)
    IF five_count GT 0 THEN hour_caution[temp_i] = 5

    store_data, 'density_1hr', data = {x:hour_start_time, y:hour_small_density}, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, 'speed_1hr', data = {x:hour_start_time, y:hour_speed}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'temperature_1hr', data = {x:hour_start_time, y:hour_temp}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'v_thermal_1hr', data = {x:hour_start_time, y:hour_small_v_th}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'north_south_1hr', data = {x:hour_start_time, y:hour_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'error_code_1hr', data = {x:hour_start_time, y:hour_error}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'caution_code_1hr', data = {x:hour_start_time, y:hour_caution}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'attitude_flag_1hr', data = {x:hour_start_time, y:hour_attitude}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'reduced_chi_squared_1hr', data = {x:hour_start_time, y:hour_red_chi}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_1hr', data = {x:hour_start_time, y:hour_count}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_1hr_v', data = {x:hour_start_time, y:hour_count_v}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_1hr_ns', data = {x:hour_start_time, y:hour_count_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_rotation_1hr', data = {x:hour_start_time, y:hour_carr_num}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_longitude_1hr', data = {x:hour_start_time, y:hour_carr_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'heliocentric_distance_1hr', data = {x:hour_start_time, y:hour_helio_dist}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEE_1hr', data = {x:hour_start_time, y:hour_hee_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEE_1hr', data = {x:hour_start_time, y:hour_hee_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEEQ_1hr', data = {x:hour_start_time, y:hour_heeq_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEEQ_1hr', data = {x:hour_start_time, y:hour_heeq_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HCI_1hr', data = {x:hour_start_time, y:hour_hci_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HCI_1hr', data = {x:hour_start_time, y:hour_hci_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}

    print, 'Calculating 10 minute averages.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;Calculate 10 minute averages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    IF new_num_records MOD 10 EQ 0 THEN k = FIX(new_num_records/10) ELSE k = FIX(new_num_records/10)+1
    small_density_copy2 = small_density
    small_avg_v_copy2 = small_avg_v
    small_temp_K_copy2 = small_temp_K
    small_v_th_copy2 = small_v_th
    SMALL_S_MAX_copy2 = SMALL_S_MAX

    ten_min_start_time = dblarr(k)
    ten_min_start_time(*) = !values.d_nan
    ten_min_small_density = dblarr(k)
    ten_min_small_density(*) = !values.f_nan
    ten_min_speed = dblarr(k)
    ten_min_speed(*) = !values.f_nan
    ten_min_temp = dblarr(k)
    ten_min_temp(*) = !values.f_nan
    ten_min_small_v_th = dblarr(k)
    ten_min_small_v_th(*) = !values.f_nan
    ten_min_count = dblarr(k)
    ten_min_count(*) = !values.f_nan
    ten_min_count_v = dblarr(k)
    ten_min_count_v(*) = !values.f_nan
    ten_min_count_ns = dblarr(k)
    ten_min_count_ns(*) = !values.f_nan
    ten_min_ns = dblarr(k)
    ten_min_ns(*) = !values.f_nan
    ten_min_error = intarr(k)
    ten_min_caution = intarr(k)
    ten_min_attitude = intarr(k, 3)
    ten_min_red_chi = dblarr(k)
    ten_min_red_chi(*) = !values.f_nan
    ten_min_carr_num = dblarr(k)
    ten_min_carr_num(*) = !values.f_nan
    ten_min_carr_lon = dblarr(k)
    ten_min_carr_lon(*) = !values.f_nan
    ten_min_helio_dist = dblarr(k)
    ten_min_helio_dist(*) = !values.f_nan
    ten_min_hee_lon = dblarr(k)
    ten_min_hee_lon(*) = !values.f_nan
    ten_min_hee_lat = dblarr(k)
    ten_min_hee_lat(*) = !values.f_nan
    ten_min_heeq_lon = dblarr(k)
    ten_min_heeq_lon(*) = !values.f_nan
    ten_min_heeq_lat = dblarr(k)
    ten_min_heeq_lat(*) = !values.f_nan
    ten_min_hci_lon = dblarr(k)
    ten_min_hci_lon(*) = !values.f_nan
    ten_min_hci_lat = dblarr(k)
    ten_min_hci_lat(*) = !values.f_nan

    i10 = 0                         ; index into 10 minute data
    found_first = 0
    on_first = 1
    FOR i1 = 0L, n_elements(small_epoch1)-1 DO BEGIN ; index into 1 minute data
        now_time = time_struct(small_epoch1(i1))
        IF now_time.year NE 1970 THEN BEGIN ; skip invalid times
            IF on_first EQ 1 THEN base_hour = now_time.hour ; LBE: change to avoid rare condition when missing data
            IF found_first EQ 0 THEN BEGIN
                base_10min = now_time.min/10
            ENDIF
            now_10min = now_time.min/10
            IF now_time.hour NE base_hour OR found_first EQ 0 OR $
              ((now_time.hour EQ base_hour) AND (now_10min NE base_10min)) THEN BEGIN ; new base
                base_hour = now_time.hour
                found_first = 1
                                ;IF i10 GT 0 THEN BEGIN ; write old variables
                IF on_first EQ 0 THEN BEGIN ; write old variables
                    ten_min_error[i10] = temp_error
                    ten_min_caution[i10] = temp_caution
                    ten_min_attitude[i10, *] = temp_attitude
                    IF n_elements(temp_red_chi) GT 0 AND FINITE(temp_red_chi) EQ 1 THEN ten_min_red_chi[i10] = temp_red_chi/ten_count ; corrected 29 Sept. 2010
                    IF n_elements(temp_density) GT 0 AND FINITE(temp_density) EQ 1 THEN ten_min_small_density[i10] = temp_density/ten_count
                    IF n_elements(temp_speed) GT 0 AND FINITE(temp_speed) EQ 1 THEN ten_min_speed[i10] = temp_speed/ten_count_v 
                    IF n_elements(temp_temp) GT 0 AND FINITE(temp_temp) EQ 1 THEN ten_min_temp[i10] = temp_temp/ten_count
                    IF n_elements(temp_v_th) GT 0 AND FINITE(temp_v_th) EQ 1 THEN ten_min_small_v_th[i10] = temp_v_th/ten_count
                    IF n_elements(temp_ns) GT 0 AND FINITE(temp_ns) EQ 1 THEN ten_min_ns[i10] = temp_ns/ten_count_ns 
                    ten_min_count[i10] = ten_count
                    ten_min_count_v[i10] = ten_count_v
                    ten_min_count_ns[i10] = ten_count_ns
                ENDIF 
                base_10min = now_10min
                IF on_first EQ 1 THEN on_first = 0 ELSE i10 = i10 + 1
                IF i10 EQ n_elements(ten_min_start_time) THEN BEGIN ; need to resize arrays
                    tmp_arr  = intarr(10)
                    tmp2_arr = intarr(10, 3)
                    ten_min_error = [ten_min_error, tmp_arr]
                    ten_min_caution = [ten_min_caution, tmp_arr]
                    ten_min_attitude = [ten_min_attitude, tmp2_arr]
                    tmp_arr = dblarr(10)
                    tmp_arr[*] = !values.f_nan
                    ten_min_count = [ten_min_count, tmp_arr]
                    ten_min_count_v = [ten_min_count_v, tmp_arr]
                    ten_min_count_ns = [ten_min_count_ns, tmp_arr]
                    ten_min_start_time = [ten_min_start_time, tmp_arr]
                    ten_min_red_chi = [ten_min_red_chi, tmp_arr]
                    ten_min_carr_num = [ten_min_carr_num, tmp_arr]
                    ten_min_carr_lon = [ten_min_carr_lon, tmp_arr]
                    ten_min_helio_dist = [ten_min_helio_dist, tmp_arr]
                    ten_min_hee_lon = [ten_min_hee_lon, tmp_arr]
                    ten_min_hee_lat = [ten_min_hee_lat, tmp_arr]
                    ten_min_heeq_lon = [ten_min_heeq_lon, tmp_arr]
                    ten_min_heeq_lat = [ten_min_heeq_lat, tmp_arr]
                    ten_min_hci_lon = [ten_min_hci_lon, tmp_arr]
                    ten_min_hci_lat = [ten_min_hci_lat, tmp_arr]
                    ten_min_small_density = [ten_min_small_density, tmp_arr]
                    ten_min_speed = [ten_min_speed, tmp_arr]
                    ten_min_temp = [ten_min_temp, tmp_arr]
                    ten_min_small_v_th = [ten_min_small_v_th, tmp_arr]
                    ten_min_ns = [ten_min_ns, tmp_arr]
                ENDIF 
                ;ten_min_start_time[i10] = time_double(string(now_time.year, format = '(I4.4)')+'-'+$
                ;                                      string(now_time.month, format = '(I2.2)')+'-'+$
                ;                                      string(now_time.date, format = '(I2.2)')+'/'+$
                ;                                      string(now_time.hour, format = '(I2.2)')+':'+$
                ;                                      string((base_10min*10), format = '(I2.2)')+':00')
                ten_min_start_time[i10] = small_epoch1[i1] ; LBE changed 9/11/09 to make actual start time
                ten_min_carr_num[i10] = carr_num[i1]
                ten_min_carr_lon[i10] = carr_lon[i1]
                ten_min_helio_dist[i10] = helio_dist[i1]
                ten_min_hee_lon[i10] = hee_lon[i1]
                ten_min_hee_lat[i10] = hee_lat[i1]
                ten_min_heeq_lon[i10] = heeq_lon[i1]
                ten_min_heeq_lat[i10] = heeq_lat[i1]
                ten_min_hci_lon[i10] = hci_lon[i1]
                ten_min_hci_lat[i10] = hci_lat[i1]
                IF finite(small_density_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN temp_density = small_density_copy2[i1] ELSE temp_density = !values.f_nan
                IF finite(small_avg_v_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN temp_speed = small_avg_v_copy2[i1] ELSE temp_speed = !values.f_nan
                IF finite(small_temp_K_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN temp_temp = small_temp_K_copy2[i1] ELSE temp_temp = !values.f_nan
                IF finite(small_v_th_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN temp_v_th = small_v_th_copy2[i1] ELSE temp_v_th = !values.f_nan
                IF finite(SMALL_S_MAX_copy2[i1]) EQ 1 AND small_std_error[i1] NE 5 AND small_std_error[i1] NE 8 $
                  THEN temp_ns = SMALL_S_MAX_copy2[i1] ELSE $
                  temp_ns = !values.f_nan
                IF finite(small_good_fit[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN temp_red_chi = small_good_fit[i1] ELSE temp_red_chi = !values.f_nan ; corrected 29 Sept. 2010
                temp_caution = small_caution_orig[i1]
                temp_attitude = attitude_flag[i1, *]
                IF small_std_error[i1] EQ 5 OR small_std_error[i1] EQ 8 THEN temp_error = 0 $
                  ELSE temp_error = small_std_error[i1]
                IF FINITE(temp_density) EQ 1 THEN ten_count = 1.0 ELSE ten_count = 0.0
                IF FINITE(temp_speed) EQ 1 THEN ten_count_v = 1.0 ELSE ten_count_v = 0.0
                IF FINITE(temp_ns) EQ 1 THEN ten_count_ns = 1.0 ELSE ten_count_ns = 0.0
            ENDIF ELSE BEGIN    ; further in same 10 min space
                IF finite(small_density_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN IF FINITE(temp_density) EQ 0 THEN temp_density = small_density_copy2[i1] $
                       ELSE temp_density = temp_density + small_density_copy2[i1]
                IF finite(small_avg_v_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN IF FINITE(temp_speed) EQ 0 THEN temp_speed =  small_avg_v_copy2[i1] $
                       ELSE temp_speed = temp_speed + small_avg_v_copy2[i1]
                IF finite(small_temp_K_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN IF FINITE(temp_temp) EQ 0 THEN temp_temp = small_temp_K_copy2[i1] $
                       ELSE  temp_temp = temp_temp + small_temp_K_copy2[i1]
                IF finite(small_v_th_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN IF FINITE(temp_v_th) EQ 0 THEN temp_v_th = small_v_th_copy2[i1] $
                       ELSE temp_v_th = temp_v_th + small_v_th_copy2[i1]
                IF finite(SMALL_S_MAX_copy2[i1]) EQ 1 AND small_std_error[i1] NE 5 AND small_std_error[i1] NE 8 THEN $
                  IF FINITE(temp_ns) EQ 0 THEN $
                  temp_ns = SMALL_S_MAX_copy2[i1] $
                  ELSE temp_ns = temp_ns + SMALL_S_MAX_copy2[i1]
                IF finite(small_good_fit[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN IF finite(temp_red_chi) EQ 0 THEN temp_red_chi = small_good_fit[i1] $
                       ELSE temp_red_chi = temp_red_chi+small_good_fit[i1] ; corrected 29 Sept. 2010
                IF small_caution_orig[i1] EQ 2 THEN temp_caution = 2
                IF small_caution_orig[i1] EQ 1 AND temp_caution NE 2 THEN temp_caution = 1
                IF attitude_flag[i1, 0] EQ 1 THEN temp_attitude[0] = 1
                IF attitude_flag[i1, 1] EQ 1 THEN temp_attitude[1] = 1
                IF attitude_flag[i1, 2] EQ 1 THEN temp_attitude[2] = 1
;; check which errors to carry over (priority is 3,7,4,6,2,1,8,5)
;; change starting 8/2014 so 5 & 8 don't go in average
                CASE small_std_error[i1] OF
                    3: temp_error = 3
                    7: IF (temp_error NE 3) THEN temp_error = 7
                    4: IF (temp_error NE 3) AND (temp_error NE 7) THEN temp_error = 4
                    6: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) THEN temp_error = 6
                    2: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) THEN temp_error = 2
                    1: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                      AND (temp_error NE 2) THEN temp_error = 1
                    ;8: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                    ;  AND (temp_error NE 2) AND (temp_error NE 1) THEN temp_error = 8
                    8: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                      AND (temp_error NE 2) AND (temp_error NE 1) THEN temp_error = 0
                    ;5: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                    ;  AND (temp_error NE 2) AND (temp_error NE 1) AND (temp_error NE 8) THEN temp_error = 5
                    5: IF (temp_error NE 3) AND (temp_error NE 7) AND (temp_error NE 4) AND (temp_error NE 6) $
                      AND (temp_error NE 2) AND (temp_error NE 1) AND (temp_error NE 8) THEN temp_error = 0
                    0: temp_error = temp_error
                ENDCASE 
                IF (FINITE(small_density_copy2[i1]) EQ 1) AND (small_density_copy2[i1] NE -1) AND $
                  small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) THEN $
                  ten_count = ten_count+1 ; Make sure time step has a valid result
                IF (FINITE(small_avg_v_copy1[i1]) EQ 1) AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 AND $
                  ((small_std_error[i1] NE 8) OR (small_std_error[i1] EQ 8 AND angle_error[i1] EQ 1)) $
                  THEN ten_count_v = ten_count_v+1
                IF (FINITE(SMALL_S_MAX_copy1[i1]) EQ 1) AND small_std_error[i1] NE 5 AND small_std_error[i1] NE 8 $
                  THEN ten_count_ns = ten_count_ns+1.0
            ENDELSE 
        ENDIF 
    ENDFOR 
; add final value
    IF i10 EQ n_elements(ten_min_error)    THEN ten_min_error =    [ten_min_error, temp_error]      $
    ELSE ten_min_error[i10] = temp_error
    IF i10 EQ n_elements(ten_min_caution)  THEN ten_min_caution =  [ten_min_caution, temp_caution]  $
    ELSE ten_min_caution[i10] = temp_caution
    IF i10 EQ n_elements(ten_min_attitude) THEN ten_min_attitude = [ten_min_attitude, temp_attitude] $
    ELSE ten_min_attitude[i10, *] = temp_attitude
    IF n_elements(temp_density) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_small_density) THEN ten_min_small_density = [ten_min_small_density, temp_density/ten_count] $
      ELSE ten_min_small_density[i10] = temp_density/ten_count
    IF n_elements(temp_speed)   GT 0 THEN $
      IF i10 EQ n_elements(ten_min_speed)         THEN ten_min_speed = [ten_min_speed, temp_speed/ten_count_v] $
      ELSE ten_min_speed[i10] = temp_speed/ten_count_v
    IF n_elements(temp_temp)    GT 0 THEN $
      IF i10 EQ n_elements(ten_min_temp)          THEN ten_min_temp = [ten_min_temp, temp_temp/ten_count] $
      ELSE ten_min_temp[i10] = temp_temp/ten_count
    IF n_elements(temp_v_th)    GT 0 THEN $
      IF i10 EQ n_elements(ten_min_small_v_th)    THEN ten_min_small_v_th = [ten_min_small_v_th, temp_v_th/ten_count] $
      ELSE ten_min_small_v_th[i10] = temp_v_th/ten_count
    IF n_elements(temp_ns)      GT 0 THEN $
      IF i10 EQ n_elements(ten_min_ns)             THEN ten_min_ns = [ten_min_ns, temp_ns/ten_count_ns] $
      ELSE ten_min_ns[i10] = temp_ns/ten_count_ns
    IF n_elements(temp_red_chi) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_red_chi)        THEN ten_min_red_chi = [ten_min_red_chi, temp_red_chi/ten_count] $
      ELSE ten_min_red_chi[i10] = temp_red_chi/ten_count ; corrected 29 Sept. 2010
    IF i10 EQ n_elements(ten_min_count)    THEN ten_min_count = [ten_min_count, ten_count]          ELSE ten_min_count[i10] = ten_count
    IF i10 EQ n_elements(ten_min_count_v)  THEN ten_min_count_v = [ten_min_count_v, ten_count_v]    ELSE ten_min_count_v[i10] = ten_count_v
    IF i10 EQ n_elements(ten_min_count_ns) THEN ten_min_count_ns = [ten_min_count_ns, ten_count_ns] ELSE ten_min_count_ns[i10] = ten_count_ns

    good_indices = intarr(n_elements(ten_min_start_time)) ; set to 1 if record should be included
    found_bad_time = 0
    FOR p = 0L, N_ELEMENTS(ten_min_start_time) - 1 DO BEGIN
        IF finite(ten_min_start_time(p)) EQ 1 THEN good_indices[p] = 1 ELSE found_bad_time = 1
    ENDFOR
    IF found_bad_time EQ 1 THEN BEGIN
        good_i = where(good_indices EQ 1, good_count)
        IF good_count GT 0 THEN BEGIN 
            ten_min_start_time    = ten_min_start_time   [good_i]
            ten_min_small_density = ten_min_small_density[good_i]
            ten_min_speed         = ten_min_speed        [good_i]
            ten_min_temp          = ten_min_temp         [good_i]
            ten_min_small_v_th    = ten_min_small_v_th   [good_i]
            ten_min_ns            = ten_min_ns           [good_i]
            ten_min_error         = ten_min_error        [good_i]
            ten_min_caution       = ten_min_caution      [good_i]
            ten_min_attitude      = ten_min_attitude     [good_i, *]
            ten_min_red_chi       = ten_min_red_chi      [good_i]
            ten_min_count         = ten_min_count        [good_i]
            ten_min_count_v       = ten_min_count_v      [good_i]
            ten_min_count_ns      = ten_min_count_ns     [good_i]
            ten_min_carr_num      = ten_min_carr_num     [good_i]
            ten_min_carr_lon      = ten_min_carr_lon     [good_i]
            ten_min_helio_dist    = ten_min_helio_dist   [good_i]
            ten_min_hee_lon       = ten_min_hee_lon      [good_i]
            ten_min_hee_lat       = ten_min_hee_lat      [good_i]
            ten_min_heeq_lon      = ten_min_heeq_lon     [good_i]
            ten_min_heeq_lat      = ten_min_heeq_lat     [good_i]
            ten_min_hci_lon       = ten_min_hci_lon      [good_i]
            ten_min_hci_lat       = ten_min_hci_lat      [good_i]
        ENDIF ELSE stop ; no valid times!
    ENDIF 

    ; check for caution eq 5
    FOR temp_i = 0, n_elements(ten_min_start_time)-2 DO BEGIN
        wanted_i = where(small_epoch1 GE ten_min_start_time[temp_i] AND small_epoch1 LT ten_min_start_time[temp_i+1], count)
        five_i = where(small_caution[wanted_i] EQ 5, five_count)
        IF five_count GT 0 THEN ten_min_caution[temp_i] = 5
    ENDFOR 
    wanted_i = where(small_epoch1 GE ten_min_start_time[temp_i], count)
    five_i = where(small_caution[wanted_i] EQ 5, five_count)
    IF five_count GT 0 THEN ten_min_caution[temp_i] = 5

    store_data, 'density_10min', data = {x:ten_min_start_time, y:ten_min_small_density}, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, 'speed_10min', data = {x:ten_min_start_time, y:ten_min_speed}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'temperature_10min', data = {x:ten_min_start_time, y:ten_min_temp}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'v_thermal_10min', data = {x:ten_min_start_time, y:ten_min_small_v_th}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'north_south_10min', data = {x:ten_min_start_time, y:ten_min_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'error_code_10min', data = {x:ten_min_start_time, y:ten_min_error}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'caution_code_10min', data = {x:ten_min_start_time, y:ten_min_caution}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'attitude_flag_10min', data = {x:ten_min_start_time, y:ten_min_attitude}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'reduced_chi_squared_10min', data = {x:ten_min_start_time, y:ten_min_red_chi}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_10min', data = {x:ten_min_start_time, y:ten_min_count}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_10min_v', data = {x:ten_min_start_time, y:ten_min_count_v}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_10min_ns', data = {x:ten_min_start_time, y:ten_min_count_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_rotation_10min', data = {x:ten_min_start_time, y:ten_min_carr_num}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_longitude_10min', data = {x:ten_min_start_time, y:ten_min_carr_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'heliocentric_distance_10min', data = {x:ten_min_start_time, y:ten_min_helio_dist}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEE_10min', data = {x:ten_min_start_time, y:ten_min_hee_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEE_10min', data = {x:ten_min_start_time, y:ten_min_hee_lat}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEEQ_10min', data = {x:ten_min_start_time, y:ten_min_heeq_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEEQ_10min', data = {x:ten_min_start_time, y:ten_min_heeq_lat}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HCI_10min', data = {x:ten_min_start_time, y:ten_min_hci_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HCI_10min', data = {x:ten_min_start_time, y:ten_min_hci_lat}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
        ;ENDELSE 
    ;ENDIF 

ENDIF


;; Un-comment below to print data in ascii form to an output folder
;calc_proton_finished_ascii, sat, small_epoch1, kev_time, small_density, small_avg_v, small_temp_K, small_v_th, SMALL_S_MAX, ;small_std_error, $
;	small_caution, small_good_fit, hour_start_time, hour_small_density, hour_speed, hour_temp, hour_small_v_th, hour_ns, $
;	hour_error, hour_caution, hour_red_chi, hour_count, ten_min_start_time, ten_min_small_density, ten_min_speed, ten_min_temp, $
;	ten_min_small_v_th, ten_min_ns, ten_min_error, ten_min_caution, ten_min_red_chi, ten_min_count


END 
