;+
;PROCEDURE: calc_proton_bulk
;PURPOSE:
;  Calculate proton bulk parameters from 1 minute resolution RA_Trigger rate.
;
;PARAMETERS:   in:  sat: 'A' or 'B'
;		    bkgd: 1 or 2 
;                   files : Array of file paths to read from (strings)
;                   which_epoch: 1 (for 1 minute resolution)
;		    time:  starting date and time
;		    table_location:  where efficiency and ESA tables are found
; 		    output_dir:  where to write the output files
;				
;
;CREATED BY: K. Simunac
;
;LAST MODIFICATION: 08/27/2009
;
;MODIFICATION HISTORY:
;    March 2008:  created
;	1 April, 2008:  added routines for 10 minute and 1 hour data averaging
;       1 April, 2008: LBE: Changed ascii output to match that on web.
;       3 April, 2008: LBE: Added good_indices.
;	3 April, 2008: KDCS put limit on index iteration for averaging.
;       4 April, 2008: LBE: Added loop for filename if epoch1 empty at beginning.
;       11 April, 2008: LBE: Changed loop variables to longs.
;       21 April, 2008: LBE: Changed gaussfit to doubles
;       22 April, 2008: LBE: Added num_days argument.
;                            Changed to take into account 1 hr buffer on each end.
;       28 April, 2008: LBE: Added tplot_dir.
;                            Added tplot variables for 10 min and 1 hr data.
;       29 April, 2008: LBE: Took out fix(); so data is now floats.
;       08 May,   2008: LBE: Added check on closed entrance system.
;	08 May,   2008: KDCS:  Added error code for low temperatures
;			       Added error/caution codes to 10 min and 1 hr data
;	12 May,   2008: KDCS:  Added check for multiple finite records in a row for
;				3 sigma check
;       21 May,   2008: LBE: Removed output_dir argument.
;       22 May,   2008: LBE: Added attitude_flag
;       30 May,   2008: LBE: Added tplot_options, 'color', 0
;       03 June,  2008: LBE: Added spacecraft positions and Carrington info.
;       04 June,  2008: LBE: Changed floats to doubles.
;       05 June,  2008: LBE: Rewrote averaging to deal with time gaps.
;	09 June,  2008: KDCS:  commented out obsolete 3 sigma check during buffer time
;       16 June,  2008: LBE: Added checks for when no valid data in day.
;	14 Aug,   2008: KDCS:  modified North/South Flow angle calculation for STEREO B
;	28 Aug,	  2008: KDCS:  modified North/South Flow angle calculation for STEREO A
;	29 Aug,   2008: KDCS:  changed index on housekeeping check from integer to long
;	02 Sep,   2008: KDCS:  added check for unrealistic north/south flow angle
;       04 Sep,   2008: LBE:   Added have_data flag.
;       22 Sep,   2008: LBE:   Updated have_data flag.
;	25 Nov,   2008: KDCS: introduced multiplicative factor to efficiency due to change in radius
;       26 Nov,   2008: LBE : Changed V6 to V7 in order to show change Kristin made.
;                             Fixed typo in sum_full_data
;	05 Dec,   2008: KDCS: Added routine to average distribution function over repeated ESA settings
;			      Added a rough correction for different orbital radii during calibration period
;       11 Dec,   2008: LBE : Removed tplot_dir and code for saving tplot variables.
;	Jan - Feb, 2009: KDCS: -realistic propagation of uncertainties added
;				-"good_fit" parameter added to store reduced chi squared from gaussfit
;				-fits with reduced chi squared greater than 5 are considered questionable,
;				so density, temperature and v_thermal assotiated with those fits are set to NaN
;				-added counter variables to averaging routines so that speed data can
;				be included even when density and v_thermal are set to NaN
;				-data on spacecraft B with thermal speed less than 12 km/s are flagged as errors
;       23 Feb. 2009: LBE: Fixed syntax errors with store_data.
;	25 Feb, 2009: KDCS:  added check for n_elements(temp_X) EQ 0 in ten minute averaging and removed ELSE clause
;				setting temp_X to NaN			
;	
;	28 Feb, 2009: KDCS: Reinstated ELSE clause that sets temp_X to NaN if first record in an averaging
;				 section is not finite, and then set counter to 0.0 
;       04 Mar, 2009: LBE : Changed attitude flag to an array.
;       05 Mar, 2009: LBE : Changed default to -1 for attitude flag.
;	06 Mar, 2009: KDCS: Changed counter to check for finite speed instead of finite density for storing
;				tplot variables.
;       06 Mar, 2009: LBE : Added 10min and 1hr value (max) for reduced chi^2
;       11 Mar, 2009: LBE : Fixed bug in yaw. Fixed bug in averages when no valid data.
;       16 Mar, 2009: LBE : Refixed bus in yaw.
;       23 Mar, 2009: LBE : Added check for bulk vel < 230 (throw out data)
;                           Added checks for outliers (B: 1/27/09)
;       10 Mar, 2009: LBE : Added outlier check (A: 4/17/07)
;       14 Apr, 2009: LBE : Added outlier checks (A: 7/11/07,
;                           11/21/07, 12/19/07, 2/6/08, 4/24/08, 4/25/08, 8/21/08, 11/09/08
;       29 Apr, 2009: LBE : Added many more outlier checks
;       05 May, 2009: LBE : Added many more outlier checks
;       07 May, 2009: LBE : Added many more outlier checks
;	08 May, 2009: KDCS:  Fixed bug in averaging section
;       15 May, 2009: LBE : Added more outlier checks
;	01 Jun, 2009: KDCS:  Revised response to error codes, i.e. when thermal speed is suspiciously
;				low the bulk speed and flow angle are NOT set to fill values, and the
;				error code assigned is 7 rather than 1
;	08 Jul, 2009: LBE : Added B outliers.
;	13 Jul, 2009: LBE : Added B outliers.
;	03 Aug, 2009: LBE : Fixed bug in check on hk system.
;	27 Aug, 2009: KDCS:  Added calls to new E/Q and efficiency tables.  Instead of multiple
;				tables for PLASTIC-B, there is just one.  The revised efficiency and E/Q tables
;				have just 64 lines instead of 128.  The main channel efficiency is a flat value for
;				all E/Q steps.   
;       09 Sep, 2009: LBE : Changed read of A eff table to look in table_location.
;	                    No longer remove fill values.
;                           Add error=8 for hand-removed times.
;                           Fixed bug (stored red_chi_sq when should be ten_min_red_chi_sq and hour_red_chi_sq)
;       29 Sep, 2009: LBE : Added error 1 and 5 to priority list for 10 min and 1 hr.
;       08 Oct, 2009: LBE : Changed outliers based on Kristin's bad_times_KDCS.xls
;-


PRO calc_proton_bulk, sat, bkgd, files, which_epoch, time, num_days, table_location

COMMON share1_high_res
COMMON com_high_res
tab = string(9B)
tplot_options, 'color', 0

num_records1 = final_count ; final count of number of records from pl_read_cdf_high_res
have_data = 1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;							  ;
; Below is a routine to calculate density, bulk speed,	  ;
; temperature and thermal speed using full	 	  ;
; resolution rate data from RA_Trigger or SF0.		  ;
;	 						  ;					 	  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; As of 31 March, 2008 ESA range is set to steps 64 to 127
; Highest proton speed is about 900 km/second (with some thermal spread)
; For table of step, E/Q, and proton velocity see ESA_table.txt


; Geometric Factors for Main Channel and S Channel (full SWS)

; Reto's Values
;---------------------------------
;A_G_Main = 3.15E-7 ;m^2 sr eV/eV 
;A_G_Sch = 6.0E-11 
;B_G_Main = 2.52E-7 ;m^2 sr eV/eV
;B_G_Sch = 6.3E-11

;Kristin's Values using Lynn's method
;-------------------------------------
; G = Active Area*Angular Extent of Slit*Bandwidth/2
; For example, for the main channel of FM1:
; 	Active Area = 8.9E-5 m^2
;	Deflection FWHM = 1.9 degrees = 0.033 radians
;	Azimuth Acceptance = 45 degrees = 0.785 radians
;	Energy Bandwidth = 6.12%
;	G = (8.9E-5)*(0.033)*(0.785)*0.0612/2 = 7.1E-8 m^2 sr eV/eV

A_G_Main = 7.1E-8 
A_G_Sch = 2.3E-11 
B_G_Main = 6.0E-8
B_G_Sch = 1.7E-11 


; Active areas -- not for calculations, but here for reference
;----------------------------------------------------------------
; S Channel value is divided by ratio of deflection bin size to
; instantaneous field of view (though not for geometric factor
; calculation above):  
;
; 1.33 degrees / 0.37 degrees = 3.6 for A
; 1.33 degrees / 0.27 degrees = 4.9 for B

 A_Area_Main = 8.9E-1 ; cm^2
 A_Area_Sch = 1.41E-3/3.6 ;cm^2
 B_Area_Main = 7.5E-1 ; cm^2
 B_Area_Sch = 1.50E-3/4.9 ;cm^2

; Dead times for SF0 and RA_TRIG
;--------------------------------
A_SF0_dead_time = 5.0E-6 ;sec
A_RA_TRIG_dead_time = 2.0E-6 ;sec
B_SF0_dead_time = 5.0E-6	;sec
B_RA_TRIG_dead_time = 2.0E-6	;sec ; CHANGED ON 02 APRIL, 2007 FROM 2.5E-6


; Read in tables of estimated efficiencies
; The location of the efficiency tables must be specified in the crib file.
;-------------------------------------------------------------------------------
eff_B_new = dblarr(128)
eff_A_new = dblarr(128)

; Read the efficiency table for spacecraft A
;----------------------------------------------------
eff_file_A_new = table_location+'leakage_A_2009_08_17.txt'
openr, in_lun_1, eff_file_A_new, /GET_LUN
line = ","
readf, in_lun_1, line ; read the header line
FOR i = 64, 127 DO BEGIN
	readf, in_lun_1, line
	parts = STRSPLIT(line, ",", /EXTRACT)
	eff_A_new[i]=parts[1]
ENDFOR
close, in_lun_1
free_lun, in_lun_1


; Read the table for spacecraft B.
;-------------------------------------------------
eff_file_B_new=table_location+'B_eff_table_2009_08_21.txt' ; this is really a multiplicative leakage table
openr, in_lun_1, eff_file_B_new, /GET_LUN
line = ","
readf, in_lun_1, line ; read past first line of header
FOR i = 64, 127 DO BEGIN
	readf, in_lun_1, line
	parts = STRSPLIT(line, ",", /EXTRACT)
	eff_B_new[i]=parts[1]
ENDFOR
close, in_lun_1
free_lun, in_lun_1

print, 'Efficiency tables have been read.'

;---------------------------------------------------------
IF sat EQ 'A' THEN BEGIN
   ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_a_full, /nan) LE 0 AND TOTAL(ra_trig_a_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        STOP
    ENDIF
    s_chan = intarr(num_records1)
    error1 = intarr(num_records1)
    rate_name = ' '
    std_error = intarr(num_records1) ; A matrix in which to store possible errors
    std_error(*) = 0

   ;; Check to see if there are RA_Trig counts.  If so, continue.  
   ;; If not, assume the full resolution rate is SF0.
   ;;
   ;; RA_Trig was set as the full resolution rate on 14 February, 2007 for A
   ;; and on 1 March, 2007 for B.

    IF TOTAL(ra_trig_a_full, /nan) GT 0 THEN BEGIN
        full_data = ra_trig_a_full
        IF bkgd EQ 2 THEN ra_trig_a_full = 0
        rate_name = '_RA_Trig'
        dead_time = A_RA_trig_dead_time
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_a
        error1 = error1_a
        IF bkgd EQ 2 THEN error1_a = 0
    ENDIF ELSE BEGIN
        full_data = sf0_a_full
        IF bkgd EQ 2 THEN sf0_a_full = 0
        rate_name = '_SF0'
        dead_time = A_SF0_dead_time
        G_Main = A_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_a
        error1 = error1_a
        IF bkgd EQ 2 THEN error1_a = 0
    ENDELSE
ENDIF ELSE BEGIN ; data is from spacecraft B
    ;; Check that there is actually data in the files.  If not, then stop.
    IF TOTAL(sf0_b_full, /nan) LE 0 AND TOTAL(ra_trig_b_full, /nan) LE 0 THEN BEGIN
        PRINT, 'No SF0 or RA_trig data for this time.'
        have_data = 0
    ENDIF
    full_data = intarr(128, 32, num_records1) ; changed from float to integer to save memory 28 Sept, 2007
    rate_name = ' '
    std_error = intarr(num_records1) ; A matrix in which to store possible errors
    std_error(*) = 0

    ;; Check to see if there are RA_Trig counts.  If so, continue.  
    ;; If not, assum SF0 is the full resolution rate.
    ;;
    ;; RA_Trig was set as the full resolution rate on 14 February, 2007 on A
    ;; and on 1 March, 2007 on B.

    IF TOTAL(ra_trig_b_full, /nan) GT 0 THEN BEGIN
        full_data = ra_trig_b_full
        IF bkgd EQ 2 THEN ra_trig_b_full = 0	
        rate_name = '_RA_Trig'
        dead_time = B_RA_trig_dead_time
        G_Main = B_G_Main
        G_Sch = A_G_Sch
        s_chan = s_chan1_b
        error1 = error1_b
        IF bkgd EQ 2 THEN error1_b = 0
    ENDIF ELSE BEGIN
        full_data = sf0_b_full
        IF bkgd EQ 2 THEN sf0_b_full = 0
        rate_name = '_SF0'
        dead_time = B_SF0_dead_time
	G_Main = B_G_Main
        G_Sch = B_G_Sch
        s_chan = s_chan1_b
        error1 = error1_b
        IF bkgd EQ 2 THEN error1_b = 0
    ENDELSE
ENDELSE

;; Get the time associated with the first file to label output files
start_time=' '
start_time=strmid(time,0,10)

print, 'Starting calculations.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove the fill value of -1 from the files.
remove_fill_high_res, -1, full_data
print, 'Removed fill values.'

;; Check for value indicating overflow.  If present, mark record with error 3.
FOR m=0L, num_records1 - 1 DO BEGIN
    FOR k = 0, 127 DO BEGIN
        FOR l = 0, 31 DO BEGIN
            IF full_data(k, l, m) GE 507904 THEN BEGIN
                full_data(k, l, m) = !values.f_nan
                error1(m) = 3
                PRINT, 'Overflow value detected.  Time, ESA, Defl:', m, k, l
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
print,'Maximum Counts:',max(full_data,/NAN)
IF max(full_data) GT 32000 THEN STOP

IF bkgd EQ 1 THEN BEGIN
;; Background Subtraction 1
;;-------------------------------------------------------------------------
;; See how many background counts there are.  Average the counts in ESA step
;; 127, DEFL steps 0, 1, 30, and 31.
    back_ground = fltarr(num_records1) ; changed integer to float (8 May, 2008)
    back_ground[*] = !values.f_nan
    delta_bkgd = fltarr(num_records1)
    delta_bkgd[*] = !values.f_nan
    full_data_no_bkgd = dblarr(128, 32, num_records1) ; changed integer to float (8 May, 2008)
    full_data_no_bkgd[*, *, *] = !values.f_nan
    delta_counts = fltarr(128,32,num_records1)
    delta_counts(*,*,*) = !values.f_nan
    delta_counts = SQRT(full_data)
    delta_counts_less_bkgd = fltarr(128,32,num_records1)
    delta_counts_less_bkgd(*,*,*) = !values.f_nan
    FOR mm = 0L, num_records1-1 DO BEGIN
        back_ground[mm] = ((full_data(mm*4096+127)+full_data(mm*4096+255)+full_data(mm*4096+3967)+full_data(mm*4096+4095))/4.0)
        IF (back_ground(mm)) LT 3 THEN back_ground(mm) = 3.0
                                ; Subtract out background
        ;full_data_no_bkgd[*, *, mm] = full_data(*, *, mm) - back_ground(mm)
	delta_bkgd(mm) = 0.25*SQRT(full_data(mm*4096+127)+full_data(mm*4096+255)+full_data(mm*4096+3967)+full_data(mm*4096+4095))
	;delta_counts_less_bkgd[*,*,mm] = SQRT(delta_counts(*,*,mm)^2.0+delta_bkgd(mm)^2.0)
		FOR nn = 0,127 DO BEGIN
			FOR oo = 0,31 DO BEGIN
			full_data_no_bkgd[nn, oo, mm] = full_data(nn, oo, mm) - back_ground(mm)
			delta_counts_less_bkgd[nn,oo,mm] = SQRT(delta_counts(nn,oo,mm)^2.0+delta_bkgd(mm)^2.0)
			ENDFOR
		ENDFOR
    ENDFOR

full_data = 0 ; reduce memory usage
print, 'Background has been subtracted out.'
ENDIF

IF bkgd EQ 2 THEN BEGIN
;; Background Subtraction 2
;;---------------------------------------------------------------------------------
;; Subtract out different background level for each ESA step -- added 31 August, 2007
;; to see if this removes contamination from WAP
    back_ground = dblarr(128, num_records1) ; changed from integer to float (8 May, 2008)
    back_ground(*, *) = 0.0
    delta_bkgd = fltarr(128,num_records1)
    delta_bkgd[*,*] = !values.f_nan
    full_data_no_bkgd = dblarr(128, 32, num_records1) ; changed from integer to float (8 May, 2008)
    full_data_no_bkgd[*, *, *] = 0
    delta_counts = fltarr(128,32,num_records1)
    delta_counts(*,*,*) = !values.f_nan
    delta_counts = SQRT(full_data)
    delta_counts_less_bkgd = fltarr(128,32,num_records1)
    delta_counts_less_bkgd(*,*,*) = !values.f_nan
    FOR mm = 0L, num_records1-1 DO BEGIN
        FOR nn = 0, 127 DO BEGIN
            back_ground[nn, mm] = ((full_data(nn, 31, mm)+full_data(nn, 0, mm))/2.0)
	    delta_bkgd[nn,mm] = 0.5*SQRT(full_data(nn, 31, mm)+full_data(nn, 0, mm))
            IF (back_ground(nn, mm)) LT 3 THEN back_ground(nn, mm) = 3.0
	; Subtract out background
            full_data_no_bkgd[nn, *, mm] = full_data(nn, *, mm)- back_ground(nn, mm)
	    delta_counts_less_bkgd[nn,*,mm]=SQRT(delta_counts(nn,*,mm)^2.0+delta_bkgd[nn,mm]^2.0)
        ENDFOR
    ENDFOR
    full_data = 0               ; reduce memory usage
    print, 'Background has been subtracted out.'
ENDIF

; Check to see if negative counts have resulted from background subtraction.
; If yes, set those values to zero.
FOR yy = 0L, n_elements(full_data_no_bkgd)-1 DO BEGIN
    IF (full_data_no_bkgd(yy) LT 0) THEN full_data_no_bkgd(yy) = 0.0
ENDFOR 

; Adjust for deflection wobble
;--------------------------------------------------------------------------------------
FOR j = 1, 31 DO BEGIN          ; DEFL Steps
    FOR i = 64, 126, 2 DO BEGIN ; Even ESA Steps only
        FOR k = 0L, num_records1-1 DO BEGIN ; Time Steps
            full_data_no_bkgd(i, j-1, k) = full_data_no_bkgd(i, j, k)
	    delta_counts_less_bkgd(i,j-1,k) = delta_counts_less_bkgd(i,j,k)
            IF j EQ 31 THEN BEGIN
		full_data_no_bkgd(i, j, k) = !values.f_nan
		delta_counts_less_bkgd(i,j,k) = !values.f_nan
	    ENDIF
        ENDFOR	
    ENDFOR
ENDFOR

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Find the deflection count peak
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
S_MAX = DBLARR(num_records1)
S_MAX(*) = 0.0
MAIN_MAX = DBLARR(num_records1) 
MAIN_MAX(*) = 0

s_count = dblarr(32, num_records1)
s_count(*,*) = 0.0
main_count = dblarr(32, num_records1)
main_count(*,*) = 0.0
defl_steps = intarr(32)

FOR z=0,31 DO BEGIN
    defl_steps(z) = z
ENDFOR

temp = dblarr(num_records1,3)
temp_2=dblarr(num_records1,3)
temp(*,*,*) = -1.0
temp_2(*,*,*) = -1.0

FOR j=0,31 DO BEGIN ; DEFL Steps
    FOR i = 64, 127 DO BEGIN
        FOR k = 0L, num_records1-1 DO BEGIN
            IF finite(full_data_no_bkgd(i, j, k)) EQ 1 THEN BEGIN
                IF i LT s_chan(k) THEN main_count(j, k) = main_count(j, k) + full_data_no_bkgd(i, j, k)
                IF i GE s_chan(k) THEN s_count(j, k) = s_count(j, k) + full_data_no_bkgd(i, j, k)
            ENDIF
        ENDFOR	
    ENDFOR
ENDFOR

FOR l = 0L, num_records1-1 DO BEGIN
    FOR m = 1, 31 DO BEGIN
        IF main_count(m, l) GT main_count(MAIN_MAX(l), l) THEN MAIN_MAX(l) = m
        IF s_count(m, l) GT s_count(S_MAX(l), l) THEN S_MAX(l) = m
    ENDFOR

    IF S_MAX(l) GE 2 AND S_MAX(l) LE 29 THEN BEGIN
        yy = S_MAX(l) - 2
        zz = S_MAX(l) + 2
        result = gaussfit(double(defl_steps(yy:zz)), double(s_count(l*32+yy:l*32+zz)), double(temp), nterms = 3)
        S_MAX(l) = temp(1)
    ENDIF ELSE BEGIN
        S_MAX(l) = !values.f_nan
    ENDELSE
    IF MAIN_MAX(l) GE 2 AND MAIN_MAX(l) LE 29 THEN BEGIN
        aa = MAIN_MAX(l) - 2
        bb = MAIN_MAX(l) + 2
        result = gaussfit(double(defl_steps(aa:bb)), double(main_count(l*32+aa:l*32+bb)), double(temp_2), nterms = 3)
        MAIN_MAX(l) = temp_2(1)
    ENDIF ELSE BEGIN
        MAIN_MAX(l) = !values.f_nan
    ENDELSE
ENDFOR

; Convert to Hz -- divide through by accumulation time of 12.8 milliseconds
;------------------------------------------------------------------------------
full_data_hz = fltarr(128,32,num_records1) ; changed from double to float to save memory 28 Sept, 2007
full_data_hz[*,*,*] = !values.f_nan
full_data_hz = FLOAT(full_data_no_bkgd)/0.0128 ; changed from double to float to save memory 28 Sept, 2007
delta_counts_hz = fltarr(128,32,num_records1)
delta_counts_hz = !values.f_nan
delta_counts_hz = delta_counts_less_bkgd/0.0128
print, 'Converted to Hz.'

IF bkgd EQ 2 THEN full_data_no_bkgd = 0 

; Correct for dead time -- real rate = recorded rate/(1-recorded rate*dead time)
delta_counts_hz = delta_counts_hz/((1.0-(full_data_hz*dead_time))^2.0)
full_data_hz = full_data_hz/(1.0-(full_data_hz*dead_time)) ; changed 1 to 1.0 13 Jan, 2009


;See if dead time correction results in negative rate.  If so, stop.
FOR k = 0L, num_records1-1 DO BEGIN
    FOR i = 0, 127 DO BEGIN
        FOR j = 0, 31 DO BEGIN
            IF full_data_hz(i, j, k) LT 0 THEN BEGIN
                PRINT, '(', i, j, k, ') Invalid countrate -- deadtime is too big.'
                STOP
            ENDIF
        ENDFOR
    ENDFOR
ENDFOR
print, 'Dead time correction completed.'

; Divide by geometric factor to account for different apertures
;----------------------------------------------------------------
full_data_hz = full_data_hz/G_Main ; Divide by G to get into to the right units
delta_counts_hz = delta_counts_hz/G_Main
FOR hh=0L, num_records1-1 DO BEGIN
    IF sat EQ 'A' THEN F_Switch = 120.0 ; approximate ratio between main and s channel based on in-flight data
    IF sat EQ 'B' THEN F_Switch = 65.0
    FOR ii = 0, 127 DO BEGIN			
        IF (ii GE s_chan(hh)) THEN BEGIN ; if data taken with S channel then multiply to normalize with main channel
            full_data_hz(ii, *, hh) = full_data_hz(ii, *, hh)*F_Switch
	    delta_counts_hz(ii,*,hh) = delta_counts_hz(ii,*,hh)*F_Switch
        ENDIF
    ENDFOR
ENDFOR
print, 'Division by geometric factor completed.'

; Sum over deflections steps for each ESA step.
; Keep track of how many total counts are in each ESA step, so we can get 
; peak ESA step.
; Also keep track of how many defl steps have non-zero counts for check
; on thermal spread.
Defl_count = INTARR(num_records1)
Defl_count(*) = 0

peak_ESA = intarr(num_records1)
peak_ESA(*) = -1
sum_full_data = dblarr(num_records1,128) ; changed from double to float to save memory
sum_full_data(*,*) = 0.0
sum_delta_counts = fltarr(num_records1,128)
sum_delta_counts(*,*) = 0.0

FOR j = 0L, num_records1-1 DO BEGIN
    FOR k = 0, 127 DO BEGIN	
        temp_count = 0		
        FOR ll = 0, 31 DO BEGIN
            IF full_data_hz(k, ll, j) GT 0 THEN BEGIN		
                sum_full_data[j, k] = sum_full_data(j, k)+(full_data_hz(k, ll, j))*(COS(-0.33816+0.02182*ll))
				; multiply by cosine of deflection angle to account for projection effect
		sum_delta_counts[j,k]=SQRT(sum_delta_counts(j,k)^2.0+(delta_counts_hz(k,ll,j)*COS(-0.33816+0.02182*ll))^2.0)
                Defl_count[j] = Defl_count(j)+1
                temp_count = temp_count + 1
            ENDIF
        ENDFOR
        IF temp_count LE 0 THEN BEGIN
		sum_full_data[j, k] = 0.0
		sum_delta_counts[j,k]=0.0
	ENDIF 
    ENDFOR		
    peak_count = 0
    FOR g = 0, 127 DO BEGIN
        IF (sum_full_data(j, g) GT peak_count) THEN peak_count = sum_full_data(j, g)
        IF (sum_full_data(j, g) GE peak_count) THEN peak_ESA(j) = g
    ENDFOR
ENDFOR
print, 'Finished summing over deflection steps.'


; Divide through by a flat efficiency for both main and s-channel based on pre-launch calibration.
IF sat EQ 'A' THEN BEGIN
	sum_full_data = sum_full_data/0.80
	sum_delta_counts = sum_delta_counts/0.80
ENDIF
IF sat EQ 'B' THEN BEGIN
	sum_full_data = sum_full_data/0.48
	sum_delta_counts = sum_delta_counts/0.48
ENDIF
 
print, 'Divison by efficiency completed.'

; Get E/Q and V associated with each ESA step so that distribution functions can be found
; The location of the ESA table must be specified in the crib file.


IF sat EQ 'B' THEN ESA_file = table_location+'ESA_table_B_2009_08_21.txt'
IF sat EQ 'A' THEN ESA_file = table_location+'ESA_table_A_2009_08_17.txt'
openr, in_lun, ESA_file, /GET_LUN
line = " "
energy = fltarr(128) ; changed from double to float (August 2009)
energy(*) = !values.f_nan
v = fltarr(128) ; changed from double to float (August 2009)
v(*) = !values.f_nan

readf, in_lun, line   ; read past first lines

FOR i = 64, 127 DO BEGIN
    readf, in_lun, line
    parts = STRSPLIT(line, " ,", /EXTRACT)
    energy[i] = parts[1]			;keV/q
    v[i] = parts[2]
ENDFOR

	
close, in_lun
free_lun, in_lun
print, 'Finished reading ESA table.'

; Multiply by another energy dependent "leakage efficiency"
IF sat EQ 'B' THEN BEGIN
FOR ii = 0L, num_records1 - 1 DO BEGIN
	FOR jj = 64, 127 DO BEGIN
		IF jj GE s_chan(ii) THEN BEGIN 
		sum_full_data(ii,jj) = sum_full_data(ii,jj)*eff_B_new(jj)
		sum_delta_counts(ii,jj) = sum_delta_counts(ii,jj)*eff_B_new(jj)
		ENDIF
	ENDFOR
ENDFOR
ENDIF ELSE BEGIN
FOR ii = 0L, num_records1 - 1 DO BEGIN
	FOR jj = 64, 127 DO BEGIN
            IF jj GE s_chan(ii) THEN BEGIN
			sum_full_data(ii,jj) = sum_full_data(ii,jj)*eff_A_new(jj)
			sum_delta_counts(ii,jj) = sum_delta_counts(ii,jj)*eff_A_new(jj)
		ENDIF
	ENDFOR
ENDFOR
ENDELSE



; Now multiply each ESA step by (delta theta*delta phi)/(v^2) to get distribution function
dist_funct = dblarr(128,num_records1)
dist_funct(*,*) = !values.d_nan
peak_dist = intarr(num_records1)
peak_dist(*) = -1
delta_dist_funct = dblarr(128,num_records1)
delta_dist_funct(*,*) = !values.d_nan

FOR i = 0L, num_records1-1 DO BEGIN
    peak_count = 0.0
    FOR j = 0, 127 DO BEGIN
        IF sum_full_data(i, j) GT 0 THEN BEGIN
            dist_funct(j, i) = 0.0218*0.7854*sum_full_data(i, j)/((v(j))^2.0) ; changed 2 to 2.0 13 Jan, 2009
	    delta_dist_funct(j,i) = (0.0218*0.7854/((v(j))^2.0))*SQRT(sum_delta_counts(i, j)^2.0+$
                                                                      ((2.0*sum_full_data(i, j)/v(j))^2.0)*(0.05*v(j))^2.0)
        ENDIF ELSE BEGIN
            dist_funct(j, i) = 0.0
	    delta_dist_funct(j,i) = 0.0
        ENDELSE

; Average repeated ESA steps
		IF sat EQ 'B' THEN BEGIN
			dist_funct(122,i) = DOUBLE(0.5*(dist_funct(121,i)+dist_funct(123,i)))
			dist_funct(125,i) = DOUBLE(0.5*(dist_funct(124,i)+dist_funct(126,i)))
			delta_dist_funct(122,i) = DOUBLE(0.5*SQRT((delta_dist_funct(121,i))^2.0+(delta_dist_funct(123,i))^2.0))
			delta_dist_funct(125,i) = DOUBLE(0.5*SQRT((delta_dist_funct(124,i))^2.0+(delta_dist_funct(126,i))^2.0))
		ENDIF ELSE BEGIN
			dist_funct(116,i) = DOUBLE(0.5*(dist_funct(115,i)+dist_funct(117,i)))
			dist_funct(118,i) = DOUBLE(0.5*(dist_funct(117,i)+dist_funct(119,i)))
			dist_funct(123,i) = DOUBLE(0.5*(dist_funct(122,i)+dist_funct(124,i)))
			dist_funct(126,i) = DOUBLE(0.5*(dist_funct(125,i)+dist_funct(127,i)))
			delta_dist_funct(116,i) = DOUBLE(0.5*SQRT((delta_dist_funct(115,i))^2.0+(delta_dist_funct(117,i))^2.0))
			delta_dist_funct(118,i) = DOUBLE(0.5*SQRT((delta_dist_funct(117,i))^2.0+(delta_dist_funct(119,i))^2.0))
			delta_dist_funct(123,i) = DOUBLE(0.5*SQRT((delta_dist_funct(122,i))^2.0+(delta_dist_funct(124,i))^2.0))
			delta_dist_funct(126,i) = DOUBLE(0.5*SQRT((delta_dist_funct(125,i))^2.0+(delta_dist_funct(127,i))^2.0))
		ENDELSE

        
        IF (dist_funct(j, i) GT peak_count) THEN peak_count = dist_funct(j, i)
        IF (dist_funct(j, i) GE peak_count) THEN peak_dist(i) = j
    ENDFOR
ENDFOR

sum_full_data = 0 ; added 28 Sept, 2007 to use save memory
sum_delta_counts = 0


; To make it easier for IDL to get a good fit to the curve, multiply the distribution by
; 1e5.  Remember to divide the density by this factor later on!
dist_funct = dist_funct*(1.0E5)
delta_dist_funct = delta_dist_funct*(1.0E5)
print, 'Distribution functions complete.'

; Fit the distribution function versus velocity data to a 1-D Gaussian.
coeff = dblarr(num_records1,3) ; A matrix in which to store the fit coefficients (changed from double to float 28 Sept, 2007)
coeff(*,*,*) = !values.d_nan
density = dblarr(num_records1)
density(*) = !values.f_nan
avg_v= dblarr(num_records1)
avg_v(*) = !values.f_nan
temp_K = dblarr(num_records1)
temp_K(*) = !values.f_nan
v_th = dblarr(num_records1)
v_th(*) = !values.f_nan
good_fit = fltarr(num_records1)
good_fit(*) = !values.f_nan

k=0L
m=0L

FOR n=0L, num_records1-1 DO BEGIN
	temp = dblarr(num_records1, 3)
	temp(*, *, *) = -1.0
	red_chi = !values.f_nan

	; increased range of points to fit when measure_errors term added on 19 November, 2007
	; added measure_errors term 19 November, 2007
	IF (peak_dist(n) LT 119) AND (peak_dist(n) GE 66) THEN BEGIN
        	k = 128*n+peak_dist(n)-2 ;2 
        	m = 128*n+peak_dist(n)+4 ;4 
        	FIT_MATRIX = GAUSSFIT(double(v(peak_dist(n)-2:peak_dist(n)+4)), double(dist_funct(k:m)), double(temp), $
                  nterms = 3, measure_errors = delta_dist_funct(k:m), chisq=red_chi)
        	coeff(3*n) = temp(0)
        	coeff(3*n+1) = temp(1)
        	coeff(3*n+2) = temp(2)

        	density(n) = coeff(3*n)*((2*!PI)^(0.5))*coeff(3*n+2)*1.0E-11 ; 1/cc ; changed from 2 pi ^ 1/2 which gave a result of 1 (13 August, 2007)
        	avg_v(n) = coeff(3*n+1)*1.0E-3 ; km/s
        	temp_K(n) = (coeff(3*n+2)^2)*(1.67E-27)/(1.38E-23) ;deg K
        	v_th(n) = SQRT(2)*(coeff(3*n+2))*1.0E-3 ; km/s
		good_fit(n) = red_chi

	
        	IF FINITE(temp(1) EQ 0) OR density(n) EQ 0 THEN BEGIN
            		FIT_MATRIX = GAUSSFIT(double(v(peak_dist(n)-4:peak_dist(n)+4)), double(dist_funct(k-2:m)), double(temp), $
                                  nterms = 3, measure_errors = delta_dist_funct(k-2:m),chisq=red_chi)            
            		coeff(3*n) = temp(0)
            		coeff(3*n+1) = temp(1)
            		coeff(3*n+2) = temp(2)
        
            		density(n) = coeff(3*n)*((2*!PI)^(0.5))*coeff(3*n+2)*1.0E-11 ; 1/cc
            		avg_v(n) = coeff(3*n+1)*1.0E-3 ; km/s
            		temp_K(n) = (coeff(3*n+2)^2)*(1.67E-27)/(1.38E-23) ;deg K
            		v_th(n) = SQRT(2)*(coeff(3*n+2))*1.0E-3 ; km/s
	    		good_fit(n) = red_chi
        	ENDIF

	        IF n GE 1 THEN BEGIN
            		IF v_th(n) GE 1.75*v_th(n-1) AND std_error(n) NE 1 THEN std_error(n) = 6
	    		IF v_th(n) LE 0.57*v_th(n-1) AND std_error(n) NE 1 THEN std_error(n) = 6 
        	ENDIF
    	ENDIF

    	IF peak_dist(n) GE 119 AND peak_dist(n) LT 127 THEN BEGIN
        	k = 128*n+peak_dist(n)-2	
        	m = 128*n+127

        	FIT_MATRIX = GAUSSFIT(double(v(peak_dist(n)-2:127)), double(dist_funct(k:m)), double(temp), $
                              nterms = 3, measure_errors = delta_dist_funct(k:m),chisq=red_chi)
        	coeff(3*n) = temp(0)
        	coeff(3*n+1) = temp(1)
        	coeff(3*n+2) = temp(2)

	        density(n) = coeff(3*n)*((2*!PI)^(0.5))*coeff(3*n+2)*1.0E-11 ; 1/cc
        	avg_v(n) = coeff(3*n+1)*1.0E-3 ; km/s
        	temp_K(n) = (coeff(3*n+2)^2)*(1.67E-27)/(1.38E-23) ;deg K
        	v_th(n) = SQRT(2)*(coeff(3*n+2))*1.0E-3 ; km/s
		good_fit(n) = red_chi
        
        	IF error1(n) GT 0 THEN std_error(n) = 2

        	IF FINITE(temp(1) EQ 0) OR density(n) EQ 0 THEN BEGIN
		
            		FIT_MATRIX = GAUSSFIT(double(v(peak_dist(n)-3:127)), double(dist_funct(k-1:m)), double(temp), $
                                  nterms = 3, measure_errors = delta_dist_funct(k-1:m),chisq=red_chi)
            		coeff(3*n) = temp(0)
            		coeff(3*n+1) = temp(1)
            		coeff(3*n+2) = temp(2)
        
            		density(n) = coeff(3*n)*((2*!PI)^(0.5))*coeff(3*n+2)*1.0E-11 ; 1/cc
            		avg_v(n) = coeff(3*n+1)*1.0E-3 ; km/s
            		temp_K(n) = (coeff(3*n+2)^2)*(1.67E-27)/(1.38E-23) ;deg K
            		v_th(n) = SQRT(2)*(coeff(3*n+2))*1.0E-3 ; km/s
	    		good_fit(n) = red_chi
        	ENDIF

	        IF n GE 1 THEN BEGIN
            		IF v_th(n) GE 1.75*v_th(n-1) AND std_error(n) NE 1 THEN std_error(n) = 6
	    		IF v_th(n) LE 0.57*v_th(n-1) AND std_error(n) NE 1 THEN std_error(n) = 6 ; added Feb 2009
        	ENDIF
    	ENDIF

    	IF peak_dist(n) EQ 127 OR peak_dist(n) LT 66 THEN BEGIN	
        	density(n) = !values.f_nan
        	avg_v(n) = !values.f_nan
        	temp_K(n) = !values.f_nan
        	v_th(n) = !values.f_nan
        	std_error(n) = 5
    	ENDIF


ENDFOR

;;; Print raw output file
;IF bkgd EQ 2 THEN BEGIN
;	label_time = time_struct(epoch1(0))
;	label_year = STRING(label_time.year, FORMAT = '(I04)')
;	label_month = STRING(label_time.month, FORMAT = '(I02)')
;	label_day = STRING(label_time.date, FORMAT = '(I02)')
;	outfile_5 = '/home/ksimunac/Output/'+'PLA_ST'+sat+'_L2_1DMax_'+label_year+label_month+label_day+'_'+string(label_time.doy, format = '(I3.3)')+'_raw.txt'
;	openw, out_lun_5, outfile_5, /GET_LUN
;	line = ' '
;	temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+'sec'+tab+'date and time'+tab+$
;		'ST'+sat+' Np [1/cc]'+tab+'ST'+sat+' Vp [km/s]'+tab+'ST'+sat+' Tkin [deg K]'+tab+'ST'+sat+' v_th [km/s]'+tab+'Red. Chi_sq'
;	printf, out_lun_5, temp_string
;
;	FOR nn=0L, num_records1-1 DO BEGIN
;		apid_time = time_struct(epoch1(nn))
;		YEAR = STRING(apid_time.year, FORMAT = '(I04)')
;		DOY = STRING(apid_time.doy, FORMAT = '(I03)')
;		HOUR = STRING(apid_time.hour, FORMAT = '(I02)')
;		MIN = STRING (apid_time.min, FORMAT = '(I02)')
;		SEC = STRING(apid_time.sec, FORMAT = '(I2.2)')
;		Np = STRING(density(nn), FORMAT = '(F0.2)')
;		V_bulk = STRING(round(avg_v(nn)), FORMAT = '(I4)')
;		T = STRING(temp_K(nn), FORMAT = '(E10.2)')
;		V_TEMP = STRING(v_th(nn), FORMAT = '(F0.1)')
;		FIT_GOOD = STRING(good_fit(nn), FORMAT = '(F0.2)')
;
;		temp_string = YEAR+tab+DOY+tab+HOUR+tab+MIN+tab+SEC+tab+STRING(apid_time.month, format ='(I2.2)')+'/'+$
;		STRING(apid_time.date, format = '(I2.2)')+'/'+string(apid_time.year, format = '(I4.4)')+' '+$
;		STRING(apid_time.hour, format = '(I2.2)')+':'+STRING(apid_time.min, format = '(I2.2)')+':'+$
;		STRING(apid_time.sec, format = '(I2.2)')+tab+Np+tab+V_bulk+tab+T+tab+V_TEMP+tab+FIT_GOOD
;
;		IF apid_time.year NE 1970 THEN printf, out_lun_5, temp_string
;	ENDFOR
;
;	CLOSE, out_lun_5
;	FREE_LUN, out_lun_5
;ENDIF



; CHECK FOR PROBLEMS (non-physical results)
; hand-checked bad times
IF sat EQ 'A' THEN BEGIN  ;outliers
   bad_i = where(time_string(epoch1) EQ '2007-03-11/23:26:33' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-04-13/06:20:26' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-04-13/06:46:26' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-04-17/08:58:26' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-04-17/09:40:26' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-04-17/10:30:26' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-05-07/19:07:23' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-05-22/05:26:21' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-14/03:58:18' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-14/04:03:18' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-18/02:42:18' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-18/02:42:18' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/10:49:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/15:26:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/15:28:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/15:44:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/15:45:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/15:48:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/15:49:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-06-30/17:18:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-07-11/13:51:54' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-07-11/14:34:54' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-07-11/14:36:54' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-09-13/14:18:45' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-09-15/12:00:45' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-09-23/05:46:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-09-23/05:47:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-10-19/01:39:40' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-11-21/05:38:36' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-12/19:48:31' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-19/00:28:30' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-19/00:34:30' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-19/00:36:30' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-19/00:41:30' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-19/00:44:30' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2007-12-19/00:45:30' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-03-01/06:50:19' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-03-01/06:51:19' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-03-01/07:06:19' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-03-01/07:26:19' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-03-04/01:27:19' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-03-28/18:26:16' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-04-02/05:39:15' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-04-02/16:41:15' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-04-24/22:27:11' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-04-24/23:02:11' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-04-25/00:14:11' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-05-21/14:48:08' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-06-01/11:46:06' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-08-01/17:43:57' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-08-21/09:09:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-08-21/09:11:55' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:03:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:06:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:09:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:10:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:12:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:13:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:15:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:27:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:31:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-09-22/22:33:51' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-12/14:32:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-12/14:36:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-12/14:41:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-13/05:29:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-13/06:22:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-13/06:44:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-13/06:58:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-13/07:17:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-13/08:52:47' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-19/04:08:46' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-10-31/23:36:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-11-09/14:05:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-11-09/14:07:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-11-09/14:25:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2008-11-09/15:07:44' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-17/20:02:28' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-24/09:06:27' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-24/09:08:27' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-24/09:09:27' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-24/09:17:27' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-24/09:22:27' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-02-24/09:23:27' OR $ ; KDCS 10/08/2009
                 time_string(epoch1) EQ '2009-04-20/15:46:19', $ ; KDCS 10/08/2009
                 bad_count)
   IF bad_count GT 0 THEN BEGIN 
       avg_v    [bad_i] = !values.f_nan
       density  [bad_i] = !values.f_nan
       temp_K   [bad_i] = !values.f_nan
       v_th     [bad_i] = !values.f_nan
       good_fit [bad_i] = !values.f_nan
       std_error[bad_i] = 8
   ENDIF
ENDIF 

IF sat EQ 'B' THEN BEGIN
    bad_i = where(time_string(epoch1) EQ '2007-03-01/00:06:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:18:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:19:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:25:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:36:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:37:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:40:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:41:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:43:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:46:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:47:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:48:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:53:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-01/00:54:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-03/11:24:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-06/07:02:41' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-06/16:20:41' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-06/16:21:41' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-06/16:23:41' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-07/15:28:40' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-07/15:29:40' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-07/17:45:40' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-07/18:15:40' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-07/19:13:40' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-10/17:01:39' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-10/17:03:39' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-12/00:04:39' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-13/15:35:38' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-13/15:44:38' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-13/22:36:38' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-14/17:02:38' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-03-25/12:30:33' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/05:36:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:10:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:11:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:13:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:16:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:26:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:32:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:33:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:36:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:37:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/08:41:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/09:10:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-04-23/09:37:24' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/14:42:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/15:03:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/15:54:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/17:43:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/17:46:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/17:49:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/17:59:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/18:28:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/18:42:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/20:35:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-07/20:53:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-08/02:01:16' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-08/02:06:16' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-08/02:12:16' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-08/02:19:16' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-19/11:49:11' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-19/11:58:11' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-20/01:47:11' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-05-24/16:08:09' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-06-02/16:59:05' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-04/02:55:10' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-10/20:49:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/00:07:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/00:54:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/00:57:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/01:12:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/01:17:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/01:30:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/02:02:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/02:22:07' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/03:31:06' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/03:34:06' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-07-11/03:37:06' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-08-06/19:03:54' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-08-06/19:46:54' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-08-18/01:31:48' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-08-18/14:26:47' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-10-17/20:08:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-12-16/13:53:55' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-12-19/12:29:54' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2007-12-22/12:11:53' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-01-04/09:09:49' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-01-09/01:07:48' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-01-11/18:04:47' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-01-26/03:20:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-01-30/05:44:44' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-05/23:27:43' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-09/12:00:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-09/12:49:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-09/13:13:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-09/15:36:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-09/15:39:42' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-02-28/03:29:38' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-03-01/02:08:37' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-03-01/02:13:37' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-03-23/22:17:32' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-04-04/05:11:30' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-04-11/09:14:28' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-04-11/09:30:28' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-04-15/11:03:27' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-05-13/15:08:19' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-05-20/00:40:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-05-20/00:41:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-05-20/00:59:17' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-05-27/17:56:15' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-06-07/19:33:11' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-06-07/20:51:11' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-06-15/00:14:08' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-07-11/01:48:58' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-08-07/18:21:47' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-09-01/18:18:35' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-10-17/08:52:19' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-10-17/09:03:19' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-10-17/09:16:19' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-10-28/15:29:14' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-10-28/15:31:14' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-10-30/13:45:13' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-11-22/09:54:05' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-11-25/10:31:05' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-11-25/11:18:05' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-11-25/11:43:05' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-11-25/11:44:05' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-12-01/02:39:04' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-12-01/02:53:04' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-12-01/06:32:04' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2008-12-23/01:53:58' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-01-27/20:01:51' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-17/17:02:46' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-17/17:09:46' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/07:08:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/07:34:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/07:43:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/07:48:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/07:50:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/07:51:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/08:35:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/08:54:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/09:02:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/09:16:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/09:33:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/13:43:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-03-31/13:55:45' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-04-18/17:19:43' OR $ ; KDCS 10/08/2009
                  time_string(epoch1) EQ '2009-05-03/16:01:41', $ ; KDCS 10/08/2009 
                  bad_count)
    IF bad_count GT 0 THEN BEGIN 
        avg_v    [bad_i] = !values.f_nan
        density  [bad_i] = !values.f_nan
        temp_K   [bad_i] = !values.f_nan
	v_th     [bad_i] = !values.f_nan
	good_fit [bad_i] = !values.f_nan
        std_error[bad_i] = 8
    ENDIF
ENDIF 

; Check for 3 standard deviations or non-physical results
FOR k = 0L, num_records1 - 1 DO BEGIN
    IF avg_v(k) GT 2000 OR avg_v(k) LE 230 THEN std_error(k) = 1 ; changed lower limit to 230 per Toni 3/23/2009 
    IF density(k) LT 0.001 OR density(k) GE 500 THEN std_error(k) = 1
    IF (v_th(k) GT 400) THEN std_error(k) = 1

    IF sat EQ 'B' AND v_th(k) LE 12 THEN BEGIN
	IF std_error(k) NE 1 THEN std_error(k) = 7
    ENDIF

    IF sat EQ 'A' AND v_th(k) LE 10 THEN BEGIN
	IF std_error(k) NE 1 THEN std_error(k) = 7
    ENDIF

    IF error1(k) GT 0 THEN std_error(k) = 2

    IF FINITE(avg_v(k)) EQ 0 AND std_error(k) NE 5 THEN std_error(k) = 1
   ; IF FINITE(density(k)) EQ 0 AND std_error(k) NE 5 THEN std_error(k) = 1
   ; IF FINITE(v_th(k)) EQ 0 AND std_error(k) NE 5 THEN std_error(k) = 1
    IF FINITE(good_fit(k)) EQ 0 THEN std_error(k) = 1 ; added Feb 2009

    IF std_error(k) EQ 1 THEN BEGIN
	avg_v[k] = !values.f_nan
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan
    ENDIF

    IF good_fit(k) GE 5 OR std_error(k) EQ 7 THEN BEGIN
	density(k) = !values.f_nan
	temp_K(k) = !values.f_nan
	v_th(k) = !values.f_nan
	good_fit(k) = !values.f_nan
    ENDIF
    
    IF FINITE(density(k)) EQ 1 THEN BEGIN
        IF k GE 29 AND k LE num_records1-31 THEN BEGIN
	IF FINITE(density(k+1)) EQ 1 AND FINITE(density(k+2)) EQ 1 THEN BEGIN
            mean_density = MEAN(density(k-29:k+30), /NAN)
            mean_avg_v = MEAN(avg_v(k-29:k+30), /NAN)
            mean_v_th = MEAN(v_th(k-29:k+30), /NAN)
	  
            stdev_density = STDDEV(density(k-29:k+30), /NAN)
            stdev_avg_v = STDDEV(avg_v(k-29:k+30), /NAN)
            stdev_v_th = STDDEV(v_th(k-29:k+30), /NAN)

            IF ABS(density(k) - mean_density) GE 3*stdev_density AND std_error(k) NE 1 THEN std_error(k) = 4
            IF ABS(avg_v(k) - mean_avg_v) GE 3*stdev_avg_v AND std_error(k) NE 1 THEN std_error(k) = 4
            IF ABS(v_th(k) - mean_v_th) GE 3*stdev_v_th AND std_error(k) NE 1 THEN std_error(k) = 4
	ENDIF
        ENDIF
			
    ENDIF

ENDFOR


; take off extra time on either side

; take extra time off bkgd 1 density values so that percent difference calculation will be matching
; up in-synch records
 
IF bkgd EQ 1 THEN BEGIN
    temp_start = time_double(time)
    temp_stop = temp_start+(num_days*24L*60*60)
    wanted_i = where(epoch1 GE temp_start AND epoch1 LT temp_stop, wanted_count)
    IF wanted_count LT 1 THEN BEGIN
        print, 'Error: calc_proton_bulk: No valid data.'
        have_data = 0
    ENDIF ELSE BEGIN 
        density_1 = density[wanted_i]
    ENDELSE 
ENDIF



IF bkgd EQ 2 THEN BEGIN
    temp_start = time_double(time)
    temp_stop = temp_start+(num_days*24L*60*60)
    wanted_i = where(epoch1 GE temp_start AND epoch1 LT temp_stop, wanted_count)
    IF wanted_count LT 1 THEN BEGIN
        print, 'Error: calc_proton_bulk: No valid data.'
        have_data = 0
    ENDIF ELSE BEGIN 
        small_epoch1 = epoch1[wanted_i]
        small_density = density[wanted_i]
        small_avg_v = avg_v[wanted_i]
        small_temp_K = temp_K[wanted_i]
        small_v_th = v_th[wanted_i]
        small_MAIN_MAX = MAIN_MAX[wanted_i]
        small_S_MAX = S_MAX[wanted_i]
        small_std_error = std_error[wanted_i]
	small_good_fit = good_fit[wanted_i]
        new_num_records = wanted_count
    ENDELSE 

    IF have_data EQ 1 THEN BEGIN 

        SMALL_S_MAX_SCALED = dblarr(new_num_records)
        SMALL_S_MAX_SCALED(*) = !values.f_nan

        FOR x = 0L, new_num_records-1 DO BEGIN
            IF SMALL_MAIN_MAX(x) EQ 0 THEN SMALL_MAIN_MAX(x) = !values.f_nan
            IF SMALL_S_MAX(x) EQ 0 THEN SMALL_S_MAX(x) = !values.f_nan		
        ENDFOR
        IF SAT EQ 'A' THEN BEGIN
            SMALL_MAIN_MAX = -23.25+1.5*SMALL_MAIN_MAX ; convert to angle (degrees) from bin number
            SMALL_S_MAX = -19.375 + 1.25*SMALL_S_MAX  ; convert to angle (degrees) from bin number
            SMALL_S_MAX_SCALED = 3.0*(SMALL_S_MAX + 3.5) ; modified 27 August, 2008  
                        ; 3.5 if shift center of distribution, 3.0 is to spread out distribution      
            
            FOR k = 0L, new_num_records-1 DO BEGIN
                                ;SMALL_S_MAX_SCALED(k) = 3.0*(SMALL_S_MAX(k) + 1.72) ; old conversion
                
                                ;SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) - (3.6613E-5)*(small_avg_v(k)^2) + $
                                ;  5.2914E-2*small_avg_v(k)-1.5385E1 + 2.0 ; 
                SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) + (-2.0854E-7)*(small_avg_v(k)^3)+(3.0371E-4)*(small_avg_v(k)^2) + $
                                 (-1.2454E-1)*small_avg_v(k)+1.0983E1 ; modified 27 August, 2008
                   ; this comes from velocity dependence of leakage
                IF ABS(SMALL_S_MAX(k)) GT 20.0 THEN SMALL_S_MAX(k) = !values.f_nan ; added 02 September, 2008
            ENDFOR
        ENDIF ELSE BEGIN
            SMALL_MAIN_MAX = 31.0 - SMALL_MAIN_MAX
            SMALL_S_MAX = 31.0 - SMALL_S_MAX ; flip distribution
            
            SMALL_MAIN_MAX = -23.25+1.5*SMALL_MAIN_MAX
            SMALL_S_MAX = -19.375 + 1.25*SMALL_S_MAX
                                ;SMALL_S_MAX_SCALED = 1.75*(SMALL_S_MAX - 4.43) ; old conversion
            SMALL_S_MAX_SCALED = 1.75*(SMALL_S_MAX - 4.62) ; modified 14 August, 2008
            FOR k = 0L, new_num_records-1 DO BEGIN
                ;SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) + 4.549E-5*(small_avg_v(k)^2)-5.344E-2*small_avg_v(k)+1.469E1-1.0 ; modified 14 August, 2008
                SMALL_S_MAX(k) = SMALL_S_MAX_SCALED(k) + (-2.6404E-7)*(small_avg_v(k)^3) + (4.3726E-4)*(small_avg_v(k)^2) + $
                                 (-2.3922E-1)*small_avg_v(k)+4.2639E1
                IF ABS(SMALL_S_MAX(k)) GT 20.0 THEN SMALL_S_MAX(k) = !values.f_nan ; added 02 September, 2008
            ENDFOR 
        ENDELSE

; Now, need to remove times when entrance system is closed
; --------------------------------------------------------
; get hk data
        sat_hk = [28]
        pl_plot_hk_from_crib, sat_hk, sat, /INST_COORD
        temp_string = 'ST'+sat+'_Analog_HK_esa_cm_dc'
        get_data, temp_string, hk_time, hk_data
; line up data
        good_i = where(finite(hk_time) EQ 1, good_count)
        IF good_count GT 0 THEN BEGIN 
            hk_time = hk_time[good_i] 
            hk_data = hk_data[good_i] 
        ENDIF ELSE stop         ; no good data
        IF array_equal(small_epoch1, hk_time) NE 1 THEN BEGIN ; make sure time arrays line up
            temp_hk_time = dblarr(n_elements(small_epoch1))
            temp_hk_data = dblarr(n_elements(small_epoch1))
            hh = 0L
            rr = 0L
            WHILE rr LT n_elements(small_epoch1) DO BEGIN 
                IF small_epoch1[rr] EQ hk_time[hh] THEN BEGIN ; OK to have more hk values than ra
                    temp_hk_time[rr] = hk_time[hh]
                    temp_hk_data[rr] = hk_data[hh]
                    rr = rr + 1
                    hh = hh + 1
                ENDIF ELSE IF small_epoch1[rr] LT hk_time[hh] THEN rr = rr + 1 ELSE hh = hh + 1
           ENDWHILE 
            hk_time = temp_hk_time
            hk_data = temp_hk_data
        ENDIF 
; remove any indices that were added (where there was ra data but no hk data
        good_i = where(hk_time NE 0, good_count)
        IF good_count NE n_elements(hk_time) THEN BEGIN
            hk_time = hk_time[good_i]
            hk_data = hk_data[good_i]
        ENDIF 
; check for entrance system closed
        num_records = n_elements(small_epoch1)
        check_indices = intarr(num_records) ; make sure we don't overwrite data
        good_i = where(finite(hk_time) EQ 1, good_count) ; good_i[0] is where we start valid data
        temp_start_i = good_i[0]
        hh = 0                  ; for hk_time
        FOR kk = 0, num_records-1 DO BEGIN 
            IF finite(hk_time[hh+temp_start_i]) EQ 1 THEN BEGIN 
                good_i = where(hk_time EQ hk_time[hh+temp_start_i], count)
                IF count NE 1 THEN BEGIN ; check error cases
                    IF count EQ 0 AND small_std_error[kk] GT 0 THEN check_indices[kk] = check_indices[kk]+1 $ ; no hk at this time, but error already set
                    ELSE IF count EQ 0 THEN BEGIN ; check if nearest value was ok (data gap)
                        check_indices[kk] = check_indices[kk]+1
                        offset = 1
                        IF hh-offset GT 0 THEN last_i = where(hk_time EQ hk_time[hh-offset], last_count) ELSE last_count = 0
                        IF hh+offset LT num_records-1 THEN next_i = where(hk_time EQ hk_time[hh+offset], next_count) ELSE $
                          next_count = 0
                        WHILE last_count EQ 0 AND next_count EQ 0 DO BEGIN 
                            offset = offset+1
                            IF hh-offset GT 0 THEN last_i = where(hk_time EQ hk_time[hh-offset], last_count) ELSE last_count = 0
                            IF hh+offset LT num_records-1 THEN next_i = where(hk_time EQ hk_time[hh+offset], next_count) ELSE $
                              next_count = 0
                        ENDWHILE 
                        IF last_count GT 0 THEN BEGIN
                            IF hk_data[last_i] LE 50 THEN BEGIN 
                                small_avg_v[kk] = !values.f_nan
                                small_density[kk] = !values.f_nan
                                small_temp_K[kk] = !values.f_nan
                                small_v_th[kk] = !values.f_nan
                                small_s_max[kk] = !values.f_nan
                                small_std_error[kk] = 5
				small_good_fit[kk] = !values.f_nan
				density_1[kk] = !values.f_nan ; added for synching up records to check percent difference
                            ENDIF 
                        ENDIF ELSE IF next_count GT 0 THEN BEGIN 
                            IF hk_data[next_i] LE 50 THEN BEGIN 
                                small_avg_v[kk] = !values.f_nan
                                small_density[kk] = !values.f_nan
                                small_temp_K[kk] = !values.f_nan
                                small_v_th[kk] = !values.f_nan
                                small_s_max[kk] = !values.f_nan
                                small_std_error[kk] = 5
				small_good_fit[kk] = !values.f_nan
				density_1[kk] = !values.f_nan ; added for synching up records to check percent difference
                            ENDIF 
                        ENDIF ELSE stop ; no valid values found in either direction, what should I do?
                    ENDIF ELSE stop ; count gt 1?
                ENDIF ELSE BEGIN ; count eq 1
                    check_indices[kk] = check_indices[kk]+1
                    IF hk_data[good_i] LE 50 THEN BEGIN
                        small_avg_v[kk] = !values.f_nan
                        small_density[kk] = !values.f_nan
                        small_temp_K[kk] = !values.f_nan
                        small_v_th[kk] = !values.f_nan
                        small_s_max[kk] = !values.f_nan
                        small_std_error[kk] = 5
			small_good_fit[kk] = !values.f_nan
			density_1[kk] = !values.f_nan ; added for synching up records to check percent difference
                    ENDIF
                ENDELSE 
            ENDIF ELSE IF finite(small_epoch1[kk]) EQ 0 THEN BEGIN ; time is invalid in error and ra
                check_indices[kk] = check_indices[kk]+1
                small_avg_v[kk] = !values.f_nan
                small_density[kk] = !values.f_nan
                small_temp_K[kk] = !values.f_nan
                small_v_th[kk] = !values.f_nan
                small_s_max[kk] = !values.f_nan
  		small_std_error[kk] = 5
		small_good_fit[kk] = !values.f_nan
		density_1[kk] = !values.f_nan ; addedfor synching up records to check percent difference
            ENDIF 
             IF kk LT num_records-1 AND hh LT n_elements(hk_time)-1 THEN IF small_epoch1[kk+1] GT hk_time[hh] THEN hh = hh + 1 ; 8/3/09
        ENDFOR 
        IF array_equal(check_indices, 1) NE 1 THEN stop

;; Get roll information and spacecraft coordinates
        good_i = where(finite(small_epoch1) EQ 1, good_count)
        IF good_count GT 0 THEN BEGIN
            temp_epoch_string = time_string(small_epoch1[good_i])
            strput, temp_epoch_string, 'T', 10 ; change time string format for SolarSoft
            roll = get_stereo_roll(temp_epoch_string, sat, yaw, pitch, system = 'HERTN')
            attitude_flag = intarr(num_records, 3) ; roll, yaw, pitch
            flag_i = where(roll  GT 1 OR roll  LT -1, count)
            IF count GT 0 THEN attitude_flag[flag_i, 0] = 1
            temp_i = where(yaw LT 0, temp_count)
            IF temp_count GT 0 THEN yaw[temp_i] = yaw[temp_i] + 360
            flag_i = where(yaw GT 181 OR yaw LT 179, count)
            IF count GT 0 THEN attitude_flag[flag_i, 1] = 1
            flag_i = where(pitch GT 1 OR pitch LT -1, count)
            IF count GT 0 THEN attitude_flag[flag_i, 2] = 1
            carr_num = get_stereo_carr_rot(temp_epoch_string, sat)
            temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'carrington', /degrees)
            carr_lon = reform(temp_value[1, *])
            temp_i = where(carr_lon LT 0, temp_c)
            IF temp_c GT 0 THEN carr_lon[temp_i] = carr_lon[temp_i] + 360 ; range should be 0-360
            helio_dist = reform(temp_value[0, *])
            temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'HEE', /degrees)
            hee_lon = reform(temp_value[1, *])
            hee_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'HEEQ', /degrees)
            heeq_lon = reform(temp_value[1, *])
            heeq_lat = reform(temp_value[2, *])
            temp_value = get_stereo_lonlat(temp_epoch_string, sat, system = 'HCI', /degrees)
            hci_lon = reform(temp_value[1, *])
            temp_i = where(hci_lon LT 0, temp_c)
            IF temp_c GT 0 THEN hci_lon[temp_i] = hci_lon[temp_i] + 360 ; range should be 0-360
            hci_lat = reform(temp_value[2, *])
        ENDIF 
    ENDIF 
ENDIF 

; print 1 minute data to file
; ----------------------------


IF bkgd EQ 2 AND have_data EQ 1 THEN BEGIN
    small_density_2 = small_density
    IF sat EQ 'A' THEN kev_time = small_epoch1 + 41.6
    IF sat EQ 'B' THEN kev_time = small_epoch1 + 42.0
    percent_diff = dblarr(N_ELEMENTS(small_density_2))
    percent_diff(*) = !values.f_nan
    caution_flag = intarr(N_ELEMENTS(small_density_2))
    caution_flag(*) = 0
    time_i = 0
    WHILE finite(small_epoch1[time_i]) EQ 0 DO BEGIN 
        time_i = time_i + 1
    ENDWHILE 
    label_time = time_struct(small_epoch1(time_i))
    label_year = STRING(label_time.year, FORMAT = '(I04)')
    label_month = STRING(label_time.month, FORMAT = '(I02)')
    label_day = STRING(label_time.date, FORMAT = '(I02)')
 


    good_indices = intarr(n_elements(small_density_2)) ; set to 1 if data should be included
    FOR p = 0L, N_ELEMENTS(small_density_2) - 1 DO BEGIN
       	    percent_diff(p) = (density_1(p)-small_density_2(p))/small_density_2(p)
           IF finite(density_1[p]) EQ 0 OR finite(small_density_2[p]) EQ 0 THEN caution_flag[p] = 2
        IF ABS(percent_diff(p)) GE 0.1 THEN caution_flag(p) = 2
        IF ABS(percent_diff(p)) GE 0.05 AND ABS(percent_diff(p)) LT 0.1 THEN caution_flag(p) = 1 
        
	IF small_std_error(p) EQ 1 OR small_std_error(p) EQ 5 THEN BEGIN
            small_density(p) = !values.f_nan
            small_avg_v(p) = !values.f_nan
            small_temp_K(p) = !values.f_nan
            small_v_th(p) = !values.f_nan
            SMALL_S_MAX(p) = !values.f_nan
	    small_good_fit(p) = !values.f_nan
        ENDIF	
    
	
        IF finite(small_avg_v(p)) EQ 1 THEN good_indices[p] = 1 ; CHANGED from small_density to small_avg_v FEB 2009
 	
    ENDFOR	

    labels = ' '
    store_data, 'density', data = {x:small_epoch1, y:small_density}, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, 'speed', data = {x:small_epoch1, y:small_avg_v}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'temperature', data = {x:small_epoch1, y:small_temp_K}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'v_thermal', data = {x:small_epoch1, y:small_v_th}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'north_south', data = {x:small_epoch1, y:SMALL_S_MAX}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, '1_keV_time', data = {x:small_epoch1, y:kev_time}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'error_code', data = {x:small_epoch1, y:small_std_error}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'caution_code', data = {x:small_epoch1, y:caution_flag}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'attitude_flag', data = {x:small_epoch1, y:attitude_flag}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'reduced_chi_squared', data = {x:small_epoch1, y:small_good_fit}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_rotation', data = {x:small_epoch1, y:carr_num}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_longitude', data = {x:small_epoch1, y:carr_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'heliocentric_distance', data = {x:small_epoch1, y:helio_dist}, dlim = {ylog:0, labels:labels, panel_size:2.}
    
    store_data, 'spacecraft_longitude_HEE', data = {x:small_epoch1, y:hee_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEE', data = {x:small_epoch1, y:hee_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEEQ', data = {x:small_epoch1, y:heeq_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEEQ', data = {x:small_epoch1, y:heeq_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HCI', data = {x:small_epoch1, y:hci_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HCI', data = {x:small_epoch1, y:hci_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}

    print, 'Calculating hourly averages.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;Calculate hourly averages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    k = FIX(new_num_records/60)+1
    small_density_copy1 = small_density
    small_avg_v_copy1 = small_avg_v
    small_temp_K_copy1 = small_temp_K
    small_v_th_copy1 = small_v_th
    SMALL_S_MAX_copy1 = SMALL_S_MAX

    hour_start_time = dblarr(k)
    hour_start_time(*) = !values.d_nan
    hour_small_density = dblarr(k)
    hour_small_density(*) = !values.f_nan
    hour_speed = dblarr(k)
    hour_speed(*) = !values.f_nan
    hour_temp = dblarr(k)
    hour_temp(*) = !values.f_nan
    hour_small_v_th = dblarr(k)
    hour_small_v_th(*) = !values.f_nan
    hour_count = dblarr(k)
    hour_count(*) = !values.f_nan
    hour_count_v = dblarr(k)
    hour_count_v(*) = !values.f_nan
    hour_count_ns = dblarr(k)
    hour_count_ns(*) = !values.f_nan
    hour_ns = dblarr(k)
    hour_ns(*) = !values.f_nan
    hour_error = intarr(k)
    hour_caution = intarr(k)
    hour_attitude = intarr(k, 3)
    hour_red_chi = dblarr(k)
    hour_red_chi(*) = !values.f_nan
    hour_carr_num = dblarr(k)
    hour_carr_num(*) = !values.f_nan
    hour_carr_lon = dblarr(k)
    hour_carr_lon(*) = !values.f_nan
    hour_helio_dist = dblarr(k)
    hour_helio_dist(*) = !values.f_nan
    hour_hee_lon = dblarr(k)
    hour_hee_lon(*) = !values.f_nan
    hour_hee_lat = dblarr(k)
    hour_hee_lat(*) = !values.f_nan
    hour_heeq_lon = dblarr(k)
    hour_heeq_lon(*) = !values.f_nan
    hour_heeq_lat = dblarr(k)
    hour_heeq_lat(*) = !values.f_nan
    hour_hci_lon = dblarr(k)
    hour_hci_lon(*) = !values.f_nan
    hour_hci_lat = dblarr(k)
    hour_hci_lat(*) = !values.f_nan
    ih = 0 ; index into 1 hour data
    found_first = 0
    on_first = 1
    FOR im = 0L, n_elements(small_epoch1)-1 DO BEGIN ; index into 1 minute data
        now_time = time_struct(small_epoch1(im))
        IF now_time.year NE 1970 THEN BEGIN ; skip invalid times
            IF found_first EQ 0 THEN BEGIN 
                base_1hr = now_time.hour
            ENDIF 
            now_1hr = now_time.hour
            IF now_time.hour NE base_1hr OR found_first EQ 0 THEN BEGIN ; new base
                found_first = 1
                ;IF ih GT 0 THEN BEGIN ; write old variables
                IF on_first EQ 0 THEN BEGIN ; write old variables
                    hour_error[ih] = temp_error
                    hour_caution[ih] = temp_caution
                    hour_attitude[ih, *] = temp_attitude
                    IF n_elements(red_chi) GT 0 AND finite(temp_red_chi) EQ 1 THEN hour_red_chi[ih] = temp_red_chi
                    IF n_elements(temp_density) GT 0 AND FINITE(temp_density) EQ 1 THEN  hour_small_density[ih] = temp_density/sixty_count
                    IF n_elements(temp_speed) GT 0 AND FINITE(temp_speed) EQ 1 THEN hour_speed[ih] = temp_speed/sixty_count_v
                    IF n_elements(temp_temp) GT 0 AND FINITE(temp_temp) EQ 1 THEN hour_temp[ih] = temp_temp/sixty_count
                    IF n_elements(temp_v_th) GT 0 AND FINITE(temp_v_th) EQ 1 THEN hour_small_v_th[ih] = temp_v_th/sixty_count
                    IF n_elements(temp_ns) GT 0 AND FINITE(temp_ns) EQ 1 THEN hour_ns[ih] = temp_ns/sixty_count_ns
                    hour_count[ih] = sixty_count
		    hour_count_v[ih] = sixty_count_v
		    hour_count_ns[ih] = sixty_count_ns
                ENDIF 
                base_1hr = now_1hr
                IF on_first EQ 1 THEN on_first = 0 ELSE ih = ih + 1 ; so don't get empty first record
                IF ih EQ n_elements(hour_start_time) THEN BEGIN  ; need to resize arrays
		    tmp_arr  = intarr(10)
                    tmp2_arr = intarr(10, 3)
                    hour_error = [hour_error, tmp_arr]
                    hour_caution = [hour_caution, tmp_arr]
                    hour_attitude = [hour_attitude, tmp2_arr]
		    tmp_arr = dblarr(10)
		    tmp_arr[*] = !values.f_nan
                    hour_count = [hour_count, tmp_arr]
		    hour_count_v = [hour_count_v, tmp_arr]
		    hour_count_ns = [hour_count_ns, tmp_arr]
                    hour_start_time = [hour_start_time, tmp_arr]
                    hour_red_chi = [hour_red_chi, tmp_arr]
                    hour_carr_num = [hour_carr_num, tmp_arr]
                    hour_carr_lon = [hour_carr_lon, tmp_arr]
                    hour_helio_dist = [hour_helio_dist, tmp_arr]
                    hour_hee_lon = [hour_hee_lon, tmp_arr]
                    hour_hee_lat = [hour_hee_lat, tmp_arr]
                    hour_heeq_lon = [hour_heeq_lon, tmp_arr]
                    hour_heeq_lat = [hour_heeq_lat, tmp_arr]
                    hour_hci_lon = [hour_hci_lon, tmp_arr]
                    hour_hci_lat = [hour_hci_lat, tmp_arr]
                    hour_small_density = [hour_small_density, tmp_arr]
                    hour_speed = [hour_speed, tmp_arr]
                    hour_temp = [hour_temp, tmp_arr]
                    hour_small_v_th = [hour_small_v_th, tmp_arr]
                    hour_ns = [hour_ns, tmp_arr]
                ENDIF 
                ;hour_start_time[ih] = time_double(string(now_time.year, format = '(I4.4)')+'-'+string(now_time.month, format = '(I2.2)')+'-'+$
                ;                                  string(now_time.date, format = '(I2.2)')+'/'+string(now_time.hour, format = '(I2.2)')+$
                ;                                  ':00:00')
                hour_start_time[ih] = small_epoch1[im] ; LBE changed 9/11/09 to make actual start time
                hour_carr_num[ih] = carr_num[im]
                hour_carr_lon[ih] = carr_lon[im]
                hour_helio_dist[ih] = helio_dist[im]
                hour_hee_lon[ih] = hee_lon[im]
                hour_hee_lat[ih] = hee_lat[im]
                hour_heeq_lon[ih] = heeq_lon[im]
                hour_heeq_lat[ih] = heeq_lat[im]
                hour_hci_lon[ih] = hci_lon[im]
                hour_hci_lat[ih] = hci_lat[im]
                IF finite(small_density_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 $
                  THEN temp_density = small_density_copy1[im] ELSE temp_density = !values.f_nan 
                IF finite(small_avg_v_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 $
                  THEN temp_speed = small_avg_v_copy1[im] ELSE temp_speed = !values.f_nan
                IF finite(small_temp_K_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 $
                  THEN temp_temp = small_temp_K_copy1[im] ELSE temp_temp = !values.f_nan
                IF finite(small_v_th_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 $
                  THEN temp_v_th = small_v_th_copy1[im] ELSE temp_v_th = !values.f_nan
                IF finite(SMALL_S_MAX_copy1[im]) EQ 1 AND small_std_error[im] NE 5  THEN temp_ns = SMALL_S_MAX_copy1[im] ELSE $
                  temp_ns = !values.f_nan
                IF finite(small_good_fit[im]) EQ 1 THEN temp_red_chi = small_good_fit[im] ELSE temp_red_chi = !values.f_nan
                temp_caution = caution_flag[im]
                temp_attitude = attitude_flag[im, *]
                temp_error = small_std_error[im]
                IF FINITE(temp_density) EQ 1 THEN sixty_count = 1.0 ELSE sixty_count = 0.0  
		IF FINITE(temp_speed) EQ 1 THEN sixty_count_v = 1.0 ELSE sixty_count_v = 0.0 
		IF FINITE(temp_ns) EQ 1 THEN sixty_count_ns = 1.0 ELSE sixty_count_ns = 0.0
            ENDIF ELSE BEGIN   ; further in same 1 hour space
                IF finite(small_density_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 THEN $
                  IF FINITE(temp_density) EQ 0 THEN temp_density = small_density_copy1[im] $
                  ELSE temp_density = temp_density + small_density_copy1[im]
                IF finite(small_avg_v_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 THEN $
                  IF FINITE(temp_speed) EQ 0 THEN temp_speed = small_avg_v_copy1[im] $
                  ELSE temp_speed = temp_speed + small_avg_v_copy1[im]
                IF finite(small_temp_K_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 THEN $
                  IF FINITE(temp_temp) EQ 0 THEN temp_temp = small_temp_K_copy1[im] $
                  ELSE temp_temp = temp_temp + small_temp_K_copy1[im]
                IF finite(small_v_th_copy1[im]) EQ 1 AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 THEN $
                  IF FINITE(temp_v_th) EQ 0 THEN temp_v_th = small_v_th_copy1[im] $
                  ELSE temp_v_th = temp_v_th + small_v_th_copy1[im]
                IF finite(SMALL_S_MAX_copy1[im]) EQ 1 AND small_std_error[im] NE 5 THEN IF FINITE(temp_ns) EQ 0 THEN $
                  temp_ns = SMALL_S_MAX_copy1[im] ELSE temp_ns = temp_ns + SMALL_S_MAX_copy1[im]
                IF finite(small_good_fit[im]) EQ 1 AND $
                  ((finite(temp_red_chi) EQ 1 AND small_good_fit[im] GT temp_red_chi) OR $
                   (finite(temp_red_chi) EQ 0)) THEN temp_red_chi = small_good_fit[im]
                IF caution_flag[im] EQ 2 THEN temp_caution = 2
                IF caution_flag[im] EQ 1 AND temp_caution NE 2 THEN temp_caution = 1
                IF attitude_flag[im, 0] EQ 1 THEN temp_attitude[0] = 1
                IF attitude_flag[im, 1] EQ 1 THEN temp_attitude[1] = 1
                IF attitude_flag[im, 2] EQ 1 THEN temp_attitude[2] = 1
;; check which errors to carry over (priority is 3,7,4,6,2,1,5)
                IF small_std_error[im] EQ 3 THEN temp_error = 3 $
                ELSE IF small_std_error[im] EQ 7 AND temp_error NE 3 THEN temp_error = 7 $
                     ELSE IF small_std_error[im] EQ 4 AND temp_error NE 3 AND temp_error NE 7 THEN temp_error = 4 $
                          ELSE IF small_std_error[im] EQ 6 AND temp_error NE 3 AND temp_error NE 7 AND temp_error NE 4 THEN temp_error = 6 $
                               ELSE IF small_std_error[im] EQ 2 AND (temp_error LT 3 OR temp_error EQ 5) THEN temp_error = 2 $
                                 ELSE IF small_std_error[im] EQ 1 AND (temp_error LT 2 OR temp_error EQ 5) THEN temp_error = 1
                IF (FINITE(small_density_copy1[im]) EQ 1) AND (small_density_copy1[im] NE -1) AND $
                  small_std_error[im] NE 1 AND small_std_error[im] NE 5 THEN $
                  sixty_count = sixty_count+1.0 ; Make sure time step has a valid result
		IF (FINITE(small_avg_v_copy1[im]) EQ 1) AND small_std_error[im] NE 1 AND small_std_error[im] NE 5 $
                  THEN sixty_count_v = sixty_count_v+1.0
		IF (FINITE(SMALL_S_MAX_copy1[im]) EQ 1) AND small_std_error[im] NE 5 THEN sixty_count_ns = sixty_count_ns+1.0
            ENDELSE 
        ENDIF 
    ENDFOR 
;; add final value
    IF ih EQ n_elements(hour_error)         THEN hour_error = [hour_error, temp_error]          ELSE hour_error[ih] = temp_error
    IF ih EQ n_elements(hour_caution)       THEN hour_caution = [hour_caution, temp_caution]    ELSE hour_caution[ih] = temp_caution
    IF ih EQ n_elements(hour_attitude)      THEN hour_attitude = [hour_attitude, temp_attitude] ELSE hour_attitude[ih, *] = temp_attitude
    IF n_elements(temp_density) GT 0 THEN $
      IF ih EQ n_elements(hour_small_density) THEN hour_small_density = [hour_small_density, temp_density/sixty_count] ELSE $
      hour_small_density[ih] = temp_density/sixty_count
    IF n_elements(temp_speed) GT 0 THEN $
      IF ih EQ n_elements(hour_speed) THEN hour_speed = [hour_speed, temp_speed/sixty_count_v] ELSE hour_speed[ih] = temp_speed/sixty_count_v
    IF n_elements(temp_temp) GT 0 THEN $
      IF ih EQ n_elements(hour_temp)  THEN hour_temp = [hour_temp, temp_temp/sixty_count]      ELSE hour_temp[ih] = temp_temp/sixty_count
    IF n_elements(temp_v_th) GT 0 THEN $
      IF ih EQ n_elements(hour_small_v_th) THEN hour_small_v_th = [hour_small_v_th, temp_v_th/sixty_count] ELSE $
      hour_small_v_th[ih] = temp_v_th/sixty_count
    IF n_elements(temp_ns) GT 0 THEN $
      IF ih EQ n_elements(hour_ns) THEN hour_ns = [hour_ns, temp_ns/sixty_count_ns]         ELSE hour_ns[ih] = temp_ns/sixty_count_ns
    IF n_elements(temp_red_chi) GT 0 THEN $
      IF ih EQ n_elements(hour_red_chi) THEN hour_red_chi = [hour_red_chi, temp_red_chi]    ELSE hour_red_chi[ih] = temp_red_chi
    IF ih EQ n_elements(hour_count)    THEN hour_count = [hour_count, sixty_count]          ELSE hour_count[ih] = sixty_count
    IF ih EQ n_elements(hour_count_v)  THEN hour_count_v = [hour_count_v, sixty_count_v]    ELSE hour_count_v[ih] = sixty_count_v
    IF ih EQ n_elements(hour_count_ns) THEN hour_count_ns = [hour_count_ns, sixty_count_ns] ELSE hour_count_ns[ih] = sixty_count_ns

    good_indices = intarr(n_elements(hour_start_time)) ; set to 1 if record should be included
    found_bad_time = 0
    FOR p = 0L, N_ELEMENTS(hour_start_time) - 1 DO BEGIN
        IF finite(hour_start_time(p)) EQ 1 THEN good_indices[p] = 1 ELSE found_bad_time = 1
    ENDFOR
    IF found_bad_time EQ 1 THEN BEGIN
        good_i = where(good_indices EQ 1, good_count)
        IF good_count GT 0 THEN BEGIN 
            hour_start_time    = hour_start_time   [good_i]
            hour_small_density = hour_small_density[good_i]
            hour_speed         = hour_speed        [good_i]
            hour_temp          = hour_temp         [good_i]
            hour_small_v_th    = hour_small_v_th   [good_i]
            hour_ns            = hour_ns           [good_i]
            hour_error         = hour_error        [good_i]
            hour_caution       = hour_caution      [good_i]
            hour_attitude      = hour_attitude     [good_i, *]
            hour_red_chi       = hour_red_chi      [good_i]
            hour_count         = hour_count        [good_i]
            hour_count_v       = hour_count_v      [good_i]
            hour_count_ns      = hour_count_ns     [good_i]
            hour_carr_num      = hour_carr_num     [good_i]
            hour_carr_lon      = hour_carr_lon     [good_i]
            hour_helio_dist    = hour_helio_dist   [good_i]
            hour_hee_lon       = hour_hee_lon      [good_i]
            hour_hee_lat       = hour_hee_lat      [good_i]
            hour_heeq_lon      = hour_heeq_lon     [good_i]
            hour_heeq_lat      = hour_heeq_lat     [good_i]
            hour_hci_lon       = hour_hci_lon      [good_i]
            hour_hci_lat       = hour_hci_lat      [good_i]
        ENDIF ELSE stop ; no valid times!
    ENDIF 

    store_data, 'density_1hr', data = {x:hour_start_time, y:hour_small_density}, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, 'speed_1hr', data = {x:hour_start_time, y:hour_speed}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'temperature_1hr', data = {x:hour_start_time, y:hour_temp}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'v_thermal_1hr', data = {x:hour_start_time, y:hour_small_v_th}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'north_south_1hr', data = {x:hour_start_time, y:hour_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'error_code_1hr', data = {x:hour_start_time, y:hour_error}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'caution_code_1hr', data = {x:hour_start_time, y:hour_caution}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'attitude_flag_1hr', data = {x:hour_start_time, y:hour_attitude}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'reduced_chi_squared_1hr', data = {x:hour_start_time, y:hour_red_chi}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_1hr', data = {x:hour_start_time, y:hour_count}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_1hr_v', data = {x:hour_start_time, y:hour_count_v}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_1hr_ns', data = {x:hour_start_time, y:hour_count_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_rotation_1hr', data = {x:hour_start_time, y:hour_carr_num}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_longitude_1hr', data = {x:hour_start_time, y:hour_carr_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'heliocentric_distance_1hr', data = {x:hour_start_time, y:hour_helio_dist}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEE_1hr', data = {x:hour_start_time, y:hour_hee_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEE_1hr', data = {x:hour_start_time, y:hour_hee_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEEQ_1hr', data = {x:hour_start_time, y:hour_heeq_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEEQ_1hr', data = {x:hour_start_time, y:hour_heeq_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HCI_1hr', data = {x:hour_start_time, y:hour_hci_lon}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HCI_1hr', data = {x:hour_start_time, y:hour_hci_lat}, dlim = {ylog:0, labels:labels, panel_size:2.}


    print, 'Calculating 10 minute averages.'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;Calculate 10 minute averages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    IF new_num_records MOD 10 EQ 0 THEN k = FIX(new_num_records/10) ELSE k = FIX(new_num_records/10)+1
    small_density_copy2 = small_density
    small_avg_v_copy2 = small_avg_v
    small_temp_K_copy2 = small_temp_K
    small_v_th_copy2 = small_v_th
    SMALL_S_MAX_copy2 = SMALL_S_MAX

    ten_min_start_time = dblarr(k)
    ten_min_start_time(*) = !values.d_nan
    ten_min_small_density = dblarr(k)
    ten_min_small_density(*) = !values.f_nan
    ten_min_speed = dblarr(k)
    ten_min_speed(*) = !values.f_nan
    ten_min_temp = dblarr(k)
    ten_min_temp(*) = !values.f_nan
    ten_min_small_v_th = dblarr(k)
    ten_min_small_v_th(*) = !values.f_nan
    ten_min_count = dblarr(k)
    ten_min_count(*) = !values.f_nan
    ten_min_count_v = dblarr(k)
    ten_min_count_v(*) = !values.f_nan
    ten_min_count_ns = dblarr(k)
    ten_min_count_ns(*) = !values.f_nan
    ten_min_ns = dblarr(k)
    ten_min_ns(*) = !values.f_nan
    ten_min_error = intarr(k)
    ten_min_caution = intarr(k)
    ten_min_attitude = intarr(k, 3)
    ten_min_red_chi = dblarr(k)
    ten_min_red_chi(*) = !values.f_nan
    ten_min_carr_num = dblarr(k)
    ten_min_carr_num(*) = !values.f_nan
    ten_min_carr_lon = dblarr(k)
    ten_min_carr_lon(*) = !values.f_nan
    ten_min_helio_dist = dblarr(k)
    ten_min_helio_dist(*) = !values.f_nan
    ten_min_hee_lon = dblarr(k)
    ten_min_hee_lon(*) = !values.f_nan
    ten_min_hee_lat = dblarr(k)
    ten_min_hee_lat(*) = !values.f_nan
    ten_min_heeq_lon = dblarr(k)
    ten_min_heeq_lon(*) = !values.f_nan
    ten_min_heeq_lat = dblarr(k)
    ten_min_heeq_lat(*) = !values.f_nan
    ten_min_hci_lon = dblarr(k)
    ten_min_hci_lon(*) = !values.f_nan
    ten_min_hci_lat = dblarr(k)
    ten_min_hci_lat(*) = !values.f_nan

    i10 = 0 ; index into 10 minute data
    found_first = 0
    on_first = 1
    FOR i1 = 0L, n_elements(small_epoch1)-1 DO BEGIN ; index into 1 minute data
        now_time = time_struct(small_epoch1(i1))
        IF now_time.year NE 1970 THEN BEGIN ; skip invalid times
            IF i10 EQ 0 THEN base_hour = now_time.hour
            IF found_first EQ 0 THEN BEGIN
	       base_10min = now_time.min/10
            ENDIF
            now_10min = now_time.min/10
            IF now_time.hour NE base_hour OR found_first EQ 0 OR $
              ((now_time.hour EQ base_hour) AND (now_10min NE base_10min)) THEN BEGIN ; new base
                base_hour = now_time.hour
		found_first = 1
                ;IF i10 GT 0 THEN BEGIN ; write old variables
                IF on_first EQ 0 THEN BEGIN ; write old variables
                    ten_min_error[i10] = temp_error
                    ten_min_caution[i10] = temp_caution
                    ten_min_attitude[i10, *] = temp_attitude
                    IF n_elements(red_chi) GT 0 AND finite(temp_red_chi) EQ 1 THEN ten_min_red_chi[i10] = temp_red_chi; ELSE ten_min_red_chi[i10] = 0.0
                    IF n_elements(temp_density) GT 0 AND FINITE(temp_density) EQ 1 THEN $
                      ten_min_small_density[i10] = temp_density/ten_count ;ELSE BEGIN
			;ten_min_small_density[i10] = 0.0
			;ten_count = 0.0
			;ENDELSE
                    IF n_elements(temp_speed) GT 0 AND FINITE(temp_speed) EQ 1 THEN ten_min_speed[i10] = temp_speed/ten_count_v ;ELSE BEGIN
			;ten_min_speed[i10] = 0.0
			;ten_count_v = 0.0
			;ENDELSE
                    IF n_elements(temp_temp) GT 0 AND FINITE(temp_temp) EQ 1 THEN ten_min_temp[i10] = temp_temp/ten_count ;ELSE ten_min_temp[i10] = 0.0
                    IF n_elements(temp_v_th) GT 0 AND FINITE(temp_v_th) EQ 1 THEN ten_min_small_v_th[i10] = temp_v_th/ten_count ;ELSE ten_min_small_v_th[i10] = 0.0
                    IF n_elements(temp_ns) GT 0 AND FINITE(temp_ns) EQ 1 THEN ten_min_ns[i10] = temp_ns/ten_count_ns ;ELSE BEGIN
			;ten_min_ns[i10] = 0.0
			;ten_count_ns = 0.0
			;ENDELSE
                    ten_min_count[i10] = ten_count
		    ten_min_count_v[i10] = ten_count_v
		    ten_min_count_ns[i10] = ten_count_ns
                ENDIF 
		base_10min = now_10min
                IF on_first EQ 1 THEN on_first = 0 ELSE i10 = i10 + 1
                IF i10 EQ n_elements(ten_min_start_time) THEN BEGIN  ; need to resize arrays
                    tmp_arr  = intarr(10)
                    tmp2_arr = intarr(10, 3)
                    ten_min_error = [ten_min_error, tmp_arr]
                    ten_min_caution = [ten_min_caution, tmp_arr]
                    ten_min_attitude = [ten_min_attitude, tmp2_arr]
                    tmp_arr = dblarr(10)
		    tmp_arr[*] = !values.f_nan
		    ten_min_count = [ten_min_count, tmp_arr]
		    ten_min_count_v = [ten_min_count_v, tmp_arr]
		    ten_min_count_ns = [ten_min_count_ns, tmp_arr]
                    ten_min_start_time = [ten_min_start_time, tmp_arr]
                    ten_min_red_chi = [ten_min_red_chi, tmp_arr]
                    ten_min_carr_num = [ten_min_carr_num, tmp_arr]
                    ten_min_carr_lon = [ten_min_carr_lon, tmp_arr]
                    ten_min_helio_dist = [ten_min_helio_dist, tmp_arr]
                    ten_min_hee_lon = [ten_min_hee_lon, tmp_arr]
                    ten_min_hee_lat = [ten_min_hee_lat, tmp_arr]
                    ten_min_heeq_lon = [ten_min_heeq_lon, tmp_arr]
                    ten_min_heeq_lat = [ten_min_heeq_lat, tmp_arr]
                    ten_min_hci_lon = [ten_min_hci_lon, tmp_arr]
                    ten_min_hci_lat = [ten_min_hci_lat, tmp_arr]
                    ten_min_small_density = [ten_min_small_density, tmp_arr]
                    ten_min_speed = [ten_min_speed, tmp_arr]
                    ten_min_temp = [ten_min_temp, tmp_arr]
                    ten_min_small_v_th = [ten_min_small_v_th, tmp_arr]
                    ten_min_ns = [ten_min_ns, tmp_arr]
                ENDIF 
                ;ten_min_start_time[i10] = time_double(string(now_time.year, format = '(I4.4)')+'-'+$
                ;                                      string(now_time.month, format = '(I2.2)')+'-'+$
                ;                                      string(now_time.date, format = '(I2.2)')+'/'+$
                ;                                      string(now_time.hour, format = '(I2.2)')+':'+$
                ;                                      string((base_10min*10), format = '(I2.2)')+':00')
                ten_min_start_time[i10] = small_epoch1[i1] ; LBE changed 9/11/09 to make actual start time
                ten_min_carr_num[i10] = carr_num[i1]
                ten_min_carr_lon[i10] = carr_lon[i1]
                ten_min_helio_dist[i10] = helio_dist[i1]
                ten_min_hee_lon[i10] = hee_lon[i1]
                ten_min_hee_lat[i10] = hee_lat[i1]
                ten_min_heeq_lon[i10] = heeq_lon[i1]
                ten_min_heeq_lat[i10] = heeq_lat[i1]
                ten_min_hci_lon[i10] = hci_lon[i1]
                ten_min_hci_lat[i10] = hci_lat[i1]
                IF finite(small_density_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN temp_density = small_density_copy2[i1] ELSE temp_density = !values.f_nan
                IF finite(small_avg_v_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN temp_speed = small_avg_v_copy2[i1] ELSE temp_speed = !values.f_nan
                IF finite(small_temp_K_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN temp_temp = small_temp_K_copy2[i1] ELSE temp_temp = !values.f_nan
                IF finite(small_v_th_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN temp_v_th = small_v_th_copy2[i1] ELSE temp_v_th = !values.f_nan
                IF finite(SMALL_S_MAX_copy2[i1]) EQ 1 AND small_std_error[i1] NE 5 THEN temp_ns = SMALL_S_MAX_copy2[i1] ELSE $
                  temp_ns = !values.f_nan
                IF finite(small_good_fit[i1]) EQ 1 THEN temp_red_chi = small_good_fit[i1] ELSE temp_red_chi = !values.f_nan
                temp_caution = caution_flag[i1]
                temp_attitude = attitude_flag[i1, *]
                temp_error = small_std_error[i1]
                IF FINITE(temp_density) EQ 1 THEN ten_count = 1.0 ELSE ten_count = 0.0
		IF FINITE(temp_speed) EQ 1 THEN ten_count_v = 1.0 ELSE ten_count_v = 0.0
		IF FINITE(temp_ns) EQ 1 THEN ten_count_ns = 1.0 ELSE ten_count_ns = 0.0
            ENDIF ELSE BEGIN    ; further in same 10 min space
                IF finite(small_density_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN IF FINITE(temp_density) EQ 0 THEN temp_density = small_density_copy2[i1] $
                       ELSE temp_density = temp_density + small_density_copy2[i1]
                IF finite(small_avg_v_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN IF FINITE(temp_speed) EQ 0 THEN temp_speed =  small_avg_v_copy2[i1] $
                       ELSE temp_speed = temp_speed + small_avg_v_copy2[i1]
                IF finite(small_temp_K_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN IF FINITE(temp_temp) EQ 0 THEN temp_temp = small_temp_K_copy2[i1] $
                       ELSE  temp_temp = temp_temp + small_temp_K_copy2[i1]
                IF finite(small_v_th_copy2[i1]) EQ 1 AND small_std_error[i1] NE 1 AND small_std_error[i1] NE 5 $
                  THEN IF FINITE(temp_v_th) EQ 0 THEN temp_v_th = small_v_th_copy2[i1] $
                       ELSE temp_v_th = temp_v_th + small_v_th_copy2[i1]
                IF finite(SMALL_S_MAX_copy2[i1]) EQ 1 AND small_std_error[i1] NE 5 THEN IF FINITE(temp_ns) EQ 0 THEN $
                  temp_ns = SMALL_S_MAX_copy2[i1] $
			ELSE temp_ns = temp_ns + SMALL_S_MAX_copy2[i1]
                IF finite(small_good_fit[i1]) EQ 1 AND $
                  ((finite(temp_red_chi) EQ 1 AND small_good_fit[i1] GT temp_red_chi) OR $
                   (finite(temp_red_chi) EQ 0)) THEN temp_red_chi = small_good_fit[i1]
                IF caution_flag[i1] EQ 2 THEN temp_caution = 2
                IF caution_flag[i1] EQ 1 AND temp_caution NE 2 THEN temp_caution = 1
                IF attitude_flag[i1, 0] EQ 1 THEN temp_attitude[0] = 1
                IF attitude_flag[i1, 1] EQ 1 THEN temp_attitude[1] = 1
                IF attitude_flag[i1, 2] EQ 1 THEN temp_attitude[2] = 1
;; check which errors to carry over (priority is 3,7,4,6,2,1,5)
                IF small_std_error[i1] EQ 3 THEN temp_error = 3 $
                ELSE IF small_std_error[i1] EQ 7 AND temp_error NE 3 THEN temp_error = 7 $
                     ELSE IF small_std_error[i1] EQ 4 AND temp_error NE 3 AND temp_error NE 7 THEN temp_error = 4 $
                          ELSE IF small_std_error[i1] EQ 6 AND temp_error NE 3 AND temp_error NE 7 AND temp_error NE 4 THEN temp_error = 6 $
                               ELSE IF small_std_error[i1] EQ 2 AND (temp_error LT 3 OR temp_error EQ 5) THEN temp_error = 2 $
                                    ELSE IF small_std_error[i1] EQ 1 AND (temp_error LT 2 OR temp_error EQ 5) THEN temp_error = 1
                IF (FINITE(small_density_copy2[i1]) EQ 1) AND (small_density_copy2[i1] NE -1) AND small_std_error[i1] NE 1 THEN $
                  ten_count = ten_count+1 ; Make sure time step has a valid result
		IF (FINITE(small_avg_v_copy2[i1]) EQ 1) AND small_std_error[i1] NE 1 THEN ten_count_v = ten_count_v+1
		IF (FINITE(SMALL_S_MAX_copy2[i1]) EQ 1) THEN ten_count_ns = ten_count_ns+1
            ENDELSE 
        ENDIF 
    ENDFOR 
; add final value
    IF i10 EQ n_elements(ten_min_error)    THEN ten_min_error =    [ten_min_error, temp_error]      $
    ELSE ten_min_error[i10] = temp_error
    IF i10 EQ n_elements(ten_min_caution)  THEN ten_min_caution =  [ten_min_caution, temp_caution]  $
    ELSE ten_min_caution[i10] = temp_caution
    IF i10 EQ n_elements(ten_min_attitude) THEN ten_min_attitude = [ten_min_attitude, temp_attitude] $
    ELSE ten_min_attitude[i10, *] = temp_attitude
    IF n_elements(temp_density) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_small_density) THEN ten_min_small_density = [ten_min_small_density, temp_density/ten_count] $
      ELSE ten_min_small_density[i10] = temp_density/ten_count
    IF n_elements(temp_speed)   GT 0 THEN $
      IF i10 EQ n_elements(ten_min_speed)         THEN ten_min_speed = [ten_min_speed, temp_speed/ten_count_v] $
      ELSE ten_min_speed[i10] = temp_speed/ten_count_v
    IF n_elements(temp_temp)    GT 0 THEN $
      IF i10 EQ n_elements(ten_min_temp)          THEN ten_min_temp = [ten_min_temp, temp_temp/ten_count] $
      ELSE ten_min_temp[i10] = temp_temp/ten_count
    IF n_elements(temp_v_th)    GT 0 THEN $
      IF i10 EQ n_elements(ten_min_small_v_th)    THEN ten_min_small_v_th = [ten_min_small_v_th, temp_v_th/ten_count] $
      ELSE ten_min_small_v_th[i10] = temp_v_th/ten_count
    IF n_elements(temp_ns)      GT 0 THEN $
      IF i10 EQ n_elements(ten_min_ns)             THEN ten_min_ns = [ten_min_ns, temp_ns/ten_count_ns] $
      ELSE ten_min_ns[i10] = temp_ns/ten_count_ns
    IF n_elements(temp_red_chi) GT 0 THEN $
      IF i10 EQ n_elements(ten_min_red_chi)        THEN ten_min_red_chi = [ten_min_red_chi, temp_red_chi] $
      ELSE ten_min_red_chi[i10] = temp_red_chi
    IF i10 EQ n_elements(ten_min_count)    THEN ten_min_count = [ten_min_count, ten_count]          ELSE ten_min_count[i10] = ten_count
    IF i10 EQ n_elements(ten_min_count_v)  THEN ten_min_count_v = [ten_min_count_v, ten_count_v]    ELSE ten_min_count_v[i10] = ten_count_v
    IF i10 EQ n_elements(ten_min_count_ns) THEN ten_min_count_ns = [ten_min_count_ns, ten_count_ns] ELSE ten_min_count_ns[i10] = ten_count_ns

    good_indices = intarr(n_elements(ten_min_start_time)) ; set to 1 if record should be included
    found_bad_time = 0
    FOR p = 0L, N_ELEMENTS(ten_min_start_time) - 1 DO BEGIN
        IF finite(ten_min_start_time(p)) EQ 1 THEN good_indices[p] = 1 ELSE found_bad_time = 1
    ENDFOR
    IF found_bad_time EQ 1 THEN BEGIN
        good_i = where(good_indices EQ 1, good_count)
        IF good_count GT 0 THEN BEGIN 
            ten_min_start_time    = ten_min_start_time   [good_i]
            ten_min_small_density = ten_min_small_density[good_i]
            ten_min_speed         = ten_min_speed        [good_i]
            ten_min_temp          = ten_min_temp         [good_i]
            ten_min_small_v_th    = ten_min_small_v_th   [good_i]
            ten_min_ns            = ten_min_ns           [good_i]
            ten_min_error         = ten_min_error        [good_i]
            ten_min_caution       = ten_min_caution      [good_i]
            ten_min_attitude      = ten_min_attitude     [good_i, *]
            ten_min_red_chi       = ten_min_red_chi      [good_i]
            ten_min_count         = ten_min_count        [good_i]
            ten_min_count_v       = ten_min_count_v      [good_i]
            ten_min_count_ns      = ten_min_count_ns     [good_i]
            ten_min_carr_num      = ten_min_carr_num     [good_i]
            ten_min_carr_lon      = ten_min_carr_lon     [good_i]
            ten_min_helio_dist    = ten_min_helio_dist   [good_i]
            ten_min_hee_lon       = ten_min_hee_lon      [good_i]
            ten_min_hee_lat       = ten_min_hee_lat      [good_i]
            ten_min_heeq_lon      = ten_min_heeq_lon     [good_i]
            ten_min_heeq_lat      = ten_min_heeq_lat     [good_i]
            ten_min_hci_lon       = ten_min_hci_lon      [good_i]
            ten_min_hci_lat       = ten_min_hci_lat      [good_i]
        ENDIF ELSE stop ; no valid times!
    ENDIF 

    store_data, 'density_10min', data = {x:ten_min_start_time, y:ten_min_small_density}, dlim = {ylog:1, labels:labels, panel_size:2.}
    store_data, 'speed_10min', data = {x:ten_min_start_time, y:ten_min_speed}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'temperature_10min', data = {x:ten_min_start_time, y:ten_min_temp}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'v_thermal_10min', data = {x:ten_min_start_time, y:ten_min_small_v_th}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'north_south_10min', data = {x:ten_min_start_time, y:ten_min_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'error_code_10min', data = {x:ten_min_start_time, y:ten_min_error}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'caution_code_10min', data = {x:ten_min_start_time, y:ten_min_caution}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'attitude_flag_10min', data = {x:ten_min_start_time, y:ten_min_attitude}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'reduced_chi_squared_10min', data = {x:ten_min_start_time, y:ten_min_red_chi}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_10min', data = {x:ten_min_start_time, y:ten_min_count}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_10min_v', data = {x:ten_min_start_time, y:ten_min_count_v}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'cycles_10min_ns', data = {x:ten_min_start_time, y:ten_min_count_ns}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_rotation_10min', data = {x:ten_min_start_time, y:ten_min_carr_num}, dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'carrington_longitude_10min', data = {x:ten_min_start_time, y:ten_min_carr_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'heliocentric_distance_10min', data = {x:ten_min_start_time, y:ten_min_helio_dist}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEE_10min', data = {x:ten_min_start_time, y:ten_min_hee_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEE_10min', data = {x:ten_min_start_time, y:ten_min_hee_lat}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HEEQ_10min', data = {x:ten_min_start_time, y:ten_min_heeq_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HEEQ_10min', data = {x:ten_min_start_time, y:ten_min_heeq_lat}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_longitude_HCI_10min', data = {x:ten_min_start_time, y:ten_min_hci_lon}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
    store_data, 'spacecraft_latitude_HCI_10min', data = {x:ten_min_start_time, y:ten_min_hci_lat}, $
                dlim = {ylog:0, labels:labels, panel_size:2.}
        ;ENDELSE 
    ;ENDIF 


ENDIF

;IF bkgd EQ 2 THEN BEGIN
;;; print 1 minute data to output folder
;    time_i = 0
;    WHILE finite(small_epoch1[time_i]) EQ 0 DO BEGIN 
;        time_i = time_i + 1
;    ENDWHILE 
;    label_time = time_struct(small_epoch1(time_i))
;    label_year = STRING(label_time.year, FORMAT = '(I04)')
;    label_month = STRING(label_time.month, FORMAT = '(I02)')
;    label_day = STRING(label_time.date, FORMAT = '(I02)')
; 	outfile_2 = '/home/ksimunac/Output/'+'PLA_ST'+sat+'_L2_1DMax_'+label_year+label_month+label_day+'_'+string(label_time.doy, format = '(I3.3)')+'.txt'
;    	openw, out_lun_2, outfile_2, /GET_LUN
;    	line = ','
;
;   	temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+'sec'+tab+'date and time'+tab+'1 keV/q time'+tab+$
;                 'ST'+sat+' Np [1/cc]'+tab+'ST'+sat+' Vp [km/s]'+tab+'ST'+sat+' Tkin [deg K]'+tab+'ST'+sat+' v_th [km/s]'+tab+$
;                 'ST'+sat+' N/S [deg]'+tab+'Error Code'+tab+'Caution'+tab+'Chi_sq'
;  	 printf, out_lun_2, temp_string
;
;
;  FOR qq = 0L, N_ELEMENTS(small_density_2) - 1 DO BEGIN
;
;	apid_time = time_struct(small_epoch1(qq))
;	kev_struct = time_struct(kev_time(qq))
;       YEAR = STRING(apid_time.year, FORMAT = '(I04)')
;        DOY = STRING(apid_time.doy, FORMAT = '(I03)')
;        HOUR = STRING(apid_time.hour, FORMAT = '(I02)')
;        MIN = STRING(apid_time.min, FORMAT = '(I02)')
;        SEC = STRING(apid_time.sec, FORMAT = '(I02)')
;        N = STRING(small_density(qq), FORMAT = '(F0.2)')
;        V = STRING(round(small_avg_v(qq)), FORMAT = '(I4)')
;        T = STRING(small_temp_K(qq), FORMAT = '(E10.2)')
;        V_TEMP = STRING(small_v_th(qq), FORMAT = '(F0.1)') 
;        NS = STRING(SMALL_S_MAX(qq), FORMAT = '(F0.1)')
;        ERROR = STRING(small_std_error(qq), FORMAT = '(I1)')
;        CAUTION = STRING(caution_flag(qq), FORMAT = '(I1)')
;	FIT_GOOD = STRING(small_good_fit(qq),FORMAT = '(F0.3)')
;
;	temp_string = YEAR+tab+DOY+tab+HOUR+tab+MIN+tab+SEC+tab+string(apid_time.month, format = '(I2.2)')+'/'+$
;                          string(apid_time.date, format = '(I2.2)')+'/'+string(apid_time.year, format = '(I4.4)')+' '$
;                         +string(apid_time.hour, format = '(I2.2)')+':'+string(apid_time.min, format = '(I2.2)')+':'$
;                         +string(apid_time.sec, format = '(I2.2)')+tab+string(kev_struct.month, format = '(I2.2)')+'/'+$
;                         string(kev_struct.date, format = '(I2.2)')+'/'+string(kev_struct.year, format = '(I4.4)')+' '$
;                         +string(kev_struct.hour, format = '(I2.2)')+':'+string(kev_struct.min, format = '(I2.2)')+':'$
;                         +string(kev_struct.sec, format = '(I2.2)')+tab+$
;                         N+tab+V+tab+T+tab+V_TEMP+tab+NS+tab+ERROR+tab+CAUTION+tab+FIT_GOOD
;        printf, out_lun_2, temp_string
;
;   ENDFOR
;	close, out_lun_2
;	free_lun, out_lun_2
;
;
;;; print 1 hour data to output folder
; 	outfile_1 = '/home/ksimunac/Output/'+'PLA_ST'+sat+'_L2_1DMax_'+label_year+label_month+label_day+'_'+string(label_time.doy, format = '(I3.3)')+'_1_hour.txt'
;    	openw, out_lun_1, outfile_1, /GET_LUN
;    	line = ','
;
;   	temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+'date and time'+tab+$
;                  'ST'+sat+' Np [1/cc]'+tab+'ST'+sat+' Vp [km/s]'+tab+'ST'+sat+' Tkin [deg K]'+tab+'ST'+sat+' v_th [km/s]'+tab+$
;                  'ST'+sat+' N/S [deg]'+tab+'Cycle Count'+tab+'Error Code'+tab+'Caution'
;   	 printf, out_lun_1, temp_string
;
;	FOR p = 0, n_elements(hour_speed) - 1 DO BEGIN
;		apid_time = time_struct(hour_start_time(p))
;        	YEAR = STRING(apid_time.year, FORMAT = '(I04)')
;	        DOY = STRING(apid_time.doy, FORMAT = '(I03)')
;        	HOUR = STRING(apid_time.hour, FORMAT = '(I02)')
;	        MIN = STRING(apid_time.min, FORMAT = '(I02)')
;	        SEC = STRING(apid_time.sec, FORMAT = '(I02)')
;       	N = STRING(hour_small_density(p), FORMAT = '(F0.2)')
;	        V = STRING(round(hour_speed(p)), FORMAT = '(I4)')
;        	T = STRING(hour_temp(p), FORMAT = '(E10.2)')
;	        V_TEMP = STRING(hour_small_v_th(p), FORMAT = '(F0.1)') 
;	       	NS = STRING(hour_ns(p), FORMAT = '(F0.1)')
;	        ERROR = STRING(hour_error(p), FORMAT = '(I1)')
;	        CAUTION = STRING(hour_caution(p), FORMAT = '(I1)')
;		CYCLE_COUNT = STRING(hour_count(p),FORMAT = '(I2)')
;
;	
;		temp_string = YEAR+tab+DOY+tab+HOUR+tab+MIN+tab+string(apid_time.month, format = '(I2.2)')+'/'+$
;                          string(apid_time.date, format = '(I2.2)')+'/'+string(apid_time.year, format = '(I4.4)')+' '$
;                          +string(apid_time.hour, format = '(I2.2)')+':'+string(apid_time.min, format = '(I2.2)')+tab $
;                        +N+tab+V+tab+T+tab+V_TEMP+tab+NS+tab+CYCLE_COUNT+tab+ERROR+tab+CAUTION
;           	IF year NE '1970' THEN printf, out_lun_1, temp_string
;
;
;	ENDFOR
;	close, out_lun_1
;	free_lun, out_lun_1
;
;
;;; print 10 minute data to output folder
; 	outfile_3 = '/home/ksimunac/Output/'+'PLA_ST'+sat+'_L2_1DMax_'+label_year+label_month+label_day+'_'+string(label_time.doy, format = '(I3.3)')+'_10_min.txt'
;    	openw, out_lun_3, outfile_3, /GET_LUN
;    	line = ','
;
;    	temp_string = 'YEAR'+tab+'DOY'+tab+'hour'+tab+'min'+tab+'date and time'+tab+$
;                  'ST'+sat+' Np [1/cc]'+tab+'ST'+sat+' Vp [km/s]'+tab+'ST'+sat+' Tkin [deg K]'+tab+'ST'+sat+' v_th [km/s]'+tab+$
;                  'ST'+sat+' N/S [deg]'+tab+'Cycle Count'+tab+'Error Code'+tab+'Caution'
;   	 printf, out_lun_3, temp_string
;
;
; 
;    FOR pp = 0L, N_ELEMENTS(ten_min_speed) - 1 DO BEGIN
;
;        apid_time = time_struct(ten_min_start_time(pp))
;        YEAR = STRING(apid_time.year, FORMAT = '(I04)')
;        DOY = STRING(apid_time.doy, FORMAT = '(I03)')
;        HOUR = STRING(apid_time.hour, FORMAT = '(I02)')
;        MIN = STRING(apid_time.min, FORMAT = '(I02)')
;        SEC = STRING(apid_time.sec, FORMAT = '(I02)')
;        N = STRING(ten_min_small_density(pp), FORMAT = '(F0.2)')
;        V = STRING(round(ten_min_speed(pp)), FORMAT = '(I4)')
;        T = STRING(ten_min_temp(pp), FORMAT = '(E10.2)')
;        V_TEMP = STRING(ten_min_small_v_th(pp), FORMAT = '(F0.1)') 
;        NS = STRING(ten_min_ns(pp), FORMAT = '(F0.1)')
;        ERROR = STRING(ten_min_error(pp), FORMAT = '(I1)')
;        CAUTION = STRING(ten_min_caution(pp), FORMAT = '(I1)')
;	CYCLE_COUNT = STRING(ten_min_count(pp),FORMAT = '(I2)')
;            
; 	temp_string = YEAR+tab+DOY+tab+HOUR+tab+MIN+tab+string(apid_time.month, format = '(I2.2)')+'/'+$
;                          string(apid_time.date, format = '(I2.2)')+'/'+string(apid_time.year, format = '(I4.4)')+' '$
;                          +string(apid_time.hour, format = '(I2.2)')+':'+string(apid_time.min, format = '(I2.2)')+tab $
;                         +N+tab+V+tab+T+tab+V_TEMP+tab+NS+tab+CYCLE_COUNT+tab+ERROR+tab+CAUTION
;        IF year NE '1970' THEN    printf, out_lun_3, temp_string
;
;
;    ENDFOR
;	close, out_lun_3
;	free_lun, out_lun_3
;ENDIF
END 
