;+
; PROCEDURE: plot_ace_swepam
;
; PURPOSE: Read ace data file
;
; INPUT: time interval in the form of tplot timespan
;
; OUTPUT: data in the form of tplot variables
;
; KEYWORDS: LEVEL2:      Data
;           REAL_TIME:   Data
;           CDF:         Data
;           PRELIM_CDF:  Data
;
; DATA USED: Daily "real time" data 
;            From:        http://www.swpc.noaa.gov/ftpdir/lists/ace/
;            File name:   "yyyymmdd_ace_mag_1m.txt"
;            File format: ASCII
; 
;            "Level2" Verified data
;            From:        http://www.srl.caltech.edu/ACE/ASC/level2/lvl2DATA_SWEPAM.html
;            File name:   "ace_swepam_yyyy.dat"
;            File format: ASCII
;
;            CDF Verified and Preliminary data
;            From:        ftp://cdaweb.gsfc.nasa.gov/pub/istp/ace/
;            File name:   "ac_k0_swe_yyyymmdd_v$$.cdf" - Verified
;                         "ac_k0_swe_yyyymmdd_v$$.cdf" - Preliminary
;            File format: CDF
;
;
; PRODUCED by: C. Mouikis
;
; MODIFICATION HISTORY:
;            12/10/2008: (CGM) data structure tag name was the same
;            for two data sets creating conflicts.
;            04/05/2009: (LBE) Changed findfile to file_search.
;            08/04/2009: (LBE) Added check for e_file eq -1.
;            05/18/2010: (LBE) Check for no data.
;            12/23/2015: (LBE) Check for one day.
;-
PRO find_dayom_from_doy, year, doy, mon, day
compile_opt strictarrsubs
  
  IF FIX(year/4.) EQ year/4. THEN BEGIN
  
    day = [$
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31]
    
    mon = [$
            1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, $
            1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, $
            2,2,2,2,2,2,2,2,2,2,2,2,2,2,2, $
            2,2,2,2,2,2,2,2,2,2,2,2,2,2, $
            3,3,3,3,3,3,3,3,3,3,3,3,3,3,3, $
            3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3, $
            4,4,4,4,4,4,4,4,4,4,4,4,4,4,4, $
            4,4,4,4,4,4,4,4,4,4,4,4,4,4,4, $
            5,5,5,5,5,5,5,5,5,5,5,5,5,5,5, $
            5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5, $
            6,6,6,6,6,6,6,6,6,6,6,6,6,6,6, $
            6,6,6,6,6,6,6,6,6,6,6,6,6,6,6, $
            7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, $
            7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, $
            8,8,8,8,8,8,8,8,8,8,8,8,8,8,8, $
            8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8, $
            9,9,9,9,9,9,9,9,9,9,9,9,9,9,9, $
            9,9,9,9,9,9,9,9,9,9,9,9,9,9,9, $
            10,10,10,10,10,10,10,10,10,10,10,10,10,10,10, $
            10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10, $
            11,11,11,11,11,11,11,11,11,11,11,11,11,11,11, $
            11,11,11,11,11,11,11,11,11,11,11,11,11,11,11, $
            12,12,12,12,12,12,12,12,12,12,12,12,12,12,12, $
            12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12 ]
    
  ENDIF ELSE BEGIN
    day = [$
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30, $
            1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16, $
            17,18,19,20,21,22,23,24,25,26,27,28,29,30,31]
    
    mon = [$
            1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, $
            1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, $
            2,2,2,2,2,2,2,2,2,2,2,2,2,2,2, $
            2,2,2,2,2,2,2,2,2,2,2,2,2, $
            3,3,3,3,3,3,3,3,3,3,3,3,3,3,3, $
            3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3, $
            4,4,4,4,4,4,4,4,4,4,4,4,4,4,4, $
            4,4,4,4,4,4,4,4,4,4,4,4,4,4,4, $
            5,5,5,5,5,5,5,5,5,5,5,5,5,5,5, $
            5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5, $
            6,6,6,6,6,6,6,6,6,6,6,6,6,6,6, $
            6,6,6,6,6,6,6,6,6,6,6,6,6,6,6, $
            7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, $
            7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, $
            8,8,8,8,8,8,8,8,8,8,8,8,8,8,8, $
            8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8, $
            9,9,9,9,9,9,9,9,9,9,9,9,9,9,9, $
            9,9,9,9,9,9,9,9,9,9,9,9,9,9,9, $
            10,10,10,10,10,10,10,10,10,10,10,10,10,10,10, $
            10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10, $
            11,11,11,11,11,11,11,11,11,11,11,11,11,11,11, $
            11,11,11,11,11,11,11,11,11,11,11,11,11,11,11, $
            12,12,12,12,12,12,12,12,12,12,12,12,12,12,12, $
            12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12 ]
    
  ENDELSE
  
  mon = mon(doy-1)
  day = day(doy-1)
  
END

PRO r_ace_swepam_onefile, path, fln
compile_opt strictarrsubs
  
  COMMON ace_swepam_data, ace_time, ace_swepam
  
  idlfind = FILE_SEARCH(path + '/idl/' + fln + '.idl', COUNT=fc)
  IF fc GT 0 THEN BEGIN
    RESTORE, idlfind(0), /VERBOSE
    RETURN
  ENDIF ELSE BEGIN
    r_ace_swepam_onefile_original, path, fln
  ENDELSE
  
  
END

PRO r_ace_swepam_onefile_original, path, fln
compile_opt strictarrsubs
  
  COMMON ace_swepam_data, ace_time, ace_swepam
  
  ;--------------------------------------------------------------------
  ; Find original ascii file
  ;--------------------------------------------------------------------
  ext='.dat*'
  find = FILE_SEARCH(path+fln+ext, count=ff)
  IF ff EQ 0 THEN BEGIN
    print, 'FILE: '+fln+ext+' was not found'
    get_err_no = 1
    get_err_msg= 'Data NOT found in time interval'
    RETURN
  ENDIF

  fln_r = find(ff-1)
  ;--------------------------------------------------------------------
  ; Open the file and count the time steps. If file is zipped and the
  ; idl version is not 5.4 then unzip the file in a temporary directory
  ;--------------------------------------------------------------------
  version=FLOAT(!version.release)
  
  IF version GE 5.3 THEN BEGIN
    OPENR, unit, fln_r, /get_lun, /COMPRESS
    zipflag=1
  ENDIF ELSE BEGIN
    IF strmid(fln_r, strlen(fln_r)-2, 2) EQ 'gz' THEN BEGIN
      fln_from_path, fln_r, fln
      unziped_filename = strmid(fln, 0, strlen(fln)-3)
      unziped_filename = getenv('CCAT_TEMP') + '/' + unziped_filename
      spawn, 'gzip -dc ' + fln_r + ' > ' + unziped_filename
      zipflag=1
      OPENR, unit, unziped_filename, /get_lun
    ENDIF ELSE BEGIN
      zipflag=0
      OPENR, unit, fln_r, /get_lun
    ENDELSE
  ENDELSE   

  dummy=''
  
  lc = 0l
  WHILE (NOT eof(unit)) DO BEGIN
    readf, unit, dummy
    IF STRMID(dummy, 0, 8) EQ 'year day' THEN BEGIN
      lc1 = lc
      lc = 0l
    ENDIF
    lc=lc+1
  ENDWHILE
  lc = lc - 1
  
  close, unit
  free_lun, unit

  IF lc EQ 0 THEN BEGIN
    get_err_no = 1
    get_err_msg = 'No data found in file: ' + fln_r
    RETURN
  ENDIF  
  
  ;--------------------------------------------------------------------
  ; Re-open the file and read data. If file is zipped and the
  ; idl version is not 5.4 then read the unziped file from temp directory
  ;--------------------------------------------------------------------
  IF zipflag EQ 1 THEN BEGIN
    IF version GE 5.3 THEN BEGIN
      OPENR, unit, fln_r, /get_lun, /COMPRESS
    ENDIF ELSE BEGIN
      OPENR, unit, unziped_filename, /get_lun
    ENDELSE
  ENDIF ELSE BEGIN
    OPENR, unit, fln_r, /get_lun
  ENDELSE
  
  ace_time = DBLARR(lc)
  dummy_str = {$,
               year:0, $
               doy:0, $
               hour:0, $
               min:0, $
               sec:0.0, $
               fp_year:0d, $
               fp_doy:0d, $
               ACEepoch:0d, $
               proton_density:0.0, $
               proton_temp:0.0, $
               He4toprotons:0.0, $
               proton_speed:0.0, $
               x_dot_GSE:0.0, $
               y_dot_GSE:0.0, $
               z_dot_GSE:0.0, $
               r_dot_RTN:0.0, $
               t_dot_RTN:0.0, $
               n_dot_RTN:0.0, $
               x_dot_GSM:0.0, $
               y_dot_GSM:0.0, $
               z_dot_GSM:0.0, $
               pos_gse_x:0.0, $
               pos_gse_y:0.0, $
               pos_gse_z:0.0, $
               pos_gsm_x:0.0, $
               pos_gsm_y:0.0, $
               pos_gsm_z:0.0 $
              }
  
  ace_swepam = REPLICATE(dummy_str, lc-1)
  
  FOR ii=0, lc1+1 DO BEGIN
    READF, unit, dummy
  ENDFOR
  
  FOR ii=0l, lc-2 DO BEGIN
    
    READF, unit, dummy
    READS, dummy, dummy_str
    
    find_dayom_from_doy, dummy_str.year, dummy_str.doy, mon, day
    
    time_str = $
      STRING(dummy_str.year, FORMAT='(i4.4)') + '-' + $
      STRING(mon, FORMAT='(i2.2)') + '-' + $
      STRING(day, FORMAT='(i2.2)') + '/' + $
      STRING(dummy_str.hour, FORMAT='(i2.2)') + ':' + $
      STRING(dummy_str.min, FORMAT='(i2.2)') + ':' + $
      STRING(dummy_str.sec, FORMAT='(i2.2)')
    
    ace_time(ii) = time_double(time_str)
    ace_swepam(ii) = dummy_str
    
  ENDFOR
  
  ;--------------------------------------------------------------------
  ; Save data in idl-save format
  ;--------------------------------------------------------------------
  
  ;IF STRUPCASE(GETENV('IDL_SAVESETS')) EQ 'YES' THEN BEGIN
    ; Get the filename from the full path
    fln_from_path, fln_r, fln_w
    ; Check if file has the gz extension and take it out
    file_zipped = strpos(fln_w, '.gz') 
    IF file_zipped GT -1 THEN fln_w = strmid(fln_w, 0, file_zipped) 
    ; Take out file extension
    fln_w = strmid(fln_w, 0, strlen(fln_w)-1-3)
    
    ;save, ace_time, ace_swepam, filename=path+'/idl/'+fln_w+'.idl', /VERBOSE
    save, ace_time, ace_swepam, filename=fln_w+'.idl', /VERBOSE
  ;ENDIF
  
  CLOSE, unit
  FREE_LUN, unit
  IF version LT 5.3 AND zipflag EQ 1 THEN BEGIN
    spawn, '/bin/rm -f ' + unziped_filename
  ENDIF
  
END


PRO read_ace_real_time, fn, time, dens, velo, temp
compile_opt strictarrsubs

COMMON get_error, get_err_no, get_err_msg, default_verbose

nn = 1440

fnfound = FILE_SEARCH(fn,  COUNT = cfnfound)

IF cfnfound EQ 0 THEN BEGIN
    
    get_err_no = 1
    RETURN

ENDIF

OPENR, unit, fn,  /GET_LUN

dummy = ':'
WHILE STRMID(dummy, 0, 1) EQ ':' OR STRMID(dummy, 0, 1) EQ '#' DO BEGIN

    READF, unit, dummy

ENDWHILE

time = DBLARR(nn)
dens = FLTARR(nn)
velo = FLTARR(nn)
temp = FLTARR(nn)

a0 = 0 & a1 = 0 & a2 = 0 & a3 = 0 & a4 = 0l & a5 = 0l & a6 = 0 & a7 = 0. & a8 = 0. & a9 = 0.
ii = 0l
WHILE NOT EOF(unit) DO BEGIN

    READS, dummy, a0, a1, a2, a3, a4, a5, a6, a7, a8, a9

    yr = STRING(a0, FORMAT = '(i4.4)')
    mo = STRING(a1, FORMAT = '(i2.2)')
    da = STRING(a2, FORMAT = '(i2.2)')


    IF a6 EQ 0 THEN BEGIN
        time(ii) = time_double(yr + '-' + mo + '-' + da) + a5
        dens(ii) = a7
        velo(ii) = a8
        temp(ii) = a9
        ii = ii + 1
    ENDIF

    READF, unit, dummy


ENDWHILE

time = time(0:ii-1)
dens = dens(0:ii-1)
velo = velo(0:ii-1)
temp = temp(0:ii-1)

CLOSE, unit
FREE_LUN, unit, /FORCE


END




PRO r_ace_swepam_cdf_prelim, path, fln,  PRELIM = PRELIM
compile_opt strictarrsubs

COMMON ace_swepam_data_prilim, ace_swe_prilim_time, ace_swe_prilim
COMMON get_error, get_err_no, get_err_msg, default_verbose
  
get_err_no = 0
get_err_msg = 'Data found in time interval'

;--------------------------------------------------------------------
; Find and read cdf file
;--------------------------------------------------------------------
find = FILE_SEARCH(path+fln, count = ff)
IF ff EQ 0 THEN BEGIN
    print, 'FILE: '+fln+' was not found'
    get_err_no = 1
    get_err_msg = 'Data NOT found in time interval'
    RETURN
ENDIF

fln_r = find(ff-1)

; Np
read_cdf, IN_FILE = fln_r, $
          VAR = 'Np', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Np_xarr = REFORM(data.x(0, *))
Np_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; Vp
read_cdf, IN_FILE = fln_r, $
          VAR = 'Vp', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Vp_xarr = REFORM(data.x(0, *))
Vp_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; Tpr
read_cdf, IN_FILE = fln_r, $
          VAR = 'Tpr', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Tpr_xarr = REFORM(data.x(0, *))
Tpr_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; He ratio
read_cdf, IN_FILE = fln_r, $
          VAR = 'He_ratio', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
He_ratio_xarr = REFORM(data.x(0, *))
He_ratio_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; Weight
read_cdf, IN_FILE = fln_r, $
          VAR = 'Weight', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Weight_xarr = REFORM(data.x(0, *))
Weight_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del


;--------------------------------------------------------------------
; Re-open the file and read data. If file is zipped and the
; idl version is not 5.4 then read the unziped file from temp directory
;--------------------------------------------------------------------
lc = N_ELEMENTS(Np_xarr)
dummy_str = {$, 
             Np:0.0, $
             Vp:0.0, $
             Tpr:0.0, $
             He_ratio:0.0, $
             weight:0.0 $
            }

ace_swe_prilim = REPLICATE(dummy_str, lc)
ace_swe_prilim_time = Np_xarr
ace_swe_prilim.Np = Np_yarr
ace_swe_prilim.Vp = Vp_yarr
ace_swe_prilim.Tpr = Tpr_yarr
ace_swe_prilim.He_ratio = He_ratio_yarr
ace_swe_prilim.Weight = Weight_yarr

END



PRO r_ace_swepam_cdf, path, fln
compile_opt strictarrsubs

COMMON ace_swepam_data, ace_swe_time, ace_swe
COMMON get_error, get_err_no, get_err_msg, default_verbose
  
get_err_no = 0
get_err_msg = 'Data found in time interval'

;--------------------------------------------------------------------
; Find and read cdf file
;--------------------------------------------------------------------
find = FILE_SEARCH(path+fln, count = ff)
IF ff EQ 0 THEN BEGIN
    print, 'FILE: '+fln+' was not found'
    get_err_no = 1
    get_err_msg = 'Data NOT found in time interval'
    RETURN
ENDIF

fln_r = find(ff-1)

; Np
read_cdf, IN_FILE = fln_r, $
          VAR = 'Np', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Np_xarr = REFORM(data.x(0, *))
Np_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; Vp
read_cdf, IN_FILE = fln_r, $
          VAR = 'Vp', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Vp_xarr = REFORM(data.x(0, *))
Vp_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; Tpr
read_cdf, IN_FILE = fln_r, $
          VAR = 'Tpr', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
Tpr_xarr = REFORM(data.x(0, *))
Tpr_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; Alpha ratio
read_cdf, IN_FILE = fln_r, $
          VAR = 'alpha_ratio', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
alpha_ratio_xarr = REFORM(data.x(0, *))
alpha_ratio_yarr = REFORM(data.y(*, 0))
store_data, mout_data, /del

; V GSE
read_cdf, IN_FILE = fln_r, $
          VAR = 'V_GSE', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
V_gse_xarr = REFORM(data.x(0, *))
V_gse_yarr = data.y
store_data, mout_data, /del

; V GSM
read_cdf, IN_FILE = fln_r, $
          VAR = 'V_GSM', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
V_gsm_xarr = REFORM(data.x(0, *))
V_gsm_yarr = data.y
store_data, mout_data, /del

; V RTN
read_cdf, IN_FILE = fln_r, $
          VAR = 'V_RTN', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
V_rtn_xarr = REFORM(data.x(0, *))
V_rtn_yarr = data.y
store_data, mout_data, /del

; SC pos GSE
read_cdf, IN_FILE = fln_r, $
          VAR = 'SC_pos_GSE', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
X_gse_xarr = REFORM(data.x(0, *))
X_gse_yarr = data.y / 6372.0    ; Re
store_data, mout_data, /del

; SC pos GSM
read_cdf, IN_FILE = fln_r, $
          VAR = 'SC_pos_GSM', $
          EPOCH = 'Epoch', OUT_DATA = mout_data

get_data, mout_data, data = data
X_gsm_xarr = REFORM(data.x(0, *))
X_gsm_yarr = data.y / 6372.0    ; Re
store_data, mout_data, /del

;--------------------------------------------------------------------
; Re-open the file and read data. If file is zipped and the
; idl version is not 5.4 then read the unziped file from temp directory
;--------------------------------------------------------------------
lc = N_ELEMENTS(Np_xarr)
dummy_str = {$, 
             Np:0.0, $
             Vp:0.0, $
             Tpr:0.0, $
             alpha_ratio:0.0, $
             V_GSE_X:0.0, $
             V_GSE_Y:0.0, $
             V_GSE_Z:0.0, $
             V_GSM_X:0.0, $
             V_GSM_Y:0.0, $
             V_GSM_Z:0.0, $
             V_RTN_X:0.0, $
             V_RTN_Y:0.0, $
             V_RTN_Z:0.0, $
             X_GSE_X:0.0, $
             X_GSE_Y:0.0, $
             X_GSE_Z:0.0, $
             X_GSM_X:0.0, $
             X_GSM_Y:0.0, $
             X_GSM_Z:0.0 $
            }

ace_swe = REPLICATE(dummy_str, lc)
ace_swe_time = Np_xarr
ace_swe.Np = Np_yarr
ace_swe.Vp = Vp_yarr
ace_swe.Tpr = Tpr_yarr
ace_swe.alpha_ratio = alpha_ratio_yarr
ace_swe.V_GSE_X = REFORM(V_gse_yarr(*, 0))
ace_swe.V_GSE_Y = REFORM(V_gse_yarr(*, 1))
ace_swe.V_GSE_Z = REFORM(V_gse_yarr(*, 2))
ace_swe.V_GSM_X = REFORM(V_gsm_yarr(*, 0))
ace_swe.V_GSM_Y = REFORM(V_gsm_yarr(*, 1))
ace_swe.V_GSM_Z = REFORM(V_gsm_yarr(*, 2))
ace_swe.V_RTN_X = REFORM(V_rtn_yarr(*, 0))
ace_swe.V_RTN_Y = REFORM(V_rtn_yarr(*, 1))
ace_swe.V_RTN_Z = REFORM(V_rtn_yarr(*, 2))
ace_swe.X_GSE_X = REFORM(X_gse_yarr(*, 0))
ace_swe.X_GSE_Y = REFORM(X_gse_yarr(*, 1))
ace_swe.X_GSE_Z = REFORM(X_gse_yarr(*, 2))
ace_swe.X_GSM_X = REFORM(X_gsm_yarr(*, 0))
ace_swe.X_GSM_Y = REFORM(X_gsm_yarr(*, 1))
ace_swe.X_GSM_Z = REFORM(X_gsm_yarr(*, 2))

END



PRO plot_ace_swepam, $
  LEVEL2 = LEVEL2, $
  REAL_TIME = REAL_TIME, $
  CDF = CDF, $
  PRELIM_CDF = PRELIM_CDF, $
  Np = Np, $                    ; Proton Number Density
  Vp = Vp, $                    ; Solar Wind Proton Speed
  Tpr = Tpr, $                  ; Proton Temperature
  alpha_ratio = alpha_ratio, $  ; alpha/proton ratio
  V_GSE = V_GSE, $              ; Solar Wind Velocity (GSE)
  V_RTN = V_RTN, $              ; Solar Wind Velocity (RTN)
  V_GSM = V_GSM, $              ; Solar Wind Velocity (GSM)
  SC_pos_GSE = SC_pos_GSE, $    ; S/C position in GSE
  SC_pos_GSM = SC_pos_GSM       ; S/C position in GSM
compile_opt strictarrsubs
  
COMMON get_error, get_err_no, get_err_msg, default_verbose

get_err_no = 0
get_err_msg = 'Data found in time interval'

IF KEYWORD_SET(LEVEL2) THEN BEGIN

    
    COMMON ace_swepam_data, ace_time, ace_swepam
    
    path = GETENV('ACEDATA')
    ACE_SWEPAM_PATH = path + '/swepam/'
    
    file_list = FILE_SEARCH(ACE_SWEPAM_PATH + '/ace_swepam*', COUNT = fcount)
    IF fcount GT 0 THEN BEGIN
        s_date_year = INTARR(fcount)
        e_date_year = INTARR(fcount)
        FOR ff = 0, fcount-1 DO BEGIN
            s_date_year(ff) = FIX(STRMID(file_list(ff), $
                                         STRPOS(file_list(ff), 'swepam_') + 7, 4))
        ENDFOR
    ENDIF ELSE BEGIN
        STOP
    ENDELSE
    
    get_timespan, time_interval
    t_s = gettime(time_interval(0)) ; start time in tplot-time
    t_e = gettime(time_interval(1)) ; end time in tplot-time  
    
    t_s_str = time_struct(t_s)  ; start_time tplot time structure
    t_e_str = time_struct(t_e)  ; end_time tplot time structure
    
    s_year = t_s_str.year
    e_year = t_e_str.year
    s_doy = t_s_str.doy
    e_doy = t_e_str.doy
    IF t_e_str.sod EQ 0 THEN e_doy = e_doy - 1
    
    s_file = WHERE(s_date_year EQ s_year)
    e_file = WHERE(s_date_year EQ e_year)
    IF e_file EQ -1 THEN e_file = s_file ; added 8/4/2009 (LBE)
    
    no_of_files = (e_file(0) - s_file(0)) + 1 ; number of days to be loaded
    
    FOR ff = 0, no_of_files-1 DO BEGIN
        
        file_name = STRMID(file_list(s_file+ff), $
                           STRPOS(file_list(s_file+ff), 'ace_swepam_'), 15)
        
        r_ace_swepam_onefile, ace_swepam_path, file_name(0)
        
        IF ff EQ 0 THEN BEGIN
            ace_swepam_all = ace_swepam
            ace_time_all = ace_time
        ENDIF ELSE BEGIN
            ace_swepam_all = [ace_swepam_all, ace_swepam]
            ace_time_all = [ace_time_all, ace_time]
        ENDELSE
        
    ENDFOR
    
    tvstr = 'ACE_SWEPAM_LEVEL2_'
    IF KEYWORD_SET(DENS) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        fillind = WHERE(ace_swepam_all.proton_density NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            density = ace_swepam_all(fillind).proton_density
            str = {x:time, y:density}
            store_data, tvstr + 'DENSITY', data = str
            options, tvstr + 'DENSITY', 'ytitle', 'n (cm!U-3!N)'
        ENDIF 
    ENDIF
    
    IF KEYWORD_SET(TEMP) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        fillind = WHERE(ace_swepam_all.proton_temp NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            temp = ace_swepam_all(fillind).proton_temp / 11600.
            str = {x:time, y:temp}
            store_data, tvstr + 'TEMPERATURE', data = str
            options, tvstr + 'TEMPERATURE', 'ytitle', 'T (eV)'
        ENDIF 
    ENDIF
    
    IF KEYWORD_SET(VEL) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.proton_speed NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            velocity = ace_swepam_all(fillind).proton_speed
            str = {x:time, y:velocity}
            store_data, tvstr + 'VELOCITY', data = str
            options, tvstr + 'VELOCITY', 'ytitle', 'V (km s!U-1!N)'
        ENDIF 
    ENDIF
    
    IF KEYWORD_SET(PRESS) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.proton_density NE -9999.9 AND $
                        ace_swepam_all.proton_speed NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            density = ace_swepam_all(fillind).proton_density
            velocity = ace_swepam_all(fillind).proton_speed
            fillind = WHERE(velocity NE -9999.9)
            time = time(fillind)
            density = density(fillind)
            velocity = velocity(fillind)
        
            mp = 1.67e-27       ; (kg)
            press = $
              density * 1.e6 * mp *$
              (velocity*1000.)^2*1.e9
            str = {x:time, y:press}
            store_data, tvstr + 'PRESSURE', data = str
            options, tvstr + 'PRESSURE', 'ytitle', 'P (nP)'
        ENDIF 
    ENDIF
    
    
    IF KEYWORD_SET(THERM_PRESS) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.proton_density NE -9999.9 AND $
                        ace_swepam_all.proton_temp NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            density = ace_swepam_all(fillind).proton_density
            temp = ace_swepam_all(fillind).proton_temp / 11600.
            therm_pressure = temp * density / 6250.
        
            str = {x:time, y:therm_pressure}
            store_data, tvstr + 'THERM_PRESS', data = str
            options, tvstr + 'THERM_PRESS', 'ytitle', 'P (nP)'
        ENDIF 
        
    ENDIF
    
    IF KEYWORD_SET(VX) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.proton_density NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            Vx_gse = ace_swepam_all(fillind).x_dot_GSE
            str = {x:time, y:Vx_gse}
            store_data, tvstr + 'VX_GSE', data = str
            options, tvstr + 'VX_GSE', 'ytitle', 'V (km s!U-1!N)'
        ENDIF 
    ENDIF
    
    IF KEYWORD_SET(VY) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.proton_density NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            Vy_gse = ace_swepam_all(fillind).y_dot_GSE
            str = {x:time, y:Vy_gse}
            store_data, tvstr + 'VY_GSE', data = str
            options, tvstr + 'VY_GSE', 'ytitle', 'V (km s!U-1!N)'
        ENDIF 
    ENDIF
    
    IF KEYWORD_SET(VZ) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.proton_density NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            Vz_gse = ace_swepam_all(fillind).z_dot_GSE
            str = {x:time, y:Vz_gse}
            store_data, tvstr + 'VZ_GSE', data = str
            options, tvstr + 'VZ_GSE', 'ytitle', 'V (km s!U-1!N)'
        ENDIF 
    ENDIF
    
    IF KEYWORD_SET(X) OR KEYWORD_SET(LEVEL2) THEN BEGIN
        
        fillind = WHERE(ace_swepam_all.pos_gsm_x NE -9999.9, count)
        IF count GT 0 THEN BEGIN 
            time = ace_time_all(fillind)
            X_gsm = ace_swepam_all(fillind).pos_gsm_x
            str = {x:time, y:X_gsm}
            store_data, tvstr + 'X_GSM', data = str
            options, tvstr + 'X_GSM', 'ytitle', 'X (km)'
        ENDIF 
    ENDIF
    
    
ENDIF

IF KEYWORD_SET(REAL_TIME) THEN BEGIN

    get_timespan, tt
    ts = time_struct(tt(0))
    te = time_struct(tt(1))
    nofdays = te.daynum - ts.daynum
    IF nofdays EQ 0 THEN nofdays = 1
    fnyr = STRING(ts.year,  FORMAT = '(i4.4)')
    fnmo = STRING(ts.month, FORMAT = '(i2.2)')
    fnda = STRING(ts.date,  FORMAT = '(i2.2)')
    path = GETENV('ACEDATA')
    path =  path + '/swepam/real_time/'
    time_start = time_double(fnyr + '/' + fnmo + '/' + fnda)
    iig = 0
    FOR ii = 0, nofdays-1 DO BEGIN
        ts = time_struct(time_start)
        fnyr = STRING(ts.year,  FORMAT = '(i4.4)')
        fnmo = STRING(ts.month, FORMAT = '(i2.2)')
        fnda = STRING(ts.date,  FORMAT = '(i2.2)')
        fn = path + fnyr + fnmo + fnda + '_ace_swepam_1m.txt'    
        get_err_no = 0
        read_ace_real_time, fn, time, dens, velo, temp
        IF get_err_no EQ 0 THEN BEGIN
            IF iig EQ 0 THEN BEGIN
                ftime = time
                fdens = dens
                fvelo = velo
                ftemp = temp
            ENDIF ELSE BEGIN
                ftime = [ftime, time]
                fdens = [fdens, dens]
                fvelo = [fvelo, velo]
                ftemp = [ftemp, temp]
            ENDELSE
            iig = iig + 1
        ENDIF
        time_start = time_start + 86400.
    ENDFOR
    IF iig GT 0 THEN BEGIN
        ftemp = ftemp * 1.6022e-19 / 1.1605e4 ; Temp in J
        mp = 1.6726e-27         ; kg
        thvel = SQRT(2 * ftemp / mp) / 1e3 ; km/s
        store_data, 'ACE_REAL_TIME_dens', data = {x:ftime, y:fdens}, dlim = {panel_size:2, ylog:1}
        store_data, 'ACE_REAL_TIME_velo', data = {x:ftime, y:fvelo}, dlim = {panel_size:2, ylog:0}
        store_data, 'ACE_REAL_TIME_temp', data = {x:ftime, y:ftemp}, dlim = {panel_size:2, ylog:1}
        store_data, 'ACE_REAL_TIME_thve', data = {x:ftime, y:thvel}, dlim = {panel_size:2, ylog:0}
        store_data, 'ACE_REAL_TIME_2thve', data = {x:ftime, y:thvel/sqrt(2.)}, dlim = {panel_size:2, ylog:0}
    ENDIF
ENDIF

IF KEYWORD_SET(CDF) THEN BEGIN

    COMMON ace_swepam_data, ace_swe_time, ace_swe
    COMMON get_error, get_err_no, get_err_msg, default_verbose
    
    get_err_no = 0
    get_err_msg = 'Data found in time interval'
    
    path = GETENV('ACEDATA')
    path = path + '/swepam/cdf/'
    
    get_timespan, time_interval
    t_s = gettime(time_interval(0)) ; start time in tplot-time
    t_e = gettime(time_interval(1)) ; end time in tplot-time  
    
    t_s_str = time_struct(t_s)  ; start_time tplot time structure
    t_e_str = time_struct(t_e)  ; end_time tplot time structure
    
    mjd_s = julday(t_s_str.month, t_s_str.date, t_s_str.year) ;start julian day
    mjd_e = julday(t_e_str.month, t_e_str.date, t_e_str.year) ; end julian day
    
    no_of_files = (mjd_e - mjd_s) + 1 ; number of days to be loaded
    
                                ;Last day is not included if hour=min=sec=0
    IF t_e_str.hour EQ 0 AND t_e_str.min EQ 0 AND t_e_str.sec EQ 0 THEN $
      no_of_files = no_of_files - 1
    
    ;--------------------------------------------------------------------
    ; Read all 1 day files that correspond to requested time interval
    ;--------------------------------------------------------------------
    fc = 0                      ; Files-found counter
    FOR nd = 0, no_of_files-1 DO BEGIN ; Loop trough all days
        
        caldat, mjd_s + nd, month, day, year ; find caledar date
        
        filename = 'ac_h0_swe_' + $
                   STRING(year, month, day, $
                          format = '(i4.4,i2.2,i2.2)') + '_v*.cdf'
        r_ace_swepam_cdf, path, filename
        
        IF get_err_no EQ 0 THEN BEGIN
            
            IF nd EQ 0 THEN BEGIN
                ace_swe_all = ace_swe
                ace_time_all = ace_swe_time
            ENDIF ELSE BEGIN
                ace_swe_all = [ace_swe_all, ace_swe]
                ace_time_all = [ace_time_all, ace_swe_time]
            ENDELSE
            fc = fc + 1
        ENDIF
        
        
    ENDFOR
    
    IF fc GT 0 THEN BEGIN
        
        tvstr = 'ACE_SWEPAM_CDF_'
        
        IF KEYWORD_SET(Np) OR KEYWORD_SET(CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all.Np NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                density = ace_swe_all(fillind).Np
                str = {x:time, y:density}
                store_data, tvstr + 'DENSITY', data = str, lim = {ylog:1}
                options, tvstr + 'DENSITY', 'ytitle', 'n (cm!U-3!N)'
            ENDIF 
        ENDIF
        
        IF KEYWORD_SET(Vp) OR KEYWORD_SET(CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all.Vp NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                density = ace_swe_all(fillind).Vp
                str = {x:time, y:density}
                store_data, tvstr + 'Vp', data = str,  lim = {yrange:[1e2, 1e3], ylog:1}
                options, tvstr + 'Vp', 'ytitle', 'Vp (km s!U-1!N)'
            ENDIF 
        ENDIF
        
        IF KEYWORD_SET(Tpr) OR KEYWORD_SET(CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all.Tpr NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                density = ace_swe_all(fillind).Tpr
                str = {x:time, y:density}
                store_data, tvstr + 'Tpr', data = str, lim = {ylog:1}
                options, tvstr + 'Tpr', 'ytitle', 'Tpr (Kelvin)'
            ENDIF 
        ENDIF
        
        IF KEYWORD_SET(alpha_ratio) OR KEYWORD_SET(CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all.alpha_ratio NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                density = ace_swe_all(fillind).alpha_ratio
                str = {x:time, y:density}
                store_data, tvstr + 'alpha_ratio', data = str, lim = {yrange:[1e-3, 1], ylog:1}
                options, tvstr + 'alpha_ratio', 'ytitle', 'alpha ratio (%)'
            ENDIF 
        ENDIF
        
        IF KEYWORD_SET(V_GSE) OR KEYWORD_SET(CDF) THEN BEGIN
            
            fillind = WHERE(ace_swe_all.V_GSE_X NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_GSE_X
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_GSE_X', data = str
                options, tvstr + 'V_GSE_X', 'ytitle', 'V!LX!N (km s!U-1!N)'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.V_GSE_Y NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_GSE_Y
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_GSE_Y', data = str
                options, tvstr + 'V_GSE_Y', 'ytitle', 'V!LY!N (km s!U-1!N)'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.V_GSE_Z NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_GSE_Z
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_GSE_Z', data = str
                options, tvstr + 'V_GSE_Z', 'ytitle', 'V!LZ!N (km s!U-1!N)'
            ENDIF 
            
        ENDIF

        
        IF KEYWORD_SET(V_GSM) OR KEYWORD_SET(CDF) THEN BEGIN
            
            fillind = WHERE(ace_swe_all.V_GSM_X NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_GSM_X
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_GSM_X', data = str
                options, tvstr + 'V_GSM_X', 'ytitle', 'V!LX!N (km s!U-1!N)'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.V_GSM_Y NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_GSM_Y
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_GSM_Y', data = str
                options, tvstr + 'V_GSM_Y', 'ytitle', 'V!LY!N (km s!U-1!N)'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.V_GSM_Z NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_GSM_Z
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_GSM_Z', data = str
                options, tvstr + 'V_GSM_Z', 'ytitle', 'V!LZ!N (km s!U-1!N)'
            ENDIF 
            
        ENDIF

        IF KEYWORD_SET(V_RTN) OR KEYWORD_SET(CDF) THEN BEGIN
            
            fillind = WHERE(ace_swe_all.V_RTN_X NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_RTN_X
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_RTN_X', data = str
                options, tvstr + 'V_RTN_X', 'ytitle', 'V!LX!N (km s!U-1!N)'
            ENDIF 

            fillind = WHERE(ace_swe_all.V_RTN_Y NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_RTN_Y
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_RTN_Y', data = str
                options, tvstr + 'V_RTN_Y', 'ytitle', 'V!LY!N (km s!U-1!N)'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.V_RTN_Z NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                velocity = ace_swe_all(fillind).V_RTN_Z
                str = {x:time, y:velocity}
                store_data, tvstr + 'V_RTN_Z', data = str
                options, tvstr + 'V_RTN_Z', 'ytitle', 'V!LZ!N (km s!U-1!N)'
            ENDIF 
            
        ENDIF
        
        IF KEYWORD_SET(PRESS) OR KEYWORD_SET(CDF) THEN BEGIN
            
            velocity = SQRT(ace_swe_all.V_GSE_X^2 + $
                            ace_swe_all.V_GSE_Y^2 + $
                            ace_swe_all.V_GSE_Z^2)
            
            fillind = WHERE(ace_swe_all.Np NE -1.0e+31 AND $
                            velocity NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                density = ace_swe_all(fillind).Np
                velocity = velocity
            
                mp = 1.67e-27   ; (kg)
                press = $
                  density * 1.e6 * mp *$
                  (velocity*1000.)^2*1.e9
                str = {x:time, y:press}
                store_data, tvstr + 'PRESSURE', data = str
                options, tvstr + 'PRESSURE', 'ytitle', 'P (nPa)'
            ENDIF 
        ENDIF
        
        IF KEYWORD_SET(X_GSE) OR KEYWORD_SET(CDF) THEN BEGIN
            
            fillind = WHERE(ace_swe_all.X_GSE_X NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                X_gse_x = ace_swe_all(fillind).X_GSE_X
                str = {x:time, y:X_gse_x}
                store_data, tvstr + 'X_GSE_X', data = str
                options, tvstr + 'X_GSE_X', 'ytitle', 'X gse'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.X_GSE_Y NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                X_gse_y = ace_swe_all(fillind).X_GSE_Y
                str = {x:time, y:X_gse_y}
                store_data, tvstr + 'X_GSE_Y', data = str
                options, tvstr + 'X_GSE_Y', 'ytitle', 'Y gse'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.X_GSE_Z NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                X_gse_z = ace_swe_all(fillind).X_GSE_Z
                str = {x:time, y:X_gse_z}
                store_data, tvstr + 'X_GSE_Z', data = str
                options, tvstr + 'X_GSE_Z', 'ytitle', 'Z gse'
            ENDIF 
            
        ENDIF
        
        IF KEYWORD_SET(X_GSM) OR KEYWORD_SET(CDF) THEN BEGIN
            
            fillind = WHERE(ace_swe_all.X_GSM_X NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                X_gsm_x = ace_swe_all(fillind).X_GSM_X
                str = {x:time, y:X_gsm_x}
                store_data, tvstr + 'X_GSM_X', data = str
                options, tvstr + 'X_GSM_X', 'ytitle', 'X gsm'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.X_GSM_Y NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                X_gsm_y = ace_swe_all(fillind).X_GSM_Y
                str = {x:time, y:X_gsm_y}
                store_data, tvstr + 'X_GSM_Y', data = str
                options, tvstr + 'X_GSM_Y', 'ytitle', 'Y gsm'
            ENDIF 
            
            fillind = WHERE(ace_swe_all.X_GSM_Z NE -1.0e+31, count)
            IF count GT 0 THEN BEGIN 
                time = ace_time_all(fillind)
                X_gsm_z = ace_swe_all(fillind).X_GSM_Z
                str = {x:time, y:X_gsm_z}
                store_data, tvstr + 'X_GSM_Z', data = str
                options, tvstr + 'X_GSM_Z', 'ytitle', 'Z gsm'
            ENDIF 
            
        ENDIF
        
    ENDIF
    
    
ENDIF

IF KEYWORD_SET(PRELIM_CDF) THEN BEGIN

    COMMON ace_swepam_data_prilim, ace_swe_prilim_time, ace_swe_prilim
    COMMON get_error, get_err_no, get_err_msg, default_verbose
    
    get_err_no = 0
    get_err_msg = 'Data found in time interval'
    
    path = GETENV('ACEDATA')
    path = path + '/swepam/cdf/'
    
    get_timespan, time_interval
    t_s = gettime(time_interval(0)) ; start time in tplot-time
    t_e = gettime(time_interval(1)) ; end time in tplot-time  
    
    t_s_str = time_struct(t_s)  ; start_time tplot time structure
    t_e_str = time_struct(t_e)  ; end_time tplot time structure
    
    mjd_s = julday(t_s_str.month, t_s_str.date, t_s_str.year) ;start julian day
    mjd_e = julday(t_e_str.month, t_e_str.date, t_e_str.year) ; end julian day
    
    no_of_files = (mjd_e - mjd_s) + 1 ; number of days to be loaded
    
                                ;Last day is not included if hour=min=sec=0
    IF t_e_str.hour EQ 0 AND t_e_str.min EQ 0 AND t_e_str.sec EQ 0 THEN $
      no_of_files = no_of_files - 1
    
    ;--------------------------------------------------------------------
    ; Read all 1 day files that correspond to requested time interval
    ;--------------------------------------------------------------------
    fc = 0                      ; Files-found counter
    FOR nd = 0, no_of_files-1 DO BEGIN ; Loop trough all days
        
        caldat, mjd_s + nd, month, day, year ; find caledar date
        
        filename = 'ac_k0_swe_' + $
                   STRING(year, month, day, $
                          format = '(i4.4,i2.2,i2.2)') + '_v*.cdf'
        r_ace_swepam_cdf_prelim, path, filename, PRELIM = PRELIM
        
        IF get_err_no EQ 0 THEN BEGIN
            
            IF fc EQ 0 THEN BEGIN
                ace_swe_all_prilim = ace_swe_prilim
                ace_time_all_prilim = ace_swe_prilim_time
            ENDIF ELSE BEGIN
                ace_swe_all_prilim = [ace_swe_all_prilim, ace_swe_prilim]
                ace_time_all_prilim = [ace_time_all_prilim, ace_swe_prilim_time]
            ENDELSE
            fc = fc + 1
        ENDIF
        
        
    ENDFOR
    
    IF fc GT 0 THEN BEGIN
        
        tvstr = 'ACE_SWEPAM_PRELIM_CDF_'
        
        IF KEYWORD_SET(Np) OR KEYWORD_SET(PRELIM_CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all_prilim.Np NE -1.0e+31)
            time = ace_time_all_prilim(fillind)
            density = ace_swe_all_prilim(fillind).Np
            str = {x:time, y:density}
            store_data, tvstr + 'DENSITY', data = str, lim = {ylog:1}
            options, tvstr + 'DENSITY', 'ytitle', 'n (cm!U-3!N)'
        ENDIF
        
        IF KEYWORD_SET(Vp) OR KEYWORD_SET(PRELIM_CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all_prilim.Vp NE -1.0e+31)
            time = ace_time_all_prilim(fillind)
            density = ace_swe_all_prilim(fillind).Vp
            str = {x:time, y:density}
            store_data, tvstr + 'Vp', data = str,  lim = {yrange:[1e2, 1e3], ylog:1}
            options, tvstr + 'Vp', 'ytitle', 'Vp (km s!U-1!N)'
        ENDIF
        
        IF KEYWORD_SET(Tpr) OR KEYWORD_SET(PRELIM_CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all_prilim.Tpr NE -1.0e+31)
            time = ace_time_all_prilim(fillind)
            density = ace_swe_all_prilim(fillind).Tpr
            str = {x:time, y:density}
            store_data, tvstr + 'Tpr', data = str, lim = {ylog:1}
            options, tvstr + 'Tpr', 'ytitle', 'Tpr (Kelvin)'
        ENDIF
        
        IF KEYWORD_SET(He_ratio) OR KEYWORD_SET(PRELIM_CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all_prilim.He_ratio NE -1.0e+31)
            time = ace_time_all_prilim(fillind)
            density = ace_swe_all_prilim(fillind).He_ratio
            str = {x:time, y:density}
            store_data, tvstr + 'He_ratio', data = str, lim = {yrange:[1e-3, 1], ylog:1}
            options, tvstr + 'He_ratio', 'ytitle', '%'
        ENDIF
        
        IF KEYWORD_SET(Weight) OR KEYWORD_SET(PRELIM_CDF) THEN BEGIN
            fillind = WHERE(ace_swe_all_prilim.Weight NE -1.0e+31)
            time = ace_time_all_prilim(fillind)
            density = ace_swe_all_prilim(fillind).Weight
            str = {x:time, y:density}
            store_data, tvstr + 'Weight', data = str
            options, tvstr + 'Weight', 'ytitle', ''
        ENDIF
        
    ENDIF
ENDIF



END
