;+
;PROCEDURE: seek_files
;PURPOSE:
;  Find the files needed for the timespan and satellites given.
;
;PARAMETERS:   in:  sat:        'A' or 'B'
;                   file_type:  'science', 'hk', 'cl' (classifier), or 'sc' (spacecraft housekeeping)
;              out: files_a:    Array of filenames (and paths) needed for
;                               given timespan 
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;
;	November 2006:  K. Simunac modified version number and prefix for housekeeping files
;	3 Jan, 2007:  K. Simunac modifies to look for latest version
;	of file
;       9 Jan, 2007: L. Ellis fixed bug with looping
;	9 Jan, 2007: K. Simunac adds type for spacecraft housekeeping
;       1 Feb, 2007: C. Mouikis - An extra day was read when time
;                                interval ended in 00:00:00 time
;       26 Feb, 2007: C. Mouikis - the level1 data environment
;                                  variable points to the cdf directory
;       26 Feb, 2007: C. Mouikis - two level1 data environment variables
;                                  are intriduced; one for each S/C
;-
PRO seek_files, sat, file_type, files

CASE file_type OF
    'science': type_string = ''
    'hk'     : type_string = 'HK_' ; K. Simunac removed underscore before HK
    'cl'     : type_string = 'CL_' ; K. Simunac removed underscore before CL
    'sc'     : type_string = 'SC_'
    ELSE     : BEGIN 
        print, "file_type must be 'science', 'hk', 'cl', or 'sc'"
        stop
    ENDELSE 
ENDCASE 

IF strcmp(sat, 'A') NE 1 AND strcmp(sat, 'B') NE 1 THEN BEGIN
    print, "sat must be 'A' or 'B'"
    stop
ENDIF 

get_timespan, times
times_struct = time_struct(times)

; Subtract a day if last day time is 00:00:00 - C. Mouikis
IF (times_struct[1].hour EQ 0) AND (times_struct[1].min) EQ 0 AND (times_struct[1].sec EQ 0) THEN BEGIN
    IF times_struct[1].doy GT 1 THEN BEGIN
        times_struct[1].doy = times_struct[1].doy - 1
    ENDIF ELSE BEGIN
        times_struct[1].year = times_struct[1].year - 1
        times_struct[1].doy = 365
        IF FIX(times_struct[1].year/4.) EQ (times_struct[1].year/4.) THEN times_struct[1].doy = 366
    ENDELSE
    
ENDIF

year = times_struct[0].year
doy = times_struct[0].doy
first = 1
WHILE (year LT times_struct[1].year) OR ((doy LE times_struct[1].doy) AND (year EQ times_struct[1].year)) DO BEGIN 
    doy_to_month_date, year, doy, month, date

    IF sat EQ 'A' THEN temp_dir = getenv('STA_PLA_L1DATA')+'/'+string(year, format = '(I4.4)')+'/' $
    ELSE temp_dir = getenv('STB_PLA_L1DATA')+'/'+string(year, format = '(I4.4)')+'/' 
    file_prefix = temp_dir+'ST'+sat+'_L1_PLA_'+type_string+string(year, format = '(I4.4)')$
                  +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')$
                  +'_'+string(doy, format = '(I3.3)')+'_V'

;    file_prefix = getenv('L1DATA')+'ST'+sat+'_L1_PLA_'+type_string+string(year, format = '(I4.4)')$
;                  +string(month, format = '(I2.2)')+string(date, format = '(I2.2)')$
;                  +'_'+string(doy, format = '(I3.3)')+'_V'
;    	print,file_prefix
    more = 1 
    version = 99 ; largest expected version number
    file_found = 0
    WHILE (more EQ 1) AND (version GT 0) DO BEGIN ; stop looking if there isn't a version 1 file
        file_string = file_prefix+string(version, format = '(I2.2)')+'.cdf'
        file_found = file_test(file_string, /read) ; returns 1 if file exists
        IF file_found EQ 1 THEN BEGIN
		more = 0  ; the most recent version has been found, stop looking for lower version numbers
       ENDIF ELSE BEGIN
	    version = version - 1 ; try the next lower version number
       ENDELSE 

    ENDWHILE

    IF file_found EQ 0 THEN file_string = ''
    IF first EQ 1 THEN BEGIN 
        files = [file_string]
        first = 0
    ENDIF ELSE files = [files, file_string]
    IF month EQ 12 AND date EQ 31 THEN BEGIN 
        doy = 1 
        year = year + 1
    ENDIF ELSE doy = doy + 1
ENDWHILE
END 
