;+
;PROCEDURE: pl_read_cdf_sc_hk
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch: 0(for variable), 1, or 5 minute resolution, 3 (for spacecraft housekeeping) 
;				
;
;CREATED BY: L. Ellis (pl_read_cdf)
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;
;	January 2007 K.Simunac adds spacecraft housekeeping variables
;	12 Jan, 2007:  modified variable and program names to avoid
;	confusion
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;-

PRO read_product_sc_hk, cdf_id, prod_name, data_var, data_index, epoch_maxrec

; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 

cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
IF varinfo.maxrec+1 GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec
cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
num_dim = size(data_var, /n_dimensions)
CASE num_dim OF 
    1: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    2: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    3: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, *, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    ELSE: BEGIN 
        print, "Lorna: Add more dimensions"
        stop
    END
ENDCASE 
END 


PRO remove_fill_sc_hk, fill_val, data

; fill_val: fill_value to be replaced by NaN
; data: array of data values
indice = where(data EQ fill_val, count)
FOR ii = 0D, count-1 DO BEGIN
    data[indice[ii]] = !values.d_nan
ENDFOR 


END


PRO init_sc_hk, sat, num_records3, sc_hk_initialized

; initialize all the housekeeping variabales (in com_sc_hk)
COMMON com_sc_hk
CASE sat OF
    'A': BEGIN 
	pl_ebox_tmp_a	= dblarr(num_records3) ; 0
	pl_inst_msg_pwr_dwn_req_a	= strarr(num_records3) ; 1
	pl_pwr_cur_a	= dblarr(num_records3) ; 2
	pl_pwr_tt_a	= strarr(num_records3) ; 3
	pl_surv_pwr_cur_a	= dblarr(num_records3) ; 4
	pl_surv_pwr_tt_a	= strarr(num_records3) ; 5
	pl_tof_tmp_a	= dblarr(num_records3) ; 6
	pl_pwr_vt_a	= dblarr(num_records3) ; 7
	pl_surv_pwr_vt_a	= dblarr(num_records3) ; 8
	
	pl_ebox_tmp_a[*] = !values.d_nan ; 0
	pl_inst_msg_pwr_dwn_req_a[*] = !values.d_nan ; 1
	pl_pwr_cur_a[*] = !values.d_nan ; 2
	pl_pwr_tt_a[*] = !values.d_nan ; 3
	pl_surv_pwr_cur_a[*] = !values.d_nan ; 4
	pl_surv_pwr_tt_a[*] = !values.d_nan ; 5
	pl_tof_tmp_a[*] = !values.d_nan ; 6
	pl_pwr_vt_a[*] = !values.d_nan ; 7
	pl_surv_pwr_vt_a[*] = !values.d_nan ; 8

    END 
    'B': BEGIN 
	pl_ebox_tmp_b	= dblarr(num_records3) ; 0
	pl_inst_msg_pwr_dwn_req_b	= strarr(num_records3) ; 1
	pl_pwr_cur_b	= dblarr(num_records3) ; 2
	pl_pwr_tt_b	= strarr(num_records3) ; 3
	pl_surv_pwr_cur_b	= dblarr(num_records3) ; 4
	pl_surv_pwr_tt_b	= strarr(num_records3) ; 5
	pl_tof_tmp_b	= dblarr(num_records3) ; 6
	pl_pwr_vt_b	= dblarr(num_records3) ; 7
	pl_surv_pwr_vt_b	= dblarr(num_records3) ; 8
	
	pl_ebox_tmp_b[*] = !values.d_nan ; 0
	pl_inst_msg_pwr_dwn_req_b[*] = !values.d_nan ; 1
	pl_pwr_cur_b[*] = !values.d_nan ; 2
	pl_pwr_tt_b[*] = !values.d_nan ; 3
	pl_surv_pwr_cur_b[*] = !values.d_nan ; 4
	pl_surv_pwr_tt_b[*] = !values.d_nan ; 5
	pl_tof_tmp_b[*] = !values.d_nan ; 6
	pl_pwr_vt_b[*] = !values.d_nan ; 7
	pl_surv_pwr_vt_b[*] = !values.d_nan ; 8
        
    END 
ENDCASE 
sc_hk_initialized = 1

END 

PRO read_sc_hk, cdf_id, sat, sc_hk_read, data3_index, epoch3maxrec, products, epoch3

COMMON com_sc_hk
CASE sat OF 
    'A': BEGIN
	
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'pl_ebox_tmp':	BEGIN ;0
			read_product_sc_hk, cdf_id, 'pl_ebox_tmp',	pl_ebox_tmp_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_ebox_tmp_a
		END
		'pl_inst_msg_pwr_dwn_req':	BEGIN ;1
			read_product_sc_hk, cdf_id, 'pl_inst_msg_pwr_dwn_req',	pl_inst_msg_pwr_dwn_req_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, ' ', pl_inst_msg_pwr_dwn_req_a
		END
		'pl_pwr_cur':	BEGIN ;2
			read_product_sc_hk, cdf_id, 'pl_pwr_cur',	pl_pwr_cur_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_pwr_cur_a
		END 
		'pl_pwr_tt':	BEGIN ;3
			read_product_sc_hk, cdf_id, 'pl_pwr_tt',	pl_pwr_tt_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, ' ', pl_pwr_tt_a
		END
		'pl_surv_pwr_cur':  BEGIN ;4
			read_product_sc_hk, cdf_id, 'pl_surv_pwr_cur',	pl_surv_pwr_cur_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_surv_pwr_cur_a
		END
		'pl_surv_pwr_tt':	BEGIN ;5
			read_product_sc_hk, cdf_id, 'pl_surv_pwr_tt',	pl_surv_pwr_tt_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, ' ', pl_surv_pwr_tt_a
		END
		'pl_tof_tmp':	BEGIN ;6
			read_product_sc_hk, cdf_id, 'pl_tof_tmp',	pl_tof_tmp_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_tof_tmp_a
		END
		'pl_pwr_vt':	BEGIN ;7
			read_product_sc_hk, cdf_id, 'pl_pwr_vt',	pl_pwr_vt_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_pwr_vt_a
		END
		'pl_surv_pwr_vt':	BEGIN ;8
			read_product_sc_hk, cdf_id, 'pl_surv_pwr_vt',	pl_surv_pwr_vt_a,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_surv_pwr_vt_a
		END
		ENDCASE
	ENDFOR

    END
    'B': BEGIN
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'pl_ebox_tmp':	BEGIN ;0
			read_product_sc_hk, cdf_id, 'pl_ebox_tmp',	pl_ebox_tmp_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_ebox_tmp_b
		END
		'pl_inst_msg_pwr_dwn_req':	BEGIN ;1
			read_product_sc_hk, cdf_id, 'pl_inst_msg_pwr_dwn_req',	pl_inst_msg_pwr_dwn_req_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, ' ', pl_inst_msg_pwr_dwn_req_b
		END
		'pl_pwr_cur':	BEGIN ;2
			read_product_sc_hk, cdf_id, 'pl_pwr_cur',	pl_pwr_cur_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_pwr_cur_b
		END 
		'pl_pwr_tt':	BEGIN ;3
			read_product_sc_hk, cdf_id, 'pl_pwr_tt',	pl_pwr_tt_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, ' ', pl_pwr_tt_b
		END
		'pl_surv_pwr_cur':  BEGIN ;4
			read_product_sc_hk, cdf_id, 'pl_surv_pwr_cur',	pl_surv_pwr_cur_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_surv_pwr_cur_b
		END
		'pl_surv_pwr_tt':	BEGIN ;5
			read_product_sc_hk, cdf_id, 'pl_surv_pwr_tt',	pl_surv_pwr_tt_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, ' ', pl_surv_pwr_tt_b
		END
		'pl_tof_tmp':	BEGIN ;6
			read_product_sc_hk, cdf_id, 'pl_tof_tmp',	pl_tof_tmp_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_tof_tmp_b
		END
		'pl_pwr_vt':	BEGIN ;7
			read_product_sc_hk, cdf_id, 'pl_pwr_vt',	pl_pwr_vt_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_pwr_vt_b
		END
		'pl_surv_pwr_vt':	BEGIN ;8
			read_product_sc_hk, cdf_id, 'pl_surv_pwr_vt',	pl_surv_pwr_vt_b,	data3_index,	epoch3maxrec
			remove_fill_sc_hk, -1, pl_surv_pwr_vt_b
		END
		ENDCASE
	ENDFOR
    END
ENDCASE
sc_hk_read = 1
END 

PRO check_indices_sc_hk, index, epoch, resolution, file_num
; checks for missing records from the cdf epoch variable
; index: array of pairs [start_index,num_records] (could be [0,10,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1,5, or 3 depending on epoch (may need to add variable)
; file_num: index of this file in the file array

first_index=0L
first_index = file_num*(1440/resolution)
IF (resolution EQ 3) THEN first_index = LONG(file_num*8640) ; resolution 3 is for one per second

num_records = 0
in_valid_area = 0
ii = 0L
index = [-1L]
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 59 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*60)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]                
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    5: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 11 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF (hour EQ time.hour AND time.min GE (min*5) AND time.min LT (min*5+5)) OR $
                      (time.hour EQ hour+1 AND time.min LE 4 AND min EQ 11) THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*12)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    3: BEGIN
	FOR hour=0L,23 DO BEGIN
	   FOR min=0L,59 DO BEGIN
		FOR sec=0L,5 DO BEGIN
		   IF ii LT n_elements(epoch) THEN BEGIN
			time = time_struct(epoch[ii])
			;cdf_epoch, epoch[ii], year, month, day, time_hour, time_min, time_sec, milli,/breakdown_epoch
			;print, ii, time_hour, time_min, time_sec
			IF (hour EQ time.hour AND min EQ time.min AND sec EQ time.sec) $
			OR (hour EQ time.hour AND min EQ time.min AND time.sec LT sec*10+10) THEN BEGIN
				IF in_valid_area EQ 0 THEN BEGIN
				   start_index = (first_index+(hour*360)+(min*6)+(sec))	
				   IF start_index LT 0 THEN STOP
				   num_records = 1
				   in_valid_area = 1
				ENDIF ELSE BEGIN ; already in valid area
				   num_records = num_records+1
				ENDELSE
				ii = ii+1
				IF sec GT 0 THEN BEGIN
					sec = sec-1
				ENDIF ELSE BEGIN
					IF (min GT 0) AND (sec EQ 0) THEN BEGIN
					min = min-1 
					sec = sec+59
					ENDIF ELSE BEGIN
						IF (hour GT 0) AND (min EQ 0) and (sec EQ 0) THEN BEGIN
						hour = hour-1
						min = min+59 
						sec = sec+59
						ENDIF
					ENDELSE
				ENDELSE
			ENDIF ELSE BEGIN
				IF in_valid_area EQ 1 THEN BEGIN
				   in_valid_area = 0
				   IF index[0] EQ -1 THEN index = [start_index, num_records] $
				   ELSE index = [index, start_index, num_records]
				ENDIF
			ENDELSE
		ENDIF ELSE BEGIN
			IF in_valid_area EQ 1 THEN BEGIN
				in_valid_area=0
				IF index[0] EQ -1 THEN index = [start_index, num_records] $
				ELSE index = [index, start_index, num_records]
			ENDIF
		ENDELSE
	      ENDFOR
	   ENDFOR
	ENDFOR

    END
    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf_sc_hk, sat, files, products, which_epoch

COMMON share1_sc_hk
COMMON com_sc_hk
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat

; initialize output variables (from common block)
need_epoch1 = 0
need_epoch5 = 0
need_epoch3 = 0
FOR ii = 0L, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
        1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
        END 
        5: BEGIN 
            num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 5
        END 
	3: BEGIN
	  ; num_records3 = DOUBLE(n_elements(files)*86401)
	   num_records3=DOUBLE(n_elements(files)*8651)
	   epoch3 = DBLARR(num_records3)
	   epoch3(*) = !values.d_nan
	   need_epoch3 = 3
	END
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 
sc_hk_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	'pl_ebox_tmp': ; 0
	'pl_inst_msg_pwr_dwn_req': ; 1
	'pl_pwr_cur': ; 2
	'pl_pwr_tt': ; 3
	'pl_surv_pwr_cur': ; 4
	'pl_surv_pwr_tt': ; 5
	'pl_tof_tmp': ; 6
	'pl_pwr_vt': ; 7
	'pl_surv_pwr_v': $; 8
	 IF sc_hk_initialized EQ 0 THEN init_sc_hk, sat, num_records3, sc_hk_initialized
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch_analog', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch_analog', temp_epoch, rec_count = epoch1info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_sc_hk, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data1_index[kk+1]-1)]
                            converted_index = converted_index+data1_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF
                END 
                5: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5', /zvariable
                    IF epoch5info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch5', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_sc_hk, data5_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data5_index[kk+1]-1)]
                            converted_index = converted_index+data5_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF 
                END 
		3:  BEGIN
		   cdf_control, cdf_id, get_var_info = epoch3info, var = 'epoch', /zvariable
		   IF epoch3info.maxrec GE 0 THEN BEGIN
			cdf_varget, cdf_id, 'epoch', temp_epoch, rec_count = epoch3info.maxrec+1, /zvariable
			converted_epoch = time_double(temp_epoch, /epoch)
			check_indices_sc_hk, data3_index, converted_epoch, which_epoch, ii
			;check_indices_sc_hk, data3_index, temp_epoch, which_epoch, ii
			converted_index = 0L
			 FOR kk = 0L, n_elements(data3_index)-1 DO BEGIN 
                            epoch3[data3_index[kk]:(data3_index[kk]+data3_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data3_index[kk+1]-1)]
                            converted_index = converted_index+data3_index[kk+1]
                            kk = kk+1
                        ENDFOR
		   ENDIF

		remove_fill_sc_hk, -1, epoch3
		END
                0: BEGIN 
                    print, "Lorna: add variable epoch"
                    stop
                END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_a = epoch5
	    IF need_epoch3 EQ 3 THEN epoch3_a = epoch3
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
	    IF need_epoch3 EQ 3 THEN epoch3_b = epoch3
        ENDELSE 

                                ; get other data
        sc_hk_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		'pl_ebox_tmp': ; 0
		'pl_inst_msg_pwr_dwn_req': ; 1
		'pl_pwr_cur': ; 2
		'pl_pwr_tt': ; 3
		'pl_surv_pwr_cur': ; 4
		'pl_surv_pwr_tt': ; 5
		'pl_tof_tmp': ; 6
		'pl_pwr_vt': ; 7
		'pl_surv_pwr_v': $; 8
			IF sc_hk_read EQ 0 THEN read_sc_hk, cdf_id, sat, sc_hk_read, data3_index, epoch3info.maxrec+1, products, epoch3
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
    IF need_epoch5 EQ 5 THEN data5_index_a = data5_index
    IF need_epoch3 EQ 3 THEN data3_index_a = data3_index 
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
    IF need_epoch5 EQ 5 THEN data5_index_b = data5_index
    IF need_epoch3 EQ 3 THEN data3_index_b = data3_index 
ENDELSE 

END 
