;+
;PROCEDURE: pl_read_cdf_mon
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch: 0(for variable), 1, or 5 minute resolution, 2 (for analog housekeeping, 
;				which can be reported every 2 seconds)
;
;CREATED BY: L. Ellis (pl_read_cdf)
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;
;	December 2006 K.Simunac modifies for monitor rates
;	4 Jan, 2007:  modified to check for error flags
;	7 Jan, 2007:  modified to output counts per second (if desired)
;	8 Jan, 2007:  modified to calculate select ratios
;	12 Jan, 2007:  modified variable and program names to avoid confusion
;		read_product -> read_product_mon
;		remove_fill -> remove_fill_mon
;		init_mom -> init_mon
;		mom_initialized -> mon_initialized
;		mom_read -> mon_read
;		check_indices -> check_indices_mon
;	24 Jan, 2007:  IMPORTANT FIX to sum_defl... index system
;       29 Mar, 2007: Set file_name to '' if no records & added check
;                     if maxrec = 0 (LBE)
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;   05-30-2007: CM
;               Overflow values corrected
;-

PRO read_product_mon, cdf_id, prod_name, data_var, data_index, epoch_maxrec

; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 

cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable

IF varinfo.maxrec+1 GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec

IF maxrec GT 0 THEN BEGIN 
    cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
    num_dim = size(data_var, /n_dimensions)

    CASE num_dim OF 
        1: BEGIN
            temp_index = 0L
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[temp_index:(temp_index+data_index[kk+1]-1)]
                temp_index = temp_index+data_index[kk+1]
                kk = kk+1
            ENDFOR 
        END 
        2: BEGIN
            temp_index = 0L
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, temp_index:(temp_index+data_index[kk+1]-1)]
                temp_index = temp_index+data_index[kk+1]
                kk = kk+1
            ENDFOR 
        END 
        3: BEGIN
            temp_index = 0L
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, *, temp_index:(temp_index+data_index[kk+1]-1)]
                temp_index = temp_index+data_index[kk+1]
                kk = kk+1
            ENDFOR 
        END 
        ELSE: BEGIN 
            print, "Lorna: Add more dimensions"
            stop
        END
    ENDCASE 
ENDIF ELSE data_index = -1
END 

PRO remove_fill_mon, fill_val, data

; fill_val: fill_value to be replaced by NaN
; data: array of data values
indice = where(data EQ fill_val, count)
FOR ii = 0L, count-1 DO BEGIN
    data[indice[ii]] = !values.d_nan
ENDFOR 

END 


PRO init_mon, sat, num_records1, num_records5, mon_initialized, E_step, products

; initialize all the monitor rate variables (in com_mon)

num_E = n_elements(E_step)


COMMON com_mon
num_rows1 = num_records1
num_rows5 = num_records5

CASE sat OF
    'A': BEGIN 
	
	error5_mon_a	= fltarr(num_records5) ; error flag science mode
	error5_mon_a[*]	= !values.f_nan
	error1_a	= fltarr(num_records1) ; error flag engineering mode
	error1_a[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_a_sci	= dblarr(32,8,num_records5) ; 0
		s_valid_a_eng	= dblarr(32,8,num_records1) ;0
		s_valid_a_sci[*,*,*] 	= !values.d_nan
		s_valid_a_eng[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_a_sci = dblarr(num_records5,32) ;0
		sum_defl_s_valid_a_eng = dblarr(num_records1,32) ;0
		sum_defl_s_valid_a_sci[*,*] = !values.d_nan ;0
		sum_defl_s_valid_a_eng[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_a_sci	= dblarr(32,8,num_records5) ; 1
		s_e_not_req_a_eng	= dblarr(32,8,num_records1) ; 1
		s_e_not_req_a_sci[*,*,*]	= !values.d_nan
		s_e_not_req_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_a_sci = dblarr(num_records5,32) ;1
		sum_defl_s_e_not_req_a_eng = dblarr(num_records1,32) ;1
		sum_defl_s_e_not_req_a_sci[*,*] = !values.d_nan ;1
		sum_defl_s_e_not_req_a_eng[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_a_sci	= dblarr(32,8,num_records5) ; 2
		s_e_req_a_eng	= dblarr(32,8,num_records1) ; 2
		s_e_req_a_sci[*,*,*]	= !values.d_nan
		s_e_req_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_a_sci = dblarr(num_records5,32) ;2
		sum_defl_s_e_req_a_eng = dblarr(num_records1,32) ;2
		sum_defl_s_e_req_a_sci[*,*] = !values.d_nan ;2
		sum_defl_s_e_req_a_eng[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_a_sci	= dblarr(32,8,num_records5) ; 3
		s_no_pos_a_eng	= dblarr(32,8,num_records1) ; 3
		s_no_pos_a_sci[*,*,*]	= !values.d_nan
		s_no_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_a_sci = dblarr(num_records5,32) ;3
		sum_defl_s_no_pos_a_eng = dblarr(num_records1,32) ;3
		sum_defl_s_no_pos_a_sci[*,*] = !values.d_nan ;3
		sum_defl_s_no_pos_a_eng[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_a_sci	= dblarr(32,8,num_records5) ; 4
		s_mult_pos_a_eng	= dblarr(32,8,num_records1) ; 4
		s_mult_pos_a_sci[*,*,*]	= !values.d_nan
		s_mult_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_a_sci = dblarr(num_records5,32) ;4
		sum_defl_s_mult_pos_a_eng = dblarr(num_records1,32) ;4
		sum_defl_s_mult_pos_a_sci[*,*] = !values.d_nan ;4
		sum_defl_s_mult_pos_a_eng[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_a_sci	= dblarr(32,8,num_records5) ; 5
		s_no_e_a_eng	= dblarr(32,8,num_records1) ; 5
		s_no_e_a_sci[*,*,*]	= !values.d_nan
		s_no_e_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_a_sci = dblarr(num_records5,32) ;5
		sum_defl_s_no_e_a_eng = dblarr(num_records1,32) ;5
		sum_defl_s_no_e_a_sci[*,*] = !values.d_nan ;5
		sum_defl_s_no_e_a_eng[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_a_sci	= dblarr(32,8,num_records5) ; 6
		s_mult_e_a_eng	= dblarr(32,8,num_records1) ; 6
		s_mult_e_a_sci[*,*,*]	= !values.d_nan
		s_mult_e_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_a_sci = dblarr(num_records5,32) ;6
		sum_defl_s_mult_e_a_eng = dblarr(num_records1,32) ;6
		sum_defl_s_mult_e_a_sci[*,*] = !values.d_nan ;6
		sum_defl_s_mult_e_a_eng[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_a_sci	= dblarr(32,8,num_records5) ; 7
		ra_sat_a_a_eng	= dblarr(32,8,num_records1) ; 7
		ra_sat_a_a_sci[*,*,*]	= !values.d_nan
		ra_sat_a_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_a_sci = dblarr(num_records5,32) ;7
		sum_defl_ra_sat_a_a_eng = dblarr(num_records1,32) ;7
		sum_defl_ra_sat_a_a_sci[*,*] = !values.d_nan ;7
		sum_defl_ra_sat_a_a_eng[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_a_sci	= dblarr(32,8,num_records5) ; 8
		ra_sat_b_a_eng	= dblarr(32,8,num_records1) ; 8
		ra_sat_b_a_sci[*,*,*]= !values.d_nan
		ra_sat_b_a_eng[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_a_sci = dblarr(num_records5,32) ;8
		sum_defl_ra_sat_b_a_eng = dblarr(num_records1,32) ;8
		sum_defl_ra_sat_b_a_sci[*,*] = !values.d_nan ;8
		sum_defl_ra_sat_b_a_eng[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_a_sci	= dblarr(32,8,num_records5) ; 9
		ra_sat_both_a_eng	= dblarr(32,8,num_records1) ; 9
		ra_sat_both_a_sci[*,*,*]	= !values.d_nan
		ra_sat_both_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_a_sci = dblarr(num_records5,32) ;9
		sum_defl_ra_sat_both_a_eng = dblarr(num_records1,32) ;9
		sum_defl_ra_sat_both_a_sci[*,*] = !values.d_nan ;9
		sum_defl_ra_sat_both_a_eng[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_a_sci	= dblarr(32,8,num_records5) ; 10
		ssd_sw_a_eng	= dblarr(32,8,num_records1) ; 10
		ssd_sw_a_sci[*,*,*]	= !values.d_nan
		ssd_sw_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_a_sci = dblarr(num_records5,32) ;10
		sum_defl_ssd_sw_a_eng = dblarr(num_records1,32) ;10
		sum_defl_ssd_sw_a_sci[*,*] = !values.d_nan ;10
		sum_defl_ssd_sw_a_eng[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_a_sci	= dblarr(32,8,num_records5) ; 11
		ssd_st_a_eng	= dblarr(32,8,num_records1) ; 11
		ssd_st_a_sci[*,*,*]	= !values.d_nan
		ssd_st_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_a_sci = dblarr(num_records5,32) ;11
		sum_defl_ssd_st_a_eng = dblarr(num_records1,32) ;11
		sum_defl_ssd_st_a_sci[*,*] = !values.d_nan ;11
		sum_defl_ssd_st_a_eng[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_eng	= dblarr(32,8,num_records1) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sf0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_a_eng[*,*] = !values.d_nan ;12		
		END
	'sfr0':  BEGIN
		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sfr0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_a_eng[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_a_sci	= dblarr(32,8,num_records5) ; 14
		stp0_a_eng	= dblarr(32,8,num_records1) ; 14
		stp0_a_sci[*,*,*]	= !values.d_nan
		stp0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_a_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_a_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_a_eng[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_a_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_a_eng	= dblarr(32,8,num_records1) ; 15
		ra_trig_a_sci[*,*,*]	= !values.d_nan
		ra_trig_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_a_eng = dblarr(num_records1,32) ;15
		sum_defl_ra_trig_a_sci[*,*] = !values.d_nan ;15
		sum_defl_ra_trig_a_eng[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_a_sci	= dblarr(32,8,num_records5) ; 16
		pos1_0_a_eng	= dblarr(32,8,num_records1) ; 16
		pos1_0_a_sci[*,*,*]	= !values.d_nan
		pos1_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_a_sci = dblarr(num_records5,32) ;16
		sum_defl_pos1_0_a_eng = dblarr(num_records1,32) ;16
		sum_defl_pos1_0_a_sci[*,*] = !values.d_nan ;16
		sum_defl_pos1_0_a_eng[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_a_sci	= dblarr(32,8,num_records5) ; 17
		pos1_1_a_eng	= dblarr(32,8,num_records1) ; 17
		pos1_1_a_sci[*,*,*]	= !values.d_nan
		pos1_1_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_a_sci = dblarr(num_records5,32) ;17
		sum_defl_pos1_1_a_eng = dblarr(num_records1,32) ;17
		sum_defl_pos1_1_a_sci[*,*] = !values.d_nan ;17
		sum_defl_pos1_1_a_eng[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_a_sci	= dblarr(32,8,num_records5) ; 18
		w_no_pos_a_eng	= dblarr(32,8,num_records1) ; 18
		w_no_pos_a_sci[*,*,*]	= !values.d_nan
		w_no_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_a_sci = dblarr(num_records5,32) ;18
		sum_defl_w_no_pos_a_eng = dblarr(num_records1,32) ;18
		sum_defl_w_no_pos_a_sci[*,*] = !values.d_nan ;18
		sum_defl_w_no_pos_a_eng[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_a_sci	= dblarr(32,8,num_records5) ; 19
		w_mult_pos_a_eng	= dblarr(32,8,num_records1) ; 19
		w_mult_pos_a_sci[*,*,*]	= !values.d_nan
		w_mult_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_a_sci = dblarr(num_records5,32) ;19
		sum_defl_w_mult_pos_a_eng = dblarr(num_records1,32) ;19
		sum_defl_w_mult_pos_a_sci[*,*] = !values.d_nan ;19
		sum_defl_w_mult_pos_a_eng[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_a_sci	= dblarr(32,8,num_records5) ; 20
		w_valid_a_eng	= dblarr(32,8,num_records1) ; 20
		w_valid_a_sci[*,*,*]	= !values.d_nan
		w_valid_a_eng[*,*,*]	= !values.d_nan
		sum_defl_w_valid_a_sci = dblarr(num_records5,32) ;20
		sum_defl_w_valid_a_eng = dblarr(num_records1,32) ;20
		sum_defl_w_valid_a_sci[*,*] = !values.d_nan ;20
		sum_defl_w_valid_a_eng[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_a_sci	= dblarr(32,8,num_records5) ; 21
		sf2_a_eng	= dblarr(32,8,num_records1) ; 21
		sf2_a_sci[*,*,*]	= !values.d_nan
		sf2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_a_eng = dblarr(num_records1,32) ;21
		sum_defl_sf2_a_sci[*,*] = !values.d_nan ;21
		sum_defl_sf2_a_eng[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sfr2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_a_eng[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_a_sci	= dblarr(32,8,num_records5) ; 23
		stp2_a_eng	= dblarr(32,8,num_records1) ; 23
		stp2_a_sci[*,*,*]	= !values.d_nan	
		stp2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_a_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_a_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_a_eng[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_a_eng	= dblarr(32,8,num_records1) ; 24
		pos2_0_a_sci[*,*,*]	= !values.d_nan	
		pos2_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_a_eng = dblarr(num_records1,32) ;24
		sum_defl_pos2_0_a_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos2_0_a_eng[*,*] = !values.d_nan ;24
		END
	'pos2_1':  BEGIN
		pos2_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_a_eng	= dblarr(32,8,num_records1) ; 25
		pos2_1_a_sci[*,*,*]	= !values.d_nan	
		pos2_1_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_a_eng = dblarr(num_records1,32) ;25
		sum_defl_pos2_1_a_sci[*,*] = !values.d_nan ;25
		sum_defl_pos2_1_a_eng[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_a_eng	= dblarr(32,8,num_records1) ; 26
		pos2_2_a_sci[*,*,*]	= !values.d_nan
		pos2_2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_a_eng = dblarr(num_records1,32) ;26
		sum_defl_pos2_2_a_sci[*,*] = !values.d_nan ;26
		sum_defl_pos2_2_a_eng[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_a_eng	= dblarr(32,8,num_records1) ; 27
		pos2_3_a_sci[*,*,*]	= !values.d_nan	
		pos2_3_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_a_eng = dblarr(num_records1,32) ;27
		sum_defl_pos2_3_a_sci[*,*] = !values.d_nan ;27
		sum_defl_pos2_3_a_eng[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_a_sci	= dblarr(32,8,num_records5) ; 28
		pos3_0_a_eng	= dblarr(32,8,num_records1) ; 28
		pos3_0_a_sci[*,*,*]	= !values.d_nan	
		pos3_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_0_a_sci = dblarr(num_records5,32) ;28
		sum_defl_pos3_0_a_eng = dblarr(num_records1,32) ;28
		sum_defl_pos3_0_a_sci[*,*] = !values.d_nan ;28
		sum_defl_pos3_0_a_eng[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_a_sci	= dblarr(32,8,num_records5) ; 29
		pos3_1_a_eng	= dblarr(32,8,num_records1) ; 29
		pos3_1_a_sci[*,*,*]	= !values.d_nan	
		pos3_1_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_1_a_sci = dblarr(num_records5,32) ;29
		sum_defl_pos3_1_a_eng = dblarr(num_records1,32) ;29
		sum_defl_pos3_1_a_sci[*,*] = !values.d_nan ;29
		sum_defl_pos3_1_a_eng[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_a_sci	= dblarr(32,8,num_records5) ; 30
		pos3_2_a_eng	= dblarr(32,8,num_records1) ; 30
		pos3_2_a_sci[*,*,*]	= !values.d_nan
		pos3_2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_sci = dblarr(num_records5,32) ;30
		sum_defl_pos3_2_a_eng = dblarr(num_records1,32) ;30
		sum_defl_pos3_2_a_sci[*,*] = !values.d_nan ;30
		sum_defl_pos3_2_a_eng[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_a_sci	= dblarr(32,8,num_records5) ; 31
		pos3_3_a_eng	= dblarr(32,8,num_records1) ; 31
		pos3_3_a_sci[*,*,*]	= !values.d_nan
		pos3_3_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_a_sci = dblarr(num_records5,32) ;31
		sum_defl_pos3_3_a_eng = dblarr(num_records1,32) ;31
		sum_defl_pos3_3_a_sci[*,*] = !values.d_nan ;31
		sum_defl_pos3_3_a_eng[*,*] = !values.d_nan ;31
		END
	'stop0':  BEGIN
		stop0_a_sci	= dblarr(32,8,num_records5) 
		stop0_a_eng	= dblarr(32,8,num_records1) 
		stop0_a_sci[*,*,*]	= !values.d_nan
		stop0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stop0_a_sci = dblarr(num_records5,32) 
		sum_defl_stop0_a_eng = dblarr(num_records1,32) 
		sum_defl_stop0_a_sci[*,*] = !values.d_nan 
		sum_defl_stop0_a_eng[*,*] = !values.d_nan 

		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_eng	= dblarr(32,8,num_records1) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sf0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_a_eng[*,*] = !values.d_nan ;12	

		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sfr0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_a_eng[*,*] = !values.d_nan ;13
	
		END

	'start0':  BEGIN
		start0_a_sci	= dblarr(32,8,num_records5) 
		start0_a_eng	= dblarr(32,8,num_records1) 
		start0_a_sci[*,*,*]	= !values.d_nan
		start0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_start0_a_sci = dblarr(num_records5,32) 
		sum_defl_start0_a_eng = dblarr(num_records1,32)
		sum_defl_start0_a_sci[*,*] = !values.d_nan
		sum_defl_start0_a_eng[*,*] = !values.d_nan

		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sfr0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_a_eng[*,*] = !values.d_nan ;13

		stp0_a_sci	= dblarr(32,8,num_records5) ; 14
		stp0_a_eng	= dblarr(32,8,num_records1) ; 14
		stp0_a_sci[*,*,*]	= !values.d_nan
		stp0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_a_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_a_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_a_eng[*,*] = !values.d_nan ;14

		END
	'stop2':  BEGIN
		stop2_a_sci	= dblarr(32,8,num_records5)
		stop2_a_eng	= dblarr(32,8,num_records1)
		stop2_a_sci[*,*,*]	= !values.d_nan
		stop2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stop2_a_sci = dblarr(num_records5,32)
		sum_defl_stop2_a_eng = dblarr(num_records1,32)
		sum_defl_stop2_a_sci[*,*] = !values.d_nan
		sum_defl_stop2_a_eng[*,*] = !values.d_nan

		sf2_a_sci	= dblarr(32,8,num_records5) ; 21
		sf2_a_eng	= dblarr(32,8,num_records1) ; 21
		sf2_a_sci[*,*,*]	= !values.d_nan
		sf2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_a_eng = dblarr(num_records1,32) ;21
		sum_defl_sf2_a_sci[*,*] = !values.d_nan ;21
		sum_defl_sf2_a_eng[*,*] = !values.d_nan ;21
		
		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sfr2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_a_eng[*,*] = !values.d_nan ;22
		END

	'start2':  BEGIN
		start2_a_sci	= dblarr(32,8,num_records5)
		start2_a_eng	= dblarr(32,8,num_records1) 
		start2_a_sci[*,*,*]	= !values.d_nan
		start2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_start2_a_sci = dblarr(num_records5,32) 
		sum_defl_start2_a_eng = dblarr(num_records1,32) 
		sum_defl_start2_a_sci[*,*] = !values.d_nan 
		sum_defl_start2_a_eng[*,*] = !values.d_nan 

		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sfr2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_a_eng[*,*] = !values.d_nan ;22

		stp2_a_sci	= dblarr(32,8,num_records5) ; 23
		stp2_a_eng	= dblarr(32,8,num_records1) ; 23
		stp2_a_sci[*,*,*]	= !values.d_nan	
		stp2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_a_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_a_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_a_eng[*,*] = !values.d_nan ;23
		END

	ENDCASE
	ENDFOR			
    END 
    'B': BEGIN 

	error5_mon_b	= fltarr(num_records5) ; error flag science mode
	error5_mon_b[*]	= !values.f_nan
	error1_b	= fltarr(num_records1) ; error flag engineering mode
	error1_b[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_b_sci	= dblarr(32,8,num_records5) ; 0
		s_valid_b_eng	= dblarr(32,8,num_records1) ;0
		s_valid_b_sci[*,*,*] 	= !values.d_nan
		s_valid_b_eng[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_b_sci = dblarr(num_records5,32) ;0
		sum_defl_s_valid_b_eng = dblarr(num_records1,32) ;0
		sum_defl_s_valid_b_sci[*,*] = !values.d_nan ;0
		sum_defl_s_valid_b_eng[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_b_sci	= dblarr(32,8,num_records5) ; 1
		s_e_not_req_b_eng	= dblarr(32,8,num_records1) ; 1
		s_e_not_req_b_sci[*,*,*]	= !values.d_nan
		s_e_not_req_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_b_sci = dblarr(num_records5,32) ;1
		sum_defl_s_e_not_req_b_eng = dblarr(num_records1,32) ;1
		sum_defl_s_e_not_req_b_sci[*,*] = !values.d_nan ;1
		sum_defl_s_e_not_req_b_eng[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_b_sci	= dblarr(32,8,num_records5) ; 2
		s_e_req_b_eng	= dblarr(32,8,num_records1) ; 2
		s_e_req_b_sci[*,*,*]	= !values.d_nan
		s_e_req_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_b_sci = dblarr(num_records5,32) ;2
		sum_defl_s_e_req_b_eng = dblarr(num_records1,32) ;2
		sum_defl_s_e_req_b_sci[*,*] = !values.d_nan ;2
		sum_defl_s_e_req_b_eng[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_b_sci	= dblarr(32,8,num_records5) ; 3
		s_no_pos_b_eng	= dblarr(32,8,num_records1) ; 3
		s_no_pos_b_sci[*,*,*]	= !values.d_nan
		s_no_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_b_sci = dblarr(num_records5,32) ;3
		sum_defl_s_no_pos_b_eng = dblarr(num_records1,32) ;3
		sum_defl_s_no_pos_b_sci[*,*] = !values.d_nan ;3
		sum_defl_s_no_pos_b_eng[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_b_sci	= dblarr(32,8,num_records5) ; 4
		s_mult_pos_b_eng	= dblarr(32,8,num_records1) ; 4
		s_mult_pos_b_sci[*,*,*]	= !values.d_nan
		s_mult_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_b_sci = dblarr(num_records5,32) ;4
		sum_defl_s_mult_pos_b_eng = dblarr(num_records1,32) ;4
		sum_defl_s_mult_pos_b_sci[*,*] = !values.d_nan ;4
		sum_defl_s_mult_pos_b_eng[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_b_sci	= dblarr(32,8,num_records5) ; 5
		s_no_e_b_eng	= dblarr(32,8,num_records1) ; 5
		s_no_e_b_sci[*,*,*]	= !values.d_nan
		s_no_e_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_b_sci = dblarr(num_records5,32) ;5
		sum_defl_s_no_e_b_eng = dblarr(num_records1,32) ;5
		sum_defl_s_no_e_b_sci[*,*] = !values.d_nan ;5
		sum_defl_s_no_e_b_eng[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_b_sci	= dblarr(32,8,num_records5) ; 6
		s_mult_e_b_eng	= dblarr(32,8,num_records1) ; 6
		s_mult_e_b_sci[*,*,*]	= !values.d_nan
		s_mult_e_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_b_sci = dblarr(num_records5,32) ;6
		sum_defl_s_mult_e_b_eng = dblarr(num_records1,32) ;6
		sum_defl_s_mult_e_b_sci[*,*] = !values.d_nan ;6
		sum_defl_s_mult_e_b_eng[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_b_sci	= dblarr(32,8,num_records5) ; 7
		ra_sat_a_b_eng	= dblarr(32,8,num_records1) ; 7
		ra_sat_a_b_sci[*,*,*]	= !values.d_nan
		ra_sat_a_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_b_sci = dblarr(num_records5,32) ;7
		sum_defl_ra_sat_a_b_eng = dblarr(num_records1,32) ;7
		sum_defl_ra_sat_a_b_sci[*,*] = !values.d_nan ;7
		sum_defl_ra_sat_a_b_eng[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_b_sci	= dblarr(32,8,num_records5) ; 8
		ra_sat_b_b_eng	= dblarr(32,8,num_records1) ; 8
		ra_sat_b_b_sci[*,*,*]= !values.d_nan
		ra_sat_b_b_eng[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_b_sci = dblarr(num_records5,32) ;8
		sum_defl_ra_sat_b_b_eng = dblarr(num_records1,32) ;8
		sum_defl_ra_sat_b_b_sci[*,*] = !values.d_nan ;8
		sum_defl_ra_sat_b_b_eng[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_b_sci	= dblarr(32,8,num_records5) ; 9
		ra_sat_both_b_eng	= dblarr(32,8,num_records1) ; 9
		ra_sat_both_b_sci[*,*,*]	= !values.d_nan
		ra_sat_both_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_b_sci = dblarr(num_records5,32) ;9
		sum_defl_ra_sat_both_b_eng = dblarr(num_records1,32) ;9
		sum_defl_ra_sat_both_b_sci[*,*] = !values.d_nan ;9
		sum_defl_ra_sat_both_b_eng[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_b_sci	= dblarr(32,8,num_records5) ; 10
		ssd_sw_b_eng	= dblarr(32,8,num_records1) ; 10
		ssd_sw_b_sci[*,*,*]	= !values.d_nan
		ssd_sw_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_b_sci = dblarr(num_records5,32) ;10
		sum_defl_ssd_sw_b_eng = dblarr(num_records1,32) ;10
		sum_defl_ssd_sw_b_sci[*,*] = !values.d_nan ;10
		sum_defl_ssd_sw_b_eng[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_b_sci	= dblarr(32,8,num_records5) ; 11
		ssd_st_b_eng	= dblarr(32,8,num_records1) ; 11
		ssd_st_b_sci[*,*,*]	= !values.d_nan
		ssd_st_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_b_sci = dblarr(num_records5,32) ;11
		sum_defl_ssd_st_b_eng = dblarr(num_records1,32) ;11
		sum_defl_ssd_st_b_sci[*,*] = !values.d_nan ;11
		sum_defl_ssd_st_b_eng[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_eng	= dblarr(32,8,num_records1) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sf0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_b_eng[*,*] = !values.d_nan ;12		
		END
	'sfr0':  BEGIN
		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sfr0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_b_eng[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_b_sci	= dblarr(32,8,num_records5) ; 14
		stp0_b_eng	= dblarr(32,8,num_records1) ; 14
		stp0_b_sci[*,*,*]	= !values.d_nan
		stp0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_b_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_b_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_b_eng[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_b_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_b_eng	= dblarr(32,8,num_records1) ; 15
		ra_trig_b_sci[*,*,*]	= !values.d_nan
		ra_trig_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_b_eng = dblarr(num_records1,32) ;15
		sum_defl_ra_trig_b_sci[*,*] = !values.d_nan ;15
		sum_defl_ra_trig_b_eng[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_b_sci	= dblarr(32,8,num_records5) ; 16
		pos1_0_b_eng	= dblarr(32,8,num_records1) ; 16
		pos1_0_b_sci[*,*,*]	= !values.d_nan
		pos1_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_b_sci = dblarr(num_records5,32) ;16
		sum_defl_pos1_0_b_eng = dblarr(num_records1,32) ;16
		sum_defl_pos1_0_b_sci[*,*] = !values.d_nan ;16
		sum_defl_pos1_0_b_eng[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_b_sci	= dblarr(32,8,num_records5) ; 17
		pos1_1_b_eng	= dblarr(32,8,num_records1) ; 17
		pos1_1_b_sci[*,*,*]	= !values.d_nan
		pos1_1_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_b_sci = dblarr(num_records5,32) ;17
		sum_defl_pos1_1_b_eng = dblarr(num_records1,32) ;17
		sum_defl_pos1_1_b_sci[*,*] = !values.d_nan ;17
		sum_defl_pos1_1_b_eng[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_b_sci	= dblarr(32,8,num_records5) ; 18
		w_no_pos_b_eng	= dblarr(32,8,num_records1) ; 18
		w_no_pos_b_sci[*,*,*]	= !values.d_nan
		w_no_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_b_sci = dblarr(num_records5,32) ;18
		sum_defl_w_no_pos_b_eng = dblarr(num_records1,32) ;18
		sum_defl_w_no_pos_b_sci[*,*] = !values.d_nan ;18
		sum_defl_w_no_pos_b_eng[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_b_sci	= dblarr(32,8,num_records5) ; 19
		w_mult_pos_b_eng	= dblarr(32,8,num_records1) ; 19
		w_mult_pos_b_sci[*,*,*]	= !values.d_nan
		w_mult_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_b_sci = dblarr(num_records5,32) ;19
		sum_defl_w_mult_pos_b_eng = dblarr(num_records1,32) ;19
		sum_defl_w_mult_pos_b_sci[*,*] = !values.d_nan ;19
		sum_defl_w_mult_pos_b_eng[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_b_sci	= dblarr(32,8,num_records5) ; 20
		w_valid_b_eng	= dblarr(32,8,num_records1) ; 20
		w_valid_b_sci[*,*,*]	= !values.d_nan
		w_valid_b_eng[*,*,*]	= !values.d_nan
		sum_defl_w_valid_b_sci = dblarr(num_records5,32) ;20
		sum_defl_w_valid_b_eng = dblarr(num_records1,32) ;20
		sum_defl_w_valid_b_sci[*,*] = !values.d_nan ;20
		sum_defl_w_valid_b_eng[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_b_sci	= dblarr(32,8,num_records5) ; 21
		sf2_b_eng	= dblarr(32,8,num_records1) ; 21
		sf2_b_sci[*,*,*]	= !values.d_nan
		sf2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_b_eng = dblarr(num_records1,32) ;21
		sum_defl_sf2_b_sci[*,*] = !values.d_nan ;21
		sum_defl_sf2_b_eng[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_b_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_b_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sfr2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_b_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_b_eng[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_b_sci	= dblarr(32,8,num_records5) ; 23
		stp2_b_eng	= dblarr(32,8,num_records1) ; 23
		stp2_b_sci[*,*,*]	= !values.d_nan	
		stp2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_b_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_b_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_b_eng[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_b_eng	= dblarr(32,8,num_records1) ; 24
		pos2_0_b_sci[*,*,*]	= !values.d_nan	
		pos2_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_b_eng = dblarr(num_records1,32) ;24
		sum_defl_pos2_0_b_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos2_0_b_eng[*,*] = !values.d_nan ;24
		END
	'pos2_1':  BEGIN
		pos2_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_b_eng	= dblarr(32,8,num_records1) ; 25
		pos2_1_b_sci[*,*,*]	= !values.d_nan	
		pos2_1_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_b_eng = dblarr(num_records1,32) ;25
		sum_defl_pos2_1_b_sci[*,*] = !values.d_nan ;25
		sum_defl_pos2_1_b_eng[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_b_eng	= dblarr(32,8,num_records1) ; 26
		pos2_2_b_sci[*,*,*]	= !values.d_nan
		pos2_2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_b_eng = dblarr(num_records1,32) ;26
		sum_defl_pos2_2_b_sci[*,*] = !values.d_nan ;26
		sum_defl_pos2_2_b_eng[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_b_eng	= dblarr(32,8,num_records1) ; 27
		pos2_3_b_sci[*,*,*]	= !values.d_nan	
		pos2_3_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_b_eng = dblarr(num_records1,32) ;27
		sum_defl_pos2_3_b_sci[*,*] = !values.d_nan ;27
		sum_defl_pos2_3_b_eng[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_b_sci	= dblarr(32,8,num_records5) ; 28
		pos3_0_b_eng	= dblarr(32,8,num_records1) ; 28
		pos3_0_b_sci[*,*,*]	= !values.d_nan	
		pos3_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_0_b_sci = dblarr(num_records5,32) ;28
		sum_defl_pos3_0_b_eng = dblarr(num_records1,32) ;28
		sum_defl_pos3_0_b_sci[*,*] = !values.d_nan ;28
		sum_defl_pos3_0_b_eng[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_b_sci	= dblarr(32,8,num_records5) ; 29
		pos3_1_b_eng	= dblarr(32,8,num_records1) ; 29
		pos3_1_b_sci[*,*,*]	= !values.d_nan	
		pos3_1_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_1_b_sci = dblarr(num_records5,32) ;29
		sum_defl_pos3_1_b_eng = dblarr(num_records1,32) ;29
		sum_defl_pos3_1_b_sci[*,*] = !values.d_nan ;29
		sum_defl_pos3_1_b_eng[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_b_sci	= dblarr(32,8,num_records5) ; 30
		pos3_2_b_eng	= dblarr(32,8,num_records1) ; 30
		pos3_2_b_sci[*,*,*]	= !values.d_nan
		pos3_2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_sci = dblarr(num_records5,32) ;30
		sum_defl_pos3_2_b_eng = dblarr(num_records1,32) ;30
		sum_defl_pos3_2_b_sci[*,*] = !values.d_nan ;30
		sum_defl_pos3_2_b_eng[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_b_sci	= dblarr(32,8,num_records5) ; 31
		pos3_3_b_eng	= dblarr(32,8,num_records1) ; 31
		pos3_3_b_sci[*,*,*]	= !values.d_nan
		pos3_3_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_b_sci = dblarr(num_records5,32) ;31
		sum_defl_pos3_3_b_eng = dblarr(num_records1,32) ;31
		sum_defl_pos3_3_b_sci[*,*] = !values.d_nan ;31
		sum_defl_pos3_3_b_eng[*,*] = !values.d_nan ;31
		END
	'stop0':  BEGIN
		stop0_b_sci	= dblarr(32,8,num_records5) 
		stop0_b_eng	= dblarr(32,8,num_records1) 
		stop0_b_sci[*,*,*]	= !values.d_nan
		stop0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stop0_b_sci = dblarr(num_records5,32) 
		sum_defl_stop0_b_eng = dblarr(num_records1,32) 
		sum_defl_stop0_b_sci[*,*] = !values.d_nan 
		sum_defl_stop0_b_eng[*,*] = !values.d_nan 

		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_eng	= dblarr(32,8,num_records1) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sf0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_b_eng[*,*] = !values.d_nan ;12	

		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sfr0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_b_eng[*,*] = !values.d_nan ;13
	

		END
	'start0':  BEGIN
		start0_b_sci	= dblarr(32,8,num_records5) 
		start0_b_eng	= dblarr(32,8,num_records1) 
		start0_b_sci[*,*,*]	= !values.d_nan
		start0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_start0_b_sci = dblarr(num_records5,32) 
		sum_defl_start0_b_eng = dblarr(num_records1,32)
		sum_defl_start0_b_sci[*,*] = !values.d_nan
		sum_defl_start0_b_eng[*,*] = !values.d_nan

		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sfr0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_b_eng[*,*] = !values.d_nan ;13

		stp0_b_sci	= dblarr(32,8,num_records5) ; 14
		stp0_b_eng	= dblarr(32,8,num_records1) ; 14
		stp0_b_sci[*,*,*]	= !values.d_nan
		stp0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_b_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_b_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_b_eng[*,*] = !values.d_nan ;14


		END
	'stop2':  BEGIN
		stop2_b_sci	= dblarr(32,8,num_records5)
		stop2_b_eng	= dblarr(32,8,num_records1)
		stop2_b_sci[*,*,*]	= !values.d_nan
		stop2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stop2_b_sci = dblarr(num_records5,32)
		sum_defl_stop2_b_eng = dblarr(num_records1,32)
		sum_defl_stop2_b_sci[*,*] = !values.d_nan
		sum_defl_stop2_b_eng[*,*] = !values.d_nan

		sf2_b_sci	= dblarr(32,8,num_records5) ; 12
		sf2_b_eng	= dblarr(32,8,num_records1) ; 12
		sf2_b_sci[*,*,*]	= !values.d_nan
		sf2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf2_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf2_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf2_b_eng[*,*] = !values.d_nan ;12		
	

		sfr2_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr2_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sfr2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr2_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr2_b_eng[*,*] = !values.d_nan ;13

		END
	'start2':  BEGIN
		start2_b_sci	= dblarr(32,8,num_records5)
		start2_b_eng	= dblarr(32,8,num_records1) 
		start2_b_sci[*,*,*]	= !values.d_nan
		start2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_start2_b_sci = dblarr(num_records5,32) 
		sum_defl_start2_b_eng = dblarr(num_records1,32) 
		sum_defl_start2_b_sci[*,*] = !values.d_nan 
		sum_defl_start2_b_eng[*,*] = !values.d_nan 

		sfr2_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr2_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sfr2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr2_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr2_b_eng[*,*] = !values.d_nan ;13

		stp2_b_sci	= dblarr(32,8,num_records5) ; 23
		stp2_b_eng	= dblarr(32,8,num_records1) ; 23
		stp2_b_sci[*,*,*]	= !values.d_nan	
		stp2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_b_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_b_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_b_eng[*,*] = !values.d_nan ;23

		END

	ENDCASE
	ENDFOR			
    END 
ENDCASE 
mon_initialized = 1

END 

PRO read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, prod_name, sci_values, eng_values, sum_sci, sum_eng
COMMON com_mon

;print,'read_one_mon called'

temp_name = prod_name+'_norm_science'

read_product_mon, cdf_id, temp_name, sci_values,  data5_index, epoch5maxrec
                                ;IF (FINITE(TOTAL(s_valid_a_sci),/NAN) NE 1) THEN BEGIN

kk =  where(sci_values GT 65535,  ckk)
IF ckk GT 0 THEN BEGIN
    sci_values(kk) =  65535
ENDIF

sum_sci=DBLARR(num_records5,32)
sum_sci[*, *] = !values.d_nan

sum_eng = DBLARR(num_records1,32)
sum_eng[*, *] = !values.d_nan

;print, 'read', temp_name
;stop

remove_fill_mon, -1, sci_values
temp = where(finite(sci_values) EQ 1,  count)

IF count GT 0 THEN BEGIN 
;    remove_fill_mon, -1, sci_values
    FOR j = 0, num_records5-1 DO BEGIN
        FOR k = 0, 31 DO BEGIN	
            sum_sci[j, k] = TOTAL(sci_values(k, *, j), /NAN)
        ENDFOR
    ENDFOR
ENDIF

;ENDIF ELSE BEGIN
;	FOR j = 0, num_records5-1 DO BEGIN
;        FOR k = 0, 31 DO BEGIN	
;            sum_sci[j, k] = !values.d_nan
;        ENDFOR
;    ENDFOR
;ENDELSE

temp_name = prod_name+'_norm_eng'
read_product_mon, cdf_id, temp_name, eng_values, data1_index, epoch1maxrec
                                ;IF (FINITE(TOTAL(s_valid_a_eng),/NAN) NE 1) THEN BEGIN
remove_fill_mon, -1, eng_values
temp = where(finite(eng_values) EQ 1,  count_2)

IF count_2 GT 0 THEN BEGIN 
;    remove_fill_mon, -1, eng_values
    FOR j = 0, num_records1-1 DO BEGIN
        FOR k = 0, 31 DO BEGIN	
            sum_eng[j, k] = TOTAL(eng_values(k, *, j), /NAN)
        ENDFOR
    ENDFOR
ENDIF	
;ENDIF ELSE BEGIN
;	FOR j = 0, num_records1-1 DO BEGIN
;        FOR k = 0, 31 DO BEGIN	
;            sum_eng[j, k] = !values.d_nan
;        ENDFOR
;    ENDFOR	
;ENDELSE

END 

PRO read_mon, cdf_id, sat, mom_read, data5_index, epoch5maxrec, data1_index, epoch1maxrec, products, num_records1, num_records5, E_step

COMMON com_mon
CASE sat OF 
    'A': BEGIN
	read_product_mon, cdf_id, 'error5_mon', error5_mon_a, data5_index, epoch5maxrec
	remove_fill_mon, -1, error5_mon_a
	IF (TOTAL(error5_mon_a, /NAN) GT 0) THEN print, 'ERROR5_MON FLAG SET'
	read_product_mon, cdf_id, 'error1', error1_a, data1_index, epoch1maxrec
	remove_fill_mon, -1, error1_a
	IF (TOTAL(error1_a, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'


        FOR ii = 0, n_elements(products)-1 DO BEGIN
            CASE products[ii] OF
                's_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_valid', s_valid_a_sci, s_valid_a_eng, sum_defl_s_valid_a_sci, sum_defl_s_valid_a_eng
                's_e_not_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_not_req', s_e_not_req_a_sci, s_e_not_req_a_eng, sum_defl_s_e_not_req_a_sci, sum_defl_s_e_not_req_a_eng	
		's_e_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_req', s_e_req_a_sci, s_e_req_a_eng, sum_defl_s_e_req_a_sci, sum_defl_s_e_req_a_eng	
		's_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_pos', s_no_pos_a_sci, s_no_pos_a_eng, sum_defl_s_no_pos_a_sci, sum_defl_s_no_pos_a_eng	
		's_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_pos', s_mult_pos_a_sci, s_mult_pos_a_eng, sum_defl_s_mult_pos_a_sci, sum_defl_s_mult_pos_a_eng	
		's_no_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_e', s_no_e_a_sci, s_no_e_a_eng, sum_defl_s_no_e_a_sci, sum_defl_s_no_e_a_eng	
		's_mult_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_e', s_mult_e_a_sci, s_mult_e_a_eng, sum_defl_s_mult_e_a_sci, sum_defl_s_mult_e_a_eng	
		'ra_sat_a': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_a', ra_sat_a_a_sci, ra_sat_a_a_eng, sum_defl_ra_sat_a_a_sci, sum_defl_ra_sat_a_a_eng	
		'ra_sat_b': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_b', ra_sat_b_a_sci, ra_sat_b_a_eng, sum_defl_ra_sat_b_a_sci, sum_defl_ra_sat_b_a_eng	
		'ra_sat_both': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_both', ra_sat_both_a_sci, ra_sat_both_a_eng, sum_defl_ra_sat_both_a_sci, sum_defl_ra_sat_both_a_eng	
		'ssd_sw': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_sw', ssd_sw_a_sci, ssd_sw_a_eng, sum_defl_ssd_sw_a_sci, sum_defl_ssd_sw_a_eng	
		'ssd_st': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_st', ssd_st_a_sci, ssd_st_a_eng, sum_defl_ssd_st_a_sci, sum_defl_ssd_st_a_eng	
 		'sf0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf0', sf0_a_sci, sf0_a_eng, sum_defl_sf0_a_sci, sum_defl_sf0_a_eng	
		'sfr0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr0', sfr0_a_sci, sfr0_a_eng, sum_defl_sfr0_a_sci, sum_defl_sfr0_a_eng	
		'stp0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp0', stp0_a_sci, stp0_a_eng, sum_defl_stp0_a_sci, sum_defl_stp0_a_eng	
		'ra_trig': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_trig', ra_trig_a_sci, ra_trig_a_eng, sum_defl_ra_trig_a_sci, sum_defl_ra_trig_a_eng	
		'pos1_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_0', pos1_0_a_sci, pos1_0_a_eng, sum_defl_pos1_0_a_sci, sum_defl_pos1_0_a_eng	
		'pos1_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_1', pos1_1_a_sci, pos1_1_a_eng, sum_defl_pos1_1_a_sci, sum_defl_pos1_1_a_eng	
		'w_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_no_pos', w_no_pos_a_sci, w_no_pos_a_eng, sum_defl_w_no_pos_a_sci, sum_defl_w_no_pos_a_eng	
		'w_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_mult_pos', w_mult_pos_a_sci, w_mult_pos_a_eng, sum_defl_w_mult_pos_a_sci, sum_defl_w_mult_pos_a_eng	
		'w_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_valid', w_valid_a_sci, w_valid_a_eng, sum_defl_w_valid_a_sci, sum_defl_w_valid_a_eng	
		'sf2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf2', sf2_a_sci, sf2_a_eng, sum_defl_sf2_a_sci, sum_defl_sf2_a_eng	
		'sfr2':	read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr2', sfr2_a_sci, sfr2_a_eng, sum_defl_sfr2_a_sci, sum_defl_sfr2_a_eng
		'stp2':	read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp2', stp2_a_sci, stp2_a_eng, sum_defl_stp2_a_sci, sum_defl_stp2_a_eng
		'pos2_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_0', pos2_0_a_sci, pos2_0_a_eng, sum_defl_pos2_0_a_sci, sum_defl_pos2_0_a_eng
		'pos2_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_1', pos2_1_a_sci, pos2_1_a_eng, sum_defl_pos2_1_a_sci, sum_defl_pos2_1_a_eng
		'pos2_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_2', pos2_2_a_sci, pos2_2_a_eng, sum_defl_pos2_2_a_sci, sum_defl_pos2_2_a_eng
		'pos2_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_3', pos2_3_a_sci, pos2_3_a_eng, sum_defl_pos2_3_a_sci, sum_defl_pos2_3_a_eng	
		'pos3_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_0', pos3_0_a_sci, pos3_0_a_eng, sum_defl_pos3_0_a_sci, sum_defl_pos3_0_a_eng	
		'pos3_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_1', pos3_1_a_sci, pos3_1_a_eng, sum_defl_pos3_1_a_sci, sum_defl_pos3_1_a_eng	
		'pos3_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_2', pos3_2_a_sci, pos3_2_a_eng, sum_defl_pos3_2_a_sci, sum_defl_pos3_2_a_eng	
		'pos3_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_3', pos3_3_a_sci, pos3_3_a_eng, sum_defl_pos3_3_a_sci, sum_defl_pos3_3_a_eng	
		'stop0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf0', sf0_a_sci, sf0_a_eng, sum_defl_sf0_a_sci, sum_defl_sf0_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_a_sci, sfr0_a_eng, sum_defl_sfr0_a_sci, sum_defl_sfr0_a_eng	
			
			FOR kk = 0D, n_elements(sf0_a_sci) - 1 DO BEGIN
			IF (sf0_a_sci[kk] NE 0) AND (FINITE(sf0_a_sci[kk]) EQ 1) AND (FINITE(sfr0_a_sci[kk]) EQ 1) THEN BEGIN
				stop0_a_sci[kk] = sfr0_a_sci[kk]/sf0_a_sci[kk]
			ENDIF ELSE BEGIN
				stop0_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf0_a_eng) - 1 DO BEGIN
			IF (sf0_a_eng[ll] NE 0) AND (FINITE(sf0_a_eng[ll]) EQ 1) AND (FINITE(sfr0_a_eng[ll]) EQ 1) THEN BEGIN
				stop0_a_eng[ll] = sfr0_a_eng[ll]/sf0_a_eng[ll]
			ENDIF ELSE BEGIN
				stop0_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_a_sci) - 1 DO BEGIN
			IF (sum_defl_sf0_a_sci[mm] NE 0) AND (FINITE(sum_defl_sf0_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_a_sci[mm] = sum_defl_sfr0_a_sci[mm]/sum_defl_sf0_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf0_a_eng) - 1 DO BEGIN
			IF (sum_defl_sf0_a_eng[nn] NE 0) AND (FINITE(sum_defl_sf0_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop0_a_eng[nn] = sum_defl_sfr0_a_eng[nn]/sum_defl_sf0_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'start0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp0', stp0_a_sci, stp0_a_eng, sum_defl_stp0_a_sci, sum_defl_stp0_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_a_sci, sfr0_a_eng, sum_defl_sfr0_a_sci, sum_defl_sfr0_a_eng	
			
			FOR kk = 0D, n_elements(stp0_a_sci) - 1 DO BEGIN
			IF (stp0_a_sci[kk] NE 0) AND (FINITE(stp0_a_sci[kk]) EQ 1) AND (FINITE(sfr0_a_sci[kk]) EQ 1) THEN BEGIN
				start0_a_sci[kk] = sfr0_a_sci[kk]/stp0_a_sci[kk]
			ENDIF ELSE BEGIN
				start0_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp0_a_eng) - 1 DO BEGIN
			IF (stp0_a_eng[ll] NE 0) AND (FINITE(stp0_a_eng[ll]) EQ 1) AND (FINITE(sfr0_a_eng[ll]) EQ 1) THEN BEGIN
				start0_a_eng[ll] = sfr0_a_eng[ll]/stp0_a_eng[ll]
			ENDIF ELSE BEGIN
				start0_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_a_sci) - 1 DO BEGIN
			IF (sum_defl_stp0_a_sci[mm] NE 0) AND (FINITE(sum_defl_stp0_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_a_sci[mm] = sum_defl_sfr0_a_sci[mm]/sum_defl_stp0_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp0_a_eng) - 1 DO BEGIN
			IF (sum_defl_stp0_a_eng[nn] NE 0) AND (FINITE(sum_defl_stp0_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start0_a_eng[nn] = sum_defl_sfr0_a_eng[nn]/sum_defl_stp0_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'stop2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf2', sf2_a_sci, sf2_a_eng, sum_defl_sf2_a_sci, sum_defl_sf2_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_a_sci, sfr2_a_eng, sum_defl_sfr2_a_sci, sum_defl_sfr2_a_eng	
			
			FOR kk = 0D, n_elements(sf2_a_sci) - 1 DO BEGIN
			IF (sf2_a_sci[kk] NE 0) AND (FINITE(sf2_a_sci[kk]) EQ 1) AND (FINITE(sfr2_a_sci[kk]) EQ 1) THEN BEGIN
				stop2_a_sci[kk] = sfr2_a_sci[kk]/sf2_a_sci[kk]
			ENDIF ELSE BEGIN
				stop2_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf2_a_eng) - 1 DO BEGIN
			IF (sf2_a_eng[ll] NE 0) AND (FINITE(sf2_a_eng[ll]) EQ 1) AND (FINITE(sfr2_a_eng[ll]) EQ 1) THEN BEGIN
				stop2_a_eng[ll] = sfr2_a_eng[ll]/sf2_a_eng[ll]
			ENDIF ELSE BEGIN
				stop2_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_a_sci) - 1 DO BEGIN
			IF (sum_defl_sf2_a_sci[mm] NE 0) AND (FINITE(sum_defl_sf2_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_a_sci[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_sf2_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf2_a_eng) - 1 DO BEGIN
			IF (sum_defl_sf2_a_eng[nn] NE 0) AND (FINITE(sum_defl_sf2_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop2_a_eng[nn] = sum_defl_sfr2_a_eng[nn]/sum_defl_sf2_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR

			END

		'start2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp2', stp2_a_sci, stp2_a_eng, sum_defl_stp2_a_sci, sum_defl_stp2_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_a_sci, sfr2_a_eng, sum_defl_sfr2_a_sci, sum_defl_sfr2_a_eng	
			
			FOR kk = 0D, n_elements(stp2_a_sci) - 1 DO BEGIN
			IF (stp2_a_sci[kk] NE 0) AND (FINITE(stp2_a_sci[kk]) EQ 1) AND (FINITE(sfr2_a_sci[kk]) EQ 1) THEN BEGIN
				start2_a_sci[kk] = sfr2_a_sci[kk]/stp2_a_sci[kk]
			ENDIF ELSE BEGIN
				start2_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp2_a_eng) - 1 DO BEGIN
			IF (stp2_a_eng[ll] NE 0) AND (FINITE(stp2_a_eng[ll]) EQ 1) AND (FINITE(sfr2_a_eng[ll]) EQ 1) THEN BEGIN
				start2_a_eng[ll] = sfr2_a_eng[ll]/stp2_a_eng[ll]
			ENDIF ELSE BEGIN
				start2_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_a_sci) - 1 DO BEGIN
			IF (sum_defl_stp2_a_sci[mm] NE 0) AND (FINITE(sum_defl_stp2_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_a_sci[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_stp2_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp2_a_eng) - 1 DO BEGIN
			IF (sum_defl_stp2_a_eng[nn] NE 0) AND (FINITE(sum_defl_stp2_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start2_a_eng[nn] = sum_defl_sfr2_a_eng[nn]/sum_defl_stp2_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END

            ENDCASE
        ENDFOR
    END
    'B': BEGIN

	read_product_mon, cdf_id, 'error5_mon', error5_mon_b, data5_index, epoch5maxrec
	remove_fill_mon, -1, error5_mon_b
	IF (TOTAL(error5_mon_b, /NAN) GT 0) THEN print, 'ERROR5_MON FLAG SET'
	read_product_mon, cdf_id, 'error1', error1_b, data1_index, epoch5maxrec
	remove_fill_mon, -1, error1_b
	IF (TOTAL(error1_b, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'

        FOR ii = 0, n_elements(products)-1 DO BEGIN 
            CASE products[ii] OF
                's_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_valid', s_valid_b_sci, s_valid_b_eng, sum_defl_s_valid_b_sci, sum_defl_s_valid_b_eng
		's_e_not_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_not_req', s_e_not_req_b_sci, s_e_not_req_b_eng, sum_defl_s_e_not_req_b_sci, sum_defl_s_e_not_req_b_eng	
		's_e_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_req', s_e_req_b_sci, s_e_req_b_eng, sum_defl_s_e_req_b_sci, sum_defl_s_e_req_b_eng	
		's_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_pos', s_no_pos_b_sci, s_no_pos_b_eng, sum_defl_s_no_pos_b_sci, sum_defl_s_no_pos_b_eng	
		's_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_pos', s_mult_pos_b_sci, s_mult_pos_b_eng, sum_defl_s_mult_pos_b_sci, sum_defl_s_mult_pos_b_eng	
		's_no_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_e', s_no_e_b_sci, s_no_e_b_eng, sum_defl_s_no_e_b_sci, sum_defl_s_no_e_b_eng	
		's_mult_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_e', s_mult_e_b_sci, s_mult_e_b_eng, sum_defl_s_mult_e_b_sci, sum_defl_s_mult_e_b_eng	
		'ra_sat_a': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_a', ra_sat_a_b_sci, ra_sat_a_b_eng, sum_defl_ra_sat_a_b_sci, sum_defl_ra_sat_a_b_eng	
 		'ra_sat_b': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_b', ra_sat_b_b_sci, ra_sat_b_b_eng, sum_defl_ra_sat_b_b_sci, sum_defl_ra_sat_b_b_eng	
		'ra_sat_both': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_both', ra_sat_both_b_sci, ra_sat_both_b_eng, sum_defl_ra_sat_both_b_sci, sum_defl_ra_sat_both_b_eng	
		'ssd_sw': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_sw', ssd_sw_b_sci, ssd_sw_b_eng, sum_defl_ssd_sw_b_sci, sum_defl_ssd_sw_b_eng	
		'ssd_st': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_st', ssd_st_b_sci, ssd_st_b_eng, sum_defl_ssd_st_b_sci, sum_defl_ssd_st_b_eng	
		'sf0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf0', sf0_b_sci, sf0_b_eng, sum_defl_sf0_b_sci, sum_defl_sf0_b_eng	
		'sfr0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr0', sfr0_b_sci, sfr0_b_eng, sum_defl_sfr0_b_sci, sum_defl_sfr0_b_eng	
		'stp0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp0', stp0_b_sci, stp0_b_eng, sum_defl_stp0_b_sci, sum_defl_stp0_b_eng	
 		'ra_trig': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_trig', ra_trig_b_sci, ra_trig_b_eng, sum_defl_ra_trig_b_sci, sum_defl_ra_trig_b_eng	
		'pos1_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_0', pos1_0_b_sci, pos1_0_b_eng, sum_defl_pos1_0_b_sci, sum_defl_pos1_0_b_eng	
		'pos1_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_1', pos1_1_b_sci, pos1_1_b_eng, sum_defl_pos1_1_b_sci, sum_defl_pos1_1_b_eng	
		'w_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_no_pos', w_no_pos_b_sci, w_no_pos_b_eng, sum_defl_w_no_pos_b_sci, sum_defl_w_no_pos_b_eng	
		'w_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_mult_pos', w_mult_pos_b_sci, w_mult_pos_b_eng, sum_defl_w_mult_pos_b_sci, sum_defl_w_mult_pos_b_eng	
		'w_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_valid', w_valid_b_sci, w_valid_b_eng, sum_defl_w_valid_b_sci, sum_defl_w_valid_b_eng	
		'sf2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf2', sf2_b_sci, sf2_b_eng, sum_defl_sf2_b_sci, sum_defl_sf2_b_eng	
		'sfr2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr2', sfr2_b_sci, sfr2_b_eng, sum_defl_sfr2_b_sci, sum_defl_sfr2_b_eng	
		'stp2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp2', stp2_b_sci, stp2_b_eng, sum_defl_stp2_b_sci, sum_defl_stp2_b_eng	
		'pos2_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_0', pos2_0_b_sci, pos2_0_b_eng, sum_defl_pos2_0_b_sci, sum_defl_pos2_0_b_eng	
		'pos2_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_1', pos2_1_b_sci, pos2_1_b_eng, sum_defl_pos2_1_b_sci, sum_defl_pos2_1_b_eng	
		'pos2_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_2', pos2_2_b_sci, pos2_2_b_eng, sum_defl_pos2_2_b_sci, sum_defl_pos2_2_b_eng	
		'pos2_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_3', pos2_3_b_sci, pos2_3_b_eng, sum_defl_pos2_3_b_sci, sum_defl_pos2_3_b_eng	
		'pos3_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_0', pos3_0_b_sci, pos3_0_b_eng, sum_defl_pos3_0_b_sci, sum_defl_pos3_0_b_eng	
		'pos3_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_1', pos3_1_b_sci, pos3_1_b_eng, sum_defl_pos3_1_b_sci, sum_defl_pos3_1_b_eng	
		'pos3_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_2', pos3_2_b_sci, pos3_2_b_eng, sum_defl_pos3_2_b_sci, sum_defl_pos3_2_b_eng	
		'pos3_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_3', pos3_3_b_sci, pos3_3_b_eng, sum_defl_pos3_3_b_sci, sum_defl_pos3_3_b_eng	
	'stop0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf0', sf0_b_sci, sf0_b_eng, sum_defl_sf0_b_sci, sum_defl_sf0_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_b_sci, sfr0_b_eng, sum_defl_sfr0_b_sci, sum_defl_sfr0_b_eng	
			
			FOR kk = 0D, n_elements(sf0_b_sci) - 1 DO BEGIN
			IF (sf0_b_sci[kk] NE 0) AND (FINITE(sf0_b_sci[kk]) EQ 1) AND (FINITE(sfr0_b_sci[kk]) EQ 1) THEN BEGIN
				stop0_b_sci[kk] = sfr0_b_sci[kk]/sf0_b_sci[kk]
			ENDIF ELSE BEGIN
				stop0_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf0_b_eng) - 1 DO BEGIN
			IF (sf0_b_eng[ll] NE 0) AND (FINITE(sf0_b_eng[ll]) EQ 1) AND (FINITE(sfr0_b_eng[ll]) EQ 1) THEN BEGIN
				stop0_b_eng[ll] = sfr0_b_eng[ll]/sf0_b_eng[ll]
			ENDIF ELSE BEGIN
				stop0_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_b_sci) - 1 DO BEGIN
			IF (sum_defl_sf0_b_sci[mm] NE 0) AND (FINITE(sum_defl_sf0_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_b_sci[mm] = sum_defl_sfr0_b_sci[mm]/sum_defl_sf0_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf0_b_eng) - 1 DO BEGIN
			IF (sum_defl_sf0_b_eng[nn] NE 0) AND (FINITE(sum_defl_sf0_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop0_b_eng[nn] = sum_defl_sfr0_b_eng[nn]/sum_defl_sf0_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR

			END
		'start0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp0', stp0_b_sci, stp0_b_eng, sum_defl_stp0_b_sci, sum_defl_stp0_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_b_sci, sfr0_b_eng, sum_defl_sfr0_b_sci, sum_defl_sfr0_b_eng	
			
			FOR kk = 0D, n_elements(stp0_b_sci) - 1 DO BEGIN
			IF (stp0_b_sci[kk] NE 0) AND (FINITE(stp0_b_sci[kk]) EQ 1) AND (FINITE(sfr0_b_sci[kk]) EQ 1) THEN BEGIN
				start0_b_sci[kk] = sfr0_b_sci[kk]/stp0_b_sci[kk]
			ENDIF ELSE BEGIN
				start0_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp0_b_eng) - 1 DO BEGIN
			IF (stp0_b_eng[ll] NE 0) AND (FINITE(stp0_b_eng[ll]) EQ 1) AND (FINITE(sfr0_b_eng[ll]) EQ 1) THEN BEGIN
				start0_b_eng[ll] = sfr0_b_eng[ll]/stp0_b_eng[ll]
			ENDIF ELSE BEGIN
				start0_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_b_sci) - 1 DO BEGIN
			IF (sum_defl_stp0_b_sci[mm] NE 0) AND (FINITE(sum_defl_stp0_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_b_sci[mm] = sum_defl_sfr0_b_sci[mm]/sum_defl_stp0_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp0_b_eng) - 1 DO BEGIN
			IF (sum_defl_stp0_b_eng[nn] NE 0) AND (FINITE(sum_defl_stp0_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start0_b_eng[nn] = sum_defl_sfr0_b_eng[nn]/sum_defl_stp0_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'stop2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf2', sf2_b_sci, sf2_b_eng, sum_defl_sf2_b_sci, sum_defl_sf2_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_b_sci, sfr2_b_eng, sum_defl_sfr2_b_sci, sum_defl_sfr2_b_eng	
			
			FOR kk = 0D, n_elements(sf2_b_sci) - 1 DO BEGIN
			IF (sf2_b_sci[kk] NE 0) AND (FINITE(sf2_b_sci[kk]) EQ 1) AND (FINITE(sfr2_b_sci[kk]) EQ 1) THEN BEGIN
				stop2_b_sci[kk] = sfr2_b_sci[kk]/sf2_b_sci[kk]
			ENDIF ELSE BEGIN
				stop2_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf2_b_eng) - 1 DO BEGIN
			IF (sf2_b_eng[ll] NE 0) AND (FINITE(sf2_b_eng[ll]) EQ 1) AND (FINITE(sfr2_b_eng[ll]) EQ 1) THEN BEGIN
				stop2_b_eng[ll] = sfr2_b_eng[ll]/sf2_b_eng[ll]
			ENDIF ELSE BEGIN
				stop2_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_b_sci) - 1 DO BEGIN
			IF (sum_defl_sf2_b_sci[mm] NE 0) AND (FINITE(sum_defl_sf2_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_b_sci[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_sf2_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf2_b_eng) - 1 DO BEGIN
			IF (sum_defl_sf2_b_eng[nn] NE 0) AND (FINITE(sum_defl_sf2_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop2_b_eng[nn] = sum_defl_sfr2_b_eng[nn]/sum_defl_sf2_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR

			END

		'start2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp2', stp2_b_sci, stp2_b_eng, sum_defl_stp2_b_sci, sum_defl_stp2_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_b_sci, sfr2_b_eng, sum_defl_sfr2_b_sci, sum_defl_sfr2_b_eng	
			
			FOR kk = 0D, n_elements(stp2_b_sci) - 1 DO BEGIN
			IF (stp2_b_sci[kk] NE 0) AND (FINITE(stp2_b_sci[kk]) EQ 1) AND (FINITE(sfr2_b_sci[kk]) EQ 1) THEN BEGIN
				start2_b_sci[kk] = sfr2_b_sci[kk]/stp2_b_sci[kk]
			ENDIF ELSE BEGIN
				start2_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp2_b_eng) - 1 DO BEGIN
			IF (stp2_b_eng[ll] NE 0) AND (FINITE(stp2_b_eng[ll]) EQ 1) AND (FINITE(sfr2_b_eng[ll]) EQ 1) THEN BEGIN
				start2_b_eng[ll] = sfr2_b_eng[ll]/stp2_b_eng[ll]
			ENDIF ELSE BEGIN
				start2_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_b_sci) - 1 DO BEGIN
			IF (sum_defl_stp2_b_sci[mm] NE 0) AND (FINITE(sum_defl_stp2_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_b_sci[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_stp2_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp2_b_eng) - 1 DO BEGIN
			IF (sum_defl_stp2_b_eng[nn] NE 0) AND (FINITE(sum_defl_stp2_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start2_b_eng[nn] = sum_defl_sfr2_b_eng[nn]/sum_defl_stp2_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END

                ENDCASE
            ENDFOR
        END
    ENDCASE
    mon_read = 1
END 

PRO check_indices_mon, index, epoch, resolution, file_num

; checks for missing records from the cdf epoch variable
; index: array of pairs [start_index,num_records] (could be [0,10,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1,5, or 3 depending on epoch (may need to add variable)
; file_num: index of this file in the file array

first_index=0L
first_index = file_num*(1440/resolution)

num_records = 0L
in_valid_area = 0
ii = 0L
index = [-1L]
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 59 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*60)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]                
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    5: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 11 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF (hour EQ time.hour AND time.min GE (min*5) AND time.min LT (min*5+5)) OR $
                      (time.hour EQ hour+1 AND time.min LE 4 AND min EQ 11) THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*12)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf_mon, sat, files, products, which_epoch, E_step

COMMON share1_mon
COMMON com_mon
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat

; initialize output variables (from common block)
need_epoch1 = 0
need_epoch5 = 0

FOR ii = 0L, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
        1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
	    num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 5

        END 
        5: BEGIN 
            num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 5
        END 
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 
mon_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	's_valid' : ;0
	's_e_not_req': ;1
	's_e_req': ;2
	's_no_pos': ;3
	's_mult_pos': ;4
	's_no_e': ;5
	's_mult_e': ;6
	'ra_sat_a': ;7
	'ra_sat_b': ;8
	'ra_sat_both': ;9
	'ssd_sw': ;10
	'ssd_st': ;11
	'sf0': ;12
	'sfr0': ;13
	'stp0': ;14
	'ra_trig': ;15
	'pos1_0': ;16
	'pos1_1': ;17
	'w_no_pos': ;18
	'w_mult_pos': ;19
	'w_valid': ;20
	'sf2': ;21
	'sfr2': ;22
	'stp2': ;23
	'pos2_0': ;24
	'pos2_1': ;25
	'pos2_2': ;26
	'pos2_3': ;27
	'pos3_0': ;28
	'pos3_1': ;29
	'pos3_2': ;30
	'pos3_3': ;31
	'stop0':
	'start0':
	'stop2':
	'start2': IF mon_initialized EQ 0 THEN init_mon, sat, num_records1, num_records5, mom_initialized, E_step, products
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
                    IF epoch1info.maxrec GT 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch1', temp_epoch, rec_count = epoch1info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_mon, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data1_index[kk+1]-1)]
                            converted_index = converted_index+data1_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF ELSE files[ii] = ''
		remove_fill_mon, -1, epoch1
		cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5_mon', /zvariable
                    IF epoch5info.maxrec GT 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch5_mon', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
			converted_epoch5 = time_double(temp_epoch, /epoch)
                        check_indices_mon, data5_index, converted_epoch5, 5, ii
                        converted_index5 = 0
                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
                              converted_epoch5[converted_index5:(converted_index5+data5_index[kk+1]-1)]
                            converted_index5 = converted_index5+data5_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF ELSE files[ii] = ''
               END 
                5: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5_mon', /zvariable
                    IF epoch5info.maxrec GT 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch5_mon', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_mon, data5_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data5_index[kk+1]-1)]
                            converted_index = converted_index+data5_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF ELSE files[ii] = ''
                END 
                0: BEGIN 
                    print, "Lorna: add variable epoch"
                    stop
                END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_a = epoch5
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
        ENDELSE 

        ; get other data
        mon_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		's_valid' : ;0
		's_e_not_req': ;1
		's_e_req': ;2
		's_no_pos': ;3
		's_mult_pos': ;4
		's_no_e': ;5
		's_mult_e': ;6
		'ra_sat_a': ;7
		'ra_sat_b': ;8
		'ra_sat_both': ;9
		'ssd_sw': ;10
		'ssd_st': ;11
		'sf0': ;12
		'sfr0': ;13
		'stp0': ;14
		'ra_trig': ;15
		'pos1_0': ;16
		'pos1_1': ;17
		'w_no_pos': ;18
		'w_mult_pos': ;19
		'w_valid': ;20
		'sf2': ;21
		'sfr2': ;22
		'stp2': ;23
		'pos2_0': ;24
		'pos2_1': ;25
		'pos2_2': ;26
		'pos2_3': ;27
		'pos3_0': ;28
		'pos3_1': ;29
		'pos3_2': ;30
		'pos3_3': ;31
		'stop0':
		'start0':
		'stop2':
		'start2': $
	IF mon_read EQ 0 THEN read_mon, cdf_id, sat, mon_read, data5_index, epoch5info.maxrec, data1_index, epoch1info.maxrec+1, products(jj), num_records1, num_records5, E_step
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
    IF need_epoch5 EQ 5 THEN data5_index_a = data5_index
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
    IF need_epoch5 EQ 5 THEN data5_index_b = data5_index
ENDELSE 

END 
