;+
;PROCEDURE: pl_read_cdf_hk_1min
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch: 0(for variable), 1, or 5 minute resolution, 2 (for analog housekeeping, 
;				which can be reported every 2 seconds)
;
;CREATED BY: L. Ellis (pl_read_cdf)
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;
;	November 2006 K.Simunac adds housekeeping variables
;	12 January, 2007:  variable and program names modified to avoid confusion
;		read_product -> read_product_hk
;		remove_fill -> remove_fill_hk
;		init_mom -> init_hk
;		moments_initialized -> hk_initialized
;		mom_read -> hk_read
;		check_indices -> check_indices_hk
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;-

PRO read_product_hk, cdf_id, prod_name, data_var, data_index, epoch_maxrec

; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 

cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
IF varinfo.maxrec+1 GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec
cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
num_dim = size(data_var, /n_dimensions)
CASE num_dim OF 
    1: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    2: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    3: BEGIN
        temp_index = 0L
        FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
            data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, *, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    ELSE: BEGIN 
        print, "Lorna: Add more dimensions"
        stop
    END
ENDCASE 
END 

PRO remove_fill_hk, fill_val, data

; fill_val: fill_value to be replaced by NaN
; data: array of data values
indice = where(data EQ fill_val, count)
FOR ii = 0L, count-1 DO BEGIN
    data[indice[ii]] = !values.d_nan
ENDFOR 

END 


PRO init_hk, sat, num_records1, num_records2, hk_initialized

; initialize all the housekeeping variabales (in com_hk)
COMMON com_hk
CASE sat OF
    'A': BEGIN 
	lrnm_spare_a	= dblarr(num_records1) ; 0
	lvc_neg12v_a	= dblarr(num_records1) ; 1
	lvc_neg5v_a	= dblarr(num_records1) ; 2
	esa_vm_pos_a	= dblarr(num_records1) ; 3
	dfl_1_vm_a	= dblarr(num_records1) ; 4
	dfl_1_tap_a	= dblarr(num_records1) ; 5
	dfl_2_vm_a	= dblarr(num_records1) ; 6
	dfl_2_tap_a	= dblarr(num_records1) ; 7
	lvc_pos2point5v_b_a	= dblarr(num_records1) ; 8
	pac_cm_dc_a	= dblarr(num_records1) ; 9
	lvc_pos12v_a	= dblarr(num_records1) ; 10
	lrpm_spare_a	= dblarr(num_records1) ; 11
	lvc_pos2point5v_a_a	= dblarr(num_records1) ; 12
	lvc_pos5v_a	= dblarr(num_records1) ; 13
	adc_avdd_a	= dblarr(num_records1) ; 14
	adc_dvdd_a	= dblarr(num_records1) ; 15
	pac_vm_a	= dblarr(num_records1) ; 16
	cal_vref_a	= dblarr(num_records1) ; 17
	pac_cm_ac_a	= dblarr(num_records1) ; 18
	mcp_vm_a	= dblarr(num_records1) ; 19
	mcp_cm_dc_a	= dblarr(num_records1) ; 20
	mcp_cm_ac_a	= dblarr(num_records1) ; 21
	ssd_cm_dc_a	= dblarr(num_records1) ; 22
	s_ch_vm_a	= dblarr(num_records1) ; 23
	s_ch_vm_tap_a	= dblarr(num_records1) ; 24
	s_ch_cm_dc_a	= dblarr(num_records1) ; 25
	esa_vm_neg_a	= dblarr(num_records1) ; 26
	esa_cm_dc_a	= dblarr(num_records1) ; 27
	dfl_1_cm_dc_a	= dblarr(num_records1) ; 28
	dfl_2_cm_dc_a	= dblarr(num_records1) ; 29
	lvc_pos2point5v_b_imon_a	= dblarr(num_records1) ; 30	
	lvc_pos12v_imon_a	= dblarr(num_records1) ; 31
	lvc_neg12v_imon_a	= dblarr(num_records1) ; 32
	lvc_pos5v_imon_a	= dblarr(num_records1) ; 33
	lvc_neg5v_imon_a	= dblarr(num_records1) ; 34
	lvc_pos2point5v_a_imon_a	= dblarr(num_records1) ; 35
	ssd_vm_a	= dblarr(num_records1) ; 36
	adc_lu_flag_a	= dblarr(num_records1) ; 37
	plug_id_a	= dblarr(num_records1) ; 38
	adc_lu_ctr_a	= dblarr(num_records1) ; 39
	dac_status_a	= dblarr(num_records1) ; 40
	adc0_agnd_a	= dblarr(num_records1) ; 41
	adc1_agnd_a	= dblarr(num_records1) ; 42
	ra_e0_a	= dblarr(num_records1) ; 43
	ra_e1_a	= dblarr(num_records1) ; 44
	tac0_tsp_a	= dblarr(num_records1) ; 45
	tac2_tsp_a	= dblarr(num_records1) ; 46
	sb0_tsp_a	= dblarr(num_records1) ; 47
	sb1_tsp_a	= dblarr(num_records1) ; 48
	tof_hv0_tsp_a	= dblarr(num_records1) ; 49
	tof_hv1_tsp_a	= dblarr(num_records1) ; 50
	s_c_0_tsp_a	= dblarr(num_records1) ; 51
	s_c_1_tsp_a	= dblarr(num_records1) ; 52
	lvc0_tsp_a	= dblarr(num_records1) ; 53
	lvc1_tsp_a	= dblarr(num_records1) ; 54
	adc0_vref_a	= dblarr(num_records1) ; 55
	adc1_vref_a	= dblarr(num_records1) ; 56
	ssd_status_a	= dblarr(num_records1) ; 57
	ssd_v_pos_analog_a	= dblarr(num_records1) ; 58
	ssd_v_neg_analog_a	= dblarr(num_records1) ; 59
	ssd_hv_bias_a	= dblarr(num_records1) ; 60
	ssd_tc0_a	= dblarr(num_records1) ; 61
	ssd_tc1_a	= dblarr(num_records1) ; 62
	ssd_tc2_a	= dblarr(num_records1) ; 63
	ssd_v_pos_dig_a	= dblarr(num_records1) ; 64

	lrnm_spare_a[*] = !values.d_nan ; 0
	lvc_neg12v_a[*]	= !values.d_nan ; 1
	lvc_neg5v_a[*]	= !values.d_nan ; 2
	esa_vm_pos_a[*]	= !values.d_nan ; 3
	dfl_1_vm_a[*]	= !values.d_nan ; 4
	dfl_1_tap_a[*]	= !values.d_nan ; 5
	dfl_2_vm_a[*]	= !values.d_nan ; 6
	dfl_2_tap_a[*]	= !values.d_nan ; 7
	lvc_pos2point5v_b_a[*]= !values.d_nan ; 8
	pac_cm_dc_a[*]	= !values.d_nan ; 9
	lvc_pos12v_a[*]	= !values.d_nan ; 10
	lrpm_spare_a[*]	= !values.d_nan ; 11
	lvc_pos2point5v_a_a[*]	= !values.d_nan ; 12
	lvc_pos5v_a[*]	= !values.d_nan ; 13
	adc_avdd_a[*]	= !values.d_nan ; 14
	adc_dvdd_a[*]	= !values.d_nan ; 15
	pac_vm_a[*]	= !values.d_nan ; 16
	cal_vref_a[*]	= !values.d_nan ; 17
	pac_cm_ac_a[*]	= !values.d_nan ; 18
	mcp_vm_a[*]	= !values.d_nan ; 19
	mcp_cm_dc_a[*]	= !values.d_nan ; 20
	mcp_cm_ac_a[*]	= !values.d_nan ; 21
	ssd_cm_dc_a[*]	= !values.d_nan ; 22
	s_ch_vm_a[*]	= !values.d_nan	; 23
	s_ch_vm_tap_a[*]	= !values.d_nan ; 24	
	s_ch_cm_dc_a[*]	= !values.d_nan	; 25
	esa_vm_neg_a[*]	= !values.d_nan	; 26
	esa_cm_dc_a[*]	= !values.d_nan	; 27
	dfl_1_cm_dc_a[*]	= !values.d_nan ; 28	
	dfl_2_cm_dc_a[*]	= !values.d_nan	; 29
	lvc_pos2point5v_b_imon_a[*]	= !values.d_nan ; 30	
	lvc_pos12v_imon_a[*]	= !values.d_nan	; 31
	lvc_neg12v_imon_a[*]	= !values.d_nan ; 32
	lvc_pos5v_imon_a[*]	= !values.d_nan ; 33
	lvc_neg5v_imon_a[*]	= !values.d_nan ; 34
	lvc_pos2point5v_a_imon_a[*]	= !values.d_nan ; 35
	ssd_vm_a[*]	= !values.d_nan ; 36
	adc_lu_flag_a[*]	= !values.d_nan ; 37
	plug_id_a[*]	= !values.d_nan	; 38
	adc_lu_ctr_a[*]	= !values.d_nan	; 39
	dac_status_a[*]	= !values.d_nan ; 40
	adc0_agnd_a[*]	= !values.d_nan	; 41
	adc1_agnd_a[*]	= !values.d_nan	; 42
	ra_e0_a[*]	= !values.d_nan ; 43
	ra_e1_a[*]	= !values.d_nan	; 44
	tac0_tsp_a[*]	= !values.d_nan	; 45
	tac2_tsp_a[*]	= !values.d_nan	; 46
	sb0_tsp_a[*]	= !values.d_nan	; 47
	sb1_tsp_a[*]	= !values.d_nan ; 48
	tof_hv0_tsp_a[*]	= !values.d_nan ; 49
	tof_hv1_tsp_a[*]	= !values.d_nan ; 50
	s_c_0_tsp_a[*]	= !values.d_nan ; 51
	s_c_1_tsp_a[*]	= !values.d_nan	; 52
	lvc0_tsp_a[*]	= !values.d_nan ; 53
	lvc1_tsp_a[*]	= !values.d_nan ; 54
	adc0_vref_a[*]	= !values.d_nan ; 55
	adc1_vref_a[*]	= !values.d_nan ; 56
	ssd_status_a[*]	= !values.d_nan ; 57
	ssd_v_pos_analog_a[*]	= !values.d_nan ; 58
	ssd_v_neg_analog_a[*]	= !values.d_nan ; 59
	ssd_hv_bias_a[*]	= !values.d_nan ; 60
	ssd_tc0_a[*]	= !values.d_nan ; 61
	ssd_tc1_a[*]	= !values.d_nan ; 62
	ssd_tc2_a[*]	= !values.d_nan ; 63
	ssd_v_pos_dig_a[*]	= !values.d_nan ; 64


    END 
    'B': BEGIN 
	lrnm_spare_b	= dblarr(num_records1) ; 0
	lvc_neg12v_b	= dblarr(num_records1) ; 1
	lvc_neg5v_b	= dblarr(num_records1) ; 2
	esa_vm_pos_b	= dblarr(num_records1) ; 3
	dfl_1_vm_b	= dblarr(num_records1) ; 4
	dfl_1_tap_b	= dblarr(num_records1) ; 5
	dfl_2_vm_b	= dblarr(num_records1) ; 6
	dfl_2_tap_b	= dblarr(num_records1) ; 7
	lvc_pos2point5v_b_b	= dblarr(num_records1) ; 8
	pac_cm_dc_b	= dblarr(num_records1) ; 9
	lvc_pos12v_b	= dblarr(num_records1) ; 10
	lrpm_spare_b	= dblarr(num_records1) ; 11
	lvc_pos2point5v_a_b	= dblarr(num_records1) ; 12
	lvc_pos5v_b	= dblarr(num_records1) ; 13
	adc_avdd_b	= dblarr(num_records1) ; 14
	adc_dvdd_b	= dblarr(num_records1) ; 15
	pac_vm_b	= dblarr(num_records1) ; 16
	cal_vref_b	= dblarr(num_records1) ; 17
	pac_cm_ac_b	= dblarr(num_records1) ; 18
	mcp_vm_b	= dblarr(num_records1) ; 19
	mcp_cm_dc_b	= dblarr(num_records1) ; 20
	mcp_cm_ac_b	= dblarr(num_records1) ; 21
	ssd_cm_dc_b	= dblarr(num_records1) ; 22
	s_ch_vm_b	= dblarr(num_records1) ; 23
	s_ch_vm_tap_b	= dblarr(num_records1) ; 24
	s_ch_cm_dc_b	= dblarr(num_records1) ; 25
	esa_vm_neg_b	= dblarr(num_records1) ; 26
	esa_cm_dc_b	= dblarr(num_records1) ; 27
	dfl_1_cm_dc_b	= dblarr(num_records1) ; 28
	dfl_2_cm_dc_b	= dblarr(num_records1) ; 29
	lvc_pos2point5v_b_imon_b	= dblarr(num_records1) ; 30	
	lvc_pos12v_imon_b	= dblarr(num_records1) ; 31
	lvc_neg12v_imon_b	= dblarr(num_records1) ; 32
	lvc_pos5v_imon_b	= dblarr(num_records1) ; 33
	lvc_neg5v_imon_b	= dblarr(num_records1) ; 34
	lvc_pos2point5v_a_imon_b	= dblarr(num_records1) ; 35
	ssd_vm_b	= dblarr(num_records1) ; 36
	adc_lu_flag_b	= dblarr(num_records1) ; 37
	plug_id_b	= dblarr(num_records1) ; 38
	adc_lu_ctr_b	= dblarr(num_records1) ; 39
	dac_status_b	= dblarr(num_records1) ; 40
	adc0_agnd_b	= dblarr(num_records1) ; 41
	adc1_agnd_b	= dblarr(num_records1) ; 42
	ra_e0_b	= dblarr(num_records1) ; 43
	ra_e1_b	= dblarr(num_records1) ; 44
	tac0_tsp_b	= dblarr(num_records1) ; 45
	tac2_tsp_b	= dblarr(num_records1) ; 46
	sb0_tsp_b	= dblarr(num_records1) ; 47
	sb1_tsp_b	= dblarr(num_records1) ; 48
	tof_hv0_tsp_b	= dblarr(num_records1) ; 49
	tof_hv1_tsp_b	= dblarr(num_records1) ; 50
	s_c_0_tsp_b	= dblarr(num_records1) ; 51
	s_c_1_tsp_b	= dblarr(num_records1) ; 52
	lvc0_tsp_b	= dblarr(num_records1) ; 53
	lvc1_tsp_b	= dblarr(num_records1) ; 54
	adc0_vref_b	= dblarr(num_records1) ; 55
	adc1_vref_b	= dblarr(num_records1) ; 56
	ssd_status_b	= dblarr(num_records1) ; 57
	ssd_v_pos_analog_b	= dblarr(num_records1) ; 58
	ssd_v_neg_analog_b	= dblarr(num_records1) ; 59
	ssd_hv_bias_b	= dblarr(num_records1) ; 60
	ssd_tc0_b	= dblarr(num_records1) ; 61
	ssd_tc1_b	= dblarr(num_records1) ; 62
	ssd_tc2_b	= dblarr(num_records1) ; 63
	ssd_v_pos_dig_b	= dblarr(num_records1) ; 64

	lrnm_spare_b[*] = !values.d_nan ; 0
	lvc_neg12v_b[*]	= !values.d_nan ; 1
	lvc_neg5v_b[*]	= !values.d_nan ; 2
	esa_vm_pos_b[*]	= !values.d_nan ; 3
	dfl_1_vm_b[*]	= !values.d_nan ; 4
	dfl_1_tap_b[*]	= !values.d_nan ; 5
	dfl_2_vm_b[*]	= !values.d_nan ; 6
	dfl_2_tap_b[*]	= !values.d_nan ; 7
	lvc_pos2point5v_b_b[*]= !values.d_nan ; 8
	pac_cm_dc_b[*]	= !values.d_nan ; 9
	lvc_pos12v_b[*]	= !values.d_nan ; 10
	lrpm_spare_b[*]	= !values.d_nan ; 11
	lvc_pos2point5v_a_b[*]	= !values.d_nan ; 12
	lvc_pos5v_b[*]	= !values.d_nan ; 13
	adc_avdd_b[*]	= !values.d_nan ; 14
	adc_dvdd_b[*]	= !values.d_nan ; 15
	pac_vm_b[*]	= !values.d_nan ; 16
	cal_vref_b[*]	= !values.d_nan ; 17
	pac_cm_ac_b[*]	= !values.d_nan ; 18
	mcp_vm_b[*]	= !values.d_nan ; 19
	mcp_cm_dc_b[*]	= !values.d_nan ; 20
	mcp_cm_ac_b[*]	= !values.d_nan ; 21
	ssd_cm_dc_b[*]	= !values.d_nan ; 22
	s_ch_vm_b[*]	= !values.d_nan	; 23
	s_ch_vm_tap_b[*]	= !values.d_nan ; 24	
	s_ch_cm_dc_b[*]	= !values.d_nan	; 25
	esa_vm_neg_b[*]	= !values.d_nan	; 26
	esa_cm_dc_b[*]	= !values.d_nan	; 27
	dfl_1_cm_dc_b[*]	= !values.d_nan ; 28	
	dfl_2_cm_dc_b[*]	= !values.d_nan	; 29
	lvc_pos2point5v_b_imon_b[*]	= !values.d_nan ; 30	
	lvc_pos12v_imon_b[*]	= !values.d_nan	; 31
	lvc_neg12v_imon_b[*]	= !values.d_nan ; 32
	lvc_pos5v_imon_b[*]	= !values.d_nan ; 33
	lvc_neg5v_imon_b[*]	= !values.d_nan ; 34
	lvc_pos2point5v_a_imon_b[*]	= !values.d_nan ; 35
	ssd_vm_b[*]	= !values.d_nan ; 36
	adc_lu_flag_b[*]	= !values.d_nan ; 37
	plug_id_b[*]	= !values.d_nan	; 38
	adc_lu_ctr_b[*]	= !values.d_nan	; 39
	dac_status_b[*]	= !values.d_nan ; 40
	adc0_agnd_b[*]	= !values.d_nan	; 41
	adc1_agnd_b[*]	= !values.d_nan	; 42
	ra_e0_b[*]	= !values.d_nan ; 43
	ra_e1_b[*]	= !values.d_nan	; 44
	tac0_tsp_b[*]	= !values.d_nan	; 45
	tac2_tsp_b[*]	= !values.d_nan	; 46
	sb0_tsp_b[*]	= !values.d_nan	; 47
	sb1_tsp_b[*]	= !values.d_nan ; 48
	tof_hv0_tsp_b[*]	= !values.d_nan ; 49
	tof_hv1_tsp_b[*]	= !values.d_nan ; 50
	s_c_0_tsp_b[*]	= !values.d_nan ; 51
	s_c_1_tsp_b[*]	= !values.d_nan	; 52
	lvc0_tsp_b[*]	= !values.d_nan ; 53
	lvc1_tsp_b[*]	= !values.d_nan ; 54
	adc0_vref_b[*]	= !values.d_nan ; 55
	adc1_vref_b[*]	= !values.d_nan ; 56
	ssd_status_b[*]	= !values.d_nan ; 57
	ssd_v_pos_analog_b[*]	= !values.d_nan ; 58
	ssd_v_neg_analog_b[*]	= !values.d_nan ; 59
	ssd_hv_bias_b[*]	= !values.d_nan ; 60
	ssd_tc0_b[*]	= !values.d_nan ; 61
	ssd_tc1_b[*]	= !values.d_nan ; 62
	ssd_tc2_b[*]	= !values.d_nan ; 63
	ssd_v_pos_dig_b[*]	= !values.d_nan ;64
        
    END 
ENDCASE 
hk_initialized = 1

END 

PRO read_hk, cdf_id, sat, hk_read, data1_index, epoch1maxrec, products

COMMON com_hk
CASE sat OF 
    'A': BEGIN
	
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'lrnm_spare':	BEGIN
			read_product_hk, cdf_id, 'lrnm_spare_eng',	lrnm_spare_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lrnm_spare_a
		END
		'lvc_-12v':  BEGIN
			read_product_hk, cdf_id, 'lvc_-12v_eng',	lvc_neg12v_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg12v_a
		END
		'lvc_-5v':  BEGIN
			read_product_hk, cdf_id, 'lvc_-5v_eng',	lvc_neg5v_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg5v_a
		END
		'esa_vm_pos':  BEGIN
			read_product_hk, cdf_id, 'esa_vm_pos_eng',	esa_vm_pos_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, esa_vm_pos_a	
		END
		'dfl_1_vm':  BEGIN
			read_product_hk, cdf_id, 'dfl_1_vm_eng',	dfl_1_vm_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_1_vm_a
		END
		'dfl_1_tap':  BEGIN
			read_product_hk, cdf_id, 'dfl_1_tap_eng',	dfl_1_tap_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_1_tap_a
		END
		'dfl_2_vm':  BEGIN
			read_product_hk, cdf_id, 'dfl_2_vm_eng',	dfl_2_vm_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_2_vm_a
		END
		'dfl_2_tap':  BEGIN
			read_product_hk, cdf_id, 'dfl_2_tap_raw',	dfl_2_tap_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_2_tap_a
		END
		'lvc_+2.5v_b':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_b_eng',lvc_pos2point5v_b_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_b_a
		END
		'pac_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'pac_cm_dc_eng',	pac_cm_dc_a,	data1_index,	epoch1maxrec 
			remove_fill_hk, -1, pac_cm_dc_a
		END
		'lvc_+12v':  BEGIN
			read_product_hk, cdf_id, 'lvc_+12v_eng',	lvc_pos12v_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos12v_a
		END
		'lrpm_spare':  BEGIN
			read_product_hk, cdf_id, 'lrpm_spare_eng',	lrpm_spare_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lrpm_spare_a
		END
		'lvc_+2.5v_a':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_a_eng',lvc_pos2point5v_a_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_a_a
		END
		'lvc_+5v':  BEGIN
			read_product_hk, cdf_id, 'lvc_+5v_eng',	lvc_pos5v_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos5v_a
		END
		'adc_avdd':  BEGIN
			read_product_hk, cdf_id, 'adc_avdd_eng',	adc_avdd_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_avdd_a
		END
		'adc_dvdd':  BEGIN
			read_product_hk, cdf_id, 'adc_dvdd_eng',	adc_dvdd_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_dvdd_a
		END
		'pac_vm':  BEGIN
			read_product_hk, cdf_id, 'pac_vm_eng',	pac_vm_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, pac_vm_a
		END
		'cal_vref':  BEGIN
			read_product_hk, cdf_id, 'cal_vref_eng',	cal_vref_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, cal_vref_a
		END
		'pac_cm_ac':  BEGIN
			read_product_hk, cdf_id, 'pac_cm_ac_raw',	pac_cm_ac_raw_a,data1_index,	epoch1maxrec
			remove_fill_hk, -1, pac_cm_ac_a
		END
		'mcp_vm':  BEGIN
			read_product_hk, cdf_id, 'mcp_vm_eng',	mcp_vm_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, mcp_vm_a
		END
		'mcp_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'mcp_cm_dc_eng',	mcp_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, mcp_cm_dc_a
		END
		'mcp_cm_ac':  BEGIN
			read_product_hk, cdf_id, 'mcp_cm_ac_raw',	mcp_cm_ad_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, mcp_cm_ac_a
		END
		'ssd_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'ssd_cm_dc_eng',	ssd_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_cm_dc_a
		END
		's_ch_vm':  BEGIN
			read_product_hk, cdf_id, 's_ch_vm_eng',	s_ch_vm_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_ch_vm_a
		END
		's_ch_vm_tap':  BEGIN
			read_product_hk, cdf_id, 's_ch_vm_tap_eng',s_ch_vm_tap_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_ch_vm_tap_a
		END
		's_ch_cm_dc':  BEGIN
			read_product_hk, cdf_id, 's_ch_cm_dc_eng',	s_ch_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_ch_cm_dc_a
		END
		'esa_vm_neg':  BEGIN
			read_product_hk, cdf_id, 'esa_vm_neg_eng',	esa_vm_neg_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, esa_vm_neg_a
		END
		'esa_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'esa_cm_dc_eng',	esa_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, esa_cm_dc_a
		END
		'dfl_1_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'dfl_1_cm_dc_eng',dfl_1_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_1_cm_dc_a
		END
		'dfl_2_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'dfl_2_cm_dc_eng',dfl_2_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_2_cm_dc_a
		END
		'lvc_+2.5v_b_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_b_imon_eng',	lvc_pos2point5v_b_imon_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_b_imon_a
		END
		'lvc_+12v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+12v_imon_eng',	lvc_pos12v_imon_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos12v_imon_a
		END
		'lvc_-12v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_-12v_imon_eng',	lvc_neg12v_imon_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg12v_imon_a
		END
		'lvc_+5v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+5v_imon_eng',	lvc_pos5v_imon_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos5v_imon_a
		END
		'lvc_-5v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_-5v_imon_eng',	lvc_neg5v_imon_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg5v_imon_a
		END
		'lvc_+2.5v_a_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_a_imon_eng',	lvc_pos2point5v_a_imon_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_a_imon_a
		END
		'ssd_vm':  BEGIN
			read_product_hk, cdf_id, 'ssd_vm_eng',	ssd_vm_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_vm_a
		END
		'adc_lu_flag':  BEGIN
			read_product_hk, cdf_id, 'adc_lu_flag_raw',adc_lu_flag_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_lu_flag_a
		END
		'plug_id':  BEGIN
			read_product_hk, cdf_id, 'plug_id_raw',	plug_id_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, plug_id_a
		END
		'adc_lu_ctr':  BEGIN
			read_product_hk, cdf_id, 'adc_lu_ctr_raw',	adc_lu_ctr_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_lu_ctr_a
		END
		'dac_status':  BEGIN
			read_product_hk, cdf_id, 'dac_status_raw',	dac_status_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dac_status_a
		END
		'adc0_agnd':  BEGIN
			read_product_hk, cdf_id, 'adc0_agnd_eng',	adc0_agnd_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc0_agnd_a
		END
		'adc1_agnd':  BEGIN
			read_product_hk, cdf_id, 'adc1_agnd_eng',	adc1_agnd_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc1_agnd_a
		END
		'ra_e0':  BEGIN
			read_product_hk, cdf_id, 'ra_e0_eng',	ra_e0_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ra_e0_a
		END
		'ra_e1':  BEGIN
			read_product_hk, cdf_id, 'ra_e1_eng',	ra_e1_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ra_e1_a
		END
		'tac0_tsp':  BEGIN
			read_product_hk, cdf_id, 'tac0_tsp_eng',	tac0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tac0_tsp_a
		END
		'tac2_tsp':  BEGIN
			read_product_hk, cdf_id, 'tac2_tsp_eng',	tac2_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tac2_tsp_a
		END
		'sb0_tsp':  BEGIN
			read_product_hk, cdf_id, 'sb0_tsp_eng',	sb0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, sb0_tsp_a
		END
		'sb1_tsp':  BEGIN
			read_product_hk, cdf_id, 'sb1_tsp_eng',	sb1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, sb1_tsp_a
		END
		'tof_hv0_tsp':  BEGIN
			read_product_hk, cdf_id, 'tof_hv0_tsp_eng',tof_hv0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tof_hv0_tsp_a
		END
		'tof_hv1_tsp':  BEGIN
			read_product_hk, cdf_id, 'tof_hv1_tsp_eng',tof_hv1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tof_hv1_tsp_a
		END
		's_c_0_tsp':  BEGIN
			read_product_hk, cdf_id, 's_c_0_tsp_eng',	s_c_0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_c_0_tsp_a
		END
		's_c_1_tsp':  BEGIN
			read_product_hk, cdf_id, 's_c_1_tsp_eng',	s_c_1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_c_1_tsp_a
		END
		'lvc0_tsp':  BEGIN
			read_product_hk, cdf_id, 'lvc0_tsp_eng',	lvc0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc0_tsp_a
		END
		'lvc1_tsp':  BEGIN
			read_product_hk, cdf_id, 'lvc1_tsp_eng',	lvc1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc1_tsp_a
		END
		'adc0_vref':  BEGIN
			read_product_hk, cdf_id, 'adc0_vref_eng',	adc0_vref_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc0_vref_a
		END
		'adc1_vref':  BEGIN
			read_product_hk, cdf_id, 'adc1_vref_eng',	adc1_vref_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc1_vref_a
		END
		'ssd_status':  BEGIN
			read_product_hk, cdf_id, 'ssd_status_raw',	ssd_status_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_status_a
		END
		'ssd_v_pos_analog':  BEGIN
			read_product_hk, cdf_id, 'ssd_v_pos_analog_eng',	ssd_v_pos_analog_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_v_pos_analog_a
		END
		'ssd_v_neg_analog':  BEGIN
			read_product_hk, cdf_id, 'ssd_v_neg_analog_eng',	ssd_v_neg_analog_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_v_neg_analog_a
		END
		'ssd_hv_bias':  BEGIN
			read_product_hk, cdf_id, 'ssd_hv_bias_eng',	ssd_hv_bias_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_hv_bias_a
		END
		'ssd_tc0':  BEGIN
			read_product_hk, cdf_id, 'ssd_tc0_eng',	ssd_tc0_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_tc0_a
		END
		'ssd_tc1':  BEGIN
			read_product_hk, cdf_id, 'ssd_tc1_eng',	ssd_tc1_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_tc1_a
		END
		'ssd_tc2':  BEGIN
			read_product_hk, cdf_id, 'ssd_tc2_eng',	ssd_tc2_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_tc2_a	
		END
		'ssd_v_pos_dig':  BEGIN
			read_product_hk, cdf_id, 'ssd_v_pos_dig_eng',	ssd_v_pos_dig_a,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_v_pos_dig_a
		END
		ENDCASE
	ENDFOR

    END
    'B': BEGIN
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'lrnm_spare':	BEGIN
			read_product_hk, cdf_id, 'lrnm_spare_eng',	lrnm_spare_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lrnm_spare_b
		END
		'lvc_-12v':  BEGIN
			read_product_hk, cdf_id, 'lvc_-12v_eng',	lvc_neg12v_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg12v_b
		END
		'lvc_-5v':  BEGIN
			read_product_hk, cdf_id, 'lvc_-5v_eng',	lvc_neg5v_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg5v_b
		END
		'esa_vm_pos':  BEGIN
			read_product_hk, cdf_id, 'esa_vm_pos_eng',	esa_vm_pos_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, esa_vm_pos_b	
		END
		'dfl_1_vm':  BEGIN
			read_product_hk, cdf_id, 'dfl_1_vm_eng',	dfl_1_vm_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_1_vm_b
		END
		'dfl_1_tap':  BEGIN
			read_product_hk, cdf_id, 'dfl_1_tap_eng',	dfl_1_tap_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_1_tap_b
		END
		'dfl_2_vm':  BEGIN
			read_product_hk, cdf_id, 'dfl_2_vm_eng',	dfl_2_vm_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_2_vm_b
		END
		'dfl_2_tap':  BEGIN
			read_product_hk, cdf_id, 'dfl_2_tap_raw',	dfl_2_tap_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_2_tap_b
		END
		'lvc_+2.5v_b':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_b_eng',lvc_pos2point5v_b_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_b_b
		END
		'pac_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'pac_cm_dc_eng',	pac_cm_dc_b,	data1_index,	epoch1maxrec 
			remove_fill_hk, -1, pac_cm_dc_b
		END
		'lvc_+12v':  BEGIN
			read_product_hk, cdf_id, 'lvc_+12v_eng',	lvc_pos12v_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos12v_b
		END
		'lrpm_spare':  BEGIN
			read_product_hk, cdf_id, 'lrpm_spare_eng',	lrpm_spare_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lrpm_spare_b
		END
		'lvc_+2.5v_a':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_a_eng',lvc_pos2point5v_a_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_a_b
		END
		'lvc_+5v':  BEGIN
			read_product_hk, cdf_id, 'lvc_+5v_eng',	lvc_pos5v_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos5v_b
		END
		'adc_avdd':  BEGIN
			read_product_hk, cdf_id, 'adc_avdd_eng',	adc_avdd_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_avdd_b
		END
		'adc_dvdd':  BEGIN
			read_product_hk, cdf_id, 'adc_dvdd_eng',	adc_dvdd_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_dvdd_b
		END
		'pac_vm':  BEGIN
			read_product_hk, cdf_id, 'pac_vm_eng',	pac_vm_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, pac_vm_b
		END
		'cal_vref':  BEGIN
			read_product_hk, cdf_id, 'cal_vref_eng',	cal_vref_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, cal_vref_b
		END
		'pac_cm_ac':  BEGIN
			read_product_hk, cdf_id, 'pac_cm_ac_raw',	pac_cm_ac_raw_b,data1_index,	epoch1maxrec
			remove_fill_hk, -1, pac_cm_ac_b
		END
		'mcp_vm':  BEGIN
			read_product_hk, cdf_id, 'mcp_vm_eng',	mcp_vm_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, mcp_vm_b
		END
		'mcp_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'mcp_cm_dc_eng',	mcp_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, mcp_cm_dc_b
		END
		'mcp_cm_ac':  BEGIN
			read_product_hk, cdf_id, 'mcp_cm_ac_raw',	mcp_cm_ad_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, mcp_cm_ac_b
		END
		'ssd_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'ssd_cm_dc_eng',	ssd_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_cm_dc_b
		END
		's_ch_vm':  BEGIN
			read_product_hk, cdf_id, 's_ch_vm_eng',	s_ch_vm_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_ch_vm_b
		END
		's_ch_vm_tap':  BEGIN
			read_product_hk, cdf_id, 's_ch_vm_tap_eng',s_ch_vm_tap_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_ch_vm_tap_b
		END
		's_ch_cm_dc':  BEGIN
			read_product_hk, cdf_id, 's_ch_cm_dc_eng',	s_ch_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_ch_cm_dc_b
		END
		'esa_vm_neg':  BEGIN
			read_product_hk, cdf_id, 'esa_vm_neg_eng',	esa_vm_neg_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, esa_vm_neg_b
		END
		'esa_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'esa_cm_dc_eng',	esa_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, esa_cm_dc_b
		END
		'dfl_1_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'dfl_1_cm_dc_eng',dfl_1_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_1_cm_dc_b
		END
		'dfl_2_cm_dc':  BEGIN
			read_product_hk, cdf_id, 'dfl_2_cm_dc_eng',dfl_2_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dfl_2_cm_dc_b
		END
		'lvc_+2.5v_b_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_b_imon_eng',	lvc_pos2point5v_b_imon_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_b_imon_b
		END
		'lvc_+12v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+12v_imon_eng',	lvc_pos12v_imon_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos12v_imon_b
		END
		'lvc_-12v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_-12v_imon_eng',	lvc_neg12v_imon_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg12v_imon_b
		END
		'lvc_+5v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+5v_imon_eng',	lvc_pos5v_imon_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos5v_imon_b
		END
		'lvc_-5v_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_-5v_imon_eng',	lvc_neg5v_imon_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_neg5v_imon_b
		END
		'lvc_+2.5v_a_imon':  BEGIN
			read_product_hk, cdf_id, 'lvc_+2.5v_a_imon_eng',	lvc_pos2point5v_a_imon_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc_pos2point5v_a_imon_b
		END
		'ssd_vm':  BEGIN
			read_product_hk, cdf_id, 'ssd_vm_eng',	ssd_vm_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_vm_b
		END
		'adc_lu_flag':  BEGIN
			read_product_hk, cdf_id, 'adc_lu_flag_raw',adc_lu_flag_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_lu_flag_b
		END
		'plug_id':  BEGIN
			read_product_hk, cdf_id, 'plug_id_raw',	plug_id_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, plug_id_b
		END
		'adc_lu_ctr':  BEGIN
			read_product_hk, cdf_id, 'adc_lu_ctr_raw',	adc_lu_ctr_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc_lu_ctr_b
		END
		'dac_status':  BEGIN
			read_product_hk, cdf_id, 'dac_status_raw',	dac_status_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, dac_status_b
		END
		'adc0_agnd':  BEGIN
			read_product_hk, cdf_id, 'adc0_agnd_eng',	adc0_agnd_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc0_agnd_b
		END
		'adc1_agnd':  BEGIN
			read_product_hk, cdf_id, 'adc1_agnd_eng',	adc1_agnd_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc1_agnd_b
		END
		'ra_e0':  BEGIN
			read_product_hk, cdf_id, 'ra_e0_eng',	ra_e0_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ra_e0_b
		END
		'ra_e1':  BEGIN
			read_product_hk, cdf_id, 'ra_e1_eng',	ra_e1_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ra_e1_b
		END
		'tac0_tsp':  BEGIN
			read_product_hk, cdf_id, 'tac0_tsp_eng',	tac0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tac0_tsp_b
		END
		'tac2_tsp':  BEGIN
			read_product_hk, cdf_id, 'tac2_tsp_eng',	tac2_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tac2_tsp_b
		END
		'sb0_tsp':  BEGIN
			read_product_hk, cdf_id, 'sb0_tsp_eng',	sb0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, sb0_tsp_b
		END
		'sb1_tsp':  BEGIN
			read_product_hk, cdf_id, 'sb1_tsp_eng',	sb1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, sb1_tsp_b
		END
		'tof_hv0_tsp':  BEGIN
			read_product_hk, cdf_id, 'tof_hv0_tsp_eng',tof_hv0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tof_hv0_tsp_b
		END
		'tof_hv1_tsp':  BEGIN
			read_product_hk, cdf_id, 'tof_hv1_tsp_eng',tof_hv1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, tof_hv1_tsp_b
		END
		's_c_0_tsp':  BEGIN
			read_product_hk, cdf_id, 's_c_0_tsp_eng',	s_c_0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_c_0_tsp_b
		END
		's_c_1_tsp':  BEGIN
			read_product_hk, cdf_id, 's_c_1_tsp_eng',	s_c_1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, s_c_1_tsp_b
		END
		'lvc0_tsp':  BEGIN
			read_product_hk, cdf_id, 'lvc0_tsp_eng',	lvc0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc0_tsp_b
		END
		'lvc1_tsp':  BEGIN
			read_product_hk, cdf_id, 'lvc1_tsp_eng',	lvc1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, lvc1_tsp_b
		END
		'adc0_vref':  BEGIN
			read_product_hk, cdf_id, 'adc0_vref_eng',	adc0_vref_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc0_vref_b
		END
		'adc1_vref':  BEGIN
			read_product_hk, cdf_id, 'adc1_vref_eng',	adc1_vref_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, adc1_vref_b
		END
		'ssd_status':  BEGIN
			read_product_hk, cdf_id, 'ssd_status_raw',	ssd_status_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_status_b
		END
		'ssd_v_pos_analog':  BEGIN
			read_product_hk, cdf_id, 'ssd_v_pos_analog_eng',	ssd_v_pos_analog_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_v_pos_analog_b
		END
		'ssd_v_neg_analog':  BEGIN
			read_product_hk, cdf_id, 'ssd_v_neg_analog_eng',	ssd_v_neg_analog_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_v_neg_analog_b
		END
		'ssd_hv_bias':  BEGIN
			read_product_hk, cdf_id, 'ssd_hv_bias_eng',	ssd_hv_bias_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_hv_bias_b
		END
		'ssd_tc0':  BEGIN
			read_product_hk, cdf_id, 'ssd_tc0_eng',	ssd_tc0_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_tc0_b
		END
		'ssd_tc1':  BEGIN
			read_product_hk, cdf_id, 'ssd_tc1_eng',	ssd_tc1_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_tc1_b
		END
		'ssd_tc2':  BEGIN
			read_product_hk, cdf_id, 'ssd_tc2_eng',	ssd_tc2_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_tc2_b
		END
		'ssd_v_pos_dig':  BEGIN
			read_product_hk, cdf_id, 'ssd_v_pos_dig_eng',	ssd_v_pos_dig_b,	data1_index,	epoch1maxrec
			remove_fill_hk, -1, ssd_v_pos_dig_b
		END
		ENDCASE
	ENDFOR
    END
ENDCASE
hk_read = 1
END 

PRO check_indices_hk, index, epoch, resolution, file_num

; checks for missing records from the cdf epoch variable
; index: array of pairs [start_index,num_records] (could be [0,10,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1,5, or 3 depending on epoch (may need to add variable)
; file_num: index of this file in the file array

first_index=0L
first_index = file_num*(1440/resolution)

num_records = 0L
in_valid_area = 0
ii = 0L
index = [-1L]
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 59 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*60)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]                
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    5: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 11 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF (hour EQ time.hour AND time.min GE (min*5) AND time.min LT (min*5+5)) OR $
                      (time.hour EQ hour+1 AND time.min LE 4 AND min EQ 11) THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*12)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf_hk_1min, sat, files, products, which_epoch

COMMON share1_hk
COMMON com_hk
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat

; initialize output variables (from common block)
need_epoch1 = 0
need_epoch5 = 0
need_epoch2 = 0
FOR ii = 0L, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
        1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
        END 
        5: BEGIN 
            num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 5
        END 
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 
hk_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	'lrnm_spare' : ;0
	'lvc_-12v': ;1
	'lvc_-5v': ;2
	'esa_vm_pos': ;3
	'dfl_1_vm': ;4
	'dfl_1_tap': ;5
	'dfl_2_vm': ;6
	'dfl_2_tap': ;7
	'lvc_+2.5v_b': ;8
	'pac_cm_dc': ;9
	'lvc_+12v': ;10
	'lrpm_spare': ;11
	'lvc_+2.5v_a': ;12
	'lvc_+5v': ;13
	'adc_avdd': ;14
	'adc_dvdd': ;15
	'pac_vm': ;16
	'cal_vref': ;17
	'pac_cm_ac': ;18
	'mcp_vm': ;19
	'mcp_cm_dc': ;20
	'mcp_cm_ac': ;21
	'ssd_cm_dc': ;22
	's_ch_vm': ;23
	's_ch_vm_tap': ;24
	's_ch_cm_dc': ;25
	'esa_vm_neg': ;26
	'esa_cm_dc': ;27
	'dfl_1_cm_dc': ;28
	'dfl_2_cm_dc': ;29
	'lvc_+2.5v_b_imon': ;30
	'lvc_+12v_imon': ;31
	'lvc_-12v_imon': ;32
	'lvc_+5v_imon': ;33
	'lvc_-5v_imon': ;34
	'lvc_+2.5v_a_imon': ;35
	'ssd_vm': ;36
	'adc_lu_flag': ;37
	'plug_id': ;38
	'adc_lu_ctr': ;39
	'dac_status': ;40
	'adc0_agnd': ;41
	'adc1_agnd': ;42
	'ra_e0': ;43
	'ra_e1': ;44
	'tac0_tsp': ;45
	'tac2_tsp': ;46
	'sb0_tsp': ;47
	'sb1_tsp': ;48
	'tof_hv0_tsp': ;49
	'tof_hv1_tsp': ;50
	's_c_0_tsp': ;51
	's_c_1_tsp': ;52
	'lvc0_tsp': ;53
	'lvc1_tsp': ;54
	'adc0_vref': ;55
	'adc1_vref': ;56
	'ssd_status': ;57
	'ssd_v_pos_analog': ;58
	'ssd_v_neg_analog': ;59
	'ssd_hv_bias': ;60
	'ssd_tc0': ;61
	'ssd_tc1': ;62
	'ssd_tc2': ;63
	'ssd_v_pos_dig': IF hk_initialized EQ 0 THEN init_hk, sat, num_records1, num_records2, hk_initialized
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch_analog', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch_analog', temp_epoch, rec_count = epoch1info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_hk, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data1_index[kk+1]-1)]
                            converted_index = converted_index+data1_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF
                END 
                5: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5', /zvariable
                    IF epoch5info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch5', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices_hk, data5_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data5_index[kk+1]-1)]
                            converted_index = converted_index+data5_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF 
                END 
                0: BEGIN 
                    print, "Lorna: add variable epoch"
                    stop
                END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_a = epoch5
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
        ENDELSE 

       ; get other data
        hk_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		'lrnm_spare' : ;0
		'lvc_-12v': ;1
		'lvc_-5v': ;2
		'esa_vm_pos': ;3
		'dfl_1_vm': ;4
		'dfl_1_tap': ;5
		'dfl_2_vm': ;6
		'dfl_2_tap': ;7
		'lvc_+2.5v_b': ;8
		'pac_cm_dc': ;9
		'lvc_+12v': ;10
		'lrpm_spare': ;11
		'lvc_+2.5v_a': ;12
		'lvc_+5v': ;13
		'adc_avdd': ;14
		'adc_dvdd': ;15
		'pac_vm': ;16
		'cal_vref': ;17
		'pac_cm_ac': ;18
		'mcp_vm': ;19
		'mcp_cm_dc': ;20
		'mcp_cm_ac': ;21
		'ssd_cm_dc': ;22
		's_ch_vm': ;23
		's_ch_vm_tap': ;24
		's_ch_cm_dc': ;25
		'esa_vm_neg': ;26
		'esa_cm_dc': ;27
		'dfl_1_cm_dc': ;28
		'dfl_2_cm_dc': ;29
		'lvc_+2.5v_b_imon': ;30
		'lvc_+12v_imon': ;31
		'lvc_-12v_imon': ;32
		'lvc_+5v_imon': ;33
		'lvc_-5v_imon': ;34
		'lvc_+2.5v_a_imon': ;35
		'ssd_vm': ;36
		'adc_lu_flag': ;37
		'plug_id': ;38
		'adc_lu_ctr': ;39
		'dac_status': ;40
		'adc0_agnd': ;41
		'adc1_agnd': ;42
		'ra_e0': ;43
		'ra_e1': ;44
		'tac0_tsp': ;45
		'tac2_tsp': ;46
		'sb0_tsp': ;47
		'sb1_tsp': ;48
		'tof_hv0_tsp': ;49
		'tof_hv1_tsp': ;50
		's_c_0_tsp': ;51
		's_c_1_tsp': ;52
		'lvc0_tsp': ;53
		'lvc1_tsp': ;54
		'adc0_vref': ;55
		'adc1_vref': ;56
		'ssd_status': ;57
		'ssd_v_pos_analog': ;58
		'ssd_v_neg_analog': ;59
		'ssd_hv_bias': ;60
		'ssd_tc0': ;61
		'ssd_tc1': ;62
		'ssd_tc2': ;63
		'ssd_v_pos_dig': IF hk_read EQ 0 THEN read_hk, cdf_id, sat, hk_read, data1_index, epoch1info.maxrec+1, products
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
    IF need_epoch5 EQ 5 THEN data5_index_a = data5_index
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
    IF need_epoch5 EQ 5 THEN data5_index_b = data5_index
ENDELSE 

END 
