;+
;PROCEDURE: pl_read_cdf_general
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch:  resolution
;		    E_step:  Array of desired energy steps (set to 0 if not applicable)
;		    crib_name:  String naming which type of product is being read (housekeeping, monitor rates, etc.)
;		    NO_ENG:  if set to 1, do not look for engineering mode data
;
;CREATED BY: L. Ellis (pl_read_cdf)
;
;LAST MODIFICATION: 3 July, 2007
;
;MODIFICATION HISTORY:
;    3 July, 2007: combined pl_read_cdf_mon, pl_read_cdf_mon_no_eng, pl_read_cdf_hk_1min,
;		   pl_read_cdf_dig_hk, pl_read_cdf_mon_full, and
;		   pl_read_cdf_sc_hk
;    18 July, 2007: changed 'first_index' to a long (LBE)
;                   changed loop in read_one_mon to long
;
;-

PRO read_product, cdf_id, prod_name, data_var, data_index, epoch_maxrec

; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 

cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable

IF varinfo.maxrec GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec

IF maxrec GT 0 THEN BEGIN 
    cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
    num_dim = size(data_var, /n_dimensions)

    CASE num_dim OF 
        1: BEGIN
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[data_index[kk]:(data_index[kk]+data_index[kk+2]-1)] = temp_data[data_index[kk+1]:(data_index[kk+1]+data_index[kk+2]-1)]
                kk = kk+2
            ENDFOR 
        END 
        2: BEGIN
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+2]-1)] = temp_data[*, data_index[kk+1]:(data_index[kk+1]+data_index[kk+2]-1)]
                kk = kk+2
            ENDFOR 
        END 
        3: BEGIN
            FOR kk = 0L, n_elements(data_index)-1 DO BEGIN 
                data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+2]-1)] = temp_data[*, *, data_index[kk+1]:(data_index[kk+1]+data_index[kk+2]-1)]
                kk = kk+2
            ENDFOR 
        END 
        ELSE: BEGIN 
            print, "Lorna: Add more dimensions"
            stop
        END
    ENDCASE 
ENDIF ELSE data_index = -1
END 

PRO remove_fill, fill_val, data

; fill_val: fill_value to be replaced by NaN
; data: array of data values
indice = where(data EQ fill_val, count)
FOR ii = 0L, count-1 DO BEGIN
    data[indice[ii]] = !values.d_nan
ENDFOR 

END 

;; BEGINNING OF VARIABLE INITIALIZATION ROUTINES

;-----------------------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------------------
PRO init_sc_hk, sat, num_records3, sc_hk_initialized
;-----------------------------------------------------------------------------------------------------
; initialize all the spacecraft housekeeping variables
;	sat:  which spacecraft
;	num_records3: the number of records we expect
;	sc_hk_initialized:  whether or not the spacecraft housekeeping variables have been initialized

COMMON com_mon
CASE sat OF
    'A': BEGIN 
	pl_ebox_tmp_a	= dblarr(num_records3) ; 0
	pl_inst_msg_pwr_dwn_req_a	= strarr(num_records3) ; 1
	pl_pwr_cur_a	= dblarr(num_records3) ; 2
	pl_pwr_tt_a	= strarr(num_records3) ; 3
	pl_surv_pwr_cur_a	= dblarr(num_records3) ; 4
	pl_surv_pwr_tt_a	= strarr(num_records3) ; 5
	pl_tof_tmp_a	= dblarr(num_records3) ; 6
	pl_pwr_vt_a	= dblarr(num_records3) ; 7
	pl_surv_pwr_vt_a	= dblarr(num_records3) ; 8
	
	pl_ebox_tmp_a[*] = !values.d_nan ; 0
	pl_inst_msg_pwr_dwn_req_a[*] = !values.d_nan ; 1
	pl_pwr_cur_a[*] = !values.d_nan ; 2
	pl_pwr_tt_a[*] = !values.d_nan ; 3
	pl_surv_pwr_cur_a[*] = !values.d_nan ; 4
	pl_surv_pwr_tt_a[*] = !values.d_nan ; 5
	pl_tof_tmp_a[*] = !values.d_nan ; 6
	pl_pwr_vt_a[*] = !values.d_nan ; 7
	pl_surv_pwr_vt_a[*] = !values.d_nan ; 8

    END 
    'B': BEGIN 
	pl_ebox_tmp_b	= dblarr(num_records3) ; 0
	pl_inst_msg_pwr_dwn_req_b	= strarr(num_records3) ; 1
	pl_pwr_cur_b	= dblarr(num_records3) ; 2
	pl_pwr_tt_b	= strarr(num_records3) ; 3
	pl_surv_pwr_cur_b	= dblarr(num_records3) ; 4
	pl_surv_pwr_tt_b	= strarr(num_records3) ; 5
	pl_tof_tmp_b	= dblarr(num_records3) ; 6
	pl_pwr_vt_b	= dblarr(num_records3) ; 7
	pl_surv_pwr_vt_b	= dblarr(num_records3) ; 8
	
	pl_ebox_tmp_b[*] = !values.d_nan ; 0
	pl_inst_msg_pwr_dwn_req_b[*] = !values.d_nan ; 1
	pl_pwr_cur_b[*] = !values.d_nan ; 2
	pl_pwr_tt_b[*] = !values.d_nan ; 3
	pl_surv_pwr_cur_b[*] = !values.d_nan ; 4
	pl_surv_pwr_tt_b[*] = !values.d_nan ; 5
	pl_tof_tmp_b[*] = !values.d_nan ; 6
	pl_pwr_vt_b[*] = !values.d_nan ; 7
	pl_surv_pwr_vt_b[*] = !values.d_nan ; 8
        
    END 
ENDCASE 
sc_hk_initialized = 1

END 
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
PRO init_dig_hk, sat, num_records1, dig_hk_initialized, products
;---------------------------------------------------------------------------
; initialize all the digital housekeeping variables
;	sat:  which spacecraft
;	num_records1:  the number of records we expect
;	dig_hk_initialized:  whether or not the digital housekeeping variables have been initialized
;	products:  which products we want -- initialize variables only for the specific products
;			we want, otherwise too much memory is used

COMMON com_mon
CASE sat OF
    'A': BEGIN 

	FOR kk=0,n_elements(products)-1 DO BEGIN
	CASE products[kk] OF
	
	'dpu_sw_version':  BEGIN
		dpu_sw_version_a	= dblarr(num_records1) ; 0
		dpu_sw_version_a[*] = !values.d_nan ; 0
		END
	'cmd_rcvd_cnt':  BEGIN
		cmd_rcvd_cnt_a	= dblarr(num_records1) ; 1
		cmd_rcvd_cnt_a[*] = !values.d_nan ; 1
		END
	'cmd_exec_cnt':  BEGIN
		cmd_exec_cnt_a	= dblarr(num_records1) ; 2
		cmd_exec_cnt_a[*] = !values.d_nan ; 2
		END
	'cmd_total_err_cnt':  BEGIN
		cmd_total_err_cnt_a	= dblarr(num_records1) ; 3
		cmd_total_err_cnt_a[*] = !values.d_nan ; 3
		END
	'cmd_parity_err_cnt':  BEGIN
		cmd_parity_err_cnt_a	= dblarr(num_records1) ; 4
		cmd_parity_err_cnt_a[*] = !values.d_nan ; 4
		END
	'cmd_frame_err_cnt':  BEGIN
		cmd_frame_err_cnt_a	= dblarr(num_records1) ; 5
		cmd_frame_err_cnt_a[*] = !values.d_nan ; 5
		END
	'cmd_illegal_err_cnt':  BEGIN
		cmd_illegal_err_cnt_a	= dblarr(num_records1) ; 6
		cmd_illegal_err_cnt_a[*] = !values.d_nan ; 6
		END
	'ic_status_reg':  BEGIN
		ic_status_reg_a	= dblarr(num_records1) ; 7
		ic_status_reg_a[*] = !values.d_nan ; 7
		END
	'ic_ctl_reg':  BEGIN
		ic_ctl_reg_a	= dblarr(num_records1) ; 8
		ic_ctl_reg_a[*] = !values.d_nan ; 8
		END
	'lbc_reset_ctl':  BEGIN
		lbc_reset_ctl_a	= dblarr(num_records1) ; 9
		lbc_reset_ctl_a[*] = !values.d_nan ; 9
		END
	'lbc_mode_ctl':  BEGIN
		lbc_mode_ctl_a	= dblarr(num_records1) ; 10
		lbc_mode_ctl_a[*] = !values.d_nan ; 10
		END
	'lbc_logic_ctl_a':  BEGIN
		lbc_logic_ctl_a_a	= dblarr(num_records1) ; 11
		lbc_logic_ctl_a_a[*] = !values.d_nan ; 11
		END
	'lbc_logic_ctl_b':  BEGIN
		lbc_logic_ctl_b_a	= dblarr(num_records1) ; 12
		lbc_logic_ctl_b_a[*] = !values.d_nan ; 12
		END
	'lbc_tac_pwr_ctl':  BEGIN
		lbc_tac_pwr_ctl_a	= dblarr(num_records1) ; 13
		lbc_tac_pwr_ctl_a[*] = !values.d_nan ; 13
		END
	'lbc_reg_seq':  BEGIN
		lbc_reg_seq_a	= dblarr(num_records1) ; 14
		lbc_reg_seq_a[*] = !values.d_nan ; 14
		END
	'lbc_event_ctl':  BEGIN
		lbc_event_ctl_a	= dblarr(num_records1) ; 15
		lbc_event_ctl_a[*] = !values.d_nan ; 15
		END
	'lbc_rlim_ch':  BEGIN
		lbc_rlim_ch_a	= dblarr(num_records1) ; 16
		lbc_rlim_ch_a[*] = !values.d_nan ; 16
		END
	'lbc_rlim_hi':  BEGIN
		lbc_rlim_hi_a	= dblarr(num_records1) ; 17
		lbc_rlim_hi_a[*] = !values.d_nan ; 17
		END
	'lbc_rlim_lo':  BEGIN
		lbc_rlim_lo_a	= dblarr(num_records1) ; 18
		lbc_rlim_lo_a[*] = !values.d_nan ; 18
		END
	'lbe_pos_ctrl':  BEGIN
		lbe_pos_ctrl_a	= dblarr(num_records1) ; 19
		lbe_pos_ctrl_a[*] = !values.d_nan ; 19
		END
	'lbe_pos_disable0':  BEGIN
		lbe_pos_disable0_a	= dblarr(num_records1) ; 20
		lbe_pos_disable0_a[*] = !values.d_nan ; 20
		END
	'lbe_pos_disable2':  BEGIN
		lbe_pos_disable2_a	= dblarr(num_records1) ; 21
		lbe_pos_disable2_a[*] = !values.d_nan ; 21
		END
	'lbe_tac0_ctrl':  BEGIN
		lbe_tac0_ctrl_a	= dblarr(num_records1) ; 22
		lbe_tac0_ctrl_a[*] = !values.d_nan ; 22
		END
	'lbe_tac0_dac':  BEGIN
		lbe_tac0_dac_a	= dblarr(num_records1) ; 23
		lbe_tac0_dac_a[*] = !values.d_nan ; 23
		END
	'lbe_tac2_ctrl':  BEGIN
		lbe_tac2_ctrl_a	= dblarr(num_records1) ; 24
		lbe_tac2_ctrl_a[*] = !values.d_nan ; 24
		END
	'lbe_tac2_dac':  BEGIN
		lbe_tac2_dac_a	= dblarr(num_records1) ; 25
		lbe_tac2_dac_a[*] = !values.d_nan ; 25
		END
	'lbe_tac0_under_hi':  BEGIN
		lbe_tac0_under_hi_a	= dblarr(num_records1) ; 26
		lbe_tac0_under_hi_a[*] = !values.d_nan ; 26
		END
	'lbe_tac0_under_lo':  BEGIN
		lbe_tac0_under_lo_a	= dblarr(num_records1) ; 27
		lbe_tac0_under_lo_a[*] = !values.d_nan ; 27
		END
	'lbe_tac0_over_hi':  BEGIN
		lbe_tac0_over_hi_a	= dblarr(num_records1) ; 28
		lbe_tac0_over_hi_a[*] = !values.d_nan ; 28
		END
	'lbe_tac0_over_lo':  BEGIN
		lbe_tac0_over_lo_a	= dblarr(num_records1) ; 29
		lbe_tac0_over_lo_a[*] = !values.d_nan ; 29
		END
	'lbe_tac2_under_hi':  BEGIN
		lbe_tac2_under_hi_a	= dblarr(num_records1) ; 30
		lbe_tac2_under_hi_a[*] = !values.d_nan ; 30
		END
	'lbe_tac2_under_lo':  BEGIN
		lbe_tac2_under_lo_a	= dblarr(num_records1) ; 31
		lbe_tac2_under_lo_a[*] = !values.d_nan ; 31
		END
	'lbe_tac2_over_hi':  BEGIN
		lbe_tac2_over_hi_a	= dblarr(num_records1) ; 32
		lbe_tac2_over_hi_a[*] = !values.d_nan ; 32
		END
	'lbe_tac2_over_lo':  BEGIN
		lbe_tac2_over_lo_a	= dblarr(num_records1) ; 33
		lbe_tac2_over_lo_a[*] = !values.d_nan ; 33
		END
	'lbe_ssd_ctrl':  BEGIN
		lbe_ssd_ctrl_a	= dblarr(num_records1) ; 34
		lbe_ssd_ctrl_a[*] = !values.d_nan ; 34
		END
	'lbe_ssd_cmd_h':  BEGIN
		lbe_ssd_cmd_h_a	= dblarr(num_records1) ; 35
		lbe_ssd_cmd_h_a[*] = !values.d_nan ; 35
		END
	'lbe_ssd_cmd_l':  BEGIN
		lbe_ssd_cmd_l_a	= dblarr(num_records1) ; 36
		lbe_ssd_cmd_l_a[*] = !values.d_nan ; 36
		END
	'lbe_ssd_dis':  BEGIN
		lbe_ssd_dis_a	= dblarr(num_records1) ; 37
		lbe_ssd_dis_a[*] = !values.d_nan ; 37
		END
	'lbe_ssd_under_hi':  BEGIN
		lbe_ssd_under_hi_a	= dblarr(num_records1) ; 38
		lbe_ssd_under_hi_a[*] = !values.d_nan ; 38
		END
	'lbe_ssd_under_lo':  BEGIN
		lbe_ssd_under_lo_a	= dblarr(num_records1) ; 39
		lbe_ssd_under_lo_a[*] = !values.d_nan ; 39
		END
	'lbe_ssd_over_hi':  BEGIN
		lbe_ssd_over_hi_a	= dblarr(num_records1) ; 40
		lbe_ssd_over_hi_a[*] = !values.d_nan ; 40
		END
	'lbe_ssd_over_lo':  BEGIN
		lbe_ssd_over_lo_a	= dblarr(num_records1) ; 41
		lbe_ssd_over_lo_a[*] = !values.d_nan ; 41
		END
	'lbe_sel_ctrl':  BEGIN
		lbe_sel_ctrl_a	= dblarr(num_records1) ; 42
		lbe_sel_ctrl_a[*] = !values.d_nan ; 42
		END
	'lbe_trig_mode':  BEGIN
		lbe_trig_mode_a	= dblarr(num_records1) ; 43
		lbe_trig_mode_a[*] = !values.d_nan ; 43
		END
	'lbe_esa_step':  BEGIN
		lbe_esa_step_a	= dblarr(num_records1) ; 44
		lbe_esa_step_a[*] = !values.d_nan ; 44
		END
	'lbe_defl_step':  BEGIN
		lbe_defl_step_a	= dblarr(num_records1) ; 45
		lbe_defl_step_a[*] = !values.d_nan ; 45
		END
	'lbe_ssd_offset0':  BEGIN
		lbe_ssd_offset0_a	= dblarr(num_records1) ; 46
		lbe_ssd_offset0_a[*] = !values.d_nan ; 46
		END
	'lbe_ssd_offset1':  BEGIN
		lbe_ssd_offset1_a	= dblarr(num_records1) ; 47
		lbe_ssd_offset1_a[*] = !values.d_nan ; 47
		END
	'lbe_ssd_offset2':  BEGIN
		lbe_ssd_offset2_a	= dblarr(num_records1) ; 48
		lbe_ssd_offset2_a[*] = !values.d_nan ; 48
		END
	'lbe_ssd_offset3':  BEGIN
		lbe_ssd_offset3_a	= dblarr(num_records1) ; 49
		lbe_ssd_offset3_a[*] = !values.d_nan ; 49
		END
	'lbe_ssd_offset4':  BEGIN
		lbe_ssd_offset4_a	= dblarr(num_records1) ; 50
		lbe_ssd_offset4_a[*] = !values.d_nan ; 50
		END
	'lbe_ssd_offset5':  BEGIN
		lbe_ssd_offset5_a	= dblarr(num_records1) ; 51
		lbe_ssd_offset5_a[*] = !values.d_nan; 51
		END
	'lbe_ssd_offset6':  BEGIN
		lbe_ssd_offset6_a	= dblarr(num_records1) ; 52
		lbe_ssd_offset6_a[*] = !values.d_nan ; 52
		END
	'lbe_ssd_offset7':  BEGIN
		lbe_ssd_offset7_a	= dblarr(num_records1) ; 53
		lbe_ssd_offset7_a[*] = !values.d_nan ; 53
		END
	'lbe_stim_enable':  BEGIN
		lbe_stim_enable_a	= dblarr(num_records1) ; 54
		lbe_stim_enable_a[*] = !values.d_nan ; 54
		END
	'lbe_stim_freq_hi':  BEGIN
		lbe_stim_freq_hi_a	= dblarr(num_records1) ; 55
		lbe_stim_freq_hi_a[*] = !values.d_nan ; 55
		END
	'lbe_stim_freq_lo':  BEGIN
		lbe_stim_freq_lo_a	= dblarr(num_records1) ; 56
		lbe_stim_freq_lo_a[*] = !values.d_nan ; 56
		END
	'lbe_sel0_window':  BEGIN
		lbe_sel0_window_a	= dblarr(num_records1) ; 57
		lbe_sel0_window_a[*] = !values.d_nan ; 57
		END
	'lbe_sel2_window':  BEGIN
		lbe_sel2_window_a	= dblarr(num_records1) ; 58
		lbe_sel2_window_a[*] = !values.d_nan ; 58
		END
	'lbe_ssd_offset8':  BEGIN
		lbe_ssd_offset8_a	= dblarr(num_records1) ; 59
		lbe_ssd_offset8_a[*] = !values.d_nan ; 59
		END
	'lbe_ssd_offset9':  BEGIN
		lbe_ssd_offset9_a	= dblarr(num_records1) ; 60
		lbe_ssd_offset9_a[*] = !values.d_nan ; 60
		END
	'lbe_ssd_offset10':  BEGIN
		lbe_ssd_offset10_a	= dblarr(num_records1) ; 61
		lbe_ssd_offset10_a[*] = !values.d_nan ; 61
		END
	'lbe_ssd_offset11':  BEGIN
		lbe_ssd_offset11_a	= dblarr(num_records1) ; 62
		lbe_ssd_offset11_a[*] = !values.d_nan ; 62
		END
	'lbe_ssd_offset12':  BEGIN
		lbe_ssd_offset12_a	= dblarr(num_records1) ; 63
		lbe_ssd_offset12_a[*] = !values.d_nan ; 63
		END
	'lbe_ssd_offset13':  BEGIN
		lbe_ssd_offset13_a	= dblarr(num_records1) ; 64
		lbe_ssd_offset13_a[*] = !values.d_nan ; 64
		END
	'lbe_ssd_offset14':  BEGIN
		lbe_ssd_offset14_a	= dblarr(num_records1) ; 65
		lbe_ssd_offset14_a[*] = !values.d_nan ; 65
		END
	'lbe_ssd_offset15':  BEGIN
		lbe_ssd_offset15_a	= dblarr(num_records1) ; 66
		lbe_ssd_offset15_a[*] = !values.d_nan ; 66
		END
	'lbe_pos_ra':  BEGIN
		lbe_pos_ra_a	=	 dblarr(num_records1) ; 67
		lbe_pos_ra_a[*] = !values.d_nan ; 67
		END
	'dacm_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
		dacm_pac_mcp_ssd_ctl_mcp_limit_a	= dblarr(num_records1) ; 68
		dacm_pac_mcp_ssd_ctl_mcp_limit_a[*] = !values.d_nan ; 68
		END
	'dacm_pac_limit':  BEGIN
		dacm_pac_limit_a	= dblarr(num_records1) ; 69
		dacm_pac_limit_a[*] = !values.d_nan ; 69
		END
	'dacm_esa_ctl':  BEGIN
		dacm_esa_ctl_a	=	 dblarr(num_records1) ; 70
		dacm_esa_ctl_a[*] = !values.d_nan ; 70
		END
	'dacm_s_ch_ctl':  BEGIN
		dacm_s_ch_ctl_a	=	 dblarr(num_records1) ; 71
		dacm_s_ch_ctl_a[*] = !values.d_nan ; 71
		END
	'dacm_dfl_1_ctl':  BEGIN
		dacm_dfl_1_ctl_a	= dblarr(num_records1) ; 72
		dacm_dfl_1_ctl_a[*] = !values.d_nan ; 72
		END
	'dacm_dfl_2_ctl':  BEGIN
		dacm_dfl_2_ctl_a	= dblarr(num_records1) ; 73
		dacm_dfl_2_ctl_a[*] = !values.d_nan ; 73
		END
	'dacm_control':  BEGIN
		dacm_control_a	= 	dblarr(num_records1) ; 74
		dacm_control_a[*] = !values.d_nan ; 74
		END
	'dacm_status':  BEGIN
		dacm_status_a	= 	dblarr(num_records1) ; 75
		dacm_status_a[*] = !values.d_nan ; 75
		END
	'dacm_adc_lu_ctr':  BEGIN
		dacm_adc_lu_ctr_a	= dblarr(num_records1) ; 76
		dacm_adc_lu_ctr_a[*] = !values.d_nan ; 76
		END
	'dacm_dfl_1_offset':  BEGIN
		dacm_dfl_1_offset_a	=	 dblarr(num_records1) ; 77
		dacm_dfl_1_offset_a[*] = !values.d_nan ; 77
		END
	'dacm_dfl_2_offset':  BEGIN
		dacm_dfl_2_offset_a	= dblarr(num_records1) ; 78
		dacm_dfl_2_offset_a[*] = !values.d_nan ; 78
		END
	'dacm_pac_coarse':  BEGIN
		dacm_pac_coarse_a	= dblarr(num_records1) ; 79
		dacm_pac_coarse_a[*] = !values.d_nan ; 79
		END
	'dacm_pac_fine':  BEGIN
		dacm_pac_fine_a	=	 dblarr(num_records1) ; 80
		dacm_pac_fine_a[*] = !values.d_nan ; 80
		END
	'dacm_pac_cm_fs_coarse':  BEGIN
		dacm_pac_cm_fs_coarse_a	=	 dblarr(num_records1) ; 81
		dacm_pac_cm_fs_coarse_a[*] = !values.d_nan ; 81
		END
	'dacm_pac_cm_fs_fine':  BEGIN
		dacm_pac_cm_fs_fine_a	= dblarr(num_records1) ; 82
		dacm_pac_cm_fs_fine_a[*] = !values.d_nan ; 82
		END
	'dacm_mcp_coarse':  BEGIN
		dacm_mcp_coarse_a	= dblarr(num_records1) ; 83
		dacm_mcp_coarse_a[*] = !values.d_nan ; 83
		END
	'dacm_mcp_fine':  BEGIN
		dacm_mcp_fine_a	= 	dblarr(num_records1) ; 84
		dacm_mcp_fine_a[*] = !values.d_nan ; 84
		END
	'dacm_mcp_cm_fs_coarse':  BEGIN
		dacm_mcp_cm_fs_coarse_a	= 	dblarr(num_records1) ; 85
		dacm_mcp_cm_fs_coarse_a[*] = !values.d_nan ; 85
		END
	'dacm_mcp_cm_fs_fine':  BEGIN
		dacm_mcp_cm_fs_fine_a	= dblarr(num_records1) ; 86
		dacm_mcp_cm_fs_fine_a[*] = !values.d_nan ; 86
		END
	'dacm_ssd_coarse':  BEGIN
		dacm_ssd_coarse_a	=	dblarr(num_records1) ; 87
		dacm_ssd_coarse_a[*] = !values.d_nan ; 87
		END
	'dacm_ssd_fine':  BEGIN
		dacm_ssd_fine_a	= dblarr(num_records1) ; 88
		dacm_ssd_fine_a[*] = !values.d_nan ; 88
		END
	'dacm_esa_coarse':  BEGIN
		dacm_esa_coarse_a	= dblarr(num_records1) ; 89
		dacm_esa_coarse_a[*] = !values.d_nan ; 89
		END
	'dacm_esa_fine':  BEGIN
		dacm_esa_fine_a	=	 dblarr(num_records1) ; 90
		dacm_esa_fine_a[*] = !values.d_nan ; 90
		END
	'dacm_s_ch_coarse':  BEGIN
		dacm_s_ch_coarse_a	=	 dblarr(num_records1) ; 91
		dacm_s_ch_coarse_a[*] = !values.d_nan ; 91
		END
	'dacm_s_ch_fine':  BEGIN
		dacm_s_ch_fine_a	= dblarr(num_records1) ; 92
		dacm_s_ch_fine_a[*] = !values.d_nan ; 92
		END
	'dacm_s_ch_tap_coarse':  BEGIN
		dacm_s_ch_tap_coarse_a	= dblarr(num_records1) ; 93
		dacm_s_ch_tap_coarse_a[*] = !values.d_nan ; 93
		END
	'dacm_s_ch_tap_fine':  BEGIN
		dacm_s_ch_tap_fine_a	= 	dblarr(num_records1) ; 94
		dacm_s_ch_tap_fine_a[*] = !values.d_nan ; 94
		END
	'dacm_dfl_1_coarse':  BEGIN
		dacm_dfl_1_coarse_a	= 	dblarr(num_records1) ; 95
		dacm_dfl_1_coarse_a[*] = !values.d_nan ; 95
		END
	'dacm_dfl_1_fine':  BEGIN
		dacm_dfl_1_fine_a	= dblarr(num_records1) ; 96
		dacm_dfl_1_fine_a[*] = !values.d_nan ; 96
		END
	'dacm_dfl_1_dfine':  BEGIN
		dacm_dfl_1_dfine_a	=	 dblarr(num_records1) ; 97
		dacm_dfl_1_dfine_a[*] = !values.d_nan ; 97
		END
	'dacm_dfl_2_coarse':  BEGIN
		dacm_dfl_2_coarse_a	= dblarr(num_records1) ; 98
		dacm_dfl_2_coarse_a[*] = !values.d_nan ; 98
		END
	'dacm_dfl_2_fine':  BEGIN
		dacm_dfl_2_fine_a	= dblarr(num_records1) ; 99
		dacm_dfl_2_fine_a[*] = !values.d_nan ; 99
		END
	'dacm_dfl_2_dfine':  BEGIN
		dacm_dfl_2_dfine_a	=	 dblarr(num_records1) ; 100
		dacm_dfl_2_dfine_a[*] = !values.d_nan ; 100
		END
	'dacr_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
		dacr_pac_mcp_ssd_ctl_mcp_limit_a	=	 dblarr(num_records1) ; 101
		dacr_pac_mcp_ssd_ctl_mcp_limit_a[*] = !values.d_nan ; 101
		END
	'dacr_pac_limit':  BEGIN
		dacr_pac_limit_a	= dblarr(num_records1) ; 102
		dacr_pac_limit_a[*] = !values.d_nan ; 102
		END
	'dacr_esa_ctl':  BEGIN
		dacr_esa_ctl_a	= dblarr(num_records1) ; 103
		dacr_esa_ctl_a[*] = !values.d_nan ; 103
		END
	'dacr_s_ch_ctl':  BEGIN
		dacr_s_ch_ctl_a	= 	dblarr(num_records1) ; 104
		dacr_s_ch_ctl_a[*] = !values.d_nan ; 104
		END
	'dacr_dfl_1_ctl':  BEGIN
		dacr_dfl_1_ctl_a	= 	dblarr(num_records1) ; 105
		dacr_dfl_1_ctl_a[*] = !values.d_nan ; 105
		END
	'dacr_dfl_2_ctl':  BEGIN
		dacr_dfl_2_ctl_a	= dblarr(num_records1) ; 106
		dacr_dfl_2_ctl_a[*] = !values.d_nan ; 106
		END
	'dacr_control':  BEGIN
		dacr_control_a	=	 dblarr(num_records1) ; 107
		dacr_control_a[*] = !values.d_nan ; 107
		END
	'dacr_status':  BEGIN
		dacr_status_a	= dblarr(num_records1) ; 108
		dacr_status_a[*] = !values.d_nan ; 108
		END
	'dacr_adc_lu_ctr':  BEGIN
		dacr_adc_lu_ctr_a	= dblarr(num_records1) ; 109
		dacr_adc_lu_ctr_a[*] = !values.d_nan ; 109
		END
	'dacr_dfl_1_offset':  BEGIN
		dacr_dfl_1_offset_a	=	 dblarr(num_records1) ; 110
		dacr_dfl_1_offset_a[*] = !values.d_nan ; 110
		END
	'dacr_dfl_2_offset':  BEGIN
		dacr_dfl_2_offset_a	=	 dblarr(num_records1) ; 111
		dacr_dfl_2_offset_a[*] = !values.d_nan ; 111
		END
	'dacr_c_pac_coarse':  BEGIN
		dacr_c_pac_coarse_a	= dblarr(num_records1) ; 112
		dacr_c_pac_coarse_a[*] = !values.d_nan ; 112
		END
	'dacr_c_pac_fine':  BEGIN
		dacr_c_pac_fine_a	= dblarr(num_records1) ; 113
		dacr_c_pac_fine_a[*] = !values.d_nan ; 113
		END
	'dacr_c_pac_cm_fs_coarse':  BEGIN
		dacr_c_pac_cm_fs_coarse_a	= 	dblarr(num_records1) ; 114
		dacr_c_pac_cm_fs_coarse_a[*] = !values.d_nan ; 114
		END
	'dacr_c_pac_cm_fs_fine':  BEGIN
		dacr_c_pac_cm_fs_fine_a	= 	dblarr(num_records1) ; 115
		dacr_c_pac_cm_fs_fine_a[*] = !values.d_nan ; 115
		END
	'dacr_c_mcp_coarse':  BEGIN
		dacr_c_mcp_coarse_a	= dblarr(num_records1) ; 116
		dacr_c_mcp_coarse_a[*] = !values.d_nan ; 116
		END
	'dacr_c_mcp_fine':  BEGIN
		dacr_c_mcp_fine_a	=	 dblarr(num_records1) ; 117
		dacr_c_mcp_fine_a[*] = !values.d_nan ; 117
		END
	'dacr_c_mcp_cm_fs_coarse':  BEGIN
		dacr_c_mcp_cm_fs_coarse_a	= dblarr(num_records1) ; 118
		dacr_c_mcp_cm_fs_coarse_a[*] = !values.d_nan ; 118
		END
	'dacr_c_mcp_cm_fs_fine':  BEGIN
		dacr_c_mcp_cm_fs_fine_a	= dblarr(num_records1) ; 119
		dacr_c_mcp_cm_fs_fine_a[*] = !values.d_nan ; 119
		END
	'dacr_c_ssd_coarse':  BEGIN
		dacr_c_ssd_coarse_a	=	 dblarr(num_records1) ; 120
		dacr_c_ssd_coarse_a[*] = !values.d_nan ; 120
		END
	'dacr_c_ssd_fine':  BEGIN
		dacr_c_ssd_fine_a	=	 dblarr(num_records1) ; 121
		dacr_c_ssd_fine_a[*] = !values.d_nan ; 121
		END
	'dacr_c_esa_coarse':  BEGIN
		dacr_c_esa_coarse_a	= dblarr(num_records1) ; 122
		dacr_c_esa_coarse_a[*] = !values.d_nan ; 122
		END
	'dacr_c_esa_fine':  BEGIN
		dacr_c_esa_fine_a	= dblarr(num_records1) ; 123
		dacr_c_esa_fine_a[*] = !values.d_nan ; 123
		END
	'dacr_c_s_ch_coarse':  BEGIN
		dacr_c_s_ch_coarse_a	= 	dblarr(num_records1) ; 124
		dacr_c_s_ch_coarse_a[*] = !values.d_nan ; 124
		END
	'dacr_c_s_ch_fine':  BEGIN
		dacr_c_s_ch_fine_a	= 	dblarr(num_records1) ; 125
		dacr_c_s_ch_fine_a[*] = !values.d_nan ; 125
		END
	'dacr_c_s_ch_tap_coarse':  BEGIN
		dacr_c_s_ch_tap_coarse_a	= dblarr(num_records1) ; 126
		dacr_c_s_ch_tap_coarse_a[*] = !values.d_nan ; 126
		END
	'dacr_c_s_ch_tap_fine':  BEGIN
		dacr_c_s_ch_tap_fine_a	=	 dblarr(num_records1) ; 127
		dacr_c_s_ch_tap_fine_a[*] = !values.d_nan ; 127
		END
	'dacr_dfl_1_coarse':  BEGIN
		dacr_dfl_1_coarse_a	= dblarr(num_records1) ; 128
		dacr_dfl_1_coarse_a[*] = !values.d_nan ; 128
		END
	'dacr_dfl_1_fine':  BEGIN
		dacr_dfl_1_fine_a	= dblarr(num_records1) ; 129
		dacr_dfl_1_fine_a[*] = !values.d_nan; 129
		END
	'dacr_dfl_1_dfine':  BEGIN
		dacr_dfl_1_dfine_a	=	 dblarr(num_records1) ; 130
		dacr_dfl_1_dfine_a[*] = !values.d_nan; 130
		END
	'dacr_dfl_2_coarse':  BEGIN
		dacr_dfl_2_coarse_a	=	 dblarr(num_records1) ; 131
		dacr_dfl_2_coarse_a[*] = !values.d_nan ; 131
		END
	'dacr_dfl_2_fine':  BEGIN
		dacr_dfl_2_fine_a	= dblarr(num_records1) ; 132
		dacr_dfl_2_fine_a[*] = !values.d_nan; 132
		END
	'dacr_dfl_2_dfine':  BEGIN
		dacr_dfl_2_dfine_a	= dblarr(num_records1) ; 133
		dacr_dfl_2_dfine_a[*] = !values.d_nan ; 133
		END
	'sw_status1':  BEGIN
		sw_status1_a	= 	dblarr(num_records1) ; 134
		sw_status1_a[*] = !values.d_nan ; 134
		END
	'sw_hv_ramp_status':  BEGIN
		sw_hv_ramp_status_a	= 	dblarr(num_records1) ; 135
		sw_hv_ramp_status_a[*] = !values.d_nan; 135
		END
	'sw_hv_enable_status':  BEGIN
		sw_hv_enable_status_a	= dblarr(num_records1) ; 136
		sw_hv_enable_status_a[*] = !values.d_nan ; 136
		END
	'sw_hv_limit_status':  BEGIN
		sw_hv_limit_status_a	=	 dblarr(num_records1) ; 137
		sw_hv_limit_status_a[*] = !values.d_nan ; 137
		END
	'sw_eepromchksum':  BEGIN
		sw_eepromchksum_a	= dblarr(num_records1) ; 138
		sw_eepromchksum_a[*] = !values.d_nan ; 138
		END
	'hkp_sampletime':  BEGIN
		hkp_sampletime_a	= dblarr(num_records1) ; 139
		hkp_sampletime_a[*] = !values.d_nan ; 139
		END
	'pac_discharge_status':  BEGIN
		pac_discharge_status_a	=	 dblarr(num_records1) ; 140
		pac_discharge_status_a[*] = !values.d_nan ; 140
		END
	'pac_discharge_cnt':  BEGIN
		pac_discharge_cnt_a	=	 dblarr(num_records1) ; 141
		pac_discharge_cnt_a[*] = !values.d_nan ; 141
		END
	'pac_discharge_conseccntr':  BEGIN
		pac_discharge_conseccntr_a	= dblarr(num_records1) ; 142
		pac_discharge_conseccntr_a[*] = !values.d_nan ; 142
		END
	'pac_discharge_safe_att':  BEGIN
		pac_discharge_safe_att_a	= dblarr(num_records1) ; 143
		pac_discharge_safe_att_a[*] = !values.d_nan ; 143
		END
	'analog_limit_status':  BEGIN
		analog_limit_status_a	= 	dblarr(num_records1) ; 144
		analog_limit_status_a[*] = !values.d_nan ; 144
		END
	'analog_lim_pt1_status':  BEGIN
		analog_lim_pt1_status_a	= 	dblarr(num_records1) ; 145
		analog_lim_pt1_status_a[*] = !values.d_nan ; 145
		END
	'analog_lim_pt1_id':  BEGIN
		analog_lim_pt1_id_a	= dblarr(num_records1) ; 146
		analog_lim_pt1_id_a[*] = !values.d_nan ; 146
		END
	'analog_lim_pt1_red':  BEGIN
		analog_lim_pt1_red_a	=	 dblarr(num_records1) ; 147
		analog_lim_pt1_red_a[*] = !values.d_nan ; 147
		END
	'analog_lim_pt2_status':  BEGIN
		analog_lim_pt2_status_a	= dblarr(num_records1) ; 148
		analog_lim_pt2_status_a[*] = !values.d_nan ; 148
		END
	'analog_lim_pt2_id':  BEGIN
		analog_lim_pt2_id_a	= dblarr(num_records1) ; 149
		analog_lim_pt2_id_a[*] = !values.d_nan ; 149
		END
	'analog_lim_pt2_red':  BEGIN
		analog_lim_pt2_red_a	=	 dblarr(num_records1) ; 150
		analog_lim_pt2_red_a[*] = !values.d_nan ; 150
		END
	'analog_lim_pt3_status':  BEGIN
		analog_lim_pt3_status_a	=	 dblarr(num_records1) ; 151
		analog_lim_pt3_status_a[*] = !values.d_nan ; 151
		END
	'analog_lim_pt3_id':  BEGIN
		analog_lim_pt3_id_a	= dblarr(num_records1) ; 152
		analog_lim_pt3_id_a[*] = !values.d_nan ; 152
		END
	'analog_lim_pt3_red':  BEGIN
		analog_lim_pt3_red_a	= dblarr(num_records1) ; 153
		analog_lim_pt3_red_a[*] = !values.d_nan ; 153
		END
	'analog_lim_pt4_status':  BEGIN
		analog_lim_pt4_status_a	= 	dblarr(num_records1) ; 154
		analog_lim_pt4_status_a[*] = !values.d_nan ; 154
		END
	'analog_lim_pt4_id':  BEGIN
		analog_lim_pt4_id_a	= 	dblarr(num_records1) ; 155
		analog_lim_pt4_id_a[*] = !values.d_nan ; 155
		END
	'analog_lim_pt4_red':  BEGIN
		analog_lim_pt4_red_a	= dblarr(num_records1) ; 156
		analog_lim_pt4_red_a[*] = !values.d_nan ; 156
		END
	'analog_lim_pt5_status':  BEGIN
		analog_lim_pt5_status_a	=	 dblarr(num_records1) ; 157
		analog_lim_pt5_status_a[*] = !values.d_nan ; 157
		END
	'analog_lim_pt5_id':  BEGIN
		analog_lim_pt5_id_a	= dblarr(num_records1) ; 158
		analog_lim_pt5_id_a[*] = !values.d_nan ; 158
		END
	'analog_lim_pt5_red':  BEGIN
		analog_lim_pt5_red_a	= dblarr(num_records1) ; 159
		analog_lim_pt5_red_a[*] = !values.d_nan ; 159
		END
	'analog_lim_pt6_status':  BEGIN
		analog_lim_pt6_status_a	=	 dblarr(num_records1) ; 160
		analog_lim_pt6_status_a[*] = !values.d_nan ; 160
		END
	'analog_lim_pt6_id':  BEGIN
		analog_lim_pt6_id_a	=	 dblarr(num_records1) ; 161
		analog_lim_pt6_id_a[*] = !values.d_nan ; 161
		END
	'analog_lim_pt6_red':  BEGIN
		analog_lim_pt6_red_a	= dblarr(num_records1) ; 162
		analog_lim_pt6_red_a[*] = !values.d_nan ; 162
		END
	'analog_lim_pt7_status':  BEGIN
		analog_lim_pt7_status_a	= dblarr(num_records1) ; 163
		analog_lim_pt7_status_a[*] = !values.d_nan ; 163
		END
	'analog_lim_pt7_id':  BEGIN
		analog_lim_pt7_id_a	= 	dblarr(num_records1) ; 164
		analog_lim_pt7_id_a[*] = !values.d_nan ; 164
		END
	'analog_lim_pt7_red':  BEGIN
		analog_lim_pt7_red_a	= 	dblarr(num_records1) ; 165
		analog_lim_pt7_red_a[*] = !values.d_nan ; 165
		END
	'analog_lim_pt8_status':  BEGIN
		analog_lim_pt8_status_a	= dblarr(num_records1) ; 166
		analog_lim_pt8_status_a[*] = !values.d_nan ; 166
		END
	'analog_lim_pt8_id':  BEGIN
		analog_lim_pt8_id_a	=	 dblarr(num_records1) ; 167
		analog_lim_pt8_id_a[*] = !values.d_nan ; 167
		END
	'analog_lim_pt8_red':  BEGIN
		analog_lim_pt8_red_a	= dblarr(num_records1) ; 168
		analog_lim_pt8_red_a[*] = !values.d_nan ; 168
		END
	'analog_lim_pt9_status':  BEGIN
		analog_lim_pt9_status_a	= dblarr(num_records1) ; 169
		analog_lim_pt9_status_a[*] = !values.d_nan ; 169
		END
	'analog_lim_pt9_id':  BEGIN
		analog_lim_pt9_id_a	=	 dblarr(num_records1) ; 170
		analog_lim_pt9_id_a[*] = !values.d_nan ; 170
		END
	'analog_lim_pt9_red':  BEGIN
		analog_lim_pt9_red_a	=	 dblarr(num_records1) ; 171
		analog_lim_pt9_red_a[*] = !values.d_nan ; 171
		END
	'analog_lim_pt10_status':  BEGIN
		analog_lim_pt10_status_a	= dblarr(num_records1) ; 172
		analog_lim_pt10_status_a[*] = !values.d_nan ; 172
		END
	'analog_lim_pt10_id':  BEGIN
		analog_lim_pt10_id_a	= dblarr(num_records1) ; 173
		analog_lim_pt10_id_a[*] = !values.d_nan ; 173
		END
	'analog_lim_pt10_red':  BEGIN
		analog_lim_pt10_red_a	= 	dblarr(num_records1) ; 174
		analog_lim_pt10_red_a[*] = !values.d_nan ; 174
		END
	'analog_lim_pt11_status':  BEGIN
		analog_lim_pt11_status_a	= 	dblarr(num_records1) ; 175
		analog_lim_pt11_status_a[*] = !values.d_nan ; 175
		END
	'analog_lim_pt11_id':  BEGIN
		analog_lim_pt11_id_a	= dblarr(num_records1) ; 176
		analog_lim_pt11_id_a[*] = !values.d_nan ; 176
		END
	'analog_lim_pt11_red':  BEGIN
		analog_lim_pt11_red_a	=	 dblarr(num_records1) ; 177
		analog_lim_pt11_red_a[*] = !values.d_nan ; 177
		END
	'analog_lim_pt12_status':  BEGIN
		analog_lim_pt12_status_a	= dblarr(num_records1) ; 178
		analog_lim_pt12_status_a[*] = !values.d_nan ; 178
		END
	'analog_lim_pt12_id':  BEGIN
		analog_lim_pt12_id_a	= dblarr(num_records1) ; 179
		analog_lim_pt12_id_a[*] = !values.d_nan ; 179
		END
	'analog_lim_pt12_red':  BEGIN
		analog_lim_pt12_red_a	=	 dblarr(num_records1) ; 180
		analog_lim_pt12_red_a[*] = !values.d_nan ; 180
		END
	'analog_lim_pt13_status':  BEGIN
		analog_lim_pt13_status_a	=	 dblarr(num_records1) ; 181
		analog_lim_pt13_status_a[*] = !values.d_nan ; 181
		END
	'analog_lim_pt13_id':  BEGIN
		analog_lim_pt13_id_a	= dblarr(num_records1) ; 182
		analog_lim_pt13_id_a[*] = !values.d_nan ; 182
		END
	'analog_lim_pt13_red':  BEGIN
		analog_lim_pt13_red_a	= dblarr(num_records1) ; 183
		analog_lim_pt13_red_a[*] = !values.d_nan ; 183
		END
	'analog_lim_pt14_status':  BEGIN
		analog_lim_pt14_status_a	= 	dblarr(num_records1) ; 184
		analog_lim_pt14_status_a[*] = !values.d_nan ; 184
		END
	'analog_lim_pt14_id':  BEGIN
		analog_lim_pt14_id_a	= 	dblarr(num_records1) ; 185
		analog_lim_pt14_id_a[*] = !values.d_nan ; 185
		END
	'analog_lim_pt14_red':  BEGIN
		analog_lim_pt14_red_a	= dblarr(num_records1) ; 186
		analog_lim_pt14_red_a[*] = !values.d_nan; 186
		END
	'analog_lim_pt15_status':  BEGIN
		analog_lim_pt15_status_a	=	 dblarr(num_records1) ; 187
		analog_lim_pt15_status_a[*] = !values.d_nan ; 187
		END
	'analog_lim_pt15_id':  BEGIN
		analog_lim_pt15_id_a	= dblarr(num_records1) ; 188
		analog_lim_pt15_id_a[*] = !values.d_nan ; 188
		END
	'analog_lim_pt15_red':  BEGIN
		analog_lim_pt15_red_a	= dblarr(num_records1) ; 189
		analog_lim_pt15_red_a[*] = !values.d_nan ; 189
		END
	'opmode':  BEGIN
		opmode_a	=	 dblarr(num_records1) ; 190
		opmode_a[*] = !values.d_nan ; 190
		END
	'sw_mode':  BEGIN
		sw_mode_a	=	 dblarr(num_records1) ; 191
		sw_mode_a[*] = !values.d_nan ; 191
		END
	'proton_mode':  BEGIN
		proton_mode_a	= dblarr(num_records1) ; 192
		proton_mode_a[*] = !values.d_nan ; 192
		END
	'active_supplyid':  BEGIN
		active_supplyid_a	= dblarr(num_records1) ; 193
		active_supplyid_a[*] = !values.d_nan ; 193
		END
	'supply_coarse_rb':  BEGIN
		supply_coarse_rb_a	= 	dblarr(num_records1) ; 194
		supply_coarse_rb_a[*] = !values.d_nan ; 194
		END
	'supply_fine_rb':  BEGIN
		supply_fine_rb_a	= 	dblarr(num_records1) ; 195
		supply_fine_rb_a[*] = !values.d_nan ; 195
		END
	'swmaint_supplystat':  BEGIN
		swmaint_supplystat_a	= dblarr(num_records1) ; 196
		swmaint_supplystat_a[*] = !values.d_nan ; 196
		END
	'swmaint_pac_coarse':  BEGIN
		swmaint_pac_coarse_a	=	 dblarr(num_records1) ; 197
		swmaint_pac_coarse_a[*] = !values.d_nan ; 197
		END
	'swmaint_pac_fine':  BEGIN
		swmaint_pac_fine_a	= dblarr(num_records1) ; 198		
		swmaint_pac_fine_a[*] = !values.d_nan ; 198
		END
	ENDCASE
	ENDFOR
    	END 
    'B': BEGIN 

	FOR kk=0,n_elements(products)-1 DO BEGIN
	CASE products[kk] OF
	
	'dpu_sw_version':  BEGIN
		dpu_sw_version_b	= dblarr(num_records1) ; 0
		dpu_sw_version_b[*] = !values.d_nan ; 0
		END
	'cmd_rcvd_cnt':  BEGIN
		cmd_rcvd_cnt_b	= dblarr(num_records1) ; 1
		cmd_rcvd_cnt_b[*] = !values.d_nan ; 1
		END
	'cmd_exec_cnt':  BEGIN
		cmd_exec_cnt_b	= dblarr(num_records1) ; 2
		cmd_exec_cnt_b[*] = !values.d_nan ; 2
		END
	'cmd_total_err_cnt':  BEGIN
		cmd_total_err_cnt_b	= dblarr(num_records1) ; 3
		cmd_total_err_cnt_b[*] = !values.d_nan ; 3
		END
	'cmd_parity_err_cnt':  BEGIN
		cmd_parity_err_cnt_b	= dblarr(num_records1) ; 4
		cmd_parity_err_cnt_b[*] = !values.d_nan ; 4
		END
	'cmd_frame_err_cnt':  BEGIN
		cmd_frame_err_cnt_b	= dblarr(num_records1) ; 5
		cmd_frame_err_cnt_b[*] = !values.d_nan ; 5
		END
	'cmd_illegal_err_cnt':  BEGIN
		cmd_illegal_err_cnt_b	= dblarr(num_records1) ; 6
		cmd_illegal_err_cnt_b[*] = !values.d_nan ; 6
		END
	'ic_status_reg':  BEGIN
		ic_status_reg_b	= dblarr(num_records1) ; 7
		ic_status_reg_b[*] = !values.d_nan ; 7
		END
	'ic_ctl_reg':  BEGIN
		ic_ctl_reg_b	= dblarr(num_records1) ; 8
		ic_ctl_reg_b[*] = !values.d_nan ; 8
		END
	'lbc_reset_ctl':  BEGIN
		lbc_reset_ctl_b	= dblarr(num_records1) ; 9
		lbc_reset_ctl_b[*] = !values.d_nan ; 9
		END
	'lbc_mode_ctl':  BEGIN
		lbc_mode_ctl_b	= dblarr(num_records1) ; 10
		lbc_mode_ctl_b[*] = !values.d_nan ; 10
		END
	'lbc_logic_ctl_a':  BEGIN
		lbc_logic_ctl_a_b	= dblarr(num_records1) ; 11
		lbc_logic_ctl_a_b[*] = !values.d_nan ; 11
		END
	'lbc_logic_ctl_b':  BEGIN
		lbc_logic_ctl_b_b	= dblarr(num_records1) ; 12
		lbc_logic_ctl_b_b[*] = !values.d_nan ; 12
		END
	'lbc_tac_pwr_ctl':  BEGIN
		lbc_tac_pwr_ctl_b	= dblarr(num_records1) ; 13
		lbc_tac_pwr_ctl_b[*] = !values.d_nan ; 13
		END
	'lbc_reg_seq':  BEGIN
		lbc_reg_seq_b	= dblarr(num_records1) ; 14
		lbc_reg_seq_b[*] = !values.d_nan ; 14
		END
	'lbc_event_ctl':  BEGIN
		lbc_event_ctl_b	= dblarr(num_records1) ; 15
		lbc_event_ctl_b[*] = !values.d_nan ; 15
		END
	'lbc_rlim_ch':  BEGIN
		lbc_rlim_ch_b	= dblarr(num_records1) ; 16
		lbc_rlim_ch_b[*] = !values.d_nan ; 16
		END
	'lbc_rlim_hi':  BEGIN
		lbc_rlim_hi_b	= dblarr(num_records1) ; 17
		lbc_rlim_hi_b[*] = !values.d_nan ; 17
		END
	'lbc_rlim_lo':  BEGIN
		lbc_rlim_lo_b	= dblarr(num_records1) ; 18
		lbc_rlim_lo_b[*] = !values.d_nan ; 18
		END
	'lbe_pos_ctrl':  BEGIN
		lbe_pos_ctrl_b	= dblarr(num_records1) ; 19
		lbe_pos_ctrl_b[*] = !values.d_nan ; 19
		END
	'lbe_pos_disable0':  BEGIN
		lbe_pos_disable0_b	= dblarr(num_records1) ; 20
		lbe_pos_disable0_b[*] = !values.d_nan ; 20
		END
	'lbe_pos_disable2':  BEGIN
		lbe_pos_disable2_b	= dblarr(num_records1) ; 21
		lbe_pos_disable2_b[*] = !values.d_nan ; 21
		END
	'lbe_tac0_ctrl':  BEGIN
		lbe_tac0_ctrl_b	= dblarr(num_records1) ; 22
		lbe_tac0_ctrl_b[*] = !values.d_nan ; 22
		END
	'lbe_tac0_dac':  BEGIN
		lbe_tac0_dac_b	= dblarr(num_records1) ; 23
		lbe_tac0_dac_b[*] = !values.d_nan ; 23
		END
	'lbe_tac2_ctrl':  BEGIN
		lbe_tac2_ctrl_b	= dblarr(num_records1) ; 24
		lbe_tac2_ctrl_b[*] = !values.d_nan ; 24
		END
	'lbe_tac2_dac':  BEGIN
		lbe_tac2_dac_b	= dblarr(num_records1) ; 25
		lbe_tac2_dac_b[*] = !values.d_nan ; 25
		END
	'lbe_tac0_under_hi':  BEGIN
		lbe_tac0_under_hi_b	= dblarr(num_records1) ; 26
		lbe_tac0_under_hi_b[*] = !values.d_nan ; 26
		END
	'lbe_tac0_under_lo':  BEGIN
		lbe_tac0_under_lo_b	= dblarr(num_records1) ; 27
		lbe_tac0_under_lo_b[*] = !values.d_nan ; 27
		END
	'lbe_tac0_over_hi':  BEGIN
		lbe_tac0_over_hi_b	= dblarr(num_records1) ; 28
		lbe_tac0_over_hi_b[*] = !values.d_nan ; 28
		END
	'lbe_tac0_over_lo':  BEGIN
		lbe_tac0_over_lo_b	= dblarr(num_records1) ; 29
		lbe_tac0_over_lo_b[*] = !values.d_nan ; 29
		END
	'lbe_tac2_under_hi':  BEGIN
		lbe_tac2_under_hi_b	= dblarr(num_records1) ; 30
		lbe_tac2_under_hi_b[*] = !values.d_nan ; 30
		END
	'lbe_tac2_under_lo':  BEGIN
		lbe_tac2_under_lo_b	= dblarr(num_records1) ; 31
		lbe_tac2_under_lo_b[*] = !values.d_nan ; 31
		END
	'lbe_tac2_over_hi':  BEGIN
		lbe_tac2_over_hi_b	= dblarr(num_records1) ; 32
		lbe_tac2_over_hi_b[*] = !values.d_nan ; 32
		END
	'lbe_tac2_over_lo':  BEGIN
		lbe_tac2_over_lo_b	= dblarr(num_records1) ; 33
		lbe_tac2_over_lo_b[*] = !values.d_nan ; 33
		END
	'lbe_ssd_ctrl':  BEGIN
		lbe_ssd_ctrl_b	= dblarr(num_records1) ; 34
		lbe_ssd_ctrl_b[*] = !values.d_nan ; 34
		END
	'lbe_ssd_cmd_h':  BEGIN
		lbe_ssd_cmd_h_b	= dblarr(num_records1) ; 35
		lbe_ssd_cmd_h_b[*] = !values.d_nan ; 35
		END
	'lbe_ssd_cmd_l':  BEGIN
		lbe_ssd_cmd_l_b	= dblarr(num_records1) ; 36
		lbe_ssd_cmd_l_b[*] = !values.d_nan ; 36
		END
	'lbe_ssd_dis':  BEGIN
		lbe_ssd_dis_b	= dblarr(num_records1) ; 37
		lbe_ssd_dis_b[*] = !values.d_nan ; 37
		END
	'lbe_ssd_under_hi':  BEGIN
		lbe_ssd_under_hi_b	= dblarr(num_records1) ; 38
		lbe_ssd_under_hi_b[*] = !values.d_nan ; 38
		END
	'lbe_ssd_under_lo':  BEGIN
		lbe_ssd_under_lo_b	= dblarr(num_records1) ; 39
		lbe_ssd_under_lo_b[*] = !values.d_nan ; 39
		END
	'lbe_ssd_over_hi':  BEGIN
		lbe_ssd_over_hi_b	= dblarr(num_records1) ; 40
		lbe_ssd_over_hi_b[*] = !values.d_nan ; 40
		END
	'lbe_ssd_over_lo':  BEGIN
		lbe_ssd_over_lo_b	= dblarr(num_records1) ; 41
		lbe_ssd_over_lo_b[*] = !values.d_nan ; 41
		END
	'lbe_sel_ctrl':  BEGIN
		lbe_sel_ctrl_b	= dblarr(num_records1) ; 42
		lbe_sel_ctrl_b[*] = !values.d_nan ; 42
		END
	'lbe_trig_mode':  BEGIN
		lbe_trig_mode_b	= dblarr(num_records1) ; 43
		lbe_trig_mode_b[*] = !values.d_nan ; 43
		END
	'lbe_esa_step':  BEGIN
		lbe_esa_step_b	= dblarr(num_records1) ; 44
		lbe_esa_step_b[*] = !values.d_nan ; 44
		END
	'lbe_defl_step':  BEGIN
		lbe_defl_step_b	= dblarr(num_records1) ; 45
		lbe_defl_step_b[*] = !values.d_nan ; 45
		END
	'lbe_ssd_offset0':  BEGIN
		lbe_ssd_offset0_b	= dblarr(num_records1) ; 46
		lbe_ssd_offset0_b[*] = !values.d_nan ; 46
		END
	'lbe_ssd_offset1':  BEGIN
		lbe_ssd_offset1_b	= dblarr(num_records1) ; 47
		lbe_ssd_offset1_b[*] = !values.d_nan ; 47
		END
	'lbe_ssd_offset2':  BEGIN
		lbe_ssd_offset2_b	= dblarr(num_records1) ; 48
		lbe_ssd_offset2_b[*] = !values.d_nan ; 48
		END
	'lbe_ssd_offset3':  BEGIN
		lbe_ssd_offset3_b	= dblarr(num_records1) ; 49
		lbe_ssd_offset3_b[*] = !values.d_nan ; 49
		END
	'lbe_ssd_offset4':  BEGIN
		lbe_ssd_offset4_b	= dblarr(num_records1) ; 50
		lbe_ssd_offset4_b[*] = !values.d_nan ; 50
		END
	'lbe_ssd_offset5':  BEGIN
		lbe_ssd_offset5_b	= dblarr(num_records1) ; 51
		lbe_ssd_offset5_b[*] = !values.d_nan; 51
		END
	'lbe_ssd_offset6':  BEGIN
		lbe_ssd_offset6_b	= dblarr(num_records1) ; 52
		lbe_ssd_offset6_b[*] = !values.d_nan ; 52
		END
	'lbe_ssd_offset7':  BEGIN
		lbe_ssd_offset7_b	= dblarr(num_records1) ; 53
		lbe_ssd_offset7_b[*] = !values.d_nan ; 53
		END
	'lbe_stim_enable':  BEGIN
		lbe_stim_enable_b	= dblarr(num_records1) ; 54
		lbe_stim_enable_b[*] = !values.d_nan ; 54
		END
	'lbe_stim_freq_hi':  BEGIN
		lbe_stim_freq_hi_b	= dblarr(num_records1) ; 55
		lbe_stim_freq_hi_b[*] = !values.d_nan ; 55
		END
	'lbe_stim_freq_lo':  BEGIN
		lbe_stim_freq_lo_b	= dblarr(num_records1) ; 56
		lbe_stim_freq_lo_b[*] = !values.d_nan ; 56
		END
	'lbe_sel0_window':  BEGIN
		lbe_sel0_window_b	= dblarr(num_records1) ; 57
		lbe_sel0_window_b[*] = !values.d_nan ; 57
		END
	'lbe_sel2_window':  BEGIN
		lbe_sel2_window_b	= dblarr(num_records1) ; 58
		lbe_sel2_window_b[*] = !values.d_nan ; 58
		END
	'lbe_ssd_offset8':  BEGIN
		lbe_ssd_offset8_b	= dblarr(num_records1) ; 59
		lbe_ssd_offset8_b[*] = !values.d_nan ; 59
		END
	'lbe_ssd_offset9':  BEGIN
		lbe_ssd_offset9_b	= dblarr(num_records1) ; 60
		lbe_ssd_offset9_b[*] = !values.d_nan ; 60
		END
	'lbe_ssd_offset10':  BEGIN
		lbe_ssd_offset10_b	= dblarr(num_records1) ; 61
		lbe_ssd_offset10_b[*] = !values.d_nan ; 61
		END
	'lbe_ssd_offset11':  BEGIN
		lbe_ssd_offset11_b	= dblarr(num_records1) ; 62
		lbe_ssd_offset11_b[*] = !values.d_nan ; 62
		END
	'lbe_ssd_offset12':  BEGIN
		lbe_ssd_offset12_b	= dblarr(num_records1) ; 63
		lbe_ssd_offset12_b[*] = !values.d_nan ; 63
		END
	'lbe_ssd_offset13':  BEGIN
		lbe_ssd_offset13_b	= dblarr(num_records1) ; 64
		lbe_ssd_offset13_b[*] = !values.d_nan ; 64
		END
	'lbe_ssd_offset14':  BEGIN
		lbe_ssd_offset14_b	= dblarr(num_records1) ; 65
		lbe_ssd_offset14_b[*] = !values.d_nan ; 65
		END
	'lbe_ssd_offset15':  BEGIN
		lbe_ssd_offset15_b	= dblarr(num_records1) ; 66
		lbe_ssd_offset15_b[*] = !values.d_nan ; 66
		END
	'lbe_pos_ra':  BEGIN
		lbe_pos_ra_b	=	 dblarr(num_records1) ; 67
		lbe_pos_ra_b[*] = !values.d_nan ; 67
		END
	'dacm_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
		dacm_pac_mcp_ssd_ctl_mcp_limit_b	= dblarr(num_records1) ; 68
		dacm_pac_mcp_ssd_ctl_mcp_limit_b[*] = !values.d_nan ; 68
		END
	'dacm_pac_limit':  BEGIN
		dacm_pac_limit_b	= dblarr(num_records1) ; 69
		dacm_pac_limit_b[*] = !values.d_nan ; 69
		END
	'dacm_esa_ctl':  BEGIN
		dacm_esa_ctl_b	=	 dblarr(num_records1) ; 70
		dacm_esa_ctl_b[*] = !values.d_nan ; 70
		END
	'dacm_s_ch_ctl':  BEGIN
		dacm_s_ch_ctl_b	=	 dblarr(num_records1) ; 71
		dacm_s_ch_ctl_b[*] = !values.d_nan ; 71
		END
	'dacm_dfl_1_ctl':  BEGIN
		dacm_dfl_1_ctl_b	= dblarr(num_records1) ; 72
		dacm_dfl_1_ctl_b[*] = !values.d_nan ; 72
		END
	'dacm_dfl_2_ctl':  BEGIN
		dacm_dfl_2_ctl_b	= dblarr(num_records1) ; 73
		dacm_dfl_2_ctl_b[*] = !values.d_nan ; 73
		END
	'dacm_control':  BEGIN
		dacm_control_b	= 	dblarr(num_records1) ; 74
		dacm_control_b[*] = !values.d_nan ; 74
		END
	'dacm_status':  BEGIN
		dacm_status_b	= 	dblarr(num_records1) ; 75
		dacm_status_b[*] = !values.d_nan ; 75
		END
	'dacm_adc_lu_ctr':  BEGIN
		dacm_adc_lu_ctr_b	= dblarr(num_records1) ; 76
		dacm_adc_lu_ctr_b[*] = !values.d_nan ; 76
		END
	'dacm_dfl_1_offset':  BEGIN
		dacm_dfl_1_offset_b	=	 dblarr(num_records1) ; 77
		dacm_dfl_1_offset_b[*] = !values.d_nan ; 77
		END
	'dacm_dfl_2_offset':  BEGIN
		dacm_dfl_2_offset_b	= dblarr(num_records1) ; 78
		dacm_dfl_2_offset_b[*] = !values.d_nan ; 78
		END
	'dacm_pac_coarse':  BEGIN
		dacm_pac_coarse_b	= dblarr(num_records1) ; 79
		dacm_pac_coarse_b[*] = !values.d_nan ; 79
		END
	'dacm_pac_fine':  BEGIN
		dacm_pac_fine_b	=	 dblarr(num_records1) ; 80
		dacm_pac_fine_b[*] = !values.d_nan ; 80
		END
	'dacm_pac_cm_fs_coarse':  BEGIN
		dacm_pac_cm_fs_coarse_b	=	 dblarr(num_records1) ; 81
		dacm_pac_cm_fs_coarse_b[*] = !values.d_nan ; 81
		END
	'dacm_pac_cm_fs_fine':  BEGIN
		dacm_pac_cm_fs_fine_b	= dblarr(num_records1) ; 82
		dacm_pac_cm_fs_fine_b[*] = !values.d_nan ; 82
		END
	'dacm_mcp_coarse':  BEGIN
		dacm_mcp_coarse_b	= dblarr(num_records1) ; 83
		dacm_mcp_coarse_b[*] = !values.d_nan ; 83
		END
	'dacm_mcp_fine':  BEGIN
		dacm_mcp_fine_b	= 	dblarr(num_records1) ; 84
		dacm_mcp_fine_b[*] = !values.d_nan ; 84
		END
	'dacm_mcp_cm_fs_coarse':  BEGIN
		dacm_mcp_cm_fs_coarse_b	= 	dblarr(num_records1) ; 85
		dacm_mcp_cm_fs_coarse_b[*] = !values.d_nan ; 85
		END
	'dacm_mcp_cm_fs_fine':  BEGIN
		dacm_mcp_cm_fs_fine_b	= dblarr(num_records1) ; 86
		dacm_mcp_cm_fs_fine_b[*] = !values.d_nan ; 86
		END
	'dacm_ssd_coarse':  BEGIN
		dacm_ssd_coarse_b	=	dblarr(num_records1) ; 87
		dacm_ssd_coarse_b[*] = !values.d_nan ; 87
		END
	'dacm_ssd_fine':  BEGIN
		dacm_ssd_fine_b	= dblarr(num_records1) ; 88
		dacm_ssd_fine_b[*] = !values.d_nan ; 88
		END
	'dacm_esa_coarse':  BEGIN
		dacm_esa_coarse_b	= dblarr(num_records1) ; 89
		dacm_esa_coarse_b[*] = !values.d_nan ; 89
		END
	'dacm_esa_fine':  BEGIN
		dacm_esa_fine_b	=	 dblarr(num_records1) ; 90
		dacm_esa_fine_b[*] = !values.d_nan ; 90
		END
	'dacm_s_ch_coarse':  BEGIN
		dacm_s_ch_coarse_b	=	 dblarr(num_records1) ; 91
		dacm_s_ch_coarse_b[*] = !values.d_nan ; 91
		END
	'dacm_s_ch_fine':  BEGIN
		dacm_s_ch_fine_b	= dblarr(num_records1) ; 92
		dacm_s_ch_fine_b[*] = !values.d_nan ; 92
		END
	'dacm_s_ch_tap_coarse':  BEGIN
		dacm_s_ch_tap_coarse_b	= dblarr(num_records1) ; 93
		dacm_s_ch_tap_coarse_b[*] = !values.d_nan ; 93
		END
	'dacm_s_ch_tap_fine':  BEGIN
		dacm_s_ch_tap_fine_b	= 	dblarr(num_records1) ; 94
		dacm_s_ch_tap_fine_b[*] = !values.d_nan ; 94
		END
	'dacm_dfl_1_coarse':  BEGIN
		dacm_dfl_1_coarse_b	= 	dblarr(num_records1) ; 95
		dacm_dfl_1_coarse_b[*] = !values.d_nan ; 95
		END
	'dacm_dfl_1_fine':  BEGIN
		dacm_dfl_1_fine_b	= dblarr(num_records1) ; 96
		dacm_dfl_1_fine_b[*] = !values.d_nan ; 96
		END
	'dacm_dfl_1_dfine':  BEGIN
		dacm_dfl_1_dfine_b	=	 dblarr(num_records1) ; 97
		dacm_dfl_1_dfine_b[*] = !values.d_nan ; 97
		END
	'dacm_dfl_2_coarse':  BEGIN
		dacm_dfl_2_coarse_b	= dblarr(num_records1) ; 98
		dacm_dfl_2_coarse_b[*] = !values.d_nan ; 98
		END
	'dacm_dfl_2_fine':  BEGIN
		dacm_dfl_2_fine_b	= dblarr(num_records1) ; 99
		dacm_dfl_2_fine_b[*] = !values.d_nan ; 99
		END
	'dacm_dfl_2_dfine':  BEGIN
		dacm_dfl_2_dfine_b	=	 dblarr(num_records1) ; 100
		dacm_dfl_2_dfine_b[*] = !values.d_nan ; 100
		END
	'dacr_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
		dacr_pac_mcp_ssd_ctl_mcp_limit_b	=	 dblarr(num_records1) ; 101
		dacr_pac_mcp_ssd_ctl_mcp_limit_b[*] = !values.d_nan ; 101
		END
	'dacr_pac_limit':  BEGIN
		dacr_pac_limit_b	= dblarr(num_records1) ; 102
		dacr_pac_limit_b[*] = !values.d_nan ; 102
		END
	'dacr_esa_ctl':  BEGIN
		dacr_esa_ctl_b	= dblarr(num_records1) ; 103
		dacr_esa_ctl_b[*] = !values.d_nan ; 103
		END
	'dacr_s_ch_ctl':  BEGIN
		dacr_s_ch_ctl_b	= 	dblarr(num_records1) ; 104
		dacr_s_ch_ctl_b[*] = !values.d_nan ; 104
		END
	'dacr_dfl_1_ctl':  BEGIN
		dacr_dfl_1_ctl_b	= 	dblarr(num_records1) ; 105
		dacr_dfl_1_ctl_b[*] = !values.d_nan ; 105
		END
	'dacr_dfl_2_ctl':  BEGIN
		dacr_dfl_2_ctl_b	= dblarr(num_records1) ; 106
		dacr_dfl_2_ctl_b[*] = !values.d_nan ; 106
		END
	'dacr_control':  BEGIN
		dacr_control_b	=	 dblarr(num_records1) ; 107
		dacr_control_b[*] = !values.d_nan ; 107
		END
	'dacr_status':  BEGIN
		dacr_status_b	= dblarr(num_records1) ; 108
		dacr_status_b[*] = !values.d_nan ; 108
		END
	'dacr_adc_lu_ctr':  BEGIN
		dacr_adc_lu_ctr_b	= dblarr(num_records1) ; 109
		dacr_adc_lu_ctr_b[*] = !values.d_nan ; 109
		END
	'dacr_dfl_1_offset':  BEGIN
		dacr_dfl_1_offset_b	=	 dblarr(num_records1) ; 110
		dacr_dfl_1_offset_b[*] = !values.d_nan ; 110
		END
	'dacr_dfl_2_offset':  BEGIN
		dacr_dfl_2_offset_b	=	 dblarr(num_records1) ; 111
		dacr_dfl_2_offset_b[*] = !values.d_nan ; 111
		END
	'dacr_c_pac_coarse':  BEGIN
		dacr_c_pac_coarse_b	= dblarr(num_records1) ; 112
		dacr_c_pac_coarse_b[*] = !values.d_nan ; 112
		END
	'dacr_c_pac_fine':  BEGIN
		dacr_c_pac_fine_b	= dblarr(num_records1) ; 113
		dacr_c_pac_fine_b[*] = !values.d_nan ; 113
		END
	'dacr_c_pac_cm_fs_coarse':  BEGIN
		dacr_c_pac_cm_fs_coarse_b	= 	dblarr(num_records1) ; 114
		dacr_c_pac_cm_fs_coarse_b[*] = !values.d_nan ; 114
		END
	'dacr_c_pac_cm_fs_fine':  BEGIN
		dacr_c_pac_cm_fs_fine_b	= 	dblarr(num_records1) ; 115
		dacr_c_pac_cm_fs_fine_b[*] = !values.d_nan ; 115
		END
	'dacr_c_mcp_coarse':  BEGIN
		dacr_c_mcp_coarse_b	= dblarr(num_records1) ; 116
		dacr_c_mcp_coarse_b[*] = !values.d_nan ; 116
		END
	'dacr_c_mcp_fine':  BEGIN
		dacr_c_mcp_fine_b	=	 dblarr(num_records1) ; 117
		dacr_c_mcp_fine_b[*] = !values.d_nan ; 117
		END
	'dacr_c_mcp_cm_fs_coarse':  BEGIN
		dacr_c_mcp_cm_fs_coarse_b	= dblarr(num_records1) ; 118
		dacr_c_mcp_cm_fs_coarse_b[*] = !values.d_nan ; 118
		END
	'dacr_c_mcp_cm_fs_fine':  BEGIN
		dacr_c_mcp_cm_fs_fine_b	= dblarr(num_records1) ; 119
		dacr_c_mcp_cm_fs_fine_b[*] = !values.d_nan ; 119
		END
	'dacr_c_ssd_coarse':  BEGIN
		dacr_c_ssd_coarse_b	=	 dblarr(num_records1) ; 120
		dacr_c_ssd_coarse_b[*] = !values.d_nan ; 120
		END
	'dacr_c_ssd_fine':  BEGIN
		dacr_c_ssd_fine_b	=	 dblarr(num_records1) ; 121
		dacr_c_ssd_fine_b[*] = !values.d_nan ; 121
		END
	'dacr_c_esa_coarse':  BEGIN
		dacr_c_esa_coarse_b	= dblarr(num_records1) ; 122
		dacr_c_esa_coarse_b[*] = !values.d_nan ; 122
		END
	'dacr_c_esa_fine':  BEGIN
		dacr_c_esa_fine_b	= dblarr(num_records1) ; 123
		dacr_c_esa_fine_b[*] = !values.d_nan ; 123
		END
	'dacr_c_s_ch_coarse':  BEGIN
		dacr_c_s_ch_coarse_b	= 	dblarr(num_records1) ; 124
		dacr_c_s_ch_coarse_b[*] = !values.d_nan ; 124
		END
	'dacr_c_s_ch_fine':  BEGIN
		dacr_c_s_ch_fine_b	= 	dblarr(num_records1) ; 125
		dacr_c_s_ch_fine_b[*] = !values.d_nan ; 125
		END
	'dacr_c_s_ch_tap_coarse':  BEGIN
		dacr_c_s_ch_tap_coarse_b	= dblarr(num_records1) ; 126
		dacr_c_s_ch_tap_coarse_b[*] = !values.d_nan ; 126
		END
	'dacr_c_s_ch_tap_fine':  BEGIN
		dacr_c_s_ch_tap_fine_b	=	 dblarr(num_records1) ; 127
		dacr_c_s_ch_tap_fine_b[*] = !values.d_nan ; 127
		END
	'dacr_dfl_1_coarse':  BEGIN
		dacr_dfl_1_coarse_b	= dblarr(num_records1) ; 128
		dacr_dfl_1_coarse_b[*] = !values.d_nan ; 128
		END
	'dacr_dfl_1_fine':  BEGIN
		dacr_dfl_1_fine_b	= dblarr(num_records1) ; 129
		dacr_dfl_1_fine_b[*] = !values.d_nan; 129
		END
	'dacr_dfl_1_dfine':  BEGIN
		dacr_dfl_1_dfine_b	=	 dblarr(num_records1) ; 130
		dacr_dfl_1_dfine_b[*] = !values.d_nan; 130
		END
	'dacr_dfl_2_coarse':  BEGIN
		dacr_dfl_2_coarse_b	=	 dblarr(num_records1) ; 131
		dacr_dfl_2_coarse_b[*] = !values.d_nan ; 131
		END
	'dacr_dfl_2_fine':  BEGIN
		dacr_dfl_2_fine_b	= dblarr(num_records1) ; 132
		dacr_dfl_2_fine_b[*] = !values.d_nan; 132
		END
	'dacr_dfl_2_dfine':  BEGIN
		dacr_dfl_2_dfine_b	= dblarr(num_records1) ; 133
		dacr_dfl_2_dfine_b[*] = !values.d_nan ; 133
		END
	'sw_status1':  BEGIN
		sw_status1_b	= 	dblarr(num_records1) ; 134
		sw_status1_b[*] = !values.d_nan ; 134
		END
	'sw_hv_ramp_status':  BEGIN
		sw_hv_ramp_status_b	= 	dblarr(num_records1) ; 135
		sw_hv_ramp_status_b[*] = !values.d_nan; 135
		END
	'sw_hv_enable_status':  BEGIN
		sw_hv_enable_status_b	= dblarr(num_records1) ; 136
		sw_hv_enable_status_b[*] = !values.d_nan ; 136
		END
	'sw_hv_limit_status':  BEGIN
		sw_hv_limit_status_b	=	 dblarr(num_records1) ; 137
		sw_hv_limit_status_b[*] = !values.d_nan ; 137
		END
	'sw_eepromchksum':  BEGIN
		sw_eepromchksum_b	= dblarr(num_records1) ; 138
		sw_eepromchksum_b[*] = !values.d_nan ; 138
		END
	'hkp_sampletime':  BEGIN
		hkp_sampletime_b	= dblarr(num_records1) ; 139
		hkp_sampletime_b[*] = !values.d_nan ; 139
		END
	'pac_discharge_status':  BEGIN
		pac_discharge_status_b	=	 dblarr(num_records1) ; 140
		pac_discharge_status_b[*] = !values.d_nan ; 140
		END
	'pac_discharge_cnt':  BEGIN
		pac_discharge_cnt_b	=	 dblarr(num_records1) ; 141
		pac_discharge_cnt_b[*] = !values.d_nan ; 141
		END
	'pac_discharge_conseccntr':  BEGIN
		pac_discharge_conseccntr_b	= dblarr(num_records1) ; 142
		pac_discharge_conseccntr_b[*] = !values.d_nan ; 142
		END
	'pac_discharge_safe_att':  BEGIN
		pac_discharge_safe_att_b	= dblarr(num_records1) ; 143
		pac_discharge_safe_att_b[*] = !values.d_nan ; 143
		END
	'analog_limit_status':  BEGIN
		analog_limit_status_b	= 	dblarr(num_records1) ; 144
		analog_limit_status_b[*] = !values.d_nan ; 144
		END
	'analog_lim_pt1_status':  BEGIN
		analog_lim_pt1_status_b	= 	dblarr(num_records1) ; 145
		analog_lim_pt1_status_b[*] = !values.d_nan ; 145
		END
	'analog_lim_pt1_id':  BEGIN
		analog_lim_pt1_id_b	= dblarr(num_records1) ; 146
		analog_lim_pt1_id_b[*] = !values.d_nan ; 146
		END
	'analog_lim_pt1_red':  BEGIN
		analog_lim_pt1_red_b	=	 dblarr(num_records1) ; 147
		analog_lim_pt1_red_b[*] = !values.d_nan ; 147
		END
	'analog_lim_pt2_status':  BEGIN
		analog_lim_pt2_status_b	= dblarr(num_records1) ; 148
		analog_lim_pt2_status_b[*] = !values.d_nan ; 148
		END
	'analog_lim_pt2_id':  BEGIN
		analog_lim_pt2_id_b	= dblarr(num_records1) ; 149
		analog_lim_pt2_id_b[*] = !values.d_nan ; 149
		END
	'analog_lim_pt2_red':  BEGIN
		analog_lim_pt2_red_b	=	 dblarr(num_records1) ; 150
		analog_lim_pt2_red_b[*] = !values.d_nan ; 150
		END
	'analog_lim_pt3_status':  BEGIN
		analog_lim_pt3_status_b	=	 dblarr(num_records1) ; 151
		analog_lim_pt3_status_b[*] = !values.d_nan ; 151
		END
	'analog_lim_pt3_id':  BEGIN
		analog_lim_pt3_id_b	= dblarr(num_records1) ; 152
		analog_lim_pt3_id_b[*] = !values.d_nan ; 152
		END
	'analog_lim_pt3_red':  BEGIN
		analog_lim_pt3_red_b	= dblarr(num_records1) ; 153
		analog_lim_pt3_red_b[*] = !values.d_nan ; 153
		END
	'analog_lim_pt4_status':  BEGIN
		analog_lim_pt4_status_b	= 	dblarr(num_records1) ; 154
		analog_lim_pt4_status_b[*] = !values.d_nan ; 154
		END
	'analog_lim_pt4_id':  BEGIN
		analog_lim_pt4_id_b	= 	dblarr(num_records1) ; 155
		analog_lim_pt4_id_b[*] = !values.d_nan ; 155
		END
	'analog_lim_pt4_red':  BEGIN
		analog_lim_pt4_red_b	= dblarr(num_records1) ; 156
		analog_lim_pt4_red_b[*] = !values.d_nan ; 156
		END
	'analog_lim_pt5_status':  BEGIN
		analog_lim_pt5_status_b	=	 dblarr(num_records1) ; 157
		analog_lim_pt5_status_b[*] = !values.d_nan ; 157
		END
	'analog_lim_pt5_id':  BEGIN
		analog_lim_pt5_id_b	= dblarr(num_records1) ; 158
		analog_lim_pt5_id_b[*] = !values.d_nan ; 158
		END
	'analog_lim_pt5_red':  BEGIN
		analog_lim_pt5_red_b	= dblarr(num_records1) ; 159
		analog_lim_pt5_red_b[*] = !values.d_nan ; 159
		END
	'analog_lim_pt6_status':  BEGIN
		analog_lim_pt6_status_b	=	 dblarr(num_records1) ; 160
		analog_lim_pt6_status_b[*] = !values.d_nan ; 160
		END
	'analog_lim_pt6_id':  BEGIN
		analog_lim_pt6_id_b	=	 dblarr(num_records1) ; 161
		analog_lim_pt6_id_b[*] = !values.d_nan ; 161
		END
	'analog_lim_pt6_red':  BEGIN
		analog_lim_pt6_red_b	= dblarr(num_records1) ; 162
		analog_lim_pt6_red_b[*] = !values.d_nan ; 162
		END
	'analog_lim_pt7_status':  BEGIN
		analog_lim_pt7_status_b	= dblarr(num_records1) ; 163
		analog_lim_pt7_status_b[*] = !values.d_nan ; 163
		END
	'analog_lim_pt7_id':  BEGIN
		analog_lim_pt7_id_b	= 	dblarr(num_records1) ; 164
		analog_lim_pt7_id_b[*] = !values.d_nan ; 164
		END
	'analog_lim_pt7_red':  BEGIN
		analog_lim_pt7_red_b	= 	dblarr(num_records1) ; 165
		analog_lim_pt7_red_b[*] = !values.d_nan ; 165
		END
	'analog_lim_pt8_status':  BEGIN
		analog_lim_pt8_status_b	= dblarr(num_records1) ; 166
		analog_lim_pt8_status_b[*] = !values.d_nan ; 166
		END
	'analog_lim_pt8_id':  BEGIN
		analog_lim_pt8_id_b	=	 dblarr(num_records1) ; 167
		analog_lim_pt8_id_b[*] = !values.d_nan ; 167
		END
	'analog_lim_pt8_red':  BEGIN
		analog_lim_pt8_red_b	= dblarr(num_records1) ; 168
		analog_lim_pt8_red_b[*] = !values.d_nan ; 168
		END
	'analog_lim_pt9_status':  BEGIN
		analog_lim_pt9_status_b	= dblarr(num_records1) ; 169
		analog_lim_pt9_status_b[*] = !values.d_nan ; 169
		END
	'analog_lim_pt9_id':  BEGIN
		analog_lim_pt9_id_b	=	 dblarr(num_records1) ; 170
		analog_lim_pt9_id_b[*] = !values.d_nan ; 170
		END
	'analog_lim_pt9_red':  BEGIN
		analog_lim_pt9_red_b	=	 dblarr(num_records1) ; 171
		analog_lim_pt9_red_b[*] = !values.d_nan ; 171
		END
	'analog_lim_pt10_status':  BEGIN
		analog_lim_pt10_status_b	= dblarr(num_records1) ; 172
		analog_lim_pt10_status_b[*] = !values.d_nan ; 172
		END
	'analog_lim_pt10_id':  BEGIN
		analog_lim_pt10_id_b	= dblarr(num_records1) ; 173
		analog_lim_pt10_id_b[*] = !values.d_nan ; 173
		END
	'analog_lim_pt10_red':  BEGIN
		analog_lim_pt10_red_b	= 	dblarr(num_records1) ; 174
		analog_lim_pt10_red_b[*] = !values.d_nan ; 174
		END
	'analog_lim_pt11_status':  BEGIN
		analog_lim_pt11_status_b	= 	dblarr(num_records1) ; 175
		analog_lim_pt11_status_b[*] = !values.d_nan ; 175
		END
	'analog_lim_pt11_id':  BEGIN
		analog_lim_pt11_id_b	= dblarr(num_records1) ; 176
		analog_lim_pt11_id_b[*] = !values.d_nan ; 176
		END
	'analog_lim_pt11_red':  BEGIN
		analog_lim_pt11_red_b	=	 dblarr(num_records1) ; 177
		analog_lim_pt11_red_b[*] = !values.d_nan ; 177
		END
	'analog_lim_pt12_status':  BEGIN
		analog_lim_pt12_status_b	= dblarr(num_records1) ; 178
		analog_lim_pt12_status_b[*] = !values.d_nan ; 178
		END
	'analog_lim_pt12_id':  BEGIN
		analog_lim_pt12_id_b	= dblarr(num_records1) ; 179
		analog_lim_pt12_id_b[*] = !values.d_nan ; 179
		END
	'analog_lim_pt12_red':  BEGIN
		analog_lim_pt12_red_b	=	 dblarr(num_records1) ; 180
		analog_lim_pt12_red_b[*] = !values.d_nan ; 180
		END
	'analog_lim_pt13_status':  BEGIN
		analog_lim_pt13_status_b	=	 dblarr(num_records1) ; 181
		analog_lim_pt13_status_b[*] = !values.d_nan ; 181
		END
	'analog_lim_pt13_id':  BEGIN
		analog_lim_pt13_id_b	= dblarr(num_records1) ; 182
		analog_lim_pt13_id_b[*] = !values.d_nan ; 182
		END
	'analog_lim_pt13_red':  BEGIN
		analog_lim_pt13_red_b	= dblarr(num_records1) ; 183
		analog_lim_pt13_red_b[*] = !values.d_nan ; 183
		END
	'analog_lim_pt14_status':  BEGIN
		analog_lim_pt14_status_b	= 	dblarr(num_records1) ; 184
		analog_lim_pt14_status_b[*] = !values.d_nan ; 184
		END
	'analog_lim_pt14_id':  BEGIN
		analog_lim_pt14_id_b	= 	dblarr(num_records1) ; 185
		analog_lim_pt14_id_b[*] = !values.d_nan ; 185
		END
	'analog_lim_pt14_red':  BEGIN
		analog_lim_pt14_red_b	= dblarr(num_records1) ; 186
		analog_lim_pt14_red_b[*] = !values.d_nan; 186
		END
	'analog_lim_pt15_status':  BEGIN
		analog_lim_pt15_status_b	=	 dblarr(num_records1) ; 187
		analog_lim_pt15_status_b[*] = !values.d_nan ; 187
		END
	'analog_lim_pt15_id':  BEGIN
		analog_lim_pt15_id_b	= dblarr(num_records1) ; 188
		analog_lim_pt15_id_b[*] = !values.d_nan ; 188
		END
	'analog_lim_pt15_red':  BEGIN
		analog_lim_pt15_red_b	= dblarr(num_records1) ; 189
		analog_lim_pt15_red_b[*] = !values.d_nan ; 189
		END
	'opmode':  BEGIN
		opmode_b	=	 dblarr(num_records1) ; 190
		opmode_b[*] = !values.d_nan ; 190
		END
	'sw_mode':  BEGIN
		sw_mode_b	=	 dblarr(num_records1) ; 191
		sw_mode_b[*] = !values.d_nan ; 191
		END
	'proton_mode':  BEGIN
		proton_mode_b	= dblarr(num_records1) ; 192
		proton_mode_b[*] = !values.d_nan ; 192
		END
	'active_supplyid':  BEGIN
		active_supplyid_b	= dblarr(num_records1) ; 193
		active_supplyid_b[*] = !values.d_nan ; 193
		END
	'supply_coarse_rb':  BEGIN
		supply_coarse_rb_b	= 	dblarr(num_records1) ; 194
		supply_coarse_rb_b[*] = !values.d_nan ; 194
		END
	'supply_fine_rb':  BEGIN
		supply_fine_rb_b	= 	dblarr(num_records1) ; 195
		supply_fine_rb_b[*] = !values.d_nan ; 195
		END
	'swmaint_supplystat':  BEGIN
		swmaint_supplystat_b	= dblarr(num_records1) ; 196
		swmaint_supplystat_b[*] = !values.d_nan ; 196
		END
	'swmaint_pac_coarse':  BEGIN
		swmaint_pac_coarse_b	=	 dblarr(num_records1) ; 197
		swmaint_pac_coarse_b[*] = !values.d_nan ; 197
		END
	'swmaint_pac_fine':  BEGIN
		swmaint_pac_fine_b	= dblarr(num_records1) ; 198		
		swmaint_pac_fine_b[*] = !values.d_nan ; 198
		END
	ENDCASE
	ENDFOR
    	END 
ENDCASE 
dig_hk_initialized = 1

END 
;-------------------------------------------------------------------------------



;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
PRO init_mon_full, sat, num_records1, mon_full_initialized, E_step, products
;-----------------------------------------------------------------------------
; initialize the full resolution monitor rate variables
;	sat:  which spacecraft
;	num_records1:  the number of records we expect
;	mon_full_initialized:  whether or not the full resolution rate variables have been initialized
;	E_step:  the desired energy steps
;	products:  the desired products

num_E = n_elements(E_step)

COMMON com_mon
num_rows1_full = num_records1
array_name = ' '

CASE sat OF
    'A': BEGIN 
	error1_full_a	= fltarr(num_records1) ; error flag
	error1_full_a[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_a_full	= dblarr(128,32,num_records1) ;0
		s_valid_a_full[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_a_full = dblarr(num_records1,128) ;0
		sum_defl_s_valid_a_full[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_a_full	= dblarr(128,32,num_records1) ; 1
		s_e_not_req_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_a_full = dblarr(num_records1,128) ;1
		sum_defl_s_e_not_req_a_full[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_a_full	= dblarr(128,32,num_records1) ; 2
		s_e_req_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_a_full = dblarr(num_records1,128) ;2
		sum_defl_s_e_req_a_full[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_a_full	= dblarr(128,32,num_records1) ; 3
		s_no_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_a_full = dblarr(num_records1,128) ;3
		sum_defl_s_no_pos_a_full[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_a_full	= dblarr(128,32,num_records1) ; 4
		s_mult_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_a_full = dblarr(num_records1,128) ;4
		sum_defl_s_mult_pos_a_full[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_a_full	= dblarr(128,32,num_records1) ; 5
		s_no_e_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_a_full = dblarr(num_records1,128) ;5
		sum_defl_s_no_e_a_full[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_a_full	= dblarr(128,32,num_records1) ; 6
		s_mult_e_a_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_a_full = dblarr(num_records1,128) ;6
		sum_defl_s_mult_e_a_full[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_a_full	= dblarr(128,32,num_records1) ; 7
		ra_sat_a_a_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_a_full = dblarr(num_records1,128) ;7
		sum_defl_ra_sat_a_a_full[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_a_full	= dblarr(128,32,num_records1) ; 8
		ra_sat_b_a_full[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_a_full = dblarr(num_records1,128) ;8
		sum_defl_ra_sat_b_a_full[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_a_full	= dblarr(128,32,num_records1) ; 9
		ra_sat_both_a_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_a_full = dblarr(num_records1,128) ;9
		sum_defl_ra_sat_both_a_full[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_a_full	= dblarr(128,32,num_records1) ; 10
		ssd_sw_a_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_a_full = dblarr(num_records1,128) ;10
		sum_defl_ssd_sw_a_full[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_a_full	= dblarr(128,32,num_records1) ; 11
		ssd_st_a_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_a_full = dblarr(num_records1,128) ;11
		sum_defl_ssd_st_a_full[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_a_full	= dblarr(128,32,num_records1) ; 12
		sf0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_a_full[*,*] = !values.d_nan ;12		
		END
	'sfr0':  BEGIN
		sfr0_a_full	= dblarr(128,32,num_records1) ; 13
		sfr0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_a_full[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_a_full	= dblarr(128,32,num_records1) ; 14
		stp0_a_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_a_full[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_a_full	= dblarr(128,32,num_records1) ; 15
		ra_trig_a_full[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_full = dblarr(num_records1,128) ;15
		sum_defl_ra_trig_a_full[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_a_full	= dblarr(128,32,num_records1) ; 16
		pos1_0_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_a_full = dblarr(num_records1,128) ;16
		sum_defl_pos1_0_a_full[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_a_full	= dblarr(128,32,num_records1) ; 17
		pos1_1_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_a_full = dblarr(num_records1,128) ;17
		sum_defl_pos1_1_a_full[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_a_full	= dblarr(128,32,num_records1) ; 18
		w_no_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_a_full = dblarr(num_records1,128) ;18
		sum_defl_w_no_pos_a_full[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_a_full	= dblarr(128,32,num_records1) ; 19
		w_mult_pos_a_full[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_a_full = dblarr(num_records1,128) ;19
		sum_defl_w_mult_pos_a_full[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_a_full	= dblarr(128,32,num_records1) ; 20
		w_valid_a_full[*,*,*]	= !values.d_nan
		sum_defl_w_valid_a_full = dblarr(num_records1,128) ;20
		sum_defl_w_valid_a_full[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_a_full	= dblarr(128,32,num_records1) ; 21
		sf2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_a_full[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_a_full	= dblarr(128,32,num_records1) ; 22
		sfr2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_a_full[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_a_full	= dblarr(128,32,num_records1) ; 23
		stp2_a_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_a_full[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_a_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_a_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_a_full[*,*] = !values.d_nan ;24	
		END
	'pos2_1':  BEGIN
		pos2_1_a_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_a_full[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_a_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_a_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_a_full[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_a_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_a_full[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_a_full	= dblarr(128,32,num_records1) ; 28
		pos3_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_a_full = dblarr(num_records1,128) ;28
		sum_defl_pos3_0_a_full[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_a_full	= dblarr(128,32,num_records1) ; 29
		pos3_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_full = dblarr(num_records1,128) ;29
		sum_defl_pos3_1_a_full[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_a_full	= dblarr(128,32,num_records1) ; 30
		pos3_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_full = dblarr(num_records1,128) ;30
		sum_defl_pos3_2_a_full[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_a_full	= dblarr(128,32,num_records1) ; 31
		pos3_3_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_a_full = dblarr(num_records1,128) ;31
		sum_defl_pos3_3_a_full[*,*] = !values.d_nan ;31
		END
	'pos2':  BEGIN 
		pos2_a_full	= dblarr(128,32,num_records1) 
		pos2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_a_full = dblarr(num_records1,128) 
		sum_defl_pos2_a_full[*,*] = !values.d_nan  

		pos2_0_a_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_a_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_a_full[*,*] = !values.d_nan ;24	

		pos2_1_a_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_a_full[*,*] = !values.d_nan ;25

		pos2_2_a_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_a_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_a_full[*,*] = !values.d_nan ;26

		pos2_3_a_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_a_full[*,*] = !values.d_nan ;27
		
		END
	'pos3':  BEGIN 
		pos3_a_full	= dblarr(128,32,num_records1) 
		pos3_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_a_full = dblarr(num_records1,128) 
		sum_defl_pos3_a_full[*,*] = !values.d_nan 

		pos3_0_a_full	= dblarr(128,32,num_records1) ; 24
		pos3_0_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_a_full = dblarr(num_records1,128) ;24
		sum_defl_pos3_0_a_full[*,*] = !values.d_nan ;24	

		pos3_1_a_full	= dblarr(128,32,num_records1) ; 25
		pos3_1_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_full = dblarr(num_records1,128) ;25
		sum_defl_pos3_1_a_full[*,*] = !values.d_nan ;25

		pos3_2_a_full	= dblarr(128,32,num_records1) ; 26
		pos3_2_a_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_full = dblarr(num_records1,128) ;26
		sum_defl_pos3_2_a_full[*,*] = !values.d_nan ;26

		pos3_3_a_full	= dblarr(128,32,num_records1) ; 27
		pos3_3_a_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_a_full = dblarr(num_records1,128) ;27
		sum_defl_pos3_3_a_full[*,*] = !values.d_nan ;27
		
		END


	'stop0':  BEGIN 
		stop0_a_full	= dblarr(128,32,num_records1) 
		stop0_a_full[*,*,*]	= !values.d_nan
		sum_defl_stop0_a_full = dblarr(num_records1,128) 
		sum_defl_stop0_a_full[*,*] = !values.d_nan  

		sf0_a_full	= dblarr(128,32,num_records1) ; 12
		sf0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_a_full[*,*] = !values.d_nan ;12	

		sfr0_a_full	= dblarr(128,32,num_records1) ; 13
		sfr0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_a_full[*,*] = !values.d_nan ;13
	
		END

	'start0':  BEGIN
		start0_a_full	= dblarr(128,32,num_records1) 
		start0_a_full[*,*,*]	= !values.d_nan
		sum_defl_start0_a_full = dblarr(num_records1,128)
		sum_defl_start0_a_full[*,*] = !values.d_nan

		sfr0_a_full	= dblarr(128,32,num_records1) ; 13
		sfr0_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_a_full[*,*] = !values.d_nan ;13

		stp0_a_full	= dblarr(128,32,num_records1) ; 14
		stp0_a_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_a_full[*,*] = !values.d_nan ;14

		END
	'stop2':  BEGIN
		stop2_a_full	= dblarr(128,32,num_records1)
		stop2_a_full[*,*,*]	= !values.d_nan
		sum_defl_stop2_a_full = dblarr(num_records1,128)
		sum_defl_stop2_a_full[*,*] = !values.d_nan

		sf2_a_full	= dblarr(128,32,num_records1) ; 21
		sf2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_a_full[*,*] = !values.d_nan ;21
		
		sfr2_a_full	= dblarr(128,32,num_records1) ; 22
		sfr2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_a_full[*,*] = !values.d_nan ;22
		END

	'start2':  BEGIN
		start2_a_full	= dblarr(128,32,num_records1) 
		start2_a_full[*,*,*]	= !values.d_nan
		sum_defl_start2_a_full = dblarr(num_records1,128) 
		sum_defl_start2_a_full[*,*] = !values.d_nan 

		sfr2_a_full	= dblarr(128,32,num_records1) ; 22
		sfr2_a_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_a_full[*,*] = !values.d_nan ;22

		stp2_a_full	= dblarr(128,32,num_records1) ; 23
		stp2_a_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_a_full[*,*] = !values.d_nan ;23
		END

	ENDCASE
	ENDFOR			
    END 
    'B': BEGIN 
	error1_full_b	= fltarr(num_records1) ; error flag
	error1_full_b[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_b_full	= dblarr(128,32,num_records1) ;0
		s_valid_b_full[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_b_full = dblarr(num_records1,128) ;0
		sum_defl_s_valid_b_full[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_b_full	= dblarr(128,32,num_records1) ; 1
		s_e_not_req_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_b_full = dblarr(num_records1,128) ;1
		sum_defl_s_e_not_req_b_full[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_b_full	= dblarr(128,32,num_records1) ; 2
		s_e_req_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_b_full = dblarr(num_records1,128) ;2
		sum_defl_s_e_req_b_full[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_b_full	= dblarr(128,32,num_records1) ; 3
		s_no_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_b_full = dblarr(num_records1,128) ;3
		sum_defl_s_no_pos_b_full[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_b_full	= dblarr(128,32,num_records1) ; 4
		s_mult_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_b_full = dblarr(num_records1,128) ;4
		sum_defl_s_mult_pos_b_full[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_b_full	= dblarr(128,32,num_records1) ; 5
		s_no_e_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_b_full = dblarr(num_records1,128) ;5
		sum_defl_s_no_e_b_full[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_b_full	= dblarr(128,32,num_records1) ; 6
		s_mult_e_b_full[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_b_full = dblarr(num_records1,128) ;6
		sum_defl_s_mult_e_b_full[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_b_full	= dblarr(128,32,num_records1) ; 7
		ra_sat_a_b_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_b_full = dblarr(num_records1,128) ;7
		sum_defl_ra_sat_a_b_full[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_b_full	= dblarr(128,32,num_records1) ; 8
		ra_sat_b_b_full[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_b_full = dblarr(num_records1,128) ;8
		sum_defl_ra_sat_b_b_full[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_b_full	= dblarr(128,32,num_records1) ; 9
		ra_sat_both_b_full[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_b_full = dblarr(num_records1,128) ;9
		sum_defl_ra_sat_both_b_full[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_b_full	= dblarr(128,32,num_records1) ; 10
		ssd_sw_b_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_b_full = dblarr(num_records1,128) ;10
		sum_defl_ssd_sw_b_full[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_b_full	= dblarr(128,32,num_records1) ; 11
		ssd_st_b_full[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_b_full = dblarr(num_records1,128) ;11
		sum_defl_ssd_st_b_full[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_b_full	= dblarr(128,32,num_records1) ; 12
		sf0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_b_full[*,*] = !values.d_nan ;12		
		END
	'sfr0':  BEGIN
		sfr0_b_full	= dblarr(128,32,num_records1) ; 13
		sfr0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_b_full[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_b_full	= dblarr(128,32,num_records1) ; 14
		stp0_b_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_b_full[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_b_full	= dblarr(128,32,num_records1) ; 15
		ra_trig_b_full[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_full = dblarr(num_records1,128) ;15
		sum_defl_ra_trig_b_full[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_b_full	= dblarr(128,32,num_records1) ; 16
		pos1_0_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_b_full = dblarr(num_records1,128) ;16
		sum_defl_pos1_0_b_full[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_b_full	= dblarr(128,32,num_records1) ; 17
		pos1_1_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_b_full = dblarr(num_records1,128) ;17
		sum_defl_pos1_1_b_full[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_b_full	= dblarr(128,32,num_records1) ; 18
		w_no_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_b_full = dblarr(num_records1,128) ;18
		sum_defl_w_no_pos_b_full[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_b_full	= dblarr(128,32,num_records1) ; 19
		w_mult_pos_b_full[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_b_full = dblarr(num_records1,128) ;19
		sum_defl_w_mult_pos_b_full[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_b_full	= dblarr(128,32,num_records1) ; 20
		w_valid_b_full[*,*,*]	= !values.d_nan
		sum_defl_w_valid_b_full = dblarr(num_records1,128) ;20
		sum_defl_w_valid_b_full[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_b_full	= dblarr(128,32,num_records1) ; 21
		sf2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_b_full[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_b_full	= dblarr(128,32,num_records1) ; 22
		sfr2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_b_full[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_b_full	= dblarr(128,32,num_records1) ; 23
		stp2_b_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_b_full[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_b_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_b_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_b_full[*,*] = !values.d_nan ;24	
		END
	'pos2_1':  BEGIN
		pos2_1_b_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_b_full[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_b_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_b_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_b_full[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_b_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_b_full[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_b_full	= dblarr(128,32,num_records1) ; 28
		pos3_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_b_full = dblarr(num_records1,128) ;28
		sum_defl_pos3_0_b_full[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_b_full	= dblarr(128,32,num_records1) ; 29
		pos3_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_full = dblarr(num_records1,128) ;29
		sum_defl_pos3_1_b_full[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_b_full	= dblarr(128,32,num_records1) ; 30
		pos3_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_full = dblarr(num_records1,128) ;30
		sum_defl_pos3_2_b_full[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_b_full	= dblarr(128,32,num_records1) ; 31
		pos3_3_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_b_full = dblarr(num_records1,128) ;31
		sum_defl_pos3_3_b_full[*,*] = !values.d_nan ;31
		END
	'pos2':  BEGIN 
		pos2_b_full	= dblarr(128,32,num_records1) 
		pos2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_b_full = dblarr(num_records1,128) 
		sum_defl_pos2_b_full[*,*] = !values.d_nan  

		pos2_0_b_full	= dblarr(128,32,num_records1) ; 24
		pos2_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_b_full = dblarr(num_records1,128) ;24
		sum_defl_pos2_0_b_full[*,*] = !values.d_nan ;24	

		pos2_1_b_full	= dblarr(128,32,num_records1) ; 25
		pos2_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_full = dblarr(num_records1,128) ;25
		sum_defl_pos2_1_b_full[*,*] = !values.d_nan ;25

		pos2_2_b_full	= dblarr(128,32,num_records1) ; 26
		pos2_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_b_full = dblarr(num_records1,128) ;26
		sum_defl_pos2_2_b_full[*,*] = !values.d_nan ;26

		pos2_3_b_full	= dblarr(128,32,num_records1) ; 27
		pos2_3_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_full = dblarr(num_records1,128) ;27
		sum_defl_pos2_3_b_full[*,*] = !values.d_nan ;27
		
		END
	'pos3':  BEGIN 
		pos3_b_full	= dblarr(128,32,num_records1) 
		pos3_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_b_full = dblarr(num_records1,128) 
		sum_defl_pos3_b_full[*,*] = !values.d_nan 

		pos3_0_b_full	= dblarr(128,32,num_records1) ; 24
		pos3_0_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_b_full = dblarr(num_records1,128) ;24
		sum_defl_pos3_0_b_full[*,*] = !values.d_nan ;24	

		pos3_1_b_full	= dblarr(128,32,num_records1) ; 25
		pos3_1_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_full = dblarr(num_records1,128) ;25
		sum_defl_pos3_1_b_full[*,*] = !values.d_nan ;25

		pos3_2_b_full	= dblarr(128,32,num_records1) ; 26
		pos3_2_b_full[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_full = dblarr(num_records1,128) ;26
		sum_defl_pos3_2_b_full[*,*] = !values.d_nan ;26

		pos3_3_b_full	= dblarr(128,32,num_records1) ; 27
		pos3_3_b_full[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_b_full = dblarr(num_records1,128) ;27
		sum_defl_pos3_3_b_full[*,*] = !values.d_nan ;27
		
		END

	'stop0':  BEGIN 
		stop0_b_full	= dblarr(128,32,num_records1) 
		stop0_b_full[*,*,*]	= !values.d_nan
		sum_defl_stop0_b_full = dblarr(num_records1,128) 
		sum_defl_stop0_b_full[*,*] = !values.d_nan  

		sf0_b_full	= dblarr(128,32,num_records1) ; 12
		sf0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_full = dblarr(num_records1,128) ;12
		sum_defl_sf0_b_full[*,*] = !values.d_nan ;12	

		sfr0_b_full	= dblarr(128,32,num_records1) ; 13
		sfr0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_b_full[*,*] = !values.d_nan ;13
	
		END

	'start0':  BEGIN
		start0_b_full	= dblarr(128,32,num_records1) 
		start0_b_full[*,*,*]	= !values.d_nan
		sum_defl_start0_b_full = dblarr(num_records1,128)
		sum_defl_start0_b_full[*,*] = !values.d_nan

		sfr0_b_full	= dblarr(128,32,num_records1) ; 13
		sfr0_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_full = dblarr(num_records1,128) ;13
		sum_defl_sfr0_b_full[*,*] = !values.d_nan ;13

		stp0_b_full	= dblarr(128,32,num_records1) ; 14
		stp0_b_full[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_full = dblarr(num_records1,128) ;14
		sum_defl_stp0_b_full[*,*] = !values.d_nan ;14

		END
	'stop2':  BEGIN
		stop2_b_full	= dblarr(128,32,num_records1)
		stop2_b_full[*,*,*]	= !values.d_nan
		sum_defl_stop2_b_full = dblarr(num_records1,128)
		sum_defl_stop2_b_full[*,*] = !values.d_nan

		sf2_b_full	= dblarr(128,32,num_records1) ; 21
		sf2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_full = dblarr(num_records1,128) ;21
		sum_defl_sf2_b_full[*,*] = !values.d_nan ;21
		
		sfr2_b_full	= dblarr(128,32,num_records1) ; 22
		sfr2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_b_full[*,*] = !values.d_nan ;22
		END

	'start2':  BEGIN
		start2_b_full	= dblarr(128,32,num_records1) 
		start2_b_full[*,*,*]	= !values.d_nan
		sum_defl_start2_b_full = dblarr(num_records1,128) 
		sum_defl_start2_b_full[*,*] = !values.d_nan 

		sfr2_b_full	= dblarr(128,32,num_records1) ; 22
		sfr2_b_full[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_full = dblarr(num_records1,128) ;22
		sum_defl_sfr2_b_full[*,*] = !values.d_nan ;22

		stp2_b_full	= dblarr(128,32,num_records1) ; 23
		stp2_b_full[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_full = dblarr(num_records1,128) ;23
		sum_defl_stp2_b_full[*,*] = !values.d_nan ;23
		END

	ENDCASE
	ENDFOR
	END
ENDCASE 
mon_full_initialized = 1

END
;--------------------------------------------------------------
 

;--------------------------------------------------------------
;--------------------------------------------------------------
PRO init_hk, sat, num_records1, hk_initialized
;--------------------------------------------------------------
; initialize the analog housekeeping variabales
;	sat:  which spacecraft
;	num_records1:  the number of records we expect
;	hk_initialized:  whether or not the analog housekeeping variables have been initialized

COMMON com_mon
CASE sat OF
    'A': BEGIN 
	lrnm_spare_a	= dblarr(num_records1) ; 0
	lvc_neg12v_a	= dblarr(num_records1) ; 1
	lvc_neg5v_a	= dblarr(num_records1) ; 2
	esa_vm_pos_a	= dblarr(num_records1) ; 3
	dfl_1_vm_a	= dblarr(num_records1) ; 4
	dfl_1_tap_a	= dblarr(num_records1) ; 5
	dfl_2_vm_a	= dblarr(num_records1) ; 6
	dfl_2_tap_a	= dblarr(num_records1) ; 7
	lvc_pos2point5v_b_a	= dblarr(num_records1) ; 8
	pac_cm_dc_a	= dblarr(num_records1) ; 9
	lvc_pos12v_a	= dblarr(num_records1) ; 10
	lrpm_spare_a	= dblarr(num_records1) ; 11
	lvc_pos2point5v_a_a	= dblarr(num_records1) ; 12
	lvc_pos5v_a	= dblarr(num_records1) ; 13
	adc_avdd_a	= dblarr(num_records1) ; 14
	adc_dvdd_a	= dblarr(num_records1) ; 15
	pac_vm_a	= dblarr(num_records1) ; 16
	cal_vref_a	= dblarr(num_records1) ; 17
	pac_cm_ac_a	= dblarr(num_records1) ; 18
	mcp_vm_a	= dblarr(num_records1) ; 19
	mcp_cm_dc_a	= dblarr(num_records1) ; 20
	mcp_cm_ac_a	= dblarr(num_records1) ; 21
	ssd_cm_dc_a	= dblarr(num_records1) ; 22
	s_ch_vm_a	= dblarr(num_records1) ; 23
	s_ch_vm_tap_a	= dblarr(num_records1) ; 24
	s_ch_cm_dc_a	= dblarr(num_records1) ; 25
	esa_vm_neg_a	= dblarr(num_records1) ; 26
	esa_cm_dc_a	= dblarr(num_records1) ; 27
	dfl_1_cm_dc_a	= dblarr(num_records1) ; 28
	dfl_2_cm_dc_a	= dblarr(num_records1) ; 29
	lvc_pos2point5v_b_imon_a	= dblarr(num_records1) ; 30	
	lvc_pos12v_imon_a	= dblarr(num_records1) ; 31
	lvc_neg12v_imon_a	= dblarr(num_records1) ; 32
	lvc_pos5v_imon_a	= dblarr(num_records1) ; 33
	lvc_neg5v_imon_a	= dblarr(num_records1) ; 34
	lvc_pos2point5v_a_imon_a	= dblarr(num_records1) ; 35
	ssd_vm_a	= dblarr(num_records1) ; 36
	adc_lu_flag_a	= dblarr(num_records1) ; 37
	plug_id_a	= dblarr(num_records1) ; 38
	adc_lu_ctr_a	= dblarr(num_records1) ; 39
	dac_status_a	= dblarr(num_records1) ; 40
	adc0_agnd_a	= dblarr(num_records1) ; 41
	adc1_agnd_a	= dblarr(num_records1) ; 42
	ra_e0_a	= dblarr(num_records1) ; 43
	ra_e1_a	= dblarr(num_records1) ; 44
	tac0_tsp_a	= dblarr(num_records1) ; 45
	tac2_tsp_a	= dblarr(num_records1) ; 46
	sb0_tsp_a	= dblarr(num_records1) ; 47
	sb1_tsp_a	= dblarr(num_records1) ; 48
	tof_hv0_tsp_a	= dblarr(num_records1) ; 49
	tof_hv1_tsp_a	= dblarr(num_records1) ; 50
	s_c_0_tsp_a	= dblarr(num_records1) ; 51
	s_c_1_tsp_a	= dblarr(num_records1) ; 52
	lvc0_tsp_a	= dblarr(num_records1) ; 53
	lvc1_tsp_a	= dblarr(num_records1) ; 54
	adc0_vref_a	= dblarr(num_records1) ; 55
	adc1_vref_a	= dblarr(num_records1) ; 56
	ssd_status_a	= dblarr(num_records1) ; 57
	ssd_v_pos_analog_a	= dblarr(num_records1) ; 58
	ssd_v_neg_analog_a	= dblarr(num_records1) ; 59
	ssd_hv_bias_a	= dblarr(num_records1) ; 60
	ssd_tc0_a	= dblarr(num_records1) ; 61
	ssd_tc1_a	= dblarr(num_records1) ; 62
	ssd_tc2_a	= dblarr(num_records1) ; 63
	ssd_v_pos_dig_a	= dblarr(num_records1) ; 64

	lrnm_spare_a[*] = !values.d_nan ; 0
	lvc_neg12v_a[*]	= !values.d_nan ; 1
	lvc_neg5v_a[*]	= !values.d_nan ; 2
	esa_vm_pos_a[*]	= !values.d_nan ; 3
	dfl_1_vm_a[*]	= !values.d_nan ; 4
	dfl_1_tap_a[*]	= !values.d_nan ; 5
	dfl_2_vm_a[*]	= !values.d_nan ; 6
	dfl_2_tap_a[*]	= !values.d_nan ; 7
	lvc_pos2point5v_b_a[*]= !values.d_nan ; 8
	pac_cm_dc_a[*]	= !values.d_nan ; 9
	lvc_pos12v_a[*]	= !values.d_nan ; 10
	lrpm_spare_a[*]	= !values.d_nan ; 11
	lvc_pos2point5v_a_a[*]	= !values.d_nan ; 12
	lvc_pos5v_a[*]	= !values.d_nan ; 13
	adc_avdd_a[*]	= !values.d_nan ; 14
	adc_dvdd_a[*]	= !values.d_nan ; 15
	pac_vm_a[*]	= !values.d_nan ; 16
	cal_vref_a[*]	= !values.d_nan ; 17
	pac_cm_ac_a[*]	= !values.d_nan ; 18
	mcp_vm_a[*]	= !values.d_nan ; 19
	mcp_cm_dc_a[*]	= !values.d_nan ; 20
	mcp_cm_ac_a[*]	= !values.d_nan ; 21
	ssd_cm_dc_a[*]	= !values.d_nan ; 22
	s_ch_vm_a[*]	= !values.d_nan	; 23
	s_ch_vm_tap_a[*]	= !values.d_nan ; 24	
	s_ch_cm_dc_a[*]	= !values.d_nan	; 25
	esa_vm_neg_a[*]	= !values.d_nan	; 26
	esa_cm_dc_a[*]	= !values.d_nan	; 27
	dfl_1_cm_dc_a[*]	= !values.d_nan ; 28	
	dfl_2_cm_dc_a[*]	= !values.d_nan	; 29
	lvc_pos2point5v_b_imon_a[*]	= !values.d_nan ; 30	
	lvc_pos12v_imon_a[*]	= !values.d_nan	; 31
	lvc_neg12v_imon_a[*]	= !values.d_nan ; 32
	lvc_pos5v_imon_a[*]	= !values.d_nan ; 33
	lvc_neg5v_imon_a[*]	= !values.d_nan ; 34
	lvc_pos2point5v_a_imon_a[*]	= !values.d_nan ; 35
	ssd_vm_a[*]	= !values.d_nan ; 36
	adc_lu_flag_a[*]	= !values.d_nan ; 37
	plug_id_a[*]	= !values.d_nan	; 38
	adc_lu_ctr_a[*]	= !values.d_nan	; 39
	dac_status_a[*]	= !values.d_nan ; 40
	adc0_agnd_a[*]	= !values.d_nan	; 41
	adc1_agnd_a[*]	= !values.d_nan	; 42
	ra_e0_a[*]	= !values.d_nan ; 43
	ra_e1_a[*]	= !values.d_nan	; 44
	tac0_tsp_a[*]	= !values.d_nan	; 45
	tac2_tsp_a[*]	= !values.d_nan	; 46
	sb0_tsp_a[*]	= !values.d_nan	; 47
	sb1_tsp_a[*]	= !values.d_nan ; 48
	tof_hv0_tsp_a[*]	= !values.d_nan ; 49
	tof_hv1_tsp_a[*]	= !values.d_nan ; 50
	s_c_0_tsp_a[*]	= !values.d_nan ; 51
	s_c_1_tsp_a[*]	= !values.d_nan	; 52
	lvc0_tsp_a[*]	= !values.d_nan ; 53
	lvc1_tsp_a[*]	= !values.d_nan ; 54
	adc0_vref_a[*]	= !values.d_nan ; 55
	adc1_vref_a[*]	= !values.d_nan ; 56
	ssd_status_a[*]	= !values.d_nan ; 57
	ssd_v_pos_analog_a[*]	= !values.d_nan ; 58
	ssd_v_neg_analog_a[*]	= !values.d_nan ; 59
	ssd_hv_bias_a[*]	= !values.d_nan ; 60
	ssd_tc0_a[*]	= !values.d_nan ; 61
	ssd_tc1_a[*]	= !values.d_nan ; 62
	ssd_tc2_a[*]	= !values.d_nan ; 63
	ssd_v_pos_dig_a[*]	= !values.d_nan ; 64


    END 
    'B': BEGIN 
	lrnm_spare_b	= dblarr(num_records1) ; 0
	lvc_neg12v_b	= dblarr(num_records1) ; 1
	lvc_neg5v_b	= dblarr(num_records1) ; 2
	esa_vm_pos_b	= dblarr(num_records1) ; 3
	dfl_1_vm_b	= dblarr(num_records1) ; 4
	dfl_1_tap_b	= dblarr(num_records1) ; 5
	dfl_2_vm_b	= dblarr(num_records1) ; 6
	dfl_2_tap_b	= dblarr(num_records1) ; 7
	lvc_pos2point5v_b_b	= dblarr(num_records1) ; 8
	pac_cm_dc_b	= dblarr(num_records1) ; 9
	lvc_pos12v_b	= dblarr(num_records1) ; 10
	lrpm_spare_b	= dblarr(num_records1) ; 11
	lvc_pos2point5v_a_b	= dblarr(num_records1) ; 12
	lvc_pos5v_b	= dblarr(num_records1) ; 13
	adc_avdd_b	= dblarr(num_records1) ; 14
	adc_dvdd_b	= dblarr(num_records1) ; 15
	pac_vm_b	= dblarr(num_records1) ; 16
	cal_vref_b	= dblarr(num_records1) ; 17
	pac_cm_ac_b	= dblarr(num_records1) ; 18
	mcp_vm_b	= dblarr(num_records1) ; 19
	mcp_cm_dc_b	= dblarr(num_records1) ; 20
	mcp_cm_ac_b	= dblarr(num_records1) ; 21
	ssd_cm_dc_b	= dblarr(num_records1) ; 22
	s_ch_vm_b	= dblarr(num_records1) ; 23
	s_ch_vm_tap_b	= dblarr(num_records1) ; 24
	s_ch_cm_dc_b	= dblarr(num_records1) ; 25
	esa_vm_neg_b	= dblarr(num_records1) ; 26
	esa_cm_dc_b	= dblarr(num_records1) ; 27
	dfl_1_cm_dc_b	= dblarr(num_records1) ; 28
	dfl_2_cm_dc_b	= dblarr(num_records1) ; 29
	lvc_pos2point5v_b_imon_b	= dblarr(num_records1) ; 30	
	lvc_pos12v_imon_b	= dblarr(num_records1) ; 31
	lvc_neg12v_imon_b	= dblarr(num_records1) ; 32
	lvc_pos5v_imon_b	= dblarr(num_records1) ; 33
	lvc_neg5v_imon_b	= dblarr(num_records1) ; 34
	lvc_pos2point5v_a_imon_b	= dblarr(num_records1) ; 35
	ssd_vm_b	= dblarr(num_records1) ; 36
	adc_lu_flag_b	= dblarr(num_records1) ; 37
	plug_id_b	= dblarr(num_records1) ; 38
	adc_lu_ctr_b	= dblarr(num_records1) ; 39
	dac_status_b	= dblarr(num_records1) ; 40
	adc0_agnd_b	= dblarr(num_records1) ; 41
	adc1_agnd_b	= dblarr(num_records1) ; 42
	ra_e0_b	= dblarr(num_records1) ; 43
	ra_e1_b	= dblarr(num_records1) ; 44
	tac0_tsp_b	= dblarr(num_records1) ; 45
	tac2_tsp_b	= dblarr(num_records1) ; 46
	sb0_tsp_b	= dblarr(num_records1) ; 47
	sb1_tsp_b	= dblarr(num_records1) ; 48
	tof_hv0_tsp_b	= dblarr(num_records1) ; 49
	tof_hv1_tsp_b	= dblarr(num_records1) ; 50
	s_c_0_tsp_b	= dblarr(num_records1) ; 51
	s_c_1_tsp_b	= dblarr(num_records1) ; 52
	lvc0_tsp_b	= dblarr(num_records1) ; 53
	lvc1_tsp_b	= dblarr(num_records1) ; 54
	adc0_vref_b	= dblarr(num_records1) ; 55
	adc1_vref_b	= dblarr(num_records1) ; 56
	ssd_status_b	= dblarr(num_records1) ; 57
	ssd_v_pos_analog_b	= dblarr(num_records1) ; 58
	ssd_v_neg_analog_b	= dblarr(num_records1) ; 59
	ssd_hv_bias_b	= dblarr(num_records1) ; 60
	ssd_tc0_b	= dblarr(num_records1) ; 61
	ssd_tc1_b	= dblarr(num_records1) ; 62
	ssd_tc2_b	= dblarr(num_records1) ; 63
	ssd_v_pos_dig_b	= dblarr(num_records1) ; 64

	lrnm_spare_b[*] = !values.d_nan ; 0
	lvc_neg12v_b[*]	= !values.d_nan ; 1
	lvc_neg5v_b[*]	= !values.d_nan ; 2
	esa_vm_pos_b[*]	= !values.d_nan ; 3
	dfl_1_vm_b[*]	= !values.d_nan ; 4
	dfl_1_tap_b[*]	= !values.d_nan ; 5
	dfl_2_vm_b[*]	= !values.d_nan ; 6
	dfl_2_tap_b[*]	= !values.d_nan ; 7
	lvc_pos2point5v_b_b[*]= !values.d_nan ; 8
	pac_cm_dc_b[*]	= !values.d_nan ; 9
	lvc_pos12v_b[*]	= !values.d_nan ; 10
	lrpm_spare_b[*]	= !values.d_nan ; 11
	lvc_pos2point5v_a_b[*]	= !values.d_nan ; 12
	lvc_pos5v_b[*]	= !values.d_nan ; 13
	adc_avdd_b[*]	= !values.d_nan ; 14
	adc_dvdd_b[*]	= !values.d_nan ; 15
	pac_vm_b[*]	= !values.d_nan ; 16
	cal_vref_b[*]	= !values.d_nan ; 17
	pac_cm_ac_b[*]	= !values.d_nan ; 18
	mcp_vm_b[*]	= !values.d_nan ; 19
	mcp_cm_dc_b[*]	= !values.d_nan ; 20
	mcp_cm_ac_b[*]	= !values.d_nan ; 21
	ssd_cm_dc_b[*]	= !values.d_nan ; 22
	s_ch_vm_b[*]	= !values.d_nan	; 23
	s_ch_vm_tap_b[*]	= !values.d_nan ; 24	
	s_ch_cm_dc_b[*]	= !values.d_nan	; 25
	esa_vm_neg_b[*]	= !values.d_nan	; 26
	esa_cm_dc_b[*]	= !values.d_nan	; 27
	dfl_1_cm_dc_b[*]	= !values.d_nan ; 28	
	dfl_2_cm_dc_b[*]	= !values.d_nan	; 29
	lvc_pos2point5v_b_imon_b[*]	= !values.d_nan ; 30	
	lvc_pos12v_imon_b[*]	= !values.d_nan	; 31
	lvc_neg12v_imon_b[*]	= !values.d_nan ; 32
	lvc_pos5v_imon_b[*]	= !values.d_nan ; 33
	lvc_neg5v_imon_b[*]	= !values.d_nan ; 34
	lvc_pos2point5v_a_imon_b[*]	= !values.d_nan ; 35
	ssd_vm_b[*]	= !values.d_nan ; 36
	adc_lu_flag_b[*]	= !values.d_nan ; 37
	plug_id_b[*]	= !values.d_nan	; 38
	adc_lu_ctr_b[*]	= !values.d_nan	; 39
	dac_status_b[*]	= !values.d_nan ; 40
	adc0_agnd_b[*]	= !values.d_nan	; 41
	adc1_agnd_b[*]	= !values.d_nan	; 42
	ra_e0_b[*]	= !values.d_nan ; 43
	ra_e1_b[*]	= !values.d_nan	; 44
	tac0_tsp_b[*]	= !values.d_nan	; 45
	tac2_tsp_b[*]	= !values.d_nan	; 46
	sb0_tsp_b[*]	= !values.d_nan	; 47
	sb1_tsp_b[*]	= !values.d_nan ; 48
	tof_hv0_tsp_b[*]	= !values.d_nan ; 49
	tof_hv1_tsp_b[*]	= !values.d_nan ; 50
	s_c_0_tsp_b[*]	= !values.d_nan ; 51
	s_c_1_tsp_b[*]	= !values.d_nan	; 52
	lvc0_tsp_b[*]	= !values.d_nan ; 53
	lvc1_tsp_b[*]	= !values.d_nan ; 54
	adc0_vref_b[*]	= !values.d_nan ; 55
	adc1_vref_b[*]	= !values.d_nan ; 56
	ssd_status_b[*]	= !values.d_nan ; 57
	ssd_v_pos_analog_b[*]	= !values.d_nan ; 58
	ssd_v_neg_analog_b[*]	= !values.d_nan ; 59
	ssd_hv_bias_b[*]	= !values.d_nan ; 60
	ssd_tc0_b[*]	= !values.d_nan ; 61
	ssd_tc1_b[*]	= !values.d_nan ; 62
	ssd_tc2_b[*]	= !values.d_nan ; 63
	ssd_v_pos_dig_b[*]	= !values.d_nan ;64
        
    END 
ENDCASE 
hk_initialized = 1

END 
;----------------------------------------------------------------------------------------


;----------------------------------------------------------------------------------------
;----------------------------------------------------------------------------------------
PRO init_data, sat, num_records1, num_records5, mon_initialized, E_step, products, NO_ENG 
;----------------------------------------------------------------------------------------
; initialize all the monitor rate variables (in com_mon)
;	sat:  which spacecraft
;	num_records1:  the number of engineering mode records we expect
;	num_records5:  the number of science mode records we expect
;	mon_initialized:  whether or not the monitor rate variables have been initialized
;	E_step:	the desired energy steps
;	products:  the desired products
;	NO_ENG:  if set to 1, engineering mode data will not be read, and the engineering mode
;		variables are not needed (saves time and memory)

num_E = n_elements(E_step)


COMMON com_mon
num_rows1 = num_records1
num_rows5 = num_records5

IF NO_ENG EQ 0 THEN BEGIN

	CASE sat OF
    	'A': BEGIN 
	
		error5_mon_a	= fltarr(num_records5) ; error flag science mode
		error5_mon_a[*]	= !values.f_nan
		error1_a	= fltarr(num_records1) ; error flag engineering mode
		error1_a[*]	= !values.f_nan	

		FOR ii = 0, n_elements(products)-1 DO BEGIN 
        	CASE products[ii] OF 
	
		's_valid':  BEGIN
		s_valid_a_sci	= dblarr(32,8,num_records5) ; 0
		s_valid_a_eng	= dblarr(32,8,num_records1) ;0
		s_valid_a_sci[*,*,*] 	= !values.d_nan
		s_valid_a_eng[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_a_sci = dblarr(num_records5,32) ;0
		sum_defl_s_valid_a_eng = dblarr(num_records1,32) ;0
		sum_defl_s_valid_a_sci[*,*] = !values.d_nan ;0
		sum_defl_s_valid_a_eng[*,*] = !values.d_nan ;0
		END
		's_e_not_req':  BEGIN
		s_e_not_req_a_sci	= dblarr(32,8,num_records5) ; 1
		s_e_not_req_a_eng	= dblarr(32,8,num_records1) ; 1
		s_e_not_req_a_sci[*,*,*]	= !values.d_nan
		s_e_not_req_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_a_sci = dblarr(num_records5,32) ;1
		sum_defl_s_e_not_req_a_eng = dblarr(num_records1,32) ;1
		sum_defl_s_e_not_req_a_sci[*,*] = !values.d_nan ;1
		sum_defl_s_e_not_req_a_eng[*,*] = !values.d_nan ;1
		END
		's_e_req':  BEGIN
		s_e_req_a_sci	= dblarr(32,8,num_records5) ; 2
		s_e_req_a_eng	= dblarr(32,8,num_records1) ; 2
		s_e_req_a_sci[*,*,*]	= !values.d_nan
		s_e_req_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_a_sci = dblarr(num_records5,32) ;2
		sum_defl_s_e_req_a_eng = dblarr(num_records1,32) ;2
		sum_defl_s_e_req_a_sci[*,*] = !values.d_nan ;2
		sum_defl_s_e_req_a_eng[*,*] = !values.d_nan ;2
		END
		's_no_pos':  BEGIN
		s_no_pos_a_sci	= dblarr(32,8,num_records5) ; 3
		s_no_pos_a_eng	= dblarr(32,8,num_records1) ; 3
		s_no_pos_a_sci[*,*,*]	= !values.d_nan
		s_no_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_a_sci = dblarr(num_records5,32) ;3
		sum_defl_s_no_pos_a_eng = dblarr(num_records1,32) ;3
		sum_defl_s_no_pos_a_sci[*,*] = !values.d_nan ;3
		sum_defl_s_no_pos_a_eng[*,*] = !values.d_nan ;3
		END
		's_mult_pos':  BEGIN
		s_mult_pos_a_sci	= dblarr(32,8,num_records5) ; 4
		s_mult_pos_a_eng	= dblarr(32,8,num_records1) ; 4
		s_mult_pos_a_sci[*,*,*]	= !values.d_nan
		s_mult_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_a_sci = dblarr(num_records5,32) ;4
		sum_defl_s_mult_pos_a_eng = dblarr(num_records1,32) ;4
		sum_defl_s_mult_pos_a_sci[*,*] = !values.d_nan ;4
		sum_defl_s_mult_pos_a_eng[*,*] = !values.d_nan ;4
		END
		's_no_e':  BEGIN
		s_no_e_a_sci	= dblarr(32,8,num_records5) ; 5
		s_no_e_a_eng	= dblarr(32,8,num_records1) ; 5
		s_no_e_a_sci[*,*,*]	= !values.d_nan
		s_no_e_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_a_sci = dblarr(num_records5,32) ;5
		sum_defl_s_no_e_a_eng = dblarr(num_records1,32) ;5
		sum_defl_s_no_e_a_sci[*,*] = !values.d_nan ;5
		sum_defl_s_no_e_a_eng[*,*] = !values.d_nan ;5
		END
		's_mult_e':  BEGIN
		s_mult_e_a_sci	= dblarr(32,8,num_records5) ; 6
		s_mult_e_a_eng	= dblarr(32,8,num_records1) ; 6
		s_mult_e_a_sci[*,*,*]	= !values.d_nan
		s_mult_e_a_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_a_sci = dblarr(num_records5,32) ;6
		sum_defl_s_mult_e_a_eng = dblarr(num_records1,32) ;6
		sum_defl_s_mult_e_a_sci[*,*] = !values.d_nan ;6
		sum_defl_s_mult_e_a_eng[*,*] = !values.d_nan ;6
		END
		'ra_sat_a':  BEGIN
		ra_sat_a_a_sci	= dblarr(32,8,num_records5) ; 7
		ra_sat_a_a_eng	= dblarr(32,8,num_records1) ; 7
		ra_sat_a_a_sci[*,*,*]	= !values.d_nan
		ra_sat_a_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_a_sci = dblarr(num_records5,32) ;7
		sum_defl_ra_sat_a_a_eng = dblarr(num_records1,32) ;7
		sum_defl_ra_sat_a_a_sci[*,*] = !values.d_nan ;7
		sum_defl_ra_sat_a_a_eng[*,*] = !values.d_nan ;7
		END
		'ra_sat_b':  BEGIN
		ra_sat_b_a_sci	= dblarr(32,8,num_records5) ; 8
		ra_sat_b_a_eng	= dblarr(32,8,num_records1) ; 8
		ra_sat_b_a_sci[*,*,*]= !values.d_nan
		ra_sat_b_a_eng[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_a_sci = dblarr(num_records5,32) ;8
		sum_defl_ra_sat_b_a_eng = dblarr(num_records1,32) ;8
		sum_defl_ra_sat_b_a_sci[*,*] = !values.d_nan ;8
		sum_defl_ra_sat_b_a_eng[*,*] = !values.d_nan ;8
		END
		'ra_sat_both':  BEGIN
		ra_sat_both_a_sci	= dblarr(32,8,num_records5) ; 9
		ra_sat_both_a_eng	= dblarr(32,8,num_records1) ; 9
		ra_sat_both_a_sci[*,*,*]	= !values.d_nan
		ra_sat_both_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_a_sci = dblarr(num_records5,32) ;9
		sum_defl_ra_sat_both_a_eng = dblarr(num_records1,32) ;9
		sum_defl_ra_sat_both_a_sci[*,*] = !values.d_nan ;9
		sum_defl_ra_sat_both_a_eng[*,*] = !values.d_nan ;9
		END
		'ssd_sw':  BEGIN
		ssd_sw_a_sci	= dblarr(32,8,num_records5) ; 10
		ssd_sw_a_eng	= dblarr(32,8,num_records1) ; 10
		ssd_sw_a_sci[*,*,*]	= !values.d_nan
		ssd_sw_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_a_sci = dblarr(num_records5,32) ;10
		sum_defl_ssd_sw_a_eng = dblarr(num_records1,32) ;10
		sum_defl_ssd_sw_a_sci[*,*] = !values.d_nan ;10
		sum_defl_ssd_sw_a_eng[*,*] = !values.d_nan ;10
		END
		'ssd_st':  BEGIN
		ssd_st_a_sci	= dblarr(32,8,num_records5) ; 11
		ssd_st_a_eng	= dblarr(32,8,num_records1) ; 11
		ssd_st_a_sci[*,*,*]	= !values.d_nan
		ssd_st_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_a_sci = dblarr(num_records5,32) ;11
		sum_defl_ssd_st_a_eng = dblarr(num_records1,32) ;11
		sum_defl_ssd_st_a_sci[*,*] = !values.d_nan ;11
		sum_defl_ssd_st_a_eng[*,*] = !values.d_nan ;11
		END
		'sf0':  BEGIN
		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_eng	= dblarr(32,8,num_records1) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sf0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_a_eng[*,*] = !values.d_nan ;12		
		END
		'sfr0':  BEGIN
		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sfr0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_a_eng[*,*] = !values.d_nan ;13
		END
		'stp0':  BEGIN
		stp0_a_sci	= dblarr(32,8,num_records5) ; 14
		stp0_a_eng	= dblarr(32,8,num_records1) ; 14
		stp0_a_sci[*,*,*]	= !values.d_nan
		stp0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_a_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_a_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_a_eng[*,*] = !values.d_nan ;14
		END
		'ra_trig':  BEGIN
		ra_trig_a_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_a_eng	= dblarr(32,8,num_records1) ; 15
		ra_trig_a_sci[*,*,*]	= !values.d_nan
		ra_trig_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_a_eng = dblarr(num_records1,32) ;15
		sum_defl_ra_trig_a_sci[*,*] = !values.d_nan ;15
		sum_defl_ra_trig_a_eng[*,*] = !values.d_nan ;15
		END
		'pos1_0':  BEGIN
		pos1_0_a_sci	= dblarr(32,8,num_records5) ; 16
		pos1_0_a_eng	= dblarr(32,8,num_records1) ; 16
		pos1_0_a_sci[*,*,*]	= !values.d_nan
		pos1_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_a_sci = dblarr(num_records5,32) ;16
		sum_defl_pos1_0_a_eng = dblarr(num_records1,32) ;16
		sum_defl_pos1_0_a_sci[*,*] = !values.d_nan ;16
		sum_defl_pos1_0_a_eng[*,*] = !values.d_nan ;16
		END
		'pos1_1':  BEGIN
		pos1_1_a_sci	= dblarr(32,8,num_records5) ; 17
		pos1_1_a_eng	= dblarr(32,8,num_records1) ; 17
		pos1_1_a_sci[*,*,*]	= !values.d_nan
		pos1_1_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_a_sci = dblarr(num_records5,32) ;17
		sum_defl_pos1_1_a_eng = dblarr(num_records1,32) ;17
		sum_defl_pos1_1_a_sci[*,*] = !values.d_nan ;17
		sum_defl_pos1_1_a_eng[*,*] = !values.d_nan ;17
		END
		'w_no_pos':  BEGIN
		w_no_pos_a_sci	= dblarr(32,8,num_records5) ; 18
		w_no_pos_a_eng	= dblarr(32,8,num_records1) ; 18
		w_no_pos_a_sci[*,*,*]	= !values.d_nan
		w_no_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_a_sci = dblarr(num_records5,32) ;18
		sum_defl_w_no_pos_a_eng = dblarr(num_records1,32) ;18
		sum_defl_w_no_pos_a_sci[*,*] = !values.d_nan ;18
		sum_defl_w_no_pos_a_eng[*,*] = !values.d_nan ;18
		END
		'w_mult_pos':  BEGIN
		w_mult_pos_a_sci	= dblarr(32,8,num_records5) ; 19
		w_mult_pos_a_eng	= dblarr(32,8,num_records1) ; 19
		w_mult_pos_a_sci[*,*,*]	= !values.d_nan
		w_mult_pos_a_eng[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_a_sci = dblarr(num_records5,32) ;19
		sum_defl_w_mult_pos_a_eng = dblarr(num_records1,32) ;19
		sum_defl_w_mult_pos_a_sci[*,*] = !values.d_nan ;19
		sum_defl_w_mult_pos_a_eng[*,*] = !values.d_nan ;19
		END
		'w_valid':  BEGIN
		w_valid_a_sci	= dblarr(32,8,num_records5) ; 20
		w_valid_a_eng	= dblarr(32,8,num_records1) ; 20
		w_valid_a_sci[*,*,*]	= !values.d_nan
		w_valid_a_eng[*,*,*]	= !values.d_nan
		sum_defl_w_valid_a_sci = dblarr(num_records5,32) ;20
		sum_defl_w_valid_a_eng = dblarr(num_records1,32) ;20
		sum_defl_w_valid_a_sci[*,*] = !values.d_nan ;20
		sum_defl_w_valid_a_eng[*,*] = !values.d_nan ;20
		END
		'sf2':  BEGIN
		sf2_a_sci	= dblarr(32,8,num_records5) ; 21
		sf2_a_eng	= dblarr(32,8,num_records1) ; 21
		sf2_a_sci[*,*,*]	= !values.d_nan
		sf2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_a_eng = dblarr(num_records1,32) ;21
		sum_defl_sf2_a_sci[*,*] = !values.d_nan ;21
		sum_defl_sf2_a_eng[*,*] = !values.d_nan ;21
		END
		'sfr2':  BEGIN
		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sfr2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_a_eng[*,*] = !values.d_nan ;22
		END
		'stp2':  BEGIN
		stp2_a_sci	= dblarr(32,8,num_records5) ; 23
		stp2_a_eng	= dblarr(32,8,num_records1) ; 23
		stp2_a_sci[*,*,*]	= !values.d_nan	
		stp2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_a_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_a_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_a_eng[*,*] = !values.d_nan ;23
		END
		'pos2_0':  BEGIN
		pos2_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_a_eng	= dblarr(32,8,num_records1) ; 24
		pos2_0_a_sci[*,*,*]	= !values.d_nan	
		pos2_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_a_eng = dblarr(num_records1,32) ;24
		sum_defl_pos2_0_a_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos2_0_a_eng[*,*] = !values.d_nan ;24
		END
		'pos2_1':  BEGIN
		pos2_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_a_eng	= dblarr(32,8,num_records1) ; 25
		pos2_1_a_sci[*,*,*]	= !values.d_nan	
		pos2_1_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_a_eng = dblarr(num_records1,32) ;25
		sum_defl_pos2_1_a_sci[*,*] = !values.d_nan ;25
		sum_defl_pos2_1_a_eng[*,*] = !values.d_nan ;25
		END
		'pos2_2':  BEGIN
		pos2_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_a_eng	= dblarr(32,8,num_records1) ; 26
		pos2_2_a_sci[*,*,*]	= !values.d_nan
		pos2_2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_a_eng = dblarr(num_records1,32) ;26
		sum_defl_pos2_2_a_sci[*,*] = !values.d_nan ;26
		sum_defl_pos2_2_a_eng[*,*] = !values.d_nan ;26
		END
		'pos2_3':  BEGIN
		pos2_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_a_eng	= dblarr(32,8,num_records1) ; 27
		pos2_3_a_sci[*,*,*]	= !values.d_nan	
		pos2_3_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_a_eng = dblarr(num_records1,32) ;27
		sum_defl_pos2_3_a_sci[*,*] = !values.d_nan ;27
		sum_defl_pos2_3_a_eng[*,*] = !values.d_nan ;27
		END
		'pos3_0':  BEGIN
		pos3_0_a_sci	= dblarr(32,8,num_records5) ; 28
		pos3_0_a_eng	= dblarr(32,8,num_records1) ; 28
		pos3_0_a_sci[*,*,*]	= !values.d_nan	
		pos3_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_0_a_sci = dblarr(num_records5,32) ;28
		sum_defl_pos3_0_a_eng = dblarr(num_records1,32) ;28
		sum_defl_pos3_0_a_sci[*,*] = !values.d_nan ;28
		sum_defl_pos3_0_a_eng[*,*] = !values.d_nan ;28
		END
		'pos3_1':  BEGIN
		pos3_1_a_sci	= dblarr(32,8,num_records5) ; 29
		pos3_1_a_eng	= dblarr(32,8,num_records1) ; 29
		pos3_1_a_sci[*,*,*]	= !values.d_nan	
		pos3_1_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_1_a_sci = dblarr(num_records5,32) ;29
		sum_defl_pos3_1_a_eng = dblarr(num_records1,32) ;29
		sum_defl_pos3_1_a_sci[*,*] = !values.d_nan ;29
		sum_defl_pos3_1_a_eng[*,*] = !values.d_nan ;29
		END
		'pos3_2':  BEGIN
		pos3_2_a_sci	= dblarr(32,8,num_records5) ; 30
		pos3_2_a_eng	= dblarr(32,8,num_records1) ; 30
		pos3_2_a_sci[*,*,*]	= !values.d_nan
		pos3_2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_sci = dblarr(num_records5,32) ;30
		sum_defl_pos3_2_a_eng = dblarr(num_records1,32) ;30
		sum_defl_pos3_2_a_sci[*,*] = !values.d_nan ;30
		sum_defl_pos3_2_a_eng[*,*] = !values.d_nan ;30
		END
		'pos3_3':  BEGIN
		pos3_3_a_sci	= dblarr(32,8,num_records5) ; 31
		pos3_3_a_eng	= dblarr(32,8,num_records1) ; 31
		pos3_3_a_sci[*,*,*]	= !values.d_nan
		pos3_3_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_a_sci = dblarr(num_records5,32) ;31
		sum_defl_pos3_3_a_eng = dblarr(num_records1,32) ;31
		sum_defl_pos3_3_a_sci[*,*] = !values.d_nan ;31
		sum_defl_pos3_3_a_eng[*,*] = !values.d_nan ;31
		END
		'pos1':  BEGIN
		pos1_a_sci	= dblarr(32,8,num_records5) 
		pos1_a_eng	= dblarr(32,8,num_records1) 
		pos1_a_sci[*,*,*]	= !values.d_nan
		pos1_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_a_sci = dblarr(num_records5,32) 
		sum_defl_pos1_a_eng = dblarr(num_records1,32) 
		sum_defl_pos1_a_sci[*,*] = !values.d_nan 
		sum_defl_pos1_a_eng[*,*] = !values.d_nan 

		pos1_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos1_0_a_eng	= dblarr(32,8,num_records1) ; 24
		pos1_0_a_sci[*,*,*]	= !values.d_nan	
		pos1_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos1_0_a_eng = dblarr(num_records1,32) ;24
		sum_defl_pos1_0_a_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos1_0_a_eng[*,*] = !values.d_nan ;24

		pos1_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos1_1_a_eng	= dblarr(32,8,num_records1) ; 25
		pos1_1_a_sci[*,*,*]	= !values.d_nan	
		pos1_1_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos1_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos1_1_a_eng = dblarr(num_records1,32) ;25
		sum_defl_pos1_1_a_sci[*,*] = !values.d_nan ;25
		sum_defl_pos1_1_a_eng[*,*] = !values.d_nan ;25
		
		END
		'pos2':  BEGIN
		pos2_a_sci	= dblarr(32,8,num_records5) 
		pos2_a_eng	= dblarr(32,8,num_records1) 
		pos2_a_sci[*,*,*]	= !values.d_nan
		pos2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_a_sci = dblarr(num_records5,32) 
		sum_defl_pos2_a_eng = dblarr(num_records1,32) 
		sum_defl_pos2_a_sci[*,*] = !values.d_nan 
		sum_defl_pos2_a_eng[*,*] = !values.d_nan 

		pos2_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_a_eng	= dblarr(32,8,num_records1) ; 24
		pos2_0_a_sci[*,*,*]	= !values.d_nan	
		pos2_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_a_eng = dblarr(num_records1,32) ;24
		sum_defl_pos2_0_a_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos2_0_a_eng[*,*] = !values.d_nan ;24

		pos2_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_a_eng	= dblarr(32,8,num_records1) ; 25
		pos2_1_a_sci[*,*,*]	= !values.d_nan	
		pos2_1_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_a_eng = dblarr(num_records1,32) ;25
		sum_defl_pos2_1_a_sci[*,*] = !values.d_nan ;25
		sum_defl_pos2_1_a_eng[*,*] = !values.d_nan ;25

		pos2_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_a_eng	= dblarr(32,8,num_records1) ; 26
		pos2_2_a_sci[*,*,*]	= !values.d_nan
		pos2_2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_a_eng = dblarr(num_records1,32) ;26
		sum_defl_pos2_2_a_sci[*,*] = !values.d_nan ;26
		sum_defl_pos2_2_a_eng[*,*] = !values.d_nan ;26

		pos2_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_a_eng	= dblarr(32,8,num_records1) ; 27
		pos2_3_a_sci[*,*,*]	= !values.d_nan	
		pos2_3_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_a_eng = dblarr(num_records1,32) ;27
		sum_defl_pos2_3_a_sci[*,*] = !values.d_nan ;27
		sum_defl_pos2_3_a_eng[*,*] = !values.d_nan ;27
		
		END
		'pos3':  BEGIN
		pos3_a_sci	= dblarr(32,8,num_records5) 
		pos3_a_eng	= dblarr(32,8,num_records1) 
		pos3_a_sci[*,*,*]	= !values.d_nan
		pos3_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_a_sci = dblarr(num_records5,32) 
		sum_defl_pos3_a_eng = dblarr(num_records1,32) 
		sum_defl_pos3_a_sci[*,*] = !values.d_nan 
		sum_defl_pos3_a_eng[*,*] = !values.d_nan 

		pos3_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos3_0_a_eng	= dblarr(32,8,num_records1) ; 24
		pos3_0_a_sci[*,*,*]	= !values.d_nan	
		pos3_0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos3_0_a_eng = dblarr(num_records1,32) ;24
		sum_defl_pos3_0_a_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos3_0_a_eng[*,*] = !values.d_nan ;24

		pos3_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos3_1_a_eng	= dblarr(32,8,num_records1) ; 25
		pos3_1_a_sci[*,*,*]	= !values.d_nan	
		pos3_1_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos3_1_a_eng = dblarr(num_records1,32) ;25
		sum_defl_pos3_1_a_sci[*,*] = !values.d_nan ;25
		sum_defl_pos3_1_a_eng[*,*] = !values.d_nan ;25

		pos3_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos3_2_a_eng	= dblarr(32,8,num_records1) ; 26
		pos3_2_a_sci[*,*,*]	= !values.d_nan
		pos3_2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos3_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos3_2_a_eng = dblarr(num_records1,32) ;26
		sum_defl_pos3_2_a_sci[*,*] = !values.d_nan ;26
		sum_defl_pos3_2_a_eng[*,*] = !values.d_nan ;26

		pos3_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos3_3_a_eng	= dblarr(32,8,num_records1) ; 27
		pos3_3_a_sci[*,*,*]	= !values.d_nan	
		pos3_3_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos3_3_a_eng = dblarr(num_records1,32) ;27
		sum_defl_pos3_3_a_sci[*,*] = !values.d_nan ;27
		sum_defl_pos3_3_a_eng[*,*] = !values.d_nan ;27
		
		END

		'stop0':  BEGIN
		stop0_a_sci	= dblarr(32,8,num_records5) 
		stop0_a_eng	= dblarr(32,8,num_records1) 
		stop0_a_sci[*,*,*]	= !values.d_nan
		stop0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stop0_a_sci = dblarr(num_records5,32) 
		sum_defl_stop0_a_eng = dblarr(num_records1,32) 
		sum_defl_stop0_a_sci[*,*] = !values.d_nan 
		sum_defl_stop0_a_eng[*,*] = !values.d_nan 

		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_eng	= dblarr(32,8,num_records1) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sf0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_a_eng[*,*] = !values.d_nan ;12	

		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sfr0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_a_eng[*,*] = !values.d_nan ;13
	
		END

		'start0':  BEGIN
		start0_a_sci	= dblarr(32,8,num_records5) 
		start0_a_eng	= dblarr(32,8,num_records1) 
		start0_a_sci[*,*,*]	= !values.d_nan
		start0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_start0_a_sci = dblarr(num_records5,32) 
		sum_defl_start0_a_eng = dblarr(num_records1,32)
		sum_defl_start0_a_sci[*,*] = !values.d_nan
		sum_defl_start0_a_eng[*,*] = !values.d_nan

		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sfr0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_a_eng[*,*] = !values.d_nan ;13

		stp0_a_sci	= dblarr(32,8,num_records5) ; 14
		stp0_a_eng	= dblarr(32,8,num_records1) ; 14
		stp0_a_sci[*,*,*]	= !values.d_nan
		stp0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_a_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_a_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_a_eng[*,*] = !values.d_nan ;14

		END
		'stop2':  BEGIN
		stop2_a_sci	= dblarr(32,8,num_records5)
		stop2_a_eng	= dblarr(32,8,num_records1)
		stop2_a_sci[*,*,*]	= !values.d_nan
		stop2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_stop2_a_sci = dblarr(num_records5,32)
		sum_defl_stop2_a_eng = dblarr(num_records1,32)
		sum_defl_stop2_a_sci[*,*] = !values.d_nan
		sum_defl_stop2_a_eng[*,*] = !values.d_nan

		sf2_a_sci	= dblarr(32,8,num_records5) ; 21
		sf2_a_eng	= dblarr(32,8,num_records1) ; 21
		sf2_a_sci[*,*,*]	= !values.d_nan
		sf2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_a_eng = dblarr(num_records1,32) ;21
		sum_defl_sf2_a_sci[*,*] = !values.d_nan ;21
		sum_defl_sf2_a_eng[*,*] = !values.d_nan ;21
		
		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sfr2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_a_eng[*,*] = !values.d_nan ;22
		END

		'start2':  BEGIN
		start2_a_sci	= dblarr(32,8,num_records5)
		start2_a_eng	= dblarr(32,8,num_records1) 
		start2_a_sci[*,*,*]	= !values.d_nan
		start2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_start2_a_sci = dblarr(num_records5,32) 
		sum_defl_start2_a_eng = dblarr(num_records1,32) 
		sum_defl_start2_a_sci[*,*] = !values.d_nan 
		sum_defl_start2_a_eng[*,*] = !values.d_nan 

		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sfr2_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_a_eng[*,*] = !values.d_nan ;22

		stp2_a_sci	= dblarr(32,8,num_records5) ; 23
		stp2_a_eng	= dblarr(32,8,num_records1) ; 23
		stp2_a_sci[*,*,*]	= !values.d_nan	
		stp2_a_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_a_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_a_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_a_eng[*,*] = !values.d_nan ;23
		END

		'sf0ra_trig': BEGIN
		
		sf0ra_trig_a_sci= dblarr(32,8,num_records5)
		sf0ra_trig_a_sci[*,*,*] = !values.d_nan
		sum_defl_sf0ra_trig_a_sci=dblarr(num_records5,32)
		sum_defl_sf0ra_trig_a_sci[*,*] = !values.d_nan

		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_eng	= dblarr(32,8,num_records1) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sf0_a_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_a_eng[*,*] = !values.d_nan ;12

		ra_trig_a_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_a_eng	= dblarr(32,8,num_records1) ; 15
		ra_trig_a_sci[*,*,*]	= !values.d_nan
		ra_trig_a_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_a_eng = dblarr(num_records1,32) ;15
		sum_defl_ra_trig_a_sci[*,*] = !values.d_nan ;15
		sum_defl_ra_trig_a_eng[*,*] = !values.d_nan ;15
		END		

		ENDCASE
		ENDFOR			
    	END 
    	'B': BEGIN 

	error5_mon_b	= fltarr(num_records5) ; error flag science mode
	error5_mon_b[*]	= !values.f_nan
	error1_b	= fltarr(num_records1) ; error flag engineering mode
	error1_b[*]	= !values.f_nan	


		FOR ii = 0, n_elements(products)-1 DO BEGIN 
        	CASE products[ii] OF 
	
		's_valid':  BEGIN
		s_valid_b_sci	= dblarr(32,8,num_records5) ; 0
		s_valid_b_eng	= dblarr(32,8,num_records1) ;0
		s_valid_b_sci[*,*,*] 	= !values.d_nan
		s_valid_b_eng[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_b_sci = dblarr(num_records5,32) ;0
		sum_defl_s_valid_b_eng = dblarr(num_records1,32) ;0
		sum_defl_s_valid_b_sci[*,*] = !values.d_nan ;0
		sum_defl_s_valid_b_eng[*,*] = !values.d_nan ;0
		END
		's_e_not_req':  BEGIN
		s_e_not_req_b_sci	= dblarr(32,8,num_records5) ; 1
		s_e_not_req_b_eng	= dblarr(32,8,num_records1) ; 1
		s_e_not_req_b_sci[*,*,*]	= !values.d_nan
		s_e_not_req_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_b_sci = dblarr(num_records5,32) ;1
		sum_defl_s_e_not_req_b_eng = dblarr(num_records1,32) ;1
		sum_defl_s_e_not_req_b_sci[*,*] = !values.d_nan ;1
		sum_defl_s_e_not_req_b_eng[*,*] = !values.d_nan ;1
		END
		's_e_req':  BEGIN
		s_e_req_b_sci	= dblarr(32,8,num_records5) ; 2
		s_e_req_b_eng	= dblarr(32,8,num_records1) ; 2
		s_e_req_b_sci[*,*,*]	= !values.d_nan
		s_e_req_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_b_sci = dblarr(num_records5,32) ;2
		sum_defl_s_e_req_b_eng = dblarr(num_records1,32) ;2
		sum_defl_s_e_req_b_sci[*,*] = !values.d_nan ;2
		sum_defl_s_e_req_b_eng[*,*] = !values.d_nan ;2
		END
		's_no_pos':  BEGIN
		s_no_pos_b_sci	= dblarr(32,8,num_records5) ; 3
		s_no_pos_b_eng	= dblarr(32,8,num_records1) ; 3
		s_no_pos_b_sci[*,*,*]	= !values.d_nan
		s_no_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_b_sci = dblarr(num_records5,32) ;3
		sum_defl_s_no_pos_b_eng = dblarr(num_records1,32) ;3
		sum_defl_s_no_pos_b_sci[*,*] = !values.d_nan ;3
		sum_defl_s_no_pos_b_eng[*,*] = !values.d_nan ;3
		END
		's_mult_pos':  BEGIN
		s_mult_pos_b_sci	= dblarr(32,8,num_records5) ; 4
		s_mult_pos_b_eng	= dblarr(32,8,num_records1) ; 4
		s_mult_pos_b_sci[*,*,*]	= !values.d_nan
		s_mult_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_b_sci = dblarr(num_records5,32) ;4
		sum_defl_s_mult_pos_b_eng = dblarr(num_records1,32) ;4
		sum_defl_s_mult_pos_b_sci[*,*] = !values.d_nan ;4
		sum_defl_s_mult_pos_b_eng[*,*] = !values.d_nan ;4
		END
		's_no_e':  BEGIN
		s_no_e_b_sci	= dblarr(32,8,num_records5) ; 5
		s_no_e_b_eng	= dblarr(32,8,num_records1) ; 5
		s_no_e_b_sci[*,*,*]	= !values.d_nan
		s_no_e_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_b_sci = dblarr(num_records5,32) ;5
		sum_defl_s_no_e_b_eng = dblarr(num_records1,32) ;5
		sum_defl_s_no_e_b_sci[*,*] = !values.d_nan ;5
		sum_defl_s_no_e_b_eng[*,*] = !values.d_nan ;5
		END
		's_mult_e':  BEGIN
		s_mult_e_b_sci	= dblarr(32,8,num_records5) ; 6
		s_mult_e_b_eng	= dblarr(32,8,num_records1) ; 6
		s_mult_e_b_sci[*,*,*]	= !values.d_nan
		s_mult_e_b_eng[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_b_sci = dblarr(num_records5,32) ;6
		sum_defl_s_mult_e_b_eng = dblarr(num_records1,32) ;6
		sum_defl_s_mult_e_b_sci[*,*] = !values.d_nan ;6
		sum_defl_s_mult_e_b_eng[*,*] = !values.d_nan ;6
		END
		'ra_sat_a':  BEGIN
		ra_sat_a_b_sci	= dblarr(32,8,num_records5) ; 7
		ra_sat_a_b_eng	= dblarr(32,8,num_records1) ; 7
		ra_sat_a_b_sci[*,*,*]	= !values.d_nan
		ra_sat_a_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_b_sci = dblarr(num_records5,32) ;7
		sum_defl_ra_sat_a_b_eng = dblarr(num_records1,32) ;7
		sum_defl_ra_sat_a_b_sci[*,*] = !values.d_nan ;7
		sum_defl_ra_sat_a_b_eng[*,*] = !values.d_nan ;7
		END
		'ra_sat_b':  BEGIN
		ra_sat_b_b_sci	= dblarr(32,8,num_records5) ; 8
		ra_sat_b_b_eng	= dblarr(32,8,num_records1) ; 8
		ra_sat_b_b_sci[*,*,*]= !values.d_nan
		ra_sat_b_b_eng[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_b_sci = dblarr(num_records5,32) ;8
		sum_defl_ra_sat_b_b_eng = dblarr(num_records1,32) ;8
		sum_defl_ra_sat_b_b_sci[*,*] = !values.d_nan ;8
		sum_defl_ra_sat_b_b_eng[*,*] = !values.d_nan ;8
		END
		'ra_sat_both':  BEGIN
		ra_sat_both_b_sci	= dblarr(32,8,num_records5) ; 9
		ra_sat_both_b_eng	= dblarr(32,8,num_records1) ; 9
		ra_sat_both_b_sci[*,*,*]	= !values.d_nan
		ra_sat_both_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_b_sci = dblarr(num_records5,32) ;9
		sum_defl_ra_sat_both_b_eng = dblarr(num_records1,32) ;9
		sum_defl_ra_sat_both_b_sci[*,*] = !values.d_nan ;9
		sum_defl_ra_sat_both_b_eng[*,*] = !values.d_nan ;9
		END
		'ssd_sw':  BEGIN
		ssd_sw_b_sci	= dblarr(32,8,num_records5) ; 10
		ssd_sw_b_eng	= dblarr(32,8,num_records1) ; 10
		ssd_sw_b_sci[*,*,*]	= !values.d_nan
		ssd_sw_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_b_sci = dblarr(num_records5,32) ;10
		sum_defl_ssd_sw_b_eng = dblarr(num_records1,32) ;10
		sum_defl_ssd_sw_b_sci[*,*] = !values.d_nan ;10
		sum_defl_ssd_sw_b_eng[*,*] = !values.d_nan ;10
		END
		'ssd_st':  BEGIN
		ssd_st_b_sci	= dblarr(32,8,num_records5) ; 11
		ssd_st_b_eng	= dblarr(32,8,num_records1) ; 11
		ssd_st_b_sci[*,*,*]	= !values.d_nan
		ssd_st_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_b_sci = dblarr(num_records5,32) ;11
		sum_defl_ssd_st_b_eng = dblarr(num_records1,32) ;11
		sum_defl_ssd_st_b_sci[*,*] = !values.d_nan ;11
		sum_defl_ssd_st_b_eng[*,*] = !values.d_nan ;11
		END
		'sf0':  BEGIN
		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_eng	= dblarr(32,8,num_records1) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sf0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_b_eng[*,*] = !values.d_nan ;12		
		END
		'sfr0':  BEGIN
		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sfr0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_b_eng[*,*] = !values.d_nan ;13
		END
		'stp0':  BEGIN
		stp0_b_sci	= dblarr(32,8,num_records5) ; 14
		stp0_b_eng	= dblarr(32,8,num_records1) ; 14
		stp0_b_sci[*,*,*]	= !values.d_nan
		stp0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_b_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_b_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_b_eng[*,*] = !values.d_nan ;14
		END
		'ra_trig':  BEGIN
		ra_trig_b_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_b_eng	= dblarr(32,8,num_records1) ; 15
		ra_trig_b_sci[*,*,*]	= !values.d_nan
		ra_trig_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_b_eng = dblarr(num_records1,32) ;15
		sum_defl_ra_trig_b_sci[*,*] = !values.d_nan ;15
		sum_defl_ra_trig_b_eng[*,*] = !values.d_nan ;15
		END
		'pos1_0':  BEGIN
		pos1_0_b_sci	= dblarr(32,8,num_records5) ; 16
		pos1_0_b_eng	= dblarr(32,8,num_records1) ; 16
		pos1_0_b_sci[*,*,*]	= !values.d_nan
		pos1_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_b_sci = dblarr(num_records5,32) ;16
		sum_defl_pos1_0_b_eng = dblarr(num_records1,32) ;16
		sum_defl_pos1_0_b_sci[*,*] = !values.d_nan ;16
		sum_defl_pos1_0_b_eng[*,*] = !values.d_nan ;16
		END
		'pos1_1':  BEGIN
		pos1_1_b_sci	= dblarr(32,8,num_records5) ; 17
		pos1_1_b_eng	= dblarr(32,8,num_records1) ; 17
		pos1_1_b_sci[*,*,*]	= !values.d_nan
		pos1_1_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_b_sci = dblarr(num_records5,32) ;17
		sum_defl_pos1_1_b_eng = dblarr(num_records1,32) ;17
		sum_defl_pos1_1_b_sci[*,*] = !values.d_nan ;17
		sum_defl_pos1_1_b_eng[*,*] = !values.d_nan ;17
		END
		'w_no_pos':  BEGIN
		w_no_pos_b_sci	= dblarr(32,8,num_records5) ; 18
		w_no_pos_b_eng	= dblarr(32,8,num_records1) ; 18
		w_no_pos_b_sci[*,*,*]	= !values.d_nan
		w_no_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_b_sci = dblarr(num_records5,32) ;18
		sum_defl_w_no_pos_b_eng = dblarr(num_records1,32) ;18
		sum_defl_w_no_pos_b_sci[*,*] = !values.d_nan ;18
		sum_defl_w_no_pos_b_eng[*,*] = !values.d_nan ;18
		END
		'w_mult_pos':  BEGIN
		w_mult_pos_b_sci	= dblarr(32,8,num_records5) ; 19
		w_mult_pos_b_eng	= dblarr(32,8,num_records1) ; 19
		w_mult_pos_b_sci[*,*,*]	= !values.d_nan
		w_mult_pos_b_eng[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_b_sci = dblarr(num_records5,32) ;19
		sum_defl_w_mult_pos_b_eng = dblarr(num_records1,32) ;19
		sum_defl_w_mult_pos_b_sci[*,*] = !values.d_nan ;19
		sum_defl_w_mult_pos_b_eng[*,*] = !values.d_nan ;19
		END
		'w_valid':  BEGIN
		w_valid_b_sci	= dblarr(32,8,num_records5) ; 20
		w_valid_b_eng	= dblarr(32,8,num_records1) ; 20
		w_valid_b_sci[*,*,*]	= !values.d_nan
		w_valid_b_eng[*,*,*]	= !values.d_nan
		sum_defl_w_valid_b_sci = dblarr(num_records5,32) ;20
		sum_defl_w_valid_b_eng = dblarr(num_records1,32) ;20
		sum_defl_w_valid_b_sci[*,*] = !values.d_nan ;20
		sum_defl_w_valid_b_eng[*,*] = !values.d_nan ;20
		END
		'sf2':  BEGIN
		sf2_b_sci	= dblarr(32,8,num_records5) ; 21
		sf2_b_eng	= dblarr(32,8,num_records1) ; 21
		sf2_b_sci[*,*,*]	= !values.d_nan
		sf2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_b_eng = dblarr(num_records1,32) ;21
		sum_defl_sf2_b_sci[*,*] = !values.d_nan ;21
		sum_defl_sf2_b_eng[*,*] = !values.d_nan ;21
		END
		'sfr2':  BEGIN
		sfr2_b_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_b_eng	= dblarr(32,8,num_records1) ; 22
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sfr2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_b_eng = dblarr(num_records1,32) ;22
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;22
		sum_defl_sfr2_b_eng[*,*] = !values.d_nan ;22
		END
		'stp2':  BEGIN
		stp2_b_sci	= dblarr(32,8,num_records5) ; 23
		stp2_b_eng	= dblarr(32,8,num_records1) ; 23
		stp2_b_sci[*,*,*]	= !values.d_nan	
		stp2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_b_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_b_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_b_eng[*,*] = !values.d_nan ;23
		END
		'pos2_0':  BEGIN
		pos2_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_b_eng	= dblarr(32,8,num_records1) ; 24
		pos2_0_b_sci[*,*,*]	= !values.d_nan	
		pos2_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_b_eng = dblarr(num_records1,32) ;24
		sum_defl_pos2_0_b_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos2_0_b_eng[*,*] = !values.d_nan ;24
		END
		'pos2_1':  BEGIN
		pos2_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_b_eng	= dblarr(32,8,num_records1) ; 25
		pos2_1_b_sci[*,*,*]	= !values.d_nan	
		pos2_1_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_b_eng = dblarr(num_records1,32) ;25
		sum_defl_pos2_1_b_sci[*,*] = !values.d_nan ;25
		sum_defl_pos2_1_b_eng[*,*] = !values.d_nan ;25
		END
		'pos2_2':  BEGIN
		pos2_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_b_eng	= dblarr(32,8,num_records1) ; 26
		pos2_2_b_sci[*,*,*]	= !values.d_nan
		pos2_2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_b_eng = dblarr(num_records1,32) ;26
		sum_defl_pos2_2_b_sci[*,*] = !values.d_nan ;26
		sum_defl_pos2_2_b_eng[*,*] = !values.d_nan ;26
		END
		'pos2_3':  BEGIN
		pos2_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_b_eng	= dblarr(32,8,num_records1) ; 27
		pos2_3_b_sci[*,*,*]	= !values.d_nan	
		pos2_3_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_b_eng = dblarr(num_records1,32) ;27
		sum_defl_pos2_3_b_sci[*,*] = !values.d_nan ;27
		sum_defl_pos2_3_b_eng[*,*] = !values.d_nan ;27
		END
		'pos3_0':  BEGIN
		pos3_0_b_sci	= dblarr(32,8,num_records5) ; 28
		pos3_0_b_eng	= dblarr(32,8,num_records1) ; 28
		pos3_0_b_sci[*,*,*]	= !values.d_nan	
		pos3_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_0_b_sci = dblarr(num_records5,32) ;28
		sum_defl_pos3_0_b_eng = dblarr(num_records1,32) ;28
		sum_defl_pos3_0_b_sci[*,*] = !values.d_nan ;28
		sum_defl_pos3_0_b_eng[*,*] = !values.d_nan ;28
		END
		'pos3_1':  BEGIN
		pos3_1_b_sci	= dblarr(32,8,num_records5) ; 29
		pos3_1_b_eng	= dblarr(32,8,num_records1) ; 29
		pos3_1_b_sci[*,*,*]	= !values.d_nan	
		pos3_1_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_1_b_sci = dblarr(num_records5,32) ;29
		sum_defl_pos3_1_b_eng = dblarr(num_records1,32) ;29
		sum_defl_pos3_1_b_sci[*,*] = !values.d_nan ;29
		sum_defl_pos3_1_b_eng[*,*] = !values.d_nan ;29
		END
		'pos3_2':  BEGIN
		pos3_2_b_sci	= dblarr(32,8,num_records5) ; 30
		pos3_2_b_eng	= dblarr(32,8,num_records1) ; 30
		pos3_2_b_sci[*,*,*]	= !values.d_nan
		pos3_2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_sci = dblarr(num_records5,32) ;30
		sum_defl_pos3_2_b_eng = dblarr(num_records1,32) ;30
		sum_defl_pos3_2_b_sci[*,*] = !values.d_nan ;30
		sum_defl_pos3_2_b_eng[*,*] = !values.d_nan ;30
		END
		'pos3_3':  BEGIN
		pos3_3_b_sci	= dblarr(32,8,num_records5) ; 31
		pos3_3_b_eng	= dblarr(32,8,num_records1) ; 31
		pos3_3_b_sci[*,*,*]	= !values.d_nan
		pos3_3_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_b_sci = dblarr(num_records5,32) ;31
		sum_defl_pos3_3_b_eng = dblarr(num_records1,32) ;31
		sum_defl_pos3_3_b_sci[*,*] = !values.d_nan ;31
		sum_defl_pos3_3_b_eng[*,*] = !values.d_nan ;31
		END
		'pos1':  BEGIN
		pos1_b_sci	= dblarr(32,8,num_records5) 
		pos1_b_eng	= dblarr(32,8,num_records1) 
		pos1_b_sci[*,*,*]	= !values.d_nan
		pos1_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_b_sci = dblarr(num_records5,32) 
		sum_defl_pos1_b_eng = dblarr(num_records1,32) 
		sum_defl_pos1_b_sci[*,*] = !values.d_nan 
		sum_defl_pos1_b_eng[*,*] = !values.d_nan 

		pos1_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos1_0_b_eng	= dblarr(32,8,num_records1) ; 24
		pos1_0_b_sci[*,*,*]	= !values.d_nan	
		pos1_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos1_0_b_eng = dblarr(num_records1,32) ;24
		sum_defl_pos1_0_b_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos1_0_b_eng[*,*] = !values.d_nan ;24

		pos1_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos1_1_b_eng	= dblarr(32,8,num_records1) ; 25
		pos1_1_b_sci[*,*,*]	= !values.d_nan	
		pos1_1_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos1_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos1_1_b_eng = dblarr(num_records1,32) ;25
		sum_defl_pos1_1_b_sci[*,*] = !values.d_nan ;25
		sum_defl_pos1_1_b_eng[*,*] = !values.d_nan ;25

		END
		'pos2':  BEGIN
		pos2_b_sci	= dblarr(32,8,num_records5) 
		pos2_b_eng	= dblarr(32,8,num_records1) 
		pos2_b_sci[*,*,*]	= !values.d_nan
		pos2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_b_sci = dblarr(num_records5,32) 
		sum_defl_pos2_b_eng = dblarr(num_records1,32) 
		sum_defl_pos2_b_sci[*,*] = !values.d_nan 
		sum_defl_pos2_b_eng[*,*] = !values.d_nan 

		pos2_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_b_eng	= dblarr(32,8,num_records1) ; 24
		pos2_0_b_sci[*,*,*]	= !values.d_nan	
		pos2_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos2_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_b_eng = dblarr(num_records1,32) ;24
		sum_defl_pos2_0_b_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos2_0_b_eng[*,*] = !values.d_nan ;24

		pos2_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_b_eng	= dblarr(32,8,num_records1) ; 25
		pos2_1_b_sci[*,*,*]	= !values.d_nan	
		pos2_1_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_b_eng = dblarr(num_records1,32) ;25
		sum_defl_pos2_1_b_sci[*,*] = !values.d_nan ;25
		sum_defl_pos2_1_b_eng[*,*] = !values.d_nan ;25

		pos2_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_b_eng	= dblarr(32,8,num_records1) ; 26
		pos2_2_b_sci[*,*,*]	= !values.d_nan
		pos2_2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_b_eng = dblarr(num_records1,32) ;26
		sum_defl_pos2_2_b_sci[*,*] = !values.d_nan ;26
		sum_defl_pos2_2_b_eng[*,*] = !values.d_nan ;26

		pos2_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_b_eng	= dblarr(32,8,num_records1) ; 27
		pos2_3_b_sci[*,*,*]	= !values.d_nan	
		pos2_3_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_b_eng = dblarr(num_records1,32) ;27
		sum_defl_pos2_3_b_sci[*,*] = !values.d_nan ;27
		sum_defl_pos2_3_b_eng[*,*] = !values.d_nan ;27
		
		END

		'pos3':  BEGIN
		pos3_b_sci	= dblarr(32,8,num_records5) 
		pos3_b_eng	= dblarr(32,8,num_records1) 
		pos3_b_sci[*,*,*]	= !values.d_nan
		pos3_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_b_sci = dblarr(num_records5,32) 
		sum_defl_pos3_b_eng = dblarr(num_records1,32) 
		sum_defl_pos3_b_sci[*,*] = !values.d_nan 
		sum_defl_pos3_b_eng[*,*] = !values.d_nan 

		pos3_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos3_0_b_eng	= dblarr(32,8,num_records1) ; 24
		pos3_0_b_sci[*,*,*]	= !values.d_nan	
		pos3_0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_pos3_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos3_0_b_eng = dblarr(num_records1,32) ;24
		sum_defl_pos3_0_b_sci[*,*] = !values.d_nan ;24	
		sum_defl_pos3_0_b_eng[*,*] = !values.d_nan ;24

		pos3_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos3_1_b_eng	= dblarr(32,8,num_records1) ; 25
		pos3_1_b_sci[*,*,*]	= !values.d_nan	
		pos3_1_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos3_1_b_eng = dblarr(num_records1,32) ;25
		sum_defl_pos3_1_b_sci[*,*] = !values.d_nan ;25
		sum_defl_pos3_1_b_eng[*,*] = !values.d_nan ;25

		pos3_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos3_2_b_eng	= dblarr(32,8,num_records1) ; 26
		pos3_2_b_sci[*,*,*]	= !values.d_nan
		pos3_2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos3_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos3_2_b_eng = dblarr(num_records1,32) ;26
		sum_defl_pos3_2_b_sci[*,*] = !values.d_nan ;26
		sum_defl_pos3_2_b_eng[*,*] = !values.d_nan ;26

		pos3_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos3_3_b_eng	= dblarr(32,8,num_records1) ; 27
		pos3_3_b_sci[*,*,*]	= !values.d_nan	
		pos3_3_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos3_3_b_eng = dblarr(num_records1,32) ;27
		sum_defl_pos3_3_b_sci[*,*] = !values.d_nan ;27
		sum_defl_pos3_3_b_eng[*,*] = !values.d_nan ;27
		
		END


		'stop0':  BEGIN
		stop0_b_sci	= dblarr(32,8,num_records5) 
		stop0_b_eng	= dblarr(32,8,num_records1) 
		stop0_b_sci[*,*,*]	= !values.d_nan
		stop0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stop0_b_sci = dblarr(num_records5,32) 
		sum_defl_stop0_b_eng = dblarr(num_records1,32) 
		sum_defl_stop0_b_sci[*,*] = !values.d_nan 
		sum_defl_stop0_b_eng[*,*] = !values.d_nan 

		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_eng	= dblarr(32,8,num_records1) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sf0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_b_eng[*,*] = !values.d_nan ;12	

		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sfr0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_b_eng[*,*] = !values.d_nan ;13
	

		END
		'start0':  BEGIN
		start0_b_sci	= dblarr(32,8,num_records5) 
		start0_b_eng	= dblarr(32,8,num_records1) 
		start0_b_sci[*,*,*]	= !values.d_nan
		start0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_start0_b_sci = dblarr(num_records5,32) 
		sum_defl_start0_b_eng = dblarr(num_records1,32)
		sum_defl_start0_b_sci[*,*] = !values.d_nan
		sum_defl_start0_b_eng[*,*] = !values.d_nan

		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sfr0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr0_b_eng[*,*] = !values.d_nan ;13

		stp0_b_sci	= dblarr(32,8,num_records5) ; 14
		stp0_b_eng	= dblarr(32,8,num_records1) ; 14
		stp0_b_sci[*,*,*]	= !values.d_nan
		stp0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_b_eng = dblarr(num_records1,32) ;14
		sum_defl_stp0_b_sci[*,*] = !values.d_nan ;14
		sum_defl_stp0_b_eng[*,*] = !values.d_nan ;14


		END
		'stop2':  BEGIN
		stop2_b_sci	= dblarr(32,8,num_records5)
		stop2_b_eng	= dblarr(32,8,num_records1)
		stop2_b_sci[*,*,*]	= !values.d_nan
		stop2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_stop2_b_sci = dblarr(num_records5,32)
		sum_defl_stop2_b_eng = dblarr(num_records1,32)
		sum_defl_stop2_b_sci[*,*] = !values.d_nan
		sum_defl_stop2_b_eng[*,*] = !values.d_nan

		sf2_b_sci	= dblarr(32,8,num_records5) ; 12
		sf2_b_eng	= dblarr(32,8,num_records1) ; 12
		sf2_b_sci[*,*,*]	= !values.d_nan
		sf2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf2_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf2_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf2_b_eng[*,*] = !values.d_nan ;12		
	

		sfr2_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr2_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sfr2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr2_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr2_b_eng[*,*] = !values.d_nan ;13

		END
		'start2':  BEGIN
		start2_b_sci	= dblarr(32,8,num_records5)
		start2_b_eng	= dblarr(32,8,num_records1) 
		start2_b_sci[*,*,*]	= !values.d_nan
		start2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_start2_b_sci = dblarr(num_records5,32) 
		sum_defl_start2_b_eng = dblarr(num_records1,32) 
		sum_defl_start2_b_sci[*,*] = !values.d_nan 
		sum_defl_start2_b_eng[*,*] = !values.d_nan 

		sfr2_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr2_b_eng	= dblarr(32,8,num_records1) ; 13
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sfr2_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr2_b_eng = dblarr(num_records1,32) ;13
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;13
		sum_defl_sfr2_b_eng[*,*] = !values.d_nan ;13

		stp2_b_sci	= dblarr(32,8,num_records5) ; 23
		stp2_b_eng	= dblarr(32,8,num_records1) ; 23
		stp2_b_sci[*,*,*]	= !values.d_nan	
		stp2_b_eng[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_b_eng = dblarr(num_records1,32) ;23
		sum_defl_stp2_b_sci[*,*] = !values.d_nan ;23
		sum_defl_stp2_b_eng[*,*] = !values.d_nan ;23

		END
		'sf0ra_trig': BEGIN
		
		sf0ra_trig_b_sci= dblarr(32,8,num_records5)
		sf0ra_trig_b_sci[*,*,*] = !values.d_nan
		sum_defl_sf0ra_trig_b_sci=dblarr(num_records5,32)
		sum_defl_sf0ra_trig_b_sci[*,*] = !values.d_nan

		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_eng	= dblarr(32,8,num_records1) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sf0_b_eng[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_eng = dblarr(num_records1,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		sum_defl_sf0_b_eng[*,*] = !values.d_nan ;12

		ra_trig_b_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_b_eng	= dblarr(32,8,num_records1) ; 15
		ra_trig_b_sci[*,*,*]	= !values.d_nan
		ra_trig_b_eng[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_b_eng = dblarr(num_records1,32) ;15
		sum_defl_ra_trig_b_sci[*,*] = !values.d_nan ;15
		sum_defl_ra_trig_b_eng[*,*] = !values.d_nan ;15
		END		

		ENDCASE
		ENDFOR			
   	 END 
	ENDCASE 

ENDIF ELSE BEGIN

CASE sat OF
    'A': BEGIN 
	
	error5_mon_a	= fltarr(num_records5) ; error flag science mode
	error5_mon_a[*]	= !values.f_nan
	error1_a	= fltarr(num_records1) ; error flag engineering mode
	error1_a[*]	= !values.f_nan	

	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        	CASE products[ii] OF 


		's_valid':  BEGIN
		s_valid_a_sci	= dblarr(32,8,num_records5) ; 0
		s_valid_a_sci[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_a_sci = dblarr(num_records5,32) ;0
		sum_defl_s_valid_a_sci[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_a_sci	= dblarr(32,8,num_records5) ; 1
		s_e_not_req_a_sci[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_a_sci = dblarr(num_records5,32) ;1
		sum_defl_s_e_not_req_a_sci[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_a_sci	= dblarr(32,8,num_records5) ; 2
		s_e_req_a_sci[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_a_sci = dblarr(num_records5,32) ;2
		sum_defl_s_e_req_a_sci[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_a_sci	= dblarr(32,8,num_records5) ; 3
		s_no_pos_a_sci[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_a_sci = dblarr(num_records5,32) ;3
		sum_defl_s_no_pos_a_sci[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_a_sci	= dblarr(32,8,num_records5) ; 4
		s_mult_pos_a_sci[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_a_sci = dblarr(num_records5,32) ;4
		sum_defl_s_mult_pos_a_sci[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_a_sci	= dblarr(32,8,num_records5) ; 5
		s_no_e_a_sci[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_a_sci = dblarr(num_records5,32) ;5
		sum_defl_s_no_e_a_sci[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_a_sci	= dblarr(32,8,num_records5) ; 6
		s_mult_e_a_sci[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_a_sci = dblarr(num_records5,32) ;6
		sum_defl_s_mult_e_a_sci[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_a_sci	= dblarr(32,8,num_records5) ; 7
		ra_sat_a_a_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_a_sci = dblarr(num_records5,32) ;7
		sum_defl_ra_sat_a_a_sci[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_a_sci	= dblarr(32,8,num_records5) ; 8
		ra_sat_b_a_sci[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_a_sci = dblarr(num_records5,32) ;8
		sum_defl_ra_sat_b_a_sci[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_a_sci	= dblarr(32,8,num_records5) ; 9
		ra_sat_both_a_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_a_sci = dblarr(num_records5,32) ;9
		sum_defl_ra_sat_both_a_sci[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_a_sci	= dblarr(32,8,num_records5) ; 10
		ssd_sw_a_sci[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_a_sci = dblarr(num_records5,32) ;10
		sum_defl_ssd_sw_a_sci[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_a_sci	= dblarr(32,8,num_records5) ; 11
		ssd_st_a_sci[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_a_sci = dblarr(num_records5,32) ;11
		sum_defl_ssd_st_a_sci[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		END
	'sfr0':  BEGIN
		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_a_sci	= dblarr(32,8,num_records5) ; 14
		stp0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_a_sci[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_a_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_a_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_a_sci[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_a_sci	= dblarr(32,8,num_records5) ; 16
		pos1_0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_a_sci = dblarr(num_records5,32) ;16
		sum_defl_pos1_0_a_sci[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_a_sci	= dblarr(32,8,num_records5) ; 17
		pos1_1_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_a_sci = dblarr(num_records5,32) ;17
		sum_defl_pos1_1_a_sci[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_a_sci	= dblarr(32,8,num_records5) ; 18
		w_no_pos_a_sci[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_a_sci = dblarr(num_records5,32) ;18
		sum_defl_w_no_pos_a_sci[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_a_sci	= dblarr(32,8,num_records5) ; 19
		w_mult_pos_a_sci[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_a_sci = dblarr(num_records5,32) ;19
		sum_defl_w_mult_pos_a_sci[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_a_sci	= dblarr(32,8,num_records5) ; 20
		w_valid_a_sci[*,*,*]	= !values.d_nan
		sum_defl_w_valid_a_sci = dblarr(num_records5,32) ;20
		sum_defl_w_valid_a_sci[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_a_sci	= dblarr(32,8,num_records5) ; 21
		sf2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_a_sci[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_a_sci	= dblarr(32,8,num_records5) ; 23
		stp2_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_a_sci[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_a_sci[*,*] = !values.d_nan ;24	
		END
	'pos2_1':  BEGIN
		pos2_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_a_sci[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_a_sci[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_a_sci[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_a_sci	= dblarr(32,8,num_records5) ; 28
		pos3_0_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_a_sci = dblarr(num_records5,32) ;28
		sum_defl_pos3_0_a_sci[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_a_sci	= dblarr(32,8,num_records5) ; 29
		pos3_1_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_sci = dblarr(num_records5,32) ;29
		sum_defl_pos3_1_a_sci[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_a_sci	= dblarr(32,8,num_records5) ; 30
		pos3_2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_sci = dblarr(num_records5,32) ;30
		sum_defl_pos3_2_a_sci[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_a_sci	= dblarr(32,8,num_records5) ; 31
		pos3_3_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_a_sci = dblarr(num_records5,32) ;31
		sum_defl_pos3_3_a_sci[*,*] = !values.d_nan ;31
		END
	'pos1':  BEGIN
		pos1_a_sci	= dblarr(32,8,num_records5) 
		pos1_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos1_a_sci = dblarr(num_records5,32) 
		sum_defl_pos1_a_sci[*,*] = !values.d_nan 
		
		pos1_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos1_0_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos1_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos1_0_a_sci[*,*] = !values.d_nan ;24	
	
		pos1_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos1_1_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos1_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos1_1_a_sci[*,*] = !values.d_nan ;25
				
		END
	'pos2':  BEGIN
		pos2_a_sci	= dblarr(32,8,num_records5) 
		pos2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos2_a_sci = dblarr(num_records5,32) 
		sum_defl_pos2_a_sci[*,*] = !values.d_nan 
		
		pos2_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_a_sci[*,*] = !values.d_nan ;24	
		
		pos2_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_a_sci[*,*] = !values.d_nan ;25
		
		pos2_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_a_sci[*,*] = !values.d_nan ;26
	
		pos2_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_a_sci[*,*] = !values.d_nan ;27
				
		END
	'pos3':  BEGIN
		pos3_a_sci	= dblarr(32,8,num_records5) 
		pos3_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_a_sci = dblarr(num_records5,32) 
		sum_defl_pos3_a_sci[*,*] = !values.d_nan 
		
		pos3_0_a_sci	= dblarr(32,8,num_records5) ; 24
		pos3_0_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_a_sci = dblarr(num_records5,32) ;24
		sum_defl_pos3_0_a_sci[*,*] = !values.d_nan ;24	
		
		pos3_1_a_sci	= dblarr(32,8,num_records5) ; 25
		pos3_1_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_a_sci = dblarr(num_records5,32) ;25
		sum_defl_pos3_1_a_sci[*,*] = !values.d_nan ;25
		
		pos3_2_a_sci	= dblarr(32,8,num_records5) ; 26
		pos3_2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_a_sci = dblarr(num_records5,32) ;26
		sum_defl_pos3_2_a_sci[*,*] = !values.d_nan ;26
		
		pos3_3_a_sci	= dblarr(32,8,num_records5) ; 27
		pos3_3_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_a_sci = dblarr(num_records5,32) ;27
		sum_defl_pos3_3_a_sci[*,*] = !values.d_nan ;27
				
		END

	'stop0':  BEGIN
		stop0_a_sci	= dblarr(32,8,num_records5) 
		stop0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_stop0_a_sci = dblarr(num_records5,32) 
		sum_defl_stop0_a_sci[*,*] = !values.d_nan 
		
		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
	
		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
			
		END

	'start0':  BEGIN
		start0_a_sci	= dblarr(32,8,num_records5) 
		start0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_start0_a_sci = dblarr(num_records5,32) 
		sum_defl_start0_a_sci[*,*] = !values.d_nan
		
		sfr0_a_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr0_a_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_a_sci[*,*] = !values.d_nan ;13
		
		stp0_a_sci	= dblarr(32,8,num_records5) ; 14
		stp0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_stp0_a_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_a_sci[*,*] = !values.d_nan ;14
		
		END
	'stop2':  BEGIN
		stop2_a_sci	= dblarr(32,8,num_records5)
		stop2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_stop2_a_sci = dblarr(num_records5,32)
		sum_defl_stop2_a_sci[*,*] = !values.d_nan
		
		sf2_a_sci	= dblarr(32,8,num_records5) ; 21
		sf2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sf2_a_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_a_sci[*,*] = !values.d_nan ;21
				
		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
		END

	'start2':  BEGIN
		start2_a_sci	= dblarr(32,8,num_records5)
		start2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_start2_a_sci = dblarr(num_records5,32) 
		sum_defl_start2_a_sci[*,*] = !values.d_nan 

		sfr2_a_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr2_a_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_a_sci[*,*] = !values.d_nan ;22
	
		stp2_a_sci	= dblarr(32,8,num_records5) ; 23
		stp2_a_sci[*,*,*]	= !values.d_nan	
		sum_defl_stp2_a_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_a_sci[*,*] = !values.d_nan ;23
		END

	'sf0ra_trig': BEGIN
		
		sf0ra_trig_a_sci= dblarr(32,8,num_records5)
		sf0ra_trig_a_sci[*,*,*] = !values.d_nan
		sum_defl_sf0ra_trig_a_sci=dblarr(num_records5,32)
		sum_defl_sf0ra_trig_a_sci[*,*] = !values.d_nan

		sf0_a_sci	= dblarr(32,8,num_records5) ; 12
		sf0_a_sci[*,*,*]	= !values.d_nan
		sum_defl_sf0_a_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_a_sci[*,*] = !values.d_nan ;12
		
		ra_trig_a_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_a_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_a_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_a_sci[*,*] = !values.d_nan ;15
		END		

		ENDCASE
		ENDFOR			
    END 
    'B': BEGIN 

	error5_mon_b	= fltarr(num_records5) ; error flag science mode
	error5_mon_b[*]	= !values.f_nan
	error1_b	= fltarr(num_records1) ; error flag engineering mode
	error1_b[*]	= !values.f_nan	


	FOR ii = 0, n_elements(products)-1 DO BEGIN 
        CASE products[ii] OF 
	
	's_valid':  BEGIN
		s_valid_b_sci	= dblarr(32,8,num_records5) ; 0
		s_valid_b_sci[*,*,*] 	= !values.d_nan
		sum_defl_s_valid_b_sci = dblarr(num_records5,32) ;0
		sum_defl_s_valid_b_sci[*,*] = !values.d_nan ;0
		END
	's_e_not_req':  BEGIN
		s_e_not_req_b_sci	= dblarr(32,8,num_records5) ; 1
		s_e_not_req_b_sci[*,*,*]	= !values.d_nan
		sum_defl_s_e_not_req_b_sci = dblarr(num_records5,32) ;1
		sum_defl_s_e_not_req_b_sci[*,*] = !values.d_nan ;1
		END
	's_e_req':  BEGIN
		s_e_req_b_sci	= dblarr(32,8,num_records5) ; 2
		s_e_req_b_sci[*,*,*]	= !values.d_nan
		sum_defl_s_e_req_b_sci = dblarr(num_records5,32) ;2
		sum_defl_s_e_req_b_sci[*,*] = !values.d_nan ;2
		END
	's_no_pos':  BEGIN
		s_no_pos_b_sci	= dblarr(32,8,num_records5) ; 3
		s_no_pos_b_sci[*,*,*]	= !values.d_nan
		sum_defl_s_no_pos_b_sci = dblarr(num_records5,32) ;3
		sum_defl_s_no_pos_b_sci[*,*] = !values.d_nan ;3
		END
	's_mult_pos':  BEGIN
		s_mult_pos_b_sci	= dblarr(32,8,num_records5) ; 4
		s_mult_pos_b_sci[*,*,*]	= !values.d_nan
		sum_defl_s_mult_pos_b_sci = dblarr(num_records5,32) ;4
		sum_defl_s_mult_pos_b_sci[*,*] = !values.d_nan ;4
		END
	's_no_e':  BEGIN
		s_no_e_b_sci	= dblarr(32,8,num_records5) ; 5
		s_no_e_b_sci[*,*,*]	= !values.d_nan
		sum_defl_s_no_e_b_sci = dblarr(num_records5,32) ;5
		sum_defl_s_no_e_b_sci[*,*] = !values.d_nan ;5
		END
	's_mult_e':  BEGIN
		s_mult_e_b_sci	= dblarr(32,8,num_records5) ; 6
		s_mult_e_b_sci[*,*,*]	= !values.d_nan
		sum_defl_s_mult_e_b_sci = dblarr(num_records5,32) ;6
		sum_defl_s_mult_e_b_sci[*,*] = !values.d_nan ;6
		END
	'ra_sat_a':  BEGIN
		ra_sat_a_b_sci	= dblarr(32,8,num_records5) ; 7
		ra_sat_a_b_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_a_b_sci = dblarr(num_records5,32) ;7
		sum_defl_ra_sat_a_b_sci[*,*] = !values.d_nan ;7
		END
	'ra_sat_b':  BEGIN
		ra_sat_b_b_sci	= dblarr(32,8,num_records5) ; 8
		ra_sat_b_b_sci[*,*,*]= !values.d_nan
		sum_defl_ra_sat_b_b_sci = dblarr(num_records5,32) ;8
		sum_defl_ra_sat_b_b_sci[*,*] = !values.d_nan ;8
		END
	'ra_sat_both':  BEGIN
		ra_sat_both_b_sci	= dblarr(32,8,num_records5) ; 9
		ra_sat_both_b_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_sat_both_b_sci = dblarr(num_records5,32) ;9
		sum_defl_ra_sat_both_b_sci[*,*] = !values.d_nan ;9
		END
	'ssd_sw':  BEGIN
		ssd_sw_b_sci	= dblarr(32,8,num_records5) ; 10
		ssd_sw_b_sci[*,*,*]	= !values.d_nan
		sum_defl_ssd_sw_b_sci = dblarr(num_records5,32) ;10
		sum_defl_ssd_sw_b_sci[*,*] = !values.d_nan ;10
		END
	'ssd_st':  BEGIN
		ssd_st_b_sci	= dblarr(32,8,num_records5) ; 11
		ssd_st_b_sci[*,*,*]	= !values.d_nan
		sum_defl_ssd_st_b_sci = dblarr(num_records5,32) ;11
		sum_defl_ssd_st_b_sci[*,*] = !values.d_nan ;11
		END
	'sf0':  BEGIN
		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		END
	'sfr0':  BEGIN
		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		END
	'stp0':  BEGIN
		stp0_b_sci	= dblarr(32,8,num_records5) ; 14
		stp0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_b_sci[*,*] = !values.d_nan ;14
		END
	'ra_trig':  BEGIN
		ra_trig_b_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_b_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_b_sci[*,*] = !values.d_nan ;15
		END
	'pos1_0':  BEGIN
		pos1_0_b_sci	= dblarr(32,8,num_records5) ; 16
		pos1_0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos1_0_b_sci = dblarr(num_records5,32) ;16
		sum_defl_pos1_0_b_sci[*,*] = !values.d_nan ;16
		END
	'pos1_1':  BEGIN
		pos1_1_b_sci	= dblarr(32,8,num_records5) ; 17
		pos1_1_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos1_1_b_sci = dblarr(num_records5,32) ;17
		sum_defl_pos1_1_b_sci[*,*] = !values.d_nan ;17
		END
	'w_no_pos':  BEGIN
		w_no_pos_b_sci	= dblarr(32,8,num_records5) ; 18
		w_no_pos_b_sci[*,*,*]	= !values.d_nan
		sum_defl_w_no_pos_b_sci = dblarr(num_records5,32) ;18
		sum_defl_w_no_pos_b_sci[*,*] = !values.d_nan ;18
		END
	'w_mult_pos':  BEGIN
		w_mult_pos_b_sci	= dblarr(32,8,num_records5) ; 19
		w_mult_pos_b_sci[*,*,*]	= !values.d_nan
		sum_defl_w_mult_pos_b_sci = dblarr(num_records5,32) ;19
		sum_defl_w_mult_pos_b_sci[*,*] = !values.d_nan ;19
		END
	'w_valid':  BEGIN
		w_valid_b_sci	= dblarr(32,8,num_records5) ; 20
		w_valid_b_sci[*,*,*]	= !values.d_nan
		sum_defl_w_valid_b_sci = dblarr(num_records5,32) ;20
		sum_defl_w_valid_b_sci[*,*] = !values.d_nan ;20
		END
	'sf2':  BEGIN
		sf2_b_sci	= dblarr(32,8,num_records5) ; 21
		sf2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_sci = dblarr(num_records5,32) ;21
		sum_defl_sf2_b_sci[*,*] = !values.d_nan ;21
		END
	'sfr2':  BEGIN
		sfr2_b_sci	= dblarr(32,8,num_records5) ; 22
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;22
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;22
		END
	'stp2':  BEGIN
		stp2_b_sci	= dblarr(32,8,num_records5) ; 23
		stp2_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_b_sci[*,*] = !values.d_nan ;23
		END
	'pos2_0':  BEGIN
		pos2_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_b_sci[*,*] = !values.d_nan ;24	
		END
	'pos2_1':  BEGIN
		pos2_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_b_sci[*,*] = !values.d_nan ;25
		END
	'pos2_2':  BEGIN
		pos2_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_b_sci[*,*] = !values.d_nan ;26
		END
	'pos2_3':  BEGIN
		pos2_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_b_sci[*,*] = !values.d_nan ;27
		END
	'pos3_0':  BEGIN
		pos3_0_b_sci	= dblarr(32,8,num_records5) ; 28
		pos3_0_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_b_sci = dblarr(num_records5,32) ;28
		sum_defl_pos3_0_b_sci[*,*] = !values.d_nan ;28
		END
	'pos3_1':  BEGIN
		pos3_1_b_sci	= dblarr(32,8,num_records5) ; 29
		pos3_1_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_sci = dblarr(num_records5,32) ;29
		sum_defl_pos3_1_b_sci[*,*] = !values.d_nan ;29
		END
	'pos3_2':  BEGIN
		pos3_2_b_sci	= dblarr(32,8,num_records5) ; 30
		pos3_2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_sci = dblarr(num_records5,32) ;30
		sum_defl_pos3_2_b_sci[*,*] = !values.d_nan ;30
		END
	'pos3_3':  BEGIN
		pos3_3_b_sci	= dblarr(32,8,num_records5) ; 31
		pos3_3_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_3_b_sci = dblarr(num_records5,32) ;31
		sum_defl_pos3_3_b_sci[*,*] = !values.d_nan ;31
		END
	'pos1':  BEGIN
		pos1_b_sci	= dblarr(32,8,num_records5) 
		pos1_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos1_b_sci = dblarr(num_records5,32) 
		sum_defl_pos1_b_sci[*,*] = !values.d_nan 
	
		pos1_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos1_0_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos1_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos1_0_b_sci[*,*] = !values.d_nan ;24	
	
		pos1_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos1_1_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos1_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos1_1_b_sci[*,*] = !values.d_nan ;25
		
		END
	'pos2':  BEGIN
		pos2_b_sci	= dblarr(32,8,num_records5) 
		pos2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos2_b_sci = dblarr(num_records5,32) 
		sum_defl_pos2_b_sci[*,*] = !values.d_nan 
		
		pos2_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos2_0_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos2_0_b_sci[*,*] = !values.d_nan ;24	
		
		pos2_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos2_1_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos2_1_b_sci[*,*] = !values.d_nan ;25
		
		pos2_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos2_2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos2_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos2_2_b_sci[*,*] = !values.d_nan ;26

		pos2_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos2_3_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos2_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos2_3_b_sci[*,*] = !values.d_nan ;27
				
		END

	'pos3':  BEGIN
		pos3_b_sci	= dblarr(32,8,num_records5) 
		pos3_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_b_sci = dblarr(num_records5,32) 
		sum_defl_pos3_b_sci[*,*] = !values.d_nan 
		
		pos3_0_b_sci	= dblarr(32,8,num_records5) ; 24
		pos3_0_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_0_b_sci = dblarr(num_records5,32) ;24
		sum_defl_pos3_0_b_sci[*,*] = !values.d_nan ;24	
		
		pos3_1_b_sci	= dblarr(32,8,num_records5) ; 25
		pos3_1_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_1_b_sci = dblarr(num_records5,32) ;25
		sum_defl_pos3_1_b_sci[*,*] = !values.d_nan ;25
		
		pos3_2_b_sci	= dblarr(32,8,num_records5) ; 26
		pos3_2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_pos3_2_b_sci = dblarr(num_records5,32) ;26
		sum_defl_pos3_2_b_sci[*,*] = !values.d_nan ;26
		
		pos3_3_b_sci	= dblarr(32,8,num_records5) ; 27
		pos3_3_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_pos3_3_b_sci = dblarr(num_records5,32) ;27
		sum_defl_pos3_3_b_sci[*,*] = !values.d_nan ;27
				
		END


	'stop0':  BEGIN
		stop0_b_sci	= dblarr(32,8,num_records5) 
		stop0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_stop0_b_sci = dblarr(num_records5,32) 
		sum_defl_stop0_b_sci[*,*] = !values.d_nan 
		
		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		
		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
			

		END
	'start0':  BEGIN
		start0_b_sci	= dblarr(32,8,num_records5) 
		start0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_start0_b_sci = dblarr(num_records5,32) 
		sum_defl_start0_b_sci[*,*] = !values.d_nan
		
		sfr0_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr0_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr0_b_sci[*,*] = !values.d_nan ;13
		
		stp0_b_sci	= dblarr(32,8,num_records5) ; 14
		stp0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_stp0_b_sci = dblarr(num_records5,32) ;14
		sum_defl_stp0_b_sci[*,*] = !values.d_nan ;14


		END
	'stop2':  BEGIN
		stop2_b_sci	= dblarr(32,8,num_records5)
		stop2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_stop2_b_sci = dblarr(num_records5,32)
		sum_defl_stop2_b_sci[*,*] = !values.d_nan
		
		sf2_b_sci	= dblarr(32,8,num_records5) ; 12
		sf2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sf2_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf2_b_sci[*,*] = !values.d_nan ;12
			

		sfr2_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;13
		
		END
	'start2':  BEGIN
		start2_b_sci	= dblarr(32,8,num_records5)
		start2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_start2_b_sci = dblarr(num_records5,32) 
		sum_defl_start2_b_sci[*,*] = !values.d_nan 
		
		sfr2_b_sci	= dblarr(32,8,num_records5) ; 13
		sfr2_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sfr2_b_sci = dblarr(num_records5,32) ;13
		sum_defl_sfr2_b_sci[*,*] = !values.d_nan ;13
		
		stp2_b_sci	= dblarr(32,8,num_records5) ; 23
		stp2_b_sci[*,*,*]	= !values.d_nan	
		sum_defl_stp2_b_sci = dblarr(num_records5,32) ;23
		sum_defl_stp2_b_sci[*,*] = !values.d_nan ;23
		
		END
	'sf0ra_trig': BEGIN
		
		sf0ra_trig_b_sci= dblarr(32,8,num_records5)
		sf0ra_trig_b_sci[*,*,*] = !values.d_nan
		sum_defl_sf0ra_trig_b_sci=dblarr(num_records5,32)
		sum_defl_sf0ra_trig_b_sci[*,*] = !values.d_nan

		sf0_b_sci	= dblarr(32,8,num_records5) ; 12
		sf0_b_sci[*,*,*]	= !values.d_nan
		sum_defl_sf0_b_sci = dblarr(num_records5,32) ;12
		sum_defl_sf0_b_sci[*,*] = !values.d_nan ;12
		
		ra_trig_b_sci	= dblarr(32,8,num_records5) ; 15
		ra_trig_b_sci[*,*,*]	= !values.d_nan
		sum_defl_ra_trig_b_sci = dblarr(num_records5,32) ;15
		sum_defl_ra_trig_b_sci[*,*] = !values.d_nan ;15
		END		

		ENDCASE
		ENDFOR			
    	END 
	ENDCASE 

ENDELSE
data_initialized = 1
END
;----------------------------------------------------------------------------------

;; END OF VARIABLE INITIALIZATION ROUTINES

;; BEGINNING OF DATA READING ROUTINES

;----------------------------------------------------------------------------------
;----------------------------------------------------------------------------------
PRO read_sc_hk, cdf_id, sat, sc_hk_read, data3_index, epoch3maxrec, products, epoch3
;-----------------------------------------------------------------------------------
; read the spacecraft housekeeping data

COMMON com_mon
CASE sat OF 
    'A': BEGIN
	
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'pl_ebox_tmp':	BEGIN ;0
			read_product, cdf_id, 'pl_ebox_tmp',	pl_ebox_tmp_a,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_ebox_tmp_a
		END
		'pl_inst_msg_pwr_dwn_req':	BEGIN ;1
			read_product, cdf_id, 'pl_inst_msg_pwr_dwn_req',	pl_inst_msg_pwr_dwn_req_a,	data3_index,	epoch3maxrec
			remove_fill, ' ', pl_inst_msg_pwr_dwn_req_a
		END
		'pl_pwr_cur':	BEGIN ;2
			read_product, cdf_id, 'pl_pwr_cur',	pl_pwr_cur_a,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_pwr_cur_a
		END 
		'pl_pwr_tt':	BEGIN ;3
			read_product, cdf_id, 'pl_pwr_tt',	pl_pwr_tt_a,	data3_index,	epoch3maxrec
			remove_fill, ' ', pl_pwr_tt_a
		END
		'pl_surv_pwr_cur':  BEGIN ;4
			read_product, cdf_id, 'pl_surv_pwr_cur',	pl_surv_pwr_cur_a,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_surv_pwr_cur_a
		END
		'pl_surv_pwr_tt':	BEGIN ;5
			read_product, cdf_id, 'pl_surv_pwr_tt',	pl_surv_pwr_tt_a,	data3_index,	epoch3maxrec
			remove_fill, ' ', pl_surv_pwr_tt_a
		END
		'pl_tof_tmp':	BEGIN ;6
			read_product, cdf_id, 'pl_tof_tmp',	pl_tof_tmp_a,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_tof_tmp_a
		END
		'pl_pwr_vt':	BEGIN ;7
			read_product, cdf_id, 'pl_pwr_vt',	pl_pwr_vt_a,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_pwr_vt_a
		END
		'pl_surv_pwr_vt':	BEGIN ;8
			read_product, cdf_id, 'pl_surv_pwr_vt',	pl_surv_pwr_vt_a,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_surv_pwr_vt_a
		END
		ENDCASE
	ENDFOR

    END
    'B': BEGIN
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'pl_ebox_tmp':	BEGIN ;0
			read_product, cdf_id, 'pl_ebox_tmp',	pl_ebox_tmp_b,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_ebox_tmp_b
		END
		'pl_inst_msg_pwr_dwn_req':	BEGIN ;1
			read_product, cdf_id, 'pl_inst_msg_pwr_dwn_req',	pl_inst_msg_pwr_dwn_req_b,	data3_index,	epoch3maxrec
			remove_fill, ' ', pl_inst_msg_pwr_dwn_req_b
		END
		'pl_pwr_cur':	BEGIN ;2
			read_product, cdf_id, 'pl_pwr_cur',	pl_pwr_cur_b,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_pwr_cur_b
		END 
		'pl_pwr_tt':	BEGIN ;3
			read_product, cdf_id, 'pl_pwr_tt',	pl_pwr_tt_b,	data3_index,	epoch3maxrec
			remove_fill, ' ', pl_pwr_tt_b
		END
		'pl_surv_pwr_cur':  BEGIN ;4
			read_product, cdf_id, 'pl_surv_pwr_cur',	pl_surv_pwr_cur_b,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_surv_pwr_cur_b
		END
		'pl_surv_pwr_tt':	BEGIN ;5
			read_product, cdf_id, 'pl_surv_pwr_tt',	pl_surv_pwr_tt_b,	data3_index,	epoch3maxrec
			remove_fill, ' ', pl_surv_pwr_tt_b
		END
		'pl_tof_tmp':	BEGIN ;6
			read_product, cdf_id, 'pl_tof_tmp',	pl_tof_tmp_b,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_tof_tmp_b
		END
		'pl_pwr_vt':	BEGIN ;7
			read_product, cdf_id, 'pl_pwr_vt',	pl_pwr_vt_b,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_pwr_vt_b
		END
		'pl_surv_pwr_vt':	BEGIN ;8
			read_product, cdf_id, 'pl_surv_pwr_vt',	pl_surv_pwr_vt_b,	data3_index,	epoch3maxrec
			remove_fill, -1, pl_surv_pwr_vt_b
		END
		ENDCASE
	ENDFOR
    END
ENDCASE
sc_hk_read = 1
END 
;---------------------------------------------------------------------------


;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
PRO read_dig_hk, cdf_id, sat, dig_hk_read, data1_index, epoch1maxrec, products
;-----------------------------------------------------------------------------
; read the digital housekeeping data


COMMON com_mon
CASE sat OF 
    'A': BEGIN
	
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
	
		'dpu_sw_version':  BEGIN
			read_product, cdf_id, 'dpu_sw_version', dpu_sw_version_a, data1_index, epoch1maxrec ; 0
			remove_fill, -1, dpu_sw_version_a; 0
			END
		'cmd_rcvd_cnt':  BEGIN
			read_product, cdf_id, 'cmd_rcvd_cnt', cmd_rcvd_cnt_a, data1_index, epoch1maxrec  ; 1
			remove_fill, -1, cmd_rcvd_cnt_a ; 1
			END
		'cmd_exec_cnt':  BEGIN
			read_product, cdf_id, 'cmd_exec_cnt', cmd_exec_cnt_a, data1_index, epoch1maxrec ; 2
			remove_fill, -1, cmd_exec_cnt_a ; 2
			END
		'cmd_total_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_total_err_cnt', cmd_total_err_cnt_a, data1_index, epoch1maxrec ; 3
			remove_fill, -1, cmd_total_err_cnt_a ; 3
			END
		'cmd_parity_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_parity_err_cnt', cmd_parity_err_cnt_a,  data1_index, epoch1maxrec ; 4
			remove_fill, -1, cmd_parity_err_cnt_a ; 4
			END
		'cmd_frame_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_frame_err_cnt', cmd_frame_err_cnt_a, data1_index, epoch1maxrec ; 5
			remove_fill, -1, cmd_frame_err_cnt_a ; 5
			END
		'cmd_illegal_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_illegal_err_cnt', cmd_illegal_err_cnt_a, data1_index, epoch1maxrec ; 6
			remove_fill, -1, cmd_illegal_err_cnt_a ; 6
			END
		'ic_status_reg':  BEGIN
			read_product, cdf_id, 'ic_status_reg', ic_status_reg_a, data1_index, epoch1maxrec ; 7
			remove_fill, -1, ic_status_reg_a ; 7
			END
		'ic_ctl_reg':  BEGIN
			read_product, cdf_id, 'ic_ctl_reg', ic_ctl_reg_a, data1_index, epoch1maxrec ; 8
			remove_fill, -1, ic_ctl_reg_a ; 8
			END
		'lbc_reset_ctl':  BEGIN
			read_product, cdf_id, 'lbc_reset_ctl', lbc_reset_ctl_a, data1_index, epoch1maxrec ; 9
			remove_fill, -1, lbc_reset_ctl_a ; 9
			END
		'lbc_mode_ctl':  BEGIN
			read_product, cdf_id, 'lbc_mode_ctl', lbc_mode_ctl_a, data1_index, epoch1maxrec ; 10
			remove_fill, -1, lbc_mode_ctl_a ; 10
			END
		'lbc_logic_ctl_a':  BEGIN
			read_product, cdf_id, 'lbc_logic_ctl_a', lbc_logic_ctl_a_a, data1_index, epoch1maxrec ; 11
			remove_fill, -1, lbc_logic_ctl_a_a ; 11
			END
		'lbc_logic_ctl_b':  BEGIN
			read_product, cdf_id, 'lbc_logic_ctl_b', lbc_logic_ctl_b_a, data1_index, epoch1maxrec ; 12
			remove_fill, -1, lbc_logic_ctl_b_a ; 12
			END
		'lbc_tac_pwr_ctl':  BEGIN
			read_product, cdf_id, 'lbc_tac_pwr_ctl', lbc_tac_pwr_ctl_a, data1_index, epoch1maxrec ; 13
			remove_fill, -1, lbc_tac_pwr_ctl_a ; 13
			END
		'lbc_reg_seq':  BEGIN
			read_product, cdf_id, 'lbc_reg_seq', lbc_reg_seq_a, data1_index, epoch1maxrec ; 14
			remove_fill, -1, lbc_reg_seq_a ; 14
			END
		'lbc_event_ctl':  BEGIN
			read_product, cdf_id, 'lbc_event_ctl', lbc_event_ctl_a, data1_index, epoch1maxrec ; 15
			remove_fill, -1, lbc_event_ctl_a ; 15
			END
		'lbc_rlim_ch':  BEGIN
			read_product, cdf_id, 'lbc_rlim_ch', lbc_rlim_ch_a, data1_index, epoch1maxrec ; 16
			remove_fill, -1, lbc_rlim_ch_a ; 16
			END
		'lbc_rlim_hi':  BEGIN
			read_product, cdf_id, 'lbc_rlim_hi', lbc_rlim_hi_a, data1_index, epoch1maxrec ; 17
			remove_fill, -1, lbc_rlim_hi_a ; 17
			END
		'lbc_rlim_lo':  BEGIN
			read_product, cdf_id, 'lbc_rlim_lo', lbc_rlim_lo_a, data1_index, epoch1maxrec ; 18
			remove_fill, -1, lbc_rlim_lo_a ; 18
			END
		'lbe_pos_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_pos_ctrl', lbe_pos_ctrl_a, data1_index, epoch1maxrec ; 19
			remove_fill, -1, lbe_pos_ctrl_a ; 19
			END
		'lbe_pos_disable0':  BEGIN
			read_product, cdf_id, 'lbe_pos_disable0', lbe_pos_disable0_a, data1_index, epoch1maxrec ; 20
			remove_fill, -1, lbe_pos_disable0_a ; 20
			END
		'lbe_pos_disable2':  BEGIN
			read_product, cdf_id, 'lbe_pos_disable2', lbe_pos_disable2_a, data1_index, epoch1maxrec ; 21
			remove_fill, -1, lbe_pos_disable2_a ; 21
			END
		'lbe_tac0_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_tac0_ctrl', lbe_tac0_ctrl_a, data1_index, epoch1maxrec ; 22
			remove_fill, -1, lbe_tac0_ctrl_a ; 22
			END
		'lbe_tac0_dac':  BEGIN
			read_product, cdf_id, 'lbe_tac0_dac', lbe_tac0_dac_a, data1_index, epoch1maxrec ; 23
			remove_fill, -1, lbe_tac0_dac_a ; 23
			END
		'lbe_tac2_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_tac2_ctrl', lbe_tac2_ctrl_a, data1_index, epoch1maxrec ; 24
			remove_fill, -1, lbe_tac2_ctrl_a ; 24
			END
		'lbe_tac2_dac':  BEGIN
			read_product, cdf_id, 'lbe_tac2_dac', lbe_tac2_dac_a, data1_index, epoch1maxrec ; 25
			remove_fill, -1, lbe_tac2_dac_a ; 25
			END
		'lbe_tac0_under_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac0_under_hi', lbe_tac0_under_hi_a, data1_index, epoch1maxrec ; 26
			remove_fill, -1, lbe_tac0_under_hi_a ; 26
			END
		'lbe_tac0_under_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac0_under_lo', lbe_tac0_under_lo_a, data1_index, epoch1maxrec ; 27
			remove_fill, -1, lbe_tac0_under_lo_a ; 27
			END
		'lbe_tac0_over_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac0_over_hi', lbe_tac0_over_hi_a, data1_index, epoch1maxrec ; 28
			remove_fill, -1, lbe_tac0_over_hi_a ; 28
			END
		'lbe_tac0_over_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac0_over_lo', lbe_tac0_over_lo_a, data1_index, epoch1maxrec ; 29
			remove_fill, -1, lbe_tac0_over_lo_a ; 29
			END
		'lbe_tac2_under_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac2_under_hi', lbe_tac2_under_hi_a, data1_index, epoch1maxrec ; 30
			remove_fill, -1, lbe_tac2_under_hi_a ; 30
			END
		'lbe_tac2_under_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac2_under_lo', lbe_tac2_under_lo_a, data1_index, epoch1maxrec ; 31
			remove_fill, -1, lbe_tac2_under_lo_a ; 31
			END
		'lbe_tac2_over_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac2_over_hi', lbe_tac2_over_hi_a, data1_index, epoch1maxrec ; 32
			remove_fill, -1, lbe_tac2_over_hi_a ; 32
			END
		'lbe_tac2_over_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac2_over_lo', lbe_tac2_over_lo_a, data1_index, epoch1maxrec ; 33
			remove_fill, -1, lbe_tac2_over_lo_a ; 33
			END
		'lbe_ssd_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_ssd_ctrl', lbe_ssd_ctrl_a, data1_index, epoch1maxrec ; 34
			remove_fill, -1, lbe_ssd_ctrl_a ; 34
			END
		'lbe_ssd_cmd_h':  BEGIN
			read_product, cdf_id, 'lbe_ssd_cmd_h', lbe_ssd_cmd_h_a, data1_index, epoch1maxrec ; 35
			remove_fill, -1, lbe_ssd_cmd_h_a ; 35
			END
		'lbe_ssd_cmd_l':  BEGIN
			read_product, cdf_id,'lbe_ssd_cmd_l', lbe_ssd_cmd_l_a, data1_index, epoch1maxrec ; 36
			remove_fill, -1, lbe_ssd_cmd_l_a ; 36
			END
		'lbe_ssd_dis':  BEGIN
			read_product, cdf_id, 'lbe_ssd_dis', lbe_ssd_dis_a, data1_index, epoch1maxrec ; 37
			remove_fill, -1, lbe_ssd_dis_a ; 37
			END
		'lbe_ssd_under_hi':  BEGIN
			read_product, cdf_id,'lbe_ssd_under_hi', lbe_ssd_under_hi_a, data1_index, epoch1maxrec ; 38
			remove_fill, -1, lbe_ssd_under_hi_a ; 38
			END
		'lbe_ssd_under_lo':  BEGIN
			read_product, cdf_id,'lbe_ssd_under_lo', lbe_ssd_under_lo_a, data1_index, epoch1maxrec ; 39
			remove_fill, -1, lbe_ssd_under_lo_a ; 39
			END
		'lbe_ssd_over_hi':  BEGIN
			read_product, cdf_id,'lbe_ssd_over_hi', lbe_ssd_over_hi_a, data1_index, epoch1maxrec ; 40
			remove_fill, -1, lbe_ssd_over_hi_a ; 40
			END
		'lbe_ssd_over_lo':  BEGIN
			read_product, cdf_id,'lbe_ssd_over_lo', lbe_ssd_over_lo_a, data1_index, epoch1maxrec ; 41
			remove_fill, -1, lbe_ssd_over_lo_a ; 41
			END
		'lbe_sel_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_sel_ctrl', lbe_sel_ctrl_a, data1_index, epoch1maxrec ; 42
			remove_fill, -1, lbe_sel_ctrl_a ; 42
			END
		'lbe_trig_mode':  BEGIN
			read_product, cdf_id, 'lbe_trig_mode', lbe_trig_mode_a, data1_index, epoch1maxrec ; 43
			remove_fill, -1, lbe_trig_mode_a ; 43
			END
		'lbe_esa_step':  BEGIN
			read_product, cdf_id, 'lbe_esa_step', lbe_esa_step_a, data1_index, epoch1maxrec ; 44
			remove_fill, -1, lbe_esa_step_a ; 44
			END
		'lbe_defl_step':  BEGIN
			read_product, cdf_id, 'lbe_defl_step', lbe_defl_step_a, data1_index, epoch1maxrec ; 45
			remove_fill, -1, lbe_defl_step_a ; 45
			END
		'lbe_ssd_offset0':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset0', lbe_ssd_offset0_a, data1_index, epoch1maxrec ; 46
			remove_fill, -1, lbe_ssd_offset0_a ; 46
			END
		'lbe_ssd_offset1':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset1', lbe_ssd_offset1_a, data1_index, epoch1maxrec ; 47
			remove_fill, -1, lbe_ssd_offset1_a ; 47
			END
		'lbe_ssd_offset2':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset2', lbe_ssd_offset2_a, data1_index, epoch1maxrec ; 48
			remove_fill, -1, lbe_ssd_offset2_a ; 48
			END
		'lbe_ssd_offset3':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset3', lbe_ssd_offset3_a, data1_index, epoch1maxrec ; 49
			remove_fill, -1, lbe_ssd_offset3_a ; 49
			END
		'lbe_ssd_offset4':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset4', lbe_ssd_offset4_a, data1_index, epoch1maxrec ; 50
			remove_fill, -1, lbe_ssd_offset4_a ; 50
			END
		'lbe_ssd_offset5':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset5', lbe_ssd_offset5_a, data1_index, epoch1maxrec ; 51
			remove_fill, -1, lbe_ssd_offset5_a ; 51
			END
		'lbe_ssd_offset6':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset6', lbe_ssd_offset6_a, data1_index, epoch1maxrec ; 52
			remove_fill, -1, lbe_ssd_offset6_a ; 52
			END
		'lbe_ssd_offset7':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset7', lbe_ssd_offset7_a, data1_index, epoch1maxrec ; 53
			remove_fill, -1, lbe_ssd_offset7_a ; 53
			END
		'lbe_stim_enable':  BEGIN
			read_product, cdf_id, 'lbe_stim_enable', lbe_stim_enable_a, data1_index, epoch1maxrec ; 54
			remove_fill, -1, lbe_stim_enable_a ; 54
			END
		'lbe_stim_freq_hi':  BEGIN
			read_product, cdf_id, 'lbe_stim_freq_hi', lbe_stim_freq_hi_a, data1_index, epoch1maxrec ; 55
			remove_fill, -1, lbe_stim_freq_hi_a ; 55
			END
		'lbe_stim_freq_lo':  BEGIN
			read_product, cdf_id, 'lbe_stim_freq_lo', lbe_stim_freq_lo_a, data1_index, epoch1maxrec ; 56
			remove_fill, -1, lbe_stim_freq_lo_a ; 56
			END
		'lbe_sel0_window':  BEGIN
			read_product, cdf_id, 'lbe_sel0_window', lbe_sel0_window_a, data1_index, epoch1maxrec ; 57
			remove_fill, -1, lbe_sel0_window_a ; 57
			END
		'lbe_sel2_window':  BEGIN
			read_product, cdf_id, 'lbe_sel2_window', lbe_sel2_window_a, data1_index, epoch1maxrec ; 58
			remove_fill, -1, lbe_sel2_window_a ; 58
			END
		'lbe_ssd_offset8':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset8', lbe_ssd_offset8_a, data1_index, epoch1maxrec ; 59
			remove_fill, -1, lbe_ssd_offset8_a ; 59
			END
		'lbe_ssd_offset9':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset9', lbe_ssd_offset9_a, data1_index, epoch1maxrec ; 60
			remove_fill, -1, lbe_ssd_offset9_a ; 60
			END
		'lbe_ssd_offset10':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset10', lbe_ssd_offset10_a, data1_index, epoch1maxrec ; 61
			remove_fill, -1, lbe_ssd_offset10_a ; 61
			END
		'lbe_ssd_offset11':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset11', lbe_ssd_offset11_a, data1_index, epoch1maxrec ; 62
			remove_fill, -1, lbe_ssd_offset11_a ; 62
			END
		'lbe_ssd_offset12':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset12', lbe_ssd_offset12_a, data1_index, epoch1maxrec ; 63
			remove_fill, -1, lbe_ssd_offset12_a ; 63
			END
		'lbe_ssd_offset13':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset13', lbe_ssd_offset13_a, data1_index, epoch1maxrec ; 64
			remove_fill, -1, lbe_ssd_offset13_a ; 64
			END
		'lbe_ssd_offset14':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset14', lbe_ssd_offset14_a, data1_index, epoch1maxrec ; 65
			remove_fill, -1, lbe_ssd_offset14_a ; 65
			END
		'lbe_ssd_offset15':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset15', lbe_ssd_offset15_a, data1_index, epoch1maxrec ; 66
			remove_fill, -1, lbe_ssd_offset15_a ; 66
			END
		'lbe_pos_ra':  BEGIN
			read_product, cdf_id, 'lbe_pos_ra', lbe_pos_ra_a, data1_index, epoch1maxrec ; 67
			remove_fill, -1, lbe_pos_ra_a ; 67
			END
		'dacm_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
			read_product, cdf_id, 'dacm_pac_mcp_ssd_ctl_mcp_limit', dacm_pac_mcp_ssd_ctl_mcp_limit_a, data1_index, epoch1maxrec ; 68
			remove_fill, -1, dacm_pac_mcp_ssd_ctl_mcp_limit_a ; 68
			END
		'dacm_pac_limit':  BEGIN
			read_product, cdf_id, 'dacm_pac_limit', dacm_pac_limit_a, data1_index, epoch1maxrec ; 69
			remove_fill, -1, dacm_pac_limit_a ; 69
			END
		'dacm_esa_ctl':  BEGIN
			read_product, cdf_id, 'dacm_esa_ctl', dacm_esa_ctl_a, data1_index, epoch1maxrec ; 70
			remove_fill, -1, dacm_esa_ctl_a ; 70
			END
		'dacm_s_ch_ctl':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_ctl', dacm_s_ch_ctl_a, data1_index, epoch1maxrec ; 71
			remove_fill, -1, dacm_s_ch_ctl_a ; 71
			END
		'dacm_dfl_1_ctl':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_ctl', dacm_dfl_1_ctl_a, data1_index, epoch1maxrec ; 72
			remove_fill, -1, dacm_dfl_1_ctl_a ; 72
			END
		'dacm_dfl_2_ctl':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_ctl', dacm_dfl_2_ctl_a, data1_index, epoch1maxrec ; 73
			remove_fill, -1, dacm_dfl_2_ctl_a ; 73
			END
		'dacm_control':  BEGIN
			read_product, cdf_id, 'dacm_control', dacm_control_a, data1_index, epoch1maxrec ; 74
			remove_fill, -1, dacm_control_a ; 74
			END
		'dacm_status':  BEGIN
			read_product, cdf_id, 'dacm_status', dacm_status_a, data1_index, epoch1maxrec ; 75
			remove_fill, -1, dacm_status_a ; 75
			END
		'dacm_adc_lu_ctr':  BEGIN
			read_product, cdf_id, 'dacm_adc_lu_ctr', dacm_adc_lu_ctr_a, data1_index, epoch1maxrec ;76
			remove_fill, -1, dacm_adc_lu_ctr_a ; 76
			END
		'dacm_dfl_1_offset':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_offset', dacm_dfl_1_offset_a, data1_index, epoch1maxrec ; 77
			remove_fill, -1, dacm_dfl_1_offset_a ; 77
			END
		'dacm_dfl_2_offset':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_offset', dacm_dfl_2_offset_a, data1_index, epoch1maxrec ; 78
			remove_fill, -1, dacm_dfl_2_offset_a ; 78
			END
		'dacm_pac_coarse':  BEGIN
			read_product, cdf_id, 'dacm_pac_coarse', dacm_pac_coarse_a, data1_index, epoch1maxrec ; 79
			remove_fill, -1, dacm_pac_coarse_a ; 79
			END
		'dacm_pac_fine':  BEGIN
			read_product, cdf_id, 'dacm_pac_fine', dacm_pac_fine_a, data1_index, epoch1maxrec ; 80
			remove_fill, -1, dacm_pac_fine_a ; 80
			END
		'dacm_pac_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacm_pac_cm_fs_coarse', dacm_pac_cm_fs_coarse_a, data1_index, epoch1maxrec ; 81
			remove_fill, -1, dacm_pac_cm_fs_coarse_a ; 81
			END
		'dacm_pac_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacm_pac_cm_fs_fine', dacm_pac_cm_fs_fine_a, data1_index, epoch1maxrec ; 82
			remove_fill, -1, dacm_pac_cm_fs_fine_a ; 82
			END
		'dacm_mcp_coarse':  BEGIN
			read_product, cdf_id, 'dacm_mcp_coarse', dacm_mcp_coarse_a, data1_index, epoch1maxrec ; 83
			remove_fill, -1, dacm_mcp_coarse_a ; 83
			END
		'dacm_mcp_fine':  BEGIN
			read_product, cdf_id, 'dacm_mcp_fine', dacm_mcp_fine_a, data1_index, epoch1maxrec ; 84
			remove_fill, -1, dacm_mcp_fine_a ; 84
			END
		'dacm_mcp_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacm_mcp_cm_fs_coarse', dacm_mcp_cm_fs_coarse_a, data1_index, epoch1maxrec ; 85
			remove_fill, -1, dacm_mcp_cm_fs_coarse_a ; 85
			END
		'dacm_mcp_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacm_mcp_cm_fs_fine', dacm_mcp_cm_fs_fine_a, data1_index, epoch1maxrec ; 86
			remove_fill, -1, dacm_mcp_cm_fs_fine_a ; 86
			END
		'dacm_ssd_coarse':  BEGIN
			read_product, cdf_id, 'dacm_ssd_coarse', dacm_ssd_coarse_a, data1_index, epoch1maxrec ; 87
			remove_fill, -1, dacm_ssd_coarse_a ; 87
			END
		'dacm_ssd_fine':  BEGIN
			read_product, cdf_id, 'dacm_ssd_fine', dacm_ssd_fine_a, data1_index, epoch1maxrec ; 88
			remove_fill, -1, dacm_ssd_fine_a ; 88
			END
		'dacm_esa_coarse':  BEGIN
			read_product, cdf_id, 'dacm_esa_coarse', dacm_esa_coarse_a, data1_index, epoch1maxrec ; 89
			remove_fill, -1, dacm_esa_coarse_a ; 89
			END
		'dacm_esa_fine':  BEGIN
			read_product, cdf_id, 'dacm_esa_fine', dacm_esa_fine_a, data1_index, epoch1maxrec ; 90
			remove_fill, -1, dacm_esa_fine_a ; 90
			END
		'dacm_s_ch_coarse':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_coarse', dacm_s_ch_coarse_a, data1_index, epoch1maxrec ; 91
			remove_fill, -1, dacm_s_ch_coarse_a ; 91
			END
		'dacm_s_ch_fine':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_fine', dacm_s_ch_fine_a, data1_index, epoch1maxrec ; 92
			remove_fill, -1, dacm_s_ch_fine_a ; 92
			END
		'dacm_s_ch_tap_coarse':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_tap_coarse', dacm_s_ch_tap_coarse_a, data1_index, epoch1maxrec ; 93
			remove_fill, -1, dacm_s_ch_tap_coarse_a ; 93
			END
		'dacm_s_ch_tap_fine':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_tap_fine', dacm_s_ch_tap_fine_a, data1_index, epoch1maxrec ; 94
			remove_fill, -1, dacm_s_ch_tap_fine_a ; 94
			END
		'dacm_dfl_1_coarse':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_coarse', dacm_dfl_1_coarse_a, data1_index, epoch1maxrec ; 95
			remove_fill, -1, dacm_dfl_1_coarse_a ; 95
			END
		'dacm_dfl_1_fine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_fine', dacm_dfl_1_fine_a, data1_index, epoch1maxrec ; 96
			remove_fill, -1, dacm_dfl_1_fine_a ; 96
			END
		'dacm_dfl_1_dfine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_dfine', dacm_dfl_1_dfine_a, data1_index, epoch1maxrec ; 97
			remove_fill, -1, dacm_dfl_1_dfine_a ; 97
			END
		'dacm_dfl_2_coarse':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_coarse', dacm_dfl_2_coarse_a, data1_index, epoch1maxrec ; 98
			remove_fill, -1, dacm_dfl_2_coarse_a ; 98
			END
		'dacm_dfl_2_fine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_fine', dacm_dfl_2_fine_a, data1_index, epoch1maxrec ; 99
			remove_fill, -1, dacm_dfl_2_fine_a ; 99
			END
		'dacm_dfl_2_dfine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_dfine', dacm_dfl_2_dfine_a, data1_index, epoch1maxrec ; 100
			remove_fill, -1, dacm_dfl_2_dfine_a ; 100
			END
		'dacr_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
			read_product, cdf_id, 'dacr_pac_mcp_ssd_ctl_mcp_limit', dacr_pac_mcp_ssd_ctl_mcp_limit_a, data1_index, epoch1maxrec ; 101
			remove_fill, -1, dacr_pac_mcp_ssd_ctl_mcp_limit_a ; 101
			END
		'dacr_pac_limit':  BEGIN
			read_product, cdf_id, 'dacr_pac_limit', dacr_pac_limit_a, data1_index, epoch1maxrec ; 102
			remove_fill, -1, dacr_pac_limit_a ; 102
			END
		'dacr_esa_ctl':  BEGIN
			read_product, cdf_id, 'dacr_esa_ctl', dacr_esa_ctl_a, data1_index, epoch1maxrec ; 103
			remove_fill, -1, dacr_esa_ctl_a ; 103
			END
		'dacr_s_ch_ctl':  BEGIN
			read_product, cdf_id, 'dacr_s_ch_ctl', dacr_s_ch_ctl_a, data1_index, epoch1maxrec  ; 104
			remove_fill, -1, dacr_s_ch_ctl_a ; 104
			END
		'dacr_dfl_1_ctl':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_ctl', dacr_dfl_1_ctl_a, data1_index, epoch1maxrec ; 105
			remove_fill, -1, dacr_dfl_1_ctl_a ; 105
			END
		'dacr_dfl_2_ctl':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_ctl', dacr_dfl_2_ctl_a, data1_index, epoch1maxrec ; 106
			remove_fill, -1, dacr_dfl_2_ctl_a ; 106
			END
		'dacr_control':  BEGIN
			read_product, cdf_id, 'dacr_control', dacr_control_a, data1_index, epoch1maxrec ; 107
			remove_fill, -1, dacr_control_a ; 107
			END
		'dacr_status':  BEGIN
			read_product, cdf_id, 'dacr_status', dacr_status_a, data1_index, epoch1maxrec ; 108
			remove_fill, -1, dacr_status_a ; 108
			END
		'dacr_adc_lu_ctr':  BEGIN
			read_product, cdf_id, 'dacr_adc_lu_ctr', dacr_adc_lu_ctr_a, data1_index, epoch1maxrec  ; 109
			remove_fill, -1, dacr_adc_lu_ctr_a ; 109
			END
		'dacr_dfl_1_offset':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_offset', dacr_dfl_1_offset_a, data1_index, epoch1maxrec ; 110
			remove_fill, -1, dacr_dfl_1_offset_a ; 110
			END
		'dacr_dfl_2_offset':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_offset', dacr_dfl_2_offset_a, data1_index, epoch1maxrec ; 111
			remove_fill, -1, dacr_dfl_2_offset_a ; 111
			END
		'dacr_c_pac_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_coarse', dacr_c_pac_coarse_a, data1_index, epoch1maxrec ; 112
			remove_fill, -1, dacr_c_pac_coarse_a ; 112
			END
		'dacr_c_pac_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_fine', dacr_c_pac_fine_a, data1_index, epoch1maxrec ; 113
			remove_fill, -1, dacr_c_pac_fine_a ; 113
			END
		'dacr_c_pac_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_cm_fs_coarse', dacr_c_pac_cm_fs_coarse_a, data1_index, epoch1maxrec ; 114
			remove_fill, -1, dacr_c_pac_cm_fs_coarse_a ; 114
			END
		'dacr_c_pac_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_cm_fs_fine', dacr_c_pac_cm_fs_fine_a, data1_index, epoch1maxrec ; 115
			remove_fill, -1, dacr_c_pac_cm_fs_fine_a ; 115
			END
		'dacr_c_mcp_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_coarse', dacr_c_mcp_coarse_a, data1_index, epoch1maxrec  ; 116
			remove_fill, -1, dacr_c_mcp_coarse_a ; 116
			END
		'dacr_c_mcp_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_fine', dacr_c_mcp_fine_a, data1_index, epoch1maxrec ; 117
			remove_fill, -1, dacr_c_mcp_fine_a ; 117
			END
		'dacr_c_mcp_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_cm_fs_coarse', dacr_c_mcp_cm_fs_coarse_a, data1_index, epoch1maxrec ; 118
			remove_fill, -1, dacr_c_mcp_cm_fs_coarse_a ; 118
			END
		'dacr_c_mcp_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_cm_fs_fine', dacr_c_mcp_cm_fs_fine_a, data1_index, epoch1maxrec ; 119
			remove_fill, -1, dacr_c_mcp_cm_fs_fine_a ; 119
			END
		'dacr_c_ssd_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_ssd_coarse', dacr_c_ssd_coarse_a, data1_index, epoch1maxrec ; 120
			remove_fill, -1, dacr_c_ssd_coarse_a ; 120
			END
		'dacr_c_ssd_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_ssd_fine', dacr_c_ssd_fine_a, data1_index, epoch1maxrec ; 121
			remove_fill, -1, dacr_c_ssd_fine_a ; 121
			END
		'dacr_c_esa_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_esa_coarse', dacr_c_esa_coarse_a, data1_index, epoch1maxrec ; 122
			remove_fill, -1, dacr_c_esa_coarse_a ; 122
			END
		'dacr_c_esa_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_esa_fine', dacr_c_esa_fine_a, data1_index, epoch1maxrec ; 123
			remove_fill, -1, dacr_c_esa_fine_a ; 123
			END
		'dacr_c_s_ch_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_coarse', dacr_c_s_ch_coarse_a, data1_index, epoch1maxrec ; 124
			remove_fill, -1, dacr_c_s_ch_coarse_a ; 124
			END
		'dacr_c_s_ch_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_fine', dacr_c_s_ch_fine_a, data1_index, epoch1maxrec ; 125
			remove_fill, -1, dacr_c_s_ch_fine_a ; 125
			END
		'dacr_c_s_ch_tap_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_tap_coarse', dacr_c_s_ch_tap_coarse_a, data1_index, epoch1maxrec ; 126
			remove_fill, -1, dacr_c_s_ch_tap_coarse_a ; 126
			END
		'dacr_c_s_ch_tap_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_tap_fine', dacr_c_s_ch_tap_fine_a, data1_index, epoch1maxrec ; 127
			remove_fill, -1, dacr_c_s_ch_tap_fine_a ; 127
			END
		'dacr_dfl_1_coarse':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_coarse', dacr_dfl_1_coarse_a, data1_index, epoch1maxrec ; 128
			remove_fill, -1, dacr_dfl_1_coarse_a ; 128
			END
		'dacr_dfl_1_fine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_fine', dacr_dfl_1_fine_a, data1_index, epoch1maxrec ; 129
			remove_fill, -1, dacr_dfl_1_fine_a ; 129
			END
		'dacr_dfl_1_dfine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_dfine', dacr_dfl_1_dfine_a, data1_index, epoch1maxrec  ; 130
			remove_fill, -1, dacr_dfl_1_dfine_a ; 130
			END
		'dacr_dfl_2_coarse':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_coarse', dacr_dfl_2_coarse_a, data1_index, epoch1maxrec  ; 131
			remove_fill, -1, dacr_dfl_2_coarse_a ; 131
			END
		'dacr_dfl_2_fine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_fine', dacr_dfl_2_fine_a, data1_index, epoch1maxrec ; 132
			remove_fill, -1, dacr_dfl_2_fine_a ; 132
			END
		'dacr_dfl_2_dfine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_dfine', dacr_dfl_2_dfine_a, data1_index, epoch1maxrec ; 133
			remove_fill, -1, dacr_dfl_2_dfine_a ; 133
			END
		'sw_status1':  BEGIN
			read_product, cdf_id, 'sw_status1', sw_status1_a, data1_index, epoch1maxrec ; 134
			remove_fill, -1, sw_status1_a ; 134
			END
		'sw_hv_ramp_status':  BEGIN
			read_product, cdf_id, 'sw_hv_ramp_status', sw_hv_ramp_status_a, data1_index, epoch1maxrec ; 135
			remove_fill, -1, sw_hv_ramp_status_a ; 135
			END
		'sw_hv_enable_status':  BEGIN
			read_product, cdf_id, 'sw_hv_enable_status', sw_hv_enable_status_a, data1_index, epoch1maxrec ; 136
			remove_fill, -1, sw_hv_enable_status_a ; 136
			END
		'sw_hv_limit_status':  BEGIN
			read_product, cdf_id, 'sw_hv_limit_status', sw_hv_limit_status_a, data1_index, epoch1maxrec ; 137
			remove_fill, -1, sw_hv_limit_status_a ; 137
			END
		'sw_eepromchksum':  BEGIN
			read_product, cdf_id, 'sw_eepromchksum', sw_eepromchksum_a, data1_index, epoch1maxrec ; 138
			remove_fill, -1, sw_eepromchksum_a ; 138
			END
		'hkp_sampletime':  BEGIN
			read_product, cdf_id, 'hkp_sampletime', hkp_sampletime_a, data1_index, epoch1maxrec ; 139
			remove_fill, -1, hkp_sampletime_a ; 139
			END
		'pac_discharge_status':  BEGIN
			read_product, cdf_id, 'pac_discharge_status', pac_discharge_status_a, data1_index, epoch1maxrec ; 140
			remove_fill, -1, pac_discharge_status_a ; 140
			END
		'pac_discharge_cnt':  BEGIN
			read_product, cdf_id, 'pac_discharge_cnt', pac_discharge_cnt_a, data1_index, epoch1maxrec ; 141
			remove_fill, -1, pac_discharge_cnt_a ; 141
			END
		'pac_discharge_conseccntr':  BEGIN
			read_product, cdf_id, 'pac_discharge_conseccntr', pac_discharge_conseccntr_a, data1_index, epoch1maxrec ; 142
			remove_fill, -1, pac_discharge_conseccntr_a ; 142
			END
		'pac_discharge_safe_att':  BEGIN
			read_product, cdf_id, 'pac_discharge_safe_att', pac_discharge_safe_att_a, data1_index, epoch1maxrec ; 143
			remove_fill, -1, pac_discharge_safe_att_a ; 143
			END
		'analog_limit_status':  BEGIN
			read_product, cdf_id, 'analog_limit_status', analog_limit_status_a, data1_index, epoch1maxrec ; 144
			remove_fill, -1, analog_limit_status_a ; 144
			END
		'analog_lim_pt1_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt1_status', analog_lim_pt1_status_a, data1_index, epoch1maxrec ; 145
			remove_fill, -1, analog_lim_pt1_status_a ; 145
			END
		'analog_lim_pt1_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt1_id', analog_lim_pt1_id_a, data1_index, epoch1maxrec ; 146
			remove_fill, -1, analog_lim_pt1_id_a ; 146
			END
		'analog_lim_pt1_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt1_red', analog_lim_pt1_red_a, data1_index, epoch1maxrec ; 147
			remove_fill, -1, analog_lim_pt1_red_a ; 147
			END
		'analog_lim_pt2_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt2_status', analog_lim_pt2_status_a, data1_index, epoch1maxrec ; 148
			remove_fill, -1, analog_lim_pt2_status_a ; 148
			END
		'analog_lim_pt2_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt2_id', analog_lim_pt2_id_a, data1_index, epoch1maxrec ; 149
			remove_fill, -1, analog_lim_pt2_id_a ; 149
			END
		'analog_lim_pt2_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt2_red', analog_lim_pt2_red_a, data1_index, epoch1maxrec ; 150
			remove_fill, -1, analog_lim_pt2_red_a ; 150
			END
		'analog_lim_pt3_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt3_status', analog_lim_pt3_status_a, data1_index, epoch1maxrec ; 151
			remove_fill, -1, analog_lim_pt3_status_a ; 151
			END
		'analog_lim_pt3_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt3_id', analog_lim_pt3_id_a, data1_index, epoch1maxrec ; 152
			remove_fill, -1, analog_lim_pt3_id_a ; 152
			END
		'analog_lim_pt3_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt3_red', analog_lim_pt3_red_a, data1_index, epoch1maxrec ; 153
			remove_fill, -1, analog_lim_pt3_red_a ; 153
			END
		'analog_lim_pt4_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt4_status', analog_lim_pt4_status_a, data1_index, epoch1maxrec ; 154
			remove_fill, -1, analog_lim_pt4_status_a ; 154
			END
		'analog_lim_pt4_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt4_id', analog_lim_pt4_id_a, data1_index, epoch1maxrec ; 155
			remove_fill, -1, analog_lim_pt4_id_a ; 155
			END
		'analog_lim_pt4_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt4_red', analog_lim_pt4_red_a, data1_index, epoch1maxrec ; 156
			remove_fill, -1, analog_lim_pt4_red_a ; 156
			END
		'analog_lim_pt5_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt5_status', analog_lim_pt5_status_a, data1_index, epoch1maxrec ; 157
			remove_fill, -1, analog_lim_pt5_status_a ; 157
			END
		'analog_lim_pt5_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt5_id', analog_lim_pt5_id_a, data1_index, epoch1maxrec ; 158
			remove_fill, -1, analog_lim_pt5_id_a ; 158
			END
		'analog_lim_pt5_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt5_red', analog_lim_pt5_red_a, data1_index, epoch1maxrec ; 159
			remove_fill, -1, analog_lim_pt5_red_a ; 159
			END
		'analog_lim_pt6_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt6_status', analog_lim_pt6_status_a, data1_index, epoch1maxrec ; 160
			remove_fill, -1, analog_lim_pt6_status_a ; 160
			END
		'analog_lim_pt6_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt6_id', analog_lim_pt6_id_a, data1_index, epoch1maxrec ; 161
			remove_fill, -1, analog_lim_pt6_id_a ; 161
			END
		'analog_lim_pt6_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt6_red', analog_lim_pt6_red_a, data1_index, epoch1maxrec ; 162
			remove_fill, -1, analog_lim_pt6_red_a ; 162
			END
		'analog_lim_pt7_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt7_status', analog_lim_pt7_status_a, data1_index, epoch1maxrec ; 163
			remove_fill, -1, analog_lim_pt7_status_a ; 163
			END
		'analog_lim_pt7_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt7_id', analog_lim_pt7_id_a, data1_index, epoch1maxrec ; 164
			remove_fill, -1, analog_lim_pt7_id_a ; 164
			END
		'analog_lim_pt7_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt7_red', analog_lim_pt7_red_a, data1_index, epoch1maxrec ; 165
			remove_fill, -1, analog_lim_pt7_red_a ; 165
			END
		'analog_lim_pt8_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt8_status', analog_lim_pt8_status_a, data1_index, epoch1maxrec ; 166
			remove_fill, -1, analog_lim_pt8_status_a ; 166
			END
		'analog_lim_pt8_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt8_id', analog_lim_pt8_id_a, data1_index, epoch1maxrec ; 167
			remove_fill, -1, analog_lim_pt8_id_a ; 167
			END
		'analog_lim_pt8_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt8_red', analog_lim_pt8_red_a, data1_index, epoch1maxrec ; 168
			remove_fill, -1, analog_lim_pt8_red_a ; 168
			END
		'analog_lim_pt9_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt9_status', analog_lim_pt9_status_a, data1_index, epoch1maxrec ; 169
			remove_fill, -1, analog_lim_pt9_status_a ; 169
			END
		'analog_lim_pt9_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt9_id', analog_lim_pt9_id_a, data1_index, epoch1maxrec ; 170
			remove_fill, -1, analog_lim_pt9_id_a ; 170
			END
		'analog_lim_pt9_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt9_red', analog_lim_pt9_red_a, data1_index, epoch1maxrec ; 171
			remove_fill, -1, analog_lim_pt9_red_a ; 171
			END
		'analog_lim_pt10_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt10_status', analog_lim_pt10_status_a, data1_index, epoch1maxrec ; 172
			remove_fill, -1, analog_lim_pt10_status_a ; 172
			END
		'analog_lim_pt10_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt10_id', analog_lim_pt10_id_a, data1_index, epoch1maxrec ; 173
			remove_fill, -1, analog_lim_pt10_id_a ; 173
			END
		'analog_lim_pt10_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt10_red', analog_lim_pt10_red_a, data1_index, epoch1maxrec ; 174
			remove_fill, -1, analog_lim_pt10_red_a ; 174
			END
		'analog_lim_pt11_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt11_status', analog_lim_pt11_status_a, data1_index, epoch1maxrec ; 175
			remove_fill, -1, analog_lim_pt11_status_a ; 175
			END
		'analog_lim_pt11_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt11_id', analog_lim_pt11_id_a, data1_index, epoch1maxrec ; 176
			remove_fill, -1, analog_lim_pt11_id_a ; 176
			END
		'analog_lim_pt11_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt11_red', analog_lim_pt11_red_a, data1_index, epoch1maxrec ; 177
			remove_fill, -1, analog_lim_pt11_red_a ; 177
			END
		'analog_lim_pt12_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt12_status', analog_lim_pt12_status_a, data1_index, epoch1maxrec ; 178
			remove_fill, -1, analog_lim_pt12_status_a ; 178
			END
		'analog_lim_pt12_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt12_id', analog_lim_pt12_id_a, data1_index, epoch1maxrec ; 179
			remove_fill, -1, analog_lim_pt12_id_a ; 179
			END
		'analog_lim_pt12_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt12_red', analog_lim_pt12_red_a, data1_index, epoch1maxrec ; 180
			remove_fill, -1, analog_lim_pt12_red_a ; 180
			END
		'analog_lim_pt13_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt13_status', analog_lim_pt13_status_a, data1_index, epoch1maxrec ; 181
			remove_fill, -1, analog_lim_pt13_status_a ; 181
			END
		'analog_lim_pt13_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt13_id', analog_lim_pt13_id_a, data1_index, epoch1maxrec ; 182
			remove_fill, -1, analog_lim_pt13_id_a ; 182
			END
		'analog_lim_pt13_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt13_red', analog_lim_pt13_red_a, data1_index, epoch1maxrec ; 183
			remove_fill, -1, analog_lim_pt13_red_a ; 183
			END
		'analog_lim_pt14_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt14_status', analog_lim_pt14_status_a, data1_index, epoch1maxrec ; 184
			remove_fill, -1, analog_lim_pt14_status_a ; 184
			END
		'analog_lim_pt14_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt14_id', analog_lim_pt14_id_a, data1_index, epoch1maxrec ; 185
			remove_fill, -1, analog_lim_pt14_id_a ; 185
			END
		'analog_lim_pt14_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt14_red', analog_lim_pt14_red_a, data1_index, epoch1maxrec ; 186
			remove_fill, -1, analog_lim_pt14_red_a ; 186
			END
		'analog_lim_pt15_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt15_status', analog_lim_pt15_status_a, data1_index, epoch1maxrec ; 187
			remove_fill, -1, analog_lim_pt15_status_a ; 187
			END
		'analog_lim_pt15_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt15_id', analog_lim_pt15_id_a, data1_index, epoch1maxrec  ; 188
			remove_fill, -1, analog_lim_pt15_id_a ; 188
			END
		'analog_lim_pt15_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt15_red', analog_lim_pt15_red_a, data1_index, epoch1maxrec  ; 189
			remove_fill, -1, analog_lim_pt15_red_a ; 189
			END
		'opmode':  BEGIN
			read_product, cdf_id, 'opmode', opmode_a, data1_index, epoch1maxrec  ; 190
			remove_fill, -1, opmode_a ; 190
			END
		'sw_mode':  BEGIN
			read_product, cdf_id, 'sw_mode', sw_mode_a, data1_index, epoch1maxrec ; 191
			remove_fill, -1, sw_mode_a ; 191
			END
		'proton_mode':  BEGIN
			read_product, cdf_id, 'proton_mode', proton_mode_a, data1_index, epoch1maxrec ; 192
			remove_fill, -1, proton_mode_a ; 192
			END
		'active_supplyid':  BEGIN
			read_product, cdf_id, 'active_supplyid', active_supplyid_a, data1_index, epoch1maxrec ; 193
			remove_fill, -1, active_supplyid_a ; 193
			END
		'supply_coarse_rb':  BEGIN
			read_product, cdf_id, 'supply_coarse_rb', supply_coarse_rb_a, data1_index, epoch1maxrec ; 194
			remove_fill, -1, supply_coarse_rb_a ; 194
			END
		'supply_fine_rb':  BEGIN
			read_product, cdf_id, 'supply_fine_rb', supply_fine_rb_a, data1_index, epoch1maxrec ; 195
			remove_fill, -1, supply_fine_rb_a ; 195
			END
		'swmaint_supplystat':  BEGIN
			read_product, cdf_id, 'swmaint_supplystat', swmaint_supplystat_a, data1_index, epoch1maxrec ; 196
			remove_fill, -1, swmaint_supplystat_a ; 196
			END
		'swmaint_pac_coarse':  BEGIN
			read_product, cdf_id, 'swmaint_pac_coarse', swmaint_pac_coarse_a, data1_index, epoch1maxrec ; 197
			remove_fill, -1, swmaint_pac_coarse_a ; 197
			END
		'swmaint_pac_fine':  BEGIN
			read_product, cdf_id, 'swmaint_pac_fine', swmaint_pac_fine_a, data1_index, epoch1maxrec ; 198		
			remove_fill, -1, swmaint_pac_fine_a ; 198
			END


		ENDCASE
	ENDFOR

    END
    'B': BEGIN
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF

		'dpu_sw_version':  BEGIN
			read_product, cdf_id, 'dpu_sw_version', dpu_sw_version_b, data1_index, epoch1maxrec ; 0
			remove_fill, -1, dpu_sw_version_b; 0
			END
		'cmd_rcvd_cnt':  BEGIN
			read_product, cdf_id, 'cmd_rcvd_cnt', cmd_rcvd_cnt_b, data1_index, epoch1maxrec  ; 1
			remove_fill, -1, cmd_rcvd_cnt_b ; 1
			END
		'cmd_exec_cnt':  BEGIN
			read_product, cdf_id, 'cmd_exec_cnt', cmd_exec_cnt_b, data1_index, epoch1maxrec ; 2
			remove_fill, -1, cmd_exec_cnt_b ; 2
			END
		'cmd_total_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_total_err_cnt', cmd_total_err_cnt_b, data1_index, epoch1maxrec ; 3
			remove_fill, -1, cmd_total_err_cnt_b ; 3
			END
		'cmd_parity_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_parity_err_cnt', cmd_parity_err_cnt_b,  data1_index, epoch1maxrec ; 4
			remove_fill, -1, cmd_parity_err_cnt_b ; 4
			END
		'cmd_frame_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_frame_err_cnt', cmd_frame_err_cnt_b, data1_index, epoch1maxrec ; 5
			remove_fill, -1, cmd_frame_err_cnt_b ; 5
			END
		'cmd_illegal_err_cnt':  BEGIN
			read_product, cdf_id, 'cmd_illegal_err_cnt', cmd_illegal_err_cnt_b, data1_index, epoch1maxrec ; 6
			remove_fill, -1, cmd_illegal_err_cnt_b ; 6
			END
		'ic_status_reg':  BEGIN
			read_product, cdf_id, 'ic_status_reg', ic_status_reg_b, data1_index, epoch1maxrec ; 7
			remove_fill, -1, ic_status_reg_b ; 7
			END
		'ic_ctl_reg':  BEGIN
			read_product, cdf_id, 'ic_ctl_reg', ic_ctl_reg_b, data1_index, epoch1maxrec ; 8
			remove_fill, -1, ic_ctl_reg_b ; 8
			END
		'lbc_reset_ctl':  BEGIN
			read_product, cdf_id, 'lbc_reset_ctl', lbc_reset_ctl_b, data1_index, epoch1maxrec ; 9
			remove_fill, -1, lbc_reset_ctl_b ; 9
			END
		'lbc_mode_ctl':  BEGIN
			read_product, cdf_id, 'lbc_mode_ctl', lbc_mode_ctl_b, data1_index, epoch1maxrec ; 10
			remove_fill, -1, lbc_mode_ctl_b ; 10
			END
		'lbc_logic_ctl_a':  BEGIN
			read_product, cdf_id, 'lbc_logic_ctl_a', lbc_logic_ctl_a_b, data1_index, epoch1maxrec ; 11
			remove_fill, -1, lbc_logic_ctl_a_b ; 11
			END
		'lbc_logic_ctl_b':  BEGIN
			read_product, cdf_id, 'lbc_logic_ctl_b', lbc_logic_ctl_b_b, data1_index, epoch1maxrec ; 12
			remove_fill, -1, lbc_logic_ctl_b_b ; 12
			END
		'lbc_tac_pwr_ctl':  BEGIN
			read_product, cdf_id, 'lbc_tac_pwr_ctl', lbc_tac_pwr_ctl_b, data1_index, epoch1maxrec ; 13
			remove_fill, -1, lbc_tac_pwr_ctl_b ; 13
			END
		'lbc_reg_seq':  BEGIN
			read_product, cdf_id, 'lbc_reg_seq', lbc_reg_seq_b, data1_index, epoch1maxrec ; 14
			remove_fill, -1, lbc_reg_seq_b ; 14
			END
		'lbc_event_ctl':  BEGIN
			read_product, cdf_id, 'lbc_event_ctl', lbc_event_ctl_b, data1_index, epoch1maxrec ; 15
			remove_fill, -1, lbc_event_ctl_b ; 15
			END
		'lbc_rlim_ch':  BEGIN
			read_product, cdf_id, 'lbc_rlim_ch', lbc_rlim_ch_b, data1_index, epoch1maxrec ; 16
			remove_fill, -1, lbc_rlim_ch_b ; 16
			END
		'lbc_rlim_hi':  BEGIN
			read_product, cdf_id, 'lbc_rlim_hi', lbc_rlim_hi_b, data1_index, epoch1maxrec ; 17
			remove_fill, -1, lbc_rlim_hi_b ; 17
			END
		'lbc_rlim_lo':  BEGIN
			read_product, cdf_id, 'lbc_rlim_lo', lbc_rlim_lo_b, data1_index, epoch1maxrec ; 18
			remove_fill, -1, lbc_rlim_lo_b ; 18
			END
		'lbe_pos_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_pos_ctrl', lbe_pos_ctrl_b, data1_index, epoch1maxrec ; 19
			remove_fill, -1, lbe_pos_ctrl_b ; 19
			END
		'lbe_pos_disable0':  BEGIN
			read_product, cdf_id, 'lbe_pos_disable0', lbe_pos_disable0_b, data1_index, epoch1maxrec ; 20
			remove_fill, -1, lbe_pos_disable0_b ; 20
			END
		'lbe_pos_disable2':  BEGIN
			read_product, cdf_id, 'lbe_pos_disable2', lbe_pos_disable2_b, data1_index, epoch1maxrec ; 21
			remove_fill, -1, lbe_pos_disable2_b ; 21
			END
		'lbe_tac0_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_tac0_ctrl', lbe_tac0_ctrl_b, data1_index, epoch1maxrec ; 22
			remove_fill, -1, lbe_tac0_ctrl_b ; 22
			END
		'lbe_tac0_dac':  BEGIN
			read_product, cdf_id, 'lbe_tac0_dac', lbe_tac0_dac_b, data1_index, epoch1maxrec ; 23
			remove_fill, -1, lbe_tac0_dac_b ; 23
			END
		'lbe_tac2_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_tac2_ctrl', lbe_tac2_ctrl_b, data1_index, epoch1maxrec ; 24
			remove_fill, -1, lbe_tac2_ctrl_b ; 24
			END
		'lbe_tac2_dac':  BEGIN
			read_product, cdf_id, 'lbe_tac2_dac', lbe_tac2_dac_b, data1_index, epoch1maxrec ; 25
			remove_fill, -1, lbe_tac2_dac_b ; 25
			END
		'lbe_tac0_under_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac0_under_hi', lbe_tac0_under_hi_b, data1_index, epoch1maxrec ; 26
			remove_fill, -1, lbe_tac0_under_hi_b ; 26
			END
		'lbe_tac0_under_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac0_under_lo', lbe_tac0_under_lo_b, data1_index, epoch1maxrec ; 27
			remove_fill, -1, lbe_tac0_under_lo_b ; 27
			END
		'lbe_tac0_over_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac0_over_hi', lbe_tac0_over_hi_b, data1_index, epoch1maxrec ; 28
			remove_fill, -1, lbe_tac0_over_hi_b ; 28
			END
		'lbe_tac0_over_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac0_over_lo', lbe_tac0_over_lo_b, data1_index, epoch1maxrec ; 29
			remove_fill, -1, lbe_tac0_over_lo_b ; 29
			END
		'lbe_tac2_under_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac2_under_hi', lbe_tac2_under_hi_b, data1_index, epoch1maxrec ; 30
			remove_fill, -1, lbe_tac2_under_hi_b ; 30
			END
		'lbe_tac2_under_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac2_under_lo', lbe_tac2_under_lo_b, data1_index, epoch1maxrec ; 31
			remove_fill, -1, lbe_tac2_under_lo_b ; 31
			END
		'lbe_tac2_over_hi':  BEGIN
			read_product, cdf_id, 'lbe_tac2_over_hi', lbe_tac2_over_hi_b, data1_index, epoch1maxrec ; 32
			remove_fill, -1, lbe_tac2_over_hi_b ; 32
			END
		'lbe_tac2_over_lo':  BEGIN
			read_product, cdf_id, 'lbe_tac2_over_lo', lbe_tac2_over_lo_b, data1_index, epoch1maxrec ; 33
			remove_fill, -1, lbe_tac2_over_lo_b ; 33
			END
		'lbe_ssd_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_ssd_ctrl', lbe_ssd_ctrl_b, data1_index, epoch1maxrec ; 34
			remove_fill, -1, lbe_ssd_ctrl_b ; 34
			END
		'lbe_ssd_cmd_h':  BEGIN
			read_product, cdf_id, 'lbe_ssd_cmd_h', lbe_ssd_cmd_h_b, data1_index, epoch1maxrec ; 35
			remove_fill, -1, lbe_ssd_cmd_h_b ; 35
			END
		'lbe_ssd_cmd_l':  BEGIN
			read_product, cdf_id,'lbe_ssd_cmd_l', lbe_ssd_cmd_l_b, data1_index, epoch1maxrec ; 36
			remove_fill, -1, lbe_ssd_cmd_l_b ; 36
			END
		'lbe_ssd_dis':  BEGIN
			read_product, cdf_id, 'lbe_ssd_dis', lbe_ssd_dis_b, data1_index, epoch1maxrec ; 37
			remove_fill, -1, lbe_ssd_dis_b ; 37
			END
		'lbe_ssd_under_hi':  BEGIN
			read_product, cdf_id,'lbe_ssd_under_hi', lbe_ssd_under_hi_b, data1_index, epoch1maxrec ; 38
			remove_fill, -1, lbe_ssd_under_hi_b ; 38
			END
		'lbe_ssd_under_lo':  BEGIN
			read_product, cdf_id,'lbe_ssd_under_lo', lbe_ssd_under_lo_b, data1_index, epoch1maxrec ; 39
			remove_fill, -1, lbe_ssd_under_lo_b ; 39
			END
		'lbe_ssd_over_hi':  BEGIN
			read_product, cdf_id,'lbe_ssd_over_hi', lbe_ssd_over_hi_b, data1_index, epoch1maxrec ; 40
			remove_fill, -1, lbe_ssd_over_hi_b ; 40
			END
		'lbe_ssd_over_lo':  BEGIN
			read_product, cdf_id,'lbe_ssd_over_lo', lbe_ssd_over_lo_b, data1_index, epoch1maxrec ; 41
			remove_fill, -1, lbe_ssd_over_lo_b ; 41
			END
		'lbe_sel_ctrl':  BEGIN
			read_product, cdf_id, 'lbe_sel_ctrl', lbe_sel_ctrl_b, data1_index, epoch1maxrec ; 42
			remove_fill, -1, lbe_sel_ctrl_b ; 42
			END
		'lbe_trig_mode':  BEGIN
			read_product, cdf_id, 'lbe_trig_mode', lbe_trig_mode_b, data1_index, epoch1maxrec ; 43
			remove_fill, -1, lbe_trig_mode_b ; 43
			END
		'lbe_esa_step':  BEGIN
			read_product, cdf_id, 'lbe_esa_step', lbe_esa_step_b, data1_index, epoch1maxrec ; 44
			remove_fill, -1, lbe_esa_step_b ; 44
			END
		'lbe_defl_step':  BEGIN
			read_product, cdf_id, 'lbe_defl_step', lbe_defl_step_b, data1_index, epoch1maxrec ; 45
			remove_fill, -1, lbe_defl_step_b ; 45
			END
		'lbe_ssd_offset0':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset0', lbe_ssd_offset0_b, data1_index, epoch1maxrec ; 46
			remove_fill, -1, lbe_ssd_offset0_b ; 46
			END
		'lbe_ssd_offset1':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset1', lbe_ssd_offset1_b, data1_index, epoch1maxrec ; 47
			remove_fill, -1, lbe_ssd_offset1_b ; 47
			END
		'lbe_ssd_offset2':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset2', lbe_ssd_offset2_b, data1_index, epoch1maxrec ; 48
			remove_fill, -1, lbe_ssd_offset2_b ; 48
			END
		'lbe_ssd_offset3':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset3', lbe_ssd_offset3_b, data1_index, epoch1maxrec ; 49
			remove_fill, -1, lbe_ssd_offset3_b ; 49
			END
		'lbe_ssd_offset4':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset4', lbe_ssd_offset4_b, data1_index, epoch1maxrec ; 50
			remove_fill, -1, lbe_ssd_offset4_b ; 50
			END
		'lbe_ssd_offset5':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset5', lbe_ssd_offset5_b, data1_index, epoch1maxrec ; 51
			remove_fill, -1, lbe_ssd_offset5_b ; 51
			END
		'lbe_ssd_offset6':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset6', lbe_ssd_offset6_b, data1_index, epoch1maxrec ; 52
			remove_fill, -1, lbe_ssd_offset6_b ; 52
			END
		'lbe_ssd_offset7':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset7', lbe_ssd_offset7_b, data1_index, epoch1maxrec ; 53
			remove_fill, -1, lbe_ssd_offset7_b ; 53
			END
		'lbe_stim_enable':  BEGIN
			read_product, cdf_id, 'lbe_stim_enable', lbe_stim_enable_b, data1_index, epoch1maxrec ; 54
			remove_fill, -1, lbe_stim_enable_b ; 54
			END
		'lbe_stim_freq_hi':  BEGIN
			read_product, cdf_id, 'lbe_stim_freq_hi', lbe_stim_freq_hi_b, data1_index, epoch1maxrec ; 55
			remove_fill, -1, lbe_stim_freq_hi_b ; 55
			END
		'lbe_stim_freq_lo':  BEGIN
			read_product, cdf_id, 'lbe_stim_freq_lo', lbe_stim_freq_lo_b, data1_index, epoch1maxrec ; 56
			remove_fill, -1, lbe_stim_freq_lo_b ; 56
			END
		'lbe_sel0_window':  BEGIN
			read_product, cdf_id, 'lbe_sel0_window', lbe_sel0_window_b, data1_index, epoch1maxrec ; 57
			remove_fill, -1, lbe_sel0_window_b ; 57
			END
		'lbe_sel2_window':  BEGIN
			read_product, cdf_id, 'lbe_sel2_window', lbe_sel2_window_b, data1_index, epoch1maxrec ; 58
			remove_fill, -1, lbe_sel2_window_b ; 58
			END
		'lbe_ssd_offset8':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset8', lbe_ssd_offset8_b, data1_index, epoch1maxrec ; 59
			remove_fill, -1, lbe_ssd_offset8_b ; 59
			END
		'lbe_ssd_offset9':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset9', lbe_ssd_offset9_b, data1_index, epoch1maxrec ; 60
			remove_fill, -1, lbe_ssd_offset9_b ; 60
			END
		'lbe_ssd_offset10':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset10', lbe_ssd_offset10_b, data1_index, epoch1maxrec ; 61
			remove_fill, -1, lbe_ssd_offset10_b ; 61
			END
		'lbe_ssd_offset11':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset11', lbe_ssd_offset11_b, data1_index, epoch1maxrec ; 62
			remove_fill, -1, lbe_ssd_offset11_b ; 62
			END
		'lbe_ssd_offset12':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset12', lbe_ssd_offset12_a, data1_index, epoch1maxrec ; 63
			remove_fill, -1, lbe_ssd_offset12_a ; 63
			END
		'lbe_ssd_offset13':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset13', lbe_ssd_offset13_a, data1_index, epoch1maxrec ; 64
			remove_fill, -1, lbe_ssd_offset13_a ; 64
			END
		'lbe_ssd_offset14':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset14', lbe_ssd_offset14_a, data1_index, epoch1maxrec ; 65
			remove_fill, -1, lbe_ssd_offset14_a ; 65
			END
		'lbe_ssd_offset15':  BEGIN
			read_product, cdf_id, 'lbe_ssd_offset15', lbe_ssd_offset15_a, data1_index, epoch1maxrec ; 66
			remove_fill, -1, lbe_ssd_offset15_a ; 66
			END
		'lbe_pos_ra':  BEGIN
			read_product, cdf_id, 'lbe_pos_ra', lbe_pos_ra_a, data1_index, epoch1maxrec ; 67
			remove_fill, -1, lbe_pos_ra_a ; 67
			END
		'dacm_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
			read_product, cdf_id, 'dacm_pac_mcp_ssd_ctl_mcp_limit', dacm_pac_mcp_ssd_ctl_mcp_limit_a, data1_index, epoch1maxrec ; 68
			remove_fill, -1, dacm_pac_mcp_ssd_ctl_mcp_limit_a ; 68
			END
		'dacm_pac_limit':  BEGIN
			read_product, cdf_id, 'dacm_pac_limit', dacm_pac_limit_b, data1_index, epoch1maxrec ; 69
			remove_fill, -1, dacm_pac_limit_b ; 69
			END
		'dacm_esa_ctl':  BEGIN
			read_product, cdf_id, 'dacm_esa_ctl', dacm_esa_ctl_b, data1_index, epoch1maxrec ; 70
			remove_fill, -1, dacm_esa_ctl_b ; 70
			END
		'dacm_s_ch_ctl':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_ctl', dacm_s_ch_ctl_b, data1_index, epoch1maxrec ; 71
			remove_fill, -1, dacm_s_ch_ctl_b ; 71
			END
		'dacm_dfl_1_ctl':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_ctl', dacm_dfl_1_ctl_b, data1_index, epoch1maxrec ; 72
			remove_fill, -1, dacm_dfl_1_ctl_b ; 72
			END
		'dacm_dfl_2_ctl':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_ctl', dacm_dfl_2_ctl_b, data1_index, epoch1maxrec ; 73
			remove_fill, -1, dacm_dfl_2_ctl_b ; 73
			END
		'dacm_control':  BEGIN
			read_product, cdf_id, 'dacm_control', dacm_control_b, data1_index, epoch1maxrec ; 74
			remove_fill, -1, dacm_control_b ; 74
			END
		'dacm_status':  BEGIN
			read_product, cdf_id, 'dacm_status', dacm_status_b, data1_index, epoch1maxrec ; 75
			remove_fill, -1, dacm_status_b ; 75
			END
		'dacm_adc_lu_ctr':  BEGIN
			read_product, cdf_id, 'dacm_adc_lu_ctr', dacm_adc_lu_ctr_b, data1_index, epoch1maxrec ;76
			remove_fill, -1, dacm_adc_lu_ctr_b ; 76
			END
		'dacm_dfl_1_offset':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_offset', dacm_dfl_1_offset_b, data1_index, epoch1maxrec ; 77
			remove_fill, -1, dacm_dfl_1_offset_b ; 77
			END
		'dacm_dfl_2_offset':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_offset', dacm_dfl_2_offset_b, data1_index, epoch1maxrec ; 78
			remove_fill, -1, dacm_dfl_2_offset_b ; 78
			END
		'dacm_pac_coarse':  BEGIN
			read_product, cdf_id, 'dacm_pac_coarse', dacm_pac_coarse_b, data1_index, epoch1maxrec ; 79
			remove_fill, -1, dacm_pac_coarse_b ; 79
			END
		'dacm_pac_fine':  BEGIN
			read_product, cdf_id, 'dacm_pac_fine', dacm_pac_fine_b, data1_index, epoch1maxrec ; 80
			remove_fill, -1, dacm_pac_fine_b ; 80
			END
		'dacm_pac_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacm_pac_cm_fs_coarse', dacm_pac_cm_fs_coarse_b, data1_index, epoch1maxrec ; 81
			remove_fill, -1, dacm_pac_cm_fs_coarse_b ; 81
			END
		'dacm_pac_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacm_pac_cm_fs_fine', dacm_pac_cm_fs_fine_b, data1_index, epoch1maxrec ; 82
			remove_fill, -1, dacm_pac_cm_fs_fine_b ; 82
			END
		'dacm_mcp_coarse':  BEGIN
			read_product, cdf_id, 'dacm_mcp_coarse', dacm_mcp_coarse_b, data1_index, epoch1maxrec ; 83
			remove_fill, -1, dacm_mcp_coarse_b ; 83
			END
		'dacm_mcp_fine':  BEGIN
			read_product, cdf_id, 'dacm_mcp_fine', dacm_mcp_fine_b, data1_index, epoch1maxrec ; 84
			remove_fill, -1, dacm_mcp_fine_b ; 84
			END
		'dacm_mcp_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacm_mcp_cm_fs_coarse', dacm_mcp_cm_fs_coarse_b, data1_index, epoch1maxrec ; 85
			remove_fill, -1, dacm_mcp_cm_fs_coarse_b ; 85
			END
		'dacm_mcp_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacm_mcp_cm_fs_fine', dacm_mcp_cm_fs_fine_b, data1_index, epoch1maxrec ; 86
			remove_fill, -1, dacm_mcp_cm_fs_fine_b ; 86
			END
		'dacm_ssd_coarse':  BEGIN
			read_product, cdf_id, 'dacm_ssd_coarse', dacm_ssd_coarse_b, data1_index, epoch1maxrec ; 87
			remove_fill, -1, dacm_ssd_coarse_b ; 87
			END
		'dacm_ssd_fine':  BEGIN
			read_product, cdf_id, 'dacm_ssd_fine', dacm_ssd_fine_b, data1_index, epoch1maxrec ; 88
			remove_fill, -1, dacm_ssd_fine_b ; 88
			END
		'dacm_esa_coarse':  BEGIN
			read_product, cdf_id, 'dacm_esa_coarse', dacm_esa_coarse_b, data1_index, epoch1maxrec ; 89
			remove_fill, -1, dacm_esa_coarse_b ; 89
			END
		'dacm_esa_fine':  BEGIN
			read_product, cdf_id, 'dacm_esa_fine', dacm_esa_fine_b, data1_index, epoch1maxrec ; 90
			remove_fill, -1, dacm_esa_fine_b ; 90
			END
		'dacm_s_ch_coarse':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_coarse', dacm_s_ch_coarse_b, data1_index, epoch1maxrec ; 91
			remove_fill, -1, dacm_s_ch_coarse_b ; 91
			END
		'dacm_s_ch_fine':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_fine', dacm_s_ch_fine_b, data1_index, epoch1maxrec ; 92
			remove_fill, -1, dacm_s_ch_fine_b ; 92
			END
		'dacm_s_ch_tap_coarse':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_tap_coarse', dacm_s_ch_tap_coarse_b, data1_index, epoch1maxrec ; 93
			remove_fill, -1, dacm_s_ch_tap_coarse_b ; 93
			END
		'dacm_s_ch_tap_fine':  BEGIN
			read_product, cdf_id, 'dacm_s_ch_tap_fine', dacm_s_ch_tap_fine_b, data1_index, epoch1maxrec ; 94
			remove_fill, -1, dacm_s_ch_tap_fine_b ; 94
			END
		'dacm_dfl_1_coarse':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_coarse', dacm_dfl_1_coarse_b, data1_index, epoch1maxrec ; 95
			remove_fill, -1, dacm_dfl_1_coarse_b ; 95
			END
		'dacm_dfl_1_fine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_fine', dacm_dfl_1_fine_b, data1_index, epoch1maxrec ; 96
			remove_fill, -1, dacm_dfl_1_fine_b ; 96
			END
		'dacm_dfl_1_dfine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_1_dfine', dacm_dfl_1_dfine_b, data1_index, epoch1maxrec ; 97
			remove_fill, -1, dacm_dfl_1_dfine_b ; 97
			END
		'dacm_dfl_2_coarse':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_coarse', dacm_dfl_2_coarse_b, data1_index, epoch1maxrec ; 98
			remove_fill, -1, dacm_dfl_2_coarse_b ; 98
			END
		'dacm_dfl_2_fine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_fine', dacm_dfl_2_fine_b, data1_index, epoch1maxrec ; 99
			remove_fill, -1, dacm_dfl_2_fine_b ; 99
			END
		'dacm_dfl_2_dfine':  BEGIN
			read_product, cdf_id, 'dacm_dfl_2_dfine', dacm_dfl_2_dfine_b, data1_index, epoch1maxrec ; 100
			remove_fill, -1, dacm_dfl_2_dfine_b ; 100
			END
		'dacr_pac_mcp_ssd_ctl_mcp_limit':  BEGIN
			read_product, cdf_id, 'dacr_pac_mcp_ssd_ctl_mcp_limit', dacr_pac_mcp_ssd_ctl_mcp_limit_b, data1_index, epoch1maxrec ; 101
			remove_fill, -1, dacr_pac_mcp_ssd_ctl_mcp_limit_b ; 101
			END
		'dacr_pac_limit':  BEGIN
			read_product, cdf_id, 'dacr_pac_limit', dacr_pac_limit_b, data1_index, epoch1maxrec ; 102
			remove_fill, -1, dacr_pac_limit_b ; 102
			END
		'dacr_esa_ctl':  BEGIN
			read_product, cdf_id, 'dacr_esa_ctl', dacr_esa_ctl_b, data1_index, epoch1maxrec ; 103
			remove_fill, -1, dacr_esa_ctl_b ; 103
			END
		'dacr_s_ch_ctl':  BEGIN
			read_product, cdf_id, 'dacr_s_ch_ctl', dacr_s_ch_ctl_b, data1_index, epoch1maxrec  ; 104
			remove_fill, -1, dacr_s_ch_ctl_b ; 104
			END
		'dacr_dfl_1_ctl':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_ctl', dacr_dfl_1_ctl_b, data1_index, epoch1maxrec ; 105
			remove_fill, -1, dacr_dfl_1_ctl_b ; 105
			END
		'dacr_dfl_2_ctl':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_ctl', dacr_dfl_2_ctl_b, data1_index, epoch1maxrec ; 106
			remove_fill, -1, dacr_dfl_2_ctl_b ; 106
			END
		'dacr_control':  BEGIN
			read_product, cdf_id, 'dacr_control', dacr_control_b, data1_index, epoch1maxrec ; 107
			remove_fill, -1, dacr_control_b ; 107
			END
		'dacr_status':  BEGIN
			read_product, cdf_id, 'dacr_status', dacr_status_b, data1_index, epoch1maxrec ; 108
			remove_fill, -1, dacr_status_b ; 108
			END
		'dacr_adc_lu_ctr':  BEGIN
			read_product, cdf_id, 'dacr_adc_lu_ctr', dacr_adc_lu_ctr_b, data1_index, epoch1maxrec  ; 109
			remove_fill, -1, dacr_adc_lu_ctr_b ; 109
			END
		'dacr_dfl_1_offset':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_offset', dacr_dfl_1_offset_b, data1_index, epoch1maxrec ; 110
			remove_fill, -1, dacr_dfl_1_offset_b ; 110
			END
		'dacr_dfl_2_offset':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_offset', dacr_dfl_2_offset_b, data1_index, epoch1maxrec ; 111
			remove_fill, -1, dacr_dfl_2_offset_b ; 111
			END
		'dacr_c_pac_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_coarse', dacr_c_pac_coarse_b, data1_index, epoch1maxrec ; 112
			remove_fill, -1, dacr_c_pac_coarse_b ; 112
			END
		'dacr_c_pac_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_fine', dacr_c_pac_fine_b, data1_index, epoch1maxrec ; 113
			remove_fill, -1, dacr_c_pac_fine_b ; 113
			END
		'dacr_c_pac_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_cm_fs_coarse', dacr_c_pac_cm_fs_coarse_b, data1_index, epoch1maxrec ; 114
			remove_fill, -1, dacr_c_pac_cm_fs_coarse_b ; 114
			END
		'dacr_c_pac_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_pac_cm_fs_fine', dacr_c_pac_cm_fs_fine_b, data1_index, epoch1maxrec ; 115
			remove_fill, -1, dacr_c_pac_cm_fs_fine_b ; 115
			END
		'dacr_c_mcp_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_coarse', dacr_c_mcp_coarse_b, data1_index, epoch1maxrec  ; 116
			remove_fill, -1, dacr_c_mcp_coarse_b ; 116
			END
		'dacr_c_mcp_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_fine', dacr_c_mcp_fine_b, data1_index, epoch1maxrec ; 117
			remove_fill, -1, dacr_c_mcp_fine_b ; 117
			END
		'dacr_c_mcp_cm_fs_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_cm_fs_coarse', dacr_c_mcp_cm_fs_coarse_b, data1_index, epoch1maxrec ; 118
			remove_fill, -1, dacr_c_mcp_cm_fs_coarse_b ; 118
			END
		'dacr_c_mcp_cm_fs_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_mcp_cm_fs_fine', dacr_c_mcp_cm_fs_fine_b, data1_index, epoch1maxrec ; 119
			remove_fill, -1, dacr_c_mcp_cm_fs_fine_b ; 119
			END
		'dacr_c_ssd_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_ssd_coarse', dacr_c_ssd_coarse_b, data1_index, epoch1maxrec ; 120
			remove_fill, -1, dacr_c_ssd_coarse_b ; 120
			END
		'dacr_c_ssd_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_ssd_fine', dacr_c_ssd_fine_b, data1_index, epoch1maxrec ; 121
			remove_fill, -1, dacr_c_ssd_fine_b ; 121
			END
		'dacr_c_esa_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_esa_coarse', dacr_c_esa_coarse_b, data1_index, epoch1maxrec ; 122
			remove_fill, -1, dacr_c_esa_coarse_b ; 122
			END
		'dacr_c_esa_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_esa_fine', dacr_c_esa_fine_b, data1_index, epoch1maxrec ; 123
			remove_fill, -1, dacr_c_esa_fine_b ; 123
			END
		'dacr_c_s_ch_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_coarse', dacr_c_s_ch_coarse_b, data1_index, epoch1maxrec ; 124
			remove_fill, -1, dacr_c_s_ch_coarse_b ; 124
			END
		'dacr_c_s_ch_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_fine', dacr_c_s_ch_fine_b, data1_index, epoch1maxrec ; 125
			remove_fill, -1, dacr_c_s_ch_fine_b ; 125
			END
		'dacr_c_s_ch_tap_coarse':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_tap_coarse', dacr_c_s_ch_tap_coarse_b, data1_index, epoch1maxrec ; 126
			remove_fill, -1, dacr_c_s_ch_tap_coarse_b ; 126
			END
		'dacr_c_s_ch_tap_fine':  BEGIN
			read_product, cdf_id, 'dacr_c_s_ch_tap_fine', dacr_c_s_ch_tap_fine_b, data1_index, epoch1maxrec ; 127
			remove_fill, -1, dacr_c_s_ch_tap_fine_b ; 127
			END
		'dacr_dfl_1_coarse':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_coarse', dacr_dfl_1_coarse_b, data1_index, epoch1maxrec ; 128
			remove_fill, -1, dacr_dfl_1_coarse_b ; 128
			END
		'dacr_dfl_1_fine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_fine', dacr_dfl_1_fine_b, data1_index, epoch1maxrec ; 129
			remove_fill, -1, dacr_dfl_1_fine_b ; 129
			END
		'dacr_dfl_1_dfine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_1_dfine', dacr_dfl_1_dfine_b, data1_index, epoch1maxrec  ; 130
			remove_fill, -1, dacr_dfl_1_dfine_b ; 130
			END
		'dacr_dfl_2_coarse':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_coarse', dacr_dfl_2_coarse_b, data1_index, epoch1maxrec  ; 131
			remove_fill, -1, dacr_dfl_2_coarse_b ; 131
			END
		'dacr_dfl_2_fine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_fine', dacr_dfl_2_fine_b, data1_index, epoch1maxrec ; 132
			remove_fill, -1, dacr_dfl_2_fine_b ; 132
			END
		'dacr_dfl_2_dfine':  BEGIN
			read_product, cdf_id, 'dacr_dfl_2_dfine', dacr_dfl_2_dfine_b, data1_index, epoch1maxrec ; 133
			remove_fill, -1, dacr_dfl_2_dfine_b ; 133
			END
		'sw_status1':  BEGIN
			read_product, cdf_id, 'sw_status1', sw_status1_b, data1_index, epoch1maxrec ; 134
			remove_fill, -1, sw_status1_b ; 134
			END
		'sw_hv_ramp_status':  BEGIN
			read_product, cdf_id, 'sw_hv_ramp_status', sw_hv_ramp_status_b, data1_index, epoch1maxrec ; 135
			remove_fill, -1, sw_hv_ramp_status_b ; 135
			END
		'sw_hv_enable_status':  BEGIN
			read_product, cdf_id, 'sw_hv_enable_status', sw_hv_enable_status_b, data1_index, epoch1maxrec ; 136
			remove_fill, -1, sw_hv_enable_status_b ; 136
			END
		'sw_hv_limit_status':  BEGIN
			read_product, cdf_id, 'sw_hv_limit_status', sw_hv_limit_status_b, data1_index, epoch1maxrec ; 137
			remove_fill, -1, sw_hv_limit_status_b ; 137
			END
		'sw_eepromchksum':  BEGIN
			read_product, cdf_id, 'sw_eepromchksum', sw_eepromchksum_b, data1_index, epoch1maxrec ; 138
			remove_fill, -1, sw_eepromchksum_b ; 138
			END
		'hkp_sampletime':  BEGIN
			read_product, cdf_id, 'hkp_sampletime', hkp_sampletime_b, data1_index, epoch1maxrec ; 139
			remove_fill, -1, hkp_sampletime_b ; 139
			END
		'pac_discharge_status':  BEGIN
			read_product, cdf_id, 'pac_discharge_status', pac_discharge_status_b, data1_index, epoch1maxrec ; 140
			remove_fill, -1, pac_discharge_status_b ; 140
			END
		'pac_discharge_cnt':  BEGIN
			read_product, cdf_id, 'pac_discharge_cnt', pac_discharge_cnt_b, data1_index, epoch1maxrec ; 141
			remove_fill, -1, pac_discharge_cnt_b ; 141
			END
		'pac_discharge_conseccntr':  BEGIN
			read_product, cdf_id, 'pac_discharge_conseccntr', pac_discharge_conseccntr_b, data1_index, epoch1maxrec ; 142
			remove_fill, -1, pac_discharge_conseccntr_b ; 142
			END
		'pac_discharge_safe_att':  BEGIN
			read_product, cdf_id, 'pac_discharge_safe_att', pac_discharge_safe_att_b, data1_index, epoch1maxrec ; 143
			remove_fill, -1, pac_discharge_safe_att_b ; 143
			END
		'analog_limit_status':  BEGIN
			read_product, cdf_id, 'analog_limit_status', analog_limit_status_b, data1_index, epoch1maxrec ; 144
			remove_fill, -1, analog_limit_status_b ; 144
			END
		'analog_lim_pt1_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt1_status', analog_lim_pt1_status_b, data1_index, epoch1maxrec ; 145
			remove_fill, -1, analog_lim_pt1_status_b ; 145
			END
		'analog_lim_pt1_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt1_id', analog_lim_pt1_id_b, data1_index, epoch1maxrec ; 146
			remove_fill, -1, analog_lim_pt1_id_b ; 146
			END
		'analog_lim_pt1_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt1_red', analog_lim_pt1_red_b, data1_index, epoch1maxrec ; 147
			remove_fill, -1, analog_lim_pt1_red_b ; 147
			END
		'analog_lim_pt2_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt2_status', analog_lim_pt2_status_b, data1_index, epoch1maxrec ; 148
			remove_fill, -1, analog_lim_pt2_status_b ; 148
			END
		'analog_lim_pt2_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt2_id', analog_lim_pt2_id_b, data1_index, epoch1maxrec ; 149
			remove_fill, -1, analog_lim_pt2_id_b ; 149
			END
		'analog_lim_pt2_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt2_red', analog_lim_pt2_red_b, data1_index, epoch1maxrec ; 150
			remove_fill, -1, analog_lim_pt2_red_b ; 150
			END
		'analog_lim_pt3_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt3_status', analog_lim_pt3_status_b, data1_index, epoch1maxrec ; 151
			remove_fill, -1, analog_lim_pt3_status_b ; 151
			END
		'analog_lim_pt3_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt3_id', analog_lim_pt3_id_b, data1_index, epoch1maxrec ; 152
			remove_fill, -1, analog_lim_pt3_id_b ; 152
			END
		'analog_lim_pt3_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt3_red', analog_lim_pt3_red_b, data1_index, epoch1maxrec ; 153
			remove_fill, -1, analog_lim_pt3_red_b ; 153
			END
		'analog_lim_pt4_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt4_status', analog_lim_pt4_status_b, data1_index, epoch1maxrec ; 154
			remove_fill, -1, analog_lim_pt4_status_b ; 154
			END
		'analog_lim_pt4_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt4_id', analog_lim_pt4_id_b, data1_index, epoch1maxrec ; 155
			remove_fill, -1, analog_lim_pt4_id_b ; 155
			END
		'analog_lim_pt4_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt4_red', analog_lim_pt4_red_b, data1_index, epoch1maxrec ; 156
			remove_fill, -1, analog_lim_pt4_red_b ; 156
			END
		'analog_lim_pt5_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt5_status', analog_lim_pt5_status_b, data1_index, epoch1maxrec ; 157
			remove_fill, -1, analog_lim_pt5_status_b ; 157
			END
		'analog_lim_pt5_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt5_id', analog_lim_pt5_id_b, data1_index, epoch1maxrec ; 158
			remove_fill, -1, analog_lim_pt5_id_b ; 158
			END
		'analog_lim_pt5_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt5_red', analog_lim_pt5_red_b, data1_index, epoch1maxrec ; 159
			remove_fill, -1, analog_lim_pt5_red_b ; 159
			END
		'analog_lim_pt6_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt6_status', analog_lim_pt6_status_b, data1_index, epoch1maxrec ; 160
			remove_fill, -1, analog_lim_pt6_status_b ; 160
			END
		'analog_lim_pt6_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt6_id', analog_lim_pt6_id_b, data1_index, epoch1maxrec ; 161
			remove_fill, -1, analog_lim_pt6_id_b ; 161
			END
		'analog_lim_pt6_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt6_red', analog_lim_pt6_red_b, data1_index, epoch1maxrec ; 162
			remove_fill, -1, analog_lim_pt6_red_b ; 162
			END
		'analog_lim_pt7_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt7_status', analog_lim_pt7_status_b, data1_index, epoch1maxrec ; 163
			remove_fill, -1, analog_lim_pt7_status_b ; 163
			END
		'analog_lim_pt7_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt7_id', analog_lim_pt7_id_b, data1_index, epoch1maxrec ; 164
			remove_fill, -1, analog_lim_pt7_id_b ; 164
			END
		'analog_lim_pt7_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt7_red', analog_lim_pt7_red_b, data1_index, epoch1maxrec ; 165
			remove_fill, -1, analog_lim_pt7_red_b ; 165
			END
		'analog_lim_pt8_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt8_status', analog_lim_pt8_status_b, data1_index, epoch1maxrec ; 166
			remove_fill, -1, analog_lim_pt8_status_b ; 166
			END
		'analog_lim_pt8_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt8_id', analog_lim_pt8_id_b, data1_index, epoch1maxrec ; 167
			remove_fill, -1, analog_lim_pt8_id_b ; 167
			END
		'analog_lim_pt8_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt8_red', analog_lim_pt8_red_b, data1_index, epoch1maxrec ; 168
			remove_fill, -1, analog_lim_pt8_red_b ; 168
			END
		'analog_lim_pt9_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt9_status', analog_lim_pt9_status_b, data1_index, epoch1maxrec ; 169
			remove_fill, -1, analog_lim_pt9_status_b ; 169
			END
		'analog_lim_pt9_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt9_id', analog_lim_pt9_id_b, data1_index, epoch1maxrec ; 170
			remove_fill, -1, analog_lim_pt9_id_b ; 170
			END
		'analog_lim_pt9_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt9_red', analog_lim_pt9_red_b, data1_index, epoch1maxrec ; 171
			remove_fill, -1, analog_lim_pt9_red_b ; 171
			END
		'analog_lim_pt10_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt10_status', analog_lim_pt10_status_b, data1_index, epoch1maxrec ; 172
			remove_fill, -1, analog_lim_pt10_status_b ; 172
			END
		'analog_lim_pt10_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt10_id', analog_lim_pt10_id_b, data1_index, epoch1maxrec ; 173
			remove_fill, -1, analog_lim_pt10_id_b ; 173
			END
		'analog_lim_pt10_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt10_red', analog_lim_pt10_red_b, data1_index, epoch1maxrec ; 174
			remove_fill, -1, analog_lim_pt10_red_b ; 174
			END
		'analog_lim_pt11_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt11_status', analog_lim_pt11_status_b, data1_index, epoch1maxrec ; 175
			remove_fill, -1, analog_lim_pt11_status_b ; 175
			END
		'analog_lim_pt11_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt11_id', analog_lim_pt11_id_b, data1_index, epoch1maxrec ; 176
			remove_fill, -1, analog_lim_pt11_id_b ; 176
			END
		'analog_lim_pt11_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt11_red', analog_lim_pt11_red_b, data1_index, epoch1maxrec ; 177
			remove_fill, -1, analog_lim_pt11_red_b ; 177
			END
		'analog_lim_pt12_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt12_status', analog_lim_pt12_status_b, data1_index, epoch1maxrec ; 178
			remove_fill, -1, analog_lim_pt12_status_b ; 178
			END
		'analog_lim_pt12_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt12_id', analog_lim_pt12_id_b, data1_index, epoch1maxrec ; 179
			remove_fill, -1, analog_lim_pt12_id_b ; 179
			END
		'analog_lim_pt12_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt12_red', analog_lim_pt12_red_b, data1_index, epoch1maxrec ; 180
			remove_fill, -1, analog_lim_pt12_red_b ; 180
			END
		'analog_lim_pt13_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt13_status', analog_lim_pt13_status_b, data1_index, epoch1maxrec ; 181
			remove_fill, -1, analog_lim_pt13_status_b ; 181
			END
		'analog_lim_pt13_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt13_id', analog_lim_pt13_id_b, data1_index, epoch1maxrec ; 182
			remove_fill, -1, analog_lim_pt13_id_b ; 182
			END
		'analog_lim_pt13_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt13_red', analog_lim_pt13_red_b, data1_index, epoch1maxrec ; 183
			remove_fill, -1, analog_lim_pt13_red_b ; 183
			END
		'analog_lim_pt14_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt14_status', analog_lim_pt14_status_b, data1_index, epoch1maxrec ; 184
			remove_fill, -1, analog_lim_pt14_status_b ; 184
			END
		'analog_lim_pt14_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt14_id', analog_lim_pt14_id_b, data1_index, epoch1maxrec ; 185
			remove_fill, -1, analog_lim_pt14_id_b ; 185
			END
		'analog_lim_pt14_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt14_red', analog_lim_pt14_red_b, data1_index, epoch1maxrec ; 186
			remove_fill, -1, analog_lim_pt14_red_b ; 186
			END
		'analog_lim_pt15_status':  BEGIN
			read_product, cdf_id, 'analog_lim_pt15_status', analog_lim_pt15_status_b, data1_index, epoch1maxrec ; 187
			remove_fill, -1, analog_lim_pt15_status_b ; 187
			END
		'analog_lim_pt15_id':  BEGIN
			read_product, cdf_id, 'analog_lim_pt15_id', analog_lim_pt15_id_b, data1_index, epoch1maxrec  ; 188
			remove_fill, -1, analog_lim_pt15_id_b ; 188
			END
		'analog_lim_pt15_red':  BEGIN
			read_product, cdf_id, 'analog_lim_pt15_red', analog_lim_pt15_red_b, data1_index, epoch1maxrec  ; 189
			remove_fill, -1, analog_lim_pt15_red_b ; 189
			END
		'opmode':  BEGIN
			read_product, cdf_id, 'opmode', opmode_b, data1_index, epoch1maxrec  ; 190
			remove_fill, -1, opmode_b ; 190
			END
		'sw_mode':  BEGIN
			read_product, cdf_id, 'sw_mode', sw_mode_b, data1_index, epoch1maxrec ; 191
			remove_fill, -1, sw_mode_b ; 191
			END
		'proton_mode':  BEGIN
			read_product, cdf_id, 'proton_mode', proton_mode_b, data1_index, epoch1maxrec ; 192
			remove_fill, -1, proton_mode_b ; 192
			END
		'active_supplyid':  BEGIN
			read_product, cdf_id, 'active_supplyid', active_supplyid_b, data1_index, epoch1maxrec ; 193
			remove_fill, -1, active_supplyid_b ; 193
			END
		'supply_coarse_rb':  BEGIN
			read_product, cdf_id, 'supply_coarse_rb', supply_coarse_rb_b, data1_index, epoch1maxrec ; 194
			remove_fill, -1, supply_coarse_rb_b ; 194
			END
		'supply_fine_rb':  BEGIN
			read_product, cdf_id, 'supply_fine_rb', supply_fine_rb_b, data1_index, epoch1maxrec ; 195
			remove_fill, -1, supply_fine_rb_b ; 195
			END
		'swmaint_supplystat':  BEGIN
			read_product, cdf_id, 'swmaint_supplystat', swmaint_supplystat_b, data1_index, epoch1maxrec ; 196
			remove_fill, -1, swmaint_supplystat_b ; 196
			END
		'swmaint_pac_coarse':  BEGIN
			read_product, cdf_id, 'swmaint_pac_coarse', swmaint_pac_coarse_b, data1_index, epoch1maxrec ; 197
			remove_fill, -1, swmaint_pac_coarse_b ; 197
			END
		'swmaint_pac_fine':  BEGIN
			read_product, cdf_id, 'swmaint_pac_fine', swmaint_pac_fine_b, data1_index, epoch1maxrec ; 198		
			remove_fill, -1, swmaint_pac_fine_b ; 198
			END

		ENDCASE
	ENDFOR
    END
ENDCASE
dig_hk_read = 1
END 
;-----------------------------------------------------------------------

;---------------------------------------------------------------------------------------------------------------
;---------------------------------------------------------------------------------------------------------------
PRO read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, prod_name, full_values, sum_full
;---------------------------------------------------------------------------------------------------------------
; check to see if full resolution record has data

COMMON com_mon

temp_name = prod_name+'_full'

read_product, cdf_id, temp_name, full_values, data1_index, epoch1maxrec

sum_full=DBLARR(num_records1,128)
sum_full[*, *] = !values.d_nan

remove_fill, -1, full_values
temp = where(finite(full_values) EQ 1,  count)

IF count GT 0 THEN BEGIN 
    FOR j = 0, num_records1-1 DO BEGIN
        FOR k = 0, 127 DO BEGIN	
            sum_full[j, k] = TOTAL(full_values(k, *, j), /NAN)
        ENDFOR
    ENDFOR
ENDIF

END 
;--------------------------------------------------------------

;------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------
PRO read_mon_full, cdf_id, sat, mon_full_read, data1_index, epoch1maxrec, products, num_records1, E_step
;------------------------------------------------------------------------------------------------------
; read full resolution rate data


COMMON com_mon
CASE sat OF 
    'A': BEGIN
	
	read_product, cdf_id, 'error1', error1_full_a, data1_index, epoch1maxrec
	remove_fill, -1, error1_full_a
	IF (TOTAL(error1_full_a, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'


        FOR ii = 0, n_elements(products)-1 DO BEGIN

            CASE products[ii] OF
                's_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_valid', s_valid_a_full, sum_defl_s_valid_a_full
                's_e_not_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_not_req', s_e_not_req_a_full, sum_defl_s_e_not_req_a_full
		's_e_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_req', s_e_req_a_full, sum_defl_s_e_req_a_full
		's_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_pos', s_no_pos_a_full, sum_defl_s_no_pos_a_full
		's_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_pos', s_mult_pos_a_full, sum_defl_s_mult_pos_a_full
		's_no_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_e', s_no_e_a_full, sum_defl_s_no_e_a_full
		's_mult_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_e', s_mult_e_a_full, sum_defl_s_mult_e_a_full
		'ra_sat_a': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_a', ra_sat_a_a_full, sum_defl_ra_sat_a_a_full
		'ra_sat_b': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_b', ra_sat_b_a_full, sum_defl_ra_sat_b_a_full
		'ra_sat_both': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_both', ra_sat_both_a_full, sum_defl_ra_sat_both_a_full
		'ssd_sw': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_sw', ssd_sw_a_full, sum_defl_ssd_sw_a_full
		'ssd_st': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_st', ssd_st_a_full, sum_defl_ssd_st_a_full
 		'sf0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf0', sf0_a_full, sum_defl_sf0_a_full
		'sfr0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr0', sfr0_a_full, sum_defl_sfr0_a_full
		'stp0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp0', stp0_a_full, sum_defl_stp0_a_full
		'ra_trig': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_trig', ra_trig_a_full, sum_defl_ra_trig_a_full
		'pos1_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_0', pos1_0_a_full, sum_defl_pos1_0_a_full
		'pos1_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_1', pos1_1_a_full, sum_defl_pos1_1_a_full
		'w_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_no_pos', w_no_pos_a_full, sum_defl_w_no_pos_a_full
		'w_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_mult_pos', w_mult_pos_a_full, sum_defl_w_mult_pos_a_full
		'w_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_valid', w_valid_a_full, sum_defl_w_valid_a_full
		'sf2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf2', sf2_a_full, sum_defl_sf2_a_full
		'sfr2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr2', sfr2_a_full, sum_defl_sfr2_a_full
		'stp2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp2', stp2_a_full, sum_defl_stp2_a_full
		'pos2_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_0', pos2_0_a_full, sum_defl_pos2_0_a_full
		'pos2_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_1', pos2_1_a_full, sum_defl_pos2_1_a_full
		'pos2_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_2', pos2_2_a_full, sum_defl_pos2_2_a_full
		'pos2_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_3', pos2_3_a_full, sum_defl_pos2_3_a_full
		'pos3_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_0', pos3_0_a_full, sum_defl_pos3_0_a_full
		'pos3_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_1', pos3_1_a_full, sum_defl_pos3_1_a_full
		'pos3_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_2', pos3_2_a_full, sum_defl_pos3_2_a_full
		'pos3_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_3', pos3_3_a_full, sum_defl_pos3_3_a_full
		'pos2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_0', pos2_0_a_full, sum_defl_pos2_0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_1', pos2_1_a_full,  sum_defl_pos2_1_a_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_2', pos2_2_a_full,  sum_defl_pos2_2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_3', pos2_3_a_full, sum_defl_pos2_3_a_full
			
			FOR kk = 0D, n_elements(pos2_0_a_full) - 1 DO BEGIN
				pos2_a_full[kk] = pos2_0_a_full[kk]+pos2_1_a_full[kk]+pos2_2_a_full[kk]+pos2_3_a_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_a_full) - 1 DO BEGIN
				sum_defl_pos2_a_full[mm] = sum_defl_pos2_0_a_full[mm]+sum_defl_pos2_1_a_full[mm]+sum_defl_pos2_2_a_full[mm]+sum_defl_pos2_3_a_full[mm]
			ENDFOR
			
			END
		'pos3':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_0', pos3_0_a_full,  sum_defl_pos3_0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_1', pos3_1_a_full,  sum_defl_pos3_1_a_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_2', pos3_2_a_full,  sum_defl_pos3_2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_3', pos3_3_a_full,  sum_defl_pos3_3_a_full
			
			FOR kk = 0D, n_elements(pos3_0_a_full) - 1 DO BEGIN
				pos3_a_full[kk] = pos3_0_a_full[kk]+pos3_1_a_full[kk]+pos3_2_a_full[kk]+pos3_3_a_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_a_full) - 1 DO BEGIN
				sum_defl_pos3_a_full[mm] = sum_defl_pos3_0_a_full[mm]+sum_defl_pos3_1_a_full[mm]+sum_defl_pos3_2_a_full[mm]+sum_defl_pos3_3_a_full[mm]
			ENDFOR
			
			END

		'stop0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf0', sf0_a_full, sum_defl_sf0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_a_full, sum_defl_sfr0_a_full
			
			FOR kk = 0D, n_elements(sf0_a_full) - 1 DO BEGIN
			IF (sf0_a_full[kk] NE 0) AND (FINITE(sf0_a_full[kk]) EQ 1) AND (FINITE(sfr0_a_full[kk]) EQ 1) THEN BEGIN
				stop0_a_full[kk] = sfr0_a_full[kk]/sf0_a_full[kk]
			ENDIF ELSE BEGIN
				stop0_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_a_full) - 1 DO BEGIN
			IF (sum_defl_sf0_a_full[mm] NE 0) AND (FINITE(sum_defl_sf0_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_a_full[mm] = sum_defl_sfr0_a_full[mm]/sum_defl_sf0_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'start0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp0', stp0_a_full, sum_defl_stp0_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_a_full, sum_defl_sfr0_a_full
			
			FOR kk = 0D, n_elements(stp0_a_full) - 1 DO BEGIN
			IF (stp0_a_full[kk] NE 0) AND (FINITE(stp0_a_full[kk]) EQ 1) AND (FINITE(sfr0_a_full[kk]) EQ 1) THEN BEGIN
				start0_a_full[kk] = sfr0_a_full[kk]/stp0_a_full[kk]
			ENDIF ELSE BEGIN
				start0_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_a_full) - 1 DO BEGIN
			IF (sum_defl_stp0_a_full[mm] NE 0) AND (FINITE(sum_defl_stp0_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_a_full[mm] = sum_defl_sfr0_a_full[mm]/sum_defl_stp0_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'stop2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf2', sf2_a_full, sum_defl_sf2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_a_full, sum_defl_sfr2_a_full
			
			FOR kk = 0D, n_elements(sf2_a_full) - 1 DO BEGIN
			IF (sf2_a_full[kk] NE 0) AND (FINITE(sf2_a_full[kk]) EQ 1) AND (FINITE(sfr2_a_full[kk]) EQ 1) THEN BEGIN
				stop2_a_full[kk] = sfr2_a_full[kk]/sf2_a_full[kk]
			ENDIF ELSE BEGIN
				stop2_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_a_full) - 1 DO BEGIN
			IF (sum_defl_sf2_a_full[mm] NE 0) AND (FINITE(sum_defl_sf2_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_a_full[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_sf2_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

		'start2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp2', stp2_a_full, sum_defl_stp2_a_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_a_full, sum_defl_sfr2_a_full
			
			FOR kk = 0D, n_elements(stp2_a_full) - 1 DO BEGIN
			IF (stp2_a_full[kk] NE 0) AND (FINITE(stp2_a_full[kk]) EQ 1) AND (FINITE(sfr2_a_full[kk]) EQ 1) THEN BEGIN
				start2_a_full[kk] = sfr2_a_full[kk]/stp2_a_full[kk]
			ENDIF ELSE BEGIN
				start2_a_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_a_full) - 1 DO BEGIN
			IF (sum_defl_stp2_a_full[mm] NE 0) AND (FINITE(sum_defl_stp2_a_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_a_full[mm] = sum_defl_sfr2_a_full[mm]/sum_defl_stp2_a_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

            ENDCASE
        ENDFOR
    END
    'B': BEGIN

	read_product, cdf_id, 'error1', error1_full_b, data1_index, epoch1maxrec
	remove_fill, -1, error1_full_b
	IF (TOTAL(error1_full_b, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'


        FOR ii = 0, n_elements(products)-1 DO BEGIN

            CASE products[ii] OF
                's_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_valid', s_valid_b_full, sum_defl_s_valid_b_full
                's_e_not_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_not_req', s_e_not_req_b_full, sum_defl_s_e_not_req_b_full
		's_e_req': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_e_req', s_e_req_b_full, sum_defl_s_e_req_b_full
		's_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_pos', s_no_pos_b_full, sum_defl_s_no_pos_b_full
		's_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_pos', s_mult_pos_b_full, sum_defl_s_mult_pos_b_full
		's_no_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_no_e', s_no_e_b_full, sum_defl_s_no_e_b_full
		's_mult_e': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  's_mult_e', s_mult_e_b_full, sum_defl_s_mult_e_b_full
		'ra_sat_a': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_a', ra_sat_a_b_full, sum_defl_ra_sat_a_b_full
		'ra_sat_b': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_b', ra_sat_b_b_full, sum_defl_ra_sat_b_b_full
		'ra_sat_both': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_sat_both', ra_sat_both_b_full, sum_defl_ra_sat_both_b_full
		'ssd_sw': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_sw', ssd_sw_b_full, sum_defl_ssd_sw_b_full
		'ssd_st': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ssd_st', ssd_st_b_full, sum_defl_ssd_st_b_full
 		'sf0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf0', sf0_b_full, sum_defl_sf0_b_full
		'sfr0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr0', sfr0_b_full, sum_defl_sfr0_b_full
		'stp0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp0', stp0_b_full, sum_defl_stp0_b_full
		'ra_trig': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'ra_trig', ra_trig_b_full, sum_defl_ra_trig_b_full
		'pos1_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_0', pos1_0_b_full, sum_defl_pos1_0_b_full
		'pos1_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos1_1', pos1_1_b_full, sum_defl_pos1_1_b_full
		'w_no_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_no_pos', w_no_pos_b_full, sum_defl_w_no_pos_b_full
		'w_mult_pos': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_mult_pos', w_mult_pos_b_full, sum_defl_w_mult_pos_b_full
		'w_valid': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'w_valid', w_valid_b_full, sum_defl_w_valid_b_full
		'sf2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sf2', sf2_b_full, sum_defl_sf2_b_full
		'sfr2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'sfr2', sfr2_b_full, sum_defl_sfr2_b_full
		'stp2':	read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'stp2', stp2_b_full, sum_defl_stp2_b_full
		'pos2_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_0', pos2_0_b_full, sum_defl_pos2_0_b_full
		'pos2_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_1', pos2_1_b_full, sum_defl_pos2_1_b_full
		'pos2_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_2', pos2_2_b_full, sum_defl_pos2_2_b_full
		'pos2_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos2_3', pos2_3_b_full, sum_defl_pos2_3_b_full
		'pos3_0': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_0', pos3_0_b_full, sum_defl_pos3_0_b_full
		'pos3_1': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_1', pos3_1_b_full, sum_defl_pos3_1_b_full
		'pos3_2': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_2', pos3_2_b_full, sum_defl_pos3_2_b_full
		'pos3_3': read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  'pos3_3', pos3_3_b_full, sum_defl_pos3_3_b_full
		'pos2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_0', pos2_0_b_full, sum_defl_pos2_0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_1', pos2_1_b_full,  sum_defl_pos2_1_b_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos2_2', pos2_2_b_full,  sum_defl_pos2_2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos2_3', pos2_3_b_full, sum_defl_pos2_3_b_full
			
			FOR kk = 0D, n_elements(pos2_0_b_full) - 1 DO BEGIN
				pos2_b_full[kk] = pos2_0_b_full[kk]+pos2_1_b_full[kk]+pos2_2_b_full[kk]+pos2_3_b_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_b_full) - 1 DO BEGIN
				sum_defl_pos2_b_full[mm] = sum_defl_pos2_0_b_full[mm]+sum_defl_pos2_1_b_full[mm]+sum_defl_pos2_2_b_full[mm]+sum_defl_pos2_3_b_full[mm]
			ENDFOR
			
			END
		'pos3':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_0', pos3_0_b_full,  sum_defl_pos3_0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_1', pos3_1_b_full,  sum_defl_pos3_1_b_full

			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'pos3_2', pos3_2_b_full,  sum_defl_pos3_2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'pos3_3', pos3_3_b_full,  sum_defl_pos3_3_b_full
			
			FOR kk = 0D, n_elements(pos3_0_b_full) - 1 DO BEGIN
				pos3_b_full[kk] = pos3_0_b_full[kk]+pos3_1_b_full[kk]+pos3_2_b_full[kk]+pos3_3_b_full[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_b_full) - 1 DO BEGIN
				sum_defl_pos3_b_full[mm] = sum_defl_pos3_0_b_full[mm]+sum_defl_pos3_1_b_full[mm]+sum_defl_pos3_2_b_full[mm]+sum_defl_pos3_3_b_full[mm]
			ENDFOR
			
			END

		'stop0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf0', sf0_b_full, sum_defl_sf0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_b_full, sum_defl_sfr0_b_full
			
			FOR kk = 0D, n_elements(sf0_b_full) - 1 DO BEGIN
			IF (sf0_b_full[kk] NE 0) AND (FINITE(sf0_b_full[kk]) EQ 1) AND (FINITE(sfr0_b_full[kk]) EQ 1) THEN BEGIN
				stop0_b_full[kk] = sfr0_b_full[kk]/sf0_b_full[kk]
			ENDIF ELSE BEGIN
				stop0_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_b_full) - 1 DO BEGIN
			IF (sum_defl_sf0_b_full[mm] NE 0) AND (FINITE(sum_defl_sf0_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_b_full[mm] = sum_defl_sfr0_b_full[mm]/sum_defl_sf0_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'start0':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp0', stp0_b_full, sum_defl_stp0_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr0', sfr0_b_full, sum_defl_sfr0_b_full
			
			FOR kk = 0D, n_elements(stp0_b_full) - 1 DO BEGIN
			IF (stp0_b_full[kk] NE 0) AND (FINITE(stp0_b_full[kk]) EQ 1) AND (FINITE(sfr0_b_full[kk]) EQ 1) THEN BEGIN
				start0_b_full[kk] = sfr0_b_full[kk]/stp0_b_full[kk]
			ENDIF ELSE BEGIN
				start0_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_b_full) - 1 DO BEGIN
			IF (sum_defl_stp0_b_full[mm] NE 0) AND (FINITE(sum_defl_stp0_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_b_full[mm] = sum_defl_sfr0_b_full[mm]/sum_defl_stp0_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'stop2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'sf2', sf2_b_full, sum_defl_sf2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_b_full, sum_defl_sfr2_b_full
			
			FOR kk = 0D, n_elements(sf2_b_full) - 1 DO BEGIN
			IF (sf2_b_full[kk] NE 0) AND (FINITE(sf2_b_full[kk]) EQ 1) AND (FINITE(sfr2_b_full[kk]) EQ 1) THEN BEGIN
				stop2_b_full[kk] = sfr2_b_full[kk]/sf2_b_full[kk]
			ENDIF ELSE BEGIN
				stop2_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_b_full) - 1 DO BEGIN
			IF (sum_defl_sf2_b_full[mm] NE 0) AND (FINITE(sum_defl_sf2_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_b_full[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_sf2_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

		'start2':  BEGIN
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                  		'stp2', stp2_b_full, sum_defl_stp2_b_full
			
			read_one_mon_full, cdf_id, data1_index, epoch1maxrec, num_records1, E_step, $
                 		'sfr2', sfr2_b_full, sum_defl_sfr2_b_full
			
			FOR kk = 0D, n_elements(stp2_b_full) - 1 DO BEGIN
			IF (stp2_b_full[kk] NE 0) AND (FINITE(stp2_b_full[kk]) EQ 1) AND (FINITE(sfr2_b_full[kk]) EQ 1) THEN BEGIN
				start2_b_full[kk] = sfr2_b_full[kk]/stp2_b_full[kk]
			ENDIF ELSE BEGIN
				start2_b_full[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_b_full) - 1 DO BEGIN
			IF (sum_defl_stp2_b_full[mm] NE 0) AND (FINITE(sum_defl_stp2_b_full[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_full[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_b_full[mm] = sum_defl_sfr2_b_full[mm]/sum_defl_stp2_b_full[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_full[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

            ENDCASE
        ENDFOR
	END
    ENDCASE
    mon_full_read = 1
END 


 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, $
	num_records1, num_records5, E_step, prod_name, sci_values, eng_values, sum_sci, sum_eng
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; check to see if monitor rate record has data (both scientific and engineering modes)

COMMON com_mon

;print,'read_one_mon called'

	temp_name = prod_name+'_norm_science'

	read_product, cdf_id, temp_name, sci_values,  data5_index, epoch5maxrec

	sum_sci=DBLARR(num_records5,32)
	sum_sci[*, *] = !values.d_nan

	sum_eng = DBLARR(num_records1,32)
	sum_eng[*, *] = !values.d_nan

	remove_fill, -1, sci_values
	temp = where(finite(sci_values) EQ 1,  count)

	IF count GT 0 THEN BEGIN 
	    FOR j = 0, num_records5-1 DO BEGIN
        	FOR k = 0, 31 DO BEGIN	
            	sum_sci[j, k] = TOTAL(sci_values(k, *, j), /NAN)
       		ENDFOR
   	 ENDFOR
	ENDIF

	temp_name = prod_name+'_norm_eng'
	read_product, cdf_id, temp_name, eng_values, data1_index, epoch1maxrec
        
	remove_fill, -1, eng_values
	temp = where(finite(eng_values) EQ 1,  count_2)

	IF count_2 GT 0 THEN BEGIN 
   	 FOR j = 0L, num_records1-1 DO BEGIN
       	 	FOR k = 0, 31 DO BEGIN	
         	   sum_eng[j, k] = TOTAL(eng_values(k, *, j), /NAN)
       		ENDFOR
   	ENDFOR
	ENDIF	

END
;-------------------------------------------------------------------------

;-------------------------------------------------------------------------------------------------------------
;-------------------------------------------------------------------------------------------------------------
PRO read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, prod_name, sci_values, sum_sci
;--------------------------------------------------------------------------------------------------------------
; check to see if monitor rate record has data (science mode only)


COMMON com_mon

;print,'read_one_mon_no_eng called'

temp_name = prod_name+'_norm_science'

read_product, cdf_id, temp_name, sci_values,  data5_index, epoch5maxrec

sum_sci=DBLARR(num_records5,32)
sum_sci[*, *] = !values.d_nan

remove_fill, -1, sci_values
temp = where(finite(sci_values) EQ 1,  count)

IF count GT 0 THEN BEGIN 
    FOR j = 0, num_records5-1 DO BEGIN
        FOR k = 0, 31 DO BEGIN	
            sum_sci[j, k] = TOTAL(sci_values(k, *, j), /NAN)
        ENDFOR
    ENDFOR
ENDIF

END 
;----------------------------------------------------------------------------

;------------------------------------------------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------------------------------
PRO read_data, cdf_id, sat, data_read, data5_index, epoch5maxrec, data1_index, epoch1maxrec, products, num_records1, num_records5, E_step, NO_ENG
;------------------------------------------------------------------------------------------------------------------------------------------------
; read the normal resolution monitor rate data


COMMON com_mon

IF NO_ENG EQ 0 THEN BEGIN

CASE sat OF 
    'A': BEGIN
	read_product, cdf_id, 'error5_mon', error5_mon_a, data5_index, epoch5maxrec
	remove_fill, -1, error5_mon_a
	IF (TOTAL(error5_mon_a, /NAN) GT 0) THEN print, 'ERROR5_MON FLAG SET'
	read_product, cdf_id, 'error1', error1_a, data1_index, epoch1maxrec
	remove_fill, -1, error1_a
	IF (TOTAL(error1_a, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'


        FOR ii = 0, n_elements(products)-1 DO BEGIN
		print, 'reading data:',products(ii)
            CASE products[ii] OF
                's_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_valid', s_valid_a_sci, s_valid_a_eng, sum_defl_s_valid_a_sci, sum_defl_s_valid_a_eng
                's_e_not_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_not_req', s_e_not_req_a_sci, s_e_not_req_a_eng, sum_defl_s_e_not_req_a_sci, sum_defl_s_e_not_req_a_eng	
		's_e_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_req', s_e_req_a_sci, s_e_req_a_eng, sum_defl_s_e_req_a_sci, sum_defl_s_e_req_a_eng	
		's_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_pos', s_no_pos_a_sci, s_no_pos_a_eng, sum_defl_s_no_pos_a_sci, sum_defl_s_no_pos_a_eng	
		's_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_pos', s_mult_pos_a_sci, s_mult_pos_a_eng, sum_defl_s_mult_pos_a_sci, sum_defl_s_mult_pos_a_eng	
		's_no_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_e', s_no_e_a_sci, s_no_e_a_eng, sum_defl_s_no_e_a_sci, sum_defl_s_no_e_a_eng	
		's_mult_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_e', s_mult_e_a_sci, s_mult_e_a_eng, sum_defl_s_mult_e_a_sci, sum_defl_s_mult_e_a_eng	
		'ra_sat_a': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_a', ra_sat_a_a_sci, ra_sat_a_a_eng, sum_defl_ra_sat_a_a_sci, sum_defl_ra_sat_a_a_eng	
		'ra_sat_b': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_b', ra_sat_b_a_sci, ra_sat_b_a_eng, sum_defl_ra_sat_b_a_sci, sum_defl_ra_sat_b_a_eng	
		'ra_sat_both': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_both', ra_sat_both_a_sci, ra_sat_both_a_eng, sum_defl_ra_sat_both_a_sci, sum_defl_ra_sat_both_a_eng	
		'ssd_sw': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_sw', ssd_sw_a_sci, ssd_sw_a_eng, sum_defl_ssd_sw_a_sci, sum_defl_ssd_sw_a_eng	
		'ssd_st': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_st', ssd_st_a_sci, ssd_st_a_eng, sum_defl_ssd_st_a_sci, sum_defl_ssd_st_a_eng	
 		'sf0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf0', sf0_a_sci, sf0_a_eng, sum_defl_sf0_a_sci, sum_defl_sf0_a_eng	
		'sfr0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr0', sfr0_a_sci, sfr0_a_eng, sum_defl_sfr0_a_sci, sum_defl_sfr0_a_eng	
		'stp0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp0', stp0_a_sci, stp0_a_eng, sum_defl_stp0_a_sci, sum_defl_stp0_a_eng	
		'ra_trig': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_trig', ra_trig_a_sci, ra_trig_a_eng, sum_defl_ra_trig_a_sci, sum_defl_ra_trig_a_eng	
		'pos1_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_0', pos1_0_a_sci, pos1_0_a_eng, sum_defl_pos1_0_a_sci, sum_defl_pos1_0_a_eng	
		'pos1_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_1', pos1_1_a_sci, pos1_1_a_eng, sum_defl_pos1_1_a_sci, sum_defl_pos1_1_a_eng	
		'w_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_no_pos', w_no_pos_a_sci, w_no_pos_a_eng, sum_defl_w_no_pos_a_sci, sum_defl_w_no_pos_a_eng	
		'w_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_mult_pos', w_mult_pos_a_sci, w_mult_pos_a_eng, sum_defl_w_mult_pos_a_sci, sum_defl_w_mult_pos_a_eng	
		'w_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_valid', w_valid_a_sci, w_valid_a_eng, sum_defl_w_valid_a_sci, sum_defl_w_valid_a_eng	
		'sf2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf2', sf2_a_sci, sf2_a_eng, sum_defl_sf2_a_sci, sum_defl_sf2_a_eng	
		'sfr2':	read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr2', sfr2_a_sci, sfr2_a_eng, sum_defl_sfr2_a_sci, sum_defl_sfr2_a_eng
		'stp2':	read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp2', stp2_a_sci, stp2_a_eng, sum_defl_stp2_a_sci, sum_defl_stp2_a_eng
		'pos2_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_0', pos2_0_a_sci, pos2_0_a_eng, sum_defl_pos2_0_a_sci, sum_defl_pos2_0_a_eng
		'pos2_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_1', pos2_1_a_sci, pos2_1_a_eng, sum_defl_pos2_1_a_sci, sum_defl_pos2_1_a_eng
		'pos2_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_2', pos2_2_a_sci, pos2_2_a_eng, sum_defl_pos2_2_a_sci, sum_defl_pos2_2_a_eng
		'pos2_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_3', pos2_3_a_sci, pos2_3_a_eng, sum_defl_pos2_3_a_sci, sum_defl_pos2_3_a_eng	
		'pos3_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_0', pos3_0_a_sci, pos3_0_a_eng, sum_defl_pos3_0_a_sci, sum_defl_pos3_0_a_eng	
		'pos3_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_1', pos3_1_a_sci, pos3_1_a_eng, sum_defl_pos3_1_a_sci, sum_defl_pos3_1_a_eng	
		'pos3_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_2', pos3_2_a_sci, pos3_2_a_eng, sum_defl_pos3_2_a_sci, sum_defl_pos3_2_a_eng	
		'pos3_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_3', pos3_3_a_sci, pos3_3_a_eng, sum_defl_pos3_3_a_sci, sum_defl_pos3_3_a_eng	
			'pos1':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos1_0', pos1_0_a_sci, pos1_0_a_eng, sum_defl_pos1_0_a_sci, sum_defl_pos1_0_a_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos1_1', pos1_1_a_sci, pos1_1_a_eng, sum_defl_pos1_1_a_sci, sum_defl_pos1_1_a_eng

			FOR kk = 0D, n_elements(pos1_0_a_sci) - 1 DO BEGIN
				pos1_a_sci[kk] = pos1_0_a_sci[kk]+pos1_1_a_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos1_0_a_sci) - 1 DO BEGIN
				sum_defl_pos1_a_sci[mm] = sum_defl_pos1_0_a_sci[mm]+sum_defl_pos1_1_a_sci[mm]
			ENDFOR
			
			END
		'pos2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos2_0', pos2_0_a_sci, pos2_0_a_eng, sum_defl_pos2_0_a_sci, sum_defl_pos2_0_a_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos2_1', pos2_1_a_sci, pos2_1_a_eng, sum_defl_pos2_1_a_sci, sum_defl_pos2_1_a_eng

			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos2_2', pos2_2_a_sci, pos2_2_a_eng, sum_defl_pos2_2_a_sci, sum_defl_pos2_2_a_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos2_3', pos2_3_a_sci, pos2_3_a_eng, sum_defl_pos2_3_a_sci, sum_defl_pos2_2_a_eng
			
			FOR kk = 0D, n_elements(pos2_0_a_sci) - 1 DO BEGIN
				pos2_a_sci[kk] = pos2_0_a_sci[kk]+pos2_1_a_sci[kk]+pos2_2_a_sci[kk]+pos2_3_a_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_a_sci) - 1 DO BEGIN
				sum_defl_pos2_a_sci[mm] = sum_defl_pos2_0_a_sci[mm]+sum_defl_pos2_1_a_sci[mm]+sum_defl_pos2_2_a_sci[mm]+sum_defl_pos2_3_a_sci[mm]
			ENDFOR
			
			END
		'pos3':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos3_0', pos3_0_a_sci, pos3_0_a_eng, sum_defl_pos3_0_a_sci, sum_defl_pos3_0_a_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos3_1', pos3_1_a_sci, pos3_1_a_eng, sum_defl_pos3_1_a_sci, sum_defl_pos3_1_a_eng

			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos3_2', pos3_2_a_sci, pos3_2_a_eng, sum_defl_pos3_2_a_sci, sum_defl_pos3_2_a_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos3_3', pos3_3_a_sci, pos3_3_a_eng, sum_defl_pos3_3_a_sci, sum_defl_pos3_3_a_eng
			
			FOR kk = 0D, n_elements(pos3_0_a_sci) - 1 DO BEGIN
				pos3_a_sci[kk] = pos3_0_a_sci[kk]+pos3_1_a_sci[kk]+pos3_2_a_sci[kk]+pos3_3_a_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_a_sci) - 1 DO BEGIN
				sum_defl_pos3_a_sci[mm] = sum_defl_pos3_0_a_sci[mm]+sum_defl_pos3_1_a_sci[mm]+sum_defl_pos3_2_a_sci[mm]+sum_defl_pos3_3_a_sci[mm]
			ENDFOR
			
			END
	

		'stop0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf0', sf0_a_sci, sf0_a_eng, sum_defl_sf0_a_sci, sum_defl_sf0_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_a_sci, sfr0_a_eng, sum_defl_sfr0_a_sci, sum_defl_sfr0_a_eng	
			
			FOR kk = 0D, n_elements(sf0_a_sci) - 1 DO BEGIN
			IF (sf0_a_sci[kk] NE 0) AND (FINITE(sf0_a_sci[kk]) EQ 1) AND (FINITE(sfr0_a_sci[kk]) EQ 1) THEN BEGIN
				stop0_a_sci[kk] = sfr0_a_sci[kk]/sf0_a_sci[kk]
			ENDIF ELSE BEGIN
				stop0_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf0_a_eng) - 1 DO BEGIN
			IF (sf0_a_eng[ll] NE 0) AND (FINITE(sf0_a_eng[ll]) EQ 1) AND (FINITE(sfr0_a_eng[ll]) EQ 1) THEN BEGIN
				stop0_a_eng[ll] = sfr0_a_eng[ll]/sf0_a_eng[ll]
			ENDIF ELSE BEGIN
				stop0_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_a_sci) - 1 DO BEGIN
			IF (sum_defl_sf0_a_sci[mm] NE 0) AND (FINITE(sum_defl_sf0_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_a_sci[mm] = sum_defl_sfr0_a_sci[mm]/sum_defl_sf0_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf0_a_eng) - 1 DO BEGIN
			IF (sum_defl_sf0_a_eng[nn] NE 0) AND (FINITE(sum_defl_sf0_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop0_a_eng[nn] = sum_defl_sfr0_a_eng[nn]/sum_defl_sf0_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'start0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp0', stp0_a_sci, stp0_a_eng, sum_defl_stp0_a_sci, sum_defl_stp0_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_a_sci, sfr0_a_eng, sum_defl_sfr0_a_sci, sum_defl_sfr0_a_eng	
			
			FOR kk = 0D, n_elements(stp0_a_sci) - 1 DO BEGIN
			IF (stp0_a_sci[kk] NE 0) AND (FINITE(stp0_a_sci[kk]) EQ 1) AND (FINITE(sfr0_a_sci[kk]) EQ 1) THEN BEGIN
				start0_a_sci[kk] = sfr0_a_sci[kk]/stp0_a_sci[kk]
			ENDIF ELSE BEGIN
				start0_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp0_a_eng) - 1 DO BEGIN
			IF (stp0_a_eng[ll] NE 0) AND (FINITE(stp0_a_eng[ll]) EQ 1) AND (FINITE(sfr0_a_eng[ll]) EQ 1) THEN BEGIN
				start0_a_eng[ll] = sfr0_a_eng[ll]/stp0_a_eng[ll]
			ENDIF ELSE BEGIN
				start0_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_a_sci) - 1 DO BEGIN
			IF (sum_defl_stp0_a_sci[mm] NE 0) AND (FINITE(sum_defl_stp0_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_a_sci[mm] = sum_defl_sfr0_a_sci[mm]/sum_defl_stp0_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp0_a_eng) - 1 DO BEGIN
			IF (sum_defl_stp0_a_eng[nn] NE 0) AND (FINITE(sum_defl_stp0_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start0_a_eng[nn] = sum_defl_sfr0_a_eng[nn]/sum_defl_stp0_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'stop2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf2', sf2_a_sci, sf2_a_eng, sum_defl_sf2_a_sci, sum_defl_sf2_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_a_sci, sfr2_a_eng, sum_defl_sfr2_a_sci, sum_defl_sfr2_a_eng	
			
			FOR kk = 0D, n_elements(sf2_a_sci) - 1 DO BEGIN
			IF (sf2_a_sci[kk] NE 0) AND (FINITE(sf2_a_sci[kk]) EQ 1) AND (FINITE(sfr2_a_sci[kk]) EQ 1) THEN BEGIN
				stop2_a_sci[kk] = sfr2_a_sci[kk]/sf2_a_sci[kk]
			ENDIF ELSE BEGIN
				stop2_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf2_a_eng) - 1 DO BEGIN
			IF (sf2_a_eng[ll] NE 0) AND (FINITE(sf2_a_eng[ll]) EQ 1) AND (FINITE(sfr2_a_eng[ll]) EQ 1) THEN BEGIN
				stop2_a_eng[ll] = sfr2_a_eng[ll]/sf2_a_eng[ll]
			ENDIF ELSE BEGIN
				stop2_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_a_sci) - 1 DO BEGIN
			IF (sum_defl_sf2_a_sci[mm] NE 0) AND (FINITE(sum_defl_sf2_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_a_sci[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_sf2_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf2_a_eng) - 1 DO BEGIN
			IF (sum_defl_sf2_a_eng[nn] NE 0) AND (FINITE(sum_defl_sf2_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop2_a_eng[nn] = sum_defl_sfr2_a_eng[nn]/sum_defl_sf2_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR

			END

		'start2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp2', stp2_a_sci, stp2_a_eng, sum_defl_stp2_a_sci, sum_defl_stp2_a_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_a_sci, sfr2_a_eng, sum_defl_sfr2_a_sci, sum_defl_sfr2_a_eng	
			
			FOR kk = 0D, n_elements(stp2_a_sci) - 1 DO BEGIN
			IF (stp2_a_sci[kk] NE 0) AND (FINITE(stp2_a_sci[kk]) EQ 1) AND (FINITE(sfr2_a_sci[kk]) EQ 1) THEN BEGIN
				start2_a_sci[kk] = sfr2_a_sci[kk]/stp2_a_sci[kk]
			ENDIF ELSE BEGIN
				start2_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp2_a_eng) - 1 DO BEGIN
			IF (stp2_a_eng[ll] NE 0) AND (FINITE(stp2_a_eng[ll]) EQ 1) AND (FINITE(sfr2_a_eng[ll]) EQ 1) THEN BEGIN
				start2_a_eng[ll] = sfr2_a_eng[ll]/stp2_a_eng[ll]
			ENDIF ELSE BEGIN
				start2_a_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_a_sci) - 1 DO BEGIN
			IF (sum_defl_stp2_a_sci[mm] NE 0) AND (FINITE(sum_defl_stp2_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_a_sci[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_stp2_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp2_a_eng) - 1 DO BEGIN
			IF (sum_defl_stp2_a_eng[nn] NE 0) AND (FINITE(sum_defl_stp2_a_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_a_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start2_a_eng[nn] = sum_defl_sfr2_a_eng[nn]/sum_defl_stp2_a_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'sf0ra_trig':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf0', sf0_a_sci, sf0_a_eng, sum_defl_sf0_a_sci, sum_defl_sf0_a_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'ra_trig', ra_trig_a_sci, ra_trig_a_eng, sum_defl_ra_trig_a_sci, sum_defl_ra_trig_a_eng
			
			FOR kk = 0D, n_elements(sf0_a_sci) - 1 DO BEGIN
			IF (ra_trig_a_sci[kk] NE 0) AND (FINITE(ra_trig_a_sci[kk]) EQ 1) AND (FINITE(sf0_a_sci[kk]) EQ 1) THEN BEGIN
				sf0ra_trig_a_sci[kk] = sf0_a_sci[kk]/ra_trig_a_sci[kk]
			ENDIF ELSE BEGIN
				sf0ra_trig_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR mm=0D, n_elements(sum_defl_sf0_a_sci) - 1 DO BEGIN
			IF (sum_defl_ra_trig_a_sci[mm] NE 0) AND (FINITE(sum_defl_ra_trig_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sf0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_sf0ra_trig_a_sci[mm] = sum_defl_sf0_a_sci[mm]/sum_defl_ra_trig_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_sf0ra_trig_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

            ENDCASE
        ENDFOR
    END
    'B': BEGIN

	read_product, cdf_id, 'error5_mon', error5_mon_b, data5_index, epoch5maxrec
	remove_fill, -1, error5_mon_b
	IF (TOTAL(error5_mon_b, /NAN) GT 0) THEN print, 'ERROR5_MON FLAG SET'
	read_product, cdf_id, 'error1', error1_b, data1_index, epoch5maxrec
	remove_fill, -1, error1_b
	IF (TOTAL(error1_b, /NAN) GT 0) THEN print, 'ERROR1 FLAG SET'

        FOR ii = 0, n_elements(products)-1 DO BEGIN 
		print, 'reading data:',products(ii)
            CASE products[ii] OF
                's_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_valid', s_valid_b_sci, s_valid_b_eng, sum_defl_s_valid_b_sci, sum_defl_s_valid_b_eng
		's_e_not_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_not_req', s_e_not_req_b_sci, s_e_not_req_b_eng, sum_defl_s_e_not_req_b_sci, sum_defl_s_e_not_req_b_eng	
		's_e_req': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_e_req', s_e_req_b_sci, s_e_req_b_eng, sum_defl_s_e_req_b_sci, sum_defl_s_e_req_b_eng	
		's_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_pos', s_no_pos_b_sci, s_no_pos_b_eng, sum_defl_s_no_pos_b_sci, sum_defl_s_no_pos_b_eng	
		's_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_pos', s_mult_pos_b_sci, s_mult_pos_b_eng, sum_defl_s_mult_pos_b_sci, sum_defl_s_mult_pos_b_eng	
		's_no_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_no_e', s_no_e_b_sci, s_no_e_b_eng, sum_defl_s_no_e_b_sci, sum_defl_s_no_e_b_eng	
		's_mult_e': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  's_mult_e', s_mult_e_b_sci, s_mult_e_b_eng, sum_defl_s_mult_e_b_sci, sum_defl_s_mult_e_b_eng	
		'ra_sat_a': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_a', ra_sat_a_b_sci, ra_sat_a_b_eng, sum_defl_ra_sat_a_b_sci, sum_defl_ra_sat_a_b_eng	
 		'ra_sat_b': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_b', ra_sat_b_b_sci, ra_sat_b_b_eng, sum_defl_ra_sat_b_b_sci, sum_defl_ra_sat_b_b_eng	
		'ra_sat_both': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_sat_both', ra_sat_both_b_sci, ra_sat_both_b_eng, sum_defl_ra_sat_both_b_sci, sum_defl_ra_sat_both_b_eng	
		'ssd_sw': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_sw', ssd_sw_b_sci, ssd_sw_b_eng, sum_defl_ssd_sw_b_sci, sum_defl_ssd_sw_b_eng	
		'ssd_st': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ssd_st', ssd_st_b_sci, ssd_st_b_eng, sum_defl_ssd_st_b_sci, sum_defl_ssd_st_b_eng	
		'sf0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf0', sf0_b_sci, sf0_b_eng, sum_defl_sf0_b_sci, sum_defl_sf0_b_eng	
		'sfr0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr0', sfr0_b_sci, sfr0_b_eng, sum_defl_sfr0_b_sci, sum_defl_sfr0_b_eng	
		'stp0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp0', stp0_b_sci, stp0_b_eng, sum_defl_stp0_b_sci, sum_defl_stp0_b_eng	
 		'ra_trig': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'ra_trig', ra_trig_b_sci, ra_trig_b_eng, sum_defl_ra_trig_b_sci, sum_defl_ra_trig_b_eng	
		'pos1_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_0', pos1_0_b_sci, pos1_0_b_eng, sum_defl_pos1_0_b_sci, sum_defl_pos1_0_b_eng	
		'pos1_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos1_1', pos1_1_b_sci, pos1_1_b_eng, sum_defl_pos1_1_b_sci, sum_defl_pos1_1_b_eng	
		'w_no_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_no_pos', w_no_pos_b_sci, w_no_pos_b_eng, sum_defl_w_no_pos_b_sci, sum_defl_w_no_pos_b_eng	
		'w_mult_pos': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_mult_pos', w_mult_pos_b_sci, w_mult_pos_b_eng, sum_defl_w_mult_pos_b_sci, sum_defl_w_mult_pos_b_eng	
		'w_valid': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'w_valid', w_valid_b_sci, w_valid_b_eng, sum_defl_w_valid_b_sci, sum_defl_w_valid_b_eng	
		'sf2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sf2', sf2_b_sci, sf2_b_eng, sum_defl_sf2_b_sci, sum_defl_sf2_b_eng	
		'sfr2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'sfr2', sfr2_b_sci, sfr2_b_eng, sum_defl_sfr2_b_sci, sum_defl_sfr2_b_eng	
		'stp2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'stp2', stp2_b_sci, stp2_b_eng, sum_defl_stp2_b_sci, sum_defl_stp2_b_eng	
		'pos2_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_0', pos2_0_b_sci, pos2_0_b_eng, sum_defl_pos2_0_b_sci, sum_defl_pos2_0_b_eng	
		'pos2_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_1', pos2_1_b_sci, pos2_1_b_eng, sum_defl_pos2_1_b_sci, sum_defl_pos2_1_b_eng	
		'pos2_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_2', pos2_2_b_sci, pos2_2_b_eng, sum_defl_pos2_2_b_sci, sum_defl_pos2_2_b_eng	
		'pos2_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos2_3', pos2_3_b_sci, pos2_3_b_eng, sum_defl_pos2_3_b_sci, sum_defl_pos2_3_b_eng	
		'pos3_0': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_0', pos3_0_b_sci, pos3_0_b_eng, sum_defl_pos3_0_b_sci, sum_defl_pos3_0_b_eng	
		'pos3_1': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_1', pos3_1_b_sci, pos3_1_b_eng, sum_defl_pos3_1_b_sci, sum_defl_pos3_1_b_eng	
		'pos3_2': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_2', pos3_2_b_sci, pos3_2_b_eng, sum_defl_pos3_2_b_sci, sum_defl_pos3_2_b_eng	
		'pos3_3': read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  'pos3_3', pos3_3_b_sci, pos3_3_b_eng, sum_defl_pos3_3_b_sci, sum_defl_pos3_3_b_eng	
		'pos1':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos1_0', pos1_0_b_sci, pos1_b_eng, sum_defl_pos1_0_b_sci, sum_defl_pos1_0_b_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos1_1', pos1_1_b_sci, pos1_1_b_eng, sum_defl_pos1_1_b_sci, sum_defl_pos1_1_b_eng
			
			FOR kk = 0D, n_elements(pos1_0_b_sci) - 1 DO BEGIN
				pos1_b_sci[kk] = pos1_0_b_sci[kk]+pos1_1_b_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos1_0_b_sci) - 1 DO BEGIN
				sum_defl_pos1_b_sci[mm] = sum_defl_pos1_0_b_sci[mm]+sum_defl_pos1_1_b_sci[mm]
			ENDFOR
			
			END
		'pos2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos2_0', pos2_0_b_sci, pos2_0_b_eng, sum_defl_pos2_0_b_sci, sum_defl_pos2_0_b_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos2_1', pos2_1_b_sci, pos2_1_b_eng, sum_defl_pos2_1_b_sci, sum_defl_pos2_1_b_eng

			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos2_2', pos2_2_b_sci, pos2_2_b_eng, sum_defl_pos2_2_b_sci, sum_defl_pos2_2_b_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos2_3', pos2_3_b_sci, pos2_3_b_eng, sum_defl_pos2_3_b_sci, sum_defl_pos2_3_b_eng
			
			FOR kk = 0D, n_elements(pos2_0_b_sci) - 1 DO BEGIN
				pos2_b_sci[kk] = pos2_0_b_sci[kk]+pos2_1_b_sci[kk]+pos2_2_b_sci[kk]+pos2_3_b_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_b_sci) - 1 DO BEGIN
				sum_defl_pos2_b_sci[mm] = sum_defl_pos2_0_b_sci[mm]+sum_defl_pos2_1_b_sci[mm]+sum_defl_pos2_2_b_sci[mm]+sum_defl_pos2_3_b_sci[mm]
			ENDFOR
			
			END

		'pos3':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos3_0', pos3_0_b_sci, pos3_0_b_eng, sum_defl_pos3_0_b_sci, sum_defl_pos3_0_b_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos3_1', pos3_1_b_sci, pos3_1_b_eng, sum_defl_pos3_1_b_sci, sum_defl_pos3_1_b_eng

			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'pos3_2', pos3_2_b_sci, pos3_2_b_eng, sum_defl_pos3_2_b_sci, sum_defl_pos3_2_b_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'pos3_3', pos3_3_b_sci, pos3_3_b_eng, sum_defl_pos3_3_b_sci, sum_defl_pos3_3_b_eng
			
			FOR kk = 0D, n_elements(pos3_0_b_sci) - 1 DO BEGIN
				pos3_b_sci[kk] = pos3_0_b_sci[kk]+pos3_1_b_sci[kk]+pos3_2_b_sci[kk]+pos3_3_b_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_b_sci) - 1 DO BEGIN
				sum_defl_pos3_b_sci[mm] = sum_defl_pos3_0_b_sci[mm]+sum_defl_pos3_1_b_sci[mm]+sum_defl_pos3_2_b_sci[mm]+sum_defl_pos3_3_b_sci[mm]
			ENDFOR
			
			END


		'stop0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf0', sf0_b_sci, sf0_b_eng, sum_defl_sf0_b_sci, sum_defl_sf0_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_b_sci, sfr0_b_eng, sum_defl_sfr0_b_sci, sum_defl_sfr0_b_eng	
			
			FOR kk = 0D, n_elements(sf0_b_sci) - 1 DO BEGIN
			IF (sf0_b_sci[kk] NE 0) AND (FINITE(sf0_b_sci[kk]) EQ 1) AND (FINITE(sfr0_b_sci[kk]) EQ 1) THEN BEGIN
				stop0_b_sci[kk] = sfr0_b_sci[kk]/sf0_b_sci[kk]
			ENDIF ELSE BEGIN
				stop0_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf0_b_eng) - 1 DO BEGIN
			IF (sf0_b_eng[ll] NE 0) AND (FINITE(sf0_b_eng[ll]) EQ 1) AND (FINITE(sfr0_b_eng[ll]) EQ 1) THEN BEGIN
				stop0_b_eng[ll] = sfr0_b_eng[ll]/sf0_b_eng[ll]
			ENDIF ELSE BEGIN
				stop0_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_b_sci) - 1 DO BEGIN
			IF (sum_defl_sf0_b_sci[mm] NE 0) AND (FINITE(sum_defl_sf0_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_b_sci[mm] = sum_defl_sfr0_b_sci[mm]/sum_defl_sf0_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf0_b_eng) - 1 DO BEGIN
			IF (sum_defl_sf0_b_eng[nn] NE 0) AND (FINITE(sum_defl_sf0_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop0_b_eng[nn] = sum_defl_sfr0_b_eng[nn]/sum_defl_sf0_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR

			END
		'start0':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp0', stp0_b_sci, stp0_b_eng, sum_defl_stp0_b_sci, sum_defl_stp0_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr0', sfr0_b_sci, sfr0_b_eng, sum_defl_sfr0_b_sci, sum_defl_sfr0_b_eng	
			
			FOR kk = 0D, n_elements(stp0_b_sci) - 1 DO BEGIN
			IF (stp0_b_sci[kk] NE 0) AND (FINITE(stp0_b_sci[kk]) EQ 1) AND (FINITE(sfr0_b_sci[kk]) EQ 1) THEN BEGIN
				start0_b_sci[kk] = sfr0_b_sci[kk]/stp0_b_sci[kk]
			ENDIF ELSE BEGIN
				start0_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp0_b_eng) - 1 DO BEGIN
			IF (stp0_b_eng[ll] NE 0) AND (FINITE(stp0_b_eng[ll]) EQ 1) AND (FINITE(sfr0_b_eng[ll]) EQ 1) THEN BEGIN
				start0_b_eng[ll] = sfr0_b_eng[ll]/stp0_b_eng[ll]
			ENDIF ELSE BEGIN
				start0_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_b_sci) - 1 DO BEGIN
			IF (sum_defl_stp0_b_sci[mm] NE 0) AND (FINITE(sum_defl_stp0_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_b_sci[mm] = sum_defl_sfr0_b_sci[mm]/sum_defl_stp0_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp0_b_eng) - 1 DO BEGIN
			IF (sum_defl_stp0_b_eng[nn] NE 0) AND (FINITE(sum_defl_stp0_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr0_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start0_b_eng[nn] = sum_defl_sfr0_b_eng[nn]/sum_defl_stp0_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'stop2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf2', sf2_b_sci, sf2_b_eng, sum_defl_sf2_b_sci, sum_defl_sf2_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_b_sci, sfr2_b_eng, sum_defl_sfr2_b_sci, sum_defl_sfr2_b_eng	
			
			FOR kk = 0D, n_elements(sf2_b_sci) - 1 DO BEGIN
			IF (sf2_b_sci[kk] NE 0) AND (FINITE(sf2_b_sci[kk]) EQ 1) AND (FINITE(sfr2_b_sci[kk]) EQ 1) THEN BEGIN
				stop2_b_sci[kk] = sfr2_b_sci[kk]/sf2_b_sci[kk]
			ENDIF ELSE BEGIN
				stop2_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(sf2_b_eng) - 1 DO BEGIN
			IF (sf2_b_eng[ll] NE 0) AND (FINITE(sf2_b_eng[ll]) EQ 1) AND (FINITE(sfr2_b_eng[ll]) EQ 1) THEN BEGIN
				stop2_b_eng[ll] = sfr2_b_eng[ll]/sf2_b_eng[ll]
			ENDIF ELSE BEGIN
				stop2_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_b_sci) - 1 DO BEGIN
			IF (sum_defl_sf2_b_sci[mm] NE 0) AND (FINITE(sum_defl_sf2_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_b_sci[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_sf2_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_sf2_b_eng) - 1 DO BEGIN
			IF (sum_defl_sf2_b_eng[nn] NE 0) AND (FINITE(sum_defl_sf2_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_stop2_b_eng[nn] = sum_defl_sfr2_b_eng[nn]/sum_defl_sf2_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR

			END

		'start2':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'stp2', stp2_b_sci, stp2_b_eng, sum_defl_stp2_b_sci, sum_defl_stp2_b_eng	
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'sfr2', sfr2_b_sci, sfr2_b_eng, sum_defl_sfr2_b_sci, sum_defl_sfr2_b_eng	
			
			FOR kk = 0D, n_elements(stp2_b_sci) - 1 DO BEGIN
			IF (stp2_b_sci[kk] NE 0) AND (FINITE(stp2_b_sci[kk]) EQ 1) AND (FINITE(sfr2_b_sci[kk]) EQ 1) THEN BEGIN
				start2_b_sci[kk] = sfr2_b_sci[kk]/stp2_b_sci[kk]
			ENDIF ELSE BEGIN
				start2_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR ll=0D, n_elements(stp2_b_eng) - 1 DO BEGIN
			IF (stp2_b_eng[ll] NE 0) AND (FINITE(stp2_b_eng[ll]) EQ 1) AND (FINITE(sfr2_b_eng[ll]) EQ 1) THEN BEGIN
				start2_b_eng[ll] = sfr2_b_eng[ll]/stp2_b_eng[ll]
			ENDIF ELSE BEGIN
				start2_b_eng[ll] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_b_sci) - 1 DO BEGIN
			IF (sum_defl_stp2_b_sci[mm] NE 0) AND (FINITE(sum_defl_stp2_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_b_sci[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_stp2_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR nn=0D, n_elements(sum_defl_stp2_b_eng) - 1 DO BEGIN
			IF (sum_defl_stp2_b_eng[nn] NE 0) AND (FINITE(sum_defl_stp2_b_eng[nn]) EQ 1) AND (FINITE(sum_defl_sfr2_b_eng[nn]) EQ 1)THEN BEGIN
				sum_defl_start2_b_eng[nn] = sum_defl_sfr2_b_eng[nn]/sum_defl_stp2_b_eng[nn]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_eng[nn] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'sf0ra_trig':  BEGIN
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                  		'sf0', sf0_b_sci, sf0_b_eng, sum_defl_sf0_b_sci, sum_defl_sf0_b_eng
			
			read_one_mon, cdf_id, data5_index, epoch5maxrec, data1_index, epoch1maxrec, num_records1, num_records5, E_step, $
                 		'ra_trig', ra_trig_b_sci, ra_trig_b_eng, sum_defl_ra_trig_b_sci, sum_defl_ra_trig_b_eng
			
			FOR kk = 0D, n_elements(sf0_b_sci) - 1 DO BEGIN
			IF (ra_trig_b_sci[kk] NE 0) AND (FINITE(ra_trig_b_sci[kk]) EQ 1) AND (FINITE(sf0_b_sci[kk]) EQ 1) THEN BEGIN
				sf0ra_trig_b_sci[kk] = sf0_b_sci[kk]/ra_trig_b_sci[kk]
			ENDIF ELSE BEGIN
				sf0ra_trig_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR mm=0D, n_elements(sum_defl_sf0_b_sci) - 1 DO BEGIN
			IF (sum_defl_ra_trig_b_sci[mm] NE 0) AND (FINITE(sum_defl_ra_trig_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sf0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_sf0ra_trig_b_sci[mm] = sum_defl_sf0_b_sci[mm]/sum_defl_ra_trig_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_sf0ra_trig_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

                ENDCASE
            ENDFOR
        END
    ENDCASE

ENDIF ELSE BEGIN

CASE sat OF 
    'A': BEGIN
	read_product, cdf_id, 'error5_mon', error5_mon_a, data5_index, epoch5maxrec
	remove_fill, -1, error5_mon_a
	IF (TOTAL(error5_mon_a, /NAN) GT 0) THEN print, 'ERROR5_MON FLAG SET'

        FOR ii = 0, n_elements(products)-1 DO BEGIN

            CASE products[ii] OF
                's_valid': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
		  's_valid', s_valid_a_sci, sum_defl_s_valid_a_sci
                's_e_not_req': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_e_not_req', s_e_not_req_a_sci, sum_defl_s_e_not_req_a_sci
		's_e_req': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_e_req', s_e_req_a_sci, sum_defl_s_e_req_a_sci
		's_no_pos': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_no_pos', s_no_pos_a_sci, sum_defl_s_no_pos_a_sci
		's_mult_pos': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_mult_pos', s_mult_pos_a_sci, sum_defl_s_mult_pos_a_sci
		's_no_e': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_no_e', s_no_e_a_sci, sum_defl_s_no_e_a_sci
		's_mult_e': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_mult_e', s_mult_e_a_sci, sum_defl_s_mult_e_a_sci
		'ra_sat_a': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_sat_a', ra_sat_a_a_sci, sum_defl_ra_sat_a_a_sci
		'ra_sat_b': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_sat_b', ra_sat_b_a_sci, sum_defl_ra_sat_b_a_sci
		'ra_sat_both': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_sat_both', ra_sat_both_a_sci, sum_defl_ra_sat_both_a_sci
		'ssd_sw': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ssd_sw', ssd_sw_a_sci, sum_defl_ssd_sw_a_sci
		'ssd_st': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ssd_st', ssd_st_a_sci, sum_defl_ssd_st_a_sci
 		'sf0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sf0', sf0_a_sci, sum_defl_sf0_a_sci
		'sfr0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sfr0', sfr0_a_sci, sum_defl_sfr0_a_sci
		'stp0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'stp0', stp0_a_sci, sum_defl_stp0_a_sci
		'ra_trig': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_trig', ra_trig_a_sci, sum_defl_ra_trig_a_sci
		'pos1_0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos1_0', pos1_0_a_sci, sum_defl_pos1_0_a_sci
		'pos1_1': read_one_mon_no_eng, cdf_id, data5_index, num_records5, E_step, $
                  'pos1_1', pos1_1_a_sci, sum_defl_pos1_1_a_sci
		'w_no_pos': read_one_mon_no_eng, cdf_id, data5_index, num_records5, E_step, $
                  'w_no_pos', w_no_pos_a_sci, sum_defl_w_no_pos_a_sci
		'w_mult_pos': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'w_mult_pos', w_mult_pos_a_sci, sum_defl_w_mult_pos_a_sci
		'w_valid': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'w_valid', w_valid_a_sci, sum_defl_w_valid_a_sci
		'sf2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sf2', sf2_a_sci, sum_defl_sf2_a_sci
		'sfr2':	read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sfr2', sfr2_a_sci, sum_defl_sfr2_a_sci
		'stp2':	read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'stp2', stp2_a_sci, sum_defl_stp2_a_sci
		'pos2_0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_0', pos2_0_a_sci, sum_defl_pos2_0_a_sci
		'pos2_1': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_1', pos2_1_a_sci, sum_defl_pos2_1_a_sci
		'pos2_2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_2', pos2_2_a_sci, sum_defl_pos2_2_a_sci
		'pos2_3': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_3', pos2_3_a_sci, sum_defl_pos2_3_a_sci
		'pos3_0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_0', pos3_0_a_sci, sum_defl_pos3_0_a_sci
		'pos3_1': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_1', pos3_1_a_sci, sum_defl_pos3_1_a_sci
		'pos3_2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_2', pos3_2_a_sci, sum_defl_pos3_2_a_sci
		'pos3_3': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_3', pos3_3_a_sci, sum_defl_pos3_3_a_sci
		'pos1':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos1_0', pos1_0_a_sci, sum_defl_pos1_0_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos1_1', pos1_1_a_sci, sum_defl_pos1_1_a_sci

			FOR kk = 0D, n_elements(pos1_0_a_sci) - 1 DO BEGIN
				pos1_a_sci[kk] = pos1_0_a_sci[kk]+pos1_1_a_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos1_0_a_sci) - 1 DO BEGIN
				sum_defl_pos1_a_sci[mm] = sum_defl_pos1_0_a_sci[mm]+sum_defl_pos1_1_a_sci[mm]
			ENDFOR
			
			END
		'pos2':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos2_0', pos2_0_a_sci, sum_defl_pos2_0_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos2_1', pos2_1_a_sci, sum_defl_pos2_1_a_sci

			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos2_2', pos2_2_a_sci, sum_defl_pos2_2_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos2_3', pos2_3_a_sci, sum_defl_pos2_3_a_sci
			
			FOR kk = 0D, n_elements(pos2_0_a_sci) - 1 DO BEGIN
				pos2_a_sci[kk] = pos2_0_a_sci[kk]+pos2_1_a_sci[kk]+pos2_2_a_sci[kk]+pos2_3_a_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_a_sci) - 1 DO BEGIN
				sum_defl_pos2_a_sci[mm] = sum_defl_pos2_0_a_sci[mm]+sum_defl_pos2_1_a_sci[mm]+sum_defl_pos2_2_a_sci[mm]+sum_defl_pos2_3_a_sci[mm]
			ENDFOR
			
			END
		'pos3':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos3_0', pos3_0_a_sci, sum_defl_pos3_0_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos3_1', pos3_1_a_sci, sum_defl_pos3_1_a_sci

			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos3_2', pos3_2_a_sci, sum_defl_pos3_2_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos3_3', pos3_3_a_sci, sum_defl_pos3_3_a_sci
			
			FOR kk = 0D, n_elements(pos3_0_a_sci) - 1 DO BEGIN
				pos3_a_sci[kk] = pos3_0_a_sci[kk]+pos3_1_a_sci[kk]+pos3_2_a_sci[kk]+pos3_3_a_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_a_sci) - 1 DO BEGIN
				sum_defl_pos3_a_sci[mm] = sum_defl_pos3_0_a_sci[mm]+sum_defl_pos3_1_a_sci[mm]+sum_defl_pos3_2_a_sci[mm]+sum_defl_pos3_3_a_sci[mm]
			ENDFOR
			
			END
	

	'stop0':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'sf0', sf0_a_sci, sum_defl_sf0_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr0', sfr0_a_sci, sum_defl_sfr0_a_sci
			
			FOR kk = 0D, n_elements(sf0_a_sci) - 1 DO BEGIN
			IF (sf0_a_sci[kk] NE 0) AND (FINITE(sf0_a_sci[kk]) EQ 1) AND (FINITE(sfr0_a_sci[kk]) EQ 1) THEN BEGIN
				stop0_a_sci[kk] = sfr0_a_sci[kk]/sf0_a_sci[kk]
			ENDIF ELSE BEGIN
				stop0_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_a_sci) - 1 DO BEGIN
			IF (sum_defl_sf0_a_sci[mm] NE 0) AND (FINITE(sum_defl_sf0_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_a_sci[mm] = sum_defl_sfr0_a_sci[mm]/sum_defl_sf0_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'start0':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'stp0', stp0_a_sci, sum_defl_stp0_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr0', sfr0_a_sci, sum_defl_sfr0_a_sci
			
			FOR kk = 0D, n_elements(stp0_a_sci) - 1 DO BEGIN
			IF (stp0_a_sci[kk] NE 0) AND (FINITE(stp0_a_sci[kk]) EQ 1) AND (FINITE(sfr0_a_sci[kk]) EQ 1) THEN BEGIN
				start0_a_sci[kk] = sfr0_a_sci[kk]/stp0_a_sci[kk]
			ENDIF ELSE BEGIN
				start0_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_a_sci) - 1 DO BEGIN
			IF (sum_defl_stp0_a_sci[mm] NE 0) AND (FINITE(sum_defl_stp0_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_a_sci[mm] = sum_defl_sfr0_a_sci[mm]/sum_defl_stp0_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			END
		'stop2':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'sf2', sf2_a_sci, sum_defl_sf2_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr2', sfr2_a_sci, sum_defl_sfr2_a_sci
			
			FOR kk = 0D, n_elements(sf2_a_sci) - 1 DO BEGIN
			IF (sf2_a_sci[kk] NE 0) AND (FINITE(sf2_a_sci[kk]) EQ 1) AND (FINITE(sfr2_a_sci[kk]) EQ 1) THEN BEGIN
				stop2_a_sci[kk] = sfr2_a_sci[kk]/sf2_a_sci[kk]
			ENDIF ELSE BEGIN
				stop2_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_a_sci) - 1 DO BEGIN
			IF (sum_defl_sf2_a_sci[mm] NE 0) AND (FINITE(sum_defl_sf2_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_a_sci[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_sf2_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

		'start2':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'stp2', stp2_a_sci, sum_defl_stp2_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr2', sfr2_a_sci, sum_defl_sfr2_a_sci
			
			FOR kk = 0D, n_elements(stp2_a_sci) - 1 DO BEGIN
			IF (stp2_a_sci[kk] NE 0) AND (FINITE(stp2_a_sci[kk]) EQ 1) AND (FINITE(sfr2_a_sci[kk]) EQ 1) THEN BEGIN
				start2_a_sci[kk] = sfr2_a_sci[kk]/stp2_a_sci[kk]
			ENDIF ELSE BEGIN
				start2_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_a_sci) - 1 DO BEGIN
			IF (sum_defl_stp2_a_sci[mm] NE 0) AND (FINITE(sum_defl_stp2_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_a_sci[mm] = sum_defl_sfr2_a_sci[mm]/sum_defl_stp2_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
	'sf0ra_trig':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'sf0', sf0_a_sci, sum_defl_sf0_a_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'ra_trig', ra_trig_a_sci, sum_defl_ra_trig_a_sci
			
			FOR kk = 0D, n_elements(sf0_a_sci) - 1 DO BEGIN
			IF (ra_trig_a_sci[kk] NE 0) AND (FINITE(ra_trig_a_sci[kk]) EQ 1) AND (FINITE(sf0_a_sci[kk]) EQ 1) THEN BEGIN
				sf0ra_trig_a_sci[kk] = sf0_a_sci[kk]/ra_trig_a_sci[kk]
			ENDIF ELSE BEGIN
				sf0ra_trig_a_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR mm=0D, n_elements(sum_defl_sf0_a_sci) - 1 DO BEGIN
			IF (sum_defl_ra_trig_a_sci[mm] NE 0) AND (FINITE(sum_defl_ra_trig_a_sci[mm]) EQ 1) AND (FINITE(sum_defl_sf0_a_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_sf0ra_trig_a_sci[mm] = sum_defl_sf0_a_sci[mm]/sum_defl_ra_trig_a_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_sf0ra_trig_a_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

            ENDCASE
        ENDFOR
    END
    'B': BEGIN

	read_product, cdf_id, 'error5_mon', error5_mon_b, data5_index, epoch5maxrec
	remove_fill, -1, error5_mon_b
	IF (TOTAL(error5_mon_b, /NAN) GT 0) THEN print, 'ERROR5_MON FLAG SET'

        FOR ii = 0, n_elements(products)-1 DO BEGIN 
            CASE products[ii] OF
                's_valid': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_valid', s_valid_b_sci, sum_defl_s_valid_b_sci
		's_e_not_req': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_e_not_req', s_e_not_req_b_sci, sum_defl_s_e_not_req_b_sci
		's_e_req': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_e_req', s_e_req_b_sci, sum_defl_s_e_req_b_sci
		's_no_pos': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_no_pos', s_no_pos_b_sci, sum_defl_s_no_pos_b_sci
		's_mult_pos': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_mult_pos', s_mult_pos_b_sci, sum_defl_s_mult_pos_b_sci
		's_no_e': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_no_e', s_no_e_b_sci, sum_defl_s_no_e_b_sci
		's_mult_e': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  's_mult_e', s_mult_e_b_sci, sum_defl_s_mult_e_b_sci
		'ra_sat_a': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_sat_a', ra_sat_a_b_sci, sum_defl_ra_sat_a_b_sci
 		'ra_sat_b': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_sat_b', ra_sat_b_b_sci, sum_defl_ra_sat_b_b_sci
		'ra_sat_both': read_one_mon_no_eng, cdf_id, data5_index, num_records5, E_step, $
                  'ra_sat_both', ra_sat_both_b_sci, sum_defl_ra_sat_both_b_sci
		'ssd_sw': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ssd_sw', ssd_sw_b_sci, sum_defl_ssd_sw_b_sci
		'ssd_st': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ssd_st', ssd_st_b_sci, sum_defl_ssd_st_b_sci
		'sf0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sf0', sf0_b_sci, sum_defl_sf0_b_sci
		'sfr0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sfr0', sfr0_b_sci, sum_defl_sfr0_b_sci
		'stp0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'stp0', stp0_b_sci, sum_defl_stp0_b_sci
 		'ra_trig': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'ra_trig', ra_trig_b_sci, sum_defl_ra_trig_b_sci
		'pos1_0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos1_0', pos1_0_b_sci, sum_defl_pos1_0_b_sci
		'pos1_1': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos1_1', pos1_1_b_sci, sum_defl_pos1_1_b_sci
		'w_no_pos': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'w_no_pos', w_no_pos_b_sci, sum_defl_w_no_pos_b_sci
		'w_mult_pos': read_one_mon_no_eng, cdf_id, data5_index, num_records5, E_step, $
                  'w_mult_pos', w_mult_pos_b_sci, sum_defl_w_mult_pos_b_sci
		'w_valid': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'w_valid', w_valid_b_sci, sum_defl_w_valid_b_sci
		'sf2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sf2', sf2_b_sci, sum_defl_sf2_b_sci
		'sfr2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'sfr2', sfr2_b_sci, sum_defl_sfr2_b_sci
		'stp2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'stp2', stp2_b_sci, sum_defl_stp2_b_sci
		'pos2_0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_0', pos2_0_b_sci, sum_defl_pos2_0_b_sci
		'pos2_1': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_1', pos2_1_b_sci, sum_defl_pos2_1_b_sci
		'pos2_2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_2', pos2_2_b_sci, sum_defl_pos2_2_b_sci
		'pos2_3': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos2_3', pos2_3_b_sci, sum_defl_pos2_3_b_sci
		'pos3_0': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_0', pos3_0_b_sci, sum_defl_pos3_0_b_sci
		'pos3_1': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_1', pos3_1_b_sci, sum_defl_pos3_1_b_sci
		'pos3_2': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_2', pos3_2_b_sci, sum_defl_pos3_2_b_sci
		'pos3_3': read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  'pos3_3', pos3_3_b_sci, sum_defl_pos3_3_b_sci
		'pos1':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos1_0', pos1_0_b_sci, sum_defl_pos1_0_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos1_1', pos1_1_b_sci, sum_defl_pos1_1_b_sci
			
			FOR kk = 0D, n_elements(pos1_0_b_sci) - 1 DO BEGIN
				pos1_b_sci[kk] = pos1_0_b_sci[kk]+pos1_1_b_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos1_0_b_sci) - 1 DO BEGIN
				sum_defl_pos1_b_sci[mm] = sum_defl_pos1_0_b_sci[mm]+sum_defl_pos1_1_b_sci[mm]
			ENDFOR
			
			END
		'pos2':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos2_0', pos2_0_b_sci, sum_defl_pos2_0_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos2_1', pos2_1_b_sci, sum_defl_pos2_1_b_sci

			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos2_2', pos2_2_b_sci, sum_defl_pos2_2_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos2_3', pos2_3_b_sci, sum_defl_pos2_3_b_sci
			
			FOR kk = 0D, n_elements(pos2_0_b_sci) - 1 DO BEGIN
				pos2_b_sci[kk] = pos2_0_b_sci[kk]+pos2_1_b_sci[kk]+pos2_2_b_sci[kk]+pos2_3_b_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos2_0_b_sci) - 1 DO BEGIN
				sum_defl_pos2_b_sci[mm] = sum_defl_pos2_0_b_sci[mm]+sum_defl_pos2_1_b_sci[mm]+sum_defl_pos2_2_b_sci[mm]+sum_defl_pos2_3_b_sci[mm]
			ENDFOR
			
			END

		'pos3':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos3_0', pos3_0_b_sci, sum_defl_pos3_0_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos3_1', pos3_1_b_sci, sum_defl_pos3_1_b_sci

			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'pos3_2', pos3_2_b_sci, sum_defl_pos3_2_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'pos3_3', pos3_3_b_sci, sum_defl_pos3_3_b_sci
			
			FOR kk = 0D, n_elements(pos3_0_b_sci) - 1 DO BEGIN
				pos3_b_sci[kk] = pos3_0_b_sci[kk]+pos3_1_b_sci[kk]+pos3_2_b_sci[kk]+pos3_3_b_sci[kk]
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_pos3_0_b_sci) - 1 DO BEGIN
				sum_defl_pos3_b_sci[mm] = sum_defl_pos3_0_b_sci[mm]+sum_defl_pos3_1_b_sci[mm]+sum_defl_pos3_2_b_sci[mm]+sum_defl_pos3_3_b_sci[mm]
			ENDFOR
			
			END

	'stop0':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'sf0', sf0_b_sci, sum_defl_sf0_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr0', sfr0_b_sci, sum_defl_sfr0_b_sci
			
			FOR kk = 0D, n_elements(sf0_b_sci) - 1 DO BEGIN
			IF (sf0_b_sci[kk] NE 0) AND (FINITE(sf0_b_sci[kk]) EQ 1) AND (FINITE(sfr0_b_sci[kk]) EQ 1) THEN BEGIN
				stop0_b_sci[kk] = sfr0_b_sci[kk]/sf0_b_sci[kk]
			ENDIF ELSE BEGIN
				stop0_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf0_b_sci) - 1 DO BEGIN
			IF (sum_defl_sf0_b_sci[mm] NE 0) AND (FINITE(sum_defl_sf0_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop0_b_sci[mm] = sum_defl_sfr0_b_sci[mm]/sum_defl_sf0_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop0_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'start0':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'stp0', stp0_b_sci, sum_defl_stp0_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr0', sfr0_b_sci, sum_defl_sfr0_b_sci
			
			FOR kk = 0D, n_elements(stp0_b_sci) - 1 DO BEGIN
			IF (stp0_b_sci[kk] NE 0) AND (FINITE(stp0_b_sci[kk]) EQ 1) AND (FINITE(sfr0_b_sci[kk]) EQ 1) THEN BEGIN
				start0_b_sci[kk] = sfr0_b_sci[kk]/stp0_b_sci[kk]
			ENDIF ELSE BEGIN
				start0_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp0_b_sci) - 1 DO BEGIN
			IF (sum_defl_stp0_b_sci[mm] NE 0) AND (FINITE(sum_defl_stp0_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start0_b_sci[mm] = sum_defl_sfr0_b_sci[mm]/sum_defl_stp0_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start0_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
		'stop2':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'sf2', sf2_b_sci, sum_defl_sf2_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr2', sfr2_b_sci, sum_defl_sfr2_b_sci
			
			FOR kk = 0D, n_elements(sf2_b_sci) - 1 DO BEGIN
			IF (sf2_b_sci[kk] NE 0) AND (FINITE(sf2_b_sci[kk]) EQ 1) AND (FINITE(sfr2_b_sci[kk]) EQ 1) THEN BEGIN
				stop2_b_sci[kk] = sfr2_b_sci[kk]/sf2_b_sci[kk]
			ENDIF ELSE BEGIN
				stop2_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_sf2_b_sci) - 1 DO BEGIN
			IF (sum_defl_sf2_b_sci[mm] NE 0) AND (FINITE(sum_defl_sf2_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_stop2_b_sci[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_sf2_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_stop2_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END

		'start2':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'stp2', stp2_b_sci, sum_defl_stp2_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'sfr2', sfr2_b_sci, sum_defl_sfr2_b_sci
			
			FOR kk = 0D, n_elements(stp2_b_sci) - 1 DO BEGIN
			IF (stp2_b_sci[kk] NE 0) AND (FINITE(stp2_b_sci[kk]) EQ 1) AND (FINITE(sfr2_b_sci[kk]) EQ 1) THEN BEGIN
				start2_b_sci[kk] = sfr2_b_sci[kk]/stp2_b_sci[kk]
			ENDIF ELSE BEGIN
				start2_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR
			
			FOR mm=0D, n_elements(sum_defl_stp2_b_sci) - 1 DO BEGIN
			IF (sum_defl_stp2_b_sci[mm] NE 0) AND (FINITE(sum_defl_stp2_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sfr2_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_start2_b_sci[mm] = sum_defl_sfr2_b_sci[mm]/sum_defl_stp2_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_start2_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
	
	'sf0ra_trig':  BEGIN
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                  		'sf0', sf0_b_sci, sum_defl_sf0_b_sci
			
			read_one_mon_no_eng, cdf_id, data5_index, epoch5maxrec, num_records5, E_step, $
                 		'ra_trig', ra_trig_b_sci, sum_defl_ra_trig_b_sci
			
			FOR kk = 0D, n_elements(sf0_b_sci) - 1 DO BEGIN
			IF (ra_trig_b_sci[kk] NE 0) AND (FINITE(ra_trig_b_sci[kk]) EQ 1) AND (FINITE(sf0_b_sci[kk]) EQ 1) THEN BEGIN
				sf0ra_trig_b_sci[kk] = sf0_b_sci[kk]/ra_trig_b_sci[kk]
			ENDIF ELSE BEGIN
				sf0ra_trig_b_sci[kk] = !values.d_nan
			ENDELSE
			ENDFOR

			FOR mm=0D, n_elements(sum_defl_sf0_b_sci) - 1 DO BEGIN
			IF (sum_defl_ra_trig_b_sci[mm] NE 0) AND (FINITE(sum_defl_ra_trig_b_sci[mm]) EQ 1) AND (FINITE(sum_defl_sf0_b_sci[mm]) EQ 1) THEN BEGIN
				sum_defl_sf0ra_trig_b_sci[mm] = sum_defl_sf0_b_sci[mm]/sum_defl_ra_trig_b_sci[mm]
			ENDIF ELSE BEGIN
				sum_defl_sf0ra_trig_b_sci[mm] = !values.d_nan
			ENDELSE
			ENDFOR
			
			END
                ENDCASE
            ENDFOR
        END
    ENDCASE

ENDELSE
    ;data_read = 1
END 
;-------------------------------------------------------------------------------------

;-------------------------------------------------------------------------
;------------------------------------------------------------------------
PRO read_hk, cdf_id, sat, hk_read, data1_index, epoch1maxrec, products
;------------------------------------------------------------------------
; read analog housekeeping data

COMMON com_mon
CASE sat OF 
    'A': BEGIN
	
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'lrnm_spare':	BEGIN
			read_product, cdf_id, 'lrnm_spare_eng',	lrnm_spare_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lrnm_spare_a
		END
		'lvc_-12v':  BEGIN
			read_product, cdf_id, 'lvc_-12v_eng',	lvc_neg12v_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg12v_a
		END
		'lvc_-5v':  BEGIN
			read_product, cdf_id, 'lvc_-5v_eng',	lvc_neg5v_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg5v_a
		END
		'esa_vm_pos':  BEGIN
			read_product, cdf_id, 'esa_vm_pos_eng',	esa_vm_pos_a,	data1_index,	epoch1maxrec
			remove_fill, -1, esa_vm_pos_a	
		END
		'dfl_1_vm':  BEGIN
			read_product, cdf_id, 'dfl_1_vm_eng',	dfl_1_vm_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_1_vm_a
		END
		'dfl_1_tap':  BEGIN
			read_product, cdf_id, 'dfl_1_tap_eng',	dfl_1_tap_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_1_tap_a
		END
		'dfl_2_vm':  BEGIN
			read_product, cdf_id, 'dfl_2_vm_eng',	dfl_2_vm_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_2_vm_a
		END
		'dfl_2_tap':  BEGIN
			read_product, cdf_id, 'dfl_2_tap_raw',	dfl_2_tap_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_2_tap_a
		END
		'lvc_+2.5v_b':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_b_eng',lvc_pos2point5v_b_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_b_a
		END
		'pac_cm_dc':  BEGIN
			read_product, cdf_id, 'pac_cm_dc_eng',	pac_cm_dc_a,	data1_index,	epoch1maxrec 
			remove_fill, -1, pac_cm_dc_a
		END
		'lvc_+12v':  BEGIN
			read_product, cdf_id, 'lvc_+12v_eng',	lvc_pos12v_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos12v_a
		END
		'lrpm_spare':  BEGIN
			read_product, cdf_id, 'lrpm_spare_eng',	lrpm_spare_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lrpm_spare_a
		END
		'lvc_+2.5v_a':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_a_eng',lvc_pos2point5v_a_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_a_a
		END
		'lvc_+5v':  BEGIN
			read_product, cdf_id, 'lvc_+5v_eng',	lvc_pos5v_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos5v_a
		END
		'adc_avdd':  BEGIN
			read_product, cdf_id, 'adc_avdd_eng',	adc_avdd_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_avdd_a
		END
		'adc_dvdd':  BEGIN
			read_product, cdf_id, 'adc_dvdd_eng',	adc_dvdd_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_dvdd_a
		END
		'pac_vm':  BEGIN
			read_product, cdf_id, 'pac_vm_eng',	pac_vm_a,	data1_index,	epoch1maxrec
			remove_fill, -1, pac_vm_a
		END
		'cal_vref':  BEGIN
			read_product, cdf_id, 'cal_vref_eng',	cal_vref_a,	data1_index,	epoch1maxrec
			remove_fill, -1, cal_vref_a
		END
		'pac_cm_ac':  BEGIN
			read_product, cdf_id, 'pac_cm_ac_raw',	pac_cm_ac_raw_a,data1_index,	epoch1maxrec
			remove_fill, -1, pac_cm_ac_a
		END
		'mcp_vm':  BEGIN
			read_product, cdf_id, 'mcp_vm_eng',	mcp_vm_a,	data1_index,	epoch1maxrec
			remove_fill, -1, mcp_vm_a
		END
		'mcp_cm_dc':  BEGIN
			read_product, cdf_id, 'mcp_cm_dc_eng',	mcp_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill, -1, mcp_cm_dc_a
		END
		'mcp_cm_ac':  BEGIN
			read_product, cdf_id, 'mcp_cm_ac_raw',	mcp_cm_ad_a,	data1_index,	epoch1maxrec
			remove_fill, -1, mcp_cm_ac_a
		END
		'ssd_cm_dc':  BEGIN
			read_product, cdf_id, 'ssd_cm_dc_eng',	ssd_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_cm_dc_a
		END
		's_ch_vm':  BEGIN
			read_product, cdf_id, 's_ch_vm_eng',	s_ch_vm_a,	data1_index,	epoch1maxrec
			remove_fill, -1, s_ch_vm_a
		END
		's_ch_vm_tap':  BEGIN
			read_product, cdf_id, 's_ch_vm_tap_eng',s_ch_vm_tap_a,	data1_index,	epoch1maxrec
			remove_fill, -1, s_ch_vm_tap_a
		END
		's_ch_cm_dc':  BEGIN
			read_product, cdf_id, 's_ch_cm_dc_eng',	s_ch_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill, -1, s_ch_cm_dc_a
		END
		'esa_vm_neg':  BEGIN
			read_product, cdf_id, 'esa_vm_neg_eng',	esa_vm_neg_a,	data1_index,	epoch1maxrec
			remove_fill, -1, esa_vm_neg_a
		END
		'esa_cm_dc':  BEGIN
			read_product, cdf_id, 'esa_cm_dc_eng',	esa_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill, -1, esa_cm_dc_a
		END
		'dfl_1_cm_dc':  BEGIN
			read_product, cdf_id, 'dfl_1_cm_dc_eng',dfl_1_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_1_cm_dc_a
		END
		'dfl_2_cm_dc':  BEGIN
			read_product, cdf_id, 'dfl_2_cm_dc_eng',dfl_2_cm_dc_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_2_cm_dc_a
		END
		'lvc_+2.5v_b_imon':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_b_imon_eng',	lvc_pos2point5v_b_imon_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_b_imon_a
		END
		'lvc_+12v_imon':  BEGIN
			read_product, cdf_id, 'lvc_+12v_imon_eng',	lvc_pos12v_imon_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos12v_imon_a
		END
		'lvc_-12v_imon':  BEGIN
			read_product, cdf_id, 'lvc_-12v_imon_eng',	lvc_neg12v_imon_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg12v_imon_a
		END
		'lvc_+5v_imon':  BEGIN
			read_product, cdf_id, 'lvc_+5v_imon_eng',	lvc_pos5v_imon_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos5v_imon_a
		END
		'lvc_-5v_imon':  BEGIN
			read_product, cdf_id, 'lvc_-5v_imon_eng',	lvc_neg5v_imon_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg5v_imon_a
		END
		'lvc_+2.5v_a_imon':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_a_imon_eng',	lvc_pos2point5v_a_imon_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_a_imon_a
		END
		'ssd_vm':  BEGIN
			read_product, cdf_id, 'ssd_vm_eng',	ssd_vm_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_vm_a
		END
		'adc_lu_flag':  BEGIN
			read_product, cdf_id, 'adc_lu_flag_raw',adc_lu_flag_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_lu_flag_a
		END
		'plug_id':  BEGIN
			read_product, cdf_id, 'plug_id_raw',	plug_id_a,	data1_index,	epoch1maxrec
			remove_fill, -1, plug_id_a
		END
		'adc_lu_ctr':  BEGIN
			read_product, cdf_id, 'adc_lu_ctr_raw',	adc_lu_ctr_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_lu_ctr_a
		END
		'dac_status':  BEGIN
			read_product, cdf_id, 'dac_status_raw',	dac_status_a,	data1_index,	epoch1maxrec
			remove_fill, -1, dac_status_a
		END
		'adc0_agnd':  BEGIN
			read_product, cdf_id, 'adc0_agnd_eng',	adc0_agnd_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc0_agnd_a
		END
		'adc1_agnd':  BEGIN
			read_product, cdf_id, 'adc1_agnd_eng',	adc1_agnd_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc1_agnd_a
		END
		'ra_e0':  BEGIN
			read_product, cdf_id, 'ra_e0_eng',	ra_e0_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ra_e0_a
		END
		'ra_e1':  BEGIN
			read_product, cdf_id, 'ra_e1_eng',	ra_e1_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ra_e1_a
		END
		'tac0_tsp':  BEGIN
			read_product, cdf_id, 'tac0_tsp_eng',	tac0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, tac0_tsp_a
		END
		'tac2_tsp':  BEGIN
			read_product, cdf_id, 'tac2_tsp_eng',	tac2_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, tac2_tsp_a
		END
		'sb0_tsp':  BEGIN
			read_product, cdf_id, 'sb0_tsp_eng',	sb0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, sb0_tsp_a
		END
		'sb1_tsp':  BEGIN
			read_product, cdf_id, 'sb1_tsp_eng',	sb1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, sb1_tsp_a
		END
		'tof_hv0_tsp':  BEGIN
			read_product, cdf_id, 'tof_hv0_tsp_eng',tof_hv0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, tof_hv0_tsp_a
		END
		'tof_hv1_tsp':  BEGIN
			read_product, cdf_id, 'tof_hv1_tsp_eng',tof_hv1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, tof_hv1_tsp_a
		END
		's_c_0_tsp':  BEGIN
			read_product, cdf_id, 's_c_0_tsp_eng',	s_c_0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, s_c_0_tsp_a
		END
		's_c_1_tsp':  BEGIN
			read_product, cdf_id, 's_c_1_tsp_eng',	s_c_1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, s_c_1_tsp_a
		END
		'lvc0_tsp':  BEGIN
			read_product, cdf_id, 'lvc0_tsp_eng',	lvc0_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc0_tsp_a
		END
		'lvc1_tsp':  BEGIN
			read_product, cdf_id, 'lvc1_tsp_eng',	lvc1_tsp_a,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc1_tsp_a
		END
		'adc0_vref':  BEGIN
			read_product, cdf_id, 'adc0_vref_eng',	adc0_vref_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc0_vref_a
		END
		'adc1_vref':  BEGIN
			read_product, cdf_id, 'adc1_vref_eng',	adc1_vref_a,	data1_index,	epoch1maxrec
			remove_fill, -1, adc1_vref_a
		END
		'ssd_status':  BEGIN
			read_product, cdf_id, 'ssd_status_raw',	ssd_status_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_status_a
		END
		'ssd_v_pos_analog':  BEGIN
			read_product, cdf_id, 'ssd_v_pos_analog_eng',	ssd_v_pos_analog_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_v_pos_analog_a
		END
		'ssd_v_neg_analog':  BEGIN
			read_product, cdf_id, 'ssd_v_neg_analog_eng',	ssd_v_neg_analog_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_v_neg_analog_a
		END
		'ssd_hv_bias':  BEGIN
			read_product, cdf_id, 'ssd_hv_bias_eng',	ssd_hv_bias_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_hv_bias_a
		END
		'ssd_tc0':  BEGIN
			read_product, cdf_id, 'ssd_tc0_eng',	ssd_tc0_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_tc0_a
		END
		'ssd_tc1':  BEGIN
			read_product, cdf_id, 'ssd_tc1_eng',	ssd_tc1_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_tc1_a
		END
		'ssd_tc2':  BEGIN
			read_product, cdf_id, 'ssd_tc2_eng',	ssd_tc2_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_tc2_a	
		END
		'ssd_v_pos_dig':  BEGIN
			read_product, cdf_id, 'ssd_v_pos_dig_eng',	ssd_v_pos_dig_a,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_v_pos_dig_a
		END
		ENDCASE
	ENDFOR

    END
    'B': BEGIN
	FOR ii = 0, n_elements(products)-1 DO BEGIN 
		CASE products[ii] OF
		'lrnm_spare':	BEGIN
			read_product, cdf_id, 'lrnm_spare_eng',	lrnm_spare_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lrnm_spare_b
		END
		'lvc_-12v':  BEGIN
			read_product, cdf_id, 'lvc_-12v_eng',	lvc_neg12v_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg12v_b
		END
		'lvc_-5v':  BEGIN
			read_product, cdf_id, 'lvc_-5v_eng',	lvc_neg5v_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg5v_b
		END
		'esa_vm_pos':  BEGIN
			read_product, cdf_id, 'esa_vm_pos_eng',	esa_vm_pos_b,	data1_index,	epoch1maxrec
			remove_fill, -1, esa_vm_pos_b	
		END
		'dfl_1_vm':  BEGIN
			read_product, cdf_id, 'dfl_1_vm_eng',	dfl_1_vm_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_1_vm_b
		END
		'dfl_1_tap':  BEGIN
			read_product, cdf_id, 'dfl_1_tap_eng',	dfl_1_tap_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_1_tap_b
		END
		'dfl_2_vm':  BEGIN
			read_product, cdf_id, 'dfl_2_vm_eng',	dfl_2_vm_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_2_vm_b
		END
		'dfl_2_tap':  BEGIN
			read_product, cdf_id, 'dfl_2_tap_raw',	dfl_2_tap_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_2_tap_b
		END
		'lvc_+2.5v_b':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_b_eng',lvc_pos2point5v_b_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_b_b
		END
		'pac_cm_dc':  BEGIN
			read_product, cdf_id, 'pac_cm_dc_eng',	pac_cm_dc_b,	data1_index,	epoch1maxrec 
			remove_fill, -1, pac_cm_dc_b
		END
		'lvc_+12v':  BEGIN
			read_product, cdf_id, 'lvc_+12v_eng',	lvc_pos12v_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos12v_b
		END
		'lrpm_spare':  BEGIN
			read_product, cdf_id, 'lrpm_spare_eng',	lrpm_spare_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lrpm_spare_b
		END
		'lvc_+2.5v_a':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_a_eng',lvc_pos2point5v_a_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_a_b
		END
		'lvc_+5v':  BEGIN
			read_product, cdf_id, 'lvc_+5v_eng',	lvc_pos5v_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos5v_b
		END
		'adc_avdd':  BEGIN
			read_product, cdf_id, 'adc_avdd_eng',	adc_avdd_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_avdd_b
		END
		'adc_dvdd':  BEGIN
			read_product, cdf_id, 'adc_dvdd_eng',	adc_dvdd_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_dvdd_b
		END
		'pac_vm':  BEGIN
			read_product, cdf_id, 'pac_vm_eng',	pac_vm_b,	data1_index,	epoch1maxrec
			remove_fill, -1, pac_vm_b
		END
		'cal_vref':  BEGIN
			read_product, cdf_id, 'cal_vref_eng',	cal_vref_b,	data1_index,	epoch1maxrec
			remove_fill, -1, cal_vref_b
		END
		'pac_cm_ac':  BEGIN
			read_product, cdf_id, 'pac_cm_ac_raw',	pac_cm_ac_raw_b,data1_index,	epoch1maxrec
			remove_fill, -1, pac_cm_ac_b
		END
		'mcp_vm':  BEGIN
			read_product, cdf_id, 'mcp_vm_eng',	mcp_vm_b,	data1_index,	epoch1maxrec
			remove_fill, -1, mcp_vm_b
		END
		'mcp_cm_dc':  BEGIN
			read_product, cdf_id, 'mcp_cm_dc_eng',	mcp_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill, -1, mcp_cm_dc_b
		END
		'mcp_cm_ac':  BEGIN
			read_product, cdf_id, 'mcp_cm_ac_raw',	mcp_cm_ad_b,	data1_index,	epoch1maxrec
			remove_fill, -1, mcp_cm_ac_b
		END
		'ssd_cm_dc':  BEGIN
			read_product, cdf_id, 'ssd_cm_dc_eng',	ssd_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_cm_dc_b
		END
		's_ch_vm':  BEGIN
			read_product, cdf_id, 's_ch_vm_eng',	s_ch_vm_b,	data1_index,	epoch1maxrec
			remove_fill, -1, s_ch_vm_b
		END
		's_ch_vm_tap':  BEGIN
			read_product, cdf_id, 's_ch_vm_tap_eng',s_ch_vm_tap_b,	data1_index,	epoch1maxrec
			remove_fill, -1, s_ch_vm_tap_b
		END
		's_ch_cm_dc':  BEGIN
			read_product, cdf_id, 's_ch_cm_dc_eng',	s_ch_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill, -1, s_ch_cm_dc_b
		END
		'esa_vm_neg':  BEGIN
			read_product, cdf_id, 'esa_vm_neg_eng',	esa_vm_neg_b,	data1_index,	epoch1maxrec
			remove_fill, -1, esa_vm_neg_b
		END
		'esa_cm_dc':  BEGIN
			read_product, cdf_id, 'esa_cm_dc_eng',	esa_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill, -1, esa_cm_dc_b
		END
		'dfl_1_cm_dc':  BEGIN
			read_product, cdf_id, 'dfl_1_cm_dc_eng',dfl_1_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_1_cm_dc_b
		END
		'dfl_2_cm_dc':  BEGIN
			read_product, cdf_id, 'dfl_2_cm_dc_eng',dfl_2_cm_dc_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dfl_2_cm_dc_b
		END
		'lvc_+2.5v_b_imon':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_b_imon_eng',	lvc_pos2point5v_b_imon_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_b_imon_b
		END
		'lvc_+12v_imon':  BEGIN
			read_product, cdf_id, 'lvc_+12v_imon_eng',	lvc_pos12v_imon_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos12v_imon_b
		END
		'lvc_-12v_imon':  BEGIN
			read_product, cdf_id, 'lvc_-12v_imon_eng',	lvc_neg12v_imon_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg12v_imon_b
		END
		'lvc_+5v_imon':  BEGIN
			read_product, cdf_id, 'lvc_+5v_imon_eng',	lvc_pos5v_imon_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos5v_imon_b
		END
		'lvc_-5v_imon':  BEGIN
			read_product, cdf_id, 'lvc_-5v_imon_eng',	lvc_neg5v_imon_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_neg5v_imon_b
		END
		'lvc_+2.5v_a_imon':  BEGIN
			read_product, cdf_id, 'lvc_+2.5v_a_imon_eng',	lvc_pos2point5v_a_imon_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc_pos2point5v_a_imon_b
		END
		'ssd_vm':  BEGIN
			read_product, cdf_id, 'ssd_vm_eng',	ssd_vm_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_vm_b
		END
		'adc_lu_flag':  BEGIN
			read_product, cdf_id, 'adc_lu_flag_raw',adc_lu_flag_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_lu_flag_b
		END
		'plug_id':  BEGIN
			read_product, cdf_id, 'plug_id_raw',	plug_id_b,	data1_index,	epoch1maxrec
			remove_fill, -1, plug_id_b
		END
		'adc_lu_ctr':  BEGIN
			read_product, cdf_id, 'adc_lu_ctr_raw',	adc_lu_ctr_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc_lu_ctr_b
		END
		'dac_status':  BEGIN
			read_product, cdf_id, 'dac_status_raw',	dac_status_b,	data1_index,	epoch1maxrec
			remove_fill, -1, dac_status_b
		END
		'adc0_agnd':  BEGIN
			read_product, cdf_id, 'adc0_agnd_eng',	adc0_agnd_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc0_agnd_b
		END
		'adc1_agnd':  BEGIN
			read_product, cdf_id, 'adc1_agnd_eng',	adc1_agnd_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc1_agnd_b
		END
		'ra_e0':  BEGIN
			read_product, cdf_id, 'ra_e0_eng',	ra_e0_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ra_e0_b
		END
		'ra_e1':  BEGIN
			read_product, cdf_id, 'ra_e1_eng',	ra_e1_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ra_e1_b
		END
		'tac0_tsp':  BEGIN
			read_product, cdf_id, 'tac0_tsp_eng',	tac0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, tac0_tsp_b
		END
		'tac2_tsp':  BEGIN
			read_product, cdf_id, 'tac2_tsp_eng',	tac2_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, tac2_tsp_b
		END
		'sb0_tsp':  BEGIN
			read_product, cdf_id, 'sb0_tsp_eng',	sb0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, sb0_tsp_b
		END
		'sb1_tsp':  BEGIN
			read_product, cdf_id, 'sb1_tsp_eng',	sb1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, sb1_tsp_b
		END
		'tof_hv0_tsp':  BEGIN
			read_product, cdf_id, 'tof_hv0_tsp_eng',tof_hv0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, tof_hv0_tsp_b
		END
		'tof_hv1_tsp':  BEGIN
			read_product, cdf_id, 'tof_hv1_tsp_eng',tof_hv1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, tof_hv1_tsp_b
		END
		's_c_0_tsp':  BEGIN
			read_product, cdf_id, 's_c_0_tsp_eng',	s_c_0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, s_c_0_tsp_b
		END
		's_c_1_tsp':  BEGIN
			read_product, cdf_id, 's_c_1_tsp_eng',	s_c_1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, s_c_1_tsp_b
		END
		'lvc0_tsp':  BEGIN
			read_product, cdf_id, 'lvc0_tsp_eng',	lvc0_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc0_tsp_b
		END
		'lvc1_tsp':  BEGIN
			read_product, cdf_id, 'lvc1_tsp_eng',	lvc1_tsp_b,	data1_index,	epoch1maxrec
			remove_fill, -1, lvc1_tsp_b
		END
		'adc0_vref':  BEGIN
			read_product, cdf_id, 'adc0_vref_eng',	adc0_vref_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc0_vref_b
		END
		'adc1_vref':  BEGIN
			read_product, cdf_id, 'adc1_vref_eng',	adc1_vref_b,	data1_index,	epoch1maxrec
			remove_fill, -1, adc1_vref_b
		END
		'ssd_status':  BEGIN
			read_product, cdf_id, 'ssd_status_raw',	ssd_status_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_status_b
		END
		'ssd_v_pos_analog':  BEGIN
			read_product, cdf_id, 'ssd_v_pos_analog_eng',	ssd_v_pos_analog_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_v_pos_analog_b
		END
		'ssd_v_neg_analog':  BEGIN
			read_product, cdf_id, 'ssd_v_neg_analog_eng',	ssd_v_neg_analog_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_v_neg_analog_b
		END
		'ssd_hv_bias':  BEGIN
			read_product, cdf_id, 'ssd_hv_bias_eng',	ssd_hv_bias_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_hv_bias_b
		END
		'ssd_tc0':  BEGIN
			read_product, cdf_id, 'ssd_tc0_eng',	ssd_tc0_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_tc0_b
		END
		'ssd_tc1':  BEGIN
			read_product, cdf_id, 'ssd_tc1_eng',	ssd_tc1_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_tc1_b
		END
		'ssd_tc2':  BEGIN
			read_product, cdf_id, 'ssd_tc2_eng',	ssd_tc2_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_tc2_b
		END
		'ssd_v_pos_dig':  BEGIN
			read_product, cdf_id, 'ssd_v_pos_dig_eng',	ssd_v_pos_dig_b,	data1_index,	epoch1maxrec
			remove_fill, -1, ssd_v_pos_dig_b
		END
		ENDCASE
	ENDFOR
    END
ENDCASE
hk_read = 1
END 
;---------------------------------------------------------------------------------------------------------

;; END OF DATA READING PROGRAMS


PRO check_indices, index, epoch, resolution, file_num

; checks for missing records from the cdf epoch variable
; index: array of pairs [start_index,num_records] (could be [0,10,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1,5, or 3 depending on epoch (may need to add variable)
; file_num: index of this file in the file array

first_index=0L
first_index = long(file_num)*(1440/resolution)

num_records = 0L
in_valid_area = 0
ii = 0L
index = [-1L]
in_synch = 1 ; needed for when duplicate time-stamps -- only takes first
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 59 DO BEGIN 
                WHILE in_synch EQ 1 DO BEGIN 
                    IF ii LT n_elements(epoch) THEN BEGIN 
                        time = time_struct(epoch[ii])
                        IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                            IF in_valid_area EQ 0 THEN BEGIN
                                start_index_time = first_index+(hour*60)+min
                                start_index_cdf = ii
                                num_records = 1
                                in_valid_area = 1
                            ENDIF ELSE BEGIN ; already in valid area
                                num_records = num_records+1
                            ENDELSE 
                            ii = ii+1
                            in_synch = 0
                        ENDIF ELSE BEGIN
                            IF in_valid_area EQ 1 THEN BEGIN
                                in_valid_area = 0
                                IF index[0] EQ -1 THEN index = [start_index_time, start_index_cdf, num_records] $
                                ELSE index = [index, start_index_time, start_index_cdf, num_records]
                            ENDIF 
                            IF time.hour LT hour OR (hour EQ time.hour AND time.min LT min) THEN ii = ii+1 ELSE in_synch = 0
                        ENDELSE 
                    ENDIF ELSE BEGIN 
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index_time, start_index_cdf, num_records] $
                            ELSE index = [index, start_index_time, start_index_cdf, num_records]                
                        ENDIF 
                        in_synch = 0
                    ENDELSE 
                ENDWHILE 
                in_synch = 1
            ENDFOR 
        ENDFOR 
    END
    5: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 11 DO BEGIN 
                WHILE in_synch EQ 1 DO BEGIN 
                    IF ii LT n_elements(epoch) THEN BEGIN 
                        time = time_struct(epoch[ii])
                        IF (hour EQ time.hour AND time.min GE (min*5) AND time.min LT (min*5+5)) OR $
                          (time.hour EQ hour+1 AND time.min LE 4 AND min EQ 11) THEN BEGIN 
                            IF in_valid_area EQ 0 THEN BEGIN
                                start_index_time = first_index+(hour*12)+min
                                start_index_cdf = ii
                                num_records = 1
                                in_valid_area = 1
                            ENDIF ELSE BEGIN ; already in valid area
                                num_records = num_records+1
                            ENDELSE 
                            ii = ii+1
                            in_synch = 0
                        ENDIF ELSE BEGIN
                            IF in_valid_area EQ 1 THEN BEGIN
                                in_valid_area = 0
                                IF index[0] EQ -1 THEN index = [start_index_time, start_index_cdf, num_records] $
                                ELSE index = [index, start_index_time, start_index_cdf, num_records]
                            ENDIF 
                            IF time.hour LT hour OR (hour EQ time.hour AND time.min LT min) THEN ii = ii+1 ELSE in_synch = 0
                        ENDELSE 
                    ENDIF ELSE BEGIN 
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index_time, start_index_cdf, num_records] $
                            ELSE index = [index, start_index_time, start_index_cdf, num_records]                
                        ENDIF 
                        in_synch = 0
                    ENDELSE 
                ENDWHILE
                in_synch = 1
            ENDFOR 
        ENDFOR 
    END
	3: BEGIN
	FOR hour=0L,23 DO BEGIN
	   FOR min=0L,59 DO BEGIN
		FOR sec=0L,5 DO BEGIN
                    WHILE in_synch EQ 1 DO BEGIN 
                        IF ii LT n_elements(epoch) THEN BEGIN
                            time = time_struct(epoch[ii])
			;cdf_epoch, epoch[ii], year, month, day, time_hour, time_min, time_sec, milli,/breakdown_epoch
			;print, ii, time_hour, time_min, time_sec
                            IF (hour EQ time.hour AND min EQ time.min AND sec EQ time.sec) $
                              OR (hour EQ time.hour AND min EQ time.min AND time.sec LT sec*10+10) THEN BEGIN
                                IF in_valid_area EQ 0 THEN BEGIN
                                    start_index_time = (first_index+(hour*360)+(min*6)+(sec))	
                                    start_index_cdf = ii
                                    IF start_index_time LT 0 THEN STOP
                                    num_records = 1
                                    in_valid_area = 1
                                ENDIF ELSE BEGIN ; already in valid area
                                    num_records = num_records+1
                                ENDELSE
                                ii = ii+1
                                in_synch = 0
                                IF sec GT 0 THEN BEGIN
                                    sec = sec-1
                                ENDIF ELSE BEGIN
                                    IF (min GT 0) AND (sec EQ 0) THEN BEGIN
                                        min = min-1 
                                        sec = sec+59
                                    ENDIF ELSE BEGIN
                                        IF (hour GT 0) AND (min EQ 0) and (sec EQ 0) THEN BEGIN
                                            hour = hour-1
                                            min = min+59 
                                            sec = sec+59
                                        ENDIF
                                    ENDELSE
                                ENDELSE
                            ENDIF ELSE BEGIN
                                IF in_valid_area EQ 1 THEN BEGIN
                                    in_valid_area = 0
                                    IF index[0] EQ -1 THEN index = [start_index_time, start_index_cdf, num_records] $
                                    ELSE index = [index, start_index_time, start_index_cdf, num_records]                
                                ENDIF
                                IF time.hour LT hour OR (hour EQ time.hour AND time.min LT min) THEN ii = ii+1 ELSE in_synch = 0
                            ENDELSE
                        ENDIF ELSE BEGIN
                            IF in_valid_area EQ 1 THEN BEGIN
                                in_valid_area = 0
                                IF index[0] EQ -1 THEN index = [start_index_time, start_index_cdf, num_records] $
                                ELSE index = [index, start_index_time, start_index_cdf, num_records]                
                            ENDIF
                            in_synch = 0
                        ENDELSE
                    ENDWHILE
                    in_synch = 1
                ENDFOR
            ENDFOR
        ENDFOR

    END

    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf_general, sat, files, products, which_epoch, E_step, crib_name, NO_ENG

PRINT, crib_name
PRINT, products

COMMON share1_mon
COMMON com_mon
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat


; initialize output variables (from common block)
need_epoch1 = 0
need_epoch5 = 0
need_epoch3 = 0
FOR ii = 0L, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
	 1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
	    num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 5
        END 
	3: BEGIN
	   num_records3=DOUBLE(n_elements(files)*8651)
	   epoch3 = DBLARR(num_records3)
	   epoch3(*) = !values.d_nan
	   need_epoch3 = 3
	END
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 



; SPACECRAFT HOUSEKEEPING
;--------------------------
IF crib_name EQ 'sc_hk' THEN BEGIN
sc_hk_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	'pl_ebox_tmp': ; 0
	'pl_inst_msg_pwr_dwn_req': ; 1
	'pl_pwr_cur': ; 2
	'pl_pwr_tt': ; 3
	'pl_surv_pwr_cur': ; 4
	'pl_surv_pwr_tt': ; 5
	'pl_tof_tmp': ; 6
	'pl_pwr_vt': ; 7
	'pl_surv_pwr_v': $; 8
	 IF sc_hk_initialized EQ 0 THEN init_sc_hk, sat, num_records3, sc_hk_initialized
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                   ; get epochs
		   cdf_control, cdf_id, get_var_info = epoch3info, var = 'epoch', /zvariable
		   IF epoch3info.maxrec GE 0 THEN BEGIN
			cdf_varget, cdf_id, 'epoch', temp_epoch, rec_count = epoch3info.maxrec, /zvariable
			converted_epoch = time_double(temp_epoch, /epoch)
			check_indices, data3_index, converted_epoch, which_epoch, ii
			 FOR kk = 0L, n_elements(data3_index)-1 DO BEGIN 
                            epoch3[data3_index[kk]:(data3_index[kk]+data3_index[kk+2]-1)] = $
                              converted_epoch[data3_index[kk+1]:(data3_index[kk+1]+data3_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR
		   ENDIF
		remove_fill, -1, epoch3

        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_sc_hk_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_sc_hk_a = epoch5
	    IF need_epoch3 EQ 3 THEN epoch3_sc_hk_a = epoch3
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_sc_hk_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_sc_hk_b = epoch5 
	    IF need_epoch3 EQ 3 THEN epoch3_sc_hk_b = epoch3
        ENDELSE 

        ; get other data
        sc_hk_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		'pl_ebox_tmp': ; 0
		'pl_inst_msg_pwr_dwn_req': ; 1
		'pl_pwr_cur': ; 2
		'pl_pwr_tt': ; 3
		'pl_surv_pwr_cur': ; 4
		'pl_surv_pwr_tt': ; 5
		'pl_tof_tmp': ; 6
		'pl_pwr_vt': ; 7
		'pl_surv_pwr_v': $; 8
			IF sc_hk_read EQ 0 THEN read_sc_hk, cdf_id, sat, sc_hk_read, data3_index, epoch3info.maxrec, products, epoch3
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_sc_hk_index_a = data1_index 
    IF need_epoch5 EQ 5 THEN data5_sc_hk_index_a = data5_index
    IF need_epoch3 EQ 3 THEN data3_sc_hk_index_a = data3_index 
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_sc_hk_index_b = data1_index
    IF need_epoch5 EQ 5 THEN data5_sc_hk_index_b = data5_index
    IF need_epoch3 EQ 3 THEN data3_sc_hk_index_b = data3_index 
ENDELSE 
ENDIF



; DIGITAL HOUSEKEEPING
;------------------------
IF crib_name EQ 'digital_hk' THEN BEGIN

dig_hk_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	   'dpu_sw_version':                  ; 0 
           'cmd_rcvd_cnt':                    ; 1
           'cmd_exec_cnt':                    ; 2
           'cmd_total_err_cnt':               ; 3
           'cmd_parity_err_cnt':              ; 4
           'cmd_frame_err_cnt':               ; 5
           'cmd_illegal_err_cnt':             ; 6
           'ic_status_reg':                   ; 7
           'ic_ctl_reg':                      ; 8
           'lbc_reset_ctl':                   ; 9
           'lbc_mode_ctl':                    ; 10
           'lbc_logic_ctl_a':                 ; 11
           'lbc_logic_ctl_b':                 ; 12
           'lbc_tac_pwr_ctl':                 ; 13
           'lbc_reg_seq':                     ; 14
           'lbc_event_ctl':                   ; 15
           'lbc_rlim_ch':                     ; 16
           'lbc_rlim_hi':                     ; 17
           'lbc_rlim_lo':                     ; 18
           'lbe_pos_ctrl':                    ; 19
           'lbe_pos_disable0':                ; 20
           'lbe_pos_disable2':                ; 21
           'lbe_tac0_ctrl':                   ; 22
           'lbe_tac0_dac':                    ; 23
           'lbe_tac2_ctrl':                   ; 24
           'lbe_tac2_dac':                    ; 25
           'lbe_tac0_under_hi':               ; 26
           'lbe_tac0_under_lo':               ; 27
           'lbe_tac0_over_hi':                ; 28
           'lbe_tac0_over_lo':                ; 29
           'lbe_tac2_under_hi':               ; 30
           'lbe_tac2_under_lo':               ; 31
           'lbe_tac2_over_hi':                ; 32
           'lbe_tac2_over_lo':                ; 33
           'lbe_ssd_ctrl':                    ; 34
           'lbe_ssd_cmd_h':                   ; 35
           'lbe_ssd_cmd_l':                   ; 36
           'lbe_ssd_dis':                     ; 37
           'lbe_ssd_under_hi':                ; 38
           'lbe_ssd_under_lo':                ; 39
           'lbe_ssd_over_hi':                 ; 40
           'lbe_ssd_over_lo':                 ; 41
           'lbe_sel_ctrl':                    ; 42
           'lbe_trig_mode':                   ; 43
           'lbe_esa_step':                    ; 44
           'lbe_defl_step':                   ; 45
           'lbe_ssd_offset0':                 ; 46
           'lbe_ssd_offset1':                 ; 47
           'lbe_ssd_offset2':                 ; 48
           'lbe_ssd_offset3':                 ; 49
           'lbe_ssd_offset4':                 ; 50
           'lbe_ssd_offset5':                 ; 51
           'lbe_ssd_offset6':                 ; 52
           'lbe_ssd_offset7':                 ; 53
           'lbe_stim_enable':                 ; 54
           'lbe_stim_freq_hi':                ; 55
           'lbe_stim_freq_lo':                ; 56
           'lbe_sel0_window':                 ; 57
           'lbe_sel2_window':                 ; 58
           'lbe_ssd_offset8':                 ; 59
           'lbe_ssd_offset9':                 ; 60
           'lbe_ssd_offset10':                ; 61
           'lbe_ssd_offset11':                ; 62
           'lbe_ssd_offset12':                ; 63
           'lbe_ssd_offset13':                ; 64
           'lbe_ssd_offset14':                ; 65
           'lbe_ssd_offset15':                ; 66
           'lbe_pos_ra':                      ; 67
           'dacm_pac_mcp_ssd_ctl_mcp_limit':  ; 68
           'dacm_pac_limit':                  ; 69
           'dacm_esa_ctl':                    ; 70
           'dacm_s_ch_ctl':                   ; 71
           'dacm_dfl_1_ctl':                  ; 72
           'dacm_dfl_2_ctl':                  ; 73
           'dacm_control':                    ; 74
           'dacm_status':                     ; 75
           'dacm_adc_lu_ctr':                 ; 76
           'dacm_dfl_1_offset':               ; 77
           'dacm_dfl_2_offset':               ; 78
           'dacm_pac_coarse':                 ; 79
           'dacm_pac_fine':                   ; 80
           'dacm_pac_cm_fs_coarse':           ; 81
           'dacm_pac_cm_fs_fine':             ; 82
           'dacm_mcp_coarse':                 ; 83
           'dacm_mcp_fine':                   ; 84
           'dacm_mcp_cm_fs_coarse':           ; 85
           'dacm_mcp_cm_fs_fine':             ; 86
           'dacm_ssd_coarse':                 ; 87
           'dacm_ssd_fine':                   ; 88
           'dacm_esa_coarse':                 ; 89
           'dacm_esa_fine':                   ; 90
           'dacm_s_ch_coarse':                ; 91
           'dacm_s_ch_fine':                  ; 92
           'dacm_s_ch_tap_coarse':            ; 93
           'dacm_s_ch_tap_fine':              ; 94
           'dacm_dfl_1_coarse':               ; 95
           'dacm_dfl_1_fine':                 ; 96
           'dacm_dfl_1_dfine':                ; 97
           'dacm_dfl_2_coarse':               ; 98
           'dacm_dfl_2_fine':                 ; 99
           'dacm_dfl_2_dfine':                ; 100
           'dacr_pac_mcp_ssd_ctl_mcp_limit':  ; 101
           'dacr_pac_limit':                  ; 102
           'dacr_esa_ctl':                    ; 103
           'dacr_s_ch_ctl':                   ; 104
           'dacr_dfl_1_ctl':                  ; 105
           'dacr_dfl_2_ctl':                  ; 106
           'dacr_control':                    ; 107
           'dacr_status':                     ; 108
           'dacr_adc_lu_ctr':                 ; 109
           'dacr_dfl_1_offset':               ; 110
           'dacr_dfl_2_offset':               ; 111
           'dacr_c_pac_coarse':               ; 112
           'dacr_c_pac_fine':                 ; 113
           'dacr_c_pac_cm_fs_coarse':         ; 114
           'dacr_c_pac_cm_fs_fine':           ; 115
           'dacr_c_mcp_coarse':               ; 116
           'dacr_c_mcp_fine':                 ; 117
           'dacr_c_mcp_cm_fs_coarse':         ; 118
           'dacr_c_mcp_cm_fs_fine':           ; 119
           'dacr_c_ssd_coarse':               ; 120
           'dacr_c_ssd_fine':                 ; 121
           'dacr_c_esa_coarse':               ; 122
           'dacr_c_esa_fine':                 ; 123
           'dacr_c_s_ch_coarse':              ; 124
           'dacr_c_s_ch_fine':                ; 125
           'dacr_c_s_ch_tap_coarse':          ; 126
           'dacr_c_s_ch_tap_fine':            ; 127
           'dacr_dfl_1_coarse':               ; 128
           'dacr_dfl_1_fine':                 ; 129
           'dacr_dfl_1_dfine':                ; 130
           'dacr_dfl_2_coarse':               ; 131
           'dacr_dfl_2_fine':                 ; 132
           'dacr_dfl_2_dfine':                ; 133
           'sw_status1':                      ; 134
           'sw_hv_ramp_status':               ; 135
           'sw_hv_enable_status':             ; 136
           'sw_hv_limit_status':              ; 137
           'sw_eepromchksum':                 ; 138
           'hkp_sampletime':                  ; 139
           'pac_discharge_status':            ; 140
           'pac_discharge_cnt':               ; 141
           'pac_discharge_conseccntr':        ; 142
           'pac_discharge_safe_att':          ; 143
           'analog_limit_status':             ; 144
           'analog_lim_pt1_status':           ; 145
           'analog_lim_pt1_id':               ; 146
           'analog_lim_pt1_red':              ; 147
           'analog_lim_pt2_status':           ; 148
           'analog_lim_pt2_id':               ; 149
           'analog_lim_pt2_red':              ; 150
           'analog_lim_pt3_status':           ; 151
           'analog_lim_pt3_id':               ; 152
           'analog_lim_pt3_red':              ; 153
           'analog_lim_pt4_status':           ; 154
           'analog_lim_pt4_id':               ; 155
           'analog_lim_pt4_red':              ; 156
           'analog_lim_pt5_status':           ; 157
           'analog_lim_pt5_id':               ; 158
           'analog_lim_pt5_red':              ; 159
           'analog_lim_pt6_status':           ; 160
           'analog_lim_pt6_id':               ; 161
           'analog_lim_pt6_red':              ; 162
           'analog_lim_pt7_status':           ; 163
           'analog_lim_pt7_id':               ; 164
           'analog_lim_pt7_red':              ; 165
           'analog_lim_pt8_status':           ; 166
           'analog_lim_pt8_id':               ; 167
           'analog_lim_pt8_red':              ; 168
           'analog_lim_pt9_status':           ; 169
           'analog_lim_pt9_id':               ; 170
           'analog_lim_pt9_red':              ; 171
           'analog_lim_pt10_status':          ; 172
           'analog_lim_pt10_id':              ; 173
           'analog_lim_pt10_red':             ; 174
           'analog_lim_pt11_status':          ; 175
           'analog_lim_pt11_id':              ; 176
           'analog_lim_pt11_red':             ; 177
           'analog_lim_pt12_status':          ; 178
           'analog_lim_pt12_id':              ; 179
           'analog_lim_pt12_red':             ; 180
           'analog_lim_pt13_status':          ; 181
           'analog_lim_pt13_id':              ; 182
           'analog_lim_pt13_red':             ; 183
           'analog_lim_pt14_status':          ; 184
           'analog_lim_pt14_id':              ; 185
           'analog_lim_pt14_red':             ; 186
           'analog_lim_pt15_status':          ; 187
           'analog_lim_pt15_id':              ; 188
           'analog_lim_pt15_red':             ; 189
           'opmode':                          ; 190
           'sw_mode':                         ; 191
           'proton_mode':                     ; 192
           'active_supplyid':                 ; 193
           'supply_coarse_rb':                ; 194
           'supply_fine_rb':                  ; 195
           'swmaint_supplystat':              ; 196
           'swmaint_pac_coarse':              ; 197
           'swmaint_pac_fine': $              ; 198
	IF dig_hk_initialized EQ 0 THEN init_dig_hk, sat, num_records1, dig_hk_initialized, products
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
			;get epochs
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch_digital', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch_digital', temp_epoch, rec_count = epoch1info.maxrec, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+2]-1)] = $
                              converted_epoch[data1_index[kk+1]:(data1_index[kk+1]+data1_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR 
                    ENDIF
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_dig_hk_a = epoch1
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_dig_hk_b = epoch1  
        ENDELSE 

 ; get other data
        dig_hk_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   	   'dpu_sw_version':                  ; 0 
           'cmd_rcvd_cnt':                    ; 1
           'cmd_exec_cnt':                    ; 2
           'cmd_total_err_cnt':               ; 3
           'cmd_parity_err_cnt':              ; 4
           'cmd_frame_err_cnt':               ; 5
           'cmd_illegal_err_cnt':             ; 6
           'ic_status_reg':                   ; 7
           'ic_ctl_reg':                      ; 8
           'lbc_reset_ctl':                   ; 9
           'lbc_mode_ctl':                    ; 10
           'lbc_logic_ctl_a':                 ; 11
           'lbc_logic_ctl_b':                 ; 12
           'lbc_tac_pwr_ctl':                 ; 13
           'lbc_reg_seq':                     ; 14
           'lbc_event_ctl':                   ; 15
           'lbc_rlim_ch':                     ; 16
           'lbc_rlim_hi':                     ; 17
           'lbc_rlim_lo':                     ; 18
           'lbe_pos_ctrl':                    ; 19
           'lbe_pos_disable0':                ; 20
           'lbe_pos_disable2':                ; 21
           'lbe_tac0_ctrl':                   ; 22
           'lbe_tac0_dac':                    ; 23
           'lbe_tac2_ctrl':                   ; 24
           'lbe_tac2_dac':                    ; 25
           'lbe_tac0_under_hi':               ; 26
           'lbe_tac0_under_lo':               ; 27
           'lbe_tac0_over_hi':                ; 28
           'lbe_tac0_over_lo':                ; 29
           'lbe_tac2_under_hi':               ; 30
           'lbe_tac2_under_lo':               ; 31
           'lbe_tac2_over_hi':                ; 32
           'lbe_tac2_over_lo':                ; 33
           'lbe_ssd_ctrl':                    ; 34
           'lbe_ssd_cmd_h':                   ; 35
           'lbe_ssd_cmd_l':                   ; 36
           'lbe_ssd_dis':                     ; 37
           'lbe_ssd_under_hi':                ; 38
           'lbe_ssd_under_lo':                ; 39
           'lbe_ssd_over_hi':                 ; 40
           'lbe_ssd_over_lo':                 ; 41
           'lbe_sel_ctrl':                    ; 42
           'lbe_trig_mode':                   ; 43
           'lbe_esa_step':                    ; 44
           'lbe_defl_step':                   ; 45
           'lbe_ssd_offset0':                 ; 46
           'lbe_ssd_offset1':                 ; 47
           'lbe_ssd_offset2':                 ; 48
           'lbe_ssd_offset3':                 ; 49
           'lbe_ssd_offset4':                 ; 50
           'lbe_ssd_offset5':                 ; 51
           'lbe_ssd_offset6':                 ; 52
           'lbe_ssd_offset7':                 ; 53
           'lbe_stim_enable':                 ; 54
           'lbe_stim_freq_hi':                ; 55
           'lbe_stim_freq_lo':                ; 56
           'lbe_sel0_window':                 ; 57
           'lbe_sel2_window':                 ; 58
           'lbe_ssd_offset8':                 ; 59
           'lbe_ssd_offset9':                 ; 60
           'lbe_ssd_offset10':                ; 61
           'lbe_ssd_offset11':                ; 62
           'lbe_ssd_offset12':                ; 63
           'lbe_ssd_offset13':                ; 64
           'lbe_ssd_offset14':                ; 65
           'lbe_ssd_offset15':                ; 66
           'lbe_pos_ra':                      ; 67
           'dacm_pac_mcp_ssd_ctl_mcp_limit':  ; 68
           'dacm_pac_limit':                  ; 69
           'dacm_esa_ctl':                    ; 70
           'dacm_s_ch_ctl':                   ; 71
           'dacm_dfl_1_ctl':                  ; 72
           'dacm_dfl_2_ctl':                  ; 73
           'dacm_control':                    ; 74
           'dacm_status':                     ; 75
           'dacm_adc_lu_ctr':                 ; 76
           'dacm_dfl_1_offset':               ; 77
           'dacm_dfl_2_offset':               ; 78
           'dacm_pac_coarse':                 ; 79
           'dacm_pac_fine':                   ; 80
           'dacm_pac_cm_fs_coarse':           ; 81
           'dacm_pac_cm_fs_fine':             ; 82
           'dacm_mcp_coarse':                 ; 83
           'dacm_mcp_fine':                   ; 84
           'dacm_mcp_cm_fs_coarse':           ; 85
           'dacm_mcp_cm_fs_fine':             ; 86
           'dacm_ssd_coarse':                 ; 87
           'dacm_ssd_fine':                   ; 88
           'dacm_esa_coarse':                 ; 89
           'dacm_esa_fine':                   ; 90
           'dacm_s_ch_coarse':                ; 91
           'dacm_s_ch_fine':                  ; 92
           'dacm_s_ch_tap_coarse':            ; 93
           'dacm_s_ch_tap_fine':              ; 94
           'dacm_dfl_1_coarse':               ; 95
           'dacm_dfl_1_fine':                 ; 96
           'dacm_dfl_1_dfine':                ; 97
           'dacm_dfl_2_coarse':               ; 98
           'dacm_dfl_2_fine':                 ; 99
           'dacm_dfl_2_dfine':                ; 100
           'dacr_pac_mcp_ssd_ctl_mcp_limit':  ; 101
           'dacr_pac_limit':                  ; 102
           'dacr_esa_ctl':                    ; 103
           'dacr_s_ch_ctl':                   ; 104
           'dacr_dfl_1_ctl':                  ; 105
           'dacr_dfl_2_ctl':                  ; 106
           'dacr_control':                    ; 107
           'dacr_status':                     ; 108
           'dacr_adc_lu_ctr':                 ; 109
           'dacr_dfl_1_offset':               ; 110
           'dacr_dfl_2_offset':               ; 111
           'dacr_c_pac_coarse':               ; 112
           'dacr_c_pac_fine':                 ; 113
           'dacr_c_pac_cm_fs_coarse':         ; 114
           'dacr_c_pac_cm_fs_fine':           ; 115
           'dacr_c_mcp_coarse':               ; 116
           'dacr_c_mcp_fine':                 ; 117
           'dacr_c_mcp_cm_fs_coarse':         ; 118
           'dacr_c_mcp_cm_fs_fine':           ; 119
           'dacr_c_ssd_coarse':               ; 120
           'dacr_c_ssd_fine':                 ; 121
           'dacr_c_esa_coarse':               ; 122
           'dacr_c_esa_fine':                 ; 123
           'dacr_c_s_ch_coarse':              ; 124
           'dacr_c_s_ch_fine':                ; 125
           'dacr_c_s_ch_tap_coarse':          ; 126
           'dacr_c_s_ch_tap_fine':            ; 127
           'dacr_dfl_1_coarse':               ; 128
           'dacr_dfl_1_fine':                 ; 129
           'dacr_dfl_1_dfine':                ; 130
           'dacr_dfl_2_coarse':               ; 131
           'dacr_dfl_2_fine':                 ; 132
           'dacr_dfl_2_dfine':                ; 133
           'sw_status1':                      ; 134
           'sw_hv_ramp_status':               ; 135
           'sw_hv_enable_status':             ; 136
           'sw_hv_limit_status':              ; 137
           'sw_eepromchksum':                 ; 138
           'hkp_sampletime':                  ; 139
           'pac_discharge_status':            ; 140
           'pac_discharge_cnt':               ; 141
           'pac_discharge_conseccntr':        ; 142
           'pac_discharge_safe_att':          ; 143
           'analog_limit_status':             ; 144
           'analog_lim_pt1_status':           ; 145
           'analog_lim_pt1_id':               ; 146
           'analog_lim_pt1_red':              ; 147
           'analog_lim_pt2_status':           ; 148
           'analog_lim_pt2_id':               ; 149
           'analog_lim_pt2_red':              ; 150
           'analog_lim_pt3_status':           ; 151
           'analog_lim_pt3_id':               ; 152
           'analog_lim_pt3_red':              ; 153
           'analog_lim_pt4_status':           ; 154
           'analog_lim_pt4_id':               ; 155
           'analog_lim_pt4_red':              ; 156
           'analog_lim_pt5_status':           ; 157
           'analog_lim_pt5_id':               ; 158
           'analog_lim_pt5_red':              ; 159
           'analog_lim_pt6_status':           ; 160
           'analog_lim_pt6_id':               ; 161
           'analog_lim_pt6_red':              ; 162
           'analog_lim_pt7_status':           ; 163
           'analog_lim_pt7_id':               ; 164
           'analog_lim_pt7_red':              ; 165
           'analog_lim_pt8_status':           ; 166
           'analog_lim_pt8_id':               ; 167
           'analog_lim_pt8_red':              ; 168
           'analog_lim_pt9_status':           ; 169
           'analog_lim_pt9_id':               ; 170
           'analog_lim_pt9_red':              ; 171
           'analog_lim_pt10_status':          ; 172
           'analog_lim_pt10_id':              ; 173
           'analog_lim_pt10_red':             ; 174
           'analog_lim_pt11_status':          ; 175
           'analog_lim_pt11_id':              ; 176
           'analog_lim_pt11_red':             ; 177
           'analog_lim_pt12_status':          ; 178
           'analog_lim_pt12_id':              ; 179
           'analog_lim_pt12_red':             ; 180
           'analog_lim_pt13_status':          ; 181
           'analog_lim_pt13_id':              ; 182
           'analog_lim_pt13_red':             ; 183
           'analog_lim_pt14_status':          ; 184
           'analog_lim_pt14_id':              ; 185
           'analog_lim_pt14_red':             ; 186
           'analog_lim_pt15_status':          ; 187
           'analog_lim_pt15_id':              ; 188
           'analog_lim_pt15_red':             ; 189
           'opmode':                          ; 190
           'sw_mode':                         ; 191
           'proton_mode':                     ; 192
           'active_supplyid':                 ; 193
           'supply_coarse_rb':                ; 194
           'supply_fine_rb':                  ; 195
           'swmaint_supplystat':              ; 196
           'swmaint_pac_coarse':              ; 197
           'swmaint_pac_fine': $              ; 198		
	IF dig_hk_read EQ 0 THEN read_dig_hk, cdf_id, sat, dig_hk_read, data1_index, epoch1info.maxrec, products
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_dig_hk_index_a = data1_index 
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_dig_hk_index_b = data1_index
ENDELSE 
ENDIF


; ANALOG HOUSEKEEPING
;---------------------
IF crib_name EQ 'analog_hk' OR crib_name EQ 'daily_hk' THEN BEGIN

hk_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	'lrnm_spare' : ;0
	'lvc_-12v': ;1
	'lvc_-5v': ;2
	'esa_vm_pos': ;3
	'dfl_1_vm': ;4
	'dfl_1_tap': ;5
	'dfl_2_vm': ;6
	'dfl_2_tap': ;7
	'lvc_+2.5v_b': ;8
	'pac_cm_dc': ;9
	'lvc_+12v': ;10
	'lrpm_spare': ;11
	'lvc_+2.5v_a': ;12
	'lvc_+5v': ;13
	'adc_avdd': ;14
	'adc_dvdd': ;15
	'pac_vm': ;16
	'cal_vref': ;17
	'pac_cm_ac': ;18
	'mcp_vm': ;19
	'mcp_cm_dc': ;20
	'mcp_cm_ac': ;21
	'ssd_cm_dc': ;22
	's_ch_vm': ;23
	's_ch_vm_tap': ;24
	's_ch_cm_dc': ;25
	'esa_vm_neg': ;26
	'esa_cm_dc': ;27
	'dfl_1_cm_dc': ;28
	'dfl_2_cm_dc': ;29
	'lvc_+2.5v_b_imon': ;30
	'lvc_+12v_imon': ;31
	'lvc_-12v_imon': ;32
	'lvc_+5v_imon': ;33
	'lvc_-5v_imon': ;34
	'lvc_+2.5v_a_imon': ;35
	'ssd_vm': ;36
	'adc_lu_flag': ;37
	'plug_id': ;38
	'adc_lu_ctr': ;39
	'dac_status': ;40
	'adc0_agnd': ;41
	'adc1_agnd': ;42
	'ra_e0': ;43
	'ra_e1': ;44
	'tac0_tsp': ;45
	'tac2_tsp': ;46
	'sb0_tsp': ;47
	'sb1_tsp': ;48
	'tof_hv0_tsp': ;49
	'tof_hv1_tsp': ;50
	's_c_0_tsp': ;51
	's_c_1_tsp': ;52
	'lvc0_tsp': ;53
	'lvc1_tsp': ;54
	'adc0_vref': ;55
	'adc1_vref': ;56
	'ssd_status': ;57
	'ssd_v_pos_analog': ;58
	'ssd_v_neg_analog': ;59
	'ssd_hv_bias': ;60
	'ssd_tc0': ;61
	'ssd_tc1': ;62
	'ssd_tc2': ;63
	'ssd_v_pos_dig': IF hk_initialized EQ 0 THEN init_hk, sat, num_records1, hk_initialized
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch_analog', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch_analog', temp_epoch, rec_count = epoch1info.maxrec, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+2]-1)] = $
                              converted_epoch[data1_index[kk+1]:(data1_index[kk+1]+data1_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR 
                    ENDIF
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_hk_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_hk_a = epoch5
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_hk_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_hk_b = epoch5 
        ENDELSE 

       ; get other data
        hk_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		'lrnm_spare' : ;0
		'lvc_-12v': ;1
		'lvc_-5v': ;2
		'esa_vm_pos': ;3
		'dfl_1_vm': ;4
		'dfl_1_tap': ;5
		'dfl_2_vm': ;6
		'dfl_2_tap': ;7
		'lvc_+2.5v_b': ;8
		'pac_cm_dc': ;9
		'lvc_+12v': ;10
		'lrpm_spare': ;11
		'lvc_+2.5v_a': ;12
		'lvc_+5v': ;13
		'adc_avdd': ;14
		'adc_dvdd': ;15
		'pac_vm': ;16
		'cal_vref': ;17
		'pac_cm_ac': ;18
		'mcp_vm': ;19
		'mcp_cm_dc': ;20
		'mcp_cm_ac': ;21
		'ssd_cm_dc': ;22
		's_ch_vm': ;23
		's_ch_vm_tap': ;24
		's_ch_cm_dc': ;25
		'esa_vm_neg': ;26
		'esa_cm_dc': ;27
		'dfl_1_cm_dc': ;28
		'dfl_2_cm_dc': ;29
		'lvc_+2.5v_b_imon': ;30
		'lvc_+12v_imon': ;31
		'lvc_-12v_imon': ;32
		'lvc_+5v_imon': ;33
		'lvc_-5v_imon': ;34
		'lvc_+2.5v_a_imon': ;35
		'ssd_vm': ;36
		'adc_lu_flag': ;37
		'plug_id': ;38
		'adc_lu_ctr': ;39
		'dac_status': ;40
		'adc0_agnd': ;41
		'adc1_agnd': ;42
		'ra_e0': ;43
		'ra_e1': ;44
		'tac0_tsp': ;45
		'tac2_tsp': ;46
		'sb0_tsp': ;47
		'sb1_tsp': ;48
		'tof_hv0_tsp': ;49
		'tof_hv1_tsp': ;50
		's_c_0_tsp': ;51
		's_c_1_tsp': ;52
		'lvc0_tsp': ;53
		'lvc1_tsp': ;54
		'adc0_vref': ;55
		'adc1_vref': ;56
		'ssd_status': ;57
		'ssd_v_pos_analog': ;58
		'ssd_v_neg_analog': ;59
		'ssd_hv_bias': ;60
		'ssd_tc0': ;61
		'ssd_tc1': ;62
		'ssd_tc2': ;63
		'ssd_v_pos_dig': IF hk_read EQ 0 THEN read_hk, cdf_id, sat, hk_read, data1_index, epoch1info.maxrec, products
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN
	data1_hk_index_a = data1_index 
ENDIF ELSE BEGIN
	data1_hk_index_b = data1_index
ENDELSE 

ENDIF


; FULL RESOLUTION RATE
;------------------------
IF crib_name EQ 'full_monitor_rates' THEN BEGIN

mon_full_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	's_valid' : ;0
	's_e_not_req': ;1
	's_e_req': ;2
	's_no_pos': ;3
	's_mult_pos': ;4
	's_no_e': ;5
	's_mult_e': ;6
	'ra_sat_a': ;7
	'ra_sat_b': ;8
	'ra_sat_both': ;9
	'ssd_sw': ;10
	'ssd_st': ;11
	'sf0': ;12
	'sfr0': ;13
	'stp0': ;14
	'ra_trig': ;15
	'pos1_0': ;16
	'pos1_1': ;17
	'w_no_pos': ;18
	'w_mult_pos': ;19
	'w_valid': ;20
	'sf2': ;21
	'sfr2': ;22
	'stp2': ;23
	'pos2_0': ;24
	'pos2_1': ;25
	'pos2_2': ;26
	'pos2_3': ;27
	'pos3_0': ;28
	'pos3_1': ;29
	'pos3_2': ;30
	'pos3_3': ;31
	'stop0':
	'start0':
	'stop2':
	'start2':
	'pos2':
	'pos3': IF mon_full_initialized EQ 0 THEN init_mon_full, sat, num_records1, mon_full_initialized, E_step, products
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch1', temp_epoch, rec_count = epoch1info.maxrec, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices, data1_index, converted_epoch, which_epoch, ii
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+2]-1)] = $
                              converted_epoch[data1_index[kk+1]:(data1_index[kk+1]+data1_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR 
                    ENDIF
		remove_fill, -1E31, epoch1
               END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_full_a = epoch1 
          ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_full_b = epoch1  
        ENDELSE 

        ; get other data
        mon_full_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		's_valid' : ;0
		's_e_not_req': ;1
		's_e_req': ;2
		's_no_pos': ;3
		's_mult_pos': ;4
		's_no_e': ;5
		's_mult_e': ;6
		'ra_sat_a': ;7
		'ra_sat_b': ;8
		'ra_sat_both': ;9
		'ssd_sw': ;10
		'ssd_st': ;11
		'sf0': ;12
		'sfr0': ;13
		'stp0': ;14
		'ra_trig': ;15
		'pos1_0': ;16
		'pos1_1': ;17
		'w_no_pos': ;18
		'w_mult_pos': ;19
		'w_valid': ;20
		'sf2': ;21
		'sfr2': ;22
		'stp2': ;23
		'pos2_0': ;24
		'pos2_1': ;25
		'pos2_2': ;26
		'pos2_3': ;27
		'pos3_0': ;28
		'pos3_1': ;29
		'pos3_2': ;30
		'pos3_3': ;31
		'stop0':
		'start0':
		'stop2':
		'start2':
		'pos2':
		'pos3': $
	IF mon_full_read EQ 0 THEN read_mon_full, cdf_id, sat, mon_full_read, data1_index, epoch1info.maxrec, products, num_records1, E_step
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_full_index_a = data1_index 
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_full_index_b = data1_index
ENDELSE 

ENDIF

; MONITOR RATES
;-----------------
IF crib_name EQ 'norm_monitor_rates' OR crib_name EQ 'set_monitor_rates' THEN BEGIN
data_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
	's_valid' : ;0
	's_e_not_req': ;1
	's_e_req': ;2
	's_no_pos': ;3
	's_mult_pos': ;4
	's_no_e': ;5
	's_mult_e': ;6
	'ra_sat_a': ;7
	'ra_sat_b': ;8
	'ra_sat_both': ;9
	'ssd_sw': ;10
	'ssd_st': ;11
	'sf0': ;12
	'sfr0': ;13
	'stp0': ;14
	'ra_trig': ;15
	'pos1_0': ;16
	'pos1_1': ;17
	'w_no_pos': ;18
	'w_mult_pos': ;19
	'w_valid': ;20
	'sf2': ;21
	'sfr2': ;22
	'stp2': ;23
	'pos2_0': ;24
	'pos2_1': ;25
	'pos2_2': ;26
	'pos2_3': ;27
	'pos3_0': ;28
	'pos3_1': ;29
	'pos3_2': ;30
	'pos3_3': ;31
	'stop0':
	'start0':
	'stop2':
	'start2': 
	'pos1':
	'pos2':
	'pos3':
	'sf0ra_trig': $
	IF data_initialized EQ 0 THEN init_data, sat, num_records1, num_records5, data_initialized, E_step, products, NO_ENG
    ENDSWITCH 
ENDFOR 

    
; read the data

print, 'read data section started'
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0L, n_elements(which_epoch)-1 DO BEGIN 
            
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
                    IF epoch1info.maxrec GT 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch1', temp_epoch, rec_count = epoch1info.maxrec, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices, data1_index, converted_epoch, which_epoch, ii
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+2]-1)] = $
                              converted_epoch[data1_index[kk+1]:(data1_index[kk+1]+data1_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR 
                    ENDIF ELSE files[ii] = ''
		remove_fill, -1, epoch1
		cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5_mon', /zvariable
                    IF epoch5info.maxrec GT 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch5_mon', temp_epoch, rec_count = epoch5info.maxrec, /zvariable
			converted_epoch5 = time_double(temp_epoch, /epoch)
                        check_indices, data5_index, converted_epoch5, 5, ii
                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+2]-1)] = $
                              converted_epoch5[data5_index[kk+1]:(data5_index[kk+1]+data5_index[kk+2]-1)]
                            kk = kk+2
                        ENDFOR 
                    ENDIF ELSE files[ii] = ''
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_a = epoch5
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
        ENDELSE 

 ; get other data
print, 'getting other data'
data_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
   		's_valid' : ;0
		's_e_not_req': ;1
		's_e_req': ;2
		's_no_pos': ;3
		's_mult_pos': ;4
		's_no_e': ;5
		's_mult_e': ;6
		'ra_sat_a': ;7
		'ra_sat_b': ;8
		'ra_sat_both': ;9
		'ssd_sw': ;10
		'ssd_st': ;11
		'sf0': ;12
		'sfr0': ;13
		'stp0': ;14
		'ra_trig': ;15
		'pos1_0': ;16
		'pos1_1': ;17
		'w_no_pos': ;18
		'w_mult_pos': ;19
		'w_valid': ;20
		'sf2': ;21
		'sfr2': ;22
		'stp2': ;23
		'pos2_0': ;24
		'pos2_1': ;25
		'pos2_2': ;26
		'pos2_3': ;27
		'pos3_0': ;28
		'pos3_1': ;29
		'pos3_2': ;30
		'pos3_3': ;31
		'stop0':
		'start0':
		'stop2':
		'start2': 
		'pos1':
		'pos2':
		'pos3':
		'sf0ra_trig': $
		IF data_read EQ 0 THEN read_data, cdf_id, sat, data_read, data5_index, epoch5info.maxrec, data1_index, epoch1info.maxrec, products(jj), num_records1, num_records5, E_step, NO_ENG
        	ELSE: BREAK
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
    IF need_epoch5 EQ 5 THEN data5_index_a = data5_index
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
    IF need_epoch5 EQ 5 THEN data5_index_b = data5_index
ENDELSE 

ENDIF

END 
