;+
;PROCEDURE: pl_read_cdf
;PURPOSE:
;  Read data from cdf files.
;
;PARAMETERS:   in:  sat        : 'A' or 'B'
;                   files      : Array of file paths to read from (strings)
;                   products   : Array of Products wanted (strings)
;                   which_epoch: 0(for variable), 1, or 5 minute resolution
;
;CREATED BY: L. Ellis
;
;LAST MODIFICATION: 09/18/2006
;
;MODIFICATION HISTORY:
;    09/18/2006: Created
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;-
PRO read_product, cdf_id, prod_name, data_var, data_index, epoch_maxrec
; prod_name: string with cdf variable name 
; data_var: the common variable in which to put the data
; data_index: the start index for records in the data_var
; epoch_maxrec: number of records we expect 
cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
IF varinfo.maxrec+1 GE epoch_maxrec THEN maxrec = varinfo.maxrec ELSE maxrec = epoch_maxrec
cdf_varget, cdf_id, prod_name, temp_data, rec_count = maxrec, /zvariable
num_dim = size(data_var, /n_dimensions)
CASE num_dim OF 
    1: BEGIN
        temp_index = 0
        FOR kk = 0, n_elements(data_index)-1 DO BEGIN 
            data_var[data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    2: BEGIN
        temp_index = 0
        FOR kk = 0, n_elements(data_index)-1 DO BEGIN 
            data_var[*, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    3: BEGIN
        temp_index = 0
        FOR kk = 0, n_elements(data_index)-1 DO BEGIN 
            data_var[*, *, data_index[kk]:(data_index[kk]+data_index[kk+1]-1)] = temp_data[*, *, temp_index:(temp_index+data_index[kk+1]-1)]
            temp_index = temp_index+data_index[kk+1]
            kk = kk+1
        ENDFOR 
    END 
    ELSE: BEGIN 
        print, "Lorna: Add more dimensions"
        stop
    END
ENDCASE 
END 

PRO remove_fill, fill_val, data, error
; fill_val: fill_value to be replaced by NaN
; data: array of data values
; error: array that matches data array, 1 means error
indice = where(data EQ fill_val, count)
FOR ii = 0, count-1 DO BEGIN
    data[indice[ii]] = !values.f_nan
ENDFOR 
indice = where(error EQ 1, count)
FOR ii = 0, count-1 DO BEGIN
    data[indice[ii]] = !values.f_nan
ENDFOR 
END 


PRO init_mom, sat, num_records1, mom_initialized
; initialize all the moments variabales (in com_mom)
COMMON com_moment
CASE sat OF
    'A': BEGIN 
        mom_meta_a             = intarr(4, num_records1)
        mom_density_main_a     = dblarr(num_records1)
        mom_density_s_a        = dblarr(num_records1)
        mom_velocity_main_a    = dblarr(3, num_records1)
        mom_velocity_s_a       = dblarr(3, num_records1)
        mom_heat_flux_main_a   = dblarr(3, num_records1)
        mom_heat_flux_s_a      = dblarr(3, num_records1)
        mom_temperature_main_a = dblarr(6, num_records1)
        mom_temperature_s_a    = dblarr(6, num_records1)
        error1_a               = intarr(num_records1)
        mom_meta_a[*, *]             = !values.f_nan
        mom_density_main_a[*]        = !values.f_nan
        mom_density_s_a[*]           = !values.f_nan
        mom_velocity_main_a[*, *]    = !values.f_nan
        mom_velocity_s_a[*, *]       = !values.f_nan
        mom_heat_flux_main_a[*, *]   = !values.f_nan
        mom_heat_flux_s_a[*, *]      = !values.f_nan
        mom_temperature_main_a[*, *] = !values.f_nan
        mom_temperature_s_a[*, *]    = !values.f_nan
    END 
    'B': BEGIN 
        mom_meta_b             = intarr(4, num_records1)
        mom_density_main_b     = dblarr(num_records1)
        mom_density_s_b        = dblarr(num_records1)
        mom_velocity_main_b    = dblarr(3, num_records1)
        mom_velocity_s_b       = dblarr(3, num_records1)
        mom_heat_flux_main_b   = dblarr(3, num_records1)
        mom_heat_flux_s_b      = dblarr(3, num_records1)
        mom_temperature_main_b = dblarr(6, num_records1)
        mom_temperature_s_b    = dblarr(6, num_records1)
        error1_b               = intarr(num_records1)
        mom_meta_b[*, *]             = !values.f_nan
        mom_density_main_b[*]        = !values.f_nan
        mom_density_s_b[*]           = !values.f_nan
        mom_velocity_main_b[*, *]    = !values.f_nan
        mom_velocity_s_b[*, *]       = !values.f_nan
        mom_heat_flux_main_b[*, *]   = !values.f_nan
        mom_heat_flux_s_b[*, *]      = !values.f_nan
        mom_temperature_main_b[*, *] = !values.f_nan
        mom_temperature_s_b[*, *]    = !values.f_nan
    END 
ENDCASE 
mom_initialized = 1
END 

PRO read_moments, cdf_id, sat, mom_read, data1_index, epoch1maxrec
COMMON com_moment
CASE sat OF 
    'A': BEGIN
        read_product, cdf_id, 'moment_meta',      mom_meta_a,             data1_index, epoch1maxrec
        read_product, cdf_id, 'density_main',     mom_density_main_a,     data1_index, epoch1maxrec
        read_product, cdf_id, 'density_s',        mom_density_s_a,        data1_index, epoch1maxrec
        read_product, cdf_id, 'velocity_main',    mom_velocity_main_a,    data1_index, epoch1maxrec
        read_product, cdf_id, 'velocity_s',       mom_velocity_s_a,       data1_index, epoch1maxrec
        read_product, cdf_id, 'heat_flux_main',   mom_heat_flux_main_a,   data1_index, epoch1maxrec
        read_product, cdf_id, 'heat_flux_s',      mom_heat_flux_s_a,      data1_index, epoch1maxrec
        read_product, cdf_id, 'temperature_main', mom_temperature_main_a, data1_index, epoch1maxrec
        read_product, cdf_id, 'temperature_s',    mom_temperature_s_a,    data1_index, epoch1maxrec
        read_product, cdf_id, 'error1',           error1_a,               data1_index, epoch1maxrec
        remove_fill, -1D, mom_meta_a, error1_a
        remove_fill, -1D, mom_density_main_a, error1_a
        remove_fill, -1D, mom_density_s_a, error1_a
        remove_fill, -1D, mom_velocity_main_a, error1_a
        remove_fill, -1D, mom_velocity_s_a, error1_a
        remove_fill, -1D, mom_heat_flux_main_a, error1_a
        remove_fill, -1D, mom_heat_flux_s_a, error1_a
        remove_fill, -1D, mom_temperature_main_a, error1_a
        remove_fill, -1D, mom_temperature_s_a, error1_a
    END
    'B': BEGIN
        read_product, cdf_id, 'moment_meta',      mom_meta_b,             data1_index, epoch1maxrec
        read_product, cdf_id, 'density_main',     mom_density_main_b,     data1_index, epoch1maxrec
        read_product, cdf_id, 'density_s',        mom_density_s_b,        data1_index, epoch1maxrec
        read_product, cdf_id, 'velocity_main',    mom_velocity_main_b,    data1_index, epoch1maxrec
        read_product, cdf_id, 'velocity_s',       mom_velocity_s_b,       data1_index, epoch1maxrec
        read_product, cdf_id, 'heat_flux_main',   mom_heat_flux_main_b,   data1_index, epoch1maxrec
        read_product, cdf_id, 'heat_flux_s',      mom_heat_flux_s_b,      data1_index, epoch1maxrec
        read_product, cdf_id, 'temperature_main', mom_temperature_main_b, data1_index, epoch1maxrec
        read_product, cdf_id, 'temperature_s',    mom_temperature_s_b,    data1_index, epoch1maxrec
        read_product, cdf_id, 'error1',           error1_b,               data1_index, epoch1maxrec
        remove_fill, -1D, mom_meta_b, error1_b
        remove_fill, -1D, mom_density_main_b, error1_b
        remove_fill, -1D, mom_density_s_b, error1_b
        remove_fill, -1D, mom_velocity_main_b, error1_b
        remove_fill, -1D, mom_velocity_s_b, error1_b
        remove_fill, -1D, mom_heat_flux_main_b, error1_b
        remove_fill, -1D, mom_heat_flux_s_b, error1_b
        remove_fill, -1D, mom_temperature_main_b, error1_b
        remove_fill, -1D, mom_temperature_s_b, error1_b
    END
ENDCASE
mom_read = 1
END 

PRO check_indices, index, epoch, resolution, file_num
; checks for missing records from the cdf epoch variable
; index: array of pairs [start_index,num_records] (could be [0,10,15,2] for times 0-9,15-16)
; epoch: array of time values read from cdf (and converted)
; resolution: 1 or 5 depending on epoch (may need to add variable)
; file_num: index of this file in the file array
first_index = file_num*(1440/resolution)
num_records = 0
in_valid_area = 0
ii = 0
index = [-1]
CASE resolution OF 
    1: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 59 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF hour EQ time.hour AND min EQ time.min THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*60)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]                
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    5: BEGIN 
        FOR hour = 0, 23 DO BEGIN 
            FOR min = 0, 11 DO BEGIN 
                IF ii LT n_elements(epoch) THEN BEGIN 
                    time = time_struct(epoch[ii])
                    IF (hour EQ time.hour AND time.min GE (min*5) AND time.min LT (min*5+5)) OR $
                      (time.hour EQ hour+1 AND time.min LE 4 AND min EQ 11) THEN BEGIN 
                        IF in_valid_area EQ 0 THEN BEGIN
                            start_index = first_index+(hour*12)+min
                            num_records = 1
                            in_valid_area = 1
                        ENDIF ELSE BEGIN ; already in valid area
                            num_records = num_records+1
                        ENDELSE 
                        ii = ii+1
                    ENDIF ELSE BEGIN
                        IF in_valid_area EQ 1 THEN BEGIN
                            in_valid_area = 0
                            IF index[0] EQ -1 THEN index = [start_index, num_records] $
                            ELSE index = [index, start_index, num_records]
                        ENDIF 
                    ENDELSE 
                ENDIF ELSE BEGIN 
                    IF in_valid_area EQ 1 THEN BEGIN
                        in_valid_area = 0
                        IF index[0] EQ -1 THEN index = [start_index, num_records] $
                        ELSE index = [index, start_index, num_records]
                    ENDIF 
                ENDELSE 
            ENDFOR 
        ENDFOR 
    END
    ELSE: BEGIN
        print, "Lorna -- add variable resolution"
        stop
    END 
ENDCASE 
END 


PRO pl_read_cdf, sat, files, products, which_epoch
COMMON share1
COMMON com_moment
; check if there are any files to read
IF n_elements(products) GT 0 AND n_elements(files) EQ 0 THEN print, "No Data Files in this Time Range for Satellite ", sat

; initialize output variables (from common block)
need_epoch1 = 0
need_epoch5 = 0
FOR ii = 0, n_elements(which_epoch)-1 DO BEGIN 
    CASE which_epoch[ii] OF 
        1: BEGIN 
            num_records1 = n_elements(files)*1441
            epoch1 = dblarr(num_records1)
            epoch1(*) = !values.f_nan
            need_epoch1 = 1
        END 
        5: BEGIN 
            num_records5 = n_elements(files)*289
            epoch5 = dblarr(num_records5)
            epoch5(*) = !values.f_nan
            need_epoch5 = 1
        END 
        0: BEGIN 
            print, "Lorna: add variable epoch"
            stop
        END 
        ELSE: BEGIN
            print, "Invalid indication of which epoch is needed: ", which_epoch
            stop
        END
    ENDCASE 
ENDFOR 
mom_initialized = 0
FOR ii = 0, n_elements(products)-1 DO BEGIN 
    SWITCH products[ii] OF 
        'D'    : 
        'VX'   : 
        'VY'   : 
        'VZ'   : 
        'VXYZ' : 
        'VT'   : 
        'TXX'   : 
        'TXY'   : 
        'TXZ'   : 
        'TYY'   : 
        'TYZ'   : 
        'TZZ'   : 
        'PXX'   : 
        'PXY'   : 
        'PXZ'   : 
        'PYY'   : 
        'PYZ'   : 
        'PZZ'   : 
        'EX'   : 
        'EY'   : 
        'EZ'   : 
        'HX'   : 
        'HY'   : 
        'HZ'   : IF mom_initialized EQ 0 THEN init_mom, sat, num_records1, mom_initialized
    ENDSWITCH 
ENDFOR 
    
; read data
FOR ii = 0, n_elements(files)-1 DO BEGIN 
    IF strcmp(files[ii], '') NE 1 THEN BEGIN 
        cdf_id = cdf_open(files[ii])
                                ; get epochs
        FOR jj = 0, n_elements(which_epoch)-1 DO BEGIN 
            CASE which_epoch[jj] OF 
                1: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch1info, var = 'epoch1', /zvariable
                    IF epoch1info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch1', temp_epoch, rec_count = epoch1info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices, data1_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data1_index)-1 DO BEGIN 
                            epoch1[data1_index[kk]:(data1_index[kk]+data1_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data1_index[kk+1]-1)]
                            converted_index = converted_index+data1_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF
                END 
                5: BEGIN 
                    cdf_control, cdf_id, get_var_info = epoch5info, var = 'epoch5', /zvariable
                    IF epoch5info.maxrec GE 0 THEN BEGIN 
                        cdf_varget, cdf_id, 'epoch5', temp_epoch, rec_count = epoch5info.maxrec+1, /zvariable
                        converted_epoch = time_double(temp_epoch, /epoch)
                        check_indices, data5_index, converted_epoch, which_epoch, ii
                        converted_index = 0
                        FOR kk = 0, n_elements(data5_index)-1 DO BEGIN 
                            epoch5[data5_index[kk]:(data5_index[kk]+data5_index[kk+1]-1)] = $
                              converted_epoch[converted_index:(converted_index+data5_index[kk+1]-1)]
                            converted_index = converted_index+data5_index[kk+1]
                            kk = kk+1
                        ENDFOR 
                    ENDIF 
                END 
                0: BEGIN 
                    print, "Lorna: add variable epoch"
                    stop
                END 
                ELSE: BEGIN
                    print, "Invalid indication of which epoch is needed: ", which_epoch
                    stop
                END
            ENDCASE 
        ENDFOR 
        IF sat EQ 'A' THEN BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_a = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_a = epoch5 
        ENDIF ELSE BEGIN 
            IF need_epoch1 EQ 1 THEN epoch1_b = epoch1 
            IF need_epoch5 EQ 5 THEN epoch5_b = epoch5 
        ENDELSE 

                                ; get other data
        mom_read = 0
        FOR jj = 0, n_elements(products)-1 DO BEGIN 
            SWITCH products[jj] OF
                'D'   : 
                'VX'  : 
                'VY'  : 
                'VZ'  : 
                'VXYZ': 
                'VT  ': 
                'TXX'   : 
                'TXY'   : 
                'TXZ'   : 
                'TYY'   : 
                'TYZ'   : 
                'TZZ'   : 
                'PXX'   : 
                'PXY'   : 
                'PXZ'   : 
                'PYY'   : 
                'PYZ'   : 
                'PZZ'   : 
                'EX'   : 
                'EY'   : 
                'EZ'   : 
                'HX'   : 
                'HY'   : 
                'HZ'   : BEGIN 
                    IF mom_read EQ 0 THEN read_moments, cdf_id, sat, mom_read, data1_index, epoch1info.maxrec+1
                    BREAK
                END 
                ELSE: BREAK 
            ENDSWITCH 
        ENDFOR 
    ENDIF 
ENDFOR 
IF sat EQ 'A' THEN BEGIN 
    IF need_epoch1 EQ 1 THEN data1_index_a = data1_index 
    IF need_epoch5 EQ 1 THEN data5_index_a = data5_index 
ENDIF ELSE BEGIN
    IF need_epoch1 EQ 1 THEN data1_index_b = data1_index
    IF need_epoch5 EQ 1 THEN data5_index_b = data5_index
ENDELSE 
END 
