;+
; PROCEDURE: pl_obmom
;
; DESCRIPTION:
;	computes OB moments for the STEREO PLASTIC instrument
;
; INPUT:
;       PROD -> product number
;	MOMENTS -> moments to compute
;	NAME -> tplot structure name
;	ENERGY -> energy range
;	ANGLE -> angle range
;       TRANS -> coordinate transformation direction
;
; CREATED BY: Lorna Ellis
;
; LAST MODIFICATION: 09/14/06
;
; MODIFICATION HISTORY:
;    09/14/06 Modified from Chris Mouikis's codif_obmom.pro
;-

FUNCTION compute_ob_density, dat, inst, specie

  IF inst EQ 0 THEN BEGIN ; CODIF

    ;sum over 3 energy ranges:
    density = TOTAL(dat.data(0,specie,*,*),3)

  ENDIF ELSE BEGIN

    density = dat.data(0,0,*) ; HIA

  ENDELSE

  density = REFORM(density)

  RETURN, density

END

FUNCTION compute_ob_velocity, dat, inst, specie

  density = compute_ob_density(dat,inst,specie)#[1.,1.,1.]

  IF inst EQ 0 THEN BEGIN ; CODIF

    V = TRANSPOSE(REFORM(TOTAL(dat.data(1:3,specie,*,*),3)))/density

  ENDIF ELSE BEGIN ; HIA

    V = TRANSPOSE(REFORM(dat.data(1:3,0,*)))/density

  ENDELSE

  RETURN, V
END


FUNCTION compute_ob_pressure, dat, inst, specie
  
  THERMAL=1
  MK2nPa = 0.013806  ; pressure is in MK/cm^3, convert to nPa
  mass=[1.0079,4002.,4.002,15.999]*1.6605e-6 ; [10^-21 kg]

  packets=n_elements(dat.time)
  pressure = FLTARR(packets,9)

  IF inst EQ 0 THEN BEGIN ; CODIF

    thisd=REFORM(dat.data(*,specie,*,*))
    P0 = thisd(7:12,*,*)*MK2nPa;[nPa]

    IF KEYWORD_SET(thermal) THEN BEGIN ; subtract dynamic pressure:
      ; this bit is questionable:
      density = compute_ob_density(dat,inst,specie)
      mdens =  mass(specie)/thisd(0,*,*)
      ;FOR i=0,2 DO mdens(0,i,*)=mdens(0,i,*)/(thisd(0,i,*)*thisd(0,i,*));(density*density)
      mflux=thisd(1:3,*,*)

      P = FLTARR(packets,6)
      P(*,0) = TRANSPOSE(TOTAL(P0(0,*,*) -mflux(0,*,*)*mflux(0,*,*)*mdens(0,*,*),2)) ;[nPa]
      P(*,1) = TRANSPOSE(TOTAL(P0(1,*,*) -mflux(1,*,*)*mflux(1,*,*)*mdens(0,*,*),2)) ;[nPa]
      P(*,2) = TRANSPOSE(TOTAL(P0(2,*,*) -mflux(2,*,*)*mflux(2,*,*)*mdens(0,*,*),2)) ;[nPa]
      P(*,3) = TRANSPOSE(TOTAL(P0(3,*,*) -mflux(0,*,*)*mflux(1,*,*)*mdens(0,*,*),2)) ;[nPa]
      P(*,4) = TRANSPOSE(TOTAL(P0(4,*,*) -mflux(0,*,*)*mflux(2,*,*)*mdens(0,*,*),2)) ;[nPa]
      P(*,5) = TRANSPOSE(TOTAL(P0(5,*,*) -mflux(1,*,*)*mflux(2,*,*)*mdens(0,*,*),2)) ;[nPa]

    ENDIF ELSE P=TRANSPOSE(TOTAL(P0,2))


  ENDIF ELSE BEGIN ; HIA

    P = TRANSPOSE(REFORM(dat.data(7:12,0,*)))*MK2nPa;[nPa]

    IF KEYWORD_SET(thermal) THEN BEGIN ; subtract dynamic pressure:

      density = compute_ob_density(dat,inst,specie)
      velocity = compute_ob_velocity(dat,inst,specie)
      mdens=density*mass(specie)

      P(*,0) = P(*,0) -velocity(*,0)*velocity(*,0)*mdens ;[nPa]
      P(*,1) = P(*,1) -velocity(*,1)*velocity(*,1)*mdens
      P(*,2) = P(*,2) -velocity(*,2)*velocity(*,2)*mdens
      P(*,3) = P(*,3) -velocity(*,0)*velocity(*,1)*mdens
      P(*,4) = P(*,4) -velocity(*,0)*velocity(*,2)*mdens
      P(*,5) = P(*,5) -velocity(*,1)*velocity(*,2)*mdens

    ENDIF

  ENDELSE

  ; Diagonalize pressure matrix for the calculation of Ppar, Pperp
  eig_val = FLTARR(packets,3)
  eig_vec = FLTARR(packets,3)

  FOR i= 0, packets-1 DO BEGIN
    tt = WHERE(FINITE(P(i,*)) EQ 0, itt)
    IF itt EQ 0 THEN BEGIN
      mat_diag, P(i,*), EIG_VAL= val, EIG_VEC= vec
      eig_val(i,*) = val
      eig_vec(i,*) = vec(*,0)
    ENDIF
  ENDFOR

  pressure(*,0:5) = P
  pressure(*,6:8) = eig_val

  RETURN, pressure

END


PRO pl_obmom, sat, name, prod, moments, $
              INST_COORD = INST_COORD

  COMMON get_error, get_err_no, get_err_msg, default_verbose
  COMMON tplot_com
  stop
  get_err_no = 0

  ;--------------------------------------------------------------------
  ; Read data
  ;--------------------------------------------------------------------
  dat = get_pl_data(prod, sat)
  
  IF get_err_no GT 0 THEN RETURN ; if product was not found, return

  mid_times = dat.time + (dat.end_time - dat.time)/2d
  packets = N_ELEMENTS(dat.time)

  ; CODIF: The HS geometric factor is included in the data. Here this
  ; factor is removed and the right geometric factor is applied
  IF inst EQ 0 THEN BEGIN
    FOR i1 = 0l, packets-1 DO BEGIN
      dat.data(*,*,*,i1) = 0.00239 * dat.data(*,*,*,i1) / $
        (dat.geom_factor(i1) * dat.eff(i1,specie))
    ENDFOR
  ENDIF


  CASE STRMID(moments,0,1) OF
    'D':BEGIN

      density = compute_ob_density(dat, inst, specie)

      datastr = {x:mid_times, y:density}
      labels=''
      store_data, name, data=datastr, $
        dlim={ylog:1,labels:labels,panel_size:2.}

    END
    'V':BEGIN

      vel = compute_ob_velocity(dat, inst, specie)

      datastr = {x:mid_times, y:vel}
      labels=['X','Y','Z']
      store_data, name, data=datastr, $
        dlim={inst_num:inst, sens:dat.sensitivity, sat:sat, $
              phase_instr:dat.phase_instr,labels:labels,panel_size:2.}

      IF NOT KEYWORD_SET(INST_COORD) THEN BEGIN
        IF inst EQ 0 THEN BEGIN
          cis_coord_trans, DATA_IN = name, TRANS='CODIF->GSE', $
            DATA_OUT = name
        ENDIF ELSE BEGIN
          cis_coord_trans, DATA_IN = name, TRANS='CODIF->GSE', $
            DATA_OUT = name
        ENDELSE
      ENDIF

    END

    'P': BEGIN
      P = compute_ob_pressure(dat,inst,specie)

      datastr = {x:mid_times, y:P(*,0:5)}
      labels = ['XX','YY','ZZ','XY','XZ','YZ']
      store_data, name, data=datastr, $
        dlim={inst_num:inst, sens:dat.sensitivity, sat:sat, $
	phase_instr:dat.phase_instr, labels:labels, panel_size:2}

      labels = ['']
      datastr = {x:mid_times, y:REFORM(P(*,6))}
      store_data, name+'_X', data=datastr, $
        dlim={inst_num:inst, sens:dat.sensitivity, sat:sat, $
	phase_instr:dat.phase_instr, labels:labels, panel_size:2}

      labels = ['']
      datastr = {x:mid_times, y:REFORM(P(*,7))}
      store_data, name+'_Y', data=datastr, $
        dlim={inst_num:inst, sens:dat.sensitivity, sat:sat, $
	phase_instr:dat.phase_instr, labels:labels, panel_size:2}

      labels = ['']
      datastr = {x:mid_times, y:REFORM(P(*,8))}
      store_data, name+'_Z', data=datastr, $
        dlim={inst_num:inst, sens:dat.sensitivity, sat:sat, $
	phase_instr:dat.phase_instr, labels:labels, panel_size:2}

      labels = ['']
      datastr = {x:mid_times, y:TOTAL((P(*,6:8)),2)/3.}
      store_data, name+'_T', data=datastr, $
        dlim={inst_num:inst, sens:dat.sensitivity, sat:sat, $
	phase_instr:dat.phase_instr, labels:labels, panel_size:2}



    END

    ELSE: BEGIN
     MESSAGE,'Moment not implemented:'+moments,/inf
     get_err_no =1
     RETURN
    END

  ENDCASE
END
