;+
; FUNCTION: get_pla_wap_priority_ssd
;
; PURPOSE: To create the data structure of the 'wap_priority_ssd' product
;
; INPUT: sat: S/C id - 'A' or 'B'
;
; OUTPUT: the data structure is returned
;
; CREATED BY: C. Mouikis
;
; LAST MODIFICATION:
;
; MODIFICATION HISTORY:
;   05-15-2007: CM
;               Actual maxrec is epochinfo.maxrec+1
;
;-

FUNCTION get_pla_wap_priority_ssd, sat, class

COMMON get_error, get_err_no, get_err_msg, default_verbose
get_err_no = 0 & get_err_msg = ''

prod        = 'wap_priority_ssd'
nenergy     =  128
nposition   =  1
ndeflection =  1

seek_files, sat, 'science', filesfound ; find files that correspond to time interval
iff = WHERE(filesfound NE '',  ciff)
IF ciff EQ 0 THEN BEGIN
    get_err_no =  1
    get_err_msg =  'No files found'
    RETURN, 0
ENDIF

;------------------------------------------------------------------
; Read CDF files for the specific product related parameters
;------------------------------------------------------------------
ifwd = 0
FOR jj = 0, N_ELEMENTS(filesfound)-1 DO BEGIN

    IF filesfound(jj) NE '' THEN BEGIN

        cdf_id = cdf_open(filesfound(jj))

        prod_name = 'epoch5_heavy'
        cdf_control, cdf_id, get_var_info = epochinfo, var = prod_name, /zvariable

        IF epochinfo.maxrec GE 0 THEN BEGIN
            
            cdf_varget, cdf_id, prod_name, epoch, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'wap_priority_ssd'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable
            cdf_varget, cdf_id, prod_name, cdf_data, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 's_chan5_heavy'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_s_chan5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
            
            prod_name = 'heavy_ion_num_summed5'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_heavy_ion_num_summed5, rec_count = epochinfo.maxrec+1, /zvariable

            prod_name = 'error5_heavy'
            cdf_control, cdf_id, get_var_info = varinfo, var = prod_name, /zvariable  
            cdf_varget, cdf_id, prod_name, cdf_error5_heavy, rec_count = epochinfo.maxrec+1, /zvariable
            
            IF ifwd EQ 0 THEN BEGIN
                converted_epoch       = REFORM(time_double(epoch, /epoch))
                data                  = TRANSPOSE(cdf_data)
                s_chan5_heavy         = REFORM(cdf_s_chan5_heavy)
                heavy_ion_num_summed5 = TRANSPOSE(cdf_heavy_ion_num_summed5)
                error5_heavy          = REFORM(cdf_error5_heavy)
            ENDIF ELSE BEGIN
                converted_epoch       = [converted_epoch, REFORM(time_double(epoch, /epoch))]
                data                  = [data,            TRANSPOSE(cdf_data)]
                s_chan5_heavy         = [s_chan5_heavy,   REFORM(cdf_s_chan5_heavy)]
                heavy_ion_num_summed5 = [heavy_ion_num_summed5, TRANSPOSE(cdf_heavy_ion_num_summed5)]
                error5_heavy          = [error5_heavy,    REFORM(cdf_error5_heavy)]
                
            ENDELSE
            
            ifwd =  ifwd + 1
            
        ENDIF

        cdf_close, cdf_id

    ENDIF
    
ENDFOR

IF ifwd EQ 0 THEN BEGIN

    get_err_no =  1
    get_err_msg =  'No data found for time interval'
    RETURN,  0

ENDIF

;------------------------------------------------------------------
; Restrict data array to classification required
;------------------------------------------------------------------
data =  REFORM(data(*, class, *))

;------------------------------------------------------------------
; Take out packets that have an error flag raised
;------------------------------------------------------------------
findfill = WHERE(error5_heavy EQ 0, cff)
IF cff GT 0 THEN BEGIN
    
    converted_epoch = converted_epoch(findfill)
    s_chan5_heavy = s_chan5_heavy(findfill)
    error5_heavy = error5_heavy(findfill)
    heavy_ion_num_summed5 = heavy_ion_num_summed5(findfill, *)
    data = data(findfill, *)
    
ENDIF

;------------------------------------------------------------------
; Check individual arrays for value range validity
;------------------------------------------------------------------

icepoch =  WHERE(converted_epoch LT (time_double('1958-01-01')) OR $
                 converted_epoch GT (time_double('2050-01-01')), cicepoch)
IF cicepoch GT 0 THEN STOP

data =  FLOAT(data)
icdata = WHERE(data EQ -1, cicdata)
IF cicdata GT 0 THEN BEGIN
    data(icdata) =  !VALUES.F_NAN
ENDIF

isch =  WHERE(s_chan5_heavy LT 0 OR s_chan5_heavy GT 127,  cisch)
;IF cisch GT 0 THEN STOP

ierr =  WHERE(error5_heavy NE 0 AND error5_heavy NE 1,  cierr)
;IF cierr GT 0 THEN STOP

;------------------------------------------------------------------
;convert energy/deflection/position/s_channel steps to actual values
;------------------------------------------------------------------
get_pla_energies, sat, esa_step             ; get energy bin values range
get_pla_theta_angles, sat, prod, theta_step ; get polar (deflection) angles range
get_pla_phi_angles, sat, prod, phi_step     ; get azimuthal (position) angles range

;position   = FLTARR(N_ELEMENTS(converted_epoch), nposition)
deflection = FLTARR(N_ELEMENTS(converted_epoch), ndeflection)
energy = FLTARR(N_ELEMENTS(converted_epoch), 128)

FOR ii = 0, N_ELEMENTS(converted_epoch)-1 DO BEGIN

    energy(ii, *) =  esa_step
    deflection(ii, *) =  theta_step
;    position(ii, *) =  phi_step

ENDFOR

;convert s_chan5_heavy to energy
s_chan5_heavy = esa_step(s_chan5_heavy)

;------------------------------------------------------------------
; Time tag of the middle of the bin
;------------------------------------------------------------------
converted_epoch = converted_epoch + 150. ; time shift by 2.5 minutes

;------------------------------------------------------------------
; Integration time
;------------------------------------------------------------------
packets = N_ELEMENTS(converted_epoch)

dt =  12.8e-3 ; ms - Integration time for one deflection

integ_t = FLTARR(packets, nenergy)
FOR ip = 0, packets-1 DO BEGIN
    FOR ien = 0, nenergy-1 DO BEGIN
        
        integ_t(ip, ien) = dt * heavy_ion_num_summed5(ip, ien) * 32.

    ENDFOR
ENDFOR

;------------------------------------------------------------------
; Construct the dara structure
;------------------------------------------------------------------
retdata = {$
          project:     'STEREO/PLASTIC', $
          prod_name:   prod, $
          units_name:  'Counts', $
          ntime:       N_ELEMENTS(converted_epoch), $
          time:        converted_epoch, $
          integ_t:     integ_t, $
          data:        data, $
          nenergy:     nenergy, $
          energy:      energy, $
          nposition:   nposition, $
;          position:    position, $
          ndeflection: ndeflection, $
          deflection:  deflection, $
          s_chan:      s_chan5_heavy, $
          error:       error5_heavy $
          }

RETURN, retdata

END
